

#ifndef _dwgsurf_h_ /* Thu Sep 29 11:34:19 1994 */
#define _dwgsurf_h_




/*
 *
 *          Copyright (C) 1994, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1994. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



// A DrawingSurface is an abstract class representing a surface that can be
// drawn on.

// Author: M. A. Sridhar


#if defined(__GNUC__)
#pragma interface
#endif


#include "base/object.h"

#include "ui/rectangl.h"
#include "ui/font.h"
#include "ui/pen.h"
#include "ui/brush.h"
#include "ui/uidefs.h"
// #include "ui/colormap.h"

#if defined(__MS_WINDOWS__)
typedef  long UI_DwgSurfHandle;

#elif defined(__OS2__)
typedef  ulong UI_DwgSurfHandle;

#elif defined(__X_MOTIF__)
struct _XGC;
typedef struct _XGC* UI_DwgSurfHandle;

#endif

class CL_EXPORT UI_VisualObject;
class CL_EXPORT UI_Bitmap;
class CL_EXPORT UI_Arc;
class CL_EXPORT UI_Chord;
class CL_EXPORT UI_PieWedge;


enum UID_DrawOptions {UID_Outline = 1, UID_Fill = 2};

class CL_EXPORT UI_DrawingSurface: public CL_Object {

public:

    UI_DrawingSurface ();

    ~UI_DrawingSurface () {};

    // 
    // --------------------------- Query ---------------------------------
    //     

    virtual double HorzPixelsPerMM () const = 0;
    // Return the number of pixels per millimeter in the horizontal
    // direction.

    virtual double VertPixelsPerMM () const = 0;
    // Return the number of pixels per millimeter in the vertical direction.

    virtual UI_Rectangle DrawingArea() const = 0;
    // Returns the rectangle enclosing the drawing surface, in pixels. The
    // origin is (0, 0).

    virtual UI_Rectangle DrawingAreaInMM () const = 0;
    // Returns the drawing area, in millimeters. The origin is (0, 0).

    virtual bool SupportsColor () const {return TRUE;};
    // Return TRUE if this DrawingSurface supports color, FALSE otherwise.
    // The default implementation returns TRUE unconditionally.
    
    // Pen, Font and Brush  are resources
    // maintained by this object. These may be accessed and modified 
    // via the following reference methods.

    virtual UI_Font& Font();
    // Return (a reference to) the font used by this surface. The return
    // value can be modified by the Font methods.

    virtual UI_Pen&  Pen();
    // Return (a reference to) the pen used by this surface. The return
    // value can be modified by the Pen methods.


    virtual UI_Brush& Brush();
    // Return (a reference to) the background brush used by this surface.
    // The return value can be modified by the Brush methods.

    
    // 
    // -------------------------- Draw  operations ------------------------
    // 


    virtual bool DrawLine (const UI_Point& p, const UI_Point& q);
    // Draw a line from {\tt p} to {\tt q}, using our pen.

    virtual void ColorDisplay (const UI_Color& c);
    // Paint client rectangle with specified color.

    virtual void ColorRectangle (const UI_Rectangle& r, const UI_Color& cs);
    // Convenience function to color the interior of the given rectangle.
    
    virtual void DrawRectangle (const UI_Rectangle& rect,
                                ushort opt = UID_Outline); 
    // Draw a rectangle using the current pen. The second parameter
    // specifies the outline and fill options.
                
    virtual void DrawEllipse (const UI_Rectangle& rect,
                              ushort opt = UID_Outline); 
    // Draw an ellipse with given bounding rectangle, using the current pen.
    // The second parameter specifies the outline and fill options.
    // (If the "rectangle" happens to be a square, then a circle is drawn.)
                
    virtual void DrawArc (const UI_Arc &arc);
    // Draw an arc with the using the parameters given to UI_Arc.

    virtual void DrawChord (const UI_Chord& chord,
                            ushort opt = UID_Outline); 
    // Draw a chord using the current pen. The second parameter
    // specifies the outline and fill options.

    virtual void DrawPieWedge (const UI_PieWedge& pie,
                               ushort opt = UID_Outline); 
    // Draw a pie wedge using the current pen. The second parameter
    // specifies the outline and fill options.
        
    virtual void DrawPolygon (UI_Point pointArray[], short numPoints,
                              ushort opt = UID_Outline);
    // Draw a closed polygon by connecting the points in the array with
    // line segments, in the order of occurrence of the points.  The second
    // parameter specifies the outline and fill options.
    // The first point in the array is taken as the starting point.

    virtual void DrawPolyLine (UI_Point pointArray[], short numPoints);
    // Draw a figure by connecting the points in the array with
    // line segments, in the order of occurrence of the points. 
    // The first point in the array is taken as the starting point.

    virtual void DrawPoint (const UI_Point& p, const UI_Color& color);
    // Draw a point at position {\tt p} with given color.

    virtual bool DrawBitmap (const UI_Bitmap& b, const UI_Point& topLeft) = 0;
    // [Pure virtual] Draw the given bitmap on this surface.
     
    // 
    // ------------------------ Text operations -------------------------
    // 

    virtual void WriteString (const CL_String& str, const UI_Rectangle& box,
                              UI_TextStyle style = UIText_Left);
    // Write the given text string within the given box, in the given
    // style. This method is guaranteed to handle a single line, not
    // necessarily multiple lines. It draws the text within the rectangle,
    // clipping the text if necessary.

    
    virtual short TextWidth (const char* s) const;
    // Return the width (in pixels) of the text string s if it were drawn
    // on this DisplaySurface using its current font.



    // Basic methods:
    
    const char* ClassName () const {return "UI_DrawingSurface";};

    // -------------------- End public protocol -------------------------

    
protected:
    UI_Pen*          _pen;       // These must be set up by the derived class.
    UI_Font*         _font;
    UI_Brush*        _brush;
    UI_DwgSurfHandle _handle;
    long             _horzPixels;
    long             _vertPixels;
    
public:

    UI_DwgSurfHandle Handle() const;
    // [For YACL internal use only]

};



inline void UI_DrawingSurface::ColorDisplay (const UI_Color& c)
{
    ColorRectangle (DrawingArea (), c);
}


inline UI_DwgSurfHandle UI_DrawingSurface::Handle () const
{
    return _handle;
}


inline UI_Font& UI_DrawingSurface::Font ()
{
    return *_font;
}

inline UI_Pen& UI_DrawingSurface::Pen ()
{
    return *_pen;
}

inline UI_Brush& UI_DrawingSurface::Brush ()
{
    return *_brush;
}


 
#endif /* _dwgsurf_h_ */
