
/*
 *@@sourcefile xdoc.h:
 *      header for xdoc.cpp.
 */

/*      Copyright (C) 1999-2000 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of the XFolder main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

/* ******************************************************************
 *                                                                  *
 *   Declarations                                                   *
 *                                                                  *
 ********************************************************************/

class XDBase;
typedef XDBase *PXDBase;

class XDIndexable;
typedef XDIndexable *PXDIndexable;

class XDDeclBase;
typedef XDDeclBase *PXDDeclBase;

class XDDefine;
typedef XDDefine *PXDDefine;

class XDFunction;
typedef XDFunction *PXDFunction;

class XDClass;
typedef XDClass *PXDClass;

class XDSomClass;
typedef XDSomClass *PXDSomClass;

class XDSourceFile;
typedef XDSourceFile *PXDSourceFile;

class XDCategory;
typedef XDCategory *PXDCategory;

/*
 *@@ XDDate:
 *      encapsulation of a date. This can be used
 *      for quick STL sorting because it supports
 *      the "<" operator.
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

struct XDDate
{
    ULONG       _ulYear,
                _ulMonth,
                _ulDay;
    /*
     *@@ XDDate:
     *      constructor.
     */

    XDDate()
    {
        memset(this, 0, sizeof(*this));
    }

    /*
     *@@ operator<:
     *
     */

    virtual bool operator<(const XDDate &X) const
    {
        return (    (_ulYear < X._ulYear)
                 || (   (_ulYear == X._ulYear)
                                    // major versions are the same:
                                    // compare minor versions then
                     && (   (_ulMonth < X._ulMonth)
                         || (   (_ulMonth == X._ulMonth)
                                    // minor versions are the same too:
                                    // compare revisions then
                             && (_ulDay < X._ulDay)
                            )
                        )
                    )
                );
    }

    /*
     *@@ operator==:
     *
     */

    bool operator== (const XDDate &X) const
    {
        return (    (_ulYear == X._ulYear)
                 && (_ulMonth == X._ulMonth)
                 && (_ulDay == X._ulDay)
               );
    }

};

/*
 *@@ XDVersion:
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

struct XDVersion
{
    ULONG       _ulMajor,
                _ulMinor,
                _ulRevision;

    /*
     *@@ XDVersion:
     *
     */

    XDVersion()
    {
        memset(this, 0, sizeof(*this));
    }

    /*
     *@@ operator<:
     *
     */

    virtual bool operator< (const XDVersion &X) const
    {

        return (    (_ulMajor < X._ulMajor)
                 || (   (_ulMajor == X._ulMajor)
                                    // major versions are the same:
                                    // compare minor versions then
                     && (   (_ulMinor < X._ulMinor)
                         || (   (_ulMinor == X._ulMinor)
                                    // minor versions are the same too:
                                    // compare revisions then
                             && (_ulRevision < X._ulRevision)
                            )
                        )
                    )
                );
    }

    /*
     *@@ operator==:
     *
     */

    bool operator== (const XDVersion &X) const
    {
        return (    (_ulMajor == X._ulMajor)
                 && (_ulMinor == X._ulMinor)
                 && (_ulRevision == X._ulRevision)
               );
    }
};

/*
 *@@ XDChangeLogEntry:
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

class XDChangeLogEntry : public BSRoot
{
    protected:
        DECLARE_CLASS(XDChangeLogEntry);

    public:
        XDDate          _date;
        XDVersion       _version;
        BOOL            _fAdded;            // if TRUE, @@added entry;
                                            // if FALSE, @@changed entry
        string          _strAuthor;         // author of change
        string          _strDescription;    // description of change

        XDDeclBase      *_pMyDecl;
                // declaration this log entry belongs to or NULL
        XDSourceFile    *_pMySourceFile;
                // source file this log entry belongs to or NULL;
                // one of these two must be != NULL

        XDChangeLogEntry(XDDeclBase *pDecl,
                         XDSourceFile *pSourceFile,
                         string &strDeclName,
                         PSZ pFound,
                         BOOL fAdded);

        /*
         *@@ XDChangeLogEntry:
         *      default constructor, added to allow
         *      subclassing.
         *
         *@@added V0.9.18 (2002-02-25) [lafaix]
         */

        XDChangeLogEntry(BSClassID &Class)
            : BSRoot(Class)
        {
            memset(this, 0, sizeof(*this));
        }

        virtual VOID WriteHTMLDescription(FILE *ThisFile);
        virtual BOOL ParseEntry(PSZ pEntry, PSZ *ppError);

        /*
         *@@ operator<:
         *
         */

        bool operator< (const XDChangeLogEntry &X) const
        {
            return (    (_version < X._version)
                     || (   (_version == X._version)
                                        // versions are the same:
                                        // compare dates then
                         && (_date < X._date)
                        )
                    );
        }

        /*
         *@@ operator==:
         *
         */

        bool operator== (const XDChangeLogEntry &X) const
        {
            return (    (_version == X._version)
                     && (_date == X._date)
                   );
        }
};

typedef XDChangeLogEntry *PXDChangeLogEntry;

/*
 *@@ XDReviewEntry:
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

class XDReviewEntry : public XDChangeLogEntry
{
    protected:
        DECLARE_CLASS(XDReviewEntry);

    public:
        XDReviewEntry(XDSourceFile *pSourceFile,
                      PSZ pFound);

        virtual VOID WriteHTMLDescription(FILE *ThisFile);
};

typedef XDReviewEntry *PXDReviewEntry;

/*
 *@@ XDArgument:
 *      stored in linked list in XDFunction.
 */

class XDArgument : public BSRoot
{
    protected:
        DECLARE_CLASS(XDArgument);

    public:
        string      _strType,
            // e.g. "unsigned long"
                    _strIdentifier,
            // e.g. "ulWhatever"
                    _strArgComment;
            // comment which was found for this arg

        XDArgument(const string &strArgument_,
                   const string &strArgComment_,
                   BOOL *pfHasNonVoidArgs);
};
typedef XDArgument *PXDArgument;

/* ******************************************************************
 *                                                                  *
 *   XDBase and subclasses                                          *
 *                                                                  *
 ********************************************************************/

/*
 *@@ XDBase:
 *      abstract base class for everything which
 *      will turn into an HTML output file.
 *
 *      This gives us the following class hierarchy:
 +
 +          XDBase (HTMLable)
 +           |
 +           +- XDDeclBase (resolveable)
 +           |   |
 +           |   +- XDIndexableBase (indexable)
 +           |   |   |
 +           |   |   +- XDDefine
 +           |   |   |
 +           |   |   +- XDFunction
 +           |   |   |
 +           |   |   +- XDClass
 +           |   |       |
 +           |   |       +- XDSomClass
 +           |   |
 +           |   +- XDGlossary
 +           |
 +           +- XDCategory
 +           |
 +           +- XDSourceFile
 *
 *      While XDBase is the root for everything that
 *      can turn into HTML, only instances of
 *      XDIndexableBase will appear in the "all decls"
 *      index. This excludes glossary entries.
 *
 *      In addition, only XDDeclBase is resolveable automatically.
 *      XDSourceFile still is a bit of an exception
 *      because it is resolved as well, but separately.
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

class XDBase : public BSRoot
{
    public:
        DECLARE_CLASS(XDBase);

    public:
        string      _strIndexNameOrig,
                    _strIndexNameHTML;
                // name of this declaration. Yes, it is illogical that
                // only XDIndexableBase is indexable and we have the index
                // name here, but XDSourceFile uses this as well.
                // XDCategory:
                // -- category name
                // XDDefine:
                // -- definition name
                // XDFunction:
                // -- if SOM function: shortended method name, e.g. "wpInitData"
                // -- if C++ member function: full name (e.g. "XDBase::XDBase")
                // -- if plain C function: == pszOrigDeclName
                // -- if struct: == pszOrigDeclName
                // XDGlossary:
                // -- readable name (while XDDeclbase::strOrigDeclName has the "@id")

        string      _strOutputFilename;
                // filename for HTML output

        XDSourceFile *_pSourceFile;
                // where this declaration resides

        XDCategory  *_pMyCategory;
                // category this declaration belongs to or NULL if none

        string      _strCommentOriginal,
                // unformatted comment
                    _strCommentNew;
                // formatted comment

        list<PXDChangeLogEntry> _ChangeLogsList;
                // list of change logs for this item;
                // the list items point into the global list
        XDChangeLogEntry    *_pAddedLog;
                // ptr to "added" log entry on that list or NULL
                // if there's none

        XDBase(const string &strIndexName,
               BSClassID &Class);

        virtual VOID CreateHTMLFile();

        virtual const char* QueryHTMLAttributes();

        /*
         *@@ QueryHTMLTitle:
         *
         */

        virtual const char* QueryHTMLTitle()
        {
            return "unknown";
        }

        virtual VOID WriteHTMLBody(FILE *File);

        /*
         *@@ WriteHTMLDescription:
         *      pure virtual method used to dump the
         *      HTML contents, to be overridden
         *      by subclasses.
         */

        virtual VOID WriteHTMLDescription(FILE* ThisFile) = 0;

        /*
         *@@ QueryOutputFile:
         *      returns the HTML output filename.
         */

        const char* QueryOutputFilename()
        {
            return (_strOutputFilename.c_str());
        }

        /*
         *@@ operator<:
         *
         *@@changed V0.9.18 (2002-03-16) [umoeller]: now sorting case-insensitively
         */

        virtual bool operator< (const XDBase &X) const
        {
            PCSZ p1 = _strIndexNameOrig.c_str(),
                 p2 = X._strIndexNameOrig.c_str();
            if (p1 && p2)
                return (stricmp(p1, p2) < 0);
            else if (p2)
                return TRUE;
            return FALSE;
        }

        /*
         *@@ operator==:
         *
         */

        virtual bool operator== (const XDBase &X) const
        {
            return (_strIndexNameOrig == X._strIndexNameOrig);
        }
};

// flags for XDDeclBase.ulDeclType
/* #define DECL_FUNCTION         1     // function
#define DECL_DEFINE           2     // #define
#define DECL_CLASS            3     // struct or class
#define DECL_GLOSSARY         4     // XDGlossary
   */

/*
 *@@ XDDeclBase:
 *      base class for declarations.
 *      One of these is created of every declaration found in a
 *      source file. A linked list of these structures is
 *      maintained in each XDSourceFile instance.
 *
 *      XDDeclBase per definition is resolveable, i.e.
 *      instances of all subclasses of this will be
 *      resolved in all comments.
 *
 *      This has two subclasses in turn. XDIndexable
 *      is used for functions, structs, classes etc.
 *      that should appear in the index. By contrast,
 *      XDGlossary is resolveable, but should not appear
 *      in the index.
 *
 *      This thing supports the "==" and "<" operators so
 *      it can be used with STL sorting.
 */

class XDDeclBase : public XDBase
{
    protected:
        DECLARE_CLASS(XDDeclBase);

    protected:
        // ULONG       _ulDeclType;
                // DECL_* flags

    public:
        string      _strOrigDeclName;
                // decl. name as found in source file,
                // -- if SOM function: e.g. "xf_wpInitData";
                // -- if C++ member function: e.g. "XDBase::XDBase";
                // -- if non-SOM function: == pszRealName
                // -- if struct: name which immediately follows "struct" keyword
                // -- if glossary entry: "@identifier" (while XDBase::strIndexName
                //              receives the readablename)

        // list<string> _IdentifiersList;
                // list of identifiers for this item;
                // this always has at least one item, but maybe
                // several if typedef's have been declared for this item.
                // The exact content of an item depends on the ulDeclType:
                // -- For methods (SOM and C++), this has the fully qualified
                //    declarator, e.g. "XFolder::wpInitData" (only one item).
                // -- For regular functions (non-methods), this is == pszOrigDeclName
                //    (only one item).
                // -- For structures, this has all the different structure names,
                //    e.g. _STRUCTURE, STRUCTURE, PSTRUCTURE, if the typical C-style
                //    "typedef" structure definition has been used.

        string      _strSearchString;
                // string to search for in comment text when resolving references.
                // This is normally the same as XDBase::_strIndexName, except
                // for two subclasses:
                // -- XDFunction appends brackets to the search string (e.g. "funcname()")
                //    if the length of the funcname is less than six. This avoids
                //    replacing all occurrences of "main" and similar words in the
                //    comments.
                // -- XDGlossary prefixes the search string with a single "@" character.
                //    This makes sure that glossary entries are only replaced when
                //    they are explicitly referenced.

        string      _strReplacementString;
                // string to replace _strSearchString with when resolving references.
                // If this is empty, _strSearchString is used. This is only set
                // for XDGlossaray to insert the glossary term (which may include
                // spaces) instead of the glossary declarator (which must be a
                // single word).

        XDDeclBase(XDSourceFile *pSourceFile_,
                   CHAR cFilenamePrefix,
                   const string &strOrigDeclName_,
                   const string &strIndexName,
                   BSClassID &Class);

        /*
         *@@ ~XDDeclBase:
         *
         */

        virtual ~XDDeclBase() { }

        virtual ULONG ParseDeclChangeLogs();

        virtual VOID Resolve();

        virtual const char* QueryHTMLAttributes();

        virtual VOID WriteHTMLDescription(FILE* ThisFile);
};

/*
 *@@ XDGlossary:
 *      XDDeclBase subclass for glossary entries.
 *
 *@@added V0.9.1 (2000-01-26) [umoeller]
 */

class XDGlossary : public XDDeclBase
{
    protected:
        DECLARE_CLASS(XDGlossary);

    public:
        string      _strReadableName;

        XDGlossary(XDSourceFile *pSourceFile_,
                   const string &strDefName_,           // identifier (no spaces)
                   const string &strReadableName,      // readable name (with spaces)
                   string &strComment);

        virtual ~XDGlossary() {};

        /*
         *@@ QueryHTMLTitle:
         *      overridden pure virtual base method.
         */

        virtual const char* QueryHTMLTitle()
        {
            return "glossary entry";
        }

        virtual VOID WriteHTMLBody(FILE *File);
};

/*
 *@@ XDTypedef:
 *      XDIndexableBase subclass for typedefs. These
 *      are simply "shadows" with an internal link
 *      to some other XDDeclBase to allow giving
 *      a definition several names.
 *
 *      Since this is not a subclass of XDIndexableBase,
 *      the typedefs are resolveable, but not
 *      indexable.
 *
 *@@added V0.9.14 (2001-07-14) [umoeller]
 */

class XDTypedef : public XDDeclBase
{
    public:
        DECLARE_CLASS(XDTypedef);

        XDDeclBase *_pLinkedTo;

        XDTypedef(XDSourceFile *pSourceFile,
                  const string &strDefName,
                  XDDeclBase *pLinkedTo);
};

/*
 *@@ XDIndexableBase:
 *
 *@@added V0.9.9 (2001-02-10) [umoeller]
 */

class XDIndexableBase : public XDDeclBase
{
    public:
        DECLARE_CLASS(XDIndexableBase);

    public:
        /*
         *@@ XDIndexableBase:
         *
         */

        XDIndexableBase(XDSourceFile *pSourceFile_,     // in: passed to XDDeclBase
                        CHAR cFilenamePrefix,           // in: passed to XDDeclBase
                        const string &strOrigDeclName_, // in: passed to XDDeclBase
                        const string &strIndexName,     // in: passed to XDDeclBase
                        BSClassID &Class)               // in: passed to XDDeclBase
                : XDDeclBase(pSourceFile_,
                             cFilenamePrefix,
                             strOrigDeclName_,
                             strIndexName,
                             Class)
        {
        }

        virtual ~XDIndexableBase() {};
};

/*
 *@@ XDDefine:
 *      XDIndexableBase subclass for simple defines.
 *
 *@@added V0.9.1 (2000-01-26) [umoeller]
 */

class XDDefine : public XDIndexableBase
{
    protected:
        DECLARE_CLASS(XDDefine);

    public:
        XDDefine(XDSourceFile *pSourceFile_,
                 const string &strDefName_,
                 const string &strDefDefinition);

        string      _strDefDefinition;      // whole definition; can be empty

        /*
         *@@ QueryHTMLTitle:
         *
         */

        virtual const char* QueryHTMLTitle()
        {
            return "define";
        }

        virtual VOID WriteHTMLDescription(FILE* ThisFile);
};

/*
 *@@ XDFunction:
 *      XDIndexableBase subclass for functions and methods.
 */

class XDFunction : public XDIndexableBase
{
    protected:
        DECLARE_CLASS(XDFunction);

    public:
        // The following fields are used for DECL_FUNCTION only:
        string      _strBeforeFunctionName,  // e.g. "ULONG SOMLINK"
                    _strFunctionReturn;      // e.g. "PVOID" (first element in line)

        string      _strPureName;           // pure name without leading class name

        BOOL        _fIsStaticFunction,
                    _fIsVirtualFunction;

        list<PXDArgument> _ArgList;

        string      _strMangledArgs;        // arguments list (e.g. "(const char *, const char*)"

        string      _strOwningClass;
                // if this is a class member, the owning class name...
                // this gets assigned for member functions defined in
                // the source code outside a class definition, and is
                // later used by main() to later resolve _pOwningClass

        XDClass     *_pOwningClass;
                // if this is a class member, the owning class.
                // This is only resolved after all source files have
                // been fully parsed... because we can't know all
                // classes before we're done with all this.

                // if this is a member function, these lists show
                // the overrides
        list<PXDFunction> _OverridesList,
                          _OverriddenByList;

        static XDFunction* CreateFunction(XDSourceFile *pSourceFile,
                                          XDClass *pOwningClass,
                                          string &strComment,
                                          string &strDeclBlock);

        XDFunction(XDSourceFile *pSourceFile_,
                   const string &strOrigDeclName_,
                   const string &strIndexName,
                   const string &strBeforeFunctionName,
                   const string &strMangledArgs,
                   list<XDArgument*> &ArgList);

        /*
         *@@ QueryHTMLTitle:
         *
         */

        virtual const char* QueryHTMLTitle()
        {
            return "function";
        }

        VOID DumpList(FILE* ThisFile,
                      PCSZ pcszTitle,
                      list<PXDFunction> &List);

        virtual VOID WriteHTMLDescription(FILE* ThisFile);
};

/*
 *@@ XDClass:
 *      XDIndexableBase subclass for class/struct declarations.
 */

class XDClass : public XDIndexableBase
{
    public:
        DECLARE_CLASS(XDClass);

    public:
        string          _strNameForResolve;
                // class name without <T> if it's a template;
                // otherwise the same as strOrigDeclName...
                // this is needed to resolve parent template
                // classes right

        // The following fields are used for DECL_STRUCT only:
        string          _strDefinition;
                // entire structure definition
                // (copied from source)

        list<string*>   _llParentNames;
                // class names of parent classes, if any
        ULONG           _cParentNames;
                // not calculated until all classes have been loaded...

        PXDClass        _pParent;
                // parent class or NULL

        list<PXDClass>  _DescendantsList;
        ULONG           _cDescendants;

        list<PXDFunction> _MemberFunctionsList;
        ULONG           _cMemberFunctions;

        XDClass(XDSourceFile *pSourceFile_,
                const string &strOrigDeclName_,
                const string &strNameForResolve,
                const string &strDefinition,
                BSClassID &Class);

        virtual VOID Resolve();     // overridden

        /*
         *@@ QueryHTMLTitle:
         *
         */

        virtual const char* QueryHTMLTitle()
        {
            return "class";
        }
        virtual VOID WriteHTMLDescription(FILE* ThisFile);
        // virtual VOID WriteHTMLDescription(FILE* ThisFile);
};

/*
 *@@ XDSomClass:
 *      special subclass for SOM classes, which
 *      can be declared in C source files using
 *      the @@somclass tag.
 */

class XDSomClass : public XDClass
{
    protected:
        DECLARE_CLASS(XDSomClass);

    public:
        string    _strSomFuncPrefix;  // e.g. "xfdisk_"

        /*
         *@@ XDSomClass:
         *
         */

        XDSomClass(XDSourceFile *pSourceFile_,
                   const string &strClassName,
                   const string &strPrefix)
            : XDClass(pSourceFile_,
                      strClassName,
                      strClassName,         // there are no templates in SOM
                      "SOM class. No definition available.",
                      tXDSomClass)
        {
            _strSomFuncPrefix = strPrefix;
            _strCommentOriginal
                = _strCommentNew
                = "SOM class. No definition available.";
        }
};

/*
 *@@ XDCategory:
 *      created from a @@category tag.
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

class XDCategory : public XDBase
{
    protected:
        DECLARE_CLASS(XDCategory);

    public:
        ULONG       _ulIndex;       // category index, starting with 1
        // string      _strName;       // name of this category item only;
                                    // the full category can be recomposed
                                    // by climbing up the parents
        string      _strLinkNameHTMLShort;
        string      _strLinkNameHTMLLong; // full category name (all parents included)
        XDCategory  *_pParent;      // parent category or NULL if none

        list<PXDCategory> _ChildCategoriesList;
                            // subcategories of this category
        ULONG       _cDirectChildCategories,
                            // items count on that list
                    _cTotalChildCategories;
                            // item count on that list and on all child lists

        list<PXDDeclBase> _MemberDeclsList;
                            // all declarations which are part of this
                            // category (i.e. appear after the category
                            // definition in the source file)
        ULONG       _cDirectMemberDecls,
                            // items count on that list
                    _cTotalMemberDecls;
                            // items count on that list and on all child lists

        XDCategory(string &strName,
                   XDCategory *pParent,
                   XDSourceFile *pFirstSourceFile,
                   string &strComment);

        /*
         *@@ QueryHTMLTitle:
         *
         */

        virtual const char* QueryHTMLTitle()
        {
            return "category";
        }

        virtual VOID WriteHTMLDescription(FILE* ThisFile);

};

/*
 *@@ XDSourceFile:
 *      structure to hold information for an
 *      input sourcefile. One of these is
 *      created for each source which is read
 *      in.
 */

class XDSourceFile : public XDBase
{
    protected:
        DECLARE_CLASS(XDSourceFile);

    public:
        ULONG           _ulFileIndex;

        ULONG               _ulCommentCount,
                            _ulTagsCount;

        /* CHAR            // _szShortFilename[CCHMAXPATH],
                    // not qualified: XDBase::_strIndexName
                        _szFilePath[CCHMAXPATH],
                    // path (last char must be "\\")
                        _szLongFilename[CCHMAXPATH];
                    // fully qualified (szFilePath + szShortFilename)
                        // _szHTMLFile[CCHMAXPATH];
          */

        string          _strFilePath,
                        _strLongFilename;

        ULONG           _ulFunctionCount;
        // string          _strFileComment;
        string          _strInclude;

        // header information
        BOOL            _fIsHeader;         // TRUE if produced from a @@header tag
        string          _strHeader;         // only if (!fIsHeader): @@header tag value of source file
        XDSourceFile    *_pMyHeader;         // only if (!fIsHeader): XDSourceFile struct with header (which has fHeader == TRUE)

        list<PXDDeclBase>    _AllDeclsList;
        list<PXDDefine>      _DefinesList;
        list<PXDFunction>    _FunctionsList;
        list<PXDClass>       _ClassesList;
        list<PXDSomClass>    _SomClassesList;

        ULONG _ulReviewCount;
        list<PXDReviewEntry> _ReviewsList;
                // list of reviews for this item;
                // the list items point into the global list
                // V0.9.18 (2002-02-25) [lafaix]

        list<PXDChangeLogEntry> _ChangeLogsList;
                // list of change logs for this item;
                // the list items point into the global list
        XDChangeLogEntry    *_pAddedLog;
                // ptr to "added" log entry on that list or NULL
                // if there's none

        XDCategory          *_pCurrentCategory;
                // current category while parsing source file;
                // do not use after this!

        XDSourceFile(const string &strShortFilename,
                     const string &strFilePath,
                     ULONG ulFileIndex_,
                     PXDSourceFile pFileForWhichThisIsHeader);

        /*
         *@@ ~XDSourceFile:
         *
         */

        ~XDSourceFile() { };

        XDDefine* ParseDefine(string &strComment,
                              PCSZ pAfterComment);

        XDClass* ParseClass(string &strComment,
                            string &strDeclBlock,
                            PCSZ *ppDeclBlockInFile);

        BOOL ParseDeclComment(XDClass *pOwningClass,
                              string &strComment,
                              PCSZ *ppAfterComment);

        ULONG ParseSourcefileChangeLogs();
        ULONG ParseSourcefileReviews();

        ULONG ParseSourcefileComment(string &strComment);

        XDCategory* CreateCategory(string &strCategory,
                                   XDCategory *pParentCategory,
                                   string &strCategoryComment);

        ULONG ParseCategoryComment(string &strComment,
                                   ULONG pos);

        ULONG ParseGlossaryComment(string &strComment,
                                   ULONG pos);

        ULONG ParseBlock(PCSZ pSearchPos,
                         PXDClass pOwningClass);

        ULONG LoadAndParse();

        BOOL WillBeWritten();

        VOID WriteHTMLSourcefileProlog(FILE *ThisFile);

        virtual const char* QueryHTMLAttributes();

        /*
         *@@ QueryHTMLTitle:
         *
         */

        virtual const char* QueryHTMLTitle()
        {
            return "source file";
        }

        virtual VOID WriteHTMLDescription(FILE* ThisFile);
};

/*
 *@@ operator==:
 *
 */

bool operator==(const PXDSourceFile &s, const string &str)
{
    return (s->_strIndexNameOrig == str);
}
