
/*
 *@@sourcefile wpi2exe.cpp:
 *      This is the source file for the WarpIN Archive -> Exe converter.
 *      This is done by writing a startup block that will scan the appended
 *      resources. The created resource format is as follows:
 *
 *      --  A STRINGTABLE containing the filenames of the included WarpIN
 *          archives. The IDs should be beginning at 1 and increment without
 *          any jumps, since scanning of resources terminates when a string
 *          cannot be loaded by WinLoadString().
 *
 *      --  One or more (same number as the strings in the table)
 *          RESOURCEs of type RT_WPIARCHIVE, also with IDs from 1 on with
 *          increments of one.
 */

/*
 *      Copyright (C) 2000 Christoph Schulte Mnting.
 *      Copyright (C) 2000 Ulrich Mller.
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR

#define INCL_DOSPROCESS
#define INCL_DOSERRORS
#include <os2.h>

// standard C library includes
// #include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <stdarg.h>
#include <string.h>

#include "setup.h"

// helpers includes
#include "helpers\ansiscrn.h"
#include "helpers\dosh.h"
#include "helpers\resh.h"

// own includes
#include "bl_wpi2exe.h"
#include "rc_types.h"


#pragma hdrstop
/*
 *@@ hprintf:
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

void hprintf(int hFile,
             const char *pcszFormatString,
             ...)
{
    char szBuf[5000];
    va_list arg_ptr;

    // allocate memory
    va_start(arg_ptr, pcszFormatString);
    vsprintf(szBuf, pcszFormatString, arg_ptr);
    write(hFile, szBuf, strlen(szBuf));
    va_end(arg_ptr);
}

/*
 *@@ printHelp:
 *      Shows a screen with a help how to use this program.
 *
 *@@added V0.9.4 [csm]
 */

void printHelp()
{
    ANSI_attrib(BOLD);
    printf("wpi2exe V" BLDLEVEL_VERSION " " __DATE__ " - Creates self-extracting WarpIN launcher\n");
    ANSI_attrib(NORMAL);
    printf("(C) 2000 Christoph Schulte Mnting\n");
    ANSI_attrib(BOLD);
    printf("Usage:");
    ANSI_attrib(NORMAL);
    printf(" wpi2exe <file>.WPI [<file2>.WPI ...]\n");
    printf("This will create an executable file called <file>.EXE from <file>.WPI.\n");
    printf("If multiple WPI files are specified, all of them will be put into the EXE file.\n");
    printf("When <file>.EXE is then started, it will extract the WPI files from itself\n");
    printf("again and attempt to start WarpIN.\n");
    printf("WarpIN will be called with the first given wpi file.\n");
    printf("RC.EXE must be on your PATH for this to work.\n");
    printf("The wpi files do not have to be located in the current directory.\n");
}

/*
 *@@ fixFilename:
 *      Fixes the given filename for .RC files by doubling the backslashes.
 *
 *@@added V0.9.4 [csm]
 */

VOID fixFilename(PCSZ pcszOrigFilename,    // in: the filename to fix
                 PSZ pszFixedFilename)     // out: fixed filename
{
    ULONG cbOrigFilename = strlen(pcszOrigFilename);
    for (unsigned int i = 0, j = 0;
         i <= cbOrigFilename;
         i++)
    {
        char ch      = pcszOrigFilename[i];
        pszFixedFilename[j++] = ch;

        if (ch == '\\')
            pszFixedFilename[j++] = ch;
    }
}

/*
 *@@ getBasename:
 *      Returns a pointer to a new buffer which contains only the basename
 *      of the given file, i.e. the name without drive and directory.
 *
 *@@added V0.9.4 [csm]
 */

VOID getBasename(PCSZ pcszFilename,     // in: (full) filename
                 PSZ pszBasename)       // out: base name
{
    // skip directory if given
    const char* pcszBegin = strrchr(pcszFilename, '\\');

    if (pcszBegin != NULL)
        pcszBegin++;
    else
        pcszBegin = (char*)pcszFilename;

    // allocate space and copy basename
    strcpy(pszBasename, pcszBegin);
}

/*
 *@@ getPrefix:
 *      extracts only the prefix
 *      of the given file, i.e. the name
 *      without drive, directory and extension.
 *
 *@@added V0.9.4 [csm]
 */

VOID getPrefix(PCSZ pcszFilename,        // in: (full) filename
               PSZ pszPrefix)            // out: prefix
{
    CHAR szBegin[CCHMAXPATH];
    getBasename(pcszFilename, szBegin);

    // strip extension if given
    const char* pcszEnd = strrchr(szBegin, '.');

    if (pcszEnd == NULL)
        pcszEnd = szBegin + strlen(szBegin);

    // allocate space and copy basename
    strncpy(pszPrefix, szBegin, pcszEnd - szBegin);
}

/*
 *@@ writeRCfile:
 *      Writes an .RC file with the prefix of apszFilenames[0].
 *      This file contains a STRINGTABLE with the names of the
 *      included wpi archives and RESOURCE entries with references
 *      to the corresponding files (see the description at the
 *      beginning of wpi2exe.cpp).
 *
 *      Returns TRUE if the .RC file has been successfully written,
 *      FALSE if there have been errors.
 *
 *@@added V0.9.4 [csm]
 */

BOOL writeRCfile(PCSZ pcszPrefix,            // in: prefix of the .RC file
                 int  iFiles,                // in: number of files to include
                 PSZ  apszFilenames[])       // in: names of files to include
{
    BOOL    fSuccess = FALSE;

    CHAR    szRCName[CCHMAXPATH];

    sprintf(szRCName, "%s.rc", pcszPrefix);

    printf("\nCreating RC file %s ... ", szRCName);

    int hRCfile = open(szRCName, O_CREAT | O_WRONLY | O_BINARY, S_IWRITE | S_IREAD);
/* #ifdef __EMX__
    FILE* pRCfile = fopen(szRCName, "wt");
#else
    FILE* pRCfile = fopen(szRCName, "w");
#endif */
    if (hRCfile == -1)
        printf("Error creating file %s.", szRCName);
    else
    {
        char* pszTemp;

        hprintf(hRCfile, "#define RT_WPIARCHIVE %u\r\n\r\n", RT_WPIARCHIVE);

        // write STRINGTABLE

        hprintf(hRCfile, "; String table with archive file names\r\n\r\n");
        hprintf(hRCfile, "STRINGTABLE\n{\r\n");
        int i;
        for (i = 0; i < iFiles; i++)
        {
            CHAR szTemp[CCHMAXPATH];
            getBasename(apszFilenames[i], szTemp);
            hprintf(hRCfile, "  %i, \"%s\"\r\n", i+1, szTemp);
        }
        hprintf(hRCfile, "}\r\n\r\n");

        // write RESOURCE entries

        hprintf(hRCfile, "; Archives\r\n\r\n");
        for (i = 0; i < iFiles; i++)
        {
            CHAR szTemp[2*CCHMAXPATH];
            fixFilename(apszFilenames[i], szTemp);
            hprintf(hRCfile, "RESOURCE RT_WPIARCHIVE %i \"%s\"\r\n",
                    i+1, szTemp);
        }
        close(hRCfile);

        printf("done.\n");

        fSuccess = TRUE;
    }

    return fSuccess;
}

/*
 *@@ createExecutable:
 *      Creates the executable by writing the launcher startup block
 *      from the program's resources to file and appends the WarpIN archives
 *      (that are compiled into a .RES file) to it.
 *
 *      Returns TRUE if the executable has been successfully created,
 *      FALSE if there have been errors.
 *
 *@@added V0.9.4 [csm]
 */

BOOL createExecutable(PCSZ pcszPrefix)       // in: prefix of the Exe to be created
{
    BOOL fResult = FALSE;

    // Write launcher executable and append resources

    // create XXX.exe in current directory
    CHAR szBuffer[2*CCHMAXPATH];
    sprintf(szBuffer, "%s.exe", pcszPrefix);
    printf("Writing stub to '%s'... ", szBuffer);
    APIRET arc = reshWriteResourceToFile(NULLHANDLE, // module
                                         RT_WPILAUNCHER,
                                         1,
                                         szBuffer,
                                         NULL);
    if (arc != NO_ERROR)
        printf("error %d occured loading/writing resource.\n", arc);
    else
    {
        printf("done.\n");

        // create XXX.res in current directory
        sprintf(szBuffer, "rc -n %s", pcszPrefix);
        printf("Executing '%s' ...\n", szBuffer);

        ANSI_attrib(BOLD);
        ANSI_bg_color(BLUE);

        BOOL fError = FALSE;
        LONG lExitCode;
        APIRET arc = doshExecVIO(szBuffer,
                                 &lExitCode);

        ANSI_bg_color(BLACK);
        ANSI_attrib(NORMAL);
        ANSI_cleol();

        if (arc != 0)
        {
            printf("\nError %d occured executing the program.", arc);
            fError = TRUE;
        }
        else if (lExitCode)
        {
            printf("\ncmd.exe reported error %d.", lExitCode);
            fError = TRUE;
        }
        else
        {
            // OK:

            // delete RES file
            sprintf(szBuffer, "%s.res", pcszPrefix);
            printf("Deleting temporary file %s\n", szBuffer);
            DosDelete(szBuffer);

            fResult = TRUE;
        }

        if (fError)
        {
            // error:
            sprintf(szBuffer, "%s.exe", pcszPrefix);
            printf("\nDeleting stub %s\n", szBuffer);

            DosDelete(szBuffer);
        }
    }

    // clean up

    // delete RC file
    sprintf(szBuffer, "%s.rc", pcszPrefix);
    printf("Deleting temporary file %s\n", szBuffer);
    DosDelete(szBuffer);

    return fResult;
}

/*
 *@@ checkForFile:
 *      Returns TRUE if the file with the given name exists, FALSE else.
 *
 *@@added V0.9.4 [csm]
 */

BOOL checkForFile(PCSZ pcszFilename)         // in: name of file to be searched
{
    BOOL fResult = FALSE;

    printf("Checking for %s ... ", pcszFilename);

    FILESTATUS3 fs3;
    if (DosQueryPathInfo(pcszFilename,
                         FIL_STANDARD,
                         &fs3,
                         sizeof(fs3))
         == NO_ERROR)
    {
        printf("OK\n");
        fResult = TRUE;
    }
    else
    {
        printf("ERROR: not found!\n");
    }

    return fResult;
}

/*
 *@@ createInstaller:
 *      Asks for confirmation before there can be any files overwritten.
 *      Then calls the functions necessary to create the executable.
 *
 *@@added V0.9.4 [csm]
 */

VOID createInstaller(int iFiles,             // in: number of files to include
                     PSZ apszFilenames[])    // in: names of files to include
{
    CHAR szPrefix[CCHMAXPATH] = {0};
    getPrefix(apszFilenames[0], szPrefix);

    // Check if files exist

    BOOL fSuccess = TRUE;
    INT  i        = 0;

    while ((i < iFiles) && fSuccess)
    {
        fSuccess = fSuccess && checkForFile(apszFilenames[i]);
        i++;
    }

    if (!fSuccess)
        printf("\nFiles are missing, terminating.\n");
    else
    {
        // warn user

        ANSI_fg_color(B_YELLOW);
        printf("\nThe following files will be created "
             "(and overwritten if they exist):\n");
        printf("%s.rc, %s.res, %s.exe\n",
               szPrefix, szPrefix, szPrefix);
        printf("Do you want to continue (y/n)? ");
        // fflush(stdout);
        char ch = doshGetChar();
        ANSI_fg_color(WHITE);

        // Go ahead if confirmed

        if ((ch == 'y') || (ch == 'Y'))
        {
            if (writeRCfile(szPrefix, iFiles, apszFilenames))
            {
                if (createExecutable(szPrefix))
                    printf("\n%s.exe has successfully been created\n",
                           szPrefix);
            }
        }
    }
}

/*
 *@@ main:
 *      Program entry point.
 *      Calls printHelp() if there are no command line parameters,
 *      else it calls createInstaller() to create the executable.
 *
 *@@added V0.9.4 [csm]
 */

int main(int   argc,                        // in: number of arguments
         char* argv[])                      // in: arguments
{
    if (argc == 1)
        printHelp();
    else
        createInstaller(argc-1, argv+1);

    return 0;
}
