/*
 *@@sourcefile arcsubsys.cpp:
 *  archive subsystem
 *
 *
 */

/*      Copyright (C) 2000-2001 Teemu Ahola.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOSPROCESS
#define INCL_DOSERRORS
#define INCL_DOSQUEUES
#define INCL_DOSSEMAPHORES
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "setup.h"
#include <deque>
#include <vector>
#include <map>

#include <io.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "libbz2\bzlib.h"

#include "base\bs_base.h"
#include "base\bs_list.h"

#include "helpers\stringh.h"
#include "helpers\xstring.h"
#include "base\bs_base.h"
#include "base\bs_string.h"

#include "helpers\datetime.h"
#include "helpers\eah.h"

// Oh yes! The new WIArchive2 which I have wait for.
#include "wiarchive\wiarchive.h"

// Base include
#include "wipengine\wipebase.h"
// File database unit include
#include "wipengine\fdu.h"
// Thread base include
#include "wipengine\threadbase.h"
// Archive subsystem include
#include "wipengine\arcsubsys.h"
// Message logging unit include.
#include "wipengine\mlu.h"

// XML related includes. Refer to helpers\xml.c.
#include "expat\expat.h" // must come before xml.h
#include "helpers\linklist.h"
#include "helpers\tree.h"
#include "helpers\xml.h"

//#define PRINT_DEBUG_INFO
int WiCallBack(short p_mode, short p_param, WIFileHeader* p_pwifh, void* p_user) {
    printf("arcsubsys - WiCallback(short p_mode = %i, short p_param = %i "
           "WIFileHeader *p_pwifh = 0x%X, void *p_user = 0x%X\n\n\r",
           p_mode, p_param, p_pwifh, p_user);
    return 0;
}


/*
 *@@ PFNFORALLFILESCALLBACK:
 *      callback prototype for WIArchive::forAllFiles
 */
/*typedef int (FNFORALLFILESCALLBACK)(WIFileHeader*, unsigned long);
typedef FNFORALLFILESCALLBACK *PFNFORALLFILESCALLBACK;*/


/*
 *@@assDefaultCallBack:
 *  Default callback function is used when the user has not provided
 *  an callback function. This function simply frees the data passed
 *  with the message to the logger thread. Note: the memory is not freed in
 *  every call-back situations.
 */
LONG assDefaultCallBack(SHORT p_sCBType,
                        LONG p_lDetail,
                        PVOID p_pData,
                        PVOID p_pUser) {
    int i = 0;
#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::assDefaultCallBack(SHORT p_sCBType = %i, "
           "LONG p_lDetail = %i, PVOID p_pData = 0x%X, "
           "PVOID p_pUser = 0x%X)\n",
           p_sCBType, p_lDetail, p_pData, p_pUser);
    fflush(stdout);
#endif

    switch (p_sCBType) {

        case CB_TYPE_FILE_ADDING_COMPLETED:
            if (p_pData != NULL)
                delete (FileVector *)p_pData;
            if (p_pUser != NULL)
                delete p_pUser;
            break;

        case CB_TYPE_FILE_QUERYING_COMPLETED:

            break;

        case CB_TYPE_FILE_DELETING_COMPLETED:
#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::assDefaultCallBack(SHORT, "
                   "LONG, PVOID, PVOID)\n"
                   "  - deletion completed\n\n");
#endif
            if (p_pData != NULL)
                delete (FileVector *)p_pData;
            if (p_pUser != NULL)
                delete p_pUser;
            break;

        case CB_TYPE_QUIT_COMPLETED:
            if (p_pData != NULL)
                delete p_pData;
            if (p_pUser != NULL)
                delete p_pUser;
            break;

        case CB_TYPE_ERR_INITIALIZATION:

            break;

        case CB_TYPE_ARCHIVE_FILE_OPENED:
            if (p_pData != NULL)
                delete (BSString *)p_pData;
            if (p_pUser != NULL)
                delete p_pUser;
            break;

        case CB_TYPE_ARCHIVE_FILE_QUERIED:
#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::assDefaultCallBack(SHORT, "
                   "LONG, PVOID, PVOID)\n"
                   "  - File nbr. %i queried form archive %i:\n"
                   "    file name: %s\n"
                   "    file install path: %s\n",
                   p_lDetail, ((FileClass*)p_pData)->_sPackage,
                   ((FileClass*)p_pData)->_bssFileName.c_str(),
                   ((FileClass*)p_pData)->_bssInstallPath.c_str());
            fflush(stdout);
#endif
            break;

        case CB_TYPE_PACKAGE_ADDED:
#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::assDefaultCallBack(SHORT, "
                   "LONG, PVOID, PVOID)\n"
                   "  - Package #%i added.\n\n",
                   p_lDetail);
            fflush(stdout);
#endif
            break;

        case CB_TYPE_STUB_MAY_BE_LOST:
#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::assDefaultCallBack(SHORT, "
                   "LONG, PVOID, PVOID)\n"
                   "  - File \"%s\" contains a stub.\n\n",
                   ((BSString*)p_pData)->c_str());
            printf(" Should we overwrite the stub? (Y/N): ");
            fflush(stdout);
            i = getchar();
            fflush(stdout);
            if (i == 'Y') {
                //delete (BSString *)p_pData;
                return RET_CB_OVERWRITE_STUB;
            }
#endif
            //delete (BSString *)p_pData;
            return RET_CB_CANCEL;
            break;

        case CB_TYPE_ARCHIVE_FILE_NOT_OPENED:
#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::assDefaultCallBack(SHORT, "
                   "LONG, PVOID, PVOID)\n"
                   "  - Error. Archive not open.\n"
                   "  - During operation #%i\n\n",
                   p_lDetail);
            fflush(stdout);
#endif
            if (p_pData != NULL)
                delete p_pData;
            if (p_pUser != NULL)
                delete p_pUser;
            break;

        case CB_TYPE_CLOSING_ARCHIVE_COMPLETED:
#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::assDefaultCallBack(SHORT, "
                   "LONG, PVOID, PVOID)\n"
                   "  - Archive closed.\n\n");
            fflush(stdout);
#endif
            if (p_pData != NULL)
                delete p_pData;
            if (p_pUser != NULL)
                delete p_pUser;
            break;
    }

    return RET_OK;
}


/*
 *@@InitSemaphores:
 *  initializes map mutex semaphore and init event semaphore by
 *  using DosCreateMutexSem and DosCreateEventSem. <I>Map mutex semaphore</I>
 *  is used to coordinate _Messages multimap usage. <I>Init event semaphore</I>
 *  is used during initialization phase and it is closed soon after
 *  the initialization is completed.
 */
VOID ArcSubSystem::InitSemaphores(VOID) {
    APIRET ulrc = 0;
    // Create mutex semaphore for coordination of using of _Files.
    ulrc = DosCreateMutexSem(NULL, &_mtxSemFiles, 0L, FALSE);
#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::InitSemaphores(VOID):\n"
           "  - DosCreateMutexSem = %i\n", ulrc);
    fflush(stdout);
#endif
    // Create mutex semaphore for coordination of using of _bssQueueName.
    ulrc = DosCreateMutexSem(NULL, &_mtxSemQueName, 0L, FALSE);
#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::InitSemaphores(VOID):\n"
           "  - DosCreateMutexSem = %i\n", ulrc);
    fflush(stdout);
#endif
    // Create event semaphore for initialization syncronization.
    ulrc = DosCreateEventSem(NULL, &_hevSemInit, 0L, FALSE);
#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::InitSemaphores(VOID):\n"
           "  - DosCreateEventSem = %i\n", ulrc);
    fflush(stdout);
#endif
    // Create event semaphore for initialization error indication.
    ulrc = DosCreateEventSem(NULL, &_hevSemError, 0L, FALSE);
#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::InitSemaphores(VOID):\n"
          "  - DosCreateEventSem = %i\n", ulrc);
    fflush(stdout);
#endif
}


/*
 *@@Start:
 *  this function calls 'execute' function which was inherit from
 *  threadbase class. After that this function waits until an event
 *  semaphore is posted from '_ThdFunction' in a newly created thread.
 *  Next the event semaphore is closed since there is no need for it
 *  anymore. If an error has been occurred during the thread initialization
 *  phase, error count semaphore is increased.
 */
USHORT ArcSubSystem::Start(VOID) {
    APIRET ulrc = 0;
    ULONG ulOwned, ulError = 0;
    PID pid;
    TID tid;
    if (_hQMessageLogger == NULLHANDLE)
        return RET_ERR_PARAMETER_ERROR;

    if (execute() == 0)
        return RET_ERR_THREAD_CREATION_FAILED;

    // Wait until the initialization is completed.
    ulrc = DosWaitEventSem(_hevSemInit, SEM_INDEFINITE_WAIT);
#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::Start(VOID):\n"
            "  - DosWaitEventSem = %i\n", ulrc);
     fflush(stdout);
#endif

    // Query post counts of error semaphore which indicates
    // if an error has been occurred during the initialization.
    ulrc = DosQueryEventSem(_hevSemError, &ulError);
#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::Start(VOID):\n"
            "  - Error semaphore post count: %i\n", ulError);
     printf("  - DosQueryEventSem = %i\n", ulrc);
     fflush(stdout);
#endif

    // If error occurred, close semaphores and return an error code.
    // Detailed description about the error can be found from the MLU.
    if (ulError > 0) {
        DosCloseEventSem(_hevSemError);
        DosCloseEventSem(_hevSemInit);
        return RET_ERR_INITIALIZATION_FAILED;
    }

    // Close the error semaphore since there is no need for it anymore.
    ulrc = DosCloseEventSem(_hevSemError);
#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::Start(VOID):\n"
            "  - DosCloseEventSem in Start of returned: %i\n", ulrc);
     fflush(stdout);
#endif

    // Close the initialization event semaphore since there is
    // no need for it anymore.
    ulrc = DosCloseEventSem(_hevSemInit);
#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::Start(VOID):\n"
            "  - DosCloseEventSem in Start of returned: %i\n", ulrc);
     fflush(stdout);
#endif

    _hevSemInit = NULLHANDLE;
    _hevSemError = NULLHANDLE;

    return RET_OK;
}


/*
 *@@ InsertFiles:
 *  This method inserts files given in p_files vector to
 *  the file database unit. Multi-user protection has been
 *  achieved by using mutex semaphores.
 */
USHORT ArcSubSystem::InsertFiles(FileVector &p_files) {
    FileVector::iterator itrBegin;

    DosRequestMutexSem(_mtxSemFiles, SEM_INDEFINITE_WAIT);

    itrBegin = p_files.begin();
    for (;itrBegin < p_files.end(); itrBegin++)
        _Files.InsertFile((FileClass)*itrBegin);

    DosReleaseMutexSem(_mtxSemFiles);
    return RET_OK;
}


/*
 *@@ InsertFiles:
 *  This method queries all files in the file database unit.
 *  Multi-user protection has been achieved by using mutex
 *  semaphores.
 */
USHORT ArcSubSystem::QueryFiles(FileVector &p_pFileVector) {

    DosRequestMutexSem(_mtxSemFiles, SEM_INDEFINITE_WAIT);

    _Files.QueryFiles(p_pFileVector);

    DosReleaseMutexSem(_mtxSemFiles);
    return RET_OK;
}


/*
 *@@ DeleteFiles:
 *  This method deletes all files in the file database unit
 *  which are listed in the give FileVector parameter. Also if
 *  a file is in the archive or atleast marked to be archived,
 *  the file will be removed also from WIArchives internal data
 *  structures and if neccessary form the archive too.
 */
USHORT ArcSubSystem::DeleteFiles(FileVector &p_files) {
    FileVector::iterator itrBegin;
    BSString bssArchivedFileName;

#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::DeleteFiles(FileVector):\n\n");
     fflush(stdout);
#endif

    DosRequestMutexSem(_mtxSemFiles, SEM_INDEFINITE_WAIT);
    itrBegin = p_files.begin();
    for (; itrBegin < p_files.end(); itrBegin++) {
        // If _bArchive flag is set, the file is also in
        // WIArchives internal lists and maybe in the archive
        // also so we must delete it too.
        if (((FileClass)*itrBegin)._bArchived == TRUE) {
            // Compose full archived file name.
            bssArchivedFileName = ((FileClass)*itrBegin)._bssInstallPath +
                ((FileClass)*itrBegin)._bssFileName;
#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::DeleteFiles(FileVector):\n"
            "  - delete file \"%s\" form package #%i\n\n",
            bssArchivedFileName.c_str(),
            ((FileClass)*itrBegin)._sPackage);
     fflush(stdout);
#endif
            // Remove the file.
            _Archive.remove(bssArchivedFileName.c_str(),
                ((FileClass)*itrBegin)._sPackage);
        }
        // Delete the file from internal filelist also.
        _Files.DeleteFile((FileClass)*itrBegin);
    }

    DosReleaseMutexSem(_mtxSemFiles);
    return RET_OK;
}


/*
 *@@ QueryArchivedFiles
 *  This method goes through all the file in an archive opened
 *  earlier and stores some information about them to the _Files
 *  attribute of this class. Every time a file is found which match
 *  to the specified package, a call-back function is called if the
 *  function was provided. If ALL_PACKAGES is specified to be
 *  the package to be selected, all files are selected to be queried.
 */
USHORT ArcSubSystem::QueryArchivedFiles(SHORT p_sPackage) {
    FileClass file;
    BSString bssFilePath;
    BSString bssInstallPath;
    list<WIFileLI *>::iterator current, end;
    list<WIFileLI *> *lstArchivedFiles = NULL;
    USHORT usFileCount = 0;

#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::QueryArchivedFiles("
            "SHORT p_sPackage = %i)\n\n",
            p_sPackage);
     fflush(stdout);
#endif

    // Get file list from an archive.
    lstArchivedFiles = _Archive.getFileList();

#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::QueryArchivedFiles(SHORT)\n"
           "  - lstArchivedFiles = 0x%X\n\n",
           lstArchivedFiles);
    fflush(stdout);
#endif

    if (lstArchivedFiles == NULL)
        return RET_ERR_PARAMETER_ERROR;

    current = (*lstArchivedFiles).begin();
    end = (*lstArchivedFiles).end();

    // Go trought the file list.
    for (; current != end; current++) {
        // If a file which package id is the same we are
        // looking for, store it's information to the
        // specified file vector.
        if (((**current)._p->package == p_sPackage) ||
            (p_sPackage == (SHORT)ALL_PACKAGES)) {
            file._bssFileName = (**current)._p->name;
            file._bssFilePath = bssFilePath;
            file._bssInstallPath = bssInstallPath;
            file._sPackage = (**current)._p->package;
            file._bArchived = TRUE;// Since these files are in
                                   // the archice, _bArchived flag
                                   // is set.
            _Files.InsertFile(file);
            usFileCount++;
            // If the call-back is definited, call it.
            if (_pfnWipeCallBack)
                (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_QUERIED,
                                    usFileCount,
                                    &file, _pUserData);
        }


    }
    return RET_OK;

}


/*
 *@@ AddNonArchivedFiles
 *  This method adds files to the archive opened earlier
 *  which are not yet in the archive. This can be checked
 *  from the _bArchived flag of the FileClass class and if
 *  it is FALSE, the file is not yet in the archive. If
 *  p_sPackage parameter is set to ALL_PACKAGES, every file,
 *  not in the archive, will be added to the archive despite
 *  of their package number.
 */
USHORT ArcSubSystem::AddNonArchivedFiles(SHORT p_sPackage) {

    BSString bssArchivedName, bssFullName;
    USHORT usFileCount = 0, i = 0;
    FileClass fc;

#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::AddArchivedFiles("
           "SHORT p_sPackage = %i)\n\n",
           p_sPackage);
    fflush(stdout);
#endif

    usFileCount = _Files.ReturnFileCount();

    // Go trought the file list.
    for (i; i < usFileCount; i++) {
        // Look for files which are in the specified package and
        // have _bArchived flag set to False.
        _Files.QueryFile(i, fc);
        if (((fc._sPackage == p_sPackage) ||
            (p_sPackage == (SHORT)ALL_PACKAGES)) &&
            (fc._bArchived == FALSE)) {
            // Clear temporary strings.
            bssArchivedName.erase();
            bssFullName.erase();

            // If install path is not empty, add it to
            // the name of the file to be used as installation
            // name.
            if (fc._bssInstallPath.empty() == FALSE) {
                bssArchivedName = fc._bssInstallPath;
                bssArchivedName += "\\";
            }
            bssArchivedName += fc._bssFileName;

            // If the path where the file can be found is not
            // not empty, the path will be concatenated with
            // the filename.
            if (fc._bssFilePath.empty() == FALSE) {
                bssFullName = fc._bssFilePath;
                bssFullName += "\\";
            }
            bssFullName += fc._bssFileName;

#ifdef PRINT_DEBUG_INFO
            printf("ArcSubSystem::AddArchivedFiles("
                   "SHORT)\n"
                   "  - original file name and path: "
                   "\"%s\"\n"
                   "  - install name of the file to be archived: "
                   "\"%s\"\n\n",
                   bssFullName.c_str(),
                   bssArchivedName.c_str());
            fflush(stdout);
#endif
            // Change archived flat to TRUE.
            fc._bArchived = TRUE;
            // And add it to the archive. Actually it goes to
            // a todo list and will be processed later.
            _Archive.add(bssFullName.c_str(),
                         bssArchivedName.c_str(),
                         fc._sPackage);
        }


    }
    return RET_OK;
}

/*
 *@@AddPackage
 *  This method add a package with a defined number to an
 *  archive. In order to add files with specified package
 *  id to the archive, a package with same id must be created
 *  first. It is not necessary to add files to the package created
 *  earlier but it must be created before adding of files to it.
 *  If package already exists, nothing is returned by this method.
 *  Package is simply not added.
 */
USHORT ArcSubSystem::AddPackage(SHORT p_sPackage) {

    char szPackageName[20];

    // Compose package name. This is mainly for internal use and
    // the user has no clue about it.
    sprintf(szPackageName, "Pck%03d", p_sPackage);

    _Archive.setPackage(p_sPackage, szPackageName);

    return RET_OK;
}


/*
 *@@_ThdFunction:
 *  this function is an implementation for a virtual function in
 *  ThreadBase class and therefore required. This function is called
 *  from inside ThreadStarted function which is called from inside
 *  _begingthread function which actually creates a new thread. In
 *  this function an unique message queue is created and after that
 *  Start function is notified by posting event semaphore. Next
 *  the function starts to wait messages in an indefinite loop.
 *  When a message arrives to the message queue needed operations
 *  are accomplishend depending the message type. After that the function
 *  calls a call-back function if provided and posts event semaphore.
 *  Also the memory allocated for the receivced data is deleted here so
 *  it is important not to send pointers to automatic variables or free
 *  allocated memory in the sender thread.
 *
 *  <B>Operations:</B>
 *      <I>Add files</I>: Adds files in a file vector to the file database
 *                        unit's database.
 *
 */
VOID ArcSubSystem::_ThdFunction() {
    BSString bssErrorMessage;
    TID tidSenderThread = 0;
    APIRET ulrc = ERROR_QUE_DUPLICATE;
    USHORT usIndexNbr = 0;
    CHAR szQueueName[256];
    REQUESTDATA request = {0};
    ULONG ulDataLength = 0;
    BYTE bPriority = 0;
    ULONG ulCommand;
    PVOID pReceivedData = NULL;
    MsgInfo *pMsgInfo = NULL;
    FileVector *pFileVector = NULL;


    // Create unique queue. DosCreateQueue returns ERROR_QUE_DUPLICATED, if there is
    // already a queue with same name.
    while (ulrc == ERROR_QUE_DUPLICATE) {
        // Format an unique queue name.
        sprintf(szQueueName, "\\QUEUES\\WIPENGINE\\ASUBSYS%i.QUE", usIndexNbr);
        // Create a new queue with FIFO priority-ordering and address-converting
        // flags.
        ulrc = DosCreateQueue(&_hQReceiver, QUE_FIFO | QUE_CONVERT_ADDRESS,
                              szQueueName);
        usIndexNbr++;
    }

    // If error occurred, return immediately. This also completes the thread.
    // Before returning, post error semaphore so starter function knows
    // that something went wrong. Sends error message to the MLU.
    // _aidSender will be NULLHANDLE.
    if (ulrc != NO_ERROR) {
#ifdef PRINT_DEBUG_INFO
        printf("Initialization error occurred in arcsubsys.cpp (_ThdFunction).\n");
        fflush(stdout);
#endif
        DosPostEventSem(_hevSemError);

        // Compose error message.
        bssErrorMessage = STATUS_MSG_FAIL;
        bssErrorMessage += " in ";
        bssErrorMessage += __FILE__;
        // Log the error to MLU. Sender in NULLHANDLE and type is critical.
        MsgLoggingUnit::LogMessage(NULLHANDLE, bssErrorMessage, TYPE_MSG_CRITICAL,
                                   RET_ERR_INITIALIZATION_FAILED, _hQMessageLogger);

        // Call the call-back function.
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ERR_INITIALIZATION, ulrc,
                                NULL, _pUserData);

        // Release initialization semaphore.
        DosPostEventSem(_hevSemInit);
        return;
    }

#ifdef PRINT_DEBUG_INFO
    printf("Message queue created with name: %s\n", szQueueName);
    fflush(stdout);
#endif

    // Store the queue name.
    DosRequestMutexSem(_mtxSemQueName, SEM_INDEFINITE_WAIT);
    _bssQueueName = szQueueName;
    DosReleaseMutexSem(_mtxSemQueName);

    // Initialize WIArchive.
    _Archive.setCallbackFunc(WiCallBack, NULL);

    // Release initialization semaphore.
    DosPostEventSem(_hevSemInit);

    while (TRUE) {
        // Wait until there is a message in the queue created earlier.
        ulrc = DosReadQueue(_hQReceiver, &request, &ulDataLength, (PPVOID)&pReceivedData,
                            0L, DCWW_WAIT, &bPriority, 0L);
        // Get the requested command.
        ulCommand = request.ulData;

#ifdef PRINT_DEBUG_INFO
         printf("Message received.\n");
         fflush(stdout);
#endif

        if (pReceivedData) {
            switch (ulCommand) {
                // QUE_MSG_ADD_FILES:
                //  Handles file adding requests.
                case QUE_MSG_ADD_FILES: {
                    // Get MsgInfo.
                    pMsgInfo = (MsgInfo *)pReceivedData;
                    // Add files to the archive.
                    HandleAddFilesMsg(*pMsgInfo);
                    // Free the message but the user data, pointed by the message's
                    // _pData pointer, is not freed here. The default call-back
                    // function frees the user data.
                    delete pMsgInfo;
                    break;
                }
                case QUE_MSG_QUERY_FILES: {
                    pMsgInfo = (MsgInfo *)pReceivedData;
                    HandleQueryFilesMsg(*pMsgInfo);
                    delete pMsgInfo;
                    break;
                }
                case QUE_MSG_DELETE_FILES: {
                    pMsgInfo = (MsgInfo *)pReceivedData;
                    HandleDeleteFilesMsg(*pMsgInfo);
                    delete pMsgInfo;
                    break;
                }
                case QUE_MSG_QUIT: {
                    pMsgInfo = (MsgInfo *)pReceivedData;
#ifdef PRINT_DEBUG_INFO
                    printf("Quit\n\n");
                    fflush(stdout);
#endif
                    // Call open message handling method.
                    HandleQuitMsg(*pMsgInfo);
                    // Free the memory received for the message.
                    delete pMsgInfo;
                    // Leave the function.
#ifdef PRINT_DEBUG_INFO
                    printf("Leaving _ThdFunction in arcsubssys.cpp.\n");
                    fflush(stdout);
#endif
                    return;
                }
                case QUE_MSG_OPEN_ARCHIVE_FILE: {
                    pMsgInfo = (MsgInfo *)pReceivedData;
                    HandleOpenArchiveFileMsg(*pMsgInfo);
                    delete pMsgInfo;
                    break;
                }
                case QUE_MSG_ADD_PACKAGE: {
                    pMsgInfo = (MsgInfo *)pReceivedData;
                    HandleAddPackageMsg(*pMsgInfo);
                    delete pMsgInfo;
                    break;
                }
                case QUE_MSG_CLOSE_ARC: {
                    pMsgInfo = (MsgInfo *)pReceivedData;
                    HandleCloseArchiveMsg(*pMsgInfo);
                    delete pMsgInfo;
                    break;
                }
            } /* switch (ulCommand) { */
        } /* switch (ulCommand) { */
    } /* while (TRUE) { */

}

/*
 *@@~ArcSubSystem:
 *  Destructor for this class. Sends QUE_MSG_QUIT message to the thread
 *  and waits until the thread is dead.
 */
ArcSubSystem::~ArcSubSystem() {
    MsgInfo *pMsgInfo = new MsgInfo;

#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::~ArcSubSystem():\n"
            "   - _hQReceiver              = %i\n"
            "   - _Files.ReturnFileCount() = %i\n"
            "   - _hQMessageLogger         = %i\n"
            "   - _mtxSemFiles             = %i\n"
            "   - _mtxSemQueName           = %i\n"
            "   - _pfnWipeCallBack         = 0x%X\n"
            "   - _pUserData               = 0x%X\n"
            "   - _bssQueueName            = %s\n",
            _hQReceiver, _Files.ReturnFileCount(), _hQMessageLogger,
            _mtxSemFiles, _mtxSemQueName, _pfnWipeCallBack,
            _pUserData, _bssQueueName.c_str(), _bssQueueName.c_str());
     fflush(stdout);
#endif

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = NULL;
    pMsgInfo->_hevSem = NULLHANDLE;
    pMsgInfo->_pUserData = NULL;
    // Send killer message to the thread.
    DosWriteQueue(_hQReceiver, QUE_MSG_QUIT, sizeof(MsgInfo), (PVOID)pMsgInfo,
                  QUE_MSG_PRIORITY_LOWEST);
    // Wait until the thread has died.
    waitThread();
}


/*
 * Message handler methods.
 */

USHORT ArcSubSystem::HandleAddFilesMsg(MsgInfo &p_msgInfo) {

    // Get a pointer to a FileVector from MsgInfo.
    FileVector *pFileVector = (FileVector *)p_msgInfo._pData;

#ifdef PRINT_DEBUG_INFO
    printf("Add files.\n");
    fflush(stdout);
#endif

    // If no archives have been opened, we cannot proceed.
    if (_bArchiveOpened != TRUE) {
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_NOT_OPENED,
                                CB_TYPE_FILE_ADDING_COMPLETED,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);

        return RET_ERR_ARCHIVE_NOT_OPENED;
    }

    // If there is (hopefully) a FileVector, proceed.
    if (pFileVector) {
        // Insert all files in the vector to the database.
        InsertFiles(*pFileVector);
        // If user has defined a call-back function, call it.
        // If user has not provided a call-back function, we will
        // use our own unless it is NULL.
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_FILE_ADDING_COMPLETED, 0,
                               p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);
    }
    return RET_OK;
}


USHORT ArcSubSystem::HandleQueryFilesMsg(MsgInfo &p_msgInfo) {

    FileVector *pFileVector = (FileVector *)p_msgInfo._pData;

    // If no archives have been opened, we cannot proceed.
    if (_bArchiveOpened != TRUE) {
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_NOT_OPENED,
                                CB_TYPE_FILE_QUERYING_COMPLETED,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);

        return RET_ERR_ARCHIVE_NOT_OPENED;
    }

    if (pFileVector) {
        QueryFiles(*pFileVector);
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_FILE_QUERYING_COMPLETED, 0,
                               p_msgInfo._pData, p_msgInfo._pUserData);
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);
#ifdef PRINT_DEBUG_INFO
        printf("Query files.\n");
        fflush(stdout);
#endif
    }
    return RET_OK;
}

USHORT ArcSubSystem::HandleDeleteFilesMsg(MsgInfo &p_msgInfo) {

    FileVector *pFileVector = (FileVector *)p_msgInfo._pData;

#ifdef PRINT_DEBUG_INFO
        printf("Delete files.\n");
        fflush(stdout);
#endif

    // If no archives have been opened, we cannot proceed.
    if (_bArchiveOpened != TRUE) {
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_NOT_OPENED,
                                CB_TYPE_FILE_DELETING_COMPLETED,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);

        return RET_ERR_ARCHIVE_NOT_OPENED;
    }

    if (pFileVector) {
        DeleteFiles(*pFileVector);
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_FILE_DELETING_COMPLETED, 0,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);
    }
    return RET_OK;
}

USHORT ArcSubSystem::HandleAddPackageMsg(MsgInfo &p_msgInfo) {

#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::HandleAddPackageMsg(MsgInfo &p_msgInfo)\n\n");
    fflush(stdout);
#endif

    // If no archives have been opened, we cannot proceed.
    if (_bArchiveOpened != TRUE) {
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_NOT_OPENED,
                                CB_TYPE_PACKAGE_ADDED,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);

        return RET_ERR_ARCHIVE_NOT_OPENED;
    }

    // Add package to the archive. Note: this method always returns
    // RET_OK, since if the package already exists, no new package
    // is added.
    AddPackage((SHORT)p_msgInfo._lData);

    // Make a call-back if a call-back function has been provided.
    if (_pfnWipeCallBack)
        (*_pfnWipeCallBack)(CB_TYPE_PACKAGE_ADDED, p_msgInfo._lData,
                           NULL, p_msgInfo._pUserData);
    // If there is an event semaphore, post it since a thread which
    // wrote this message may want us to notify that quitting was
    // completed.
    if (p_msgInfo._hevSem != NULLHANDLE)
        DosPostEventSem(p_msgInfo._hevSem);

    return RET_OK;
}


USHORT ArcSubSystem::HandleQuitMsg(MsgInfo &p_msgInfo) {

    // Close the queue.
    DosCloseQueue(_hQReceiver);
    _hQReceiver = NULLHANDLE;
#ifdef PRINT_DEBUG_INFO
    printf("Message queue closed.\n");
    fflush(stdout);
#endif
    // Make a call-back if a call-back function has been provided.
    if (_pfnWipeCallBack)
        (*_pfnWipeCallBack)(CB_TYPE_QUIT_COMPLETED, 0,
                           p_msgInfo._pData, p_msgInfo._pUserData);
    // If there is an event semaphore, post it since a thread which
    // wrote this message may want us to notify that quitting was
    // completed.
    if (p_msgInfo._hevSem != NULLHANDLE)
        DosPostEventSem(p_msgInfo._hevSem);

    return RET_OK;
}


/*
 *@@ HandleOpenArchiveFileMsg:
 *  This method is a message handling method for
 *  QUE_MSG_OPEN_ARCHIVE_FILE messages. This method is called
 *  only from _ThdFunction method.
 */
USHORT ArcSubSystem::HandleOpenArchiveFileMsg(MsgInfo &p_msgInfo) {
    int openRc = 0;
    WIArcHeader ArcHeader;
    BSString bssErrorMessage;
    FILE *pFile = NULL;
    LONG lRc = 0;

    // Get archive filename.
    BSString *pbssFileName = (BSString *)p_msgInfo._pData;
    if (pbssFileName) {
        // Check if no filename has been defined. If true, use default one.
        if (pbssFileName->empty())
            *pbssFileName = STR_DEFAULT_FILE_NAME;

        // Because this engine does not yet support archive files with
        // stub code, we will notify the user that he/she will loose
        // the stub if we modify it. If the file does not exist, then
        // we don't have to care since this engine can add additional
        // stub to the archive.
        // So first check if the file ends to ".exe".
        if (pbssFileName->find(".exe",   // in: string to find
                               pbssFileName->size() - 4)
            != pbssFileName->npos) {
            // Ok, this is 'exe' file. So check whether this file is
            // a new one or exiting one. This is done by opening file
            // in read-only mode. If the opening fails, the file does not
            // yet exist.
            pFile = fopen(pbssFileName->c_str(), "r");
            // So it exists. The stub will be lost if we continue so ask
            // user what to do.
            if (pFile != NULL) {
                lRc = (*_pfnWipeCallBack)(CB_TYPE_STUB_MAY_BE_LOST,
                                          0, p_msgInfo._pData,
                                          p_msgInfo._pUserData);

                fclose(pFile);
                pFile = NULL;
                if (lRc != RET_CB_OVERWRITE_STUB) {
                    // Post specified semaphore.
                    if (p_msgInfo._hevSem != NULLHANDLE) // @added 2001-10-4 [tahola]
                        DosPostEventSem(p_msgInfo._hevSem);
                    // So user wants to save the stub so we will leave.
                    return RET_OK;
                }

            }

        }

        // Check if the file exists.
        pFile = fopen(pbssFileName->c_str(), "r");
        if (pFile != NULL) {
            _bNewFile = FALSE;
            fclose(pFile);
            pFile = NULL;
        }
        else
            _bNewFile = TRUE;

        // First store all EAs.
        _peaList = eaPathReadAll(pbssFileName->c_str());

        // Open the archive file in RW mode. If archive not exists
        // a new one will be created.
        openRc = _Archive.open(pbssFileName->c_str(), 1);
#ifdef PRINT_DEBUG_INFO
        printf("archive opening return value: %i\n", openRc);
        fflush(stdout);
#endif
        // If opening not succeeded, call the call-back with
        // an error code and also return it.
        if (openRc != 0) {
            // Compose error message.
            bssErrorMessage = "Opening of an archive \"";
            bssErrorMessage += *pbssFileName;
            bssErrorMessage += "\" failed.";
            // Log the error to MLU.
            MsgLoggingUnit::LogMessage(_hQReceiver, bssErrorMessage,
                                       TYPE_MSG_ERROR,
                                       DETAIL_MSG_ARCUNIT_ARC_OPENING_FAILED,
                                       _hQMessageLogger);
            if (_pfnWipeCallBack)
                (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_OPENED,
                                    RET_ERR_ARCHIVE_OPENING_FAILED,
                                    p_msgInfo._pData, p_msgInfo._pUserData);

            // Free the EA list.
            eaFreeList(_peaList);
            _peaList = NULL;

            return RET_ERR_ARCHIVE_OPENING_FAILED;
        }
        // The archive is now open so update _bArchiveOpened
        // flag.
        _bArchiveOpened = TRUE;

        // Store the archive filename.
        _bssFullArchiveName = *pbssFileName;
        // Compose header information.
        strcpy(ArcHeader.name_app, APPLICATION_NAME);
        strcpy(ArcHeader.name_dev, DEVELOPER_NAME);
        strcpy(ArcHeader.inet_address, "N/A");
        ArcHeader.rev_app = APPLICATION_REVISION;
        // Set the header information to the archive file.
        _Archive.setArcHeader(ArcHeader);

        // Query all exiting files in the archive.
        QueryArchivedFiles(ALL_PACKAGES);

        // Call callback function.
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_OPENED, 0,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // Post specified semaphore.
        if (p_msgInfo._hevSem != NULLHANDLE)
            DosPostEventSem(p_msgInfo._hevSem);
    }
    return RET_OK;
}

USHORT ArcSubSystem::HandleCloseArchiveMsg(MsgInfo &p_msgInfo) {

    PEALIST peaList = NULL;

#ifdef PRINT_DEBUG_INFO
    printf("Closing archive file \"%s\".\n",
           _bssFullArchiveName.c_str());
    fflush(stdout);
#endif

    // If no archives have been opened, we cannot proceed.
    if (_bArchiveOpened != TRUE) {
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_NOT_OPENED,
                                CB_TYPE_CLOSING_ARCHIVE_COMPLETED,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);

        return RET_ERR_ARCHIVE_NOT_OPENED;
    }

    // Add all files not in the archive to the archive.
    AddNonArchivedFiles(ALL_PACKAGES);

    // Close the archive.
    _Archive.close();

    // Make a call-back if a call-back function has been provided.
    if (_pfnWipeCallBack)
        (*_pfnWipeCallBack)(CB_TYPE_CLOSING_ARCHIVE_COMPLETED, 0,
                           p_msgInfo._pData, p_msgInfo._pUserData);
    // If there is an event semaphore, post it since a thread which
    // wrote this message may want us to notify that quitting was
    // completed.
    if (p_msgInfo._hevSem != NULLHANDLE)
        DosPostEventSem(p_msgInfo._hevSem);

    // Update the EA list.
    UpdateEas(_peaList, &peaList);

    // Write some EAs.
    eaPathWriteAll(_bssFullArchiveName.c_str(), peaList);

    // Free the EA lists.
    eaFreeList(_peaList);
    eaFreeList(peaList);
    _peaList = NULL;

    _bArchiveOpened = FALSE;

    return RET_OK;
}


USHORT ArcSubSystem::HandleExtractFilesMsg(MsgInfo &p_msgInfo) {

    int rc = 0;
    // Get the vector which defines the files to be extracted.
    FileVector *fileVector = (FileVector)p_msgInfo._pData;
    // Get the index which describes whitch package is to be
    // extracted.
    USHORT usPackage = (USHORT)p_msgInfo._lData;

#ifdef PRINT_DEBUG_INFO
    printf("ArcSubSystem::HandleExtractFiles(MsgInfo &)");
    fflush(stdout);
#endif


    // If no archives have been opened, we cannot proceed.
    if (_bArchiveOpened != TRUE) {
        if (_pfnWipeCallBack)
            (*_pfnWipeCallBack)(CB_TYPE_ARCHIVE_FILE_NOT_OPENED,
                                CB_TYPE_EXTRACTING_FILES_COMPLETED,
                                p_msgInfo._pData, p_msgInfo._pUserData);
        // If the message contains semaphore, post it.
        if (p_msgInfo._hevSem)
            DosPostEventSem(p_msgInfo._hevSem);

        return RET_ERR_ARCHIVE_NOT_OPENED;
    }

    // Initialize WIArchive call-back.
    _Archive.setCallbackFunc(WiCallBack, fileVector);

    rc = _Archive.unpack(usPackage);
    switch (rc) {

        case WIERR_INVALID_INDEX:
            // invalid index
            break;

        case 0:
            break;

        default:
            // error occurred
            break;

    }

}


/*
 * EA helpers:
 */


USHORT ArcSubSystem::UpdateEas(PEALIST p_peaOldList, PEALIST *p_ppeaNewList) {

    PEALIST peaList = NULL, peaPrev = NULL;
    PSZ pszEaValue = NULL, pszTempValue = NULL, pszTemp2Value = NULL;
    PEABINDING peaBinding = NULL;
    CHAR szTimeStamp[30];
    DATETIME dateTime;
    USHORT usCodePage = 0;

    // Create time stamp.
    DosGetDateTime(&dateTime);
    sprintf(szTimeStamp, "%4u/%02u/%02u %02u:%02u.%02u",
            dateTime.year,
            dateTime.month,
            dateTime.day,
            dateTime.hours,
            dateTime.minutes,
            dateTime.seconds);

    // Create an ea binding for subject.
    pszEaValue = ReturnEaValue(EA_NAME_SUBJECT, _peaList);
    if (pszEaValue == NULL)
        pszEaValue = strdup("WarpIN Archive");
    peaBinding = eaCreateBindingFromPSZ(EA_NAME_SUBJECT,
                                        pszEaValue);
    free(pszEaValue);
    peaList = (PEALIST)malloc(sizeof(EALIST));
    peaList->peab = peaBinding;
    peaList->next = NULL;
    peaPrev = peaList;

    // Create an ea binding for type.
    pszEaValue = ReturnEaValue(EA_NAME_TYPE, _peaList);
    if (pszEaValue == NULL)
        pszEaValue = strdup("WarpIN Archive");
    peaBinding = eaCreateBindingFromPSZ(EA_NAME_TYPE,
                                        pszEaValue);
    free(pszEaValue);
    peaList = (PEALIST)malloc(sizeof(EALIST));
    peaList->peab = peaBinding;
    peaList->next = peaPrev;
    peaPrev = peaList;

    // Create an ea binding for history.
    pszEaValue = ReturnMvEaValue(EA_NAME_HISTORY, _peaList,
                                 &usCodePage);

    // If this is a new file, use Created text.
    if (_bNewFile == TRUE) {
        pszTempValue = (PSZ)malloc(_bssArchiveAuthor.size() +
                                   1 + strlen(EA_HISTORY_ITEM_CREATED) +
                                   1 + strlen(szTimeStamp) + 2 + 1);

        sprintf(pszTempValue, "%s %s %s", _bssArchiveAuthor.c_str(),
                EA_HISTORY_ITEM_CREATED,
                szTimeStamp);
    }
    // If not, use Changed text.
    else {
        pszTempValue = (PSZ)malloc(_bssArchiveAuthor.size() +
                                   1 + strlen(EA_HISTORY_ITEM_CHANGED) +
                                   1 + strlen(szTimeStamp) + 2 + 1);

        sprintf(pszTempValue, "%s %s %s", _bssArchiveAuthor.c_str(),
                EA_HISTORY_ITEM_CHANGED,
                szTimeStamp);
    }

    pszTemp2Value = pszTempValue;
    // If there is already something in history EA, append our
    // new string to it.
    if (pszEaValue != NULL) {
        pszTemp2Value = (PSZ)malloc(strlen(pszEaValue) + strlen(pszTempValue) + 3);
        sprintf(pszTemp2Value, "%s\n\r%s", pszEaValue, pszTempValue);
        free(pszTempValue);
    }
    peaBinding = eaCreateMVBindingFromPSZ(EA_NAME_HISTORY, "\n\r",
                                          pszTemp2Value,
                                          usCodePage);
    free(pszTemp2Value);
    peaList = (PEALIST)malloc(sizeof(EALIST));
    peaList->peab = peaBinding;
    peaList->next = peaPrev;
    peaPrev = peaList;

    *p_ppeaNewList = peaList;

    return RET_OK;
}

PSZ ArcSubSystem::ReturnEaValue(PSZ p_pszEaName, PEALIST p_peaList) {

    if (p_peaList) {
        PEALIST list = p_peaList;
        while (list != 0){
            PEALIST next = list->next;
            // Check if the EA exists.
            if (strcmp(list->peab->pszName, p_pszEaName) == 0)
                return eaCreatePSZFromBinding(list->peab);
            list = next;
        }
    }
    return NULL;
}

PSZ ArcSubSystem::ReturnMvEaValue(PSZ p_pszEaName, PEALIST p_peaList,
                                  USHORT *p_pusCodePage) {

    if (p_peaList) {
        PEALIST list = p_peaList;
        while (list != 0){
            PEALIST next = list->next;
            // Check if the EA exists.
            if (strcmp(list->peab->pszName, p_pszEaName) == 0)
                return eaCreatePSZFromMVBinding(list->peab,
                                                "\n\r",
                                                p_pusCodePage);

            list = next;
        }
    }
    return NULL;
}



/*
 * Interface methods.
 */


/*
 *@@CloseArchive:
 *  This method closes earlier opened archive file which eventually
 *  causes all added files to be stored to the archive and the
 *  script being created and stored to the archive.
 */
USHORT ArcSubSystem::CloseArchive(HEV p_hevSem,
                                  PVOID p_pUserData
                                 ) {

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = NULL;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE). This message is sent with lowest priority
    // so if some other thread sends other messages they will be
    // performed before this message.
    return DosWriteQueue(_hQReceiver, QUE_MSG_CLOSE_ARC, sizeof(MsgInfo),
                         (PVOID)pMsgInfo,
                         QUE_MSG_PRIORITY_LOWEST);


}


/*
 *@@InsertFiles:
 *  Inserts files in the specified FileVector to the specified archive.
 *  The FileVector will be free so it should be dynamically allocated and
 *  should not be freed in the calling thread. Also the specified semaphore
 *  will be signalled by the archive after adding of the files has been
 *  completed. This method is actually a dummy interface method for sending
 *  QUE_MSG_ADD_FILES message to the specified archive subsystem.
 */
USHORT ArcSubSystem::InsertFiles(FileVector *p_pFileVector,
                                 HEV p_hevSem,
                                 PVOID p_pUserData
                                ) {
    if (!p_pFileVector)
        return RET_ERR_PARAMETER_ERROR;

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = p_pFileVector;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE).
    return DosWriteQueue(_hQReceiver, QUE_MSG_ADD_FILES, sizeof(MsgInfo),
                        (PVOID)pMsgInfo, QUE_MSG_PRIORITY_NORMAL);
}


/*
 *@@DeleteFiles:
 *  Deletes files, specified in the FileVector, from the specified archive.
 *  The FileVector will be free so it should be dynamically allocated and
 *  should not be freed in the calling thread. Also the specified semaphore
 *  will be signalled by the archive after removing of the files has been
 *  completed. This method is actually a dummy interface method for sending
 *  QUE_MSG_DELETE_FILES message to the specified archive subsystem.
 */
USHORT ArcSubSystem::DeleteFiles(FileVector *p_pFileVector,
                                 HEV p_hevSem,
                                 PVOID p_pUserData
                                ) {

    if (!p_pFileVector)
        return RET_ERR_PARAMETER_ERROR;

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = p_pFileVector;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE).
    return DosWriteQueue(_hQReceiver, QUE_MSG_DELETE_FILES, sizeof(MsgInfo),
                        (PVOID)pMsgInfo, QUE_MSG_PRIORITY_NORMAL);
}


/*
 *@@QueryFiles:
 *  Returns all files from the specified archive. A pointer to a FileVector
 *  must be provided. Also the specified semaphore will be signalled by
 *  the archive after removing of the files has been completed. This method
 *  is actually a dummy interface method for sending QUE_MSG_DELETE_FILES
 *  message to the specified archive subsystem.
 */
USHORT ArcSubSystem::QueryFiles(FileVector *p_pFileVector,
                                HEV p_hevSem,
                                PVOID p_pUserData
                                ) {

    if (!p_pFileVector)
        return RET_ERR_PARAMETER_ERROR;

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = p_pFileVector;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE).
    return DosWriteQueue(_hQReceiver, QUE_MSG_QUERY_FILES, sizeof(MsgInfo),
                        (PVOID)pMsgInfo, QUE_MSG_PRIORITY_LOWEST);


}


USHORT ArcSubSystem::OpenArchiveFile(BSString &p_bbsFileName,
                                     HEV p_hevSem,
                                     PVOID p_pUserData
                                     ) {

#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::OpenArchiveFile(BSString &p_fileName "
            "(.c_str()) = \"%s\", HEV p_hevSem = %i)\n",
            p_bbsFileName.c_str(), p_hevSem);
     fflush(stdout);
#endif


    BSString *fileName = new BSString;

    *fileName = p_bbsFileName;

    if (fileName->empty())
        *fileName = STR_DEFAULT_FILE_NAME;

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = fileName;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE).
    return DosWriteQueue(_hQReceiver, QUE_MSG_OPEN_ARCHIVE_FILE,
                         sizeof(MsgInfo), (PVOID)pMsgInfo,
                         QUE_MSG_PRIORITY_NORMAL);


}

USHORT ArcSubSystem::QueryArchivedFiles(FileVector *p_pFileVector,
                                        SHORT p_sPackage,
                                        HEV p_hevSem,
                                        PVOID p_pUserData
                                        ) {

#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::QueryArchivedFiles("
            "FileVector *p_pFileVector = 0x%X, "
            "SHORT p_sPackage = %i, "
            "HEV p_hevSem = %i)\n\n",
            p_pFileVector, p_sPackage, p_hevSem);
     fflush(stdout);
#endif

    if (p_pFileVector == NULL)
        return RET_ERR_PARAMETER_ERROR;

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = p_pFileVector;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_lData = p_sPackage;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE).
    return DosWriteQueue(_hQReceiver, QUE_MSG_QUERY_ARCHIVED_FILES,
                         sizeof(MsgInfo), (PVOID)pMsgInfo,
                         QUE_MSG_PRIORITY_NORMAL);


}


USHORT ArcSubSystem::AddPackage(SHORT p_sPackage,
                                HEV p_hevSem,
                                PVOID p_pUserData
                               ) {

#ifdef PRINT_DEBUG_INFO
     printf("ArcSubSystem::AddPackage("
            "SHORT p_sPackage = %i, "
            "HEV p_hevSem = %i)\n\n",
            p_sPackage, p_hevSem);
     fflush(stdout);
#endif

    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = NULL;
    pMsgInfo->_hevSem = p_hevSem;
    pMsgInfo->_lData = p_sPackage;
    pMsgInfo->_pUserData = p_pUserData;

    // Send message to the specified archive sub system which ID is
    // p_aidASS (HMQUEUE).
    return DosWriteQueue(_hQReceiver, QUE_MSG_ADD_PACKAGE,
                         sizeof(MsgInfo), (PVOID)pMsgInfo,
                         QUE_MSG_PRIORITY_NORMAL);


}



