/*
 *              WicPM - WarpIN Archive Creator for PM
 *                  (c) Copyright Teemu Ahola 2000
 *
 *              WarpIN Script Handling Class 0.9.6
 *
 *  Special thanks to Ulrich Mller about his great helper functions for
 *  string handling.
 *
 *
 *@@sourcefile: script.cpp
 *  Source code file for Script C++ class.
 *  This file contains source codes for WarpIN Script handling class.
 *  This file <U>is</U> platform independent. More detailed information about
 *  WarpIN script can be found from WarpIN programmer guide.
 *
 * This class is used for script handling in WicPM (WarpIN Archive Creator
 * for PM). This class is intended to be characteristic of real WarpIN script
 * so different classes are like WarpIN script's tags and attributes. When
 * creating new attribute specific structures can be used. With this class
 * script can be created and script text can be parsed back to objects.
 *
 *@todo:
 *  - Inlining and optimization. The classes are terribly slow so in older
 *    computers script parsing and generating takes a lot of time.
 *
 *@added v.0.9.5 [tahola]: Message logging system.
 *@changed v.0.9.6 [tahola]: New conding conversion.
 */


#include <os2.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <direct.h>

#include "setup.h"
// #include <list>
#include "helpers\stringh.h"
#include "helpers\xstring.h"

#include "base\bs_base.h"           // V0.9.14 (2001-07-07) [umoeller]
#include "base\bs_string.h"
#include "base\bs_list.h"       // V0.9.14 (2001-07-07) [umoeller]

#include "wicpm\script.hpp"



// Script tags and attributes
    // PCK tag
#define SZ_TAG_PACKAGE                  "PCK"
#define SZ_ATTRIBUTE_INDEX              "INDEX"
#define SZ_ATTRIBUTE_PACKAGEID          "PACKAGEID"
#define SZ_ATTRIBUTE_TITLE              "TITLE"
#define SZ_ATTRIBUTE_TARGET             "TARGET"
#define SZ_KEYWORD_BASE                 "BASE"
#define SZ_KEYWORD_FIXED                "FIXED"
#define SZ_KEYWORD_SELECT               "SELECT"
#define SZ_KEYWORD_NODESELECT           "NODESELECT"
#define SZ_ATTRIBUTE_REQUIRES           "REQUIRES"
#define SZ_ATTRIBUTE_CONFIGSYS          "CONFIGSYS"
#define SZ_MODIFIER_UNIQUE              "UNIQUE"
#define SZ_MODIFIER_ADDRIGHT            "ADDRIGHT"
#define SZ_MODIFIER_ADDLEFT             "ADDLEFT"
#define SZ_MODIFIER_ADDTOP              "ADDTOP"
#define SZ_MODIFIER_ADDAFTER            "ADDAFTER"
#define SZ_MODIFIER_ADDBEFORE           "ADDBEFORE"
#define SZ_MODIFIER_REMOVELINE          "REMOVELINE"
#define SZ_MODIFIER_REMOVEPART          "REMOVEPART"
#define SZ_ATTRIBUTE_REGISTERCLASS      "REGISTERCLASS"
#define SZ_ATTRIBUTE_REPLACECLASS       "REPLACECLASS"
#define SZ_ATTRIBUTE_CREATEOBJECT       "CREATEOBJECT"
#define SZ_PARAMETER_REPLACE            "REPLACE"
#define SZ_ATTRIBUTE_EXECUTE            "EXECUTE"
#define SZ_FLAG_CONFIGSYS               "CONFIGSYS"
#define SZ_FLAG_REGISTERCLASS           "REGISTERCLASS"
#define SZ_FLAG_CREATEOBJECT            "CREATEOBJECT"
#define SZ_CONTEXT_CONFIGSYS            "CONFIGSYS"
#define SZ_CONTEXT_REGISTERCLASS        "REGISTERCLASS"
#define SZ_CONTEXT_CREATEOBJECT         "CREATEOBJECT"
#define SZ_ATTRIBUTE_CLEARPROFILE       "CLEARPROFILE"
#define SZ_ATTRIBUTE_WRITEPROFILE       "WRITEPROFILE"
#define SZ_ATTRIBUTE_KILLPROCESS        "KILLPROCESS"
#define SZ_ATTRIBUTE_EXTERNAL           "EXTERNAL"
#define SZ_PARAMETER_REQUIRED           "REQUIRED"
#define SZ_TAG_ENDPACKAGE               "/PCK"
    // BODY tag
#define SZ_TAG_BODY                     "BODY"
#define SZ_TAG_ENDBODY                  "/BODY"
    // GROUP tag
#define SZ_TAG_GROUP                    "GROUP"
#define SZ_ATTRIBUTE_EXPANDED           "EXPANDED"
// TITLE attribute from PCK tag.
#define SZ_TAG_ENDGROUP                 "/GROUP"
    // HEAD tag
#define SZ_TAG_HEAD                     "HEAD"
#define SZ_TAG_ENDHEAD                  "/HEAD"
    // MSG tag
#define SZ_TAG_MSG                      "MSG"
#define SZ_TAG_ENDMSG                   "/MSG"
    // NEXTBUTTON tag
#define SZ_TAG_NEXTBUTTON               "NEXTBUTTON"
#define SZ_ATTRIBUTE_TARGET             "TARGET"
#define SZ_TAG_ENDNEXTBUTTON            "/NEXTBUTTON"
    // PAGE tag
#define SZ_TAG_PAGE                     "PAGE"
#define SZ_ATTRIBUTE_INDEX              "INDEX"
#define SZ_ATTRIBUTE_TYPE               "TYPE"
#define SZ_TYPE_TEXT                    "TEXT"
#define SZ_TYPE_README                  "README"
#define SZ_TYPE_CONTAINER               "CONTAINER"
#define SZ_TYPE_CONFIGURE               "CONFIGURE"
#define SZ_TAG_ENDPAGE                  "/PAGE"
    // README tag
#define SZ_TAG_README                   "README"
#define SZ_ATTRIBUTE_FORMAT             "FORMAT"
#define SZ_FORMAT_PLAIN                 "PLAIN"
#define SZ_FORMAT_FLOW                  "FLOW"
#define SZ_FORMAT_HTML                  "HTML"
#define SZ_TAG_ENDREADME                "/README"
    // TEXT tag
#define SZ_TAG_TEXT                     "TEXT"
#define SZ_TAG_ENDTEXT                  "/TEXT"
    // TITLE tag
#define SZ_TAG_TITLE                    "TITLE"
#define SZ_TAG_ENDTITLE                 "/TITLE"
    // VARPROMPT tag
#define SZ_TAG_VARPROMPT                "VARPROMPT"
#define SZ_ATTRIBUTE_NAME               "NAME"
#define SZ_ATTRIBUTE_TYPE               "TYPE"
#define SZ_TYPE_NUM                     "NUM"
#define SZ_TYPE_ALPHA                   "ALPHA"
#define SZ_TYPE_PATH                    "PATH"
#define SZ_TYPE_FAIL                    "FAIL"
#define SZ_TAG_ENDVARPROMPT             "/VARPROMPT"
    // WARPIN tag
#define SZ_TAG_WARPIN                   "WARPIN"
#define SZ_ATTRIBUTE_VERSION            "VERSION"
#define SZ_ATTRIBUTE_OS                 "OS"
#define SZ_OS_OS22X                     "OS2_2X"
#define SZ_OS_OS23X                     "OS2_3X"
#define SZ_OS_OS24X                     "OS2_4X"
#define SZ_ATTRIBUTE_CODEPAGE           "CODEPAGE"
#define SZ_TAG_ENDWARPIN                "/WARPIN"

// Message types.
#define SZ_MESSAGE_INFO                 "Informative"
#define SZ_MESSAGE_WARNING              "Warning"
#define SZ_MESSAGE_ERROR                "Error"
#define SZ_MESSAGE_MSG                  "Message"




ULONG LtoALength(LONG lInteger)
    {
    CHAR szTemp[32];
    _ltoa(lInteger, szTemp, 10);
    return strlen(szTemp);
    }


//
//
//  Templates:
//
//

/*
 *@@CopyTables:
 *  This is a function template for copying table of classes to another table. This
 *  way we don't have to create so much overloaded CopyTable functions.
 *
 *@@added v.0.9.5 (2000-08-22) [tahola]
 */
template <class T>
T *CopyTable(T *p_destTable,//in: Pointer to destination table.
              T *p_srcTable,//in: Pointer to source table.
              USHORT p_usSize,//in: Size of source table.
              USHORT p_usSkipIndex//in: Index to be skipped. If 0 no skipping.
              )
    {
    USHORT srcIndex, destIndex;
    if (!p_destTable)
        return NULL;
    if (!p_srcTable)
        return p_destTable;
    for (srcIndex = 0, destIndex = 0; srcIndex < p_usSize; srcIndex ++, destIndex ++)
        {
        if (p_srcTable[srcIndex].ReturnIndex() == p_usSkipIndex)
            if (srcIndex < p_usSize - 1)
                srcIndex ++;
            else
                break;
        p_destTable[destIndex] = p_srcTable[srcIndex];
        }

    return p_destTable;
    }


/*
 *@@RebuildIndexes:
 *  Rebuilds indexes after deletion. The order of the indexes does not matter.
 *  Example:
 *      Old indexing: 1, 4, 3, 5
 *      New indexing: 1, 3, 2, 4
 *
 *@@added v.0.9.5 (2000-08-22) [tahola]
 */
template <class T>
USHORT RebuildIndexes(T *p_table, // A pointer to a table of objects.
                      USHORT p_usTableSize // The size of the table.
                      )
    {
    USHORT usIndex, usIndex2, usIndex3, usMissing, usMax = 0, i;
    BOOL bFinded = FALSE, bReady = FALSE;

    // First find largest index number.
    for (usIndex = 0; usIndex < p_usTableSize; usIndex ++)
        if (p_table[usIndex].ReturnIndex() > usMax)
            usMax = p_table[usIndex].ReturnIndex();

    // Do this until it is sure that there is no skipped indexes left.
    while (!bReady)
        {
        bReady = TRUE;
        // Go through all index values from 1 to usMax.
        for (usIndex = 1; usIndex <= usMax; usIndex ++)
            {
            // Check that Index number will be found.
            for (usIndex2 = 0; usIndex2 < p_usTableSize; usIndex2 ++)
                if (usIndex == p_table[usIndex2].ReturnIndex())
                    bFinded = TRUE;
            // If it did not found sub 1 from every index number above
            // the index number which did not found.
            if (bFinded == FALSE)
                {
                for (usIndex3 = usIndex + 1; usIndex3 <= usMax; usIndex3 ++)
                    for (usIndex2 = 0; usIndex2 < p_usTableSize; usIndex2 ++)
                        if (usIndex3 == p_table[usIndex2].ReturnIndex())
                            p_table[usIndex2].SetIndex(usIndex3 - 1);
                usMax --;
                }
            if (bFinded == FALSE)
                bReady = FALSE;
            bFinded = FALSE;
            }
        }
    return NO_ERROR;
    }


/*
 *@@FindUnusedIndex:
 *  Returns first unused index number.
 *  Ex. 1, 2, 4, 5
 *      returns: 3
 *
 *@@added v.0.9.5 (2000-08-22) [tahola]
 */
template <class T>
USHORT FindUnusedIndex(T *p_table, // A pointer to a table of objects.
                       USHORT p_usTableSize // The size of the table.
                       )
    {
    USHORT usIndex, usIndex2, usMax = 0;
    BOOL bFinded = FALSE;

    // First find the largest index number.
    for (usIndex = 0; usIndex < p_usTableSize; usIndex ++)
        if (p_table[usIndex].ReturnIndex() > usMax)
            usMax = p_table[usIndex].ReturnIndex();

    for (usIndex = 1; usIndex <= usMax; usIndex ++)
        {
        // Go through all the objects on the table.
        for (usIndex2 = 0; usIndex2 < p_usTableSize; usIndex2 ++)
            if (usIndex == p_table[usIndex2].ReturnIndex())
                {
                bFinded = TRUE;
                break;
                }
        // If Index found from the table, continue.
        if (bFinded == TRUE)
            bFinded = FALSE;
        // If not break.
        else
            break;
        }

    return usIndex;
    }


//
//
//  Handy functions.
//
//


/*
 *@@AddErrorToTable:
 *  Adds error information structure to table. Returns error code if
 *  necessary.
 */
USHORT AddErrorToTable(SCRIPTERROR error, // Structure to be added.
                      SCRIPTERROR **pErrorTable, // Table of error structures.
                      USHORT usErrorTableSize) // Size of the table.
    {
    SCRIPTERROR *pErrorTemp = NULL;

    // Create new array for error structures.
    if (!(pErrorTemp = new SCRIPTERROR[usErrorTableSize + 1]))
        return 0;

    // If there is error structures in pErrorTable copy them to new
    // table.
    if (pErrorTable && (usErrorTableSize > 0))
        {
        memcpy(pErrorTemp, pErrorTable, sizeof(SCRIPTERROR) * usErrorTableSize);
        pErrorTemp[usErrorTableSize] = error; // @@Fixed Teemu Ahola 15.4.2000
        }
    else
        pErrorTemp[0] = error;

    *pErrorTable = pErrorTemp;

    return usErrorTableSize + 1;
    }


/*
 *@@AddCharacterAfterEOL:
 *  This function adds character (ex. \t, \n)  before each line in the string.
 */
void AddCharacterAfterEOL(PSZ *pszText, // Pointer to PSZ pointer.
                          CHAR szCharacter // The character to be added.
                          )
    {
    PSZ pszTemp = *pszText, pszTemp1 = NULL, pszReturn = NULL;
    ULONG ulOffset = 0, ulTempOffset = 0;

    if (pszText == NULL || (*pszText) == NULL)
        return;

    // Add the character to the beginning of the string.
    pszTemp1 = strhins(pszTemp, 0, &szCharacter);
    delete [] pszTemp;
    pszTemp = pszTemp1;
    // Repeat.
    while (TRUE)
        {
        // Find next line.
        pszReturn = strhFindNextLine(pszTemp + ulOffset, &ulOffset);
        // If no more lines, break.
        if (*pszReturn == '\0')
            break;
        // Set ulOffset to beginning of next line.
        ulOffset += ulTempOffset;
        ulTempOffset = ulOffset;
        // Add the character.
        pszTemp1 = strhins(pszTemp, ulOffset, &szCharacter);
        delete [] pszTemp;
        pszTemp = pszTemp1;
        }
    *pszText = pszTemp;
    }




INDEX INDEX::operator+(unsigned short usIndex)
    {
   unsigned short usCount = 0;
   INDEX TempIndex = *this;

   // Find the last used index number.
   while ((TempIndex.tusIndex[usCount] != 0) && (usCount < 10))
    usCount ++;

   TempIndex.tusIndex[usCount - 1] += usIndex;

   return TempIndex;
   }


INDEX &INDEX::operator=(unsigned short usIndex)
    {
   unsigned short usCount = 0;

   // Find the last used index number.
   while ((this->tusIndex[usCount] != 0) && (usCount < 10))
    usCount ++;

   if (usCount < 10)
       this->tusIndex[usCount] = usIndex;

   return (*this);
   }

INDEX::INDEX(unsigned short usIndex)
    {
   unsigned short usCount = 0;
   this->tusIndex[usCount++] = usIndex;
   while (usCount < 10)
    this->tusIndex[usCount++] = 0;
   }


INDEX::INDEX()
    {
   unsigned short usCount = 0;
   while (usCount < 10)
    this->tusIndex[usCount++] = 0;
   }


short INDEX::operator==(const INDEX &p_Index)
    {
   unsigned short usCount = 0;

   while (usCount < 10)
    {
      if (this->tusIndex[usCount] < p_Index.tusIndex[usCount])
        return -1;
      else if (this->tusIndex[usCount] > p_Index.tusIndex[usCount])
        return 1;
      else
        usCount ++;
      }

   return 0;
   }

INDEX& INDEX::DeleteLastIndex()
    {
   unsigned short usCount = 0;
   while ((this->tusIndex[usCount] != 0) && (usCount < 10))
    usCount++;

   if ((usCount < 10) && (usCount > 0))
    usCount--;
   this->tusIndex[usCount] = 0;

   return (*this);
   }



/****************************************************************************
 ****************************************************************************
        @@Message class:

    <B>Description:</B>
        Message class is a simple message data handling class which is used
        in MessageLogger and all script classes.

    Member functions:
        Message()                           Constructor.
        virtual ~Message()                  Virtual destructor.
        USHORT SetType(USHORT)              Sets message type. In Script
                                            classes there are three different
                                            types: informative, warning and
                                            error.
        USHORT SetMessageString(BSString &) Sets message string.
        USHORT SetLineNumber(ULONG)         Sets line number of script where
                                            message is related to. Can also
                                            be 0.
        USHORT SetMessageValue(USHORT)      Sets message value.
        USHORT ReturnType()                 Returns message type.
        BSString &ReturnMessageString()     Returns message string.
        ULONG ReturnLineNumber()            Returns Line number.
        USHORT ReturnMessageValue()         Returns message valuer.

****************************************************************************/



/*
 *@@Message:
 *      Constructor for Message class.Initializes _usType and _usLineNumber
 *      variables.
 */
Message::Message() : BSRoot(tMessage)
    {
    _usType = MSG_TYPE_UNDEF;
    _ulLineNumber = 0;
    };


/*
 *@@~Message:
 *      Destructor for Message class. Sets variables to zero and clears
 *      message string.
 */
Message::~Message()
    {
    _usType = MSG_TYPE_UNDEF;
    _ulLineNumber = 0;
    _usMessageValue = 0;
    _strMessageText = ""; // Clear text string.
    };


/*
 *@@SetType:
 *      Sets message type. In Script classes only three defined types are
 *      used: MSG_TYPE_INFO, MSG_TYPE_WARNING and MSG_TYPE_ERROR.
 */
USHORT Message::SetType(USHORT p_usType // Message type number.
                       )
    {
    _usType = p_usType;
    return NO_ERROR;
    }


/*
 *@@SetMessageString:
 *      Sets message string.
 */
USHORT Message::
        SetMessageText(const BSString &p_strMsgText // Reference to message string.
                      )
    {
    _strMessageText = p_strMsgText;
    return NO_ERROR;
    }


/*
 *@@SetMessageText:
 *      Sets message string.
 */
USHORT Message::SetMessageText(const PSZ p_pszMsgText // Pointer to message text.
                              )
    {
    _strMessageText = p_pszMsgText;
    return NO_ERROR;
    }


/*
 *@@SetLineNumber:
 *      Sets line number. Some messages may need line number for example
 *      there is a syntax error in a script and the line number of the
 *      syntax error is in _ulLineNumber.
 */
USHORT Message::SetLineNumber(ULONG p_ulLineNumber // Line number.
                             )
    {
    _ulLineNumber = p_ulLineNumber;
    return NO_ERROR;
    }


/*
 *@@SetMessageValue:
 *      Sets message value. Gives more detailed information about message.
 */
USHORT Message::SetMessageValue(USHORT p_usMsgValue // Message value.
                               )
    {
    _usMessageValue = p_usMsgValue;
    return NO_ERROR;
    }


/*
 *@@ReturnType:
 *      Return message type. (See SetType)
 */
USHORT Message::ReturnType()
    {
    return _usType;
    }


/*
 *@@ ReturnLineNumber
 *      Return line number. (See SetLineNumber)
 */
ULONG Message::ReturnLineNumber()
    {
    return _ulLineNumber;
    }


/*
 *@@ReturnMessageValue:
 *      Returns message value.
 */
USHORT Message::ReturnMessageValue()
    {
    return _usMessageValue;
    }


/*
 *@@ReturnMessageString:
 *      Return message string. (See SetMessageText)
 */
BSString &Message::ReturnMessageText()
    {
    return _strMessageText;
    }


/*
 *@@ComposeMessage:
 *      Composes message. New BSString is allocated.
 *      Formats the string to "message type": "message text".
 */
BSString &Message::ComposeMessage()
    {
    BSString *strMsg = new BSString;

    switch(_usType)
        {
        case MSG_TYPE_INFO:
            {
            *strMsg += SZ_MESSAGE_INFO;
            break;
            }
        case MSG_TYPE_WARNING:
            {
            *strMsg += SZ_MESSAGE_WARNING;
            break;
            }
        case MSG_TYPE_ERROR:
            {
            *strMsg += SZ_MESSAGE_ERROR;
            break;
            }
        default:
            {
            *strMsg += SZ_MESSAGE_MSG;
            break;
            }
        }

    *strMsg += ": ";
    CHAR szValue[128];
    sprintf(szValue, "Message value: %i", _usMessageValue);
    *strMsg += szValue;
    *strMsg += "  " + _strMessageText;
    if (_ulLineNumber > 0)
        {
        CHAR szLineNumber[256];
        sprintf(szLineNumber, "at line: %i", _ulLineNumber);
        *strMsg += szLineNumber;
        }
    return *strMsg;
    }





/****************************************************************************
 ****************************************************************************
        @@MessageLogger class:

    Member functions:

****************************************************************************/


/*
 *@@MessageLogger:
 *  Default constructor for MessageLogger class.
 *  Initializes variables.
 */
MessageLogger::MessageLogger() : _lstMessage(SHADOW)
    {
    _pParentLogger = NULL;
    _usMessages = 0;
    }


/*
 *@@MessageLogger:
 *  Copy constructor for MessageLogger class.
 */
MessageLogger::MessageLogger(MessageLogger &p_pLogger // Reference to a object.
                            ) : _lstMessage(SHADOW)
    {
    _pParentLogger = &p_pLogger;
    _usMessages = 0;
    }


/*
 *@@~MessageLogger:
 *  Destructor for MessageLogger class.
 *  Deletes all logged messages and clears other variables.
 */
MessageLogger::~MessageLogger()
    {
    DeleteLogs();
    _pParentLogger = NULL;
    _usMessages = 0;
    }


/*
 *@@operator=:
 *  Assignment operator for MessageLogger.
 */
MessageLogger &MessageLogger::operator=(MessageLogger &p_msgLog // Reference to a object.
                                       )
    {
    list<Message*>::iterator start, end;
    Message *pMsg;

    // Delete current logged messages.
    DeleteLogs();

    _pParentLogger = p_msgLog._pParentLogger;
    _usMessages = p_msgLog._usMessages;

    // Query list of messages from object logger and log all messages to
    // local message list.
    list<Message*> lstMsg = p_msgLog.ReturnMessageList();
    start = lstMsg.begin(); end = lstMsg.end();
    for (; start != end; start++)
        {
        pMsg = *start;
        LogMessage(*pMsg);
        }

    return (*this);
    }


/*
 *@@ReturnMessageList:
 *  Return constant list of pointers to messages.
 */
const list<Message*> MessageLogger::ReturnMessageList()
    {
    return _lstMessage;
    }


USHORT MessageLogger::DeleteLogs()
    {
    // Go through all messages in the list and delete them.
    /*if (_usMessages == 0)
        return NO_ERROR;*/
    list<Message*>::iterator start, end;
    start = _lstMessage.begin();  end = _lstMessage.end();
    _lstMessage.erase(start, end);
    _usMessages = 0;
    return NO_ERROR;
    }


USHORT MessageLogger::LogMessage(Message &p_Msg)
    {
    if (_pParentLogger)
        // If there is a parent logger, use it to log a message.
        _pParentLogger->LogMessage(p_Msg);
    else
        {
        // If not, put the message to local message list.
        Message *msg = new(Message);
        *msg = p_Msg;
        _lstMessage.push_back(msg);
        _usMessages ++;
        }
    return NO_ERROR;
    }


USHORT MessageLogger::SetParentLogger(MessageLogger &p_MsgLog)
    {
    list<Message*>::iterator start, end;
    Message *pMsg = NULL;
    start = _lstMessage.begin ();  end = _lstMessage.end ();

    _pParentLogger = &p_MsgLog;

    if (_pParentLogger)
        {
        for (; start != end; start++)
            {
            pMsg = *start;
            _pParentLogger->LogMessage(*pMsg);
            }
        DeleteLogs();
        _usMessages = 0;
        }

    return NO_ERROR;
    }


/*
 *@@ReturnComposedMessageString:
 *      Returns string which is composed from message text and line number
 *      in Message objects which are in message list. Every line is
 *      string is formatted as follow: MESSAGE_TYPE: MESSAGE_TEXT LINE_NBR.
 *@@changed v.0.9.6 [tahola]: Do not use parent message logger any more.
 */
BSString &MessageLogger::ReturnComposedMessageString()
    {
    // If there is a parent logger, use it.
    //if (_pParentLogger)
    //    return _pParentLogger->ReturnComposedMessageString();

    BSString *MsgString = new BSString;
    Message *pMsg = NULL;
    list<Message*>::iterator start, end;

    // Go through local list of messages and compose them to string.
    start = _lstMessage.begin ();  end = _lstMessage.end ();
    for (; start != end; start++)
        {
        pMsg = *start;
        *MsgString += pMsg->ComposeMessage() + "\n";
        }
    return *MsgString;
    }




/****************************************************************************
 ****************************************************************************
        @@ WarpINScriptBase class:

    <B>Description:</B>
        Base class for all script classes. Includes some virtual functions
        which must be implemented in derived classes.

    Member functions:
        USHORT SetIndex(USHORT)     Sets index number.
*****************************************************************************/

/*
 *@@SetIndex:
 *      Sets usIndex number.
 */
USHORT WarpINScriptBase::SetIndex(const USHORT p_usIndex)
    {
        _usIndex = p_usIndex;
        return (0);     // V0.9.14 (2001-07-07) [umoeller]
    }


/*
 *@@ReturnIndex:
 *      Returns UsIndex number.
 */

USHORT WarpINScriptBase::ReturnIndex()
    {
    return _usIndex;
    }


WarpINScriptBase::~WarpINScriptBase()
    {

    }




/****************************************************************************
 ****************************************************************************
        ScriptIndex class:
            :public WarpINScriptBase

    Member functions:
        ScripitIndex()              Constructor.
        ~ScriptIndex()              Destructor.
        SetIndexAttribute(USHORT)   Sets title attribute.
        ReturnIndexAttribute        Returns index attribute.
        ReturnIndex                 Returns object ID number.
        MakeScriptText              Returns pointer to script text.
        BuildFromText               Builds object accordingly to script text.
****************************************************************************/


/*
 *@@ScriptIndex:
 *      Constructor for ScriptIndex class. Initializes usIndex  by setting
 *      it to 0.
 */
ScriptIndex::ScriptIndex(USHORT usIndex)
    {
    usIndexAttribute = 0;
    SetIndex(usIndex);
    }



/*
 *@@~ScriptIndex:
 *      Destructor for ScriptIndex class. Does nothing special.
 */

ScriptIndex::~ScriptIndex()
    {
    }


/*
 *@@operator=:
 *@@added v.0.9.5 (2000-08-20) [tahola]
 */
ScriptIndex &ScriptIndex::operator=(ScriptIndex &p_index)
    {
    SetIndex(p_index.ReturnIndex());
    usIndexAttribute = p_index.usIndexAttribute;
    return (*this);
    }


/*
 *@@SetIndexAttribute:
 *      Sets usIndexAttribute.
 */

USHORT ScriptIndex::SetIndexAttribute(USHORT usIndexAttribute)
    {
    this->usIndexAttribute = usIndexAttribute;
    return NO_ERROR;
    }


/*
 *@@ReturnIndex:
 *  Returns usIndexAttribute number.
 */

USHORT ScriptIndex::ReturnIndexAttribute()
    {
    return usIndexAttribute;
    }


/*
 *@@MakeScriptText:
 *      Returns pointer to script text.
 */
PSZ ScriptIndex::MakeScriptText(BOOL bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    ulTextLength = LtoALength((LONG)usIndexAttribute);

    if (bAttribute)
        // Length of the attribute, =, " and "
        ulTextLength += strlen(SZ_ATTRIBUTE_INDEX) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bAttribute)
        sprintf(pszScriptText, "%s=%d", SZ_ATTRIBUTE_INDEX, usIndexAttribute);
    else
        sprintf(pszScriptText, "%d", usIndexAttribute);

    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds INDEX accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 *@@changed v.0.9.5 (2000-08-30) [tahola]: modified message logging.
 */
PSZ ScriptIndex::BuildFromText(PSZ pszScriptText,
                               ULONG ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    LONG lValue = 0;


    pszAttr = strhGetNumAttribValue(pszScriptText, SZ_ATTRIBUTE_INDEX, &lValue);
    if (pszAttr)
        this->SetIndexAttribute((USHORT)lValue);
    else
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_INDEX_NOTFOUND);
        msg.SetLineNumber(ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    return pszAttr + LtoALength(lValue);
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptPackageID class:

    <B>Description:</B>
        This class is for PACKAGEID attribute in PCK tag. This attribute
        contains information from vendor , application name, package identifier
        and version numbering. Different ScriptPackageID objects are identified
        from each others by usIndex. So when creating new object unique index
        number should be used. This class does not check if index is not
        valid ie. if same index number is already in use.

    Member functions:
        ScriptPackageID     constructor
        ~ScriptPackageID    destructor
        SetVendor           sets vendor text
        SetApplication      sets application text
        SetMajor            sets major version number
        SetMinor            sets minor version number
        SetRevision         sets revision version number
        ReturnVendor        returns const pointer to vendor string
        ReturnApplication   returns const pointer to application string
        ReturnPackage       returns const pointer to package string
        ReturnMajor         returns value of major version number
        ReturnMinor         returns value of minor version number
        ReturnRevision      returns value of revision version number
        MakeScriptText      returns pointer to script text
        BuildFromText       builds package ID datas from text
 ****************************************************************************/


/*
 *@@ScriptPackageID:
 *  This is default constructor for ScriptPackageID class. This initializes
 *  all local variables.
 */
ScriptPackageID::ScriptPackageID(USHORT p_usIndex)
    {
    // Set first character is char tables to '\0' (end of line).
    *_szVendor = '\0';
    *_szApplication = '\0';
    *_szPackage = '\0';

    _usMajor = 0;
    _usMinor = 0;
    _usRevision = 0;

    SetIndex(p_usIndex);
    }

/*
 *@@~ScriptPackageID:
 *  This is default destructor for ScriptPackageID class. This does not do
 *  anything special.
 */
ScriptPackageID::~ScriptPackageID()
    {
    }



/*
 *@@SetVendor:
 *  This function sets vendor string.
 */
USHORT ScriptPackageID::SetVendor(const PSZ p_pszVendor)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszVendor) < MAXVENDORNAMELENGTH)
        strcpy(this->_szVendor, p_pszVendor);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_VENDOR_STRTOOLONG);
        msg.SetMessageText(p_pszVendor);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }



/*
 *@@SetApplication:
 *  This function sets application string.
 */
USHORT ScriptPackageID::SetApplication(const PSZ p_pszApplication)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszApplication) < MAXAPPNAMELENGTH)
        strcpy(this->_szApplication, p_pszApplication);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_APPL_STRTOOLONG);
        msg.SetMessageText(p_pszApplication);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }

/*
 *@@SetPackage:
 *  This function sets package string.
 */
USHORT ScriptPackageID::SetPackage(const PSZ p_pszPackage)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszPackage) < MAXPCKNAMELENGTH)
        strcpy(this->_szPackage, p_pszPackage);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_PACKAGE_STRTOOLONG);
        msg.SetMessageText(p_pszPackage);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }



/*
 *@@SetMajor:
 *  This function sets major number.
 */
USHORT ScriptPackageID::SetMajor(USHORT p_usMajor)
    {
    this->_usMajor = p_usMajor;
    return NO_ERROR;
    }


/*
 *@@SetMinor:
 *  This function sets minor number.
 */
USHORT ScriptPackageID::SetMinor(USHORT p_usMinor)
    {
    this->_usMinor = p_usMinor;
    return NO_ERROR;
    }



/*
 *@@SetRevision:
 *  This function sets revision number.
 */
USHORT ScriptPackageID::SetRevision(USHORT p_usRevision)
    {
    this->_usRevision = p_usRevision;
    return NO_ERROR;
    }


/*
 *@@SetPackageID:
 *  For compatibility only. Does actually nothing. Must be removed at some point.
 *  Useless. No need anymore. Nobody will need it. Nobody cares about it.
 */
USHORT ScriptPackageID::SetPackageID(PACKAGEIDDATA packageIDData)
    {
    return NO_ERROR;
    }


/*
 *@@ReturnVendor:
 *  This function returns const pointer to vendor string.
 */
const PSZ ScriptPackageID::ReturnVendor()
    {
    return (PSZ)_szVendor;
    }



/*
 *@@ReturnApplication:
 *  This function returns const pointer to application string.
 */
const PSZ ScriptPackageID::ReturnApplication()
    {
    return (PSZ)_szApplication;
    }



/*
 *@@ReturnPackage:
 *  This function returns const pointer to package string.
 */
const PSZ ScriptPackageID::ReturnPackage()
    {
    return (PSZ)_szPackage;
    }


/*
 *@@ReturnMajor:
 *  This function returns major version number.
 */
USHORT ScriptPackageID::ReturnMajor()
    {
    return _usMajor;
    }


/*
 *@@ReturnMinor:
 *  This function returns minor version number.
 */
USHORT ScriptPackageID::ReturnMinor()
    {
    return _usMinor;
    }


/*
 *@@ReturnRevision:
 *  This function returns revision version number.
 */
USHORT ScriptPackageID::ReturnRevision()
    {
    return _usRevision;
    }


/*
 *@@MakeScriptText:
 *  This function returns pointer to script text. If bShowAttribute is FALSE
 *  PACKAGEID="" text will not be included to the script text.
 */
PSZ ScriptPackageID::MakeScriptText(BOOL p_bShowAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    XSTRING xstrScriptText;
    CHAR szTemp[1024];

    xstrInit(&xstrScriptText, 0);
    szTemp[0]='\0';
    ulTextLength = strlen(_szVendor) + strlen(_szApplication) + sizeof(_szPackage);
    ulTextLength += LtoALength(_usMajor) + LtoALength(_usMinor) +
        LtoALength(_usRevision);
    ulTextLength += 5;

    if (p_bShowAttribute)
        // Length of the attribute, =, " and "
        ulTextLength += strlen(SZ_ATTRIBUTE_PACKAGEID) + 3;

    if (p_bShowAttribute)
        sprintf(szTemp, "%s=\"%s\\%s\\%s\\%i\\%i\\%i\"", SZ_ATTRIBUTE_PACKAGEID,
            _szVendor, _szApplication, _szPackage, _usMajor, _usMinor, _usRevision);
    else
        sprintf(szTemp, "%s\\%s\\%s\\%i\\%i\\%i",
            _szVendor, _szApplication, _szPackage, _usMajor, _usMinor, _usRevision);

    xstrcat(&xstrScriptText, szTemp, 0);
    return xstrScriptText.psz;
    }


/*
 *@@BuildFromText:
 *  Builds package ID accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptPackageID::BuildFromText(PSZ pszScriptText,
                                   ULONG ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL, pszTemp2 = NULL, pszTemp3 = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    ULONG ulVersion = 0;
    ULONG ulAttrLen = 0;

    pszAttr = strhGetTextAttr(pszScriptText, SZ_ATTRIBUTE_PACKAGEID, &ulOffset);

    if (!pszAttr)
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_NOTFOUND);
        msg.SetLineNumber(ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    pszTemp2 = pszAttr;
    pszTemp = strchr(pszAttr, '\\');
    if (pszTemp) // Set vendor
        {
        pszTemp3 = strhSubstr(pszTemp2, pszTemp);
        if (strlen(pszTemp3) == 0)
            {
            // There is no vendor string so report it.
            msg.SetType(MSG_TYPE_WARNING);
            msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_VENDOR_NOTFOUND);
            msg.SetLineNumber(ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        this->SetVendor(pszTemp3);
        delete [] pszTemp3;
        pszTemp2 = ++pszTemp;
        pszTemp = strchr(pszTemp, '\\');
        if (pszTemp) // Set application
            {
            pszTemp3 = strhSubstr(pszTemp2, pszTemp);
            if (strlen(pszTemp3) == 0)
                {
                // There is no application string so report it.
                msg.SetType(MSG_TYPE_WARNING);
                msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_APPL_NOTFOUND);
                msg.SetLineNumber(ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            this->SetApplication(pszTemp3);
            delete [] pszTemp3;
            pszTemp2 = ++pszTemp;
            pszTemp = strchr(pszTemp, '\\');
            if (pszTemp) // Set package
                {
                pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                if (strlen(pszTemp3) == 0)
                    {
                    // There is no package string so report it.
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_PACKAGE_NOTFOUND);
                    msg.SetLineNumber(ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                this->SetPackage(pszTemp3);
                delete [] pszTemp3;
                pszTemp2 = ++pszTemp;
                pszTemp = strchr(pszTemp, '\\');
                if (pszTemp) // Set major
                    {
                    pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                    if (strlen(pszTemp3) == 0)
                        {
                        // There is no major number so report it.
                        msg.SetType(MSG_TYPE_WARNING);
                        msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_MAJOR_NOTFOUND);
                        msg.SetLineNumber(ulLineNumber);
                        _messageLogger.LogMessage(msg);
                        }
                    sscanf(pszTemp3, "%d", &ulVersion);
                    this->SetMajor((USHORT)ulVersion);
                    delete [] pszTemp3;
                    pszTemp2 = ++pszTemp;
                    pszTemp = strchr(pszTemp, '\\');
                    if (pszTemp) // Set minor and revision
                        {
                        pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                        if (strlen(pszTemp3) == 0)
                            {
                            // There is no minor number so report it.
                            msg.SetType(MSG_TYPE_WARNING);
                            msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_MINOR_NOTFOUND);
                            msg.SetLineNumber(ulLineNumber);
                            _messageLogger.LogMessage(msg);
                            }
                        sscanf(pszTemp3, "%d", &ulVersion);
                        this->SetMinor((USHORT)ulVersion);
                        delete [] pszTemp3;
                        pszTemp2 = ++pszTemp;

                        // Set revision
                        if (strlen(pszTemp) == 0) // There is \\ but no value.
                            this->SetRevision(0);
                        else
                            {
                            pszTemp3 = strhSubstr(pszTemp2, pszTemp + strlen(pszTemp));
                            sscanf(pszTemp3, "%d", &ulVersion);
                            this->SetRevision((USHORT)ulVersion);
                            delete [] pszTemp3;
                            }
                        }
                    else // Minor and revision
                        {
                        pszTemp3 = strhSubstr(pszTemp2, pszTemp2 + strlen(pszTemp2));
                        sscanf(pszTemp3, "%d", &ulVersion);
                        this->SetMinor((USHORT)ulVersion);
                        delete [] pszTemp3;
                        this->SetRevision(0);
                        }
                    }
                else // Major warning.
                    {
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_NOTFOUND);
                    msg.SetLineNumber(ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                }
            else // Package warning.
                {
                msg.SetType(MSG_TYPE_WARNING);
                msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_PACKAGE_NOTFOUND);
                msg.SetLineNumber(ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        else // Application warning.
            {
            msg.SetType(MSG_TYPE_WARNING);
            msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_APPL_NOTFOUND);
            msg.SetLineNumber(ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        }
    else // Vendor warning.
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_PACKAGEID_VENDOR_NOTFOUND);
        msg.SetLineNumber(ulLineNumber);
        _messageLogger.LogMessage(msg);
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptTitle class:

    <B>Description:</B>
        This class is for TITLE attribute in PCK tag. This class contains
        information about title. Title is used to specify what will be
        displayed in WarpIN's container page.

    Class member functions:
        ScripitTitle(USHORT)            Constructor.
        ~ScriptTitle()                  Destructor.
        USHORT SetTitle(const PSZ)      Sets title text.
        PSZ ReturnTitle()               Returns const pointer to title text.
        PSZ MakeScriptText()            Returns pointer to script text.
        PSZ BuildFromText(PSZ, ULONG)   Builds object accordingly to script text.
 ****************************************************************************/


/*
 *@@ScriptTitle:
 *  Constructor for ScriptTitle class. Initializes szTitle string by setting
 *  first char to '\0'.
 */
ScriptTitle::ScriptTitle(USHORT p_usIndex)
    {
    *_szTitle = '\0';
    SetIndex(p_usIndex);
    }


/*
 *@@~ScriptTitle:
 *  destructor for ScriptTitle class. Does nothing special.
 */

ScriptTitle::~ScriptTitle()
    {
    }


/*
 *@@SetTitle:
 *  Sets title text.
 */
USHORT ScriptTitle::SetTitle(const PSZ p_pszTitle)
    {
    Message msg;
    USHORT rc = NO_ERROR;

    if (strlen(p_pszTitle) < MAXTITLELENGTH)
        strcpy(this->_szTitle, p_pszTitle);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_TITLE_TITLE_STRTOOLONG);
        msg.SetMessageText(p_pszTitle);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@ReturnTitle:
 *  Returns const pointer to title text.
 */
const PSZ ScriptTitle::ReturnTitle()
    {
    return _szTitle;
    }

/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptTitle::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
{
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    ulTextLength = strlen(_szTitle);
    CHAR szTemp[1024];
    szTemp[0]='\0';

    if (p_bAttribute)
        // Length of the attribute, =, " and "
        ulTextLength += strlen(SZ_ATTRIBUTE_TITLE) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s\"", SZ_ATTRIBUTE_TITLE, _szTitle);
    else
        sprintf(pszScriptText, "%s", _szTitle);

    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds package ID accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptTitle::BuildFromText(PSZ pszScriptText,
                               ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    ULONG ulAttrLen = 0;


    pszAttr = strhGetTextAttr(pszScriptText, SZ_ATTRIBUTE_TITLE, &ulOffset);
    if (pszAttr)
        this->SetTitle(pszAttr);
    else
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_TITLE_TITLE_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return pszScriptText + ulOffset + ulAttrLen + 1;
    }




/****************************************************************************
 ****************************************************************************
        @@ScriptTarget class:

    <B>Description:</B>
        This class is for TARGET attribute in PCK tag. This class contains
        information about target path where files will be installed by
        default. Same path will be displayed in WarpIN's container page.


    Member functions:
        ScripitTarget(USHORT)           Constructor.
        ~ScriptTarget()                 Destructor.
        USHORT SetTarget(const PSZ)     Sets title text.
        const PSZ ReturnTarget()        Returns const pointer to title text.
        PSZ MakeScriptText(BOOL)        Returns pointer to script text.
        PSZ BuildFromText(PSZ, ULONG)   Construct object from script text.
 ***************************************************************************/


/*
 *@@ScriptTarget:
 *  Constructor for ScriptTarget class. Initializes szTarget string by
 *  setting first char to '\0'.
 */
ScriptTarget::ScriptTarget(USHORT p_usIndex)
    {
    *_szTarget = '\0';
    SetIndex(p_usIndex);
    }



/*
 *@@~ScriptTarget:
 *  destructor for ScriptTarget class.
 */
ScriptTarget::~ScriptTarget()
    {
    *_szTarget = '\0';
    }


/*
 *@@SetTarget:
 *  Sets target text.
 */
USHORT ScriptTarget::SetTarget(const PSZ p_pszTarget)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszTarget) < MAXTARGETLENGTH)
        strcpy(this->_szTarget, p_pszTarget);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_TARGET_TARGET_STRTOOLONG);
        msg.SetMessageText(p_pszTarget);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@ReturnTitle:
 *  Returns const pointer to target text.
 */
const PSZ ScriptTarget::ReturnTarget()
    {
    return _szTarget;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptTarget::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    XSTRING xstrScriptText;
    CHAR szTemp[1024];
    szTemp[0]='\0';
    xstrInit(&xstrScriptText, 0);

    if (p_bAttribute)
        sprintf(szTemp, "%s=\"%s\"", SZ_ATTRIBUTE_TARGET, _szTarget);
    else
        sprintf(szTemp, "%s\"", _szTarget);

    xstrcat(&xstrScriptText, szTemp, 0);
    return xstrScriptText.psz;
    }



/*
 *@@BuildFromText:
 *  Builds target object accordingly to script text. If any errors is found
 *  NULL pointer is returned.If no errors occurred, pointer to after end of
 *  tag is returned so multiple searches can be done.
 */
PSZ ScriptTarget::BuildFromText(PSZ p_pszScriptText,
                               ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    ULONG ulAttrLen = 0;


    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_TARGET, &ulOffset);
    if (pszAttr)
        this->SetTarget(pszAttr);
    else
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_TARGET_TARGET_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }



/****************************************************************************
 ****************************************************************************
        @@ScriptKeyWord class:

    <B>Description:</B>
        This class is for keywords in PCK tag. This class contains
        all four keywords: BASE, FIXED, SELECT and NODESELECT. Detailed
        information about the meaning of the keywords can be found from
        <I>WarpIN Programmer's Guide and Reference</I>.


    Member functions:
        ScripitKeyWord(USHORT)          Constructor.
        ~ScriptKeyWord                  Destructor.
        USHORT SetBase(BOOL)            Sets BASE keyword.
        USHORT SetFixed(BOOL)           Sets FIXED keyword.
        USHORT SetSelect(BOOL)          Sets SELECT keyword.
        USHORT SetNoDeselect(BOOL)      Sets BODESELECT keyword.
        USHORT SetKeyWord(USHORT)       Sets all keywords simultaniously.
        BOOL ReturnBase()               Returns BASE keyword state.
        BOOL ReturnFixed()              Returns FIXED keyword state.
        BOOL ReturnSelect()             Returns SELECT keyword state.
        BOOL ReturnNoDeselect()         Returns NODESELECT keyword state.
        PSZ MakeScriptText(BOOL)        Returns pointer to script text.
        PSZ BuildFromText(PSZ, ULONG)   Builds object from text.
 ***************************************************************************/


/*
 *@@ScriptKeyWord:
 *  Constructor for ScriptKeyWord class. Initializes object by setting
 *  usIndex to given value on uses default value (1).
 */
ScriptKeyWord::ScriptKeyWord(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    _bBase = FALSE;
    _bFixed = FALSE;
    _bSelect = FALSE;
    _bNoDeselect = FALSE;
    }


/*
 *@@~ScriptKeyWord:
 *  Destructor for ScriptKeyWord class.
 */
ScriptKeyWord::~ScriptKeyWord()
    {
    _bBase = FALSE;
    _bFixed = FALSE;
    _bSelect = FALSE;
    _bNoDeselect = FALSE;
    }


/*
 *@@SetBase:
 *  Sets BASE keyword state.
 */
USHORT ScriptKeyWord::SetBase(BOOL p_bBase)
    {
    _bBase = p_bBase;
    return NO_ERROR;
    }


/*
 *@@ ScriptKeyWord::SetFixed
 *  Sets FIXED keyword state.
 */
USHORT ScriptKeyWord::SetFixed(BOOL p_bFixed)
    {
    _bFixed = p_bFixed;
    return NO_ERROR;
    }


/*
 *@@SetSelect:
 *  Sets SELECT keyword state.
 */
USHORT ScriptKeyWord::SetSelect(BOOL p_bSelect)
    {
    _bSelect = p_bSelect;
    return NO_ERROR;
    }


/*
 *@@SetNoDeselect:
 *  Sets NODESELECT keyword state.
 */
USHORT ScriptKeyWord::SetNoDeselect(BOOL p_bNoDeselect)
    {
    _bNoDeselect = p_bNoDeselect;
    return NO_ERROR;
    }


/*
 *@@SetKeyWord:
 *  Sets all keyword states acconding to usKeyWords. Different keywords can
 *  be combined with OR. Example: KEYWORD_BASE | KEYWORD_FIXED.
 */
USHORT ScriptKeyWord::SetKeyWord(USHORT p_usKeyWords)
    {
    if (p_usKeyWords & KEYWORD_BASE)
        _bBase = TRUE;
    else
        _bBase = FALSE;

    if (p_usKeyWords & KEYWORD_FIXED)
        _bFixed = TRUE;
    else
        _bFixed = FALSE;

    if (p_usKeyWords & KEYWORD_SELECT)
        _bSelect = TRUE;
    else
        _bSelect = FALSE;

    if (p_usKeyWords & KEYWORD_NODESELECT)
        _bNoDeselect = TRUE;
    else
        _bNoDeselect = FALSE;

    return NO_ERROR;
    }


/*
 *@@ReturnBase:
 *  Returns bBase state.
 */
BOOL ScriptKeyWord::ReturnBase()
    {
    return _bBase;
    }


/*
 *@@ReturnFixed:
 *  Returns bFixed state.
 */
BOOL ScriptKeyWord::ReturnFixed()
    {
    return _bFixed;
    }


/*
 *@@ReturnSelect:
 *  Returns bSelect state.
 */
BOOL ScriptKeyWord::ReturnSelect()
    {
    return _bSelect;
    }


/*
 *@@ReturnNoDeselect:
 *  Returns bBase state.
 */
BOOL ScriptKeyWord::ReturnNoDeselect()
    {
    return _bNoDeselect;
    }


/*
 *@@ReturnKeyWord:
 *  Returns all keyword states.
 */
USHORT ScriptKeyWord::ReturnKeyWord()
    {
    USHORT usReturnValue = 0;

    if (_bBase)
        usReturnValue |= KEYWORD_BASE;
    if (_bFixed)
        usReturnValue |= KEYWORD_FIXED;
    if (_bSelect)
        usReturnValue |= KEYWORD_SELECT;
    if (_bNoDeselect)
        usReturnValue |= KEYWORD_NODESELECT;

    return usReturnValue;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptKeyWord::MakeScriptText(BOOL p_bAttribute, BOOL p_bLineFeed)
    {
    ULONG ulSpaces = 0;
    XSTRING xstrScriptText;
    xstrInit(&xstrScriptText, 0);

    if (_bBase)
        ulSpaces ++;
    if (_bFixed)
        ulSpaces ++;
    if (_bSelect)
        ulSpaces ++;
    if (_bNoDeselect)
        ulSpaces ++;

    if (_bBase)
        {
        xstrcat(&xstrScriptText, SZ_KEYWORD_BASE, 0);
        if (p_bLineFeed && ulSpaces > 0)
            xstrcat(&xstrScriptText, "\n", 0);
        else if (ulSpaces > 0)
            xstrcat(&xstrScriptText, " ", 0);
        ulSpaces--;
        }
    if (_bFixed)
        {
        xstrcat(&xstrScriptText, SZ_KEYWORD_FIXED, 0);
        if (p_bLineFeed && ulSpaces > 0)
            xstrcat(&xstrScriptText, "\n", 0);
        else if (ulSpaces > 0)
            xstrcat(&xstrScriptText, " ", 0);
        ulSpaces--;
        }
    if (_bSelect)
        {
        xstrcat(&xstrScriptText, SZ_KEYWORD_SELECT, 0);
        if (p_bLineFeed && ulSpaces > 0)
           xstrcat(&xstrScriptText, "\n", 0);
        else if (ulSpaces > 0)
            xstrcat(&xstrScriptText, " ", 0);
        ulSpaces--;
        }
    if (_bNoDeselect)
        {
        xstrcat(&xstrScriptText, SZ_KEYWORD_NODESELECT, 0);
        if (p_bLineFeed && ulSpaces > 0)
            xstrcat(&xstrScriptText, "\n", 0);
        else if (ulSpaces > 0)
            xstrcat(&xstrScriptText, " ", 0);
        ulSpaces--;
        }

    return xstrScriptText.psz;
    }


/*
 *@@BuildFromText:
 *  Builds keyword object accordingly to script text. If any errors is found
 *  NULL is returned otherwice pointer to just after end of tag is returned so
 *  multiple searches can be done.
 */
PSZ ScriptKeyWord::BuildFromText(PSZ p_pszScriptText,
                               ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszKWord = NULL;
    BOOL KWFound = FALSE;


    pszKWord = strhFindWord(p_pszScriptText, SZ_KEYWORD_BASE, "\x0d\x0a ()/\\-,.",
                            "\x0d\x0a ()/\\-,.:;");
    if (pszKWord)
        {
        KWFound = TRUE;
        this->SetBase(TRUE);
        }
    else
        this->SetBase(FALSE);

    pszKWord = strhFindWord(p_pszScriptText, SZ_KEYWORD_FIXED, "\x0d\x0a ()/\\-,.",
                            "\x0d\x0a ()/\\-,.:;");
    if (pszKWord)
        {
        KWFound = TRUE;
        this->SetFixed(TRUE);
        }
    else
        this->SetFixed(FALSE);

    pszKWord = strhFindWord(p_pszScriptText, SZ_KEYWORD_SELECT, "\x0d\x0a ()/\\-,.",
                            "\x0d\x0a ()/\\-,.:;");
    if (pszKWord)
        {
        KWFound = TRUE;
        this->SetSelect(TRUE);
        }
    else
        this->SetSelect(FALSE);

    pszKWord = strhFindWord(p_pszScriptText, SZ_KEYWORD_NODESELECT, "\x0d\x0a ()/\\-,.",
                            "\x0d\x0a ()/\\-,.:;");
    if (pszKWord)
        {
        KWFound = TRUE;
        this->SetNoDeselect(TRUE);
        }
    else
        this->SetNoDeselect(FALSE);

    if (KWFound)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_KEYWORD_PCKKW_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    return p_pszScriptText;
    }




/****************************************************************************
 ****************************************************************************
        @@ScriptRequires class:

    <B>Description:</B>
        This class is for REQUIRES attribute in PCK tag. This class contains
        Detailed information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScripitRequires(USHORT)     Constructor.
        ~ScriptRequires()           Destructor.
        SetVendor                   Sets vendor text.
        SetApplication              Sets application text.
        SetPackage                  Sets package text.
        SetMajor                    Sets major number.
        SetMinor                    Sets minor number.
        SetRevision                 Sets revision number.
        SetRequires                 Sets all fields simultaneously.
        ReturnVendor                Returns const pointer to vendor string.
        ReturnApplication           Returns const pointer to application string.
        ReturnPackage               Returns const pointer to package string.
        ReturnMajor                 Returns major number.
        ReturnMinor                 Returns minor number.
        ReturnRevision              Returns revision number.
        ReturnRequires              Returns all fields.
        MakeScriptText              Returns pointer to script text.
 ***************************************************************************/


/*
 *@@ScriptRequires:
 *  Constructor for ScriptRequires class. Initializes object by setting
 *  fields to zero.
 */
ScriptRequires::ScriptRequires(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    *_szVendor = '\0';
    *_szApplication = '\0';
    *_szPackage = '\0';
    _usMajor = 0;
    _usMinor = 0;
    _usRevision = 0;
    _usPckIndex = 0;
    }


/*
 *@@~ScriptRequires:
 *  Destructor for ScriptRequires class.
 */
ScriptRequires::~ScriptRequires()
    {
    *_szVendor = '\0';
    *_szApplication = '\0';
    *_szPackage = '\0';
    _usMajor = 0;
    _usMinor = 0;
    _usRevision = 0;
    _usPckIndex = 0;
    }


/*
 *@@SetVendor:
 *  Sets vendor text.
 */
USHORT ScriptRequires::SetVendor(const PSZ p_pszVendor)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszVendor) < MAXVENDORNAMELENGTH)
        strcpy(this->_szVendor, p_pszVendor);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_VENDOR_STRTOOLONG);
        msg.SetMessageText(p_pszVendor);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@SetApplication:
 *  Sets application text.
 */
USHORT ScriptRequires::SetApplication(const PSZ p_pszApplication)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszApplication) < MAXAPPNAMELENGTH)
        strcpy(this->_szApplication, p_pszApplication);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_APPL_STRTOOLONG);
        msg.SetMessageText(p_pszApplication);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@SetPackage:
 *  Sets package text.
 */
USHORT ScriptRequires::SetPackage(const PSZ p_pszPackage)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszPackage) < MAXPCKNAMELENGTH)
        strcpy(this->_szPackage, p_pszPackage);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_PACKAGE_STRTOOLONG);
        msg.SetMessageText(p_pszPackage);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@SetMajor
 *  This function sets major number.
 */
USHORT ScriptRequires::SetMajor(USHORT p_usMajor)
    {
    this->_usMajor = p_usMajor;
    return NO_ERROR;
    }


/*
 *@@SetMinor:
 *  This function sets minor number.
 */
USHORT ScriptRequires::SetMinor(USHORT p_usMinor)
    {
    this->_usMinor = p_usMinor;
    return NO_ERROR;
    }


/*
 *@@SetRevision:
 *  This function sets revision number.
 */
USHORT ScriptRequires::SetRevision(USHORT p_usRevision)
    {
    this->_usRevision = p_usRevision;
    return NO_ERROR;
    }


/*
 *@@SetPckIndex:
 *  This function sets required package number.
 */
USHORT ScriptRequires::SetPckIndex(USHORT p_usPckIndex)
    {
    this->_usPckIndex = p_usPckIndex;
    return NO_ERROR;
    }


/*
 *@@SetRequires:
 *  This function sets all datas.
 */
USHORT ScriptRequires::SetRequires(REQUIRESDATA p_requiresData)
    {
    USHORT rc = NO_ERROR;

    if (SetVendor(p_requiresData.szVendor) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetApplication(p_requiresData.szApplication) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetPackage(p_requiresData.szPackage) != NO_ERROR)
        rc = STRING_TOO_LONG;
    SetMajor(p_requiresData.usMajor);
    SetMinor(p_requiresData.usMinor);
    SetRevision(p_requiresData.usRevision);

    return rc;
    }


/*
 *@@ReturnVendor:
 *  Returns const pointer to vendor text.
 */
const PSZ ScriptRequires::ReturnVendor()
    {
    return _szVendor;
    }


/*
 *@@ReturnApplication:
 *  Returns const pointer to application text.
 */
const PSZ ScriptRequires::ReturnApplication()
    {
    return _szApplication;
    }


/*
 *@@ReturnPackage:
 *  Returns const pointer to package text.
 */
const PSZ ScriptRequires::ReturnPackage()
    {
    return _szPackage;
    }


/*
 *@@ReturnMajor:
 *  Returns major number.
 */
USHORT ScriptRequires::ReturnMajor()
    {
    return _usMajor;
    }


/*
 *@@ReturnMinor:
 *  Returns minor number.
 */
USHORT ScriptRequires::ReturnMinor()
    {
    return _usMinor;
    }


/*
 *@@ReturnRevision:
 *  Returns revision number.
 */
USHORT ScriptRequires::ReturnRevision()
    {
    return _usRevision;
    }


/*
 *@@ ScriptRequires::ReturnPckIndex
 *  Returns revision number.
 */
USHORT ScriptRequires::ReturnPckIndex()
    {
    return _usPckIndex;
    }


/*
 *@@ReturnRequires:
 *  This function returns all datas.
 */
REQUIRESDATA ScriptRequires::ReturnRequires()
    {
    REQUIRESDATA requiresData;
    strcpy(requiresData.szVendor, this->_szVendor);
    strcpy(requiresData.szApplication, this->_szApplication);
    strcpy(requiresData.szPackage, this->_szPackage);
    requiresData.usMajor = this->_usMajor;
    requiresData.usMinor = this->_usMinor;
    requiresData.usRevision = this->_usRevision;
    return requiresData;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptRequires::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    ulTextLength = strlen(_szVendor) + strlen(_szApplication) +
                    strlen(_szPackage);
    ulTextLength += LtoALength(_usMajor) + LtoALength(_usMinor) +
                    LtoALength(_usRevision);
    ulTextLength += 5;

    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_REQUIRES) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s\\%s\\%s\\%i\\%i\\%i\"",
            SZ_ATTRIBUTE_REQUIRES, _szVendor, _szApplication, _szPackage,
            _usMajor, _usMinor, _usRevision);
    else
        sprintf(pszScriptText, "%s\\%s\\%s\\%i\\%i\\%i",
            _szVendor, _szApplication, _szPackage, _usMajor, _usMinor,
            _usRevision);
    return pszScriptText;
    }



/*
 *@@BuildFromText:
 *  Builds requires accordingly to script text. If any errors is found in script
 *  text, NULL is returned otherwice pointer to after end of tag is returned so
 *  multiple searches can be done.
 */
PSZ ScriptRequires::BuildFromText(PSZ p_pszScriptText,
                                  ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL, pszTemp2 = NULL, pszTemp3 = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    ULONG ulVersion = 0;
    BOOL bError = FALSE;
    ULONG ulAttrLen = 0;

    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_REQUIRES, &ulOffset);

    if (!pszAttr)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    pszTemp2 = pszAttr;
    pszTemp = strchr(pszAttr, '\\');
    if (pszTemp) // Set vendor
        {
        pszTemp3 = strhSubstr(pszTemp2, pszTemp);
        this->SetVendor(pszTemp3);
        delete [] pszTemp3;
        pszTemp2 = ++pszTemp;
        pszTemp = strchr(pszTemp, '\\');
        if (pszTemp) // Set application
            {
            pszTemp3 = strhSubstr(pszTemp2, pszTemp);
            this->SetApplication(pszTemp3);
            delete [] pszTemp3;
            pszTemp2 =  ++pszTemp;
            pszTemp = strchr(pszTemp, '\\');
            if (pszTemp) // Set package
                {
                pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                this->SetPackage(pszTemp3);
                delete [] pszTemp3;
                pszTemp2 = ++pszTemp;
                pszTemp = strchr(pszTemp, '\\');
                if (pszTemp) // Set major
                    {
                    pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                    sscanf(pszTemp3, "%d", &ulVersion);
                    this->SetMajor((USHORT)ulVersion);
                    delete [] pszTemp3;
                    pszTemp2 = ++pszTemp;
                    pszTemp = strchr(pszTemp, '\\');
                    if (pszTemp) // Set minor and revision
                        {
                        if (pszTemp3 = strhSubstr(pszTemp2, pszTemp))
                            {
                            sscanf(pszTemp3, "%d", &ulVersion);
                            this->SetMinor((USHORT)ulVersion);
                            delete [] pszTemp3;
                            pszTemp2 = ++pszTemp;
                            }
                        else
                            {
                            msg.SetType(MSG_TYPE_ERROR);
                            msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_MINOR_NOTFOUND);
                            msg.SetLineNumber(p_ulLineNumber);
                            _messageLogger.LogMessage(msg);
                            bError = TRUE;
                            }

                        // Set revision
                        if (strlen(pszTemp) == 0) // There is \\ but no value.
                            {
                            msg.SetType(MSG_TYPE_INFO);
                            msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_REV_NOTFOUND);
                            msg.SetLineNumber(p_ulLineNumber);
                            _messageLogger.LogMessage(msg);
                            this->SetRevision(0);
                            }
                        else
                            {
                            pszTemp3 = strhSubstr(pszTemp2, pszTemp + strlen(pszTemp));
                            sscanf(pszTemp3, "%d", &ulVersion);
                            this->SetRevision((USHORT)ulVersion);
                            delete [] pszTemp3;
                            }
                        }
                    else // Minor and revision
                        {
                        if (pszTemp3 =
                            strhSubstr(pszTemp2, pszTemp2 + strlen(pszTemp2)))
                            {
                            sscanf(pszTemp3, "%d", &ulVersion);
                            this->SetMinor((USHORT)ulVersion);
                            delete [] pszTemp3;
                            this->SetRevision(0);
                            }
                        else
                            {
                            msg.SetType(MSG_TYPE_ERROR);
                            msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_MINOR_NOTFOUND);
                            msg.SetLineNumber(p_ulLineNumber);
                            _messageLogger.LogMessage(msg);
                            bError = TRUE;
                            }
                        }
                    }
                else // Major error
                    {
                    msg.SetType(MSG_TYPE_ERROR);
                    msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_MAJOR_NOTFOUND);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    bError = TRUE;
                    }
                }
            else // Package error
                {
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_PACKAGE_NOTFOUND);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                bError = TRUE;
                }
            }
        else // Application error
            {
            msg.SetType(MSG_TYPE_ERROR);
            msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_APPL_NOTFOUND);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            bError = TRUE;
            }
        }
    else // Vendor error
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_VENDOR_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        bError = TRUE;
        }
    if (bError == TRUE)
        return NULL;

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptConfigSys class:

    <B>Description:</B>
        This class is for CONFIGSYS attributes in PCK tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.


    Member functions:
        ScriptConfigSys Constructor.
        ~ScriptConfigSys Destructor.
        SetStatement    Sets statement text.
        SetUnique       Sets unique modifier state.
        SetAddRight     Sets addright state.
        SetAddLeft      Sets addleft state.
        SetAddAfter     Sets addafter state.
        SetAddBefore    Sets addbefore state.
        SetRemoveLine   Sets removeline state.
        SetRemovePart   Sets removepart state.
        SetSearchString Sets search string.
        SetConfigSys    Sets all fields.
        ReturnStatement Returns const pointer to statement string.
        ReturnUnique    Returns state of unique modifier.
        ReturnAddRight  Returns state of addright modifier.
        ReturnAddLeft   Returns addleft modifier state.
        ReturnAddAfter  Returns addafter state.
        ReturnAddBefore Returns addbefore state.
        ReturnRemoveLine Returns removeline state.
        ReturnRemovePart Returns removepart state.
        ReturnSearchString Returns const pointer to search string.
        MakeScriptText  Returns pointer to script text.
        BuildFromText   Builds object from text.
 ***************************************************************************/


/*
 *@@ScriptConfigSys:
 *  Constructor for ScriptConfigSys class. Initializes object by setting
 *  fields to zero.
 */
ScriptConfigSys::ScriptConfigSys(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    *_szStatement = '\0';
    *_szSearchString = '\0';
    _bUnique = FALSE;
    _bAddRight = FALSE;
    _bAddLeft = FALSE;
    _bAddRight = FALSE;
    _bAddTop = FALSE;
    _bAddAfter = FALSE;
    _bAddBefore = FALSE;
    _bRemoveLine = FALSE;
    _bRemovePart = FALSE;
    }


/*
 *@@~ScriptConfigSys:
 *  Destructor for ScriptConfigSys class.
 */
ScriptConfigSys::~ScriptConfigSys()
    {
    *_szStatement = '\0';
    *_szSearchString = '\0';
    _bUnique = FALSE;
    _bAddRight = FALSE;
    _bAddLeft = FALSE;
    _bAddRight = FALSE;
    _bAddTop = FALSE;
    _bAddAfter = FALSE;
    _bAddBefore = FALSE;
    _bRemoveLine = FALSE;
    _bRemovePart = FALSE;
    }


/*
 *@@~SetStatement:
 *  This function sets statement string.
 */
USHORT ScriptConfigSys::SetStatement(const PSZ p_pszStatement)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszStatement) < MAXSTATEMENTLENGTH)
        strcpy(this->_szStatement, p_pszStatement);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_STATEMENT_STRTOOLONG);
        msg.SetMessageText(p_pszStatement);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@SetUnique:
 *  This function sets unique state. Returns error code if error occurred.
 *
 *  <B>Return codes:</B>
 *      ERR_CONFIGSYS_ADDRIGHT_SETTED:      ADDRIGHT modifier cannot
 *                                          be used with UNIQUE.
 *      ERR_CONFIGSYS_ADDLEFT_SETTED:       ADDLEFT modifier cannot
 *                                          be used with UNIQUE.
 *      ERR_CONFIGSYS_REMOVELINE_SETTED:    REMOVELINE modifier cannot
 *                                          be used with UNIQUE.
 *      ERR_CONFIGSYS_REMOVEPART_SETTED:    REMOVEPART modifier cannot
 *                                          be used with UNOQUE.
 *      The codes can be ORed with each others.
 */
USHORT ScriptConfigSys::SetUnique(BOOL p_bUnique)
    {
    Message msg;
    BSString string = SZ_MODIFIER_UNIQUE;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bUnique)
        {
        if (_bAddRight)
            {
            usRc |= ERR_CONFIGSYS_ADDRIGHT_SETTED;
            string += SZ_MODIFIER_ADDRIGHT;
            string += "|";
            }
        if (_bAddLeft)
            {
            usRc |= ERR_CONFIGSYS_ADDLEFT_SETTED;
            string += SZ_MODIFIER_ADDLEFT;
            string += "|";
            }
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bUnique = p_bUnique;

    return usRc;
    }


/*
 *@@SetAddRight:
 *  This function sets addright state. Returns error code if error occurred.
 *
 */
USHORT ScriptConfigSys::SetAddRight(BOOL p_bAddRight)
    {
    Message msg;
    BSString string = SZ_MODIFIER_ADDRIGHT;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bAddRight)
        {
        if (_bUnique)
            {
            usRc |= ERR_CONFIGSYS_UNIQUE_SETTED;
            string += SZ_MODIFIER_UNIQUE;
            string += "|";
            }
        if (_bAddLeft)
            {
            usRc |= ERR_CONFIGSYS_ADDLEFT_SETTED;
            string += SZ_MODIFIER_ADDLEFT;
            string += "|";
            }
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bAddRight = p_bAddRight;

    return usRc;
    }


/*
 *@@SetAddLeft:
 *  This function sets addleft state. Returns error code if error occurred.
 */
USHORT ScriptConfigSys::SetAddLeft(BOOL p_bAddLeft)
    {
    Message msg;
    BSString string = SZ_MODIFIER_ADDLEFT;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bAddLeft)
        {
        if (_bUnique)
            {
            usRc |= ERR_CONFIGSYS_UNIQUE_SETTED;
            string += SZ_MODIFIER_UNIQUE;
            string += "|";
            }
        if (_bAddRight)
            {
            usRc |= ERR_CONFIGSYS_ADDRIGHT_SETTED;
            string += SZ_MODIFIER_ADDRIGHT;
            string += "|";
            }
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bAddLeft = p_bAddLeft;

    return usRc;
    }

/*
 *@@SetAddAfter:
 *  This function sets addafter state. Returns error code if error occurred.
 */
USHORT ScriptConfigSys::SetAddAfter(BOOL p_bAddAfter)
    {
    Message msg;
    BSString string = SZ_MODIFIER_ADDAFTER;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bAddAfter)
        {
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bAddAfter = p_bAddAfter;

    return usRc;
    }


/*
 *@@SetAddTop:
 *  This function sets addtop state. Returns error code if error occurred.
 *
 */
USHORT ScriptConfigSys::SetAddTop(BOOL p_bAddTop)
    {
    Message msg;
    BSString string = SZ_MODIFIER_ADDTOP;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bAddTop)
        {
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bAddTop = p_bAddTop;

    return usRc;
    }


/*
 *@@SetAddBefore:
 *  This function sets addbefore state. Returns error code if error occurred.
 */
USHORT ScriptConfigSys::SetAddBefore(BOOL p_bAddBefore)
    {
    Message msg;
    BSString string = SZ_MODIFIER_ADDBEFORE;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bAddBefore)
        {
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bAddBefore = p_bAddBefore;

    return usRc;
    }


/*
 *@@SetRemoveLine:
 *  This function sets removeline state. Returns error code if error occurred.
 */
USHORT ScriptConfigSys::SetRemoveLine(BOOL p_bRemoveLine)
    {
    Message msg;
    BSString string = SZ_MODIFIER_REMOVELINE;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bRemoveLine)
        {
        if (_bRemovePart)
            {
            usRc |= ERR_CONFIGSYS_REMOVEPART_SETTED;
            string += SZ_MODIFIER_REMOVEPART;
            string += "|";
            }
        if (_bUnique)
            {
            usRc |= ERR_CONFIGSYS_UNIQUE_SETTED;
            string += SZ_MODIFIER_UNIQUE;
            string += "|";
            }
        if (_bAddRight)
            {
            usRc |= ERR_CONFIGSYS_ADDRIGHT_SETTED;
            string += SZ_MODIFIER_ADDRIGHT;
            string += "|";
            }
        if (_bAddLeft)
            {
            usRc |= ERR_CONFIGSYS_ADDLEFT_SETTED;
            string += SZ_MODIFIER_ADDLEFT;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bRemoveLine = p_bRemoveLine;

    return usRc;
    }


/*
 *@@ScriptConfigSys::SetRemovePart
 *  This function sets removepart state. Returns error code if error occurred.
 */
USHORT ScriptConfigSys::SetRemovePart(BOOL p_bRemovePart)
    {
    Message msg;
    BSString string = SZ_MODIFIER_REMOVEPART;
    string += "|";
    USHORT usRc = NO_ERROR;
    if (p_bRemovePart)
        {
        if (_bRemoveLine)
            {
            usRc |= ERR_CONFIGSYS_REMOVELINE_SETTED;
            string += SZ_MODIFIER_REMOVELINE;
            string += "|";
            }
        if (_bUnique)
            {
            usRc |= ERR_CONFIGSYS_UNIQUE_SETTED;
            string += SZ_MODIFIER_UNIQUE;
            string += "|";
            }
        if (_bAddRight)
            {
            usRc |= ERR_CONFIGSYS_ADDRIGHT_SETTED;
            string += SZ_MODIFIER_ADDRIGHT;
            string += "|";
            }
        if (_bAddLeft)
            {
            usRc |= ERR_CONFIGSYS_ADDLEFT_SETTED;
            string += SZ_MODIFIER_ADDLEFT;
            string += "|";
            }
        }
    if (usRc)
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
        msg.SetMessageText(string);
        _messageLogger.LogMessage(msg);
        return usRc;
        }

    this->_bRemovePart = p_bRemovePart;

    return usRc;
    }


/*
 *@@SetSearchString:
 *  This function sets search string.
 */
USHORT ScriptConfigSys::SetSearchString(const PSZ p_pszSearchString)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszSearchString) < MAXSEARCHSTRINGLENGTH)
        strcpy(this->_szSearchString, p_pszSearchString);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REQUIRES_PACKAGE_STRTOOLONG);
        msg.SetMessageText(p_pszSearchString);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetConfigSys:
 *  This function sets statement string.
 */
USHORT ScriptConfigSys::SetConfigSys(CONFIGSYSDATA configSys)
    {
    USHORT rc = NO_ERROR;

    if (SetStatement(configSys.szStatement) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetSearchString(configSys.szSearchString) != NO_ERROR)
        rc = STRING_TOO_LONG;
    SetUnique(FALSE);
    SetAddRight(FALSE);
    SetAddLeft(FALSE);
    SetAddTop(FALSE);
    SetAddAfter(FALSE);
    SetAddBefore(FALSE);
    SetRemoveLine(FALSE);
    SetRemovePart(FALSE);

    if (configSys.usModifiers & MODIFIER_UNIQUE)
        SetUnique(TRUE);
    if (configSys.usModifiers & MODIFIER_ADDRIGHT)
        SetAddRight(TRUE);
    if (configSys.usModifiers & MODIFIER_ADDLEFT)
        SetAddLeft(TRUE);
    if (configSys.usModifiers & MODIFIER_ADDTOP)
        SetAddTop(TRUE);
    if (configSys.usModifiers & MODIFIER_ADDAFTER)
        SetAddAfter(TRUE);
    if (configSys.usModifiers & MODIFIER_ADDBEFORE)
        SetAddBefore(TRUE);
    if (configSys.usModifiers & MODIFIER_REMOVELINE)
        SetRemoveLine(TRUE);
    if (configSys.usModifiers & MODIFIER_REMOVEPART)
        SetRemovePart(TRUE);
    return rc;
    }


/*
 *@@ReturnStatement:
 *  Returns const pointer to statement string.
 */
const PSZ ScriptConfigSys::ReturnStatement()
    {
    return _szStatement;
    }


/*
 *@@ReturnUnique:
 *  Returns the state of the unique modifier.
 */
BOOL ScriptConfigSys::ReturnUnique()
    {
    return _bUnique;
    }


/*
 *@@ReturnAddRight:
 *  Returns the state of the addright modifier.
 */
BOOL ScriptConfigSys::ReturnAddRight()
    {
    return _bAddRight;
    }


/*
 *@@ReturnAddLeft:
 *  Returns the state of the addleft modifier.
 */
BOOL ScriptConfigSys::ReturnAddLeft()
    {
    return _bAddLeft;
    }



/*
 *@@ReturnAddAfter:
 *  Returns the state of the addafter modifier.
 */
BOOL ScriptConfigSys::ReturnAddAfter()
    {
    return _bAddAfter;
    }


/*
 *@@ReturnAddTop:
 *  Returns the state of the addtop modifier.
 */
BOOL ScriptConfigSys::ReturnAddTop()
    {
    return _bAddTop;
    }


/*
 *@@ReturnAddBefore:
 *  Returns the state of the addbefore modifier.
 */
BOOL ScriptConfigSys::ReturnAddBefore()
    {
    return _bAddBefore;
    }




/*
 *@@ReturnRemoveLine:
 *  Returns the state of the removeline modifier.
 */
BOOL ScriptConfigSys::ReturnRemoveLine()
    {
    return _bRemoveLine;
    }




/*
 *@@ReturnRemovePart:
 *  Returns the state of the removepart modifier.
 */
BOOL ScriptConfigSys::ReturnRemovePart()
    {
    return _bRemovePart;
    }



/*
 *@@ReturnSearchString:
 *  Returns const pointer to search string string.
 */
const PSZ ScriptConfigSys::ReturnSearchString()
    {
    return _szSearchString;
    }




/*
 *@@ReturnConfigSys:
 *  This function returns statement string.
 */
CONFIGSYSDATA ScriptConfigSys::ReturnConfigSys()
    {
    CONFIGSYSDATA configSys;
    strcpy(configSys.szStatement, this->_szStatement);
    strcpy(configSys.szSearchString, this->_szSearchString);

    if (_bUnique)
        configSys.usModifiers |= MODIFIER_UNIQUE;
    if (_bAddRight)
        configSys.usModifiers |= MODIFIER_ADDRIGHT;
    if (_bAddLeft)
        configSys.usModifiers |= MODIFIER_ADDLEFT;
    if (_bAddAfter)
        configSys.usModifiers |= MODIFIER_ADDAFTER;
    if (_bAddBefore)
        configSys.usModifiers |= MODIFIER_ADDBEFORE;
    if (_bAddTop)
        configSys.usModifiers |= MODIFIER_ADDTOP;
    if (_bRemoveLine)
        configSys.usModifiers |= MODIFIER_REMOVELINE;
    if (_bRemovePart)
        configSys.usModifiers |= MODIFIER_REMOVEPART;

    return configSys;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptConfigSys::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    XSTRING xstrScriptText;
    PSZ pszFirstModifier = NULL;
    PSZ pszSecondModifier = NULL;
    USHORT ulSpaces = 0;
    CHAR szTemp[1024];
    xstrInit(&xstrScriptText, 0);

    ulTextLength = strlen(_szStatement);

    if (_bUnique)
        {
        ulTextLength += strlen(SZ_MODIFIER_UNIQUE);
        ulSpaces += 3;
        pszFirstModifier = SZ_MODIFIER_UNIQUE;
        }
    if (_bAddRight)
        {
        ulTextLength += strlen(SZ_MODIFIER_ADDRIGHT);
        ulSpaces += 3;
        pszFirstModifier = SZ_MODIFIER_ADDRIGHT;
        }
    if (_bAddLeft)
        {
        ulTextLength += strlen(SZ_MODIFIER_ADDLEFT);
        ulSpaces += 3;
        pszFirstModifier = SZ_MODIFIER_ADDLEFT;
        }
    if (_bAddTop)
        {
        ulTextLength += strlen(SZ_MODIFIER_ADDTOP);
        ulSpaces ++;
        if (pszFirstModifier)
            pszSecondModifier = SZ_MODIFIER_ADDTOP;
        else
            pszFirstModifier = SZ_MODIFIER_ADDTOP;
        }
    if (_bAddAfter)
        {
        ulTextLength += strlen(SZ_MODIFIER_ADDAFTER);
        ulSpaces += 3;
        ulTextLength += strlen(_szSearchString);
        if (pszFirstModifier)
            pszSecondModifier = SZ_MODIFIER_ADDAFTER;
        else
            pszFirstModifier = SZ_MODIFIER_ADDAFTER;
        }
    if (_bAddBefore)
        {
        ulTextLength += strlen(SZ_MODIFIER_ADDBEFORE);
        ulSpaces += 3;
        ulTextLength = strlen(_szSearchString);
        if (pszFirstModifier)
            pszSecondModifier = SZ_MODIFIER_ADDBEFORE;
        else
            pszFirstModifier = SZ_MODIFIER_ADDBEFORE;
        }
    if (_bRemoveLine)
        {
        ulTextLength += strlen(SZ_MODIFIER_REMOVELINE);
        ulSpaces += 3;
        pszFirstModifier = SZ_MODIFIER_REMOVELINE;
        }
    if (_bRemovePart)
        {
        ulTextLength += strlen(SZ_MODIFIER_REMOVEPART);
        ulSpaces += 3;
        pszFirstModifier = SZ_MODIFIER_REMOVEPART;
        }

    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_CONFIGSYS) + 3;

    szTemp[0] = '\0';

    if (p_bAttribute)
        sprintf(szTemp, "%s=\"%s", SZ_ATTRIBUTE_CONFIGSYS, _szStatement);
    else
        sprintf(szTemp, "%s", _szStatement);
    xstrcat(&xstrScriptText, szTemp, 0);

    if (pszFirstModifier)
        {
        xstrcat(&xstrScriptText, " | ", 0);
        xstrcat(&xstrScriptText, pszFirstModifier, 0);
        }
    if (pszSecondModifier)
        {
        xstrcat(&xstrScriptText, " ", 0);
        xstrcat(&xstrScriptText, pszSecondModifier, 0);
        }
    if (strlen(_szSearchString) != 0)
        {
        xstrcat(&xstrScriptText, "(", 0);
        xstrcat(&xstrScriptText, _szSearchString, 0);
        xstrcat(&xstrScriptText, ")", 0);
        }
    if(p_bAttribute)
        xstrcat(&xstrScriptText, "\"", 0);

    return xstrScriptText.psz;
    }



/*
 *@@ScriptConfigSys:: BuildFromText
 *  Builds ScriptConfigSys object accordingly to script text.
 */
PSZ ScriptConfigSys::BuildFromText(PSZ p_pszScriptText,
                                  ULONG p_ulLineNumber)
    {
    Message msg;
    BSString string;
    PSZ pszTemp = NULL, pszTemp2 = NULL, pszModifiers = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    BOOL bReplaceFound = FALSE;
    BOOL bVertical = FALSE;
    BOOL bVerticalAllowed = FALSE;
    ULONG ulAttrLen = 0;

    // Initialize class attributes.
    *_szStatement = '\0';
    *_szSearchString = '\0';
    _bUnique = FALSE;
    _bAddRight = FALSE;
    _bAddLeft = FALSE;
    _bAddRight = FALSE;
    _bAddTop = FALSE;
    _bAddAfter = FALSE;
    _bAddBefore = FALSE;
    _bRemoveLine = FALSE;
    _bRemovePart = FALSE;


    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_CONFIGSYS, &ulOffset);

    if (!pszAttr)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    pszTemp = strchr(pszAttr, '|');
    if (!pszTemp) // There are no modifiers.
        SetStatement(pszAttr);
    else // There are modifiers.
        {
        pszModifiers = pszTemp + 1;
        // Remove spaces before '|'.
        pszTemp --;
        while ((*pszTemp == ' ') && (pszTemp > pszAttr))
            pszTemp --;

        // Get statement text.
        pszTemp2 = strhSubstr(pszAttr, pszTemp + 1);
        SetStatement(pszTemp2);
        delete [] pszTemp2;

        // Find UNIQUE modifier.
        pszTemp2 = strhistr(pszModifiers, SZ_MODIFIER_UNIQUE);
        if (pszTemp2) // There is UNIQUE modifier.
            {
            bReplaceFound = TRUE;
            bVerticalAllowed = TRUE;
            SetUnique(TRUE);
            pszTemp = strhExtract(pszModifiers, '(', ')', NULL);
            SetSearchString(pszTemp);
            delete [] pszTemp;
            }
        // Find ADDRIGHT modifier.
        if (strhistr(pszModifiers, SZ_MODIFIER_ADDRIGHT))
            { // There is ADDRIGHT modifier.
            if (!bReplaceFound) // Check that there is not
                {              // any replacement modifier.
                SetAddRight(TRUE);
                bVerticalAllowed = TRUE;
                bReplaceFound = TRUE;
                }
            else
                {
                string = SZ_MODIFIER_ADDRIGHT;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        // Find ADDLEFT modifier.
        if (strhistr(pszModifiers, SZ_MODIFIER_ADDLEFT))
            { // There is ADDLEFT modifier.
            if (!bReplaceFound) // Check that there is not
                {              // any replacement modifier.
                SetAddLeft(TRUE);
                bVerticalAllowed = TRUE;
                bReplaceFound = TRUE;
                }
            else
                {
                string = SZ_MODIFIER_ADDLEFT;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        // Find REMOVELINE modifier.
        if (strhistr(pszModifiers, SZ_MODIFIER_REMOVELINE))
            { // There is REMOVELINE modifier.
            if (!bReplaceFound) // Check that there is not
                {              // any replacement modifier.
                SetRemoveLine(TRUE);
                bVerticalAllowed = FALSE;
                bReplaceFound = TRUE;
                }
            else
                {
                string = SZ_MODIFIER_REMOVELINE;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        // Find REMOVEPART modifier.
        if (strhistr(pszModifiers, SZ_MODIFIER_REMOVEPART))
            { // There is REMOVEPART modifier.
            if (!bReplaceFound) // Check that there is not
                {              // any replacement modifier.
                SetRemovePart(TRUE);
                bVerticalAllowed = FALSE;
                bReplaceFound = TRUE;
                }
            else
                {
                string = SZ_MODIFIER_REMOVEPART;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        // Find ADDTOP modifier.
        if (strhistr(pszModifiers, SZ_MODIFIER_ADDTOP))
            { // There is ADDTOP modifier.
            if (bVerticalAllowed)
                {
                SetAddTop(TRUE);
                bVertical = TRUE;
                }
            else
                {
                string = SZ_MODIFIER_ADDTOP;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
           }
        // Find ADDAFTER modifier.
        pszTemp2 = strhistr(pszModifiers, SZ_MODIFIER_ADDAFTER);
        if (pszTemp2)
            { // There is ADDAFTER modifier.
            if (!bVertical && bVerticalAllowed)
                {
                SetAddAfter(TRUE);
                bVertical = TRUE;
                pszTemp = strhExtract(pszTemp2, '(', ')', NULL);
                SetSearchString(pszTemp);
                delete [] pszTemp;
                }
            else
                {
                string = SZ_MODIFIER_ADDAFTER;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        // Find ADDBEFORE modifier.
        pszTemp2 = strhistr(pszModifiers, SZ_MODIFIER_ADDBEFORE);
        if (pszTemp2)
            { // There is ADDBEFORE modifier.
            if (!bVertical && bVerticalAllowed)
                {
                SetAddBefore(TRUE);
                bVertical = TRUE;
                pszTemp = strhExtract(pszTemp2, '(', ')', NULL);
                SetSearchString(pszTemp);
                delete [] pszTemp;
                }
            else
                {
                string = SZ_MODIFIER_ADDBEFORE;
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_CFGSYS_MODIFIER_WRONGCOMB);
                msg.SetMessageText(string);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptRegisterClass class:

    <B>Description:</B>
        This class is for REGISTERCLASS attributes in PCK tag. Detailed
        information about the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScriptRegisterClass             Constructor.
        ~ScriptRegisterClass            Destructor.
        SetClassName                    Sets class name text.
        SetDLLPath                      Sets DLL path name text.
        ReturnClassName                 Returns const pointer to class name string.
        ReturnDLLPath                   Returns const pointer to DLL name string.
        MakeScriptText                  Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptRegisterClass:
 *  Constructor for ScriptRegisterClass class. Initializes object by setting
 *  strings to zero and sets index number too.
 */
ScriptRegisterClass::ScriptRegisterClass(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);

    *_szClassName = '\0';
    *_szDLLPath = '\0';
    }


/*
 *@@~ScriptRegisterClass:
 *  Destructor for ScriptRegisterClass class.
 */
ScriptRegisterClass::~ScriptRegisterClass()
    {
    *_szClassName = '\0';
    *_szDLLPath = '\0';
    }


/*
 *@@SetClassName:
 *  Sets class name string text.
 */
USHORT ScriptRegisterClass::SetClassName(const PSZ p_pszClassName)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszClassName) < MAXCLASSNAMELENGTH)
        strcpy(this->_szClassName, p_pszClassName);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_CLASSNAME_STRTOOLONG);
        msg.SetMessageText(p_pszClassName);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@SetDLLPath:
 *  Sets DLL path string text.
 */
USHORT ScriptRegisterClass::SetDLLPath(const PSZ p_pszDLLPath)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszDLLPath) < CCHMAXPATH)
        strcpy(this->_szDLLPath, p_pszDLLPath);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_DLLPATH_STRTOOLONG);
        msg.SetMessageText(p_pszDLLPath);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@ ScriptRegisterClass::SetRegisterClass
 */
USHORT ScriptRegisterClass::SetRegisterClass(REGISTERCLASSDATA p_registerClass)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (SetClassName(p_registerClass.szClassname) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetDLLPath(p_registerClass.szDLLPath) != NO_ERROR)
        rc = STRING_TOO_LONG;

    return STRING_TOO_LONG;
    }


/*
 *@@ReturnClassName:
 *  Returns const pointer to class name string.
 */
const PSZ ScriptRegisterClass::ReturnClassName()
    {
    return _szClassName;
    }


/*
 *@@ReturnDLLPath:
 *  Returns const pointer to DLL path string.
 */
const PSZ ScriptRegisterClass::ReturnDLLPath()
    {
    return _szDLLPath;
    }


/*
 *@@ReturnRegisterClass:
 */
REGISTERCLASSDATA ScriptRegisterClass::ReturnRegisterClass()
    {
    REGISTERCLASSDATA registerClass;
    strcpy(registerClass.szClassname, this->_szClassName);
    strcpy(registerClass.szDLLPath, this->_szDLLPath);
    return registerClass;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptRegisterClass::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    ulTextLength = strlen(_szClassName) + strlen(_szDLLPath);
    ulTextLength += 3;

    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_REGISTERCLASS) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s|%s\"", SZ_ATTRIBUTE_REGISTERCLASS,
            _szClassName, _szDLLPath);
    else
        sprintf(pszScriptText, "%s|%s", _szClassName, _szDLLPath);

    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds ScriptRegisterClass object accordingly to script text.
 *  Pointer to after end of tag is returned so multiple searches can be done.
 */
PSZ ScriptRegisterClass::BuildFromText(PSZ p_pszScriptText,
                                       ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszAttr = NULL;
    PSZ pszTemp = NULL;
    PSZ pszTemp2 = NULL;
    PSZ pszTemp3 = NULL;
    ULONG ulOffset = 0;
    ULONG ulAttrLen = 0;

    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_REGISTERCLASS,
                             &ulOffset);

    if (!pszAttr)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    pszTemp = strchr(pszAttr, '|');
    if (pszTemp)
        {
        pszTemp2 = pszTemp - 1;
        // Remove leading spaces.
        while ((*pszTemp2 == ' ') && (pszTemp2 > pszAttr))
            pszTemp2 --;
        if (pszTemp3 = strhSubstr(pszAttr, pszTemp2 + 1))
            {
            SetClassName(pszTemp3);
            delete [] pszTemp3;
            }
        else
            {
            msg.SetType(MSG_TYPE_INFO);
            msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_CLASSNAME_NOTFOUND);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            }

        pszTemp2 = pszTemp + 1;
        // Remove trailing spaces.
        while ((*pszTemp2 == ' ') && (*pszTemp2))
            pszTemp2 ++;
        if (strlen(pszTemp2) > 1)
            SetDLLPath(pszTemp2);
        else
            {
            msg.SetType(MSG_TYPE_INFO);
            msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_DLLPATH_NOTFOUND);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        }
    else // Seems that DLL path is missing but get class name.
        {
        while (*pszTemp =! '\0')
            pszTemp ++;
        // Remove leading spaces.
        while ((*pszTemp == ' ') && (pszTemp > pszAttr))
            pszTemp --;
        if (pszTemp3 = strhSubstr(pszAttr, pszTemp))
            {
            SetClassName(pszTemp3);
            delete [] pszTemp3;
            }
        else
            {
            msg.SetType(MSG_TYPE_INFO);
            msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_CLASSNAME_NOTFOUND);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptReplaceClass class:

    <B>Description:</B>
        This class is for REPLACECLASS attribute in CONFIGSYS tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScriptReplaceClass Constructor.
        ~ScriptReplaceClass Destructor.
        SetClassName    Sets class name text.
        SetNewClassName Sets new class name text.
        ReturnClassName Returns const pointer to class name string.
        ReturnNewClassName Returns const pointer to new class name string.
        MakeScriptText  Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptReplaceClass:
 *  Constructor for ScriptReplaceClass class.
 */
ScriptReplaceClass::ScriptReplaceClass(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    *_szClassName = '\0';
    *_szNewClassName = '\0';
    }


/*
 *@@~ScriptReplaceClass:
 *  Destructor for ScriptRegisterClass class.
 */
ScriptReplaceClass::~ScriptReplaceClass()
    {
    *_szClassName = '\0';
    *_szNewClassName = '\0';
    }


/*
 *@@SetClassName:
 *  Sets class name string text.
 */
USHORT ScriptReplaceClass::SetClassName(const PSZ p_pszClassName)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszClassName) < MAXCLASSNAMELENGTH)
        strcpy(this->_szClassName, p_pszClassName);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REPLACE_CLASSNAME_STRTOOLONG);
        msg.SetMessageText(p_pszClassName);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetNewClassName:
 *  Sets new class name string text.
 */
USHORT ScriptReplaceClass::SetNewClassName(const PSZ p_pszNewClassName)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszNewClassName) < MAXCLASSNAMELENGTH)
        strcpy(this->_szNewClassName, p_pszNewClassName);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REPLACE_NEWNAME_STRTOOLONG);
        msg.SetMessageText(p_pszNewClassName);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetReplaceClass:
 */
USHORT ScriptReplaceClass::SetReplaceClass(REPLACECLASSDATA p_replaceClass)
    {
    USHORT rc = NO_ERROR;
    if (SetClassName(p_replaceClass.szOldClassName) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetNewClassName(p_replaceClass.szNewClassName) != NO_ERROR)
        rc = STRING_TOO_LONG;
    return rc;
    }


/*
 *@@ReturnClassName:
 *  Returns const pointer to class name string.
 */
const PSZ ScriptReplaceClass::ReturnClassName()
    {
    return _szClassName;
    }


/*
 *@@ ScriptReplaceClass::ReturnNewClassName
 *  Returns const pointer to new class name string.
 */
const PSZ ScriptReplaceClass::ReturnNewClassName()
    {
    return _szNewClassName;
    }



/*
 *@@ReturnReplaceClass:
 */
REPLACECLASSDATA ScriptReplaceClass::ReturnReplaceClass()
    {
    REPLACECLASSDATA replaceClass;
    strcpy(replaceClass.szOldClassName, this->_szClassName);
    strcpy(replaceClass.szNewClassName, this->_szNewClassName);
    return replaceClass;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptReplaceClass::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    ulTextLength = strlen(_szClassName) + strlen(_szNewClassName);
    ulTextLength += 3;

    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_REPLACECLASS) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s|%s\"", SZ_ATTRIBUTE_REPLACECLASS,
            _szClassName, _szNewClassName);
    else
        sprintf(pszScriptText, "%s|%s", _szClassName, _szNewClassName);

    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds ScriptReplaceClass object accordingly to script text.
 *  If any errors is found in script text, error informations are returned
 *  in table where to *pScriptError points. Pointer to after end of tag is
 *  returned so multiple searches can be done.
 */
PSZ ScriptReplaceClass::BuildFromText(PSZ p_pszScriptText,
                                      ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszAttr = NULL;
    PSZ pszTemp = NULL;
    PSZ pszTemp2 = NULL;
    PSZ pszTemp3 = NULL;
    ULONG ulOffset = 0;
    ULONG ulAttrLen = 0;

    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_REPLACECLASS,
                             &ulOffset);

    if (!pszAttr)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_ATTRIBUTE_REPLACE_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    pszTemp = strchr(pszAttr, '|');
    if (pszTemp)
        {
        pszTemp2 = pszTemp - 1;
        // Remove leading spaces.
        while ((*pszTemp2 == ' ') && (pszTemp2 > pszAttr))
            pszTemp2 --;
        pszTemp3 = strhSubstr(pszAttr, pszTemp2 + 1);
        SetClassName(pszTemp3);
        delete [] pszTemp3;

        pszTemp2 = pszTemp + 1;
        // Remove trailing spaces.
        while ((*pszTemp2 == ' ') && (*pszTemp2))
            pszTemp2 ++;
        if (pszTemp2)
            SetNewClassName(pszTemp2);
        else
            {
            msg.SetType(MSG_TYPE_ERROR);
            msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_CLASSNAME_MISSING);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        }
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_REGISTER_CLASSNAME_MISSING);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptCreateObject class:

    <B>Description:</B>
        This class is for CREATEOBJECT attribute in CONFIGSYS tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScriptCreateObject  Constructor.
        ~ScriptCreateObject Destructor.
        SetClassName        Sets class name text.
        SetTitle            Sets title text.
        SetLocation         Sets location text.
        SetConfig           Sets config text.
        SetReplace          Sets replace parameter state.
        ReturnClassName     Returns const pointer to class name string.
        ReturnTitle         Returns const pointer to title string.
        ReturnLocation      Returns const pointer to location string.
        ReturnConfig        Returns const pointer to config string.
        ReturnReplace       Returns the state of the replace parameter.
        MakeScriptText      Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptCreateObject:
 *  Constructor for ScriptCreateObject class. Initializes object by setting
 *  strings to zero and sets index number.
 */
ScriptCreateObject::ScriptCreateObject(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);

    *_szClassName = '\0';
    *_szTitle = '\0';
    *_szLocation = '\0';
    *_szConfig = '\0';
    _bReplace = FALSE;
    }


/*
 *@@~ScriptCreateObject:
 *  Destructor for ScriptCreateObject class.
 */
ScriptCreateObject::~ScriptCreateObject()
    {
    *_szClassName = '\0';
    *_szTitle = '\0';
    *_szLocation = '\0';
    *_szConfig = '\0';
    _bReplace = FALSE;
    }


/*
 *@@SetClassName:
 *  Sets class name string text.
 */
USHORT ScriptCreateObject::SetClassName(const PSZ p_pszClassName)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszClassName) < MAXCLASSNAMELENGTH)
        strcpy(this->_szClassName, p_pszClassName);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_CLASSNAME_STRTOOLONG);
        msg.SetMessageText(p_pszClassName);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }




/*
 *@@SetTitle:
 *  Sets new title string text.
 */
USHORT ScriptCreateObject::SetTitle(const PSZ p_pszTitle)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszTitle) < MAXCLASSTITLELENGTH)
        strcpy(this->_szTitle, p_pszTitle);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_TITLE_STRTOOLONG);
        msg.SetMessageText(p_pszTitle);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetLocation:
 *  Sets new title string text.
 */
USHORT ScriptCreateObject::SetLocation(const PSZ p_pszLocation)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszLocation) < CCHMAXPATH)
        strcpy(this->_szLocation, p_pszLocation);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_LOCATION_STRTOOLONG);
        msg.SetMessageText(p_pszLocation);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetConfig:
 *  Sets new config string text.
 */
USHORT ScriptCreateObject::SetConfig(const PSZ p_pszConfig)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszConfig) < MAXCLASSCONFIGLENGTH)
        strcpy(this->_szConfig, p_pszConfig);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_CONFIG_STRTOOLONG);
        msg.SetMessageText(p_pszConfig);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetReplace:
 *  Sets new config string text.
 */
USHORT ScriptCreateObject::SetReplace(BOOL p_bReplace)
    {
    this->_bReplace = p_bReplace;
    return (0);     // V0.9.14 (2001-07-07) [umoeller]
    }


/*
 *@@SetCreateObject:
 */
USHORT ScriptCreateObject::SetCreateObject(CREATEOBJECTDATA p_createObject)
    {
    USHORT rc = NO_ERROR;
    if (SetClassName(p_createObject.szClassName) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetTitle(p_createObject.szTitle) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetLocation(p_createObject.szLocation) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetConfig(p_createObject.szConfig) != NO_ERROR)
        rc = STRING_TOO_LONG;

    return rc;
    }


/*
 *@@ReturnClassName:
 *  Returns const pointer to class name string.
 */
const PSZ ScriptCreateObject::ReturnClassName()
    {
    return _szClassName;
    }


/*
 *@@ReturnTitle:
 *  Returns const pointer to title string.
 */
const PSZ ScriptCreateObject::ReturnTitle()
    {
    return _szTitle;
    }


/*
 *@@ReturnLocation:
 *  Returns const pointer to location string.
 */
const PSZ ScriptCreateObject::ReturnLocation()
    {
    return _szLocation;
    }


/*
 *@@ReturnConfig:
 *  Returns const pointer to config string.
 */
const PSZ ScriptCreateObject::ReturnConfig()
    {
    return _szConfig;
    }


/*
 *@@ReturnReplace:
 *  Returns the state of the replace parameter.
 */
BOOL ScriptCreateObject::ReturnReplace()
    {
    return _bReplace;
    }


/*
 *@@ReturnCreateObject:
 */
CREATEOBJECTDATA ScriptCreateObject::ReturnCreateObject()
    {
    CREATEOBJECTDATA createObject;
    strcpy(createObject.szClassName, this->_szClassName);
    strcpy(createObject.szTitle, this->_szTitle);
    strcpy(createObject.szLocation, this->_szLocation);
    strcpy(createObject.szConfig, this->_szConfig);
    createObject.bReplace = this->_bReplace;
    return createObject;
    }


/*
 *@@ ScriptCreateObject::MakeScriptText
 *  Returns pointer to script text.
 *
 *@@changed v0.9.5 (2000-08-02) [tahola]
 */
PSZ ScriptCreateObject::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    PSZ pszReplace = "";

    ulTextLength = strlen(_szClassName) + strlen(_szTitle) + strlen(_szLocation);
    if ((ulTextLength += strlen(_szConfig)) > 0)
        ulTextLength += 1; // No spaces allowed.
    ulTextLength += 2;     // v0.9.5 (2000-08-02) [tahola]
    if (_bReplace)
        {
        ulTextLength += strlen(SZ_PARAMETER_REPLACE) + 2;
        pszReplace = new CHAR[strlen(SZ_PARAMETER_REPLACE) + 3];
        strcpy(pszReplace, SZ_PARAMETER_REPLACE);
        strcat(pszReplace, " ");
        }
    ulTextLength += 3;

    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_CREATEOBJECT) + 3;

    pszScriptText = new CHAR[ulTextLength + 2];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s%s|%s|%s",  // No spaces allowed.
            SZ_ATTRIBUTE_CREATEOBJECT, pszReplace, // v0.9.5 (2000-08-2) [tahola]
            _szClassName, _szTitle, _szLocation);
    else
        sprintf(pszScriptText, "%s%s|%s|%s", pszReplace, _szClassName,
            _szTitle, _szLocation); // No spaces allowed.
                                  // v0.9.5 (2000-08-2) [tahola]
    if (strlen(_szConfig) != 0)
        {
        strcat(pszScriptText, "|");      // No spaces allowed.
        strcat(pszScriptText, _szConfig); // v0.9.5 (2000-08-2) [tahola]
        }

    if (p_bAttribute)
        strcat(pszScriptText, "\"");

    if (_bReplace)
        delete [] pszReplace;

    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds ScriptCreateObject accordingly to script text.
 */
PSZ ScriptCreateObject::BuildFromText(PSZ p_pszScriptText,
                                      ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL, pszTemp2 = NULL, pszTemp3 = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    ULONG ulVersion = 0;
    ULONG ulAttrLen = 0;

    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_CREATEOBJECT,
                             &ulOffset);

    if (!pszAttr)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    pszTemp = strhFindWord(pszAttr, SZ_PARAMETER_REPLACE, "\x0d\x0a ()/\\-,.",
                          "\x0d\x0a ()/\\-,.:;");
    if (pszTemp)
        SetReplace(TRUE);

    pszTemp2 = pszAttr;
    pszTemp = strchr(pszAttr, '|');
    if (pszTemp) // Set class name.
        {
        pszTemp3 = strhSubstr(pszTemp2, pszTemp);
        SetClassName(pszTemp3);
        delete [] pszTemp3;
        pszTemp2 = ++pszTemp;
        pszTemp = strchr(pszTemp, '|');
        if (pszTemp) // Set title.
            {
            pszTemp3 = strhSubstr(pszTemp2, pszTemp);
            SetTitle(pszTemp3);
            delete [] pszTemp3;
            pszTemp2 = ++pszTemp;
            pszTemp = strchr(pszTemp, '|');
            if (pszTemp) // Set location.
                {
                pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                SetLocation(pszTemp3);
                delete [] pszTemp3;
                pszTemp2 = ++pszTemp;
                pszTemp = strchr(pszTemp, '|');
                if (pszTemp) // Set config.
                    {
                    pszTemp3 = strhSubstr(pszTemp2, pszTemp);
                    SetConfig(pszTemp3);
                    delete [] pszTemp3;
                    }
                }
            else
                {
                if (pszTemp2)
                    SetLocation(pszTemp2);
                else
                    {
                    msg.SetType(MSG_TYPE_ERROR);
                    msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_LOCATION_MISSING);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                }
            }
        else // Title error.
            {
            msg.SetType(MSG_TYPE_ERROR);
            msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_TITLE_MISSING);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        }
    else // Class name error.
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CREATE_CLASSNAME_MISSING);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptExecute class:

    <B>Description:</B>
        This class is for EXECUTE attribute in CONFIGSYS tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.


    Member functions:
        ScriptExecute           Constructor.
        ~ScriptExecute          Destructor.
        SetExecFileParams       Sets class name text.
        ReturnExecFileParams    Returns const pointer to exec file params string.
        MakeScriptText          Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptExecute:
 *  Constructor for ScriptExecute class. Initializes object by setting
 *  strings to zero and sets index number too.
 */
ScriptExecute::ScriptExecute(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    *_szExecFileParams = '\0';
    _usContext = 0;
    }


/*
 *@@~ScriptExecute:
 *  Destructor for ScriptExecute class.
 */
ScriptExecute::~ScriptExecute()
    {
    }


/*
 *@@SetClassName:
 *  Sets exec file params string text.
 */
USHORT ScriptExecute::SetExecFileParams(const PSZ p_pszExecFileParams)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszExecFileParams) < CCHMAXPATH)
        strcpy(this->_szExecFileParams, p_pszExecFileParams);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_EXECUTE_EXEFILEPARAM_STRTOOLONG);
        msg.SetMessageText(p_pszExecFileParams);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@ ScriptExecute::SetContext
 *  Sets object's index.
 */
USHORT ScriptExecute::SetContext(USHORT p_usContext)
    {
    if ((p_usContext == FLAG_CONFIGSYS) || (p_usContext == FLAG_REGISTERCLASS) ||
        (p_usContext == FLAG_CREATEOBJECT) || (p_usContext == FLAG_NOCONTEXT))
        this->_usContext = p_usContext;
    return NO_ERROR;
    }


/*
 *@@SetExecute:
 */
USHORT ScriptExecute::SetExecute(EXECUTEDATA p_execData)
    {
    USHORT rc = NO_ERROR;
    if (SetExecFileParams(p_execData.szExecFileParams) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetContext(p_execData.fusContext) != NO_ERROR)
        rc = STRING_TOO_LONG;
    return rc;
    }


/*
 *@@ReturnszExecFileParams:
 *  Returns const pointer to szExecFileParams string.
 */
const PSZ ScriptExecute::ReturnExecFileParams()
    {
    return _szExecFileParams;
    }


/*
 *@@ReturnContext:
 *  Returns index number.
 */
USHORT ScriptExecute::ReturnContext()
    {
    return _usContext;
    }


/*
 *@@ReturnExecute:
 */
EXECUTEDATA ScriptExecute::ReturnExecute()
    {
    EXECUTEDATA execData;
    strcpy(execData.szExecFileParams, this->_szExecFileParams);
    execData.fusContext = this->_usContext;
    return execData;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptExecute::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    PSZ pszTemp = NULL;
    XSTRING strTemp;
    xstrInit(&strTemp, 0);

    ulTextLength = strlen(_szExecFileParams);

    if (_usContext == FLAG_CONFIGSYS)
        {
        ulTextLength += strlen(SZ_FLAG_CONFIGSYS) + 1;
        xstrcat(&strTemp, SZ_FLAG_CONFIGSYS, 0);
        xstrcat(&strTemp, "|", 0);
        }
    else if (_usContext == FLAG_REGISTERCLASS)
        {
        ulTextLength += strlen(SZ_FLAG_REGISTERCLASS) + 1;
        xstrcat(&strTemp, SZ_FLAG_REGISTERCLASS, 0);
        xstrcat(&strTemp, "|", 0);
        }
    else if (_usContext == FLAG_CREATEOBJECT)
        {
        ulTextLength += strlen(SZ_FLAG_CREATEOBJECT) + 1;
        xstrcat(&strTemp, SZ_FLAG_CREATEOBJECT, 0);
        xstrcat(&strTemp, "|", 0);
        }

    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_EXECUTE) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    xstrcat(&strTemp, _szExecFileParams, 0);

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s\"", SZ_ATTRIBUTE_EXECUTE, pszTemp);
    else
        sprintf(pszScriptText, "%s", strTemp.psz);

    xstrClear(&strTemp);
    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds execute object  accordingly to script text.
 */
PSZ ScriptExecute::BuildFromText(PSZ p_pszScriptText,
                                ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    PSZ pszFlag = NULL;
    ULONG ulOffset = 0, ulAttrLen = 0;
    CHAR szExecFile[CCHMAXPATH];


    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_EXECUTE, &ulOffset);
    if (pszAttr)
        {
        pszTemp = strchr(pszAttr, '|');
        if (pszTemp) // There seems to be a context flag before execfile.
            {
            // Take the flag and check which one it is.
            pszFlag = strhSubstr(pszAttr, pszTemp);
            if (strhFindWord(pszFlag, SZ_FLAG_CONFIGSYS, "\x0d\x0a ()/\\-,.",
                             "\x0d\x0a ()/\\-,.:;|"))
                this->SetContext(FLAG_CONFIGSYS);
            else if (strhFindWord(pszFlag, SZ_FLAG_REGISTERCLASS, "\x0d\x0a ()/\\-,.",
                                  "\x0d\x0a ()/\\-,.:;|"))
                this->SetContext(FLAG_REGISTERCLASS);
            else if (strhFindWord(pszFlag, SZ_FLAG_CREATEOBJECT, "\x0d\x0a ()/\\-,.",
                                  "\x0d\x0a ()/\\-,.:;|"))
                this->SetContext(FLAG_CREATEOBJECT);
            else // There is something in the context but it cannot be recognised.
                {
                msg.SetType(MSG_TYPE_INFO);
                msg.SetMessageValue(MSG_ATTRIBUTE_EXECUTE_CONTEXT_MISSING);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            delete [] pszFlag;
            // Take the rest of the attribute.
            pszTemp ++; // Jump over '\'
            while (*pszTemp == ' ') // Jump over leading spaces.
                pszTemp ++;
            pszFlag = strhdup(pszTemp, NULL);
            if (strlen(pszFlag) < CCHMAXPATH)
                this->SetExecFileParams(pszFlag);
            delete [] pszFlag;
            }
        else        // There is no flags in the attribute.
            {
            this->SetContext(FLAG_NOCONTEXT);
            if (pszAttr)
                this->SetExecFileParams(pszAttr);
            else
                {
                msg.SetType(MSG_TYPE_ERROR);
                msg.SetMessageValue(MSG_ATTRIBUTE_EXECUTE_EXEFILEPARAM_MISSING);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        }
    else
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_ATTRIBUTE_EXECUTE_MISSING);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }
    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }






/****************************************************************************
 ****************************************************************************
        @@ScriptClearProfile class:

    <B>Description:</B>
        This class is for CLEARPROFILE attribute in CONFIGSYS tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScriptClearProfile      Constructor.
        ~ScriptClearProfile     Destructor.
        SetProfileString        Sets profile strting.
        SetApplication          Sets application string.
        SetKey                  Sets key string.
        SetProfile              Sets all fileds.
        ReturnProfileString     Returns const pointer to profile string.
        ReturnApplication       Returns const pointer to application string.
        ReturnKey               Returns const pointer to key string.
        ReturnProfile           Returns profile data.
        MakeScriptText          Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptClearProfile:
 *  Constructor for ScriptClearProfile class. Initializes object by setting
 *  strings to zero and sets index number too.
 */
ScriptClearProfile::ScriptClearProfile(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    *_szProfileString = '\0';
    *_szApplication = '\0';
    *_szKey = '\0';
    }


/*
 *@@~ScriptClearProfile:
 *  Destructor for ScriptClearProfile class.
 */
ScriptClearProfile::~ScriptClearProfile()
    {
    }


/*
 *@@SetProfileString:
 *  Sets _szProfileString string text.
 */
USHORT ScriptClearProfile::SetProfileString(const PSZ p_pszProfile)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszProfile) < CCHMAXPATH)
        strcpy(this->_szProfileString, p_pszProfile);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_PROFILE_STRTOOLONG);
        msg.SetMessageText(p_pszProfile);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@ ScriptClearProfile::SetApplication
 *  Sets _szApplication string text.
 */
USHORT ScriptClearProfile::SetApplication(const PSZ p_pszApplication)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszApplication) < MAXPROFILEAPPLICATIONLENGTH)
        strcpy(this->_szApplication, p_pszApplication);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_APPL_STRTOOLONG);
        msg.SetMessageText(p_pszApplication);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetKey:
 *  Sets szKey string text.
 */
USHORT ScriptClearProfile::SetKey(const PSZ p_pszKey)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszKey) < MAXPROFILEKEYLENGTH)
        strcpy(this->_szKey, p_pszKey);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_KEY_STRTOOLONG);
        msg.SetMessageText(p_pszKey);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetProfile:
 *  Sets all string texts.
 */
USHORT ScriptClearProfile::SetProfile(PROFILEDATA p_profile)
    {
    USHORT rc = NO_ERROR;
    if (SetProfileString(p_profile.szProfile) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetApplication(p_profile.szApplication) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetKey(p_profile.szKey) != NO_ERROR)
        rc = STRING_TOO_LONG;
    return rc;
    }


/*
 *ReturnProfileString:
 *  Returns const pointer to szProfileString string.
 */
const PSZ ScriptClearProfile::ReturnProfileString()
    {
    return _szProfileString;
    }


/*
 *@@ReturnApplication:
 *  Returns const pointer to szApplication string.
 */
const PSZ ScriptClearProfile::ReturnApplication()
    {
    return _szApplication;
    }


/*
 *@@ReturnKey:
 *  Returns const pointer to szKey string.
 */
const PSZ ScriptClearProfile::ReturnKey()
    {
    return _szKey;
    }


/*
 *@@ReturnProfile
 *  Sets all string texts.
 */
PROFILEDATA ScriptClearProfile::ReturnProfile()
    {
    PROFILEDATA profile;

    strcpy(profile.szProfile, this->_szProfileString);
    strcpy(profile.szApplication, this->_szApplication);
    strcpy(profile.szKey, this->_szKey);

    return profile;
    }


/*
 *@@MakeScriptText:
 */
PSZ ScriptClearProfile::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    ulTextLength = strlen(_szProfileString) + strlen(_szApplication) +
        strlen(_szKey) + 2;


    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_CLEARPROFILE) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s\\%s\\%s\"", SZ_ATTRIBUTE_CLEARPROFILE,
                _szProfileString, _szApplication, _szKey);
    else
        sprintf(pszScriptText, "%s\\%s\\%s", _szProfileString, _szApplication,
            _szKey);

    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds clearprofile object  accordingly to script text.
 */
PSZ ScriptClearProfile::BuildFromText(PSZ p_pszScriptText,
                                      ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszTemp1 = NULL;
    PSZ pszAttr = NULL;
    PSZ pszString = NULL;
    ULONG ulOffset = 0, ulAttrLen = 0;


    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_CLEARPROFILE, &ulOffset);
    if (pszAttr)
        {
        pszTemp = strchr(pszAttr, '\\');
        if (pszTemp)
            {
            // Get profile string.
            pszString = strhSubstr(pszAttr, pszTemp);
            if (pszString)
                this->SetProfileString(pszString);
            else
                {
                msg.SetType(MSG_TYPE_WARNING);
                msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_PROFILE_MISSING);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            delete [] pszString;
            pszTemp1 = strchr(pszTemp + 2, '\\');
            if (pszTemp1)
                {
                pszString = strhSubstr(pszTemp + 1, pszTemp1);
                if (pszString)
                    this->SetApplication(pszString);
                else
                    {
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_APPL_MISSING);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                delete [] pszString;

                if (*(pszTemp1 + 1) != '\0')
                    this->SetKey(pszTemp1 + 1);
                else
                    {
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_KEY_MISSING);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                }
            else
                {
                if (*(pszTemp + 1) != '\0')
                    this->SetApplication(pszTemp + 1);
                else
                    {
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_APPL_MISSING);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                }
            }
        else
            {
            // Get profile string.
            if (pszAttr)
                this->SetProfileString(pszAttr);
            else
                {
                msg.SetType(MSG_TYPE_WARNING);
                msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_PROFILE_MISSING);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            delete [] pszString;
            }
        }
    else
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_CLRPROFILE_MISSING);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }




/****************************************************************************
 ****************************************************************************
        ScriptWriteProfile class:

    <B>Description:</B>
        This class is for WRITEPROFILE attribute in CONFIGSYS tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScriptWriteProfile      Constructor.
        ~ScriptWriteProfile     Destructor.
        SetProfileString        Sets profile strting.
        SetApplication          Sets application string.
        SetString               Sets string to be stored to INI file.
        SetKey                  Sets key string.
        SetProfile              Sets all fileds.
        ReturnProfileString     Returns const pointer to profile string.
        ReturnApplication       Returns const pointer to application string.
        ReturnKey               Returns const pointer to key string.
        ReturnProfile           Returns profile data.
        MakeScriptText          Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptWriteProfile:
 *  Constructor for ScriptWriteProfile class. Initializes object by setting
 *  strings to zero and sets index number too.
 */
ScriptWriteProfile::ScriptWriteProfile(USHORT p_usIndex)
    {
    SetIndex(p_usIndex);
    *_szProfileString = '\0';
    *_szApplication = '\0';
    *_szKey = '\0';
    _pszString = NULL;
    }


/*
 *@@~ScriptWriteProfile:
 *  Destructor for ScriptWriteProfile class.
 */
ScriptWriteProfile::~ScriptWriteProfile()
    {
    *_szProfileString = '\0';
    *_szApplication = '\0';
    *_szKey = '\0';
    if (_pszString)
        delete [] _pszString;
    _pszString = NULL;
    }


/*
 *@@operator=:
 *  Assignment operator for ScriptWriteProfile class.
 */
ScriptWriteProfile &ScriptWriteProfile::operator=
        (ScriptWriteProfile &p_writeProfile)
    {
    SetIndex(p_writeProfile.ReturnIndex());
    SetProfileString(p_writeProfile.ReturnProfileString());
    SetApplication(p_writeProfile.ReturnApplication());
    SetKey(p_writeProfile.ReturnKey());
    SetString(p_writeProfile.ReturnString());
    return (*this);
    }


/*
 *@@SetProfileString:
 *  Sets szProfileString string text.
 */
USHORT ScriptWriteProfile::SetProfileString(const PSZ p_pszProfile)
    {

    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszProfile) < CCHMAXPATH)
        strcpy(this->_szProfileString, p_pszProfile);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_PROFILE_STRTOOLONG);
        msg.SetMessageText(p_pszProfile);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetApplication:
 *  Sets szApplication string text.
 */
USHORT ScriptWriteProfile::SetApplication(const PSZ p_pszApplication)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszApplication) < MAXPROFILEAPPLICATIONLENGTH)
        strcpy(this->_szApplication, p_pszApplication);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_APPL_STRTOOLONG);
        msg.SetMessageText(p_pszApplication);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }



/*
 *@@SetKey:
 *  Sets szKey string text.
 */
USHORT ScriptWriteProfile::SetKey(const PSZ p_pszKey)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszKey) < MAXPROFILEKEYLENGTH)
        strcpy(this->_szKey, p_pszKey);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_KEY_STRTOOLONG);
        msg.SetMessageText(p_pszKey);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }

    return rc;
    }


/*
 *@@SetString:
 *  Sets pszString text.
 */
USHORT ScriptWriteProfile::SetString(const PSZ p_pszString)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (p_pszString)
        this->_pszString = strhdup(p_pszString, NULL);
    return (0);     // V0.9.14 (2001-07-07) [umoeller]
    }


/*
 *@@SetProfile:
 *  Sets all string texts.
 */
USHORT ScriptWriteProfile::SetProfile(PROFILEDATA p_profile)
    {
    USHORT rc = NO_ERROR;
    if (SetProfileString(p_profile.szProfile) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetApplication(p_profile.szApplication) != NO_ERROR)
        rc = STRING_TOO_LONG;
    if (SetKey(p_profile.szKey) != NO_ERROR)
        rc = STRING_TOO_LONG;
    return rc;
    }


/*
 *@@ReturnProfileString:
 *  Returns const pointer to szProfileString string.
 */
const PSZ ScriptWriteProfile::ReturnProfileString() const
    {
    return (PSZ)_szProfileString;
    }


/*
 *@@ReturnApplication:
 *  Returns const pointer to szApplication string.
 */
const PSZ ScriptWriteProfile::ReturnApplication() const
    {
    return (PSZ)_szApplication;
    }



/*
 *@@ReturnKey:
 *  Returns const pointer to szKey string.
 */
const PSZ ScriptWriteProfile::ReturnKey() const
    {
    return (PSZ)_szKey;
    }


/*
 *@@ReturnString:
 *  Returns const pointer to szProfileString string.
 */
const PSZ ScriptWriteProfile::ReturnString() const
    {
    return _pszString;
    }


/*
 *@@ ScriptWriteProfile::ReturnProfile
 *  Sets all string texts.
 */
PROFILEDATA ScriptWriteProfile::ReturnProfile()
    {
    PROFILEDATA profile;

    strcpy(profile.szProfile, this->_szProfileString);
    strcpy(profile.szApplication, this->_szApplication);
    strcpy(profile.szKey, this->_szKey);

    return profile;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text. The text is based on the object's datas.
 */
PSZ ScriptWriteProfile::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    if (!_pszString)
        return FALSE;

    ulTextLength = strlen(_szProfileString) + strlen(_szApplication) +
        strlen(_szKey) + strlen(_pszString) + 3;


    if (p_bAttribute)
        ulTextLength += strlen(SZ_ATTRIBUTE_WRITEPROFILE) + 3;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (p_bAttribute)
        sprintf(pszScriptText, "%s=\"%s\\%s\\%s|%s\"", SZ_ATTRIBUTE_WRITEPROFILE,
                _szProfileString, _szApplication, _szKey, _pszString);
    else
        sprintf(pszScriptText, "%s\\%s\\%s|%s", _szProfileString, _szApplication,
               _szKey, _pszString);


    return pszScriptText;
    }


/*
 *@@BuildFromText:
 *  Builds clearprofile object  accordingly to script text.
 */
PSZ ScriptWriteProfile::BuildFromText(PSZ p_pszScriptText,
                                      ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszTemp1 = NULL;
    PSZ pszAttr = NULL;
    PSZ pszString = NULL;
    ULONG ulOffset = 0, ulAttrLen = 0;


    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_WRITEPROFILE, &ulOffset);
    if (pszAttr)
        {
        pszTemp = strchr(pszAttr, '\\');
        if (pszTemp)
            {
            // Get profile string.
            pszString = strhSubstr(pszAttr, pszTemp);
            if (pszString)
                this->SetProfileString(pszString);
            else
                {
                msg.SetType(MSG_TYPE_WARNING);
                msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_PROFILE_MISSING);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            delete [] pszString;
            pszTemp1 = strchr(pszTemp + 2, '\\');
            if (pszTemp1)
                {
                pszString = strhSubstr(pszTemp + 1, pszTemp1);
                if (pszString)
                    this->SetApplication(pszString);
                else
                    {
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_APPL_MISSING);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                delete [] pszString;
                pszTemp = pszTemp1;
                pszTemp1 = strchr(pszTemp + 2, '|');
                if (pszTemp1)
                    {
                    pszString = strhSubstr(pszTemp + 1, pszTemp1);
                    if (pszString)
                        this->SetKey(pszString);
                    else
                        {
                        msg.SetType(MSG_TYPE_WARNING);
                        msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_KEY_MISSING);
                        msg.SetLineNumber(p_ulLineNumber);
                        _messageLogger.LogMessage(msg);
                        }
                    delete [] pszString;
                    pszTemp1 ++;
                    // Remove leading spaces.
                    while ((*pszTemp1 == ' ') && (*pszTemp1))
                        pszTemp1 ++;
                    this->SetString(pszTemp1);
                    }
                else
                    {
                    msg.SetType(MSG_TYPE_WARNING);
                    msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_STRING_MISSING);
                    msg.SetLineNumber(p_ulLineNumber);
                    _messageLogger.LogMessage(msg);
                    }
                }
            else
                {
                msg.SetType(MSG_TYPE_WARNING);
                msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_KEY_MISSING);
                msg.SetLineNumber(p_ulLineNumber);
                _messageLogger.LogMessage(msg);
                }
            }
        else
            {
            msg.SetType(MSG_TYPE_WARNING);
            msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_APPL_MISSING);
            msg.SetLineNumber(p_ulLineNumber);
            _messageLogger.LogMessage(msg);
            }
        }
    else
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_WRITEPROFILE_MISSING);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptKillProcess class:

    <B>Description:</B>
        This class is for KILLPROCESS attribute in CONFIGSYS tag. Detailed
        information about the meaning of the attribute can be found
        from <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScripitKillProcess   Constructor.
        ~ScriptKillProcess   Destructor.
        SetFileName     Sets file name text.
        ReturnFileName    Returns const pointer to title text.
        MakeScriptText  Returns pointer to script text.
        BuildFromText   Construct object from script text.
 ***************************************************************************/


/*
 *@@ScriptKillProcess:
 *  Constructor for ScriptKillProcess class. Initializes szFileName string by
 *  setting first char to '\0'.
 */
ScriptKillProcess::ScriptKillProcess(USHORT p_usIndex)
    {
    *_szFileName = '\0';
    SetIndex(p_usIndex);
    }


/*
 *@@~ScriptKillProcess:
 *  destructor for ScriptKillProcess class. Does nothing special.
 */
ScriptKillProcess::~ScriptKillProcess()
    {
    *_szFileName = '\0';
    }


/*
 *@@SetFileName:
 *  Sets target text.
 */
USHORT ScriptKillProcess::SetFileName(const PSZ p_pszFileName)
    {
    Message msg;
    USHORT rc = NO_ERROR;
    if (strlen(p_pszFileName) < CCHMAXPATH)
        strcpy(this->_szFileName, p_pszFileName);
    else
        {
        msg.SetType(MSG_TYPE_ERROR);
        msg.SetMessageValue(MSG_ATTRIBUTE_KILLPROCESS_FILENAME_STRTOOLONG);
        msg.SetMessageText(p_pszFileName);
        _messageLogger.LogMessage(msg);
        rc = STRING_TOO_LONG;
        }
    return rc;
    }


/*
 *@@ReturnFileName:
 *  Returns const pointer to file name text.
 */
const PSZ ScriptKillProcess::ReturnFileName()
    {
    return _szFileName;
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptKillProcess::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    ULONG ulTextLength = 0;
    XSTRING strScriptText;
    xstrInit(&strScriptText, 0);
    ulTextLength = strlen(_szFileName);
    CHAR szTemp[1024];
    *szTemp = '\0';

    if (p_bAttribute)
        // Length of the attribute, =, " and "
        ulTextLength += strlen(SZ_ATTRIBUTE_KILLPROCESS) + 3;

    if (p_bAttribute)
        sprintf(szTemp, "%s=\"%s\"", SZ_ATTRIBUTE_KILLPROCESS, _szFileName);
    else
        sprintf(szTemp, "%s\"", _szFileName);

    xstrcat(&strScriptText, szTemp, 0);
    return strScriptText.psz;
    }



/*
 *@@BuildFromText:
 *  Builds target object accordingly to script text.
 */
PSZ ScriptKillProcess::BuildFromText(PSZ p_pszScriptText,
                                    ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    ULONG ulOffset = 0;
    ULONG ulAttrLen = 0;


    pszAttr = strhGetTextAttr(p_pszScriptText, SZ_ATTRIBUTE_KILLPROCESS, &ulOffset);
    if (pszAttr)
        this->SetFileName(pszAttr);
    else
        {
        msg.SetType(MSG_TYPE_WARNING);
        msg.SetMessageValue(MSG_ATTRIBUTE_KILLPROCESS_MISSING);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }

    ulAttrLen = strlen(pszAttr);
    delete [] pszAttr;
    return p_pszScriptText + ulOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        @@ScriptPackage class:

    <B>Description:</B>
        This class is for CONFIGSYS tag. Detailed information about the
        meaning of the attribute can be found from
        <I>WarpIN Programmer's Guide and Reference</I>.

    Member functions:
        ScriptPackage           Constructor.
        ~ScriptPackage          Destructor.
        MakeScriptText          Returns pointer to script text.
        BuildFromText
 ***************************************************************************/


/*
 *@@ScriptPackage:
 *  Constructor for ScriptPackage class.
 */
ScriptPackage::ScriptPackage(USHORT p_usIndex, BOOL p_bAllowIndexArrangement)
    {
    SetIndex(p_usIndex);

    this->_bAllowIndexArrangement = p_bAllowIndexArrangement;

    _Index.SetIndexAttribute(0);

    _pszDescription = NULL;

    _pRequires = NULL;
    _usRequiresCount = 0;
    _pConfigSys = NULL;
    _usConfigSysCount = 0;
    _pRegisterClass = NULL;
    _usRegisterClassCount = 0;
    _pReplaceClass = NULL;
    _usReplaceClassCount = 0;
    _pCreateObject = NULL;
    _usCreateObjectCount = 0;
    _pExecute = NULL;
    _usExecuteCount = 0;
    _pProfile = NULL;
    _usProfileCount = 0;
    _pWriteProfile = NULL;
    _usWriteProfileCount = 0;
    _pKillProcess = NULL;
    _usKillProcessCount = 0;
    }




/*
 *@@ScriptPackage:
 *  Copy constructor for ScriptPackage class.
 */
ScriptPackage::ScriptPackage(ScriptPackage &p_package)
    {
    USHORT usIndex;

    SetIndex(p_package.ReturnIndex());

    _pszDescription = NULL;

    _Index.SetIndexAttribute(0);

    _pRequires = NULL;
    _usRequiresCount = 0;
    _pConfigSys = NULL;
    _usConfigSysCount = 0;
    _pRegisterClass = NULL;
    _usRegisterClassCount = 0;
    _pReplaceClass = NULL;
    _usReplaceClassCount = 0;
    _pCreateObject = NULL;
    _usCreateObjectCount = 0;
    _pExecute = NULL;
    _usExecuteCount = 0;
    _pProfile = NULL;
    _usProfileCount = 0;
    _pWriteProfile = NULL;
    _usWriteProfileCount = 0;
    _pKillProcess = NULL;
    _usKillProcessCount = 0;

    // Copy pszDescription.
    if (p_package._pszDescription)
        _pszDescription = strdup(p_package._pszDescription);

    // Copy requires objects.
    _pRequires = new ScriptRequires[p_package._usRequiresCount];
    _usRequiresCount = p_package._usRequiresCount;
    for (usIndex = 0; usIndex < _usRequiresCount; usIndex++)
        _pRequires[usIndex] = p_package._pRequires[usIndex];

    // Copy configsys objects.
    _pConfigSys = new ScriptConfigSys[p_package._usConfigSysCount];
    _usConfigSysCount = p_package._usConfigSysCount;
    for (usIndex = 0; usIndex < _usConfigSysCount; usIndex++)
        _pConfigSys[usIndex] = p_package._pConfigSys[usIndex];

    // Copy registerclass objects.
    _pRegisterClass = new ScriptRegisterClass[p_package._usRegisterClassCount];
    _usRegisterClassCount = p_package._usRegisterClassCount;
    for (usIndex = 0; usIndex < _usRegisterClassCount; usIndex++)
        _pRegisterClass[usIndex] = p_package._pRegisterClass[usIndex];

    // Copy replaceclass objects.
    _pReplaceClass = new ScriptReplaceClass[p_package._usReplaceClassCount];
    _usReplaceClassCount = p_package._usReplaceClassCount;
    for (usIndex = 0; usIndex < _usReplaceClassCount; usIndex++)
        _pReplaceClass[usIndex] = p_package._pReplaceClass[usIndex];

    // Copy createobject objects.
    _pCreateObject = new ScriptCreateObject[p_package._usCreateObjectCount];
    _usCreateObjectCount = p_package._usCreateObjectCount;
    for (usIndex = 0; usIndex < _usCreateObjectCount; usIndex++)
        _pCreateObject[usIndex] = p_package._pCreateObject[usIndex];

    // Copy execute objects.
    _pExecute = new ScriptExecute[p_package._usExecuteCount];
    _usExecuteCount = p_package._usExecuteCount;
    for (usIndex = 0; usIndex < _usExecuteCount; usIndex++)
        _pExecute[usIndex] = p_package._pExecute[usIndex];

    // Copy profile objects.
    _pProfile = new ScriptClearProfile[p_package._usProfileCount];
    _usProfileCount = p_package._usProfileCount;
    for (usIndex = 0; usIndex < _usProfileCount; usIndex++)
        _pProfile[usIndex] = p_package._pProfile[usIndex];

    // Copy writeprofile objects.
    _pWriteProfile = new ScriptWriteProfile[p_package._usWriteProfileCount];
    _usWriteProfileCount = p_package._usWriteProfileCount;
    for (usIndex = 0; usIndex < _usWriteProfileCount; usIndex++)
        _pWriteProfile[usIndex] = p_package._pWriteProfile[usIndex];

    // Copy killprocess objects.
    _pKillProcess = new ScriptKillProcess[p_package._usKillProcessCount];
    _usKillProcessCount = p_package._usKillProcessCount;
    for (usIndex = 0; usIndex < _usKillProcessCount; usIndex++)
        _pKillProcess[usIndex] = p_package._pKillProcess[usIndex];
    }




/*
 *@@&operator=:
 *  Assignment operator for ScriptPackage class.
 *
 *@@changed V0.9.5 (2000-08-22) [tahola]: removed const
 */
ScriptPackage &ScriptPackage::operator=(ScriptPackage &p_package)
    {
    USHORT usIndex;
    ScriptIndex PckIndex = p_package._Index;

    SetIndex(p_package.ReturnIndex());

    this->_PckID = p_package._PckID;

    this->_Title = p_package._Title;

    this->_Target = p_package._Target;

    this->_KeyWords = p_package._KeyWords;

    this->_Index.SetIndexAttribute(PckIndex.ReturnIndexAttribute());

    // Copy pszDescription.
    if (p_package._pszDescription)
        _pszDescription = strdup(p_package._pszDescription);

    // Copy requires objects.
    if (p_package._pRequires)
        {
        _pRequires = new ScriptRequires[p_package._usRequiresCount];
        _usRequiresCount = p_package._usRequiresCount;
        for (usIndex = 0; usIndex < _usRequiresCount; usIndex++)
            _pRequires[usIndex] = p_package._pRequires[usIndex];
        }
    // Copy configsys objects.
    if (p_package._pConfigSys)
        {
        _pConfigSys = new ScriptConfigSys[p_package._usConfigSysCount];
        _usConfigSysCount = p_package._usConfigSysCount;
        for (usIndex = 0; usIndex < _usConfigSysCount; usIndex++)
            _pConfigSys[usIndex] = p_package._pConfigSys[usIndex];
        }
    // Copy registerclass objects.
    if (p_package._pRegisterClass)
        {
        _pRegisterClass = new ScriptRegisterClass[p_package._usRegisterClassCount];
        _usRegisterClassCount = p_package._usRegisterClassCount;
        for (usIndex = 0; usIndex < _usRegisterClassCount; usIndex++)
            _pRegisterClass[usIndex] = p_package._pRegisterClass[usIndex];
        }
    // Copy replaceclass objects.
    if (p_package._pReplaceClass)
        {
        _pReplaceClass = new ScriptReplaceClass[p_package._usReplaceClassCount];
        _usReplaceClassCount = p_package._usReplaceClassCount;
        for (usIndex = 0; usIndex < _usReplaceClassCount; usIndex++)
            _pReplaceClass[usIndex] = p_package._pReplaceClass[usIndex];
        }
    // Copy createobject objects.
    if (p_package._pCreateObject)
        {
        _pCreateObject = new ScriptCreateObject[p_package._usCreateObjectCount];
        _usCreateObjectCount = p_package._usCreateObjectCount;
        for (usIndex = 0; usIndex < _usCreateObjectCount; usIndex++)
            _pCreateObject[usIndex] = p_package._pCreateObject[usIndex];
        }
    // Copy execute objects.
    if (p_package._pExecute)
        {
        _pExecute = new ScriptExecute[p_package._usExecuteCount];
        _usExecuteCount = p_package._usExecuteCount;
        for (usIndex = 0; usIndex < _usExecuteCount; usIndex++)
            _pExecute[usIndex] = p_package._pExecute[usIndex];
        }
    // Copy profile objects.
    if (p_package._pProfile)
        {
        _pProfile = new ScriptClearProfile[p_package._usProfileCount];
        _usProfileCount = p_package._usProfileCount;
        for (usIndex = 0; usIndex < _usProfileCount; usIndex++)
            _pProfile[usIndex] = p_package._pProfile[usIndex];
        }
    // Copy writeprofile objects.
    if (p_package._pWriteProfile)
        {
        _pWriteProfile = new ScriptWriteProfile[p_package._usWriteProfileCount];
        _usWriteProfileCount = p_package._usWriteProfileCount;
        for (usIndex = 0; usIndex < _usWriteProfileCount; usIndex++)
            _pWriteProfile[usIndex] = p_package._pWriteProfile[usIndex];
        }
    // Copy killprocess objects.
    if (p_package._pKillProcess)
        {
        _pKillProcess = new ScriptKillProcess[p_package._usKillProcessCount];
        _usKillProcessCount = p_package._usKillProcessCount;
        for (usIndex = 0; usIndex < _usKillProcessCount; usIndex++)
            _pKillProcess[usIndex] = p_package._pKillProcess[usIndex];
        }


    return (*this);
    }



/*
 *@@~ScriptPackage:
 *  Destructor for ScriptPackage class.
 */
ScriptPackage::~ScriptPackage()
    {
    ClearAll;
    }


/*
 *@@ClearAll:
 *  Clears all fields.
 *
 *@@ added V0.9.5 (2000-08-22) [tahola]
 */
USHORT ScriptPackage::ClearAll()
    {
    if (_pszDescription)
        delete [] _pszDescription;
    _pszDescription = NULL;
    if (_pRequires)
        delete [] _pRequires;
    _pRequires = NULL;
    _usRequiresCount = 0;
    if (_pConfigSys)
        delete [] _pConfigSys;
    _pConfigSys = NULL;
    _usConfigSysCount = 0;
    if (_pRegisterClass)
        delete [] _pRegisterClass;
    _pRegisterClass = NULL;
    _usRegisterClassCount = 0;
    if (_pCreateObject)
        delete [] _pCreateObject;
    _pCreateObject = NULL;
    _usCreateObjectCount = 0;
    if (_pExecute)
        delete [] _pExecute;
    _pExecute = NULL;
    _usExecuteCount = 0;
    if (_pProfile)
        delete [] _pProfile;
    _pProfile = NULL;
    _usProfileCount = 0;
    if (_pWriteProfile)
        delete [] _pWriteProfile;
    _pProfile = NULL;
    _usProfileCount = 0;
    if (_pKillProcess)
        delete [] _pKillProcess;
    _pKillProcess = NULL;
    _usKillProcessCount = 0;

    return NO_ERROR;
    }


/*
 *@@SetDescription:
 *  Sets description string. If parameter is NULL, old pszDescription
 *  will be deleted.
 *
 *  <B>Return values:</B>
 *      NO_ERROR: No errors occurred.
 *      MEM_ALLOC_ERROR: Error occurred during memory allocation.
 */
USHORT ScriptPackage::SetDescription(
                        const PSZ p_pszDescription // in: a pointer to a string.
                                    )
    {
    ULONG ulStringLength;
    if (this->_pszDescription)
        delete [] this->_pszDescription;
    this->_pszDescription = NULL;

    if (p_pszDescription)
        {
        ulStringLength = strlen(p_pszDescription);
        if (!(this->_pszDescription = (PSZ)new CHAR[ulStringLength + 1]))
            return MEM_ALLOC_ERROR;
        strcpy(this->_pszDescription, p_pszDescription);
        }
    return NO_ERROR;
    }


/*
 *@@ReturnDescription:
 *  Returns constant pointer to description string.
 */
const PSZ ScriptPackage::ReturnDescription()
    {
    return _pszDescription;
    }


/*
 *@@ ScriptPackage::SetPackageIndex
 *  Sets object's index.
 */
USHORT ScriptPackage::SetPackageIndex(USHORT p_usPackageIndex)
    {
    return _Index.SetIndexAttribute(p_usPackageIndex);
    }


/*
 *@@SetPackageID
 */
USHORT ScriptPackage::SetPackageID(PACKAGEIDDATA p_packageIDData)
    {
    return _PckID.SetPackageID(p_packageIDData);
    }


/*
 *@@SetTitle:
 */
USHORT ScriptPackage::SetTitle(const PSZ p_pszTitle)
    {
    return _Title.SetTitle(p_pszTitle);
    }


/*
 *@@ReturnTitle:
 */
const PSZ ScriptPackage::ReturnTitleText()
    {
    return _Title.ReturnTitle();
    }


/*
 *@@SetTarget:
 */
USHORT ScriptPackage::SetTarget(const PSZ p_pszTarget)
    {
    return _Target.SetTarget(p_pszTarget);
    }


/*
 *@@SetKeyWords:
 */
USHORT ScriptPackage::SetKeyWords(USHORT p_usKeyWords)
    {
    return _KeyWords.SetKeyWord(p_usKeyWords);
    }


/*
 *@@ReturnPackageIDObject:
 *  Returns pointer to PackageID object.
 */
ScriptPackageID *ScriptPackage::ReturnPackageIDObject()
    {
    return &_PckID;
    }


/*
 *@@ReturnTitleObject:
 *  Returns pointer to PackageID object.
 */
ScriptTitle *ScriptPackage::ReturnTitleObject()
    {
    return &_Title;
    }


/*
 *@@ReturnTargetObject:
 *  Returns pointer to Target object.
 */
ScriptTarget *ScriptPackage::ReturnTargetObject()
    {
    return &_Target;
    }


/*
 *@@ReturnTargetText:
 *  Returns pointer to title text.
 */
const PSZ ScriptPackage::ReturnTargetText()
    {
    return _Target.ReturnTarget();
    }


/*
 *@@ReturnKeyWordObject:
 *  Returns pointer to PackageID object.
 */
ScriptKeyWord *ScriptPackage::ReturnKeyWordObject()
    {
    return &_KeyWords;
    }


/*
 *@@ReturnPackageIndex:
 *  Returns package index number.
 */
ScriptIndex *ScriptPackage::ReturnPackageIndex()
    {
    return &_Index;
    }


/*
 *@@ReturnPackageIndex:
 *  Returns package index number.
 */
USHORT ScriptPackage::ReturnPackageIndexNumber()
    {
    return _Index.ReturnIndexAttribute();
    }


/*
 *@@MakeScriptText:
 *  Returns pointer to script text.
 */
PSZ ScriptPackage::MakeScriptText(BOOL p_bAttribute,
                                BOOL bLineFeed) // V0.9.14 (2001-07-07) [umoeller]
    {
    XSTRING strScript;
    xstrInit(&strScript, 0);
    PSZ pszTemp = NULL;
    CHAR szTag[MAXTAGLENGTH];
    CHAR szAttribute[MAXATTRIBUTELENGTH];
    USHORT usIndex;

    if (p_bAttribute)
        {
        sprintf(szTag, "<%s ", SZ_TAG_PACKAGE);
        xstrcat(&strScript, szTag, 0);
        }
    //sprintf(szAttribute, "%s=%i\n", SZ_ATTRIBUTE_INDEX, Index.ReturnIndexAttribute());
    xstrcat(&strScript, "\t", 0);
    pszTemp = _Index.MakeScriptText(p_bAttribute);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    xstrcat(&strScript, "\t", 0);
    pszTemp = _PckID.MakeScriptText(p_bAttribute);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    xstrcat(&strScript, "\t", 0);
    pszTemp = _Title.MakeScriptText(p_bAttribute);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    xstrcat(&strScript, "\t", 0);
    pszTemp = _Target.MakeScriptText(p_bAttribute);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    xstrcat(&strScript, "\t", 0);
    pszTemp = _KeyWords.MakeScriptText(p_bAttribute);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    for (usIndex = 0; usIndex < _usRequiresCount; usIndex ++)
        {
        pszTemp = _pRequires[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    for (usIndex = 0; usIndex < _usConfigSysCount; usIndex ++)
        {
        pszTemp = _pConfigSys[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    for (usIndex = 0; usIndex < _usRegisterClassCount; usIndex ++)
        {
        pszTemp = _pRegisterClass[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }


    for (usIndex = 0; usIndex < _usReplaceClassCount; usIndex ++)
        {
        pszTemp = _pReplaceClass[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }


    for (usIndex = 0; usIndex < _usCreateObjectCount; usIndex ++)
        {
        pszTemp = _pCreateObject[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    for (usIndex = 0; usIndex < _usExecuteCount; usIndex ++)
        {
        pszTemp = _pExecute[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }


    for (usIndex = 0; usIndex < _usProfileCount; usIndex ++)
        {
        pszTemp = _pProfile[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    for (usIndex = 0; usIndex < _usWriteProfileCount; usIndex ++)
        {
        pszTemp = _pWriteProfile[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    for (usIndex = 0; usIndex < _usKillProcessCount; usIndex ++)
        {
        pszTemp = _pKillProcess[usIndex].MakeScriptText(p_bAttribute);
        xstrcat(&strScript, "\t", 0);
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }


    if (p_bAttribute)
        xstrcat(&strScript, ">", 0);
    if (_pszDescription)
        xstrcat(&strScript, _pszDescription, 0);

    if (p_bAttribute)
        {
        sprintf(szTag, "<%s>", SZ_TAG_ENDPACKAGE);
        xstrcat(&strScript, szTag, 0);
        }

    return strScript.psz;
    }


/*
 *@@CreateRequiresObject:
 *  Creates new requires object and returns pointer to it.
 */
ScriptRequires *ScriptPackage::CreateRequiresObject()
    {
    ScriptRequires *requires, *pNewRequires, *pTemp;

    // Create new table for ScriptRequires objects.
    if (!(requires = new ScriptRequires[++_usRequiresCount]))
        {
        _usRequiresCount --;
        return NULL;
        }

    // Set new object's index to first unused index.
    requires[_usRequiresCount - 1].SetIndex(FindUnusedIndex(this->_pRequires,
        (USHORT)(_usRequiresCount - 1)));

    CopyTable(requires, this->_pRequires, (USHORT)(_usRequiresCount - 1), (USHORT)0);
    delete [] this->_pRequires;
    this->_pRequires = requires;
    pNewRequires = &this->_pRequires[_usRequiresCount - 1];
    return pNewRequires;
    }


/*
 *@@CreateRequiresObject
 *  Creates requires object and sets datas and returns pointer to it.
 *  Also sets datas according to requiresData.
 */
ScriptRequires *ScriptPackage::CreateRequiresObject(REQUIRESDATA p_requiresData)
    {
    ScriptRequires *RequiresObject = CreateRequiresObject();

    RequiresObject->SetRequires(p_requiresData);
    return RequiresObject;
    }


/*
 *@@ReturnRequiresObject
 *  Returns object which index is given usIndex.
 */
ScriptRequires *ScriptPackage::ReturnRequiresObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usRequiresCount; Index ++)
        if (_pRequires[Index].ReturnIndex() == p_usIndex)
            return &_pRequires[Index];
    return NULL;
    }


/*
 *@@DeleteRequiresObject:
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptPackage::DeleteRequiresObject(USHORT p_usIndex)
    {
    ScriptRequires *TempRequires = NULL;

    if (_usRequiresCount > 1)
        {
        TempRequires = new ScriptRequires[_usRequiresCount - 1];
        CopyTable(TempRequires, _pRequires, _usRequiresCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempRequires, (USHORT)(_usRequiresCount - 1));
        }
    delete [] _pRequires;
    _pRequires = TempRequires;
    _usRequiresCount --;
    return 0;
    }


/*
 *@@DeleteRequiresObject:
 *  Deletes object which address is *ScriptRequiresObject.
 */
USHORT ScriptPackage::DeleteRequiresObject(ScriptRequires *p_pScriptRequiresObject)
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptRequiresObject)
        return 1;

    // Check that ScriptRequiresObject really exist on Requires table.
    while (TRUE)
        {
        if (&_pRequires[Index] == p_pScriptRequiresObject)
            break;
        Index ++;
        if (Index >= _usRequiresCount)
            return 2;
        }

    usDelete = _pRequires[Index].ReturnIndex();
    rc = DeleteRequiresObject(usDelete);

    return rc;
    }


/*
 *@@ReturnRequiresObjectCount:
 *  Returns count number of ScriptRequiresObjects.
 */
USHORT ScriptPackage::ReturnRequiresObjectCount()
    {
    return _usRequiresCount;
    }


/*
 *@@CreateConfigSysObject:
 *  Creates new ConfigSys object and returns pointer to it.
 */
ScriptConfigSys *ScriptPackage::CreateConfigSysObject()
    {
    ScriptConfigSys *Object, *NewObject;

    if (!(Object = new ScriptConfigSys[++_usConfigSysCount]))
        {
        _usConfigSysCount --;
        return NULL;
        }

    Object[_usConfigSysCount - 1].SetIndex(FindUnusedIndex(this->_pConfigSys,
        (USHORT)(_usConfigSysCount - 1)));

    CopyTable(Object, this->_pConfigSys, (USHORT)(_usConfigSysCount - 1), (USHORT)0);
    delete [] this->_pConfigSys;
    this->_pConfigSys = Object;
    NewObject = &this->_pConfigSys[_usConfigSysCount - 1];
    return NewObject;
    }


/*
 *@@CreateConfigSysObject:
 *  Creates ConfigSys object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptConfigSys *ScriptPackage::CreateConfigSysObject(CONFIGSYSDATA p_data)
    {
    ScriptConfigSys *ConfigSysObject = CreateConfigSysObject();

    ConfigSysObject->SetConfigSys(p_data);
    return ConfigSysObject;
    }


/*
 *@@ReturnConfigSysObject:
 *  Returns object which index is given usIndex.
 */
ScriptConfigSys *ScriptPackage::ReturnConfigSysObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usConfigSysCount; Index ++)
        if (_pConfigSys[Index].ReturnIndex() == p_usIndex)
            return &_pConfigSys[Index];
    return NULL;
    }


/*
 *@@DeleteConfigSysObject:
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptPackage::DeleteConfigSysObject(USHORT p_usIndex)
    {
    ScriptConfigSys *TempConfigSys = NULL;

    if (_usConfigSysCount > 1)
        {
        TempConfigSys = new ScriptConfigSys[_usConfigSysCount - 1];
        CopyTable(TempConfigSys, _pConfigSys, _usConfigSysCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempConfigSys, (USHORT)(_usConfigSysCount - 1));
        }
    delete [] _pConfigSys;
    _pConfigSys = TempConfigSys;
    _usConfigSysCount --;
    return 0;
    }


/*
 *@@DeleteConfigSysObject:
 *  Deletes object which address is *ScriptConfigSysObject.
 */
USHORT ScriptPackage::DeleteConfigSysObject(ScriptConfigSys *p_pScriptConfigSysObject)
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptConfigSysObject)
        return 1;

    // Check that ScriptConfigSysObject really exist on ConfigSys table.
    while (TRUE)
        {
        if (&_pConfigSys[Index] == p_pScriptConfigSysObject)
            break;
        Index ++;
        if (Index >= _usConfigSysCount)
            return 2;
        }

    usDelete = _pConfigSys[Index].ReturnIndex();
    rc = DeleteConfigSysObject(usDelete);


    return rc;
    }


/*
 *@@ReturnConfigSysObjectCount:
 *  Returns count number of ScriptConfigSysObjects.
 */
USHORT ScriptPackage::ReturnConfigSysObjectCount()
    {
    return _usConfigSysCount;
    }


/*
 *@@CreateRegisterClassObject:
 *  Creates new RegisterClass object and returns pointer to it.
 */
ScriptRegisterClass *ScriptPackage::CreateRegisterClassObject()
    {
    ScriptRegisterClass *Object, *NewObject;

    if (!(Object = new ScriptRegisterClass[++_usRegisterClassCount]))
        {
        _usRegisterClassCount --;
        return NULL;
        }

    Object[_usRegisterClassCount - 1].SetIndex(FindUnusedIndex(this->_pRegisterClass,
        (USHORT)(_usRegisterClassCount - 1)));

    CopyTable(Object, this->_pRegisterClass, (USHORT)(_usRegisterClassCount - 1),
              (USHORT)0);
    delete [] this->_pRegisterClass;
    this->_pRegisterClass = Object;
    NewObject = &this->_pRegisterClass[_usRegisterClassCount - 1];
    return NewObject;
    }


/*
 *@@CreateRegisterClassObject:
 *  Creates RegisterClass object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptRegisterClass *ScriptPackage::CreateRegisterClassObject(REGISTERCLASSDATA p_data)
    {
    ScriptRegisterClass *RegisterClassObject = CreateRegisterClassObject();

    RegisterClassObject->SetRegisterClass(p_data);
    return RegisterClassObject;
    }


/*
 *@@ReturnRegisterClassObject:
 *  Returns object which index is given p_usIndex.
 */
ScriptRegisterClass *ScriptPackage::ReturnRegisterClassObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usRegisterClassCount; Index ++)
        if (_pRegisterClass[Index].ReturnIndex() == p_usIndex)
            return &_pRegisterClass[Index];
    return NULL;
    }


/*
 *@@DeleteRegisterClassObject:
 *  Deletes object which index is given in p_usIndex.
 */
USHORT ScriptPackage::DeleteRegisterClassObject(USHORT p_usIndex)
    {
    ScriptRegisterClass *TempRegisterClass = NULL;

    if (_usRegisterClassCount > 1)
        {
        TempRegisterClass = new ScriptRegisterClass[_usRegisterClassCount - 1];
        CopyTable(TempRegisterClass, _pRegisterClass, _usRegisterClassCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempRegisterClass, (USHORT)(_usRegisterClassCount - 1));
        }
    delete [] _pRegisterClass;
    _pRegisterClass = TempRegisterClass;
    _usRegisterClassCount --;
    return 0;
    }


/*
 *@@DeleteRegisterClassObject:
 *  Deletes object which address is *ScriptRegisterClassObject.
 */
USHORT ScriptPackage::DeleteRegisterClassObject(
                      ScriptRegisterClass *p_pScriptRegisterClassObject
                                               )
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptRegisterClassObject)
        return 1;

    // Check that ScriptRegisterClassObject really exist on RegisterClass table.
    while (TRUE)
        {
        if (&_pRegisterClass[Index] == p_pScriptRegisterClassObject)
            break;
        Index ++;
        if (Index >= _usRegisterClassCount)
            return 2;
        }

    usDelete = _pRegisterClass[Index].ReturnIndex();
    rc = DeleteRegisterClassObject(usDelete);


    return rc;
    }


/*
 *@@ReturnRegisterClassObjectCount:
 *  Returns count number of ScriptRegisterClassObjects.
 */
USHORT ScriptPackage::ReturnRegisterClassObjectCount()
    {
    return _usRegisterClassCount;
    }


/*
 *@@CreateReplaceClassObject:
 *  Creates new ReplaceClass object and returns pointer to it.
 */
ScriptReplaceClass *ScriptPackage::CreateReplaceClassObject()
    {
    ScriptReplaceClass *Object, *NewObject;

    if (!(Object = new ScriptReplaceClass[++_usReplaceClassCount]))
        {
        _usReplaceClassCount --;
        return NULL;
        }

    Object[_usReplaceClassCount - 1].SetIndex(FindUnusedIndex(this->_pReplaceClass,
        (USHORT)(_usReplaceClassCount - 1)));

    CopyTable(Object, this->_pReplaceClass, (USHORT)(_usReplaceClassCount - 1),
              (USHORT)0);
    delete [] this->_pReplaceClass;
    this->_pReplaceClass = Object;
    NewObject = &this->_pReplaceClass[_usReplaceClassCount - 1];
    return NewObject;
    }


/*
 *@@CreateReplaceClassObject:
 *  Creates ReplaceClass object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptReplaceClass *ScriptPackage::CreateReplaceClassObject(REPLACECLASSDATA p_data)
    {
    ScriptReplaceClass *ReplaceClassObject = CreateReplaceClassObject();

    ReplaceClassObject->SetReplaceClass(p_data);
    return ReplaceClassObject;
    }


/*
 *@@ReturnReplaceClassObject:
 *  Returns object which index is given usIndex.
 */
ScriptReplaceClass *ScriptPackage::ReturnReplaceClassObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usReplaceClassCount; Index ++)
        if (_pReplaceClass[Index].ReturnIndex() == p_usIndex)
            return &_pReplaceClass[Index];
    return NULL;
    }


/*
 *@@DeleteReplaceClassObject:
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptPackage::DeleteReplaceClassObject(USHORT p_usIndex)
    {
    ScriptReplaceClass *TempReplaceClass = NULL;

    if (_usReplaceClassCount > 1)
        {
        TempReplaceClass = new ScriptReplaceClass[_usReplaceClassCount - 1];
        CopyTable(TempReplaceClass, _pReplaceClass, _usReplaceClassCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempReplaceClass, (USHORT)(_usReplaceClassCount - 1));
        }
    delete [] _pReplaceClass;
    _pReplaceClass = TempReplaceClass;
    _usReplaceClassCount --;
    return 0;
    }


/*
 *@@DeleteReplaceClassObject:
 *  Deletes object which address is *ScriptReplaceClassObject.
 */
USHORT ScriptPackage::DeleteReplaceClassObject(
                      ScriptReplaceClass *p_pScriptReplaceClassObject
                                              )
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptReplaceClassObject)
        return 1;

    // Check that ScriptReplaceClassObject really exist on ReplaceClass table.
    while (TRUE)
        {
        if (&_pReplaceClass[Index] == p_pScriptReplaceClassObject)
            break;
        Index ++;
        if (Index >= _usReplaceClassCount)
            return 2;
        }

    usDelete = _pReplaceClass[Index].ReturnIndex();
    rc = DeleteReplaceClassObject(usDelete);


    return rc;
    }


/*
 *@@ReturnReplaceClassObjectCount:
 *  Returns count number of ScriptReplaceClassObjects.
 */
USHORT ScriptPackage::ReturnReplaceClassObjectCount()
    {
    return _usReplaceClassCount;
    }


/*
 *@@CreateCreateObjectObject:
 *  Creates new CreateObject object and returns pointer to it.
 */
ScriptCreateObject *ScriptPackage::CreateCreateObjectObject()
    {
    ScriptCreateObject *Object, *NewObject;

    if (!(Object = new ScriptCreateObject[++_usCreateObjectCount]))
        {
        _usCreateObjectCount --;
        return NULL;
        }

    Object[_usCreateObjectCount - 1].SetIndex(FindUnusedIndex(this->_pCreateObject,
        (USHORT)(_usCreateObjectCount - 1)));

    CopyTable(Object, this->_pCreateObject, (USHORT)(_usCreateObjectCount - 1), (USHORT)0);
    delete [] this->_pCreateObject;
    this->_pCreateObject = Object;
    NewObject = &this->_pCreateObject[_usCreateObjectCount - 1];
    return NewObject;
    }


/*
 *@@CreateCreateObjectObject:
 *  Creates CreateObject object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptCreateObject *ScriptPackage::CreateCreateObjectObject(CREATEOBJECTDATA p_data)
    {
    ScriptCreateObject *CreateObjectObject = CreateCreateObjectObject();

    CreateObjectObject->SetCreateObject(p_data);
    return CreateObjectObject;
    }


/*
 *@@ReturnCreateObjectObject:
 *  Returns object which index is given usIndex.
 */
ScriptCreateObject *ScriptPackage::ReturnCreateObjectObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usCreateObjectCount; Index ++)
        if (_pCreateObject[Index].ReturnIndex() == p_usIndex)
            return &_pCreateObject[Index];
    return NULL;
    }


/*
 *@@ ScriptPackage::DeleteCreateObjectObject
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptPackage::DeleteCreateObjectObject(USHORT p_usIndex)
    {
    ScriptCreateObject *TempCreateObject = NULL;

    if (_usCreateObjectCount > 1)
        {
        TempCreateObject = new ScriptCreateObject[_usCreateObjectCount - 1];
        CopyTable(TempCreateObject, _pCreateObject, _usCreateObjectCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempCreateObject, (USHORT)(_usCreateObjectCount - 1));
        }
    delete [] _pCreateObject;
    _pCreateObject = TempCreateObject;
    _usCreateObjectCount --;
    return 0;
    }



/*
 *@@DeleteCreateObjectObject:
 *  Deletes object which address is given as an attribute.
 */
USHORT ScriptPackage::DeleteCreateObjectObject(
                      ScriptCreateObject *p_pScriptCreateObjectObject
                                              )
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptCreateObjectObject)
        return 1;

    // Check that ScriptCreateObjectObject really exist on CreateObject table.
    while (TRUE)
        {
        if (&_pCreateObject[Index] == p_pScriptCreateObjectObject)
            break;
        Index ++;
        if (Index >= _usCreateObjectCount)
            return 2;
        }

    usDelete = _pCreateObject[Index].ReturnIndex();
    rc = DeleteCreateObjectObject(usDelete);

    return rc;
    }


/*
 *@@ReturnCreateObjectObjectCount:
 *  Returns count number of ScriptCreateObjectObjects.
 */
USHORT ScriptPackage::ReturnCreateObjectObjectCount()
    {
    return _usCreateObjectCount;
    }


/*
 *@@CreateClearProfileObject:
 *  Creates new ClearProfile object and returns pointer to it.
 */
ScriptClearProfile *ScriptPackage::CreateClearProfileObject()
    {
    ScriptClearProfile *Object, *NewObject;

    if (!(Object = new ScriptClearProfile[++_usProfileCount]))
        {
        _usProfileCount --;
        return NULL;
        }

    Object[_usProfileCount - 1].SetIndex(FindUnusedIndex(this->_pProfile,
        (USHORT)(_usProfileCount - 1)));

    CopyTable(Object, this->_pProfile, (USHORT)(_usProfileCount - 1), (USHORT)0);
    delete [] this->_pProfile;
    this->_pProfile = Object;
    NewObject = &this->_pProfile[_usProfileCount - 1];
    return NewObject;
    }


/*
 *@@CreateClearProfileObject:
 *  Creates Profile object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptClearProfile *ScriptPackage::CreateClearProfileObject(PROFILEDATA p_data)
    {
    ScriptClearProfile *ProfileObject = CreateClearProfileObject();

    ProfileObject->SetProfile(p_data);
    return ProfileObject;
    }


/*
 *@@ReturnClearProfileObject:
 *  Returns object which index is given as a parameter.
 */
ScriptClearProfile *ScriptPackage::ReturnClearProfileObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usProfileCount; Index ++)
        if (_pProfile[Index].ReturnIndex() == p_usIndex)
            return &_pProfile[Index];
    return NULL;
    }


/*
 *@@DeleteClearProfileObject:
 *  Deletes object which index is given.
 */
USHORT ScriptPackage::DeleteClearProfileObject(USHORT p_usIndex)
    {
    ScriptClearProfile *TempProfile = NULL;

    if (_usProfileCount > 1)
        {
        TempProfile = new ScriptClearProfile[_usProfileCount - 1];
        CopyTable(TempProfile, _pProfile, _usProfileCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempProfile, (USHORT)(_usProfileCount - 1));
        }
    delete [] _pProfile;
    _pProfile = TempProfile;
    _usProfileCount --;
    return 0;
    }


/*
 *@@DeleteClearProfileObject:
 *  Deletes object which address is given.
 */
USHORT ScriptPackage::DeleteClearProfileObject(ScriptClearProfile
                                               *p_pScriptClearProfileObject)
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptClearProfileObject)
        return 1;

    // Check that ScriptClearProfileObject really exist on Profile table.
    while (TRUE)
        {
        if (&_pProfile[Index] == p_pScriptClearProfileObject)
            break;
        Index ++;
        if (Index >= _usProfileCount)
            return 2;
        }

    usDelete = _pProfile[Index].ReturnIndex();
    rc = DeleteClearProfileObject(usDelete);

    return rc;
    }


/*
 *@@ReturnClearProfileObjectCount:
 *  Returns count number of ScriptClearProfileObjects.
 */
USHORT ScriptPackage::ReturnClearProfileObjectCount()
    {
    return _usProfileCount;
    }


/*
 *@@CreateWriteProfileObject:
 *  Creates new WriteProfile object and returns pointer to it.
 */
ScriptWriteProfile *ScriptPackage::CreateWriteProfileObject()
    {
    ScriptWriteProfile *Object, *NewObject;

    if (!(Object = new ScriptWriteProfile[++_usWriteProfileCount]))
        {
        _usWriteProfileCount --;
        return NULL;
        }

    Object[_usWriteProfileCount - 1].SetIndex(FindUnusedIndex(this->_pWriteProfile,
        (USHORT)(_usWriteProfileCount - 1)));

    CopyTable(Object, this->_pWriteProfile, (USHORT)(_usWriteProfileCount - 1),
             (USHORT)0);
    delete [] this->_pWriteProfile;
    this->_pWriteProfile = Object;
    NewObject = &this->_pWriteProfile[_usWriteProfileCount - 1];
    return NewObject;
    }


/*
 *@@CreateWriteProfileObject:
 *  Creates WriteProfile object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptWriteProfile *ScriptPackage::CreateWriteProfileObject(PROFILEDATA p_data)
    {
    ScriptWriteProfile *WriteProfileObject = CreateWriteProfileObject();

    WriteProfileObject->SetProfile(p_data);
    return WriteProfileObject;
    }


/*
 *@@ReturnWriteProfileObject:
 *  Returns object which index is given usIndex.
 */
ScriptWriteProfile *ScriptPackage::ReturnWriteProfileObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usWriteProfileCount; Index ++)
        if (_pWriteProfile[Index].ReturnIndex() == p_usIndex)
            return &_pWriteProfile[Index];
    return NULL;
    }


/*
 *@@DeleteWriteProfileObject:
 *  Deletes object which index is given in as a parameter.
 */
USHORT ScriptPackage::DeleteWriteProfileObject(USHORT p_usIndex)
    {
    ScriptWriteProfile *TempWriteProfile = NULL;

    if (_usWriteProfileCount > 1)
        {
        TempWriteProfile = new ScriptWriteProfile[_usWriteProfileCount - 1];
        CopyTable(TempWriteProfile, _pWriteProfile, _usWriteProfileCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempWriteProfile, (USHORT)(_usWriteProfileCount - 1));
        }
    delete [] _pWriteProfile;
    _pWriteProfile = TempWriteProfile;
    _usWriteProfileCount --;
    return 0;
    }


/*
 *@@DeleteWriteProfileObject:
 *  Deletes object which address is given as a parameter.
 */
USHORT ScriptPackage::DeleteWriteProfileObject(
                      ScriptWriteProfile *p_pScriptWriteProfileObject
                                              )
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptWriteProfileObject)
        return 1;

    // Check that ScriptWriteProfileObject really exist on WriteProfile table.
    while (TRUE)
        {
        if (&_pWriteProfile[Index] == p_pScriptWriteProfileObject)
            break;
        Index ++;
        if (Index >= _usWriteProfileCount)
            return 2;
        }

    usDelete = _pWriteProfile[Index].ReturnIndex();
    rc = DeleteWriteProfileObject(usDelete);
    return rc;
    }


/*
 *@@ReturnWriteProfileObjectCount:
 *  Returns count number of ScriptWriteProfileObjects.
 */
USHORT ScriptPackage::ReturnWriteProfileObjectCount()
    {
    return _usWriteProfileCount;
    }


/*
 *@@CreateExecuteObject:
 *  Creates new Execute object and returns pointer to it.
 */
ScriptExecute *ScriptPackage::CreateExecuteObject()
    {
    ScriptExecute *Object, *NewObject;

    if (!(Object = new ScriptExecute[++_usExecuteCount]))
        {
        _usExecuteCount --;
        return NULL;
        }

    Object[_usExecuteCount - 1].SetIndex(FindUnusedIndex(this->_pExecute,
        (USHORT)(_usExecuteCount - 1)));

    CopyTable(Object, this->_pExecute, (USHORT)(_usExecuteCount - 1), (USHORT)0);
    delete [] this->_pExecute;
    this->_pExecute = Object;
    NewObject = &this->_pExecute[_usExecuteCount - 1];
    return NewObject;
    }


/*
 *@@CreateExecuteObject:
 *  Creates Execute object and sets datas and returns pointer to it.
 */
ScriptExecute *ScriptPackage::CreateExecuteObject(PSZ p_pszExecute)
    {
    ScriptExecute *ExecuteObject = CreateExecuteObject();

    ExecuteObject->SetExecFileParams(p_pszExecute);
    return ExecuteObject;
    }


/*
 *@@ReturnExecuteObject:
 *  Returns object which index is given usIndex.
 */
ScriptExecute *ScriptPackage::ReturnExecuteObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usExecuteCount; Index ++)
        if (_pExecute[Index].ReturnIndex() == p_usIndex)
            return &_pExecute[Index];
    return NULL;
    }


/*
 *@@DeleteExecuteObject:
 *  Deletes object which index is given as a parameter.
 */
USHORT ScriptPackage::DeleteExecuteObject(USHORT p_usIndex)
    {
    ScriptExecute *TempExecute = NULL;

    if (_usExecuteCount > 1)
        {
        TempExecute = new ScriptExecute[_usExecuteCount - 1];
        CopyTable(TempExecute, _pExecute, _usExecuteCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempExecute, (USHORT)(_usExecuteCount - 1));
        }
    delete [] _pExecute;
    _pExecute = TempExecute;
    _usExecuteCount --;
    return 0;
    }


/*
 *@@DeleteExecuteObject:
 *  Deletes object which address is given as a parameter.
 */
USHORT ScriptPackage::DeleteExecuteObject(ScriptExecute *p_pScriptExecuteObject)
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptExecuteObject)
        return 1;

    // Check that ScriptExecuteObject really exist on Execute table.
    while (TRUE)
        {
        if (&_pExecute[Index] == p_pScriptExecuteObject)
            break;
        Index ++;
        if (Index >= _usExecuteCount)
            return 2;
        }

    usDelete = _pExecute[Index].ReturnIndex();
    rc = DeleteExecuteObject(usDelete);

    return rc;
    }


/*
 *@@ReturnExecuteObjectCount:
 *  Returns count number of ScriptExecuteObjects.
 */
USHORT ScriptPackage::ReturnExecuteObjectCount()
    {
    return _usExecuteCount;
    }


/*
 *@@CreateKillProcessObject:
 *  Creates new KillProcess object and returns pointer to it.
 */
ScriptKillProcess *ScriptPackage::CreateKillProcessObject()
    {
    ScriptKillProcess *Object, *NewObject;

    if (!(Object = new ScriptKillProcess[++_usKillProcessCount]))
        {
        _usKillProcessCount --;
        return NULL;
        }

    Object[_usKillProcessCount - 1].SetIndex(FindUnusedIndex(this->_pKillProcess,
        (USHORT)(_usKillProcessCount - 1)));

    CopyTable(Object, this->_pKillProcess, (USHORT)(_usKillProcessCount - 1), (USHORT)0);
    delete [] this->_pKillProcess;
    this->_pKillProcess = Object;
    NewObject = &this->_pKillProcess[_usKillProcessCount - 1];
    return NewObject;
    }


/*
 *@@CreateKillProcessObject:
 *  Creates KillProcess object and sets datas and returns pointer to it.
 *  Also sets datas according to data.
 */
ScriptKillProcess *ScriptPackage::CreateKillProcessObject(const PSZ p_pszFileName)
    {
    ScriptKillProcess *KillProcessObject = CreateKillProcessObject();

    KillProcessObject->SetFileName(p_pszFileName);
    return KillProcessObject;
    }


/*
 *@@ReturnKillProcessObject:
 *  Returns object which index is given usIndex.
 */
ScriptKillProcess *ScriptPackage::ReturnKillProcessObject(USHORT p_usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->_usKillProcessCount; Index ++)
        if (_pKillProcess[Index].ReturnIndex() == p_usIndex)
            return &_pKillProcess[Index];
    return NULL;
    }


/*
 *@@DeleteKillProcessObject:
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptPackage::DeleteKillProcessObject(USHORT p_usIndex)
    {
    ScriptKillProcess *TempKillProcess = NULL;

    if (_usKillProcessCount > 1)
        {
        TempKillProcess = new ScriptKillProcess[_usKillProcessCount - 1];
        CopyTable(TempKillProcess, _pKillProcess, _usKillProcessCount, p_usIndex);
        if (_bAllowIndexArrangement)
            RebuildIndexes(TempKillProcess, (USHORT)(_usKillProcessCount - 1));
        }
    delete [] _pKillProcess;
    _pKillProcess = TempKillProcess;
    _usKillProcessCount --;
    return 0;
    }


/*
 *@@DeleteKillProcessObject:
 *  Deletes object which address is *ScriptKillProcessObject.
 */
USHORT ScriptPackage::DeleteKillProcessObject(
                      ScriptKillProcess *p_pScriptKillProcessObject
                                             )
    {
    USHORT Index = 0, usDelete;
    USHORT rc;

    if (!p_pScriptKillProcessObject)
        return 1;

    // Check that ScriptKillProcessObject really exist on KillProcess table.
    while (TRUE)
        {
        if (&_pKillProcess[Index] == p_pScriptKillProcessObject)
            break;
        Index ++;
        if (Index >= _usKillProcessCount)
            return 2;
        }

    usDelete = _pKillProcess[Index].ReturnIndex();
    rc = DeleteKillProcessObject(usDelete);
    return rc;
    }


/*
 *@@ReturnKillProcessObjectCount:
 *  Returns count number of ScriptKillProcessObjects.
 */
USHORT ScriptPackage::ReturnKillProcessObjectCount()
    {
    return _usKillProcessCount;
    }


/*
 *@@ ScriptPackage:: BuildFromText
 *  Builds package accordingly to script text.
 */
PSZ ScriptPackage::BuildFromText(PSZ p_pszScriptText,
                                 ULONG p_ulLineNumber)
    {
    Message msg;
    PSZ pszEnd = NULL;
    ULONG ulRc = 0;
    ULONG ulOffsetBeginTag = 0;
    ULONG ulOffsetBeginBlock = 0;
    ULONG ulSearchOffset = 0;
    PSZ pszText = p_pszScriptText;
    PSZ pszBlock = NULL;
    PSZ pszAttrs = NULL;
    PSZ pszTemp = NULL;
    ScriptRequires *pRequires = NULL;
    ScriptRequires TmpRequires;
    ScriptExecute *pExecute = NULL;
    ScriptExecute TmpExecute;
    ScriptClearProfile *pClearProfile = NULL, TmpClearProfile;
    ScriptWriteProfile *pWriteProfile = NULL, TmpWriteProfile;
    ScriptConfigSys *pConfigSys = NULL, TmpConfigSys;
    ScriptKillProcess *pKillProcess = NULL, TmpKillProcess;
    ScriptCreateObject *pCreateObject = NULL, TmpCreateObject;
    ScriptRegisterClass *pRegClass = NULL, TmpRegClass;
    ScriptReplaceClass *pRepClass = NULL, TmpRepClass;




    ulRc = strhGetBlock(pszText, &ulSearchOffset, SZ_TAG_PACKAGE, &pszBlock,
                        &pszAttrs, &ulOffsetBeginTag, &ulOffsetBeginBlock);

    pszEnd = pszText + ulSearchOffset;

    // Check that tag was founded. If not return with error message.
    if (ulRc != 0)
        {
        msg.SetType(MSG_TYPE_INFO);
        msg.SetMessageValue(MSG_TAG_PCK_NOTFOUND);
        msg.SetLineNumber(p_ulLineNumber);
        _messageLogger.LogMessage(msg);
        return NULL;
        }


    SetDescription(pszBlock);

    pszTemp = pszAttrs;
    _Index.BuildFromText(pszTemp, 0);
    _PckID.BuildFromText(pszTemp, 0);
    _Title.BuildFromText(pszTemp, 0);
    _Target.BuildFromText(pszTemp, 0);
    _KeyWords.BuildFromText(pszTemp, 0);


    // Find all REQUIRES attributes.
    pszTemp = pszAttrs;
    do
        {
        // Try to build object from text.
        pszTemp = TmpRequires.BuildFromText(pszTemp, 0);
        // If it succeded, create new object and copy temp object to
        // created object.
        if (pszTemp)
            {
            pRequires = CreateRequiresObject();
            TmpRequires.SetIndex(pRequires->ReturnIndex());
            *pRequires = TmpRequires;
            pRequires->_messageLogger.SetParentLogger(_messageLogger);
            TmpRequires._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);


    // Find all EXECUTE attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpExecute.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pExecute = CreateExecuteObject();
            TmpExecute.SetIndex(pExecute->ReturnIndex());
            *pExecute = TmpExecute;
            pExecute->_messageLogger.SetParentLogger(_messageLogger);
            TmpExecute._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);


    // Find all CLEARPROFILE attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpClearProfile.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pClearProfile = CreateClearProfileObject();
            TmpClearProfile.SetIndex(pClearProfile->ReturnIndex());
            *pClearProfile = TmpClearProfile;
            pClearProfile->_messageLogger.SetParentLogger(_messageLogger);
            TmpClearProfile._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);


    // Find all WRITEPROFILE attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpWriteProfile.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pWriteProfile = CreateWriteProfileObject();
            TmpWriteProfile.SetIndex(pWriteProfile->ReturnIndex());
            *pWriteProfile = TmpWriteProfile;
            pWriteProfile->_messageLogger.SetParentLogger(_messageLogger);
            TmpWriteProfile._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);


    // Find all CONFIGSYS attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpConfigSys.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pConfigSys = CreateConfigSysObject();
            TmpConfigSys.SetIndex(pConfigSys->ReturnIndex());
            *pConfigSys = TmpConfigSys;
            pConfigSys->_messageLogger.SetParentLogger(_messageLogger);
            TmpConfigSys._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);


    // Find all KILLPROCESS attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpKillProcess.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pKillProcess = CreateKillProcessObject();
            TmpKillProcess.SetIndex(pKillProcess->ReturnIndex());
            *pKillProcess = TmpKillProcess;
            pKillProcess->_messageLogger.SetParentLogger(_messageLogger);
            TmpKillProcess._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);

    // Find all CREATEOBJECT attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpCreateObject.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pCreateObject = CreateCreateObjectObject();
            TmpCreateObject.SetIndex(pCreateObject->ReturnIndex());
            *pCreateObject = TmpCreateObject;
            pCreateObject->_messageLogger.SetParentLogger(_messageLogger);
            TmpCreateObject._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);

    // Find all REGISTERCLASS attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpRegClass.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pRegClass = CreateRegisterClassObject();
            TmpRegClass.SetIndex(pRegClass->ReturnIndex());
            *pRegClass = TmpRegClass;
            pRegClass->_messageLogger.SetParentLogger(_messageLogger);
            TmpRegClass._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);

    // Find all REPLACECLASS attributes.
    pszTemp = pszAttrs;
    do
        {
        pszTemp = TmpRepClass.BuildFromText(pszTemp, 0);
        if (pszTemp)
            {
            pRepClass = CreateReplaceClassObject();
            TmpRepClass.SetIndex(pRepClass->ReturnIndex());
            *pRepClass = TmpRepClass;
            pRepClass->_messageLogger.SetParentLogger(_messageLogger);
            TmpRepClass._messageLogger.DeleteLogs();
            }

        } while (pszTemp != NULL);


    BSString message = _messageLogger.ReturnComposedMessageString();


    return pszEnd;
    }






/****************************************************************************
 ****************************************************************************
        ScriptExpanded class:

 ****************************************************************************
 *  ScriptExpanded class member functions:
 *  ScripitExpanded Constructor.
 *  ~ScriptExpanded Destructor.
 *  SetExpanded     Sets EXPANDED keyword.
 *  SetIndex        Sets object ID. Should be used for identification.
 *  ReturnExpanded  Returns EXPANDED attribute state.
 *  ReturnIndex     Returns object ID number.
 *  MakeScriptText  Returns pointer to script text.
 *  BuildFromText   Builds object from text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptExpanded::ScriptExpanded
 *  Constructor for ScriptExpanded class. Initializes object by setting
 *  usIndex to given value on uses default value (1).
 */
ScriptExpanded::ScriptExpanded(USHORT usIndex)
    {
    this->usIndex = usIndex;
    bExpanded = FALSE;
    }




/***************************************************************************
 *@@ ScriptExpanded::~ScriptExpanded
 *  Destructor for ScriptExpanded class.
 */
ScriptExpanded::~ScriptExpanded()
    {
    }



/***************************************************************************
 *@@ ScriptExpanded::SetExpanded
 *  Sets EXPANDED attribute state.
 */
void ScriptExpanded::SetExpanded(BOOL bExpanded)
    {
    this->bExpanded = bExpanded;
    }



/***************************************************************************
 *@@ ScriptExpanded::SetIndex
 *  Sets object's index.
 */
void ScriptExpanded::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptExpanded::ReturnExpanded
 *  Returns bExpanded state.
 */
BOOL ScriptExpanded::ReturnExpanded()
    {
    return bExpanded;
    }



/***************************************************************************
 *@@ ScriptExpanded::ReturnIndex
 *  Returns index of current object.
 */
USHORT ScriptExpanded::ReturnIndex()
    {
    return usIndex;
    }



/***************************************************************************
 *@@ ScriptExpandeds::MakeScriptText
 *  Returns pointer to script text.
 */
PSZ ScriptExpanded::MakeScriptText(BOOL bAttribute)
    {
    ULONG ulTextLength = 0;
    ULONG ulSpaces = 0;
    PSZ pszScriptText = NULL;

    if (bExpanded)
        ulTextLength += strlen(SZ_ATTRIBUTE_EXPANDED);

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bExpanded)
        strcat(pszScriptText, SZ_ATTRIBUTE_EXPANDED);

    return pszScriptText;
    }



/***************************************************************************
 *@@ScriptExpanded::BuildFromText
 *  Builds keyword object accordingly to script text. If any errors is found
 *  in script text, error informations are returned in table where to
 *  *pScriptError points. Pointer to after end of tag is returned so multiple
 *  searches can be done.
 */
PSZ ScriptExpanded::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    SCRIPTERROR Error;
    PSZ pszExpanded = NULL;


    pszExpanded = strhFindWord(pszScriptText, SZ_ATTRIBUTE_EXPANDED, "\x0d\x0a ()/\\-,.",
                              "\x0d\x0a ()/\\-,.:;");
    if (pszExpanded)
        this->SetExpanded(TRUE);
    else
        this->SetExpanded(FALSE);

    return pszScriptText;
    }







/****************************************************************************
 ****************************************************************************
        ScriptGroup class:

 ****************************************************************************
 *  ScriptGroup class member functions:
 *  ScriptGroup             Constructor.
 *  ~ScriptGroup            Destructor.
 *  SetIndex                Sets index number.
 *  ReturnIndex             Returns index number.
 *  SetTitleText            Sets group's title text.
 *  ReturnTitleText         Returns a pointer to title text.
 *  SetExpanded             Sets expanded attribute's state.
 *  ReturnExpanded          Returns expanded attribute's state.
 *  CreatePackageObject     Creates a new ScriptPackage object.
 *  ReturnPackageObject     Returns an object which index is given as parameter.
 *  DeletePackageObject     Deletes ScriptPackage object.
 *  DeleteAllPackageObjects Deletes all ScriptPackage objects.
 *  MakeScriptText          Returns a pointer to script text.
 *  BuildFromText
 *  AddSubGroupObject
 *  ReturnSubGroupObject
 *  DeleteSubGroupObject
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptGroup::ScriptGroup
 *  Constructor for ScriptGroup class.
 */
ScriptGroup::ScriptGroup(USHORT usIndex, BOOL bVirtualGroup,
                         BOOL bAllowIndexArrangement)
    {
    this->usIndex = usIndex;

    this->bVirtualGroup = bVirtualGroup;

    this->bAllowIndexArrangement = bAllowIndexArrangement;

    this->pSubGroup = NULL;
    this->pPackage = NULL;
    this->usSubGroupCount = 0;
    this->usPackageCount = 0;
    }




/***************************************************************************
 *@@ ScriptGroup::ScriptGroup
 *  Constructor for ScriptGroup class.
 */
ScriptGroup::ScriptGroup(ScriptGroup &Group)
    {
    (*this) = Group;
    }


/***************************************************************************
 *@@ ScriptGroup::~ScriptGroup
 *  Destructor for ScriptGroup class.
 *
 *@@ changed v0.9.5 (2000-08-22) [tahola]: clearning of the fields are moved
 *                                         to ClearAll() function.
 */
ScriptGroup::~ScriptGroup()
    {
    ClearAll();
    }


/*
 *@@ ClearAll
 *  Clears all fields.
 *
 *@@added v.0.9.5 (2000-08-22) [tahola]
 */
USHORT ScriptGroup::ClearAll()
    {
    this->usIndex = 1;

    this->bVirtualGroup = FALSE;

    this->bAllowIndexArrangement = TRUE;

    if (pSubGroup)
        delete [] this->pSubGroup;
    this->pSubGroup = NULL;
    if (pPackage)
        delete [] this->pPackage;
    this->pPackage = NULL;
    this->usSubGroupCount = 0;
    this->usPackageCount = 0;
    SetTitle("");
    SetExpanded(FALSE);
    return (0);     // V0.9.14 (2001-07-07) [umoeller]
    }


/***************************************************************************
 *@@ ScriptGroup::ScriptGroup &operator=
 *  Assignment operator for ScriptGroup class.
 */
ScriptGroup &ScriptGroup::operator=(ScriptGroup &Group)
    {
    USHORT usIndex = 0;
    ScriptPackage *pPck = NULL;
    ScriptGroup *pGrp = NULL;

    this->usIndex = Group.usIndex;

    this->bVirtualGroup = Group.bVirtualGroup;

    this->Title = Group.Title;

    this->Expanded = Group.Expanded;

    this->pSubGroup = NULL;
    this->usSubGroupCount = 0;

    this->pPackage = NULL;
    this->usPackageCount = 0;

    if (Group.pSubGroup)
        {
        pGrp = new ScriptGroup[Group.usSubGroupCount];
        this->usSubGroupCount = Group.usSubGroupCount;
        for (usIndex = 0; usIndex < Group.usSubGroupCount; usIndex ++)
            pGrp[usIndex] = Group.pSubGroup[usIndex];
        this->pSubGroup = pGrp;
        }

    if (Group.pPackage)
        {
        pPck = new ScriptPackage[Group.usPackageCount];
        this->usPackageCount = Group.usPackageCount;
        for (usIndex = 0; usIndex < Group.usPackageCount; usIndex ++)
            pPck[usIndex] = Group.pPackage[usIndex];
        this->pPackage = pPck;
        }
    return (*this);
    }



/***************************************************************************
 *@@ ScriptGroup::SetVirtual
 */
void ScriptGroup::SetVirtual(BOOL bVirtual)
    {
    this->bVirtualGroup = bVirtual;
    }




/***************************************************************************
 *@@ ScriptGroup::SetIndex
 *  Sets usIndex number.
 */

void ScriptGroup::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptGroup::ReturnIndex
 *  Returns usIndex number.
 */

USHORT ScriptGroup::ReturnIndex()
    {
    return usIndex;
    }


/***************************************************************************
 *@@ ScriptGroup::SetTitle
 */
void ScriptGroup::SetTitle(const PSZ pszTitle)
    {
    Title.SetTitle(pszTitle);
    }


/***************************************************************************
 *@@ ScriptGroup::SetExpanded
 */
void ScriptGroup::SetExpanded(BOOL bExpanded)
    {
    Expanded.SetExpanded(bExpanded);
    }



/***************************************************************************
 *@@ ScriptGroup::ReturnTitle
 */
const PSZ ScriptGroup::ReturnTitle()
    {
    return Title.ReturnTitle();
    }



/***************************************************************************
 *@@ ScriptGroup::MakeScriptText
 *  Returns pointer to newly allocated script text.
 */
PSZ ScriptGroup::MakeScriptText(BOOL bTag)
    {
    XSTRING strScript;
    xstrInit(&strScript, 0);
    PSZ pszTemp = NULL;
    CHAR szTag[MAXTAGLENGTH];
    CHAR szAttribute[MAXATTRIBUTELENGTH];
    USHORT usIndex;

    if (!bVirtualGroup)
        {
        if (bTag)
            {
            sprintf(szTag, "<%s ", SZ_TAG_GROUP);
            xstrcat(&strScript, szTag, 0);
            }

        // Add TITLE attribute.
        xstrcat(&strScript, " ", 0);
        pszTemp = Title.MakeScriptText(bTag);
        xstrcat(&strScript, pszTemp, 0);
        delete [] pszTemp;
        // Add EXPANDED attribute.
        xstrcat(&strScript, " ", 0);
        pszTemp = Expanded.MakeScriptText(bTag);
        xstrcat(&strScript, pszTemp, 0);
        delete [] pszTemp;

        if (bTag)
            xstrcat(&strScript, ">", 0);
        xstrcat(&strScript, "\n", 0);
        }

        // Add PACKAGE attributes.
    for (usIndex = 0; usIndex < usPackageCount; usIndex ++)
        {
        pszTemp = pPackage[usIndex].MakeScriptText(bTag);
        if (!bVirtualGroup)
            AddCharacterAfterEOL(&pszTemp, '\t');
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    // Add GROUP attributes.
    for (usIndex = 0; usIndex < usSubGroupCount; usIndex ++)
        {
        pszTemp = pSubGroup[usIndex].MakeScriptText(bTag);
        if (!bVirtualGroup)
            AddCharacterAfterEOL(&pszTemp, '\t');
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    if (bTag && !bVirtualGroup)
        {
        sprintf(szTag, "<%s>", SZ_TAG_ENDGROUP);
        xstrcat(&strScript, szTag, 0);
        }

    return strScript.psz;
    }







/****************************************
 *  ScriptGroup object handling.
 ****************************************
 * Here are functions when handling ScriptGroup objects.
 */

/***************************************************************************
 *@@ ScriptGroup::CopyTable
 *  Copies source table to destination table.
 */
ScriptGroup *ScriptGroup::CopyTable(ScriptGroup *DestTable, // in: Pointer to destination table.
                                    ScriptGroup *SrcTable, // in: Pointer to source table.
                                    USHORT usSize, // in: Size of source table.
                                    USHORT usSkipIndex) // in: Index to be skipped. If 0 no skipping.
    {
    USHORT SrcIndex, DestIndex;
    if (!DestTable)
        return NULL;
    if (!SrcTable)
        return DestTable;
        for (SrcIndex = 0, DestIndex = 0; SrcIndex < usSize; SrcIndex ++, DestIndex ++)
        {
        if (SrcTable[SrcIndex].ReturnIndex() == usSkipIndex)
            if (SrcIndex < usSize - 1)
                SrcIndex ++;
            else
                break;
        DestTable[DestIndex] = SrcTable[SrcIndex];
        }

    return DestTable;
    }




/***************************************************************************
 *@@ ScriptGroup::ReBuildIndexes
 *  Rebuilds indexes after deletion. The order of the indexes does not matter.
 *  Example:
 *      Old indexing: 1, 4, 3, 5
 *      New indexing: 1, 3, 2, 4
 */
void ScriptGroup::ReBuildIndexes(ScriptGroup *Table, USHORT usTableSize)
    {
    USHORT Index, Index2, Index3, usMissing, usMax = 0, i;
    BOOL bFinded = FALSE, bReady = FALSE;

    // First find largest index number.
    for (Index = 0; Index < usTableSize; Index ++)
        if (Table[Index].ReturnIndex() > usMax)
            usMax = Table[Index].ReturnIndex();

    // Do this until it is sure that there is no skipped indexes left.
    while (!bReady)
        {
        bReady = TRUE;
        // Go through all index values from 1 to usMax.
        for (Index = 1; Index <= usMax; Index ++)
            {
            // Check that Index number will be found.
            for (Index2 = 0; Index2 < usTableSize; Index2 ++)
                if (Index == Table[Index2].ReturnIndex())
                    bFinded = TRUE;
            // If did not found, sub 1 from every index number above the index number
            // which did not found.
            if (bFinded == FALSE)
                {
                for (Index3 = Index + 1; Index3 <= usMax; Index3 ++)
                    for (Index2 = 0; Index2 < usTableSize; Index2 ++)
                        if (Index3 == Table[Index2].ReturnIndex())
                            Table[Index2].SetIndex(Index3 - 1);
                usMax --;
                }
            if (bFinded == FALSE)
                bReady = FALSE;
            bFinded = FALSE;
            }
        }
    }


/***************************************************************************
 *@@ ScriptGroup::FindUnusedIndex
 *  Returns first unused index number.
 */
USHORT ScriptGroup::FindUnusedIndex(ScriptGroup *Table, USHORT usTableSize)
    {
    USHORT Index, Index2, usMax = 0;
    BOOL bFinded = FALSE;

    // First find largest index number.
    for (Index = 0; Index < usTableSize; Index ++)
        if (Table[Index].ReturnIndex() > usMax)
            usMax = Table[Index].ReturnIndex();

    for (Index = 1; Index <= usMax; Index ++)
        {
        // Go through all the indexes of the table.
        for (Index2 = 0; Index2 < usTableSize; Index2 ++)
            if (Index == Table[Index2].ReturnIndex())
                {
                bFinded = TRUE;
                break;
                }
        // If Index found from the table, continue.
        if (bFinded == TRUE)
            bFinded = FALSE;
        // If not break.
        else
            break;
        }

    return Index;
    }




/***************************************************************************
 *@@ ScriptGroup::CreateSubGroupObject
 *  Creates new Group object and returns pointer to it.
 */
ScriptGroup *ScriptGroup::CreateSubGroupObject()
    {
    ScriptGroup *Object, *NewObject;

    if (!(Object = new ScriptGroup[++usSubGroupCount]))
        {
        usSubGroupCount --;
        return NULL;
        }

    Object[usSubGroupCount - 1].SetIndex(FindUnusedIndex(this->pSubGroup,
        usSubGroupCount - 1));

    CopyTable(Object, this->pSubGroup, usSubGroupCount - 1);
    delete [] this->pSubGroup;
    this->pSubGroup = Object;
    NewObject = &this->pSubGroup[usSubGroupCount - 1];
    return NewObject;
    }



/***************************************************************************
 *@@ ScriptGroup::ReturnSubGroupObject
 *  Returns object which index is given usIndex.
 */
ScriptGroup *ScriptGroup::ReturnSubGroupObject(USHORT usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->usSubGroupCount; Index ++)
        if (pSubGroup[Index].ReturnIndex() == usIndex)
            return &pSubGroup[Index];
    return NULL;
    }




/***************************************************************************
 *@@ ScriptGroup::DeleteSubGroupObject
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptGroup::DeleteSubGroupObject(USHORT usIndex)
    {
    ScriptGroup *TempGroup = NULL;

    if (usSubGroupCount > 1)
        {
        TempGroup = new ScriptGroup[usSubGroupCount - 1];
        CopyTable(TempGroup, pSubGroup, usSubGroupCount, usIndex);
        if (bAllowIndexArrangement)
            ReBuildIndexes(TempGroup, usSubGroupCount - 1);
        }
    delete [] pSubGroup;
    pSubGroup = TempGroup;
    usSubGroupCount --;
    return 0;
    }



/***************************************************************************
 *@@ ScriptGroup:: ReturnSubGroupObjectCount
 *  Returns count number of ScriptGroup objects.
 */
USHORT ScriptGroup::ReturnSubGroupObjectCount()
    {
    return usSubGroupCount;
    }


/***************************************************************************
 *@@ ScriptGroup::MoveSubGroupObjects
 *  Note: destination group must not be one of the source group objects
 *  otherwice this could lead to problems because source group objects will
 *  be deleted after copy operation.
 */
USHORT ScriptGroup::MoveSubGroupObjects(ScriptGroup *pDestGroup)
    {
    USHORT usIndex;
    ScriptGroup *pTempGroup = NULL;

    for (usIndex = 0; usIndex < usSubGroupCount; usIndex ++)
        {
        pTempGroup = pDestGroup->CreateSubGroupObject();
        pSubGroup[usIndex].SetIndex(pTempGroup->ReturnIndex());
        *pTempGroup = pSubGroup[usIndex];
        }

    usSubGroupCount = 0;
    delete [] pSubGroup;
    pSubGroup = NULL;
    return 0;
    }




/****************************************
 *  ScriptPackage object handling.
 ****************************************
 * Here are functions when handling ScriptPackage objects.
 */

/***************************************************************************
 *@@ ScriptGroup::CopyTable
 *  Copies source table to destination table.
 */
ScriptPackage *ScriptGroup::CopyTable(ScriptPackage *DestTable, // in: Pointer to destination table.
                                ScriptPackage *SrcTable, // in: Pointer to source table.
                                USHORT usSize, // in: Size of source table.
                                USHORT usSkipIndex) // in: Index to be skipped. If 0, no skipping.
    {
    USHORT SrcIndex, DestIndex;
    if (!DestTable)
        return NULL;
    if (!SrcTable)
        return DestTable;
        for (SrcIndex = 0, DestIndex = 0; SrcIndex < usSize; SrcIndex ++, DestIndex ++)
        {
        if (SrcTable[SrcIndex].ReturnIndex() == usSkipIndex)
            if (SrcIndex < usSize - 1)
                SrcIndex ++;
            else
                break;
        DestTable[DestIndex] = SrcTable[SrcIndex];
        }

    return DestTable;
    }




/***************************************************************************
 *@@ ScriptGroup::ReBuildIndexes
 *  Rebuilds indexes after deletion. The order of the indexes does not matter.
 *  Example:
 *      Old indexing: 1, 4, 3, 5
 *      New indexing: 1, 3, 2, 4
 */
void ScriptGroup::ReBuildIndexes(ScriptPackage *Table, USHORT usTableSize)
    {
    USHORT Index, Index2, Index3, usMissing, usMax = 0, i;
    BOOL bFinded = FALSE, bReady = FALSE;

    // First find largest index number.
    for (Index = 0; Index < usTableSize; Index ++)
        if (Table[Index].ReturnIndex() > usMax)
            usMax = Table[Index].ReturnIndex();

    // Do this until it is sure that there is no skipped indexes left.
    while (!bReady)
        {
        bReady = TRUE;
        // Go through all index values from 1 to usMax.
        for (Index = 1; Index <= usMax; Index ++)
            {
            // Check that Index number will be found.
            for (Index2 = 0; Index2 < usTableSize; Index2 ++)
                if (Index == Table[Index2].ReturnIndex())
                    bFinded = TRUE;
            // If did not found, decrease 1 from every index number above the index number
            // which did not found.
            if (bFinded == FALSE)
                {
                for (Index3 = Index + 1; Index3 <= usMax; Index3 ++)
                    for (Index2 = 0; Index2 < usTableSize; Index2 ++)
                        if (Index3 == Table[Index2].ReturnIndex())
                            Table[Index2].SetIndex(Index3 - 1);
                usMax --;
                }
            if (bFinded == FALSE)
                bReady = FALSE;
            bFinded = FALSE;
            }
        }
    }


/***************************************************************************
 *@@ ScriptGroup::FindUnusedIndex
 *  Returns first unused index number.
 */
USHORT ScriptGroup::FindUnusedIndex(ScriptPackage *Table, USHORT usTableSize)
    {
    USHORT Index, Index2, usMax = 0;
    BOOL bFinded = FALSE;

    // First find largest index number.
    for (Index = 0; Index < usTableSize; Index ++)
        if (Table[Index].ReturnIndex() > usMax)
            usMax = Table[Index].ReturnIndex();

    for (Index = 1; Index <= usMax; Index ++)
        {
        // Go through all the indexes of the table.
        for (Index2 = 0; Index2 < usTableSize; Index2 ++)
            if (Index == Table[Index2].ReturnIndex())
                {
                bFinded = TRUE;
                break;
                }
        // If Index found from the table, continue.
        if (bFinded == TRUE)
            bFinded = FALSE;
        // If not break.
        else
            break;
        }

    return Index;
    }




/***************************************************************************
 *@@ ScriptGroup::CreatePackageObject
 *  Creates new Package object and returns pointer to it.
 */
ScriptPackage *ScriptGroup::CreatePackageObject()
    {
    ScriptPackage *Object, *NewObject;

    if (!(Object = new ScriptPackage[++usPackageCount]))
        {
        usPackageCount --;
        return NULL;
        }

    Object[usPackageCount - 1].SetIndex(FindUnusedIndex(this->pPackage,
        usPackageCount - 1));

    CopyTable(Object, this->pPackage, usPackageCount - 1);
    delete [] this->pPackage;
    this->pPackage = Object;
    NewObject = &this->pPackage[usPackageCount - 1];
    return NewObject;
    }



/***************************************************************************
 *@@ ScriptGroup::CreatePackageObject
 *  Creates new Package object and returns pointer to it. The usIndex is not
 *  changed.
 */
ScriptPackage *ScriptGroup::CreatePackageObject(ScriptPackage *pPck)
    {
    ScriptPackage *pTempPck = CreatePackageObject();
    USHORT usTmpIndex = pTempPck->ReturnIndex();
    *pTempPck = *pPck;
    pTempPck->SetIndex(usTmpIndex);
    return pTempPck;
    }




/***************************************************************************
 *@@ ScriptGroup::ReturnPackageObject
 *  Returns object which index is given usIndex.
 */
ScriptPackage *ScriptGroup::ReturnPackageObject(USHORT usIndex)
    {
    USHORT Index;

    for (Index = 0; Index < this->usPackageCount; Index ++)
        if (pPackage[Index].ReturnIndex() == usIndex)
            return &pPackage[Index];
    return NULL;
    }




/***************************************************************************
 *@@ ScriptGroup::DeletePackageObject
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptGroup::DeletePackageObject(USHORT usIndex)
    {
    ScriptPackage *TempPackage = NULL;

    if (usPackageCount > 1)
        {
        TempPackage = new ScriptPackage[usPackageCount - 1];
        CopyTable(TempPackage, pPackage, usPackageCount, usIndex);
        if (bAllowIndexArrangement)
            ReBuildIndexes(TempPackage, usPackageCount - 1);
        }
    delete [] pPackage;
    pPackage = TempPackage;
    usPackageCount --;
    return 0;
    }



/***************************************************************************
 *@@ ScriptGroup:: ReturnPackageObjectCount
 *  Returns count number of ScriptPackage objects.
 */
USHORT ScriptGroup::ReturnPackageObjectCount()
    {
    return usPackageCount;
    }




/***************************************************************************
 *@@ ScriptGroup::DeleteAllPackageObjects
 *  Deletes all ScriptPackage objects.
 */
USHORT ScriptGroup::DeleteAllPackageObjects()
    {
    if (pPackage)
        delete [] pPackage;
    pPackage = NULL;
    usPackageCount = 0;
    return 0;
    }



/***************************************************************************
 *@@ ScriptGroup::CopyPackageObjectToGroup
 *  Copies ScriptPackage object pPackageObject to sub group which index
 *  is usIndex.
 */
USHORT ScriptGroup::CopyPackageObjectToGroup(ScriptPackage *pPackageObject,
                                            USHORT usIndex)
    {
    ScriptPackage *pPck = NULL;
    ScriptGroup *pSubGroup = ReturnSubGroupObject(usIndex);
    if ((!pSubGroup) || (!pPackageObject))
        return 1;
    pPck = pSubGroup->CreatePackageObject();
    *pPck = *pPackageObject;
    return 0;
    }



/***************************************************************************
 *@@ ScriptGroup::MovePackageObjectToGroup
 *  Moves ScriptPackage object to sub group. Both package and sub group must
 *  be in the same ScriptGroup object.
 */
USHORT ScriptGroup::MovePackageObjectToGroup(USHORT usPckIndex, // The index of the package.
                                            USHORT usGrpIndex // The index of the group.
                                            )
    {
    ScriptPackage *pPackage = ReturnPackageObject(usPckIndex);
    ScriptGroup *pSubGroup = ReturnSubGroupObject(usGrpIndex);
    ScriptPackage *pPck = NULL;
    if ((!pSubGroup) || (!pPackage))
        return 1;
    pPck = pSubGroup->CreatePackageObject();
    *pPck = *pPackage;
    DeletePackageObject(usPckIndex);
    return 0;
    }



/***************************************************************************
 *@@ ScriptGroup::MovePackageObjectToGroup
 *  Moves ScriptPackage object to sub group.
 */
USHORT ScriptGroup::MovePackageObjectToGroup(USHORT usPckIndex, // The index of the package.
                                            ScriptGroup *pDestGroupObj // The index of the group.
                                            )
    {
    ScriptPackage *pPackage = ReturnPackageObject(usPckIndex);
    ScriptPackage *pPck = NULL;
    if ((!pDestGroupObj) || (!pPackage))
        return 1;
    pPck = pDestGroupObj->CreatePackageObject();
    *pPck = *pPackage;
    DeletePackageObject(usPckIndex);
    return 0;
    }




/***************************************************************************
 *@@ ScriptGroup::BuildFromText
 *  Builds package accordingly to script text.
 */
PSZ ScriptGroup::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    CHAR szPckTag[256], szGrpTag[256];
    PSZ pszEnd = NULL;
    ULONG ulRc = 0;
    ULONG ulOffsetBeginTag = 0;
    ULONG ulOffsetBeginBlock = 0;
    ULONG ulSearchOffset = 0;
    PSZ pszText = pszScriptText;
    PSZ pszBlock = NULL;
    PSZ pszAttrs = NULL;
    PSZ pszTemp = NULL;
    PSZ pszGroup = NULL;
    PSZ pszPackage = NULL;
    ScriptPackage *pPackage = NULL, TmpPackage;
    ScriptGroup *pGroup = NULL, TmpGroup;

    sprintf(szPckTag, "<%s", SZ_TAG_PACKAGE);
    sprintf(szGrpTag, "<%s", SZ_TAG_GROUP);


    // Check that pointers are not NULL.
    if (!pszScriptText)
        {
        return NULL;
        }

    // If this is not virtual group, get the attribute block between starting and ending
    // tags of the GROUP tag.
    if (!bVirtualGroup)
        {
        ulRc = strhGetBlock(pszText, &ulSearchOffset, SZ_TAG_GROUP, &pszBlock,
                            &pszAttrs, &ulOffsetBeginTag, &ulOffsetBeginBlock);
        pszTemp = Title.BuildFromText(pszAttrs, 0);
        pszTemp = Expanded.BuildFromText(pszAttrs, NULL, NULL, 0);
        }
    else
        pszBlock = pszScriptText;

    while (NULL != (pszPackage = strhistr(pszBlock, szPckTag)))
        {
        // Check if there is a GROUP tag in the block.
        pszGroup = strhistr(pszBlock, szGrpTag);
        // If there is a group before package.
        if ((pszGroup) && (pszGroup < pszPackage))
            {
            // Clear TmpGroup before build it from the text.
            TmpGroup.ClearAll();
            pszBlock = TmpGroup.BuildFromText(pszBlock, NULL, NULL, 0);
            if (pszBlock)
                {
                pGroup = CreateSubGroupObject();
                TmpGroup.SetIndex(pGroup->ReturnIndex());
                *pGroup = TmpGroup;
                }
            }
        // There is package before group.
        else
            {
            // Clear TmpPackage before build it from the text by calling
            // destructor.
            TmpPackage.ClearAll();
            pszBlock = TmpPackage.BuildFromText(pszBlock, 0);
            if (pszBlock)
                {
                pPackage = CreatePackageObject();
                TmpPackage.SetIndex(pPackage->ReturnIndex());
                *pPackage = TmpPackage;
                TmpPackage._messageLogger.DeleteLogs();
                }
            }
        }
    if (bVirtualGroup)
        pszEnd = pszScriptText + strlen(pszScriptText);
    else
        pszEnd = pszScriptText + ulSearchOffset;
    delete [] pszAttrs;
    return pszEnd;
    }





/****************************************************************************
 ****************************************************************************
        ScriptTitleTag class:

 ****************************************************************************
 *  ScriptTitleTag class member functions:
 *  ScripitTitleTag    Constructor.
 *  ~ScriptTitleTag    Destructor.
 *  SetTitle        Sets title text.
 *  SetIndex        Sets title object ID. Should be used for identification.
 *  ReturnTitle     Returns const pointer to title text.
 *  ReturnIndex     Returns object ID number.
 *  MakeScriptText  Returns pointer to script text.
 *  BuildFromText   Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptTitleTag::ScriptTitleTag
 *  Constructor for ScriptTitleTag class. Initializes szTitle string by setting
 *  first char to '\0'.
 */
ScriptTitleTag::ScriptTitleTag(USHORT usIndex)
    {
    *szTitle = '\0';
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptTitleTag::~ScriptTitleTag
 *  destructor for ScriptTitleTag class. Does nothing special.
 */

ScriptTitleTag::~ScriptTitleTag()
    {
    *szTitle = '\0';
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptTitleTag::SetTitle
 *  Sets title text.
 */

void ScriptTitleTag::SetTitle(const PSZ pszTitle)
    {
    if (strlen(szTitle) < MAXTITLELENGTH)
        strcpy(this->szTitle, pszTitle);
    }



/***************************************************************************
 *@@ ScriptTitleTag::SetIndex
 *  Sets title text.
 */

void ScriptTitleTag::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptTitleTag::ReturnTitle
 *  Returns const pointer to title text.
 */

const PSZ ScriptTitleTag::ReturnTitle()
    {
    return szTitle;
    }



/***************************************************************************
 *@@ ScriptTitleTag::ReturnIndex
 *  Sets title text.
 */

USHORT ScriptTitleTag::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptTitleTag::MakeScriptText
 *  Returns pointer to script text.
 */
PSZ ScriptTitleTag::MakeScriptText(BOOL bTag)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    ulTextLength = strlen(szTitle);
    CHAR szTag[128];
    CHAR szEndTag[128];

    if (bTag)
        {
        // The length of the attribute, <, and >.
        ulTextLength += strlen(SZ_TAG_TITLE) + 2;
        sprintf(szTag, "<%s>", SZ_TAG_TITLE);
        // The length of the ending attribute, < and >.
        ulTextLength += strlen(SZ_TAG_ENDTITLE) + 2;
        sprintf(szEndTag, "<%s>", SZ_TAG_ENDTITLE);
        }

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bTag)
        sprintf(pszScriptText, "%s%s%s", szTag, szTitle, szEndTag);
    else
        sprintf(pszScriptText, "%s", szTitle);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptTitleTag::BuildFromText
 *  Builds package ID accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptTitleTag::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    SCRIPTERROR Error;
    PSZ pszBlock = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulAttrLen = 0;
    ULONG ulRc = 0;


    ulRc = strhGetBlock(pszScriptText, &ulSearchOffset, SZ_TAG_TITLE, &pszBlock,
                        NULL, NULL, NULL);
    if (pszBlock)
        this->SetTitle(pszBlock);
    else
        return NULL;

    ulAttrLen = strlen(pszBlock);
    delete [] pszBlock;
    return pszScriptText + ulSearchOffset + ulAttrLen + 1;
    }




/****************************************************************************
 ****************************************************************************
        ScriptMessageTag class:

 ****************************************************************************
 *  ScriptMessageTag class member functions:
 *  ScripitMessageTag    Constructor.
 *  ~ScriptMessageTag    Destructor.
 *  SetMessage        Sets title text.
 *  SetIndex        Sets title object ID. Should be used for identification.
 *  ReturnMessage     Returns const pointer to title text.
 *  ReturnIndex     Returns object ID number.
 *  MakeScriptText  Returns pointer to script text.
 *  BuildFromText   Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptMessageTag::ScriptMessageTag
 *  Constructor for ScriptMessageTag class. Initializes szMessage string by setting
 *  first char to '\0'.
 */
ScriptMessageTag::ScriptMessageTag(USHORT usIndex)
    {
    *szMessage = '\0';
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptMessageTag::~ScriptMessageTag
 *  destructor for ScriptMessageTag class. Does nothing special.
 */

ScriptMessageTag::~ScriptMessageTag()
    {
    *szMessage = '\0';
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptMessageTag::SetMessage
 *  Sets title text.
 */

void ScriptMessageTag::SetMessage(const PSZ pszMessage)
    {
    if (strlen(szMessage) < MAXMESSAGELENGTH)
        strcpy(this->szMessage, pszMessage);
    }



/***************************************************************************
 *@@ ScriptMessageTag::SetIndex
 *  Sets title text.
 */

void ScriptMessageTag::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptMessageTag::ReturnMessage
 *  Returns const pointer to title text.
 */

const PSZ ScriptMessageTag::ReturnMessage()
    {
    return szMessage;
    }



/***************************************************************************
 *@@ ScriptMessageTag::ReturnIndex
 *  Sets title text.
 */

USHORT ScriptMessageTag::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptMessageTag::MakeScriptText
 *  Returns pointer to script text.
 */
PSZ ScriptMessageTag::MakeScriptText(BOOL bTag)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    ulTextLength = strlen(szMessage);
    CHAR szTag[128];
    CHAR szEndTag[128];

    if (strlen(szMessage) == 0)
        return NULL;

    if (bTag)
        {
        // The length of the attribute, <, and >.
        ulTextLength += strlen(SZ_TAG_MSG) + 2;
        sprintf(szTag, "<%s>", SZ_TAG_MSG);
        // The length of the ending attribute, < and >.
        ulTextLength += strlen(SZ_TAG_ENDMSG) + 2;
        sprintf(szEndTag, "<%s>", SZ_TAG_ENDMSG);
        }

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bTag)
        sprintf(pszScriptText, "%s%s%s", szTag, szMessage, szEndTag);
    else
        sprintf(pszScriptText, "%s", szMessage);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptMessageTag::BuildFromText
 *  Builds message tag accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptMessageTag::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    SCRIPTERROR Error;
    PSZ pszBlock = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulAttrLen = 0;
    ULONG ulRc = 0;


    ulRc = strhGetBlock(pszScriptText, &ulSearchOffset, SZ_TAG_MSG, &pszBlock,
                        NULL, NULL, NULL);
    if (pszBlock)
        this->SetMessage(pszBlock);
    else
        return NULL;

    ulAttrLen = strlen(pszBlock);
    delete [] pszBlock;
    return pszScriptText + ulSearchOffset + ulAttrLen + 1;
    }






/****************************************************************************
 ****************************************************************************
        ScriptHead class:

 ****************************************************************************
 *  ScriptHead class member functions:
 *  ScriptHead              Constructor.
 *  ~ScriptHead             Destructor.
 *  SetIndex                Sets index number.
 *  ReturnIndex             Returns index number.
 *  SetTitleTagText         Sets title tag text.
 *  ReturnTitleTagText      Returns a pointer to title tag text.
 *  SetMessageTagText       Sets message tag text.
 *  ReturnMessageTagText    Returns a pointer to message tag text.
 *  MakeScriptText          Returns a pointer to script text.
 *  BuildFromText
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptGroup::ScriptGroup
 *  Constructor for ScriptGroup class.
 */
ScriptHead::ScriptHead(USHORT usIndex)
    {
    this->usIndex = usIndex;
    VirtualGroup.SetVirtual(TRUE);
    }




/***************************************************************************
 *@@ ScriptHead::~ScriptHead
 *  Destructor for ScriptHead class.
 */
ScriptHead::~ScriptHead()
    {
    this->usIndex = 1;
    SetTitleTagText("");
    SetMessageTagText("");
    }


/***************************************************************************
 *@@ ScriptHead::SetTitleTagText
 */
void ScriptHead::SetTitleTagText(const PSZ pszTitle)
    {
    Title.SetTitle(pszTitle);
    }


/***************************************************************************
 *@@ ScriptHead::ReturnTitleTagText
 */
const PSZ ScriptHead::ReturnTitleTagText()
    {
    return Title.ReturnTitle();
    }


/***************************************************************************
 *@@ ScriptHead::ReturnTitleTagObject
 */
ScriptTitleTag *ScriptHead::ReturnTitleTagObject()
    {
    return &Title;
    }



/***************************************************************************
 *@@ ScriptHead::SetMessageTagText
 */
void ScriptHead::SetMessageTagText(const PSZ pszMessage)
    {
    Message.SetMessage(pszMessage);
    }


/***************************************************************************
 *@@ ScriptHead::ReturnMessageTagText
 */
const PSZ ScriptHead::ReturnMessageTagText()
    {
    return Message.ReturnMessage();
    }


/***************************************************************************
 *@@ ScriptHead::ReturnMessageTagObject
 */
ScriptMessageTag *ScriptHead::ReturnMessageTagObject()
    {
    return &Message;
    }


/***************************************************************************
 *@@ ScriptHead::ReturnVirtualGroupObject
 */
ScriptGroup *ScriptHead::ReturnVirtualGroupObject()
    {
    return &VirtualGroup;
    }



/***************************************************************************
 *@@ ScriptHead::MakeScriptText
 *  Returns pointer to newly allocated script text.
 */
PSZ ScriptHead::MakeScriptText(BOOL bTag)
    {
    XSTRING strScript;
    xstrInit(&strScript, 0);
    PSZ pszTemp = NULL;
    CHAR szTag[MAXTAGLENGTH];
    CHAR szAttribute[MAXATTRIBUTELENGTH];
    USHORT usIndex;

    if (bTag)
       {
       sprintf(szTag, "<%s>\n", SZ_TAG_HEAD);
       xstrcat(&strScript, szTag, 0);
       }

    // Add TITLE tag.
    xstrcat(&strScript, "  ", 0);
    pszTemp = Title.MakeScriptText(bTag);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    // Add MESSAGE tag.
    xstrcat(&strScript, "\n  ", 0);
    pszTemp = Message.MakeScriptText(bTag);
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;

    // Add GROUP tag. This is actually a virtual group so it won't produce
    // any text but if it owns packages or groups, their text will be written.
    pszTemp = VirtualGroup.MakeScriptText(bTag);
    if (pszTemp)
        AddCharacterAfterEOL(&pszTemp, '\t');
    xstrcat(&strScript, "\n", 0);
    xstrcat(&strScript, pszTemp, 0);
    xstrcat(&strScript, "\n", 0);
    delete [] pszTemp;

    if (bTag)
        {
        sprintf(szTag, "<%s>", SZ_TAG_ENDHEAD);
        xstrcat(&strScript, szTag, 0);
        }

    return strScript.psz;
    }





/***************************************************************************
 *@@ ScriptHead:: BuildFromText
 *  Builds head accordingly to script text.
 *  Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptHead::BuildFromText(PSZ pszScriptText,
                              SCRIPTERROR **ppScriptError,
                              USHORT *pusErrorTableSize,
                              ULONG ulLineNumber)
    {
    PSZ pszEnd = NULL;
    ULONG ulRc = 0;
    ULONG ulOffsetBeginBlock = 0;
    ULONG ulSearchOffset = 0;
    PSZ pszText = pszScriptText;
    PSZ pszBlock = NULL;
    PSZ pszTemp = NULL;

    // Check that pointers are not NULL.
    if (!pszScriptText)
        {
        return NULL;
        }


    ulRc = strhGetBlock(pszText, // Text to be searched.
                        &ulSearchOffset, // In: start offset/out: beging offs. of tag.
                        SZ_TAG_HEAD, // Tag string to be searched.
                        &pszBlock, // Out: Text block between opening and closing tags.
                        NULL, // HEAD tag has no attributes.
                        NULL,
                        &ulOffsetBeginBlock // Offset where the beginning of the block
                                            // was found.
                        );

    pszEnd = pszText + ulSearchOffset;

    // Check that tag was founded. If not return with error message.
    if (ulRc != 0)
        {
        return NULL;
        }


    pszTemp = pszBlock;
    Message.BuildFromText(pszTemp, NULL, NULL, 0);
    Title.BuildFromText(pszTemp, NULL, NULL, 0);
    VirtualGroup.BuildFromText(pszTemp, NULL, NULL, 0);


    return pszEnd;
    }





/****************************************************************************
 ****************************************************************************
 @@       ScriptNextButton class:

 ****************************************************************************
 *  ScriptNextButton class member functions:
 *  ScripitNextButton   Constructor.
 *  ~ScriptNextButton   Destructor.
 *  SetTarget           Sets target text.
 *  SetNextButton       Sets next button text.
 *  ReturnTarget        Returns target.
 *  ReturnNextButton    Returns const pointer to next button text.
 *  MakeScriptText      Returns pointer to script text.
 *  BuildFromText       Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptNextButton::ScriptNextButton
 *  Constructor for ScriptNextButton class. Initializes szNextButtonText
 *  string by setting first char to '\0'.
 */
ScriptNextButton::ScriptNextButton(USHORT usIndex)
    {
    *szNextButtonText = '\0';
    usTarget = 0;
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptNextButton::~ScriptNextButton
 *  destructor for ScriptNextButton class. Does nothing special.
 */

ScriptNextButton::~ScriptNextButton()
    {
    *szNextButtonText = '\0';
    usTarget = 0;
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptNextButton::SetNextButton
 *  Sets title text.
 */

void ScriptNextButton::SetNextButton(const PSZ pszText)
    {
    if (strlen(pszText) < MAXNEXTBTNTEXTLENGTH)
        strcpy(this->szNextButtonText, pszText);
    }



/***************************************************************************
 *@@ ScriptNextButton::SetIndex
 *  Sets title text.
 */

void ScriptNextButton::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptNextButton::SetTarget
 *  Sets target.
 */

void ScriptNextButton::SetTarget(USHORT usTarget)
    {
    this->usTarget = usTarget;
    }



/***************************************************************************
 *@@ ScriptNextButton::ReturnNextButton
 *  Returns const pointer to next button text.
 */

const PSZ ScriptNextButton::ReturnNextButton()
    {
    return szNextButtonText;
    }



/***************************************************************************
 *@@ ScriptNextButton::ReturnIndex
 *  Returns index number.
 */

USHORT ScriptNextButton::ReturnIndex()
    {
    return usIndex;
    }



/***************************************************************************
 *@@ ScriptNextButton::ReturnTarget
 *  Returns target number.
 */

USHORT ScriptNextButton::ReturnTarget()
    {
    return usTarget;
    }





/***************************************************************************
 *@@ ScriptNextButton::MakeScriptText
 *  Returns pointer to script text.
 */
PSZ ScriptNextButton::MakeScriptText(BOOL bTag)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    ulTextLength = strlen(szNextButtonText);
    CHAR szTag[128];
    CHAR szEndTag[128];

    if (ulTextLength == 0)
        return NULL;

    if (bTag)
        {
        // The length of the tag, <, >, attribute, target, = and a space.
        ulTextLength += strlen(SZ_TAG_NEXTBUTTON) + 2 +
            strlen(SZ_ATTRIBUTE_TARGET) + LtoALength((LONG)usTarget) + 2;
        sprintf(szTag, "<%s %s=%i>", SZ_TAG_NEXTBUTTON, SZ_ATTRIBUTE_TARGET, usTarget);
        // The length of the ending attribute, < and >.
        ulTextLength += strlen(SZ_TAG_ENDNEXTBUTTON) + 2;
        sprintf(szEndTag, "<%s>", SZ_TAG_ENDNEXTBUTTON);
        }

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bTag)
        sprintf(pszScriptText, "%s%s%s", szTag, szNextButtonText, szEndTag);
    else
        sprintf(pszScriptText, "%s", szNextButtonText);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptNextButton::BuildFromText
 *  Builds nextbutton tag accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptNextButton::BuildFromText(PSZ pszScriptText,
                                   SCRIPTERROR **ppScriptError,
                                   USHORT *pusErrorTableSize,
                                   ULONG ulLineNumber)
    {
    SCRIPTERROR Error;
    PSZ pszBlock = NULL;
    PSZ pszAttribute = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulAttrValue = 0;
    ULONG ulAttrLen = 0;
    ULONG ulRc = 0;


    ulRc = strhGetBlock(pszScriptText, &ulSearchOffset, SZ_TAG_NEXTBUTTON, &pszBlock,
                        &pszAttribute, NULL, NULL);
    if (pszBlock && pszAttribute)
        {
        this->SetNextButton(pszBlock);
        if (strhGetNumAttribValue(pszAttribute, SZ_ATTRIBUTE_TARGET, (PLONG)&ulAttrValue))
            this->SetTarget(ulAttrValue);

        }
    else
        return NULL;

    ulAttrLen = strlen(pszBlock);
    delete [] pszBlock;
    return pszScriptText + ulSearchOffset + ulAttrLen + 1;
    }




/****************************************************************************
 ****************************************************************************
        ScriptText class:

 ****************************************************************************
 *  ScriptText class member functions:
 *  ScripitText    Constructor.
 *  ~ScriptText    Destructor.
 *  SetText        Sets title text.
 *  SetIndex        Sets title object ID. Should be used for identification.
 *  ReturnText     Returns const pointer to title text.
 *  ReturnIndex     Returns object ID number.
 *  MakeScriptText  Returns pointer to script text.
 *  BuildFromText   Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptText::ScriptText
 *  Constructor for ScriptText class.
 */
ScriptText::ScriptText(USHORT usIndex)
    {
    pszText = NULL;
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptText::~ScriptText
 *  destructor for ScriptText class.
 */

ScriptText::~ScriptText()
    {
    if (this->pszText)
        delete [] this->pszText;
    this->pszText = NULL;
    this->usIndex = 1;
    }


ScriptText &ScriptText::operator=(const ScriptText &Text)
    {
    this->SetIndex(Text.usIndex);
    this->SetText(Text.pszText);
    return (*this);
    }



/***************************************************************************
 *@@ ScriptText::SetText
 *  Sets text.
 */

void ScriptText::SetText(const PSZ pszText)
    {
    if (this->pszText)
        delete [] this->pszText;
    if (pszText)
        {
        this->pszText = new CHAR[strlen(pszText) + 1];
        strcpy(this->pszText, pszText);
        }
    }



/***************************************************************************
 *@@ ScriptText::SetIndex
 *  Sets title text.
 */

void ScriptText::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptText::ReturnText
 *  Returns const pointer to text.
 */

const PSZ ScriptText::ReturnText()
    {
    return pszText;
    }



/***************************************************************************
 *@@ ScriptText::ReturnIndex
 *  Returns index number.
 */

USHORT ScriptText::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptText::MakeScriptText
 *  Returns pointer to script text.
 */
PSZ ScriptText::MakeScriptText(BOOL bTag)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    ulTextLength = strlen(pszText);
    CHAR szTag[128];
    CHAR szEndTag[128];

    if (ulTextLength == 0)
        return NULL;

    if (bTag)
        {
        // The length of the attribute, <, and >.
        ulTextLength += strlen(SZ_TAG_TEXT) + 2;
        sprintf(szTag, "<%s>", SZ_TAG_TEXT);
        // The length of the ending attribute, < and >.
        ulTextLength += strlen(SZ_TAG_ENDTEXT) + 2;
        sprintf(szEndTag, "<%s>", SZ_TAG_ENDTEXT);
        }

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bTag)
        sprintf(pszScriptText, "%s%s%s", szTag, pszText, szEndTag);
    else
        sprintf(pszScriptText, "%s", pszText);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptText::BuildFromText
 *  Builds text tag accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptText::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    SCRIPTERROR Error;
    PSZ pszBlock = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulAttrLen = 0;
    ULONG ulRc = 0;


    ulRc = strhGetBlock(pszScriptText, &ulSearchOffset, SZ_TAG_TEXT, &pszBlock,
                        NULL, NULL, NULL);

    if (pszBlock)
        this->SetText(pszBlock);
    else
        return NULL;

    ulAttrLen = strlen(pszBlock);
    delete [] pszBlock;
    return pszScriptText + ulSearchOffset + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        ScriptReadMe class:

 ****************************************************************************
 *  ScriptReadMe class member functions:
 *  ScripitReadMe    Constructor.
 *  ~ScriptReadMe    Destructor.
 *  SetReadMe        Sets title text.
 *  SetIndex        Sets title object ID. Should be used for identification.
 *  ReturnReadMe     Returns const pointer to title text.
 *  ReturnIndex     Returns object ID number.
 *  MakeScriptText  Returns pointer to script text.
 *  BuildFromText   Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptReadMe::ScriptReadMe
 *  Constructor for ScriptReadMe class.
 */
ScriptReadMe::ScriptReadMe(USHORT usIndex)
    {
    pszReadMe = NULL;
    this->usIndex = usIndex;
    }


/***************************************************************************
 *@@ ScriptReadMe::ScriptReadMe::operator=
 */
ScriptReadMe &ScriptReadMe::operator=(const ScriptReadMe &ReadMe)
    {
    this->usIndex = ReadMe.usIndex;
    SetReadMe(ReadMe.pszReadMe);
    SetFormat(ReadMe.usFormat);
    return (*this);
    }




/***************************************************************************
 *@@ ScriptReadMe::~ScriptReadMe
 *  destructor for ScriptReadMe class.
 */

ScriptReadMe::~ScriptReadMe()
    {
    if (pszReadMe)
        delete [] pszReadMe;
    pszReadMe = NULL;
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptReadMe::SetReadMe
 *  Sets text.
 */

void ScriptReadMe::SetReadMe(const PSZ pszReadMe)
    {
    if (pszReadMe)
        {
        if (this->pszReadMe)
            delete [] this->pszReadMe;
        this->pszReadMe = new CHAR[strlen(pszReadMe) + 1];
        strcpy(this->pszReadMe, pszReadMe);
        }
    else
        {
        if (this->pszReadMe)
            delete [] this->pszReadMe;
        this->pszReadMe = NULL;
        }
    }



/***************************************************************************
 *@@ ScriptReadMe::SetIndex
 *  Sets title text.
 */

void ScriptReadMe::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }


/***************************************************************************
 *@@ ScriptReadMe::SetFormat
 */

void ScriptReadMe::SetFormat(USHORT usFormat)
    {
    this->usFormat = usFormat;
    }



/***************************************************************************
 *@@ ScriptReadMe::ReturnReadMe
 *  Returns const pointer to readme text.
 */

const PSZ ScriptReadMe::ReturnReadMe()
    {
    return pszReadMe;
    }



/***************************************************************************
 *@@ ScriptReadMe::ReturnIndex
 *  Returns index number.
 */

USHORT ScriptReadMe::ReturnIndex()
    {
    return usIndex;
    }



/***************************************************************************
 *@@ ScriptReadMe::ReturnFormat
 */

USHORT ScriptReadMe::ReturnFormat()
    {
    return usFormat;
    }




/***************************************************************************
 *@@ ScriptReadMe::MakeScriptText
 *  Returns pointer to script text.
 */
PSZ ScriptReadMe::MakeScriptText(BOOL bTag)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    PSZ pszFormat = NULL;
    PSZ pszFormatType = NULL;
    CHAR szTag[128];
    CHAR szEndTag[128];


    if (!pszReadMe)
        return NULL;

    if ((ulTextLength = strlen(pszReadMe)) == 0)
        return NULL;

    if (bTag)
        {
        // The length of the attribute, <, and >.
        ulTextLength += strlen(SZ_TAG_README) + 2;

        if (usFormat != README_FORMAT_NONE)
            {
            // FORMAT attribute, = and space lengths.
            ulTextLength += strlen(SZ_ATTRIBUTE_FORMAT) + 2;
            pszFormat = SZ_ATTRIBUTE_FORMAT;
            }

        switch(usFormat)
            {
            case README_FORMAT_PLAIN:
                {
                ulTextLength += strlen(SZ_FORMAT_PLAIN);
                pszFormatType = SZ_FORMAT_PLAIN;
                break;
                }
            case README_FORMAT_FLOW:
                {
                ulTextLength += strlen(SZ_FORMAT_FLOW);
                pszFormatType = SZ_FORMAT_FLOW;
                break;
                }
            case README_FORMAT_HTML:
                {
                ulTextLength += strlen(SZ_FORMAT_HTML);
                pszFormatType = SZ_FORMAT_HTML;
                break;
                }
            default:
                break;
            }

        if (usFormat != README_FORMAT_NONE)
            sprintf(szTag, "<%s %s=%s>", SZ_TAG_README, pszFormat, pszFormatType);
        else
            sprintf(szTag, "<%s>", SZ_TAG_README);
        // The length of the ending attribute, < and >.
        ulTextLength += strlen(SZ_TAG_ENDREADME) + 2;
        sprintf(szEndTag, "<%s>", SZ_TAG_ENDREADME);
        }

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bTag)
        sprintf(pszScriptText, "%s%s%s", szTag, pszReadMe, szEndTag);
    else
        sprintf(pszScriptText, "%s", pszReadMe);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptReadMe::BuildFromText
 *  Builds text tag accordingly to script text. If any errors is found in script
 *  text, error informations are returned in table where to *pScriptError
 *  points. Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptReadMe::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    SCRIPTERROR Error;
    PSZ pszBlock = NULL;
    PSZ pszAttribute = NULL;
    PSZ pszFormat = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulAttrLen = 0;
    ULONG ulRc = 0;


    ulRc = strhGetBlock(pszScriptText, &ulSearchOffset, SZ_TAG_README, &pszBlock,
                        &pszAttribute, NULL, NULL);

    if (pszBlock)
        this->SetReadMe(pszBlock);
    else
        return NULL;

    if (pszAttribute)
        {
        pszFormat = strhGetTextAttr(pszAttribute, SZ_ATTRIBUTE_FORMAT, NULL);
        if (!pszFormat)
            usFormat = README_FORMAT_NONE;
        else
            {
            if (strcmp(pszFormat, SZ_FORMAT_PLAIN) == 0)
                usFormat = README_FORMAT_PLAIN;
            else if (strcmp(pszFormat, SZ_FORMAT_FLOW) == 0)
                usFormat = README_FORMAT_FLOW;
            else if (strcmp(pszFormat, SZ_FORMAT_HTML) == 0)
                usFormat = README_FORMAT_HTML;
            else
                usFormat = README_FORMAT_NONE;
            }
        }
    else
        usFormat = README_FORMAT_NONE;

    ulAttrLen = strlen(pszBlock);
    delete [] pszBlock;
    return pszScriptText + ulSearchOffset + ulAttrLen + 1;
    }




/****************************************************************************
 ****************************************************************************
        ScriptPage class:

 ****************************************************************************
 *  ScriptPage class member functions:
 *  ScriptPage              Constructor.
 *  ~ScriptPage             Destructor.
 *  SetIndex                Sets index number.
 *  ReturnIndex             Returns index number.
 *  SetText
 *  ReturnText
 *  SetReadMeText
 *  ReturnReadMeText
 *  MakeScriptText
 *  BuildFromText
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptGroup::ScriptGroup
 *  Constructor for ScriptGroup class.
 */
ScriptPage::ScriptPage(USHORT usIndex)
     : BSRoot(tScriptPage)
    {
    this->usIndex = usIndex;
    usPageIndex = 1;
    usPageType = PAGE_TYPES_TEXT;
    }




/***************************************************************************
 *@@ ScriptPage::~ScriptPage
 *  Destructor for ScriptPage class.
 */
ScriptPage::~ScriptPage()
    {
    this->usIndex = 1;
    usPageIndex = 1;
    usPageType = PAGE_TYPES_TEXT;
    }


/***************************************************************************
 *@@ ScriptPage::SetIndex
 */
void ScriptPage::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }


/***************************************************************************
 *@@ ScriptPage::ReturnIndex
 */
USHORT ScriptPage::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptPage::SetPageIndex
 */
void ScriptPage::SetPageIndex(USHORT usPageIndex)
    {
    if (usPageIndex < 1)
        return;
    this->usPageIndex = usPageIndex;
    }


/***************************************************************************
 *@@ ScriptPage::ReturnPageIndex
 */
USHORT ScriptPage::ReturnPageIndex()
    {
    return usPageIndex;
    }



/***************************************************************************
 *@@ ScriptPage::SetPageType
 */
void ScriptPage::SetPageType(USHORT usPageType)
    {
    this->usPageType = usPageType;
    }


/***************************************************************************
 *@@ ScriptPage::ReturnPageType
 */
USHORT ScriptPage::ReturnPageType()
    {
    return usPageType;
    }





/***************************************************************************
 *@@ ScriptPage::ReturnTitleTagObject
 */
ScriptText *ScriptPage::ReturnTextObject()
    {
    return &Text;
    }



/***************************************************************************
 *@@ ScriptPage::SetText
 */
void ScriptPage::SetText(const PSZ pszText)
    {
    Text.SetText(pszText);
    }


/***************************************************************************
 *@@ ScriptPage::ReturnText
 */
const PSZ ScriptPage::ReturnText()
    {
    return Text.ReturnText();
    }



/***************************************************************************
 *@@ ScriptPage::ReturnNextButtonObject
 */
ScriptNextButton *ScriptPage::ReturnNextButtonObject()
    {
    return &NextButton;
    }




/***************************************************************************
 *@@ ScriptPage::ReturnReadMeObject
 */
ScriptReadMe *ScriptPage::ReturnReadMeObject()
    {
    return &ReadMe;
    }



/***************************************************************************
 *@@ ScriptPage::SetReadMeText
 */
void ScriptPage::SetReadMeText(const PSZ pszReadMe)
    {
    ReadMe.SetReadMe(pszReadMe);
    }


/***************************************************************************
 *@@ ScriptPage::ReturnReadMeText
 */
const PSZ ScriptPage::ReturnReadMeText()
    {
    return ReadMe.ReturnReadMe();
    }




/***************************************************************************
 *@@ ScriptPage::MakeScriptText
 *  Returns pointer to newly allocated script text.
 */
PSZ ScriptPage::MakeScriptText(BOOL bTag)
    {
    XSTRING strScript;
    xstrInit(&strScript, 0);
    PSZ pszTemp = NULL;
    PSZ pszType = NULL;
    CHAR szTag[MAXTAGLENGTH];
    CHAR szAttribute[MAXATTRIBUTELENGTH];
    USHORT usIndex;

    if (bTag)
        {
        switch (usPageType)
            {
            case PAGE_TYPES_TEXT:
                pszType = SZ_TYPE_TEXT;
                break;
            case PAGE_TYPES_README:
                pszType = SZ_TYPE_README;
                break;
            case PAGE_TYPES_CONTAINER:
                pszType = SZ_TYPE_CONTAINER;
                break;
            case PAGE_TYPES_CONFIGURE:
                pszType = SZ_TYPE_CONFIGURE;
                break;
            default:
                pszType = "";
                break;
            }
        sprintf(szTag, "<%s %s=%i %s=%s>\n", SZ_TAG_PAGE, SZ_ATTRIBUTE_INDEX,
            usPageIndex, SZ_ATTRIBUTE_TYPE, pszType);
        xstrcat(&strScript, szTag, 0);
        }

    // Add NEXTBUTTON tag.
    pszTemp = NextButton.MakeScriptText(bTag);
    AddCharacterAfterEOL(&pszTemp, '\t');
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    // Add TEXT tag.
    xstrcat(&strScript, "\n", 0);
    pszTemp = Text.MakeScriptText(bTag);
    AddCharacterAfterEOL(&pszTemp, '\t');
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    // Add README tag.
    xstrcat(&strScript, "\n", 0);
    pszTemp = ReadMe.MakeScriptText(bTag);
    AddCharacterAfterEOL(&pszTemp, '\t');
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    if (bTag)
        {
        sprintf(szTag, "<%s>", SZ_TAG_ENDPAGE);
        xstrcat(&strScript, szTag, 0);
        }

    return strScript.psz;
    }





/***************************************************************************
 *@@ ScriptPage:: BuildFromText
 *  Builds head accordingly to script text.
 *  Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptPage::BuildFromText(PSZ pszScriptText,
                              SCRIPTERROR **ppScriptError,
                              USHORT *pusErrorTableSize,
                              ULONG ulLineNumber)
    {
    PSZ pszEnd = NULL;
    ULONG ulRc = 0;
    ULONG ulOffsetBeginBlock = 0;
    ULONG ulSearchOffset = 0;
    LONG lIndex = 0;
    PSZ pszText = pszScriptText;
    PSZ pszBlock = NULL;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    PSZ pszType = NULL;

    // Check that pointers are not NULL.
    if (!pszScriptText)
        {
        return NULL;
        }


    ulRc = strhGetBlock(pszText, // Text to be searched.
                        &ulSearchOffset, // In: start offset/out: beging offs. of tag.
                        SZ_TAG_PAGE, // Tag string to be searched.
                        &pszBlock, // Out: Text block between opening and closing tags.
                        &pszAttr,
                        NULL,
                        NULL);

    pszEnd = pszText + ulSearchOffset;

    // Check that tag was founded. If not, return.
    if (ulRc != 0)
        {
        return NULL;
        }


    strhGetNumAttribValue(pszAttr, SZ_ATTRIBUTE_INDEX, &lIndex);

    SetPageIndex((USHORT)lIndex);

    if (pszAttr)
        {
        pszType = strhGetTextAttr(pszAttr, SZ_ATTRIBUTE_TYPE, NULL);
        if (pszType)
            {
            if (strcmp(pszType, SZ_TYPE_TEXT) == 0)
                usPageType = PAGE_TYPES_TEXT;
            else if (strcmp(pszType, SZ_TYPE_README) == 0)
                usPageType = PAGE_TYPES_README;
            else if (strcmp(pszType, SZ_TYPE_CONTAINER) == 0)
                usPageType = PAGE_TYPES_CONTAINER;
            else if (strcmp(pszType, SZ_TYPE_CONFIGURE) == 0)
                usPageType = PAGE_TYPES_CONFIGURE;
            }
        }

    pszTemp = pszBlock;
    NextButton.BuildFromText(pszTemp, NULL, NULL, 0);
    Text.BuildFromText(pszTemp, NULL, NULL, 0);
    ReadMe.BuildFromText(pszTemp, NULL, NULL, 0);

    return pszEnd;
    }





/****************************************************************************
 ****************************************************************************
        ScriptBody class:

 ****************************************************************************
 *  ScriptBody class member functions:
 *  ScriptBody             Constructor.
 *  ~ScriptBody            Destructor.
 *  SetIndex                Sets index number.
 *  ReturnIndex             Returns index number.
 *  CreatePageObject
 *  ReturnPageObject
 *  DeletePageObject
 *  MakeScriptText          Returns a pointer to script text.
 *  BuildFromText
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptBody::ScriptBody
 *  Constructor for ScriptBody class.
 */
ScriptBody::ScriptBody(USHORT usIndex, BOOL bAllowIndexArrangement)
    {
    this->usIndex = usIndex;

    this->bAllowIndexArrangement = bAllowIndexArrangement;

    this->pPage = NULL;
    this->usPageCount = 0;
    }




/***************************************************************************
 *@@ ScriptBody::ScriptBody
 *  Constructor for ScriptBody class.
 */
ScriptBody::ScriptBody(const ScriptBody &Body)
    {
    (*this) = Body;
    }


/***************************************************************************
 *@@ ScriptBody::~ScriptBody
 *  Destructor for ScriptBody class.
 */
ScriptBody::~ScriptBody()
    {
    this->usIndex = 1;

    this->bAllowIndexArrangement = TRUE;

    if (pPage)
        delete [] this->pPage;
    this->pPage = NULL;
    this->usPageCount = 0;
    }



/***************************************************************************
 *@@ ScriptBody::ScriptBody &operator=
 *  Assignment operator for ScriptBody class.
 */
ScriptBody &ScriptBody::operator=(const ScriptBody &Body)
    {
    USHORT usIndex = 0;
    ScriptPage *pPage = NULL;

    this->ScriptBody::~ScriptBody();

    this->usIndex = Body.usIndex;

    if (Body.pPage)
        {
        pPage = new ScriptPage[Body.usPageCount];
        this->usPageCount = Body.usPageCount;
        for (usIndex = 0; usIndex < Body.usPageCount; usIndex ++)
            pPage[usIndex] = Body.pPage[usIndex];
        this->pPage = pPage;
        }
    return (*this);
    }



/***************************************************************************
 *@@ ScriptBody::SetIndex
 *  Sets usIndex number.
 */

void ScriptBody::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptBody::ReturnIndex
 *  Returns usIndex number.
 */

USHORT ScriptBody::ReturnIndex()
    {
    return usIndex;
    }



/***************************************************************************
 *@@ ScriptBody::MakeScriptText
 *  Returns pointer to newly allocated script text.
 */
PSZ ScriptBody::MakeScriptText(BOOL bTag)
    {
    XSTRING strScript;
    xstrInit(&strScript, 0);;
    PSZ pszTemp = NULL;
    CHAR szTag[MAXTAGLENGTH];
    USHORT usIndex;

    if (bTag)
        {
        sprintf(szTag, "<%s>", SZ_TAG_BODY);
        xstrcat(&strScript, szTag, 0);
        xstrcat(&strScript, "\n", 0);
        }


        // Add PAGE tags.
    for (usIndex = 0; usIndex < usPageCount; usIndex ++)
        {
        pszTemp = pPage[usIndex].MakeScriptText(bTag);
        AddCharacterAfterEOL(&pszTemp, '\t');
        xstrcat(&strScript, pszTemp, 0);
        xstrcat(&strScript, "\n", 0);
        delete [] pszTemp;
        }

    if (bTag)
        {
        sprintf(szTag, "<%s>", SZ_TAG_ENDBODY);
        xstrcat(&strScript, szTag, 0);
        }

    return strScript.psz;
    }





/****************************************
 *  ScriptBody object handling.
 ****************************************
 * Here are functions when handling ScriptBody objects.
 */

/***************************************************************************
 *@@ ScriptBody::CopyTable
 *  Copies source table to destination table.
 */
ScriptPage *ScriptBody::CopyTable(ScriptPage *DestTable, // in: Pointer to destination table.
                                  ScriptPage *SrcTable, // in: Pointer to source table.
                                  USHORT usSize, // in: Size of source table.
                                  USHORT usSkipIndex) // in: Index to be skipped. If 0 no skipping.
    {
    USHORT SrcIndex, DestIndex;
    if (!DestTable)
        return NULL;
    if (!SrcTable)
        return DestTable;
        for (SrcIndex = 0, DestIndex = 0; SrcIndex < usSize; SrcIndex ++, DestIndex ++)
        {
        if (SrcTable[SrcIndex].ReturnIndex() == usSkipIndex)
            if (SrcIndex < usSize - 1)
                SrcIndex ++;
            else
                break;
        DestTable[DestIndex] = SrcTable[SrcIndex];
        }

    return DestTable;
    }




/***************************************************************************
 *@@ ScriptBody::ReBuildIndexes
 *  Rebuilds indexes after deletion. The order of the indexes does not matter.
 *  Example:
 *      Old indexing: 1, 4, 3, 5
 *      New indexing: 1, 3, 2, 4
 */
void ScriptBody::ReBuildIndexes(ScriptPage *Table, USHORT usTableSize)
    {
    USHORT Index, Index2, Index3, usMissing, usMax = 0, i;
    BOOL bFinded = FALSE, bReady = FALSE;

    // First find largest index number.
    for (Index = 0; Index < usTableSize; Index ++)
        if (Table[Index].ReturnIndex() > usMax)
            usMax = Table[Index].ReturnIndex();

    // Do this until it is sure that there is no skipped indexes left.
    while (!bReady)
        {
        bReady = TRUE;
        // Go through all index values from 1 to usMax.
        for (Index = 1; Index <= usMax; Index ++)
            {
            // Check that Index number will be found.
            for (Index2 = 0; Index2 < usTableSize; Index2 ++)
                if (Index == Table[Index2].ReturnIndex())
                    bFinded = TRUE;
            // If did not found, sub 1 from every index number above the index number
            // which did not found.
            if (bFinded == FALSE)
                {
                for (Index3 = Index + 1; Index3 <= usMax; Index3 ++)
                    for (Index2 = 0; Index2 < usTableSize; Index2 ++)
                        if (Index3 == Table[Index2].ReturnIndex())
                            Table[Index2].SetIndex(Index3 - 1);
                usMax --;
                }
            if (bFinded == FALSE)
                bReady = FALSE;
            bFinded = FALSE;
            }
        }
    }


/***************************************************************************
 *@@ ScriptBody::FindUnusedIndex
 *  Returns first unused index number.
 */
USHORT ScriptBody::FindUnusedIndex(ScriptPage *Table, USHORT usTableSize)
    {
    USHORT Index, Index2, usMax = 0;
    BOOL bFinded = FALSE;

    // First find largest index number.
    for (Index = 0; Index < usTableSize; Index ++)
        if (Table[Index].ReturnIndex() > usMax)
            usMax = Table[Index].ReturnIndex();

    for (Index = 1; Index <= usMax; Index ++)
        {
        // Go through all the indexes of the table.
        for (Index2 = 0; Index2 < usTableSize; Index2 ++)
            if (Index == Table[Index2].ReturnIndex())
                {
                bFinded = TRUE;
                break;
                }
        // If Index found from the table, continue.
        if (bFinded == TRUE)
            bFinded = FALSE;
        // If not break.
        else
            break;
        }

    return Index;
    }




/***************************************************************************
 *@@ ScriptBody::CreatePageObject
 *  Creates new Page object and returns pointer to it.
 */
ScriptPage *ScriptBody::CreatePageObject()
    {
    ScriptPage *Object, *NewObject;

    if (!(Object = new ScriptPage[++usPageCount]))
        {
        usPageCount --;
        return NULL;
        }

    Object[usPageCount - 1].SetIndex(FindUnusedIndex(this->pPage,
        usPageCount - 1));

    CopyTable(Object, this->pPage, usPageCount - 1);
    delete [] this->pPage;
    this->pPage = Object;
    NewObject = &this->pPage[usPageCount - 1];
    return NewObject;
    }



/***************************************************************************
 *@@ ScriptBody::ReturnPageObject
 *  Returns object which index is given usIndex.
 */
ScriptPage *ScriptBody::ReturnPageObject(USHORT usIndex)
    {
    USHORT Index, usTemp;

    for (Index = 0; Index < this->usPageCount; Index ++)
        {
        usTemp = pPage[Index].ReturnIndex();
        if (usTemp == usIndex)
            return &pPage[Index];
        }
    return NULL;
    }




/***************************************************************************
 *@@ ScriptBody::DeletePageObject
 *  Deletes object which index is given in usIndex.
 */
USHORT ScriptBody::DeletePageObject(USHORT usIndex)
    {
    ScriptPage *TempPage = NULL;

    if (usPageCount > 1)
        {
        TempPage = new ScriptPage[usPageCount - 1];
        CopyTable(TempPage, pPage, usPageCount, usIndex);
        if (bAllowIndexArrangement)
            ReBuildIndexes(TempPage, usPageCount - 1);
        }
    delete [] pPage;
    pPage = TempPage;
    usPageCount --;
    return 0;
    }



/***************************************************************************
 *@@ ScriptBody:: ReturnPageObjectCount
 *  Returns count number of ScriptPage objects.
 */
USHORT ScriptBody::ReturnPageObjectCount()
    {
    return usPageCount;
    }




/***************************************************************************
 *@@ ScriptBody::DeleteAllPageObjects
 *  Deletes all ScriptPage objects.
 */
USHORT ScriptBody::DeleteAllPageObjects()
    {
    if (pPage)
        delete [] pPage;
    pPage = NULL;
    usPageCount = 0;
    return 0;
    }




/***************************************************************************
 *@@ ScriptBody::BuildFromText
 *  Builds body accordingly to script text.
 */
PSZ ScriptBody::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    PSZ pszEnd = NULL;
    ULONG ulRc = 0;
    ULONG ulSearchOffset = 0;
    PSZ pszText = pszScriptText;
    PSZ pszBlock = NULL;
    PSZ pszTemp = NULL;
    PSZ pszBody = NULL;
    ScriptPage *pPage = NULL, TmpPage;

    // Check that pointers are not NULL.
    if (!pszScriptText)
        {
        return NULL;
        }

    ulRc = strhGetBlock(pszText, &ulSearchOffset, SZ_TAG_BODY, &pszBlock,
                        NULL, NULL, NULL);

    // Find all PAGE tags.
    pszTemp = pszBlock;
    do
        {
        // Try to build object from text.
        pszTemp = TmpPage.BuildFromText(pszTemp, NULL, NULL, 0);
        // If it succeded, create new object and copy temp object to
        // created object.
        if (pszTemp)
            {
            pPage = CreatePageObject();
            TmpPage.SetIndex(pPage->ReturnIndex());
            *pPage = TmpPage;
            }

        } while (pszTemp != NULL);

    pszEnd = pszScriptText + ulSearchOffset;
    return pszEnd;
    }




/****************************************************************************
 ****************************************************************************
        ScriptVersion class:

 ****************************************************************************
 *  ScriptVersion class member functions:
 *  ScripitVersion    Constructor.
 *  ~ScriptVersion    Destructor.
 *  SetVersion        Sets title text.
 *  SetIndex          Sets title object ID. Should be used for identification.
 *  ReturnVersion     Returns const pointer to title text.
 *  ReturnIndex       Returns object ID number.
 *  MakeScriptText    Returns pointer to script text.
 *  BuildFromText     Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptVersion::ScriptVersion
 *  Constructor for ScriptVersion class. Initializes szMessage string by setting
 *  first char to '\0'.
 */
ScriptVersion::ScriptVersion(USHORT usIndex)
    {
    usMajor = 0;
    usMinor = 0;
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptVersion::~ScriptVersion
 *  destructor for ScriptVersion class. Does nothing special.
 */

ScriptVersion::~ScriptVersion()
    {
    usMajor = 0;
    usMinor = 0;
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptVersion::SetMajor
 */

void ScriptVersion::SetMajor(USHORT usMajor)
    {
    this->usMajor = usMajor;
    }



/***************************************************************************
 *@@ ScriptVersion::SetMinor
 */

void ScriptVersion::SetMinor(USHORT usMinor)
    {
    this->usMinor = usMinor;
    }




/***************************************************************************
 *@@ ScriptVersion::SetIndex
 *  Sets title text.
 */

void ScriptVersion::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptVersion::ReturnMajor
 */
USHORT ScriptVersion::ReturnMajor()
    {
    return usMajor;
    }



/***************************************************************************
 *@@ ScriptVersion::ReturnMinor
 */
USHORT ScriptVersion::ReturnMinor()
    {
    return usMinor;
    }



/***************************************************************************
 *@@ ScriptVersion::ReturnIndex
 *  Returns index number.
 */
USHORT ScriptVersion::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptVersion::MakeScriptText
 *  Returns pointer to newly allocated buffer containing script text.
 */
PSZ ScriptVersion::MakeScriptText(BOOL bAttribute)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    if (usMajor == 0 && usMinor == 0)
        return NULL;

    if (bAttribute)
        // The length of the attribute, =, " and  ".
        ulTextLength += strlen(SZ_ATTRIBUTE_VERSION) + 3;

    ulTextLength += LtoALength((LONG)usMajor) + LtoALength((LONG)usMinor) + 1;

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bAttribute)
        sprintf(pszScriptText, "%s=\"%i.%i\"", SZ_ATTRIBUTE_VERSION, usMajor, usMinor);
    else
        sprintf(pszScriptText, "%i.%i", usMajor, usMinor);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptVersion::BuildFromText
 *  Builds version attribute accordingly to script text. If any errors is
 *  found in script text. Pointer to after end of tag is returned so multiple
 *  searches can be done.
 */
PSZ ScriptVersion::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    PSZ pszVersion = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulValue = 0, ulAttrLen = 0;

    pszVersion = strhGetTextAttr(pszScriptText, SZ_ATTRIBUTE_VERSION, &ulValue);
    if (pszVersion)
        sscanf(pszVersion, "%d.%d", &usMajor, &usMinor);
    else
        return NULL;

    ulAttrLen = strlen(pszVersion);
    delete [] pszVersion;
    return pszScriptText + ulValue + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        ScriptOS class:

 ****************************************************************************
 *  ScriptOS class member functions:
 *  ScripitOS    Constructor.
 *  ~ScriptOS    Destructor.
 *  SetOS        Sets title text.
 *  SetIndex          Sets title object ID. Should be used for identification.
 *  ReturnOS     Returns const pointer to title text.
 *  ReturnIndex       Returns object ID number.
 *  MakeScriptText    Returns pointer to script text.
 *  BuildFromText     Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptOS::ScriptOS
 *  Constructor for ScriptOS class. Initializes szMessage string by setting
 *  first char to '\0'.
 */
ScriptOS::ScriptOS(USHORT usIndex)
    {
    usOS = 0;
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptOS::~ScriptOS
 *  destructor for ScriptOS class. Does nothing special.
 */

ScriptOS::~ScriptOS()
    {
    usOS = 0;
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptOS::SetMajor
 */

void ScriptOS::SetOS(USHORT usOS)
    {
    this->usOS = usOS;
    }



/***************************************************************************
 *@@ ScriptOS::SetIndex
 *  Sets title text.
 */

void ScriptOS::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptOS::ReturnOS
 */
USHORT ScriptOS::ReturnOS()
    {
    return usOS;
    }



/***************************************************************************
 *@@ ScriptOS::ReturnIndex
 *  Returns index number.
 */
USHORT ScriptOS::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptOS::MakeScriptText
 *  Returns pointer to newly allocated buffer containing script text.
 */
PSZ ScriptOS::MakeScriptText(BOOL bAttribute)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;
    PSZ pszOS = NULL;

    if (bAttribute)
        // The length of the attribute, =, " and  ".
        ulTextLength += strlen(SZ_ATTRIBUTE_OS) + 3;

    switch (usOS)
        {
        case OS_OS2_2X:
            pszOS = SZ_OS_OS22X;
            break;
        case OS_OS2_3X:
            pszOS = SZ_OS_OS23X;
            break;
        case OS_OS2_4X:
            pszOS = SZ_OS_OS24X;
            break;
        default:
            pszOS = SZ_OS_OS22X;
            break;
        }

    ulTextLength += strlen(pszOS);

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bAttribute)
        sprintf(pszScriptText, "%s=\"%s\"", SZ_ATTRIBUTE_OS, pszOS);
    else
        sprintf(pszScriptText, "%s", pszOS);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptOS::BuildFromText
 *  Builds version attribute accordingly to script text. If any errors is
 *  found in script text. Pointer to after end of tag is returned so multiple
 *  searches can be done.
 */
PSZ ScriptOS::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    PSZ pszOS = NULL;
    ULONG ulSearchOffset = 0;
    ULONG ulValue = 0, ulAttrLen = 0;

    pszOS = strhGetTextAttr(pszScriptText, SZ_ATTRIBUTE_OS, &ulValue);
    if (pszOS)
        {
        if (strcmp(pszOS, SZ_OS_OS22X) == 0)
            usOS = OS_OS2_2X;
        else if (strcmp(pszOS, SZ_OS_OS23X) == 0)
            usOS = OS_OS2_3X;
        else if (strcmp(pszOS, SZ_OS_OS24X) == 0)
            usOS = OS_OS2_4X;

        }
    else
        return NULL;

    ulAttrLen = strlen(pszOS);
    delete [] pszOS;
    return pszScriptText + ulValue + ulAttrLen + 1;
    }





/****************************************************************************
 ****************************************************************************
        ScriptCodePage class:

 ****************************************************************************
 *  ScriptCodePage class member functions:
 *  ScripitCodePage    Constructor.
 *  ~ScriptCodePage    Destructor.
 *  SetCodePage        Sets title text.
 *  SetIndex          Sets title object ID. Should be used for identification.
 *  ReturnCodePage     Returns const pointer to title text.
 *  ReturnIndex       Returns object ID number.
 *  MakeScriptText    Returns pointer to script text.
 *  BuildFromText     Builds object accordingly to script text.
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptCodePage::ScriptCodePage
 *  Constructor for ScriptCodePage class. Initializes szMessage string by setting
 *  first char to '\0'.
 */
ScriptCodePage::ScriptCodePage(USHORT usIndex)
    {
    usCodePage = 0;
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptCodePage::~ScriptCodePage
 *  destructor for ScriptCodePage class. Does nothing special.
 */

ScriptCodePage::~ScriptCodePage()
    {
    usCodePage = 0;
    this->usIndex = 1;
    }


/***************************************************************************
 *@@ ScriptCodePage::SetCodePage
 */

void ScriptCodePage::SetCodePage(USHORT usCodePage)
    {
    this->usCodePage = usCodePage;
    }



/***************************************************************************
 *@@ ScriptCodePage::SetIndex
 *  Sets title text.
 */

void ScriptCodePage::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }



/***************************************************************************
 *@@ ScriptCodePage::ReturnCodePage
 */
USHORT ScriptCodePage::ReturnCodePage()
    {
    return usCodePage;
    }



/***************************************************************************
 *@@ ScriptCodePage::ReturnIndex
 *  Returns index number.
 */
USHORT ScriptCodePage::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptCodePage::MakeScriptText
 *  Returns pointer to newly allocated buffer containing script text.
 */
PSZ ScriptCodePage::MakeScriptText(BOOL bAttribute)
    {
    ULONG ulTextLength = 0;
    PSZ pszScriptText = NULL;

    if (usCodePage == 0)
        return NULL;

    if (bAttribute)
        // The length of the attribute and =.
        ulTextLength += strlen(SZ_ATTRIBUTE_CODEPAGE) + 1;

    ulTextLength += (USHORT)LtoALength(usCodePage);

    pszScriptText = new CHAR[ulTextLength + 1];
    *pszScriptText = '\0';

    if (bAttribute)
        sprintf(pszScriptText, "%s=%i", SZ_ATTRIBUTE_CODEPAGE, usCodePage);
    else
        sprintf(pszScriptText, "%i", usCodePage);

    return pszScriptText;
    }


/***************************************************************************
 *@@ScriptCodePage::BuildFromText
 */
PSZ ScriptCodePage::BuildFromText(PSZ pszScriptText,
                               SCRIPTERROR **ppScriptError,
                               USHORT *pusErrorTableSize,
                               ULONG ulLineNumber)
    {
    PSZ pszCP = NULL;
    ULONG ulSearchOffset = 0;
    LONG lValue = 0;
    ULONG ulAttrLen = 0;

    pszCP = strhGetNumAttribValue(pszScriptText, SZ_ATTRIBUTE_CODEPAGE, &lValue);
    if (pszCP)
        usCodePage = (USHORT)lValue;
    else
        return NULL;

    return pszCP + LtoALength(lValue);
    }







/****************************************************************************
 ****************************************************************************
        ScriptWarpIN class:

 ****************************************************************************
 *  ScriptWarpIN class member functions:
 *  ScriptWarpIN              Constructor.
 *  ~ScriptWarpIN             Destructor.
 *  SetIndex                Sets index number.
 *  ReturnIndex             Returns index number.
 *  SetOSValue
 *  ReturnOSValue
 *  SetVersionNumber
 *  ReturnVersionNumber
 *  SetCodePageNumber
 *  ReturnCodePageNumber
 *  ReturnOSObject
 *  ReturnVersionObject
 *  ReturnCodePageObject
 *  ReturnHeadObject
 *  ReturnBodyObject
 *  MakeScriptText
 *  BuildFromText
 ***************************************************************************/


/***************************************************************************
 *@@ ScriptGroup::ScriptGroup
 *  Constructor for ScriptGroup class.
 */
ScriptWarpIN::ScriptWarpIN(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }




/***************************************************************************
 *@@ ScriptWarpIN::~ScriptWarpIN
 *  Destructor for ScriptWarpIN class.
 */
ScriptWarpIN::~ScriptWarpIN()
    {
    this->usIndex = 1;
    SetOSValue(OS_OS2_2X);
    SetCodePageNumber(0);
    SetVersionNumber(0, 0);
    }


/***************************************************************************
 *@@ ScriptWarpIN::SetIndex
 */
void ScriptWarpIN::SetIndex(USHORT usIndex)
    {
    this->usIndex = usIndex;
    }


/***************************************************************************
 *@@ ScriptWarpIN::ReturnIndex
 */
USHORT ScriptWarpIN::ReturnIndex()
    {
    return usIndex;
    }




/***************************************************************************
 *@@ ScriptWarpIN::SetOSValue
 */
void ScriptWarpIN::SetOSValue(USHORT usOS)
    {
    OS.SetOS(usOS);
    }


/***************************************************************************
 *@@ ScriptWarpIN::ReturnOSValue
 */
USHORT ScriptWarpIN::ReturnOSValue()
    {
    return OS.ReturnOS();
    }



/***************************************************************************
 *@@ ScriptWarpIN::SetVersionNumber
 */
void ScriptWarpIN::SetVersionNumber(USHORT usMajor, USHORT usMinor)
    {
    Version.SetMajor(usMajor);
    Version.SetMinor(usMinor);
    }


/***************************************************************************
 *@@ ScriptWarpIN::ReturnVersionNumber
 *  Returns both major and minor version number in ULONG format. The 16 most
 *  significant bits contains major number and the rest minor number.
 */
ULONG ScriptWarpIN::ReturnVersionNumber(USHORT *pusMajor, USHORT *pusMinor)
    {
    if (pusMajor)
        *pusMajor = Version.ReturnMajor();
    if (pusMinor)
        *pusMinor = Version.ReturnMinor();
    return ((ULONG)Version.ReturnMajor() << 16) + Version.ReturnMinor();
    }



/***************************************************************************
 *@@ ScriptWarpIN::SetCodePageNumber
 */
void ScriptWarpIN::SetCodePageNumber(USHORT usCP)
    {
    CodePage.SetCodePage(usCP);
    }


/***************************************************************************
 *@@ ScriptWarpIN::ReturnCodePageNumber
 */
USHORT ScriptWarpIN::ReturnCodePageNumber()
    {
    return CodePage.ReturnCodePage();
    }



/***************************************************************************
 *@@ ScriptWarpIN::ReturnOSObject
 */
ScriptOS *ScriptWarpIN::ReturnOSObject()
    {
    return &OS;
    }



/***************************************************************************
 *@@ ScriptWarpIN::ReturnVersionObject
 */
ScriptVersion *ScriptWarpIN::ReturnVersionObject()
    {
    return &Version;
    }




/***************************************************************************
 *@@ ScriptWarpIN::ReturnCodePageObject
 */
ScriptCodePage *ScriptWarpIN::ReturnCodePageObject()
    {
    return &CodePage;
    }




/***************************************************************************
 *@@ ScriptWarpIN::ReturnHeadObject
 */
ScriptHead *ScriptWarpIN::ReturnHeadObject()
    {
    return &Head;
    }




/***************************************************************************
 *@@ ScriptWarpIN::ReturnBodyObject
 */
ScriptBody *ScriptWarpIN::ReturnBodyObject()
    {
    return &Body;
    }





/***************************************************************************
 *@@ ScriptWarpIN::MakeScriptText
 *  Returns pointer to newly allocated script string.
 */
PSZ ScriptWarpIN::MakeScriptText(BOOL bTag)
    {
    XSTRING strScript;
    xstrInit(&strScript, 0);
    PSZ pszTemp = NULL;
    PSZ pszType = NULL;
    CHAR szTag[MAXTAGLENGTH];
    CHAR szAttribute[MAXATTRIBUTELENGTH];
    USHORT usIndex;

    if (bTag)
        {
        xstrcat(&strScript, "<", 0);
        xstrcat(&strScript, SZ_TAG_WARPIN, 0);
        pszTemp = Version.MakeScriptText();
        if (pszTemp)
            {
            xstrcat(&strScript, " ", 0);
            xstrcat(&strScript, pszTemp, 0);
            delete pszTemp;
            }
        pszTemp = OS.MakeScriptText();
        if (pszTemp)
            {
            xstrcat(&strScript, " ", 0);
            xstrcat(&strScript, pszTemp, 0);
            delete pszTemp;
            }
        pszTemp = CodePage.MakeScriptText();
        if (pszTemp)
            {
            xstrcat(&strScript, " ", 0);
            xstrcat(&strScript, pszTemp, 0);
            delete pszTemp;
            }
        xstrcat(&strScript, ">", 0);
        }

    xstrcat(&strScript, "\n", 0);
    // Add HEAD tag.
    pszTemp = Head.MakeScriptText(bTag);
    AddCharacterAfterEOL(&pszTemp, '\t');
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    // Add BODY tag.
    xstrcat(&strScript, "\n", 0);
    pszTemp = Body.MakeScriptText(bTag);
    AddCharacterAfterEOL(&pszTemp, '\t');
    xstrcat(&strScript, pszTemp, 0);
    delete [] pszTemp;
    xstrcat(&strScript, "\n", 0);

    if (bTag)
        {
        sprintf(szTag, "<%s>", SZ_TAG_ENDWARPIN);
        xstrcat(&strScript, szTag, 0);
        }

    return strScript.psz;
    }





/***************************************************************************
 *@@ ScriptWarpIN:: BuildFromText
 *  Builds head accordingly to script text.
 *  Pointer to after end of tag is returned so multiple searches can be
 *  done.
 */
PSZ ScriptWarpIN::BuildFromText(PSZ pszScriptText,
                              SCRIPTERROR **ppScriptError,
                              USHORT *pusErrorTableSize,
                              ULONG ulLineNumber)
    {
    PSZ pszEnd = NULL;
    ULONG ulRc = 0;
    ULONG ulOffsetBeginBlock = 0;
    ULONG ulSearchOffset = 0;
    LONG lIndex = 0;
    PSZ pszText = pszScriptText;
    PSZ pszBlock = NULL;
    PSZ pszTemp = NULL;
    PSZ pszAttr = NULL;
    PSZ pszType = NULL;

    // Check that pointers are not NULL.
    if (!pszScriptText)
        {
        return NULL;
        }


    ulRc = strhGetBlock(pszText, // Text to be searched.
                        &ulSearchOffset, // In: start offset/out: beging offs. of tag.
                        SZ_TAG_WARPIN, // Tag string to be searched.
                        &pszBlock, // Out: Text block between opening and closing tags.
                        &pszAttr,
                        NULL,
                        NULL);

    pszEnd = pszText + ulSearchOffset;

    // Check that tag was founded. If not, return.
    if (ulRc != 0)
        return NULL;

    Version.BuildFromText(pszAttr, NULL, NULL, 0);
    OS.BuildFromText(pszAttr, NULL, NULL, 0);
    CodePage.BuildFromText(pszAttr, NULL, NULL, 0);

    Head.BuildFromText(pszBlock, NULL, NULL, 0);
    Body.BuildFromText(pszBlock, NULL, NULL, 0);


    pszTemp = pszBlock;

    return pszEnd;
    }






