/*
 *              WicPM - WarpIN Archive Creator for PM
 *                  (c) Copyright Teemu Ahola 2000
 *
 * gui_trad_cntr.cpp
 *  This file contains functions used for container initialization and
 *  other container specified job in traditional mode.
 *
 *  Copyright (C) 1999-2000 Teemu Ahola.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 */

#define INCL_WIN
#define INCL_PM
#include <os2.h>

#include <string.h>
#include <stdio.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"

#include "helpers\comctl.h"
#include "helpers\cnrh.h"
#include "helpers\linklist.h"
// #include <list>
#include "helpers\stringh.h"
#include "helpers\xstring.h"

#include "base\bs_base.h"           // V0.9.14 (2001-07-07) [umoeller]
#include "base\bs_string.h"
#include "base\bs_list.h"       // V0.9.14 (2001-07-07) [umoeller]

#include "wicpm\script.hpp"

#include "wiarchive\wiarchive.h"

#include "wicpm\wicpm.h"
#include "wicpm\gui_objwnd.h"
#include "wicpm\gui_trad_cntr.h"
#include "wicpm\gui_wicpm.h"
#include "wicpm\wicpmres.h"



/****************************************************************************
 *@@ InvalidateRecords
 *  Used for invalidating container record structures which are not valid
 *  anymore.
 */
INT guiInvalidateRecords(HWND hwndContainer, PVOID pvRecordArray, USHORT usRecords)
    {
    WinSendMsg (hwndContainer, CM_INVALIDATERECORD, MPFROMP(pvRecordArray),
                MPFROM2SHORT(usRecords, CMA_TEXTCHANGED));
    return RET_CNTR_OK;
    }




/****************************************************************************
 *@@ SetContainerFieldInfo
 *  Sets container field infos.
 *
 */
USHORT guiSetContainerFieldInfo(PCONTAINERINFO pContainerInfo)
    {
    FIELDINFOINSERT fieldInfoInsert;
    PFIELDINFO pFieldInfo;
    CNRINFO cnrInfo;
    USHORT usReturnCode;
    HWND hwndContainer = pContainerInfo->hwndContainer;

    if (!pContainerInfo)
        return RET_CNTR_NULLPOINTERERROR;

    pFieldInfo = pContainerInfo->pFieldInfo;

    cnrInfo.flWindowAttr = CV_DETAIL | CA_DETAILSVIEWTITLES;

    // Use title in container and set the title text.
    cnrInfo.pszCnrTitle = pContainerInfo->pszContainerTitle;
    if (cnrInfo.pszCnrTitle)
        cnrInfo.flWindowAttr |= CA_CONTAINERTITLE | CA_TITLEREADONLY |
                                CA_TITLESEPARATOR;

    fieldInfoInsert.cb = sizeof(FIELDINFOINSERT);
    fieldInfoInsert.pFieldInfoOrder = (PFIELDINFO)CMA_FIRST;
    fieldInfoInsert.cFieldInfoInsert = pContainerInfo->ulColumns;
    fieldInfoInsert.fInvalidateFieldInfo = FALSE;

    // Insert field infos to the container.
    WinSendMsg (hwndContainer, CM_INSERTDETAILFIELDINFO,
                MPFROMP (pFieldInfo), MPFROMP (&fieldInfoInsert));
    // Set containers infos.
    WinSendMsg (hwndContainer, CM_SETCNRINFO, &cnrInfo,
               MPFROMLONG (CMA_FLWINDOWATTR | CMA_CNRTITLE));

    // Set column sizes to match column text sizes.
    WinSendMsg (hwndContainer, CM_INVALIDATERECORD, MPFROMP (NULL),
                MPFROM2SHORT (0, CMA_REPOSITION | CMA_ERASE | CMA_TEXTCHANGED));

    return RET_CNTR_OK;
    }





/****************************************************************************
 *@@ InitializePckContainer
 *  Allocates memory for package container control field infos and sets
 *  container field info.
 */
INT guiInitializePckContainer(PCONTAINERINFO pContainerInfo, PNLSSTRINGS pStrings)
    {
    PFIELDINFO pFirstFieldInfo, pFieldInfo;
    PSZ pszString;
    HWND hwndContainer = pContainerInfo->hwndContainer;
    pContainerInfo->ulColumns = CONT_COLUMN_PCK;

    if (!pContainerInfo)
        {
        return RET_CNTR_NULLPOINTERERROR;
        }

    // Allocate memory for detail field info.
    if (!(pFirstFieldInfo = (PFIELDINFO) WinSendMsg (hwndContainer, CM_ALLOCDETAILFIELDINFO,
        MPFROMLONG (pContainerInfo->ulColumns), NULL)))
        {
        return RET_CNTR_MEMALLOCERROR;
        }
    pFieldInfo = pFirstFieldInfo;

    // Package number column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_ULONG | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Package";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, sPackage);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // Package name column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Package ID";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, pszPackageName);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // Application name column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Application name";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, pszApplication);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // Author name column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Author";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, pszAuthor);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // Version column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Version";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, pszVersion);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // Target path column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Target path";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, pszTargetPath);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // File number column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_ULONG | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Files";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, ulFiles);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    // Total size of files  column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_ULONG | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Total size";
    pFieldInfo->offStruct = FIELDOFFSET (PCKRECORD, ulTotalSize);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    pContainerInfo->pFieldInfo = pFirstFieldInfo;
    pContainerInfo->pszContainerTitle = "Packages";


    return guiSetContainerFieldInfo(pContainerInfo);
    }





/****************************************************************************
 *@@ InitializeFileContainer
 *  Allocates memory for file container control field infos and sets
 *  container field info.
 */
INT guiInitializeFileContainer(PCONTAINERINFO pContainerInfo, PNLSSTRINGS pStrings)
    {
    PFIELDINFO pFirstFieldInfo, pFieldInfo;
    PSZ pszString;
    HWND hwndContainer = pContainerInfo->hwndContainer;
    pContainerInfo->ulColumns = CONT_COLUMN_FILE;

    if (!pContainerInfo)
        {
        return RET_CNTR_NULLPOINTERERROR;
        }

    // Allocate memory for detail field info.
    if (!(pFirstFieldInfo = (PFIELDINFO) WinSendMsg (hwndContainer, CM_ALLOCDETAILFIELDINFO,
        MPFROMLONG (pContainerInfo->ulColumns), NULL)))
        {
        return RET_CNTR_MEMALLOCERROR;
        }
    pFieldInfo = pFirstFieldInfo;

        // Filename column of the container.
    pFieldInfo = pFirstFieldInfo;
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Filename";
    pFieldInfo->offStruct = FIELDOFFSET (FILERECORD, pszFileName);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

        // File path column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "File path";
    pFieldInfo->offStruct = FIELDOFFSET (FILERECORD, pszFilePath);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

        // File install path column of the container.
        // This field is editable for user.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "File install path";
    pFieldInfo->offStruct = FIELDOFFSET (FILERECORD, pszFileInstallPath);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

        // File size column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_ULONG | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR  | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "File size";
    pFieldInfo->offStruct = FIELDOFFSET (FILERECORD, ulFileSize);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

    pContainerInfo->pFieldInfo = pFirstFieldInfo;
    pContainerInfo->pszContainerTitle = "Files";


    return guiSetContainerFieldInfo(pContainerInfo);
    }



/****************************************************************************
 *@@ InitializePageContainer
 *  Allocates memory for page container control field infos and sets
 *  container field info.
 */
INT guiInitializePageContainer(PCONTAINERINFO pContainerInfo, PNLSSTRINGS pStrings)
    {
    PFIELDINFO pFirstFieldInfo, pFieldInfo;
    PSZ pszString;
    HWND hwndContainer = pContainerInfo->hwndContainer;
    pContainerInfo->ulColumns = CONT_COLUMN_PAGE;

    if (!pContainerInfo)
        {
        return RET_CNTR_NULLPOINTERERROR;
        }

    // Allocate memory for detail field info.
    if (!(pFirstFieldInfo = (PFIELDINFO) WinSendMsg (hwndContainer, CM_ALLOCDETAILFIELDINFO,
        MPFROMLONG (pContainerInfo->ulColumns), NULL)))
        {
        return RET_CNTR_MEMALLOCERROR;
        }
    pFieldInfo = pFirstFieldInfo;

        // Page index column of the container.
    pFieldInfo = pFirstFieldInfo;
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_ULONG | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Page index";
    pFieldInfo->offStruct = FIELDOFFSET (PAGERECORD, usPageIndex);
    pFieldInfo->pUserData = NULL;
    pFieldInfo = pFieldInfo->pNextFieldInfo;

        // Page type column of the container.
    pFieldInfo->cb = sizeof (FIELDINFO);
    pFieldInfo->flData = CFA_STRING | CFA_HORZSEPARATOR | CFA_CENTER |
                         CFA_SEPARATOR | CFA_FIREADONLY | CFA_FITITLEREADONLY;
    pFieldInfo->pTitleData = "Page type";
    pFieldInfo->offStruct = FIELDOFFSET (PAGERECORD, pszPageType);
    pFieldInfo->pUserData = NULL;

    pContainerInfo->pFieldInfo = pFirstFieldInfo;

    return guiSetContainerFieldInfo(pContainerInfo);
    }




USHORT guiSetPckContainerRecords(HWND hwndContainer, PLINKLIST plDBPck)
    {
    ULONG cbRecordData;
    RECORDINSERT recIn;
    PPCKRECORD pPckRecord, pFirstPckRecord;
    DBPackage *pDBPck = NULL;
    PLISTNODE pNode = NULL;

    recIn.cb = sizeof (RECORDINSERT);
    recIn.pRecordOrder = (PRECORDCORE)CMA_END;
    recIn.pRecordParent = NULL;
    recIn.zOrder = CMA_TOP;
    recIn.fInvalidateRecord = FALSE;
    recIn.cRecordsInsert = (ULONG)lstCountItems(plDBPck);

    cbRecordData = (LONG)(sizeof (PCKRECORD) - sizeof (RECORDCORE));
    pFirstPckRecord = (PPCKRECORD)WinSendMsg (hwndContainer, CM_ALLOCRECORD, MPFROMLONG
                                  (cbRecordData), MPFROMLONG (lstCountItems(plDBPck)));

    pNode = lstQueryFirstNode(plDBPck);
    pPckRecord = pFirstPckRecord;
    while(pNode)
        {
        pDBPck = (DBPackage*)pNode->pItemData;
        pPckRecord->recordCore.cb = sizeof(RECORDCORE);
        pPckRecord->sPackage = pDBPck->usPckIndexAttr;
        pPckRecord->pszAuthor = pDBPck->szAuthor;
        pPckRecord->pszApplication = pDBPck->szApplication;
        pPckRecord->pszPackageName = pDBPck->szPackageID;
        pPckRecord->pszVersion = pDBPck->szVersion;
        pPckRecord->ulFiles = pDBPck->ulFiles;
        pPckRecord->ulTotalSize = pDBPck->ulTotalSize;
        pPckRecord->pszTargetPath = pDBPck->szPath;
        pPckRecord = (PPCKRECORD)pPckRecord->recordCore.preccNextRecord;
        pNode = pNode->pNext;
        }


    WinSendMsg (hwndContainer, CM_INSERTRECORD, MPFROMP (pFirstPckRecord), &recIn);

    WinSendMsg (hwndContainer, CM_INVALIDATERECORD, MPFROMP (NULL),
               MPFROM2SHORT (0, CMA_REPOSITION | CMA_TEXTCHANGED));

    return RET_CNTR_OK;
    }



/****************************************************************************
 *@@guiUpdatePckContRecord
 *  Updates package container record's data and invalidates container.
 *  If you are planning to update multiple records, use bInvalidate == TRUE
 *  in the last record so the container will not flicker.
 */
USHORT guiUpdatePckContRecord(HWND hwndContainer, // Handler of container.
                             PPCKRECORD pPckRecord, // Package record to be updated.
                             DBPackage *pDBPck, // New data for the package record.
                             BOOL bInvalidate // Invalidate enabling flag.
                             )
    {
    if (!pPckRecord || !pPckRecord)
        return RET_CNTR_NULLPOINTERERROR;

    pPckRecord->sPackage = pDBPck->usPckIndexAttr;
    pPckRecord->pszAuthor = pDBPck->szAuthor;
    pPckRecord->pszApplication = pDBPck->szApplication;
    pPckRecord->pszPackageName = pDBPck->szPackageID;
    pPckRecord->pszVersion = pDBPck->szVersion;
    pPckRecord->ulFiles = pDBPck->ulFiles;
    pPckRecord->ulTotalSize = pDBPck->ulTotalSize;
    pPckRecord->pszTargetPath = pDBPck->szPath;

    // If bInvalidate is TRUE invalidate whole container.
    if (bInvalidate)
        guiInvalidateRecords(hwndContainer, NULL, 0);

    return RET_CNTR_OK;
    }




USHORT guiSetFileContainerRecords(HWND hwndContainer, PLINKLIST plDBFile)
    {
    ULONG cbRecordData;
    RECORDINSERT recIn;
    PFILERECORD pFileRecord, pFirstFileRecord;
    DBFile *pDBFile = NULL;
    PLISTNODE pNode = NULL;

    recIn.cb = sizeof (RECORDINSERT);
    recIn.pRecordOrder = (PRECORDCORE)CMA_END;
    recIn.pRecordParent = NULL;
    recIn.zOrder = CMA_TOP;
    recIn.fInvalidateRecord = FALSE;
    recIn.cRecordsInsert = (ULONG)lstCountItems(plDBFile);

    cbRecordData = (LONG)(sizeof(FILERECORD) - sizeof(RECORDCORE));
    pFirstFileRecord = (PFILERECORD)WinSendMsg(hwndContainer, CM_ALLOCRECORD, MPFROMLONG
        (cbRecordData), MPFROMLONG (lstCountItems(plDBFile)));

    pNode = lstQueryFirstNode(plDBFile);
    pFileRecord = pFirstFileRecord;
    while(pNode)
        {
        pDBFile = (DBFile*)pNode->pItemData;
        pFileRecord->recordCore.cb = sizeof(RECORDCORE);
        pFileRecord->pszFileName = pDBFile->szFileName;
        pFileRecord->pszFilePath = pDBFile->szFilePath;
        pFileRecord->pszFileInstallPath = pDBFile->szFileInstallPath;
        pFileRecord->ulFileSize = pDBFile->ulFileSize;
        pFileRecord->pDBFile = pDBFile;
        pFileRecord = (PFILERECORD)pFileRecord->recordCore.preccNextRecord;
        pNode = pNode->pNext;
        }


    WinSendMsg(hwndContainer, CM_INSERTRECORD, MPFROMP (pFirstFileRecord), &recIn);

    WinSendMsg(hwndContainer, CM_INVALIDATERECORD, MPFROMP (NULL),
               MPFROM2SHORT (0, CMA_REPOSITION | CMA_TEXTCHANGED));

    return RET_CNTR_OK;
    }




/****************************************************************************
 *@@guiSetPageContainerRecords
 *  Sets page container records.
 */
USHORT guiSetPageContainerRecords(HWND hwndContainer, // Handler of container.
                                  list<ScriptPage*> *listPageAddToCntr, // List of pointers to page objects.
                                  USHORT usItemsCount // The size of the list.
                                  )
    {
    ULONG cbRecordData;
    RECORDINSERT recIn;
    PPAGERECORD pPageRecord, pFirstPageRecord;
    ScriptPage *tempPage = NULL;

    if (!listPageAddToCntr || usItemsCount == 0)
        return RET_CNTR_NULLPOINTERERROR;

    recIn.cb = sizeof (RECORDINSERT);
    recIn.pRecordOrder = (PRECORDCORE)CMA_END;
    recIn.pRecordParent = NULL;
    recIn.zOrder = CMA_TOP;
    recIn.fInvalidateRecord = FALSE;
    recIn.cRecordsInsert = (ULONG)usItemsCount;

    cbRecordData = (LONG)(sizeof(PAGERECORD) - sizeof(RECORDCORE));
    pFirstPageRecord = (PPAGERECORD)WinSendMsg(hwndContainer, CM_ALLOCRECORD, MPFROMLONG
        (cbRecordData), MPFROMLONG (usItemsCount));

    // Get first and last item of the list.
    list<ScriptPage*>::iterator PageStart = listPageAddToCntr->begin(),
                                PageEnd = listPageAddToCntr->end();

    pPageRecord = pFirstPageRecord;
    for (; PageStart != PageEnd; PageStart++)
        {
        tempPage = *PageStart;
        pPageRecord->recordCore.cb = sizeof(RECORDCORE);
        switch (tempPage->ReturnPageType())
            {
            case PAGE_TYPES_TEXT:
                {
                pPageRecord->pszPageType = "TEXT";
                break;
                }
            case PAGE_TYPES_README:
                {
                pPageRecord->pszPageType = "README";
                break;
                }
            case PAGE_TYPES_CONTAINER:
                {
                pPageRecord->pszPageType = "CONTAINER";
                break;
                }
            case PAGE_TYPES_CONFIGURE:
                {
                pPageRecord->pszPageType = "CONFIGURE";
                break;
                }
            }
        pPageRecord->usPageIndex = tempPage->ReturnPageIndex();
        pPageRecord = (PPAGERECORD)pPageRecord->recordCore.preccNextRecord;
        }


    WinSendMsg(hwndContainer, CM_INSERTRECORD, MPFROMP (pFirstPageRecord), &recIn);

    WinSendMsg(hwndContainer, CM_INVALIDATERECORD, MPFROMP (NULL),
               MPFROM2SHORT (0, CMA_REPOSITION | CMA_TEXTCHANGED));

    return RET_CNTR_OK;
    }


/****************************************************************************
 *@@guiUpdatePageContRecord
 *  Updates page container record's data and invalidates container.
 *  If you are planning to update multiple records, use bInvalidate == TRUE
 *  in the last record so the container will not flicker.
 */
USHORT guiUpdatePageContRecord(HWND hwndContainer, // Handler of container.
                               PPAGERECORD pPageRecord, // Page record to be updated.
                               ScriptPage *pPage, // New data for the page record.
                               BOOL bInvalidate // Invalidate enabling flag.
                               )
    {
    if (!pPageRecord || !pPage)
        return RET_CNTR_NULLPOINTERERROR;

    switch (pPage->ReturnPageType())
            {
            case PAGE_TYPES_TEXT:
                {
                pPageRecord->pszPageType = "TEXT";
                break;
                }
            case PAGE_TYPES_README:
                {
                pPageRecord->pszPageType = "README";
                break;
                }
            case PAGE_TYPES_CONTAINER:
                {
                pPageRecord->pszPageType = "CONTAINER";
                break;
                }
            case PAGE_TYPES_CONFIGURE:
                {
                pPageRecord->pszPageType = "CONFIGURE";
                break;
                }
            }
        pPageRecord->usPageIndex = pPage->ReturnPageIndex();

    // If bInvalidate is TRUE invalidate whole container.
    if (bInvalidate)
        guiInvalidateRecords(hwndContainer, NULL, 0);

    return RET_CNTR_OK;
    }







/******************************************************************************
 *@@guiRemoveContainerRecords
 *  Removes container records which are listed in an array of pointers to
 *  RECORDCORE structures.
 */
USHORT guiRemoveContainerRecords(HWND hwndContainer, // Handler of cntr.
                                 PVOID pvRecords, // An array of pointers.
                                 SHORT sRecords // Number of records.
                                 )
    {

    WinSendMsg(hwndContainer, CM_REMOVERECORD, MPFROMP(&pvRecords),
                MPFROM2SHORT((USHORT) sRecords, CMA_FREE | CMA_INVALIDATE));
    return RET_CNTR_OK;
    }
