/**
 * $Id: wiarchive.cpp,v 1.9 2001/07/01 20:49:04 jens Exp $
 *
 * Archive handling class for WarpIN, WIArchive. The class handles all the
 * grunt work of managing a compressed archive, including pretty advanced file
 * management. Just create a new instance of this class for each and every
 * archive file that needs to be processed.
 *
 * WarpIN utilizes a concept called 'packages', where files are separated in
 * logical groups.  This could be used in a situation where the user can
 * choose what should be installed  Let's say an archive contains the source,
 * the binaries and the manual for an application. A real power user might
 * just want the source and compile the application herself, but an ordinary
 * user might want only the binaries and the documentation. This is very easy
 * to accomplish with WarpIN. Just add the files into separate packages, and
 * at install time only install the selected packages.
 *
 * This file Copyright (C) 1999-2001 Jens B&auml;ckman, Ulrich M&ouml;ller
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, in version 2 as it comes in the COPYING
 * file of this distribution.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

// BUG:  In the store() method, you always lose the first 8 bytes somehow. It
//       just drives me crazy - where is that damn bug?!?
// TODO: The CBM_ERR_WRITE and CBM_ERR_READ messages are never sent, although
//       they probably should be.

#include "wiarchive2/wiarchive.h"

/****************************************************************************
 * Constructor
 */
WIArchive::WIArchive()
{
    modified     = false;
    callback     = NULL;
    extendedData = NULL;
    lastError    = 0;
    tempName[0]  = 0;

    // WARNING! Light wizardry ahead! Creating CRC table for later usage.  If
    // you don't know about CRC - don't bother trying to understand this very
    // small code segment, just leave it intact and everything will be fine.
    unsigned long i, j, r;
    for(i = 0; i < 256; i++) {
        r = i;
        for (j = 0; j < 8; j++)
            if (r & 1) r = (r >> 1) ^ 0xEBD88320;
            else       r >>= 1;
        crcTable[i] = r;
    }
}

/****************************************************************************
 * Destructor
 */
WIArchive::~WIArchive()
{
    close();
}

/****************************************************************************
 * Adds a file to be processed and added to the archive into one of the
 * internal lists. No file handling action is actually taken until the
 * archive is closed. Reasons for this is extremely reduced workload and a
 * significantly smaller risk to corrupt already existing archives when only
 * modifying contents.
 *
 * Arguments - fileName: Name and path of the file to add
 *             name: Name as it will appear in the archive
 *             package: Which package number the file belongs to
 *    Return - false if the package specified does not exist
 */
bool WIArchive::addFile(const char *fileName, const char *name, short package)
{
    list<WIFile *>::iterator current, end;
    list<WIPackHeader *>::iterator currentPackage, endPackage;
    WIFile *entry;

    // Locate the package associated with this file
    currentPackage = packList.begin();  endPackage = packList.end();
    for (; currentPackage != endPackage; currentPackage++) {
        if ((**currentPackage).number == package)  break;
    }
    if (currentPackage == endPackage)  return false;

    // Create a new entry in the todo list
    entry = new WIFile(name, fileName, package);
    todoList.push_back(entry);
    modified = true;
    (**currentPackage).files++;     // Increase file count for package

    // If the file exists in the current file list, we shall remove it
    current = fileList.begin();  end = fileList.end();
    for (; current != end; current++)
    {
        if ((**current).name == name && (**current).package == package) {
            delete (WIFile *)*current;
            fileList.erase(current);
            (**currentPackage).files--;     // Number of files for package unchanged
            break;
        }
    }

    return true;
}

/****************************************************************************
 * Creates a new package or changes the name of a previously created one
 *
 * Arguments - package: Which package number this is all about
 *             name: Name of the package
 */
void WIArchive::addPackage(short package, const char *name)
{
    list<WIPackHeader *>::iterator current, end;
    WIPackHeader *p;
    int i = 0;

    // See if we already have a package with this number
    current = packList.begin ();  end = packList.end ();
    for (; current != end; current++)
        if ((**current).number == package)
            {  p = *current;  i++;  break;  }
    if (i == 0)  p = new WIPackHeader();

    // Store this information
    p->number = package;
    strcpy(p->name, name);
    if (i == 0)  packList.push_back(p);
}

/****************************************************************************
 * Release all the memory we have allocated
 */
void WIArchive::cleanup()
{
    list<WIFile *>::iterator current, end, temp;

    // Reset all of our state variables to their original values
    extendedData   = NULL;
    modified       = false;
    archiveName[0] = 0;
    script         = NULL;
    tempName[0]    = 0;
    archivePos     = 0;

    // Empty all the lists we have created
    current = fileList.begin();  end = fileList.end();
    while (current != end) {
        temp = current;  temp++;
        delete (WIFile *)*current;
        fileList.erase(current);
        current = temp;
    }
    current = todoList.begin();  end = todoList.end();
    while (current != end) {
        temp = current;  temp++;
        delete (WIFile *)*current;
        fileList.erase(current);
        current = temp;
    }

    // Do the same for other allocated resources
    if (extendedData != NULL) {
        delete [] extendedData;
        extendedData = NULL;
    }
}

/****************************************************************************
 * Close the archive and process all changes made (if any)
 *
 * Arguments - performChanges: Update the archive if it is opened with write
 *                             access and any changes have been made
 *                             (default = true)
 *    Return - True if all went OK, otherwise false
 */
bool WIArchive::close(bool performChanges)
{
    if (performChanges == true) {
        if (modified == true && archiveAccess == WRITE) {
            if (update() == false)  return false;
        }
    }

    cleanup();
    return true;
}

/****************************************************************************
 * Apply the BZ2 compression algorithm on this file.  Do your worst to make
 * it small, and store the result in the destination file.
 *
 * Arguments - header: File header being processed
 *             in: Source file
 *             out: Destination file
 *    Return - True if all went well, otherwise false
 */
bool WIArchive::compress(WIFileHeader &header, fstream &in, fstream &out)
{
    unsigned long crc, i;
    char *inBuffer, *outBuffer;
    bz_stream z;            // BZ2 compression library control structure
    int status, count;
    unsigned long readBytes;

    // Do some basic initialization before we get started
    header.compSize = 0;
    in.seekg(0, ios::end);
    header.origSize = in.tellg();
    in.seekg(0, ios::beg);
    header.method   = 1;    // Method = 1: Compress with BZ2
    inBuffer        = new char[WIARCHIVE_BUFFERSIZE];
    outBuffer       = new char[WIARCHIVE_BUFFERSIZE];
    crc             = 0xFFFFFFFF;
    readBytes       = 0;

    // Initialize all the variables, flags, pointers and all the other things
    // that the compression library needs in order to function.
    z.bzalloc   = NULL;                 // Special routine for allocation memory
    z.bzfree    = NULL;                 // ------- || -------- freeing memory
    z.opaque    = NULL;                 // First arguments for the above routines
    z.avail_in  = 0;                    // Number of bytes available for reading
    z.avail_out = WIARCHIVE_BUFFERSIZE; // ------------- || ------------ storage
    z.next_out  = outBuffer;            // Where to store all the compressed bits
    BZ2_bzCompressInit (&z, 9, 0, 0);   // Initialize the compression library
                                        //   param0: BZ2 control structure
                                        //   param1: Which block size to use. Higher
                                        //           value gives better compression
                                        //           but raises memory used. Valid
                                        //           values are 1 -> 9.
                                        //   param2: Specifies how much debug info
                                        //           to show. Valid values are 0 -> 4.
                                        //   param3: Strange parameter. Just leave
                                        //           this one as zero and everything
                                        //           will work just fine. Promise.

    // Begin the compression-o-rama circus now. This is just too wierd to give
    // any comments. If you understand it, you may change stuff here. If that
    // is not the case, just leave it intact and hope it works.
    for (;;) {
        // Show the progress in bytes we have made so far
        if (callback != NULL) {
            callback(CBM_PERCENTAGE, readBytes, &header);
        }

        // Read file fragment from disk
        if (z.avail_in == 0) {
            z.next_in = inBuffer;
            in.read(inBuffer, WIARCHIVE_BUFFERSIZE);
            if (in.bad()); // TODO - Read error handling
            z.avail_in = in.gcount();
            readBytes += z.avail_in;
        }
        if (z.avail_in == 0 || z.avail_in == 0xFFFFFFFF)  break;

        // Calculate the CRC value of this block
        for (i = 0; i < z.avail_in; i++) {
            crc = crcTable[(crc ^ inBuffer[i]) & 0xFF] ^ (crc >> 8);
        }

        // Do the real compression things
        status = BZ2_bzCompress(&z, BZ_RUN);
        if (status < 0) {
            // We have some sort of situation here.
            if (callback != NULL)  callback(CBM_ERR_COMPLIB, status, &header);
            return false;
        }
        count = WIARCHIVE_BUFFERSIZE - z.avail_out;
        if (count > 0) {
            out.write(outBuffer, count);
            header.compSize += count;
            z.next_out  = outBuffer;
            z.avail_out = WIARCHIVE_BUFFERSIZE;
        }
    }

    // Easy, right? Flush the pending compressed data and we are done here.
    if (readBytes > 0) {
        do {
            status = BZ2_bzCompress(&z, BZ_FINISH);
            count  = WIARCHIVE_BUFFERSIZE - z.avail_out;
            if (count > 0) {
                out.write(outBuffer, count);
                header.compSize += count;
                z.next_out  = outBuffer;
                z.avail_out = WIARCHIVE_BUFFERSIZE;
            }
        } while (status != BZ_STREAM_END);
    }

    // It seems like we're done here. Clean things up here and go home.
    BZ2_bzCompressEnd (&z);
    delete [] inBuffer;
    delete [] outBuffer;
    crc ^= 0xFFFFFFFF;
    header.crc = crc;
    return true;
}

/****************************************************************************
 * Let the BZ2 library decompress a previously compressed and stored file.
 *
 * Arguments - header: File header being processed
 *             in: Source file
 *             out: Destination file
 *             package: The package which the file belongs to
 *             bytesDone: How many bytes that already has been decompressed
 *                        of this package
 *    Return - True if everything went file, otherwise false
 */
bool WIArchive::expand(WIFileHeader &header, fstream &in, fstream &out,
                       WIPackHeader &package, long bytesDone)
{
    bz_stream z;
    char *inBuffer, *outBuffer;
    unsigned long i, bytes, bytesProcessed, crc;
    long bytesLeft;
    int status;

    // Allocate memory and such small things
    inBuffer       = new char[WIARCHIVE_BUFFERSIZE];
    outBuffer      = new char[WIARCHIVE_BUFFERSIZE];
    crc            = 0xFFFFFFFF;
    bytesLeft      = header.compSize;
    bytesProcessed = 0;

    // Initialize all the variables, flags, pointers and all the other things
    // that the compression library needs in order to function.
    z.bzalloc   = NULL;                 // Special routine for allocation memory
    z.bzfree    = NULL;                 // ------- || -------- freeing memory
    z.opaque    = NULL;                 // First arguments for the above routines
    z.avail_in  = 0;                    // Number of bytes available for reading
    z.avail_out = WIARCHIVE_BUFFERSIZE; // ------------- || ------------ storage
    z.next_out  = outBuffer;            // Where to store all the compressed bits
    BZ2_bzDecompressInit (&z, 0, 0);    // Initialize the compression library
                                        //   param0: BZ2 control structure
                                        //   param1: Specifies how much debug info
                                        //           to show. Valid values are 0 -> 4.
                                        //   param2: If this value is set to 1, the
                                        //           memory requirements for doing all
                                        //           decompression are halved, but so
                                        //           is the speed. Just leave it at 0.

    // Oh yes, it all begins here - decompressing file contents...
    if (callback != NULL) {
        callback(CBM_PERCENTAGE, bytesDone, &header);
    }
    while (bytesLeft) {
        // Read file information
        if (bytesLeft > WIARCHIVE_BUFFERSIZE)
            bytes = WIARCHIVE_BUFFERSIZE;
        else
            bytes = bytesLeft;
        bytesLeft -= bytes;
        in.read(inBuffer, bytes);
        z.avail_in = bytes;
        z.next_in  = inBuffer;

        // While we have bytes left in the inbuffer, let's continue working
        while (z.avail_in > 0) {
            // Decompress more information
            status = BZ2_bzDecompress(&z);
            if (status < 0) {
                // Oops, BZ just told us about an error...
                if (callback != NULL) {
                    callback(CBM_ERR_COMPLIB, status, &header);
                }
                return false;
            }

            if (WIARCHIVE_BUFFERSIZE - z.avail_out) {
                // Write decompressed data to file
                out.write(outBuffer, WIARCHIVE_BUFFERSIZE - z.avail_out);
                bytesProcessed += (WIARCHIVE_BUFFERSIZE - z.avail_out);

                // Calculate CRC value
                for (i = 0; i < (WIARCHIVE_BUFFERSIZE - z.avail_out); i++) {
                    crc = crcTable[(crc ^ outBuffer[i]) & 0xFF] ^ (crc >> 8);
                }

                // Send a message to callback function with percentage value
                if (callback != NULL) {
                    callback(CBM_PERCENTAGE, bytesDone + bytesProcessed, &header);
                }

                z.avail_out = WIARCHIVE_BUFFERSIZE;
                z.next_out  = outBuffer;
            }
        }
        if (status == BZ_STREAM_END)  break;
    }

    // Flush the remaining uncompressed data
    while ((header.compSize) != 0 && (status != BZ_STREAM_END)) {
        status = BZ2_bzDecompress(&z);

        // Write decompressed data to file
        out.write(outBuffer, WIARCHIVE_BUFFERSIZE - z.avail_out);
        bytesProcessed += (WIARCHIVE_BUFFERSIZE - z.avail_out);

        // Calculate CRC value
        for (i = 0; i < (WIARCHIVE_BUFFERSIZE - z.avail_out); i++) {
            crc = crcTable[(crc ^ outBuffer[i]) & 0xFF] ^ (crc >> 8);
        }

        // Send a message to callback function with percentage value
        if (callback != NULL) {
            callback(CBM_PERCENTAGE, bytesDone + bytesProcessed, &header);
        }

        z.avail_out = WIARCHIVE_BUFFERSIZE;
        z.next_out  = outBuffer;
    }

    // We're done.  Yee-hah...
    BZ2_bzDecompressEnd(&z);
    delete [] inBuffer;
    delete [] outBuffer;
    if (callback != NULL) {
        callback(CBM_PERCENTAGE, bytesDone + header.origSize, &header);
    }

    // Check if the file is calculated
    crc ^= 0xFFFFFFFF;
    if (crc != header.crc) {
        // Noooo! The compressed file is/was/will be/wants to be broken!
        if (callback != NULL) {
            callback(CBM_ERR_CRC, 0, &header);
        }
        return false;
    } else {
        return true;
    }
}

/****************************************************************************
 * Read a stored file from archive and write it to disk.
 *
 * Arguments - header: File header being processed
 *             in: Source file
 *             out: Destination file
 *             package: The package which the file belongs to
 *             bytesDone: How many bytes that already has been decompressed
 *                        of this package
 *    Return - True if everything went file, otherwise false
 */
bool WIArchive::extract(WIFileHeader &header, fstream &in, fstream &out,
                        WIPackHeader &package, long bytesDone)
{
    unsigned long crc, bytes, i;
    long size;
    char *buffer;

    // Do some basic initialization before we get started
    buffer = new char[WIARCHIVE_BUFFERSIZE];
    size   = header.compSize;
    crc    = 0xFFFFFFFF;

    // Copy all the information
    if (callback != NULL) {
        callback(CBM_PERCENTAGE, bytesDone, &header);
    }
    while (size) {
        // Find out how many bytes to copy
        if (size > WIARCHIVE_BUFFERSIZE)
            bytes = WIARCHIVE_BUFFERSIZE;
        else
            bytes = size;
        size -= bytes;

        // Read, calculate CRC and write...
        in.read(buffer, bytes);
        for (i = 0; i < bytes; i++) {
            crc = crcTable[(crc ^ buffer[i]) & 0xFF] ^ (crc >> 8);
        }
        out.write(buffer, bytes);

        // Report progress
        if (callback != NULL) {
            callback(CBM_PERCENTAGE, bytesDone + header.compSize - size, &header);
        }
    }

    // Extraction finished! Check if CRC matches...
    delete [] buffer;
    if (callback != NULL) {
        callback(CBM_PERCENTAGE, bytesDone + header.origSize, &header);
    }
    crc ^= 0xFFFFFFFF;
    if (crc != header.crc) {
        // Oh, no... The archive seems to be broken in some way.
        if (callback != NULL) {
            callback(CBM_ERR_CRC, 0, &header);
        }
        return false;
    } else {
        return true;
    }
}

/****************************************************************************
 * This calls pfnWICallback for all files which belong to the specified
 * package. If you want to list all the files, you need to call this method
 * for each package (by going thru the package list).
 *
 * This method accesses the archive file on disk, because the class-internal
 * list does not have the WIFileHeader's, which we therefore must read in
 * here. For large archives, this function can take a bit of time.
 *
 * The callback function is called with these arguments:
 *   pwifh: current file header
 *   ulUser: the userSpecified argument that was used to call this function.
 *           Can be used to store some external pointers or similar.
 *
 * Arguments - package:
 *             callback:
 *             userSpecified:
 *    Return - The number of files found (0 -> n). If a -1 is returned, some
 *             sort of error occured.
 */
short WIArchive::forAllFiles(WIPackHeader *package,
                             PFNFORALLFILESCALLBACK callback,
                             unsigned long userSpecified)
{
    WIFileHeader fileHeader;
    short foundFiles = 0;
    short files;

    if (package == NULL)  return 0;

    // Set the read position to the first file header
    archive.seekg(package->pos, ios::beg);
    files = package->files;

    // Examine all files
    while (files--) {
        // Read the fileheader and verify it
        fileHeader.read(archive);
        if (fileHeader.verify() == false)  return (-1);
        foundFiles++;

        // It seems to be OK - send it to the callback
        (*callback)(&fileHeader, userSpecified);
        archive.seekg(fileHeader.compSize, ios::cur);
    }

    return foundFiles;
}

/****************************************************************************
 * Creates directories necessary for installation
 *
 * Arguments - path: The path for the file
 */
void WIArchive::makeDirectories(const char *path)
{
    char tempName[256];
    char *cp;
    char c;

    // Copy the string before the code begins
    strcpy(tempName, path);
    cp = tempName;

    // ----- OS/2 and Windows algorithm
    #if defined (XP_OS2) || defined (XP_WIN)
        // Check if there are any directories involved in this path
        if ((strchr(tempName, '\\') != 0) || (strchr(tempName, '/') != 0)) {
            // Check if we have a drive letter in the path. If so, advance past
            // it  before we do anything else aroung
            if (*(cp+1) == ':')  cp += 3;

            // Now, create the directories
            while (*cp != '\0') {
                if (*cp == '\\' || *cp == '/') {
                    c = *cp;
                    *cp = '\0';
                    #if defined (XP_OS2_VACPP)
                        mkdir (tempName);
                    #elif defined (XP_WIN)
                        CreateDirectory(tempName, NULL);
                    #else
                        mkdir (tempName, 0);
                    #endif
                    *cp = c;
                }
                cp++;
            }
        }
    #endif // XP_OS2 || XP_WIN

    // ----- General UNIX creation algorithm
    #if defined(XP_UNIX)
        // Check if there are any directories involved in this path
        if (strchr(tempName, '/') != 0) {
            // Yes, there is a path here. Create the directories.
            while (*cp != '\0') {
                if (*cp == '/') {
                    c = *cp;
                    *cp = '\0';
                    mkdir (tempName, 0);
                    *cp = c;
                }
                cp++;
            }
        }
    #endif // XP_UNIX
}

/****************************************************************************
 * Open an archive file and scan for packages and files.  If the package
 * does not exist, the function return false if the access is set to READ.
 * If the file does exist and access is set to WRITE, we create a temporary
 * archive that recieves all the changes.  After all modifications have been
 * done, the old archive is removed and replaced by the new one.
 *
 * Arguments - name: Archive file name
 *             arcAccess: One of the two access modes WIArchive::READ and
 *                        WIArchive::WRITE.
 *    Return - true if everything went well, otherwise false
 */
bool WIArchive::open(const char *name, access arcAccess)
{
    strcpy(archiveName, name);
    archiveAccess = arcAccess;

    archive.open(archiveName, ios::in | ios::binary);
    if (archive.is_open()) {
        // The file exists - check if it's a valid archive
        if (readHeader (archive) == false) {
            return false;           // Apparently it wasn't valid
        } else {
            // Read all the files and packages in this archive
            readPackages(archive);
            readFiles(archive);
        }

        // If we are allowed to read and write, then create a new archive
        if (archiveAccess == WRITE)
        {
            if (archive.is_open())  archive.close();
            oldArchive.open(archiveName, ios::in | ios::binary);

            // Create a temporary archive
            char *cp, tempSpace[256];
            strcpy(tempSpace, archiveName);
            #if defined(XP_OS2) || defined (XP_WIN)
                cp = strrchr(tempSpace, '\\');
                if (cp == NULL) {
                    if (tempSpace[1] == ':')
                        cp = &tempSpace[2];
                    else
                        cp = tempSpace;
                }
            #endif // XP_OS2 || XP_WIN
            #if defined(XP_UNIX)
                cp = strrchr (tempSpace, '/');
                if (cp == NULL)  cp = tempSpace;
            #endif // XP_UNIX
            strcat (cp, "_witemp_");
            strcpy(tempName, tempSpace);
            archive.open(tempName, ios::out | ios::binary);
            if (archive.bad())  return false;
        }
    } else {
        // Are we permitted to create a new archive?
        if (archiveAccess == READ) {
            return false;           // No, we're obviously not
        }
        if (archive.is_open())  archive.close();
        archive.open(archiveName, ios::out | ios::binary);
        if (archive.bad())  return false;
    }

    // Everything went well - return a true
    return true;
}


/****************************************************************************
 * Reads all the file headers from file and store them in a nice list
 *
 * Arguments - file: The archive file to scan for headers
 */
void WIArchive::readFiles(fstream &file) {
    list<WIPackHeader *>::iterator current, end;
    WIFileHeader fileHeader;
    WIFile *wifile;
    unsigned long pos;
    int i;

    // Walk through all the packages
    current = packList.begin ();  end = packList.end ();
    for (; current != end; current++) {
        i = (**current).files;
        pos = (**current).pos + archivePos; // There could be an auto-extract thing in
                                            // front of the archive - therefore this
        file.seekg(pos, ios::beg);
        while (i--) {
            fileHeader.read(file);
            wifile = new WIFile;
            wifile->package = fileHeader.package;
            wifile->name = fileHeader.name;
            fileList.push_back(wifile);
            file.seekg(fileHeader.compSize, ios::cur);
        }
    }
}

/****************************************************************************
 * Scan the archive for a header.  If we do not find it within the first
 * 500k bytes, let's assume that this isn't a valid archive.  On the other
 * hand, if we find the header we check if we have a correct version and
 * decompress any script we may find.
 *
 * Arguments - file: File to read header from
 *    Return - true if everything went file, false if not.
 */
bool WIArchive::readHeader(fstream &file)
{
    bz_stream z;
    long l;

    // Scan the first 500kb for a valid archive header
    for (l = 0; l < 500000; l++) {
        file.seekg (l, ios::beg);
        header.read(file);
        if (header.verify() == false)  continue;
        if (header.revisionNeeded > WIARCHIVE_REVISION) {
            lastError = WIERR_OUTDATED_ARCHIVE;
            return false;
        }
        break;
    }
    if (l >= 500000) {
        // Sorry, no correct archive header found
        lastError = WIERR_INVALID_HEADER;
        return false;
    }

    // It seems like we have a perfectly healthy archive here.  Let's do all
    // the other stuff we do with healthy archives!
    archivePos = l;
    if (header.compScriptSize > 0) {
        // Apparently, we have a compressed installation script here!  Why not
        // be a good chap an decompress it?
        char *tempBuffer = new char[header.compScriptSize];
        char *tempScript = new char[header.origScriptSize + 1];
        file.read (tempBuffer, header.compScriptSize);
        if (file.bad()) ;  // TODO: Error handling!

        // ----- Decompress the installation script
        z.bzalloc   = 0;
        z.bzfree    = 0;
        z.opaque    = NULL;
        z.bzalloc   = NULL;
        z.bzfree    = NULL;
        z.next_in   = tempBuffer;
        z.avail_out = header.origScriptSize + 1;
        z.avail_in  = header.compScriptSize;
        z.next_out  = tempScript;
        BZ2_bzDecompressInit(&z, 0, 0);
        while (BZ2_bzDecompress (&z) != BZ_STREAM_END);
        BZ2_bzDecompressEnd(&z);
        tempScript[header.origScriptSize] = '\0';
        script = tempScript;

        // Deallocate memory
        delete [] tempBuffer;
        delete [] tempScript;
    }

    // Check for extra data
    if (header.extendedDataSize > 0) {
        // Woohoo! Extended data! Read and store, please.
        extendedData = new char[header.extendedDataSize];
        file.read (extendedData, header.extendedDataSize);
    }

    return true;
}

/****************************************************************************
 * Scans an archive for package headers and puts them into a list
 *
 * Arguments - file: Archive file to read from
 */
void WIArchive::readPackages(fstream &file) {
    WIPackHeader *p;
    int i;

    i = header.packs;
    while (i--) {
        p = new WIPackHeader();
        p->read(file);
        packList.push_back(p);
    }
}

/****************************************************************************
 * Remove a file from the archive.  As usual, no actual removing takes place
 * until the close is performed.
 *
 * Arguments - package: The package number to search
 *             filename: Which file to remove
 */
void WIArchive::removeFile(short package, const char *filename)
{
    list<WIPackHeader *>::iterator currentPackage, endPackage;
    list<WIFile *>::iterator currentFile, endFile;

    // Locate the package associated with this file
    currentPackage = packList.begin();  endPackage = packList.end();
    for (; currentPackage != endPackage; currentPackage++) {
        if ((**currentPackage).number == package)  break;
    }

    // We have two lists to check for file existance - the ToDo list and the
    // archive master file list. Let's begin by examining the ToDo list.
    currentFile = todoList.begin();  endFile = todoList.end();
    for (; currentFile != endFile; currentFile++) {
        if ((**currentFile).name == filename && (**currentFile).package == package) {
            delete (WIFile *)*currentFile;
            todoList.erase(currentFile);
            (**currentPackage).files--;     // Decrease file count for package
            break;
        }
    }

    // OK, now scan the master file list
    currentFile = fileList.begin();  endFile = fileList.end();
    for (; currentFile != endFile; currentFile++)
    {
        if ((**currentFile).name == filename && (**currentFile).package == package) {
            delete (WIFile *)*currentFile;
            fileList.erase(currentFile);
            modified = true;    // Files removed, therefore changed
            (**currentPackage).files--;     // Decrease file count for package
            break;
        }
    }
}

/****************************************************************************
 * Remove a package (and all associated files) from the archive.
 *
 * Arguments - packageNumber: The wanted package number.
 */
void WIArchive::removePackage(short packageNumber)
{
    // TODO
    modified = true;
}

/****************************************************************************
 * Gives this WarpIN archive a new callback function. Also sends a test
 * message to the new CBF just for testing purposes.
 *
 * Arguments - function: The new callback function.
 */
void WIArchive::setCallback(PFNWICALLBACK function)
{
    callback = function;
    if (function != NULL)  callback(CBM_HELLO, 0, NULL);
}

/****************************************************************************
 * This file will be stored with no compression at all.  All the regular
 * things like CRC calculations and all that will be performed, but there
 * will be no compression applied.
 *
 * Arguments - header: File header being processed
 *             in: Source file
 *             out: Destination file
 */
void WIArchive::store(WIFileHeader &header, fstream &in, fstream &out)
{
    unsigned long crc, bytes, i;
    long bytesLeft;
    char *buffer;

    // Do some basic initialization before we get started
    header.method = 0;    // Method = 0: Store
    buffer        = new char[WIARCHIVE_BUFFERSIZE];
    bytesLeft     = header.origSize;
    crc           = 0xFFFFFFFF;

    // Store the file without applying any compression
    while (bytesLeft) {
        if (bytesLeft > WIARCHIVE_BUFFERSIZE)
            bytes = WIARCHIVE_BUFFERSIZE;
        else
            bytes = bytesLeft;
        bytesLeft -= bytes;
        in.read(buffer, bytes);
        if (callback != NULL) {
            callback(CBM_PERCENTAGE, in.tellg(), &header);
        }
        for (i = 0; i < bytes; i++) {
            crc = crcTable[(crc ^ buffer[i]) & 0xFF] ^ (crc >> 8);
        }
        out.write(buffer, bytes);
    }

    // It seems like we're done here
    delete [] buffer;
    crc ^= 0xFFFFFFFF;
    header.crc = crc;
    header.compSize = header.origSize;
}

/****************************************************************************
 * Expands all the files in a package to the path specified in the header
 * installation path.
 *
 * Arguments - package: Which package to expand
 *    Return - true if everything went well, otherwise false
 */
bool WIArchive::unpack(short package)
{
    list<WIPackHeader *>::iterator currentPackage, endPackage;
    WIFileHeader fileHeader;
    char fileName[256];
    short files;
    fstream file;
    int value;
    bool checker;
    long done;          // Number of decompressed bytes done in this package

    // Find the package we are ordered to install
    currentPackage = packList.begin();  endPackage = packList.end();
    for (; currentPackage != endPackage; currentPackage++) {
        if ((**currentPackage).number == package)  break;
    }
    if (currentPackage == endPackage)  return false;
    archive.seekg((**currentPackage).pos + archivePos, ios::beg);
    files = (**currentPackage).files;

    // Decompress all the files in this package
    done = 0;
    while (files--) {
        // Read file header and check if it's valid
        fileHeader.read(archive);
        if (fileHeader.verify() != true) {
            // Ooops, corrupt file header... Report the error through the wise
            // and almighty callback function (if we have one).
            if (callback != NULL) {
                callback(CBM_ERR_READ, CBREC_CANCEL, NULL);
            }
            return false;
        }

        // Create the directories for the installation path
        strcpy(fileName, header.path);
        strcat(fileName, fileHeader.name);
        makeDirectories(fileName);

        // Check with the callback function if we should proceed or skip this
        // file. If no callback is defined, assume that we have a go.
        if (callback != NULL) {
            value = callback(CBM_NEXTFILE, 0, &fileHeader);
        } else {
            value = CBRC_PROCEED;
        }
        if (value == CBRC_SKIP) {
            // Sorry, we're not allowed to extract this file. Next!
            archive.seekg(fileHeader.compSize, ios::cur);
            continue;
        }

        // Open the file and decompress/extract it
        file.open(fileName, ios::out | ios::binary);
        switch (fileHeader.method) {
            case 0:     // Stored
                checker = extract(fileHeader, archive, file, (**currentPackage), done);
                break;
            case 1:     // Compressed with BZ2
                checker = expand (fileHeader, archive, file, (**currentPackage), done);
                break;
            default:    // Unknown method
                return false;
        }
        done += (**currentPackage).origSize;
        if (file.is_open())  file.close();
        if (checker != true) {
            // For some reason, we could not finish this thing up. Let's exit
            // from this place as quickly as possible.
            return false;
        }

        // Set file creation and modification time
        utimbuf utb;
        utb.actime  = fileHeader.creation;
        utb.modtime = fileHeader.lastwrite;
        utime((char*)fileName, &utb);
    }

    return true;
}

/****************************************************************************
 * Inside this method the actual adding of files, copying of old files and
 * removing unused/unwanted/obsolete files takes place.
 */
bool WIArchive::update()
{
    list<WIPackHeader *>::iterator currentPackage, endPackage;
    list<WIFile *>::iterator currentFile, endFile, fileTemp;
    int packageMax = 0;         // Highest package number
    int package;                // Current package being processed
    WIFileHeader fileHeader;
    fstream file;               // File used for reading into archive
    long pos, pos2;             // Position in archive
    long l;
    int i;

    archive.tellg();
    archive.seekg(0, ios::beg);
    writeHeader(archive);
    writePackages(archive);

    // Find the highest package index in the archive and place that value
    // inside packageMax.  This saves a lot of unneccesary looping.  Very
    // clever indeed, Ulrich.
    currentPackage = packList.begin();  endPackage = packList.end();
    header.packs = 0;
    for (; currentPackage != endPackage; currentPackage++) {
        header.packs++;
        if ((**currentPackage).number > packageMax)
            packageMax = (**currentPackage).number;
    }

    // Start walking through the packages
    for (package = 1; package <= packageMax; package++) {
        // Look for this package in the package list.  When we are done, the
        // currentPackage will hold the package we are working on
        currentPackage = packList.begin();  endPackage = packList.end();
        for (; currentPackage != endPackage; currentPackage++) {
            if ((**currentPackage).number == package)  break;
        }

        // Reset package information
        (**currentPackage).origSize = 0;
        (**currentPackage).compSize = 0;
        pos = (**currentPackage).pos + archivePos;
        (**currentPackage).pos = archive.tellg() - (long)archivePos;

        // Let's begin by copying the old files in this package to the new
        // archive.  Later on, we will add the new files to the archive.  Now,
        // set the old archive reading position to the package start and start
        // looking for files to copy.
        // Note: We only need to do this if we have an old archive.
        if (tempName != "") {
            oldArchive.seekg(pos, ios::beg);
            for (i = 0; i < (**currentPackage).oldFiles; i++) {
                // Read file header from the old archive
                fileHeader.read(oldArchive);

                // Check if this file is supposed to be inside the new archive by
                // searching the old file list for a matching entry.
                currentFile = fileList.begin();  endFile = fileList.end();
                for (; currentFile != endFile; currentFile++) {
                    if (((**currentFile).package == package)
                            && ((**currentFile).name == fileHeader.name)) {
                        // Yep, this is one of them files.  Copy it and erase from
                        // this list so we don't have to look at it the next time
                        // we scan the entire list.
                        fileHeader.write(archive);
                        l = fileHeader.compSize;
                        char buffer[WIARCHIVE_BUFFERSIZE];
                        while (l) {
                            long lBytesThis = l;
                            if (lBytesThis > WIARCHIVE_BUFFERSIZE)  lBytesThis = WIARCHIVE_BUFFERSIZE;
                            oldArchive.read(buffer, lBytesThis);
                            archive.write(buffer, lBytesThis);
                            l -= lBytesThis;
                        }
                        (**currentPackage).origSize += fileHeader.origSize;
                        (**currentPackage).compSize += fileHeader.compSize;

                        // Time to erase it from the list
                        fileTemp = currentFile;  fileTemp++;
                        delete (WIFile *)*currentFile;
                        fileList.erase(currentFile);
                        currentFile = fileTemp;
                        break;
                    }
                }

                // If the file couldn't be found insude the list, we need to skip
                // all the compressed bytes. Any attempt of reading them and trying
                // to use it as valid data is bound to fail.
                if (currentFile == endFile) {
                    oldArchive.seekg(fileHeader.compSize, ios::cur);
                }
            }
        }

        // The time has come to compress all the fresh, newly-added files and
        // stuff them deep inside this archive.  It's a dirty job, but it seems
        // like WIArchive's just got to do it.
        currentFile = todoList.begin();  endFile = todoList.end();
        for (; currentFile != endFile; ) {
            // Check if the package number matches. If it doesn't, we will
            // have to process this file later on.
            if ((**currentPackage).number != (**currentFile).package) {
                currentFile++;
                continue;
            }

            // Everything seems OK. Let's open the chosen file for reading.
            file.open((**currentFile).extra, ios::binary | ios::in);
            if (file.bad())  return false;

            // Initialize the file header and write a preliminary version of
            // it into the destination archive.
            pos = archive.tellg();
            strcpy(fileHeader.name, (**currentFile).name);
            fileHeader.package = (**currentFile).package;
            fileHeader.write(archive);

            // Get the date and time the file was last modified and created
            struct stat statBuf;
            stat(fileHeader.name, &statBuf);
            fileHeader.lastwrite = statBuf.st_mtime;
            fileHeader.creation  = statBuf.st_ctime;

            // Try to compress the file.  In case we end up with a larger file
            // than what we started with, we have to store the file as-is to
            // not blow the archive size up.
            if (compress(fileHeader, file, archive) == false)
                // Something happened - probably bad. Hurry - exit!
                return false;
            /* TODO
            BUG WARNING! When storing files the first 8 bytes always
            vanishes like a backup stored in /dev/null or similar. I
            have not found this yet - therefore, we disable the storing
            ability. Doing this, we may loose up to 60 bytes or something
            similar. When the bug is found, reactivate this code section.

            if (fileHeader.compSize > fileHeader.origSize) {
                archive.seekg(pos, ios::beg);
                fileHeader.write(archive);
                file.seekg(0, ios::beg);
                store(fileHeader, file, archive);
            }
            */
            if (file.is_open())  file.close();

            // Report to the callback the resulting compressed file size
            if (callback != NULL) {
                callback(CBM_UPDATING, fileHeader.compSize, &fileHeader);
            }

            // Well, it seems like we made it all the way through all this
            // compression. Finish everything up by writing the correct file
            // header and doing a few other cleaning-up things.
            (**currentPackage).origSize += fileHeader.origSize;
            (**currentPackage).compSize += fileHeader.compSize;
            pos2 = archive.tellg();
            archive.seekg(pos, ios::beg);
            fileHeader.write(archive);
            archive.seekg(pos2, ios::beg);

            // Remove this file from the todo list for shorter scan times in
            // future passes.
            fileTemp = currentFile;  fileTemp++;
            todoList.erase(currentFile);
            // TODO: This line makes everything go up in a bang
            //delete (WIFile *)*currentFile;
            currentFile = fileTemp;
        }
    }

    // Finally we're done here. Clean a few things up and then exit.
    archive.seekg(0, ios::beg);
    writeHeader(archive);
    writePackages(archive);
    if (archive.is_open())  archive.close();
    modified = false;

    // Remove the old archive and rename the temporary file to the real name
    if (tempName[0] != 0) {
        if (oldArchive.is_open())  oldArchive.close();
        unlink(archiveName);
        rename(tempName, archiveName);
        tempName[0] = 0;
    }

    return true;
}

/****************************************************************************
 * Compress the installation script and write archive header to file
 *
 * Arguments - file: Place to store archive header
 */
void WIArchive::writeHeader(fstream &file)
{
    char *tempBuffer = NULL;
    bz_stream z;

    // Prepare the archive header for writing
    header.revisionNeeded = WIARCHIVE_REVISION_NEEDED;
    if (script == "") {
        // No script found
        header.compScriptSize = header.origScriptSize = 0;
    } else {
        // Compress this script and write it to disk
        int scriptSize = header.origScriptSize + 200;
        tempBuffer = new char[scriptSize];
        header.origScriptSize = strlen(script);

        // ----- Compress the installation script
        z.bzalloc   = 0;
        z.bzfree    = 0;
        z.opaque    = NULL;
        z.bzalloc   = NULL;
        z.bzfree    = NULL;
        z.next_in   = (char *)script;
        z.avail_in  = header.origScriptSize;
        z.next_out  = tempBuffer;
        z.avail_out = scriptSize;
        BZ2_bzCompressInit(&z, 1, 0, 30);
        BZ2_bzCompress(&z, BZ_FINISH);
        header.compScriptSize = scriptSize - z.avail_out;
        BZ2_bzCompressEnd(&z);
    }

    // Put all of this into storage
    header.write(file);
    if (header.origScriptSize > 0) {
        file.write(tempBuffer, header.compScriptSize);
        delete [] tempBuffer;
    }
    if (header.extendedDataSize > 0)
        file.write(extendedData, header.extendedDataSize);
}

/****************************************************************************
 * Write all the created packages to file
 *
 * Arguments - file: Where to store all of these darn packages
 */
void WIArchive::writePackages(fstream &file)
{
    list<WIPackHeader *>::iterator current, end;

    current = packList.begin();  end = packList.end();
    for (; current != end; current++) {
        (**current).write(file);
    }
}

/****************************************************************************
 *
 */
void bz_internal_error ( int errcode )
{
}
