
/*
 *@@sourcefile stub.cpp:
 *      source code of the standard WPI executable stub.
 *
 *      A WPI stub is simply a standard OS/2 .EXE file.
 *      With this stub, it is a PM program. This stub
 *      can be used with wic -u to turn any WPI file
 *      into an executable file; the actual WPI data
 *      will then simply be appended after the executable
 *      stub.
 *
 *      See StartInstallation() for what happens here.
 *
 *@@added V0.9.14 (2001-07-14) [umoeller]
 */

/*
 *      This file Copyright (C) 2001-2002 Yuri Dario, Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */


#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#ifndef __OS2_H__
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WINSHELLDATA
#define INCL_WINPOINTERS
#define INCL_WINSTATICS
#define INCL_WINSYS
#define INCL_WINFRAMEMGR
#include <os2.h>
#endif // __OS2_H__

#include <stdio.h>
#include <time.h>               // needed for WIFileHeader

#ifdef __IBMCPP__
#include <direct.h>
#endif

#include "setup.h"              // added V0.9.2 (2000-03-15) [umoeller]

#include "bldlevel.h"

#define __STRIP_DOWN_EXECUTABLE__
#include "helpers\dosh.h"
#include "helpers\exeh.h"
#include "helpers\ensure.h"
#include "helpers\standards.h"

#include "base\bs_base.h"
#include "base\bs_list.h"

// back-end includes
#include "wiarchive\wiarchive.h"

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

CHAR    G_szArchiveToLaunch[CCHMAXPATH];
// CHAR    G_szErrorMessage[1000];
HAB     G_hab;
HMQ     G_hmq;

/*
 *@@ G_apcszWarpINFiles:
 *      the list of files to be deleted from temp.
 */

const char *G_apcszWarpINFiles[] =
        {
            "warpin.exe",
            "wpirtl.dll",
            "warpin.hlp",
            "warpin.tmf",
            "warpin.ini"        // created after startup
        };


/* ******************************************************************
 *
 *  Helpers
 *
 ********************************************************************/

/*
 *@@ StartSession:
 *      an exact copy of doshQuickStartSession.
 *
 *@@added V0.9.14 (2001-07-24) [umoeller]
 */

APIRET StartSession(const char *pcszPath,       // in: program to start
                    const char *pcszParams,     // in: parameters for program
                    USHORT usPgmCtl,   // in: STARTDATA.PgmControl
                    PULONG pulSID,     // out: session ID (req.)
                    PPID ppid)         // out: process ID (req.)
{
    APIRET      arc;
    const char  *pcszQueueName = "\\queues\\warpin_stub.que";
    HQUEUE      hq = 0;
    PID         qpid = 0;

    if (    (!(arc = DosCreateQueue(&hq,
                                    QUE_FIFO | QUE_CONVERT_ADDRESS,
                                    pcszQueueName)))
         && (!(arc = DosOpenQueue(&qpid, &hq, pcszQueueName)))
       )
    {
        STARTDATA   SData;
        CHAR        szObjBuf[CCHMAXPATH];

        memset(&SData, 0, sizeof(SData));
        SData.Length  = sizeof(STARTDATA);
        SData.Related = SSF_RELATED_CHILD; //INDEPENDENT;
        SData.FgBg    = SSF_FGBG_BACK;
                // V0.9.3 (2000-05-03) [umoeller]
        SData.TraceOpt = SSF_TRACEOPT_NONE;

        SData.PgmTitle = (PSZ)pcszPath;       // title for window
        SData.PgmName = (PSZ)pcszPath;
        SData.PgmInputs = (PSZ)pcszParams;

        SData.TermQ = (PSZ)pcszQueueName;
        // SData.Environment = 0;
        SData.InheritOpt = SSF_INHERTOPT_PARENT;
        SData.SessionType = SSF_TYPE_DEFAULT;
        // SData.IconFile = 0;
        // SData.PgmHandle = 0;

        SData.PgmControl = usPgmCtl;

        // SData.InitXPos  = 30;
        // SData.InitYPos  = 40;
        // SData.InitXSize = 200;
        // SData.InitYSize = 140;
        // SData.Reserved = 0;
        SData.ObjectBuffer  = szObjBuf;
        SData.ObjectBuffLen = (ULONG)sizeof(szObjBuf);

        if (!(arc = DosStartSession(&SData, pulSID, ppid)))
        {
            REQUESTDATA rqdata;
            ULONG       DataLength = 0;
            PULONG      DataAddress;
            BYTE        elpri;

            rqdata.pid = qpid;
            DosReadQueue(hq,                // in: queue handle
                         &rqdata,           // out: pid and ulData
                         &DataLength,       // out: size of data returned
                         (PVOID*)&DataAddress, // out: data returned
                         0,                 // in: remove first element in queue
                         0,                 // in: wait for queue data (block thread)
                         &elpri,            // out: element's priority
                         0);                // in: event semaphore to be posted
        }

        DosCloseQueue(hq);
    }

    return (arc);
}

/*
 *@@ QueryCurrentDir:
 *      an exact copy of doshQueryCurrentDir.
 */

APIRET QueryCurrentDir(PSZ pszBuf)
{
    APIRET arc = NO_ERROR;
    ULONG   ulCurDisk = 0;
    ULONG   ulMap = 0;
    if (!(arc = DosQueryCurrentDisk(&ulCurDisk, &ulMap)))
    {
        ULONG   cbBuf = CCHMAXPATH - 3;
        *pszBuf = ulCurDisk + 'A' - 1;
        pszBuf[1] = ':';
        pszBuf[2] = '\\';
        arc = DosQueryCurrentDir(0, pszBuf + 3, &cbBuf);
    }

    return (arc);
}

/*
 *@@ SetCurrentDir:
 *      an exact copy of doshSetCurrentDir.
 */

APIRET SetCurrentDir(const char *pcszDir)
{
    #define toupper(c)  ((c) & ~32)

    APIRET  arc = NO_ERROR;
    if (!pcszDir)
        return (ERROR_INVALID_PARAMETER);

    if (*pcszDir != 0)
        if (*(pcszDir+1) == ':')
        {
            // drive given:
            CHAR    cDrive = toupper(*(pcszDir));
            // change drive
            arc = DosSetDefaultDisk( (ULONG)(cDrive - 'A' + 1) );
                    // 1 = A:, 2 = B:, ...
        }

    arc = DosSetCurrentDir((PSZ)pcszDir);

    return (arc);       // V0.9.9 (2001-04-04) [umoeller]
}

/*
 *@@ ComposePath:
 *
 *@@added V0.9.14 (2001-07-24) [umoeller]
 *@@changed V0.9.15 (2001-08-31) [umoeller]: fixed root dir
 */

VOID ComposePath(PSZ pszBuf,            // out: buffer
                 PCSZ pcszPath,         // in: path (without trailing '\')
                 PCSZ pcszFilename)     // in: short filename
{
    ULONG ulPathLen = strlen(pcszPath);
    memcpy(pszBuf, pcszPath, ulPathLen + 1);
    // append "\" if there's none yet
    // c:\      ulPathLen == 3
    if (pszBuf[ulPathLen - 1] != '\\') // V0.9.15 (2001-08-31) [umoeller]
        pszBuf[ulPathLen++] = '\\';
    strcpy(pszBuf + ulPathLen, pcszFilename);
}

/*
 *@@ ComposeDatabase:
 *
 *@@added V0.9.14 (2001-07-24) [umoeller]
 */

VOID ComposeDatabase(PSZ pszBuf,
                     PCSZ pcszPath)         // in: path
{
    ULONG ulBootDrive;
    DosQuerySysInfo(QSV_BOOT_DRIVE, QSV_BOOT_DRIVE,
                    &ulBootDrive,
                    sizeof(ulBootDrive));
    CHAR szTemp[] = "DATBAS_?.INI";
    szTemp[7] = (CHAR)ulBootDrive + 'A' - 1;
    ComposePath(pszBuf,         // V0.9.16 (2001-10-28) [umoeller]
                pcszPath,
                szTemp);
}

/*
 *@@ DeleteFiles:
 *      deletes the files specified in the given
 *      array of file names from the specified
 *      temp dir.
 */

APIRET DeleteFiles(PCSZ pcszTempDir,     // in: directory
                   PCSZ *apcszFiles,     // in: array of file names
                   ULONG cFiles)                // in: array item count (NOT array size)
{
    CHAR szPath[CCHMAXPATH];
    ULONG i;
    APIRET arc;
    for (i = 0;
         i < cFiles;
         ++i)
    {
        ComposePath(szPath,
                    pcszTempDir,
                    apcszFiles[i]);
        if ((arc = DosDelete(szPath)))
            break;
    }

    return (arc);
}

/*
 *@@ Substr:
 *      this creates a new PSZ containing the string
 *      from pBegin to pEnd, excluding the pEnd character.
 *      The new string is null-terminated. The caller
 *      must free() the new string after use.
 *
 *      Example:
 +              "1234567890"
 +                ^      ^
 +                p1     p2
 +          Substr(p1, p2)
 *      would return a new string containing "2345678".
 *
 *@@changed V0.9.9 (2001-04-04) [umoeller]: fixed crashes with invalid pointers
 *@@changed V0.9.9 (2001-04-04) [umoeller]: now using memcpy for speed
 */

PSZ Substr(const char *pBegin,      // in: first char
           const char *pEnd)        // in: last char (not included)
{
    PSZ pszSubstr = NULL;

    if (pEnd > pBegin)      // V0.9.9 (2001-04-04) [umoeller]
    {
        ULONG cbSubstr = (pEnd - pBegin);
        if (pszSubstr = (PSZ)malloc(cbSubstr + 1))
        {
            memcpy(pszSubstr, pBegin, cbSubstr);        // V0.9.9 (2001-04-04) [umoeller]
            *(pszSubstr + cbSubstr) = '\0';
        }
    }

    return (pszSubstr);
}

/********************************************************************
 *
 *   Executable functions
 *
 ********************************************************************/

    typedef struct _EXECUTABLE
    {
        // executable opened by doshOpen
        HFILE               hfExe;

        LXHEADER            LXHeader;

        // module analysis (always set):
        ULONG               ulExeFormat;
                // one of:
                // EXEFORMAT_OLDDOS        1
                // EXEFORMAT_NE            2
                // EXEFORMAT_PE            3
                // EXEFORMAT_LX            4
                // EXEFORMAT_TEXT_BATCH    5
                // EXEFORMAT_TEXT_REXX     6

    } EXECUTABLE, *PEXECUTABLE;

VOID ExecClose(PEXECUTABLE pExec);

/*
 *@@ ExecOpen:
 *      a stripped-down copy of doshExecOpen.
 */

APIRET ExecOpen(const char* pcszExecutable,
                PEXECUTABLE pExec)
{
    APIRET  arc = NO_ERROR;

    HFILE   hFile;
    ULONG   ulAction = 0;

    memset(pExec, 0, sizeof(EXECUTABLE));

    if (!(arc = DosOpen((PSZ)pcszExecutable,
                        &hFile,
                        &ulAction,                      // out: action taken
                        0,                              // in: new file (ignored for read-mode)
                        0,                              // in: new file attribs (ignored)
                        // open-flags
                        OPEN_ACTION_FAIL_IF_NEW
                           | OPEN_ACTION_OPEN_IF_EXISTS,
                        // open-mode
                        OPEN_FLAGS_FAIL_ON_ERROR        // report errors to caller
                           | OPEN_FLAGS_SEQUENTIAL
                           | OPEN_FLAGS_NOINHERIT
                           | OPEN_SHARE_DENYNONE
                           | OPEN_ACCESS_READONLY,      // read-only mode
                        NULL)))                         // no EAs
    {
        // file opened successfully:

        ULONG           ulLocal = 0;

        // read old DOS EXE header
        DOSEXEHEADER    DosExeHeader;
        ULONG           ulBytesRead = 0,
                        cbDosExeHeader;

        if (    (!(arc = DosSetFilePtr(hFile,
                                       0L,
                                       FILE_BEGIN,
                                       &ulLocal)))      // out: new offset
             && (!(arc = DosRead(hFile,
                                 &DosExeHeader,
                                 sizeof(DOSEXEHEADER),
                                 &cbDosExeHeader)))
           )
        {
            ULONG ulNewHeaderOfs = 0;       // V0.9.12 (2001-05-03) [umoeller]
            BOOL  fLoadNewHeader = FALSE;

            // now check if we really have a DOS header
            if (DosExeHeader.usDosExeID != 0x5a4d)
            {
                // arc = ERROR_INVALID_EXE_SIGNATURE;

                // V0.9.12 (2001-05-03) [umoeller]
                // try loading new header directly; there are
                // drivers which were built with NOSTUB, and
                // the exe image starts out with the NE or LX
                // image directly
                fLoadNewHeader = TRUE;
                        // ulNewHeaderOfs is 0 now
            }
            else
            {
                // we have a DOS header:
                if (DosExeHeader.usRelocTableOfs >= 0x40)
                {
                    // we have a new header offset:
                    fLoadNewHeader = TRUE;
                    ulNewHeaderOfs = DosExeHeader.ulNewHeaderOfs;
                }
            }

            if (fLoadNewHeader)
            {
                // either LX or PE or NE:
                // read in new header...
                // ulNewHeaderOfs is now either 0 (if no DOS header
                // was found) or pDosExeHeader->ulNewHeaderOfs
                // V0.9.12 (2001-05-03) [umoeller]
                CHAR    achNewHeaderType[2] = "";

                if (    (!(arc = DosSetFilePtr(hFile,
                                               ulNewHeaderOfs,
                                               FILE_BEGIN,
                                               &ulLocal)))
                        // read two chars to find out header type
                     && (!(arc = DosRead(hFile,
                                         &achNewHeaderType,
                                         sizeof(achNewHeaderType),
                                         &ulBytesRead)))
                   )
                {
                    // reset file ptr
                    DosSetFilePtr(hFile,
                                  ulNewHeaderOfs,
                                  FILE_BEGIN,
                                  &ulLocal);

                    if (    (achNewHeaderType[0] == 'L')
                         && (achNewHeaderType[1] == 'X')
                       )
                    {
                        // OS/2 Linear Executable:
                        pExec->ulExeFormat = EXEFORMAT_LX;
                        // read in LX header
                        arc = DosRead(hFile,
                                      &pExec->LXHeader,
                                      sizeof(LXHEADER),
                                      &ulBytesRead);
                    }
                    else
                        // strange type:
                        arc = ERROR_INVALID_EXE_SIGNATURE;

                } // end if (!(arc = DosSetFilePtr(hFile,
            }
        } // end if (!(arc = DosSetFilePtr(hFile,

        // store exec's HFILE
        pExec->hfExe = hFile;
    } // end if (!(arc = DosOpen((PSZ)pcszExecutable,

    if (arc != NO_ERROR)
        // error: clean up
        ExecClose(pExec);

    return (arc);
}

/*
 *@@ ExecClose:
 *
 */

VOID ExecClose(PEXECUTABLE pExec)
{
    if (pExec)
        if (pExec->hfExe)
            DosClose(pExec->hfExe);
}

/*
 *@@ QueryWarpINBldLevel:
 *      examines the BLDLEVEL information in the given
 *      executable.
 *
 *      Returns:
 *
 *      --  NO_ERROR (0): executable exists and is at least
 *          WarpIN 0.9.14. This still doesn't mean that it's
 *          the same bldlevel as the executable stub. The
 *          caller should check the "Version" output for that.
 *
 *      --  WIERR_ANCIENT_WARPIN_INSTALLED: WarpIN is installed,
 *          but the version is older than 0.9.14 and thus doesn't
 *          support self-installing executables.
 *
 *      plus the various error codes from ExecOpen and such
 *      if the file doesn't exist or is broken or whatever.
 *
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now returning APIRET
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now checking version level here
 *@@changed V0.9.18 (2002-03-03) [umoeller]: squeezed every possible byte out of this mess
 */

APIRET QueryWarpINBldLevel(const char *pcszProgram,
                           WIVersion &Version,      // out: if NO_ERROR, bldlevel version
                           PSZ pszBldLevel)         // out: bldlevel string (ptr can be NULL)
{
    EXECUTABLE  Exec;
    APIRET      arc;

    if (pszBldLevel)
        *pszBldLevel = 0;

    // check installed version
    if (!(arc = ExecOpen(pcszProgram, &Exec)))
    {
        if (    (Exec.ulExeFormat == EXEFORMAT_LX)
            // check if we have a non-resident name table
             && (Exec.LXHeader.ulNonResdNameTblOfs)
           )
        {
            ULONG       ulLocal = 0,
                        ulBytesRead = 0;

            // move EXE file pointer to offset of non-resident name table
            // (from LX header)
            if (!(arc = DosSetFilePtr(Exec.hfExe,     // file is still open
                                      Exec.LXHeader.ulNonResdNameTblOfs,      // ofs determined above
                                      FILE_BEGIN,
                                      &ulLocal)))
            {
                // allocate memory as necessary
                CHAR szNameTable[500]; // should suffice, because each entry
                                        // may only be 255 bytes in length
                if (!(arc = DosRead(Exec.hfExe,
                                    szNameTable,
                                    sizeof(szNameTable),
                                    &ulBytesRead)))
                {
                    ULONG cb;
                    if (!(cb = szNameTable[0]))
                        // first byte (length byte) is null:
                        arc = ERROR_INVALID_DATA;
                    else
                    {
                        // now copy the string, which is in Pascal format
                        PSZ pszDescription = szNameTable + 1;
                        pszDescription[cb] = '\0';

                        PSZ    pStartOfVendor,
                               pStartOfInfo,
                               pEndOfVendor;

                        // @#VENDOR:VERSION#@ DESCRIPTION
                        // but skip the first byte, which has the string length
                        if (    (pStartOfVendor = strstr(pszDescription, "@#"))
                             && (pStartOfInfo = strstr(pStartOfVendor + 2, "#@"))
                             && (pEndOfVendor = strchr(pStartOfVendor + 2, ':'))
                           )
                        {
                            PSZ pszVersion = pEndOfVendor + 1;
                            *pStartOfInfo = '\0';

                            if (pszBldLevel)
                                strcpy(pszBldLevel, pszVersion);

                            char    *dot1 = NULL,
                                    *dot2 = NULL;

                            _Pmpf((__FUNCTION__ ": version is %s", pszVersion));

                            // find first dot in major.minor.build
                            if ((dot1 = strchr(pszVersion, '.')))
                            {
                                // ok
                                *dot1 = 0; // terminate first token

                                // search second dot
                                if ((dot2 = strchr(dot1+1, '.')))
                                {
                                    // ok, found
                                    *dot2 = 0;
                                    // extract version info
                                    Version._ulMajor = atoi(pszVersion);
                                    Version._ulMinor = atoi(dot1+1);
                                    Version._ulRevision = atoi(dot2+1);

                                    // check at least 0.9.14 (for temp installer)
                                    if (     (Version._ulMajor == 0)
                                         &&  (Version._ulMinor < 10)
                                         &&  (Version._ulRevision < 14)
                                       )
                                    {
                                        // this is too old:
                                        arc = WIERR_ANCIENT_WARPIN_INSTALLED;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        else
            arc = ERROR_INVALID_EXE_SIGNATURE;

        // moved this down; this frees the structures!
        // V0.9.14 (2001-08-03) [umoeller]
        ExecClose(&Exec);
    }

    return arc;

}

void fastcat(PSZ psz,           // in/out: buffer
             PULONG pulOfs,     // in/out: offset in buffer
             PCSZ pcsz)         // in: string to append
{
    ULONG len = strlen(pcsz);
    memcpy(psz + *pulOfs,
           pcsz,
           len + 1);
    *pulOfs += len;
}

/* ******************************************************************
 *
 *  WarpIN Launch
 *
 ********************************************************************/

/*
 *@@ QueryWarpINPath:
 *      retrieves the path of an existing WarpIN installation
 *      from OS2.INI and checks it for validity.
 *
 *      Returns:
 *
 *      --  NO_ERROR:  pszBuf has received the path of an
 *          existing WarpIN installation, and WARPIN.EXE
 *          exists in that directory.
 *
 *      --  ERROR_FILE_NOT_FOUND, ERROR_PATH_NOT_FOUND:
 *          The OS2.INI entry existed, but points to an
 *          invalid directory, or WarpIN.EXE doesn't exist
 *          there.
 *
 *      --  WIERR_NO_WARPIN_INSTALLED: the OS2.INI entry
 *          doesn't even exist. This means that WarpIN was
 *          never installed on the system.
 *
 *@@added V0.9.16 (2001-11-25) [umoeller]
 */

APIRET QueryWarpINPath(PSZ pszBuf,        // out: WarpIN path
                       ULONG cbBuf)       // in: sizeof(*pszBuf)
{
    FILESTATUS3 fs3;
    if (3 < PrfQueryProfileString(HINI_USER,
                                  "WarpIN",
                                  "Path",
                                  "",
                                  pszBuf,
                                  cbBuf))
    {
        // check for WARPIN.EXE in that directory
        CHAR szTemp[500];
        ComposePath(szTemp,
                    pszBuf,
                    "WARPIN.EXE");
        return (DosQueryPathInfo(szTemp,
                                 FIL_STANDARD,
                                 &fs3,
                                 sizeof(fs3)));
    }

    return (WIERR_NO_WARPIN_INSTALLED);

}

/*
 *@@ launchWarpIn:
 *      launches WarpIN with the indicated wpi archive. This is
 *      done by calling QuickStartSession().
 *
 *      This gets called either for the WarpIN that was extracted
 *      to $(TEMP) or for the existing WarpIN, if that's installed.
 *
 *@@added V0.9.4 [csm]
 *@@changed V0.9.16 (2001-10-28) [umoeller]: fixed plenty of startup bugs
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now returning APIRET
 */

APIRET LaunchWarpIn(PCSZ pcszWarpINPath,
                    BOOL fRecent,
                    BOOL fNewInstall)
{
    // allocate enough space
    CHAR    szProgram[2 * CCHMAXPATH];
    CHAR    szParams[2 * CCHMAXPATH] = "-I ";      // ignore self-setup, no ini management
    ULONG   ulOfs = 3;
    APIRET  arc = NO_ERROR;

    ComposePath(szProgram,
                pcszWarpINPath,
                "WARPIN.EXE");

    // change to warpin directory, to find dll's
    DosSetExtLIBPATH(pcszWarpINPath, BEGIN_LIBPATH);

    // check if parameters are supported
    if (!fRecent)
    {
        szParams[0] = '\0';     // no extra params before .14
        ulOfs = 0;
    }
    else
    {
        // szParams has -I: ignore self-setup
        if (fNewInstall)
            fastcat(szParams, &ulOfs, "-S ");        // started from stub V0.9.14 (2001-08-09) [umoeller]

        // if we have _any_ existing WarpIN version,
        // use that path for the database to use
        CHAR szDatabase[CCHMAXPATH] = "";
        if (!QueryWarpINPath(szDatabase, sizeof(szDatabase)))
            // V0.9.16 (2001-11-25) [umoeller]
        {
            fastcat(szParams, &ulOfs, "-d");
            fastcat(szParams, &ulOfs, szDatabase);
            fastcat(szParams, &ulOfs, " ");
        }
    }

    // append the archive name;
    // if this contains spaces, enclose it in quotes
    // V0.9.16 (2001-01-23) [umoeller]
    if (strchr(G_szArchiveToLaunch, ' '))
    {
        fastcat(szParams, &ulOfs, "\"");
        fastcat(szParams, &ulOfs, G_szArchiveToLaunch);
        fastcat(szParams, &ulOfs, "\"");
    }
    else
        fastcat(szParams, &ulOfs, G_szArchiveToLaunch);

    // now go!
    _Pmpf((__FUNCTION__ ": launching \"%s\" \"%s\"",
            szProgram,
            szParams));

    ULONG ulSessionID;
    PID pidProcessID;
    arc = StartSession(szProgram,   // program
                       szParams,    // params
                       // FALSE,  // foreground
                       SSF_CONTROL_VISIBLE,    // PgmControl
                       &ulSessionID,
                       &pidProcessID);

    // ok
    return arc;
}

/* ******************************************************************
 *
 *  WarpIN Install
 *
 ********************************************************************/

/*
 *@@ ExtractWarpIn:
 *      extracts WarpIN files from the given archive to the given
 *      temporary directory.
 *      This verify that the binary is at least build 0.9.14.
 *
 *      Changes the current working directory.
 *
 *      Returns:
 *
 *      --  NO_ERROR: WarpIN was successfully extracted to the temp dir.
 *
 *      --  WIERR_ANCIENT_WARPIN_INSTALLED: WarpIN was found in the archive,
 *          but it is not at least V0.9.14.
 *
 *      --  WIERR_INVALID_INDEX: returned from WIArchive::unpack
 *          if package 30000 doesn't exist in the archive. That
 *          means that the archive doesn't contain WarpIN itself.
 *
 *      plus the other error codes from WIArchive::open, SetCurrentDir,
 *      WIArchive::unpack, QueryWarpINBldLevel.
 *
 *@@changed V0.9.18 (2002-03-03) [umoeller]: fixed error reports
 */

APIRET ExtractWarpIn(PCSZ pcszArchive,       // in: archive name
                     PCSZ pcszTmpDir)       // in: TEMP dir
{
    WIArchive   Arc;
    APIRET      arc;

    // try to open this archive
    if (!(arc = Arc.open(pcszArchive)))
        // returns WIERR_* error code
    {
        if (!(arc = SetCurrentDir(pcszTmpDir)))
        {
            // unpack files
            if (!(arc = Arc.unpack(WARPIN_SELF_PACKAGE_FOR_STUB)))       // 30000
                    // returns WIERR_INVALID_INDEX if the package doesn't
                    // exist
            {
                // check version of temp install
                WIVersion Version;
                if (arc = QueryWarpINBldLevel("WARPIN.EXE", Version, NULL))
                {
                    // we can get WIERR_ANCIENT_WARPIN_INSTALLED if the
                    // version is not at least 0.9.14
                    // but in any case, remove the temp files then
                    _Pmpf(("Removing temporary files...\n"));
                    DeleteFiles(pcszTmpDir,
                                G_apcszWarpINFiles,
                                sizeof(G_apcszWarpINFiles) / sizeof(G_apcszWarpINFiles[0]));
                }
            }

            // close archive
            Arc.close();
        }
    }

    // return status
    return arc;
}

/*
 *@@ InstallCheckFilesInSameDir:
 *      helper routine for UnpackTempWarpIN which runs
 *      thru all files in the given directory which
 *      match the given file mask and attempts to
 *      unpack WarpIN to pcszTempDir.
 *
 *      Returns:
 *
 *      --  NO_ERROR (0): successfully unpacked.
 *
 *      --  ERROR_FILE_NOT_FOUND: no such file mask
 *          existed, or file existed, but did not
 *          contain package 30000.
 *
 *      plus other fatal error codes from DosFindNext.
 *
 *@@added V0.9.18 (2002-03-03) [umoeller]
 */

APIRET InstallCheckFilesInSameDir(PCSZ pcszCWD,        // in: current directory
                                  PCSZ pcszFileMask,   // in: file mask (e.g. *.wpi)
                                  PCSZ pcszTempDir)    // in: $(TEMP)
{
    APIRET          arc;

    HDIR            hdir = HDIR_CREATE;
    FILEFINDBUF3    buf3;
    ULONG           entries = 1;

    char            currentWPIPath[CCHMAXPATH];
    ComposePath(currentWPIPath, pcszCWD, "*.wpi");

    arc = DosFindFirst(currentWPIPath,
                       &hdir,
                       FILE_NORMAL,
                       &buf3,
                       sizeof(buf3),
                       &entries,
                       FIL_STANDARD);

    while (entries && !arc)
    {
        ComposePath(currentWPIPath,
                    pcszCWD,
                    buf3.achName);

        // @@todo we should ignore pcszFileIgnore,
        // or we'll try to extract this twice always

        if (!(arc = ExtractWarpIn(currentWPIPath,
                                  pcszTempDir)))
        {
            // alright, this one worked:
            // report success
            DosFindClose(hdir);
            return (NO_ERROR);
        }

        // search next file
        entries = 1;
        arc = DosFindNext(hdir,
                          &buf3,
                          sizeof(buf3),
                          &entries);
            // gives us ERROR_NO_MORE_FILES eventually
    }

    // close search
    DosFindClose(hdir);

    if (arc == ERROR_NO_MORE_FILES)
        arc = ERROR_FILE_NOT_FOUND;

    return (arc);
}

HWND    G_hwndNotify;

/*
 *@@ UnpackThread:
 *      this does the actual unpacking and stuff on a second
 *      thread so we won't block the PM SIQ while this is done.
 *
 *      Note that we are still using the single-thread runtime
 *      libraries, so watch out with runtime calls here.
 *
 *@@added V0.9.18 (2002-03-03) [umoeller]
 */

VOID APIENTRY UnpackThread(ULONG ulParam)
{
    PCSZ pcszTempDir = (PCSZ)ulParam;

    char cwd[CCHMAXPATH];
    // save current path
    QueryCurrentDir(cwd);

    // try extracting from this executable
    APIRET arc = ExtractWarpIn(G_szArchiveToLaunch,
                               pcszTempDir);

    _Pmpf((__FUNCTION__ ": ExtractWarpIn returned %d", arc));

    if (arc)
    {
        // failed:

        // the above ExtractWarpIN cannot return WIERR_ANCIENT_WARPIN
        // because we ran it on our own executable, so this must be
        // at least 0.9.14!

        // We should only continue though if ExtractWarpIN returned
        // WIERR_INVALID_INDEX (i.e. WarpIN is not contained in the
        // executable itself). Otherwise the archive is probably
        // broken anyway, and anything else would fail also.

        if (arc == WIERR_INVALID_INDEX)
        {
            // package not included, try searching other archives
            // in same directory
            arc = InstallCheckFilesInSameDir(cwd, "*.wpi", pcszTempDir);

            _Pmpf((__FUNCTION__ ": InstallCheckFilesInSameDir(*.wpi) returned %d", arc));

            if (arc = ERROR_FILE_NOT_FOUND)
            {
                arc = InstallCheckFilesInSameDir(cwd, "warpin-*.exe", pcszTempDir);

                _Pmpf((__FUNCTION__ ": InstallCheckFilesInSameDir(warpin-*.exe) returned %d", arc));
            }
        }
    }

    WinPostMsg(G_hwndNotify,
               WM_USER,
               (MPARAM)arc,
               0);
}

/*
 *@@ UnpackTempWarpIN:
 *      extracts the WarpIN files to $(TEMP).
 *
 *      This attempts to do two things.
 *
 *      1)  First we check package 30000 in our
 *          own EXE.
 *
 *      2)  If that fails, we try looking for
 *          *.wpi files in the same directory
 *          and do the same.
 *
 *      3)  If that fails, we try looking for
 *          warpin-*.exe in the same directory
 *          and do the same.
 *
 *      Returns:
 *
 *      --  NO_ERROR (0): WarpIN was found either in the
 *          EXE or in another warpin-* file.
 *
 *      --  ERROR_FILE_NOT_FOUND: recent WarpIN was not
 *          found (neither in current exe, nor in *.wpi,
 *          nor in warpin-*.exe).
 *
 *      In addition, this can return all the error codes
 *      from ExtractWarpIn. If that returns something other
 *      than WIERR_INVALID_INDEX, since that means that
 *      the archive is most probably broken, we will _not_
 *      perform steps 2 and 3 but return that error instead.
 *      This fixes the stupid "No sufficient WarpIN" error
 *      message that people kept getting no matter what
 *      error really occured.
 *
 *@@changed V0.9.16 (2001-10-28) [umoeller]: fixed plenty of startup bugs
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now returning APIRET; fixed error handling
 *@@changed V0.9.18 (2002-03-03) [umoeller]: fixed hangs by moving unpack to a separate thread
 */

APIRET UnpackTempWarpIN(PCSZ pcszTempDir)
{
    APIRET arc;

    // set wait pointer, create a dull temporary window
    #define CX_MSG          600
    #define CY_MSG          80

    ULONG ulFrameFlags = FCF_DLGBORDER | FCF_NOBYTEALIGN;
    HWND hwndClient;
    HWND hwndFrame = WinCreateStdWindow(HWND_DESKTOP,
                                        WS_ANIMATE,
                                        &ulFrameFlags,
                                        WC_STATIC,
                                        NULL,
                                        SS_TEXT | DT_CENTER | DT_VCENTER,
                                        0,
                                        0,
                                        &hwndClient);

    LONG lColor = 0;            // black
    WinSetPresParam(hwndClient,
                    PP_FOREGROUNDCOLOR,
                    sizeof(ULONG),
                    &lColor);

    lColor = 0xFFFFFF;
    WinSetPresParam(hwndClient,
                    PP_BACKGROUNDCOLOR,
                    sizeof(ULONG),
                    &lColor);

    CHAR    szFont[] = "9.WarpSans Bold";
    WinSetPresParam(hwndClient,
                    PP_FONTNAMESIZE,
                    sizeof(szFont),     // including 0
                    szFont);

    WinSetWindowText(hwndClient, "Please wait while WarpIN unpacks itself...");

    // show the window
    WinSetWindowPos(hwndFrame,
                    HWND_TOP,
                    (WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - CX_MSG)
                         / 2,
                    (WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN) - CY_MSG)
                         / 2,
                    CX_MSG,
                    CY_MSG,
                    SWP_MOVE | SWP_SIZE | SWP_ACTIVATE | SWP_SHOW);

    // run unpack on a separate thread
    // V0.9.18 (2002-03-03) [umoeller]:
    if (G_hwndNotify = WinCreateWindow(HWND_OBJECT,
                                       WC_BUTTON,
                                       (PSZ)"",
                                       0,
                                       0,0,0,0,
                                       0,
                                       HWND_BOTTOM,
                                       0,
                                       0,
                                       NULL))
    {
        TID tidUnpack;
        DosCreateThread(&tidUnpack,
                        UnpackThread,
                        (ULONG)pcszTempDir,
                        CREATE_READY,
                        3*96000); // plenty of stack

        QMSG qmsg;
        BOOL fQuit = FALSE;
        while (WinGetMsg(G_hab,
                         &qmsg, 0, 0, 0))
        {
            // current message for our object window?
            if (    (qmsg.hwnd == G_hwndNotify)
                 && (qmsg.msg == WM_USER)
               )
            {
                fQuit = TRUE;
                arc = (ULONG)qmsg.mp1;
            }

            WinDispatchMsg(G_hab, &qmsg);
            if (fQuit)
                break;
        }

        WinDestroyWindow(G_hwndNotify);
    }

    WinDestroyWindow(hwndFrame);

    return (arc);
}

/*
 *@@ AppendErrorMsg:
 *
 *@@added V0.9.18 (2002-03-03) [umoeller]
 */

VOID AppendErrorMsg(PSZ *ppsz,          // in/out: string (buf can be NULL)
                    PCSZ pcszMsg)       // in: string to append
{
    ULONG   ulMsgLen = strlen(pcszMsg);
    ULONG   cb = ulMsgLen + 1;      // null terminator
    ULONG   ulPos = 0;
    PSZ     pszOld;

    if (pszOld = *ppsz)
    {
        ulPos = strlen(*ppsz);      // append to tail
        cb += ulPos;             // and raise required mem
    }

    // avoid realloc, that costs us another 800 bytes
    // or so in the code segment
    if (*ppsz = (PSZ)malloc(cb))
    {
        if (pszOld)
        {
            memcpy(*ppsz, pszOld, ulPos);
            free(pszOld);
        }

        memcpy(*ppsz + ulPos,       // ulPos will be 0 if string was empty
               pcszMsg,
               ulMsgLen + 1);       // null terminator
    }
}

/*
 *@@ AppendWhereToGet:
 *
 *@@added V0.9.18 (2002-03-03) [umoeller]
 */

VOID AppendWhereToGet(PSZ *ppszError)
{
    AppendErrorMsg(ppszError,
            "The latest WarpIN can be obtained from http://warpin.netlabs.org. ");
}

/*
 *@@ StartInstallation:
 *      starts installation process, called from main().
 *
 *      The procedure here is as follows:
 *
 *      1)  The stub checks for whether WarpIN is already
 *          installed on the system, by looking at OS2.INI.
 *
 *          a)  If so, we check that version's buildlevel.
 *
 *              aa) If the build level is at least our buildlevel,
 *                  that version is used for install always.
 *                  Go to 3).
 *
 *              bb) If the build level is lower than 0.9.14,
 *                  the installed version cannot handle executable
 *                  archives, so we must go to 2).
 *
 *              cc) If that build level is at least 0.9.14 but
 *                  lower than that of the stub, we _try_ to
 *                  install a better version under 2).
 *
 *          b)  If OS2.INI doesn't contain the WarpIN key,
 *              go to 2).
 *
 *      2)  We then attempt to install WarpIN from an
 *          archive. See UnpackTempWarpIN().
 *
 *          If that fails and WarpIN >= 0.9.14 is already
 *          installed, we use that. If the installed
 *          WarpIN is < 0.9.14, we always fail because
 *          because the executable cannot be installed.
 *
 *          If we find a better WarpIN than the installed
 *          one, we use that always. This is new with 0.9.18
 *          because otherwise WarpIN can't properly upgrade
 *          itself either, and using the old WarpIN for
 *          installing is a stupid idea anyway because the
 *          archive might need new features and the user
 *          would still get an error message.
 *
 *      3)  WarpIN is started with our own executable name
 *          (which is the WPI file) as a parameter. If an
 *          existing copy is used, we pass -d for the
 *          name of the existing database (even for an
 *          old WarpIN copy).
 *
 *      4)  If the TEMP WarpIN copy was used, its files are
 *          destroyed again.
 *
 *      Returns NULL on success, or an error message otherwise.
 *
 *@@changed V0.9.16 (2001-10-28) [umoeller]: fixed plenty of startup bugs
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now returning much better error messages
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now using WarpIN from archive if older but >= 0.9.14 installed
 */

PSZ StartInstallation(const char *pcszTempDir)
{
    CHAR    szProgram[CCHMAXPATH];

    CHAR    szWarpINPath[CCHMAXPATH];           // V0.9.16 (2001-10-28) [umoeller]

    PCSZ    pcszWarpINPath = NULL;
    WIVersion Version;
    CHAR    szBldLevel[255];
    BOOL    fNewInstall = FALSE,
            fTryIfBetter = FALSE;

    APIRET  arc;
    BOOL    fHandled = FALSE;
    PSZ     pszError = NULL;

    // query current install path
    arc = QueryWarpINPath(szWarpINPath, sizeof(szWarpINPath));

    _Pmpf((__FUNCTION__ ": QueryWarpINPath returned %d", arc));

    if (!arc)
    {
        // path found, check installed version
        ComposePath(szProgram,          // V0.9.16 (2001-10-28) [umoeller]
                    szWarpINPath,
                    "WARPIN.EXE");

        // check installed version
        arc = QueryWarpINBldLevel(szProgram, Version, szBldLevel);

        _Pmpf((__FUNCTION__ ": QueryWarpINBldLevel returned %d", arc));

        if (!arc)
        {
            // use the INI path, this is recent
            pcszWarpINPath = szWarpINPath;

            // but if the version number is lower than that
            // of the stub, check if maybe we can still
            // use a better warpin from the executable, the
            // package might need it
            // V0.9.18 (2002-03-03) [umoeller]
            _Pmpf((__FUNCTION__ ": existing has %d.%d.%d",
                        Version._ulMajor,
                        Version._ulMinor,
                        Version._ulRevision));

            if (    (Version._ulMajor < VERSION_MAJOR)
                 || (    (Version._ulMajor == VERSION_MAJOR)
                      && (    (Version._ulMinor < VERSION_MINOR)
                           || (    (Version._ulMinor == VERSION_MINOR)
                                && (Version._ulRevision < VERSION_REVISION)
                              )
                         )
                    )
                )
            {
                fTryIfBetter = TRUE;
            }
        }

        // else: warpin.exe exists, but cannot be
        // accessed, or is older than 0.9.14
        // (WIERR_ANCIENT_WARPIN_INSTALLED)
    }

    // V0.9.18 (2002-03-03) [umoeller]
    // --   NO_ERROR: we can use the existing installation
    // --   WIERR_NO_WARPIN_INSTALLED: os2.ini entry doesn't
    //      exist
    // --   WIERR_ANCIENT_WARPIN_INSTALLED: warpin exists,
    //      but is too old
    // --   ERROR_FILE_NOT_FOUND, ERROR_PATH_NOT_FOUND: os2.ini
    //      directory is invalid... in that case we should warn
    //      the user in order not to give him stupid error messages!
    if (    (arc != NO_ERROR)
         && (arc != WIERR_NO_WARPIN_INSTALLED)
         && (arc != WIERR_ANCIENT_WARPIN_INSTALLED)
       )
    {
        // this means that the os2.ini entry is broken!
        // so delete the entry and tell the user we did so
        PrfWriteProfileString(HINI_USER,
                              "WarpIN",
                              "Path",
                              NULL);

        AppendErrorMsg(&pszError,
            "The WarpIN OS2.INI entry points to the directory \"");
        AppendErrorMsg(&pszError, szWarpINPath);
        AppendErrorMsg(&pszError,
            "\", but that directory is invalid. To avoid further problems, "
            "the WarpIN OS2.INI entry has been deleted. If you did have an existing "
            "WARPIN.EXE in that directory, you can run that executable to "
            "re-register it as the active WarpIN installation. Otherwise "
            "you should install a fresh WarpIN. ");
        AppendWhereToGet(&pszError);
        fHandled = TRUE;
    }

    if (    (arc == WIERR_NO_WARPIN_INSTALLED)
         || (arc == WIERR_ANCIENT_WARPIN_INSTALLED)
         || (fTryIfBetter)     // we should check, but the existing is 0.9.14 or higher
       )
    {
        // OK, we need a new one...

        // recent warpin not found: try to install it
        if (arc = UnpackTempWarpIN(pcszTempDir))
        {
            if (fTryIfBetter)
            {
                // we have a version 0.9.14 or higher already but
                // can't find a better one: don't complain then
                arc = 0;
                // pcszWarpINPath still points to the old one
            }
            else if (arc == ERROR_FILE_NOT_FOUND)
            {
                // no WarpIN found anywhere:
                // report an error depending on what we looked for

                if (szBldLevel[0])  // we found a local WarpIN, but it wasn't
                                         // recent enough:
                {
                    AppendErrorMsg(&pszError,
                        "The WarpIN that is currently installed on your system is "
                        "too old to support this archive. Your version in \"");
                    AppendErrorMsg(&pszError,
                        szWarpINPath);
                    AppendErrorMsg(&pszError,
                        "\" is ");
                    AppendErrorMsg(&pszError,
                        szBldLevel);
                    AppendErrorMsg(&pszError,
                        ".");
                    AppendErrorMsg(&pszError,
                        ", but self-installing executables "
                        "require at least WarpIN version 0.9.14 to install. ");
                }
                else
                    // we didn't even find a local WarpIN:
                    AppendErrorMsg(&pszError,
                        "To install this archive, you need to install WarpIN 0.9.14 or "
                        "higher first. ");

                AppendWhereToGet(&pszError);

                // reset error code so we won't mess with it below
                fHandled = TRUE;
            }
        }
        else
        {
            // WarpIN installed:
            // check (again) installed version

            // V0.9.18 (2002-03-03) [umoeller]
            // why?!? UnpackTempWarpIN has checked this many times
            // already!
            /*
            ComposePath(szProgram,              // V0.9.16 (2001-10-28) [umoeller]
                        pcszTempDir,
                        "WARPIN.EXE");
            if (QueryWarpINBldLevel(szProgram,
                                    &major,
                                    &minor,
                                    &build))
            */
            {
                pcszWarpINPath = pcszTempDir;
                fNewInstall = TRUE;
            }
        }
    }

    if (    !arc
         && pcszWarpINPath      // points to either old or temp WarpIN now
       )
    {
        // WarpIn installed, extract archive
        if (!(arc = LaunchWarpIn(pcszWarpINPath,
                                 TRUE,
                                 fNewInstall)))
        {
            // if it is a new install for WarpIN, move database file and install log
            if (fNewInstall)
            {
                // MoveDatabaseEtc(pcszTempDir, major, minor, build);
                // remove temporary files
                _Pmpf(("Removing temporary files...\n"));
                DeleteFiles(pcszTempDir,
                            G_apcszWarpINFiles,
                            sizeof(G_apcszWarpINFiles) / sizeof(G_apcszWarpINFiles[0]));
            }
        }
    }

    if (arc && !fHandled)
    {
        // unhandled error left:

        AppendErrorMsg(&pszError,
                       "An unhandled error occured in the self-extracting archive. "
                       "The following error was reported: ");
        CHAR szTemp[50];
        _ltoa(arc, szTemp, 10);
        AppendErrorMsg(&pszError, szTemp);
    }

    return (pszError);
}

/*
 *@@ main:
 *      Initialize enviroment
 */

void main(void) // int argc, char *argv[])
{
    G_hab = WinInitialize(0);
    G_hmq = WinCreateMsgQueue(G_hab, 0);

    // query archive name; this is our own executable
    PPIB pib = NULL;
    DosGetInfoBlocks(NULL, &pib);
    DosQueryModuleName(pib->pib_hmte,
                       sizeof(G_szArchiveToLaunch),
                       G_szArchiveToLaunch);

    // find a temporary directory
    PCSZ pcszTempDir;
    if (DosScanEnv("TMP", &pcszTempDir))
        if (DosScanEnv("TEMP", &pcszTempDir))
            if (DosScanEnv("TMPDIR", &pcszTempDir))
                pcszTempDir = "C:\\";

    // start process
    PSZ pszError;
    if (pszError = StartInstallation(pcszTempDir))
    {
        WinMessageBox(HWND_DESKTOP, HWND_DESKTOP,
                      pszError,
                      "WarpIN: Error",
                      1,
                      MB_OK | MB_ERROR | MB_MOVEABLE);
        // free(pszError);
    }

    // free resources
    // WinDestroyMsgQueue(G_hmq);
    // WinTerminate(G_hab);
}
