/*
 * $Id: selfin.c,v 1.2 2001/07/15 20:33:47 jens Exp $
 *
 * Extracts the contents of an installation file.
 *
 * While WarpIN is a very good utility for installing other applications, you
 * can't install WarpIN itself using WarpIN. This small utility, named SelfIN,
 * will remedy that. By using the same powerful BZ2 compression code that is
 * used in WarpIN itself, the installer will be very small. Best of all, you no
 * longer have to use Zip archives for distributing WarpIN. Finally, WarpIN can
 * be installed by using (an extremely compact version of) itself.
 *
 * This file Copyright (C) 2001 Jens B&auml;ckman
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, in version 2 as it comes in the COPYING
 * file of this distribution.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "selfin/common.h"
#include "selfin/selfin.h"

/****************************************************************************
 * Let the BZ2 library decompress a previously compressed and stored file.
 *
 * Arguments - filename: Name of the file where decompressed data is to be
 *                       stored. Path and such should be here as well.
 *             in:
 *             bytesCompressed:
 *             bytesOriginal:
 **/
static void expand(char *filename, FILE *in, unsigned long bytesCompressed,
                   unsigned long bytesOriginal)
{
    #define BUFFERSIZE 0x10000
    unsigned char outBuffer[BUFFERSIZE];/* Output data buffer           */
    unsigned char inBuffer[BUFFERSIZE]; /* Input data buffer            */
    unsigned long bytes;                /* Bytes in this block          */
    bz_stream bz;                       /* BZ2 control structure        */
    int status;                         /* BZ2 status value             */
    FILE *out;                          /* File handle to unpacked file */

    makeDirectories(filename);
    out = fopen(filename, "wb");

    /* Make sure that BZ2 knows what's going on here. */
    bz.bzalloc   = NULL;
    bz.bzfree    = NULL;
    bz.opaque    = NULL;
    bz.avail_in  = 0;
    bz.avail_out = BUFFERSIZE;
    bz.next_out  = outBuffer;
    BZ2_bzDecompressInit(&bz, 0, 0);

    while (bytesCompressed) {
        /* Read compressed file data. */
        if (bytesCompressed > BUFFERSIZE)
            bytes = BUFFERSIZE;
        else
            bytes = bytesCompressed;
        bytesCompressed -= bytes;
        fread(inBuffer, bytes, 1, in);
        bz.avail_in = bytesCompressed;
        bz.next_in  = inBuffer;

        /* While we have bytes left in the inbuffer, let's continue working. */
        while (bz.avail_in > 0) {
            /* Decompress more information. */
            status = BZ2_bzDecompress(&bz);
            if (status < 0) {
                /* TODO: BZ2 error! */
                fclose(out);
                return;
            }

            /* Write data (if we have any). */
            if (BUFFERSIZE - bz.avail_out) {
                // Write decompressed data to file
                fwrite(outBuffer, BUFFERSIZE - bz.avail_out, 1, out);
                bytesOriginal -= (BUFFERSIZE - bz.avail_out);
                bz.avail_out = BUFFERSIZE;
                bz.next_out  = outBuffer;
            }
        }
        if (status == BZ_STREAM_END)  break;
    }

    /* Flush the remaining uncompressed data. */
    while ((bytesOriginal != 0) && (status != BZ_STREAM_END)) {
        status = BZ2_bzDecompress(&bz);
        fwrite(outBuffer, BUFFERSIZE - bz.avail_out, 1, out);
        bytesOriginal -= (BUFFERSIZE - bz.avail_out);
        bz.avail_out = BUFFERSIZE;
        bz.next_out  = outBuffer;
    }

    fclose(out);
    BZ2_bzDecompressEnd(&bz);
}

/****************************************************************************
 * Install ourselves.
 *
 * Arguments - filename: Name of the application containing the installation
 *                       package information. Passing argv[0] to this would
 *                       not be considered stupid.
 *             installPath: Path where the application will be installed.
 *    Return - Anyone of the SELFIN_ values listed in selfin.h are valid.
 **/
int selfinstall(char *filename, char *installPath)
{
    unsigned long crc;  /* File CRC value                     */
    unsigned long pos;  /* Position in file where our data is */
    unsigned long files;/* Number of files to decompress      */
    FILE *in;           /* Input file handle                  */
    char c;

    /* Start by making an integrity check of ourselves. */
    in = fopen(filename, "rb");
    fseek(in, sizeof(unsigned long)*2, SEEK_END);
    fread(&pos, sizeof(unsigned long), 1, in);
    fread(&crc, sizeof(unsigned long), 1, in);
    if (crc != calculateCrc(in, ftell(in) - sizeof(unsigned long))) {
        fclose(in);
        return SELFIN_CORRUPT;
    }

    /* Add finalslash to installpath if not there */
    c = installPath[strlen(installPath) - 1];
    if (c != '\\' && c != '/')
        strcat(installPath, "\\");

    /* Now... Install these damn files! */
    fseek(in, pos, SEEK_SET);
    fread(&files, sizeof(unsigned long), 1, in);
    while (files--) {
        unsigned long sizeCompressed;
        unsigned long sizeOriginal;
        unsigned char nameLength;
        unsigned char name[256];
        unsigned char tempName[256];

        fread(&sizeCompressed, sizeof(unsigned long), 1, in);
        fread(&sizeOriginal, sizeof(unsigned long), 1, in);
        fread(&nameLength, 1, 1, in);
        fread(&name, nameLength, 1, in);
        name[nameLength] = 0;
        strcpy(tempName, installPath);
        strcat(tempName, name);
        expand(tempName, in, sizeCompressed, sizeOriginal);
    }

    fclose(in);
    return SELFIN_OK;
}

