/******************************************************************************
 *@@sourcefile WIURL.cpp:
 *      Implements WIURL class to store server name, server port and
 *      document path as separate data fields.
 *
 *@@header "net/WIURL.hpp"
 */

#include <stdio.h>      // for sscanf()
#include <string.h>     // for strnicmp()

#include "net\WIURL.hpp"

WIURL::WIURL()
{
    _fIsEmpty         = TRUE;
    _pszServerName    = NULL;
    _pszServerAddress = NULL;
    _usAddressLength  = 0;
    _pszDocumentPath  = NULL;
}

/******************************************************************************
 *@@ WIURL:
 *      <P>This constructor parses <CODE>pcszURL</CODE> and splits it
 *      into server name, server port and document path. The expected format is
 *      <BR><CODE>[ &quot;http://&quot; ] server_name [ &quot;:&quot;
 *                [ server_port ] ] [ document_path ]</CODE>
 *      <BR>where document_path must begin with <CODE>&quot;/&quot;</CODE>
 *      if present.
 *
 *      The constructor throws an <CODE>WIInvalidURLException</CODE>
 *      if the given URL seems to be invalid (server name is empty,
 *      or server port is not an <CODE>USHORT</CODE>. Further testing
 *      is not done.
 *
 *      If no server_port is given, or it is empty, the default value
 *      <SAMP>80</SAMP> is assumed.
 *      <BR>If no document_path is given, the default value
 *      <CODE>&quot;/&quot;</CODE> is used.
 */

WIURL::WIURL(PCSZ pcszURL)              // in: URL to be parsed
{
    // Copy the pointer (not the string)

    PCSZ ptrURL = pcszURL;

    // If URL starts with the scheme "http://", skip that

    if (strnicmp(pcszURL, "http://", strlen("http://")) == 0)
        ptrURL += strlen("http://");

    // Allocate enough memory to handle broken URLs

    ULONG len = strlen(ptrURL);

    PSZ pszServer  = new CHAR[len+1];
    PSZ pszPort    = new CHAR[len+1];
    PSZ pszDocPath = new CHAR[len+1];

    // Scan URL

    BOOL fServerFailure = FALSE;
    BOOL fPortFailure   = FALSE;

    int iTokens = sscanf(ptrURL, "%[^:/]%[^/]%s",
                         pszServer, pszPort, pszDocPath);
    if (iTokens < 1)
        fServerFailure = TRUE;
    else
    {
        if (iTokens == 1)
        {
            strcpy(pszPort, "");
            sscanf(ptrURL, "%[^/]%s", pszServer, pszDocPath);
        }

        // Check server name and server port

        if (strlen(pszServer) == 0)
            fServerFailure = TRUE;
        else
        {
            // Check port in detail

            _usServerPort = 80;
            if (strlen(pszPort) > strlen(":"))
            {
                ULONG ulOffset;
                LONG  lPort;

                sscanf(pszPort, "%*c%li%ln", &lPort, &ulOffset);

                // Port number out of range or contains non-alpha?

                if ((lPort < 0) || (lPort > 0xFFFF) ||
                    (ulOffset < strlen(pszPort)))
                    fPortFailure = TRUE;
                else
                    _usServerPort = (USHORT) lPort;
            }
        }
    }

    // On failure, throw exception
    // On success, set server name and document path

    if (fServerFailure || fPortFailure)
    {
        WIInvalidURLException exception;
        string& str = exception._strDescription;

        str += "\n    Invalid URL \"";
        str += pcszURL;
        str += '"';
        if (fServerFailure)
            str += "\n    Reason: Server name not specified";
        if (fPortFailure)
        {
            str += "\n    Reason: Invalid server port ";
            str += (pszPort + 1);
        }

        delete pszServer;
        delete pszPort;
        delete pszDocPath;

        throw exception;
    }
    else
    {
        delete pszPort;

        _fIsEmpty        = FALSE;
        _pszServerName   = pszServer;
        _pszDocumentPath = pszDocPath;

        // Fix empty document path

        if (strlen(_pszDocumentPath) == 0)
            strcpy(_pszDocumentPath, "/");
    }

    _usAddressLength  = 0;
    _pszServerAddress = NULL;
}

/******************************************************************************
 *@@ WIURL:
 *      <P>The copy constructor just copies all data from the given instance.
 */

WIURL::WIURL(const WIURL& wiURL)        // in: instance to copy from
{
    _fIsEmpty = wiURL._fIsEmpty;

    // Allocate memory and copy data

    _pszServerName    = new CHAR[strlen(wiURL._pszServerName) + 1];
    strcpy(_pszServerName, wiURL._pszServerName);

    _usAddressLength = wiURL._usAddressLength;
    if (_usAddressLength > 0)
    {
        _pszServerAddress = new CHAR[wiURL._usAddressLength];
        memcpy(_pszServerAddress, wiURL._pszServerAddress, _usAddressLength);
    }
    else
        _pszServerAddress = NULL;

    _pszDocumentPath  = new CHAR[strlen(wiURL._pszDocumentPath) + 1];
    strcpy(_pszDocumentPath, wiURL._pszDocumentPath);

    _usServerPort = wiURL._usServerPort;
}

/******************************************************************************
 *@@ ~WIURL:
 *      <P>Frees all allocated memory.
 */

WIURL::~WIURL()
{
    // Play it safe

    if (_pszServerName != NULL)
        delete _pszServerName;
    if (_pszServerAddress != NULL)
        delete _pszServerAddress;
    if (_pszDocumentPath != NULL)
        delete _pszDocumentPath;
}

VOID
WIURL::SetServerAddress(PCSZ pcszServerAddress, USHORT usAddressLength)
{
    if (_pszServerAddress != NULL)
    {
        delete _pszServerAddress;
    }

    _usAddressLength = usAddressLength;
    _pszServerAddress = new CHAR[_usAddressLength];
    memcpy(_pszServerAddress, pcszServerAddress, _usAddressLength);
}
