/******************************************************************************
 *@@sourcefile WINet.cpp:
 *      Document me!
 */

#include <stdarg.h>
#include "net\WINet.hpp"
#include "net\WIP_HTTP.hpp"

#include <sys\types.h>
#include <netdb.h>

/*@@ PrintDebug:
 *      <P>Debugging interface. I use this to make debug output independent
 *      of the actual interface used.
 */

#ifdef DEBUG
#include <stdio.h>

VOID
PrintDebug(const char* format, ...)
{
    va_list arg_ptr;

    va_start(arg_ptr, format);
    vfprintf(stderr, format, arg_ptr);
    va_end(arg_ptr);
}
#endif // DEBUG

/******************************************************************************
 *@@ WINet:
 *      <P>This constructor sets the external callback function and
 *      initializes the other data fields with default values.
 */

WINet::WINet(PFNWINETCALLBACK pfnExternalCallback)
{
    _pfnExternalCallback = pfnExternalCallback;
}

/******************************************************************************
 *@@ WINet:
 *      <P>Copy constructor for WINet class.
 */

WINet::WINet(const WINet& wiNet)
{
    _pfnExternalCallback = wiNet._pfnExternalCallback;
}

BOOL
WINet::Callback(USHORT       usMsgType,
                ULONG        ulMsgValue,
                const WIURL& wiURL)
{
    if (_pfnExternalCallback != NULL)
        return _pfnExternalCallback(usMsgType, ulMsgValue, wiURL);
    else
        return false;
}

/******************************************************************************
 *@@ FetchURL:
 *      <P>Tries to fetch the given URL and store it to the given file.
 *      Returns TRUE if fetching has succeeded, FALSE else. If fResume
 *      is set to TRUE, an previously interrupted download will be
 *      continued (if supported by protocol, client and server).
 *
 *      If the given URL is invalid, an WIInvalidURLException is thrown.
 */

BOOL
WINet::FetchURL(WIURL& wiURL,
                PCSZ   pcszFileName,
                BOOL   fResume)
{
    // Throw WIInvalidURLException if wiURL is empty

    if (wiURL.IsEmpty())
    {
        WIInvalidURLException excpt;
        excpt._strDescription += " thrown by WINet::FetchURL(WIURL&, PCSZ)"
                                 "\n    Reason: The WIURL instance has not "
                                 "been assigned an URL yet.";
        throw excpt;
    }

    // Look up host

    BOOL fSuccess = LookupHost(wiURL);
    if (fSuccess)
    {
        // Create an instance of the desirced protocol class

        WIProtocol* pwiProtocol = NULL;

        // Currently, only HTTP protocol is implemented

        pwiProtocol = new WIP_HTTP(this);

        // Fetch URL and delete WIProtocol child

        fSuccess = pwiProtocol->Fetch(wiURL, pcszFileName, fResume);

        if (fSuccess)
            Callback(CBM_NET_TRANSFER_COMPLETE, 0, wiURL);
        else
            Callback(CBM_NET_TRANSFER_BROKEN, 0, wiURL);

        if (pwiProtocol != NULL)
            delete pwiProtocol;
    }

    return fSuccess;
}

BOOL
WINet::LookupHost(WIURL& wiURL)
{
    BOOL fSuccess = false;

    Callback(CBM_NET_DNSLOOKUP, 0, wiURL);

    struct hostent* Hostent = gethostbyname(wiURL.GetServerName());
    if (Hostent != NULL)
    {
        wiURL.SetServerAddress(Hostent->h_addr, Hostent->h_length);
        fSuccess = true;
    }
    else
    {
        Callback(CBM_NET_DNSLOOKUP_FAILED, h_errno, wiURL);
    }

    return fSuccess;
}
