
/*
 *@@sourcefile gui_installing.cpp:
 *      this contains the PM-specific stuff for while
 *      the front end is installing packages. See
 *      guiStartInstall, which is the main entry point
 *      into all this.
 *
 *      This has been separated from gui.cpp (99-10-26) [umoeller].
 *
 *@@header "frontend\gui.h"
 */

/*
 *      This file Copyright (C) 1998-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_REXXSAA
#define INCL_WIN
#define INCL_WINWORKPLACE
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#include <os2.h>

#ifdef __IBMCPP__
    #include <rexxsaa.h>    // EMX has this in os2.h already
#endif

#include <stdlib.h>
#include <stdio.h>           // included with wiarchive.h
#ifdef __IBMCPP__
    #include <direct.h>
#endif
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h
#include <io.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"

// include's from helpers
#include "helpers\datetime.h"
#include "helpers\dosh.h"
#include "helpers\cnrh.h"
#include "helpers\comctl.h"
#include "helpers\gpih.h"
#include "helpers\nls.h"
#include "helpers\nlscache.h"
#include "helpers\winh.h"
#include "helpers\stringh.h"
#include "helpers\textview.h"
#include "helpers\threads.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"

#include "base\bs_config.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"

#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_package_db.h"
#include "engine\fe_archive.h"
#include "engine\fe_job.h"
#include "engine\fe_database.h"

#include "engine\fe_engine.h"

// #include "engine\fe_rexx.h"

#include "frontend\warpin.h"
#include "frontend\calbacks.h"

#include "frontend\dlgids.h"
#include "frontend\gui.h"

#include "helpers\except.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

// LONG                G_lErrorRC = -1;             // changed (99-11-03) [umoeller]
            // removed V0.9.12 (2001-05-22) [umoeller]

ULONG               G_ulInstallTimeBegin = 0,
                    G_ulSecsElapsed = 0,
                    G_ulPercent = 0;

/* ******************************************************************
 *
 *    GUI functions on thread 1
 *
 ********************************************************************/

/*
 *@@ ConfirmCancelInstall:
 *      helper func which is used every time
 *      the user tries to cancel installation.
 *      This wants this decision confirmed
 *      and does all the cleanup, if neccessary.
 */

BOOL ConfirmCancelInstall(FELocals *pLocals,
                          HWND hwndOwner)
{
    ULONG ulmbrc = MBID_NO;
    BOOL  brc = FALSE;

    if (!G_WpiGlobals.fInstallComplete)
    {
        TID     tidInstall;

        // suspend Install thread if it's running;
        // we don't want the unpacking of files to
        // continue while the dialog is being
        // displayed
        if ((tidInstall = thrQueryID(&G_WpiGlobals.tiInstallThread)))
            DosSuspendThread(tidInstall);

        ulmbrc = pLocals->MessageBox(108, // "Warning"
                                   109, // "Damage, Danger, Really cancel?"
                                   MSG_CONFIRM_YESNO_DEFYES);

        if (ulmbrc == MBID_YES)
        {
            // terminate install thread
            G_WpiGlobals.tiInstallThread.fExit = TRUE;

            WinPostMsg(hwndOwner, WM_CLOSE, 0, 0);
            brc = TRUE;
        }

        if (tidInstall)
            DosResumeThread(tidInstall);
                    // if the exit flag has been set above,
                    // this will quickly exit
    }
    else
    {
        WinPostMsg(hwndOwner, WM_CLOSE, 0, 0);
        brc = TRUE;
    }

    return (brc);
}

/*
 *@@ fnwpFileExistsOrLocked:
 *      window proc for "file exists" confirmation
 *      dialog. This is only for an additional
 *      confirmation when "Cancel" or "Confirmations"
 *      are pressed.
 *
 *@@changed V0.9.0 (99-10-26) [umoeller]: renamed from fnwpFileExists
 *@@changed V0.9.0 (99-10-26) [umoeller]: added support for file-locked dlg
 */

MRESULT EXPENTRY fnwpFileExistsOrLocked(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    FEFileError *pfi = (FEFileError *)WinQueryWindowPtr(hwndDlg, QWL_USER);
    // GUILocals   *pLocals = (GUILocals*)&pfi->_Locals;

    switch (msg)
    {
        case WM_COMMAND:
        {
            switch (SHORT1FROMMP(mp1))
            {
                case ID_WIMI_PREFERENCES:
                    gshrPreferencesDlg((GUILocals*)&pfi->_Locals, hwndDlg);     // only in file exists
                break;

                case DID_CANCEL:
                {
                    if (ConfirmCancelInstall((GUILocals*)&pfi->_Locals, hwndDlg))
                        WinDismissDlg(hwndDlg, DID_CANCEL);
                    else
                        // continue
                        break;
                }

                default:
                    mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            }
        }
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ ShowFileLockedDlg:
 *      this gets called from fnwpInstallingStatus if a
 *      file to be overwritten is locked by another process
 *      (in a response to a guiFileLockedError call).
 *
 *      This must return one of the following:
 *      --  CBRC_SKIP:    skip locked file
 *      --  CBRC_RETRY:   retry same file
 *      --  CBRC_UNLOCK:  DosReplaceModule
 *      --  CBRC_DEFER:   defer thru CONFIG.SYS
 *      --  CBRC_ABORT:   abort install
 *
 *@@added V0.9.0 (99-10-26) [umoeller]
 */

int ShowFileLockedDlg(HWND hwndOwner,
                      FEFileError *pfi)
{
    int irc = CBRC_SKIP;

    // last_sel is the global (static) flag
    // in which we remember the last user selection
    static int      last_sel = ID_WIDI_EXISTS_SKIP;

    HWND hwndConfirm = gshrLoadConfirmDlg(hwndOwner,
                                          ID_WID_FILELOCKED,
                                          fnwpFileExistsOrLocked,
                                          pfi);

    // set file name (replace %1 in static text)
    HWND        hwndItem = WinWindowFromID(hwndConfirm, ID_WIDI_EXISTS_TXT1);
    PSZ         pszMessage = winhQueryWindowText(hwndItem);
    ULONG       ulOfs = 0;
    strhFindReplace(&pszMessage, &ulOfs, "%1", pfi->_strFilename.c_str());
    WinSetWindowText(hwndItem, pszMessage);
    free(pszMessage);

    // disable "defer"; not implemented yet
    WinEnableControl(hwndConfirm, ID_WIDI_LOCKED_DEFER, FALSE);

    // check last selection item
    winhSetDlgItemChecked(hwndConfirm, last_sel, 1);
    WinSetFocus(HWND_DESKTOP, WinWindowFromID(hwndConfirm, last_sel));

    // run confirmation dialog
    ULONG ulrc = WinProcessDlg(hwndConfirm);

    if (ulrc == DID_OK)
    {
        // and now, do what?
        if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_EXISTS_SKIP))
        {
            last_sel = ID_WIDI_EXISTS_SKIP;
            irc = CBRC_SKIP;
        }
        else if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_LOCKED_RETRY))
        {
            last_sel = ID_WIDI_LOCKED_RETRY;
            irc = CBRC_RETRY;
        }
        else if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_LOCKED_UNLOCK))
        {
            last_sel = ID_WIDI_LOCKED_UNLOCK;
            irc = CBRC_UNLOCK;
        }
        else if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_LOCKED_DEFER))
        {
            last_sel = ID_WIDI_LOCKED_DEFER;
            irc = CBRC_DEFER;
        }
    }
    else
        // "Cancel":
        irc = CBRC_ABORT;

    WinDestroyWindow(hwndConfirm);

    return (irc);
}

/*
 *@@ ShowFileExistsDlg:
 *      this gets called from fnwpInstallingStatus if a
 *      file already exists on disk and we need to prompt
 *      the user.
 *
 *      This must return one of the following:
 *      --  CBRC_PROCEED    replace this file
 *      --  CBRC_SKIP       skip this file
 *      --  CBRC_ABORT      abort installation
 *
 *@@changed V0.9.0 (99-10-26) [umoeller]: renamed from FileExistsDlg
 *@@changed V0.9.0 (99-10-26) [umoeller]: fixed "Cancel" not working
 *@@changed V0.9.0 (99-10-26) [umoeller]: exported stuff to LoadConfirmDlg
 */

int ShowFileExistsDlg(HWND hwndOwner,
                      FEFileError *pfi)
{
    int irc = CBRC_SKIP;

    // last_sel is the global (static) flag
    // in which we remember the last user selection
    static int      last_sel = ID_WIDI_EXISTS_SKIP;

    HWND hwndConfirm = gshrLoadConfirmDlg(hwndOwner,
                                          ID_WID_FILEEXISTS,
                                          fnwpFileExistsOrLocked,
                                          pfi);

    // set file name (replace %1 and %2 in static text)
    HWND        hwndItem = WinWindowFromID(hwndConfirm, ID_WIDI_EXISTS_TXT1);
    PSZ         pszMessage = winhQueryWindowText(hwndItem);
    PSZ         pLastBackslash = strrchr(pfi->_strFilename.c_str(), '\\');
    ULONG       ulOfs = 0;
    strhFindReplace(&pszMessage, &ulOfs, "%1", pLastBackslash+1);
    *pLastBackslash = 0;
    ulOfs = 0;
    strhFindReplace(&pszMessage, &ulOfs, "%2", pfi->_strFilename.c_str());
    WinSetWindowText(hwndItem, pszMessage);
    free(pszMessage);

    // check last selection item
    winhSetDlgItemChecked(hwndConfirm, last_sel, 1);
    WinSetFocus(HWND_DESKTOP, WinWindowFromID(hwndConfirm, last_sel));

    // run confirmation dialog
    ULONG ulrc = WinProcessDlg(hwndConfirm);

    if (ulrc == DID_OK)
    {
        // and now, do what?
        if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_EXISTS_SKIP))
        {
            last_sel = ID_WIDI_EXISTS_SKIP;
            irc = CBRC_SKIP;
        } else if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_EXISTS_RENAME))
        {
            last_sel = ID_WIDI_EXISTS_RENAME;
            irc = CBRC_PROCEED;
        }
    }
    else
        // "Cancel":
        irc = CBRC_ABORT;

    WinDestroyWindow(hwndConfirm);

    // last_sel always has the action to do now
    return (irc);
}

/*
 *@@ fnwpInstallingStatus:
 *      window proc for the "Install" window after we're
 *      done with all the pages and the Install thread is
 *      running.
 *
 *      This runs on thread 1.
 *
 *      The various gui* callbacks on the Install thread
 *      keep posting messages to us.
 *
 *      This gets started via WinProcessDlg from guiStartInstall.
 *      If this dialog returns TRUE via WinDismissDlg, this
 *      means installation went OK.
 *
 *@@changed V0.9.0 (99-10-26) [umoeller]: renamed from fnwpInstall
 *@@changed V0.9.0 (99-10-26) [umoeller]: error string handling reworked completely
 *@@changed V0.9.0 (99-10-26) [umoeller]: added WPIM_FILELOCKEDERROR
 */

MRESULT EXPENTRY fnwpInstallingStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    static HWND         hwndProgressBar = NULLHANDLE;

    GUIInstallEngine *pEngine = (GUIInstallEngine*)WinQueryWindowPtr(hwndDlg, QWL_USER);
    GUILocals* pLocals;
    if (pEngine)
        pLocals = (GUILocals*)&pEngine->_Locals;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      initialize some more controls.
         *      mp2 has ptr to GUIInstallEngine.
         */

        case WM_INITDLG:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            pEngine = (GUIInstallEngine*)mp2;
            WinSetWindowPtr(hwndDlg, QWL_USER, mp2);

            winhSetControlsFont(hwndDlg, 0, 500,
                        gshrQueryDefaultFont());

            // create a progress bar
            hwndProgressBar = WinWindowFromID(hwndDlg, ID_WIDI_PROGRESSBAR);
            ctlProgressBarFromStatic(hwndProgressBar,
                        PBA_ALIGNCENTER | PBA_BUTTONSTYLE);  // (*UM#4)

            // start the timer
            G_ulInstallTimeBegin = dtGetULongTime();
            WinStartTimer(pEngine->_Locals._habThread1, hwndDlg, 1, 1000);
        break;

        /*
         *@@ WPIM_UPDATEPACKAGE:
         *      user msg posted from the Install thread
         *      when we advance to the next package.
         *      Parameters:
         *          PSZ mp1  current FEInstallJob*
         *          PSZ mp2  string with "pck 1 out of 9"
         *
         *      Strings must be free()'d here.
         */

        case WPIM_UPDATEPACKAGE:
            if (mp1 != NULL)
            {
                FEInstallJob *pJob = (FEInstallJob*)mp1;
                CHAR szSizeRounded[20];
                nlsThousandsULong(szSizeRounded, pJob->QueryPckSizeRounded(),
                                  G_szThousand[0]);
                ustring strPck;
                strPck._printf("%s (%s KB)",
                               pJob->QueryPckTitle().GetBuffer(),
                               szSizeRounded);
                pLocals->SetWindowTextU(WinWindowFromID(hwndDlg, ID_WIDI_CURPCKTITLE),
                                        strPck);

                pLocals->SetWindowTextU(WinWindowFromID(hwndDlg, ID_WIDI_TARGETPATHDIR),
                                        pJob->QueryTargetPath());
                /*  V0.9.20 (2002-07-03) [umoeller]
                string strTarget;
                strTarget.assignUtf8(pLocals->_pCodecGui, pJob->QueryTargetPath());
                WinSetWindowText(WinWindowFromID(hwndDlg, ID_WIDI_TARGETPATHDIR),
                                 strTarget.c_str());
                */
            }
            if (mp2 != NULL)
            {
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURPCKCOUNT, (PSZ)mp2);
                free(mp2);
            }
        break;

        /*
         *@@ WPIM_UPDATEFILE:
         *      user msg posted from the Install thread
         *      when we advance to the next file.
         *      Parameters:
         *          PSZ mp1  current file name
         *          PSZ mp2  string with "file 1 out of 9"
         *
         *      Both strings must be free()'d here.
         */

        case WPIM_UPDATEFILE:
        {
            if (mp1 != NULL)
            {
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURFILENAME, (PSZ)mp1);
                free(mp1);
            }
            if (mp2 != NULL)
            {
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURFILECOUNT, (PSZ)mp2);
                free(mp2);
            }
            ULONG ulDrive, ulMap;
            APIRET arc = DosQueryCurrentDisk(&ulDrive, &ulMap);
            CHAR szMsg[100] = "?";
            if (arc == NO_ERROR)
            {
                double d = 0;
                arc = doshQueryDiskFree(ulDrive, &d);
                if (arc == NO_ERROR)
                {
                    CHAR sz2[20];
                    sprintf(szMsg, nlsGetString(WPSI_DRIVEXFREE), // "Drive %c: %s KB free",
                                UCHAR(ulDrive & 0xFF) + 'A' - 1,
                                nlsThousandsDouble(sz2,
                                                   d / 1024,
                                                   G_szThousand[0]));
                }
            }
            WinSetDlgItemText(hwndDlg, ID_WIDI_TARGETPATHFREE, szMsg);
        }
        break;

        /*
         *@@ WPIM_UPDATEBYTES:
         *      user msg posted from the Install thread
         *      when the "bytes" display needs updating.
         *      Parameters:
         *          SHORT mp1  current percentage
         */

        case WPIM_UPDATEBYTES:
            G_ulPercent = (ULONG)mp1;
            WinPostMsg(hwndProgressBar, WM_UPDATEPROGRESSBAR,
                       (MPARAM)G_ulPercent,      // current value
                       (MPARAM)100);           // maximum value
        break;

        /*
         * WM_TIMER:
         *      the timer is only running on the
         *      "Install" page to update the
         *      elapsed time once a second.
         *      We also attempt to have a meaningful
         *      "time to go" display, but this isn't
         *      always working right.
         */

        case WM_TIMER:
        {
            ULONG ulTime2 = dtGetULongTime();

            G_ulSecsElapsed = (ulTime2 - G_ulInstallTimeBegin) / 1000;

            ULONG   ulHours = G_ulSecsElapsed / 60 /60;
            ULONG   ulMinutes = ((G_ulSecsElapsed / 60) - (ulHours * 60));
            ULONG   ulSecs = G_ulSecsElapsed
                                - (ulHours * 60 * 60)
                                - (ulMinutes * 60);
            CHAR    szTime[30];
            sprintf(szTime, "%02d%c%02d%c%02d %s",
                    ulHours,    G_szTimeSep[0],
                    ulMinutes,  G_szTimeSep[0],
                    ulSecs,
                    nlsGetString(WPSI_ELAPSED));

            WinSetDlgItemText(hwndDlg, ID_WIDI_TIMEELAPSED, szTime);

            // calculate time to go
            if ((G_ulSecsElapsed) && (G_ulPercent > 10))
            {
                ULONG   ulSecsToGo = (G_ulSecsElapsed * (100 - G_ulPercent) ) / G_ulPercent;

                ULONG   ulHours = ulSecsToGo / 60 /60;
                ULONG   ulMinutes = ((ulSecsToGo / 60) - (ulHours * 60));
                ULONG   ulSecs = ulSecsToGo
                                    - (ulHours * 60 * 60)
                                    - (ulMinutes * 60);
                CHAR    szTime[30];
                sprintf(szTime, "%02d%c%02d%c%02d %s",
                        ulHours,    G_szTimeSep[0],
                        ulMinutes,  G_szTimeSep[0],
                        ulSecs,
                        nlsGetString(WPSI_TOGO));

                WinSetDlgItemText(hwndDlg, ID_WIDI_TIMELEFT, szTime);
            }
        }
        break;

        /*
         *@@ WPIM_FILEEXISTS:
         *      this is posted by the guiFileExists callback
         *      on the Install thread if a file exists already.
         *
         *      Parameters:
         *      -- PFILEEXISTS mp1: more info.
         *
         *      Note that guiFileExists waits until the global
         *      iErrorRC variable is set to anything but -1.
         *      The Install thread then resumes operation
         *      depending on that value.
         *
         *      This must set iErrorRC to one of the following:
         *
         *      --  CBRC_PROCEED:   replace this file
         *
         *      --  CBRC_SKIP:      skip this file
         *
         *      --  CBRC_ABORT:     stop processing.
         *
         *      Those are the return values of ShowFileExistsDlg.
         */

        case WPIM_FILEEXISTS:
            mrc = (MPARAM)ShowFileExistsDlg(hwndDlg,
                                            (FEFileError*)mp1);
        break;

        /*
         *@@ WPIM_INSTALLERROR:
         *      posted from the Install thread also.
         *      Parameters:
         *      --  PSZ mp1:    the error message (static buffer (99-10-26) [umoeller])
         *      --  ULONG mp2:  CBREC_* flags (wiarchive.h), allowed return codes.
         *
         *      Note that guiFileExists waits until the global
         *      iErrorRC variable is set to anything but -1.
         *      The Install thread then resumes operation
         *      depending on that value.
         *
         *      Depending on mp2, the following values may be returned:
         *      --  CBRC_RETRY:     have the install thread try again.
         *      --  CBRC_IGNORE:    ignore the error, go on (skip the file).
         *      --  CBRC_ABORT:     have guiFileExists terminate the
         *                          install thread and quit.
         */

        case WPIM_INSTALLERROR:
        {
            ULONG ulErrRsp;

            switch ((ULONG)mp2)
            {
                case CBREC_RETRYCANCEL:         ulErrRsp = MB_RETRYCANCEL; break;
                case CBREC_ABORTRETRYIGNORE:    ulErrRsp = MB_ABORTRETRYIGNORE; break;
                default:                        ulErrRsp = MB_CANCEL;
            }

            ULONG ulmbrc = WinMessageBox(HWND_DESKTOP,      // parent
                                         hwndDlg,       // owner
                                         (PCSZ)mp1,
                                         "WarpIN: Install Error",   // xxx
                                         0,
                                         MB_ICONEXCLAMATION | ulErrRsp | MB_MOVEABLE);

            switch (ulmbrc)
            {
                case MBID_RETRY:  mrc = (MPARAM)CBRC_RETRY; break;
                case MBID_IGNORE: mrc = (MPARAM)CBRC_IGNORE; break;
                // case MBID_CANCEL: mrc = (MPARAM)CBRC_ABORT; break;
                // case MBID_ABORT:  mrc = (MPARAM)CBRC_ABORT; break;
                default:          mrc = (MPARAM)CBRC_ABORT; break;
            }
        }
        break;

        /*
         *@@ WPIM_FILELOCKEDERROR:
         *      posted from guiFileLockedError when a file is
         *      locked and cannot be overwritten.
         *
         *      Parameters:
         *      -- PFILEEXISTS mp1: more info.
         *
         *      This must set iErrorRC to one of these:
         *
         *      --  CBRC_SKIP:    ignore the error, go on (skip the file).
         *
         *      --  CBRC_RETRY:   have the engine retry.
         *
         *      --  CBRC_UNLOCK:  have the engine unlock the file (DosReplaceModule).
         *
         *      --  CBRC_DEFER:   have the engine defer processing thru CONFIG.SYS.
         *
         *      --  CBRC_ABORT:   have guiFileExists terminate the
         *                          install thread and quit.
         *
         *      This was added (99-10-26) [umoeller].
         *
         *@@changed V0.9.14 (2001-08-09) [umoeller]: added unlock of WarpIN help
         */

        case WPIM_FILELOCKEDERROR:
        {
            FEFileError *pfi = (FEFileError*)mp1;

            // check if this file is the WarpIN help file
            // V0.9.14 (2001-08-09) [umoeller]
            if (G_hwndHelpInstance)
            {
                /*
                string str(pLocals->_pCodecProcess, pLocals->QueryWarpINPath());
                str += "\\warpin.hlp";
                if (!stricmp(pfi->_strFilename.c_str(), str.c_str()))
                */

                // replaced with compareI, which works with non-ASCII chars;
                // V0.9.20 (2002-07-03) [umoeller]

                ustring ustrFilename(pLocals->_pCodecProcess, pfi->_strFilename);
                ustring ustrHelp(pLocals->QueryWarpINPath());
                ustrHelp.appendUtf8("\\warpin.hlp");
                if (!ustrFilename.compareI(ustrHelp))
                {
                    // this is the WarpIN help file:
                    winhDestroyHelp(G_hwndHelpInstance, NULLHANDLE);
                    G_hwndHelpInstance = NULLHANDLE;
                    mrc = (MPARAM)CBRC_RETRY;
                    break;
                }
            }

            mrc = (MPARAM)ShowFileLockedDlg(hwndDlg,
                                            pfi);
        }
        break;

        /*
         *@@ WPIM_DONEWITHINSTALL:
         *      posted (!) from the Install thread
         *      when all files have been unpacked.
         *
         *      The Install thread terminates itself
         *      automatically. We need to dismiss
         *      the dialog and continue working
         *      on thread 1.
         *
         *      mp1 and mp2 are both NULL.
         *
         *      NOTE: even though we converted all
         *      WinPostMsg to WinSendMsg, this msg
         *      is still _posted_ (V0.9.12 (2001-05-22) [umoeller])
         */

        case WPIM_DONEWITHINSTALL:
            // stop timer on container page
            WinStopTimer(pEngine->_Locals._habThread1, hwndDlg, 1);

            // get outta here
            WinDismissDlg(hwndDlg, DID_OK);
        break;

        case WM_COMMAND:
            ConfirmCancelInstall(pLocals,
                                 hwndDlg);
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

    } // end switch

    return (mrc);
}

/*
 *@@ guiStartInstall:
 *      this gets called from warpin.cpp to actually
 *      start the install process. This gets called
 *      in "install" modes after all information has
 *      been collected from the user.
 *
 *      Required callback.
 *
 *      It is the responsibility of this function to
 *      show up the "Installing..." dialog
 *      (progress bar, current packages, etc.)
 *      and call wpiStartInstallThread, which will
 *      process the FEInstallJob list.
 *
 *      If this returns TRUE, this means the packages
 *      are to be stored in the database afterwards
 *      (i.e. installation went OK).
 *
 *      If this returns FALSE, packages are not
 *      stored, because installation was cancelled or
 *      errors occured.
 */

BOOL guiStartInstall(GUIInstallEngine &Engine)
{
    G_hwndMainFrame = gshrLoadDlg(HWND_DESKTOP,     // parent
                                  HWND_DESKTOP,
                                  fnwpInstallingStatus,
                                  NULLHANDLE,
                                  ID_WID_INSTALLING_STATUS,
                                  &Engine);       // create param
                // this calls wpiStartInstallThread in turn

    // add ourselves to the tasklist
    winhAddToTasklist(G_hwndMainFrame,
                      G_hptrMain);

    winhCenterWindow(G_hwndMainFrame);
    WinShowWindow(G_hwndMainFrame, TRUE);

    // start Install thread
    wpiStartInstallThread(Engine);
    // go!!
    ULONG ulrc = WinProcessDlg(G_hwndMainFrame);

    WinDestroyWindow(G_hwndMainFrame);
    G_hwndMainFrame = NULLHANDLE;
    return (ulrc == DID_OK);
}

/* ******************************************************************
 *
 *    GUIInstallEngine implementation
 *
 ********************************************************************/

/*  General note for the following functions:
 *      Most of these are callbacks from the
 *      main Install thread on which the "Unpack"
 *      methods of the WIArchive class are
 *      running. In other words, it's the Install
 *      thread which creates all the files.
 *      Since all the following functions therefore
 *      run on the Install thread, and since the
 *      Install thread does _not_ have a message
 *      queue, we better not manipulate windows
 *      here. Instead, we post msgs to the main
 *      window which does the updating on thread 1.
 *
 *      This runs on the Install thread.
 */

/*
 *@@ GUIInstallEngine:
 *      the constructor.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

GUIInstallEngine::GUIInstallEngine(GUILocals &Locals)     // in: ref to FELocals object
                    : FEInstallEngine(*Locals._pDatabase,
                                      Locals)
{
    // moved global status window here V0.9.20 (2002-07-03) [umoeller]
    _ulCurrentBoldID = 0;

    _hwndGlobalStatus = gshrLoadDlg(HWND_DESKTOP, HWND_DESKTOP,
                                    WinDefDlgProc,
                                    NULLHANDLE, ID_WID_GLOBALSTATUS,
                                    NULL);
    winhSetControlsFont(_hwndGlobalStatus, 0, 1000,
                        gshrQueryDefaultFont());

    // winhEnableDlgItem(G_hwndGlobalStatus, ID_WIDI_REMOVING, FALSE);
        // i don't think this is true any more  V0.9.14 (2001-07-26) [umoeller]

    WinShowWindow(_hwndGlobalStatus, TRUE);
}

/*
 *@@ ~GUIInstallEngine:
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

GUIInstallEngine::~GUIInstallEngine()
{
    if (_hwndGlobalStatus)
    {
        WinDestroyWindow(_hwndGlobalStatus);
        _hwndGlobalStatus = NULLHANDLE;
    }
}

/*
 *@@ guiUpdateGlobalStatus:
 *      this makes the specified dlg item in the
 *      "Global status" dialog bold and puts the
 *      small arrow next to it.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *@@added V0.9.0 (99-11-01) [umoeller]
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

VOID GUIInstallEngine::UpdateGlobalStatus(BYTE bStatus)
{
    ULONG   ulToAdd = 0,
            ulToRemove = 0;
    PCSZ    pszCurrentConfig = NULL;

    CheckJobs(&ulToAdd, &ulToRemove);

    winhEnableDlgItem(_hwndGlobalStatus, ID_WIDI_REMOVING, (ulToRemove != 0));
    winhEnableDlgItem(_hwndGlobalStatus, ID_WIDI_UNPACKING, (ulToAdd != 0));
    winhEnableDlgItem(_hwndGlobalStatus, ID_WIDI_CONFIGSYS,
                      QueryInstallVar(_Locals._varConfigSys));
    winhEnableDlgItem(_hwndGlobalStatus, ID_WIDI_REGISTERING,
                      QueryInstallVar(_Locals._varWPSClasses));
    winhEnableDlgItem(_hwndGlobalStatus, ID_WIDI_CREATINGOBJECTS,
                      QueryInstallVar(_Locals._varWPSObjects));

    if (_ulCurrentBoldID)
    {
        winhSetWindowFont(WinWindowFromID(_hwndGlobalStatus,
                                          _ulCurrentBoldID),
                          gshrQueryDefaultFont());
    }

    ULONG   ulDlgID = 0;
    BOOL    fDestroyConfigStatus = FALSE;

    switch (bStatus)
    {
        case GSTAT_LOADINGARCHIVE:
            ulDlgID = ID_WIDI_LOADINGARCHIVE;
            fDestroyConfigStatus = TRUE;
        break;

        case GSTAT_COLLECTING:
            ulDlgID = ID_WIDI_COLLECTING;
            fDestroyConfigStatus = TRUE;
        break;

        case GSTAT_REMOVING:
            ulDlgID = ID_WIDI_REMOVING;
            fDestroyConfigStatus = TRUE;
        break;

        case GSTAT_UNPACKING:
            ulDlgID = ID_WIDI_UNPACKING;
            fDestroyConfigStatus = TRUE;
        break;

        case GSTAT_CONFIGSYS:
            ulDlgID = ID_WIDI_CONFIGSYS;
            pszCurrentConfig = nlsGetString(WPSI_UPDATINGCONFIGSYS);
                            // "Updating CONFIG.SYS");
        break;

        case GSTAT_WPSCLASSES:
            ulDlgID = ID_WIDI_REGISTERING;
            pszCurrentConfig = nlsGetString(WPSI_REGISTERINGCLASSES);
                            // "Registering WPS classes");
        break;

        case GSTAT_WPSOBJECTS:
            ulDlgID = ID_WIDI_CREATINGOBJECTS;
            pszCurrentConfig = nlsGetString(WPSI_CREATINGOBJECTS);
                            // "Creating WPS objects");
        break;

        case GSTAT_UPDATINGDATABASE:
            ulDlgID = ID_WIDI_UPDATINGDATABASE;
        break;

        case GSTAT_DONEWITHALL:
            fDestroyConfigStatus = TRUE;
        break;
    }

    HWND    hwndArrow = WinWindowFromID(_hwndGlobalStatus, ID_WIDI_ARROW);
    if (bStatus != GSTAT_DONEWITHALL)
    {
        HWND hwndItem;
        if (hwndItem = WinWindowFromID(_hwndGlobalStatus, ulDlgID))
        {
            SWP     swpArrow, swpBold;
            WinQueryWindowPos(hwndArrow,
                              &swpArrow);
            WinQueryWindowPos(hwndItem,
                              &swpBold);
            WinSetWindowPos(hwndArrow, HWND_TOP,
                            swpArrow.x, swpBold.y, swpArrow.cx, swpArrow.cy,
                            SWP_MOVE);
            winhSetWindowFont(hwndItem,
                              gshrQueryBoldFont());
            _ulCurrentBoldID = ulDlgID;
        }
    }
    else
        WinShowWindow(hwndArrow, FALSE);

    static HWND hwndConfigStatus = NULLHANDLE;

    if (pszCurrentConfig)
    {
        // configuring: then show configuration window also
        if (hwndConfigStatus == NULLHANDLE)
        {
            // not loaded yet:
            hwndConfigStatus = gshrLoadDlg(HWND_DESKTOP,
                                          HWND_DESKTOP,
                                          WinDefDlgProc,
                                          NULLHANDLE,      // our own EXE
                                          ID_WID_CONFIGURINGSTATUS,
                                          NULL);
            winhCenterWindow(hwndConfigStatus);
        }
        HWND hwndInfoInStatus = WinWindowFromID(hwndConfigStatus, ID_WIDI_INFOTEXT);

        WinSetWindowText(hwndInfoInStatus, pszCurrentConfig);
        WinShowWindow(hwndConfigStatus, TRUE);
    }

    if ( (fDestroyConfigStatus) && (hwndConfigStatus) )
    {
        WinDestroyWindow(hwndConfigStatus);
        hwndConfigStatus = NULLHANDLE;
    }
}

/*
 *@@ guiVarPrompt:
 *      this gets called by wpiResolveMacros if an
 *      environment variable needs to be resolved and
 *      a value could not be found.
 *
 *      This must return the new value for that variable
 *      in a new buffer allocated using malloc().
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      WARNING: This now runs on the "parse script" thread.
 *
 *      Throws:
 *      -- BSCancelExcpt.
 *
 *@@changed V0.9.4 (2000-07-26) [umoeller]: text wasn't displayed right. Fixed.
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

PSZ GUIInstallEngine::VarPrompt(FEVarPrompt &VarPrompt)    // in: VARPROMPT definition
{
    GUILocals *pLocals = (GUILocals *)&_Locals;
    PSZ     pszValue;
    BOOL    fContinue = FALSE;
    do
    {
        fContinue = FALSE;

        // set help panel for fnwpDialogWithHelp
        DLGWITHHELP dwh;
        dwh.pLocals = pLocals;
        dwh.ulHelpPanel = IDHI_VARPROMPT_DLG;
        HWND hwndDlg = gshrLoadDlg(HWND_DESKTOP, HWND_DESKTOP,
                                   gshr_fnwpDialogWithHelp,
                                   NULLHANDLE, ID_WID_ENTERVARIABLE,
                                   &dwh);

        winhCenterWindow(hwndDlg);
        winhSetControlsFont(hwndDlg, 0, 2000,
                        gshrQueryDefaultFont());

        PSZ pszText;

        // ### handle fail condition here

        // description given?
        if (VarPrompt._strDescription())
            // yes: use that
            pszText = strdup(VarPrompt._strDescription.c_str());
        else
            // use default text from dlg template...
            pszText = winhQueryWindowText(WinWindowFromID(hwndDlg, ID_WIDI_EV_INFOTEXT));

        HWND hwndTextView = txvReplaceWithTextView(hwndDlg,
                                                   ID_WIDI_EV_INFOTEXT,
                                                   WS_VISIBLE | WS_TABSTOP
                                                        | XS_VSCROLL | XS_AUTOVHIDE,
                                                   2);  // border
        WinSendMsg(hwndTextView,
                   TXM_SETWORDWRAP,
                   (MPARAM)(TRUE),
                   0);

        txvStripLinefeeds(&pszText, 4);

        // set description text
        WinSetWindowText(hwndTextView, pszText);
        free(pszText);

        CHAR szVarName[300];
        sprintf(szVarName, "SET %s=", VarPrompt._strVarName.c_str());
        WinSetDlgItemText(hwndDlg, ID_WIDI_EV_SETTEXT, szVarName);

        HWND hwndEntryField = WinWindowFromID(hwndDlg, ID_WIDI_EV_ENTRYFIELD);
        winhSetEntryFieldLimit(hwndEntryField, 255);

        if (WinProcessDlg(hwndDlg) == DID_OK)
        {
            pszValue = winhQueryWindowText(hwndEntryField);
        }
        else
            throw BSCancelExcpt();

        WinDestroyWindow(hwndDlg);

        if (!VarPrompt.Validate(pszValue))
        {
            if (pszValue == NULL)
                pszValue = strdup("NULL");

            ustring str(pLocals->_pCodecGui, pszValue);
            if (_Locals.MessageBox(102,      // error
                                   185,      // %1 not valid
                                   MSG_ERROR_RETRYCANCEL,
                                   &str,
                                   1)
                    == MBID_CANCEL)
                throw BSCancelExcpt();

            fContinue = TRUE;
        }

        if (fContinue)
            free(pszValue);
    } while (fContinue);

    return (pszValue);
}

/*
 *@@ guiUpdatePackage:
 *      GUI callback from warpin.cpp
 *      to update the "Package" display.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

void GUIInstallEngine::UpdatePackage(FEInstallJob *pJob,
                                     ULONG ulPckThis,
                                     ULONG ulPcksTotal)
{
    CHAR szMsg[100];
    sprintf(szMsg,
            nlsGetString(WPSI_PACKAGEX),  // "Package %d out of %d"
            ulPckThis,
            ulPcksTotal);
    WinPostMsg(G_hwndMainFrame, WPIM_UPDATEPACKAGE,
               pJob,
               strdup(szMsg));
        // the window proc will release the string memory
}

/*
 *@@ guiUpdateFile:
 *      GUI callback from warpin.cpp
 *      to update the "File" display.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

void GUIInstallEngine::UpdateFile(WIFileHeader *pwifh,
                                  ULONG ulFileThis,
                                  ULONG ulFilesTotal)
{
    CHAR szSizeRounded[20];
    nlsThousandsULong(szSizeRounded,
                      ((pwifh->origsize + 512) / 1024),
                      G_szThousand[0]);

    CHAR    szFile[CCHMAXPATH + 100];
    CHAR    szMsg[100];
    sprintf(szFile, "%s (%s KB)", pwifh->name, szSizeRounded);
    sprintf(szMsg,
            nlsGetString(WPSI_FILEX), // "File %d out of %d",
            ulFileThis,
            ulFilesTotal);
    WinPostMsg(G_hwndMainFrame, WPIM_UPDATEFILE,
               strdup(szFile),
               strdup(szMsg));
        // the window proc will release the string memory
}

/*
 *@@ guiUpdateBytes:
 *      GUI callback from warpin.cpp
 *      to update the progress display.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

void GUIInstallEngine::UpdateBytes(double dBytesDone,
                                   double dBytesTotal)
{
    SHORT sPercent = (SHORT)((dBytesDone / dBytesTotal) * 100);
    WinPostMsg(G_hwndMainFrame, WPIM_UPDATEBYTES,
               (MPARAM)sPercent, 0);
}

/*
 *@@ guiFileExists:
 *      callback from FEInstallEngine::Install to confirm
 *      whether a file should be replaced.
 *
 *      This only gets called if EngineWICallback has
 *      determined that the user should be prompted
 *      for what to do, depending on the overwrite
 *      settings in FELocals.
 *
 *      This must return one of the following:
 *
 *      --  CBRC_PROCEED:   replace this file
 *
 *      --  CBRC_SKIP:      skip this file
 *
 *      If the user presses "Cancel", it's our own
 *      responsibility to terminate everthing.
 *      We must _not_ return from this function then.
 *
 *      This callback should also offer the user an
 *      opportunity to reset the three flags in FELocals
 *      so that prompting can be disabled for subsequent
 *      files.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@changed V0.9.12 (2001-05-22) [umoeller]: converted all WinPostMsg to WinSendMsg
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

int GUIInstallEngine::OnFileExists(FEFileError *pfi)
{
    // we cannot do a WinSendMsg, because this would
    // require the Install thread to have a PM message
    // queue.
    // WinPostMsg works though, so we rely on a global
    // variable to get the return code. This is safe
    // since this variable is only set by thread 1
    // after it receives our message.

    // changed V0.9.12 (2001-05-22) [umoeller]: this
    // thread now has a PM message queue, so use
    // WinSendMsg to fix serialization problems on SMP.
    int irc = (int)WinSendMsg(G_hwndMainFrame,
                              WPIM_FILEEXISTS,
                              (MPARAM)pfi,     // pass structure to main window
                              (MPARAM)0);

    // wait until the user has selected something
    // in the "File exists" dialog on thread 1;
    // WPIM_FILEEXISTS in fnwpPages (above) sets
    // iErrorRC to either CBRC_CANCEL, CBRC_SKIP
    // or CBRC_PROCEED
    // DosWaitEventSem(G_hevGUIDone, SEM_INDEFINITE_WAIT);

    if (irc == CBRC_ABORT)
        // "Cancel" pressed: we need to terminate
        // our thread then, or OS/2 will hang
        _endthread();       // @@todo this doesn't work

    return (irc);
}

/*
 *@@ guiFileError:
 *      callback from FEInstallEngine::Install when
 *      some major error occurs.
 *
 *      FEInstallEngine now composes a meaningful
 *      error message for us, which we only need to
 *      display.
 *
 *      iErrorRsp signals if the error is recoverable:
 *
 *      --  CBREC_RETRYCANCEL
 *
 *      --  CBREC_ABORTRETRYIGNORE
 *
 *      --  CBREC_CANCEL
 *
 *      This must return one of the following:
 *
 *      --  CBRC_RETRY     retry the failing operation
 *
 *      If the user presses "Cancel", it's our own
 *      responsibility to terminate the Install thread.
 *      We must _not_ return from this function then.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@changed V0.9.0 (99-10-26) [umoeller]: function prototype changed
 *@@changed V0.9.0 (99-10-26) [umoeller]: string and code handling reworked completely
 *@@changed V0.9.1 (2000-02-06) [umoeller]: renamed from guiInstallError
 *@@changed V0.9.12 (2001-05-22) [umoeller]: converted all WinPostMsg to WinSendMsg
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 *@@changed V0.9.20 (2002-07-03) [umoeller]: moved lots of code to FEInstallEngine
 */

int GUIInstallEngine::OnFileError(FEInstallJob *pJob,    // in: current install job, ptr can be NULL!
                                  const ustring &strErrorMsg,   // in: error message from engine
                                  int iErrorRsp)      // in: possible response (CBREC_* flags)
{
    // we cannot do a WinSendMsg, because this would
    // require the Install thread to have a PM message
    // queue.
    // WinPostMsg works though, so we rely on a global
    // variable to get the return code. This is safe
    // since this variable is only set by thread 1
    // after it receives our message.
    // changed V0.9.12 (2001-05-22) [umoeller]: this
    // thread now has a PM message queue, so use
    // WinSendMsg to fix serialization problems on SMP.

    GUILocals *pLocals = (GUILocals*)&_Locals;

    // add button explanations from message file
    ULONG ulMsg = 0;
    switch (iErrorRsp)
    {
        case CBREC_RETRYCANCEL:
            ulMsg = 111;
        break;

        case CBREC_ABORTRETRYIGNORE:
            ulMsg = 112;
        break;

        default:
            ulMsg = 113;
    }

    // append msg after existing one
    ustring ustrMsg2,
            ustrTotal(strErrorMsg);
    _Locals.GetMessage(ustrMsg2,
                       ulMsg);
    ustrTotal += ustrMsg2;

    string strMsg3;
    strMsg3.assignUtf8(pLocals->_pCodecGui, ustrTotal);

    // post msg to fnwpInstallingStatus to display msg
    int irc = (int)WinSendMsg(G_hwndMainFrame,
                              WPIM_INSTALLERROR,
                              (MPARAM)strMsg3.c_str(),
                              (MPARAM)iErrorRsp);

    return (irc);
}

/*
 *@@ OnFileError:
 *      callback from FEInstallEngine::Install when
 *      the disk is full. This is fatal, and we
 *      cannot recover. This should display a
 *      message to the user; the engine will
 *      exit afterwards.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

VOID GUIInstallEngine::OnDiskFull(char cDriveLetter)
{
    // send msg to fnwpInstallingStatus to display msg
    GUILocals *pLocals = (GUILocals*)&_Locals;
    string strBuf;
    ustring austr;
    austr._printf("%c:", cDriveLetter);
    pLocals->GetMessageA(strBuf,
                         289, //  Disk %1 is full. Installation cannot continue.
                         &austr,
                         1);
    WinSendMsg(G_hwndMainFrame,
               WPIM_INSTALLERROR,
               (MPARAM)strBuf.c_str(),
               (MPARAM)CBREC_CANCEL);
}

/*
 *@@ guiFileLockedError:
 *      this is a special case of file error when
 *      the FEInstallEngine::Install has detected
 *      that a file should be overwritten which is
 *      currently in use by another process.
 *
 *      This must return one of these:
 *
 *      --  CBRC_SKIP:    ignore the error, go on (skip the file).
 *
 *      --  CBRC_RETRY:   have the engine retry.
 *
 *      --  CBRC_UNLOCK:  have the engine unlock the file (DosReplaceModule).
 *
 *      --  CBRC_DEFER:   have the engine defer processing thru CONFIG.SYS.
 *
 *      If the user presses "Cancel", it's our own
 *      responsibility to terminate the Install thread.
 *      We must _not_ return from this function then.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@added V0.9.0 (99-10-26) [umoeller]
 *@@changed V0.9.12 (2001-05-22) [umoeller]: converted all WinPostMsg to WinSendMsg
 *@@changed V0.9.12 (2001-05-31) [umoeller]: _really_ converted WinPostMsg to WinSendMsg, previous broke "locked" support
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

int GUIInstallEngine::OnFileLockedError(FEFileError *pfi,
                                        int iErrorCode)      // in: usually 32 == ERROR_SHARING_VIOLATION
{
    // V0.9.12 (2001-06-03) [umoeller]: this was still WinPostMsg
    int irc = (int)WinSendMsg(G_hwndMainFrame,
                              WPIM_FILELOCKEDERROR,
                              (MPARAM)pfi,
                              (MPARAM)0);

    if (irc == CBRC_ABORT)
    {
        // "Cancel" pressed: we need to terminate
        // our thread then, or OS/2 will hang
        WinPostMsg(G_hwndMainFrame, WM_CLOSE, 0, 0);
        _endthread();       // @@todo this doesn't work
    }

    return (irc);
}

/*
 *@@ guiDoneWithInstall:
 *      callback when all files have been
 *      unpacked successfully.
 *      After this, the Install thread will
 *      terminate itself (warpin.cpp).
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      This runs on the Install thread.
 *
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUIInstallEngine implementation method
 */

void GUIInstallEngine::DoneWithInstall(void)
{
    WinPostMsg(G_hwndMainFrame,
               WPIM_DONEWITHINSTALL,
               NULL,
               NULL);
}


