
/*
 *@@sourcefile gui_db_thread.cpp:
 *      this has the GUI stuff for the database thread.
 *
 *      This has been separated from gui_db.cpp to reduce
 *      source code dependencies and speed up compile times.
 *
 *@@added V0.9.1 (2000-02-08) [umoeller]
 */

/*
 *      This file Copyright (C) 1998-2001 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_WINWORKPLACE
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#include <os2.h>

#include <stdio.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h
#include <io.h>

#include "setup.h"

// include's from helpers
#include "helpers\configsys.h"
#include "helpers\datetime.h"
#include "helpers\dosh.h"
#include "helpers\cnrh.h"
#include "helpers\comctl.h"
#include "helpers\except.h"
#include "helpers\gpih.h"
#include "helpers\nls.h"
#include "helpers\nlscache.h"
#include "helpers\winh.h"
#include "helpers\stringh.h"
#include "helpers\threads.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_package.h"
#include "engine\fe_package_db.h"
#include "engine\fe_database.h"

#include "frontend\warpin.h"
#include "frontend\calbacks.h"

#include "frontend\dlgids.h"
#include "frontend\gui.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

// extern HWND     G_hwndDBPackagesDlg;
            // in gui_db.cpp

// BOOL    G_DBThreadUnlockFile = FALSE;       // return code for WPIM_FILELOCKEDERROR

/*
 *@@ fnwpDatabaseStatus:
 *      this is the window procedure for the Database
 *      thread status window which pops up while the
 *      database thread is running. Gets created from
 *      guiStartVerify or guiStartRemove.
 *
 *      This runs on thread 1 and receives messages from
 *      guiDatabaseCallback (running on the Database thread,
 *      which received this hwnd as a user param).
 *
 *      When a window is created using this function,
 *      the Database thread is started anew. mp2 of
 *      WM_INITDIALOG (pCreate of WinLoadDlg) must
 *      have a PDATABASESTATUSINFO, which we're copying
 *      here.
 *
 *      This is a bit sick because this dialog must operate
 *      both modally and non-modally.
 *
 *      --  This gets called in "install" mode when packages
 *          have been selected for de-installation. We are
 *          then operating modally and guiStartRemove is in
 *          a WinProcessDlg call.
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked database thread callbacks
 *@@changed V0.9.1 (2000-01-03) [umoeller]: fixed package recc status display
 *@@changed V0.9.3 (2000-05-03) [umoeller]: added error checking to report save
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added "unlock file" support
 *@@changed V0.9.18 (2002-03-08) [umoeller]: adjusted for new FEDatabaseStatus implementation
 */

MRESULT EXPENTRY fnwpDatabaseStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    GUIDatabaseStatus *pdbsi = (GUIDatabaseStatus*)WinQueryWindowPtr(hwndDlg, QWL_USER);
    GUILocals *pLocals;
    if (pdbsi)
        pLocals = (GUILocals*)&pdbsi->_Locals;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      since this can only be created when WM_COMMAND
         *      is evaluated in fnwpDBPackagesDlg, and since this
         *      must come from a context menu item, the global
         *      variable preccSource still has the DATABASERECORD
         *      of the application or package which should be
         *      worked on.
         *
         *      mp2 (pCreate) has the DATABASESTATUSINFO structure
         *      for this dialog.
         */

        case WM_INITDLG:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            // create progress bar
            ctlProgressBarFromStatic(WinWindowFromID(hwndDlg,
                                                      ID_WIDI_DBT_PROGRESSBAR),
                                      PBA_ALIGNCENTER | PBA_BUTTONSTYLE);

            // get the create struct which was passed in mp2
            pdbsi = (GUIDatabaseStatus*)mp2;
            WinSetWindowPtr(hwndDlg, QWL_USER, pdbsi);

            WinSetWindowText(hwndDlg, pdbsi->_strWindowTitle.c_str());

            // disable "Save Report As" button
            winhEnableDlgItem(hwndDlg, ID_WIDI_DBT_SAVEREPORTAS, FALSE);
        break;

        /*
         * WPIM_UPDATEPACKAGE:
         *      this gets posted by guiDatabaseCallback.
         *
         *      mp1 has the GUIDatabaseStatus*.
         *
         *      In there, pPackage has been set to the
         *      new package, or is NULL if we're done.
         *
         *      No return value.
         */

        case WPIM_UPDATEPACKAGE:
        {
            // now that the package has changed,
            // update the visible status of the
            // previous package

            if (pdbsi->_pPrevPackage)
            {
                if (pdbsi->_hwndDBPackagesDlg)
                {
                    // database mode:
                    // running for second package (or we're done),
                    // get database record
                    PDATABASERECORD precc;
                    if (precc = (PDATABASERECORD)pdbsi->_pPrevPackage->_pvGuiData)
                    {
                        // non-modal mode:
                        if (pdbsi->_pPrevPackage->_ulStatus == 1)
                            // error occured:
                            precc->recc.hptrIcon
                                = precc->recc.hptrMiniIcon
                                = pLocals->_hptrError;
                        else
                            // status should be 2:
                            precc->recc.hptrIcon
                                = precc->recc.hptrMiniIcon
                                = pLocals->_hptrOK;
                        WinSendMsg(WinWindowFromID(pdbsi->_hwndDBPackagesDlg,
                                                   ID_WIDI_PACKAGESCNR),
                                   CM_INVALIDATERECORD,
                                   (MPARAM)&precc,
                                   MPFROM2SHORT(1, CMA_ERASE));
                    }
                }
            }

            if (pdbsi->_pPackage)
            {
                // not yet done:

                // store the new package as the previous
                pdbsi->_pPrevPackage = pdbsi->_pPackage;
                // create package ID on the stack
                /* FEPackageID PckID(*pLocals,
                                  pStatus->_pPackage, __FILE__, __LINE__,
                                  FALSE);    // don't allow short format
                */

                WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_APPLICATION,
                                  pdbsi->_pPackage->_PckID._strApplicationGui.c_str());
                WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_PACKAGE,
                                  pdbsi->_pPackage->_PckID._strPackageNameGui.c_str());
            } // end if (pStatus)
            else
            {
                // pStatus == NULL: we're done
                pdbsi->_pPrevPackage = NULL;

                WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_PROGRESSBAR),
                           WM_UPDATEPROGRESSBAR,
                           (MPARAM)1,
                           (MPARAM)1);      // 100%

                // if no errors were found:
                if (!pdbsi->_fErrorsFound)
                    WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_REPORTMLE),
                               MLM_INSERT,
                               (MPARAM)nlsGetString(WPSI_DONE),
                               MPNULL);

                // enable "Save Report As" button
                winhEnableDlgItem(hwndDlg, ID_WIDI_DBT_SAVEREPORTAS, TRUE);

                // rename "Cancel" button
                WinSetDlgItemText(hwndDlg, DID_CANCEL, "~Close"); // @@todo localize
                // empty "File" static text
                WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_FILE, "");

                // if we're operating non-modally (i.e. in database mode),
                // we must close the logger that was opened from the
                // database gui
                if (!pdbsi->_hwndDBPackagesDlg)
                {
                    pLocals->CloseLogger();
                }
            }
            // post semaphore
            // DosPostEventSem(G_hevGUIDone);
        }
        break;

        /*
         * WPIM_UPDATEFILE:
         *      this gets posted by guiDatabaseCallback.
         *      mp1 has the GUIDatabaseStatus*.
         *
         *      No return value.
         */

        case WPIM_UPDATEFILE:
        {
            GUIDatabaseStatus *pStatus = (GUIDatabaseStatus*)mp1;

            if (pStatus)
            {
                WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_FILE,
                                  pStatus->_pwifh->name);

                WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_PROGRESSBAR),
                           WM_UPDATEPROGRESSBAR,
                           (MPARAM)(pStatus->_ulCurrentFile),      // current value
                           (MPARAM)(pStatus->_cFiles));    // maximum value
            }

            // post semaphore
            // DosPostEventSem(G_hevGUIDone);
        }
        break;

        /*
         * WPIM_ADDSTRING:
         *      generic message from guiDatabaseCallback to
         *      add a string. mp1 points to a static buffer
         *      containing the string.
         *
         *      mp2 is null if this is just a report, but no
         *      error message. Otherwise it contains "1".
         *
         *      No return value.
         */

        case WPIM_ADDSTRING:
        {
            APIRET  arc = (APIRET)mp2;
            WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_REPORTMLE),
                       MLM_INSERT,
                       (MPARAM)mp1,
                       MPNULL);

            if (arc)
                pdbsi->_fErrorsFound = TRUE;

            // post semaphore
            // DosPostEventSem(G_hevGUIDone);
        }
        break;

        /*
         * WM_COMMAND:
         *      intercept the "Save as" button
         */

        case WM_COMMAND:
        {
            switch (SHORT1FROMMP(mp1))
            {
                case ID_WIDI_DBT_SAVEREPORTAS:
                {
                    FILEDLG fd;
                    memset(&fd, 0, sizeof(FILEDLG));
                    fd.cbSize = sizeof(FILEDLG);
                    fd.fl = FDS_SAVEAS_DIALOG
                              | FDS_ENABLEFILELB
                              | FDS_CENTER;
                    sprintf(fd.szFullFile, "*.log");
                    if (    WinFileDlg(HWND_DESKTOP,    // parent
                                       hwndDlg,
                                       &fd)
                        && (fd.lReturn == DID_OK)
                       )
                    {
                        PSZ pszLogText = winhQueryWindowText(WinWindowFromID(hwndDlg,
                                                                             ID_WIDI_DBT_REPORTMLE));
                        CHAR szBackup[CCHMAXPATH];
                        APIRET arc = doshWriteTextFile(fd.szFullFile,
                                                       pszLogText,
                                                       NULL,
                                                       szBackup);
                        free(pszLogText);

                        if (arc)
                        {
                            PSZ pszError = doshQuerySysErrorMsg(arc);
                            ustring str;
                            str.assignCP(pLocals->_pCodecProcess,
                                         pszError);
                            gshrMLEMessageBox(pLocals,
                                              102,  // "error"
                                              188,  // "writing report",
                                              FALSE,    // OK only
                                              &str,
                                              1);
                            free(pszError);
                        }
                    }
                }
                break;

                default:
                    // includes the "Close" button;
                    // dismiss the dialog
                    mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            }
        }
        break;

        /*
         * WPIM_FILELOCKEDERROR:
         *      (PSZ) mp1 has the full file name.
         *      V0.9.4 (2000-07-01) [umoeller]
         *
         *      This must return TRUE if the file
         *      should be unlocked on the DB thread.
         */

        case WPIM_FILELOCKEDERROR:
        {
            ustring str;
            str.assignCP(pLocals->_pCodecProcess,
                         (PSZ)mp1);
            mrc = (MPARAM)(pLocals->MessageBox(102,  // error
                                         191,  // unlock %1?
                                         MSG_CONFIRM_YESNO_DEFYES,
                                         &str,
                                         1)
                               == MBID_YES);

            // post semaphore
            // DosPostEventSem(G_hevGUIDone);
        }
        break;

        /*
         * WPIM_DELETE_EXISTINGFILENEWER:
         *      mp1 has the PGUIFILEERROR.
         *
         *      This must return TRUE if the file
         *      is to be deleted anyway, or FALSE if not.
         */

        case WPIM_DELETE_EXISTINGFILENEWER:
        {
            FEFileError *pfi = (FEFileError *)mp1;
            HWND hwndConfirm = gshrLoadConfirmDlg(hwndDlg,
                                                  ID_WID_DELETE_FILENEWER,
                                                  WinDefDlgProc,
                                                  pfi);
            if (hwndConfirm)
            {
                // set file name (replace %1 in static text)
                HWND        hwndItem = WinWindowFromID(hwndConfirm, ID_WIDI_EXISTS_TXT1);
                PSZ         pszMessage = winhQueryWindowText(hwndItem);
                ULONG       ulOfs = 0;
                strhFindReplace(&pszMessage, &ulOfs, "%1", pfi->_strFilename.c_str());
                WinSetWindowText(hwndItem, pszMessage);
                free(pszMessage);

                ULONG ulrc = WinProcessDlg(hwndConfirm);

                switch (ulrc)
                {
                    case DID_OK:    mrc = (MPARAM)TRUE;       break;
                    case ID_WIDI_NOBUTTON: mrc = (MPARAM)FALSE; break;
                    case DID_CANCEL: throw(new BSCancelExcpt); break;
                }

                WinDestroyWindow(hwndConfirm);
            }

            // post semaphore
            // DosPostEventSem(G_hevGUIDone);
        }
        break;

        case WM_DESTROY:
            // clean up copy of DATABASESTATUSINFO we
            // created in WM_CREATE
            delete pdbsi;
            WinSetWindowPtr(hwndDlg, QWL_USER, NULL);
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ ComposeFullPath:
 *      composes the full path for the current package
 *      and WIFileHeader, puts it into str and returns
 *      str.c_str().
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

PCSZ GUIDatabaseStatus::ComposeFullPath(string &str)
{
    if ((_pPackage) && (_pwifh))
    {
        GUILocals *pLocals = (GUILocals*)&_Locals;

        str.assignUtf8(pLocals->_pCodecGui, _pPackage->QueryTargetPath());
        if (str[str.length() - 1] != '\\')
            str += '\\';
        str += _pwifh->name;
    }

    return str.c_str();
}

/*
 *@@ SendString:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID GUIDatabaseStatus::SendString(BOOL fIsErrorMsg)
{
    _strMsg += "\r\n";
    WinSendMsg(_hwndStatus,
               WPIM_ADDSTRING,
               (MPARAM)_strMsg.c_str(),
               (MPARAM)fIsErrorMsg);
}

/*
 *@@ guiDatabaseCallback:
 *      this callback gets called on the Database
 *      thread during de-installation/verification.
 *
 *      Required callback.
 *
 *      Note: this is _not_ running on thread 1, but
 *      on the Database thread. Be careful with global
 *      variables.
 *
 *      The callbacks come in as follows:
 *
 *      1) DBC_NEXTPACKAGE with pPckInfo pointing to the package,
 *
 *      2) DBC_NEXTFILE with pHeader pointing to the current header;
 *
 *      3) _possibly_ one of DBC_FILENOTFOUND, DBC_EXISTINGFILENEWER,
 *         or DBC_EXISTINGFILEOLDER; if the existing file is OK,
 *         no callback occurs;
 *
 *      4) next file: go back to 2).
 *
 *      5) next package (only for datVerifyApplication): go back to 1).
 *
 *      5) After all files have been processed, another DBC_NEXTPACKAGE
 *         comes in with (pPckInfo == 0).
 *
 *      This must return 0 always, EXCEPT for :
 *
 *      --  with DBC_FILELOCKED report code, this must return TRUE or FALSE,
 *          depending on whether the file in question should be unlocked;
 *
 *      --  with DBC_DELETE_EXISTINGFILENEWER, this must return TRUE or
 *          FALSE depending on whether the newer file should be deleted.
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked database thread callbacks
 *@@changed V0.9.1 (2000-01-05) [umoeller]: fixed file error message, which reported "not found" always
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added "unlock file" support
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added DEEXECUTE support
 *@@changed V0.9.9 (2001-04-06) [umoeller]: added DBC_DELETE_EXISTINGFILENEWER
 *@@changed V0.9.12 (2001-05-22) [umoeller]: converted all WinPostMsg to WinSendMsg
 *@@changed V0.9.18 (2002-03-08) [umoeller]: removed excessive string buffer
 *@@changed V0.9.18 (2002-03-08) [umoeller]: removed all config callbacks, turned those into virtual methods
 */

ULONG GUIDatabaseStatus::Callback(ULONG ulReportCode,
                                        // in: DBC_* code
                                  ULONG ulExtra)
                                        // in: extra info
{
    ULONG       ulrc = 0;

    // string      str;        // V0.9.18 (2002-03-08) [umoeller]
    string      strPath;
    CHAR        szExistingDate[40] = "",
                szExistingTime[40] = "",
                szDatabaseDate[40] = "",
                szDatabaseTime[40] = "";
    BOOL        fIsErrorMsg = FALSE;

    _strMsg.erase();

    // we cannot do a WinSendMsg, because this would
    // require the Install thread to have a PM message
    // queue.
    // WinPostMsg works though, so we rely on a global
    // variable to get the return code. This is safe
    // since this variable is only set by thread 1
    // after it receives our message.

    // changed V0.9.12 (2001-05-22) [umoeller]: this
    // thread now has a PM message queue, so use
    // WinSendMsg to fix serialization problems on SMP.

    switch (ulReportCode)
    {
        /*
         * DBC_PACKAGEREMOVED:
         *      this results from datRemovePackage.
         *      Note that in this case, this callback does not
         *      necessarily get called from the Database thread,
         *      but from any thread which calls datRemovePackage.
         *      pPckInfo has the package that was just removed
         *      from the database.
         *
         *      ulUser, in this one case, has the FEPackage*
         *      which was removed. (99-11-02) [umoeller]
         */

        case DBC_PACKAGEREMOVED:
        {
            FEDBPackage *pPackageRemoved = (FEDBPackage*)ulExtra;
            // results from datRemovePackage;
            // in this case, we're running on the
            // thread which called that func, which
            // _can_, but does not have to be the
            // Database thread
            if (_hwndDBPackagesDlg)
                WinSendMsg(_hwndDBPackagesDlg,
                           WPIM_PACKAGEREMOVED,
                           (MPARAM)pPackageRemoved,
                           (MPARAM)NULL);
                // do not wait
        }
        break;

        /*
         * DBC_NEXTPACKAGE:
         *      Database thread is progressing to next package.
         *      Status.pPackage has the new package,
         *      Status.pwifh is NULL at this point.
         */

        case DBC_NEXTPACKAGE:
            WinSendMsg(_hwndStatus,
                       WPIM_UPDATEPACKAGE,
                       (MPARAM)this,
                       (MPARAM)0);

            // wait for thread 1 to set semaphore
            // DosWaitEventSem(G_hevGUIDone, SEM_INDEFINITE_WAIT);
        break;

        /*
         * DBC_NEXTFILE:
         *      Database thread is progressing
         *      to next file (useful for progress bars).
         *      This comes in for _every_ file in the
         *      database file list; there might be a second
         *      call with of the following error codes afterwards.
         *      Status.pPackage has the current package,
         *      Status.pwifh has the new WIFileHeader.
         */

        case DBC_NEXTFILE:
            WinSendMsg(_hwndStatus, WPIM_UPDATEFILE,
                      (MPARAM)this,
                      (MPARAM)0);

            // wait for thread 1 to set semaphore
            // DosWaitEventSem(G_hevGUIDone, SEM_INDEFINITE_WAIT);
        break;

        /*
         * DBC_FILEERROR:
         *      the database thread encountered an error
         *      while processing a file. This is what
         *      happens if DosQueryPathInfo or setting file
         *      attributes fails for some reason.
         *      This can happen in both "verify" and "deinstall"
         *      mode.
         *      pPckInfo has the same package info as in the
         *      previous call with DBC_NEXTPACKAGE, pHeader
         *      has the current file (as in the previous DBC_NEXTFILE
         *      call).
         *      ulExtra has the APIRET which occured.
         */

        case DBC_FILEERROR:
        {
            PSZ pszSysError;
            if (pszSysError = doshQuerySysErrorMsg(ulExtra))
            {
                _strMsg._printf(nlsGetString(WPSI_FILEERROR),
                            // "Error with File ""%s"". OS/2 reported: ""%s"""
                            ComposeFullPath(strPath),
                            pszSysError);
                free (pszSysError);
            }
            else
                _strMsg._printf("Error getting error message for OS/2 error %d",
                            ulExtra);

            fIsErrorMsg = TRUE;
        }
        break;

        /*
         * DBC_EXISTINGFILENEWER:
         *      the last write date of a file on
         *      disk is newer than that stored in the database
         *      (verify mode only).
         *      This is probably not serious.
         *      pPckInfo has the same package info as in the
         *      previous call with DBC_NEXTPACKAGE, pHeader
         *      has the current file (as in the previous DBC_NEXTFILE
         *      call).
         *      In this case, ulExtra is pointer to a FILESTATUS3
         *      structure with information about the existing file on disk.
         */

        case DBC_EXISTINGFILENEWER:

        /*
         * DBC_EXISTINGFILEOLDER:
         *      the last write date of a file on
         *      disk is _older_ than that stored in the database
         *      (verify mode only).
         *      This could lead to problems.
         *      pPckInfo, pHeader, and ulExtra are set like with
         *      DBC_EXISTINGFILENEWER.         */

        case DBC_EXISTINGFILEOLDER:
        {
            FILESTATUS3* pfs3 = (FILESTATUS3*)(ulExtra);
            FDATE       fdateLastWrite;
            FTIME       ftimeLastWrite;
            nlsFileDate(szExistingDate,
                        &pfs3->fdateLastWrite,
                        G_ulDateFormat, G_szDateSep[0]);
            nlsFileTime(szExistingTime,
                        &pfs3->ftimeLastWrite,
                        G_ulTimeFormat,
                        G_szTimeSep[0]);
            // convert file header's time information
            time_t lastwrite = _pwifh->lastwrite;
            wpiCTime2FTime(&lastwrite,
                           &fdateLastWrite,
                           &ftimeLastWrite);
            nlsFileDate(szDatabaseDate,
                        &fdateLastWrite,
                        G_ulDateFormat, G_szDateSep[0]);
            nlsFileTime(szDatabaseTime,
                        &ftimeLastWrite,
                        G_ulTimeFormat,
                        G_szTimeSep[0]);
            _strMsg._printf((ulReportCode == DBC_EXISTINGFILENEWER)
                            ? nlsGetString(WPSI_EXISTINGFILENEWER)
                            : nlsGetString(WPSI_EXISTINGFILEOLDER),
                        ComposeFullPath(strPath),
                        szExistingDate,
                        szExistingTime,
                        szDatabaseDate,
                        szDatabaseTime);
        }
        break;

        /*
         * DBC_DELETE_EXISTINGFILENEWER:
         *      file to be deleted is newer than
         *      file stored in database.
         *
         *      ulExtra has GUIFILEERROR pointer.
         *
         *      Must return TRUE if the file is to
         *      be deleted anyway.
         *
         *added V0.9.9 (2001-04-06) [umoeller]
         */

        case DBC_DELETE_EXISTINGFILENEWER:
            ulrc = (ULONG)WinSendMsg(_hwndStatus,
                                     WPIM_DELETE_EXISTINGFILENEWER,
                                     (MPARAM)ulExtra,     // GUIFILEERROR ptr
                                     (MPARAM)0);
        break;

        /*
         * DBC_DELETEERROR:
         *      a file could not be deleted
         *      (de-install mode only).
         *      pPckInfo and pHeader are set like with
         *      DBC_EXISTINGFILENEWER.
         *      In this case, ulExtra is the APIRET of DosDelete.
         */

        case DBC_DELETEERROR:
        {
            PSZ pszSysError = doshQuerySysErrorMsg(ulExtra);
            _strMsg._printf(nlsGetString(WPSI_DELETEERROR),
                            // "File \"%s\" could not be deleted. OS/2 reported: \"%s\"\r"
                        ComposeFullPath(strPath),
                        pszSysError);
            if (pszSysError)
                free (pszSysError);
            fIsErrorMsg = TRUE;
        }
        break;

        /*
         * DBC_DELETEDIRERROR:
         *      a directory could not be removed
         *      (de-install mode only).
         *      In this case, ulExtra is the APIRET of DosDeleteDir.
         */

        /* case DBC_DELETEDIRERROR:
        {
            PSZ pszSysError = doshQuerySysErrorMsg(ulExtra);
            _strMsg._printf(nlsGetString(WPSI_DELETEDIRERROR),
                            // "Directory \"%s\" could not be removed, %d files left
                        _pPackage->QueryTargetPath(),
                        _cFilesInDir);
            if (pszSysError)
                free (pszSysError);
            fIsErrorMsg = TRUE;
        }
        break; */ // removed V0.9.19 (2002-04-14) [umoeller]

        /*
         * DBC_FILELOCKED:
         *      called when a locked file was encountered
         *      during delete.
         *
         *      Must return TRUE if the file is to be
         *      unlocked.
         */

        case DBC_FILELOCKED:
        {
            ulrc = (ULONG)WinSendMsg(_hwndStatus,
                                     WPIM_FILELOCKEDERROR,
                                     (MPARAM)ComposeFullPath(strPath),
                                     0);
        }
        break;
    }

    if (_strMsg())
    {
        // something string-like was added for posting:
        SendString(fIsErrorMsg);
    }

    return (ulrc);
}

#define CONVERT(s) \
    string str ## s(((GUILocals *)&_Locals)->_pCodecGui, pCfg->_ustr ## s); \
    PCSZ pcsz ## s = str ## s.c_str()

/*
 *@@ OnObjectDeleted:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void GUIDatabaseStatus::OnObjectDeleted(BSDeleteWPSObject *pCfg,
                                        BOOL fSuccess)
{
    BOOL    fIsErrorMsg = FALSE;

    CONVERT(ObjectID);

    if (fSuccess)
    {
        _strMsg._printf(nlsGetString(WPSI_OBJECTDELETED), // "Deleted object \"%s\"\n",
                        pcszObjectID);
    }
    else
    {
        _strMsg._printf(nlsGetString(WPSI_ERRORDELETINGOBJECT),
                        pcszObjectID);
        fIsErrorMsg = TRUE;
    }

    SendString(fIsErrorMsg);
}

/*
 *@@ OnClassUnreplaced:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void GUIDatabaseStatus::OnClassUnreplaced(BSUnreplaceClass *pCfg,
                                          BOOL fSuccess)
{
    GUILocals *pLocals = (GUILocals *)&_Locals;

    CONVERT(OldClassName);
    CONVERT(NewClassName);

    if (fSuccess)
    {
        _strMsg._printf(nlsGetString(WPSI_CLASSUNREPLACED),
                        // "Undid replacement of WPS class ""%s"" with ""%s"""
                        pcszOldClassName,
                        pcszNewClassName);
    }
    else
    {
        _strMsg._printf(nlsGetString(WPSI_ERRORUNREPLACINGCLASS),
                        // "Error undoing replacement of WPS class ""%s"" with ""%s"""
                        pcszOldClassName,
                        pcszNewClassName);
    }

    SendString(fSuccess);
}

/*
 *@@ OnClassDeregistered:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void GUIDatabaseStatus::OnClassDeregistered(BSDeregisterClass *pCfg,
                                        BOOL fSuccess)
{
    CONVERT(ClassName);

    if (fSuccess)
    {
        _strMsg._printf(nlsGetString(WPSI_CLASSDEREGISTERED), // "Deregistered class \"%s\"\n",
                        pcszClassName);
    }
    else
    {
        _strMsg._printf(nlsGetString(WPSI_ERRORDEREGISTERINGCLASS),
                        pcszClassName);
    }

    SendString(fSuccess);
}

/*
 *@@ OnCfgSysUndone:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void GUIDatabaseStatus::OnCfgSysUndone(BSCfgSysManip *pCfg,
                                        BOOL fSuccess)
{
    CONVERT(NewLine);

    if (fSuccess)
    {
        _strMsg._printf(nlsGetString(WPSI_CFGSYSUNDONE),
                        // "CONFIG.SYS change undone (line: ""%s"")"
                        pcszNewLine);
    }
    else
    {
        _strMsg._printf(nlsGetString(WPSI_ERRORUNDOINGCFGSYS),
                        // "Error undoing CONFIG.SYS change (line: ""%s"")"
                        pcszNewLine);
    }

    SendString(fSuccess);
}

/*
 *@@ OnProfileCleared:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void GUIDatabaseStatus::OnProfileCleared(BSClearProfile *pCfg,
                                         BOOL fSuccess)
{
    string strKey;
    strKey = pCfg->DescribePrfKey(*((GUILocals *)&_Locals)->_pCodecGui);

    if (fSuccess)
    {
        _strMsg._printf(nlsGetString(WPSI_PROFILECLEARED),
                        strKey.c_str());
    }
    else
    {
        _strMsg._printf(nlsGetString(WPSI_ERRORCLEARINGPROFILE),
                        strKey.c_str());
    }

    SendString(fSuccess);
}

/*
 *@@ OnDeExecute:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *      iCode is
 *
 *      --  -1: info msg that deexecute is about to
 *          start.
 *
 *      --  0: deexecute was successful.
 *
 *      --  any other: OS/2 error code when deexecute
 *          failed.
 *
 *      This always comes in twice.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void GUIDatabaseStatus::OnDeExecute(BSDeExecute *pCfg,
                                    int iCode)
{
    CONVERT(Executable);
    CONVERT(Params);

    switch (iCode)
    {
        case -1:
            _strMsg._printf(nlsGetString(WPSI_DEEXECUTING),
                            pcszExecutable,
                            pcszParams);
        break;

        case 0:
            _strMsg = nlsGetString(WPSI_DEEXECUTEDONE);
        break;

        default:
            // error:
            _strMsg._printf(nlsGetString(WPSI_DEEXECUTEERROR),
                            pcszExecutable,
                            pcszParams);
        break;
    }

    SendString((iCode <= 0));       // success
}

/*
 *@@ OnCannotDeleteDirError:
 *      required GUI implementation of the respective
 *      pure virtual FEDatabaseStatus method.
 *
 *      Comes in when WarpIN cannot delete a directory
 *      on deinstall.
 *
 *      The directory in question is _pPackage->QueryTargetPath().
 *
 *@@added V0.9.19 (2002-04-14) [umoeller]
 *@@changed V0.9.20 (2002-07-22) [umoeller]: changed prototype
 */

void GUIDatabaseStatus::OnCannotDeleteDirError(const ustring &ustrDirectory,
                                               ULONG cDirsInDir,
                                               ULONG cFilesInDir)
{
     GUILocals *pLocals = (GUILocals*)&_Locals;
     string strDir(pLocals->_pCodecGui, ustrDirectory);
     _strMsg._printf(nlsGetString(WPSI_DELETEDIRERROR),
                     // "Directory ""%s"" could not be removed because there are %d file(s) left in it. Remove it manually."
                     strDir.c_str(),
                     cFilesInDir);

     SendString(FALSE);      // error
}

