
/*
 *@@sourcefile fe_package.cpp:
 *      this implements the package classes (FEPackageBase
 *      and subclasses plus the FEPackageID helper class).
 *
 *      Before V0.9.0, 99-10-31, we only had a single "PackageInfo" class.
 *      Now this is much more complicated, as C++ code should be. ;-)
 *
 *@@header "engine\fe_package.h"
 */

/*
 *      This file Copyright (C) 1999-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#include <os2.h>

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"

// include's from helpers
#include "helpers\nls.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"

#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_package_db.h"

#pragma hdrstop

DEFINE_CLASS(FEPackageID, BSRoot);
DEFINE_CLASS(FEPackageBase, BSRoot);

/* ******************************************************************
 *
 *  PackageID class
 *
 ********************************************************************/

/*
 *@@ Init:
 *      protected helper method used by the constructors.
 *
 *      Returns 0 on success, an error message number otherwise.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 *@@changed V0.9.18 (2002-03-08) [umoeller]: now handling GUI codepage conversions
 *@@changed V0.9.20 (2002-07-03) [umoeller]: rewrote version scanning to finally handle fixlevels
 */

ULONG FEPackageID::Init(FELocals &Locals,
                        const ustring &ustrPackageID,       // in: utf-8 ID string
                        BOOL fAllowShort)
{
    ULONG   ulMsg = 0;

    PSZ pszID;

    if (pszID = strhdup(ustrPackageID.GetBuffer(), NULL))
    {
        PSZ     p,
                p2;

        BOOL    fAllOK = FALSE;

        // get author
        if (p = strchr(pszID, '\\'))
        {
            // p points to after author now:
            _ustrAuthor.assignUtf8(pszID, p);

            // get application
            if (p2 = strchr(p + 1, '\\'))
            {
                // p points to after application now:
                _ustrApplication.assignUtf8(p + 1, p2);

                // four ULONGs for the versions
                // V0.9.20 (2002-07-03) [umoeller]
                ULONG aul[4] = {0};
                ULONG c = 0;

                // get package name
                p = p2;
                if (!(p2 = strchr(p + 1, '\\')))
                {
                    // '\' after package not found:
                    // this is only valid in "allow short" mode
                    if (fAllowShort)
                        _ustrPackageName.assignUtf8(p + 1);
                }
                else
                {
                    // p points to after package now:
                    _ustrPackageName.assignUtf8(p + 1, p2);

                    // now go for the version numbers...
                    // V0.9.20 (2002-07-03) [umoeller]

                    do
                    {
                        // author\app\package\major\minor\rev\fix
                        //                   ^ p2

                        p = p2;

                        // author\app\package\major\minor\rev\fix
                        //                   ^ p
                        //                         ^ p2

                        if (p2 = strchr(p + 1, '\\'))
                            *p2 = '\0';

                        aul[c++] = atoi(p + 1);

                    } while (p2);
                }

                if (_ustrPackageName())
                {
                    if (    (c >= 2)        // we must at least have a minor version
                         || (fAllowShort)
                       )
                    {
                        // all OK: store in instance data
                        _version._ulMajor = aul[0];
                        _version._ulMinor = aul[1];
                        _version._ulRevision = aul[2];
                        _version._ulFixlevel = aul[3];

                        // make copies with GUI codepage for speed
                        // so that we don't have to convert all the
                        // time in the PM containers
                        BSUniCodec *codec = &Locals.QueryGuiCodec();
                        _strAuthorGui.assignUtf8(codec, _ustrAuthor);
                        _strApplicationGui.assignUtf8(codec, _ustrApplication);
                        _strPackageNameGui.assignUtf8(codec, _ustrPackageName);

                        _strVersion._printf("%u.%u.%u.%u",
                                            _version._ulMajor,
                                            _version._ulMinor,
                                            _version._ulRevision,
                                            _version._ulFixlevel);

                        return 0;     // success
                    }
                    else
                        ulMsg = 158;        // invalid minor version

                } // end valid package name
                else
                    ulMsg = 160;            // invalid package name

            }
            else
                ulMsg = 161;  // Invalid application name

        }
        else
            ulMsg = 162; // Invalid author name

        free(pszID);
    }
    else
        ulMsg = 163;  // package ID string is NULL

    return ulMsg;
}

/*
 *@@ PackageID:
 *      this constructor splits the five- or six-part package ID
 *      into the six member buffers.
 *
 *      The package ID must have the following format:
 +          author\application\packagename\majorversion\minorversion\revision
 *
 *      If "revision" is not specified (that is, the ID string
 *      only contains four backslashes), the ulVersionRevision
 *      member is set to 0.
 *
 *      This constructor throws a FEConstructorExcpt upon
 *      parsing errors, which gets caught in main() only. You
 *      should therefore use this constructor in a try-catch
 *      block yourself.
 *
 *      If (fAllowShort == FALSE), this throws an exception even
 *      if the version numbers are not specified.
 *      If (fAllowShort == TRUE), this throws an exception only
 *      if the author\application\packagename part is broken and
 *      thus accepts a package ID without version numbers.
 *
 *      Throws:
 *      -- FEConstructorExcpt.
 *
 *@@changed V0.9.0 (99-10-26) [umoeller]: made PackageID a separate class and this the default constructor
 *@@changed V0.9.1 (2000-01-03) [umoeller]: version revision was scanned wrong; fixed
 *@@changed V0.9.1 (2000-01-03) [umoeller]: now having safe defaults for all members
 *@@changed V0.9.3 (2000-06-05) [umoeller]: added fAllowShort
 *@@changed V0.9.18 (2002-03-08) [umoeller]: now only accepting ustring as input
 */

FEPackageID::FEPackageID(FELocals &Locals,
                         const ustring &ustrPackageID,
                         const char *pszParentSource,   // in: caller's file (__FILE__)
                         int iParentLine,               // in: caller's line (__LINE__)
                         BOOL fAllowShort)              // in: allow short specification?
    : BSRoot(tFEPackageID)
{
    ULONG ulMsg;
    if (ulMsg = Init(Locals, ustrPackageID, fAllowShort))
        throw FEConstructorExcpt(Locals,
                                 "PackageID",
                                 pszParentSource,
                                 iParentLine,
                                 ulMsg);
}

/*
 *@@ Compare:
 *      this compares the member ID with that of the
 *      specified package ID.
 *
 *      This returns a ULONG bit field as follows:
 *
 *      If IDCOMP_SAMEPCK (0x10000)is set, this means
 *      that the author, application, and package-name
 *      fields are equal.
 *
 *      In that case, one of the following bits is
 *      also set:
 *
 *      -- IDCOMP_SAMEVERSION (0x00001): both version numbers
 *                  are exactly the same.
 *      -- IDCOMP_THISNEWER (0x00002): "this" has a higher
 *                  version number than "id".
 *      -- IDCOMP_THISOLDER (0x00004): "this" has a lower
 *                  version number than "id".
 *
 *      If IDCOMP_SAMEPCK is not set, no other bits will
 *      be set (i.e. 0 will always be returned).
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 */

ULONG FEPackageID::Compare(const FEPackageID &id)
                   const
{
    ULONG ulrc = 0;
    if (    (_ustrAuthor == id._ustrAuthor)
         && (_ustrApplication == id._ustrApplication)
         && (_ustrPackageName == id._ustrPackageName)
       )
    {
        // author, application, and package name match:
        ulrc = IDCOMP_SAMEPCK | _version.Compare(id._version);
    }

    return ulrc;
}

/*
 *@@ CreateStringSix:
 *      creates a six-part package ID in the specified
 *      string, which is cleared first.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

VOID FEPackageID::CreateStringSix(ustring &ustr)
                  const
{
    ustr._printf("%s\\%s\\%s\\%u\\%u\\%u\\%u",
                 _ustrAuthor.GetBuffer(),
                 _ustrApplication.GetBuffer(),
                 _ustrPackageName.GetBuffer(),
                 _version._ulMajor,
                 _version._ulMinor,
                 _version._ulRevision,
                 _version._ulFixlevel);
}

/* ******************************************************************
 *
 *  FEPackageBase implementation
 *
 ********************************************************************/

/*
 *@@ FEPackageBase:
 *      protected default constructor.
 *
 *      This is only to be used by descendants.
 *
 *@@changed V0.9.12 (2001-05-17) [umoeller]: removed declaration parameter
 */

FEPackageBase::FEPackageBase(FEPckDeclBase *pDecl,
                             BSClassID &Class)
    : BSRoot(Class),
      _RequiresIDs(STORE),
      _pDecl(pDecl)

{
    memset(&_PackHeader, 0, sizeof(_PackHeader));

    _pszFilesList = 0;
    _cbFilesList = 0;
}

/*
 *@@ ~FEPackageBase:
 *      destructor.
 */

FEPackageBase::~FEPackageBase()
{
    if (_pszFilesList)
        free(_pszFilesList);
}

/*
 *@@ QueryTargetPath:
 *      returns the target path of the package as a const C string.
 *
 *      This is either the original target path as specified
 *      in the script (if this is an archive package) or the
 *      target path as stored in the database (for database
 *      packages).
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 *@@changed V0.9.12 (2001-05-17) [umoeller]: adjusted for _pDecl move to subclasses
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed, this must return a ustring
 */

const ustring& FEPackageBase::QueryTargetPath() const
{
    return _pDecl->_ustrTargetPath;
}

/*
 *@@ QueryPackageID:
 *      returns the package's ID as a const C string.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

const ustring& FEPackageBase::QueryPackageID() const
{
    return _pDecl->_ustrID;
}

/*
 *@@ QueryTitle:
 *      returns the package's title as a const C string.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

const ustring& FEPackageBase::QueryTitle() const
{
    return _pDecl->_ustrTitle;
}

/*
 *@@ AppendRequiresID:
 *      appends a new package to the private list of
 *      requirements.
 *
 *      strID must be a five- or six-part package ID.
 *
 *@@added V0.9.18 (2002-02-06) [umoeller]
 */

VOID FEPackageBase::AppendRequiresID(FELocals &Locals,
                                     const ustring &ustrID)
{
    FEPackageID *pID;
    if (pID = new FEPackageID(Locals,
                              ustrID,
                              __FILE__, __LINE__,
                              FALSE))       // do not allow short format
    {
        // store in logger, as previously, so we dump this
        // into the database
        _logRequiresIDs.Append(ustrID);
        // and create a package ID from this!
        _RequiresIDs.push_back(pID);
    }
}

