
/*
 *@@sourcefile fe_job.cpp:
 *      this implements the job classes, based on FEJobBase,
 *      within the WarpIN installer engine (FEInstallEngine).
 *
 *      See @jobs for details.
 *
 *@@added V0.9.0 (99-11-01) [umoeller]
 *@@header "engine\fe_job.h"
 */

/*
 *      This file Copyright (C) 1999-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"

#include "bldlevel.h"

#include "helpers\nls.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"

#include "engine\fe_archive.h"
#include "engine\fe_job.h"
#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_package_db.h"
#include "engine\fe_database.h"

#pragma hdrstop

DEFINE_CLASS(FEJobBase, BSRoot);
DEFINE_CLASS(FEInstallJob, FEJobBase);
DEFINE_CLASS(FEGroupJob, FEJobBase);

/* ******************************************************************
 *
 *  FEJobBase implementation
 *
 ********************************************************************/

/*
 *@@ FEJobBase:
 *      protected default constructor.
 *
 *@@changed V0.9.20 (2002-07-03) [umoeller]: removed onJobSelectionChanged callback, added locals
 */

FEJobBase::FEJobBase(FEGroupJob *pGroupJob_,
                     FELocals &Locals,
                     BSClassID &Class)
    : BSRoot(Class),
      _Locals(Locals)
{
    _Selection = JOB_UNDEFINED;
    _pGroupJob = pGroupJob_;
    if (_pGroupJob)
        // we're part of a group:
        // store ourselves in the group's member list
        _pGroupJob->_GroupMembersList.push_back(this);
    _pvGuiData = 0;
}

/*
 *@@ IsInstallJob:
 *      virtual method overridden by FEInstallJob only.
 *      The default returns NULL.
 *
 *      This can be used to quickly check if a given
 *      FEJobBase instance is actually of the FEInstallJob
 *      subclass (and not FEGroupJob) and to typecast the
 *      pointer at the same time if it is.
 */

FEInstallJob* FEJobBase::IsInstallJob()
{
    return NULL;
}

/*
 *@@ IsGroupJob:
 *      virtual method overridden by FEGroupJob only.
 *      The default returns NULL.
 *
 *      This can be used to quickly check if a given
 *      FEJobBase instance is actually of the FEGroupJob
 *      subclass (and not FEInstallJob) and to typecast
 *      the pointer at the same time if it is.
 */

FEGroupJob* FEJobBase::IsGroupJob()
{
    return NULL;
}

/*
 *@@ QueryJobID:
 *      returns the current job status, which is one
 *      of:
 *
 *      -- JOB_INSTALL_IGNORE: well, do nothing
 *      -- JOB_INSTALL: install package
 *      -- JOB_DEINSTALL: remove package
 *
 *      If this job is an FEGroupJob, in addition:
 *
 *      -- JOB_GROUP_MIXED: sub-packages have
 *         varying flags.
 */

enJobSelection FEJobBase::QuerySelection() const
{
    return _Selection;
}

/* ******************************************************************
 *
 *  FEInstallJob implementation
 *
 ********************************************************************/

/*
 *@@ FEInstallJob:
 *      constructor to create an install job from a
 *      given package. This is used while the front
 *      end sets up the packages from the archive.
 *
 *      All the constructor arguments are required.
 *
 *      Note: To give the ulInstall* fields a meaningful
 *      value, you must explicitly call FEInstallJob::CheckInstall.
 *      This cannot be done by the constructor because while
 *      all the jobs are being created, we know little about
 *      the database yet.
 *
 *@@changed V0.9.14 (2001-07-24) [umoeller]: added callback param
 *@@changed V0.9.20 (2002-07-03) [umoeller]: removed onJobSelectionChanged callback, added locals
 */

FEInstallJob::FEInstallJob(FEArcPackagePck &Package,  // in: new member package
                           FEGroupJob *pGroupJob,      // in: the group we belong to or NULL
                           FELocals &Locals)
              : FEJobBase(pGroupJob,
                          Locals,
                          tFEInstallJob),
                _ArcPackage(Package)
{
    _Selection = JOB_INSTALL_IGNORE;

    _pDBPackageInstalled = NULL;

    _ulInstallStatus = INSTALLED_UNKNOWN;
}

/*
 *@@ FEInstallJob:
 *      destructor.
 */

FEInstallJob::~FEInstallJob()
{
    /* if (_pPackageID)
        delete _pPackageID;
    if (_pPackageIDInstalled)
        delete _pPackageIDInstalled; */
}

/*
 *@@ IsInstallJob:
 *      since we're of the FEInstallJob class,
 *      we return "this" instead of the base class's NULL.
 */

FEInstallJob* FEInstallJob::IsInstallJob()
{
    return this;
}

/*
 *@@ CheckInstall:
 *      this queries the given database for whether
 *      the package of this job is already installed.
 *
 *      After this call, the ulInstall* fields have
 *      a meaningful value.
 *
 *      This returns TRUE if the package is already installed.
 *      In that case, version is set to the version number
 *      of the package that is installed.
 *
 *      FEInstallEngine::AppendArchive calls this for every
 *      install job in an archive.
 *
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added version
 */

BOOL FEInstallJob::CheckInstall(FEDatabase *pDatabase,
                                WIVersion &version)
{
    BOOL brc = FALSE;

    _ulInstallStatus = INSTALLED_NO;

    if (_pDBPackageInstalled = pDatabase->FindDBPackage(_ArcPackage._PckID))
    {
        // package found:

        // compare versions
        ULONG ul = _pDBPackageInstalled->_PckID.Compare(_ArcPackage._PckID);

        if (ul & IDCOMP_SAMEVERSION)
            _ulInstallStatus = INSTALLED_IS_SAME;
        else if (ul & IDCOMP_THISNEWER)
            // _pDBPackageInstalled is newer than pArcPackage:
            _ulInstallStatus = INSTALLED_IS_NEWER;
        else if (ul & IDCOMP_THISOLDER)
            // _pDBPackageInstalled is older than pArcPackage:
            _ulInstallStatus = INSTALLED_IS_OLDER;

        version = _pDBPackageInstalled->_PckID._version;

        // take over the list of directories that were created
        // in the previous install; otherwise the directory
        // will not be removed when the updated version is
        // uninstalled
        _ArcPackage._logDirsCreated = _pDBPackageInstalled->_logDirsCreated;

        brc = TRUE;
    }

    return brc;
}

/*
 *@@ QueryTargetPath:
 *      returns a const pointer to the target path
 *      member. This is in the current process
 *      codepage.
 *
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed return, this must be a ustring!
 */

const ustring& FEInstallJob::QueryTargetPath()
               const
{
    return _ustrTargetPath;
}

/*
 *@@ QueryPckPackageID:
 *      returns the package ID of the
 *      install job's archive package.
 *
 *      This has only been added so that this
 *      can be queried without having to include
 *      all the package headers.
 *
 *@@added V0.9.9 (2001-03-30) [umoeller]
 */

const ustring& FEInstallJob::QueryPckPackageID()
               const
{
    return _ArcPackage.QueryPackageID();
}

/*
 *@@ QueryPckSizeRounded:
 *      returns the _ulSizeRounded member of the
 *      install job's archive package.
 *
 *      This has only been added so that this
 *      can be queried without having to include
 *      all the package headers.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

ULONG FEInstallJob::QueryPckSizeRounded()
      const
{
    return _ArcPackage._ulSizeRounded;
}

/*
 *@@ QueryPckFilesCount:
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

ULONG FEInstallJob::QueryPckFilesCount()
            const
{
    return _ArcPackage._PackHeader.files;
}

/*
 *@@ QueryPckOrigSize:
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

ULONG FEInstallJob::QueryPckOrigSize()
            const
{
    return _ArcPackage._PackHeader.origsize;
}


/*
 *@@ QueryPckTitle:
 *      invokes FEPackageBase::QueryTitle on the
 *      install job's archive package.
 *
 *      This has only been added so that this
 *      can be queried without having to include
 *      all the package headers.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

const ustring& FEInstallJob::QueryPckTitle()
            const
{
    return _ArcPackage.QueryTitle();
}

/*
 *@@ SetTargetPath:
 *      sets the new target path for the package
 *      and updates other packages' target paths
 *      if BASE attributes have been set for them.
 *
 *      Returns TRUE if the drive letter has changed.
 *      The caller should update the drives dialog then.
 *
 *@@changed V0.9.7 (2001-01-07) [umoeller]: fixed BASE handling not to update paths of installed packages
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed input, this must be a ustring
 *@@changed V0.9.20 (2002-07-03) [umoeller]: made jobs list an optional pointer
 */

BOOL FEInstallJob::SetTargetPath(const ustring &ustrNewTargetPath,
                                        // in: new target path
                                 list<FEJobBase*> *pJobsList)
                                        // in: list of all install jobs or NULL; this is processed
                                        // if "this" has the BASE attribute
{
    BOOL    brc = FALSE;

    // if our member package is a base package
    // and the caller has given us a list,
    // update all other packages
    if (    (_ArcPackage._pDecl->_ulPathType & PATH_BASE)
         && (pJobsList)
       )
    {
        ustring strSearch(_ustrTargetPath);
        list<FEJobBase*>::iterator JobStart = pJobsList->begin(),
                                   JobEnd = pJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThat = pTemp->IsInstallJob();
            if (    (pJobThat)
                 && (pJobThat != this)
                 && (pJobThat->_ulInstallStatus == INSTALLED_NO)
                        // only if package is not installed yet...
                        // because otherwise it already has the target
                        // path of the installed package, which we must
                        // not change!!
               )
            {
                ustring strTargetPathThat(pJobThat->QueryTargetPath());
                size_type ulOfs = 0;
                strTargetPathThat._find_replace(strSearch,
                                                   // search string: our old target path
                                                ustrNewTargetPath,
                                                   // replace with new path
                                                &ulOfs);
                pJobThat->_ustrTargetPath = strTargetPathThat;
            }
        }
    }

    if (_ustrTargetPath[0] != ustrNewTargetPath[0])
        // drive letter changed:
        // update drive info, because the drive might
        // have changed
        brc = TRUE;

    _ustrTargetPath = ustrNewTargetPath;

    return brc;
}

/*
 *@@ DefaultSelect:
 *      this selects the job for installation according
 *      to the default selection which was specified in
 *      the script. A package is however not selected
 *      if a newer version is already installed.
 *
 *      This calls FEInstallJob::Select in turn.
 *
 *      Returns FALSE if CheckInstall hasn't been
 *      called yet.
 *
 *@@added V0.9.0 (99-11-06) [umoeller]
 *@@changed V0.9.12 (2001-05-31) [umoeller]: fixed NODESELECT
 */

BOOL FEInstallJob::DefaultSelect()
{
    if (    (    (_ArcPackage._pDecl->_Selected == FEPckDeclBase::SELECTED)
              || (_ArcPackage._pDecl->_Selected == FEPckDeclBase::NODESELECT)
                    // NODESELECT was missing V0.9.12 (2001-05-31) [umoeller]
            )
         && (_ulInstallStatus != INSTALLED_IS_NEWER)
       )
        Select(JOB_INSTALL);
    else
        Select(JOB_INSTALL_IGNORE);

    return TRUE;
}

/*
 *@@ RepairSelect:
 *      this compares the job's package version with
 *      the installed version, if any, and selects
 *      the package for installation if it's not
 *      yet installed, older, or same version --
 *      but not if the installed version is newer.
 *
 *      This calls FEInstallJob::Select in turn.
 *
 *      Returns FALSE if CheckInstall hasn't been
 *      called yet.
 *
 *@@added V0.9.0 (99-11-06) [umoeller]
 */

BOOL FEInstallJob::RepairSelect()
{
    switch (_ulInstallStatus)
    {
        case INSTALLED_UNKNOWN:
            return FALSE;

        case INSTALLED_NO:
            // not installed:
        case INSTALLED_IS_NEWER:
            // newer package installed:
            // deselect package per default, we don't want
            // to overwrite with outdated version
            Select(JOB_INSTALL_IGNORE);
        break;

        case INSTALLED_IS_OLDER:
        case INSTALLED_IS_SAME:
            Select(JOB_INSTALL);
        break;
    }

    return TRUE;
}

/*
 *@@ UpdateSelect:
 *      this compares the job's package version with
 *      the installed version, if any, and selects
 *      the package only if the one in the archive
 *      is newer than the installed version.
 *
 *      This calls FEInstallJob::Select in turn.
 *
 *      Returns FALSE if CheckInstall hasn't been
 *      called yet.
 *
 *@@added V0.9.0 (99-11-06) [umoeller]
 */

BOOL FEInstallJob::UpdateSelect()
{
    switch (_ulInstallStatus)
    {
        case INSTALLED_UNKNOWN:
            return FALSE;

        case INSTALLED_IS_OLDER:
            Select(JOB_INSTALL);
        break;

        default:
            Select(JOB_INSTALL_IGNORE);
    }

    return TRUE;
}

/*
 *@@ Select:
 *      this virtual FEJobBase method is overridden to
 *      set the job status for this job.
 *
 *      ulJobNew can be one of the following:
 *
 *      -- JOB_INSTALL:   select the member package for installation.
 *      -- JOB_DEINSTALL:    select the member package for removal (de-install).
 *                  If this is specified and the package isn't installed,
 *                  JOB_INSTALL_IGNORE is chosen instead.
 *      -- JOB_INSTALL_IGNORE: do nothing. If the package is not yet installed,
 *                  it will not be installed. If the package is
 *                  already installed, it will not be deinstalled.
 *
 *      This calls guiJobSelectionChanged if anything has changed
 *      and automatically updates the group to which this job
 *      belongs to, if any.
 *
 *@@changed V0.9.1 (2000-01-05) [umoeller]: it was impossible to deinstall "no-deselect" packages; fixed
 *@@changed V0.9.1 (2000-01-08) [umoeller]: renamed from "SetJobID"
 */

BOOL FEInstallJob::Select(enJobSelection selNew)
{
    BOOL brc = TRUE;

    if (_Selection != selNew)
    {
        // prohibit ignore if package must be installed
        if (    (selNew == JOB_INSTALL_IGNORE)
             && (_ArcPackage._pDecl->_Selected == FEPckDeclBase::NODESELECT)
           )
            selNew = JOB_INSTALL;
        // prohibit deinstall?
        else if (selNew == JOB_DEINSTALL)
        {
            // prohibit deinstall if package is not installed
            if (_ulInstallStatus == INSTALLED_NO)
                selNew = JOB_INSTALL_IGNORE;
        }

        _Selection = selNew;

        _Locals.OnJobSelectionChanged(this);
                // V0.9.20 (2002-07-03) [umoeller]

        if (_pGroupJob)
        {
            _pGroupJob->Update();
        }
    }

    return brc;
}

/*
 *@@ ToggleSelection:
 *      similar to FEInstallJob::Select, but this toggles
 *      the selection.
 *
 *      If the member package is already installed, this
 *      circles thru the three job statuses. Otherwise,
 *      "deinstall" is not used, of course.
 *
 *      This is a virtual method defined abstractly by
 *      FEJobBase. This way, packages can toggle differently
 *      from groups.
 *
 *      This returns the new selection status.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: PckInfoList param removed
 */

enJobSelection FEInstallJob::ToggleSelection()
{
    switch (_Selection)
    {
        case JOB_INSTALL_IGNORE:
            Select(JOB_INSTALL);
        break;

        case JOB_INSTALL:
            if (    (_ulInstallStatus != INSTALLED_UNKNOWN)
                 && (_ulInstallStatus != INSTALLED_NO)
               )
                // package already installed: deinstall then
                Select(JOB_DEINSTALL);
            else
                Select(JOB_INSTALL_IGNORE);
        break;

        case JOB_DEINSTALL:
            Select(JOB_INSTALL_IGNORE);
        break;
    }

    return _Selection;
}

/*
 *@@ Is2BeInstalled:
 *      returns TRUE if the job represents a
 *      package (not a group) and has been
 *      selected for installation.
 *
 *@@changed V0.9.18 (2002-02-06) [umoeller]: rewritten
 */

BOOL FEInstallJob::Is2BeInstalled() const
{
    if (_Selection == JOB_INSTALL)
        return _ArcPackage.IsA(FEArcPackagePck::tFEArcPackagePck);
        // if (_ArcPackage.IsPackage())
        //     brc = TRUE;

    return FALSE;
}

/*
 *@@ Is2BeRemoved:
 *      returns TRUE if the job represents a
 *      package (not a group) and has been
 *      selected for de-installation.
 *
 *@@changed V0.9.18 (2002-02-06) [umoeller]: rewritten
 */

BOOL FEInstallJob::Is2BeRemoved() const
{
    if (_Selection == JOB_DEINSTALL)
        return _ArcPackage.IsA(FEArcPackagePck::tFEArcPackagePck);
        // if (_ArcPackage.IsPackage())
        //     brc = TRUE;

    return FALSE;
}

/*
 *@@ Unpack:
 *      simple wrapper around FEArcPackagePck::Unpack,
 *      which in turn calls WIArchive::unpack. This
 *      unpacks all files in the member package and
 *      does not return until all files have been unpacked
 *      or a fatal error occured.
 *
 *      That method is called on the archive to which
 *      the member package belongs, with the package
 *      index of the member package.
 *
 *      This produces tons of calls to the callback that
 *      was specified with the FEArcPackagePck constructor.
 *
 *      Gets called on the WarpIN install thread for each
 *      package that is to be installed.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

VOID FEInstallJob::Unpack(PFNWICALLBACK pWICallback,
                          void *pvUser)
{
    _ArcPackage.Unpack(pWICallback,
                       pvUser);
}

/*
 *@@ AddFileHeader:
 *      simple wrapper around FEArcPackagePck::AddFileHeader.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

VOID FEInstallJob::AddFileHeader(WIFileHeader* pwifh)
{
    _ArcPackage.AddFileHeader(pwifh);
}

/*
 *@@ Store:
 *      this stores the member package in the given database.
 *
 *      This gets called from warpin.cpp after installation
 *      was complete and successful to store the package
 *      information in the global database.
 *
 *      We store the whole package data in the database then,
 *      using the package ID stored in _pszID.
 *      This better be valid.
 *
 *      If this package is already stored in the database,
 *      even if it's an older version, the old data is
 *      overwritten without further notice.
 *
 *      The format is so that the package info can later
 *      be recreated as an instance of FEDBPackage simply
 *      by passing the package ID to the FEDBPackage::FEDBPackage
 *      constructor, which in turn calls FEDBPackage::Load.
 *
 *      The data stored here must match exactly the data which
 *      is restored in FEDBPackage::Load.
 *
 *      This returns FALSE upon errors.
 *
 *@@changed V0.9.0 (99-10-26) [umoeller]: six-part ID support added
 *@@changed V0.9.0 (99-11-07) [umoeller]: added install date/time
 *@@changed V0.9.1 (2000-01-07) [umoeller]: changed "requires" storage
 *@@changed V0.9.1 (2000-02-03) [umoeller]: added "execute" loggers support
 *@@changed V0.9.2 (2000-03-11) [umoeller]: added description and codepage
 *@@changed V0.9.20 (2002-07-22) [umoeller]: added "installed using"
 */

BOOL FEInstallJob::Store(FEDatabase &Database)
{
    BOOL    brc = FALSE;

    // normalize package ID by creating an FEPackageID
    // and re-parsing that
    FEPackageID &pckid = _ArcPackage._PckID;
    ustring strIDSix;
    pckid.CreateStringSix(strIDSix);
    const char *pcszIDSix = strIDSix.GetBuffer();

    // check if another version of this package has
    // been stored already
    const FEDBPackage *pPckInstalled;
    if (pPckInstalled = Database.FindDBPackage(pckid))
        // yes: remove that first
        Database.RemovePackage(pPckInstalled);

    HINI hiniDB = Database._hiniDatabase;

    // moved all the package-specific stuff to FEArcPackagePck
    // V0.9.18 (2002-02-06) [umoeller]
    _ArcPackage.Store(hiniDB,
                      pcszIDSix);

    PrfWriteProfileString(hiniDB,
                          pcszIDSix,
                          WPIKEY_TARGETPATH,
                          _ustrTargetPath.GetBuffer());

    // added "installed using"
    // V0.9.20 (2002-07-22) [umoeller]
    PrfWriteProfileString(hiniDB,
                          pcszIDSix,
                          WPIKEY_INSTALLEDUSING,
                          BLDLEVEL_VERSION);

    // store current date/time as install date/time
    DATETIME dt;
    DosGetDateTime(&dt);
    PrfWriteProfileData(hiniDB,
                        pcszIDSix,
                        WPIKEY_INSTALLDATETIME,
                        &dt,
                        sizeof(DATETIME));

    return brc;
}

/* ******************************************************************
 *
 *  FEGroupJob implementation
 *
 ********************************************************************/

/*
 *@@ FEGroupJob:
 *
 *@@changed V0.9.20 (2002-07-03) [umoeller]: removed onJobSelectionChanged callback, added locals
 */

FEGroupJob::FEGroupJob(FEArcPackageGroup *pGroup_, // in: group package (required)
                       FEGroupJob *pGroupJob,      // in: our own group (or NULL)
                       FELocals &Locals)
            : FEJobBase(pGroupJob,
                        Locals,
                        tFEGroupJob),
              _GroupMembersList(SHADOW) // STORE)
                            // fixed V0.9.15 (2001-08-26) [umoeller]:
                            // list must be in SHADOW mode
{
    _pGroupPackage = pGroup_;
}

/*
 *@@ IsGroupJob:
 *      since we're of the FEGroupJob class,
 *      we return "this" instead of the base class's NULL.
 */

FEGroupJob* FEGroupJob::IsGroupJob()
{
    return this;
}

/*
 *@@ ToggleSelection:
 *      this toggles the selection for the group and all
 *      its members.
 *
 *      This is a virtual method defined abstractly by
 *      FEJobBase. This way, packages can toggle differently
 *      from groups.
 *
 *      This returns the new selection status.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: PckInfoList param removed
 */

enJobSelection FEGroupJob::ToggleSelection()
{
    enJobSelection selNew;

    switch (_Selection)
    {
        case JOB_INSTALL:
            selNew = JOB_DEINSTALL;
        break;

        case JOB_DEINSTALL:
            selNew = JOB_INSTALL_IGNORE;
        break;

        case JOB_INSTALL_IGNORE:
            selNew = JOB_INSTALL;
        break;

        case JOB_GROUP_MIXED:
            selNew = JOB_INSTALL;
        break;
    }

    Select(selNew);

    return selNew;
}

/*
 *@@ Select:
 *      this virtual FEJobBase method is overridden to
 *      set the job status for this group job and all of
 *      its member jobs.
 *
 *      ulJobNew can be one of the following:
 *
 *      -- JOB_INSTALL:   select the member package for installation.
 *      -- JOB_DEINSTALL:    select the member package for removal (de-install).
 *      -- JOB_INSTALL_IGNORE: do nothing. If the package is not yet installed,
 *                        it will not be installed. If the package is
 *                        already installed, it will not be deinstalled.
 *
 *      This calls guiJobSelectionChanged if anything has changed
 *      and automatically updates the group to which this group job
 *      belongs to, if any.
 *
 *@@changed V0.9.1 (2000-01-08) [umoeller]: renamed from "SetJobID"
 */

ULONG FEGroupJob::Select(enJobSelection selNew)
{
    ULONG ulrc = 0;
    list<FEJobBase*>::iterator JobStart = _GroupMembersList.begin(),
                               JobEnd = _GroupMembersList.end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        (**JobStart).Select(selNew);
            // this will call Update in turn
        ++ulrc;
    }

    return ulrc;
}

/*
 *@@ Update:
 *      this gets called by FEInstallJob::Select
 *      if the selection has changed for a package
 *      which belongs to a group, so that the group
 *      can update itself according to the members.
 */

VOID FEGroupJob::Update()
{
    const ULONG FOUND_MIXED     = 0x0001,
                FOUND_INSTALL   = 0x0002,
                FOUND_DEINSTALL = 0x0004,
                FOUND_IGNORE    = 0x0008;

    ULONG   ulFound = 0;

    // check our members for selections;
    // we will OR the flags above in ulFound
    // so if all our members have the same
    // selections, we'll get a "straight" value
    // in ulValue, otherwise more than one bit
    // will be set
    list<FEJobBase*>::iterator JobStart = _GroupMembersList.begin(),
                               JobEnd = _GroupMembersList.end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pJobThis = *JobStart;
        switch (pJobThis->QuerySelection())
        {
            case JOB_INSTALL:
                ulFound |= FOUND_INSTALL;
            break;

            case JOB_DEINSTALL:
                ulFound |= FOUND_DEINSTALL;
            break;

            case JOB_INSTALL_IGNORE:
                ulFound |= FOUND_IGNORE;
            break;

            case JOB_GROUP_MIXED:
                ulFound |= FOUND_MIXED;
            break;
        }
    }

    // now check if we have a "straight" one-bit value
    // and set our own value accordingly
    if (ulFound == FOUND_INSTALL)
        _Selection = JOB_INSTALL;
    else if (ulFound == FOUND_DEINSTALL)
        _Selection = JOB_DEINSTALL;
    else if (ulFound == FOUND_IGNORE)
        _Selection = JOB_INSTALL_IGNORE;
    else
        // any other combination:
        _Selection = JOB_GROUP_MIXED;

    _Locals.OnJobSelectionChanged(this);
            // V0.9.20 (2002-07-03) [umoeller]

    if (_pGroupJob)
    {
        // we belong to a group ourselves:
        // update parent group too (this will
        // climb up the group tree until the
        // root has been reached)
        _pGroupJob->Update();
    }
}


