
/*
 *@@sourcefile fe_database.cpp:
 *      this implements the FEDatabase class (declared in fe_database.h).
 *
 *      Creating a database instance (using FEDataBase::FEDataBase)
 *      will open the database file and initialize FEDBPackage instances.
 *
 *      After creation, you can, for example, invoke FEDatabase::FindPackageID
 *      or FEDatabase::FindDBPackage to query the database.
 *
 *      Also, the database has methods for verification and de-installation.
 *      See FEDatabase::VerifyPackages and FEDatabase::DeinstallPackages.
 *
 *      Currently, the database is stored in a possibly large
 *      INI file in the directory where WARPIN.EXE also resides.
 *
 *      With V0.9.0 (99-10-31) [umoeller], the database has been transformed
 *      into a C++ class so that more than one database may possibly
 *      be created (even though this is not used yet).
 *
 *      The functions in here are _not_ thread-safe, so the caller
 *      must synchronize access himself.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: renamed from database.cpp
 *@@header "engine\fe_database.h"
 */

/*
 *      This file Copyright (C) 1999-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSMODULEMGR
#define INCL_DOSERRORS
// #define INCL_WIN
#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <io.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#ifdef __IBMCPP__
    #include <direct.h>
#endif

#include <string.h>
#include <ctype.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h

#include "setup.h"

// include's from helpers
#include "helpers\configsys.h"
#include "helpers\dosh.h"
#include "helpers\except.h"
#include "helpers\nls.h"
#include "helpers\prfh.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"
#include "engine\fe_package.h"
#include "engine\fe_package_db.h"
#include "engine\fe_database.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

DEFINE_CLASS(FEDatabaseStatus, BSRoot);
DEFINE_CLASS(FEDatabase, BSRoot);

/* ******************************************************************
 *
 *  FEDatabase implementation
 *
 ********************************************************************/

/*
 *@@ FEDatabase:
 *      database constructor. This opens the database and
 *      constructs the public member list of FEDBPackage
 *      instances by creating as many instances as entries
 *      are found in the database. So after constructing
 *      the database, you get all database packages constructed
 *      as well.
 *
 *      Throws:
 *      -- FEDatabaseExcpt.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this was datInitialize
 *@@changed V0.9.1 (2000-01-08) [umoeller]: bugfixing mode added
 *@@changed V0.9.18 (2002-03-08) [umoeller]: rewrote bugfixing mode
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added Unicode handling
 */

FEDatabase::FEDatabase(FELocals &Locals,
                       const ustring &ustrDatabasePath) // in: path of database (without trailing "\")
    : BSRoot(tFEDatabase),
      _PackageIDsList(STORE),
      _DBPackagesList(SHADOW),           // switch list into shadow mode;
                                         // FEDatabase::RemovePackage spec says that
                                         // the package is still valid after RemovePackage
      _Locals(Locals)
{
    _hiniDatabase = NULLHANDLE;

    // check the "one-instance" semaphore, which
    // we create just for testing that WarpIN is
    // started only once
    static const char *pcszSem = "\\SEM32\\WARPIN\\PROTECT.SEM";
    APIRET arc;
    if (arc = DosCreateMutexSem(pcszSem,
                                &_hmtxOneInstance,
                                DC_SEM_SHARED,
                                FALSE))
    {
        if (arc == ERROR_DUPLICATE_NAME)
            throw FEFatalErrorExcpt(Locals, 155);   // "another instance"

        CHAR sz[500];
        sprintf(sz, "error %d creating %s", arc, pcszSem);
        throw BSExcptBase(sz);
    }

    string strDatabasePath(Locals._pCodecProcess,
                           ustrDatabasePath);
    PCSZ pszWarpINPath = strDatabasePath.c_str();

    CHAR szOldDatabaseFile[2*CCHMAXPATH];
    sprintf(szOldDatabaseFile,
            "%s\\database.ini",
            pszWarpINPath);

    _strDatabaseFile._printf("%s\\datbas_%c.ini",
                             pszWarpINPath,
                             doshQueryBootDrive());

    // check if old-style database still exists
    if (access(szOldDatabaseFile, 0) == 0)
        // yes: rename
        rename(szOldDatabaseFile, _strDatabaseFile.c_str());

    if (!(_hiniDatabase = PrfOpenProfile(Locals._habThread1,
                                         _strDatabaseFile.c_str())))
    {
        // error opening database INI file:
        ustring str(Locals._pCodecProcess,
                    _strDatabaseFile);
        throw FEFatalErrorExcpt(Locals,
                                157,
                                &str, 1);
    }

    // cache the INI apps so we won't have to re-read
    // them all the time; this is terminally slow if
    // the database is on a remote drive
    // V0.9.18 (2002-03-03) [umoeller]
    PSZ         pszPackageList = NULL;
    ULONG       cPackages = 0;

    if (!(arc = prfhQueryKeysForApp(_hiniDatabase,
                                    NULL, // query applications list
                                    &pszPackageList)))
    {
        // search list for pszPackageSearch
        PSZ pPackage = pszPackageList;

        // codec from loading stuff from database;
        // this gets recreated whenever the codepage
        // changes
        BSUniCodec  *pCodec = NULL;
        ULONG       ulCodepage;

        list<string*> listBrokenPackages(STORE);
        list<ustring*> listBrokenPackagesU(STORE);

        while (*pPackage)
        {
            // any packages are installed:
            // create a new package info from database

            // pPackage has the current key now;
            // store this in the private list
            // V0.9.18 (2002-03-03) [umoeller]
            string *pstr = new string(pPackage);

            ustring ustrConvID;

            // codepage V0.9.2 (2000-03-11) [umoeller]
            // V0.9.18 (2002-03-08) [umoeller]: load this
            // first so we can do string conversions!
            ulCodepage = PrfQueryProfileInt(_hiniDatabase,
                                            pPackage,
                                            WPIKEY_CODEPAGE,
                                            // 850);      // default
                                            0);        // use 0 for utf-8!
                                                   // V0.9.18 (2002-03-08) [umoeller]
            if (ulCodepage)
            {
                // legacy package: V0.9.18 (2002-03-08) [umoeller]
                if (!pCodec)
                    pCodec = new BSUniCodec(ulCodepage);
                // codepage changed from last loop?
                else if (pCodec->QueryCodepage() != ulCodepage)
                {
                    delete pCodec;
                    pCodec = new BSUniCodec(ulCodepage);
                }
                ustrConvID.assignCP(pCodec,
                                    pPackage);
            }
            else
                // package was stored in Unicode:
                ustrConvID.assignUtf8(pPackage);

            try
            {
                FEDBPackage* pPackageNew = new FEDBPackage(*this,
                                                           // codepaged string:
                                                           *pstr,
                                                           pCodec,
                                                           ustrConvID);

                                    // this can throw a FEDatabaseExcpt

                _PackageIDsList.push_back(pstr);
                _DBPackagesList.push_back(pPackageNew);
            }
            // catch all exception thrown by the constructor
            // and nuke the respective packages...
            catch(BSExcptBase &X)
            {
                // broken package:
                // append to list
                listBrokenPackages.push_back(pstr);

                ustring *pustr = new ustring(ustrConvID);
                pustr->appendUtf8(": ");
                pustr->append(X._ustrDescription);
                listBrokenPackagesU.push_back(pustr);
            }

            // else: package (application)
            pPackage += strlen(pPackage) + 1;
            ++cPackages;
        } // end while (*pPackage)

        if (listBrokenPackages.count())
        {
            ustring ustr;
            list<string*>::iterator brokenBegin = listBrokenPackages.begin(),
                                    brokenEnd = listBrokenPackages.end();
            list<ustring*>::iterator brokenBeginU = listBrokenPackagesU.begin(),
                                    brokenEndU = listBrokenPackagesU.end();
            for (;
                 brokenBegin != brokenEnd;
                 ++brokenBegin, ++brokenBeginU)
            {
                ustr.appendUtf8("    ");
                ustr.append(**brokenBeginU);
                ustr.appendUtf8("\n");
            }

            ULONG rc = Locals.MessageBox(102,
                                         275, // The following broken packages were found in the database:
                                         MSG_CONFIRM_YESNOCANCEL_DEFYES,
                                         &ustr, 1);
            switch (rc)
            {
                case MBID_YES:
                    brokenBegin = listBrokenPackages.begin(),
                    brokenEnd = listBrokenPackages.end();
                    for (;
                         brokenBegin != brokenEnd;
                         ++brokenBegin)
                    {
                        PCSZ pcszApp = (**brokenBegin).c_str();
                        // bugfixing mode:
                        PrfWriteProfileString(_hiniDatabase,
                                              pcszApp,
                                              NULL,             // delete whole application
                                              NULL);
                    }
                break;

                case MBID_CANCEL:
                    throw BSCancelExcpt();
                break;
            }
        }

        if (pCodec)
            delete pCodec;

        free(pszPackageList);
    }
}

/*
 *@@ ~FEDatabase:
 *      database destructor.
 */

FEDatabase::~FEDatabase()
{
    PrfCloseProfile(_hiniDatabase);
    DosCloseMutexSem(_hmtxOneInstance);
    _hmtxOneInstance = NULLHANDLE;
}

/*
 *@@ QueryDatabaseFile:
 *      returns the fully qualified filename of the
 *      INI file that the database was loaded from.
 *
 *@@added V0.9.14 (2001-08-09) [umoeller]
 */

const string& FEDatabase::QueryDatabaseFile()
                          const
{
    return _strDatabaseFile;
}

/*
 *@@ EnumPackages:
 *      this enumerates the packages which are stored in
 *      the database.
 *
 *      This works in two modes:
 *
 *      --  If pszPackageSearch is NULL, the first package
 *          in the database is returned.
 *
 *      --  If pszPackageSearch is != NULL, the package
 *          which comes after pszPackageSearch in the
 *          database is returned.
 *
 *      In any case, this function returns a const string* to
 *      an internal buffer which must not be freed (V0.9.18).
 *      If no (more) packages are found, NULL is returned.
 *
 *      The return value and pszPackageIDSearch are both
 *      in the "package ID" format (five- or six-part
 *      package ID, separated by four backslashes).
 *
 *      This does _not_ work on the member database lists,
 *      but only on the database INI file directly.
 *
 *      However, you can pass the return PSZ to
 *      FEDatabase::FindDBPackage to get a database member
 *      package.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this was datEnumPackages
 *@@changed V0.9.1 (2000-01-08) [umoeller]: this used to be private; now it's public
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now caching package IDs for speed, returning PCSZ
 */

const string *FEDatabase::EnumPackages(const string *pstrPackageIDSearch)
                          const
{
    list<string*>::iterator idBegin = _PackageIDsList.begin(),
                            idEnd = _PackageIDsList.end();

    const string *pFound = NULL;

    for (;
         idBegin != idEnd;
         ++idBegin)
    {
        string *pString = *idBegin;

        if (    // return first entry?
                (!pstrPackageIDSearch)
                // previous one matched search string?
             || (pFound)
           )
            return pString;

        if (*pString == *pstrPackageIDSearch)
            // return this on the next loop
            pFound = pString;
    }

    return NULL;

    /*
    PCSZ pszReturn = NULL;

    APIRET arc;
    PSZ pszPackageList = NULL;

    if (!(arc = prfhQueryKeysForApp(_hiniDatabase,
                                    NULL, // query applications list
                                    &pszPackageList)))
    {
        if (pszPackageIDSearch == NULL)
            // first package queried:
            pszReturn = strdup(pszPackageList); // first app is null-terminated
        else
        {
            // search list for pszPackageSearch
            PSZ pPackage = pszPackageList;

            while (*pPackage != 0)
            {
                // pPackage has the current key now
                if (strcmp(pszPackageIDSearch, pPackage) == 0)
                {
                    // pPackage found: return next
                    pPackage += strlen(pPackage)+1;
                    if (*pPackage)
                        // not last:
                        pszReturn = strdup(pPackage);

                    // in any case, stop search
                    break;
                }

                // else: next package (application)
                pPackage += strlen(pPackage)+1;
            }
        }
        free(pszPackageList);
    }

    // return found package or NULL
    return pszReturn;

    */
}

/*
 *@@ FindPackageID:
 *      this returns the package ID for the package
 *      which matches the given attributes. This
 *      can be used to find out whether a given
 *      package has already been installed and if
 *      so, which version is installed.
 *
 *      Since WarpIN does not allow several packages
 *      with the same author/application/name, but
 *      different version numbers, the returned
 *      information is guaranteed to be unique.
 *
 *      This returns NULL if no matching package was found.
 *
 *      The return value must be delete'd after use.
 *
 *      This does _not_ work on the member database lists,
 *      but only on the database INI file directly.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this was datFindPackage
 *@@changed V0.9.0 (99-11-03) [umoeller]: now returning FEPackageID*
 *@@changed V0.9.18 (2002-03-03) [umoeller]: no longer returning copy, but const FEPackageID*; speedup
 */

/*
 *@@ FindDBPackage:
 *      kinda similar to FEDatabase::FindPackageID,
 *      but this searches the member list of FEDBPackage
 *      instances and returns the list item which matches
 *      the given attributes.
 *
 *      Since WarpIN does not allow several packages
 *      with the same author/application/name, but
 *      different version numbers, the returned
 *      information is guaranteed to be unique.
 *
 *      The returned instance comes from the database
 *      packages list, so you better not delete() the
 *      returned package.
 *
 *      If none is found, NULL is returned.
 *
 *@@added V0.9.0 (99-11-04) [umoeller]
 *@@changed V0.9.18 (2002-03-03) [umoeller]: optimized
 */

const FEDBPackage* FEDatabase::FindDBPackage(const ustring &ustrAuthor,
                                             const ustring &ustrApplication,
                                             const ustring &ustrPackageName)
                               const
{
    list<FEDBPackage*>::iterator PckBegin = _DBPackagesList.begin(),
                                 PckEnd = _DBPackagesList.end();

    for (; PckBegin != PckEnd; ++PckBegin)
    {
        FEDBPackage *pPackageThis = *PckBegin;
        if (    (pPackageThis->_PckID._ustrAuthor == ustrAuthor)
             && (pPackageThis->_PckID._ustrApplication == ustrApplication)
             && (pPackageThis->_PckID._ustrPackageName == ustrPackageName)
           )
        {
            // matches: return that one
            return pPackageThis;
        }
    }

    return NULL;
}

/*
 *@@ FindDBPackage:
 *      second version of FindDBPackage, which takes a
 *      package ID as input.
 *
 *      Note that this does NOT check the version numbers
 *      in the package ID, but will return the first database
 *      package which matches author, application, and
 *      package name.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

const FEDBPackage* FEDatabase::FindDBPackage(const FEPackageID &PckID)
                               const
{
    return FindDBPackage(PckID._ustrAuthor,
                         PckID._ustrApplication,
                         PckID._ustrPackageName);
}

/*
 *@@ IsPackageRequired:
 *      returns the number of packages from the database which
 *      require the specified package to be installed, or returns
 *      0 if the specified package is not required by any other
 *      package.
 *
 *      The packages from the database which require the specified
 *      packages are appended to the specified list so they can
 *      be evaluated by the caller. The list receives pointers
 *      to the existing packages in the database, so IT MUST NOT
 *      BE IN STORE MODE.
 *
 *      All packages on *pIgnoreList are completely ignored during
 *      these checks. That is, if a package requires PckIDThis,
 *      but that package is also on *pIgnoreList, that requirement
 *      is not added to the list.
 *
 *      pIgnoreList can be NULL if you don't want any packages
 *      to be ignored. However, this can be useful if
 *      you have several packages to be deinstalled
 *      and all those packages should be ignored in the checks,
 *      because if the requiring packages are de-installed as
 *      well, there's no problem.
 *
 *@@added V0.9.4 (2000-07-01) [umoeller]
 *@@changed V0.9.18 (2002-02-06) [umoeller]: speed optimizations
 */

ULONG FEDatabase::IsPackageRequired(FEPackageID &PckIDThis,     // in: package to check
                                    list<FEDBPackage*> &RequiredByList, // out: packages which require this
                                    list<FEDBPackage*> *pIgnoreList) // in: packages to ignore in checks
                  const
{
    ULONG ulrc = 0;

    // check if a package in the database
    // requires the given package:

    // enumerate all packages in the database
    /* const string *pstrDBPckIDThis = EnumPackages(NULL);
    while (pstrDBPckIDThis) */

    list<FEDBPackage*>::iterator pckBegin = _DBPackagesList.begin(),
                                 pckEnd = _DBPackagesList.end();
    for (;
         pckBegin != pckEnd;
         ++pckBegin)
    {
        // get database package from that ID
        /* FEPackageID DBPckIDThis(_pLocals,
                                pstrDBPckIDThis->c_str(), __FILE__, __LINE__,
                                FALSE);    // don't allow short format
        const FEDBPackage *pDBPckThis
            = FindDBPackage(DBPckIDThis);
        if (!pDBPckThis)
            throw FEFatalErrorExcpt("database package not found"); */

        const FEDBPackage *pDBPckThis = *pckBegin;

        // decode "requires" logger strings for that DBPackage
        // PSZ     pszLogStart = pDBPckThis->_logRequiresIDs._pabLogString,
        //         pszDBRequiresIDThis = pszLogStart;

        // use pre-resolved package IDs in the packages now
        // V0.9.18 (2002-02-06) [umoeller]
        list<FEPackageID*>::iterator DBRequiresStart = pDBPckThis->_RequiresIDs.begin(),
                                     DBRequiresEnd = pDBPckThis->_RequiresIDs.end();

        for (;
             DBRequiresStart != DBRequiresEnd;
             ++DBRequiresStart)
        // while (pszDBRequiresIDThis < pszLogStart + pDBPckThis->_logRequiresIDs._cbLogString)
        {
            // ULONG   cbLogThis = strlen(pszDBRequiresIDThis);

            // pszDBRequiresIDThis now has a requirement for the DB package;
            // check if that ID is the same as that of the job which is
            // to be removed
            // FEPackageID DBRequiresIDThis(_pLocals,
               //                           pszDBRequiresIDThis, __FILE__, __LINE__,
                  //                        FALSE);    // don't allow short format

            FEPackageID *pDBRequiresIDThis = *DBRequiresStart;

            if (pDBRequiresIDThis->Compare(PckIDThis))
            {
                // matches:
                // check if this package is on the "ignore" list
                BOOL fIgnore = FALSE;
                if (pIgnoreList)
                {
                    list<FEDBPackage*>::iterator IgnoreStart = pIgnoreList->begin(),
                                                 IgnoreEnd = pIgnoreList->end();
                    for (; IgnoreStart != IgnoreEnd; ++IgnoreStart)
                    {
                        FEDBPackage *pIgnorePck = *IgnoreStart;
                        FEPackageID &IgnoreID = pIgnorePck->_PckID;
                                // V0.9.18 (2002-03-03) [umoeller]

                        if (IgnoreID.Compare(pDBPckThis->_PckID))
                        {
                            fIgnore = TRUE;
                            break;
                        }
                    }
                }

                if (!fIgnore)
                {
                    // requiring package is not on ignore list:
                    // store the DB package in the list which was
                    // given to us
                    RequiredByList.push_back((FEDBPackage*)pDBPckThis);
                    ++ulrc;
                }
            }

            // pszDBRequiresIDThis += cbLogThis + 1;    // go beyond null byte
        }

        // next package in the database
        // pstrDBPckIDThis = EnumPackages(pstrDBPckIDThis);
    }

    return ulrc;
}

/*
 *@@ RemovePackage:
 *      this removes a single package from the database.
 *      Note that this removes _only_ the database
 *      information. It does _not_ de-install the
 *      package (which is not possible after calling
 *      this function, because all database info
 *      for that package will be lost).
 *
 *      To fully deinstall packages (i.e. delete files,
 *      undo system config) call FEDatabase::DeinstallOrVerify,
 *      which calls this method in turn.
 *
 *      Note that pPckInfo is removed from WPIGLOBALS.DBPckList,
 *      but is not deleted itself. It's still valid after
 *      this call.
 *
 *      @@todo this isn't pretty and causes memory
 *      leaks. _DBPackagesList should be in STORE mode and
 *      the package should be removed.
 *
 *      Returns FALSE upon errors.
 *
 *      This no longer calls guiDatabaseCallback (0.9.14)...
 *      caller must update its GUI itself if this
 *      returns TRUE.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this was datRemovePackage
 *@@changed V0.9.14 (2001-07-24) [umoeller]: removed gui callback
 *@@changed V0.9.18 (2002-03-08) [umoeller]: adjusted INI string for new package IDs
 */

BOOL FEDatabase::RemovePackage(const FEDBPackage *pPackage)
{
    BOOL brc = FALSE;

    if (PrfWriteProfileString(_hiniDatabase,
                              // for deletion, use the exact INI app that
                              // the DB package was created from
                              // V0.9.18 (2002-03-08) [umoeller]
                              pPackage->_strAppLoadedFrom.c_str(),
                              NULL,             // delete whole application
                              NULL))
    {
        // OK, package deleted from database:

        // remove it from the list in memory too
        _DBPackagesList.remove((FEDBPackage*)pPackage);
                // this calls delete on the package!
                // @@todo does it?

        brc = TRUE;
    }

    return brc;
}

/*
 *@@ UndoConfig:
 *      called by FEDatabase::DeinstallOrVerify for each package
 *      to be de-installed. This removes config.sys
 *      entries, wps objects and classes, etc.
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added de-executes
 */

VOID FEDatabase::UndoConfig(HAB hab,
                            FEDatabaseStatus &Status,
                            BSConfigSys *pConfigSys,    // in: config.sys to manipulate
                            PBOOL pfConfigSysChanged)   // out: config.sys changed?
{
    BSFileLogger *pLogFile = Status._Locals._pLogFile;

    /*
     *  1) delete WPS objects:
     *
     */

    // use a reverse_iterator for deleting the
    // objects because there might be a folder
    // first which contains the other objects
    // and we'd get error messages then
    list<BSConfigBase*>::reverse_iterator
            RConfigStart = Status._pPackage->_listUndoConfig.rbegin(),
            RConfigEnd = Status._pPackage->_listUndoConfig.rend();
    for (; RConfigStart != RConfigEnd; ++RConfigStart)
    {
        // now typecast this to the proper BSConfigBase
        // subclass we need
        DYNAMIC_CAST(BSDeleteWPSObject, pDelThis, *RConfigStart);
        if (!pDelThis)
            // not proper subclass:
            continue;

        if (pDelThis->IsSelected()) // V0.9.4 (2000-07-01) [umoeller]
        {
            try
            {
                pDelThis->Delete(*_Locals._pCodecProcess, pLogFile);
                Status.OnObjectDeleted(pDelThis, TRUE);
                /* Status.Callback(DBC_OBJECTDELETED,
                                (ULONG)pDelThis->_pszObjectID); */
            }
            catch(BSConfigExcpt& X)
            {
                // error deleting WPS object:
                // call GUI callback with error
                Status.OnObjectDeleted(pDelThis, FALSE);
                /* Status.Callback(DBC_OBJECTDELETEERROR,
                                (ULONG)pDelThis->_pszObjectID); */
                Status._pPackage->_ulStatus = 1;         // == errors found
            }
        }
    }

    /*
     *  2)   unreplace classes
     *
     */

    list<BSConfigBase*>::iterator
        ConfigStart = Status._pPackage->_listUndoConfig.begin(),
        ConfigEnd = Status._pPackage->_listUndoConfig.end();
    for (; ConfigStart != ConfigEnd; ++ConfigStart)
    {
        // now typecast this to the proper BSConfigBase
        // subclass we need
        DYNAMIC_CAST(BSUnreplaceClass, pReplThis, *ConfigStart);
        // BSUnreplaceClass *pReplThis = (**ConfigStart).IsUnreplaceClass();
        if (!pReplThis)
            // not proper subclass:
            continue;

        if (pReplThis->IsSelected()) // V0.9.4 (2000-07-01) [umoeller]
        {
            try
            {
                pReplThis->Unreplace(*_Locals._pCodecProcess, pLogFile);
                Status.OnClassUnreplaced(pReplThis, TRUE);
                /* Status.Callback(DBC_CLASSUNREPLACED,
                                (ULONG)pReplThis->_pszOldClassName); */
                Status._Locals._fWPSClassesChanged = TRUE;
            }
            catch(BSConfigExcpt& X)
            {
                // error deregistering class:
                // call GUI callback with error
                Status.OnClassUnreplaced(pReplThis, FALSE);
                /* Status.Callback(DBC_UNREPLACECLASSERROR,
                                (ULONG)pReplThis->_pszOldClassName); */
                Status._pPackage->_ulStatus = 1;         // == errors found
            }
        }
    }

    /*
     *  3)   then deregister classes
     *
     */

    ConfigStart = Status._pPackage->_listUndoConfig.begin(),
    ConfigEnd = Status._pPackage->_listUndoConfig.end();
    for (; ConfigStart != ConfigEnd; ++ConfigStart)
    {
        // now typecast this to the proper BSConfigBase
        // subclass we need
        DYNAMIC_CAST(BSDeregisterClass, pDeregThis, *ConfigStart);
        // BSDeregisterClass *pDeregThis = (**ConfigStart).IsDeregisterClass();
        if (!pDeregThis)
            // not proper subclass:
            continue;

        if (pDeregThis->IsSelected()) // V0.9.4 (2000-07-01) [umoeller]
        {
            try
            {
                pDeregThis->Deregister(*_Locals._pCodecProcess, pLogFile);
                /*
                Status.Callback(DBC_CLASSDEREGISTERED,
                                (ULONG)pDeregThis->_pszClassName); */
                Status.OnClassDeregistered(pDeregThis, TRUE);
                Status._Locals._fWPSClassesChanged = TRUE;
            }
            catch(BSConfigExcpt& X)
            {
                // error deregistering class:
                // call GUI callback with error
                Status.OnClassDeregistered(pDeregThis, FALSE);
                /* Status.Callback(DBC_DEREGISTERCLASSERROR,
                                (ULONG)(X._strDescription.GetBuffer())); */
                Status._pPackage->_ulStatus = 1;         // == errors found
            }
        }
    }

    /*
     *  4) undo CONFIG.SYS changes:
     *
     */

    ConfigStart = Status._pPackage->_listUndoConfig.begin(),
    ConfigEnd = Status._pPackage->_listUndoConfig.end();
    for (; ConfigStart != ConfigEnd; ++ConfigStart)
    {
        // now typecast this to the proper BSConfigBase
        // subclass we need
        DYNAMIC_CAST(BSCfgSysManip, pManipThis, *ConfigStart);
        // BSCfgSysManip *pManipThis = (**ConfigStart).IsCfgSysManip();
        if (!pManipThis)
            // not proper subclass:
            continue;

        if (pManipThis->IsSelected()) // V0.9.4 (2000-07-01) [umoeller]
        {
            try
            {
                // the items on this list have been
                // created by the FEPackage constructor
                // just according to what needs to be undone
                pConfigSys->Manipulate(*_Locals._pCodecProcess,
                                       *pManipThis,
                                       NULL,   // no logger here
                                       pLogFile);
                *pfConfigSysChanged = TRUE;
                Status.OnCfgSysUndone(pManipThis, TRUE);
                /* Status.Callback(DBC_CFGSYSUNDONE,
                                (ULONG)pManipThis); */
                Status._Locals._fConfigSysChanged = TRUE;
            }
            catch (BSConfigExcpt& X)
            {
                // error deregistering class:
                // call GUI callback with error
                Status.OnCfgSysUndone(pManipThis, FALSE);
                /* Status.Callback(DBC_UNDOCFGSYSERROR,
                                (ULONG)pManipThis); */
                Status._pPackage->_ulStatus = 1;         // == errors found
            }
        }
    }

    /*
     *  5) clear profile entries
     *
     */

    ConfigStart = Status._pPackage->_listUndoConfig.begin(),
    ConfigEnd = Status._pPackage->_listUndoConfig.end();
    for (; ConfigStart != ConfigEnd; ++ConfigStart)
    {
        // now typecast this to the proper BSConfigBase
        // subclass we need
        DYNAMIC_CAST(BSClearProfile, pManipThis, *ConfigStart);
        // BSClearProfile *pManipThis = (**ConfigStart).IsClearProfile();
        if (!pManipThis)
            // not proper subclass:
            continue;

        if (pManipThis->IsSelected()) // V0.9.4 (2000-07-01) [umoeller]
        {
            try
            {
                pManipThis->Clear(*_Locals._pCodecProcess,
                                  hab,
                                  pLogFile);
                Status.OnProfileCleared(pManipThis, TRUE);
                /* Status.Callback(DBC_PROFILECLEARED,
                                (ULONG)pManipThis); */
            }
            catch (BSConfigExcpt& X)
            {
                // error deregistering class:
                // call GUI callback with error
                Status.OnProfileCleared(pManipThis, FALSE);
                /* Status.Callback(DBC_CLEARPROFILEERROR,
                                (ULONG)pManipThis);
                */
                Status._pPackage->_ulStatus = 1;         // == errors found
            }
        }
    }

    /*
     *  6) call de-executes V0.9.9 (2001-03-30) [umoeller]
     *
     */

    ConfigStart = Status._pPackage->_listUndoConfig.begin(),
    ConfigEnd = Status._pPackage->_listUndoConfig.end();
    for (; ConfigStart != ConfigEnd; ++ConfigStart)
    {
        // now typecast this to the proper BSConfigBase
        // subclass we need
        DYNAMIC_CAST(BSDeExecute, pManipThis, *ConfigStart);
        // BSDeExecute *pManipThis = (**ConfigStart).IsDeExecute();
        if (!pManipThis)
            // not proper subclass:
            continue;

        if (pManipThis->IsSelected()) // V0.9.4 (2000-07-01) [umoeller]
        {
            try
            {
                Status.OnDeExecute(pManipThis,
                                   -1);     // start
                /* Status.Callback(DBC_DEEXECUTING,
                                (ULONG)pManipThis); */
                pManipThis->Execute(*_Locals._pCodecProcess,
                                    pLogFile);
                Status.OnDeExecute(pManipThis,
                                   0);      // done, no error
                /* Status.Callback(DBC_DEEXECUTE_DONE,
                                (ULONG)pManipThis); */
            }
            catch (BSConfigExcpt& X)
            {
                // error de-executing:
                // call GUI callback with error
                /* Status.Callback(DBC_DEEXECUTE_ERROR,
                                (ULONG)pManipThis); */
                Status.OnDeExecute(pManipThis,
                                   X._iErrorCode);
                Status._pPackage->_ulStatus = 1;         // == errors found
            }
        }
    }
}

/*
 *@@ VerifyFile:
 *      called from FEDatabase::DeinstallOrVerify to
 *      verify a single file.
 *
 *      The "Status" field in the thread info has
 *      the current package and file.
 *
 *      Returns FALSE if an error was found or TRUE
 *      if the file is OK.
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked database thread
 *@@changed V0.9.9 (2001-04-06) [umoeller]: didn't work with root dir target path
 */

BOOL FEDatabase::VerifyFile(FEDatabaseStatus &Status)
{
    BOOL brc = FALSE;

    FEDBPackage *pPackage = Status._pPackage;
    WIFileHeader *pwifh = Status._pwifh;

    CHAR szFullFile[2*CCHMAXPATH];

    // fixed root directory V0.9.9 (2001-04-06) [umoeller]
    string strTemp(_Locals._pCodecProcess,
                   pPackage->QueryTargetPath());
    strcpy(szFullFile, strTemp.c_str());
    if (szFullFile[strlen(szFullFile) - 1] != '\\')
        strcat(szFullFile, "\\");
    strcat(szFullFile, pwifh->name);

    FILESTATUS3 fs3;
    APIRET arc;
    if (arc = DosQueryPathInfo(szFullFile, FIL_STANDARD, &fs3, sizeof(fs3)))
    {
        // error: complain at callback
        Status.Callback(DBC_FILEERROR,
                        arc);            // error code
    }
    else
    {
        // file found: compare last write dates
        // of database and real file
        FDATE fdateHeaderLastWrite;
        FTIME ftimeHeaderLastWrite;
        int iComp = wpiCompareFileDates(pwifh,
                                        &fs3,
                                        &fdateHeaderLastWrite,
                                        &ftimeHeaderLastWrite);
        if (iComp < 0)
            Status.Callback(DBC_EXISTINGFILENEWER,
                            (ULONG)&fs3);
        else if (iComp > 0)
            Status.Callback(DBC_EXISTINGFILEOLDER,
                            (ULONG)&fs3);
        else
            // else 0: file dates are the same
            brc = TRUE;
    }
    return brc;
}

/*
 *@@ DeinstallFile:
 *      called from FEDatabase::DeinstallOrVerify to
 *      deinstall (= delete) a single file.
 *
 *      The "Status" field in the thread info has
 *      the current package and file.
 *
 *      Returns FALSE if an error was found or TRUE
 *      if the file is OK.
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked database thread
 *@@changed V0.9.9 (2001-04-06) [umoeller]: didn't work with root dir target path
 */

BOOL FEDatabase::DeinstallFile(FEDatabaseStatus &Status)
{
    BOOL brc = FALSE;

    FEDBPackage *pPackage = Status._pPackage;
    WIFileHeader *pwifh = Status._pwifh;
    BSFileLogger *pLogFile = Status._Locals._pLogFile;

    CHAR szFullFile[2*CCHMAXPATH];

    // fixed root directory V0.9.9 (2001-04-06) [umoeller]
    string strTemp;
    strTemp.assignUtf8(_Locals._pCodecProcess, pPackage->QueryTargetPath());
    strcpy(szFullFile, strTemp.c_str());
    if (szFullFile[strlen(szFullFile) - 1] != '\\')
        strcat(szFullFile, "\\");
    strcat(szFullFile, pwifh->name);

    // check file date; do not delete if newer V0.9.9 (2001-04-06) [umoeller]
    BOOL fProceed = TRUE;

    FILESTATUS3 fs3;
    APIRET arc;
    if (arc = DosQueryPathInfo(szFullFile, FIL_STANDARD, &fs3, sizeof(fs3)))
    {
        // error: complain at callback
        Status.Callback(DBC_FILEERROR,
                        arc);            // error code
        if (pLogFile)
            pLogFile->Write("Error %d checking file \"%s\"",
                                     arc,
                                     szFullFile);
        fProceed = FALSE;
    }
    else
    {
        // file found: compare last write dates
        // of database and real file
        FDATE fdateHeaderLastWrite;
        FTIME ftimeHeaderLastWrite;
        int iComp = wpiCompareFileDates(pwifh,
                                        &fs3,
                                        &fdateHeaderLastWrite,
                                        &ftimeHeaderLastWrite);
        if (iComp < 0)
        {
            if (pLogFile)
            {
                CHAR        szExistingDate[40] = "",
                            szExistingTime[40] = "",
                            szDatabaseDate[40] = "",
                            szDatabaseTime[40] = "";
                FDATE       fdateLastWrite;
                FTIME       ftimeLastWrite;

                nlsFileDate(szExistingDate,
                            &(fs3.fdateLastWrite),
                            2, '-');
                nlsFileTime(szExistingTime,
                            &(fs3.ftimeLastWrite),
                            1, ':');
                // convert file header's time information
                time_t lastwrite = pwifh->lastwrite;
                wpiCTime2FTime(&lastwrite,
                               &fdateLastWrite,
                               &ftimeLastWrite);
                nlsFileDate(szDatabaseDate,
                            &fdateLastWrite,
                            2, '-');
                nlsFileTime(szDatabaseTime,
                            &ftimeLastWrite,
                            1, ':');

                pLogFile->Write("File \"%s\" is newer than in database:",
                                         szFullFile);
                pLogFile->Write("  Existing file: date %s, time %s",
                                         szExistingDate, szExistingTime);
                pLogFile->Write("  Database: date %s, time %s",
                                         szDatabaseDate, szDatabaseTime);
            }

            FEFileError gfe(Status._Locals,
                            &pPackage->QueryTargetPath(),
                            pwifh);
            fProceed = Status.Callback(DBC_DELETE_EXISTINGFILENEWER,
                                       (ULONG)&gfe);
            if (pLogFile)
            {
                pLogFile->Write("  User selected: %s",
                                         (fProceed) ? "delete anyway" : "skip");
            }
        }
    }
    // end check file date V0.9.9 (2001-04-06) [umoeller]

    if (fProceed)
    {
        // reset the file's attributes, because
        // DosDelete fails if this has READONLY
        BOOL fRetry = FALSE;
        do
        {
            fRetry = FALSE;
            APIRET arc;
            if (arc = doshSetPathAttr(szFullFile, 0))
            {
                if (arc == ERROR_SHARING_VIOLATION)
                {
                    // file is locked:
                    if (Status.Callback(DBC_FILELOCKED,
                                        arc)                // error code
                            == TRUE)
                        // unlock:
                    {
                        #ifndef __EMX__
                            // we use the undocumented DosReplaceModule API.
                            DosReplaceModule(szFullFile,        // old module
                                             NULL,              // new module
                                             NULL);             // backup module
                        #endif

                        fRetry = TRUE;
                    }
                }

                if (!fRetry)
                {
                    // other error or no unlock:
                    if (pLogFile)
                        pLogFile->Write("Error %d checking file \"%s\"",
                                                 arc,
                                                 szFullFile);
                    // complain at callback
                    Status.Callback(DBC_FILEERROR,
                                    arc);                // error code
                }
            }
            else
            {
                // file exists: delete it
                arc = DosDelete(szFullFile);
                if (arc != NO_ERROR)
                {
                    // report error
                    if (pLogFile)
                        pLogFile->Write("Error deleting file \"%s\"",
                                                 szFullFile);

                    Status.Callback(DBC_DELETEERROR,
                                    arc);            // error code
                }
                else
                {
                    if (pLogFile)
                        pLogFile->Write("Deleted file \"%s\"",
                                                 szFullFile);
                    brc = TRUE;
                }
            }
        } while (fRetry);
    }
    else
        pLogFile->Write("Skipped file \"%s\"",
                                 szFullFile);

    return brc;
}

/*
 *@@ DeinstallOrVerify:
 *      this method either compares packages with the actual files
 *      on disk or removes (deinstalls) them. PackagesList is
 *      expected to contain a (shadow) list of packages to
 *      work on.
 *
 *      If ulTask == DBT_VERIFY, this will verify only.
 *
 *      By contrast, if ulTask == DBT_DEINSTALL, this will
 *      completely deinstall each package on the given list.
 *
 *      If flDeinstall is 0, only the files will be removed.
 *      You can however OR any of the following:
 *
 *      -- DBDEINST_CONFIGSYS:  remove CONFIG.SYS entries
 *
 *      -- DBDEINST_WPSCLASSES: de-install WPS classes
 *
 *      -- DBDEINST_WPSOBJECTS: destroy WPS objects
 *
 *      This calls guiDatabaseCallback all the time.
 *
 *      Each package on the given list will also be removed
 *      from the database if deinstall was successful, by
 *      calling FEDatabase::RemovePackage.
 *
 *      The PackagesList itself is _not_ modified. However,
 *      if this was called with DBT_DEINSTALL, the package
 *      pointers on the list are probably no longer valid.
 *      The caller should therefore simply delete the list
 *      after this function returns.
 *
 *      Besides, the list must be in shadow mode and contain
 *      copy pointers to the actual packages in the database.
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked database thread
 *@@changed V0.9.1 (2000-01-05) [umoeller]: reworked configuration undoing
 *@@changed V0.9.1 (2000-01-05) [umoeller]: CONFIG.SYS was always rewritten to disk; fixed
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added individual undo-config selection
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added deletion of empty subdirs
 *@@changed V0.9.6 (2000-10-27) [umoeller]: CONFIG.SYS backup filename wasn't reported right, fixed.
 *@@changed V0.9.14 (2001-07-26) [umoeller]: made this a FEDatabase method
 *@@changed V0.9.20 (2002-07-22) [umoeller]: now only removing directories that were created during install
 *@@changed V0.9.20 (2002-07-22) [umoeller]: no longer reporting package error for directories that could not be deleted
 */

VOID FEDatabase::DeinstallOrVerify(HAB hab,
                                   FEDatabaseStatus &Status,
                                   list<FEDBPackage*> &PackagesList,
                                   ULONG ulTask,
                                   ULONG flDeinstall) // in: DBDEINST_* flags
{
    BSFileLogger *pLogFile = Status._Locals._pLogFile;

    // install the LOUD exception handler (except.h)
    TRY_LOUD(excpt1)
    {
        // to this list, we'll add the target directories of
        // the packages which we have removed
        // list<FEDBPackage*> TargetDirList(SHADOW);

        list<ustring*> TargetDirList(STORE);
                    // replaced V0.9.20 (2002-07-22) [umoeller]

        // if we're in de-install mode and we're to undo
        // changes to CONFIG.SYS, load CONFIG.SYS now
        // (by creating a BSConfigSys instance)
        BSConfigSys*        pConfigSys = 0;
        BOOL                fConfigSysChanged = FALSE;
        if (    (flDeinstall & DBT_UNDOCONFIG)
             && (ulTask == DBT_DEINSTALL)
           )
            pConfigSys = new(BSConfigSys);

        // PASS 1: count packages and files
        list<FEDBPackage*>::iterator PckBegin = PackagesList.begin(),
                                     PckEnd = PackagesList.end();
        for (; (PckBegin != PckEnd); ++PckBegin)
        {
            ++Status._cPackages;
            Status._cFiles += (**PckBegin)._PackHeader.files;
        }

        // PASS 2: go thru all packages given to us
        PckBegin = PackagesList.begin();
        PckEnd = PackagesList.end();
        for (; PckBegin != PckEnd; ++PckBegin)
        {
            FEDBPackage *pPackage = *PckBegin;

            if (pLogFile)
            {
                pLogFile->Write("Deinstalling package \"%s\"",
                                 pPackage->QueryPackageID().GetBuffer());
                pLogFile->IncIndent(+4);
            }

            pPackage->_ulStatus = 2;
                    // package status = "no errors found"
                    // (was either 0 or 2 before)

            // set package in status; this is used
            // for the callback and all subsequent actions
            Status._pPackage = pPackage;
            Status._pwifh = 0;        // for now, changed later

            // report package to callback
            Status.Callback(DBC_NEXTPACKAGE,
                            0);

            string strTemp;
            strTemp.assignUtf8(_Locals._pCodecProcess, pPackage->QueryTargetPath());
            if (!doshQueryDirExist(strTemp.c_str()))
            {
                // target directory not found:
                // call GUI callback with error
                Status.Callback(DBC_TARGETDIRNOTFOUND,
                                0);
            }

            // in de-install mode, undo CONFIG.SYS/WPS classes/WPS objects
            if (    (pPackage->_ulStatus == 2)        // no errors found
                 && (ulTask == DBT_DEINSTALL) // de-install mode
                 && (flDeinstall & DBT_UNDOCONFIG) // undo configuration
               )
            {
                UndoConfig(hab,
                           Status,
                           pConfigSys,
                           &fConfigSysChanged);
            } // end configuration (CONFIG.SYS, WPS classes, WPS objects)

            // only in verify mode or if files are to be
            // removed, go thru files list
            if (    (ulTask == DBT_VERIFY)
                 || (flDeinstall & DBT_DELETEFILES)
               )
            {
                // get list of file headers from package
                ULONG ulHeaderCount = 0;            // count of headers in this package
                list<WIFileHeaderLI*> *pFileHeadersList
                       = pPackage->GetFileHeaders(&ulHeaderCount);

                // now verify or delete all files
                if (pFileHeadersList)
                {
                    list<WIFileHeaderLI*>::iterator HeaderStart = pFileHeadersList->begin(),
                                                    HeaderEnd = pFileHeadersList->end();

                    for (; HeaderStart != HeaderEnd; ++HeaderStart)
                    {
                        WIFileHeader *pwifh = (**HeaderStart)._p;
                        Status._pwifh = pwifh;

                        // report file
                        Status.Callback(DBC_NEXTFILE,
                                        NULL);               // ulExtra

                        switch (ulTask)
                        {
                            case DBT_VERIFY:
                                if (!VerifyFile(Status))
                                    pPackage->_ulStatus = 1;         // == errors found
                            break;

                            case DBT_DEINSTALL:
                                if (!DeinstallFile(Status))
                                    pPackage->_ulStatus = 1;         // == errors found
                            break;
                        }

                        ++Status._ulCurrentFile;
                    }

                    delete pFileHeadersList;
                }
                // now done with the files of this package

                // in de-install mode, prepare removal
                // of directories too: this must be done
                // at the very end, because several packages
                // might use the same directory, so we collect
                // these for all packages first
                if (    (pPackage->_ulStatus == 2)        // no errors found
                     && (ulTask == DBT_DEINSTALL)
                     && (flDeinstall & DBT_DELETEFILES) // files are to be removed
                   )
                {
                    // run thru the logger of directories that were created
                    // during install and add them to the list of directories
                    // to be removed, if they aren't added yet
                    // reworked V0.9.20 (2002-07-22) [umoeller]
                    PSZ     pszLogStart = pPackage->_logDirsCreated._pabLogString,
                            pszLogThis = pszLogStart;
                    ustring ustrLogThis;
                    while (pszLogThis < pszLogStart + pPackage->_logDirsCreated._cbLogString)
                    {
                        ULONG cbLogThis = strlen(pszLogThis);

                        ustrLogThis.assignUtf8(pszLogThis);

                        BOOL    fFound = FALSE;
                        list<ustring*>::iterator dirStart = TargetDirList.begin(),
                                                 dirEnd = TargetDirList.end();
                        for (;
                             dirStart != dirEnd;
                             ++dirStart)
                        {
                            ustring *pstrDirThis = *dirStart;
                            if (!pstrDirThis->compareI(ustrLogThis))
                            {
                                // matches:
                                fFound = TRUE;
                                break;
                            }
                        }

                        if (!fFound)
                        {
                            TargetDirList.push_back(new ustring(ustrLogThis));
                        }

                        pszLogThis += cbLogThis + 1;    // go beyond null byte
                    }

                    /*      replaced V0.9.20 (2002-07-22) [umoeller]
                    // check if the target dir of this package is
                    // already in the list
                    list<FEDBPackage*>::iterator TargetDirFirst = TargetDirList.begin(),
                                                 TargetDirLast = TargetDirList.end();
                    BOOL fFound = FALSE;
                    for (; TargetDirFirst != TargetDirLast; ++TargetDirFirst)
                    {
                        FEDBPackage    *pTargetThis = *TargetDirFirst;
                        if (pTargetThis->QueryTargetPath() == pPackage->QueryTargetPath())
                        {
                            fFound = TRUE;
                            break;
                        }
                    }
                    if (!fFound)
                        // not in list: add
                        TargetDirList.push_back(pPackage);
                    */

                    // remove this package from the database;
                    // this will call guiDatabaseCallback again
                    if (RemovePackage(pPackage))
                    {
                           // but the package pointer is still valid
                           // after this, per definition, because
                           // database package list is in SHADOW mode
                        // and call the GUI callback
                        Status.Callback(DBC_PACKAGEREMOVED,
                                        (ULONG)pPackage);

                        // @@todo remove package from todo list
                   }

                   if (pLogFile)
                       pLogFile->Write("Removed package \"%s\" from the database",
                                        pPackage->QueryPackageID().GetBuffer());
                }

                // increase package counter
                ++Status._ulCurrentPackage;
            }

            if (pLogFile)
            {
                pLogFile->IncIndent(-4);
                pLogFile->Write("Done with package \"%s\"",
                                 pPackage->QueryPackageID().GetBuffer());
            }

        } // end for (; (PckBegin != PckEnd); ++PckBegin)

        // done with packages!

        // remove all directories on the target dir list
        // which have been collected from all packages above
        // (this list is empty for verify mode)

        /* replaced V0.9.20 (2002-07-22) [umoeller]
        list<FEDBPackage*>::iterator TargetDirFirst = TargetDirList.begin(),
                                     TargetDirLast = TargetDirList.end();
        for (; TargetDirFirst != TargetDirLast; ++TargetDirFirst)
        {
            FEDBPackage *pPackage = *TargetDirFirst;
            Status._pPackage = pPackage;
            Status._pwifh = 0;

            ULONG cDirsInDir = 0;
            ULONG cFilesInDir = 0;
            string strTemp(_Locals._pCodecProcess,
                           pPackage->QueryTargetPath());
            const char *pcszDir = strTemp.c_str();
            APIRET arc = doshDeleteDir(pcszDir,
                                       DOSHDELDIR_RECURSE,
                                       &cDirsInDir,
                                       &cFilesInDir);
        */

        Status._pPackage = NULL;
        Status._pwifh = 0;

        list<ustring*>::iterator dirFirst = TargetDirList.begin(),
                                 dirLast = TargetDirList.end();
        for (;
             dirFirst != dirLast;
             ++dirFirst)
        {
            ustring *pstrDir = *dirFirst;
            ULONG cDirsInDir = 0;
            ULONG cFilesInDir = 0;
            string strTemp(_Locals._pCodecProcess,
                           *pstrDir);
            const char *pcszDir = strTemp.c_str();

            APIRET arc;
            if (arc = doshDeleteDir(pcszDir,
                                    DOSHDELDIR_RECURSE,
                                    &cDirsInDir,
                                    &cFilesInDir))
            {
                if (pLogFile)
                    pLogFile->Write("Error %d removing directory \"%s\", %d dirs and %d files present",
                                    arc,
                                    pcszDir,
                                    cDirsInDir,
                                    cFilesInDir);

                // fixed callback
                // V0.9.19 (2002-04-14) [umoeller]
                Status.OnCannotDeleteDirError(*pstrDir,
                                              cDirsInDir,
                                              cFilesInDir);

                // (*TargetDirFirst)->_ulStatus = 1;         // == errors found
                        // no longer report error
            }
            else
                if (pLogFile)
                    pLogFile->Write("Removed directory \"%s\"",
                                    pcszDir);
        }

        TargetDirList.clear();

        // write BSConfigSys back to disk
        if (fConfigSysChanged)
        {
            if (pConfigSys->Flush(&Status._Locals._strConfigSysBackup,
                                  pLogFile) == 0)
                Status._Locals._fConfigSysChanged = TRUE;
        }

        if (pConfigSys)
            delete pConfigSys;     // V0.9.14 (2001-07-24) [umoeller]

        // report NULL package (== we're done)
        Status._pPackage = NULL;
        Status.Callback(DBC_NEXTPACKAGE,
                        0);
    }
    CATCH (excpt1)
    {
    } END_CATCH();
}

