
/*
 *@@sourcefile fe_archive.cpp:
 *      this implements the FEArchive class, which is new with
 *      V0.9.0 (99-10-31) [umoeller].
 *
 *      The entire archive/package/script/packages handling has
 *      been reworked greatly for V0.9.9 to allow for replacing
 *      the script parsing. As a result, another layer had to
 *      be introduced between the FEArchive class and the package
 *      classes for script handling.
 *
 *      See FEArchive for details.
 *
 *@@header "engine\fe_archive.h"
 *@@added V0.9.0 (99-10-31) [umoeller]
 */

/*
 *      This file Copyright (C) 1999-2001 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

// Changes by Cornelis Bockemhl (cbo) for REXX implementation:
// -> search for "(cbo" to find all changes
// Most changes follow immediately after a GetBlock call and perform
// REXX calls in order to replace =("macro args") macros
// Be careful: In some cases the GetBlock() call had to get offset
// attribute(s) added. In some other cases the if(!GetBlock()) logic
// is reversed (in order to get the "positive" code block first).

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOSERRORS
#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#ifdef __IBMCPP__
    #include <direct.h>
#endif
#include <io.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"
#include "bldlevel.h"           // needed for version checks

// include's from helpers
#include "helpers\dosh.h"
#include "helpers\nls.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"

#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_archive.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  Static variables
 *
 ********************************************************************/

DEFINE_CLASS(FEArchive, BSRoot);

/* ******************************************************************
 *
 *  FEArchive implementation
 *
 ********************************************************************/

/*
 *@@ FEArchive:
 *      FEArchive constructor. This must be created by
 *      passing the WIArchive file name.
 *
 *      NOTE: The archive file name must be fully qualified.
 *
 *      This constructor calls the back end by initializing
 *      the member instance of WIArchive from the given file name.
 *
 *      As opposed to versions before V0.9.9, it no longer parses
 *      the script because this should be deferred (and not done at
 *      all for dependent archives).
 *
 *      To parse the script, call FEArchive::ParseScript afterwards.
 *      Only that call will set the _pScript member.
 *
 *      See fe_archive.cpp for an introduction.
 *
 *      This throws:
 *
 *      -- FEFatalErrorExcpt (if archive could not be opened).
 *
 *@@added V0.9.0 (99-10-31) [umoeller]
 *@@changed V0.9.2 (2000-03-10) [umoeller]: added external archive support
 *@@changed V0.9.2 (2000-03-11) [umoeller]: added more error checking
 *@@changed V0.9.3 (2000-04-28) [umoeller]: finally accepting tabs in scripts
 *@@changed V0.9.4 (2000-07-26) [umoeller]: added strNext to args list
 *@@changed V0.9.19 (2002-04-14) [umoeller]: now using ustring for archive name
 */

FEArchive::FEArchive(FELocals &Locals,
                     const ustring &ustrArcFilename_,   // in: archive filename
                     FEArchive *pReferencedFrom)        // in: archive this archive was
                                                        // referenced from; NULL means this
                                                        // is no external archive (default)
    : BSRoot(tFEArchive),
      _Locals(Locals)
{
    _pPackagesList = new list<FEArcPackageBase*>(STORE);
            // switch packages list to STORE mode (auto-free)

    _pReferencedFrom = pReferencedFrom;

    // if this is the base archive (not referenced from anywhere):
    // get the global archive path
    // (cbo 2000-05-23)
    ustring ustrArcFilename(ustrArcFilename_);
    if (NULL == pReferencedFrom)
        ExtractArchivePath(ustrArcFilename);

    // prepend archive path in any case
    PrependArchivePath(ustrArcFilename);

    // _pPackHeadersList = 0;

    _pArcHeader = NULL;

    _pScript = 0;

    // open the archive in param 1 (WIArchive back-end)
    int irc;
    string strArchiveName(_Locals._pCodecProcess, ustrArcFilename);
    PCSZ pcszArchiveName = strArchiveName.c_str();
    Locals.Log("Calling WIArchive::open(\"%s\")", pcszArchiveName);
    if (irc = _Arc.open(pcszArchiveName))
    {
        ustring aStrings[3];
        aStrings[0] = ustrArcFilename;
        ULONG   cpsz = 1;
        ULONG   ulMsg = 0;

        switch (irc)
        {
            case WIERR_UNSPECIFIED_ERROR:
                ulMsg = 220; break;

            case WIERR_IO_OPEN:
                ulMsg = 221; break;

            case WIERR_IO_READ:
                ulMsg = 222; break;

            case WIERR_IO_WRITE:
                ulMsg = 223; break;

            case WIERR_IO_SEEK:
                ulMsg = 224; break;

            case WIERR_IO_CLOSE:
                ulMsg = 225; break;

            case WIERR_FILE_MAKEDIRECTORY:
                ulMsg = 226; break;

            case WIERR_ARCHIVE_CORRUPT:
                ulMsg = 227; break;

            case WIERR_INTERNAL:
                ulMsg = 228; break;

            case WIERR_CANNOT_CREATE:
                ulMsg = 229; break;

            case WIERR_UNKNOWN_METHOD:
                ulMsg = 230; break;

            case WIERR_INVALID_INDEX:
                ulMsg = 231; break;

            case WIERR_FILENOTFOUND:
                if (pReferencedFrom)
                {
                    // external archive:
                    ulMsg = 184;    // unable to open archive %1 from %2.
                    aStrings[1] = pReferencedFrom->_ustrArcFilename;
                    cpsz = 2;
                }
                else
                    ulMsg = 182;    // unable to open archive %1.
            break;

            case WIERR_OUTDATED_ARCHIVE:
                ulMsg = 168;    // archive %1 is outdated
            break;

            case WIERR_BZDECOMPRESS:
                ulMsg = 181;
            break;

            case WIERR_INVALID_HEADER:
                ulMsg = 183;    // archive %1 has invalid header
            break;

            case WIERR_NEWER_ARCHIVE:
                ulMsg = 208;    // archive %1 is for a newer WarpIN
            break;

            case WIERR_NOT_ENOUGH_MEMORY:
                ulMsg = 214;
            break;

            default:
                aStrings[1]._itoa10( irc, Locals._cThousands);
                throw FEFatalErrorExcpt(Locals,
                                        213,       // error %2 opening %1
                                        aStrings,
                                        2);
        }
        // error opening archive:
        throw FEFatalErrorExcpt(Locals, ulMsg, aStrings, cpsz);
    }
    else
    {
        // WPI file opened OK: call back-end to get more info

        _ustrArcFilename = ustrArcFilename;

        _pArcHeader = _Arc.getArcHeader();

        // _pPackHeadersList = _Arc.getPackList();
    }
}

/*
 *@@ ~FEArchive:
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

FEArchive::~FEArchive()
{
    delete _pPackagesList;
}

/*
 *@@ CreatePackages:
 *      little helper function called from FEArchive::ParseScript
 *      to create packages from script package declarations.
 *
 *      This recurses for groups.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

VOID FEArchive::CreatePackages(list<FEPckDeclBase*> &DeclarationsList,
                               FEArcPackageGroup *pParentGroup) // in: parent group, NULL unless we're recursing
{
    // static level = 0;

    _Locals.IncIndent(+4);

    // go thru all declarations given to us
    list<FEPckDeclBase*>::iterator  DeclStart = DeclarationsList.begin(),
                                    DeclEnd   = DeclarationsList.end();
    for (;
         DeclStart != DeclEnd;
         ++DeclStart)
    {
        FEPckDeclBase *pDeclThis = *DeclStart;

        if (pDeclThis->_Type == GROUP)
        {
            // declaration is for group: create a group then
            // and recurse for subpackages/-groups
            FEGroupDecl *pGroupDeclThis = (FEGroupDecl*)pDeclThis;

            FEArcPackageGroup *pGroupNew = new FEArcPackageGroup(*this,
                                                                 pParentGroup, // initially NULL
                                                                 *pGroupDeclThis);
            _Locals.Log("Created group \"%s\"",
                        pGroupDeclThis->_ustrTitle.GetBuffer());

            _pPackagesList->push_back(pGroupNew);

            // recurse!!
            CreatePackages(pGroupDeclThis->_DeclarationsList,
                           pGroupNew);
        }
        else
        {
            // regular package:
            FEPckDeclBase *pPckDeclThis = (FEPckDeclBase*)pDeclThis;

            FEArchive *pArchive = this;

            // check if it's external
            if (pPckDeclThis->_Type != INTERNAL)
            {
                // yes:
                if (pPckDeclThis->_Type != REQEXTERNAL)
                {
                    // external, but not required:

                    // (cbo 2000-05-23) prepend path name for checking existence
                    ustring ustrArchiveName = pPckDeclThis->_ustrExternalArchive;
                    PrependArchivePath(ustrArchiveName);

                    string str;
                    str.assignUtf8(_Locals._pCodecProcess, ustrArchiveName);

                    // not required: check for existence first,
                    // because the constructor throws exceptions...
                    if (access(str.c_str(), 0) != 0)
                        // does not exist:
                        // go for next package
                        continue;
                }

                // (remark 2000-05-23) use the pure archive name here, not
                // the one with the path prepended: The path is prepended
                // in the constructor while this pure archive name is
                // stored in _szArcFilename
                pArchive = new FEArchive(_Locals,
                                         pPckDeclThis->_ustrExternalArchive,
                                         this); // FEArchive *pReferencedFrom
            }

            // create a package then
            FEArcPackagePck *pPckNew = new FEArcPackagePck(*pArchive,
                                                           pParentGroup, // initially NULL
                                                           *pPckDeclThis);

            _Locals.Log("Created package \"%s\"",
                        pPckDeclThis->_ustrID.GetBuffer());

            _pPackagesList->push_back(pPckNew);
        }
    }

    _Locals.IncIndent(-4);

}

/*
 *@@ ParseScript:
 *      this parses the script and sets _pScript to a
 *      FEScriptBase instance (one of the subclasses)
 *      by calling FEScriptBase::CreateScript.
 *
 *      If that doesn't throw, we create packages from
 *      the package declarations in the script by calling
 *      FEArchive::CreatePackages. That's what we need
 *      the callback param for.
 *
 *      See fe_archive.cpp for an introduction.
 *
 *      Note: Only call this for the "root" (internal)
 *      archive, never for external archives.
 *
 *      pConfirmRexxAllowed gets passed to FEScriptBase::CreateScript
 *      and will get called if there's REXX code in the script.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 *@@changed V0.9.12 (2001-05-31) [umoeller]: added REXX confirmation
 *@@changed V0.9.15 (2001-08-26) [umoeller]: fixed excessive allocations for script buffer
 *@@changed V0.9.20 (2002-07-03) [umoeller]: added REXX instance
 */

VOID FEArchive::ParseScript(FERexx &Rexx)       // in: REXX instance
{
    // get the installation profile from the archive
    PCSZ pcszProfileTemp;
    if (!(pcszProfileTemp = _Arc.getScript()))
        throw FEFatalErrorExcpt(_Locals, 103);

    // create FEScriptBase subclass instance depending
    // on script format and parse the entire thing
    if (!(_pScript = FEScriptBase::CreateScript(pcszProfileTemp,
                                                _Locals,
                                                Rexx)
       ))
        throw FEFatalErrorExcpt(_Locals, 202);    // internal script error

    _Locals.Log("Creating packages...");

    // now go thru all the script's package declarations and
    // create packages from that...
    CreatePackages(_pScript->_DeclarationsList,     // initial list
                   NULL);                            // group (initially NULL)
            // this recurses!

    // resolve package requirements
    ResolveRequirements();
}

/*
 *@@ ExtractArchivePath:
 *      extracts any path information from the passed file name
 *      and stores it in _strArchivePath.
 *
 *      The path is also assigned to the environment variable
 *      WI_ARCHIVE_PATH so REXX script may access it.
 *      This environment assignment is put in _strArchivePathEnv
 *      for reasons I cannot see.
 *
 *      The passed in archive name is stripped for its path.
 *
 *@@added V0.9.3 (2000-05-23) [cbo]
 *@@changed V0.9.13 (2001-06-09) [bird]: rewritten, this failed with various path/filename combinations
 */

VOID FEArchive::ExtractArchivePath(ustring& ustrArchiveName)
{
    CHAR    szArchiveName[CCHMAXPATH];

    string strTemp;
    strTemp.assignUtf8(_Locals._pCodecProcess, ustrArchiveName);

    if (!DosQueryPathInfo(strTemp.c_str(),
                          FIL_QUERYFULLNAME,
                          szArchiveName,
                          sizeof(szArchiveName)))
    {
        PSZ     pSlash,
                pFore = strrchr(szArchiveName, '/');
        if (    (!(pSlash = strrchr(szArchiveName, '\\')))
             || (pSlash < pFore)
           )
            pSlash = pFore;

        if (!pSlash)
        {
            // urg! this should _never_ happen!!!
            throw FEFatalErrorExcpt(_Locals,
                                    277,            // V0.9.19 (2002-04-14) [umoeller]
                                    &ustrArchiveName, 1);
        }

        // make filename and archive path.
        ustrArchiveName.assignCP(_Locals._pCodecProcess, ++pSlash);
        *pSlash = '\0';
        _Locals._ustrArchivePath.assignCP(_Locals._pCodecProcess, szArchiveName);

        // put the archive path into the environment variable WI_ARCHIVE_PATH,
        // for use of any REXX scripts
        _Locals._strArchivePathEnv = "WI_ARCHIVE_PATH=";
        _Locals._strArchivePathEnv.appendUtf8(_Locals._pCodecProcess,
                                              _Locals._ustrArchivePath);
        putenv(_Locals._strArchivePathEnv.c_str());
    }
    else
        throw FEFatalErrorExcpt(_Locals,
                                277,            // V0.9.19 (2002-04-14) [umoeller]
                                &ustrArchiveName, 1);

}

/*
 *@@ PrependArchivePath:
 *      prepends the archive path in _strArchivePath to the
 *      passed archive file name, thus making it a fully
 *      qualified file name.
 *
 *@@added V0.9.3 (2000-05-23) [cbo]
 */

VOID FEArchive::PrependArchivePath(ustring& ustrArchiveName)
{
    ustrArchiveName = _Locals._ustrArchivePath + ustrArchiveName;
}

/*
 *@@ ResolveRequirements:
 *      this gets called by FEArchive::ParseScript after
 *      FEArchive::CreatePackages (so that all packages
 *      have been parsed) to resolve package requirements.
 *
 *      This step is only the prerequisite for checking
 *      dependencies; this does not check dependencies
 *      yet because FEArchive doesn't know about the
 *      database.
 *
 *      In the script, package requirements can be specified
 *      either by package index or by full five- or six-part
 *      package ID. For all packages, we will now initialize
 *      the FEPackageBase::_logRequiresIDs member, which hasn't
 *      been initialized yet.
 *
 *      Since package requirements can also be specified by
 *      package index in the script, we can only resolve this
 *      after all packages have been parsed.
 *
 *      If a full package ID has been specified, this will
 *      simply be copied.
 *
 *      If a package index has been specified, that package's
 *      ID will be copied.
 *
 *      Throws:
 *      -- FEFatalErrorExcpt if an invalid index has been
 *         specified.
 *
 *@@added V0.9.1 (2000-01-07) [umoeller]
 */

VOID FEArchive::ResolveRequirements()
{
    // go thru all packages of this archive
    list<FEArcPackageBase*>::iterator PckStart = _pPackagesList->begin(),
                                      PckEnd = _pPackagesList->end();
    for (; PckStart != PckEnd; ++PckStart)
    {
        // only for "real" packages; groups have
        // no dependencies:

        // V0.9.18 (2002-02-06) [umoeller]
        DYNAMIC_CAST(FEArcPackagePck, pPckPckThis, *PckStart);

        // FEArcPackageBase *pPckThis = *PckStart;
        /// FEArcPackagePck *pPckPckThis;
        // if ( (pPckPckThis = pPckThis->IsPackage()) )

        if (pPckPckThis)
        {
            // decode logger strings
            PSZ     pszLogStart = pPckPckThis->_pDecl->_logRequiresStrings._pabLogString,
                    pszLogThis = pszLogStart;
            int     iCount = 0;

            while (pszLogThis < pszLogStart + pPckPckThis->_pDecl->_logRequiresStrings._cbLogString)
            {
                ULONG   cbLogThis = strlen(pszLogThis);
                ULONG   ulIndex = atoi(pszLogThis);

                if (ulIndex)
                {
                    // decimal: copy from package with that index
                    const FEArcPackagePck *pPckSource;
                    if (pPckSource = FindArcPackage(ulIndex))
                    {
                        pPckPckThis->AppendRequiresID(_Locals,
                                                      pPckSource->_pDecl->_ustrID);
                        // pPckPckThis->_logRequiresIDs.Append(pPckSource->_pDecl->_strID);
                    }
                    else
                    {
                        ustring strError;
                        strError._printf("%u", ulIndex);
                        throw FEFatalErrorExcpt(_Locals,
                                                257, // Invalid package index "%1" specified with REQUIRES attribute.
                                                &strError, 1);
                    }
                }
                else
                {
                    // not decimal: simply copy ID
                    // pPckPckThis->_logRequiresIDs.Append(pszLogThis);
                    ustring ustr;
                    ustr.assignUtf8(pszLogThis);
                    pPckPckThis->AppendRequiresID(_Locals,
                                                  ustr);
                }

                pszLogThis += cbLogThis + 1;    // go beyond null byte
                ++iCount;
            }
        }
    }
}

/*
 *@@ FindArcPackage:
 *      returns the package with the specified index
 *      from the archive or NULL if not found.
 *
 *      This returns one of the items from the
 *      _PackagesList member.
 *
 *@@added V0.9.1 (2000-01-07) [umoeller]
 */

FEArcPackagePck* FEArchive::FindArcPackage(ULONG ulIndex)
{
    list<FEArcPackageBase*>::iterator PckStart = _pPackagesList->begin(),
                                      PckEnd = _pPackagesList->end();
    for (; PckStart != PckEnd; ++PckStart)
    {
        // FEArcPackageBase *pPckThis = *PckStart;
        // FEArcPackagePck *pPckPckThis;

        // V0.9.18 (2002-02-06) [umoeller]
        DYNAMIC_CAST(FEArcPackagePck, pPckPckThis, *PckStart);

        if (    (pPckPckThis)
             && (pPckPckThis->_pDecl->_ulIndexInArchive == ulIndex)
           )
            return pPckPckThis;
    }

    return NULL;
}

/*
 *@@ FindPackHeader:
 *      returns the WIPackHeader from the member WIArchive
 *      with the specified package index or NULL if not
 *      found.
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 */

WIPackHeader* FEArchive::FindPackHeader(ULONG ulIndex)
{
    list<WIPackHeaderLI*> *pPackHeadersList = _Arc.getPackList();
    list<WIPackHeaderLI*>::iterator start = pPackHeadersList->begin(),
                                    end = pPackHeadersList->end();
    for (; start != end; ++start)
    {
        WIPackHeader *pThis = (**start)._p;

        if (pThis->number == ulIndex)
            return pThis;
    }

    return NULL;
}

/*
 *@@ IsInArchive:
 *      this checks whether the package with the
 *      specified five- or six-part package ID is
 *      in the archive.
 *
 *      If so, the corresponding item from the
 *      _PackagesList member is returned and
 *      *pulComp is set to the return value of
 *      FEPackageID::Compare. If the IDCOMP_THISOLDER
 *      flag is set, that means that the package in the
 *      archive has an insufficient version number
 *      compared to pszPckID.
 *
 *      If not, NULL is returned.
 *
 *      Note that pszID is understood as the
 *      _minimum_ package ID. If any higher
 *      version is in the archive, this is
 *      also returned.
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.18 (2002-02-06) [umoeller]: now using FEPackageID for speed
 */

FEArcPackagePck* FEArchive::IsInArchive(const FEPackageID &PckID,
                                        PULONG pulComp) // out: result of FEPackageID::Compare;
                                                        // ptr can be NULL if not desired
                            const
{
    // FEPackageID     PckIDSearch(_pLocals,
    //                             pszPckID, __FILE__, __LINE__,
    //                             FALSE);    // don't allow short format
            // this may throw a FEConstructorExcpt

    list<FEArcPackageBase*>::iterator PckStart = _pPackagesList->begin(),
                                      PckEnd = _pPackagesList->end();
    for (; PckStart != PckEnd; ++PckStart)
    {
        // FEArcPackageBase *pPckThis = *PckStart;
        // FEArcPackagePck *pPckPckThis;
        // if (pPckPckThis = pPckThis->IsPackage())

        // V0.9.18 (2002-02-06) [umoeller]
        DYNAMIC_CAST(FEArcPackagePck, pPckPckThis, *PckStart);

        if (pPckPckThis)
        {
            // get package ID for the one in the archive
            // FEPackageID     PckIDThis(_pLocals,
            //                           pPckPckThis->_pDecl->_strID.c_str(),
            //                           __FILE__, __LINE__,
            //                           FALSE);    // don't allow short format
                    // this may throw a FEConstructorExcpt

            // compare with the one to search for
            ULONG ulComp = pPckPckThis->_PckID.Compare(PckID);
            if (ulComp & IDCOMP_SAMEPCK)
            {
                // store result
                if (pulComp)
                    *pulComp = ulComp;
                // return this archive
                return pPckPckThis;
            }
        }
    }

    return NULL;
}

/*
 *@@ FindPageInfo:
 *      returns the FEPageInfo from the script with the
 *      specified index, or throws an FEFatalErrorExcpt
 *      if no such page exists.
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 *@@changed V0.9.20 (2002-07-06) [umoeller]: moved here from FEInstallEngine
 */

FEPageInfo* FEArchive::FindPageInfo(LONG lIndex)
{
    list<FEPageInfo*>::iterator PageStart = _pScript->_PageInfoList.begin(),
                                PageEnd = _pScript->_PageInfoList.end();
    for (; PageStart != PageEnd; ++PageStart)
    {
        FEPageInfo *p = *PageStart;
        if (p->_lPageIndex == lIndex)
            // page found:
            return p;
    }

    ustring str;
    str._itoa10(lIndex, _Locals._cThousands);
    throw FEFatalErrorExcpt(_Locals,
                            267, // Cannot find page %1 in the installation script.
                            &str, 1);

    return NULL;
}


