/**
 * $Id: wiarchive.h,v 1.10 2001/07/01 20:49:03 jens Exp $
 *
 * Archive handling class for WarpIN, WIArchive (header file)
 *
 * This file Copyright (C) 1999-2000 Jens B&auml;ckman, Ulrich M&ouml;ller.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, in version 2 as it comes in the COPYING
 * file of this distribution.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

// Since reading code is boring, I give you this. Laugh, or I'll rip your
// lungs out and say nasty things about your pets.
//
// Two strings walk into a bar. The bartender says, "What'll it be?" The
// first string says, "I'll have a beer.quag fulk boorg jdk^CjfdLk jk3s
// d#f67howe%^U r89nvy~~owmc63^Dz x.xvcu" The second says, "You'll have
// to excuse my friend, he's not null-terminated."
//
// That's a cool one. (UM).

#ifndef WIARCHIVE_H
#define WIARCHIVE_H
#include "wiarchive2\wixp.h"
#include "wixp.h"

#include <string.h>
#include <time.h>

#if defined(__IBMCPP__) || defined(XP_WIN)
    #include <sys\utime.h>
#else
    #include <utime.h>
#endif

#include <list>

// STL uses namespaces in compilers that support them
#if !defined(XP_OS2_VACPP) && !defined(XP_OS2_EMX)
    #include <fstream>
    using namespace std;
#else
    #include <fstream.h>
#endif

// Unlink is defined in different places for different compilers
#ifdef XP_UNIX
    #include <unistd.h>
#else
    #include <io.h>
#endif

// System call: mkdir();
#ifdef XP_WIN_BCC
    #include <dir.h>
#endif

// The following is needed for mkdir with VAC++
#ifdef XP_OS2_VACPP
    #include <direct.h>
#endif

// These header files are for getting/setting file time
#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#include "bzlib.h"

#include "setup.h"

// Some compiler does not define the boolean values

// ----- WIArchive constants ------------------------------------------------
const short WIARCHIVE_REVISION = 3;
const short WIARCHIVE_REVISION_NEEDED = 2;

const short WIARCHIVE_OS_OS2   = 1;
const short WIARCHIVE_OS_UNIX  = 2;
const short WIARCHIVE_OS_WIN32 = 3;

const long  WIARCHIVE_BUFFERSIZE = 65536;

// ----- WIArchive control structures definitions ---------------------------
class WIHeader {
    public:
        // ----- Public methods
        WIHeader();
        WIHeader & operator=(const WIHeader &);
        void read(fstream &);
        bool verify();
        void write(fstream &);

        // ----- Public member variables
        short  revisionNeeded;      // Minimum WarpIN version required
        char   path[256];           // Installation path

        char   dummy[256];          // Empty space for compatibility
        short  revision;            // Application revision
        short  os;                  // Operating system to install to
        short  packs;               // Number of packages in archive
        short  origScriptSize;      // Size of installation script
        short  compScriptSize;      // Size of compressed script
        long   extendedDataSize;    // Extended data

    private:
        // ----- Private member variables
        unsigned char v1, v2, v3, v4;   // Archive verification data
};

class WIPackHeader {
    public:
        // ----- Public methods
        WIPackHeader();
        void read(fstream &);
        void write(fstream &);

        short  number;      // Package number
        short  files;       // Number of files in package
        short  oldFiles;    // Number of files in package before changes
                            // (this value is not stored in file)
        long   pos;         // Position in archive
        long   origSize;    // Size of package (original)
        long   compSize;    // Size of package (compressed)
        char   name[32];    // Name of package
};

class WIFileHeader {
    public:
        // ----- Public methods
        WIFileHeader();
        void read(fstream &);
        bool verify();
        void write(fstream &);

        // ----- Public member variables
        short method;               // Compression method used
        short package;              // Which package it belongs to
        long  origSize;             // Size of file (original)
        long  compSize;             // Size of file (compressed)
        unsigned long crc;          // File CRC checksum
        char  name[256];            // Filename
        time_t lastwrite;           // File last write date/time
        time_t creation;            // File creation date/time
                                    // we don't need last access date/time
        char  extended;             // Size of extended information (if any)
    private:
        // ----- Private methods
        short calculateChecksum();

        // ----- Private member variables
        unsigned short magic;       // Must be WIFH_MAGIC for security
        short checksum;             // Header checksum (add all public bytes)
};

class WIFile {
    public:
        // ----- Public methods
        WIFile(const char *n = NULL, const char *e = NULL, short s = 0) {
            name = n;
            extra = e;
            package = s;
        }

        // ----- Public member variables
        const char *name;
        const char *extra;
        short       package;
};

// ----- Callback function definition and related stuff ---------------------
// The callback function should accept three different parameters:
//      short         mode   one of the CBM_* flags below
//      unsigned long param  meaning depends on CBM_* value
//      WIFileHeader* pwifh  the file header of the file in question
typedef int (FNWICALLBACK)(short, unsigned long, WIFileHeader*);
typedef FNWICALLBACK *PFNWICALLBACK;

/*
 *@@ PFNFORALLFILESCALLBACK:
 *      callback prototype for WIArchive::forAllFiles
 */
typedef int (FNFORALLFILESCALLBACK)(WIFileHeader*, unsigned long);
typedef FNFORALLFILESCALLBACK *PFNFORALLFILESCALLBACK;

// ----- Callback messages ("mode" parameter)
const int CBM_HELLO        = 1;
    // This is the first message sent when you set a new callback function,
    // mostly as an opportunity to check that your CBF works like it should.
    //
    // param: NULL
    // pwift: NULL
    // Return: Ignored
const int CBM_PERCENTAGE   = 2;
    // Sent when any progress in compressing/decompressing has been made. The
    // progress value is different depending on the operational mode. If we
    // compress a file, this is the number of bytes precessed of the current
    // file. In case decompression is commencing, this is the number of bytes
    // decompressed of the current package.
    //
    // param: Progress value
    // pwifh: Current file header
    // Return: Ignored.
const int CBM_UPDATING     = 3;
    // After compression has finished on a file, this message will be sent to
    // the frontend for updating of the achieved compression and such things.
    //
    // param: Size of file after compression
    // pwifh: Current file header
    // Return: Ignored.
const int CBM_NEXTFILE     = 4;
    // This comes just before the WIArchive class attempts to open a new
    // output file for writing. This allows the front-end to do two things:
    //   a) update the "current file" display
    //   b) check for whether that file exists already:
    //      -- if not, we return CBRC_PROCEED;
    //      -- if it does exist, we ask the user for whether the file may be
    //         overwritten; if so, we try to delete it. Otherwise, we return
    //         CBRC_SKIP, and the back-end does not touch the file.
    // This new approach (Alpha #3) is required because sometimes the target
    // file which already exists is _locked_, e.g. if it's a system DLL in use.
    // We then need to do a DosReplaceModule, which we cannot do in the back-end
    // if we want to keep it platform-independent. Sorry for the hassle.
    //
    // param: NULL
    // pwifh: Current file header
    // Return: CBRC_PROCEED or CBRC_SKIP
const int CBM_ERR_MEMORY   = 5;
    // Somehow, there wasn't enough memory to do something.  Not even in swap
    // space?  Well...  We're here, anyway.  This is a fatal error - TERMINATE!
    // Do not return when recieving this message!
    //
    // param: NULL
    // pwfih: NULL
    // Return: TERMINATE, no return
const int CBM_ERR_WRITE    = 6;
            // error writing into output file; this is most probable when the
            // target disk is full.
            // "param" then has one of the CBREC_* values below, which identify
            // whether the error is recoverable. Depending on that value,
            // we may return one of the CBRC_* values.
const int CBM_ERR_READ     = 7;
    // Error when reading from file/archive. One possible reason for this message
    // is that the archive is broken and contains faulty file headers.
    //
    // param: One of the CBREC_* values
    // pwfih: NULL
    // Return: CBRC_ABORT, CBRC_RETRY or CBRC_IGNORE
const int CBM_ERR_COMPLIB  = 8;
    // While compressing/decompressing, a strange error occurred. This usually
    // means the archive is broken, sometimes badly. There is no use in trying
    // to complete the operation - let's quit this and take a cup of coffee.
    // Maybe even a beer. I personally recommend a very tasty ale, Usher's Ruby
    // Ale. You just got to try this. Oh, I'm getting a little bit offtopic...
    //
    // param: Error value from BZ2 compression library.
    // pwifh: Current file header.
    // Return: Ignored.
const int CBM_ERR_CRC      = 9;
    // During decompression of a file, the calculated checksum did not match
    // the stored value. This usually means a broken archive when downloading
    // it or a bad media error. Anyway, the file might be really bad.
    //
    // param: NULL
    // pwifh: Current file header
    // Return: Ignored.

// ----- Callback error recovery flags
const int CBREC_CANCEL           = 1;   // Severe error - Cancel only
const int CBREC_RETRYCANCEL      = 2;   // Cancel/Retry
const int CBREC_ABORTRETRYIGNORE = 3;   // Abort/Retry/Skip file

// ----- Callback return values
const int CBRC_ABORT   = 1;
const int CBRC_RETRY   = 2;
const int CBRC_IGNORE  = 3;
const int CBRC_PROCEED = 4;
const int CBRC_SKIP    = 5;

const int WIERR_OUTDATED_ARCHIVE  = 1;
const int WIERR_INVALID_HEADER    = 2;

// ----- WIArchive class definition -----------------------------------------
class WIArchive {
    public:
        // ----- Constant enumerators
        enum access { READ, WRITE };

        // ----- Public methods ---------------------------------------------
        WIArchive();
        ~WIArchive();
        bool addFile(const char *, const char *, short);
        void addPackage(short, const char *);
        bool close(bool = true);
        short forAllFiles(WIPackHeader *, PFNFORALLFILESCALLBACK, unsigned long);
        bool open(const char *, access = WRITE);
        void removeFile(short, const char *);
        void removePackage(short);
        bool unpack(short);

        // ----- Getters ----------------------------------------------------
        list<WIFile *>       *getFileList()  { return &fileList; }
        WIHeader             *getHeader()    { return &header;   }
        list<WIPackHeader *> *getPackList()  { return &packList; }
        const char           *getScript()    { return script;    }
        int                   getLastError() { return lastError; }

        // ----- Setters ----------------------------------------------------
        void setCallback(PFNWICALLBACK);
        void setHeader(const WIHeader &h)   { header = h; }
        void setScript(const char *s)       { script = s; }

    protected:
        // ----- Protected methods ------------------------------------------
        void cleanup();
        bool compress(WIFileHeader &, fstream &, fstream &);
        bool expand (WIFileHeader &, fstream &, fstream &, WIPackHeader &, long);
        bool extract(WIFileHeader &, fstream &, fstream &, WIPackHeader &, long);
        void makeDirectories(const char *);
        void readFiles(fstream &);
        bool readHeader(fstream &);
        void readPackages(fstream &);
        void store(WIFileHeader &, fstream &, fstream &);
        bool update();
        void writeHeader(fstream &);
        void writePackages(fstream &);

        // ----- Protected member variables ---------------------------------
        WIHeader header;            // Archive header
        char     archiveName[256];  // Name of archive file
        char     tempName[256];     // Name of temporary archive file
        access   archiveAccess;     // Which mode the archive is opened in
        bool     modified;          // Tells if the archive has been modified
        fstream  archive;           // File stream for archive
        fstream  oldArchive;        // File stream for old version of archive
        int      archivePos;        // Where in the file this archive begins
        char    *extendedData;      // Some extra archive information
        const char *script;         // Installation script
        unsigned long crcTable[256];// Checksum code lookup table
        PFNWICALLBACK callback;     // Callback function
        int lastError;              // The last error value
        list <WIFile *> fileList;   // Contains a list of all the archived files
        list <WIFile *> todoList;   // All the files we need to add to this archive
        list <WIPackHeader *>packList;  // List of all packages in archive
};

#endif // WIARCHIVE_H

