
/*
 * fe_base.h:
 *      header file for fe_base.cpp. See remarks there.
 *
 *@@include #include "os2.h"
 *@@include #include "helpers\nls.h"
 *@@include #include "base\bs_base.h"
 *@@include #include "base\bs_string.h"
 *@@include #include "base\bs_list.h"
 *@@include #include "base\bs_errors.h"
 *@@include #include "engine\fe_base.h"
 */

/*
 *      Copyright (C) 1998-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#ifndef WARPIN_FEBASE_HEADER_INCLUDED
    #define WARPIN_FEBASE_HEADER_INCLUDED

    class BSFileLogger;
    class BSExecute;
    class BSKillProcess;
    class FEJobBase;

    /* ******************************************************************
     *
     *  FEInstallVar declaration
     *
     ********************************************************************/

    /*
     *@@ FEInstallVar:
     *      representation of an install variable.
     *      Presently, an install variable is a
     *      keyword with an assigned boolean that
     *      is either YES or NO. These things are
     *      now used for determining whether
     *      configuration is to be performed.
     *
     *      Presently, only predefined variables
     *      exist. At a later point, we might allow
     *      using any variables in the script to
     *      make system configuration more flexible.
     *
     *      The predefined install variables are
     *      part of FELocals and created and owned
     *      by FELocals::FELocals. There is no
     *      global list of install variables in
     *      the engine; instead, each FEPckDeclBase
     *      holds a list of references to those
     *      variables that are used in the script.
     *
     *@@added V0.9.18 (2002-03-08) [umoeller]
     */

    class FEInstallVar : public BSRoot
    {
        public:
            DECLARE_CLASS(FEInstallVar);

            const string    _strVarName;

        private:
            ustring         _ustrDescription;
            BOOL            _fIsPublic;
            BOOL            _fValue;

        public:

            /*
             *@@ FEInstallVar:
             *      the constructor.
             */

            FEInstallVar(const char *pcszVarNameA,      // in: ASCII variable name
                         const char *pcszDescriptionU,  // in: UTF-8 variable description
                         BOOL fIsPublic)
                : BSRoot(tFEInstallVar),
                  _strVarName(pcszVarNameA)
            {
                _ustrDescription.assignUtf8(pcszDescriptionU);
                _fIsPublic = fIsPublic;
                _fValue = TRUE;
            }

            ~FEInstallVar()
            {}

            inline BOOL IsPublic() const
            {
                return _fIsPublic;
            }

            inline BOOL GetValue() const
            {
                return _fValue;
            }

            inline void SetValue(BOOL f)
            {
                _fValue = f;
            }

            inline const ustring& GetDescription() const
            {
                return _ustrDescription;
            }
    };

    /* removed V0.9.20 (2002-07-03) [umoeller]
    typedef enum _enShowMessage
    {
        MSG_WARNING_OK,                 // just a warning, continue ("OK")
        MSG_ERROR_ABORT,                // error, show "Cancel" only
        MSG_ERROR_ABORTRETRYIGNORE,     // need a decision (abort, retry, ignore);
                                        // default is abort
        MSG_ERROR_RETRYCANCEL,          // need a decision (retry, cancel)
                                        // default is cancel
        MSG_CONFIRM_YESNOCANCEL_DEFYES, // confirmation with yes, no, cancel;
                                        // default is yes
        MSG_CONFIRM_YESNO_DEFYES        // confirmation with yes, no;
                                        // default is yes
    } enShowMessage;
    */

    #define MSG_WARNING_OK                  MB_OK
    #define MSG_ERROR_ABORT                 MB_CANCEL
    #define MSG_ERROR_ABORTRETRYIGNORE      MB_ABORTRETRYIGNORE
    #define MSG_ERROR_RETRYCANCEL           MB_RETRYCANCEL
    #define MSG_CONFIRM_YESNOCANCEL_DEFYES  MB_YESNOCANCEL
    #define MSG_CONFIRM_YESNO_DEFYES        MB_YESNO

    // flags for the "File exists" fields in WPIGLOBALS:
    #define FE_PROMPT      0        // 0: prompt
    #define FE_SKIP        1        // 1: skip
    #define FE_OVERWRITE   2        // 2: overwrite

    /*
     *@@ FELocals:
     *      abstract base class for the state of the system while
     *      the front-end is running.
     *
     *      This is an unsorted mess for many things. Basically
     *      this contains things which are needed by the engine
     *      in both "install" and "database" mode, while
     *      FEInstallEngine is only created in "install" mode.
     *
     *      Since the constructor is protected, and since this
     *      has several pure virtual methods, the class cannot
     *      be used as such. Since this is used to encapsulate
     *      GUI-specific things, the GUI is responsible for
     *      deriving a class from this and implementing the
     *      following methods:
     *
     *      --  ShowMessage
     *
     *      --  Sleep
     *
     *      See @class_rels for an overview of how the front-end
     *      classes interact.
     *
     *      WARNING: You must create an instance of a subclass
     *      of this on thread 1, or the internal anchor block
     *      cannot be initialized properly.
     *
     *@@added V0.9.14 (2001-07-26) [umoeller]
     *@@changed V0.9.18 (2002-03-03) [umoeller]: renamed from FESystem, made class abstract
     */

    class FELocals : public BSRoot
    {
        public:
            DECLARE_CLASS(FELocals);

            // anchor block of WarpIN thread 1;
            // needed by many parts, among them the
            // database
            HAB                 _habThread1;

            // the following flags are for when a file exists:
                                        // 0: prompt
                                        // 1: skip
                                        // 2: overwrite
            ULONG               _ulIfSameDate,       // default: 2
                                _ulIfExistingOlder,  // default: 2
                                _ulIfExistingNewer;  // default: 0

            string              _strDefaultAppsPath,
                                _strDefaultToolsPath;
                    // moved these here from WPIGLOBALS
                    // V0.9.19 (2002-04-14) [umoeller]

        protected:
            PVOID               _pvMessageFile;
                        // PTMFMSGFILE really, after having opened the
                        // message file in the constructor

            ustring             _ustrWarpINPath;
                        // WarpIN path

            // file name of log file; defaults to INSTALL.LOG in the
            // WarpIN directory.
            // This may be changed by the user in the GUI; if the
            // first byte is \0, logging is disabled.
            // made this a ustring too V0.9.20 (2002-07-03) [umoeller]
            ustring             _ustrLogFilename;

        public:

            /*
             *@@ QueryWarpINPath:
             *      returns the directory where WARPIN.EXE resides.
             *
             *@@added V0.9.18 (2002-03-08) [umoeller]
             */

            inline const ustring& QueryWarpINPath()
            {
                return _ustrWarpINPath;
            }

            /*
             *@@ QueryLogFilename:
             *      returns the log file name.
             *
             *@@added V0.9.20 (2002-07-03) [umoeller]
             */

            inline const ustring& QueryLogFilename()
            {
                return _ustrLogFilename;
            }

            /*
             *@@ SetLogFilename:
             *      sets the log file name.
             *
             *@@added V0.9.20 (2002-07-03) [umoeller]
             */

            inline void SetLogFilename(const ustring &ustr)
            {
                _ustrLogFilename = ustr;
            }

        public:
            CHAR                _cThousands;
                        // NLS thousands separator char (for log file)

            void*               _paEntities;
            ULONG               _cEntities;     // copied from constructor

            BOOL                _fConfigSysChanged,
                        // TRUE after CONFIG.SYS has been changed
                                _fWPSClassesChanged,
                        // TRUE after WPS class list has been changed
                                _fWPSObjectsCreated;
                        // TRUE after WPS objects have been created

            string              _strConfigSysBackup;
                        // backup file name of original CONFIG.SYS;
                        // made this a global variable V0.9.6 (2000-10-27) [umoeller]

            list<string*>       _TempFilenamesList;
                        // list of all temp files ever created while
                        // WarpIN was running, separated by '|'

            BSUniCodec          *_pCodecProcess;
                        // uconv object for current process codepage
                        // V0.9.18 (2002-03-08) [umoeller]
            BSUniCodec          *_pCodecGui;
                        // uconv object for GUI codepage
                        // moved this here from GUILocals
                        // V0.9.19 (2002-05-07) [umoeller]
            BSUniCodec          *_pCodec850;
                        // uconv object for getting TMF strings
                        // (always codepage 850 presently)

            BSFileLogger        *_pLogFile;

            // moved the following two here from FEArchive
            // V0.9.19 (2002-05-07) [umoeller]
            ustring             _ustrArchivePath;
                    // path for all archives (cbo 2000-05-23)
                    // now using ustring V0.9.19 (2002-04-14) [umoeller]
            string              _strArchivePathEnv;
                    // WI_ARCHIVE_PATH definition in environment
                    // for use with REXX scripts (using process codepage)

            FEInstallVar
                    _varConfigSys,
                    _varWPSClasses,
                    _varWPSObjects,
                    _varWriteProfiles,
                    _varClearProfiles,
                    _varExecutes,
                    _varDeExecutes,
                    _varKillProcesses;

        protected:
            FELocals(HAB habThread1,
                     CHAR cThousands,
                     void* paEntities,
                     ULONG cEntities);

        public:

            void GetSettingsFromEnvironment();

            /*
             *@@ ShowMessage:
             *      pure virtual function to be overridden
             *      by the GUI implementation subclass, which
             *      must display the given message.
             *
             *      Note that both strings are UTF-8 encoded.
             */

            virtual ULONG ShowMessage(const ustring &ustrTitle,
                                      const ustring &ustrMessage,
                                      ULONG fl) = 0;

            /*
             *@@ ConfirmRexxAllowed:
             *      pure virtual functin to be overridden
             *      by the GUI implementation subclass,
             *      which must return TRUE if REXX code
             *      is allowed by the user.
             *
             *@@added V0.9.20 (2002-07-03) [umoeller]
             */

            virtual BOOL ConfirmRexxAllowed(const ustring &ustrScript) = 0;

            /*
             *@@ OnJobSelectionChanged:
             *      pure virtual functin to be overridden
             *      by the GUI implementation subclass,
             *      which gets called whenever job selections
             *      change in the engine.
             */

            virtual VOID OnJobSelectionChanged(FEJobBase *pJob) = 0;

            /*
             *@@ Sleep:
             *      pure virtual function to be overridden
             *      by the GUI implementation subclass,
             *      which must sleep for the given time.
             */

            virtual VOID Sleep(ULONG ulMilliseconds) = 0;

            /*
             *@@ Execute:
             *      pure virtual function to be overridden
             *      by the GUI implementation subclass,
             *      which gets called from the engine for
             *      each executable which is run because of
             *      an EXECUTE tag.
             */

            virtual APIRET Execute(BSExecute *pExec) = 0;

            /*
             *@@ QueryGuiCodec:
             *      pure virtual function to be overridden
             *      by the GUI implementation subclass,
             *      which must return a reference to the
             *      GUI codec to be used for translating
             *      strings to the GUI.
             */

            virtual BSUniCodec& QueryGuiCodec() = 0;

    /* typedef BOOL APIENTRY FNEXECUTE(FEInstallEngine &Engine,
                                    BSExecute *pExec,
                                    BSFileLogger *pLogFile */
        public:
            virtual ~FELocals();

            APIRET GetMessage(ustring &strBuf,
                              ULONG ulMsgNumber,
                              ustring *paStrings = NULL,
                              ULONG ulTable = 0);

            ULONG MessageBox(ULONG ulTitleMsg,
                             ULONG ulMsgMsg,
                             ULONG fl,
                             ustring *paStrings = NULL,
                             ULONG cTable = 0);

            void OpenLogger();
            void Log(const char *pcszFormat,
                     ...);
            void IncIndent(int i);
            void CloseLogger();

            BOOL KillOneProcess(BSKillProcess *pKill);

            VOID GetCopyrightInfo(ustring &ustr,
                                  const string &strDatabaseFile);
    };

    #ifdef WIARCHIVE_INCLUDE

    /*
     *@@ FEFileError:
     *      little helper class for quickly generating
     *      file error information in both install and
     *      database mode.
     *
     *@@changed V0.9.1 (2000-02-06) [umoeller]: renamed from FILEEXISTS
     *@@changed V0.9.14 (2001-07-26) [umoeller]: renamed from GUIFILEERROR, added proper constructor
     */

    class FEFileError : public BSRoot
    {
        public:
            DECLARE_CLASS(FEFileError);

            FELocals        &_Locals;

            string          _strFilename;
                                        // fully q'd pathname of existing file
            FILESTATUS3     _fs3Existing;    // file status of existing file

            WIFileHeader*   _pwifh;          // file header in archive
            FDATE           _fDateHeader;    // date
            FTIME           _fTimeHeader;    // and time of *pwifh

            int             _iComp;          // < 0:  existing file is newer
                                            // == 0: existing file is same write date
                                            // > 0:  existing file is older

            FEFileError(FELocals &Locals,
                        const ustring *pstrBasePath,
                        WIFileHeader* pwifh);
    };

    #endif

    /* ******************************************************************
     *
     *  Exception declarations
     *
     ********************************************************************/

    /*
     *@@ FEConstructorExcpt:
     *      exception used when errors occur in constructors.
     *
     *@@changed V0.9.0 (99-11-06) [umoeller]: renamed from ConstructorExcpt
     */

    class FEConstructorExcpt : public BSExcptBase
    {
        public:
            FEConstructorExcpt(FELocals &Locals,
                               const char *strClass,
                               const char *pszSourceFile,
                               int iSourceLine,
                               int iMessage);
    };

    /*
     *@@ FEFatalErrorExcpt:
     *      if this is thrown, WarpIN will display an error message
     *      and exit.
     *
     *@@changed V0.9.0 (99-11-06) [umoeller]: renamed from FatalErrorExcpt
     */

    class FEFatalErrorExcpt : public BSExcptBase
    {
        public:
            FEFatalErrorExcpt(FELocals &Locals,
                              const ustring &ustrMessage);
            FEFatalErrorExcpt(FELocals &Locals,
                              int iMessage,
                              ustring *paStrings = NULL,
                              int iCount = 0);
    };

    /*
     *@@ FEEnvironmentExcept:
     *      exception thrown by GetEnvironment (warpin.cpp)
     *      if an invalid environment variable value
     *      has been encountered.
     */

    class FEEnvironmentExcept : public BSExcptBase
    {
        public:
            FEEnvironmentExcept(FELocals &Locals,
                                const char *pcszCommand,
                                const char *pcszValue);
    };

    /* ******************************************************************
     *
     *  Helpers
     *
     ********************************************************************/

    #ifdef WIARCHIVE_INCLUDE
        int wpiCompareFileDates(WIFileHeader* pHeader,
                                FILESTATUS3 *pfs3,
                                FDATE *pfdateLastWrite,
                                FTIME *pftimeLastWrite);

        void wpiCTime2FTime(time_t *pCTime,
                            FDATE *pfDate,
                            FTIME *pfTime);
    #endif

#endif
