////////////////////////////////////////////////////////////////
// File - GT64_DIAG.C
//
// o A simple diagnostics program that lets you access the
//   Galileo 64 registers and local memory. 
// o This program is meant to be used as an example for using the GT64_LIB.H API,
//   you may use it as a skeleton for your driver, or 'cut & paste' parts
//   of it into your device driver code.
// 
////////////////////////////////////////////////////////////////

#include "../../../include/windrvr.h"
#include "../lib/gt64_lib.h"
#include "../../../samples/shared/pci_diag_lib.h"
#include <stdio.h>

// input of command from user
static char line[256];

void GT64_EditReg(GT64_HANDLE hGT)
{
    struct 
    {
        CHAR *name;
        DWORD dwOffset;
        DWORD dwVal;
    } fields[160];
    struct
    {
        CHAR *name;
        int iPosition;
    } group[20];

    int cmd, cmd2;
    int i = 0;
    int j = 0;
    int iGroupNum;
    int iFieldsNum;
    BOOL f111 = hGT->gt64Ver==GT64_VERSION_64111;
    BOOL f120 = hGT->gt64Ver==GT64_VERSION_64120;
    BOOL f130 = hGT->gt64Ver==GT64_VERSION_64130;

    // CPU/Local Master Interface Configuration
    group[j].name ="CPU/Local Master Interface Configuration"; group[j++].iPosition = i;
    fields[i].name ="CPU/Local Master Interface Configuration"; fields[i++].dwOffset = 0x000;
    if (f120 || f130) 
    {
        fields[i].name ="Multi-GT Register"; fields[i++].dwOffset = 0x120;
    }
    // Processor Address Space
    group[j].name ="Processor Address Space"; group[j++].iPosition = i;
    fields[i].name ="RAS[1:0] Low Decode Address"; fields[i++].dwOffset = 0x008;
    fields[i].name ="RAS[1:0] High Decode Address"; fields[i++].dwOffset = 0x010;
    fields[i].name ="RAS[3:2] Low Decode Address"; fields[i++].dwOffset = 0x018;
    fields[i].name ="RAS[3:2] High Decode Address"; fields[i++].dwOffset = 0x020;
    fields[i].name ="CS[2:0] Low Decode Address"; fields[i++].dwOffset = 0x028;
    fields[i].name ="CS[2:0] High Decode Address"; fields[i++].dwOffset = 0x030;
    fields[i].name ="CS[3] & Boot CS Low Decode Address"; fields[i++].dwOffset = 0x038;
    fields[i].name ="CS[3] & Boot CS High Decode Address"; fields[i++].dwOffset = 0x040;
    fields[i].name ="PCI I/O Low Decode Address"; fields[i++].dwOffset = 0x048;
    fields[i].name ="PCI I/O High Decode Address"; fields[i++].dwOffset = 0x050;
    fields[i].name ="PCI Memory 0 Low Decode Address"; fields[i++].dwOffset = 0x058;
    fields[i].name ="PCI Memory 0 High Decode Address"; fields[i++].dwOffset = 0x060;
    fields[i].name ="Internal Space Decode"; fields[i++].dwOffset = 0x068;
    fields[i].name ="Bus Error Address Low Processor"; fields[i++].dwOffset = 0x070;
    fields[i].name ="Read Only 0"; fields[i++].dwOffset = 0x078;
    fields[i].name ="PCI Memory 1 Low Decode Address"; fields[i++].dwOffset = 0x080;
    fields[i].name ="PCI Memory 1 High Decode Address"; fields[i++].dwOffset = 0x088;
    if (f120 || f130)
    {
        fields[i].name ="PCI_1 I/O Low Decode Address"; fields[i++].dwOffset = 0x090;
        fields[i].name ="PCI_1 I/O High Decode Address"; fields[i++].dwOffset = 0x098;
        fields[i].name ="PCI_1 Memory 0 Low Decode Address"; fields[i++].dwOffset = 0x0a0;
        fields[i].name ="PCI_1 Memory 0 High Decode Address"; fields[i++].dwOffset = 0x0a8;
        fields[i].name ="PCI_1 Memory 1 Low Decode Address"; fields[i++].dwOffset = 0x0b0;
        fields[i].name ="PCI_1 Memory 1 High Decode Address"; fields[i++].dwOffset = 0x0b8;
        fields[i].name ="SCS[1:0]* Address Remap"; fields[i++].dwOffset = 0x0d0;
        fields[i].name ="SCS[3:2]* Address Remap"; fields[i++].dwOffset = 0x0d8;
        fields[i].name ="CS[2:0]* Remap"; fields[i++].dwOffset = 0x0e0;
        fields[i].name ="CS[3]* & Boot CS* Remap"; fields[i++].dwOffset = 0x0e8;
        fields[i].name ="PCI_0 IO Remap"; fields[i++].dwOffset = 0x0f0;
        fields[i].name ="PCI_0 Memory 0 Remap"; fields[i++].dwOffset = 0x0f8;
        fields[i].name ="PCI_0 Memory 1 Remap"; fields[i++].dwOffset = 0x100;
        fields[i].name ="PCI_1 IO Remap"; fields[i++].dwOffset = 0x108;
        fields[i].name ="PCI_1 Memory 0 Remap"; fields[i++].dwOffset = 0x110;
        fields[i].name ="PCI_1 Memory 1 Remap"; fields[i++].dwOffset = 0x118;
        //CPU Sync Barrier
        group[j].name ="CPU Sync Barrier"; group[j++].iPosition = i;
        fields[i].name ="PCI_0 Sync Barrier Virtual Register"; fields[i++].dwOffset = 0x0c0;
        fields[i].name ="PCI_1 Sync Barrier Virtual Register"; fields[i++].dwOffset = 0x0c8;
    }
    //DRAM and Device Address Space
    group[j].name ="DRAM and Device Address Space"; group[j++].iPosition = i;
    fields[i].name ="RAS[0] Low Decode Address"; fields[i++].dwOffset = 0x400;
    fields[i].name ="RAS[0] High Decode Address"; fields[i++].dwOffset = 0x404;
    fields[i].name ="RAS[1] Low Decode Address"; fields[i++].dwOffset = 0x408;
    fields[i].name ="RAS[1] High Decode Address"; fields[i++].dwOffset = 0x40c;
    fields[i].name ="RAS[2] Low Decode Address"; fields[i++].dwOffset = 0x410;
    fields[i].name ="RAS[2] High Decode Address"; fields[i++].dwOffset = 0x414;
    fields[i].name ="RAS[3] Low Decode Address"; fields[i++].dwOffset = 0x418;
    fields[i].name ="RAS[3] High Decode Address"; fields[i++].dwOffset = 0x41c;
    fields[i].name ="CS[0] Low Decode Address"; fields[i++].dwOffset = 0x420;
    fields[i].name ="CS[0] High Decode Address"; fields[i++].dwOffset = 0x424;
    fields[i].name ="CS[1] Low Decode Address"; fields[i++].dwOffset = 0x428;
    fields[i].name ="CS[1] High Decode Address"; fields[i++].dwOffset = 0x42c;
    fields[i].name ="CS[2] Low Decode Address"; fields[i++].dwOffset = 0x430;
    fields[i].name ="CS[2] High Decode Address"; fields[i++].dwOffset = 0x434;
    fields[i].name ="CS[3] Low Decode Address"; fields[i++].dwOffset = 0x438;
    fields[i].name ="CS[3] High Decode Address"; fields[i++].dwOffset = 0x43c;
    fields[i].name ="Boot CS Low Decode Address"; fields[i++].dwOffset = 0x440;
    fields[i].name ="Boot CS High Decode Address"; fields[i++].dwOffset = 0x444;
    fields[i].name ="Address Decode Error"; fields[i++].dwOffset = 0x470;
    //DRAM Configuration
    group[j].name ="DRAM Configuration"; group[j++].iPosition = i;
    if (f111)
    {
        fields[i].name ="DRAM Configuration"; fields[i++].dwOffset = 0x448;
    }
    else
    {
        fields[i].name ="SDRAM Bank0 Parameters"; fields[i++].dwOffset = 0x44c;
        fields[i].name ="SDRAM Bank1 Parameters"; fields[i++].dwOffset = 0x450;
        fields[i].name ="SDRAM Bank2 Parameters"; fields[i++].dwOffset = 0x454;
        fields[i].name ="SDRAM Bank3 Parameters"; fields[i++].dwOffset = 0x458;
    }
    //DRAM Parameters
    group[j].name ="DRAM Parameters"; group[j++].iPosition = i;
    fields[i].name ="DRAM Bank0 Parameters"; fields[i++].dwOffset = 0x44c;
    fields[i].name ="DRAM Bank1 Parameters"; fields[i++].dwOffset = 0x450;
    fields[i].name ="DRAM Bank2 Parameters"; fields[i++].dwOffset = 0x454;
    fields[i].name ="DRAM Bank3 Parameters"; fields[i++].dwOffset = 0x458;
    //Device Parameters
    group[j].name ="Device Parameters"; group[j++].iPosition = i;
    fields[i].name ="Device Bank0 Parameters"; fields[i++].dwOffset = 0x45c;
    fields[i].name ="Device Bank1 Parameters"; fields[i++].dwOffset = 0x460;
    fields[i].name ="Device Bank2 Parameters"; fields[i++].dwOffset = 0x464;
    fields[i].name ="Device Bank3 Parameters"; fields[i++].dwOffset = 0x468;
    fields[i].name ="Device Boot Bank Parameters"; fields[i++].dwOffset = 0x46c;
    if (f130)
    {
        // ECC
        group[j].name ="ECC"; group[j++].iPosition = i;
        fields[i].name ="ECC Upper_Data"; fields[i++].dwOffset = 0x480;
        fields[i].name ="ECC Lower Data"; fields[i++].dwOffset = 0x484;
        fields[i].name ="ECC From Memory"; fields[i++].dwOffset = 0x488;
        fields[i].name ="ECC Calculeted"; fields[i++].dwOffset = 0x48c;
        fields[i].name ="ECC Error Report"; fields[i++].dwOffset = 0x490;
    }
    //DMA Record
    group[j].name ="DMA Record"; group[j++].iPosition = i;
    fields[i].name ="Channel 0 DMA Byte Count"; fields[i++].dwOffset = 0x800;
    fields[i].name ="Channel 1 DMA Byte Count"; fields[i++].dwOffset = 0x804;
    fields[i].name ="Channel 2 DMA Byte Count"; fields[i++].dwOffset = 0x808;
    fields[i].name ="Channel 3 DMA Byte Count"; fields[i++].dwOffset = 0x80c;
    fields[i].name ="Channel 0 DMA Source Address"; fields[i++].dwOffset = 0x810;
    fields[i].name ="Channel 1 DMA Source Address"; fields[i++].dwOffset = 0x814;
    fields[i].name ="Channel 2 DMA Source Address"; fields[i++].dwOffset = 0x818;
    fields[i].name ="Channel 3 DMA Source Address"; fields[i++].dwOffset = 0x81c;
    fields[i].name ="Channel 0 DMA Destination Address"; fields[i++].dwOffset = 0x820;
    fields[i].name ="Channel 1 DMA Destination Address"; fields[i++].dwOffset = 0x824;
    fields[i].name ="Channel 2 DMA Destination Address"; fields[i++].dwOffset = 0x828;
    fields[i].name ="Channel 3 DMA Destination Address"; fields[i++].dwOffset = 0x82c;
    fields[i].name ="Channel 0 Next Record Pointer"; fields[i++].dwOffset = 0x830;
    fields[i].name ="Channel 1 Next Record Pointer"; fields[i++].dwOffset = 0x834;
    fields[i].name ="Channel 2 Next Record Pointer"; fields[i++].dwOffset = 0x838;
    fields[i].name ="Channel 3 Next Record Pointer"; fields[i++].dwOffset = 0x83c;
    if (f120 || f130)
    {
        fields[i].name ="Channel 0 Current Descriptor Pointer"; fields[i++].dwOffset = 0x870;
        fields[i].name ="Channel 1 Current Descriptor Pointer"; fields[i++].dwOffset = 0x874;
        fields[i].name ="Channel 2 Current Descriptor Pointer"; fields[i++].dwOffset = 0x878;
        fields[i].name ="Channel 3 Current Descriptor Pointer"; fields[i++].dwOffset = 0x87c;
    }
    //DMA Channel Control
    group[j].name ="DMA Channel Control"; group[j++].iPosition = i;
    fields[i].name ="Channel 0 Control"; fields[i++].dwOffset = 0x840;
    fields[i].name ="Channel 1 Control"; fields[i++].dwOffset = 0x844;
    fields[i].name ="Channel 2 Control"; fields[i++].dwOffset = 0x848;
    fields[i].name ="Channel 3 Control"; fields[i++].dwOffset = 0x84c;
    //DMA Arbiter
    group[j].name ="DMA Arbiter"; group[j++].iPosition = i;
    fields[i].name ="Arbiter Control"; fields[i++].dwOffset = 0x860;
    //Timer Counter
    group[j].name ="Timer Counter"; group[j++].iPosition = i;
    fields[i].name ="Timer /Counter 0"; fields[i++].dwOffset = 0x850;
    fields[i].name ="Timer /Counter 1"; fields[i++].dwOffset = 0x854;
    fields[i].name ="Timer /Counter 2"; fields[i++].dwOffset = 0x858;
    fields[i].name ="Timer /Counter 3"; fields[i++].dwOffset = 0x85c;
    fields[i].name ="Timer /Counter Control"; fields[i++].dwOffset = 0x864;
    //PCI Internal
    group[j].name ="PCI Internal"; group[j++].iPosition = i;
    if (f111)
    {
        fields[i].name ="Command"; fields[i++].dwOffset = 0xc00;
        fields[i].name ="Time Out & Retry"; fields[i++].dwOffset = 0xc04;
        fields[i].name ="RAS[1:0] Bank Size"; fields[i++].dwOffset = 0xc08;
        fields[i].name ="RAS[3:2] Bank Size"; fields[i++].dwOffset = 0xc0c;
        fields[i].name ="CS[2:0] Bank Size"; fields[i++].dwOffset = 0xc10;
        fields[i].name ="CS[3] & Boot CS Bank Size"; fields[i++].dwOffset = 0xc14;
        fields[i].name ="SErr Mask"; fields[i++].dwOffset = 0xc28;
        fields[i].name ="Interrupt Acknowledge"; fields[i++].dwOffset = 0xc34;
        fields[i].name ="Base Address Registers Enable"; fields[i++].dwOffset = 0xc3c;
        fields[i].name ="Configuration Address"; fields[i++].dwOffset = 0xcf8;
        fields[i].name ="Configuration Data"; fields[i++].dwOffset = 0xcfc;
    }
    else
    {
        fields[i].name ="PCI_0 Command"; fields[i++].dwOffset = 0xc00;
        fields[i].name ="PCI_1 Command"; fields[i++].dwOffset = 0xc80;
        fields[i].name ="PCI_0 Time Out & Retry"; fields[i++].dwOffset = 0xc04;
        fields[i].name ="PCI_1 Time Out & Retry"; fields[i++].dwOffset = 0xc84;
        fields[i].name ="PCI_0 SCS[1:0]* Bank Size"; fields[i++].dwOffset = 0xc08;
        fields[i].name ="PCI_1 SCS[1:0]* Bank Size"; fields[i++].dwOffset = 0xc88;
        fields[i].name ="PCI_0 SCS[3:2]* Bank Size"; fields[i++].dwOffset = 0xc0c;
        fields[i].name ="PCI_1 SCS[3:2]* Bank Size"; fields[i++].dwOffset = 0xc8c;
        fields[i].name ="PCI_0 CS[2:0]* Bank Size"; fields[i++].dwOffset = 0xc10;
        fields[i].name ="PCI_1 CS[2:0]* Bank Size"; fields[i++].dwOffset = 0xc90;
        fields[i].name ="PCI_0 CS[3]* & Boot CS* Bank Size"; fields[i++].dwOffset = 0xc14;
        fields[i].name ="PCI_1 CS[3]* & Boot CS* Bank Size"; fields[i++].dwOffset = 0xc94;
        fields[i].name ="PCI_0 Base Address Registers Enable"; fields[i++].dwOffset = 0xc3c;
        fields[i].name ="PCI_1 Base Address Registers Enable"; fields[i++].dwOffset = 0xcbc;
        fields[i].name ="PCI_0 Prefetch/Max Burst Size"; fields[i++].dwOffset = 0xc40;
        fields[i].name ="PCI_1 Prefetch/Max Burst Size"; fields[i++].dwOffset = 0xcc0;
        fields[i].name ="PCI_0 SCS[1:0]* Base Address Remap"; fields[i++].dwOffset = 0xc48;
        fields[i].name ="PCI_1 SCS[1:0]* Base Address Remap"; fields[i++].dwOffset = 0xcc8;
        fields[i].name ="PCI_0 SCS[3:2]* Base Address Remap"; fields[i++].dwOffset = 0xc4c;
        fields[i].name ="PCI_1 SCS[3:2]* Base Address Remap"; fields[i++].dwOffset = 0xccc;
        fields[i].name ="PCI_0 CS[2:0]* Base Address Remap"; fields[i++].dwOffset = 0xc50;
        fields[i].name ="PCI_1 CS[2:0]* Base Address Remap"; fields[i++].dwOffset = 0xcd0;
        fields[i].name ="PCI_0 CS[3]* & Boot CS* Address Remap"; fields[i++].dwOffset = 0xc54;
        fields[i].name ="PCI_1 CS[3]* & Boot CS* Address Remap"; fields[i++].dwOffset = 0xcd4;
        fields[i].name ="PCI_0 Swapped SCS[1:0]* Base Address Remap"; fields[i++].dwOffset = 0xc58;
        fields[i].name ="PCI_1 Swapped SCS[1:0]* Base Address Remap"; fields[i++].dwOffset = 0xcd8;
        fields[i].name ="PCI_0 Swapped SCS[3:2]* Base Address Remap"; fields[i++].dwOffset = 0xc5c;
        fields[i].name ="PCI_1 Swapped SCS[3:2]* Base Address Remap"; fields[i++].dwOffset = 0xcdc;
        fields[i].name ="PCI_0 Swapped CS[3]* & BootCS* Base Address Remap"; fields[i++].dwOffset = 0xc64;
        fields[i].name ="PCI_1 Swapped CS[3]* & BootCS* Base Address Remap"; fields[i++].dwOffset = 0xce4;
        fields[i].name ="PCI_0 Configuration Address"; fields[i++].dwOffset = 0xcf8;
        fields[i].name ="PCI_1 Configuration Address"; fields[i++].dwOffset = 0xcf0;
        fields[i].name ="PCI_0 Configuration Data Virtual Register"; fields[i++].dwOffset = 0xcfc;
        fields[i].name ="PCI_1 Configuration Data Virtual Register"; fields[i++].dwOffset = 0xcf4;
    }
    //Interrupts
    group[j].name ="Interrupts"; group[j++].iPosition = i;
    fields[i].name ="Interrupt Cause"; fields[i++].dwOffset = 0xc18;
    if (f111)
    {
        fields[i].name ="CPU/Local Master Mask"; fields[i++].dwOffset = 0xc1c;
        fields[i].name ="PCI Mask"; fields[i++].dwOffset = 0xc24;
        fields[i].name ="SErr Mask"; fields[i++].dwOffset = 0xc28;
        fields[i].name ="Interrupt Acknowledge"; fields[i++].dwOffset = 0xc34;
    }
    else
    {
        fields[i].name ="High Interrupt Cause Register"; fields[i++].dwOffset = 0xc98;
        fields[i].name ="CPU Interrupt Mask Register"; fields[i++].dwOffset = 0xc1c;
        fields[i].name ="CPU High Interrupt Mask Register"; fields[i++].dwOffset = 0xc9c;
        fields[i].name ="PCI_0 Interrupt Cause Mask Register"; fields[i++].dwOffset = 0xc24;
        fields[i].name ="PCI_0 High Interrupt Cause Mask Register"; fields[i++].dwOffset = 0xca4;
        fields[i].name ="PCI_0 SErr0 Mask"; fields[i++].dwOffset = 0xc28;
        fields[i].name ="PCI_1 SErr1 Mask"; fields[i++].dwOffset = 0xca8;
        fields[i].name ="PCI_0 Interrupt Acknowledge Virtual Register"; fields[i++].dwOffset = 0xc34;
        fields[i].name ="PCI_1 Interrupt Acknowledge Virtual Register"; fields[i++].dwOffset = 0xc30;
        fields[i].name ="CPU Select Cause Register"; fields[i++].dwOffset = 0xc70;
        fields[i].name ="PCI_0 Interrupt Select Register"; fields[i++].dwOffset = 0xc74;
    }
    group[j].iPosition = i;
    iGroupNum = j;
    iFieldsNum = i;

    do
    {
        int group_row = 0;

        printf ("\n");
        printf ("Edit Galileo 64 registers\n");
        printf ("----------------------------\n");
        while (group_row<iGroupNum)
        {
            printf ("%2d. %s\n", group_row+1, group[group_row].name);
            group_row++;
        }
        printf ("99. Back to main menu\n");
        printf ("Choose group of registers to read from / write to, or 99 to exit: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        if (cmd>=1 && cmd <=group_row)
        {
            j = cmd-1;
            do
            {
                int fields_row = group[j].iPosition;

                printf ("\n");
                printf ("Edit Galileo 64 %s registers\n", group[j].name);
                printf ("--------------------------------------------------\n");
                i = 1;
                while (fields_row<group[j+1].iPosition)
                {   
                    char buf[10];
                    fields[fields_row].dwVal = GT64_ReadReg(hGT, fields[fields_row].dwOffset);
                    sprintf(buf, "%08x",fields[fields_row].dwVal);
                    printf ("%2d. %-30s : %s     \n",i, fields[fields_row].name, buf);
                    fields_row++;
                    i++;
                }
                printf ("99. Back to previous menu\n");
                printf ("Choose register to write to, or 99 to exit: ");
                cmd2 = 0;
                gets (line);
                sscanf (line, "%d",&cmd2);
                if (cmd2>=1 && cmd2<=fields_row - group[j].iPosition)
                {
                    i = cmd2 - 1 + group[j].iPosition;
                    printf 
                        ("Enter value to write to %s register (or 'X' to cancel)\n> ",fields[i].name);
                    gets (line);
                    if (toupper (line[0])!='X')
                    {
                        DWORD dwVal;
                        dwVal = 0;
                        sscanf (line,"%x",&dwVal);
                        GT64_WriteReg(hGT, fields[i].dwOffset, dwVal);
                    }
                }
            } while (cmd2!=99);
        }
    } while (cmd!=99);
}

char *GT64_GetAddrRangeName(GT64_ADDR addrSpace)
{
    return 
        addrSpace==GT64_ADDR_BAR0 ? "RAS [1:0] - (BAR0)" :
        addrSpace==GT64_ADDR_BAR1 ? "RAS [3:2] - (BAR1)" :
        addrSpace==GT64_ADDR_BAR2 ? "CS [2:0] - (BAR2)" :
        addrSpace==GT64_ADDR_BAR3 ? "BootCS/Cs3 - (BAR3)" :
        addrSpace==GT64_ADDR_BAR4 ? "Internal Registers (Memory) - (BAR4)" :
        addrSpace==GT64_ADDR_BAR5 ? "Internal Registers (I/O) - (BAR5)" : "Invalid";
}

void GT64_BoardAccess(GT64_HANDLE hGT)
{
    int cmd, cmd2;
    DWORD addr, data, i;
    GT64_ADDR ad_sp = GT64_ADDR_BAR0;

    for (; ad_sp<=GT64_ADDR_BAR5 && !GT64_IsAddrSpaceActive(hGT, ad_sp); ad_sp++)
    if (ad_sp>GT64_ADDR_BAR5)
    {
        printf ("No active memory spaces on board!\n");
        return;
    }

    do
    {
        printf ("Access the board's offset ranges\n");
        printf ("--------------------------------\n");
        printf ("(Access to invalid offset may hang the computer!)\n");
        printf ("1. Change active memory space: %s\n",GT64_GetAddrRangeName(ad_sp));
        printf ("2. Read from board\n");
        printf ("3. Write to board\n");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            printf ("Choose memory space:\n");
            printf ("--------------------\n");
            for (i=GT64_ADDR_BAR0; i<=GT64_ADDR_BAR5; i++)
            {
                printf ("%d. %s", i, GT64_GetAddrRangeName(i));
                if (GT64_IsAddrSpaceActive(hGT, i)) printf ("\n");
                else printf (" - space not active\n");
            }
            printf ("Enter option: ");
            cmd2 = 99;
            gets (line);
            sscanf (line, "%d",&cmd2);
            if (cmd2>=GT64_ADDR_BAR0 && cmd2<=GT64_ADDR_BAR5)
            {
                int new_ad_sp = cmd2;
                if (GT64_IsAddrSpaceActive(hGT, new_ad_sp)) ad_sp = new_ad_sp;
                else printf ("Chosen space not active!\n");
            }
            break;
        case 2:
            printf ("Enter offset to read from: ");
            gets (line);
            sscanf (line, "%x", &addr);
            data = GT64_ReadDWord(hGT, ad_sp, addr);
            printf ("Value read: %x\n", data);
            break;
        case 3:
            printf ("Enter offset to write to: ");
            gets (line);
            sscanf (line, "%x", &addr);
            printf ("Enter data to write : ");
            gets (line);
            sscanf (line, "%x",&data);
            GT64_WriteDWord(hGT, ad_sp, addr, data);
            break;
        }
    } while (cmd!=99);
}

void WINAPI GT64_IntHandlerRoutine(GT64_HANDLE hGT, GT64_INT_RESULT *intResult)
{
    printf ("Got interrupt number %d\n", intResult->dwCounter);
}

void GT64_EnableDisableInterrupts(GT64_HANDLE hGT)
{
    int cmd;

    do
    {
        printf ("Enable / Disable interrupts\n");
        printf ("---------------------------\n");
        printf ("1. %s interrupts\n", GT64_IntIsEnabled(hGT) ? "Disable" : "Enable");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            if (GT64_IntIsEnabled(hGT))
            {
                printf ("Disabling interrupt Int\n");
                GT64_IntDisable(hGT);
            }
            else
            {
                printf ("Enabling interrupts\n");
                if (!GT64_IntEnable(hGT, GT64_IntHandlerRoutine))
                    printf ("failed enabling interrupts\n");
            }
            break;
        }
    } while (cmd!=99);
}


GT64_HANDLE GT64_LocateAndOpenBoard(DWORD dwVendorID, DWORD dwDeviceID, BOOL fUseInt)
{
    DWORD cards, my_card;
    GT64_HANDLE hGT = NULL;
    GT64_VERSION gt64Ver;
    DWORD i;

    if (dwVendorID==0)
    {
        printf ("Enter VendorID: ");
        gets (line);
        sscanf (line, "%x",&dwVendorID);
        if (dwVendorID==0) return NULL;

        printf ("Enter DeviceID: ");
        gets (line);
        sscanf (line, "%x",&dwDeviceID);
        if (dwDeviceID==0) return NULL;
    }

    cards = GT64_CountCards (dwVendorID, dwDeviceID);
    if (cards==0) 
    {
        printf("%s", GT64_ErrorString);
        return NULL;
    }
    else if (cards==1) my_card = 1;
    else
    {
        printf("Found %d matching PCI cards\n", cards);
        printf("Select card (1-%d): ", cards);
        i = 0;
        gets (line);
        sscanf (line, "%d",&i);
        if (i>=1 && i <=cards) my_card = i;
        else 
        {
            printf ("Choice out of range\n");
            return NULL;
        }
    }
    printf("Enter your bridge type:\n");
    printf("1. GT-64111\n");
    printf("2. GT-64120\n");
    printf("3. GT-64130\n");
    i = 0;
    gets (line);
    sscanf (line, "%d",&i);
    switch (i) 
    {
        case 1:
            gt64Ver = GT64_VERSION_64111;
            break;
        case 2:
            gt64Ver = GT64_VERSION_64120;
            break;
        case 3:
            gt64Ver = GT64_VERSION_64130;
            break;
        default:
            printf ("Choice out of range\n");
            return NULL;
    }
    if (GT64_Open (&hGT, gt64Ver, dwVendorID, dwDeviceID, my_card - 1, fUseInt ? GT64_OPEN_USE_INT : 0))
        printf ("Galileo 64 PCI card found!\n");
    else printf ("%s", GT64_ErrorString);
    return hGT;
}

int main(int argc, char *argv[])
{
    int cmd;
    GT64_HANDLE hGT = NULL;
    HANDLE hWD;
    BOOL fUseInt = FALSE; // by default - do not install interrupts

    printf ("Galileo 64 diagnostic utility.\n");
    printf ("Application accesses hardware using " WD_PROD_NAME ".\n");

    // make sure WinDriver is loaded
    if (!PCI_Get_WD_handle(&hWD)) return 0;
    WD_Close (hWD);

    hGT = GT64_LocateAndOpenBoard(0x11ab, 0, fUseInt);
    do
    {
        printf ("\n");
        printf ("Galileo 64 main menu\n");
        printf ("-------------------\n");
        printf ("1. Scan PCI bus\n");
        printf ("2. Set opening board %s interrupts\n", fUseInt ? "without" : "with");
        printf ("3. Locate/Choose Galileo 64 board (%s interrupts)\n", fUseInt ? "with" : "without");
        if (hGT)
        {
            printf ("4. PCI configuration registers\n");
            printf ("5. Galileo 64 local registers\n");
            printf ("6. Access address spaces on the board\n");
            if (hGT->fUseInt)
                printf ("7. Enable / Disable interrupts\n");
        }
        printf ("99. Exit\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1: // Scan PCI bus
            PCI_Print_all_cards_info();
            break;
        case 2: // Set open board with / without interrupts
            fUseInt = !fUseInt;
            break;
        case 3: // Locate Galileo 64 board
            if (hGT) GT64_Close(hGT);
            hGT = GT64_LocateAndOpenBoard(0, 0, fUseInt);
            if (!hGT) printf ("GT64 card open failed!\n");
            break;
        case 4: // PCI configuration registers
            if (hGT) PCI_EditConfigReg(hGT->pciSlot);
            break;
        case 5: // Galileo 64 local registers
            if (hGT) GT64_EditReg(hGT);
            break;
        case 6: // Access address spaces on the board
            if (hGT) GT64_BoardAccess(hGT);
            break;
       case 7: // Enable / Disable interrupts
            if (hGT && hGT->fUseInt) GT64_EnableDisableInterrupts(hGT);
            break;
        }
    } while (cmd!=99);

    if (hGT) GT64_Close(hGT);

    return 0;
}

                                      
