/////////////////////////
/////// AMCCDIAG.C //////
/////////////////////////

#include "../../include/windrvr.h"
#include "../lib/amcclib.h"
#include "../../samples/shared/pci_diag_lib.h"
#include <stdio.h>

// input of command from user
static char line[256];

void AMCC_EditReg(AMCCHANDLE hAmcc)
{
    struct 
    {
        CHAR *name;
        DWORD dwOffset;
        DWORD dwBytes;
        DWORD dwVal;
    } fields[30];

    int cmd;
    int i;
    int field_count;

    i = 0;
    fields[i].name = "OMB1"; fields[i].dwOffset = 0x0; fields[i++].dwBytes = 4;
    fields[i].name = "OMB2"; fields[i].dwOffset = 0x4; fields[i++].dwBytes = 4;
    fields[i].name = "OMB3"; fields[i].dwOffset = 0x8; fields[i++].dwBytes = 4;
    fields[i].name = "OMB4"; fields[i].dwOffset = 0xc; fields[i++].dwBytes = 4;
    fields[i].name = "IMB1"; fields[i].dwOffset = 0x10; fields[i++].dwBytes = 4;
    fields[i].name = "IMB2"; fields[i].dwOffset = 0x14; fields[i++].dwBytes = 4;
    fields[i].name = "IMB3"; fields[i].dwOffset = 0x18; fields[i++].dwBytes = 4;
    fields[i].name = "IMB4"; fields[i].dwOffset = 0x1c; fields[i++].dwBytes = 4;
    fields[i].name = "FIFO"; fields[i].dwOffset = 0x20; fields[i++].dwBytes = 4;
    fields[i].name = "MWAR"; fields[i].dwOffset = 0x24; fields[i++].dwBytes = 4;
    fields[i].name = "MWTCR"; fields[i].dwOffset = 0x28; fields[i++].dwBytes = 4;
    fields[i].name = "MRAR"; fields[i].dwOffset = 0x2c; fields[i++].dwBytes = 4;
    fields[i].name = "MRTCR"; fields[i].dwOffset = 0x30; fields[i++].dwBytes = 4;
    fields[i].name = "MBEF"; fields[i].dwOffset = 0x34; fields[i++].dwBytes = 4;
    fields[i].name = "INTCSR"; fields[i].dwOffset = 0x38; fields[i++].dwBytes = 4;
    fields[i].name = "BMCSR"; fields[i].dwOffset = 0x3c; fields[i++].dwBytes = 4;
    field_count = i;
    do
    {
        int row;
        int col;
        int row_count = field_count/2 + field_count%2;

        printf ("\n");
        printf ("Edit 5933 operation registers\n");
        printf ("--------------------------------\n");
        for (row = 0; row<row_count; row++)
        {
            for (col = 0; col<=1; col++)
            {
                if (col==0) i = row;
                else i = row + row_count;

                if (i<field_count)
                {
                    char buf[10];
                    if (fields[i].dwOffset!=FIFO_ADDR)
                    {
                        fields[i].dwVal = AMCC_ReadRegDWord(hAmcc, fields[i].dwOffset);
                        sprintf(buf, "%08x",fields[i].dwVal);
                    }
                    else
                        strcpy (buf,"Write   ");
                    printf ("%2d. %6s : %s     ",i+1, fields[i].name, buf);
                }
                if (col==1) printf ("\n");
            }
        }

        printf ("20. FIFO read.\n");
        printf ("99. Back to main menu\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        if (cmd>=1 && cmd <=16)
        {
            i = cmd-1;
            printf ("Enter value to write to %s register (or 'X' to cancel): ",fields[i].name);
            gets (line);
            if (toupper (line[0])!='X')
            {
                DWORD dwVal;
                dwVal = 0;
                sscanf(line,"%x",&dwVal);
                if ((dwVal>0xff && fields[i].dwBytes==1)|| 
                    (dwVal>0xffff && fields[i].dwBytes==2)|| 
                    (dwVal>0xffffff && fields[i].dwBytes==3))
                {
                    printf ("Error: value too big for register\n");
                }
                else
                {
                    AMCC_WriteRegDWord(hAmcc, fields[i].dwOffset, dwVal);
                }
            }
        }
        else if (cmd==20)
        {
            DWORD dwVal;

            dwVal = AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR);
            if (dwVal & BIT5) 
            {
                printf ("Could not read FIFO - inbound FIFO empty\n");
            }
            else
            {
                dwVal = AMCC_ReadRegDWord(hAmcc, FIFO_ADDR);
                printf ("FIFO : %08x\n", dwVal);
            }
        }
    } while (cmd!=99);
}

BOOL Diag_test_OGMB(AMCCHANDLE hAmcc, DWORD nOGMB, DWORD nByte, DWORD nTestSize, DWORD dwVal)
{
    DWORD dwValRead;
    DWORD dwAddr = OMB1_ADDR + nOGMB*4 + nByte;
    DWORD dwMBEFMask;
    // reset MBEF
    AMCC_WriteRegDWord(hAmcc, BMCSR_ADDR, AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR) | BIT27);
    if (AMCC_ReadRegDWord(hAmcc, MBEF_ADDR)!=0) 
    {
        printf ("MBEF doesn't reset\n");
        return FALSE;
    }

    switch (nTestSize)
    {
    case 1:
        AMCC_WriteRegByte(hAmcc, dwAddr, (BYTE) dwVal);
        dwValRead = AMCC_ReadRegByte(hAmcc, dwAddr); 
        dwMBEFMask = 0x1;
        break;
    case 2:
        AMCC_WriteRegWord(hAmcc, dwAddr, (WORD) dwVal);
        dwValRead = AMCC_ReadRegWord(hAmcc, dwAddr);
        dwMBEFMask = 0x3;
        break;
    case 4:
        AMCC_WriteRegDWord(hAmcc, dwAddr, dwVal);
        dwValRead = AMCC_ReadRegDWord(hAmcc, dwAddr);
        dwMBEFMask = 0xf;
        break;
    }
    if (dwVal!=dwValRead) 
    {
        printf ("OGMB write->read gave different values\n");
        return FALSE;
    }

    dwValRead = AMCC_ReadRegDWord(hAmcc, MBEF_ADDR);
    if (dwValRead != (dwMBEFMask << (nOGMB*4 + nByte)))
    {
        printf ("MBEF didn't set bits after writing to OGMB\n");
        return FALSE;
    }

    return TRUE;
}

void AMCC_RunDiag(AMCCHANDLE hAmcc)
{
    int i,j;
    DWORD dwVal;
    BOOL fTestOk, fTestAllOk;

    printf ("\n");
    printf ("Run diagnostics\n");
    printf ("---------------\n");
    printf ("\n");
    printf ("Testing OGMB: ");

    fTestAllOk = TRUE;
    // test OGMB1-4 registers
    fTestOk = TRUE;
    for (i=0; i<4 && fTestOk; i++)
    {
        // test as BYTES
        for (j=0; j<4 && fTestOk; j++)
        {
            for (dwVal=0; dwVal<=0xff && fTestOk; dwVal++)
            {
                if (!Diag_test_OGMB(hAmcc, i, j, 1, dwVal)) fTestOk = FALSE;
            }
        }

        // test as WORDS
        for (j=0; j<2 && fTestOk; j++)
        {
            for (dwVal=0; dwVal<=0xf000 && fTestOk; dwVal+=0xf9)
            {
                if (!Diag_test_OGMB(hAmcc, i, j*2, 2, dwVal)) fTestOk = FALSE;
            }
        }

        // test as DWORD
        for (dwVal=0; dwVal<=0xf0000000 && fTestOk; dwVal+=0x00f98765)
        {
            if (!Diag_test_OGMB(hAmcc, i, 0, 4, dwVal)) fTestOk = FALSE;
        }
    }

    if (fTestOk)
    {
        printf ("Passed.\n");
    }
    else
    {
        printf ("Failed!\n");
        fTestAllOk = FALSE;
    }

    // test FIFO
    fTestOk = TRUE;
    printf ("FIFO test: ");
    // make sure FIFO is empty
    for (i=0; i<9; i++)
    {
        if (AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR) & BIT5) 
            break;
        AMCC_ReadRegDWord(hAmcc, FIFO_ADDR);
    }
    if (i==9)
    {
        printf ("Could not empty FIFO (performed 8 reads already)\n");
        fTestOk = FALSE;
    }
    else
    {
        DWORD dwPattern[8] = 
        {
            0x12345678, 0x87654321, 0xabcdef90, 0x09fedcba, 
            0x0f1e2d3c, 0x4b5a6978, 0x8796a5b4, 0xc3d2e1f0
        };

        // turn on FIFO loopback
        AMCC_WriteRegDWord(hAmcc, BMCSR_ADDR, BIT28 | AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR));

        for (i=0; i<8; i++)
        {
            AMCC_WriteRegDWord(hAmcc, FIFO_ADDR, dwPattern[i]);
        }

        for (i=0; i<8 && fTestOk; i++)
        {
            DWORD dwFifoStatus;

            dwFifoStatus = AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR) & (BIT3 | BIT4 | BIT5);
            if (i==0 && dwFifoStatus!=(BIT3 | BIT4))
            {
                printf ("BMCSR fifo status error (FIFO full)\n");
                fTestOk = FALSE;
            }
            else if (i>0 && i<=4 && dwFifoStatus!=(BIT4))
            {
                printf ("BMCSR fifo status error (FIFO half full)\n");
                fTestOk = FALSE;
            }
            else if (i>4 && dwFifoStatus!=0)
            {
                printf ("BMCSR fifo status error (FIFO half emtpy)\n");
                fTestOk = FALSE;
            }

            if (AMCC_ReadRegDWord(hAmcc, FIFO_ADDR)!=dwPattern[i])
            {
                printf ("FIFO read different value from what was written\n");
                fTestOk = FALSE;
            }
        }
        if (fTestOk)
        {
            if ((AMCC_ReadRegDWord(hAmcc, BMCSR_ADDR) & (BIT3 | BIT4 | BIT5))!=BIT5)
            {
                printf ("BMCSR error in status (FIFO empty)");
                fTestOk = FALSE;
            }
        }
    }

    if (fTestOk)
    {
        printf ("Passed.\n");
    }
    else
    {
        printf ("Failed!\n");
        fTestAllOk = FALSE;
    }
    
    // Add test to write to the address MRAR MWAR MRTCR MWTCR (also status)

    if (fTestAllOk)
    {
        printf ("Board passed Diagnostic test.\n");
    }
    else
    {
        printf ("Diagnostic test failed!\n");
    }
}

char *AMCC_GetAddrRangeName(AMCC_ADDR addrSpace)
{
    return 
        addrSpace==AMCC_ADDR_SPACE0 ? "Addr Space 0 - (BAR1)" :
        addrSpace==AMCC_ADDR_SPACE1 ? "Addr Space 1 - (BAR2)" :
        addrSpace==AMCC_ADDR_SPACE2 ? "Addr Space 2 - (BAR3)" :
        addrSpace==AMCC_ADDR_SPACE3 ? "Addr Space 3 - (BAR4)" :
        addrSpace==AMCC_ADDR_NOT_USED ? "Not used Addr Space - (BAR5)" : "Invalid";
}

void AMCC_BoardAccess(AMCCHANDLE hAmcc)
{
    int cmd, cmd2, i;
    typedef enum {AMCC_MODE_BYTE, AMCC_MODE_WORD, AMCC_MODE_DWORD} AMCC_MODE;
    DWORD addr, data;
    AMCC_ADDR ad_sp = AMCC_ADDR_SPACE0;
    AMCC_MODE ad_mode = AMCC_MODE_DWORD;

    for (; ad_sp<AMCC_ADDR_NOT_USED && !AMCC_IsAddrSpaceActive(hAmcc, ad_sp); ad_sp++)
    if (ad_sp==AMCC_ADDR_NOT_USED)
    {
        printf ("No active memory spaces on board!\n");
        return;
    }

    do
    {
        printf ("\n");
        printf ("Access the board's memory ranges\n");
        printf ("--------------------------------\n");
        printf ("1. Change active memory space: %s\n",AMCC_GetAddrRangeName(ad_sp));
        printf ("2. Toggle active mode: %s\n", 
            ad_mode==AMCC_MODE_BYTE ? "BYTE (8 bit)" :
            ad_mode==AMCC_MODE_WORD ? "WORD (16 bit)" : "DWORD (32 bit)");
        printf ("3. Read from board\n");
        printf ("4. Write to board\n");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            printf ("Choose memory space:\n");
            printf ("--------------------\n");
            for (i=AMCC_ADDR_SPACE0; i<AMCC_ADDR_NOT_USED; i++)
            {
                printf ("%d. %s", i, AMCC_GetAddrRangeName(i));
                if (AMCC_IsAddrSpaceActive(hAmcc, i)) printf ("\n");
                else printf (" - space not active\n");
            }
            printf ("Enter option: ");
            cmd2 = 99;
            gets (line);
            sscanf (line, "%d",&cmd2);
            if (cmd2>=AMCC_ADDR_SPACE0 && cmd2<AMCC_ADDR_NOT_USED)
            {
                int new_ad_sp = cmd2;
                if (AMCC_IsAddrSpaceActive(hAmcc, new_ad_sp)) ad_sp = new_ad_sp;
                else printf ("Chosen space not active!\n");
            }
            break;
        case 2:
            ad_mode = (ad_mode + 1) % 3;
            break;
        case 3:
            printf ("Enter offset to read from: ");
            gets (line);
            sscanf (line, "%x", &addr);
            switch (ad_mode)
            {
            case AMCC_MODE_BYTE:
                data = AMCC_ReadByte(hAmcc, ad_sp, addr);
                break;
            case AMCC_MODE_WORD:
                data = AMCC_ReadWord(hAmcc, ad_sp, addr);
                break;
            case AMCC_MODE_DWORD:
                data = AMCC_ReadDWord(hAmcc, ad_sp, addr);
                break;
            }
            printf ("Value read: %x\n", data);
            break;
        case 4:
            printf ("Enter offset to write to: ");
            gets (line);
            sscanf (line, "%x", &addr);
            printf ("Enter data to write %s: ",
                ad_mode==AMCC_MODE_BYTE ? "BYTE (8 bit)" :
                ad_mode==AMCC_MODE_WORD ? "WORD (16 bit)" : "DWORD (32 bit)");
            gets (line);
            sscanf (line, "%x",&data);
            switch (ad_mode)
            {
            case AMCC_MODE_BYTE:
                AMCC_WriteByte(hAmcc, ad_sp, addr, (BYTE) data);
                break;
            case AMCC_MODE_WORD:
                AMCC_WriteWord(hAmcc, ad_sp, addr, (WORD) data);
                break;
            case AMCC_MODE_DWORD:
                AMCC_WriteDWord(hAmcc, ad_sp, addr, data);
                break;
            }
            break;
        }
    } while (cmd!=99);
}

void WINAPI AMCC_IntHandlerRoutine(AMCCHANDLE hAmcc, AMCC_INT_RESULT *intResult)
{
    printf ("Got interrupt number %d\n", intResult->dwCounter);
}

void AMCC_EnableDisableInterrupts(AMCCHANDLE hAmcc)
{
    int cmd;

    printf ("WARNING!!!\n");
    printf ("----------\n");
    printf ("Your hardware has level sensitive interrupts.\n");
    printf ("You must modify the source code of AMCC_IntEnable(), in the file amcclib.c,\n");
    printf ("to acknowledge the interrupt before enabling interrupts.\n");
    printf ("Without this modification, your PC will HANG upon interrupt!\n");

    do
    {
        printf ("Enable / Disable interrupts\n");
        printf ("---------------------------\n");
        printf ("1. %s interrupts\n", AMCC_IntIsEnabled(hAmcc) ? "Disable" : "Enable");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            if (AMCC_IntIsEnabled(hAmcc))
            {
                printf ("Disabling interrupt Int\n");
                AMCC_IntDisable(hAmcc);
            }
            else
            {
                printf ("Enabling interrupts\n");
                if (!AMCC_IntEnable(hAmcc, AMCC_IntHandlerRoutine))
                    printf ("failed enabling interrupts\n");
            }
            break;
        }
    } while (cmd!=99);
}

void AMCC_AccessNVRAM(AMCCHANDLE hAmcc)
{
    int cmd;
    DWORD dwAddr;

    do
    {
        printf ("\n");
        printf ("Access the board's nvRAM\n");
        printf ("------------------------\n");
        printf ("1. Read nvRAM data\n");
        printf ("2. Program nvRAM\n");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            printf ("Read nvRAM\n");
            for (dwAddr=0; dwAddr<AMCC_NVRAM_SIZE; dwAddr++)
            {
                BYTE data;
                if ((dwAddr % 16) == 0) printf ("%04x- ", dwAddr);
                if (!AMCC_ReadNVByte(hAmcc, dwAddr, &data))
                {
                    printf ("\nerror while reading nvRAM!\n");
                    break;
                }
                printf (" %02x",data);
                if ((dwAddr % 16) == 15) printf ("\n");
            }
            break;
        case 2:
            printf ("This option is not yet implemented\n");
            break;
        }
    } while (cmd!=99);
}

AMCCHANDLE AMCC_LocateAndOpenBoard(DWORD dwVendorID, DWORD dwDeviceID, BOOL fUseInt)
{
    DWORD cards, my_card;
    AMCCHANDLE hAmcc = NULL;

    if (dwVendorID==0)
    {
        printf ("Enter VendorID: ");
        gets (line);
        sscanf (line, "%x",&dwVendorID);
        if (dwVendorID==0) return NULL;

        printf ("Enter DeviceID: ");
        gets (line);
        sscanf (line, "%x",&dwDeviceID);
    }
    cards = AMCC_CountCards (dwVendorID, dwDeviceID);
    if (cards==0) 
    {
        printf("%s", AMCC_ErrorString);
        return NULL;
    }
    else if (cards==1) my_card = 1;
    else
    {
        DWORD i;

        printf("Found %d matching PCI cards\n", cards);
        printf("Select card (1-%d): ", cards);
        i = 0;
        gets (line);
        sscanf (line, "%d",&i);
        if (i>=1 && i <=cards) my_card = i;
        else 
        {
            printf ("Choice out of range\n");
            return NULL;
        }
    }
    if (AMCC_Open (&hAmcc, dwVendorID, dwDeviceID, my_card - 1, fUseInt ? AMCC_OPEN_USE_INT : 0))
        printf ("AMCC MatchMaker PCI card found!\n");
    else printf ("%s", AMCC_ErrorString);
    return hAmcc;
}

int main(int argc, char *argv[])
{
    int cmd;
    AMCCHANDLE hAmcc = NULL;
    HANDLE hWD;
    BOOL fUseInt = FALSE; // by default - do not install interrupts

    printf ("AMCC 5933 DK1 diagnostic utility.\n");
    printf ("Application access hardware using " WD_PROD_NAME ".\n");

    // make sure WinDriver is loaded
    if (!PCI_Get_WD_handle(&hWD)) return 0;
    WD_Close (hWD);

    hAmcc = AMCC_LocateAndOpenBoard(0x10e8, 0x4750, fUseInt);

    do
    {
        printf ("\n");
        printf ("AMCC 5933 main menu\n");
        printf ("-------------------\n");
        printf ("1. Scan PCI bus\n");
        printf ("2. Set opening board %s interrupts\n", fUseInt ? "without" : "with");
        printf ("3. Locate/Choose PCI matchmaker board (%s interrupts)\n", fUseInt ? "with" : "without");
        if (hAmcc)
        {
            printf ("4. PCI configuration registers\n");
            printf ("5. MatchMaker operation registers\n");
            printf ("6. Access memory ranges on the board\n");
            if (hAmcc->fUseInt)
                printf ("7. Enable / Disable interrupts\n");
            printf ("8. Access nvRAM\n");
            printf ("9. Run diagnostics\n");
        }
        printf ("99. Exit\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf(line, "%d",&cmd);
        switch (cmd)
        {
        case 1: // Scan PCI bus
            PCI_Print_all_cards_info();
            break;
        case 2: // Set open board with / without interrupts
            fUseInt = !fUseInt;
            break;
        case 3: // Locate PCI matchmaker board
            if (hAmcc) AMCC_Close(hAmcc);
            hAmcc = AMCC_LocateAndOpenBoard(0, 0, fUseInt);
            if (!hAmcc) printf ("AMCC card open failed!\n");
            break;
        case 4: // PCI configuration registers
            if (hAmcc) PCI_EditConfigReg(hAmcc->pciSlot);
            break;
        case 5: // MatchMaker operation registers
            if (hAmcc) AMCC_EditReg(hAmcc);
            break;
        case 6: // MatchMaker access address spaces
            if (hAmcc) AMCC_BoardAccess(hAmcc);
            break;
        case 7: // Enable / Disable interrupts
            if (hAmcc && hAmcc->fUseInt) AMCC_EnableDisableInterrupts(hAmcc);
            break;
        case 8: // access nvRAM
            if (hAmcc) AMCC_AccessNVRAM(hAmcc);
            break;
        case 9: // Run diagnostics
            if (hAmcc) AMCC_RunDiag(hAmcc);
            break;
        }
    } while (cmd!=99);

    if (hAmcc) AMCC_Close(hAmcc);

    return 0;
}

                                      
