 /***************************************************************************
 *                                                                          *
 * File Name        : JOYTEST.CPP                                           *
 *                                                                          *
 * Description      : This program provides sample code for using the       *
 *                    OS/2 Joystick Device Driver API's.                    *
 *                                                                          *
 * Entry Points     : main()                                                *
 *                                                                          *
 * Copyright        : COPYRIGHT IBM CORPORATION, 1995                       *
 *                                                                          *
 *       DISCLAIMER OF WARRANTIES.  The following [enclosed] code is        *
 *       sample code created by IBM Corporation. This sample code is not    *
 *       part of any standard or IBM product and is provided to you solely  *
 *       for  the purpose of assisting you in the development of your       *
 *       applications.  The code is provided "AS IS", without               *
 *       warranty of any kind.  IBM shall not be liable for any damages     *
 *       arising out of your use of the sample code, even if they have been *
 *       advised of the possibility of such damages.                        *
 *                                                                          *
 ****************************************************************************/

#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSMEMMGR

#include <os2.h>
#include <stdlib.h>
#include <iostream.h>

#include "joyos2.h"

/*
 * Type of joysticks:
 *    A standard joystick has only 2 sets of analog values for the X-axis
 *    and Y-axis.
 *    An extended feature joystick has 2 sets of analog values for the
 *    X-axis and Y-axis and the third set of value for the throttle or
 *    pedal or hatch control.
 */
#define JOY_STANDARD    0
#define JOY_EXTENDED    1

ostream &
operator << (ostream &os, const GAME_2DPOS_STRUCT &pos)
{
	os << pos.x << "," << pos.y;
	return os;
}

ostream &
operator << (ostream &os, const GAME_3POS_STRUCT &pos)
{
	os << pos.lower << "," << pos.centre << "," << pos.upper;
	return os;
}

istream &
operator >> (istream &is, GAME_3POS_STRUCT &pos)
{
	is >> pos.lower >> pos.centre >> pos.upper;
	return is;
}

int
main (int argc, char *argv[])
{
	APIRET			rc;               /* return code */
	HFILE			hGame;               /* handle to device GAME$ */
	ULONG			action;              /* action taken by DosOpen */
	ULONG			version;             /* version of the driver */
	GAME_PARM_STRUCT	gameParms;     /* status on the installed joystick */
	GAME_CALIB_STRUCT	gameCalib;     /* calibration values */
	GAME_STATUS_STRUCT	gameStatus; /* data of joysticks and buttons */
	ULONG			parmLen, dataLen;    /* length of parameters */
   USHORT      usJoyType;           /* type of joystick */

	cout << "OS/2 Joystick Sample v1.0" << endl;
  	cout << endl;

   /*
    * Open the device "GAME$". "GAME$" was loaded by the OS/2
    * Joystick Device Driver (GAMEDD.SYS) at initialization time.
    */
	rc = DosOpen(
		GAMEPDDNAME,
		&hGame,
		&action,
		0,
		FILE_READONLY,
		FILE_OPEN,
		OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE,
		NULL);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not open device" << endl;
		exit(-1);
	}

   /*
    * API function 01h: GAME_GET_VERSION
    */
	cout << "\nGetting the version of the OS/2 Joystick device driver:" << endl;
	dataLen = sizeof(version);
	rc = DosDevIOCtl(
		hGame,
		IOCTL_CAT_USER,  GAME_GET_VERSION,
		NULL, 0, NULL,
		&version, dataLen, &dataLen);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
		exit(-1);
	}
   cout << "\t" << hex << version << dec << endl;

	if (version < 0x2a)
	{
		cerr << "ERROR: Incorrect version of driver" << endl;
		exit(-1);
	}

   /*
    * API function 02h: GAME_GET_PARMS
    */
	cout << "\nGetting the device parameters:" << endl;
	dataLen = sizeof(gameParms);
	rc = DosDevIOCtl(
		hGame,
		IOCTL_CAT_USER,  GAME_GET_PARMS,
		NULL, 0, NULL,
		&gameParms, dataLen, &dataLen);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
		exit(-1);
	}

	if ((gameParms.useA == 0) && (gameParms.useB == 0))
	{
		cout << "\tNo joysticks detected" << endl;
		DosClose(hGame);
		exit(0);
	}

   /*
    * Keep only bits defined X and Y axis, they are bit 1 and bit 2
    */
   USHORT usTmp1 = gameParms.useA & GAME_USE_BOTH_NEWMASK;
   USHORT usTmp2 = gameParms.useB & GAME_USE_BOTH_NEWMASK;

   /*
    * Determine what type of joystick is installed:
    *
    * A standard joystick has only 2 sets of analog values for the
    * X axis and Y axis.  A joystick with extended feature has 3 sets
    * of analog values for the X axis, Y axis and the third set of
    * value is for the throttle or hatch or pedal control.
    *
    * Based on the design of the device driver, if both bit 1 and 2
    * are set in usTmp1, this means there is only 1 standard joystick.
    * If bit 1 and 2 are set in usTmp1 and usTmp2, this means there
    * are 2 standard joysticks.
    *
    * If bit 1 and 2 are set in usTmp1 and either bit 1 or bit 2
    * is set in usTmp2, this means there is 1 joystick with extended
    * features such as a ThrustMaster FCS or a joystick with throttle
    * control or with a hatch button.
    *
    */
	if ( usTmp1 == GAME_USE_BOTH_NEWMASK )
   {
      if ( usTmp2 == GAME_USE_BOTH_NEWMASK )
      {
 		   cout << "\tTwo standard joysticks detected" << endl;
         usJoyType = JOY_STANDARD;
      }
      else if ( !usTmp2 )
      {
 		      cout << "\tOne standard joystick detected" << endl;
            usJoyType = JOY_STANDARD;
      }
      else if ( usTmp2 )
      {
            cout << "\tOne joystick with extended feature detected" << endl;
            cout << "\t(the third calibration/axis value is for ";
            cout << "\n\tthrottle or pedal or hatch control)" <<endl;
            usJoyType = JOY_EXTENDED;
      }
   } // end if

   /*
    * API function 04h: GAME_GET_CALIB
    */
	cout << "\nGetting the calibration values:" << endl;
	dataLen = sizeof(gameCalib);
	rc = DosDevIOCtl(
		hGame,
		IOCTL_CAT_USER,  GAME_GET_CALIB,
		NULL, 0, NULL,
		&gameCalib, dataLen, &dataLen);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
		exit(-1);
	}

   /*
    * Standard joystick
    */
   if ( usJoyType == JOY_STANDARD ) {
	   if (gameParms.useA)
	   	cout << "\tAx: " << gameCalib.Ax << "\tAy: " << gameCalib.Ay << endl;
	   if (gameParms.useB)
	   	cout << "\tBx: " << gameCalib.Bx << "\tBy: " << gameCalib.By << endl;
	   cout << endl;
   }
   /*
    * Joystick with extended feature
    */
   else {
		cout << "\tAx: " << gameCalib.Ax << endl;
      cout << "\tAy: " << gameCalib.Ay << endl;
      cout << "\tAz: " << gameCalib.By << endl;
   }

   /*
    * API function 10h: GAME_GET_STATUS
    */
	cout << "\nGetting the current joystick status:" << endl;
	dataLen = sizeof(gameStatus);
	rc = DosDevIOCtl(
		hGame,
		IOCTL_CAT_USER, GAME_GET_STATUS,
		NULL, 0, NULL,
		&gameStatus, dataLen, &dataLen);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
		exit(-1);
	}

   /*
    * Standard joystick
    */
   if ( usJoyType == JOY_STANDARD ) {
	   if (gameParms.useA)
	   	cout << "\tA (x's, y's): " << gameStatus.curdata.A << endl;
	   if (gameParms.useB)
	   	cout << "\tB (x's, y's): " << gameStatus.curdata.B << endl;
   }
   /*
    * Joystick with extended feature
    */
   else {
	   cout << "\tA (x's, y's, z's): " << gameStatus.curdata.A;
      cout <<","<< gameStatus.curdata.B.y << endl;
   }

   /*
    * API function 11h: GAME_GET_STATUS_BUTWAIT
    */
	cout << "\nGetting the joystick status when the next button is pressed:" << endl;
	dataLen = sizeof(gameStatus);
	rc = DosDevIOCtl(
		hGame,
		IOCTL_CAT_USER, GAME_GET_STATUS_BUTWAIT,
		NULL, 0, NULL,
		&gameStatus, dataLen, &dataLen);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
		exit(-1);
	}

   /*
    * Standard joystick
    */
   if ( usJoyType == JOY_STANDARD ) {
	   if (gameParms.useA)
	   	cout << "\tA (x's, y's): " << gameStatus.curdata.A << endl;
	   if (gameParms.useB)
	   	cout << "\tB (x's, y's): " << gameStatus.curdata.B << endl;
   }
   /*
    * Joystick with extended feature
    */
   else {
	   cout << "\tA (x's, y's, z's): " << gameStatus.curdata.A;
      cout <<","<< gameStatus.curdata.B.y << endl;
   }

   /*
    * API function 03h: GAME_SET_PARMS
    */
	cout << "\nSetting the device parameters:" << endl;
   cout << "\tChange sampling rate to 1 Hz" << endl;
	gameParms.sampDiv = 32;
	parmLen = sizeof(gameParms);
	rc = DosDevIOCtl(
		hGame,
		IOCTL_CAT_USER,  GAME_SET_PARMS,
		&gameParms, parmLen, &parmLen,
		NULL, 0, NULL);
	if (rc != 0)
	{
		cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
		exit(-1);
	}

	for (;;)
	{
     /*
      * API function 12h: GAME_GET_STATUS_SAMPWAIT
      */
    	cout << "\nContinuing status until button 1 is pressed:" << endl;
		dataLen = sizeof(gameStatus);
		rc = DosDevIOCtl(
			hGame,
			IOCTL_CAT_USER, GAME_GET_STATUS_SAMPWAIT,
			NULL, 0, NULL,
			&gameStatus, dataLen, &dataLen);
		if (rc != 0)
		{
			cerr << "ERROR " << rc << ": Could not call IOCtl" << endl;
			exit(-1);
		}

		if (gameStatus.b1cnt > 0)
			break;

      /*
       * Standard joystick
       */
      if ( usJoyType == JOY_STANDARD ) {
	      if (gameParms.useA)
	      	cout << "\tA (x's, y's): " << gameStatus.curdata.A << endl;
	      if (gameParms.useB)
	      	cout << "\tB (x's, y's): " << gameStatus.curdata.B << endl;
      }
      /*
       * Joystick with extended feature
       */
      else {
	      cout << "\tA (x's, y's, z's): " << gameStatus.curdata.A;
         cout <<","<< gameStatus.curdata.B.y << endl;
      }
	   cout << "\tButton 2 count: " << gameStatus.b2cnt << endl;
	}

   cout << "Program terminated normally" << endl;

   /*
    * Close the device "GAME$"
    */
	DosClose(hGame);
}
