/****************************************************************************
 *                                                                          *
 * SOURCE FILE NAME: Robot.c                                                *
 *                                                                          *
 * DESCRIPTIVE NAME: Manage a 3D world and the views into it                *
 *                                                                          *
 * COMPONENT: Robot                                                         *
 *                                                                          *
 * COPYRIGHT: (C) Copyright Argonaut Software. 1994.  All rights reserved.  *
 *            (C) Copyright IBM, 1995.  All rights reserved.                *
 *                                                                          *
 * FUNCTIONS:                                                               *
 *    LoadVUE                                                               *
 *    ApplyVUE                                                              *
 *    WorldAllocate                                                         *
 *    WorldFree                                                             *
 *    WorldUpdate                                                           *
 *    ViewBufferSet                                                         *
 *    ViewRender                                                            *
 *    ViewBoundsCallback                                                    *
 *    ViewScreenUpdate                                                      *
 *                                                                          *
 * BRENDER FUNCTIONS                                                        *
 *    BrFloatToScalar                                                       *
 *    BrIntToScalar                                                         *
 *    BrScalarToFixed                                                       *
 *    BrActorAdd                                                            *
 *    BrActorAllocate                                                       *
 *    BrActorLoad                                                           *
 *    BrActorSearch                                                         *
 *    BrFmtScriptMaterialLoadMany                                           *
 *    BrLightEnable                                                         *
 *    BrMapFindHook                                                         *
 *    BrMaterialAddMany                                                     *
 *    BrMatrix34RotateX                                                     *
 *    BrMatrix34Translate                                                   *
 *    BrMatrix34RotateY                                                     *
 *    BrMatrix34PostRotateZ                                                 *
 *    BrMemAllocate                                                         *
 *    BrModelLoadMany                                                       *
 *    BrModelAddMany                                                        *
 *    BrPixelmapFree                                                        *
 *    BrPixelmapAllocate                                                    *
 *    BrPixelmapMatch                                                       *
 *    BrPixelmapFill                                                        *
 *    BrPixelmapRectangleFill                                               *
 *    BrTableFindHook                                                       *
 *    BrZbSetRenderBoundsCallback                                           *
 *    BrZbSceneRender                                                       *
 *                                                                          *
 * DIVE FUNCTIONS                                                           *
 *    DiveAllocImageBuffer                                                  *
 *    DiveBeginImageBufferAccess                                            *
 *    DiveBlitImage                                                         *
 *    DiveEndImageBufferAccess                                              *
 *    DiveFreeImageBuffer                                                   *
 *                                                                          *
 ****************************************************************************/

#define INCL_WIN
#define INCL_GPI
#define INCL_DOS
#define INCL_OS2MM

#include <os2.h>                  /* Master include file for OS/2           */
#include <stdlib.h>               /* Standard C information                 */
#include <stdio.h>                /* Standard I/O                           */
#include <string.h>               /* String information                     */
#include <limits.h>               /* Limits defines                         */
#include <os2me.h>                /* OS/2 multimedia                        */
#include <dive.h>                 /* Display engine information             */
#include <fourcc.h>               /* Color space definitions                */

#include <brender.h>              /* Master include file for BRender        */
#include <brassert.h>             /* Assertion macros                       */

#include "world.h"                /* 3D world and views into it             */
#include "app.h"                  /* Application prototypes and defines     */

/*
 * Forward declarations
 */
static void SetupTexture( char        *material,
                          char        *pix_file,
                          br_pixelmap *shade_table );

static void SetupModel( char *name,
                        char *file );

static void BR_CALLBACK ViewBoundsCallback( br_actor    *actor,
                                            br_model    *model,
                                            br_material *material,
                                            br_uint_8    style,
                                            br_matrix4  *model_to_screen,
                                            br_int_32   *bounds );

/*
 * Dark green background
 */
#define BACK_COLOUR 80

/*
 * Keeps a copy of the current view for use by the bounds callback
 */
static brwin_view *bounds_view;

#define MAXFRAMES 100
#define MAXTRANSFORMS 2000

int total_frames;

struct ani_transform
{
	br_actor *actor;
	br_matrix34 mat;
};

struct ani_frame
{
	struct ani_transform * transforms;
	int                    ntransforms;
};

struct ani_frame frames[MAXFRAMES];
struct ani_transform transforms[MAXTRANSFORMS];


/*
 * FUNCTION NAME: LoadVUE
 *
 * DESCRIPTION:   Load transformations for actor animation
 */
void LoadVUE( char *file_name, br_actor *root )
{
   FILE     *file;
   br_model *model;
   char     buff[1024];
   char     name[80];
   float    v[4][3];
   struct   ani_transform *t = transforms;
   int      f = -1,n,i;

   file = fopen(file_name, "r");

   while ( !feof(file) )
   {
      fgets( buff, 1024, file );

      /*
       * Is it a frame specification line?
       */
      if ( sscanf( buff, "frame %d", &i ) > 0 )
      {
         if ( f > 0 )
            frames[f].ntransforms = t - frames[f].transforms;
         f = i;
         frames[f].transforms = t;
      }

      /*
       * Or a transform specification?
       */
      if (sscanf(buff,
                 "transform \"%[^\"]\" %f %f %f %f %f %f %f %f %f %f %f %f",
                 name,
                 &v[0][0],&v[0][1],&v[0][2],
                 &v[1][0],&v[1][1],&v[1][2],
                 &v[2][0],&v[2][1],&v[2][2],
                 &v[3][0],&v[3][1],&v[3][2])>0)
      {
         int k,l;

         /*
          * Found a transform so convert it to scalars
          */
         for (k = 0; k<4; k++)
            for (l = 0; l<3; l++)
               t->mat.m[k][l] = BrFloatToScalar(v[k][l]);

         t->actor = BrActorSearch(root, name);

         if(t->actor != NULL)
            t++;
      }
   }

   fclose( file );

   if ( f > 0 )
     frames[f].ntransforms = t - frames[f].transforms;

   total_frames = f+1;

} /* End of LoadVUE () */


/*
 * FUNCTION NAME: ApplyVUE
 *
 * DESCRIPTION:   Play transformations
 *                These give the appearance of a robot walking.
 */
void ApplyVUE( int frame_delta )
{
   static current_frame = 0;            /* Current frame                   */

   /*
    * Play any VUE file
    */
   if ( total_frames != 0 )
   {
      int i;

      /*
       * Clamp frame number to valid range
       */
      if ( current_frame < 0 )
         current_frame = total_frames-1;

      if ( current_frame >= total_frames )
         current_frame = 0;

      /*
       * Apply animation to world
       */
      for ( i=0; i < frames[current_frame].ntransforms; i++ )
         frames[current_frame].transforms[i].actor->t.t.mat =
            frames[current_frame].transforms[i].mat;

      /*
       * Move to next frame
       */
      current_frame += frame_delta;
   }

} /* End of ApplyVUE() */


/*
 * FUNCTION NAME:  WorldAllocate
 *
 * DESCRIPTION:    Create a new 3D world
 */
brwin_world *WorldAllocate(void)
{
   brwin_world *world;                   /* Ptr to world data structure    */
   br_camera   *camera_data;             /* Ptr to camera data             */
   br_light    *light_data;              /* Ptr to light data              */
   br_actor    *a;                       /* Ptr to robot actor heirarchy   */
   br_material *materials[20];           /* Array of pointers to materials */
   br_model    *models[50];              /* Ptr to array of model data     */
   int         nmodels;                  /* Number of models               */
   int         nmaterials;               /* Number of materials            */

   /*
    * Allocate world structure
    */
   world = BrMemAllocate( sizeof(*world), BR_MEMORY_APPLICATION );

   world->update = WORLD_UPDATE_FOREGROUND;

   world->root = BrActorAllocate( BR_ACTOR_NONE, NULL );

   /*
    * Load materials from material script
    */
   BrTableFindHook( BrTableFindFailedLoad );
   BrMapFindHook( BrMapFindFailedLoad );

   nmaterials = BrFmtScriptMaterialLoadMany( "robo.msc",  /* Materials file */
                                             materials,   /* Ptr to array   */
                                             BR_ASIZE(materials) ); /* Max  */
   BrMaterialAddMany( materials, nmaterials );

   /*
    * Load all the models
    */
   nmodels = BrModelLoadMany( "robo.dat", models, BR_ASIZE(models) );
   BrModelAddMany( models, nmodels );

   /*
    * Load all the actors
    */
   world->actor = BrActorAdd( world->root, BrActorAllocate(BR_ACTOR_NONE,NULL) );

   a = BrActorLoad( "robo.act" );
   BrActorAdd( world->actor, a );

   /*
    * Load the animation file
    */
   LoadVUE( "robo.vue", a );

   /*
    * Turn to face camera
    */
   BrMatrix34RotateX( &world->actor->t.t.mat, BR_ANGLE_DEG(-90.0) );

   /*
    * Camera
    */
   world->camera_pivot = BrActorAdd( world->root, BrActorAllocate(BR_ACTOR_NONE, NULL) );
   world->camera_pivot->t.type = BR_TRANSFORM_EULER;
   world->camera_pivot->t.t.euler.e.order = BR_EULER_XYZ_S;

   world->camera = BrActorAdd( world->camera_pivot, BrActorAllocate(BR_ACTOR_CAMERA, NULL) );
   camera_data = world->camera->type_data;

   camera_data->type          = BR_CAMERA_PERSPECTIVE;
   camera_data->field_of_view = BR_ANGLE_DEG(45.0);
   camera_data->hither_z      = BR_SCALAR(50.0);
   camera_data->yon_z         = BR_SCALAR(3000.0);
   camera_data->aspect        = BR_SCALAR(1.8);

   BrMatrix34Translate( &world->camera->t.t.mat, BR_SCALAR(0.0), BR_SCALAR(0.0), BR_SCALAR(1700) );

   /*
    * Light
    */
   world->light = BrActorAdd( world->root, BrActorAllocate(BR_ACTOR_LIGHT, NULL) );
   light_data                = world->light->type_data;
   light_data->type          = BR_LIGHT_DIRECT;
   light_data->attenuation_c = BR_SCALAR(1.0);

   BrMatrix34RotateY( &world->light->t.t.mat, BR_ANGLE_DEG(45.0) );
   BrMatrix34PostRotateZ( &world->light->t.t.mat, BR_ANGLE_DEG(45.0) );

   BrLightEnable( world->light );

   return world;

} /* End of WorldAllocate() */


/*
 * FUNCTION NAME:  WorldFree
 *
 * DESCRIPTION:    Release the 3D world
 */
void WorldFree( brwin_world *world )
{
   /* XXX Relies on memory allocator to release everything */

} /* End of WorldFree() */


/*
 * FUNCTION NAME:  WorldUpdate
 *
 * DESCRIPTION:    Time based update of the 3D world
 */
void WorldUpdate(brwin_world *world)
{
   ApplyVUE(1);

} /* End of WorldUpdate () */


/*
 * FUNCTION NAME:  ViewBufferSet
 *
 * DESCRIPTION:    Setup the view buffer
 */
void ViewBufferSet( brwin_view *view )
{
   ULONG rc;
   LONG  i, j;
   PBYTE pbBufTemp;


   /*
    * Limit the width and height
    */
   if ( view->ulWidth < 2 )
      view->ulWidth = 2;

   if ( view->ulHeight < 2 )
      view->ulHeight = 2;

   /*
    * Release any old maps
    */
   if ( view->pbBuffer )
      rc = DiveFreeImageBuffer( view->hDive, view->ulBufferNumber );

   if ( view->colour_buffer )
       BrPixelmapFree( view->colour_buffer );

   if ( view->depth_buffer )
      BrPixelmapFree( view->depth_buffer );

   /*
    * Associate image buffer with Dive
    */
   rc = DiveAllocImageBuffer( view->hDive,      /* Dive instance           */
                         &view->ulBufferNumber, /* Allocated buffer number */
                         FOURCC_LUT8,           /* Color space             */
                         view->ulWidth,         /* Width                   */
                         view->ulHeight,        /* Height                  */
                         0,                     /* Pick optimum line size  */
                         0 );                   /* Dive allocates buffer   */

   rc = DiveBeginImageBufferAccess( view->hDive,
                                    view->ulBufferNumber,
                                    &(view->pbBuffer),
                                    &(view->ulScanLineBytes),
                                    &(view->ulScanLines) );

   view->colour_buffer = BrPixelmapAllocate( BR_PMT_INDEX_8,
                                             (br_uint_16)view->ulWidth,
                                             (br_uint_16)view->ulHeight,
                                             view->pbBuffer,
                                             BR_PMAF_INVERTED);

   rc = DiveEndImageBufferAccess( view->hDive,
                                  view->ulBufferNumber );

   view->depth_buffer = BrPixelmapMatch( view->colour_buffer,
                                         BR_PMMATCH_DEPTH_16);

   /*
    * Remember camera aspect ratios
    */
    view->aspect = BrScalarToFixed( BR_DIV(BrIntToScalar(view->ulWidth),
                                    BrIntToScalar(view->ulHeight)));

   /*
    * Clear buffers
    */

   memset( view->pbBuffer, BACK_COLOUR, view->ulScanLines * view->ulScanLineBytes );

   BrPixelmapFill(view->depth_buffer, 0xFFFF);

   view->last_dirty_rectangle = 0;

} /* End of ViewBufferSet() */


/*
 * FUNCTION NAME: ViewRender
 *
 * DESCRIPTION:   Render an image of the world into test_colour_buffer
 *                and accumulate a list of changes.
 */
void ViewRender(brwin_view *view)
{
   br_renderbounds_cbfn *old_cb;     /* Old Callback                       */
   int                  i;           /* Counter for dirty rectangle list   */
   brwin_dirty_rect     *dp;         /* Ptr to dirty rectangle list        */

   /*
    * Don't do anything if off-screen buffer has not been setup yet
    */
   if(view->pbBuffer == NULL)
      return;

   /*
    * Set camera position according to controls
    */
   view->world->camera_pivot->t.t.euler.e.a  = TrackingValues[MOUSETRACK_LEFT].y *  130;
   view->world->camera_pivot->t.t.euler.e.b  = TrackingValues[MOUSETRACK_LEFT].x * -130;

   view->world->camera->t.t.translate.t.v[2] = BR_SCALAR(1700.0) +
        BrIntToScalar(TrackingValues[MOUSETRACK_RIGHT].y * 10);

   view->world->camera->t.t.translate.t.v[0] =
        BrIntToScalar(TrackingValues[MOUSETRACK_LEFT | MOUSETRACK_RIGHT].x * -10);
   view->world->camera->t.t.translate.t.v[1] =
        BrIntToScalar(TrackingValues[MOUSETRACK_LEFT | MOUSETRACK_RIGHT].y * -10);

   /*
    * Clear screen and Z buffer using dirty rectangle list
    */
   for(i=0, dp = view->dirty_rectangles; i< view->last_dirty_rectangle; i++, dp++)
   {
      BrPixelmapRectangleFill( view->colour_buffer,
                              (br_int_16)dp->min_x,
                              (br_int_16)dp->min_y,
                              (br_int_16)(dp->max_x - dp->min_x),
                              (br_int_16)(dp->max_y-dp->min_y),
                              BACK_COLOUR );

      BrPixelmapRectangleFill( view->depth_buffer,
                              (br_int_16)dp->min_x,
                              (br_int_16)dp->min_y,
                              (br_int_16)(dp->max_x - dp->min_x),
                              (br_int_16)(dp->max_y-dp->min_y),
                              0xFFFFFFFF);
   }

   /*
    * Reset dirty list
    */
   view->last_dirty_rectangle = 0;
   view->render_bounds.min_x  = INT_MAX;
   view->render_bounds.min_y  = INT_MAX;
   view->render_bounds.max_x  = INT_MIN;
   view->render_bounds.max_y  = INT_MIN;

   /*
    * Hook dirty rectangle callback, and render view
    */
   ((br_camera *)view->world->camera->type_data)->aspect =
                               BrFixedToScalar(view->aspect);

   bounds_view = view;
   old_cb = BrZbSetRenderBoundsCallback( ViewBoundsCallback );
   BrZbSceneRender( view->world->root,
                    view->world->camera,
                    view->colour_buffer,
                    view->depth_buffer);
   BrZbSetRenderBoundsCallback(old_cb);

} /* End of ViewRender() */


/*
 * FUNCTION NAME: ViewBoundsCallback
 *
 * DESCRIPTION:   Callback function used by renderer to pass back
 *                dirty rectangles
 */
static void BR_CALLBACK ViewBoundsCallback( br_actor    *actor,
                                            br_model    *model,
                                            br_material *material,
                                            br_uint_8    style,
                                            br_matrix4  *model_to_screen,
                                            br_int_32   *bounds)
{
   brwin_dirty_rect *dp;

   /*
    * Push rectangle out a bit
    */
   bounds[BR_BOUNDS_MIN_X] -= 1;
   bounds[BR_BOUNDS_MAX_X] += 1;
   bounds[BR_BOUNDS_MIN_Y] -= 1;
   bounds[BR_BOUNDS_MAX_Y] += 1;

   /*
    * Accumulate total bounding rectangle
    */
   if ( bounds[BR_BOUNDS_MIN_X] < bounds_view->render_bounds.min_x )
      bounds_view->render_bounds.min_x = bounds[BR_BOUNDS_MIN_X];

   if ( bounds[BR_BOUNDS_MIN_Y] < bounds_view->render_bounds.min_y )
      bounds_view->render_bounds.min_y = bounds[BR_BOUNDS_MIN_Y];

   if ( bounds[BR_BOUNDS_MAX_X] > bounds_view->render_bounds.max_x )
      bounds_view->render_bounds.max_x = bounds[BR_BOUNDS_MAX_X];

   if ( bounds[BR_BOUNDS_MAX_Y] > bounds_view->render_bounds.max_y )
      bounds_view->render_bounds.max_y = bounds[BR_BOUNDS_MAX_Y];

   /*
    * If list of rectangles is full, merge current into last
    */
   if(bounds_view->last_dirty_rectangle >= MAX_DIRTY_RECTANGLES)
   {
      dp = bounds_view->dirty_rectangles + bounds_view->last_dirty_rectangle-1;

      if ( bounds[BR_BOUNDS_MIN_X] < dp->min_x )
         dp->min_x = bounds[BR_BOUNDS_MIN_X];

      if ( bounds[BR_BOUNDS_MIN_Y] < dp->min_y )
         dp->min_y = bounds[BR_BOUNDS_MIN_Y];

      if ( bounds[BR_BOUNDS_MAX_X] > dp->max_x )
         dp->max_x = bounds[BR_BOUNDS_MAX_X];

      if ( bounds[BR_BOUNDS_MAX_Y] > dp->max_y )
         dp->max_y = bounds[BR_BOUNDS_MAX_Y];

      return;
   }

   /*
    * Add this rectangle to list
    */
   dp = bounds_view->dirty_rectangles + bounds_view->last_dirty_rectangle;

   dp->min_x = bounds[BR_BOUNDS_MIN_X];
   dp->min_y = bounds[BR_BOUNDS_MIN_Y];
   dp->max_x = bounds[BR_BOUNDS_MAX_X];
   dp->max_y = bounds[BR_BOUNDS_MAX_Y];

   bounds_view->last_dirty_rectangle++;

} /* End of ViewBoundsCallback() */


/*
 * FUNCTION NAME: ViewScreenUpdate
 *
 * DESCRIPTION:   Update screen from offscreen bitmap - will attempt to
 *                only update the changed area, unless force_all is true
 *
 */
void ViewScreenUpdate( brwin_view *view )
{
   ULONG rc;

   /*
    * Don't do anything if off-screen buffer has not been setup yet
    */
   if(view->pbBuffer == NULL)
      return;

   /*
    * Blit image to window
    */
   rc = DiveBlitImage( view->hDive,
                       view->ulBufferNumber,
                       DIVE_BUFFER_SCREEN );

} /* End of ViewScreenUpdate() */


