/****************************************************************************
 *                                                                          *
 * SOURCE FILE NAME: App.c                                                  *
 *                                                                          *
 * DESCRIPTIVE NAME: Simple test harness for using BRender under OS/2 DIVE  *
 *                                                                          *
 * COMPONENT: Robot                                                         *
 *                                                                          *
 * COPYRIGHT: (C) Copyright Argonaut Software. 1994.  All rights reserved.  *
 *            (C) Copyright IBM, 1995.  All rights reserved.                *
 *                                                                          *
 * FUNCTIONS:                                                               *
 *    main                                                                  *
 *    AppWndProc                                                            *
 *    PaletteInit                                                           *
 *    DrawingThread                                                         *
 *    BrPMWarning                                                           *
 *    BrPMError                                                             *
 *                                                                          *
 * BRENDER FUNCTIONS                                                        *
 *    BrBegin                                                               *
 *    BrEnd                                                                 *
 *    BrErrorHandlerSet                                                     *
 *    BrPixelmapFree                                                        *
 *    BrPixelmapLoad                                                        *
 *    BrZbBegin                                                             *
 *    BrZbEnd                                                               *
 *                                                                          *
 * DIVE FUNCTIONS:                                                          *
 *    DiveClose                                                             *
 *    DiveOpen                                                              *
 *    DiveQueryCaps                                                         *
 *    DiveSetupBlitter                                                      *
 *                                                                          *
 ****************************************************************************/

#define INCL_WIN
#define INCL_GPI
#define INCL_DEV
#define INCL_DOS
#define INCL_OS2MM

#include <os2.h>                  /* Master include file for OS/2           */
#include <stdlib.h>               /* Standard C information                 */
#include <os2me.h>                /* OS/2 multimedia                        */
#include <dive.h>                 /* Display engine information             */
#include <fourcc.h>               /* Color space definitions                */

#include <process.h>              /* Thread process information             */
#include <brender.h>              /* Master include file for BRender        */

#include "world.h"                /* 3D world and views into it             */
#include "app.h"                  /* Application prototypes and defines     */


/*
 * Globals
 */
HAB    hab;
HPAL   hPal = NULLHANDLE;

/*
 * Rendering Thread Globals
 */
static brwin_world *BrenderWorld;
static brwin_view  *BrenderView;

BOOL   AppIsActive;
int    threadRun = TRUE;

static HMTX hmtxPS = NULLHANDLE;  /* PS access semaphore                     */

ULONG  PaletteEntries[PAL_SIZE];

BOOL   HasPaletteManager;

/*
 * Mouse tracking info
 */
int    TrackingFlags = 0;
POINTL TrackingValues[4];

/*
 * error message box id
 */
#define MSG_BOX_ID     256

#define DT_STACKSIZE (32 * 1024)

static void BR_CALLBACK BrPMError(char *message);
static void BR_CALLBACK BrPMWarning(char *message);

br_errorhandler BrPMErrorHandler =
{
   "Presentation Manager ErrorHandler",
   BrPMWarning,
   BrPMError,
};


/*
 * FUNCTION NAME: main
 *
 * DESCRIPTION:   Main routine...initializes window and message queue
 *
 */
int main( )
{
   HMQ       hmqApp;                           /* Handle to message queue   */
   QMSG      qmsg;                             /* Message structure         */
   ULONG     ctldata;                          /* Window flag control data  */
   CHAR      szClassName[] = "BRender";        /* Class name                */
   CHAR      szMessage[]   = "BRender";        /* Message name              */
   DIVE_CAPS DiveCaps = {0};                   /* DIVE capabilities         */
   FOURCC    fccFormats[100] = {0};            /* Color format code         */
   HPS       hps;
   HDC       hdc;

   /*
    * Initialize the renderer
    */
   BrBegin();
   BrZbBegin( BR_PMT_INDEX_8, BR_PMT_DEPTH_16 );

   if(getenv("BRENDER_PATH") == NULL)
   putenv("BRENDER_PATH=dat;../dat;../../dat;../../../dat");

   BrErrorHandlerSet( &BrPMErrorHandler );

   /*
    * Create the 3D world
    */
   BrenderWorld        = WorldAllocate();
   BrenderView         = calloc( sizeof(*BrenderView), 1 );
   BrenderView->world  = BrenderWorld;

   /*
    * Create hab, hmq
    */
   hab = WinInitialize(0LU);
   hmqApp = WinCreateMsgQueue(hab, 0L);

   /*
    * Get the screen capabilities.  If only 16 colors are supported,
    * the application will be terminated.
    */
   DiveCaps.ulStructLen    = sizeof(DIVE_CAPS);
   DiveCaps.ulFormatLength = 100;
   DiveCaps.pFormatData    = fccFormats;
   if ( DiveQueryCaps( &DiveCaps, DIVE_BUFFER_SCREEN ) )
   {
      WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                     (PSZ)"usage:  The sample can not run on this system environment.",
                     (PSZ)"ROBOT.EXE - Robot Sample", 0, MB_OK | MB_INFORMATION );
      WinDestroyMsgQueue( hmqApp );
      WinTerminate( hab );
      BrZbEnd();
      BrEnd();
      free( BrenderView );
      return(1);
   }

   if ( DiveCaps.ulDepth < 8 )
   {
      WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                     (PSZ)"usage:  The sample can not run on this system environment.",
                     (PSZ)"ROBOT.EXE - Robot Sample", 0, MB_OK | MB_INFORMATION );
      WinDestroyMsgQueue( hmqApp );
      WinTerminate( hab );
      BrZbEnd();
      BrEnd();
      free( BrenderView );
      return(1);
   }

   /*
    * Open DIVE instance.
    */
   if ( DiveOpen( &(BrenderView->hDive), FALSE, 0 ) )
   {
      WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                     (PSZ)"usage:  The sample can not run on this system environment.",
                     (PSZ)"ROBOT.EXE - Robot Sample", 0, MB_OK | MB_INFORMATION );
      WinDestroyMsgQueue( hmqApp );
      WinTerminate( hab );
      BrZbEnd();
      BrEnd();
      free( BrenderView );
      return(1);
   }

   /*
    * Register a class
    */
   WinRegisterClass( hab,
                     (PSZ)szClassName,
                     (PFNWP)AppWndProc,
                     CS_SIZEREDRAW,
                     0L);

   /*
    * Create a standard pm window and check that creation was successful.
    */
   ctldata = FCF_STANDARD & ~(FCF_SHELLPOSITION | FCF_MENU);

   BrenderView->hwndFrame = WinCreateStdWindow( HWND_DESKTOP,
                                                WS_VISIBLE,
                                                &ctldata,
                                                szClassName,
                                                szMessage,
                                                0LU,
                                                NULLHANDLE,
                                                ID_WINDOW,
                                                &(BrenderView->hwndClient) );

   /*
    * Set our own position
    */
   WinSetWindowPos(BrenderView->hwndFrame, NULLHANDLE, 10, 10, 450, 350,
                   SWP_MOVE | SWP_SIZE | SWP_SHOW | SWP_ACTIVATE);

   /*
    * Ask to be notified of visible region changes to our window.
    */
   WinSetVisibleRegionNotify( BrenderView->hwndClient, TRUE );

   /*
    * Send an invalidation message to the client.
    */
   WinPostMsg( BrenderView->hwndFrame, WM_VRNENABLED, 0L, 0L );

   /*
    * Poll messages from event queue
    */
   while( WinGetMsg( hab, (PQMSG)&qmsg, (HWND)NULLHANDLE, 0LU, 0LU ) )
      WinDispatchMsg( hab, (PQMSG)&qmsg );

   /*
    * End notification of visible region changes to our window.
    */
   WinSetVisibleRegionNotify( BrenderView->hwndClient, FALSE );

   /*
    * Clean up
    */
   if(BrenderView->hwndFrame)
      WinDestroyWindow( BrenderView->hwndFrame );

   if(hmqApp)
      WinDestroyMsgQueue( hmqApp );

   /*
    * Close down renderer
    */
   BrZbEnd();
   BrEnd();

   /*
    * Close down Dive
    */
   if (BrenderView->hDive)
      DiveClose( BrenderView->hDive );

   free( BrenderView );

   /*
    * Restore palette
    */
   hps = WinGetPS ( HWND_DESKTOP );
   hdc = GpiQueryDevice ( hps );
   Gre32EntrY3 ( hdc, 0L, 0x000060C7L );
   WinInvalidateRect ( HWND_DESKTOP, (PRECTL)NULL, TRUE );
   WinReleasePS ( hps );

   WinTerminate( hab );
   return 0;
}


/*
 * FUNCTION NAME:  AppWndProc
 *
 * DESCRIPTION:    This routine processes WM_COMMAND, WM_PAINT.   It passes
 *                 everything else to the Default Window Procedure.
 */
MRESULT EXPENTRY AppWndProc( HWND   hwnd,
                             ULONG  msg,
                             MPARAM mp1,
                             MPARAM mp2 )
{
   void    *stack;
   HPS     hps;                        /* Presentation space handle         */
   HRGN    hrgn;                       /* Region handle                     */
   RGNRECT rgnCtl;
   RECTL   rcls[25];
   SWP     swp;
   POINTL  pointl;
   SETUP_BLITTER SetupBlitter;         /* Structure for DiveSetupBlitter    */
   ULONG rc;

   switch ( msg )
   {
     case WM_VRNDISABLED:
        /*
         * The window's visible region is changing.
         * Disable blitting during this time.
         */
        DiveSetupBlitter( BrenderView->hDive, 0 );
        break;

     case WM_VRNENABLED:
        /*
         * No change to window's visible region.
         * Blitting is enabled.
         */
        hps = WinGetPS( hwnd );

        if ( !hps )
           break;

        hrgn = GpiCreateRegion( hps, 0L, NULL );

        if ( hrgn )
        {
           /*
            * NOTE: If mp1 is zero, then this was just a move message.
            * Illustrate the visible region on a WM_VRNENABLED.
            */
           WinQueryVisibleRegion ( hwnd, hrgn );
           rgnCtl.ircStart     = 0;
           rgnCtl.crc          = 50;
           rgnCtl.ulDirection  = 1;

           /*
            * Get the all ORed rectangles
            */
           if ( GpiQueryRegionRects( hps, hrgn, NULL, &rgnCtl, rcls) )
           {
              /*
               * Now find the window position and size, relative to parent.
               */
              WinQueryWindowPos( BrenderView->hwndClient, &swp );

              /*
               * Store the window dimensions
               */
               BrenderView->ulWidth  = swp.cx;
               BrenderView->ulHeight = swp.cy;

              /*
               * Convert the point to offset from desktop lower left.
               */
              pointl.x = swp.x;
              pointl.y = swp.y;

              WinMapWindowPoints( BrenderView->hwndFrame, HWND_DESKTOP,
                                   &pointl, 1 );

              /*
               * Tell DIVE about the new settings.
               */
              SetupBlitter.ulStructLen       = sizeof ( SETUP_BLITTER );
              SetupBlitter.fccSrcColorFormat = FOURCC_LUT8;
              SetupBlitter.ulSrcWidth        = BrenderView->ulWidth;
              SetupBlitter.ulSrcHeight       = BrenderView->ulHeight;
              SetupBlitter.ulSrcPosX         = 0;
              SetupBlitter.ulSrcPosY         = 0;
              SetupBlitter.fInvert           = TRUE;
              SetupBlitter.ulDitherType      = 1;
              SetupBlitter.fccDstColorFormat = FOURCC_SCRN;
              SetupBlitter.ulDstWidth        = BrenderView->ulWidth;
              SetupBlitter.ulDstHeight       = BrenderView->ulHeight;
              SetupBlitter.lDstPosX          = 0;
              SetupBlitter.lDstPosY          = 0;
              SetupBlitter.lScreenPosX       = pointl.x;
              SetupBlitter.lScreenPosY       = pointl.y;
              SetupBlitter.ulNumDstRects     = rgnCtl.crcReturned;
              SetupBlitter.pVisDstRects      = rcls;

              DiveSetupBlitter( BrenderView->hDive, &SetupBlitter );
           }
           else
           {
              DiveSetupBlitter( BrenderView->hDive, 0 );
           }

           GpiDestroyRegion( hps, hrgn );
        }
        WinReleasePS( hps );

        WinRequestMutexSem( hmtxPS, SEM_INDEFINITE_WAIT );

        ViewBufferSet( BrenderView );

        ViewRender( BrenderView );

        ViewScreenUpdate( BrenderView );

        DosReleaseMutexSem( hmtxPS );

        break;

     case WM_CREATE:
        /*
         * Create semaphore that controls access to the
         * device shared between window proc. and renderer.
         */
        DosCreateMutexSem( (PSZ)NULL, &hmtxPS, 0UL, FALSE );

        PaletteInit();

        threadRun = TRUE;

        stack = malloc(DT_STACKSIZE);
        _beginthread( DrawingThread, stack, DT_STACKSIZE, NULL);
        break;

     case WM_ACTIVATE:
        AppIsActive = SHORT1FROMMP(mp1);
        break;

     case WM_BUTTON1DOWN:
        if ( AppIsActive )
        {
           TrackingFlags |= MOUSETRACK_LEFT;
           WinSetCapture( HWND_DESKTOP, hwnd );
        } else
        return WinDefWindowProc( hwnd, msg, mp1, mp2 );
        break;

     case WM_BUTTON1UP:
        TrackingFlags &= ~MOUSETRACK_LEFT;
        WinSetCapture( HWND_DESKTOP, NULLHANDLE );
        break;

     case WM_BUTTON2DOWN:
        if ( AppIsActive )
        {
           TrackingFlags |= MOUSETRACK_RIGHT;
           WinSetCapture( HWND_DESKTOP, hwnd );
        } else
        return WinDefWindowProc( hwnd, msg, mp1, mp2 );
        break;

     case WM_BUTTON2UP:
        TrackingFlags &= ~MOUSETRACK_RIGHT;
        WinSetCapture( HWND_DESKTOP, NULLHANDLE );
        break;

     case WM_MOUSEMOVE:
        if ( TrackingFlags )
        {
           TrackingValues[TrackingFlags].x =
                         SHORT1FROMMP(mp1) - BrenderView->ulWidth/2;
           TrackingValues[TrackingFlags].y =
                         SHORT2FROMMP(mp1) - BrenderView->ulHeight/2;
        }
        break;

     default:
        return WinDefWindowProc( hwnd, msg, mp1, mp2 );
   }

   /*
    * Return 0 for handled message
    */
   return MRFROMLONG( 0L );

} /* End of AppWndProc() */


/*
 * FUNCTION NAME: PaletteInit
 *
 * DESCRIPTION:   Creates the application palette
 *
 * NOTES:         There are many ways in which the palette can be set.
 *                This method was selected because it is fastest for
 *                games developers.  This method sets the physical
 *                palette in the correct order, unlike GPI which maps the
 *		  palette entries.
 * 		  GPI and DIVE can also be used to set the palette.
 * 		  For DIVE, use DiveSetSourcePalette here and
 *                DiveSetDestinationPalette under the WM_REALIZE message.
 * 		  DIVE only sets the logical palette.
 *
 */
LONG PaletteInit(void)
{
   char        *cp;               /* Copied palette entries from pixelmap   */
   br_pixelmap *pal;              /* Brender pixelmap palette               */
   int         i;                 /* For loop counter                       */
   ULONG       rc;                /* Return code                            */
   LONG        alCaps;
   BOOL        HasPaletteManager;
   HPS         hps;
   HDC         hdc;
   LONG        lRealColors[10];
   int         j;

   /*
    * Load palette from file
    */
   pal = BrPixelmapLoad( "os2pal.pal" );

   if ( pal == NULL )
      BR_ERROR0( "Could not load palette" );

   if ( pal->type != BR_PMT_RGBX_888 ||
        pal->width != 1              ||
        pal->height != PAL_SIZE )
      BR_ERROR0("Incorrect palette");
		
   WinRequestMutexSem( hmtxPS, SEM_INDEFINITE_WAIT );

   /*
    * Copy palette entries from pixelmap (BRender's RGB is same as OS/2)
    */
   cp = pal->pixels;

   for(i=0; i < PAL_SIZE; i++)
   {
      PaletteEntries[i] = *((ULONG *)cp);
      cp += pal->row_bytes;
   }

   BrenderView->ulNumColor = 246;

   hps = WinGetPS ( HWND_DESKTOP );
   hdc = GpiQueryDevice ( hps );

   rc = GpiQueryRealColors( hps,
                            0,
                            0,
                            10,
                            lRealColors );
   for ( i=0; i<10; i++ )
   {
      PaletteEntries[i] = lRealColors[i];
   }

   GpiQueryRealColors( hps,
                       0,
                       246,
                       10,
                       lRealColors );
   j=0;
   for ( i=246; i<256; i++ )
   {
      PaletteEntries[i] = lRealColors[j++];
   }

   rc=GpiCreateLogColorTable ( hps,
                               LCOL_PURECOLOR | LCOL_REALIZABLE,
                               LCOLF_CONSECRGB,
                               0,
                               256,
                               (PLONG)PaletteEntries );
   Gre32EntrY3 ( hdc,
                 0L,
                 0x000060C6L );
   WinInvalidateRect ( HWND_DESKTOP,
                       (PRECTL)NULL,
                       TRUE );
   WinReleasePS ( hps );

   DosReleaseMutexSem( hmtxPS );

   /*
    * Release palette pixelmap
    */
   BrPixelmapFree( pal );

   return 0;

} /* End of PaletteInit() */


/*
 * FUNCTION NAME: DrawingThread
 *
 * DESCRIPTION:   Background thread that carries out the rendering
 *
 */
static void _Optlink DrawingThread( void *arg )
{
   HAB     habThread2;   /* Anchor block handle for thread */
   HMQ     hmqThread2;   /* Queue handle for thread        */

   habThread2 = WinInitialize( 0LU );
   hmqThread2 = WinCreateMsgQueue( habThread2, 0L );

   /*
    * Renderer runs at highest IDLETIME priority
    */
   DosSetPriority(PRTYS_THREAD, PRTYC_IDLETIME, 0, 0);

   while ( threadRun )
   {
      WorldUpdate( BrenderView->world );

      WinRequestMutexSem( hmtxPS, SEM_INDEFINITE_WAIT );

      ViewRender( BrenderView );
      ViewScreenUpdate( BrenderView );

      DosReleaseMutexSem( hmtxPS );
   }

   if ( hmqThread2 )
      WinDestroyMsgQueue( hmqThread2 );

   WinTerminate( habThread2 );

   _endthread();

} /* End of DrawingThread() */


/*
 * FUNCTION NAME: BrPMWarning
 *
 * DESCRIPTION:   BRender Error handling function
 *
 */
static void BR_CALLBACK BrPMWarning(char *message)
{
   WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, message, "WARNING", 256, MB_OK | MB_APPLMODAL | MB_WARNING);
   DosExit(1,10);
} /* End of BrPMWarning() */


/*
 * FUNCTION NAME: BrPMError
 *
 * DESCRIPTION:   BRender Error handling function
 *
 */
static void BR_CALLBACK BrPMError(char *message)
{
   WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, message, "ERROR", 256, MB_OK | MB_APPLMODAL | MB_ERROR);
   DosExit(1,10);
} /* End of BrPMError() */
