/*** IWZVCOMP.CMD                                                                     */
/***                                                                                  */
/*** Licensed Material - Property of IBM                                              */
/*** IBM VisualAge COBOL :  5639-B92                                                  */
/*** (C) Copyright IBM Corp. 1995, 1998.  All rights reserved                         */
/*** US Government Users Restricted Rights - Use, duplication or disclosure           */
/*** restricted by GSA ADP Schedule Contract with IBM Corp.                           */
/***                                                                                  */
/*** Program Requirement: REXX installed                                              */
/***                                                                                  */
/*** Language Used:  REXX                                                             */
/*** Program Invocation:  rexx IWZVCOMP cob2_options                                  */
/***                               -PP"CICS(cics_options)" cobolfile1...              */
/***                               linkfile1...                                       */
/***                                                                                  */
/*** Function:                                                                        */
/***   Determines whether CICS preprocessing is required by a COBOL source program,   */
/***   based on whether -PP"CICS()" was specified.                                    */
/***   Does the preprocessing, followed by compile.  Uses temporary files to handle   */
/***   file extension requirements of preprocessors.  Sets APOST/QUOTE option for     */
/***   CICS preprocessor and compiler.                                                */
/***                                                                                  */
/*** INPUT: cob2_options (can be -Q"xxx", -I"xxx", -c, B"xxx" etc.)                   */
/***        cics_options (CICS for Windows NT preprocessor options)                   */
/***        cobolfile1... names of the files to compile)                              */
/***        linkfile1... (names of files to use in the link)                          */
/***                                                                                  */
/***        The cob2_options and cics_options must not contain double quotes (") and  */
/***        any parens in them must be matched.                                       */
/***                                                                                  */
/*** OUTPUT:  filename.PPR (output from preprocessing, input to compiler)             */
/***          filename.OBJ and filename.LST (compiler listing)                        */
/***                                                                                  */
/*** Assumptions:                                                                     */
/***   - File to be preprocessed and compiled is either in current directory or is    */
/***     fully qualified                                                              */
/***   - If COPY statement is used by program, the -I"pathname" compiler option or    */
/***     SYSLIB environment variable has been specified, or the copy file is in the   */
/***     current directory                                                            */
/***   - For SQL coprocessing, DB2 database has already been started                  */
/***                                                                                                     */

'@ECHO OFF'
trace o
q.mytrace = 0

   parse arg q.parm
   /* set variables to be used if link is requested              */

   q.compparm = ''
   q.linkparm = ''

   call RxFuncAdd 'SysFileDelete' , 'RexxUtil', 'SysFileDelete'
   call RxFuncAdd 'SysFileTree','RexxUtil','SysFileTree'
   call RxFuncAdd 'SysFileSearch','RexxUtil','SysFileSearch'
   call RxFuncAdd 'SysTempFileName','RexxUtil','SysTempFileName'
   call RxFuncAdd 'RVAR','iwzmrt','RVAR'

   q.maxgoodrc = 4

   returncode = 0


   call build_msgs

   /* parse the input parameter to get the compiler options, preprocessor options and   */
   /* name of file to preprocess/compile                                                */
   call parseparm

   returncode = max(RESULT,returncode)
   if returncode > q.maxgoodrc then
     do
       call end_program returncode
     end

   call target_main
   call comp_link_parms

   /*
    * If CICS preprocessing was requested, the subroutine PARSEPARM picked out all the .CBL files
    * from the input parameter and put them into variable q.cbls.  Now walk through them one at
    * a time to do the CICS preprocessing converting them to q.pprs
    */
   if pos('CICS', translate(q.sourcetype)) > 0 then
     do
       /* Call CICS preprocessor script */
       call cicsprep
       returncode = max(RESULT,returncode)
       if returncode > q.maxgoodrc then
         do
           call end_program returncode
         end
     end /* end cics preprocessing requested */
   else
     q.pprs = q.cbls

   /* Do the compile and link */
   call cobcomp
   returncode = max(RESULT,returncode)
   if returncode > q.maxgoodrc then
       do
         call end_program returncode
       end

/* call  display_q */
   exit returncode
/*******************************************************************************/
target_main: procedure expose q.
/*
 * only process this routine if it is not comp_only
 * set targetfn to the first name in the OBJ list if target is not specified
 * always produce a DLL if it is CICS application
 * set target extension to EXE if it is not producing a DLL
 * change target extension to IBMCOB if it is running CICSNT
 * use target name to be the MAIN program name if it is not given with -MAIN:
 */

if \q.comp_only then do
  /* set target name */
  if q.target == '' then
     q.target = word(q.objs, 1)
  parse value filespec('name', q.target) with fn '.' ext
  q.targetfn = fn

  /* set target extension */
  if q.cicstype \= '' then do
     q.dll = 1
  end
  q.targetext='exe'
  if q.dll then
     q.targetext='dll'
  if q.cicstype == 'NT' then
     q.targetext='ibmcob'

  q.target = q.targetfn'.'q.targetext
  /* informing change of the target */
  if translate(ext) \= translate(q.targetext) then
    call sayit q.true 'MSG005' q.target||q.sep

  /* getting information of what is the MAIN program*/
  if q.main == '' then
     q.main = q.targetfn

  /* if no DEFfile given, use target name as the name of DEFfile */
  if q.deffile == '' then
     q.deffile = q.targetfn
  else do
     parse value filespec('name',q.deffile) with fn '.' ext
     q.deffile = fn
  end

end
return
/****************************************************************************************************/
comp_link_parms: procedure expose q.
/*
 * setting the non-CICS compiler and linker options
 * adding -qlib to compiler options
 * OS2:   /free /nol /o:target
 * WinNT: /free /nol iwzrwin1.obj iwzrwin2.obj /o:target       (non DLL)
 * WinNT: /free /nol iwzrwiH1.obj iwzrwin2.obj /o:target       (non DLL, -HOST)
 * if DLL is to be produced:
 *   OS2:   /free /nol /dll target.def /o:target
 *   WinNT: /free /nol iwzrwin3.obj /dll target.exp /o:target
 */

   if \q.showcompmsgs then   /* messages should be shown via adata error feedback */
     q.compparm = add_option('-qnoterminal' q.compparm)

   if \q.link_only then do
     q.compparm = add_option('-qlib' q.compparm)
   end
   if \q.comp_only then
   do
     /* adding linker options */
     /*#14051: #13968 replaced ILINK w COB2, the following stmts are no longer needed
      *#14051:   q.linkparm = ADD_OPTION('/free' q.linkparm)
      *#14051:   q.linkparm = ADD_OPTION('/nol'  q.linkparm)
      *#14001: insert '/O:' in front of target if not building a DLL nor .IBMCOB'
      */

     if q.dll then do
       q.compparm = ADD_OPTION('-dll:'q.target  q.compparm)
       q.linkparm = ADD_OPTION('-dll:'q.target  q.linkparm)
     end
     else
       q.linkparm = q.linkparm '/O:'q.target         /*#14001*/

/******************** #14051 *******************************************
 * the following statments are removed in #14051 because these are ILINK's options supplied by COB2
 *   if q.windows then do
 *     cobdir = getenv('COBOLMAIN')
 *     if q.dll then do /* WINNT DLL */
 *       q.linkparm = ADD_OPTION(cobdir'\LIB\IWZRWIN3.OBJ' q.linkparm)
 *       q.linkparm = ADD_OPTION(q.deffile'.exp' q.linkparm)
 *     end              /* WINNT DLL */
 *     else do /(* WinNT exe *)/
 *       if pos('-HOST', translate(q.compparm)) > 0 then
 *         q.linkparm = ADD_OPTION(cobdir'\LIB\IWZRWIH1.OBJ' q.linkparm)
 *       else
 *         q.linkparm = ADD_OPTION(cobdir'\LIB\IWZRWIN1.OBJ' q.linkparm)
 *       q.linkparm = ADD_OPTION(cobdir'\LIB\IWZRWIN2.OBJ' q.linkparm)
 *     end     /(* WinNT exe *)/
 *   end
 *   else do  /* OS2 */
 *     if q.dll then do /* OS2 DLL */
 *********************#14051 ******************************************/
     if \q.windows & q.dll then do /* OS2 DLL */ /*#14051*/
         q.linkparm = q.linkparm q.deffile'.def'
       end              /* OS2 DLL */
 /*  end *14501*/  /* OS2 */
   end

return
/****************************************************************************************************/
   cicsprep: procedure expose q.

   call sayit q.compile 'MSG001' q.cbls||q.sep
   /* say 'Starting CICS preprocessing for' trlcbl '...' */
   if q.cicstype == 'AD' then
     call cicsadprep
   else
     call cicsntprep

   /* loop through the cobol source input files and preprocess them */
   q.pprs = ''
   cicsrc = 0
   do i = 1 to words(q.cbls)
     file = word(q.cbls, i)
     if \file_exist(file) then
       call end_program q.maxgoodrc+4 q.true 'MSG016' file||q.sep
     else do
       filename = parse_filename(file)
       if q.cicstype == 'AD' then
         trancmd = 'faaprpdc' filename q.cicsopt                       /*CICSAD*/
       else
         trancmd = 'cicstran' q.cicsopt filename                       /*CICSNT*/
       if (\q.link_only) then do
         if (\q.compile) then
           cicsrc = exec_cmd(q.true q.show q.compile trancmd) /* show trancmd only */
         else
           cicsrc = max(cicsrc, translate_cbl(filename trancmd))
       end /*if (link_only=='no') then do*/
       q.pprs = strip(q.pprs filename'.PPR')
     end
   end /* loop through input files */

/* say 'CICS preprocessing complete, return code =' cicsrc*/
   call sayit q.compile 'MSG002' cicsrc||q.sep
   return cicsrc
/****************************************************************************************************/
cicsntprep: procedure expose q.
   call cicsntenv

   /* add -lIBMCOB to q.cicsopt, remove COBOL2 from q.cicsopt */
   q.cicsopt = ADD_OPTION('-lIBMCOB' q.cicsopt)
   q.cicsopt = DEL_OPTION('COBOL2' q.cicsopt)

   cicspath = getenv('CICSPATH')

   if \q.link_only then do
     q.compparm = q.compparm '-qthread'
     q.compparm = ADD_OPTION('-I'cicspath'\INCLUDE ' q.compparm)
   end /*  if link_only=='no' then do*/

   if \q.comp_only then do
     q.linkparm = ADD_OPTION(getenv('COBOLMAIN')'\LIB\IWZRWIN4.OBJ' q.linkparm)
     q.linkparm = ADD_OPTION(cicspath'\LIB\cicsprIBMCOB.lib' q.linkparm)
   end /*if comp_only=='no' then do*/

return
/****************************************************************************************************/
cicsadprep:  procedure expose q.

   call cicsadenv
   /* set cobsw to '+S5' to capture FAAPRPDC's output; i.e., the return code*/
   cobsw=value('COBSW','+S5',q.environ)

   /* add COBOL2 to q.cicsopt, removing -lIBMCOB from q.cicsopt  */
   q.cicsopt = ADD_OPTION('COBOL2' q.cicsopt)
   q.cicsopt = DEL_OPTION('-lIBMCOB' q.cicsopt)

   if \q.comp_only then do
     q.linkparm = ADD_OPTION('FAACBID.LIB'  q.linkparm)
     q.linkparm = ADD_OPTION('FAAOTSCB.LIB' q.linkparm)
     q.linkparm = ADD_OPTION('FAASR32.LIB'  q.linkparm)
     if \q.windows then do   /* OS2   CICS */
       q.linkparm = ADD_OPTION('IWZRLIB.LIB' q.linkparm)
       q.linkparm = ADD_OPTION('OS2386.LIB' q.linkparm)
       q.linkparm = ADD_OPTION('PA2CLOCK.LIB' q.linkparm)
     end                     /* OS2   CICS */
     else                    /* WinNT CICS */
       q.linkparm = ADD_OPTION(getenv('COBOLMAIN')'\LIB\IWZRWIN4.OBJ' q.linkparm)
   end /*if comp_only=='no' then do*/

return
/****************************************************************************************************/
   cobcomp:  procedure expose q.
/*
 * delete all pre-existed OBJs
 * COB2 -C compparm cbllist
 * do the linking if not comp_only:
 *   process the DEF file if DLL is to be produced
 *   ILINK objs linkparm
 */
       comprc = 0

    /* say 'Starting COBOL compile of' q.cbls '...'*/
       call sayit (q.compile|q.link) 'MSG003' cbls_and_objs()||q.sep

       /* user can control whether the compiler creates an .obj file, using NOCOMPILE(rc)*/
       /* compiler option (which could be specified within the source code).  So we want */
       /* to make very sure that the link doesn't proceed with an old .obj. If an .obj   */
       /* exists before the compile, erase it                                            */
       call del_old_objs \q.link_only q.cbls

       rc = exec_cmd(\q.link_only q.show q.compile 'COB2 -C' q.compparm q.pprs)
       if (rc <= q.maxgoodrc) & (\q.comp_only) then do
         if q.windows & q.cicstype \= '' then do
           if \file_exist(q.deffile'.def') then do
             rc = exec_cmd(q.windows q.show q.link 'ILIB /nol /gd:'q.deffile q.objs)
             call MOD_DEF q.link
           end /*if \file_exist(q.deffile'.def') then do*/
           /*#14001: use target name as name for .exp & .lib
            *if \file_exist(q.deffile'.exp') | \file_exist(q.deffile'.lib') then
            *  rc = exec_cmd(q.windows q.show q.link 'ILIB /nol /gi:'q.deffile q.deffile'.def')
            *q.linkparm = ADD_OPTION(q.deffile'.def' q.linkparm)
            *q.linkparm = ADD_OPTION(q.deffile'.exp' q.linkparm)
            */
           if \file_exist(q.targetfn'.exp') | \file_exist(q.targetfn'.lib') then
             rc = exec_cmd(q.windows q.show q.link 'ILIB /nol /gi:'q.targetfn q.deffile'.def')
           q.linkparm = ADD_OPTION(q.deffile'.def' q.linkparm)
           q.linkparm = ADD_OPTION(q.targetfn'.exp' q.linkparm)
/* including q.deffile.lib will trigger the ILIB after ILINK to produce final .imp & .lib
 * but ILIB was having problem with CICSAD where all FAA*.lib are included, it couldn't find them
 * so the following statement is commented out for now until COB2 is fixed 2/11/98
           q.linkparm = ADD_OPTION(q.deffile'.lib' q.linkparm)
 */
         end  /* if q.windows & q.cicstype \= '' */
         else do  /* OS2 */
           if \q.windows & q.dll & \file_exist(q.deffile'.def') then
             call MOD_DEF q.link
         end      /* OS2 */
         /*#13968: use COB2 instead of ILINK
          *rc = exec_cmd(q.true q.show q.link 'ILINK ' q.objs q.linkparm)
          */
         rc = exec_cmd(q.true q.show q.link 'COB2  ' q.objs q.linkparm)
       end /* rc from COB2 <= 4 */
       comprc = rc
       call sayit (q.compile|q.link) 'MSG004' comprc||q.sep
       /* say 'COBOL compile complete, return code =' comprc*/

       /* the compiler can give negative return codes.  Fatal error when this happens */
       if comprc < 0 then
         if comprc = -1 then comprc = 12
         else comprc = 16

   return comprc
/****************************************************************************************************/
   /* parse input parameter */
   parseparm:  procedure expose q.
   /* 0: no; 1: yes */

   q.cicsopt   = ''
   q.cicstype  = ''
   q.sourcetype = ''
   q.show    = 0
   q.compile = 1
   q.link    = 1
   q.comp_only = 0
   q.link_only = 0
   q.showcompmsgs = 0
   q.objs = ''
   q.cbls = ''
   q.target = ''
   q.main = ''
   q.dll = 0
   q.deffile = ''

   optrc = 0
   /*
    *Feature 13532: Adding support of accepting response file as input arguments to IWZVCOMP.
    *i.e., IWZVCOMP @res_file
    */
   Do While pos('@', q.parm) > 0
     q.parm = expand_resfile('@' q.parm)
   end
/* say 'iwzvcomp' strip(q.parm) */
   /*
    * keep a copy of the input argument
    * will be used as input to COB2 for the non-CICS application
    */
   q.keepparm = q.parm
   /* process the options.  they begin with '-' */
   q.parm = strip(q.parm)
   do while q.parm <> '' & translate(substr(q.parm,1,1)) == '-'

      /* figure out what kind of option it is */
      select
      when translate(substr(q.parm,1,4)) == '-PP"' then
         call ppoptions
      /* #13968: use COB2 instead of ILINK & process -host -g -p -cmain
       * as common options in compparm & linkparm to COB2
       */
      when wordpos(translate(word(q.parm,1)), '-HOST -G -P -CMAIN') > 0 then do
          parse var q.parm opt q.parm
          q.compparm = q.compparm opt
          q.linkparm = q.linkparm opt
        end
    /*#13968: combine the following statements in the preceeding block
     *When translate(substr(q.parm,1,3)) == '-P ' then
     *  do
     * *#13532 q.linkparm = ADD_OPTION('/DE' q.linkparm)
     * *#13532 q.linkparm = ADD_OPTION('IWZPAN40.OBJ' q.linkparm)
     *  parse var q.parm opt q.parm
     * *#13756 q.compparm = q.compparm opt
     *  q.linkparm = q.linkparm opt
     *  end
     */
      when translate(substr(q.parm,1,4)) == '-DLL' then
        do
          q.dll = 1
          parse var q.parm opt q.parm
/*        #14001: add to compparm & linkparm in comp_link_parm after target is determined
 *        q.compparm = q.compparm opt
 *        q.linkparm = q.linkparm opt
 */
        end
      when translate(substr(q.parm,1,6)) == '-MAIN:' then
        do
          parse var q.parm opt q.parm
          parse var opt front':'q.main
          q.compparm = q.compparm opt
          q.linkparm = q.linkparm opt
        end
      when translate(word(q.parm,1)) == '-#' then
        do
          q.show = 1
          q.compile = 0
          q.link    = 0
          parse var q.parm opt q.parm
          q.compparm = q.compparm opt
          q.linkparm = q.linkparm opt
        end
      when translate(word(q.parm,1)) == '-V' then
        do
          q.show = 1
          parse var q.parm opt q.parm
          q.compparm = q.compparm opt
          q.linkparm = q.linkparm opt
        end
      when translate(word(q.parm,1)) == '-C' then
        do
          q.link    = 0
          q.comp_only = 1
          parse var q.parm opt q.parm
        end
      when translate(word(q.parm,1)) == '-L' then
        do
          q.compile = 0
          q.link_only = 1
          parse var q.parm opt q.parm
        end
      when translate(substr(q.parm,1,3)) == '-B"' then do
        q.linkparm = q.linkparm option_value('-B"' '"' q.parm)
        if pos('/DLL', translate(q.linkparm)) > 0 then
           q.dll = 1
        end
      otherwise
        q.compparm = q.compparm compoptions(q.parm)
      end /* select */

      q.parm = strip(q.parm)

   end /* while loop to process options */

   /* assume that the rest of the parm has the name(s) of the file(s) to preprocess/compile and link */
   /* pws - assume multiple cobol source files; other file type are assumed to be link files.        */
   /* loop through the rest of the parm to identify the files to compile vs. link.  Note:  the files */
   /* recognized as input to the compiler are those with extensions .cbl, .sqb, and .ccp.            */
   do while q.parm <> ''

     q.parm = strip(q.parm)
     parse var q.parm opt q.parm

     select
        when pos('.OBJ',translate(opt)) <> 0  then call OBJ_LIST opt
        when pos('.LIB',translate(opt)) <> 0  then q.linkparm = q.linkparm opt
        when pos('.DEF',translate(opt)) <> 0  then q.deffile = opt
        when pos('.MAP',translate(opt)) <> 0  then q.linkparm = q.linkparm opt
        when pos('.EXP',translate(opt)) <> 0  then q.linkparm = q.linkparm opt
        when pos('.EXE',translate(opt)) <> 0  then q.target = opt
        when pos('.DLL',translate(opt)) <> 0  then do
          q.target = opt
          q.dll = 1
        end
        when pos('.IBMCOB',translate(opt)) <> 0  then do
          q.target = opt
          q.dll = 1
        end
        otherwise /* assume a cobol source file */
          do
            q.cbls = strip(q.cbls opt)
            call OBJ_LIST opt
          end
     end  /* end select  */
   end  /* while loop to process file names */

   /* Check the compiler options to make sure everything is there to enable       */
   /* compiler error message feedback (done via adata).  If not make sure we      */
   /* show the vanilla compiler messages.                                         */
   if pos(' NOADATA',translate(q.keepparm))>0 | pos('ADEXIT(IWZRMGUX)',translate(q.keepparm))=0 then
       q.showcompmsgs = 1

   if q.mytrace ==1  then
   do
     call display_q
     say 'Trace complete'
   end  /* end do */

   return optrc
/****************************************************************************************************/
   GET_TEMPDIR:  procedure expose q.

   /* make sure the temporary directory name is in a usable form:        */
   /* - if there are multiple directories specified, use the first one   */
   /* - if the directory ends in \ and\or ;, strip it off                */
   /* - make sure hpfs directory names are handled properly              */
   /* Note:  this is used only for CICS preprocessing                    */
   tempdir  = getenv('TMP')
   tempdirhpfs = 'n'
   if length(tempdir) >0 then /* if temp directory env var set up */
     if substr(tempdir,1,1) = '"' then /* if hpfs name */
       do
         endpos = pos('"',tempdir,2) /* look for closing quote */
         if endpos = 0 then /* no closing quote */
           do
        /*   say 'Error:  no closing quote found in TMP environment variable'*/
             call sayit q.true 'MSG010'
             sqlrc = q.maxgoodrc + 4
             return sqlrc
           end
         else /* closing quote found in hpfs name */
           do
             tempdir = substr(tempdir,1,endpos-1) /* temporarily remove ending quote */
             if lastpos('\',tempdir) = length(tempdir) then  /* if temp directory name ends in backslash */
               tempdir = substr(tempdir,1,length(tempdir)-1) /* remove backslash */
             tempdir = tempdir || '"' /* put closing quote back on */
             tempdirhpfs = 'y'
           end
       end
     else /* not an hpfs name */
       do
         if pos(';',tempdir) > 0 then  /* could be multiple paths strung together */
           tempdir = substr(tempdir,1,pos(';',tempdir)-1)  /* pick up first path */
         if lastpos('\',tempdir) = length(tempdir) then  /* if temp directory name ends in backslash */
           tempdir = substr(tempdir,1,length(tempdir)-1) /* remove backslash */
       end
   else /* no temp directory specified */
     do
    /* say 'Error:  Must set TMP environment variable to a temporary directory name in order to compile.'*/
       call end_program q.maxgoodrc+4 q.true 'MSG008'
     end

   RETURN
/*********************************************************************************************************/
   END_PROGRAM:  procedure expose q.
   parse arg returncode text
/* if text \= '' then say text*/
/* say 'Processing halted.'*/
   if text \= '' then call sayit text
   call sayit q.true 'MSG006'
   EXIT returncode
   RETURN returncode

/*********************************************************************************************************/
OBJ_LIST: procedure expose q.
/* parse opt into objfn '.' fileext
   then appending objfn.obj to q.objs
 */

parse arg opt

parse value filespec('name',opt) with objfn '.' fileext
nobj = objfn'.obj'
q.objs = strip(q.objs nobj)
return
/*****************************************************************************************/
MOD_DEF: procedure expose q.
/*
 * called during the link step after ILIB /gd to pick out the main-entry point
 * Since ILIB /gd is not available on OS2, it calls MKDEF to create the initial DEF file
 * Candidates for mainentry are: 1. name entered thru -main:
 *                               2. name of the target
 *                               3. name of the OBJ if there is only one OBJ
 * find the mainentry in deffile, add "@1" to signify it is mainentry
 * and insert FAA_PROGRAM_ID @2 if it is using CICSAD.
 * If mainentry can't be identified, we want to leave the original deffile unaltered
 * for user to debug
 */
parse arg doit

if doit then do
  tmpdef = date(d)||time(s)'.def'
  deff = q.deffile'.def'
  call mkdef q.link deff q.objs   /* make sure deffile exist*/
  call rename deff tmpdef

  /* setting up the candidate list */
  name.1 = q.main
  name.2 = q.targetfn
  name.0 = 2      /* 2 entries in name. */
  msg = name.1||q.sep||name.2||q.sep||'having more than one OBJs'||q.sep
  if words(q.objs) == 1 then do
    name.3 = parse_filename(word(q.objs, 1))
    name.0 = 3    /* 3 entries in name. */
    msg = name.1||q.sep||name.2||q.sep||name.3'.obj'||q.sep
  end

  /* matching entry points in DEF file with the entries in candidate list */
  index = 0
  found.0 = 0
  Do while (index < name.0 & found.0 < 1)
    index = index + 1
    mainentry = q.pre_mainentry||strip(translate(name.index))||q.post_mainentry
    rc = SysFileSearch(mainentry, tmpdef, 'found.')
  end /*Do while (index < name.0 & found.0 < 1) */

  target_found = 0
  /*
   * change tmpdef to deffile with the CICS syntax: adding "@1" after mainentry
   * and FAA_PROGRAM_ID @2 if running CICSAD
   */
  if (found.0 >= 1) then do
    if q.windows then
      rc = lineout(deff, 'LIBRARY '||q.target )
    Do while (lines(tmpdef))
      line = linein(tmpdef)
      if (target_found < 1) then do
        words_in_line = words(line)
        if (words_in_line > 1 ) then
          if ( (word(line, words_in_line) == 'DECORATED') ,
          & pos(mainentry,translate(word(line, words_in_line -1))) == 1) then do
            posat = pos(' DECORATED', line)
            line = substr(line,1, posat-1) q.key_mainentry
            target_found = 1
          end     /* if ( (word(line, words(line)) == 'DECORATED') then do */
      end
      if \q.windows then
        parse var line line 'DECORATED'      /* os2 */
      rc = lineout(deff, line)

      if (q.cicstype == 'AD') & (pos('@1', line) > 0) then
        rc = lineout(deff, q.tab||q.pre_mainentry||'FAA_PROGRAM_ID @2')
    end /* Do while */
  end /* if (found.0 >= 1) then do */

  rc = stream(tmpdef, 'C','CLOSE')
  rc = stream(deff,'C','CLOSE')
  if (target_found < 1) then do
  /*
   * We didn't find the main entry point in the DEF file, so let's
   * restore the original DEF from tmpdef for the user (or service) to
   * look at it in the event if this error.
   */
    call rename tmpdef deff
    call sayit q.true 'MSG009' msg
    call end_program q.maxgoodrc+4 q.true 'MSG017' q.deffile'.def'||q.sep
  end
  else /* deffile successfully modified to CICS format */
    rc = SysFileDelete(tmpdef)
end /*if doit then do*/
return
/*******************************************************************************/
ADD_OPTION: procedure expose dummy
parse arg opt parm

newopt = translate(opt)
if (pos(newopt, translate(parm)) < 1) then
  parm = parm opt
return parm
/*******************************************************************************/
/* remove opt from parm */
DEL_OPTION: procedure expose dummy
parse arg opt parm

opt = translate(opt)
opt_at = wordpos(opt, translate(parm))
Do while opt_at > 0
  head = substr(parm, 1, wordindex(parm, opt_at) - 1)
  tail = substr(parm, wordindex(parm, opt_at) + length(opt))
  parm = head || tail
  opt_at = wordpos(opt, translate(parm))
end
return parm
/********************************************************************************/
ppoptions:  procedure expose q.
/*
 * process the preprocessor options
 * fill in the values for CICSOPT and determine the sourcetype
 * set parm & keepparm to exclude the -PP phrase thru the call to option_value
 */

  ppopt = option_value('-PP"' '"' q.parm)
  do while ppopt <> ''
    lprenpos = find_op(1 '(' ppopt)

    /* find keyword */
    keyword = translate(substr(ppopt,1,lprenpos-1))

    /* find closing pren  */
    rprenpos = find_op(1 ')' ppopt)  /* first right pren  */

    /* find next left pren  */
    nlprenpos = pos('(', ppopt, lprenpos+1)

    /* look for other () sets */
    do while (nlprenpos < rprenpos) & (nlprenpos <> 0)
      rprenpos = find_op(rprenpos+1 ')' ppopt)
      nlprenpos = pos('(',ppopt,nlprenpos+1)
    end /* do */
    opt   = substr(ppopt,lprenpos+1,rprenpos-lprenpos-1)
    ppopt = strip(substr(ppopt,rprenpos+1))

    select
      when keyword = 'CICSNT' then
        do
          q.sourcetype = q.sourcetype 'CICSNT'
          q.cicsopt = strip(q.cicsopt opt)
          q.cicstype= 'NT'
        end
      when keyword = 'CICS' then
        do
          q.sourcetype = q.sourcetype 'CICSAD'
          q.cicsopt = strip(q.cicsopt opt)
          q.cicstype= 'AD'
        end
      otherwise
        do
       /* say 'Error: Unrecognized preprocessor option' keyword ', was discarded'*/
          call sayit q.true 'MSG011' keyword||q.sep
        end
     end  /* select */
   end  /* end of -pp loop     */
return
/*****************************************************************************/
compoptions: procedure expose q.
/*
 * return the quoted or un-quoted compiler option to be included in q.compparm
 * set parm to exclude the option just processed thru the call to option_value
 */
parse arg parmstr
         q.parm = q.parm || ' ' /* add a blank to the end of the parm to make the IF easier */
         blankpos = pos(' ',q.parm)
         quotepos = pos('"',q.parm)
         if quotepos>0 & quotepos<blankpos then /* if parm has quotes */
           do
             prequote = substr(q.parm, 1, quotepos)
             opt = prequote|| option_value(prequote '"' q.parm)||'"'
           end
         else /* parm does not have quotes */
             parse var q.parm opt q.parm
return opt
/***************************************************************************/
option_value: procedure expose q.
/*
 * returns the value of the option delimited by KEYSTART and KEYEND
 * also set parm to a new value ready for parsing for a new option
 */
  parse arg keystart keyend parmstr
  keylength = length(keystart) + 1
  qpos = find_op(keylength keyend q.parm)  /* find ending */

  opt = substr(q.parm,1,qpos) /* option, including keystart and keyend*/

  /* This is a special preprocessing option not understood by COB2 */
  /* or the linker... remove it from  keepparm.                    */
         if keystart == '-PP"' then do
            ppstart = pos(keystart, translate(q.keepparm))
            q.keepparm = delstr(q.keepparm, ppstart, length(opt))
            end

  q.parm = strip(substr(q.parm,qpos+1))

  opt = substr(opt,keylength,qpos-keylength) /* remove surrounding keys */
return opt
/*************************************************************************/
find_op: procedure expose q.
  parse arg startpos op str
  findat = pos(op,str,startpos)
  if findat == 0 then      /* not finding  */
  do
  /* say 'Error: Unmatched ' op*/
     call sayit q.true 'MSG012' op||q.sep
     exit q.maxgoodrc+4
  end /* end do  */
return findat
/*************************************************************************/
/* set environment for CICSAD */
cicsadenv: procedure expose q.

cicsset = translate(getenv('CICSSET'))

if (cicsset \= 'ON') then do
  if \q.windows then
    call 'CICSENV'     /* os2 */
  else do              /* Windows */
    ibmcicsad = getenv('IBMCICSAD')
    if (ibmcicsad == '') then do
    /* call end_program q.maxgoodrc+4 'Error: CICSAD is not installed on this machine.'*/
      call end_program q.maxgoodrc+4 q.true 'MSG015' 'CICSAD'||q.sep
    end /* CICSAD not installed */
    infile = ibmcicsad'\RUNTIME\CICSENV.CMD'
    if (\file_exist(infile)) then
      call end_program q.maxgoodrc+4 q.true 'MSG016' infile||q.sep
    else do
      /* read through ibmcicsad\RUNTIME\CICSENV.CMD, process the SET commands */
      /* straight execution of CICSENV.CMD won't get the environment variables*/
      /* set because it's like executing in a separate shell                  */
      Do while (lines(infile))
        line = translate(linein(infile))
        firstword = word(line, 1)
        if (pos('@', firstword) = 1) then
          firstword = substr(firstword, 2)
        if firstword == 'SET' then call setadenv line
      end /* Do while (lines(infile)) */
      rc = stream(infile,'C','CLOSE')
    end /* CICSAD installed, %ibmcicsad% defined */
  end /* windows */
end /*if (cicsset \= 'ON') then do */

return
/*************************************************************************/
setadenv:  procedure expose q.
parse arg line
parse var line word1 env'='envval
/*trace r*/
env = strip(env)
envval=strip(envval)
Do While (pos('%', envval) > 0)
  first = pos('%', envval)
  second = find_op(first+1 '%' envval)
  head   = substr(envval, 1, first - 1)
  subenv = substr(envval, first+1, second - first -1)
  tail   = substr(envval, second+1)
  /*
  subenvval = value(subenv,,q.environ)
  */
  subenvval = getenv(subenv)
  envval = head||subenvval||tail
end /* Do while pos('%', envval) > 0 */
oenv = value(env,envval,q.environ)
/*trace o*/
/*say "Setting" env "to" value(env,,'ENVIRONMENT')*/
return
/****************************************************************************************/
translate_cbl:  procedure expose q.
/* backup original fn.cbl to tmpcbl, fn.ccp to tmpccp                                    */
/* fn.trl contains the log of the CICSAD translation.  Pre-existed fn.trl won't be saved */
/* rename fn.cbl to fn.ccp, remove fn.trl ready to contain msg from CICSTRAN             */
/* process trancmd                                                                       */
/* rename translated fn.cbl to fn.ppr if trancmd was successful                          */
/* remove temperory files and restore backuped original .cbl, .ccp...                    */

parse arg filename trancmd
       tempname = date(d)||time(s)
       trlcbl = filename || '.cbl'
       tmpcbl = tempname || '.cbl'
       trlccp = filename || '.ccp'
       tmpccp = tempname || '.ccp'
       ppfile = filename || '.ppr'

       call rename trlccp tmpccp
       if file_exist(trlcbl) then
         '@copy' trlcbl trlccp q.nul
       else
         '@copy' tmpccp trlccp q.nul
       call rename trlcbl tmpcbl

       cicsrc = exec_cmd(q.true q.show q.compile trancmd)

        /* check to see if trlccp got converted to trlcbl */
        if (\file_exist(trlcbl)) then do
          cicsrc = max(cicsrc,q.maxgoodrc+8)
          /* say 'Error:  Failure converting' trlcbl */
          call sayit q.true 'MSG007' trlcbl||q.sep
        end
        if cicsrc <> 0 then do
          if cicsrc = '' | datatype(cicsrc) \= 'NUM' then do /* returncode not set! */
            /* say 'Error:  Null return code returned from CICS preprocessing.' trlcbl*/
            call sayit q.true 'MSG013' trlcbl||q.sep
            cicsrc = 16
            end
          else do /* return code was set */
            cicsrc = max(cicsrc,q.maxgoodrc+4)
            /*say 'Error:  Return code' cicsrc 'returned from CICS preprocessing' trlcbl*/
            call sayit q.true 'MSG014' cicsrc||q.sep||trlcbl||q.sep
            end
          call SysFileDelete trlcbl
        end      /* cicsrc <> 0 */
        else do  /* cicsrc == 0 */
          /* Rename the translated CBL to take on .ppr extension */
          call rename trlcbl ppfile
        end       /* cicsrc == 0 */

        /* cleanup, restore original .cbl .ccp*/
        call rename tmpccp trlccp
        call rename tmpcbl trlcbl
return cicsrc
/******************************************************************************/
/* set PATH=%CICSPATH%\BIN;%path% if CICSNT is installed                      */
cicsntenv:
   cicspath = getenv('CICSPATH')
   if cicspath == '' then do
     /* call end_program q.maxgoodrc+4 'Error:  CICSNT is not installed.'*/
     call end_program q.maxgoodrc+4 q.true 'MSG015' 'CICSNT'||q.sep
   end /* CICSNT is not installed */
   call addenv cicspath'\BIN' 'PATH'
return
/******************************************************************************/
/* appending VARSTR to the front of the ENV and make sure it only appears once*/
/* in the setting of ENV                                                      */
addenv:  procedure expose q.
  parse arg varstr env
  varstr = translate(varstr)';'
  envval = translate(strip(value(env,,q.environ)))';'
  envval = translate(getenv(env))';'
  Do While (pos(varstr, envval) > 0)
    envval = pre_key(varstr envval) || post_key(varstr envval)
  end /*Do While (pos(varstr, envval) > 0)*/
  envval = replace_key(varstr||envval ';;' ';')
  oenvval = VALUE(env,envval, q.environ)
/*say env 'is set to: ' value(env,,q.environ)*/
return
/******************************************************************************/
/* return the substring prior to KEY; if KEY not found, returning NIL */
pre_key: procedure expose dummy
  parse arg key string
  str = ''
  if (key \= '') then do
    key_at = pos(key, string)
    if (key_at > 1)  then
      str = substr(string, 1, key_at - 1)
  end     /* key \= '' */
return str
/******************************************************************************/
/* return the substring following KEY; if KEY not found, returning entire string */
post_key: procedure expose dummy
  parse arg key string
  str = string
  if (key \= '') then do
    key_at = pos(key, string)
    if key_at > 0 then
      str = substr(string, key_at + length(key))
  end /*if (key \= '') then do*/
return str
/******************************************************************************/
replace_key: procedure expose dummy
  parse arg string key1 key2
  if key1 \= '' then do
    Do while (pos(key1, string) > 0)
      string = pre_key(key1 string) || key2 || post_key(key1 string)
    end /*Do while (pos(key1, string) > 0)*/
  end /*if key1 \= '' then do*/
return string
/******************************************************************************/
file_exist:  procedure expose dummy
  parse arg file
  call SysFileTree file, 'found', 'F'
  IF RESULT = '0' THEN do
    IF found.0 = 0 THEN
      exist = 0  /* file not there */
    else
      exist = 1  /* file there */
  end
return exist
/******************************************************************************/
exec_cmd:  procedure expose rc
parse arg doit show run command
rc = 0
if doit then do
  if (show) then
    say command
  if (run) then
    address cmd command
end /*if doit then do*/
return rc
/******************************************************************************/
parse_filename:  procedure expose q.
parse arg fn
      /* Get file's path and verify file exists */
      filedrv = filespec('drive',fn)
      filepath = filespec('path',fn)
      if filepath \= '' then filepath = substr(filepath,1,length(filepath)-1)
      parse value filespec('name',fn) with filename '.' fileext

 /*currdir = directory()
   currdrv = filespec('drive',currdir)
  */
      filedir = filedrv || filepath
      if filedir = '' then filedir = directory()

      /* Construct fully qualified filename */
      file = filedir || '\' || filename
      if fileext \= '' then file = file || '.' || fileext

      if q.mytrace == 1 then
        do
          say 'file to be parsed: ' fn
          say 'filedrv=' filedrv
          say 'filepath=' filepath
          say 'filedir=' filedir
          say 'filename=' filename
          say 'file=' file
        end
  /*     if (\file_exist(file)) then
          call end_program q.maxgoodrc+4 q.true 'MSG016' file||q.sep
   */
/*        call end_program q.maxgoodrc+4 'Error:  File' file 'not found'*/
return filename
/******************************************************************************/
parse_extension:  procedure expose q.
parse arg fn
      /* Get file's path and verify file exists */
      filedrv = filespec('drive',fn)
      filepath = filespec('path',fn)
      if filepath \= '' then filepath = substr(filepath,1,length(filepath)-1)
      parse value filespec('name',fn) with filename '.' fileext

      filedir = filedrv || filepath
      if filedir = '' then filedir = directory()

      /* Construct fully qualified filename */
      file = filedir || '\' || filename
      if fileext \= '' then file = file || '.' || fileext

      if q.mytrace == 1 then
        do
          say 'file to be parsed: ' fn
          say 'filedrv=' filedrv
          say 'filepath=' filepath
          say 'filedir=' filedir
          say 'filename=' filename
          say 'file=' file
        end
return fileext
/******************************************************************************/
sayit: procedure expose q.
/* parse q.num by q.sep into text.                                            */
/* parse parms by q.sep into parm.                                            */
/* construct message text by appending parts of text and parm alternately     */
/* put out constructed message                                                */
parse arg doit num parms

if doit then do
  texts = q.num
  m = 0
  Do while texts <> ''
    m = m + 1
    text.m = pre_key(q.sep texts)
    texts = post_key(q.sep texts)
  end /*Do while text <> ''*/

  p = 0
  Do while parms <> ''
    p = p + 1
    parm.p = pre_key(q.sep parms)
    parms = post_key(q.sep parms)
  end /*Do while parms <> ''*/

  msg = ''
  Do i = 1 to m
    msg = msg||text.i
    if i <= p then
      msg = msg||parm.i
  end /*Do i = 1 to m */
  say msg
end /*if doit then do*/
return
/******************************************************************************/
EN_US_msg:  procedure expose q.
q.MSG001 = 'Starting CICS preprocessing for '||q.sep||'...'||q.sep
q.MSG002 = 'CICS preprocessing complete, return code = '||q.sep||'.'||q.sep
q.MSG003 = 'Starting COBOL compile for '||q.sep||'...'||q.sep
q.MSG004 = 'COBOL compile complete, return code = '||q.sep||'.'||q.sep
q.MSG005 = 'Target of the build is named to '||q.sep||'.'||q.sep
q.MSG006 = 'Processing halted.'||q.sep
q.MSG007 = 'Error:  Failure converting '||q.sep||'.'||q.sep
q.MSG008 = 'Error:  Must set TMP environment variable to a temporary directory name in order to compile.'||q.sep
q.MSG009 = "Error:  Main entry point can't be decuced from main:"||q.sep||", target:"||q.sep||' and '||q.sep||'.'||q.sep
q.MSG010 = 'Error:  no closing quote found in TMP environment variable.'||q.sep
q.MSG011 = 'Error:  Unrecognized preprocessor option, '||q.sep||', was discarded.'||q.sep
q.MSG012 = 'Error:  Unmatched '||q.sep||'.'||q.sep
q.MSG013 = 'Error:  Null return code returned from CICS preprocessing.'||q.sep
q.MSG014 = 'Error:  Return code '||q.sep||' returned from CICS preprocessing '||q.sep
q.MSG015 = 'Error: '||q.sep||' is not installed on this machine.'||q.sep
q.MSG016 = 'Error:  File '||q.sep||' not found!'||q.sep
q.MSG017 = '        Please correct '||q.sep||' and rebuild!'||q.sep
return
/******************************************************************************/
JA_JP_msg:  procedure expose q.
q.MSG001 = 'JP:Starting CICS preprocessing for'||q.sep||'...'||q.sep
q.MSG002 = 'JP:CICS preprocessing complete, return code ='||q.sep
q.MSG003 = 'JP:Starting COBOL compile for'||q.sep||'...'||q.sep
q.MSG004 = 'JP:COBOL compile complete, return code ='||q.sep
q.MSG005 = 'JP:Target of the build is named to'||q.sep
q.MSG006 = 'JP:Processing halted.'||q.sep
q.MSG007 = 'JP:Error:  Failure converting'||q.sep
q.MSG008 = 'JP:Error:  Must set TMP environment variable to a temporary directory name in order to compile.'||q.sep
q.MSG009 = "JP:Error:  Main entry point can't be decuced from main:"||q.sep||", target:"||q.sep||'and'||q.sep
q.MSG010 = 'JP:Error:  no closing quote found in TMP environment variable.'||q.sep
q.MSG011 = 'JP:Error:  Unrecognized preprocessor option,'||q.sep||', was discarded.'||q.sep
q.MSG012 = 'JP:Error:  Unmatched'||q.sep
q.MSG013 = 'JP:Error:  Null return code returned from CICS preprocessing'||q.sep
q.MSG014 = 'JP:Error:  Return code'||q.sep||'returned from CICS preprocessing'||q.sep
q.MSG015 = 'JP:Error: '||q.sep||'is not installed on this machine.'||q.sep
q.MSG016 = 'JP:Error:  File'||q.sep||'not found!'||q.sep
q.MSG017 = 'JP:        Please correct'||q.sep||'and rebuild!'||q.sep
return
/******************************************************************************/
build_msgs:  procedure expose q.

q.true  = 1
q.false = 0
q.sep = '/\'
q.tab = '          '
q.windows=Windows() /* 1 for True; 0 for false */
if q.windows then do
  q.environ='ENVIRONMENT'
  q.pre_mainentry='_'
  q.post_mainentry='@'
  q.key_mainentry='@1 DECORATED'
  q.nul='> nul'
end
else do /* os2 */
  q.environ='OS2ENVIRONMENT'
  q.pre_mainentry=''
  q.post_mainentry=''
  q.key_mainentry='@1'
  q.nul='> nul 2> nul'
end     /* os2  */
langlvl = translate(getenv('LANG'))
if langlvl == 'EN_US' then
  call EN_US_msg
else
  call JA_JP_msg
return
/******************************************************************************/
getenv: procedure expose q.
/*
 *Defect 13532: Instead of reading environment variables, RVAR is called to
 *read registry infomation
 */
parse arg env
/*
say "env in getenv: " env
envval = value(env,,q.environ)
*/
envval = RVAR(env)
if envval == '' then
  envval = value(env,,q.environ)
envval = strip(envval)
return envval
/******************************************************************************/
/* return 1 if running Windows; 0 if not */
windows: procedure expose dummy
rt = 1
tmpver = date(d)||time(s)'.ver'
'@ver > ' tmpver
rc = SysFileSearch('Windows', tmpver, 'vertxt')
if vertxt.0 < 1 then
  rt = 0
rc = stream(tmpver, 'C', 'CLOSE')
call SysFileDelete tmpver
return rt
/******************************************************************************/
/* create fn.def for OS2 if it does not already exist                         */
/* get entry names from the name specified in the PROGRAM-ID statement in the */
/* cbl source if there is one, otherwise use the name of the OBJ              */
/* This routine is called inside MOD_DEF                                      */
mkdef: procedure expose q.
parse arg doit file obj_list
if doit then do
if (\file_exist(file)) then do
  rc = lineout(file,';Module definition file for CICS for OS/2 programs')
  rc = lineout(file,'LIBRARY INITINSTANCE')
  rc = lineout(file,'PROTMODE')
  rc = lineout(file,'DATA NONSHARED')
  rc = lineout(file,'CODE LOADONCALL')
  rc = lineout(file,'EXPORTS')
  obj_list=translate(obj_list)
  Do while obj_list <> ''
    parse var obj_list obj'.OBJ' obj_list
    obj_list=strip(obj_list)
    /* look for PROGRAM-ID statement in the corresponding source
     * mask away the quotes around the name found and write it out to the DEFfile
     * if name is not found and put out the name of the OBJ
     */
    output.0=0
    source = find_source(obj q.cbls)
    rc = SysFileSearch(' PROGRAM-ID', source, 'output')
    if output.0 > 0 then do
      line = translate(output.1)
      parse var line ' PROGRAM-ID.' entry'.'
      entry = translate(entry,'','"')
      entry = translate(entry,'',"'")
      entry = strip(entry)
      if entry == '' then entry = obj
      rc = lineout(file,q.tab||entry 'DECORATED')
    end
    else
      rc = lineout(file,q.tab||obj 'DECORATED')
  end /*Do while obj_list <> ''*/
  rc = stream(file,'C','CLOSE')
end /*if (\file_exist(file)) then do*/
end /*if doit then do*/
return
/******************************************************************************/
find_source: procedure expose dummy
/*
 * given obj name, try to look up it's entry on cbllist to find out the extension
 * if neither is found on cbllist, return obj.nothere
 */
parse arg obj cbllist
cbllist = translate(cbllist)
source = ''
Do i = 1 to words(cbllist)
  entry = word(cbllist, i)
  entryfn = parse_filename(entry)
  entryext = parse_extension(entry)
  if entryfn == translate(obj) then
    source = entryfn'.'entryext
end /*Do i = 1 to words(cbllist)*/
if source == '' then
  source = obj'.nothere'
return source
/******************************************************************************/
/* cbllst: name1.PPR name2.PPR name3.PPR....                                 */
/* parse cbllst, delete name1.obj name2.obj name3.obj...                     */
del_old_objs: procedure expose dummy
parse arg doit cbllst
if doit then do
  Do while cbllst <> ''
    parse var cbllst file'.'ext cbllst
    file=strip(file)
    if file_exist(file'.obj') then
      call SysFileDelete file'.obj'
  end /*Do while cbllst <> ''*/
end /*if doit then do*/
return
/******************************************************************************/
display_q: procedure expose q.
say 'q.sep           ' q.sep
say 'q.windows       ' q.windows
say 'q.dll           ' q.dll
say 'q.targetfn      ' q.targetfn
say 'q.targetext     ' q.targetext
say 'q.target        ' q.target
say 'q.main          ' q.main
say 'q.objs          ' q.objs
say 'q.cbls          ' q.cbls
say 'q.pprs          ' q.pprs
say 'q.cicstype      ' q.cicstype
say 'q.environ       ' q.environ
say 'q.pre_mainentry ' q.pre_mainentry
say 'q.post_mainentry' q.post_mainentry
say 'q.key_mainentry ' q.key_mainentry
say 'q.maxgoodrc     ' q.maxgoodrc
say 'q.sourcetype    ' q.sourcetype
say 'q.cicsopt       ' q.cicsopt
say 'q.keepparm      ' q.keepparm
say 'q.compparm      ' q.compparm
say 'q.linkparm      ' q.linkparm
say 'q.parm          ' q.parm
say 'q.mytrace       ' q.mytrace
return
/******************************************************************************/
rename: procedure expose q.
parse arg ofile nfile
call SysFileDelete nfile
'@copy' ofile nfile q.nul
call SysFileDelete ofile
return
/******************************************************************************/
cbls_and_objs: procedure expose q.
/*
 * build a complete list of files to be compile and link; include all the files
 * in q.cbls and for every name in q.objs, see if it's source is in the q.cbls
 * if not, include that name
 */
text = q.cbls
Do i = 1 to words(q.objs)
  obj = word(q.objs, i)
  objfn = parse_filename(obj)
  source = translate(find_source(objfn q.cbls))
  if parse_extension(source) <> 'CBL' & parse_extension(source) <> 'CCP' then
    text = text obj
end /*Do i = 1 to words(q.objs)*/
return text
/******************************************************************************/
expand_resfile: procedure expose q.
/*
 *expanding ori_str with contents in the response file signified by KEY
 *i.e., 111111 @res_file 22222  was the ori_str,
 *      aaaaaaa                 was in res_file,
 *      111111 aaaaaaa 22222    would be the expanded string to be returned to called
 */
  parse arg key ori_str
/*trace r*/
  exp = ori_str
  keyat = pos(key, ori_str)
  if keyat > 0 then do
    front = substr(ori_str, 1, keyat - 1)
    back  = substr(ori_str, keyat)
    parse var back res_file back
    res_file = substr(res_file, 2)
    if \file_exist(res_file) then
       call end_program q.maxgoodrc+4 q.true 'MSG016' res_file||q.sep
    content = ''
    Do while (lines(res_file)) & content == ''
      content = linein(res_file)
    end
    exp = front||content back
  end
/*trace o*/
return exp
/******************************************************************************/
