

#include <math.h>
 
#include "RTTypes.h"
#include "RTMatMac.h"
#include "RTVecOps.h"





/*
 *******************************************************************************
 *
 *  Name:  Cross_Product
 *
 *  Purpose:  this routine calculates the cross product between two input
 *            vectors, A and B, and places the resultant vector in C.
 *
 *
 *
 *  Input Parameters
 *
 *     A - first vector in cross product
 *     B - second vector in cross product
 *
 *
 *  Output Parameters
 *
 *     C - vector result of cross product
 *
 *******************************************************************************
 */
void
   Cross_Product( Vector *A, Vector *B, Vector *C)
      {

       C->x = (A->y * B->z) - (A->z * B->y);

       C->y = (A->z * B->x) - (A->x * B->z);

       C->z = (A->x * B->y) - (A->y * B->x);
      }



/*
 *******************************************************************************
 *
 *  Name:  NCross_Product
 *
 *  Purpose:  this routine calculates the sum of the cross product between
 *            adjacent points in a polygon.  Theresultant vector that is
 *            returned will be normal to the face of the polygon.
 *
 *
 *
 *  Input Parameters
 *
 *     n - number of points in input array
 *     pts - points array representing polygon
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   NCross_Product(int n, Vector pts[])
      {
 //	   Vector Pa, Pb;
       Vector Pcross;
       Vector V_sum;
       int i;

/*
 *    Pa = Vector_Difference(&pts[1], &pts[0]);
 */
     vset(&Pcross, (float) 0.0,  (float) 0.0,  (float) 0.0);
     vset(&V_sum,  (float) 0.0,  (float) 0.0,  (float) 0.0);

       for (i=0; i<=n-1; i++)  {
           Cross_Product(&pts[(i+1)%n], &pts[i], &Pcross);
           V_sum = Vector_Sum(&V_sum, &Pcross);
/*
 *         if (i != n)
 *            Pb = Vector_Difference(&pts[i], &pts[i+1]);
 *         else
 *            Pb = Vector_Difference(&pts[1], &pts[0]);
 *         Cross_Product(&Pb, &Pa, &Pcross);
 *         V_sum = Vector_Sum(&V_sum, &Pcross);
 *         Pa = Pb;
 */
          }
/*
 *     Cross_Product(&pts[0], &pts[n-1], &Pcross);
 *     V_sum = Vector_Sum(&V_sum, &Pcross);
 */
       return(V_sum);
      }



/*
 *******************************************************************************
 *
 *  Name: Dot_Product
 *
 *  Purpose:  this routine calculates the dot product between two input
 *            vectors, A and B, and returns the resulting scaler value
 *
 *
 *
 *  Input Parameters
 *
 *     A - first vector in dot product
 *     B - second vector in dot product
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
float
   Dot_Product(Vector *A, Vector *B)
      {
       float dp;

       dp = (A->x * B->x) + (A->y * B->y) + (A->z * B->z);

       return(dp);
      }



/*
 *******************************************************************************
 *
 *  Name: VectorScaler_Division
 *
 *  Purpose:  this routine divides the specified scaler quantity into the
 *            specified vector and returns the resultant vector.
 *
 *
 *
 *  Input Parameters
 *
 *     vec - specified vector
 *     scaler - specified scaler
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   VectorScaler_Division(Vector *vec, float scaler)
      {
       Vector v;

       v.x = vec->x / scaler;
       v.y = vec->y / scaler;
       v.z = vec->z / scaler;

       return(v);
      }



/*
 *******************************************************************************
 *
 *  Name: VectorScaler_Product
 *
 *  Purpose:  this routine multiplys the specified scaler quantity into the
 *            specified vector and returns the resultant vector.
 *
 *
 *
 *  Input Parameters
 *
 *     vec - specified vector
 *     scaler - specified scaler
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   VectorScaler_Product(Vector *vec, float scaler)
      {
       Vector v;

       v.x = vec->x * scaler;
       v.y = vec->y * scaler;
       v.z = vec->z * scaler;

       return(v);
      }




/*
 *******************************************************************************
 *
 *  Name: Vector_Difference
 *
 *  Purpose:  this routine calculates the vector difference between two input
 *            vectors, v1 and v2, and returns the resulting vector
 *
 *
 *
 *  Input Parameters
 *
 *     v1 - first vector in vector difference
 *     v2 - second vector in vector difference
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   Vector_Difference(Vector *v1, Vector *v2)
      {
       Vector vdiff;

       vdiff.x = v1->x - v2->x;
       vdiff.y = v1->y - v2->y;
       vdiff.z = v1->z - v2->z;

       return(vdiff);
      }



/*
 *******************************************************************************
 *
 *  Name:  Vector_Sum
 *
 *  Purpose:  this routine calculates the vector sum between two input
 *            vectors, v1 and v2, and returns the resulting vector
 *
 *
 *
 *  Input Parameters
 *
 *     v1 - first vector in vector sum
 *     v2 - second vector in vector sum
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   Vector_Sum(Vector *v1, Vector *v2)
      {
       Vector vsum;

       vsum.x = v1->x + v2->x;
       vsum.y = v1->y + v2->y;
       vsum.z = v1->z + v2->z;

       return(vsum);
      }



/*
 *******************************************************************************
 *
 *  Name:  Vector_Average
 *
 *  Purpose:
 *
 *
 *
 *  Input Parameters
 *
 *     v1 - first vector in vector average
 *     v2 - second vector in vector average
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   Vector_Average(Vector *v1, Vector *v2)
      {
       Vector vaverage;

       vaverage.x = (v1->x + v2->x) / ((float)2.0);
       vaverage.y = (v1->y + v2->y) / ((float)2.0);
       vaverage.z = (v1->z + v2->z) / ((float)2.0);

       return(vaverage);
      }



/*
 *******************************************************************************
 *
 *  Name:  Vector_eq
 *
 *  Purpose:  this routine determines if the two specified vectors are
 *            equal in all three components.
 *
 *
 *
 *  Input Parameters
 *
 *     v1 - first vector
 *     v2 - second vector
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Boolean
   Vector_eq(Vector *v1, Vector *v2)
      {
       Boolean vequal = FALSE;

       if ( (fabs(v1->x - v2->x) < 0.2)  &&
            (fabs(v1->y - v2->y) < 0.2)  &&
            (fabs(v1->z - v2->z) < 0.2) )
          vequal = TRUE;

       return(vequal);
      }


/*
 *******************************************************************************
 *
 *  Name:  vgt_lim
 *
 *  Purpose:  this routine determines if any element of the first vector
 *            exceeds the value of the corresponding element of the second
 *            vector.
 *
 *
 *
 *  Input Parameters
 *
 *     v1 - first vector
 *     v2 - second vector
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Boolean
   vgt_lim(Vector *v1, Vector *v2)
      {
       Boolean vgt = FALSE;

       if ( (v1->x > v2->x) ||
            (v1->y > v2->y) ||
            (v1->z > v2->z)    )
          vgt = TRUE;

       return(vgt);
      }


/*
 *******************************************************************************
 *
 *  Name:  Vector_Put_Col
 *
 *  Purpose:  this routine places a vector in a matrix column.  It is used
 *            to create the the G matrix.
 *
 *
 *
 *  Input Parameters
 *
 *     vec - specified input vector
 *     col - matrix column to place vector in
 *
 *
 *  Output Parameters
 *
 *     matrix - matrix to which input vector was added
 *
 *******************************************************************************
 * 
void
   Vector_Put_Col(Vector *vec, float matrix[][4], int col)
      {
       matrix[0][col] = vec->x;
       matrix[1][col] = vec->y;
       matrix[2][col] = vec->z;
      }
 */



/*
 *******************************************************************************
 *
 *  Name:  vector_len
 *
 *  Purpose:  this routine calculates and returns the length of a vector.
 *
 *
 *
 *  Input Parameters
 *
 *     vec - specified vector
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
float
   vector_len(Vector *vec)
      {
       float vector_length = (float) 0.0;

       vector_length = (vec->x * vec->x) + 
                       (vec->y * vec->y) + 
                       (vec->z * vec->z);

       vector_length = sqrt(vector_length);

       return(vector_length);
       }



/*
 *******************************************************************************
 *
 *  Name: vproduct
 *
 *  Purpose:  this routine multiplies two vectors together as three scaler
 *            components.
 *
 *
 *
 *  Input Parameters
 *
 *     v1 - first vector
 *     v2 - second vector
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   vproduct(Vector *vec1, Vector *vec2)
      {
       Vector vec;

       vec.x = vec1->x * vec2->x;
       vec.y = vec1->y * vec2->y;
       vec.z = vec1->z * vec2->z;

       return (vec);
      }



/*
 *******************************************************************************
 *
 *  Name: vmax
 *
 *  Purpose:  this routine compares the components of two vectors and returns
 *            a vector that contains the maximum components of the two.
 *
 *
 *
 *  Input Parameters
 *
 *     vec1 - first vector to be compared
 *     vec2 - second vector to be compared
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   vmax(Vector *vec1, Vector *vec2)
      {
       Vector max_vector;

       max_vector.x = MAX(vec1->x, vec2->x);
       max_vector.y = MAX(vec1->y, vec2->y);
       max_vector.z = MAX(vec1->z, vec2->z);

       return (max_vector);
      }



/*
 *******************************************************************************
 *
 *  Name: vmin
 *
 *  Purpose:  this routine compares the components of two vectors and returns
 *            a vector that contains the minimum components of the two.
 *
 *
 *
 *  Input Parameters
 *
 *     vec1 - first vector to be compared
 *     vec2 - second vector to be compared
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   vmin(Vector *vec1, Vector *vec2)
      {
       Vector min_vector;

       min_vector.x = MIN(vec1->x, vec2->x);
       min_vector.y = MIN(vec1->y, vec2->y);
       min_vector.z = MIN(vec1->z, vec2->z);

       return (min_vector);
      }



/*
 *******************************************************************************
 *
 *  Name:  vset
 *
 *  Purpose:  this routine initializes the specified vector to the specified
 *            component values.
 *
 *
 *
 *  Input Parameters
 *
 *     vec - vector to be initialized
 *     x - input x component
 *     y - input y component
 *     z - input z component
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
void
   vset(Vector *vec, float x, float y, float z)
      {
       vec->x = x;
       vec->y = y;
       vec->z = z;
      }



/*
 *******************************************************************************
 *
 *  Name:  vclamp
 *
 *  Purpose:  this routine clamps a vector at the maximum value specified.
 *
 *
 *
 *  Input Parameters
 *
 *     vec - vector to be clamped
 *     max - clamp value
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
void
   vclamp(Vector *vec, float max)
      {
       vec->x = (vec->x > max)? max : vec->x;
       vec->y = (vec->y > max)? max : vec->y;
       vec->z = (vec->z > max)? max : vec->z;
      }



float
   vmax_element(Vector *vec)
      {
       float max_element;

       max_element = MAX(vec->x, vec->y);
       max_element = MAX(max_element, vec->z);

       return(max_element);
      }


