/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Sample BIDI Protocol Converter                          */
/* AUTHOR         : Pat D                                                   */
/* FILENAME       : xlatecmd.c                                              */
/* DATE WRITTEN   : 12-17-93                                                */
/* DESCRIPTION    : Routines to translate messages from the printer to the  */
/*                  Protocol Converter                                      */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                           Routines In File                               */
/*                      SplProtXlateCmd()                                   */
/*                      ParseData()                                         */
/*                      StoreData()                                         */
/*                      FindQueries()                                       */
/*                      SignalQueries()                                     */
/****************************************************************************/

#define INCL_BASE
#define INCL_DOS
#define INCL_SPL
#define INCL_SPLBIDI
#define INCL_SPLERRORS
#include <os2.h>
#include <string.h>
#include "protcnv.h"
#include "protcnv1.h"
#include "cnvproto.h"

#define BIDI_UNKNOWN 0xFFFFFFFF

/*
 * Local prototype definitions
 */

ULONG ParseData ( PVOID pInData, ULONG cbInData, PRESPPKT *ppRespPkt );

ULONG StoreData ( PSZ pszPortName, PRESPPKT pRespPkt, PULONG pulCmdCode,
                  PVOID pInData, ULONG cbInData, PVOID pOutData,
                  PULONG pcbOutData, PBOOL pfPartial );

BOOL  FindQueries ( ULONG ulCmdCode, PREQINST pReqList, PBOOL pfPassthru,
                    PBOOL pfAlerts, PVOID pInData, ULONG cbInData,
                    PRESPPKT pRespPkt, PPORTINST pPortInst);

ULONG SignalQueries ( PSZ pszPortName );

/****************************************************************************
 *
 * FUNCTION NAME = SplProtXlateCmd
 *
 * DESCRIPTION   = Converts a protocol specific message received from the
 *                 printer into generic command structure that can be used by
 *                 the port driver and spooler
 *
 * INPUT         = pszPortName - Name of the port which received the message
 *               = pfnBaseProtXlateCmd - Base Converter ProtXlateCmd routine addr
 *               = pInData     - Command sequence from printer
 *               = cbInData    - Length in bytes of data in pInData
 *               = pAlertInfo  - Alert information buffer (PDALERTINFO struct)
 *               = pOutData    - Generic command structure that can be used
 *                               with the spooler and port driver
 *               = pcbOutData  - Points to the length of output buffer passed in
 *                               On entry, this is set to the length of the
 *                                  output buffer passed in
 *                               On exit, this is updated with the length of
 *                                  the structure copied into pOutData
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Unable to translate cmd)
 *
 ****************************************************************************/

ULONG APIENTRY SplProtXlateCmd ( PSZ pszPortName,
                                 PFN pfnBaseProtXlateCmd,
                                 PVOID pInData,
                                 ULONG cbInData,
                                 PVOID pAlertInfo,
                                 PVOID pOutData,
                                 PULONG pcbOutData ) {

    ULONG rc = 0;
    PRESPPKT pRespPkt = NULL;
    ULONG ulCmdCode = FALSE;
    PPDALERTINFO pPDAlertInfo = (PPDALERTINFO) pAlertInfo;
    BOOL fPartial = FALSE;

    /*
     * Check for valid port name
     */
    if (!CheckPort( pszPortName )) {
        *pcbOutData = 0;
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Check for valid Alert info
     */
    if (!pPDAlertInfo) {
        *pcbOutData = 0;
        return(ERROR_INVALID_PARAMETER);
    }
    /*
     * Parse input data into generic command information
     *  The information returned can be either the full response or a partial
     *   one.
     */
    rc = ParseData ( pInData , cbInData, &pRespPkt );
    if (rc) {
        *pcbOutData = 0;
        return(rc);
    }
    /*
     * Store parsed input in the protocol converter.  It will also copy the
     * data into the output buffer if necessary.  It will also check for
     * waiting responses.
     */
    rc = StoreData ( pszPortName, pRespPkt, &ulCmdCode, pInData, cbInData,
                     pOutData, pcbOutData, &fPartial );
    if (rc) {
        return(rc);
    }
    /*
     * If command code is set, then a full response was received
     * Signal the waiting queries and update Alertinfo structure.
     */
    if (ulCmdCode && !fPartial) {
        rc = SignalQueries ( pszPortName );
        if (ulCmdCode == BIDI_READ_ALERT) {
            pPDAlertInfo->ulFlags = PD_ALERT;
            pPDAlertInfo->ulCommand = 0xFFFFFFFF;
        } else {
            pPDAlertInfo->ulFlags = PD_RESPONSE;
            pPDAlertInfo->ulCommand = ulCmdCode;
        }
        pPDAlertInfo->ulCommand = ulCmdCode;
    } else if (ulCmdCode && fPartial) {
        pPDAlertInfo->ulFlags = PD_PARTIAL | PD_RESPONSE;
        pPDAlertInfo->ulCommand = ulCmdCode;
    } else {
        pPDAlertInfo->ulFlags = 0;
        pPDAlertInfo->ulCommand = 0xFFFFFFFF;
    }
    pPDAlertInfo->ulVersion = 0;

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = ParseData
 *
 * DESCRIPTION   = Parse command sequence from printer into generic command
 *                 structure.  Because there might be a partial response,
 *                 all responses might not be command structures.
 *
 * INPUT         = pInData     - Command sequence from printer
 *                                For skeleton protocol converter...
 *                               | ulType | Data              |
 *               = cbInData    - Length in bytes of data in pInData
 *               = ppRespPkt   - Pointer to storage for the response packet
 *                               created by ParseData
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Response could not be parsed)
 *
 ****************************************************************************/

ULONG ParseData ( PVOID pInData, ULONG cbInData, PRESPPKT *ppRespPkt ) {

    PRESPPKT pRespPkt = NULL;
    ULONG   cbRespPkt = 0;

    /*
     * Check for the existence of command sequence
     */
    if (pInData && cbInData) {
        cbRespPkt += sizeof (RESPPKT) + cbInData;
      EnterCnvSem();
        pRespPkt = AllocCnvMem ( cbRespPkt );
      LeaveCnvSem();
        /*
         * If memory allocated, build response packet
         */
        if (pRespPkt) {
            pRespPkt->cb = cbRespPkt;
            pRespPkt->ulType = *(PULONG)pInData;
            pRespPkt->cbData = cbInData - sizeof (ULONG);
            memcpy ( &(pRespPkt->Data[0]), (PBYTE)pInData+sizeof(ULONG),
                     pRespPkt->cbData );
        } else {
            return(ERROR_NOT_ENOUGH_MEMORY);
        }
    } else {
        return(ERROR_INVALID_PARAMETER);
    }

    *ppRespPkt = pRespPkt;

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = StoreData
 *
 * DESCRIPTION   = Store Data in the protocol converter
 *
 * INPUT         = pszPortName - Name of port which received the message
 *               = pRespPkt    - Pointer to the response packet from ParseData
 *               = pulCmdCode  - Pointer to storage for the command code
 *               = pInData     - Command sequence from printer
 *               = cbInData    - Length in bytes of data in pInData
 *               = pOutData    - Return buffer for the generic command structure
 *               = pcbOutData  - Points to length of output buffer
 *               = pfPartial   - Pointer to flag indicating whether partial
 *                               response has been received
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Data could not be stored)
 *
 * NOTE: Only alerts should be sent back to the caller
 *
 ****************************************************************************/

ULONG StoreData ( PSZ pszPortName, PRESPPKT pRespPkt, PULONG pulCmdCode,
                  PVOID pInData, ULONG cbInData, PVOID pOutData,
                  PULONG pcbOutData, PBOOL pfPartial ) {

    PPORTINST pPortInst;
    ULONG ulCmdCode = 0;
    ULONG rc = 0;
    ULONG cbOutData = 0;
    BOOL fPassthru = FALSE;
    BOOL fAlerts = TRUE;

  EnterCnvSem();
    pPortInst = pPortList;

    /*
     * Find port instance in protocol converter list
     */
    while (pPortInst) {
        if (!strcmp(pPortInst->pszPort, pszPortName)) {
            break;
        }
        pPortInst = pPortInst->pNext;
    }
    /*
     * If no port instance found, return error
     */
    if (!pPortInst) {
      LeaveCnvSem();
        *pcbOutData = 0;
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Switch on the type of the response packet
     */
    switch (pRespPkt->ulType) {
    case RESP_Q_DEVICE:
        ulCmdCode = BIDI_Q_DEVICE;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQDevice ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_INTERPRETER:
        ulCmdCode = BIDI_Q_INTERPRETER;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQInterpreter ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_INPUTBINS:
        ulCmdCode = BIDI_Q_INPUTBINS;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQInputbins ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_OUTPUTBINS:
        ulCmdCode = BIDI_Q_OUTPUTBINS;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQOutputbins ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_FONTS:
        ulCmdCode = BIDI_Q_FONTS;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQFonts ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_JOBS_COMPLETE:
        ulCmdCode = BIDI_Q_JOBS_COMPLETE;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQJobscomp ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_JOBS_QUEUED:
        ulCmdCode = BIDI_Q_JOBS_QUEUED;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQJobsqueued ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_Q_STATUS:
        ulCmdCode = BIDI_Q_STATUS;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        rc = CacheQStatus ( pPortInst, pRespPkt );
        *pfPartial = FALSE;
        break;
    case RESP_READ_ALERTSESSION:
        ulCmdCode = BIDI_READ_ALERT;
        /*
         * Set flag to indicate not all alerts read
         */
        fAlerts = FALSE;
        /*
         * Store data in port instance
         */
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        *pfPartial = FALSE;
        break;
    default:
        ulCmdCode = BIDI_UNKNOWN;
        /*
         * Set flag to indicate not all alerts read
         */
        fAlerts = FALSE;
        FindQueries( ulCmdCode, pPortInst->pReqList, &fPassthru, &fAlerts,
                     pInData, cbInData, pRespPkt, pPortInst);
        break;
    }

    /*
     * If passthru session started and no read passthru waiting, add passthru
     * response to list for the port
     */
    if (pPortInst->fPassthruSes && !fPassthru) {
        AddPassRespInst ( pInData, cbInData, pPortInst );
    }
  LeaveCnvSem();
    /*
     * Return the command code and output data size
     */
    *pcbOutData = cbOutData;
    *pulCmdCode = ulCmdCode;

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = SignalQueries
 *
 * DESCRIPTION   = Signal waiting queries that information is available
 *
 * INPUT         = pszPortName - Name of port to signal
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Waiting queries could not be signalled)
 *
 ****************************************************************************/

ULONG SignalQueries ( PSZ pszPortName ) {

    PPORTINST pPortInst;
    ULONG     ulPostCnt;

  EnterCnvSem();
    /*
     * Get Instance data for port
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(ERROR_FILE_NOT_FOUND);
    }
    LeaveCnvSem();
    /*
     * Post semaphore to signal queries then reset semaphore to hold queries
     * if response didn't match.
     */
    DosPostEventSem ( pPortInst->semQueries );
    DosResetEventSem ( pPortInst->semQueries, &ulPostCnt );

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = FindQueries
 *
 * DESCRIPTION   = Find Queries that match the response from the printer
 *
 * INPUT         = ulCmdCode - Generic command code
 *                 pReqList  - The head of the list of requests to the printer
 *                 pfPassthru - Pointer to flag indicating whether response
 *                              should be a passthru response..
 *                              (Command code didn't match known responses)
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful,Query was found)
 *
 * RETURN-ERROR  = 0(Failure, Query can't be found)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

BOOL  FindQueries ( ULONG ulCmdCode, PREQINST pReqList, PBOOL pfPassthru,
                    PBOOL pfAlerts, PVOID pInData, ULONG cbInData,
                    PRESPPKT pRespPkt, PPORTINST pPortInst ) {

    PREQINST      pReqInst = pReqList;
    BOOL          fRespFound = FALSE;
    PVOID pNewData;
    ULONG cbData;
    ULONG ulSize = 0;

    /*
     * Loop until end of the request list for a port
     */
    while (pReqInst) {
        /*
         * Check for the passthru command code
         */
        if (pReqInst->ulCmdCode == BIDI_READ_PASSTHRU) {
            /*
             * Not looking for the passthru, set flag and pointer
             *  will only get the first passthru request.
             */
            pReqInst->fRespRecv = 0;
            cbData = cbInData;
            if (pReqInst->ulType == TYPE_WITHOUT_WRAPPER) {
                /*
                 * If enough room in output buffer, copy data to it
                 *   If not enough room, then buffer data for next call
                 */
                ulSize = cbData - GetWrapperSize();
                if (ulSize <= *(pReqInst->pcbOutData)) {
                    pNewData = RemoveWrapper ( pInData, &cbData );
                    memcpy ( pReqInst->pOutData, pNewData, cbData );
                    pReqInst->rc = 0;
                    *pfPassthru = TRUE;
                } else {
                    pReqInst->rc = NERR_BufTooSmall;
                }
                *(pReqInst->pcbOutData) = ulSize;
            } else {
                /*
                 * If enough room in output buffer, copy data to it
                 *   If not enough room, then buffer data for next call
                 */
                if (cbInData <= *(pReqInst->pcbOutData)) {
                    memcpy ( pReqInst->pOutData, pInData, cbInData );
                    pReqInst->rc = 0;
                    *pfPassthru = TRUE;
                } else {
                    pReqInst->rc = NERR_BufTooSmall;
                }
                *(pReqInst->pcbOutData) = cbInData;
            }
            fRespFound = TRUE;
        /*
         * If command codes are the same, then clear flag indicating response
         * has been received.  For queries that may have more than variation,
         * the pInData will have to be checked to find the correct query.
         */
        } else if ( pReqInst->ulCmdCode == ulCmdCode ) {
            pReqInst->fRespRecv = 0;
            /*
             * If enough room in output buffer, copy data to it
             * Only if the response if complete, for partial responses,
             *   information will have to be stored.
             */
            if (pRespPkt->cbData <= *(pReqInst->pcbOutData)) {
                memcpy ( pReqInst->pOutData, pRespPkt->Data, pRespPkt->cbData );
                *(pReqInst->pcbOutData) = pRespPkt->cbData;
                pReqInst->rc = 0;
            } else {
                pReqInst->rc = NERR_BufTooSmall;
            }
            fRespFound = TRUE;
        }
        pReqInst = pReqInst->pNext;
    }

    return(fRespFound);

}
