/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Sample BIDI Protocol Converter                          */
/* AUTHOR         : Pat D                                                   */
/* FILENAME       : util.c                                                  */
/* DATE WRITTEN   : 12-17-93                                                */
/* DESCRIPTION    : Utility routines for use with structures, memory        */
/*                  management, etc.                                        */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                           Routines In File                               */
/*                      AllocCnvMem()                                       */
/*                      FreeCnvMem()                                        */
/*                      EnterCnvSem()                                       */
/*                      LeaveCnvSem()                                       */
/*                      CnvError()                                          */
/*                      CnvPanic()                                          */
/*                      CnvWarning()                                        */
/****************************************************************************/

#define INCL_BASE
#define INCL_DOS
#define INCL_SPL
#define INCL_SPLBIDI
#define INCL_SPLERRORS
#include <os2.h>
#include <string.h>
#include "protcnv.h"
#include "protcnv1.h"
#include "cnvproto.h"

/*
 * Local Function prototypes
 */
VOID SetReqList ( PREQINST pReqList );

/****************************************************************************/
/*                      Common Routines                                     */
/****************************************************************************/

/****************************************************************************
 *
 * FUNCTION NAME = AllocCnvMem
 *
 * DESCRIPTION   = Allocate memory from protocol converter heap
 *
 * INPUT         = cb  - Amount of memory to allocate in bytes
 *
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to allocated memory
 *
 * RETURN-ERROR  = NULL(Failure, memory could not be allocated)
 *
 ****************************************************************************/

PVOID AllocCnvMem ( ULONG cb ) {

    PVOID pMem = NULL;
    ULONG rc = 0;

    rc = DosSubAllocMem ( pHeapBase, &pMem, cb );
    if (rc) {
       CnvError ( 1 , 1 );
       return NULL;
    } else {
       return(pMem);
    }

}

/****************************************************************************
 *
 * FUNCTION NAME = FreeCnvMem
 *
 * DESCRIPTION   = Free memory from the protocol converter heap
 *
 * INPUT         = pMem - Pointer to the start of the memory block
 *               = cb - Size of the memory block to free
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0, Successful
 *
 * RETURN-ERROR  = rc, Failure
 *
 ****************************************************************************/

ULONG FreeCnvMem ( PVOID pMem, ULONG cb ) {

    ULONG rc = 0;
    PFREEMEM    pFreeMem = (PFREEMEM) pMem;

    /*
     * Set signature and size [For debugging/validation]
     */
    if (cb >= sizeof(USHORT)) {
        pFreeMem->signature = FR_SIGNATURE;
    }
    if (cb >= (sizeof(USHORT)*2)) {
        pFreeMem->cb = (USHORT) cb;
    }
    /*
     * Free the memory
     */
    rc = DosSubFreeMem ( pHeapBase, pMem, cb );
    if (rc) {
        CnvPanic ( 2 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = EnterCnvSem
 *
 * DESCRIPTION   = Enter the protocol converter semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosRequestMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be requested)
 *
 ****************************************************************************/

ULONG EnterCnvSem ( VOID ) {

    ULONG rc = 0;

    rc = DosRequestMutexSem ( semProtCnv , SEM_INDEFINITE_WAIT );
    if (rc) {
        CnvError ( 3 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = LeaveCnvSem
 *
 * DESCRIPTION   = Leave the protocol converter semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosReleaseMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 ****************************************************************************/

ULONG LeaveCnvSem ( VOID ) {

    ULONG rc = 0;

    rc = DosReleaseMutexSem ( semProtCnv );
    if (rc) {
        CnvError ( 4 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = CnvError
 *
 * DESCRIPTION   = Routine called on a error condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID CnvError ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = CnvPanic
 *
 * DESCRIPTION   = Routine called on a panic condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID CnvPanic ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = CnvWarning
 *
 * DESCRIPTION   = Routine called on a warning condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID CnvWarning ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************/
/*                      Converter Specific Routines                         */
/****************************************************************************/

/****************************************************************************
 *
 * FUNCTION NAME = AllocatePort
 *
 * DESCRIPTION   = Allocate the instance of a port
 *
 * INPUT         = pszPortName - Name of the port to create
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to Port Instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure)
 *
 ****************************************************************************/

PPORTINST AllocatePort ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;
    ULONG cb;
    ULONG cbName;
    PSZ pszName;
    ULONG rc = 0;

    cb = sizeof ( PORTINST );
    /*
     * Allocate and fill port instance
     */
    pPortInst = (PPORTINST) AllocCnvMem ( cb );
    if (pPortInst) {
        memset ( pPortInst, 0, cb );
        pPortInst->signature = CN_SIGNATURE;
        pPortInst->cb = cb ;
        /*
         * Allocate memory for the portname separate from the port instance
         */
        cbName = strlen(pszPortName) + 1;
        pszName = (PSZ) AllocCnvMem ( cbName );
        if (pszName) {
            strcpy ( pszName, pszPortName );
            pPortInst->pszPort = pszName;
        } else {
            FreeCnvMem ( pPortInst, cb );
            return(NULL);
        }
        /*
         * Create event sem on which queries with FLG_WAIT set will wait
         */
        rc = DosCreateEventSem ( 0, &pPortInst->semQueries, 0, FALSE );
        if (rc) {
            FreeCnvMem ( pszName, cbName );
            FreeCnvMem ( pPortInst, cb );
            return(NULL);
        }
        pPortInst->ulNextAlertId = 1;
    } else {
        return ( NULL );
    }

    return(pPortInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePort
 *
 * DESCRIPTION   = Free Memory and resources associated with a port instance
 *
 * INPUT         = pPortInst - pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful)
 *
 * RETURN-ERROR  = 0(Failure)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

BOOL FreePort ( PPORTINST pPortInst ) {

    PSZ pszPort;
    BOOL rc = FALSE;

    if (pPortInst) {
        pPortInst->fsStatus |= CN_PORT_FREED;
        /*
         * Check for waiting requests, free if any found
         */
        if (pPortInst->pReqList) {
            SetReqList ( pPortInst->pReqList );
            DosPostEventSem ( pPortInst->semQueries );
        }
        DosCloseEventSem ( pPortInst->semQueries );
        /*
         * Free port name string
         */
        pszPort = pPortInst->pszPort;
        if (pszPort) {
            FreeCnvMem ( pszPort, strlen (pszPort)+1 );
        }
        /*
         * Free cached passthru session responses
         */
        if (pPortInst->pPassRespList) {
            FreePassRespList ( pPortInst->pPassRespList );
        }
        FreeCnvMem ( pPortInst, pPortInst->cb );
        rc = TRUE;
    }

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreeAllPorts
 *
 * DESCRIPTION   = Free all protocol converter ports at module exit
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, All ports could not be freed)
 *
 ****************************************************************************/

ULONG FreeAllPorts ( VOID ) {

    PPORTINST pPortInst;
    PPORTINST pPortNext;
    ULONG rc = 0;

  EnterCnvSem();
    pPortInst = pPortList;

    /*
     * Loop through list and free each port
     */
    while (pPortInst) {
        pPortNext = pPortInst->pNext;
        FreePort ( pPortInst );
        pPortInst = pPortNext;
    }
  LeaveCnvSem();
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPort
 *
 * DESCRIPTION   = Add a port instance to the list in the protocol converter
 *
 * INPUT         = pszName   - Name of the port
 *               = pPortInst - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful)
 *
 * RETURN-ERROR  = 0(Failure, Port already existed)
 *
 ****************************************************************************/

BOOL AddPort ( PSZ pszName, PPORTINST pPortInst ) {

    /*
     * If port already is in the list or no data is passed in, return failure
     */
    if (FindPort(pszName) || !pPortInst) {
        return(0);
    }
    /*
     * If list exists, add port to end of list
     *  else, start list and set end
     */
    if (pPortLast) {
        pPortLast->pNext = pPortInst;
        pPortLast = pPortInst;
    } else {
        pPortList = pPortInst;
        pPortLast = pPortInst;
    }
    return(1);
}

/****************************************************************************
 *
 * FUNCTION NAME = FindPort
 *
 * DESCRIPTION   = Find port in the protocol converter list
 *
 * INPUT         = pszName - Name of the port to find
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to port instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure, Port not found)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

PPORTINST FindPort ( PSZ pszName ) {

    PPORTINST pPortInst;

    pPortInst = pPortList;

    /*
     * Loop through port list, stopping at element containing pszName
     */
    while (pPortInst) {
        if (!strcmp(pPortInst->pszPort, pszName)) {
            return(pPortInst);
        }
        pPortInst = pPortInst->pNext;
    }

    return(NULL);

}

/****************************************************************************
 *
 * FUNCTION NAME = CheckPort
 *
 * DESCRIPTION   = Check for port in the protocol converter list
 *
 * INPUT         = pszName - Name of the port to find
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful)
 *
 * RETURN-ERROR  = 0(Failure, Port not found)
 *
 ****************************************************************************/

BOOL CheckPort ( PSZ pszName ) {

    /*
     * Call find port to determine if port is in the list
     */
  EnterCnvSem();
    if (FindPort( pszName )) {
      LeaveCnvSem();
        return(1);
    } else {
      LeaveCnvSem();
        return(0);
    }

}

/****************************************************************************
 *
 * FUNCTION NAME = RemovePort
 *
 * DESCRIPTION   = Remove a port from the list contained in the protocol
 *                 converter.
 *
 * INPUT         = pszName - Name of port to remove from the protocol converter
 *                            list.
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to port instance removed(Successful)
 *
 * RETURN-ERROR  = NULL(Failure, port could not be removed)
 *
 ****************************************************************************/

PPORTINST RemovePort ( PSZ pszName ) {

    PPORTINST pPortInst;
    PPORTINST pPortPrev = NULL;

  EnterCnvSem();
    pPortInst = pPortList;
    /*
     * While not at the end of the list
     */
    while (pPortInst) {
        /*
         * Compare name of port to the name of the port instance
         */
        if (!strcmp(pPortInst->pszPort, pszName)) {
            /*
             * Determine if it isn't the head of the list ( previous entry )
             */
            if (pPortPrev) {
                pPortPrev->pNext = pPortInst->pNext;
            }
            /*
             * Compare to the head of the list, reset head of list if same
             */
            if (pPortInst == pPortList) {
                pPortList = pPortInst->pNext;
            }
            /*
             * Compare to the tail of the list
             */
            if (pPortInst == pPortLast) {
                /*
                 * Check for previous entry, if none then no more entries in
                 * list
                 */
                if (pPortPrev) {
                    pPortLast = pPortPrev;
                } else {
                    pPortLast = NULL;
                }
            }
            /*
             * Free the resources in the port instance after it is removed
             * from the port instance list
             */
            FreePort ( pPortInst );
          LeaveCnvSem();
            return(pPortInst);
        }
        pPortPrev = pPortInst;
        pPortInst = pPortInst->pNext;
    }

  LeaveCnvSem();

    return(NULL);

}

/****************************************************************************
 *
 * FUNCTION NAME = NewRequest
 *
 * DESCRIPTION   = Create a new request instance and add it to the protocol
 *                 converter list
 *
 * INPUT         = pszPortName - Name of the port to which the commands were
 *                                sent
 *               = ulCommand   - Generic Command Code
 *               = pInData     - Information required by the command
 *               = cbInData    - Length in bytes of data in pInData
 *               = cCmds       - Count of commands sent to the printer
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to new request structure(Successful)
 *
 * RETURN-ERROR  = NULL(Failure, request structure could not be created)
 *
 ****************************************************************************/

PREQINST NewRequest ( PSZ pszPortName,
                      ULONG ulCommand,
                      PVOID pInData,
                      ULONG cbInData,
                      PVOID pOutData,
                      PULONG pcbOutData,
                      ULONG cCmds ) {

    PREQINST pNewReqInst = NULL;
    PPORTINST pPortInst = NULL;
    ULONG cb = 0;

  EnterCnvSem();
    /*
     * Find port instance data
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(NULL);
    }
    cb = sizeof (REQINST);
    pNewReqInst = (PREQINST) AllocCnvMem ( cb );
    /*
     * If memory has been allocated, fill it with needed information
     */
    if (pNewReqInst) {
        memset ( pNewReqInst, 0, cb );
        pNewReqInst->signature = RQ_SIGNATURE;
        pNewReqInst->cb = cb;
        pNewReqInst->pszPortName = pszPortName;
        pNewReqInst->ulCmdCode = ulCommand;
        pNewReqInst->pInData = pInData;
        pNewReqInst->cbInData = cbInData;
        pNewReqInst->pOutData = pOutData;
        pNewReqInst->pcbOutData = pcbOutData;
        pNewReqInst->cCmdsNeeded = cCmds;
    } else {
      LeaveCnvSem();
      return(NULL);
    }
    /*
     * Add new request to the request list
     */
    AddReq ( pPortInst, pNewReqInst );

  LeaveCnvSem();
    return(pNewReqInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddReq
 *
 * DESCRIPTION   = Add the request instance to the protocol converter list
 *
 * INPUT         = pPortInst - Pointer to the port instance containing the head
 *                             and tail of the request list
 *                 pReqInst - Pointer to the request instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, request structure could not be added)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

ULONG AddReq ( PPORTINST pPortInst, PREQINST pReqInst ) {

    PREQINST pReqLast = NULL;

    if (pPortInst && pReqInst) {
        /*
         * If head of list exists, add to the end of the list
         * If not, set head and tail of list to new request pointer
         */
        if (pPortInst->pReqList) {
            pReqLast = pPortInst->pReqLast;
            if (pReqLast) {
                pReqInst->pNext = pReqLast->pNext;
                pReqLast->pNext = pReqInst;
            }
        } else {
            pPortInst->pReqList = pReqInst;
            pPortInst->pReqLast = pReqInst;
        }
    }

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = RemoveReq
 *
 * DESCRIPTION   = Remove the request instance from the protocol converter
 *                  list
 *
 * INPUT         = pReqInst - Pointer to the request instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, request structure could not be removed)
 *
 ****************************************************************************/

ULONG RemoveReq ( PSZ pszPortName, PREQINST pReqInst ) {

    PPORTINST pPortInst = NULL;
    PREQINST pReqList = NULL;
    PREQINST pReqLast = NULL;
    PREQINST pReqCurr = NULL;
    PREQINST pReqPrev = NULL;

  EnterCnvSem();
    /*
     * Find port instance data
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(ERROR_FILE_NOT_FOUND);
    }
    pReqList = pPortInst->pReqList;
    pReqLast = pPortInst->pReqLast;
    pReqCurr = pReqList;

    /*
     * While not at the end of the list
     */
    while (pReqCurr) {
        /*
         * Is current request the request to be deleted
         */
        if (pReqCurr == pReqInst) {
            /*
             * Determine if it isn't the head of the list ( previous entry )
             */
            if (pReqPrev) {
                pReqPrev->pNext = pReqCurr->pNext;
            }
            /*
             * Compare to the head of the list, reset head of list if same
             */
            if (pReqCurr == pReqList) {
                pPortInst->pReqList = pReqCurr->pNext;
            }
            /*
             * Compare to the tail of the list
             */
            if (pReqCurr == pReqLast) {
                /*
                 * Check for previous entry, if none then no more entries in
                 * list
                 */
                if (pReqPrev) {
                    pPortInst->pReqLast = pReqPrev;
                } else {
                    pPortInst->pReqLast = NULL;
                }
            }
            /*
             * Free the resources in the request instance after it is removed
             * from the request instance list
             */
            FreeCnvMem ( pReqCurr, pReqCurr->cb );
          LeaveCnvSem();
            return(0);
        }
        pReqPrev = pReqCurr;
        pReqCurr = pReqCurr->pNext;
    }

  LeaveCnvSem();

    return(ERROR_NOT_ENOUGH_MEMORY);

}

/****************************************************************************
 *
 * FUNCTION NAME = SetReqList
 *
 * DESCRIPTION   = Loop through request instance list and set rc to
 *                 PMERR_SPL_PORT_SHUTDOWN and clear fRespRecv
 *
 * INPUT         = pReqList - Pointer to the request instance list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID SetReqList ( PREQINST pReqList ) {

    PREQINST pReqInst = pReqList;

    while (pReqInst) {
        pReqInst->rc = PMERR_SPL_PORT_SHUTDOWN;
        pReqInst->fRespRecv = 0;
        pReqInst = pReqInst->pNext;
    }
    return;

}

/****************************************************************************
 *
 * FUNCTION NAME = GetWrapperSize
 *
 * DESCRIPTION   = Get packet wrapper size
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Size of Wrapper ( Successful )
 *
 * RETURN-ERROR  = 0 ( Failure )
 *
 ****************************************************************************/

ULONG GetWrapperSize ( VOID ) {

    return ( sizeof (ULONG) );

}

/****************************************************************************
 *
 * FUNCTION NAME = AddWrapper
 *
 * DESCRIPTION   = Add packet wrapper to the data when TYPE_WITHOUT_WRAPPER
 *
 * INPUT         = pBuf    - Pointer to buffer
 *                 pData   - Pointer to data
 *                 cbData  - Size of data
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to buffer containing the wrapper and the data
 *
 * RETURN-ERROR  = NULL
 *
 ****************************************************************************/

PVOID AddWrapper ( PVOID pBuf, PVOID pData, ULONG cbData ) {

    PBIDIDATA pBidiData = (PBIDIDATA) pBuf;

    pBidiData->ulCommand = BIDI_SEND_PASSTHRU;
    memcpy ( pBidiData->Data, pData, cbData );

    return (pBidiData);

}

/****************************************************************************
 *
 * FUNCTION NAME = RemoveWrapper
 *
 * DESCRIPTION   = Remove packet wrapper from the data when TYPE_WITHOUT_WRAPPER
 *
 * INPUT         = pData   - Pointer to data
 *                 pcbData - Pointer to size of data
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

PVOID RemoveWrapper ( PVOID pData, PULONG pcbData ) {

    PBIDIDATA pBidiData = (PBIDIDATA) pData;

    *pcbData -= GetWrapperSize();
    return ((PVOID)&(pBidiData->Data));

}

