/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Sample BIDI Protocol Converter                          */
/* AUTHOR         : Pat D                                                   */
/* FILENAME       : sendcmd.c                                               */
/* DATE WRITTEN   : 12-17-93                                                */
/* DESCRIPTION    : Routines to send command to the printer from the        */
/*                  Protocol Converter                                      */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                           Routines In File                               */
/*                      SplProtSendCmd()                                    */
/*                      SemdCmd()                                           */
/*                      GetResp()                                           */
/*                      GetQSW()                                            */
/*                      BuildSetShutdownCmd()                               */
/*                      BuildSetProtocolCmd()                               */
/*                      BuildStartPassthruCmd()                             */
/*                      BuildEndPassthruCmd()                               */
/*                      StartCnvUpdateThread()                              */
/*                      CnvUpdateThread()                                   */
/****************************************************************************/

#define INCL_BASE
#define INCL_SPL
#define INCL_SPLBIDI
#define INCL_SPLERRORS
#define INCL_SPLFSE
#define INCL_SPLDOSPRINT
#include <os2.h>
#include <string.h>
#include "protcnv.h"
#include "protcnv1.h"
#include "cnvproto.h"

#define PRINTER_NAME "Test Printer"
#define REVISION     "1.2"
#define SERIALNUM    "TST0001"

typedef ULONG (APIENTRY FNSENDCMD) ( PSZ, ULONG, ULONG, PVOID, ULONG );
typedef FNSENDCMD *PFNSENDCMD;

/*
 * Local prototype definitions
 */

ULONG SendCmd ( PFN pfnPdSendCmd,
                PSZ pszPortName,
                ULONG ulType,
                ULONG ulCommand,
                PREQINST  pReqInst,
                PMULTICMD pMultiCmd );

ULONG GetResp ( PSZ pszPortName,
                PREQINST pReqInst );

ULONG GetQSW ( PSZ pszPortName, PVOID pOutData, PULONG pcbOutData );

ULONG BuildSetShutdownCmd ( PSZ pszPortName, PMULTICMD *ppMultiCmd );

ULONG BuildSetProtocolCmd ( PVOID pInData, PSZ pszPortName);

ULONG StartCnvUpdateThread ( PFN pfnPdSendCmd, PSZ pszPortName, ULONG ulType,
                             ULONG ulCommand, PMULTICMD pMultiCmd );

VOID EXPENTRY CnvUpdateThread ( PCNVSEND pCnvSend );
ULONG InitDevice ( PSZ pszPortName, PVOID pOutData, PULONG pcbOutData );

/****************************************************************************
 *
 * FUNCTION NAME = SplProtSendCmd
 *
 * DESCRIPTION   = Converts generic commands and passes BIDI specific commands
 *                 to the printer.
 *
 * INPUT         = pszPortName  - The name of the port to send the commands
 *               = ulType       - type of query and options
 *               = ulCommand    - Generic command code
 *               = pfnPdSendCmd - Port Driver send routine address
 *               = pfnBaseProtSendCmd - Base Converter ProtSendCmd routine addr
 *               = pInData      - Information required by the command
 *               = cbInData     - Length in bytes of data in pInData
 *               = pOutData     - Return buffer
 *               = pcbOutData   - Points to length of output buffer
 *                                On entry this is set to length of output
 *                                  buffer passed in.
 *                                On exit this is updated with length of data
 *                                  returned.
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, rc contains error code)
 *
 ****************************************************************************/

ULONG APIENTRY SplProtSendCmd ( PSZ pszPortName,
                                ULONG ulType,
                                ULONG ulCommand,
                                PFN pfnPdSendCmd,
                                PFN pfnBaseProtSendCmd,
                                PVOID pInData,
                                ULONG cbInData,
                                PVOID pOutData,
                                PULONG pcbOutData ) {

    PMULTICMD pMultiCmd = NULL;
    ULONG rc = 0;
    ULONG rc1 = 0;
    ULONG fNeedResp = 0L;
    BOOL fGotResp = FALSE;
    BOOL fUpdate = FALSE;
    PREQINST pReqInst = NULL;
    ULONG cCmds = 0;

    /*
     * Check for set command
     */
    if ( ulCommand < BIDI_Q_BASE ) {
        /*
         * Cache commands are invalid for set commands
         */
        if ((ulType == TYPE_CACHE_ONLY) || (ulType == TYPE_CACHE_UPDATE)) {
            return(ERROR_INVALID_PARAMETER);
        }
        /*
         * Check for invalid set type
         */
        if ((ulType == TYPE_WITHOUT_WRAPPER) &&
            (ulCommand != BIDI_SEND_PASSTHRU)) {
            return(ERROR_INVALID_PARAMETER);
        }
    } else {
        /*
         * Check output pointers for validity
         */
        if ( !pcbOutData ) {
            return(ERROR_INVALID_PARAMETER);
        }
        if ( !pOutData ) {
            *pcbOutData = 0;
            return(ERROR_INVALID_PARAMETER);
        }
        /*
         * Check for invalid read type
         */
        if ((ulType == TYPE_WITHOUT_WRAPPER) &&
            (ulCommand != BIDI_READ_PASSTHRU)) {
            *pcbOutData = 0;
            return(ERROR_INVALID_PARAMETER);
        }
    }
    /*
     * Determine Command and call appropriate routine.  These routines will
     * set the necessary information in the Protocol Converter and build
     * any commands to send to the Port Driver.
     */
    switch (ulCommand) {
        case BIDI_INIT_PROTCNV:
            rc = BuildSetProtocolCmd ( pInData, pszPortName );
            break;
        case BIDI_STARTJOB:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetStartjobCmd ( pszPortName, pInData, cbInData,
                                         &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_Q_JOBID:
            /*
             * Check for valid size of output buffer
             */
            if ( *pcbOutData < 2*sizeof(ULONG) ) {
                return(NERR_BufTooSmall);
            }
            rc = BuildQJobidCmd ( pszPortName, pOutData, pcbOutData,
                                  &pMultiCmd );
            /*
             * For sample, no response is needed from the printer.
             */
            fGotResp = TRUE;
            break;
        case BIDI_ENDJOB:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetEndjobCmd ( pszPortName, pInData, cbInData,
                                       &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_HOLDJOB:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetHoldjobCmd ( pszPortName, pInData, cbInData,
                                        &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_RELEASEJOB:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetReleasejobCmd ( pszPortName, pInData, cbInData,
                                           &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_CANCELJOB:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetCanceljobCmd ( pszPortName, pInData, cbInData,
                                          &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_PACKET_SIZE:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetPacketsizeCmd ( pszPortName, pInData, cbInData,
                                           &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_Q_SW:
            rc = GetQSW ( pszPortName, pOutData, pcbOutData );
            fGotResp = TRUE;
            break;
        case BIDI_Q_DEVICE:
            /*
             * The following is for test purposes only.  Use ProcTypeQDevice
             * in actual converters
             */
            rc = InitDevice ( pszPortName, pOutData, pcbOutData );
            fGotResp = TRUE;
//          rc = ProcTypeQDevice ( pszPortName, ulType, pOutData, pcbOutData,
//                                 &fGotResp, &fNeedResp, &rc1, &pMultiCmd );
//          fUpdate = TRUE;
            break;
        case BIDI_Q_INTERPRETER:
            rc = ProcTypeQInterpreter ( pszPortName, ulType, pInData, cbInData,
                                        pOutData, pcbOutData, &fGotResp,
                                        &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_INPUTBINS:
            rc = ProcTypeQInputbins ( pszPortName, ulType, pInData, cbInData,
                                      pOutData, pcbOutData, &fGotResp,
                                      &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_OUTPUTBINS:
            rc = ProcTypeQOutputbins ( pszPortName, ulType, pInData, cbInData,
                                       pOutData, pcbOutData, &fGotResp,
                                       &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_FONTS:
            rc = ProcTypeQFonts ( pszPortName, ulType, pInData, cbInData,
                                  pOutData, pcbOutData, &fGotResp,
                                  &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_JOBS_COMPLETE:
            rc = ProcTypeQJobscomp ( pszPortName, ulType, pInData, cbInData,
                                     pOutData, pcbOutData, &fGotResp,
                                     &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_JOBS_QUEUED:
            rc = ProcTypeQJobsqueued ( pszPortName, ulType, pInData, cbInData,
                                       pOutData, pcbOutData, &fGotResp,
                                       &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_STATUS:
            rc = ProcTypeQStatus ( pszPortName, ulType, pInData, cbInData,
                                   pOutData, pcbOutData, &fGotResp,
                                   &fNeedResp, &rc1, &pMultiCmd );
            fUpdate = TRUE;
            break;
        case BIDI_Q_RESPONSE_FMT:
            /*
             * Check for valid size of output buffer
             */
            if ( *pcbOutData < sizeof(PRTRESPONSE) ) {
                return(NERR_BufTooSmall);
            }
            rc = BuildQRespfmtCmd ( pOutData, pcbOutData );
            fGotResp = TRUE;
            break;
        case BIDI_READ_ALERT:
            /*
             * For now, do nothing for BIDI_READ_ALERT.
             * If implementing extended alert categories,
             *   you should support giving the extended alert
             *   back to the spooler.
             */
            rc = ERROR_NOT_SUPPORTED;
            break;
        case BIDI_START_PASSTHRU:
            rc = BuildStartPassthruCmd ( pszPortName );
            break;
        case BIDI_SEND_PASSTHRU:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
            case TYPE_WITHOUT_WRAPPER:
              rc = BuildSendPassthruCmd ( pszPortName, pInData, cbInData,
                                          ulType, &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_END_PASSTHRU:
            rc = BuildEndPassthruCmd ( pszPortName );
            break;
        case BIDI_READ_PASSTHRU:
            /*
             * Check for cached data
             */
            rc = GetReadPassthruCmd ( pszPortName, ulType, pOutData, pcbOutData,
                                      &fGotResp );
            /*
             * If no response received, set need response flag
             */
            if (!rc && !fGotResp) {
                if ((ulType != TYPE_CACHE_ONLY) &&
                    (ulType != TYPE_CACHE_UPDATE))  {
                    fNeedResp = REQ_READ_PASSTHRU;
                } else {
                    *pcbOutData = 0;
                    return(ERROR_INFO_NOT_AVAIL);
                }
            }
            break;
        case BIDI_RESET:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetResetCmd ( pszPortName, pInData, cbInData,
                                      &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_SHUTDOWN:
            rc = BuildSetShutdownCmd ( pszPortName , &pMultiCmd );
            break;
        case BIDI_TERM:
            switch (ulType) {
            /*
             * For sample, consider Long and Short commands to be the same
             *  Actual converters may create different commands to be sent
             */
            case TYPE_SHORT_WAIT:
            case TYPE_LONG_WAIT:
              rc = BuildSetTermCmd ( pszPortName, pInData, cbInData, &pMultiCmd );
              break;
            default:
              rc = ERROR_INVALID_PARAMETER;
              break;
            }
            break;
        case BIDI_Q_PORT:
        case BIDI_WAIT_ALERT:
        case BIDI_Q_PORTDRV:
        case BIDI_INIT_PORTDRV:
        case BIDI_INIT:
        case BIDI_RESPONSE_FMT:
        case BIDI_SET_SW:
        case BIDI_SET_PORTDRV:
        default:
            rc = ERROR_INVALID_FUNCTION;
            break;
    }
    /*
     * If cache should be updated, start thread to update cache.
     *   Check to make sure no serious error occrurred.
     */
    if (ulType == TYPE_CACHE_UPDATE && ((rc == ERROR_INFO_NOT_AVAIL) ||
                                        !rc) && fUpdate ) {
        /*
         * If unable to build command, return no error ( cache returned )
         */
        if (rc1) {
            return(0);
        }
        /*
         * Start thread to update cache
         */
        rc1 = StartCnvUpdateThread ( pfnPdSendCmd, pszPortName, ulType,
                                     ulCommand, pMultiCmd );
        /*
         * Return no error even if an error has occurred
         */
        return(0);
    }
    /*
     * If Response is returned, create a request packet
     */
    if (!rc && fNeedResp) {
        if (pMultiCmd) {
            cCmds = pMultiCmd->cCmdStructs;
        } else {
            cCmds = 1;
        }
        pReqInst = NewRequest ( pszPortName, ulCommand, pInData, cbInData,
                                pOutData, pcbOutData, cCmds );
        /*
         * Set Flag for responses needed ( bit set for each command sent )
         *   and ulType
         */
        if (pReqInst) {
            pReqInst->fRespRecv = fNeedResp;
            pReqInst->ulType = ulType;
        } else {
            return(ERROR_NOT_ENOUGH_MEMORY);
        }
    }
    /*
     * If commands exist, Send them to the port driver
     */
    if (!rc && pMultiCmd) {
        rc = SendCmd ( pfnPdSendCmd, pszPortName, ulType, ulCommand, pReqInst,
                       pMultiCmd );
      EnterCnvSem();
        FreeCnvMem ( pMultiCmd, pMultiCmd->cb );
      LeaveCnvSem();
    }
    /*
     * If Command is a query, Get response ( waiting if necessary )
     */
    if (!rc && !fGotResp && ulCommand >= BIDI_Q_BASE) {
        rc = GetResp ( pszPortName, pReqInst );
    }
    /*
     * Remove request packet from list when response is received
     */
    if (pReqInst) {
        RemoveReq ( pszPortName, pReqInst );
    }

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = SendCmd
 *
 * DESCRIPTION   = Routine which calls the Port Driver send command for the
 *                 number of commands given
 *
 * INPUT         = pfnPdSendCmd - Port Driver send routine address
 *               = pszPortName  - The name of the port to send the commands
 *               = ulType       - Type of set/query and options
 *               = ulCommand    - Generic Command Code
 *               = pReqInst     - Pointer to the request instance structure
 *               = pMultiCmd    - Structure containing the commands to send
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, rc contains error code)
 *
 ****************************************************************************/

ULONG SendCmd ( PFN pfnPdSendCmd,
                PSZ pszPortName,
                ULONG ulType,
                ULONG ulCommand,
                PREQINST pReqInst,
                PMULTICMD pMultiCmd ) {

    ULONG i;
    PCMDSTRUCT pCmdStruct;
    PFNSENDCMD pfnSendCmd = (PFNSENDCMD) pfnPdSendCmd;
    ULONG cCmdStructs = pMultiCmd->cCmdStructs;
    ULONG ulFlags = 0;
    ULONG rc = 0;

    /*
     * Set number of commands to send in request instance structure
     */
    if (pReqInst) {
        pReqInst->cCmdsNeeded = cCmdStructs;
    }
    /*
     * Move to command structures
     */
    pCmdStruct = (PCMDSTRUCT)(((PBYTE)pMultiCmd) + (sizeof(ULONG) * 2));
    /*
     * Loop through commands and send them to the printer
     */
    for (i = 0; i < cCmdStructs && !rc; i++) {
        if (pReqInst) {
            (pReqInst->cCmdsSent)++;
        }
        ulFlags = FLG_DATACHANNEL;
        rc = (*pfnSendCmd) ( pszPortName, ulFlags, ulCommand, pCmdStruct->Cmd,
                             pCmdStruct->cb );
        /*
         * Advance to next command structure
         */
        pCmdStruct = (PCMDSTRUCT)(((PBYTE)pCmdStruct) + pCmdStruct->cb +
                       sizeof(ULONG));
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = GetResp
 *
 * DESCRIPTION   = Routine which gets response from command
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *               = pReqInst    - Pointer to the request instance structure
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Response was not received)
 *
 ****************************************************************************/

ULONG GetResp ( PSZ pszPortName, PREQINST pReqInst ) {

    PPORTINST pPortInst;
    HEV  semQueries;
    ULONG rc = 0;

    /*
     * Find Port instance
     */
  EnterCnvSem();
    pPortInst = FindPort ( pszPortName );
  LeaveCnvSem();
    if (!pPortInst) {
        return(ERROR_FILE_NOT_FOUND);
    }
    semQueries = pPortInst->semQueries;

    /*
     * Check for valid input
     */
    if (!pReqInst) {
        return(ERROR_INVALID_PARAMETER);
    }
    /*
     * Loop while all responses have not been received for a command
     */
    while (pReqInst->fRespRecv) {
        rc = DosWaitEventSem ( semQueries, CNV_TIMEOUT );
        if (rc) {
            return(rc);
        }
    }
    /*
     * Get return code from FindQueries()
     */
    rc = pReqInst->rc;

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = GetQSW
 *
 * DESCRIPTION   = Get the query software data for the protocol converter
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *               = pOutData    - Return buffer
 *               = pcbOutData  - Points to length of output buffer
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, unable to build command)
 *
 ****************************************************************************/

ULONG GetQSW ( PSZ pszPortName, PVOID pOutData, PULONG pcbOutData ) {

    PPRTSW pPrtSW = (PPRTSW)pOutData;

    /*
     * Check for valid port name
     */
    if (!CheckPort(pszPortName )) {
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Check for valid size of output buffer
     */
    if ( *pcbOutData < sizeof(PRTSW) ) {
        return(NERR_BufTooSmall);
    }
    /*
     * For sample, set ulJobFlags to:
     *      Printer indicates true end of job 0x0004
     *      Cancel job requires all jobs in printer to be resubmitted 0x0020
     *      Data packets require a wrapper 0x0040
     *      Printer can start a job at a given page 0x0080
     */
    pPrtSW->flJob = PRTSW_JOB_TRUE_END |
                    PRTSW_JOB_MUST_RESUBMIT |
                    PRTSW_JOB_WRAPPER_REQUIRED |
                    PRTSW_JOB_STARTPAGE ;
    /*
     * For sample, set ulDeviceFlags to:
     *      Printer sends alerts when a page completes 0x0001
     *      Printer sends alerts when a job begins printing 0x0002
     *      Printer sends alerts when a job is stacked 0x0004
     */
    pPrtSW->flDevice = PRTSW_DEV_PAGEPRINT_ALERT |
                       PRTSW_DEV_JOBBEGIN_ALERT |
                       PRTSW_DEV_JOBSTACK_ALERT ;
    *pcbOutData = sizeof(PRTSW);

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = BuildSetShutdownCmd
 *
 * DESCRIPTION   = Shutdown BIDI port, releasing all threads waiting for this
 *                 port
 *
 * INPUT         = pszPortName - Name of the port to shutdown
 *               = ppMultiCmd    - Structure containing the commands to send
 *
 * OUTPUT        = Calls RemovePort() to remove port from list and release
 *                 threads waiting on the port
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, port shutdown didn't occur)
 *
 ****************************************************************************/

ULONG BuildSetShutdownCmd ( PSZ pszPortName, PMULTICMD *ppMultiCmd ) {

    ULONG cb;
    PMULTICMD pMultiCmd = NULL;

    if (!RemovePort( pszPortName )) {
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Build empty command to force call to SplPdSendCmd
     */
    cb = (sizeof(ULONG) + sizeof(MULTICMD));
  EnterCnvSem();
    pMultiCmd = (PMULTICMD) AllocCnvMem ( cb );
  LeaveCnvSem();
    /*
     * If memory is available, create command structure
     */
    if (pMultiCmd) {
        memset ( pMultiCmd, 0, cb );
        pMultiCmd->cb = cb;
        pMultiCmd->cCmdStructs = 1;
        pMultiCmd->CmdStructs[0].cb = 0;
    } else {
        *ppMultiCmd = NULL;
        return ( ERROR_NOT_ENOUGH_MEMORY );
    }
    *ppMultiCmd = pMultiCmd;

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = BuildSetProtocolCmd
 *
 * DESCRIPTION   = Initialize protocol converter for a port(BIDI_INIT_PROTCNV)
 *
 * INPUT         = pInData     - A PRTPORT structure from BIDI_Q_PORT
 *               = pszPortname - Name of port to initialize protocol
 *
 * OUTPUT        = Creates port instance if this converter can handle
 *                 the printer connected to the given port.
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, DLL should not be loaded)
 *
 ****************************************************************************/

ULONG BuildSetProtocolCmd ( PVOID pInData, PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;
    PPRTPORT  pPrtPort = (PPRTPORT) pInData;
    PSZ pszProtocol = NULL;
    ULONG     rc;

    //
    // If this is a protocol converter extension,
    //    this would return ERROR_NOT_SUPPORTED if
    //    the specific printer model is not supported
    //    by this protocol converter extension
    //
    rc = ERROR_NOT_SUPPORTED;

    if (pPrtPort) {
        /*
         * Check for Protocol string, convert offset to pointer
         */
        if (pPrtPort->ulpszProtocol) {
            pszProtocol = GETPTRFROMOFFSET(ulpszProtocol,pPrtPort);
        }
        /*
         * Check for valid protocol ( sample only allows NPAP and PJL )
         */
        if (((pPrtPort->flBidiProtocol & PRTPORT_TYPE_NPAP) ||
             (pPrtPort->flBidiProtocol & PRTPORT_TYPE_PJL)) ||
            (pszProtocol && (!strcmp(pszProtocol,"NPAP") ||
                             !strcmp(pszProtocol,"PJL")))) {
            EnterCnvSem ();
            /*
             * Don't allocate port if all ready there
             */
            pPortInst = FindPort ( pszPortName );
            if (!pPortInst) {
                /*
                 * Allocate new port structure
                 */
                pPortInst = AllocatePort ( pszPortName );
                if (pPortInst) {
                    pPortInst->ulBidiFlags = pPrtPort->flBidiCapabilities;
                    /*
                     * Add port to list, if unable, then free port structure
                     */
                    if (!AddPort ( pszPortName, pPortInst )) {
                        FreePort ( pPortInst );
                    }
                }
            }
            LeaveCnvSem();
        }
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = StartCnvUpdateThread
 *
 * DESCRIPTION   = Routine which builds the parameter structure and starts the
 *                 update command thread
 *
 * INPUT         = pfnPdSendCmd - Port Driver send routine address
 *               = pszPortName  - The name of the port to send the commands
 *               = ulType       - Type of set/query and options
 *               = ulCommand    - Generic Command Code
 *               = pMultiCmd    - Structure containing the commands to send
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, rc contains error code)
 *
 ****************************************************************************/

ULONG StartCnvUpdateThread ( PFN pfnPdSendCmd, PSZ pszPortName, ULONG ulType,
                             ULONG ulCommand, PMULTICMD pMultiCmd ) {

    ULONG rc = 0;
    ULONG cb;
    TID tid;
    PCNVSEND pCnvSend = NULL;

    cb = sizeof(CNVSEND) + strlen(pszPortName) + 1;
  EnterCnvSem();
    pCnvSend = AllocCnvMem ( cb );
  LeaveCnvSem();
    /*
     * Build structure to pass to thread
     */
    if (pCnvSend) {
        memset( pCnvSend, 0, cb );
        pCnvSend->signature = CS_SIGNATURE;
        pCnvSend->cb = cb;
        pCnvSend->pfnPdSendCmd = pfnPdSendCmd;
        pCnvSend->ulType = ulType;
        pCnvSend->ulCommand = ulCommand;
        pCnvSend->pMultiCmd = pMultiCmd;
        /*
         * Port name is at end of allocated buffer since original may be
         *  freed before thread is finished.
         */
        pCnvSend->pszPortName = (PSZ)((PBYTE)pCnvSend + sizeof(CNVSEND));
        strcpy ( pCnvSend->pszPortName, pszPortName );
        /*
         * Create update thread with an 8K stack size
         */
        rc = DosCreateThread ( &tid, (PFNTHREAD)CnvUpdateThread,
                               (ULONG) pCnvSend, 0, 8192 );
    } else {
        rc = ERROR_NOT_ENOUGH_MEMORY;
    }

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = CnvUpdateThread
 *
 * DESCRIPTION   = Routine which runs as a thread and calls SendCmd() to
 *                 send the update commands to the printer
 *
 * INPUT         = pCnvSend   - Structure containing parameters for SendCmd()
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID EXPENTRY CnvUpdateThread ( PCNVSEND pCnvSend ) {

    ULONG rc = 0;

    rc = SendCmd ( pCnvSend->pfnPdSendCmd,
                   pCnvSend->pszPortName,
                   pCnvSend->ulType,
                   pCnvSend->ulCommand,
                   NULL,
                   pCnvSend->pMultiCmd );
    /*
     * Free memory allocated by parent thread
     */
  EnterCnvSem();
    FreeCnvMem ( pCnvSend->pMultiCmd, (pCnvSend->pMultiCmd)->cb );
    FreeCnvMem ( pCnvSend, pCnvSend->cb );
  LeaveCnvSem();
    DosExit ( EXIT_THREAD, rc );

}

/****************************************************************************
 *
 * FUNCTION NAME = InitDevice
 *
 * DESCRIPTION   = Initialize Device string for test purposes
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *               = pOutData    - Return buffer
 *               = pcbOutData  - Points to length of output buffer
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, unable to build command)
 *
 ****************************************************************************/
ULONG InitDevice ( PSZ pszPortName, PVOID pOutData, PULONG pcbOutData ) {

    ULONG cb = 0;
    ULONG rc = 0;
    PPRTDEVCHAR pPrtDevChar = (PPRTDEVCHAR)pOutData;
    PBYTE pTemp = NULL;

    /*
     * Check for valid port name
     */
    if (!CheckPort(pszPortName )) {
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Check for valid size of output buffer
     */
    if ( *pcbOutData < sizeof(PRTDEVCHAR) ) {
        return(NERR_BufTooSmall);
    }
    /*
     * Compute total size needed
     */
    cb = sizeof(PRTDEVCHAR) + strlen (PRINTER_NAME) + strlen(REVISION) +
             strlen(SERIALNUM) + 3;
    if (*pcbOutData < cb) {
        rc = ERROR_MORE_DATA;
        *pcbOutData = sizeof ( PRTDEVCHAR );
    } else {
        *pcbOutData = cb;
    }
    /*
     * Setup return buffer
     */
    memset(pPrtDevChar,0,sizeof(PRTDEVCHAR));
    pPrtDevChar->majorRev = 1;
    pPrtDevChar->minorRev = 0;
    pPrtDevChar->cLogicalConnections = 1;
    pPrtDevChar->cPhysicalConnections = 1;
    pPrtDevChar->cActiveConnections = 1;
    pPrtDevChar->maxJobsComplete = 1;
    pPrtDevChar->markingTech = 0x00;
    pPrtDevChar->colorCaps = 0x00;
    pPrtDevChar->colorlevels = 1;
    pPrtDevChar->duplexCaps = 0;
    pPrtDevChar->lengthUnits = 0x03;
    pPrtDevChar->interpreters = 1;
    pPrtDevChar->inputs = 1;
    pPrtDevChar->outputs = 1;
    pPrtDevChar->msgLanguage = 0x00;
    pPrtDevChar->maxOutstandingCmds = 0xFFFF;
    pPrtDevChar->memory = 4;
    if (!rc) {
        pTemp = (PBYTE)((PBYTE)pPrtDevChar + sizeof(PRTDEVCHAR));
        strcpy ( pTemp, PRINTER_NAME );
        pPrtDevChar->ulpszPrinterName = (ULONG) pTemp;
        PTRTOOFFSET(ulpszPrinterName,pPrtDevChar);
        pTemp += strlen ( PRINTER_NAME ) + 1;
        strcpy ( pTemp, REVISION );
        pPrtDevChar->ulpszRevision = (ULONG) pTemp;
        PTRTOOFFSET(ulpszRevision,pPrtDevChar);
        pTemp += strlen ( REVISION ) + 1;
        strcpy ( pTemp, SERIALNUM );
        pPrtDevChar->ulpszSerialNo = (ULONG) pTemp;
        PTRTOOFFSET(ulpszSerialNo,pPrtDevChar);
    }
    return(rc);

}
