/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Sample BIDI Protocol Converter                          */
/* AUTHOR         : Pat D                                                   */
/* FILENAME       : passthru.c                                              */
/* DATE WRITTEN   : 02-28-94                                                */
/* DESCRIPTION    : Routines needed to implement passthru sessions          */
/*                  (Should be common to all protocol converters)           */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                           Routines In File                               */
/*                      AllocPassRespInst()                                 */
/*                      AddPassRespInst()                                   */
/*                      FreePassRespInst()                                  */
/*                      FreePassRespList()                                  */
/*                      GetReadPassthruCmd()                                */
/*                      BuildSendPassthruCmd()                              */
/*                      BuildStartPassthruCmd()                             */
/*                      BuildEndPassthruCmd()                               */
/****************************************************************************/

#define INCL_BASE
#define INCL_DOS
#define INCL_SPL
#define INCL_SPLBIDI
#define INCL_SPLERRORS
#include <os2.h>
#include <string.h>
#include "protcnv.h"
#include "protcnv1.h"
#include "cnvproto.h"

/****************************************************************************
 *
 * FUNCTION NAME = AllocPassRespInst
 *
 * DESCRIPTION   = Allocate the instance of a passthru response
 *
 * INPUT         = pInData
 *               = cbInData
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to Passthru Response Instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

PPASSRESPINST AllocPassRespInst ( PVOID pInData, ULONG cbInData ) {

    PPASSRESPINST pPassInst = NULL;
    PVOID pNewData = NULL;

    pPassInst = (PPASSRESPINST) AllocCnvMem ( sizeof (PASSRESPINST) );
    if (pPassInst) {
        memset ( pPassInst, 0, sizeof(PASSRESPINST) );
        pPassInst->signature = PR_SIGNATURE;
        pPassInst->cb = sizeof(PASSRESPINST);
        pNewData = (PSZ) AllocCnvMem ( cbInData );
        if (pInData) {
            memcpy ( pNewData, pInData, cbInData );
            pPassInst->pInData = pNewData;
            pPassInst->cbInData = cbInData;
        } else {
            FreeCnvMem ( pPassInst, sizeof(PASSRESPINST));
            return(NULL);
        }
    } else {
        return ( NULL );
    }

    return(pPassInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPassRespInst
 *
 * DESCRIPTION   = Add a passthru response instance to the list in the protocol
 *                 converter
 *
 * INPUT         = pInData
 *               = cbInData
 *               = pPortInst - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful)
 *
 * RETURN-ERROR  = 0(Failure, Port already existed)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

BOOL AddPassRespInst ( PVOID pInData, ULONG cbInData, PPORTINST pPortInst ) {

    PPASSRESPINST pPassInst = NULL;
    PPASSRESPINST pPassLast = pPortInst->pPassRespLast;

    /*
     * Create new passthru response instance
     */
    if (!(pPassInst = AllocPassRespInst ( pInData, cbInData ))) {
        return(0);
    }
    /*
     * If list exists, add port to end of list
     *  else, start list and set end
     */
    if (pPassLast) {
        pPassLast->pNext = pPassInst;
        pPortInst->pPassRespLast = pPassInst;
    } else {
        pPortInst->pPassRespList = pPassInst;
        pPortInst->pPassRespLast = pPassInst;
    }
    /*
     * Check for maximum number of responses
     */
    if (pPortInst->cPassRespInst == MAX_PASSTHRU_RESP) {
        /*
         * Remove oldest(first) response from the list
         */
        pPassInst = pPortInst->pPassRespList;
        pPortInst->pPassRespList = pPassInst->pNext;
        FreePassRespInst ( pPassInst );
    } else {
        (pPortInst->cPassRespInst)++;
    }

    return(1);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePassRespInst
 *
 * DESCRIPTION   = Free Memory and resources associated with a passthru
 *                 response instance
 *
 * INPUT         = pPassRespInst - pointer to the passthru response instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful)
 *
 * RETURN-ERROR  = 0(Failure)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

BOOL FreePassRespInst ( PPASSRESPINST pPassRespInst ) {

    BOOL rc = FALSE;

    if (pPassRespInst) {
        if (pPassRespInst->pInData) {
            FreeCnvMem ( pPassRespInst->pInData, pPassRespInst->cbInData );
        }
        FreeCnvMem ( pPassRespInst, pPassRespInst->cb );
      rc = TRUE;
    }

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePassRespList
 *
 * DESCRIPTION   = Free all passthru responses in a list
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, All ports could not be freed)
 *
 * NOTE: Caller must be in the protocol converter semaphore
 *
 ****************************************************************************/

ULONG FreePassRespList ( PPASSRESPINST pPassRespList ) {

    PPASSRESPINST pPassRespInst;
    PPASSRESPINST pPassRespNext;
    ULONG rc = 0;

    pPassRespInst = pPassRespList;

    /*
     * Loop through list and free each port
     */
    while (pPassRespInst) {
        pPassRespNext = pPassRespInst->pNext;
        rc = FreePassRespInst ( pPassRespInst );
        pPassRespInst = pPassRespNext;
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = GetReadPassthruCmd
 *
 * DESCRIPTION   = Determine if there is any data in the Passthru buffer
 *                 that has not already been read.  If so, return the data
 *                 to the caller.  [No commands will be sent to the port
 *                 driver.]
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *               = pOutData    - Return buffer
 *               = pcbOutData  - Points to length of output buffer
 *                               On entry this is set to length of output
 *                                 buffer passed in.
 *                               On exit this is updated with length of data
 *                                 returned.
 *               = pfGotResp   - Pointer to boolean value indicating whether
 *                               response was obtained
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 1(Successful)
 *
 * RETURN-ERROR  = 0(Failure, DLL should not be loaded)
 *
 ****************************************************************************/

ULONG GetReadPassthruCmd ( PSZ pszPortName, ULONG ulType, PVOID pOutData,
                           PULONG pcbOutData, PBOOL pfGotResp) {

    PPORTINST pPortInst = NULL;
    PVOID pNewData;
    PPASSRESPINST pPassRespInst = NULL;
    ULONG rc = 0;
    ULONG ulSize = 0;
    ULONG cbData;

  EnterCnvSem();
    /*
     * Find port instance data
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Is there a passthru session started
     */
    if (!(pPortInst->fPassthruSes)) {
      LeaveCnvSem();
        return(PMERR_SPL_INV_HSPL);
    }
    if (pPortInst->pPassRespList) {
        pPassRespInst = pPortInst->pPassRespList;
        cbData = pPassRespInst->cbInData;
        if (ulType == TYPE_WITHOUT_WRAPPER) {
            /*
             * If enough room in output buffer, copy data to it
             */
            ulSize = cbData - GetWrapperSize();
            if (ulSize <= *pcbOutData) {
                pNewData = RemoveWrapper ( pPassRespInst->pInData, &cbData );
                memcpy ( pOutData, pNewData, cbData );
            } else {
                rc = NERR_BufTooSmall;
            }
            *pcbOutData = ulSize;
        } else {
            /*
             * If enough room in output buffer, copy data to it
             */
            ulSize = pPassRespInst->cbInData;
            if (ulSize <= *pcbOutData) {
                memcpy ( pOutData, pPassRespInst->pInData, ulSize );
            } else {
                rc = NERR_BufTooSmall;
            }
            *pcbOutData = ulSize;
        }
        /*
         * If no error, do processing for passthru read
         */
        if (!rc) {
            /*
             * Remove first passthru buffer from list
             */
            pPortInst->pPassRespList = pPassRespInst->pNext;
            if (pPassRespInst == pPortInst->pPassRespLast) {
                pPortInst->pPassRespLast = NULL;
            }
            FreePassRespInst ( pPassRespInst );
            (pPortInst->cPassRespInst)--;
        }
        *pfGotResp = TRUE;
    } else {
        *pfGotResp = FALSE;
    }

  LeaveCnvSem();

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = BuildSendPassthruCmd
 *
 * DESCRIPTION   = Build the set passthru command for the printer
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *               = pInData     - Information required by the command
 *               = cbInData    - Length in bytes of data in pInData
 *               = pMultiCmd    - Structure containing the commands to send
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Unable to build command)
 *
 ****************************************************************************/

ULONG BuildSendPassthruCmd ( PSZ pszPortName, PVOID pInData, ULONG cbInData,
                             ULONG ulType, PMULTICMD *ppMultiCmd ) {

    PPORTINST pPortInst = NULL;
    ULONG cb = cbInData;
    ULONG cbData = cbInData;
    ULONG cbTemp;
    ULONG rc = 0;
    PMULTICMD pMultiCmd = NULL;

  EnterCnvSem();
    /*
     * Find port instance data
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * Is there a passthru session started
     */
    if (!(pPortInst->fPassthruSes)) {
      LeaveCnvSem();
        return(PMERR_SPL_INV_HSPL);
    }
    /*
     * Verify data and size is passed in
     */
    if (pInData && cbInData) {
        cb += sizeof (MULTICMD);
        if (ulType == TYPE_WITHOUT_WRAPPER) {
            cbTemp = GetWrapperSize();
            cb += cbTemp;
            cbData += cbTemp;
        }
        pMultiCmd = (PMULTICMD) AllocCnvMem ( cb );
        /*
         * If memory is available, create command structure
         */
        if (pMultiCmd) {
            pMultiCmd->cb = cb;
            pMultiCmd->cCmdStructs = 1;
            pMultiCmd->CmdStructs[0].cb = cbData;
            if (ulType == TYPE_WITHOUT_WRAPPER) {
                if (!AddWrapper ( (PVOID)&(pMultiCmd->CmdStructs[0].Cmd),
                                  pInData, cbInData )) {
                    FreeCnvMem ( pMultiCmd, cb );
                    pMultiCmd = NULL;
                    rc = ERROR_NOT_ENOUGH_MEMORY;
                }
            } else {
                memcpy ( &(pMultiCmd->CmdStructs[0].Cmd) , pInData, cbInData );
            }
        } else {
            pMultiCmd = NULL;
            rc = ERROR_NOT_ENOUGH_MEMORY;
        }
    }
    *ppMultiCmd = pMultiCmd;
  LeaveCnvSem();

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = BuildStartPassthruCmd
 *
 * DESCRIPTION   = Start a passthru session
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Unable to build command)
 *
 ****************************************************************************/

ULONG BuildStartPassthruCmd ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;

  EnterCnvSem();
    /*
     * Find port instance data
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * If passthru session active, return error
     */
    if (pPortInst->fPassthruSes) {
      LeaveCnvSem();
        return(PMERR_SPL_INV_HSPL);
    } else {
        pPortInst->fPassthruSes = TRUE;
    }
  LeaveCnvSem();

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = BuildEndPassthruCmd
 *
 * DESCRIPTION   = End the current passthru session
 *
 * INPUT         = pszPortName - The name of the port to send the commands
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, Unable to build command)
 *
 ****************************************************************************/

ULONG BuildEndPassthruCmd ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;

  EnterCnvSem();
    /*
     * Find port instance data
     */
    if (!(pPortInst = FindPort( pszPortName ))) {
      LeaveCnvSem();
        return(ERROR_FILE_NOT_FOUND);
    }
    /*
     * If no passthru session active, return error
     */
    if (!pPortInst->fPassthruSes) {
      LeaveCnvSem();
        return(PMERR_SPL_INV_HSPL);
    } else {
        /*
         * Free resources and reset values
         */
        pPortInst->fPassthruSes = FALSE;
        FreePassRespList ( pPortInst->pPassRespList );
        pPortInst->pPassRespList = NULL;
        pPortInst->pPassRespLast = NULL;
        pPortInst->cPassRespInst = 0;
    }
  LeaveCnvSem();

    return(0);

}

