/****************************************************************************/
/*  Copyright (C) 1995 IBM Corporation                                      */
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = par1284.C
 *
 * DESCRIPTIVE NAME = par1284 port driver worker routines used
 *                     to communicate with an par1284 attached
 *                     printer
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1994, 1995
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.2
 *
 * DATE
 *
 * DESCRIPTION
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *  DATE      FLAG        APAR   CHANGE DESCRIPTION
 *  --------  ----------  -----  --------------------------------------
 *  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 ****************************************************************************/

#include    "pdrconst.h"
#include    "pdrtypes.h"
#include    "pdrproto.h"


int acrtused=1;                     /* Define variable to say this is a DLL */

/*
** Local Functions
*/
VOID EXPENTRY ControlThread( VOID );
ULONG         GetDeviceID( PPORTINST pPortInst ) ;
ULONG         SetMode( HFILE hFile, USHORT usMode, PUSHORT pCurMode);
TID           CreateReadThread( PPORTINST pPortInst );
VOID EXPENTRY ParReadThread( PPORTINST pPortInst );
ULONG         CheckAllPorts( PPRTALERT pPrtAlert, PULONG pcbBuf,
                             PBOOL pfAlertReturned );
VOID EXPENTRY GetDevidThread( PPORTINST pPortInst );


/*
** par1284 functions
*/

/*****************************************************************************
 * FUNCTION NAME = PdOpen
 *
 * DESCRIPTION   = Open connection to PAR1284 Device Driver
 *
 * INPUT         = pPortInst -> port instance struct for PAR1284 printer
 *                              to open a connection with
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = 0 - Success
 *
 * RETURN-ERROR  = error
 *
 * NOTE            Must be in port driver semaphore on entry/exit,
 *                 but leaves Sem and re-enters it.
 *
 **************************************************************************/

ULONG PdOpen( PPORTINST pPortInst )
{
  BOOL                     fSuccess;
  ULONG                    rc;
  PSZ                      pszPortName;
  PSZ                      pszNewPrinter;
  HFILE                    hFile;
  ULONG                    ActionTaken;
  ULONG                    cRetry;

  #ifdef DEBUG_ALERT
    char logbuf[260];
  #endif


  pszPortName = pPortInst->pszPortName;
  rc          = 0;
  hFile       = 0;
  /*
  ** Set timestamp to make certain ControlThread does not close port
  */
  pPortInst->ulTimeLastCmd = time();

  if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PDOPEN))
  {
     DBPRINTF ((logbuf, "PdOpen Entry for %s", pszPortName));
  }

  /*
  ** Only allow 1 thread to open the port
  */
  if (pPortInst->flStatus & PF_OPEN_IN_PROGRESS) {
    /*
     * Hold caller until other thread opening this port completes.
     * If it takes longer than 30 seconds, return an error.
     */
    cRetry = 0;
    do
    {
       LeavePdrSem();
        if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PDOPEN))
        {
          DBPRINTF ((logbuf, "PdOpen waiting for other thread to finish flStatus=%lX port=%s", pPortInst->flStatus, pPortInst->pszPortName));
        }
        DosSleep( 2000 );
       EnterPdrSem();
    } while ( (pPortInst->flStatus & PF_OPEN_IN_PROGRESS) && (cRetry < 15) );

    if ((pPortInst->flStatus & PF_OPEN_IN_PROGRESS) ||
        !(pPortInst->flStatus & PF_PORT_OPEN) ) {
      /*
       * Set ERROR INIT IN PROGRESS      @BUGBUG
       */
      rc = PMERR_SPL_INIT_IN_PROGRESS;
    }

  } else {

    pPortInst->flStatus |= PF_OPEN_IN_PROGRESS;
   LeavePdrSem();
    /*
    ** Get timeouts from system INI
    */
    GetTimeOuts( pszPortName, pPortInst );

    rc = DosOpen ( pszPortName,               /* LPT port name           */
                  &hFile,                     /* File handle             */
                  &ActionTaken,               /* Action taken            */
                  0,                          /* File primary allocation */
                  FILE_NORMAL,                /* File attribute          */
                  OPEN_ACTION_OPEN_IF_EXISTS  /* Open function type      */
                  |OPEN_ACTION_CREATE_IF_NEW,
                  OPEN_FLAGS_FAIL_ON_ERROR    /* Open mode of the file   */
                  | OPEN_FLAGS_NONSPOOLED
                  | OPEN_SHARE_DENYNONE
                  | OPEN_ACCESS_READWRITE,
                  NULL);                      /* Ptr to EA buffer        */
    if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PDOPEN))
    {
       DBPRINTF ((logbuf, "PdOpen DosOpen rc=%d hFile=%d", rc, hFile));
    }
   EnterPdrSem();
    if (!rc)
    {
       pPortInst->hFile           = hFile;
       pPortInst->flStatus       |= PF_PORT_OPEN;
       //
       // Don't fail PdOpen if GetDeviceID returns an error
       // The printer might not be bidi-capable.
       //
       GetDeviceID( pPortInst );
       pPortInst->ulTimeLastCmd   = time();
    }
    pPortInst->flStatus &= ~PF_OPEN_IN_PROGRESS;
  }

  if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PDOPEN))
  {
     DBPRINTF ((logbuf, "PdOpen rc=%d hFile=%d",rc, hFile));
  }
  return(rc);

} /* end function PdOpen */

/***************************************************************************
 * FUNCTION NAME = PdWrite
 *
 * DESCRIPTION   = Send data to the parallel attached printer.
 *
 * INPUT         = pPortInst   -> destination port structure
 *                 pchData     -> buffer to write to port
 *                 cbData      -  size in bytes to write
 *                 pcbWritten  -> ULONG to get #bytes written
 *
 * OUTPUT        = rc from DosWrite
 *
 * RETURN-NORMAL = 0 - Success
 *
 * RETURN-ERROR  = other
 *
 * NOTES         = Must have pPortInst->hPortSem before calling!
 *
 **************************************************************************/
ULONG PdWrite(PPORTINST pPortInst, PVOID pchData,
                         ULONG cbData, PULONG pcbWritten)
{
  ULONG rc;

  #ifdef DEBUG_ALERT
    char logbuf[260];
    DATETIME    TimeBefore;
    DATETIME    TimeAfter;
  #endif


  #ifdef DEBUG_ALERT
   if (flChangeDebugLog & FL_PDRDB_ENABLE_PDWRITE)
   {
      sprintf( logbuf,
               "PdWrite cb=%d hFile=%d\r\n",
               cbData, pPortInst->hFile );
      LogCall( logbuf );
      if (flChangeDebugLog & FL_PDRDB_ENABLE_WRITE_DUMP)
      {
         DumpHex( pchData, cbData );
      }
   }
  #endif /* DEBUG */

  //
  // Check PF_CMD_CHNL_ACTIVE to ensure we are sending the data  @BUGBUG
  //   down the correct ECP channel( data/command channel ).
  //

  /*
   * Init bytes written to zero
   */
  *pcbWritten = 0 ;

  #ifdef DEBUG_ALERT
   DosGetDateTime( &TimeBefore );
  #endif /* DEBUG */

  rc = DosWrite(pPortInst->hFile, pchData, cbData, pcbWritten);

  #ifdef DEBUG_ALERT
   DosGetDateTime( &TimeAfter );
  #endif /* DEBUG */

  #ifdef DEBUG_ALERT
   if (flChangeDebugLog & FL_PDRDB_ENABLE_PDWRITE)
   {
      sprintf( logbuf,
               "PdWrite rc=%d *pcbWritten=%d TimeBefore=%02d.%02d TimeAfter=%02d.%02d\r\n",
               rc,
               *pcbWritten,
               TimeBefore.seconds,
               TimeBefore.hundredths,
               TimeAfter.seconds,
               TimeAfter.hundredths
               );
      LogCall( logbuf );
   }
  #endif /* DEBUG */

  return (rc);
}                                               /* end function PdWrite */


/***************************************************************************
 * FUNCTION NAME = PdClose
 *
 * DESCRIPTION   = Disconnect, unBind, Shutdown, and unload LMDLL.DLL
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = 0 - Success
 *
 * NOTE            Must be in port driver semaphore on entry/exit,
 *                 but leaves Sem and re-enters it.
 *
 **************************************************************************/

ULONG PdClose( PPORTINST pPortInst )
{
  ULONG    rc;
  #ifdef DEBUG_ALERT
    char logbuf[260];
  #endif

  if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PDOPEN))
  {
     DBPRINTF ((logbuf, "PdClose called for hFile=%d pPortInst=%lX", pPortInst->hFile, (ULONG)pPortInst));
  }

  if (pPortInst->hFile) {

    LeavePdrSem();
     rc = DosClose( pPortInst->hFile );

     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PDOPEN))
     {
        DBPRINTF ((logbuf, "PdClose DosClose rc=%d hFile=%d", rc, pPortInst->hFile));
     }

    EnterPdrSem();
  }
  if (pPortInst->signature == PT_SIGNATURE) {
     pPortInst->hFile     = 0;
     pPortInst->flStatus &= ~PF_PORT_OPEN;
  }

  return(rc);

}   /* end function PdClose */

/****************************************************************************
 *
 * FUNCTION NAME = HandleQPort
 *
 * DESCRIPTION   = Process BIDI_Q_PORT command
 *                 This tries to connect to the Par1284 printer and
 *                 returns the printer device ID.
 *
 *
 * INPUT         = pszDeviceName   - name of port printer is attached
 *                 ulFlags         - query options(cache, short wait,...)
 *                 pOutData        - receives PRTPORT structure, which includes
 *                                   variable length strings packed
 *                                   immediately after the PRTPORT structure
 *                 pcbOutData      - Points to length of pOutData(in bytes)
 *                                   On entry this is set to length of buffer
 *                                    passed in.
 *                                   On exit this is updated with the length
 *                                    of available data, which may be more
 *                                    than put into pOutData if ERROR_MORE_DATA
 *                                    or NERR_BufTooSmall returned.
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG HandleQPort( PSZ    pszDeviceName,
                   ULONG  ulFlags,
                   PVOID  pOutData,
                   PULONG pcbOutData )
{
    ULONG       rc;
    ULONG       cbNeeded;
    PSZ         pszConnectName;
    PPORTINST   pPortInst;
    PPRTPORT    pPrtPort;
    TID         tid;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif



    if (!(flChangeDebugLog & FL_PDRDB_DISABLE_Q_PORT))
    {
       DBPRINTF ((logbuf, "HandleQPort called for %s", pszDeviceName));
    }
    rc = 0;

   EnterPdrSem();
    /*
     * Add port to the list of supported ports if it is not yet
     *   in the port list.
     */
    pPortInst = AddPortInst ( pszDeviceName );
    if (pPortInst ) {

       /*
       ** Set TimeLastCmd so that we don't close this
       **   port from our ControlThread
       */
       pPortInst->ulTimeLastCmd = time();

       /*
       ** open Par1284 device if not yet opened
       ** PdOpen will call GetDeviceID()
       */
       if ( !(pPortInst->flStatus & PF_PORT_OPEN) ) {
          rc =  PdOpen(pPortInst);
       } else if ( ulFlags == TYPE_LONG_WAIT ) {
          /*
          ** If port already open, only get current deviceID
          **   if LONG_WAIT specified, otherwise return
          **   cached information.
          */
          rc = GetDeviceID( pPortInst );
       } else if ( ulFlags == TYPE_CACHE_UPDATE ) {
          /*
          ** CacheUpdate asks to give current cache and start
          **   a thread to query current state to update cache.
          */
          tid = 0;

          DosCreateThread( &tid,
                           (PFNTHREAD)GetDevidThread,
                           (ULONG)pPortInst,          // Parameter
                           0,                         // Execution Flag
                           DEFAULT_STACKSIZE);
       }
       /*
       ** Try to load new spooler Bidi APIs
       */
       if (fLoadedPMSPL == FALSE)
       {
          LoadPMSPL();
       }
       if (pfnPrtQuery)
       {
          /*
          ** Set port flag indicating that bidi spooler is installed
          */
          pPortInst->flStatus |= PF_Q_PORT_CALLED;
       } else if (rc == 0) {

          /*
          ** If we opened the port just to get the device list
          **   then we need to close the port now so others can use it.
          */
          PdClose(pPortInst);
       }
       if (!rc)
       {
          cbNeeded = sizeof (PRTPORT) + strlen(pPortInst->szDeviceID) + 1;
          if (*pcbOutData >= sizeof(PRTPORT) ) {
              pPrtPort = (PPRTPORT) pOutData;
              memset ( pPrtPort, 0, sizeof (PRTPORT));
              pPrtPort->flBidiCapabilities = pPortInst->flBidiCapabilities;
              pPrtPort->flBidiProtocol     = pPortInst->flBidiProtocol;
              pPrtPort->ulPortType = PRTPORT_PORT_TYPE_LPT;
              pPrtPort->ulBidiLevel = 0;
              pPrtPort->ulMaxSendSize = 4096;
              pPrtPort->ulMaxReceiveSize = 4096;
              pPrtPort->ulMaxHeldResponses = 0;
              pPrtPort->ulpszDeviceID = sizeof(PRTPORT);
              if (*pcbOutData >= cbNeeded) {
                 pszConnectName = (PSZ)pPrtPort + pPrtPort->ulpszDeviceID;
                 strcpy( pszConnectName, pPortInst->szDeviceID);
              } else {
                 /*
                 ** The entire deviceID string would not fit, so
                 **   return all that will fit into caller's buffer.
                 */
                 rc = ERROR_MORE_DATA;
              }
          } else {
              /*
              ** Buffer given won't fix PRTPORT structure - don't return anything
              */
              rc = NERR_BufTooSmall;
          }
          *pcbOutData = cbNeeded;
       }
    } else {
        rc = ERROR_NOT_ENOUGH_MEMORY;
    }
   LeavePdrSem();

    return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = HandleWaitAlert
 *
 * DESCRIPTION   = Process BIDI_WAIT_ALERT command
 *                 This holds caller until data is received from one
 *                 of the ports supported by this port driver.
 *
 * INPUT         = pszDeviceName   - name of port printer is attached
 *                 ulFlags         - query options
 *                 ulCommand       - function code for query
 *                 pInData         - command specific input data
 *                 cbInData        - length in bytes of pInData
 *                 pOutData        - returned query structure, format depends
 *                                   on ulCommand
 *                 pcbOutData      - Points to length of pOutData(in bytes)
 *                                   On entry this is set to length of buffer
 *                                    passed in.
 *                                   On exit this is updated with the length
 *                                    of available data, which may be more
 *                                    than put into pOutData
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG HandleWaitAlert( PSZ    pszDeviceName,
                       ULONG  ulFlags,
                       ULONG  ulCommand,
                       PVOID  pInData,
                       ULONG  cbInData,
                       PVOID  pOutData,
                       PULONG pcbOutData )
{
    ULONG         rc;
    PDALERTINFO   PdAlertInfo;
    PBYTE         pPrinterData;
    ULONG         cbActual;
    ULONG         ulPostCount;
    PPORTINST     pPortInst;
    PWAITALERTBUF pWaitBuf;           /* Current WAITALERTBUF we are using  */
    PALERTBUF     pAlertBuf;
    ULONG         cRetry;
    BOOL          fAlertReturned;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    /*
     * Find port instance
     */
    rc             = 0;
    pPrinterData   = NULL;
    fAlertReturned = FALSE;

    if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
    {
       DBPRINTF ((logbuf, "HandleWaitAlert called for %s", pszDeviceName));
    }

   EnterPdrSem();
    pPortInst = FindPortInst ( pszDeviceName );
    if (!pPortInst ) {
        rc = ERROR_FILE_NOT_FOUND;
    } else if (!PdrWaitAlert.pCurrBuf) {
       /*
        * Must allocate WaitAlertBuffer for this port driver
        */
       rc = InitWaitAlertBufs();
    }
    if (!rc) {
       /*
       ** Go through list of LPT ports and start a read thread for
       **  any that have had a BIDI_Q_PORT issued for it.
       ** Also check for PowerON condition(from deviceID changing)
       **  and return CommStatChange alert to make spooler re-init port.
       ** fAlertReturned is set to TRUE if we need to return to caller.
       */
       rc = CheckAllPorts( (PPRTALERT)pOutData, pcbOutData, &fAlertReturned );

    }
   LeavePdrSem();
    if (rc || fAlertReturned) {

       #ifdef DEBUG_ALERT
        {
         if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
         {
            sprintf( logbuf,
                     "HandleWaitAlert rc=%d fPowerOn=%d on %s pPortInst=%lX\r\n",
                     rc, fAlertReturned, pszDeviceName, (ULONG)pPortInst );
            LogCall( logbuf );
         }
        }
       #endif /* DEBUG */

       return(rc);
    }

    PdAlertInfo.ulFlags = 0;
    /*
     * Only return to caller when an alert is read
     */
    while ( !(PdAlertInfo.ulFlags & PD_ALERT) && !rc) {
        memset( &PdAlertInfo, 0, sizeof(PDALERTINFO));
        cbActual = 0;
        pPrinterData = NULL;
       EnterPdrSem();
        if ((PdrWaitAlert.pCurrBuf->cAlertsRemaining == 0) &&
            (PdrWaitAlert.pXlateBuf->cAlertsRemaining == 0) &&
            (!PdrWaitAlert.pXlateBuf->fPdMoreAlerts)) {
          LeavePdrSem();
           if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
           {
              DBPRINTF ((logbuf, "HandleWaitAlert Waiting for %s", pszDeviceName));
           }
           rc = DosWaitEventSem ( PdrWaitAlert.hevWaitAlert, SEM_INDEFINITE_WAIT );
           if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
           {
              DBPRINTF ((logbuf, "HandleWaitAlert WokeUp rc=%d", rc ));
           }
           rc = DosResetEventSem(PdrWaitAlert.hevWaitAlert, &ulPostCount);
          EnterPdrSem();
        }
        rc = 0;
        /*
        ** Go through list of LPT ports and start a read thread for
        **  any that have had a BIDI_Q_PORT issued for it.
        ** Also check for PowerON condition(from deviceID changing)
        **  and return PowerOn alert to make spooler re-init port.
        ** fAlertReturned is set to TRUE if we need to return to caller.
        */
        fAlertReturned = FALSE;
        rc = CheckAllPorts( (PPRTALERT)pOutData, pcbOutData, &fAlertReturned );
        if (fAlertReturned) {

          LeavePdrSem();
           #ifdef DEBUG_ALERT
            {
             if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
             {
                sprintf( logbuf,
                         "HandleWaitAlert rc=%d fPowerOn=%d on %s pPortInst=%lX\r\n",
                         rc, fAlertReturned, pszDeviceName, (ULONG)pPortInst );
                LogCall( logbuf );
             }
            }
           #endif /* DEBUG */

           return(rc);
        }

        if ((PdrWaitAlert.pXlateBuf->cAlertsRemaining == 0) &&
            (PdrWaitAlert.pCurrBuf->cAlertsRemaining) &&
            (!PdrWaitAlert.pXlateBuf->fPdMoreAlerts)) {
           //
           // Swap buffers so we can process the alerts stored
           //  in pCurrBuf.
           // This thread(BIDI_WAIT_ALERT) always processes pXlateBuf.
           // When data is received from a port, it is always
           //  stored in pCurrBuf.
           // When swapping buffers, the new pCurrBuf will be reset
           //  to allow new alerts to be stored in it.
           //
           pWaitBuf = PdrWaitAlert.pCurrBuf;
           PdrWaitAlert.pCurrBuf  = PdrWaitAlert.pXlateBuf;
           PdrWaitAlert.pXlateBuf = pWaitBuf;
           memset( PdrWaitAlert.pCurrBuf, 0, sizeof(WAITALERTBUF));
           PdrWaitAlert.pCurrBuf->cbBufSize = DEF_WAITALERTBUF_SIZE;
           PdrWaitAlert.pCurrBuf->cbBufUsed = sizeof(WAITALERTBUF) -
                                              sizeof(ALERTBUF);
        }
        if ((PdrWaitAlert.pXlateBuf->cAlertsRemaining) ||
            (PdrWaitAlert.pXlateBuf->fPdMoreAlerts)) {
           pWaitBuf = PdrWaitAlert.pXlateBuf;
           //
           // Special case the first alert in the buffer.
           //
           pAlertBuf = pWaitBuf->pLastProcessed;
           if (!pAlertBuf) {
              pAlertBuf = &pWaitBuf->FirstAlertBuf;
           }
           //
           // If the last time we called SplProtXlateCmd we received
           //   PD_MOREALERTS bit set, then we must call XlateCmd
           //   again with PD_NEXTALERT set to let the protocol converter
           //   give us the additional alert from the previous buffer.
           //   This can happen if more than 1 alert is in the buffer
           //   we pass to XlateCmd().
           //
           if (PdrWaitAlert.pXlateBuf->fPdMoreAlerts) {
              cbActual = 0;
              PdAlertInfo.ulFlags = PD_NEXTALERT;
              //
              // Temp bug workaround - some code does not accept  @BUGBUG
              //   NULL for pInBuf on SplProtXlateCmd().
              // For now just point it to our stack since it is readonly
              //
              //pPrinterData = NULL;
              pPrinterData = (PBYTE)&PdAlertInfo;
              #ifdef DEBUG_ALERT
               {
                if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
                {
                   sprintf( logbuf,
                            "HandleWaitAlert PD_NEXTALERT being set for ProtXlateCmd on %s\r\n",
                            pszDeviceName );
                   LogCall( logbuf );
                }
               }
              #endif /* DEBUG */
           } else {
              //
              // Get past the last processed alert buffer and
              //   reset the header data.
              // Special case the first alert in the buffer.
              //
              if (pWaitBuf->pLastProcessed) {
                 pAlertBuf = (PALERTBUF)((PBYTE)pAlertBuf +
                                                sizeof(ALERTBUF) +
                                                pAlertBuf->cbData);
              }
              pPrinterData = (PBYTE)pAlertBuf + sizeof(ALERTBUF);
              cbActual = pAlertBuf->cbData;
              pWaitBuf->pLastProcessed = pAlertBuf;
              pWaitBuf->cAlertsRemaining--;

           }
           //
           // Set portname to the one we received the alert from
           //
           pszDeviceName = pAlertBuf->pszAlertPortName;
          LeavePdrSem();
           rc = MySplProtXlateCmd ( pszDeviceName,
                                    (PFN)NULL,
                                    pPrinterData,
                                    cbActual,
                                    &PdAlertInfo,
                                    pOutData,
                                    pcbOutData );
           #ifdef DEBUG_ALERT
            {
             if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
             {
                sprintf( logbuf,
                         "PAR1284 HandleWaitAlert ProtXlate rc=%d on %s ulFlags=%lX cb=%d\r\n",
                         rc, pszDeviceName, (ULONG)PdAlertInfo.ulFlags, cbActual );
                LogCall( logbuf );
             }
            }
           #endif /* DEBUG */
          EnterPdrSem();
           /*
           ** Must check for the protocol converter telling us
           **  that more than one alert was stored in the
           **  buffer pased to SplProtXlateCmd.
           ** In this case, we need to call the converter again
           **  with a NULL buffer to let the converter return
           **  the next alert to the caller of BIDI_WAIT_ALERT
           **/
           if (PdAlertInfo.ulFlags & PD_MOREALERTS) {
              PdrWaitAlert.pXlateBuf->fPdMoreAlerts = TRUE;
           } else {
              PdrWaitAlert.pXlateBuf->fPdMoreAlerts = FALSE;
           }
          LeavePdrSem();
        } else {
           //
           // We woke up for some reason.
           // For now, just sleep and wait for next wake-up
           //
          LeavePdrSem();
           #ifdef DEBUG_ALERT
            {
             if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
             {
                sprintf( logbuf,
                         "HandleWaitAlert wokeup for no reason - sleeping 1 second\r\n");
                LogCall( logbuf );
             }
            }
           #endif /* DEBUG */
           DosSleep( 1000 );
        }
    }
    return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = HandleQPortDRV
 *
 * DESCRIPTION   = Process BIDI_Q_PORTDRV command to return port driver
 *                 settings data structure for the given port.
 *
 * INPUT         = pszPortName   - Name of port
 *                 pPortSettings   -> buffer to get port settings
 *                 pcbPortSettings -> length of buffer(on entry)
 *                                    Set to length of data put
 *                                    into buffer(on exit)
 *
 * OUTPUT        = 0   - Success, pPortSettings and *pcbPortSettings updated
 *                 Other - failed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *                 Called by:
 *                  1) SplPdQuery(BIDI_Q_PORTDRV) under bidi spooler process
 *                  2) OpenPar1284PortDlg() if bidi spooler not installed or
 *                     spooler not enabled.
 *
 ****************************************************************************/

ULONG HandleQPortDRV ( PSZ           pszPortName,
                       PPORTSETTINGS pPortSettings,
                       PULONG        pcbPortSettings )
{

    ULONG                    cbNeeded;
    ULONG                    rc;
    ULONG                    fSuccess;
    PSZ                      pszDeviceID;
    PSZ                      pszTemp;
    ULONG                    flPortStatus;
    ULONG                    i;
    PBYTE                    pByte;
    PPORTINST                pPortInst;
    CHAR                     chDesc[CCHMAXPATH];
    CHAR                     chAppName[CCHMAXPATH];
    CHAR                     szBuf[20];
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif



    if (!(flChangeDebugLog & FL_PDRDB_DISABLE_QPORTDRV))
    {
       DBPRINTF ((logbuf, "HandleQPortDRV called for %s", pszPortName));
    }
    rc            = 0;
    flPortStatus  = 0;

   EnterPdrSem();
    pPortInst = AddPortInst ( pszPortName );
    if (pPortInst )
    {
       /*
       ** Set TimeLastCmd so that we don't close this
       **   port from our ControlThread
       */
       pPortInst->ulTimeLastCmd = time();
       flPortStatus = pPortInst->flStatus;
    } else {
        rc = ERROR_FILE_NOT_FOUND;
    }
    /*
    ** Must get printer device ID.
    ** This requires the port to be open.
    ** If port already open, just return current information;  the user
    **   can select "Refresh" pushbutton to force us to query the deviceID.
    */
    if ( !rc && !(pPortInst->flStatus & PF_PORT_OPEN))
    {
       rc = PdOpen(pPortInst);
    }
    if (rc)
    {
     LeavePdrSem();
      DBPRINTF ((logbuf, "HandleQPortDRV after PdOpen check rc=%d(%s)", rc, pszPortName));
      return(rc);
    }
    cbNeeded      = 0;

    /*
    ** If we opened the PAR1284 just to get the deviceID
    **   then we need to close the LPT now so others can use it.
    */
    if ( !(flPortStatus & PF_PORT_OPEN) &&
         !(flPortStatus & PF_OPEN_IN_PROGRESS) )
    {
       if (pPortInst && !(pPortInst->flStatus & PF_Q_PORT_CALLED))
       {
          PdClose(pPortInst);
       }
    }
    /*
    ** Get port information from INI file
    */
    strcpy (chAppName, APPNAME_LEAD_STR);
    strcat (chAppName, pszPortName);
    /*
     * Get size of return data
     */
    cbNeeded += sizeof(PORTSETTINGS)
                + strlen(pszPortName) + 1
                + strlen(chDesc) + 1
                + strlen(pPortInst->szDeviceID) + 1;
    /*
     * Check size of return buffer
     */
    if (cbNeeded > *pcbPortSettings) {
       rc = NERR_BufTooSmall;
    }

    if (!rc)
    {
       /*
        * Copy values and strings to the buffer
        */
       pPortSettings->signature          = PAR12_SIGNATURE;
       pPortSettings->ulVersion          = 1;
       pPortSettings->flStatus           = pPortInst->flStatus ;
       pPortSettings->flBidiCapabilities = pPortInst->flBidiCapabilities ;
       pPortSettings->flBidiProtocol     = pPortInst->flBidiProtocol ;
       pPortSettings->flJob              = pPortInst->flJob ;
       pPortSettings->flDevice           = pPortInst->flDevice ;
       pPortSettings->ulModeSelected     = pPortInst->ulModeSelected ;
       pPortSettings->ulCurrentMode      = pPortInst->ulCurrentMode ;
       pPortSettings->ulPrintTimeOut     = pPortInst->ulPrintTimeOut ;
       pPortSettings->ulNoQueryTimeOut   = pPortInst->ulNoQueryTimeOut ;
       pPortSettings->ulNoJobTimeOut     = pPortInst->ulNoJobTimeOut ;
       memcpy( &pPortSettings->TimeOut , &pPortInst->TimeOut, sizeof(PPTIMEOUTCHANNEL) );

       pszTemp = (PSZ)((PBYTE)pPortSettings + sizeof(PORTSETTINGS));

       strcpy( pszTemp, pszPortName );
       pPortSettings->ulpszPortName = (ULONG)pszTemp;
       pszTemp += strlen(pszTemp) + 1;

       strcpy( pszTemp, pPortInst->szDeviceID );
       pPortSettings->ulpszDeviceID = (ULONG)pszTemp;
       pszTemp += strlen(pszTemp) + 1;

       /*
       ** Get port's share access flag (part of initialization string)
       */
       strcpy (chAppName, APPNAME_LEAD_STR);
       strcat (chAppName, pszPortName);
       if ( (PrfQueryProfileString (HINI_SYSTEMPROFILE, chAppName,
                                    KEY_INITIALIZATION, NULL,
                                   (PSZ)szBuf,
                                   20))
             && (szBuf[0] == '1') ) {
           pPortSettings->fShareAccess    = TRUE ;
       } else {
           pPortSettings->fShareAccess    = FALSE;
       }

       /*
        * Change pointers to offsets
        */
       PTRTOOFFSET(ulpszPortName,pPortSettings);
       PTRTOOFFSET(ulpszDeviceID,pPortSettings);
    }
   LeavePdrSem();

    /*
     * Set size of return buffer
     */
    *pcbPortSettings = cbNeeded;

    if (!(flChangeDebugLog & FL_PDRDB_DISABLE_QPORTDRV))
    {
       DBPRINTF ((logbuf, "HandleQPortDRV rc=%d(%s) cbNeeded=%d", rc, pszPortName, cbNeeded));
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = HandleSetPortDRV
 *
 * DESCRIPTION   = Process BIDI_SET_PORTDRV command to save port driver
 *                 settings data structure for the given port.
 *
 * INPUT         = pszPortName   - Name of port
 *                 pPortSettings  -> buffer containing port settings
 *                 cbPortSettings -> length of buffer(on entry)
 *
 * OUTPUT        = 0   - Success, INI entries and instance data updated
 *                 Other - failed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *                 Called by:
 *                  1) SplPdSet(BIDI_SET_PORTDRV) under bidi spooler process
 *                  2) SavePortDlgSettings() if bidi spooler not installed or
 *                     spooler not enabled.
 *
 ****************************************************************************/

ULONG HandleSetPortDRV ( PSZ           pszPortName,
                         PPORTSETTINGS pPortSettings,
                         ULONG         cbPortSettings )
{
    ULONG        rc;
    PPORTINST    pPortInst;
    CHAR         chBuf[CCHMAXPATH];
    CHAR         chAppName[CCHMAXPATH];
    PSZ          pszInitialization ;
    HFILE        hFile ;
    ULONG        ActionTaken ;
    USHORT       usParamPacket= 0;
    USHORT       usDataPacket = 0;
    ULONG        ulReturned1;
    ULONG        ulReturned2;
    BOOL         fReInit; /* TRUE when settings change requires Port Init */

    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif



    if (!(flChangeDebugLog & FL_PDRDB_DISABLE_QPORTDRV))
    {
       DBPRINTF ((logbuf, "HandleSetPortDRV called for %s", pszPortName));
    }

    rc      = 0;
    fReInit = FALSE;

    if ( (pPortSettings->signature != PAR12_SIGNATURE) ||
         (pPortSettings->ulVersion != 1) )
    {
       return(ERROR_INVALID_PARAMETER);
    }

   EnterPdrSem();
    pPortInst = AddPortInst ( pszPortName );
    if (pPortInst )
    {
        pPortInst->ulPrintTimeOut = pPortSettings->ulPrintTimeOut ;
        pPortInst->ulNoQueryTimeOut = pPortSettings->ulNoQueryTimeOut ;
        pPortInst->ulNoJobTimeOut = pPortSettings->ulNoJobTimeOut ;
        /*
        ** Check for changes that require us to re-query device:  @BUGBUG
        ** example is ulModeSelected changing or low-level timeouts
        **    being altered.
        */
        if (pPortInst->ulModeSelected != pPortSettings->ulModeSelected)
        {
           pPortInst->ulModeSelected = pPortSettings->ulModeSelected ;
           fReInit = TRUE;
        }
        if (memcmp( &pPortInst->TimeOut , &pPortSettings->TimeOut, sizeof(PPTIMEOUTCHANNEL)))
        {
           memcpy( &pPortInst->TimeOut , &pPortSettings->TimeOut, sizeof(PPTIMEOUTCHANNEL) );
           fReInit = TRUE;
        }
    } else {
        rc = ERROR_FILE_NOT_FOUND;
    }
   LeavePdrSem();
    if (rc)
    {
      if (!(flChangeDebugLog & FL_PDRDB_DISABLE_QPORTDRV))
      {
         DBPRINTF ((logbuf, "HandleSetPortDRV failed rc=%d(%s)", rc, pszPortName));
      }
      return(rc);
    }
    /*
    ** Set port data in INI file
    */
    strcpy (chAppName, APPNAME_LEAD_STR);
    strcat (chAppName, pszPortName);

    _itoa (pPortSettings->ulPrintTimeOut, chBuf, 10);
    strcat (chBuf, ";");
    if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                               chAppName,
                               KEY_TIMEOUT_PRINT,
                               chBuf))
    {
       DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI TimeOut"));
    }

    _itoa (pPortSettings->ulNoQueryTimeOut, chBuf, 10);
    strcat (chBuf, ";");
    if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                               chAppName,
                               KEY_TIMEOUT_QUERY,
                               chBuf))
    {
       DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI TimeOut"));
    }

    _itoa (pPortSettings->ulNoJobTimeOut, chBuf, 10);
    strcat (chBuf, ";");
    if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                               chAppName,
                               KEY_TIMEOUT_JOB,
                               chBuf))
    {
       DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI TimeOut"));
    }

    if (fReInit)
    {
       /*
       ** Either the selected mode or the read/write timeouts
       **   were changed.
       */
       _itoa (pPortSettings->ulModeSelected, chBuf, 10);
       strcat (chBuf, ";");
       if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                  chAppName,
                                  KEY_MODE_SELECTED,
                                  chBuf))
       {
          DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI R/W TimeOut %s=%s", KEY_TIMEOUT_READ_INTER, chBuf));
       }

       _itoa (pPortSettings->TimeOut.ulReadIntTimeOut, chBuf, 10);
       strcat (chBuf, ";");
       if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                  chAppName,
                                  KEY_TIMEOUT_READ_INTER,
                                  chBuf))
       {
          DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI R/W TimeOut %s=%s", KEY_TIMEOUT_READ_INTER, chBuf));
       }
       _itoa (pPortSettings->TimeOut.ulReadIdleTimeOut, chBuf, 10);
       strcat (chBuf, ";");
       if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                  chAppName,
                                  KEY_TIMEOUT_READ_ELAPSED,
                                  chBuf))
       {
          DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI R/W TimeOut %s %s", KEY_TIMEOUT_READ_ELAPSED, chBuf));
       }
       _itoa (pPortSettings->TimeOut.ulWriteIntTimeOut, chBuf, 10);
       strcat (chBuf, ";");
       if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                  chAppName,
                                  KEY_TIMEOUT_WRITE_INTER,
                                  chBuf))
       {
          DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI R/W TimeOut %s %s", KEY_TIMEOUT_WRITE_INTER, chBuf));
       }
       _itoa (pPortSettings->TimeOut.ulWriteIdleTimeOut, chBuf, 10);
       strcat (chBuf, ";");
       if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                  chAppName,
                                  KEY_TIMEOUT_WRITE_ELAPSED,
                                  chBuf))
       {
          DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI R/W TimeOut %s %s", KEY_TIMEOUT_WRITE_ELAPSED, chBuf));
       }
    }


    if (pPortSettings->fShareAccess != pPortInst->fShareAccess) {

       if (pPortSettings->fShareAccess)
          pszInitialization = "1;" ;
       else
          pszInitialization = "0;" ;

       pPortInst->fShareAccess = pPortSettings->fShareAccess;

          /*
          ** If user selects to change parallel port access mode,
          **   we must issue IOCtl to PRINT02.SYS to set the
          **   share mode now. (Open port direct, NONSPOOLED!).
          **   It could be possible the user will use programs
          **   that do not print, but still require the parallel
          **   port's hardware registers
          */
       if ( DosOpen(pszPortName,                  /* port "LPT1"          */
                    &hFile,
                    &ActionTaken,
                    0L,                           /* filesize             */
                    0L,                           /* attribute            */
                    FILE_OPEN|FILE_CREATE,        /* open flags           */

                    OPEN_FLAGS_NONSPOOLED |       /* open mode            */
                    OPEN_FLAGS_FAIL_ON_ERROR |
                    OPEN_ACCESS_WRITEONLY |
                    OPEN_SHARE_DENYNONE,
                    NULL ) == 0)                  /* No EAs               */
       {
             /*
             ** flag byte: bit 0 off -> don't allow more than one process
             **                         to access parallel port at one time
             **                  on  -> allow multiple processes to access
             **                         parallel port simultaneously
             */
          usDataPacket = (USHORT)pPortSettings->fShareAccess ;

          ulReturned1 = 1 ;             /* length of in/output parameters */
          ulReturned2 = 1 ;

          DosDevIOCtl(
                 hFile,                    /* port file handle            */
                 0x5,                      /* Category number             */
                 0x51,                     /* Function - set share access */
                 (PVOID) &usParamPacket,   /* Not needed for IOCTL        */
                 sizeof (BYTE),
                 &ulReturned1,
                 (PVOID) &usDataPacket,    /* 1 = allow multiple access   */
                 sizeof (BYTE),
                 &ulReturned2);

          DosClose(hFile) ;
       }
          /*
          ** Write initialization string to ini file
          */
       if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                   chAppName,
                                   KEY_INITIALIZATION,
                                   pszInitialization))
       {
          DBPRINTF ((logbuf, "HandleSetPortDRV failed setting INI ShareAccess"));
          rc = PMERR_INI_WRITE_FAIL;
       }
    }

    if (fReInit)
    {
        /*
        ** Must re-initialize the printer using new mode/timeouts.
        */
        ulReturned1 = sizeof(chBuf);
        ulReturned2 = PrtQuery( NULL,
                                pszPortName,
                                TYPE_LONG_WAIT,
                                BIDI_Q_PORT,
                                NULL,
                                0,
                                chBuf,
                               &ulReturned1 );

    }
    if (!(flChangeDebugLog & FL_PDRDB_DISABLE_QPORTDRV))
    {
       DBPRINTF ((logbuf, "HandleSetPortDRV rc=%d(%s)", rc, pszPortName));
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = StartPassthru - INTERNAL API
 *
 * DESCRIPTION   = Process BIDI_START_PASSTHRU command
 *                 This allocates a buffer to start copying any data read for
 *                 the passthru session owner.
 *
 * INPUT         = pszDeviceName   -  name of port printer is attached
 *                 ulFlags         -  query options
 *                 ulCommand       -  function code BIDI_START_PASSTHRU
 *                 pulPassthruWait -> #secs to retry if another app has passthru
 *                 cbInData        -  length in bytes of pInData(must be 4)
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG StartPassthru( PSZ    pszDeviceName,
                       ULONG  ulFlags,
                       ULONG  ulCommand,
                       PULONG pulPassthruWait,
                       ULONG  cbInData )
{
    ULONG         rc;
    PPORTINST     pPortInst;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    /*
     * Find port instance
     */
    rc             = 0;

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "StartPassthru called for %s\r\n",
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */

   EnterPdrSem();
    pPortInst = FindPortInst ( pszDeviceName );
    if (!pPortInst ) {
        rc = ERROR_FILE_NOT_FOUND;
    } else if (pPortInst->pPassthruBuf) {
        rc = ERROR_TIMEOUT;
    } else {
        rc = DosAllocMem((PVOID)&pPortInst->pPassthruBuf,
                          DEFAULT_BUFSIZE,
                          PAG_READ|PAG_WRITE|PAG_COMMIT);
    }
    if (rc == 0) {
       pPortInst->ulcbPassthruBuf  = DEFAULT_BUFSIZE;
       pPortInst->ulcbPassthruData = 0;
    }
   LeavePdrSem();

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "StartPassthru rc=%d for %s\r\n",
                 rc,
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */
    return(rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = SendPassthru - INTERNAL API
 *
 * DESCRIPTION   = Process BIDI_SEND_PASSTHRU command
 *                 This sends data to the printer.
 *                 If we are in the middle of sending a job to the printer
 *                 then we fail this call.
 *
 * INPUT         = pszDeviceName   -  name of port printer is attached
 *                 ulFlags         -  query options
 *                 ulCommand       -  function code for query
 *                 pBuf            -> buffer to send to printer
 *                 cbInData        -  length in bytes of pBuf
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG SendPassthru( PSZ    pszDeviceName,
                       ULONG  ulFlags,
                       ULONG  ulCommand,
                       PBYTE  pBuf,
                       ULONG  cbInData )
{
    ULONG         rc;
    PPORTINST     pPortInst;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    /*
     * Find port instance
     */
    rc             = 0;

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "SendPassthru called for %s\r\n",
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */

   EnterPdrSem();
    pPortInst = FindPortInst ( pszDeviceName );
    if (!pPortInst ) {
        rc = ERROR_FILE_NOT_FOUND;
    } else {
       /*
        * Here we check for whether a job is printing or not
        *   because this port driver's passthru commands will not
        *   be allowed in the middle of sending a print job.
        */
       if (pPortInst->ulJobPrinting == PDR_PRINTING ) {
          //
          // Since there is only one channel to the printer
          //  we cannot send commands while in the middle of sending
          //  a print job
          //
          rc = ERROR_INFO_NOT_AVAIL;
       }
    }
   LeavePdrSem();
    if (rc == 0) {
       /*
        * Use our existing routine to write the data
        */
       rc = SplPdSendCmd( pszDeviceName,
                          ulFlags,
                          ulCommand,
                          pBuf,
                          cbInData );
    }
    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "SendPassthru rc=%d for %s\r\n",
                 rc,
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */

    return(rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = EndPassthru - INTERNAL API
 *
 * DESCRIPTION   = Process BIDI_END_PASSTHRU command
 *                 This frees the passthru session buffer.
 *
 * INPUT         = pszDeviceName   -  name of port printer is attached
 *                 ulFlags         -  query options
 *                 ulCommand       -  function code BIDI_END_PASSTHRU
 *                 pInData         -> not used
 *                 cbInData        -  length in bytes of pInData
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG EndPassthru( PSZ    pszDeviceName,
                   ULONG  ulFlags,
                   ULONG  ulCommand,
                   PULONG pulPassthruWait,
                   ULONG  cbInData )
{
    ULONG         rc;
    PPORTINST     pPortInst;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    /*
     * Find port instance
     */
    rc             = 0;

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "EndPassthru called for %s\r\n",
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */
    if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
    {
       DBPRINTF ((logbuf, "EndPassthru called for %s", pszDeviceName));
    }

   EnterPdrSem();
    pPortInst = FindPortInst ( pszDeviceName );
    if (!pPortInst ) {
        rc = ERROR_FILE_NOT_FOUND;
    } else if (!pPortInst->pPassthruBuf) {
        rc = PMERR_SPL_INV_HSPL;
    } else {
        rc = DosFreeMem((PVOID)pPortInst->pPassthruBuf );
        pPortInst->pPassthruBuf     = NULL;
        pPortInst->ulcbPassthruBuf  = 0;
        pPortInst->ulcbPassthruData = 0;
    }
   LeavePdrSem();

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "EndPassthru rc=%d for %s\r\n",
                 rc,
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */

    return(rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = ReadPassthru - INTERNAL API
 *
 * DESCRIPTION   = Process BIDI_READ_PASSTHRU command
 *                 This returns a buffer of data from the print to the caller.
 *
 * INPUT         = pszDeviceName   -  name of port printer is attached
 *                 ulFlags         -  query options
 *                 ulCommand       -  function code BIDI_READ_PASSTHRU
 *                 pInData         - command specific input data
 *                 cbInData        - length in bytes of pInData
 *                 pOutBuf         - gets data read from printer
 *                 pcbOutData      - Points to length of pOutData(in bytes)
 *                                   On entry this is set to length of buffer
 *                                    passed in.
 *                                   On exit this is updated with the length
 *                                    of available data, which may be more
 *                                    than put into pOutData
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG ReadPassthru( PSZ    pszDeviceName,
                       ULONG  ulFlags,
                       ULONG  ulCommand,
                       PVOID  pInData,
                       ULONG  cbInData,
                       PBYTE  pOutBuf,
                       PULONG pcbOutBuf )
{
    ULONG         rc;
    ULONG         cbToCopy;
    ULONG         ulcbData;
    PBYTE         pbTemp;
    PPORTINST     pPortInst;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    /*
     * Find port instance
     */
    rc             = 0;

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "ReadPassthru called for %s\r\n",
                 pszDeviceName );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */

    if ( !ChkMem( pcbOutBuf, sizeof(ULONG), CHK_WRITE) ||
         !ChkMem( pOutBuf, *pcbOutBuf, CHK_WRITE) ) {
       rc = ERROR_INVALID_PARAMETER;
       #ifdef DEBUG_ALERT
        if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
        {
           sprintf( logbuf,
                    "ReadPassthru invalid pOutBuf(%lX) or pcbOutBuf(%lX)\r\n",
                    (ULONG)pOutBuf, (ULONG)pcbOutBuf );
           LogCall( logbuf );
        }
       #endif /* DEBUG_ALERT */
       return(rc);
    }

   EnterPdrSem();
    pPortInst = FindPortInst ( pszDeviceName );
    if (!pPortInst ) {
        rc = ERROR_FILE_NOT_FOUND;
    } else if (!pPortInst->pPassthruBuf) {
        rc = PMERR_SPL_INV_HSPL;
    } else if (!pPortInst->ulcbPassthruData) {
        rc = ERROR_INFO_NOT_AVAIL;

    } else {
        ulcbData = pPortInst->ulcbPassthruData;
        if (ulcbData > *pcbOutBuf) {
           rc = ERROR_MORE_DATA;
           cbToCopy = *pcbOutBuf;
        } else {
           cbToCopy = pPortInst->ulcbPassthruData;
        }
        *pcbOutBuf = pPortInst->ulcbPassthruData;
        memcpy( pOutBuf, pPortInst->pPassthruBuf, cbToCopy );
        pbTemp = pPortInst->pPassthruBuf + cbToCopy;
        if (cbToCopy < pPortInst->ulcbPassthruData) {
           pPortInst->ulcbPassthruData -= cbToCopy;
           memcpy( pPortInst->pPassthruBuf, pbTemp, pPortInst->ulcbPassthruData);
        } else {
           pPortInst->ulcbPassthruData = 0;
        }
    }
   LeavePdrSem();

    #ifdef DEBUG_ALERT
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_PASSTHRU))
     {
        sprintf( logbuf,
                 "ReadPassthru rc=%d for %s cbRead=%d\r\n",
                 rc,
                 pszDeviceName,
                 cbToCopy
                 );
        LogCall( logbuf );
     }
    #endif /* DEBUG_ALERT */

    return(rc);
}



/****************************************************************************
 *
 * FUNCTION NAME = CreateControlThread - INTERNAL API
 *
 * DESCRIPTION   = Create thread to monitor IR port connections
 *
 * INPUT         = None
 *
 * OUTPUT        = tid   - new thread ID, 0 if failed to create thread
 *
 * NOTE          = Only called once when BIDI_INIT_PORTDRV command recieved
 *                 Not in pdrsem on entry/exit
 *
 ****************************************************************************/

TID   CreateControlThread( VOID )
{
   TID   tid;
   ULONG rc;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif


   tid = 0;

   rc = DosCreateThread( &tid,
                         (PFNTHREAD)ControlThread,
                         0,
                         0,                         // Execution Flag
                         DEFAULT_STACKSIZE);
   if (!(flChangeDebugLog & FL_PDRDB_DISABLE_CONTROLTHRD))
   {
      DBPRINTF ((logbuf, "CreateControlThread rc=%d tidControlThread=%d",rc, tid));
   }
   /*
   ** Keep global thread ID for debugging
   */
   tidControlThread = tid;

   return tid;
}


/****************************************************************************
 *
 * FUNCTION NAME = ControlThread - INTERNAL API
 *
 * DESCRIPTION   = Thread to monitor and control all ports.
 *                 Only 1 of these threads will exist.
 *
 * INPUT         = None
 *
 * OUTPUT        = None, exits when port driver shutdown by
 *                 the BIDI_SHUTDOWN command
 *
 ****************************************************************************/

VOID EXPENTRY ControlThread( VOID )
{
    ULONG       rc;
    ULONG       ulPostCount;
    ULONG       ulWakeupTime;     /* Time(in secs) to wake up again */
    ULONG       ulWaitTime;       /* #msecs to wait for EventSem    */
    ULONG       ulNextTimeOut;    /* Largest of Job or Query time to wakeup */
    ULONG       ulCurrTime;       /* Current time in seconds */
    ULONG       ulCheckQueryTime; /* Time to close port based on last query */
    ULONG       ulCheckJobTime;   /* Time to close port based on last job   */
    PPORTINST   pPortInst;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    /*
    ** Create non-shared event sem for this thread
    */
    rc = DosCreateEventSem((PSZ)NULL,&hevControl,
                           0, FALSE);
    do
    {
       rc = DosResetEventSem(hevControl, &ulPostCount);
      EnterPdrSem();
       ulCurrTime  = time();
       ulWakeupTime = SEM_INDEFINITE_WAIT;
       for (pPortInst = pPortInstList; pPortInst; pPortInst = pPortInst->pNext)
       {
          /*
          ** Search through list of ports
          ** If any need to be closed, close them now.
          ** If ulNoQueryTimeOut is set to 0, never close the port.
          */
          if ( (pPortInst->flStatus & PF_PORT_OPEN) &&
               !(pPortInst->flStatus & PF_OPEN_IN_PROGRESS) &&
               (pPortInst->ulJobPrinting != PDR_PRINTING) &&
               (pPortInst->ulNoQueryTimeOut) )
          {

             ulCheckQueryTime = pPortInst->ulTimeLastCmd + pPortInst->ulNoQueryTimeOut;
             ulCheckJobTime   = pPortInst->ulTimeLastJob + pPortInst->ulNoJobTimeOut;
             if ( (ulCurrTime > ulCheckQueryTime) &&
                  (ulCurrTime > ulCheckJobTime) &&
                  pPortInst->hFile )
             {
                /*
                ** Our timeout for printing a job and waiting for
                **   the next query has expired.
                ** We will close the connection, and the next time
                **   we get a query or job in, we will generate a
                **   CommStatChange alert to allow the protocol converter
                **   to re-enable alerts with the printer.
                ** Note that if a job confirmation does not come before
                **   we close the connection, the spooler will keep the
                **   job as "In Printer" status.
                */
                if (!(flChangeDebugLog & FL_PDRDB_DISABLE_CONTROLTHRD))
                {
                   DBPRINTF ((logbuf, "ControlThread closing port not in use TimeLastJob=%d TimeLastQuery=%d", pPortInst->ulTimeLastJob, pPortInst->ulTimeLastCmd));
                }
                PdClose(pPortInst);
             } else {
                /*
                ** Now determine how long we should wait until the next
                **   time we check.
                ** If a port is printing a job
                **   we wait indefinitely because SplPdClose will wake
                **   this thread up.
                ** If any port is not printing a job
                **   we will check for the next time we should close the port.
                ** If we are not waiting indefinitely
                **   set fControlWakeupPending so that SplPdSendCmd does not
                **   keep waking up the control thread.
                */
                if ( ulCheckQueryTime > ulCheckJobTime )
                   ulNextTimeOut = ulCheckQueryTime;
                else
                   ulNextTimeOut = ulCheckJobTime;

                if (ulNextTimeOut < ulWakeupTime)
                   ulWakeupTime = ulNextTimeOut;
             }
          }

       }
       /*
       ** We have checked all the ports.
       ** Now determine how long we should wait until the next time we check.
       ** If a port is printing a job
       **   we wait indefinitely because SplPdClose will wake this thread up.
       ** If any port is not printing a job
       **   we will check for the next time we should close the port.
       ** If we are not waiting indefinitely
       **   set fControlWakeupPending so that SplPdSendCmd does not keep waking
       **   up the control thread and convert ulWakeupTime from seconds
       **   to milliseconds.
       */
       if (ulWakeupTime == (ULONG)SEM_INDEFINITE_WAIT)
       {
          fControlWakeupPending = FALSE;
          ulWaitTime = SEM_INDEFINITE_WAIT;
       } else {
          fControlWakeupPending = TRUE;
          ulWakeupTime -= ulCurrTime ;
          ulWaitTime    = ulWakeupTime * 1000;
          /*
          ** If waitTime is 0, wait at least 1/4 second
          */
          if (!ulWaitTime) {
             ulWaitTime = 250;
          }
       }
      LeavePdrSem();
       if (!fShutdownInProgress)
       {
          if (!(flChangeDebugLog & FL_PDRDB_DISABLE_CONTROLTHRD))
          {
             DBPRINTF ((logbuf, "ControlThread about to Wait %d ms", ulWaitTime));
          }
          rc = DosWaitEventSem ( hevControl, ulWaitTime );
       }
    } while ( !fShutdownInProgress );
}


/****************************************************************************
 *
 * FUNCTION NAME = GetDeviceID - INTERNAL API
 *
 * DESCRIPTION   = Set parallel port into bidi mode and query its deviceID.
 *
 * INPUT         = pPortInst -> LPT port instance to update with deviceID
 *
 * OUTPUT          0     = pPortInst updated
 *                 other = unable to get deviceID
 *
 * NOTE            Must be in port driver semaphore on entry/exit,
 *                 but leaves Sem and re-enters it.
 *
 ****************************************************************************/

ULONG GetDeviceID( PPORTINST pPortInst )
{
  ULONG                    rc;
  HFILE                    hFile;
  UCHAR                    uchParm;
  USHORT                   usComMode;
  ULONG                    flStatus;     /* port status flags on entry */
  ULONG                    cbParmLen;
  ULONG                    cbDataLen;
  ULONG                    cbID;
  ULONG                    cRetry;
  PPTIMEOUTCHANNEL         PPTimeoutChannel;
  CHAR                     chBuf[STR_LEN_PORTNAME];
  CHAR                     chInitVal[STR_LEN_PORTNAME];
  USHORT                   usDataPacket = 0;
  ULONG                    ulReturned;
  BOOL                     fResetReadTimeout;
  PSZ                      pszDeviceID;
  PBYTE                    pDevIDBuf;
  CHAR                     szByteDevID[84];

  #ifdef DEBUG_ALERT
    char logbuf[360];
  #endif

   //
   // Allocate buffer to store variable-length device ID data
   //
   rc = DosAllocMem((PVOID)&pDevIDBuf,
                     DEFAULT_BUFSIZE,
                     PAG_READ|PAG_WRITE|PAG_COMMIT);
   if (rc) {
      DBPRINTF ((logbuf, "GetDeviceID DosAllocMem failed rc=%d", rc));
      return(rc);
   }

   hFile = pPortInst->hFile;
   //
   // Avoid sending COMM_STAT_CHANGE alert when the spooler is
   //   enabled by setting flag when first time we attempt
   //   to put printer in bidi mode.
   // If flag was not set and we received a new deviceID
   //   then do not post alert to spooler.
   //
   flStatus = pPortInst->flStatus;
   pPortInst->flStatus |= PF_SETMODE_CALLED;

  LeavePdrSem();
   fResetReadTimeout = FALSE;

   /*
   ** Check pPortInst->ulModeSelected to ensure bidi has not been
   **    disabled for this port.
   ** If bidi not disabled then put printer into nibble mode
   **    to get its deviceID.  Some printers can be put into
   **    byte or ECP mode but still only return a deviceID
   **    when in nibble mode.
   **/
   usComMode = 0;
   if (pPortInst->ulModeSelected == PARMODE_DISABLE_BIDI) {
      rc = SetMode( hFile, PARMODE_DISABLE_BIDI, &usComMode );
   } else {
      rc = SetMode( hFile, PARMODE_NIBBLE, &usComMode );
      /*
      ** If unable to put printer into nibble mode,
      **  then set printer into compatible mode
      */
      if (rc) {
         SetMode( hFile, PARMODE_DISABLE_BIDI, &usComMode );
      }
   }

   /*
    * Check returned ComMode, it must be > 1 for bidi nibble/byte/ECP mode
    */
   if (usComMode > CURMODE_COMPATIBLE) {
      //
      // Set timeout and command channel  @BUGBUG
      // Later we will have to set the ECP command channel for non-data writes
      //
      // ulReadIdleTimeOut may have to be temporarily increased to allow getting
      //   the entire 1284 device ID string back.  Some printers take a few
      //   seconds to return the entire string.  After getting the device ID
      //   we will set the ulReadIdleTimeOut value smaller.
      //
      uchParm   = 0;
      cbParmLen = 1;
      PPTimeoutChannel.ulReadIdleTimeOut   = pPortInst->TimeOut.ulReadIdleTimeOut;
      PPTimeoutChannel.ulReadIntTimeOut    = pPortInst->TimeOut.ulReadIntTimeOut;
      PPTimeoutChannel.ulWriteIdleTimeOut  = pPortInst->TimeOut.ulWriteIdleTimeOut;
      PPTimeoutChannel.ulWriteIntTimeOut   = pPortInst->TimeOut.ulWriteIntTimeOut;
      PPTimeoutChannel.ulLogChannel  = 1;
      cbDataLen = sizeof( PPTimeoutChannel );
      rc = DosDevIOCtl(
                        hFile,
                        5L,
                        0x53L,
                        &uchParm,
                        sizeof(uchParm),
                        &cbParmLen,
                        &PPTimeoutChannel,
                        sizeof(PPTimeoutChannel),
                        &cbDataLen);

      #ifdef DEBUG_ALERT
       if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
       {
          DBPRINTF ((logbuf, "GetDeviceID SetTimeOutChannel before getting ID rc=%d Mode=%d", rc, usComMode));
       }

       //
       // Get timeout and command channel
       //
       uchParm   = 0;
       cbParmLen = 1;
       PPTimeoutChannel.ulReadIdleTimeOut   = 0;
       PPTimeoutChannel.ulReadIntTimeOut    = 0;
       PPTimeoutChannel.ulWriteIdleTimeOut  = 0;
       PPTimeoutChannel.ulWriteIntTimeOut   = 0;
       PPTimeoutChannel.ulLogChannel        = 0;
       cbDataLen = sizeof( PPTimeoutChannel );
       rc = DosDevIOCtl(
                         hFile,
                         5L,
                         0x73L,
                         &uchParm,
                         sizeof(uchParm),
                         &cbParmLen,
                         &PPTimeoutChannel,
                         sizeof(PPTimeoutChannel),
                         &cbDataLen);
       if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
       {
          DBPRINTF ((logbuf, "GetDeviceID GetTimeOutChannel rc=%d ReadIdle=%d ReadInt=%d WriteIdle=%d WriteInt=%d LogChannel=%d",
                     rc, PPTimeoutChannel.ulReadIdleTimeOut, PPTimeoutChannel.ulReadIntTimeOut,
                     PPTimeoutChannel.ulWriteIdleTimeOut, PPTimeoutChannel.ulWriteIntTimeOut,
                     PPTimeoutChannel.ulLogChannel));
       }
      #endif /* DEBUG_ALERT */
      //
      // Get DeviceID
      //
      cRetry = 0;
      cbID   = 0;
      do
      {
        uchParm   = 0;
        cbParmLen = 1;
        pDevIDBuf[0] = 0;
        pDevIDBuf[1] = 0;
        pDevIDBuf[2] = 0;
        if (cbID || cRetry) {
           //
           // cbID is set only if we are retrying because the entire deviceID
           //   string was not returned.  We will temporarily increase
           //   ulReadIdleTimeOut to the following values:
           //     1  second  - initial value, works for some printers
           //     5  seconds - for 2nd try(cRetry = 1)
           //    20  seconds - for 3rd try
           //    45  seconds - for 4th and last try
           //
           //   the entire 1284 device ID string back.  Some printers take a few
           //   seconds to return the entire string.  After getting the device ID
           //   we will set the ulReadIdleTimeOut value smaller.
           //
           fResetReadTimeout = TRUE;
           uchParm   = 0;
           cbParmLen = 1;
           PPTimeoutChannel.ulReadIdleTimeOut   = pPortInst->TimeOut.ulReadIdleTimeOut * ( (5*cRetry) + (10*(cRetry-1)) ) ;
           PPTimeoutChannel.ulReadIntTimeOut    = pPortInst->TimeOut.ulReadIntTimeOut;
           PPTimeoutChannel.ulWriteIdleTimeOut  = pPortInst->TimeOut.ulWriteIdleTimeOut;
           PPTimeoutChannel.ulWriteIntTimeOut   = pPortInst->TimeOut.ulWriteIntTimeOut;
           PPTimeoutChannel.ulLogChannel  = 1;
           cbDataLen = sizeof( PPTimeoutChannel );
           rc = DosDevIOCtl(
                             hFile,
                             5L,
                             0x53L,
                             &uchParm,
                             sizeof(uchParm),
                             &cbParmLen,
                             &PPTimeoutChannel,
                             sizeof(PPTimeoutChannel),
                             &cbDataLen);
           if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
           {
              DBPRINTF ((logbuf, "GetDeviceID SetTimeOutChannel to longer IdleReadTimeout=%d rc=%d Mode=%d", PPTimeoutChannel.ulReadIdleTimeOut, rc, usComMode));
           }
        }
        //
        // Set buffersize to entire length of device ID string if known.
        // This will make par1284.sys return the device ID string quicker by
        //   returning when cbID bytes are transferred without waiting
        //   the IdleReadTimeout value.
        //
        if (cbID) {
           cbDataLen = cbID;
        } else {
           cbDataLen = DEFAULT_BUFSIZE;
        }
        //
        // Initialize return buffer so we always get correct
        //   returned string length.
        // We only need to initialize the first 3 bytes to 0
        //
        memset( pDevIDBuf, 0, 3 );
        //
        // Issue Get DeviceID IOCTL
        //
        rc = DosDevIOCtl(
                          hFile,
                          5L,
                          0x74L,
                          &uchParm,
                          sizeof(uchParm),
                          &cbParmLen,
                          pDevIDBuf,
                          cbDataLen,
                          &cbDataLen);

        cbID = pDevIDBuf[1] + 256 * pDevIDBuf[0];
        if ( (rc == 0) && (cbID < DEFAULT_BUFSIZE) )
        {
           pDevIDBuf[cbID] = '\0';
        } else {
           pDevIDBuf[2] = '\0';
        }
        //
        // Store the length of the returned deviceID string
        //
        cbDataLen = strlen( pDevIDBuf+2 );
        #ifdef DEBUG_ALERT
         {
          if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
          {
             CHAR  c;
             //
             // Truncate deviceid logged at 300 bytes
             //
             c = pDevIDBuf[300];
             pDevIDBuf[300] = '\0';
             sprintf( logbuf,
                      "GetDeviceID GetDeviceID rc=%d Len=%d strlen=%d ID=%s\r\n",
                      rc, cbID, cbDataLen, pDevIDBuf+2 );
             LogCall( logbuf );
             pDevIDBuf[300] = c;
          }
         }
        #endif /* DEBUG */
        //
        // Truncate stored device ID string at 511 bytes  @BUGBUG
        //  so it will fit in our PORTINST structure
        //
        pDevIDBuf[511] = '\0';

        /*
        ** Sometimes the first query of deviceID fails to return
        **   the entire string.  We will retry here.
        ** The length prefix must equal the returned string length.
        ** We only retry 3 times as described above.
        */
        if ( !rc && ( (cbDataLen+2) < cbID ) ) {
           if (cRetry < 3) {
              cRetry++;
              DosSleep( 500 );
           } else {
              cRetry = 0;
           }
        } else {
           cRetry = 0;
        }

      } while ( cRetry );


      if (fResetReadTimeout) {
         //
         // Reset read timeout to smaller value when printing if we changed it.
         // For now just hardcode the values, later give UI to let user set them.
         // Later we will have to set the ECP command channel for non-data writes
         //
         // The ulReadIdleTimeOut was temporarily increased to allow getting
         //   the entire 1284 device ID string back.  Some printers take a few
         //   seconds to return the entire string.  After getting the device ID
         //   we will are setting the ulReadIdleTimeOut value smaller.
         //
         uchParm   = 0;
         cbParmLen = 1;
         PPTimeoutChannel.ulReadIdleTimeOut   = pPortInst->TimeOut.ulReadIdleTimeOut ;
         PPTimeoutChannel.ulReadIntTimeOut    = pPortInst->TimeOut.ulReadIntTimeOut;
         PPTimeoutChannel.ulWriteIdleTimeOut  = pPortInst->TimeOut.ulWriteIdleTimeOut;
         PPTimeoutChannel.ulWriteIntTimeOut   = pPortInst->TimeOut.ulWriteIntTimeOut;
         PPTimeoutChannel.ulLogChannel  = 1;
         cbDataLen = sizeof( PPTimeoutChannel );
         rc = DosDevIOCtl(
                           hFile,
                           5L,
                           0x53L,
                           &uchParm,
                           sizeof(uchParm),
                           &cbParmLen,
                           &PPTimeoutChannel,
                           sizeof(PPTimeoutChannel),
                           &cbDataLen);
         if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
         {
            DBPRINTF ((logbuf, "GetDeviceID SetTimeOutChannel after getting ID rc=%d Mode=%d", rc, usComMode));
         }
      }

      /*
      ** Put printer into best possible(or selected) mode after getting deviceID
      ** Check pPortInst->ulModeSelected to ensure bidi has not been
      **    disabled for this port, and to check if the user specified
      **    a specific mode for their printer.
      */
      if (pPortInst->ulModeSelected == PARMODE_AUTODETECT) {
         rc = SetMode( hFile, PARMODE_ECP, &usComMode );
         if (rc) {
            rc = SetMode( hFile, PARMODE_BYTE, &usComMode );
         }
         //
         // Some printers give erroneous signals that make the
         //   host think they can be put into ECP mode.
         // Some systems can be set into Byte mode but cannot
         //   accurately read byte mode.
         //
         // Verify that ECP or Byte mode can be used by querying the
         //   first 80 characters of the device ID.
         // If it does not match the first 80 chars of the
         //   device ID returned in nibble mode,
         // then ECP or Byte mode should not be used.
         //
         if (rc == 0) {
            memset( szByteDevID, 0, sizeof(szByteDevID) );
            uchParm   = 0;
            cbParmLen = 1;
            if (cbID > 80) {
               cbDataLen = 80;
            } else {
               cbDataLen = cbID;
            }
            //
            // Issue Get DeviceID IOCTL
            //
            rc = DosDevIOCtl(
                              hFile,
                              5L,
                              0x74L,
                              &uchParm,
                              sizeof(uchParm),
                              &cbParmLen,
                              szByteDevID,
                              cbDataLen,
                              &cbDataLen);
            if (rc == 0) {
               //
               // Compare up to the first 80 chars of the device ID
               //  and if different, then Byte mode will not be supported.
               //
               if (cbID > 80) {
                  cbDataLen = 80;
               } else {
                  cbDataLen = cbID;
               }
               if ( memcmp( pDevIDBuf, szByteDevID, cbDataLen) ) {
                  rc = ERROR_NOT_SUPPORTED;
               }
            }

            #ifdef DEBUG_ALERT
             {
              if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
              {
                 sprintf( logbuf,
                          "GetDeviceID ECP/ByteMode GetDevID check rc=%d Len=%d ID=%s\r\n",
                          rc, cbDataLen, szByteDevID+2 );
                 LogCall( logbuf );
              }
             }
            #endif /* DEBUG */
         }
         if (rc) {
            rc = SetMode( hFile, PARMODE_NIBBLE, &usComMode );
         }
      } else {
         /*
         ** User selected a specific bidi mode.
         ** If unable to put printer into that mode,
         **  then set printer into compatible mode
         */
         rc = SetMode( hFile, pPortInst->ulModeSelected, &usComMode );
         if (rc) {
            SetMode( hFile, PARMODE_DISABLE_BIDI, &usComMode );
         }
      }

     EnterPdrSem();
      /*
      ** Check to see if the printer deviceID has changed.
      ** Save the deviceID with the port instance.
      */
      if ( strcmp( pPortInst->szDeviceID, pDevIDBuf+2) ) {

         strcpy( pPortInst->szDeviceID, pDevIDBuf+2);
         //
         // Only post COMM_STAT_CHANGE alert if this is NOT the
         //   first time we tried to put printer into bidi mode.
         //
         if (flStatus & PF_SETMODE_CALLED) {
            pPortInst->flStatus |= PF_COMM_STAT_CHANGE;
         }
         #ifdef DEBUG_ALERT
          {
           if (!(flChangeDebugLog & FL_PDRDB_DISABLE_GETDEVICEID))
           {
              sprintf( logbuf,
                       "GetDeviceID New printer detected on %s flStatus=%lX\r\n",
                       pPortInst->pszPortName,
                       pPortInst->flStatus );
              LogCall( logbuf );
           }
          }
         #endif /* DEBUG */
      }
      /*
      ** Return that bidi printer connected and whether ECP is avail
      **   and whether the printer's current mode has bidi active.
      */
      if (usComMode >= CURMODE_ECP) {
         pPortInst->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE |
                                         PRTPORT_CAPS_BIDI_ACTIVE |
                                         PRTPORT_CAPS_CMD_CHANNEL_AVAIL;
      } else if (usComMode > CURMODE_COMPATIBLE) {
         pPortInst->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE |
                                         PRTPORT_CAPS_BIDI_ACTIVE;
      } else {
         pPortInst->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE;
      }
      /*
      ** For now, search the 1284 DeviceID for supporting
      **  a bidi protocol( NPAP, PJL, ... )
      */
      pPortInst->flBidiProtocol = 0;
      pszDeviceID = pPortInst->szDeviceID;
      if (*pszDeviceID)
      {
         if (strstr(pszDeviceID, "NPAP"))
         {
            pPortInst->flBidiProtocol |= PRTPORT_TYPE_NPAP;
         }
         if (strstr(pszDeviceID, "PJL"))
         {
            pPortInst->flBidiProtocol |= PRTPORT_TYPE_PJL;
         }
         if (strstr(pszDeviceID, "HP LaserJet") ||
            (strstr(pszDeviceID, "DeskJet")) )
         {
            pPortInst->flBidiProtocol |= PRTPORT_TYPE_PJL;
         }
      }
      /*
      ** Clear flag to indicate completion of CommStatChange alert.
      ** We are done because the bidi printer is back.
      */
      pPortInst->flStatus &= ~PF_COMALERT_POSTED;
      /*
      ** Indicate whether we have a printer in bidi mode
      */
      if (usComMode > CURMODE_COMPATIBLE) {
         pPortInst->flStatus |= (PF_BIDI_WAS_ACTIVE+PF_BIDI_CHECKED);
      } else {
         pPortInst->flStatus |= (PF_BIDI_CHECKED);
      }
      pPortInst->ulCurrentMode = usComMode;
      /*
      ** Must wake up HandleWaitAlert thread so we can read from
      **   the newly opened port.
      */
      DosPostEventSem(PdrWaitAlert.hevWaitAlert);
   } else {
      /*
      ** Unable to put printer into a bidi mode.
      ** Make Application name string( "PM_PortName" )
      */
      strcpy (chBuf, APPNAME_LEAD_STR);
      strcat (chBuf, pPortInst->pszPortName);

      /*
       * Set write timeout for PRINT01/PRINT02 ( not bidi port )
       */
      if (PrfQueryProfileString (HINI_SYSTEMPROFILE, chBuf,
                                   KEY_TIMEOUT_PRINT, NULL, chInitVal,
                                   STR_LEN_PORTNAME)) {
          usDataPacket = (USHORT)atoi (chInitVal);
          ulReturned = 2L ;
          DosDevIOCtl(
                      hFile,                    /* Hfile passed into SplPdInit */
                      0x5,                      /* Category number             */
                      0x4e,                     /* Function number             */
                      NULL,                     /* pParams - not needed        */
                      0,                        /* sizeof *pParms              */
                      NULL,                     /* &size in/out                */
                      (PVOID) &usDataPacket,    /* points to LPT timeout value */
                      sizeof (USHORT),          /* sizeof usDataPacket         */
                      &ulReturned);            /* &size in/out                */
      }

     EnterPdrSem();
      /*
      ** Test for printer previously being a bidi printer.
      ** If so, then someone turned it off(or replaced it with non-bidi printer)
      ** For now, we treat this as "printer not responding" instead of
      **  automatically going into non-bidi mode.
      ** Only post CommStatChange once, until the bidi printer come back.
      */
      if ( (pPortInst->flStatus & PF_BIDI_WAS_ACTIVE) &&
           (pPortInst->ulModeSelected != PARMODE_DISABLE_BIDI) ) {

         pPortInst->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE |
                                         PRTPORT_CAPS_NOT_RESPONDING;
         if ( !(pPortInst->flStatus & PF_COMALERT_POSTED) ) {
            pPortInst->flStatus |= PF_COMM_STAT_CHANGE;
            /*
            ** Must wake up HandleWaitAlert thread so we can return the
            **   CommStatChange alert.
            */
            DosPostEventSem(PdrWaitAlert.hevWaitAlert);
         }
      } else {

         pPortInst->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE;
      }
      pPortInst->szDeviceID[0]  = '\0';
      pPortInst->flBidiProtocol = 0;
      pPortInst->ulCurrentMode  = CURMODE_COMPATIBLE;

      rc = ERROR_BAD_COMMAND;
   }
   DosFreeMem( pDevIDBuf );

   return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = SetMode - INTERNAL API
 *
 * DESCRIPTION   = Set parallel port into bidi mode and return its current mode.
 *
 * INPUT         = hFile  -  handle to LPT port
 *                 usMode -  mode to put paralle port into
 *                           Values are:
 *                           1 - compatability mode(unidirectional)
 *                           2 - nibble mode
 *                           3 - byte mode
 *                           4 - ECP without RLE for data transfers
 *                           5 - ECP with RunLengthEncoding
 *                           6 - EPP mode
 *                 pMode  -> ushort to receive current mode(even if setmode fails)
 *
 * OUTPUT          0     = successfully set LPT port in requested mode
 *                 other = failure setting mode
 *
 * NOTE            Must not be in port driver semaphore on entry/exit
 *
 ****************************************************************************/

ULONG SetMode( HFILE hFile, USHORT usMode, PUSHORT pCurMode)
{
  ULONG     rc;
  ULONG     rc2;
  UCHAR     uchParm;
  ULONG     cbParmLen;
  ULONG     cbDataLen;

   #ifdef DEBUG_ALERT
     char logbuf[160];
   #endif

   //
   // Try to set into given mode
   //
   uchParm   = 0;
   cbParmLen = 1;
   cbDataLen = sizeof( usMode );
   rc = DosDevIOCtl(
                     hFile,
                     5L,
                     0x52L,
                     &uchParm,
                     sizeof(uchParm),
                     &cbParmLen,
                     &usMode,
                     sizeof(usMode),
                     &cbDataLen);
   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_SETMODE))
     {
        sprintf( logbuf,
                 "SetMode rc=%d RequestedMode=%d hFile=%d\r\n",
                 rc, usMode, hFile );
        LogCall( logbuf );
        //
        // Get current mode for debug version
        //
        uchParm   = 0;
        cbParmLen = 1;
        usMode    = 0;
        cbDataLen = sizeof( usMode );
        rc2 = DosDevIOCtl(
                          hFile,
                          5L,
                          0x72L,
                          &uchParm,
                          sizeof(uchParm),
                          &cbParmLen,
                          &usMode,
                          sizeof(usMode),
                          &cbDataLen);
        sprintf( logbuf,
                 "SetMode - GetCurMode rc=%d Mode=%d\r\n",
                 rc2, usMode );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */

   //
   // Par1284.sys can fail the SetMode IOCtl
   // but still return the mode wanted in QueryMode
   //
   if (rc) {
     *pCurMode = CURMODE_COMPATIBLE;
   } else {
     *pCurMode = usMode;
   }

   return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = CreateReadThread - INTERNAL API
 *
 * DESCRIPTION   = Create thread to read from an LPT port
 *
 * INPUT         = pPortInst -> port to read from
 *
 * OUTPUT        = tid   - new thread ID, 0 if failed to create thread
 *
 * NOTE          = Must be in SplSem on entry/exit
 *
 ****************************************************************************/

TID   CreateReadThread( PPORTINST pPortInst )
{
   TID   tid;
   ULONG rc;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif


   tid = 0;

   rc = DosCreateThread( &tid,
                         (PFNTHREAD)ParReadThread,
                         (ULONG)pPortInst,          // Parameter
                         0,                         // Execution Flag
                         DEFAULT_STACKSIZE);
   if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
   {
      DBPRINTF ((logbuf, "CreateReadThread rc=%d tidReadThread=%d",rc, tid));
   }

   pPortInst->tidReadThread = tid;

   return tid;
}


/****************************************************************************
 *
 * FUNCTION NAME = ParReadThread - INTERNAL API
 *
 * DESCRIPTION   = Thread to read from an LPT port
 *                 Only 1 of these threads will exist per LPT port.
 *
 * INPUT         = pPortInst -> port to read from
 *
 * OUTPUT        = None, exits when port driver shutdown by
 *                 the BIDI_SHUTDOWN command
 *
 ****************************************************************************/

VOID EXPENTRY ParReadThread( PPORTINST pPortInst )
{
    ULONG         rc;
    ULONG         rc2;
    ULONG         cbRead;
    ULONG         cbToCopy;
    ULONG         cErrors;
    PSZ           pszPort;
    HFILE         hFile;
    PWAITALERTBUF pWaitBuf;           /* Current WAITALERTBUF we are using  */
    PALERTBUF     pAlertBuf;
    PALERTBUF     pNewAlert;
    PBYTE         pBuf;
    ULONG         cbNeeded;
    ULONG         ulPostCount;
    ULONG         ulWaitTime;
    ULONG         ulLastReadTime; /* timestamp of last bytes read */
    USHORT        usComMode;
    HEV           hevReadThread;
    CHAR          szReadBuf[CCHMAXPATH];

    #ifdef DEBUG_ALERT
      char logbuf[360];
      DATETIME    TimeBefore;
      DATETIME    TimeAfter;
    #endif

   EnterPdrSem();
    if (pPortInst->signature == PT_SIGNATURE) {
       hFile = pPortInst->hFile;
       pszPort = pPortInst->pszPortName;
       hevReadThread = pPortInst->hevReadThread;
    } else {
       pPortInst = NULL;
    }
   LeavePdrSem();

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
      {
         sprintf( logbuf,
                  "ParReadThread started for %s\r\n",
                  pPortInst ? pszPort : "Invalid" );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */

    if (!pPortInst) {
       return;
    }

    cErrors = 0;
    ulLastReadTime = time();
    ulWaitTime = 0; // reset hevReadThread wait time

    do
    {
      cbRead = 0;
      #ifdef DEBUG_ALERT
       DosGetDateTime( &TimeBefore );
      #endif /* DEBUG */

      rc = DosRead( hFile, szReadBuf, sizeof(szReadBuf), &cbRead );

      #ifdef DEBUG_ALERT
       DosGetDateTime( &TimeAfter );
      #endif /* DEBUG */
      /*
      ** If error received by DosRead
      ** then assume no valid data was returned.
      */
      if (rc) {
         cbRead = 0;
      }
      #ifdef DEBUG_ALERT
       if (((flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD) && (rc || cbRead)) ||
           (flChangeDebugLog & FL_PDRDB_ENABLE_READWAIT) )
       {
          sprintf( logbuf,
                   "ParReadThread DosRead %s rc=%d cbRead=%d cErrors=%d TimeBefore=%02d.%02d TimeAfter=%02d.%02d\r\n",
                   pszPort, rc, cbRead, cErrors,
                   TimeBefore.seconds,
                   TimeBefore.hundredths,
                   TimeAfter.seconds,
                   TimeAfter.hundredths
                   );
          LogCall( logbuf );
          DumpHex( szReadBuf, cbRead );
       }
      #endif /* DEBUG */
      if (cbRead) {
         /*
         ** Must put this data onto the WaitAlertThead buffer
         */
         ulLastReadTime = time();
         ulWaitTime = 0; // reset hevReadThread wait time
        EnterPdrSem();
         /*
         ** If a passthru session is active
         ** Then copy data read in to the passthru buffer.
         */
         if (pPortInst && (pPortInst->signature == PT_SIGNATURE) &&
             pPortInst->pPassthruBuf ) {
            //
            // Copy whatever will fit into the passthru buffer
            // If we need a larger passthru buffer then later      @BUGBUG
            // allow increasing it(but limit the amount we should store).
            //
            if ( (cbRead + pPortInst->ulcbPassthruData) > pPortInst->ulcbPassthruBuf ) {
               cbToCopy = pPortInst->ulcbPassthruBuf - pPortInst->ulcbPassthruData;
            } else {
               cbToCopy = cbRead;
            }
            memcpy( pPortInst->pPassthruBuf, szReadBuf, cbToCopy );
            pPortInst->ulcbPassthruData += cbToCopy ;
         }

         pWaitBuf  = PdrWaitAlert.pCurrBuf;
         if (pWaitBuf) {
            //
            // Handle case where entire buffer won't fit @BUGBUG
            // For this sample, any alert that will not fit is tossed.
            //
            cbNeeded = pWaitBuf->cbBufUsed + cbRead + sizeof(ALERTBUF);
            if (cbNeeded <= pWaitBuf->cbBufSize)
            {
              //
              // There is room to store this alert data in the WaitAlertBuf
              //
              if (pWaitBuf->pLastStored)
              {
                 pAlertBuf = pWaitBuf->pLastStored;
                 pNewAlert = (PALERTBUF)((PBYTE)pAlertBuf +
                                                sizeof(ALERTBUF) +
                                                pAlertBuf->cbData);
              } else {
                 pNewAlert = &pWaitBuf->FirstAlertBuf;
              }
              pNewAlert->pszAlertPortName = pszPort;
              pNewAlert->cbData = cbRead ;
              pBuf = (PBYTE)pNewAlert + sizeof(ALERTBUF);
              memcpy( pBuf, szReadBuf, cbRead );
              pWaitBuf->cbBufUsed = cbNeeded;
              pWaitBuf->pLastStored = pNewAlert;
              pWaitBuf->cAlertsRemaining++;
            }
            DosPostEventSem(PdrWaitAlert.hevWaitAlert);
         }
        LeavePdrSem();

      } else {
         /*
         ** DosRead returned without data.
         ** If an error occurred, increment our error count.
         ** Just sleep for a little while until something is there
         **  to read   @BUGBUG
         */
         if (rc) {
            cErrors++;
         }
         /*
         ** If the read handle is invalid
         ** then we should terminate this read thread.
         */
         if (rc == ERROR_INVALID_HANDLE) {
           break;
         }
         /*
         ** If we have not done a successful read for 30 seconds
         **   and we are not printing a job
         ** Then check to see if the printer is still connected.
         ** We could either try SetMode() to just put it into bidi mode @BUGBUG
         **   or call GetDeviceID() to verify it is the same printer
         **   as the one we were configured for.
         */
         if ( (pPortInst->ulJobPrinting != PDR_PRINTING) &&
              (pPortInst->flStatus & PF_BIDI_WAS_ACTIVE) &&
              !(pPortInst->flStatus & PF_COMALERT_POSTED) &&
              ((time() - ulLastReadTime) > 30) ) {
            #ifdef DEBUG_ALERT
             {
              if (flChangeDebugLog & FL_PDRDB_ENABLE_READWAIT)
              {
                 sprintf( logbuf,
                          "ParReadThread about to check for changed printer LastRead=%d CurTime=%d\r\n",
                          ulLastReadTime, time() );
                 LogCall( logbuf );
              }
             }
            #endif /* DEBUG */
            rc = SetMode( pPortInst->hFile,
                          (USHORT)pPortInst->ulCurrentMode,
                          &usComMode);
            ulLastReadTime = time();
            if (rc || (usComMode < CURMODE_NIBBLE) ) {
               #ifdef DEBUG_ALERT
                {
                 if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
                 {
                    sprintf( logbuf,
                             "ParReadThread SetMode() failed rc=%d Mode=%d\r\n",
                             rc, usComMode );
                    LogCall( logbuf );
                 }
                }
               #endif /* DEBUG */
               pPortInst->flStatus |= PF_COMM_STAT_CHANGE;
               DosPostEventSem(PdrWaitAlert.hevWaitAlert);
            }
         }
      } /* endif no data returned from DosRead */
      /*
      ** We don't want to continuously call DosRead.
      ** We will stagger our waits as follows:
      ** If data was read
      **    Wait 1 second( 1/10 sec if printing a job with wrappers )
      ** Else
      **    add 2 seconds to wait time, for max waittime 32 seconds
      **
      ** If SplPdSendCmd called
      **    hevReadThread posted to let us get printer response.
      */
      if (ulWaitTime) {
         ulWaitTime += 2000;
         if (ulWaitTime > 32000) {
            ulWaitTime = 32000;
         }
      } else {
         /*
         ** If we have not been waiting and we are printing  @BUGBUG
         **   a job to a printer that puts wrappers around the
         **   job data, the CNV might be waiting for an ack on
         **   the last buffer sent to the printer.
         **   We will only wait 100 ms in this case instead of
         **   the normal 1000 ms(1 second).
         */
         if ( (pPortInst->ulJobPrinting == PDR_PRINTING) &&
              (pPortInst->flJob & PRTSW_JOB_WRAPPER_REQUIRED) ) {
            ulWaitTime = 100;
         } else {
            ulWaitTime = 1000;
         }
      }
      #ifdef DEBUG_ALERT
       {
        if (flChangeDebugLog & FL_PDRDB_ENABLE_READWAIT)
        {
           sprintf( logbuf,
                    "ParReadThread %s About to Wait %d ms hev=%lX\r\n",
                    pszPort, ulWaitTime, (ULONG)hevReadThread );
           LogCall( logbuf );
        }
       }
      #endif /* DEBUG */
      rc = DosWaitEventSem ( hevReadThread, ulWaitTime );
      if (rc == 0) {
         /*
          * This thread's event sem was posted because someone expected
          *   a response from the printer.
          * Reset our ulWaitTime so that if no data is available
          *   we retry our read within an acceptable amount of time.
          */
         ulWaitTime = 0; // reset hevReadThread wait time
      }

      rc2 = DosResetEventSem(hevReadThread, &ulPostCount);
      #ifdef DEBUG_ALERT
       {
        if (flChangeDebugLog & FL_PDRDB_ENABLE_READWAIT)
        {
           sprintf( logbuf,
                    "ParReadThread %s WokeUp rc=%d ResetEventSem rc=%d PostCount=%d\r\n",
                    pszPort, rc, rc2, ulPostCount );
           LogCall( logbuf );
        }
       }
      #endif /* DEBUG */

    } while ( cErrors < 20 );

    /*
     * Must clear readthreadID so we create another readthread
     *   when needed.
     */
   EnterPdrSem();
    if (pPortInst->signature == PT_SIGNATURE) {
       if (rc == ERROR_INVALID_HANDLE) {
          #ifdef DEBUG_ALERT
           {
            if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
            {
               sprintf( logbuf,
                        "ParReadThread InvalidHandle - closing port %s\r\n",
                        pszPort );
               LogCall( logbuf );
            }
           }
          #endif /* DEBUG */
          PdClose(pPortInst);
       }
       pPortInst->tidReadThread = 0;
       DosPostEventSem(PdrWaitAlert.hevWaitAlert);
    }
   LeavePdrSem();
}

/****************************************************************************
 *
 * FUNCTION NAME = CheckAllPorts - INTERNAL API
 *
 * DESCRIPTION   = Check all ports during HandleWaitAlert call.
 *                 Start ReadThread for port if needed.
 *                 Return PowerOn alert to caller of BIDI_WAIT_ALERT
 *                   if any port has recently changed printers.
 *
 * INPUT         = pPrtAlert -> caller of BIDI_WAIT_ALERT buffer to get alert
 *                 pcbBuf    -> size of buffer on entry, set to size of
 *                              powerOn alert on exit if alert returned
 *                 pfAlertReturned -> BOOL set to TRUE if powerOn alert
 *                                    is placed into pPrtAlert buffer
 *
 * OUTPUT        = 0                - success checking ports
 *                 NERR_BufTooSmall - unable to store PowerOn alert in buffer
 *
 * NOTE          = Must be in PdrSem on entry/exit
 *
 ****************************************************************************/

ULONG CheckAllPorts( PPRTALERT pPrtAlert, PULONG pcbBuf,
                     PBOOL pfAlertReturned )
{
   ULONG         rc;
   ULONG         cbNeeded;
   PPORTINST     pPortTemp;
   PSZ           pszPort;
   PSZ           pszTemp;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif

   rc = 0;

   pPortTemp = pPortInstList;
   while (pPortTemp) {
      if ( (pPortTemp->flStatus & PF_Q_PORT_CALLED) &&
           (pPortTemp->ulCurrentMode > CURMODE_COMPATIBLE) &&
           !(pPortTemp->tidReadThread) &&
           (pPortTemp->hFile) ) {

         CreateReadThread( pPortTemp );
      }
      //
      // Handle case where the printer ID has changed
      //  (most likely a new printer attached).
      // Return a Communications Status Change alert.
      // This will cause the spooler to re-initialize the printer,
      //   possibly putting it into bidi-mode for the first time.
      //
      if (pPortTemp->flStatus & PF_COMM_STAT_CHANGE ) {
         pPortTemp->flStatus &= ~PF_COMM_STAT_CHANGE;
         pPortTemp->flStatus |= PF_COMALERT_POSTED;
         pszPort = pPortTemp->pszPortName;
         cbNeeded = sizeof( PRTALERT ) + strlen(pszPort) + 1;
         if (*pcbBuf >= cbNeeded) {
            /*
            ** Only variable length field we return is ulpszPortName.
            ** This is the offset from the beginning of the buffer
            **   to where the portname string is located.
            */
            pPrtAlert->ulpszPortName   = sizeof( PRTALERT );
            pPrtAlert->ulCategory      = PRTALERT_CATEGORY_CORE;
            pPrtAlert->ulType          = PRTALERT_TYPE_COMM_STATUS_CHANGED;
            pPrtAlert->interpreterID   = PRTALERT_INTERPRETER_CURRENT;
            pPrtAlert->bReserved       = 0;
            pPrtAlert->usSeverity      = PRTALERT_SEV_INFORMATIONAL;
            pPrtAlert->ulValue         = 0;
            pPrtAlert->ulPrinterJobID  = 0;
            pPrtAlert->ulpszAlertMsg   = 0;
            pszTemp = (PSZ)(pPrtAlert->ulpszPortName + (ULONG)pPrtAlert);
            strcpy( pszTemp, pszPort );
         } else {
            rc = NERR_BufTooSmall;
         }
         #ifdef DEBUG_ALERT
          {
           if (flChangeDebugLog & FL_PDRDB_ENABLE_WAITALERT)
           {
              sprintf( logbuf,
                       "CheckAllPorts PowerOn returned for %s rc=%d\r\n",
                       pszPort, rc );
              LogCall( logbuf );
           }
          }
         #endif /* DEBUG */
         *pcbBuf = cbNeeded;
         *pfAlertReturned = TRUE;
         break;
      }
      pPortTemp = pPortTemp->pNext;
   }
   return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = GetDevidThread - INTERNAL API
 *
 * DESCRIPTION   = Thread to get current deviceID of port
 *
 * INPUT         = pPortInst -> port to query
 *
 * OUTPUT        = None, but pPortInst is updated with current information
 *                 from connected printer
 *
 ****************************************************************************/

VOID EXPENTRY GetDevidThread( PPORTINST pPortInst )
{
    ULONG         rc;
    PSZ           pszPort;

    #ifdef DEBUG_ALERT
      char logbuf[360];
    #endif

   EnterPdrSem();
    if (pPortInst->signature == PT_SIGNATURE) {
       pszPort = pPortInst->pszPortName;
    } else {
       pszPort = NULL;
    }
   LeavePdrSem();

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
      {
         sprintf( logbuf,
                  "GetDevidThread started for %s\r\n",
                  pszPort ? pszPort : "Invalid" );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */

    rc = GetDeviceID( pPortInst );

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
      {
         sprintf( logbuf,
                  "GetDevidThread GetDeviceID rc=%d for %s\r\n",
                  rc, pszPort ? pszPort : "Invalid" );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */
}

/****************************************************************************
 *
 * FUNCTION NAME = RecheckDevID - INTERNAL API
 *
 * DESCRIPTION   = Get current deviceID of port after first successful write
 *
 * INPUT         = pPortInst -> port to query
 *
 * OUTPUT        = None, but pPortInst is updated with current information
 *                 from connected printer
 *
 * NOTE          = Not in PDR sem on entry/exit
 *
 ****************************************************************************/

VOID EXPENTRY RecheckDevID( PPORTINST pPortInst )
{
    ULONG         rc;
    PSZ           pszPort;

    #ifdef DEBUG_ALERT
      char logbuf[360];
    #endif

   EnterPdrSem();
    if (pPortInst->signature == PT_SIGNATURE) {
       //
       // Set flag so we only try to put printer into bidi mode once.
       // This avoids us trying to put a unidirectional printer into
       //   bidi mode during each job.
       //
       pPortInst->flStatus |= PF_BIDI_CHECKED;
       pszPort = pPortInst->pszPortName;
    } else {
       pszPort = NULL;
    }

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
      {
         sprintf( logbuf,
                  "RecheckDevID started for %s\r\n",
                  pszPort ? pszPort : "Invalid" );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */

    if (pszPort) {
       //
       // Must be in PdrSem to call GetDeviceID
       //
       rc = GetDeviceID( pPortInst );
    }

   LeavePdrSem();

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_READTHREAD)
      {
         sprintf( logbuf,
                  "RecheckDevID GetDeviceID rc=%d for %s\r\n",
                  rc, pszPort ? pszPort : "Invalid" );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */
}


/*
** END file par1284.c
*/
