/****************************************************************************/
/*  Copyright (C) 1993 IBM Corporation                                      */
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = dialogs.c
 *
 * DESCRIPTIVE NAME = par1284 port driver routines to display settings
 *                     dialogs to user
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1994, 1995
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.2
 *
 * DATE
 *
 * DESCRIPTION
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *  DATE      FLAG        APAR   CHANGE DESCRIPTION
 *  --------  ----------  -----  --------------------------------------
 *  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 ****************************************************************************/

#define     INCL_WINMLE
#include    "pdrconst.h"
#include    "pdrtypes.h"
#include    "pdrproto.h"

/*
** Local functions
*/
MRESULT EXPENTRY Par1284Dlg( HWND hDlg,
                            USHORT msg,
                            MPARAM mp1,
                            MPARAM mp2 );
MRESULT EXPENTRY Par1284AdvancedDlg( HWND hDlg,
                            USHORT msg,
                            MPARAM mp1,
                            MPARAM mp2 );
VOID             InitPortSettingsDlg( HWND hDlg, MPARAM mp2 );
BOOL             SavePortDlgSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct);
VOID             ResetPortDlgSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct );
VOID             InitAdvSettingsDlg( HWND hDlg, MPARAM mp2 );
VOID             ResetAdvDlgSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct );
BOOL             SaveAdvSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct );
VOID             InitAdvDlgValues( PPORTDLGSTRUCT pPortDlgStruct );
VOID EXPENTRY    QueryDriverThread( PPORTDLGSTRUCT pPortDlgStruct );

/****************************************************************************
 *
 * FUNCTION NAME = OpenPar1284PortDlg
 *
 * DESCRIPTION   = Display port settings dialog
 *
 * INPUT         = hab     - anchor block handle
 *                 pPortDlgStruct - pointer to the Par1284 data structure
 *
 * OUTPUT        = TRUE  - if port settings changed
 *                 FALSE - if port settings not changed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *
 ****************************************************************************/

ULONG OpenPar1284PortDlg ( HAB hab, PPORTDLGSTRUCT pPortDlgStruct )
{
   ULONG          rc;
   PPORTSETTINGS  pPortSettings;
   PPORTDLGSTRUCT pOldPortDlgStruct;
   ULONG          cbBuf;
   ULONG          cbNeeded;
   ULONG          cRetries;
   BOOL           fRetry;
   BOOL           fAdded;
   TID            tid;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif

   /*
    * Check to see if port settings are already being displayed and
    *       set focus to the open dialog
    */
  EnterPdrSem();
   pOldPortDlgStruct = FindPortDlgStruct ( pPortDlgStruct );
   if (pOldPortDlgStruct) {
      LeavePdrSem();
       WinShowWindow ( pOldPortDlgStruct->hwndDlg, TRUE );
       WinSetFocus ( HWND_DESKTOP, pOldPortDlgStruct->hwndDlg );
       return(FALSE);
   }
   fAdded = AddPortDlgStruct ( pPortDlgStruct );
  LeavePdrSem();

   /*
   ** Must query port driver for current settings
   **
   ** NOTE: you should start another thread that will issue
   **       the PrtQuery because it can take a long time.
   **       The thread should get the dialog handle from the
   **       pPortDlgStruct, then update the dialog when all
   **       the data from PrtQuery has been returned.
   */
   rc = DosCreateThread( &tid,
                         (PFNTHREAD)QueryDriverThread,
                         (ULONG)pPortDlgStruct,     // Parameter
                         0,                         // Execution Flag
                         DEFAULT_STACKSIZE);
   if (rc)
   {
      DisplayError (HWND_DESKTOP, ID_IR_SETTINGS_ERROR,
                    MB_OK | MB_APPLMODAL | MB_MOVEABLE);
      if (pPortSettings)
      {
         DosFreeMem(pPortSettings);
      }
      if (fAdded) {
          RemovePortDlgStruct ( pPortDlgStruct );
      }
      return(FALSE);
   }


   WinDlgBox  (HWND_DESKTOP,
               HWND_DESKTOP,
               (PFNWP)Par1284Dlg,
               (HMODULE)hPdrMod,
               IDD_PORTPAR1284,
               pPortDlgStruct);

  EnterDlgSem( pPortDlgStruct );
   if (pPortDlgStruct->pPortSettings) {
      DosFreeMem(pPortDlgStruct->pPortSettings);
      pPortDlgStruct->pPortSettings = NULL;
   }
  LeaveDlgSem( pPortDlgStruct );

  EnterPdrSem();
   if (fAdded) {
      RemovePortDlgStruct ( pPortDlgStruct );
   }
  LeavePdrSem();

   return pPortDlgStruct->flModified;

}



/****************************************************************************
 *
 * FUNCTION NAME = Par1284Dlg
 *
 * DESCRIPTION   = Port settings dialog procedure
 *
 * INPUT         = hDlg: HWND
 *                 msg:  message
 *                 mp1:  message parameter
 *                 mp2:     "       "
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

MRESULT EXPENTRY Par1284Dlg( HWND hDlg,
                            USHORT msg,
                            MPARAM mp1,
                            MPARAM mp2 )
{
    PPORTDLGSTRUCT pPortDlgStruct;

    switch (msg) {

        case WM_INITDLG:
            InitPortSettingsDlg( hDlg, mp2 );
            break;

        case IDC_SETTINGS_UPDATED:
            pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER));
            if (pPortDlgStruct) {
               InitAdvDlgValues( pPortDlgStruct );
            }
            break;

        case WM_COMMAND:
            /*
            ** Check validity of pPortDlgStruct
            */
            pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER));

            switch (SHORT1FROMMP(mp1)) {
               case IDC_OK:
                  /*
                  ** Call function to save any changed port settings
                  */
                  if (pPortDlgStruct) {
                     if (SavePortDlgSettings(hDlg, pPortDlgStruct)) {
                        WinDismissDlg(hDlg, 0);
                     }
                  } else {
                     WinDismissDlg(hDlg, 0);
                  }

                  break;

               case IDC_RESET:
                  if (pPortDlgStruct) {
                     ResetPortDlgSettings( hDlg, pPortDlgStruct);
                  }

                  break;

               case IDC_DEFAULT:
                  WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_PRINT_VALUE),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_QUERY_VALUE),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_JOB_VALUE),
                                    MPFROMSHORT(NULL));

                  WinSendDlgItemMsg (hDlg, IDC_PPA_SHAREHW,
                                    BM_SETCHECK,
                                    MPFROMSHORT(FALSE),
                                    MPFROMSHORT(NULL));

                  break;

               case IDC_CANCEL:
                  WinDismissDlg(hDlg, MBID_CANCEL);
                  break;

               case IDC_ADVANCED:
                  if (pPortDlgStruct) {
                     if (pPortDlgStruct->hwndAdvanced) {
                        /*
                        ** Bring advanced settings dialog to foreground if up
                        */
                        WinShowWindow ( pPortDlgStruct->hwndAdvanced, TRUE );
                        WinSetFocus ( HWND_DESKTOP, pPortDlgStruct->hwndAdvanced );
                     } else {
                        WinDlgBox  (HWND_DESKTOP,
                                    HWND_DESKTOP,
                                    (PFNWP)Par1284AdvancedDlg,
                                    (HMODULE)hPdrMod,
                                    IDD_ADVANCEDOPTS,
                                    pPortDlgStruct);
                     }
                  }
                  break;

               default:
                  return(WinDefDlgProc(hDlg, msg, mp1, mp2) );
            }
           break;

        case WM_HELP:
            InitializeHelp();
            if (hwndHelp)
            {
               WinSendMsg (hwndHelp, HM_DISPLAY_HELP,
                                         (MPARAM)IDH_DLG_EXTENDED, NULL);
               return (MRESULT)TRUE;
            }
            break;

        case WM_DESTROY:

            /*
            ** Clear dialog hwnd in pPortDlgStruct
            */
            pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER));
            if (pPortDlgStruct)
            {
               pPortDlgStruct->hwndDlg = 0;
            }
               /*
               ** if we have a help instance destroy it.
               */
            if (HelpAlreadyInitialized)
            {
               CALLDestroyHelpInstance(hwndHelp);
               hwndHelp = (HWND) NULL;
               HelpAlreadyInitialized = FALSE;
               HelpStubHookIsSet=FALSE;
            }
            ReleaseHelpStubHook();
             break;

        default:
            return(WinDefDlgProc(hDlg, msg, mp1, mp2) );
    }
    return FALSE;
}

/****************************************************************************
 *
 * FUNCTION NAME = Par1284AdvancedDlg
 *
 * DESCRIPTION   = Advanced port settings dialog procedure
 *
 * INPUT         = hDlg: HWND
 *                 msg:  message
 *                 mp1:  message parameter
 *                 mp2:     "       "
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

MRESULT EXPENTRY Par1284AdvancedDlg( HWND hDlg,
                            USHORT msg,
                            MPARAM mp1,
                            MPARAM mp2 )
{
    PPORTDLGSTRUCT pPortDlgStruct;
    TID            tid;


    switch (msg) {

        case WM_INITDLG:
            InitAdvSettingsDlg( hDlg, mp2 );
            break;

        case IDC_SETTINGS_UPDATED:
            pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER));
            if (pPortDlgStruct) {
               InitAdvDlgValues( pPortDlgStruct );
            }
            break;

        case WM_COMMAND:
            /*
            ** Check validity of pPortDlgStruct
            */
            pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER));

            switch (SHORT1FROMMP(mp1)) {
               case IDC_OK:

                  /*
                  ** Call function to save any changed port settings
                  */
                  if (pPortDlgStruct) {
                     if (SaveAdvSettings(hDlg, pPortDlgStruct)) {
                        WinDismissDlg(hDlg, 0);
                     }
                  } else {
                     WinDismissDlg(hDlg, 0);
                  }

                  break;

               case IDC_RESET:
                  if (pPortDlgStruct) {
                     ResetAdvDlgSettings( hDlg, pPortDlgStruct);
                  }

                  break;

               case IDC_DEFAULT:
                  WinSendDlgItemMsg (hDlg, IDC_PPE_READ_INTCHAR,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_READ_INTER),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPE_READ_ELAPSED,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_READ_ELAPSED),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_INTCHAR,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_WRITE_INTER),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_ELAPSED,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_WRITE_ELAPSED),
                                    MPFROMSHORT(NULL));
                  break;

               case IDC_CANCEL:
                  WinDismissDlg(hDlg, MBID_CANCEL);
                  break;

               case IDC_REFRESH:
                  DosCreateThread( &tid,
                                   (PFNTHREAD)QueryDriverThread,
                                   (ULONG)pPortDlgStruct,     // Parameter
                                   0,                         // Execution Flag
                                   DEFAULT_STACKSIZE);
                  break;

               default:
                  return(WinDefDlgProc(hDlg, msg, mp1, mp2) );
            }
           break;

        case WM_HELP:
            InitializeHelp();
            if (hwndHelp)
            {
               WinSendMsg (hwndHelp, HM_DISPLAY_HELP,
                                         (MPARAM)IDH_DLG_EXTENDED, NULL);
               return (MRESULT)TRUE;
            }
            break;

        case WM_DESTROY:

            /*
            ** Clear dialog hwnd in pPortDlgStruct
            */
            pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER));
            if (pPortDlgStruct)
            {
               pPortDlgStruct->hwndAdvanced = 0;
            }
               /*
               ** if we have a help instance destroy it.
               */
            if (HelpAlreadyInitialized)
            {
               CALLDestroyHelpInstance(hwndHelp);
               hwndHelp = (HWND) NULL;
               HelpAlreadyInitialized = FALSE;
               HelpStubHookIsSet=FALSE;
            }
            ReleaseHelpStubHook();
             break;

        default:
            return(WinDefDlgProc(hDlg, msg, mp1, mp2) );
    }
    return FALSE;
}

/****************************************************************************
 *
 * FUNCTION NAME = InitPortSettingsDlg
 *
 * DESCRIPTION   = Initialize port settings dialog when WM_INITDLG received
 *
 * INPUT         = hDlg      - dialog hwnd
 *                 mp2       -> PORTSETTINGS structure
 *
 * OUTPUT        = None
 *
 ****************************************************************************/
VOID  InitPortSettingsDlg( HWND hDlg, MPARAM mp2 )
{
   PPORTDLGSTRUCT pPortDlgStruct;
   PPORTSETTINGS  pPortSettings;
   CHAR           szTitle[ CCHMAXPATH + 1];
   CHAR           szTemp[ CCHMAXPATH  ];
   BOOL           fSuccess;


   pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)mp2);
   if (pPortDlgStruct)
   {
      WinSetWindowULong (hDlg, QWL_USER, (ULONG)pPortDlgStruct);
      pPortDlgStruct->hwndDlg = hDlg ;

      WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      WinSetWindowText (WinWindowFromID (hDlg,
                                         (USHORT)IDC_PPA_PORTNAME),
                        pPortDlgStruct->pszPortName);
      /*
       * Make dialog title consistent with the shell
       *    (PortName - Properties)
       */
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         PDR_ID_PROPERTIES,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          strcpy ( szTitle, pPortDlgStruct->pszPortName );
          if (pPortDlgStruct->pszComputerName && *pPortDlgStruct->pszComputerName) {
             strcat( szTitle, "(" );
             strcat( szTitle, pPortDlgStruct->pszComputerName) ;
             strcat( szTitle, ")" );
          }
          strcat ( szTitle, " - " );
          strcat ( szTitle, szTemp );
          WinSetWindowText (hDlg, szTitle);
      }

     EnterDlgSem( pPortDlgStruct );
      pPortSettings = pPortDlgStruct->pPortSettings;
      if (pPortSettings) {
          WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                                 SPBM_SETCURRENTVALUE,
                                 MPFROMSHORT(pPortSettings->ulPrintTimeOut),
                                 MPFROMSHORT(NULL));
          WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                                 SPBM_SETCURRENTVALUE,
                                 MPFROMSHORT(pPortSettings->ulNoQueryTimeOut),
                                 MPFROMSHORT(NULL));
          WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                                 SPBM_SETCURRENTVALUE,
                                 MPFROMSHORT(pPortSettings->ulNoJobTimeOut),
                                 MPFROMSHORT(NULL));

          WinSendDlgItemMsg (hDlg, IDC_PPA_SHAREHW,
                                 BM_SETCHECK,
                                 MPFROMSHORT(pPortSettings->fShareAccess),
                                 MPFROMSHORT(NULL));
      }
     LeaveDlgSem( pPortDlgStruct );

       /*
       ** create help hook for helps
       */
       SetHelpStubHook();
   }

}

/****************************************************************************
 *
 * FUNCTION NAME = SavePortDlgSettings
 *
 * DESCRIPTION   = save any changed values for this port in INI file
 *                 called when closing port settings(properties) dialog
 *
 * INPUT         = pPortDlgStruct -> this port's information
 *
 * OUTPUT        = TRUE          -  successful
 *                 FALSE         -  PrfWrite failed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *
 ****************************************************************************/

BOOL SavePortDlgSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct )
{
   PPORTSETTINGS pPortSettings;
   PORTSETTINGS  NewPortSettings;
   BOOL          fModified;
   ULONG         rc;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif


   fModified        = FALSE;
   rc               = 0;

   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "SavePortDlgSettings hDlg=%lX pPortDlgStruct=%lX\r\n",
                 (ULONG)hDlg,
                 (ULONG)pPortDlgStruct );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */

  EnterDlgSem( pPortDlgStruct );
   pPortSettings = pPortDlgStruct->pPortSettings;
   if (pPortSettings) {
      memcpy (&NewPortSettings, pPortSettings, sizeof(NewPortSettings));
      WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.ulPrintTimeOut),
                        (MPARAM)NULL);
      WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.ulNoQueryTimeOut),
                        (MPARAM)NULL);
      WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.ulNoJobTimeOut),
                        (MPARAM)NULL);
      NewPortSettings.fShareAccess =
                     SHORT1FROMMP( WinSendDlgItemMsg (hDlg, IDC_PPA_SHAREHW,
                              BM_QUERYCHECK,
                              MPFROMSHORT(NULL),
                              MPFROMSHORT(NULL)) );

      if ( (NewPortSettings.ulPrintTimeOut != pPortSettings->ulPrintTimeOut) ||
           (NewPortSettings.ulNoQueryTimeOut != pPortSettings->ulNoQueryTimeOut) ||
           (NewPortSettings.ulNoJobTimeOut != pPortSettings->ulNoJobTimeOut) ||
           (NewPortSettings.fShareAccess != pPortSettings->fShareAccess) )
      {
         fModified = TRUE;
      }
      pPortDlgStruct->flModified = fModified;
   }
  LeaveDlgSem( pPortDlgStruct );

   if (fModified)
   {
      NewPortSettings.ulpszPortName = 0 ;
      NewPortSettings.ulpszDeviceID = 0 ;

      rc = PMERR_SPL_SPOOLER_NOT_INSTALLED;
      if (pfnPrtSet)
      {
         rc = MyPrtSet( pPortDlgStruct->pszComputerName,
                        pPortDlgStruct->pszPortName,
                        TYPE_LONG_WAIT,
                        BIDI_SET_PORTDRV,
                        &NewPortSettings,
                        sizeof(NewPortSettings) );
      }
      if ( rc )
      {
         /*
         ** PrtSet can fail if the spooler is disabled.
         ** If Bidi spooler APIs failed or not available
         ** then call routine directly.
         */
         rc = HandleSetPortDRV ( pPortDlgStruct->pszPortName,
                                  &NewPortSettings,
                                  sizeof(NewPortSettings) );
      }
   }

   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "SaveAdvDlgSettings rc=%d\r\n",
                 rc );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */

   if (rc)
   {
      return(FALSE);
   }
   return TRUE;
}

/****************************************************************************
 *
 * FUNCTION NAME = ResetPortDlgSettings
 *
 * DESCRIPTION   = save any changed values for this port in INI file
 *                 called when closing port settings(properties) dialog
 *
 * INPUT         = pPortDlgStruct -> this port's information
 *
 * OUTPUT        = TRUE          -  successful
 *                 FALSE         -  PrfWrite failed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *
 ****************************************************************************/

VOID ResetPortDlgSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct )
{
   PPORTSETTINGS pPortSettings;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif


   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "ResetPortDlgSettings hDlg=%lX pPortDlgStruct=%lX\r\n",
                 (ULONG)hDlg,
                 (ULONG)pPortDlgStruct );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */

   EnterDlgSem( pPortDlgStruct );
    pPortSettings = pPortDlgStruct->pPortSettings;
    if (pPortSettings) {
       WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->ulPrintTimeOut),
                         MPFROMSHORT(NULL));
       WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->ulNoQueryTimeOut),
                         MPFROMSHORT(NULL));
       WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->ulNoJobTimeOut),
                         MPFROMSHORT(NULL));
       WinSendDlgItemMsg (hDlg, IDC_PPA_SHAREHW,
                         BM_SETCHECK,
                         MPFROMSHORT(pPortSettings->fShareAccess),
                         MPFROMSHORT(NULL));
    }
   LeaveDlgSem( pPortDlgStruct );

   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "ResetPortDlgSettings done\r\n" );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */
}

/****************************************************************************
 *
 * FUNCTION NAME = InitAdvSettingsDlg
 *
 * DESCRIPTION   = Initialize advanced port settings dialog when WM_INITDLG received
 *
 * INPUT         = hDlg      - dialog hwnd
 *                 mp2       -> PORTSETTINGS structure
 *
 * OUTPUT        = None
 *
 ****************************************************************************/
VOID  InitAdvSettingsDlg( HWND hDlg, MPARAM mp2 )
{
   PPORTDLGSTRUCT pPortDlgStruct;
   PPORTSETTINGS  pPortSettings;
   PSZ            pszDesc;
   PSZ            pszDeviceID;
   CHAR           szTitle[ CCHMAXPATH + 1];
   CHAR           szTemp[ CCHMAXPATH  ];
   BOOL           fSuccess;

   pPortDlgStruct = ValidatePortDlg((PPORTDLGSTRUCT)mp2);
   if (pPortDlgStruct)
   {
      WinSetWindowULong (hDlg, QWL_USER, (ULONG)pPortDlgStruct);
      pPortDlgStruct->hwndAdvanced = hDlg ;

      WinSendDlgItemMsg (hDlg, IDC_PPE_READ_INTCHAR,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT_MS),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      WinSendDlgItemMsg (hDlg, IDC_PPE_READ_ELAPSED,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT_MS),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_INTCHAR,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT_MS),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_ELAPSED,
                             SPBM_SETLIMITS,
                             MPFROMP(TIMEOUT_UPPER_LIMIT_MS),
                             MPFROMP(TIMEOUT_LOWER_LIMIT));

      /*
       * Make dialog title consistent with the shell
       *    (PortName - Properties)
       */
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         PDR_ID_ADV_PROPERTIES,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          strcpy ( szTitle, pPortDlgStruct->pszPortName );
          if (pPortDlgStruct->pszComputerName && *pPortDlgStruct->pszComputerName)
          {
             strcat ( szTitle, "(" );
             strcat ( szTitle, pPortDlgStruct->pszComputerName);
             strcat ( szTitle, ")" );
          }
          strcat ( szTitle, " - " );
          strcat ( szTitle, szTemp );
          WinSetWindowText (hDlg, szTitle);
      }

      /*
       * Fill in drop-down combo-box with modes to select
       */
      WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_DELETEALL, NULL, NULL );
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         ID_MODE_AUTODETECT,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_INSERTITEM, MPFROMLONG(PARMODE_AUTODETECT), szTemp );
      }
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         ID_MODE_COMPATIBLE,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_INSERTITEM, MPFROMLONG(PARMODE_DISABLE_BIDI), szTemp );
      }
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         ID_MODE_NIBBLE,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_INSERTITEM, MPFROMLONG(PARMODE_NIBBLE), szTemp );
      }
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         ID_MODE_BYTE,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_INSERTITEM, MPFROMLONG(PARMODE_BYTE), szTemp );
      }
      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         ID_MODE_ECP,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_INSERTITEM, MPFROMLONG(PARMODE_ECP), szTemp );
      }

      fSuccess = WinLoadString(pPortDlgStruct->hAB,
                         hPdrMod,
                         ID_MODE_ECP_RLE,
                         CCHMAXPATH, szTemp);
      if (fSuccess) {
          WinSendDlgItemMsg( hDlg,  IDC_PPE_MODE_SELECTED, LM_INSERTITEM, MPFROMLONG(PARMODE_ECP_RLE), szTemp );
      }


      InitAdvDlgValues( pPortDlgStruct );
   }

    /*
    ** create help hook for helps
    */
    SetHelpStubHook();

}

/****************************************************************************
 *
 * FUNCTION NAME = ResetAdvDlgSettings
 *
 * DESCRIPTION   = reset advanced settings to original state
 *
 * INPUT         = pPortDlgStruct -> this port's information
 *
 * OUTPUT        = TRUE          -  successful
 *                 FALSE         -  PrfWrite failed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *
 ****************************************************************************/

VOID ResetAdvDlgSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct )
{
   PPORTSETTINGS pPortSettings;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif


   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "ResetAdvDlgSettings hDlg=%lX pPortDlgStruct=%lX\r\n",
                 (ULONG)hDlg,
                 (ULONG)pPortDlgStruct );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */

   EnterDlgSem( pPortDlgStruct );
    pPortSettings = pPortDlgStruct->pPortSettings;
    if (pPortSettings) {
      WinSendDlgItemMsg (hDlg, IDC_PPE_MODE_SELECTED,
                         LM_SELECTITEM,
                         MPFROMLONG(pPortSettings->ulModeSelected),
                         MPFROMSHORT((SHORT)TRUE));
       WinSendDlgItemMsg (hDlg, IDC_PPE_READ_INTCHAR,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->TimeOut.ulReadIntTimeOut),
                         MPFROMSHORT(NULL));
       WinSendDlgItemMsg (hDlg, IDC_PPE_READ_ELAPSED,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->TimeOut.ulReadIdleTimeOut),
                         MPFROMSHORT(NULL));
       WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_INTCHAR,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->TimeOut.ulWriteIntTimeOut),
                         MPFROMSHORT(NULL));
       WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_ELAPSED,
                          SPBM_SETCURRENTVALUE,
                         MPFROMSHORT(pPortSettings->TimeOut.ulWriteIdleTimeOut),
                         MPFROMSHORT(NULL));
    }
   LeaveDlgSem( pPortDlgStruct );

   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "ResetAdvDlgSettings done\r\n" );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */
}

/****************************************************************************
 *
 * FUNCTION NAME = SaveAdvSettings
 *
 * DESCRIPTION   = save any changed values for this port in INI file
 *                 called when closing port settings(properties) dialog
 *
 * INPUT         = pPortDlgStruct -> this port's information
 *
 * OUTPUT        = TRUE          -  successful
 *                 FALSE         -  PrfWrite failed
 *
 * NOTE          = Not in PdrSem on entry/exit
 *
 ****************************************************************************/

BOOL SaveAdvSettings( HWND hDlg, PPORTDLGSTRUCT pPortDlgStruct )
{
   PPORTSETTINGS pPortSettings;
   PORTSETTINGS  NewPortSettings;
   BOOL          fModified;
   ULONG         rc;
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif


   fModified        = FALSE;
   rc               = 0;

   #ifdef DEBUG_ALERT
    {
     if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
     {
        sprintf( logbuf,
                 "SaveAdvDlgSettings hDlg=%lX pPortDlgStruct=%lX\r\n",
                 (ULONG)hDlg,
                 (ULONG)pPortDlgStruct );
        LogCall( logbuf );
     }
    }
   #endif /* DEBUG */

  EnterDlgSem( pPortDlgStruct );
   pPortSettings = pPortDlgStruct->pPortSettings;
   if (pPortSettings) {
      memcpy (&NewPortSettings, pPortSettings, sizeof(NewPortSettings));
      WinSendDlgItemMsg (hDlg, IDC_PPE_READ_INTCHAR, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.TimeOut.ulReadIntTimeOut),
                        (MPARAM)NULL);
      WinSendDlgItemMsg (hDlg, IDC_PPE_READ_ELAPSED, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.TimeOut.ulReadIdleTimeOut),
                        (MPARAM)NULL);
      WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_INTCHAR, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.TimeOut.ulWriteIntTimeOut),
                        (MPARAM)NULL);
      WinSendDlgItemMsg (hDlg, IDC_PPE_WRITE_ELAPSED, SPBM_QUERYVALUE,
                        (MPARAM)&(NewPortSettings.TimeOut.ulWriteIdleTimeOut),
                        (MPARAM)NULL);
      NewPortSettings.ulModeSelected = LONGFROMMR(
                        WinSendDlgItemMsg (hDlg, IDC_PPE_MODE_SELECTED,
                             LM_QUERYSELECTION,
                             MPFROMLONG(LIT_FIRST),
                             0));
      #ifdef DEBUG_ALERT
       {
        if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
        {
           sprintf( logbuf,
                    "SaveAdvDlgSettings Old ReadInt=%d ReadElapsed=%d WriteInt=%d WriteElapsed=%d ModeSelected=%d\r\n",
                    pPortSettings->TimeOut.ulReadIntTimeOut,
                    pPortSettings->TimeOut.ulReadIdleTimeOut,
                    pPortSettings->TimeOut.ulWriteIntTimeOut,
                    pPortSettings->TimeOut.ulWriteIdleTimeOut,
                    pPortSettings->ulModeSelected );
           LogCall( logbuf );
           sprintf( logbuf,
                    "SaveAdvDlgSettings New ReadInt=%d ReadElapsed=%d WriteInt=%d WriteElapsed=%d ModeSelected=%d\r\n",
                    NewPortSettings.TimeOut.ulReadIntTimeOut,
                    NewPortSettings.TimeOut.ulReadIdleTimeOut,
                    NewPortSettings.TimeOut.ulWriteIntTimeOut,
                    NewPortSettings.TimeOut.ulWriteIdleTimeOut,
                    NewPortSettings.ulModeSelected );
           LogCall( logbuf );
        }
       }
      #endif /* DEBUG */
      if ( (NewPortSettings.TimeOut.ulReadIntTimeOut != pPortSettings->TimeOut.ulReadIntTimeOut) ||
           (NewPortSettings.TimeOut.ulReadIdleTimeOut != pPortSettings->TimeOut.ulReadIdleTimeOut) ||
           (NewPortSettings.TimeOut.ulWriteIntTimeOut != pPortSettings->TimeOut.ulWriteIntTimeOut) ||
           (NewPortSettings.TimeOut.ulWriteIdleTimeOut != pPortSettings->TimeOut.ulWriteIdleTimeOut) ||
           (NewPortSettings.ulModeSelected != pPortSettings->ulModeSelected) )
      {
         fModified = TRUE;
      }
      pPortDlgStruct->flModified = fModified;
   }
  LeaveDlgSem( pPortDlgStruct );

   if (fModified)
   {
      NewPortSettings.ulpszPortName = 0 ;
      NewPortSettings.ulpszDeviceID = 0 ;

      rc = PMERR_SPL_SPOOLER_NOT_INSTALLED;
      if (pfnPrtSet)
      {
         rc = MyPrtSet( pPortDlgStruct->pszComputerName,
                        pPortDlgStruct->pszPortName,
                        TYPE_LONG_WAIT,
                        BIDI_SET_PORTDRV,
                        &NewPortSettings,
                        sizeof(NewPortSettings) );

      }
      if ( rc )
      {
         /*
         ** PrtSet can fail if the spooler is disabled.
         ** If Bidi spooler APIs failed or not available
         ** then call routine directly.
         */
         rc = HandleSetPortDRV ( pPortDlgStruct->pszPortName,
                                 &NewPortSettings,
                                 sizeof(NewPortSettings) );
      }
   }
   if (!(flChangeDebugLog & FL_PDRDB_DISABLE_QPORTDRV))
   {
      DBPRINTF ((logbuf, "SavePortDlgSettings rc=%d",rc));
   }
   if (rc)
   {
      return(FALSE);
   }
   return TRUE;
}

/****************************************************************************
 *
 * FUNCTION NAME = InitAdvDlgValues
 *
 * DESCRIPTION   = Initialize advanced settings dialog with data
 *                 when retrieved from port driver.
 *
 * INPUT         = pPortDlgStruct -> port dialog structure
 *
 * OUTPUT        = None
 *
 ****************************************************************************/
VOID  InitAdvDlgValues( PPORTDLGSTRUCT pPortDlgStruct )
{
   PPORTSETTINGS  pPortSettings;
   CHAR           szTemp[ CCHMAXPATH  ];
   ULONG          rc;
   ULONG          stringID;
   HWND           hwndProps;
   HWND           hwndAdv;


  EnterDlgSem( pPortDlgStruct );
   pPortSettings = pPortDlgStruct->pPortSettings;
   if (pPortSettings) {
      hwndProps = pPortDlgStruct->hwndDlg;
      hwndAdv   = pPortDlgStruct->hwndAdvanced;
      if (hwndAdv)
      {
         WinSendDlgItemMsg (hwndAdv, IDC_PPE_MODE_SELECTED,
                                LM_SELECTITEM,
                                MPFROMLONG(pPortSettings->ulModeSelected),
                                MPFROMSHORT((SHORT)TRUE));
         switch (pPortSettings->ulCurrentMode) {

           case CURMODE_COMPATIBLE:
                stringID = ID_MODE_COMPATIBLE;
                break;
           case CURMODE_NIBBLE:
                stringID = ID_MODE_NIBBLE;
                break;
           case CURMODE_BYTE:
                stringID = ID_MODE_BYTE;
                break;
           case CURMODE_ECP:
                stringID = ID_MODE_ECP;
                break;
           case CURMODE_ECP_RLE:
                stringID = ID_MODE_ECP_RLE;
                break;
           default :
                stringID = ID_MODE_COMPATIBLE;
                break;
         }
         rc = WinLoadString(pPortDlgStruct->hAB,
                            hPdrMod,
                            stringID,
                            CCHMAXPATH, szTemp);
         if (rc) {
            WinSendDlgItemMsg (hwndAdv, IDC_PPE_MODE_CURRENT,
                                   EM_SETTEXTLIMIT,
                                   MPFROM2SHORT(260, 0),
                                   NULL);
            WinSetDlgItemText(hwndAdv, IDC_PPE_MODE_CURRENT,
                                   szTemp );
         }
         WinSendDlgItemMsg (hwndAdv, IDC_PPE_READ_INTCHAR,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->TimeOut.ulReadIntTimeOut),
                                MPFROMSHORT(NULL));
         WinSendDlgItemMsg (hwndAdv, IDC_PPE_READ_ELAPSED,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->TimeOut.ulReadIdleTimeOut),
                                MPFROMSHORT(NULL));
         WinSendDlgItemMsg (hwndAdv, IDC_PPE_WRITE_INTCHAR,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->TimeOut.ulWriteIntTimeOut),
                                MPFROMSHORT(NULL));
         WinSendDlgItemMsg (hwndAdv, IDC_PPE_WRITE_ELAPSED,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->TimeOut.ulWriteIdleTimeOut),
                                MPFROMSHORT(NULL));

         #ifdef DEBUG_ALERT
          {
           char logbuf[260];
           ULONG ulcb;

               ulcb =
               WinSendDlgItemMsg (hwndAdv, IDC_MLE_DEVICEID,
                                      MLM_DELETE,
                                      MPFROMLONG(0),
                                      MPFROMLONG(520));
               DBPRINTF ((logbuf, "InitAdvDlgValues MLM_DELETE rc=%d",ulcb));
          }
         #endif

         WinSendDlgItemMsg (hwndAdv, IDC_MLE_DEVICEID,
                                MLM_INSERT,
                                MPFROMLONG(GETPTRFROMOFFSET( ulpszDeviceID, pPortSettings)),
                                MPFROMSHORT(NULL));
      }
      /*
      ** Also reset "Properties" dialog with new values
      */
      if (hwndProps)
      {
         WinSendDlgItemMsg (hwndProps, IDC_PPA_PRINTTIMEOUT,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->ulPrintTimeOut),
                                MPFROMSHORT(NULL));
         WinSendDlgItemMsg (hwndProps, IDC_PPA_QUERYTIMEOUT,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->ulNoQueryTimeOut),
                                MPFROMSHORT(NULL));
         WinSendDlgItemMsg (hwndProps, IDC_PPA_JOBTIMEOUT,
                                SPBM_SETCURRENTVALUE,
                                MPFROMSHORT(pPortSettings->ulNoJobTimeOut),
                                MPFROMSHORT(NULL));

         WinSendDlgItemMsg (hwndProps, IDC_PPA_SHAREHW,
                                BM_SETCHECK,
                                MPFROMSHORT(pPortSettings->fShareAccess),
                                MPFROMSHORT(NULL));
      }
   }
  LeaveDlgSem( pPortDlgStruct );
}

/****************************************************************************
 *
 * FUNCTION NAME = QueryDriverThread
 *
 * DESCRIPTION   = Thread to query port driver settings and post
 *                 dialog with results.
 *
 * INPUT         = pPortDlgStruct -> port dialog structure
 *
 * OUTPUT        = None
 *
 ****************************************************************************/
VOID EXPENTRY QueryDriverThread( PPORTDLGSTRUCT pPortDlgStruct )
{
   ULONG         rc;
   PPORTSETTINGS pPortSettings;
   PPORTDLGSTRUCT pOldPortDlgStruct;
   ULONG         cbBuf;
   ULONG         cbNeeded;
   ULONG         cRetries;
   BOOL          fRetry;
   BIGXRR2       RegRec;      /* exception registration record              */
   PPIB          pPib ;       /* -> process ID block */
   PTIB          pTib ;       /* -> thread  ID block */
   HWND          hwndDlg;
   CHAR          szPort[CCHMAXPATH];
   CHAR          szComputer[CCHMAXPATH];
   #ifdef DEBUG_ALERT
     char logbuf[260];
   #endif



   /*
    *  Set exception handler in case invalid pointer.
    *  When setjmp returns a nonzero number, it is returning from a longjmp
    *  [an exception has occurred]
    */
   if ( setjmp(RegRec.JBuf)) {
      /*
       * We are back here after an exception
       * Return to caller
       */
      return;
   }

   /* Get thread information block pointer
    * Save this thread's exception chain pointer because we must
    *   restore this if we handle an exception
    */
   rc = DosGetInfoBlocks(&pTib, &pPib);

   if (rc) {
       RegRec.pTib         = (PTIB)NULL ;
       RegRec.pExceptChain = (PVOID)NULL ;
   } else {
       RegRec.pTib         = pTib ;
       RegRec.pExceptChain = pTib->tib_pexchain;
   }

   RegRec.pXRR = NULL;
   RegRec.Handler = &ExceptRoutine;

   rc = DosSetExceptionHandler((PEXCEPTIONREGISTRATIONRECORD)&RegRec);

   pPortDlgStruct = ValidatePortDlg(pPortDlgStruct);
   if (pPortDlgStruct) {
      strcpy( szPort, pPortDlgStruct->pszPortName );
      if (pPortDlgStruct->pszComputerName) {
         strcpy( szComputer, pPortDlgStruct->pszComputerName );
      } else {
         szComputer[0] = '\0' ;
      }
   }
   hwndDlg = 0;
   pPortSettings = NULL;
   cbBuf  = DEFAULT_BUFSIZE ;
   fRetry = FALSE;
   do
   {
      rc = DosAllocMem ((PPVOID) &pPortSettings,
                        cbBuf,
                        PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_TILE );
      if (rc) {
         DBPRINTF ((logbuf, "QueryDriverThread DosAllocMem failed rc=%d",rc));
         break;

      } else {
         //
         // Must get around another thread initting this port @BUGBUG
         //    We will get rc=PMERR_SPL_INIT_IN_PROGRESS if
         //    another thread is in middle of initialization.
         //
         cRetries = 0;
         do
         {
           cbNeeded = cbBuf ;

           //
           // Force the port driver to re-set its cache based
           // on the current printer connected.
           //
           rc = MyPrtQuery( szComputer,
                            szPort,
                            TYPE_LONG_WAIT,
                            BIDI_Q_PORT,
                            NULL,
                            0,
                            pPortSettings,
                           &cbNeeded );
           //
           // Now issue the call to actually get the current
           // settings for the port.
           //
           cbNeeded = cbBuf ;
           rc = MyPrtQuery( szComputer,
                            szPort,
                            TYPE_LONG_WAIT,
                            BIDI_Q_PORTDRV,
                            NULL,
                            0,
                            pPortSettings,
                           &cbNeeded );

         } while ( (rc == PMERR_SPL_INIT_IN_PROGRESS) &&
                   (cRetries++ < 10) && !(DosSleep( 1000 ) ) );

         if ((rc == ERROR_MORE_DATA) && (fRetry == FALSE)) {

            fRetry = TRUE ;
            cbBuf = cbNeeded ;
            DosFreeMem( pPortSettings );
            pPortSettings = NULL ;

         } else {

            fRetry = FALSE;
         }
      }
   } while ( (rc == ERROR_MORE_DATA) && fRetry );

   if (rc)
   {
      /*
       * PrtQuery failed.
       * This will happen if using the original Warp spooler(PMSPL.DLL),
       *   so for now just call the routine to get the port settings.
       * This will open the device driver, get the settings,
       *   then close the DD.
       */
      if (!(flChangeDebugLog & FL_PDRDB_DISABLE_DIALOG))
      {
         DBPRINTF ((logbuf, "QueryDriverThread PrtQuery failed rc=%d, trying HandleQPortDRV",rc));
      }
      rc = HandleQPortDRV ( szPort,
                            pPortSettings,
                            &cbNeeded );
   }
   if (rc)
   {
      DBPRINTF ((logbuf, "QueryDriverThread HandleQPortDRV failed rc=%d",rc));
   } else {
      /*
      ** Validate port dialog structure again,
      ** then initialize Port Settings dialog structure to
      **  current port settings
      */
      pPortDlgStruct = ValidatePortDlg(pPortDlgStruct);
      if (pPortDlgStruct) {
        EnterDlgSem( pPortDlgStruct );
         if (pPortDlgStruct->pPortSettings) {

            DBPRINTF ((logbuf, "QueryDriverThread freeing old portSettings=%lX cb=%d",(ULONG)pPortDlgStruct->pPortSettings,pPortDlgStruct->cbPortSettings));
            DosFreeMem( pPortDlgStruct->pPortSettings );
         }
         pPortDlgStruct->pPortSettings    = pPortSettings;
         pPortDlgStruct->cbPortSettings   = cbBuf ;
         pPortDlgStruct->ulPrintTimeOut   = pPortSettings->ulPrintTimeOut ;
         pPortDlgStruct->ulNoQueryTimeOut = pPortSettings->ulNoQueryTimeOut ;
         pPortDlgStruct->ulNoJobTimeOut   = pPortSettings->ulNoJobTimeOut ;
         /*
         ** Post the advanced properties dialog if it is up,
         ** otherwise post the normal settings dialog
         */
         if (pPortDlgStruct->hwndAdvanced)
         {
            hwndDlg = pPortDlgStruct->hwndAdvanced;
         } else {
            hwndDlg = pPortDlgStruct->hwndDlg;
         }
        LeaveDlgSem( pPortDlgStruct );
      }
   }
   if (hwndDlg) {
      WinPostMsg( hwndDlg,
                  IDC_SETTINGS_UPDATED,
                  (MPARAM)pPortSettings,
                  (MPARAM)pPortDlgStruct);
   }

   DosUnsetExceptionHandler( (PEXCEPTIONREGISTRATIONRECORD)&RegRec );

}

#ifdef NB_DIALOGS
          pdi->hwndJobProp = WinLoadDlg( HWND_DESKTOP,
                                         WinQueryActiveWindow(HWND_DESKTOP),
                                         (PFNWP)JobPropertiesDlgProc,
                                         globals.hModule,
                                         IDD_JOB_PROPERTIES,
                                         (PVOID)pdi );

          DBPRINTIF(( bShow, "%s(): hwndJobProp = '%x'\n", __FUNCTION__, pdi->hwndJobProp ));

          CreateHelpInstance( pdi, FALSE );

          rc = WinProcessDlg( pdi->hwndJobProp );
          DBPRINTIF(( bShow, "%s(): WinProcessDlg( hwndJobProp = %x ); rc= %d\n", __FUNCTION__, pdi->hwndJobProp, rc ));

          // We must call WinDestroyWindow() ourselves for WinProcessDlg()
          // does not send it on exit as WinDlgBox() does
          DBPRINTIF(( bShow, "%s(): WinDestroyWindow( hwndJobProp = %x )\n", __FUNCTION__, pdi->hwndJobProp ));
          WinDestroyWindow( pdi->hwndJobProp );

          // If user asked to save the new job props out
          if (rc == DID_OK)

MRESULT EXPENTRY JobPropertiesDlgProc( HWND hwnd, ULONG  msg, MPARAM mp1, MPARAM mp2 )
    case WM_INITDLG:
    {
      ULONG ulDlgIdx;
      LONG  lClr;
      CHAR  szTitle[LEN_STRING];
      CHAR  szVersion[32];
      HWND  hwndTitlebar;
      ULONG cxDesktop, cyDesktop;
      RECTL rctl;

      pdi = (PDLGINSTANCE)mp2;
      // store the pointer to instance data
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)pdi );
      // Set Titlebar text (e.g. "Job Propeties: HP DeskJet 660c: Driver Version 5.67")
      WinSetWindowText ( hwnd, szTitle );
      // Fill in push buttons' text in Job Properties dialog from driver's stringtable
      WinSetDlgItemText( hwnd, DID_OK,     globals.pbStringTable[ulDlgIdx + DIALOG_TEXT_SAVE]   );
      WinSetDlgItemText( hwnd, DID_CANCEL, globals.pbStringTable[ulDlgIdx + DIALOG_TEXT_CANCEL] );
      WinSetDlgItemText( hwnd, DID_HELP,   globals.pbStringTable[ulDlgIdx + DIALOG_TEXT_HELP]   );
      pNBI = pdi->pNBI;

      pdi->pNBI->ulJobPropNBID = IDD_JOB_PROPERTIES_NB;
      pdi->pNBI->hwndJobPropNB = WinWindowFromID( hwnd, IDD_JOB_PROPERTIES_NB );

      // Select "Common Job Properties" to be topmost page
      WinSendMsg( pdi->pNBI->hwndJobPropNB,
                  BKM_TURNTOPAGE,
                  MPFROMLONG( pNBI->PageInfoJobProps.ulPageID ),
                  MPVOID );

      // Set text limit of entry field to a much larger value
      WinSendDlgItemMsg( hwnd, IDD_CONTEXT_HELP,
                         EM_SETTEXTLIMIT,
                         MPFROMLONG( MAX_CTX_HELP_STRING ),
                         MPVOID );

      // done initializing dlg box
      return( 0L );

    case HM_QUERY_KEYS_HELP:
    return (MRESULT)PANEL_JOB_PROPERTIES;

    case WM_HELP:
    {
#ifdef DEBUG
      USHORT usCmd    = SHORT1FROMMP( mp1 );
      USHORT usSource = SHORT1FROMMP( mp2 );
#endif

      pdi = (PDLGINSTANCE) WinQueryWindowULong( hwnd, QWL_USER );

      DBPRINTIF (( FALSE, "%s(); WM_HELP: usCmd=%d, usSource=%d\n", __FUNCTION__, usCmd, usSource ));

      // send help along if we have a valid help instance
      if( pdi->hwndHelp )
      {
        WinSendMsg( pdi->hwndHelp, HM_DISPLAY_HELP,
                    MPFROMSHORT( PANEL_JOB_PROPERTIES ),
                    MPFROMSHORT( HM_RESOURCEID ));
      } /* end if */

      return( 0L );

    } /* end case */

    case WM_DESTROY:
    {
      pdi = (PDLGINSTANCE) WinQueryWindowULong( hwnd, QWL_USER );
      DBPRINTIF (( FALSE, "%s(); WM_DESTROY\n", __FUNCTION__ ));

      // Check for valid pointers
      assertF( pdi );
      assertF( pdi->pDevice );

      // assign local copy of current device information
      pDevice = pdi->pDevice;

      // Reset OS/2 system SYSCLR_ACTIVETITLE back to defaults
      // WinSetSysColors( HWND_DESKTOP, LCOL_RESET, LCOLF_CONSECRGB,
      //                  SYSCLR_ACTIVETITLE, 0L, NULL );

      // free up context help linked list
      DestroyContextHelp( pdi->pHelpJobProp );

      // Delete all pages in the Job Properties notebook
      WinSendDlgItemMsg( hwnd,
                         IDD_JOB_PROPERTIES_NB,
                         BKM_DELETEPAGE,
                         0L, (MPARAM)BKA_ALL);

      // if user is displaying job props and this is not from
      // a "Device Defaults" call from printer props then we
      // have to free up user defined lists (Trays, Paper sizes etc.)
      if( !pdi->bDeviceDefaults )
      {
        // Free up linked list of user-defined resources
        FreeUserDefinedTrayList( pDevice );
        FreeUserDefinedFormList( pDevice );
        FreeUserDefinedFontList( pDevice );
        FreeConnectionList( pDevice );

        // Free up notebook information structure
        GplMemoryFree( pdi->pNBI );

      } /* end if */

    } /* end case */
    break;

    case WM_COMMAND:
    {
      // get instance data
      pdi = (PDLGINSTANCE) WinQueryWindowULong( hwnd, QWL_USER );

      switch( SHORT1FROMMP( mp1 ) )
      {
        case DID_OK:
        {

#ifdef DEBUG
          DebugOutputJobProperties( FALSE, "DRIVDATA saved from DPDM_CHANGEPROP", &pdi->JobProperties );
#endif

          SaveJobProperties( pdi, &(pdi->JobProperties) );

          // Free help manager here because it fails at WM_DESTROY
          if( !pdi->bDeviceDefaults )
          {
            DestroyHelpInstance( pdi, FALSE );
          } /* end if */

          WinDismissDlg( hwnd, DID_OK );

        } /* end case */
        break;

        case DID_CANCEL:
        {
          // Free help manager here because it fails at WM_DESTROY
          if( !pdi->bDeviceDefaults )
          {
            DestroyHelpInstance( pdi, FALSE );
          } /* end if */

          WinDismissDlg( hwnd, DID_CANCEL );
        } /* end case */
        break;

      } /* end switch */

      return(0L);

    } /* end case */

  } /* end switch */

  // default for most cases
  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end JobPropertiesDlgProc */

#endif
