/****************************************************************************/
/*  Copyright (C) 1993 IBM Corporation                                      */
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Sample BIDI Port Driver                                 */
/* FILENAME       : util.c                                                  */
/* DATE WRITTEN   : 02-08-94                                                */
/* DESCRIPTION    : Utility routines for use with structures, memory        */
/*                  management, etc.                                        */
/*                                                                          */
/****************************************************************************/

#include    "pdrconst.h"
#include    "pdrtypes.h"
#include    "pdrproto.h"

/*
** Global Variable for debugging
*/
#ifdef DEBUG_ALERT
 HFILE    hFileAlerts;      /* Handle to file to store logged data */
 BOOL     fLogAlerts=1;     /* Set to false to no longer log info  */
 BOOL     fDBterminal;      /* TRUE if sending output to PMDD.SYS  */
#endif

/*
** Defines for converting DataTime into seconds past Jan 1,1970
*/
#define JULIANJAN 0
#define JULIANFEB 31
#define JULIANMAR (JULIANFEB + 28)
#define JULIANAPR (JULIANMAR + 31)
#define JULIANMAY (JULIANAPR + 30)
#define JULIANJUN (JULIANMAY + 31)
#define JULIANJUL (JULIANJUN + 30)
#define JULIANAUG (JULIANJUL + 31)
#define JULIANSEP (JULIANAUG + 31)
#define JULIANOCT (JULIANSEP + 30)
#define JULIANNOV (JULIANOCT + 31)
#define JULIANDEC (JULIANNOV + 30)
#define FEB     2
#define SECPERMIN 60
#define SECPERHOUR (SECPERMIN*60)
#define SECPERDAY (SECPERHOUR*24)
#define SECPERYEAR (SECPERDAY*365)
#define STARTYEAR 1970


const ULONG JulianMonth[] = { 0,
                              JULIANJAN,
                              JULIANFEB,
                              JULIANMAR,
                              JULIANAPR,
                              JULIANMAY,
                              JULIANJUN,
                              JULIANJUL,
                              JULIANAUG,
                              JULIANSEP,
                              JULIANOCT,
                              JULIANNOV,
                              JULIANDEC };


/*
** Local Functions
*/

/****************************************************************************
 *
 * FUNCTION NAME = AllocPdrMem
 *
 * DESCRIPTION   = Allocate memory from port driver heap
 *
 * INPUT         = cb  - Amount of memory to allocate in bytes
 *
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to allocated memory
 *
 * RETURN-ERROR  = NULL(Failure, memory could not be allocated)
 *
 ****************************************************************************/

PVOID AllocPdrMem ( ULONG cb ) {

    PVOID pMem = NULL;
    ULONG rc = 0;

    rc = DosSubAllocMem ( pHeapBase, &pMem, cb );
    if (rc) {
       PdrError ( 1 , 1 );
       return NULL;
    } else {
       return(pMem);
    }

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePdrMem
 *
 * DESCRIPTION   = Free memory from the port driver heap
 *
 * INPUT         = pMem - Pointer to the start of the memory block
 *               = cb - Size of the memory block to free
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0, Successful
 *
 * RETURN-ERROR  = rc, Failure
 *
 ****************************************************************************/

ULONG FreePdrMem ( PVOID pMem, ULONG cb ) {

    ULONG rc = 0;
    PFREEMEM    pFreeMem = (PFREEMEM) pMem;

    /*
     * Set signature and size [For debugging/validation]
     */
    if (cb >= sizeof(USHORT)) {
        pFreeMem->signature = FR_SIGNATURE;
    }
    if (cb >= (sizeof(USHORT)*2)) {
        pFreeMem->cb = (USHORT) cb;
    }
    /*
     * Free the memory
     */
    rc = DosSubFreeMem ( pHeapBase, pMem, cb );
    if (rc) {
        PdrPanic ( 2 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = EnterPdrSem
 *
 * DESCRIPTION   = Enter the port driver semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosRequestMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be requested)
 *
 ****************************************************************************/

ULONG EnterPdrSem ( VOID ) {

    ULONG rc = 0;

    rc = DosRequestMutexSem ( semPortDrv , SEM_INDEFINITE_WAIT );
    if (rc) {
        PdrError ( 3 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = LeavePdrSem
 *
 * DESCRIPTION   = Leave the protocol converter semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosReleaseMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 ****************************************************************************/

ULONG LeavePdrSem ( VOID ) {

    ULONG rc = 0;

    rc = DosReleaseMutexSem ( semPortDrv );
    if (rc) {
        PdrError ( 4 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPdOpenInst
 *
 * DESCRIPTION   = Add new port instance to the list in the port driver
 *
 * INPUT         = pPdOpenInst       - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG AddPdOpenInst ( PPDOPENINST pPdOpenInst ) {

    if (pPdOpenInstList) {
        pPdOpenInst->pNext = pPdOpenInstList;
    }
    pPdOpenInstList = pPdOpenInst;

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = RemovePdOpenInst
 *
 * DESCRIPTION   = Remove port instance from the list in the port driver
 *
 * INPUT         = pPdOpenInst       - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG RemovePdOpenInst ( PPDOPENINST pPdOpenInst ) {

    PPDOPENINST pPdCurr = pPdOpenInstList;
    PPDOPENINST pPdPrev = NULL;

    while (pPdCurr) {
        if (pPdCurr == pPdOpenInst) {
            if (pPdPrev) {
                pPdPrev->pNext = pPdCurr->pNext;
            } else {
                pPdOpenInstList = pPdCurr->pNext;
            }
            return(0);
        }
        pPdPrev = pPdCurr;
        pPdCurr = pPdCurr->pNext;
    }

    return(ERROR_INVALID_HANDLE);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPortInst
 *
 * DESCRIPTION   = Add new port instance to the list in the port driver
 *
 * INPUT         = pszPortName  - Name of the port to add
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, port could not be added)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

PPORTINST AddPortInst ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;

    /*
     * See if port has been added previously
     */
    pPortInst = FindPortInst ( pszPortName );
    if ( pPortInst ) {
        return( pPortInst );
    }
    /*
     * Allocate a new port structure
     */
    if (!(pPortInst = NewPortInst ( pszPortName ))) {
        return(NULL);
    }
    /*
     * If port list exists, add to front of list
     */
    if (pPortInstList) {
        pPortInst->pNext = pPortInstList;
    }
    pPortInstList = pPortInst;

    return(pPortInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = NewPortInst
 *
 * DESCRIPTION   = Allocate new port instance
 *
 * INPUT         = pszPortName  - Name of the new port
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to port instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

PPORTINST NewPortInst ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;
    ULONG     cb;
    ULONG     rc = 0;

    cb = sizeof( PORTINST ) + strlen ( pszPortName ) + 1;
    if (!(pPortInst = AllocPdrMem( cb ))) {
        return(NULL);
    }
    memset ( pPortInst, 0, cb );
    pPortInst->signature = PT_SIGNATURE;
    pPortInst->cb = cb;
    pPortInst->ulNoQueryTimeOut = DEF_TIMEOUT_QUERY_VALUE;
    pPortInst->ulNoJobTimeOut   = DEF_TIMEOUT_JOB_VALUE;
    pPortInst->pszPortName = (PSZ)((PBYTE)pPortInst + sizeof(PORTINST));
    strcpy ( pPortInst->pszPortName, pszPortName );
    //
    // Make LMParms part point back to beginning of structure
    //
    pPortInst->LMParms.pPortInst = pPortInst;
    rc = DosCreateMutexSem ( 0,
                             &(pPortInst->hPortSem),
                             0,
                             FALSE );       // Create Unowned
    if (rc) {
        FreePdrMem ( pPortInst, cb );
        return(NULL);
    }

    return(pPortInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = FindPortInst
 *
 * DESCRIPTION   = Find port instance in the port driver
 *
 * INPUT         = pszPortName   -  Name of port to find
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to port instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

PPORTINST FindPortInst ( PSZ pszPortName ) {

    PPORTINST pPortCurr = pPortInstList;

    while (pPortCurr) {
        if (!strcmp(pPortCurr->pszPortName, pszPortName)) {
            return(pPortCurr);
        }
        pPortCurr = pPortCurr->pNext;
    }

    return(NULL);

}

/****************************************************************************
 *
 * FUNCTION NAME = RemovePortInst
 *
 * DESCRIPTION   = Remove port instance from the list in the port driver
 *
 * INPUT         = pszPortName   -  Name of port to remove
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG RemovePortInst ( PSZ pszPortName ) {

    PPORTINST pPortCurr = pPortInstList;
    PPORTINST pPortPrev = NULL;

    while (pPortCurr) {
        if (!strcmp(pPortCurr->pszPortName, pszPortName)) {
            if (pPortPrev) {
                pPortPrev->pNext = pPortCurr->pNext;
            } else {
                pPortInstList = pPortCurr->pNext;
            }
            /*
            ** For now, we will not actually remove the port instance
            **  from memory, just remove it from our list.
            ** This is done because some of our threads might not
            **  be validating the port instance structure all the time.
            ** This is just a precaution to avoid traps;  if the
            **  code proves to be very good at revalidating pPortInst
            **  structures then we can free up this memory here.
            */
            pPortCurr->pNext = NULL;
            // FreePortInst ( pPortCurr );
            return(0);
        }
        pPortPrev = pPortCurr;
        pPortCurr = pPortCurr->pNext;
    }

    return(ERROR_INVALID_HANDLE);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePortInst
 *
 * DESCRIPTION   = Free port instance memory
 *
 * INPUT         = pPortInst     -  Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0 (Successful)
 *
 * RETURN-ERROR  = rc (Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG FreePortInst ( PPORTINST pPortInst ) {

    ULONG rc = 0;

    /*
     * Close port semaphore
     */
    rc = DosCloseMutexSem ( pPortInst->hPortSem );
    FreePdrMem ( pPortInst, pPortInst->cb );

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreeAllPortInst
 *
 * DESCRIPTION   = Free port instance memory list
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0 (Successful)
 *
 * RETURN-ERROR  = rc (Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG FreeAllPortInst ( VOID ) {

    ULONG rc = 0;
    PPORTINST pPortInst = pPortInstList;
    PPORTINST pPortNext = NULL;

    while (pPortInst) {
        pPortNext = pPortInst->pNext;
        rc = FreePortInst ( pPortInst );
        pPortInst = pPortNext;
    }

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = ValidatePdOpenInst
 *
 * DESCRIPTION   = Validate port instance from the list in the port driver
 *
 * INPUT         = pPdOpenInst       - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 ****************************************************************************/

PPDOPENINST ValidatePdOpenInst ( PPDOPENINST pPdOpenInst ) {

    PPDOPENINST pPdCurr = NULL;

  EnterPdrSem();

    pPdCurr = pPdOpenInstList;
    while (pPdCurr) {
        if (pPdCurr == pPdOpenInst) {
          LeavePdrSem();
            return(pPdCurr);
        }
        pPdCurr = pPdCurr->pNext;
    }

  LeavePdrSem();
    return(NULL);

}

/****************************************************************************
 *
 * FUNCTION NAME = PdrError
 *
 * DESCRIPTION   = Routine called on a error condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID PdrError ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = PdrPanic
 *
 * DESCRIPTION   = Routine called on a panic condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID PdrPanic ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = PdrWarning
 *
 * DESCRIPTION   = Routine called on a warning condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID PdrWarning ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = GetConnectionName
 *
 * DESCRIPTION   = Retrieve default connection name from system INI
 *
 * INPUT         = pszPort     - name of infrared port
 *               = pszBuf      - receives connectionname or set to null string

 *
 * RETURN-NORMAL = TRUE  - connection name found in INI file
 *
 * RETURN-ERROR  = FALSE - no connection name in INI file
 *
 ****************************************************************************/
BOOL GetConnectionName( PSZ pszPort,  PSZ pszBuf )
{
   BOOL     fSuccess;
   CHAR     chBuf[CCHMAXPATH];


   strcpy (chBuf, APPNAME_LEAD_STR);
   strcat (chBuf, pszPort);

   *pszBuf = '\0';

   fSuccess = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                     chBuf,
                                     KEY_CONNECTION,
                                     NULL,
                                     (PSZ)pszBuf,
                                     MAX_CONNECTNAME_SZ);
   return(fSuccess);
}


/****************************************************************************
 *
 * FUNCTION NAME = GetTimeOuts
 *
 * DESCRIPTION   = Retrieve timeouts for printer from system INI
 *
 * INPUT         = pszPort     -> name of infrared port
 *               = pPortInst   -> PORTINST structure to receive timeout values

 *
 * RETURN-NORMAL = TRUE  - timeouts found
 *
 * RETURN-ERROR  = FALSE - timeouts not in INI, defaults used
 *
 * NOTE          = NOT in PDR Sem, but updates pPortInst timeouts
 *                 anyway.
 *
 *
 *
 ****************************************************************************/
BOOL GetTimeOuts( PSZ pszPort, PPORTINST pPortInst )
{
   BOOL     fSuccess;
   CHAR     chAppName[CCHMAXPATH];
   CHAR     chBuf[CCHMAXPATH];
   PSZ      pszSemi;
   ULONG    ulTimeOut;


   strcpy (chAppName, APPNAME_LEAD_STR);
   strcat (chAppName, pszPort);

   ulTimeOut = 0;
   fSuccess = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                     chAppName,
                                     KEY_TIMEOUT_PRINT,
                                     NULL,
                                     chBuf,
                                     10 );
   if (fSuccess)
   {
     /*
     ** Remove trailing semi-colon
     */
     pszSemi = strchr( chBuf, ';' );
     if (pszSemi)
     {
        *pszSemi = '\0';
     }
     ulTimeOut = atoi(chBuf);
   }
   if (!ulTimeOut)
   {
     ulTimeOut = DEF_TIMEOUT_PRINT_VALUE;
   }
   pPortInst->ulPrintTimeOut = ulTimeOut;

   ulTimeOut = 0;
   fSuccess = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                     chAppName,
                                     KEY_TIMEOUT_QUERY,
                                     NULL,
                                     chBuf,
                                     10 );
   if (fSuccess)
   {
     /*
     ** Remove trailing semi-colon
     */
     pszSemi = strchr( chBuf, ';' );
     if (pszSemi)
     {
        *pszSemi = '\0';
     }
     ulTimeOut = atoi(chBuf);
   }
   if (!ulTimeOut)
   {
     ulTimeOut = DEF_TIMEOUT_QUERY_VALUE;
   }
   pPortInst->ulNoQueryTimeOut = ulTimeOut;

   ulTimeOut = 0;
   fSuccess = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                     chAppName,
                                     KEY_TIMEOUT_JOB,
                                     NULL,
                                     chBuf,
                                     10 );
   if (fSuccess)
   {
     /*
     ** Remove trailing semi-colon
     */
     pszSemi = strchr( chBuf, ';' );
     if (pszSemi)
     {
        *pszSemi = '\0';
     }
     ulTimeOut = atoi(chBuf);
   }
   if (!ulTimeOut)
   {
     ulTimeOut = DEF_TIMEOUT_JOB_VALUE;
   }
   pPortInst->ulNoJobTimeOut = ulTimeOut;

   return(fSuccess);
}


/****************************************************************************
 *
 * FUNCTION NAME = FormatDateTime
 *
 * DESCRIPTION   = Format the date and time into the given buffer in
 *                  the following format:
 *                    month-day hours:minutes:seconds
 *                  All fields will be 2 numbers(zero-preceded) and
 *                    the string will have 1 trailing blank
 *
 *                 Used by LogCall for debug purposes
 *
 * INPUT         = psz         - string buffer to get null terminated
 *                               formatted date and time
 * OUTPUT        = none        -
 *
 ****************************************************************************/
#ifdef DEBUG_ALERT

VOID FormatDateTime ( PSZ psz ) {

    DATETIME    DateTime;


    DosGetDateTime( &DateTime );
    psz[0]  = (DateTime.month / 10) + '0' ;
    psz[1]  = (DateTime.month % 10) + '0' ;
    psz[2]  = '-' ;
    psz[3]  = (DateTime.day   / 10) + '0' ;
    psz[4]  = (DateTime.day   % 10) + '0' ;
    psz[5]  = ' ' ;

    psz[6]  = (DateTime.hours / 10) + '0' ;
    psz[7]  = (DateTime.hours % 10) + '0' ;
    psz[8]  = ':' ;
    psz[9]  = (DateTime.minutes   / 10) + '0' ;
    psz[10] = (DateTime.minutes   % 10) + '0' ;
    psz[11] = ':' ;
    psz[12] = (DateTime.seconds   / 10) + '0' ;
    psz[13] = (DateTime.seconds   % 10) + '0' ;
    psz[14] = ' ' ;
    psz[15] = '\0' ;

}

/****************************************************************************
 *
 * FUNCTION NAME = ShowHexByte
 *
 * DESCRIPTION   = Convert hex number to string
 *                 Used by DumpHex for debug purposes
 *
 * INPUT         = pszBuf    - buffer area to get converted number
 *                             This does NOT get null terminated!
 *                 bValue    - byte to format
 *
 * OUTPUT        = NONE
 *
 ****************************************************************************/
VOID ShowHexByte ( PSZ pszBuf, BYTE bValue )
{
    USHORT uValue;

    uValue = (USHORT)( bValue / 16 );

    if (uValue > 9)
       pszBuf[0] = (char) ((uValue - 10) + 'A');
    else
       pszBuf[0] = (char) (uValue + '0');

    uValue = (USHORT)( bValue % 16 );

    if (uValue > 9)
       pszBuf[1] = (char) ((uValue - 10) + 'A');
    else
       pszBuf[1] = (char) (uValue + '0');
}

/****************************************************************************
 *
 * FUNCTION NAME = ShowHexAscii
 *
 * DESCRIPTION   = Display hex byte as its Ascii value or as '.' if not valid
 *                 Used by DumpHex for debug purposes
 *
 * INPUT         = pszBuf    - buffer area to get converted number
 *                             This does NOT get null terminated!
 *                 bValue    - byte to format
 *
 * OUTPUT        = NONE
 *
 ****************************************************************************/
VOID ShowHexAscii ( PSZ pszBuf, BYTE bValue )
{

    if ( (bValue >= ' ') && (bValue <= '~') )
       pszBuf[0] = (char) (bValue);
    else
       pszBuf[0] = '.';
}

/****************************************************************************
 *
 * FUNCTION NAME = DumpHex
 *
 * DESCRIPTION   = Output hex/ascii representation of buffer into
 *                 log file
 *
 * INPUT         = pbHexChars  - buffer containing data to format
 *                 cb          - amount of data to format
 *
 * OUTPUT        = None
 *
 ****************************************************************************/
VOID DumpHex ( PBYTE pbHexChars, ULONG cb )
{
    CHAR   logbuf[70];
    ULONG  i;
    ULONG  numLines;
    ULONG  curLine;
    ULONG  charsLeftInLine;
    PBYTE  pbBytes;

    if (!cb) {
       return;
    }
    numLines = (cb / 16);

    if (cb % 16) {
      numLines++;
    }
    for (curLine = 1; curLine <= numLines; curLine++)
    {
       if (curLine < numLines) {
          charsLeftInLine = 16;
       } else {
          charsLeftInLine = cb % 16;
          if (charsLeftInLine == 0)
             charsLeftInLine = 16;
       }
       memset( logbuf, ' ', 70 );
       pbBytes = pbHexChars + ( (curLine-1) * 16 );
       for (i=0; i < charsLeftInLine; i++)
       {
          //
          // Create a line with following format:
          //
          // 00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF  0123456789ABCDEF
          //
          // Where double numbers are hex values and
          //       single numbers are ascii representation of hex byte,
          //              or "." if not printable
          //
          ShowHexByte ( logbuf+(3*i), pbBytes[i] );
          ShowHexAscii( logbuf+48+i,  pbBytes[i] );

       }
       logbuf[48+16] = '\r';
       logbuf[49+16] = '\n';
       logbuf[50+16] = '\0';
       LogCall( logbuf );
    }
}

/****************************************************************************
 *
 * FUNCTION NAME = LogCall    - DEBUG API
 *
 * DESCRIPTION   = Log msg describing Bidi call in debug file C:\\ALERTIR
 *
 * INPUT         = pszMsg    - -> null terminated msg to log.
 *                                This can have newline characters.
 *
 * OUTPUT        = none
 *
 * NOTE          = Need not be in PdrSem on entry
 *
 ****************************************************************************/
VOID LogCall( PSZ pszMsg )
{

    ULONG       ActionTaken;
    ULONG       ulOpenAction;
    ULONG       cWritten;
    ULONG       rc ;
    CHAR        buf2[260];
    PPIB        ppib;
    PTIB        ptib;


    if (fLogAlerts)
    {
      /*
       * Get Local Process Info
       */
      DosGetInfoBlocks ( &ptib, &ppib );

      if ( hFileAlerts == 0)
      {
        ulOpenAction = OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW ;
        //
        // Allow setting the debug IR alerts filename.
        // Default to debug console if not specified( "DB" )
        //
        buf2[0] = '\0';
        if (!PrfQueryProfileString(HINI_SYSTEMPROFILE,
                                   "PM_SPOOLER_DEBUG",
                                   "ALERTSFILE",
                                   (PSZ)NULL,
                                   buf2,
                                   sizeof(buf2) - 1) )
        {
           strcpy( buf2, "DB" );
        }
        //
        // Special filename "DB" makes us send to debug terminal
        //   using PMDD.SYS( need /C1 in config.sys )
        //
        if (strcmp(buf2, "DB") == 0 )
        {
           fDBterminal = TRUE;
           /**********************************/
           /* Open PMDD Device Driver Output */
           /**********************************/
           rc = DosOpen ("\\DEV\\SINGLEQ$",          /* File path name          */
                         &hFileAlerts,               /* File handle             */
                         &ActionTaken,               /* Action taken            */
                         0,                          /* File primary allocation */
                         FILE_NORMAL,                /* File attribute          */
                         OPEN_ACTION_OPEN_IF_EXISTS, /* Open function type      */
                         OPEN_FLAGS_NO_LOCALITY      /* Open mode of the file   */
                         | OPEN_FLAGS_NOINHERIT
                         | OPEN_SHARE_DENYNONE
                         | OPEN_ACCESS_READONLY,
                         NULL);                      /* Ptr to EA buffer        */
        } else {

           fDBterminal = FALSE;
           rc = DosOpen( buf2, &hFileAlerts, &ActionTaken,
                        0, 0,
                        ulOpenAction,
                        OPEN_FLAGS_NONSPOOLED | OPEN_FLAGS_FAIL_ON_ERROR |
                        OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYNONE,
                        NULL);
        }
        if (rc)
        {
          hFileAlerts = (ULONG)-1 ;
        }
      }
      if (hFileAlerts != (ULONG)-1)
      {
           FormatDateTime( buf2 );

           strcat( buf2, pszMsg );
           //
           // Re-use callers buffer - we know it is scratch buffer @BUGBUG
           //
           sprintf( buf2+strlen(buf2)-2, " PID=%d TID=%d\r\n",
                    ppib->pib_ulpid, ptib->tib_ordinal );

           if (fDBterminal)
           {
              //
              // Must use IOCtl to send to PMDD.SYS
              //
              ULONG  cbParmInOut = 0;
              ULONG  cbDataInOut = 0;

              DosDevIOCtl (hFileAlerts,       /* Device:   PMDD            */
                           0x03,              /* Catagory: SCRN/PTR DRAW   */
                           0x77,              /* Function: DEBUG OUTPUT    */
                           buf2,              /* Ptr to parms              */
                           strlen(buf2),      /* Parms Max Length          */
                           &cbParmInOut,      /* Ptr to Parm Length In/Out */
                           NULL,              /* Ptr to Data Area          */
                           0L,                /* Data Max Length           */
                           &cbDataInOut);     /* Ptr of Data Length In/Out */
           } else {

              /*
               * Append to file, don't overwrite other App's log entries
               */
              DosSetFilePtr( hFileAlerts, 0, FILE_END, &cWritten );
              DosWrite( hFileAlerts, buf2, strlen(buf2), &cWritten );
           }
      }
    } else {
      if ( (hFileAlerts != 0) && (hFileAlerts != (ULONG)-1) )
      {
         DosClose( hFileAlerts );
         hFileAlerts = 0;
      }

    }
}
#else
VOID LogCall( PSZ pszMsg )
{
}
VOID DumpHex ( PSZ pszHexChars, ULONG cb )
{
}
#endif /* DEBUG_ALERT */


/****************************************************************************
 *
 * FUNCTION NAME = JulianDate
 *
 * DESCRIPTION   = Convert day, month to the Julian Date
 *
 * INPUT         = day - Current day of the month
 *               = month - Current month of the year
 *
 * OUTPUT        = Calculates the Julian Date using an array of monthly offsets
 *
 * RETURN-NORMAL = Results of the calculation
 *
 * RETURN-ERROR  = 0
 *
 ****************************************************************************/

ULONG JulianDate (UCHAR day, UCHAR month, USHORT year) {

    ULONG ulDay = 0;

    ulDay += (ULONG) day + JulianMonth [ month ];

    // Check for leap year and add one day if necessary

    if ((((year % 4) == 0) && ((year % 100) != 0)) && (month > FEB)) {
        ulDay++;
    } /* endif */
    return (ulDay);

}

/****************************************************************************
 *
 * FUNCTION NAME = time
 *
 * DESCRIPTION   = returns the number of seconds past Jan 1, 1970
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosGetDateTime and converts the results to seconds
 *
 * RETURN-NORMAL = number of seconds past Jan 1, 1970
 *
 * RETURN-ERROR  = 0
 *
 ****************************************************************************/

ULONG time ( VOID ) {

    DATETIME DateTime;
    ULONG    ulTime;

    DosGetDateTime ( &DateTime );
    ulTime = DateTime.seconds;
    ulTime += (DateTime.minutes * SECPERMIN);
    ulTime += (DateTime.hours * SECPERHOUR);

    // Determine the Julian Date for the day, subtract 1 since you still are part
    // of that day

    ulTime += ((JulianDate (DateTime.day, DateTime.month, DateTime.year) - 1)
                 * SECPERDAY);

    // Determine interval in years from the start year

    ulTime += ((DateTime.year - STARTYEAR) * SECPERYEAR);

    // Check for the year 2000, it is not a leap year

    if (DateTime.year < 2000) {
        ulTime += ((((DateTime.year - STARTYEAR) + 2) / 4) * SECPERDAY);
    } else {
        ulTime += (((((DateTime.year - STARTYEAR) + 2) / 4) - 1) * SECPERDAY);
    }
    return ulTime;

}

/****************************************************************************
 *
 * FUNCTION NAME = GenerateUniquePort
 *
 * DESCRIPTION   = Generate a port name that is not yet in use
 *
 * INPUT         = pszPortName - buffer containing the default portname
 *                               This gets an ascii number appended to it
 *                               to make it a unique portname
 *
 * OUTPUT        = pszPortName gets updated
 *
 * RETURN-NORMAL = TRUE  - able to generate unique name
 *
 * RETURN-ERROR  = FALSE - unable to create unique port name
 *
 ****************************************************************************/

BOOL  GenerateUniquePortName( PSZ pszPortName )
{
    BOOL   fPortExists;
    PSZ    pszEndPortName;
    USHORT i;
    CHAR   chPortData[20];

    /*
    ** Generate a unique port name by adding numbers to the
    **   end of pszPortName
    ** We arbitrarily pick 50 as the max number of infrared
    **   ports you can install.  This can be changed.
    */
    pszEndPortName = pszPortName + strlen( pszPortName );
    i = 1;
    fPortExists = TRUE;
    while ( (i < 50) && fPortExists )
    {
       _itoa( i, pszEndPortName, 4);
       fPortExists = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                            APPNAME_PM_SPOOLER_PORT,
                                            pszPortName,
                                            NULL,
                                            chPortData,
                                            sizeof(chPortData) - 1);
       i++;
    }
    return(!fPortExists);
}

/*
** The following functions are for functions only exported by
**   the bidi spooler(PMSPL.DLL) available in a fixpak.
** These functions are not exported by the original Warp spooler,
**   so this port driver calls MyXXXX API instead of directly
**   calling an API in PMSPL.DLL that might not exist.
** Global(per-process) variable fLoadedPMSPL is set if we have
**   checked for the following APIs in PMSPL.DLL:
**   PrtQuery PrtSet SplProtSendCmd SplProtXlateCmd
*/
VOID LoadPMSPL( VOID )
{
   ULONG rc;
   HMODULE  hmod;
   CHAR  szFailName[80];

   rc = DosLoadModule( szFailName, 80, "PMSPL", &hmod ) ;
   if (rc == 0)
   {
     /*
      * Ordinal for PrtQuery is 603 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 603, NULL, (PFN *)&pfnPrtQuery ) ;
     /*
      * Ordinal for PrtSet is 604 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 604, NULL, (PFN *)&pfnPrtSet ) ;
     /*
      * Ordinal for SplProtSendCmd is 605 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 605, NULL, (PFN *)&pfnProtSendCmd ) ;
     /*
      * Ordinal for SplProtXlateCmd is 606 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 606, NULL, (PFN *)&pfnProtXlateCmd ) ;
   }
   fLoadedPMSPL = TRUE;
}

ULONG  MyPrtQuery ( PSZ    pszComputerName,
                    PSZ    pszDeviceName,
                    ULONG  ulType,
                    ULONG  ulCommand,
                    PVOID  pInData,
                    ULONG  cbInData,
                    PVOID  pOutData,
                    PULONG pcbOutData )
{
   ULONG rc;

   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnPrtQuery)
   {
      rc = (*pfnPrtQuery)( pszComputerName,
                           pszDeviceName,
                           ulType,
                           ulCommand,
                           pInData,
                           cbInData,
                           pOutData,
                           pcbOutData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}


ULONG MyPrtSet   ( PSZ    pszComputerName,
                   PSZ    pszDeviceName,
                   ULONG  ulType,
                   ULONG  ulCommand,
                   PVOID  pInData,
                   ULONG  cbInData )
{
   ULONG rc;


   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnPrtSet)
   {
      rc = (*pfnPrtSet)( pszComputerName,
                         pszDeviceName,
                         ulType,
                         ulCommand,
                         pInData,
                         cbInData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}

ULONG MySplProtSendCmd( PSZ    pszPortName,
                        ULONG  ulType,
                        ULONG  ulCommand,
                        PFN    pfnPdSendCmd,
                        PFN    pfnBaseProtSendCmd,
                        PVOID  pInData,
                        ULONG  cbInData,
                        PVOID  pOutData,
                        PULONG pcbOutData )
{
   ULONG rc;


   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnProtSendCmd)
   {
      rc = (*pfnProtSendCmd)( pszPortName,
                              ulType,
                              ulCommand,
                              pfnPdSendCmd,
                              pfnBaseProtSendCmd,
                              pInData,
                              cbInData,
                              pOutData,
                              pcbOutData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}

ULONG MySplProtXlateCmd( PSZ    pszPortName,
                         PFN    pfnBaseProtXlateCmd,
                         PVOID  pInData,
                         ULONG  cbInData,
                         PVOID  pAlertInfo,
                         PVOID  pOutData,
                         PULONG pcbOutData )
{
   ULONG rc;


   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnProtXlateCmd)
   {
      rc = (*pfnProtXlateCmd)( pszPortName,
                               pfnBaseProtXlateCmd,
                               pInData,
                               cbInData,
                               pAlertInfo,
                               pOutData,
                               pcbOutData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = AddDataInfoBuf
 *
 * DESCRIPTION   = Put DATAINFOBUF structure onto a list.
 *                 The list can be the WaitingConfirm or Confirmed list.
 *
 * INPUT         = ppHead    -> location holding pointer to head of list
 *                 pNewItem  -> new DATAINFO structure to add to list
 *
 * OUTPUT        = None
 *
 * NOTES         = Must be in PdrSem on entry
 *
 ****************************************************************************/

VOID AddDataInfoBuf ( PDATAINFO *ppHead, PDATAINFO pNewItem )
{
    PDATAINFO pLast;

    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif

    #ifdef DEBUG
     {
       sprintf( logbuf, "AddDataInfoBuf head=%lX, New=%lX, pHead=%lX\r\n",
                (ULONG)*ppHead, (ULONG)pNewItem, (ULONG)ppHead );
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    pNewItem->pNext = NULL;

    if (*ppHead) {
       pLast = *ppHead;
       while (pLast->pNext)
       {
          pLast = pLast->pNext;
       }
       pLast->pNext = pNewItem;
    } else {
       *ppHead = pNewItem;
    }

} /* end AddDataInfoBuf */

/****************************************************************************
 *
 * FUNCTION NAME = RemoveDataInfoBuf
 *
 * DESCRIPTION   = Remove DATAINFOBUF structure from a list.
 *                 The list can be the WaitingConfirm or Confirmed list.
 *
 * INPUT         = ppHead    -> location holding pointer to head of list
 *                 pOldItem  -> DATAINFO structure to remove from list
 *
 * OUTPUT        = TRUE  - OldItem was on LIST, else FALSE
 *
 * NOTES         = Must be in PdrSem on entry
 *
 ****************************************************************************/

BOOL RemoveDataInfoBuf ( PDATAINFO *ppHead, PDATAINFO pOldItem )
{
    PDATAINFO pLast;
    BOOL      fFound;

    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif

    #ifdef DEBUG
     {
       sprintf( logbuf, "RemoveDataInfoBuf head=%lX, Old=%lX, pHead=%lX\r\n",
                (ULONG)*ppHead, (ULONG)pOldItem, (ULONG)ppHead );
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    fFound = FALSE;

    if (*ppHead) {
       if (*ppHead == pOldItem) {
          *ppHead = pOldItem->pNext;
          fFound = TRUE;
       } else {

          pLast = *ppHead;
          while ( pLast && !fFound )
          {
             if (pLast->pNext == pOldItem) {
                fFound = TRUE;
             } else {
                pLast = pLast->pNext;
             }
          }
          if (fFound) {
             pLast->pNext = pOldItem->pNext;
          } else {
             #ifdef DEBUG
              {
                sprintf( logbuf, "RemoveDataInfoBuf NOT IN LIST!\r\n");
                LogCall( logbuf );
              }
             #endif /* DEBUG */
          }
       }

    } else {
       #ifdef DEBUG
        {
          sprintf( logbuf, "RemoveDataInfoBuf NOT IN LIST!\r\n");
          LogCall( logbuf );
        }
       #endif /* DEBUG */
    }
    #ifdef DEBUG
     {
       sprintf( logbuf, "RemoveDataInfoBuf found=%d NewHead=%lX\r\n",
                fFound, (ULONG)*ppHead );
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    return(fFound);

} /* end RemoveDataInfoBuf */

/****************************************************************************
 *
 * FUNCTION NAME = FreeConfirmedBufs
 *
 * DESCRIPTION   = Return count of bytes confirmed by printer
 *                 by removing items on the pLMParms->pConfirmedBufs list
 *
 * INPUT         = ppHead    -> location holding pointer to head of list
 *
 * OUTPUT        = cbConfirmed - sum of data in all confirmed buffers.
 *
 * NOTES         = Must be in PdrSem on entry, NULLs *ppHead
 *
 ****************************************************************************/

ULONG FreeConfirmedBufs ( PDATAINFO *ppHead )
{
    PDATAINFO pLast;
    PDATAINFO pTemp;
    ULONG     cbConfirmed;

    #ifdef DEBUG
     CHAR  logbuf[260];            // For calling LogCall
     ULONG cBuffers=0;
    #endif

    #ifdef DEBUG
     {
       sprintf( logbuf, "FreeConfirmedBufs head=%lX\r\n",
                (ULONG)*ppHead );
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    cbConfirmed = 0;
    pLast = *ppHead;
    *ppHead = NULL;

    while (pLast)
    {
       cbConfirmed += pLast->ulDataSize;
       #ifdef DEBUG
        {
          cBuffers++;
          sprintf( logbuf, "FreeConfirmedBufs cBufs=%d cbConfirmed=%d freeing=%lX\r\n",
                   cBuffers, cbConfirmed, (ULONG)pLast );
          LogCall( logbuf );
        }
       #endif /* DEBUG */
       pTemp = pLast;
       pLast = pLast->pNext;
       DosFreeMem( (PVOID)pTemp );
    }

    return(cbConfirmed);
}


/****************************************************************************
 *
 * FUNCTION NAME = AddPortDlgStruct
 *
 * DESCRIPTION   = Add PORTDLGSTRUCT to list of open dialogs.  If found,
 *                 return the structure from the list.
 *
 * INPUT         = pPortDlgStruct - Pointer to structure contain the
 *                                  computer name and the port name of the
 *                                  PORTDLGSTRUCT to find in the list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = pOldPortDlgStruct - Pointer to PORTDLGSTRUCT from list
 *
 * RETURN-ERROR  = NULL - No PORTDLGSTRUCT found
 *
 ****************************************************************************/

BOOL AddPortDlgStruct ( PPORTDLGSTRUCT pPortDlgStruct ) {

    /*
     * Make sure we have a structure to add
     */
    if (pPortDlgStruct) {
       EnterPdrSem();
        pPortDlgStruct->pNext = pOpenDlgList;
        pOpenDlgList = pPortDlgStruct;
       LeavePdrSem();
    }

    return(TRUE);

} /* end AddPortDlgStruct */

/****************************************************************************
 *
 * FUNCTION NAME = RemovePortDlgStruct
 *
 * DESCRIPTION   = Add PORTDLGSTRUCT to list of open dialogs.  If found,
 *                 return the structure from the list.
 *
 * INPUT         = pPortDlgStruct - Pointer to structure contain the
 *                                  computer name and the port name of the
 *                                  PORTDLGSTRUCT to find in the list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = pOldPortDlgStruct - Pointer to PORTDLGSTRUCT from list
 *
 * RETURN-ERROR  = NULL - No PORTDLGSTRUCT found
 *
 ****************************************************************************/

BOOL RemovePortDlgStruct ( PPORTDLGSTRUCT pPortDlgStruct ) {

    PPORTDLGSTRUCT pOldPortDlgStruct = NULL;
    PPORTDLGSTRUCT pPrevPortDlgStruct = NULL;
    BOOL fRemoved = FALSE;

    /*
     * Make sure we have a structure to remove
     */
    if (pPortDlgStruct) {
       EnterPdrSem();
        pOldPortDlgStruct = pOpenDlgList;
        /*
         * Go through the list and remove structure if found
         */
        while (pOldPortDlgStruct) {
            if (pOldPortDlgStruct == pPortDlgStruct) {
                /*
                 * If no previous structure, then it is the head of the list
                 */
                if (pPrevPortDlgStruct) {
                    pPrevPortDlgStruct->pNext = pPortDlgStruct->pNext;
                } else {
                    pOpenDlgList = pPortDlgStruct->pNext;
                }
                fRemoved = TRUE;
                break;
            }
        }
       LeavePdrSem();
    }

    return(fRemoved);

} /* end RemovePortDlgStruct */

/****************************************************************************
 *
 * FUNCTION NAME = FindPortDlgStruct
 *
 * DESCRIPTION   = Find PORTDLGSTRUCT in list of open dialogs.  If found,
 *                 return the structure from the list.
 *
 * INPUT         = pPortDlgStruct - Pointer to structure contain the
 *                                  computer name and the port name of the
 *                                  PORTDLGSTRUCT to find in the list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = pOldPortDlgStruct - Pointer to PORTDLGSTRUCT from list
 *
 * RETURN-ERROR  = NULL - No PORTDLGSTRUCT found
 *
 ****************************************************************************/

PPORTDLGSTRUCT FindPortDlgStruct ( PPORTDLGSTRUCT pPortDlgStruct ) {

    PPORTDLGSTRUCT pOldPortDlgStruct = NULL;
    PSZ pszComputerName = NULL;
    PSZ pszPortName = NULL;
    BOOL fComputerMatch;

    if (pPortDlgStruct) {
        pszComputerName = pPortDlgStruct->pszComputerName;
        pszPortName = pPortDlgStruct->pszPortName;
       EnterPdrSem();
        pOldPortDlgStruct = pOpenDlgList;
        while (pOldPortDlgStruct) {
            fComputerMatch = FALSE;
            /*
             * Make sure computer names match
             */
            if ((pszComputerName && pOldPortDlgStruct->pszComputerName)) {
                if (!strcmp(pszComputerName,pOldPortDlgStruct->pszComputerName)) {
                    fComputerMatch = TRUE;
                }
            } else if (!pszComputerName && !(pOldPortDlgStruct->pszComputerName)) {
                fComputerMatch = TRUE;
            }
            /*
             * If computers match, make sure port names match
             */
            if (fComputerMatch &&
                   (pszPortName && pOldPortDlgStruct->pszPortName)) {
                if (!strcmp(pszPortName,pOldPortDlgStruct->pszPortName)) {
                    break;
                }
            }
            pOldPortDlgStruct = pOldPortDlgStruct->pNext;
        }
       LeavePdrSem();
    }

    return(pOldPortDlgStruct);

} /* end FindPortDlgStruct */
