/****************************************************************************/
/*  Copyright (C) 1995 IBM Corporation                                      */
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = infrared.C
 *
 * DESCRIPTIVE NAME = infrared port driver worker routines used
 *                     to communicate with an infrared attached
 *                     printer
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1994, 1995
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.2
 *
 * DATE
 *
 * DESCRIPTION
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *  DATE      FLAG        APAR   CHANGE DESCRIPTION
 *  --------  ----------  -----  --------------------------------------
 *  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 ****************************************************************************/

#define     INCL_SPLDOSPRINT

#include    "pdrconst.h"
#include    "pdrtypes.h"
#include    "pdrproto.h"


int acrtused=1;                     /* Define variable to say this is a DLL */

/*
** Local Functions
*/
VOID  InitPortSettingsDlg( HWND hDlg,
                           MPARAM mp2 );
ULONG FindConnectName( LP_DEVICE_DISCOVERY_LIST pDiscList,
                       PSZ                      pszConnectName,
                       LP_HDEVICE               phDevice,
                       PSZ                     *ppszPrinter );
ULONG  CalcStructLength ( HAB hab, USHORT usID );
VOID   CopyStruct ( HAB hab, USHORT usID, PCH pBuf, PULONG pulBeginStruct,
                    PULONG pulBeginText );
VOID   RemoveLeadTrailBlanks ( PCH pTarget, PCH pSource );
VOID   DE ( PCH str );
VOID EXPENTRY ControlThread( VOID );


/*
 * InfraRed structure with callback routines for IR Driver
 */
LM_CALLBACKS CallBacks  =
{
   (LMCB_CONN_INDIC) lmConnectIndic
  ,(LMCB_DISC_INDIC) lmDisconnectIndic
  ,(LMCB_DATA_INDIC) lmDataIndic
  ,(LMCB_EXPD_INDIC) lmExpDataIndic
  ,(LMCB_QOS_INDIC)  lmQosIndic
  ,(LMCB_CONN_CONF)  lmConnectConf
  ,(LMCB_TX_CONF)    lmTransmitConf
  ,(LMCB_QOS_CONF)   lmQosConf
  ,(LMCB_DSCVR_CONF) lmDiscoverConf
  ,(LMCB_DSCVR_INDIC)lmDiscoverIndic
};

/*
** new infrared functions
*/

/***************************************************************************
 * FUNCTION NAME = InfraredWrite
 *
 * DESCRIPTION   = Send data to the infrared dll LMDLL.DLL
 *
 * INPUT         = pPortInst   -> port structure printint to
 *                 pchData     -> buffer to write
 *                 cbData      -  bytes to write
 *                 pcbWritten  -> total bytes written on exit
 *
 * OUTPUT        = 0   - write successful, *pcbWritten has bytes actually sent
 *                 other - error, *pcbWritten still has bytes actually sent
 *
 * NOTES         = Must have pPortInst->hPortSem before calling to ensure
 *                  only 1 write is sent at a time.
 *
 *
 **************************************************************************/
ULONG APIENTRY InfraredWrite( PPORTINST  pPortInst, PVOID pchData,
                              ULONG cbData, PULONG pcbWritten)
{
  ULONG     ulReturn = 0L;
  ULONG     ulRC;
  INT       rc;
  PBYTE     pbyte;
  PDATAINFO pDataInfo;                 /* data info header pointer&handle*/
  PDATAINFO pLast;
  PDATAINFO pLastTmp;
  ULONG     ulBufferSize  = cbData;
  PBYTE     pbData        = pchData;
  ULONG     ulBytesToWrite;
  ULONG     ulRetryCount  = 0;
  ULONG     ulPostCount;
  ULONG     cbConfirmed;      /* #bytes confirmed for pdWrite */
  PLMPARMS  pLMParms;
  CHAR      szCaption[40];  /* SplMessageBox caption */

  #ifdef DEBUG
   CHAR logbuf[130];            // For calling LogCall
  #endif

  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredWrite Entry cb=%d pPortInst=%lX\r\n",
              cbData, (ULONG)pPortInst);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  *pcbWritten = 0L;
  cbConfirmed = 0;
  rc          = 0;
  if (!pPortInst || (pPortInst->signature != PT_SIGNATURE) ||
      !(pPortInst->flStatus & PF_PORT_OPEN) )
  {
    #ifdef DEBUG
     {
       sprintf( logbuf, "InfraredWrite invalid pPortInst or PortNotOpen=%lX\r\n", (ULONG)pPortInst);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    return(ERROR_WRITE_FAULT);       // return  DOS_ERROR error
  }
  pLMParms    = &pPortInst->LMParms;

  /*
  ** Writes to the infrared port are asynchronous.
  ** We allocate a buffer(with some header info),
  **  place it onto LMPARMS->pWaitingConfirmBufs list
  **  and pass the buffer to LM_DataRequest2().
  ** This returns immediately, but the data is not transmitted
  **  to the printer until lmTransmitConf() callback routine is called.
  ** lmTransmitConf is given the structure given to LM_DataRequest2()
  **  and a flag indicating if the transfer was successful.
  **  The flag seems to return failure only if the connection
  **  is broken(does not timeout right now).  We get negative
  **  acknowledgements for all packets before lmDisconnectIndic
  **  gets called(which sets pLMParms->fDisconnected)
  **
  ** lmTransmitConf tries to pull the buffer from the waiting list.
  **   If on the waiting list, the buffer is put onto the confirmed
  **      list for InfraredWrite to handle.
  **   If a negative ack or buffer not on waiting list
  **      then the buffer is freed.
  ** Here( in InfraredWrite ) we wait(for xx seconds) until
  **  the last buffer sent to LM_DataRequest2() is confirmed.
  ** If we get a transmit confirmation while waiting in InfraredWrite
  **  then lmTransmitConf puts the buffer onto the pConfirmedBufs list.
  **  If the last buffer sent is confirmed
  **    then LMParms->hevWriteComplete is posted to wakeup InfraredWrite.
  **         InfraredWrite counts all the buffers confirmed and frees them.
  ** If InfraredWrite times out, it will return the number of bytes
  **  confirmed, but there may still be buffers waiting in LMDLL to
  **  be transmitted.
  **  If lmTransmitConf gets called when noone is in InfraredWrite()
  **   then it will still put the buffers on the confirmed list for
  **        InfraredWrite to check the next time it is called.
  **  If InfraredWrite is called again after all buffers were not confirmed
  **   then it counts and removes all buffers from the confirmed list
  **   and returns immediately if any were confirmed since the prior call.
  **   If there are still buffers waiting to be confirmed, yet no buffers
  **      were confirmed since the last InfraredWrite call
  **      then InfraredWrite just waits for the prior buffers to be confirmed.
  **   We do not try to send any data given by the caller if we already
  **   have buffers waiting for confirmation because we could send the same
  **   data twice to the printer.
  **
  ** If we get a transmit confirmation failure
  **  then lmTransmitConf removes the buffer from the Waiting list, frees
  **  the buffer(not putting it onto the confirmed list) and
  **  posts LMParms->hevWriteComplete.
  ** We typically get called later with lmDisconnectIndic.
  **  For now, InfraredWrite will assume all printers must be resent
  **  the entire job when the IR connection is broken.
  **  This is due to the IR disconnection causing a printer reset.
  **  When InfraredWrite notices the disconnection while printing a job,
  **  InfraredWrite will give a dialog to the user asking them if they
  **  want to Cancel or Retry the print job.
  **  If Cancel is chosen, InfraredWrite will simulate a SplPdAbortDoc
  **   and all writes for this job will be tossed.
  **  If Retry is chosen, InfraredWrite will call SplControlDevice
  **   to restart the print job, and simulate a SplPdAbortDoc.
  **   This will cause the current job to be aborted, then resent
  **   to the printer in its entirety.
  **/

 EnterPdrSem();
  cbConfirmed = FreeConfirmedBufs( &pLMParms->pConfirmedBufs );
 LeavePdrSem();

  /*
  ** If prior buffers were confirmed since the last write
  **   return successful for the amount of data confirmed.
  */
  if (cbConfirmed) {
     if (cbConfirmed > cbData) {
        #ifdef DEBUG
         {
           sprintf( logbuf, "InfraredWrite cbConfirmed > cbData! cbConfirmed=%d cbData=%d\r\n", cbConfirmed, cbData );
           LogCall( logbuf );
         }
        #endif /* DEBUG */
        *pcbWritten = cbData;
     }  else {
        *pcbWritten = cbConfirmed;
     }
     #ifdef DEBUG
      {
        sprintf( logbuf, "InfraredWrite returning previously confirmed cb=%d\r\n", cbConfirmed );
        LogCall( logbuf );
      }
     #endif /* DEBUG */
     return(0);
  }

  /*
  ** If we were disconnected from the printer while printing
  **   then ask user if they want to Retry or Abort the job.
  **   We cannot continue the job where we left off because
  **   when the IR connection is broken, most printers reset
  **   themselves.  We must resend the entire job in this case.
  */
  if (pLMParms->fDisconnected)
  {
     /*
     ** The other device disconnected from us so close connection
     **   and allow reconnect with IR device.
     */
     #ifdef DEBUG
      {
        sprintf( logbuf, "InfraredWrite fDisconnected set\r\n");
        LogCall( logbuf );
      }
     #endif /* DEBUG */
     if (pPortInst->ulJobPrinting == PDR_NOT_PRINTING) {
        /*
        ** We are not printing a job, we are sending a bidi command.
        ** Do not give error popup to user in this case.
        ** Close the connection so that the next attempt to write
        **   will try to reconnect.
        */
       EnterPdrSem();
        InfraredClose(&pPortInst->LMParms, 2, &pPortInst->hFile);
        pPortInst->flStatus &= ~PF_PORT_OPEN;
       LeavePdrSem();
        *pcbWritten = 0;
        return(ERROR_WRITE_FAULT);

     } else {
        /*
        ** We were printing a job.
        ** Close the connection during SplPdClose()
        */
        pPortInst->ulJobPrinting = PDR_ABORTED;
        szCaption[0] = '\0';
        WinLoadString((HAB)-1, hPdrMod, ID_FIRST_DESC_LINES, sizeof(szCaption)-1, szCaption);
        ulRC = SplMessageBox( pPortInst->pszPortName,
                              SPLINFO_WARNING,
                              SPLDATA_PRINTERJAM,
                              NULL,
                              szCaption,
                              0,
                              MB_RETRYCANCEL );

        if (ulRC == MBID_RETRY) {
           /*
           ** Tell spooler to resend this job
           */
           ulRC = SplControlDevice( NULL, pPortInst->pszPortName, PRD_RESTART );
           #ifdef DEBUG
            {
              sprintf( logbuf, "InfraredWrite SplControlDevice rc=%d\r\n", rc);
              LogCall( logbuf );
            }
           #endif /* DEBUG */

        } else {
           /*
           ** Cancel the print job;  we already set PDR_ABORTED
           */
        }
        *pcbWritten = cbData;
     }
     return( 0 );   // let caller think everything is OK
  }

  /*
  ** If prior buffers were confirmed since the last write
  **   return successful for the amount of data confirmed.
  */
  if (cbConfirmed || rc) {
     if (cbConfirmed > cbData) {
        #ifdef DEBUG
         {
           sprintf( logbuf, "InfraredWrite cbConfirmed > cbData! cbConfirmed=%d cbData=%d\r\n", cbConfirmed, cbData );
           LogCall( logbuf );
         }
        #endif /* DEBUG */
        *pcbWritten = cbData;
     }  else {
        *pcbWritten = cbConfirmed;
     }
     #ifdef DEBUG
      {
        sprintf( logbuf, "InfraredWrite returning after IRopen rc=%d confirmed cb=%d\r\n", rc, cbConfirmed );
        LogCall( logbuf );
      }
     #endif /* DEBUG */
     return(rc);
  }

  if (pLMParms->ushConnection && !pLMParms->fDisconnected)
  {

    /*
    ** Reset the event sem back to a Wait state
    */
    ulRC = DosResetEventSem(pLMParms->hevWriteComplete, &ulPostCount);
    ASSERTT(ulRC);

    ulRC = 0L;
    pbyte = NULL;
    if (pLMParms->pWaitingConfirmBufs) {
       /*
       ** There are buffers waiting to be confirmed by the printer.
       ** Don't send any new buffers until old ones are confirmed.
       */
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredWrite pWaitingConfirmBufs=%lX\r\n", pLMParms->pWaitingConfirmBufs);
         LogCall( logbuf );
       }
      #endif /* DEBUG */

    } else {
       /*
       ** All previously sent buffers have been acknowledged.
       ** Begin sending new buffers.
       */
       pLMParms->ulBuffersSent       = 0;
       pLMParms->ulBuffersConfirmed  = 0;
       pLMParms->ulBytesConfirmed    = 0;
       while (ulBufferSize && !ulRC)
       {
         ulRC = DosAllocMem((PVOID)&pbyte, pLMParms->ConnectParms.maxDataSize +
                            sizeof(DATAINFO),
                            OBJ_TILE | PAG_READ|PAG_WRITE|PAG_COMMIT);
         ASSERTT (ulRC);
         if (!ulRC)
         {
           pDataInfo = (PDATAINFO)pbyte;     /* data header&handle pointer     */
           /**
            ** Store our instance pointer in data header
            ** Sem in this struct is used to block
            ** the pdWrite until the writes complete
            */
           pDataInfo->pLMParms    = pLMParms;
           pDataInfo->ulDataType  = 0L;                // init data type
           pbyte                 += sizeof(DATAINFO);  // bump over header

           ulBytesToWrite         = MIN(ulBufferSize,
                                        pLMParms->ConnectParms.maxDataSize - 4);
           pDataInfo->ulDataSize  = ulBytesToWrite;

           memcpy( pbyte, pbData, ulBytesToWrite);
           #ifdef DEBUG
            {
              sprintf( logbuf, "InfraredWrite loop BytesToWrite=%d pBuf=%lX\r\n",ulBytesToWrite, (ULONG)pDataInfo);
              LogCall( logbuf );
            }
           #endif /* DEBUG */
          EnterPdrSem();
           AddDataInfoBuf( &pLMParms->pWaitingConfirmBufs, pDataInfo );
          LeavePdrSem();
           /*
           ** if this is going to be the last buffer
           ** mark it with the IRDAATYPE_LASTBUFFER flag.
           ** Then we will wait for the last buffer to
           ** be confirmed.
           */
           if (ulBufferSize <= ulBytesToWrite)
           {
              pDataInfo->ulDataType |= IRDATATYPE_LASTBUFFER;
           }
           rc = STSFAILED;
           while (pLMParms &&
                  pLMParms->ushConnection && !pLMParms->fDisconnected)
           {
             rc = (*pLMParms->lmpfns.pfnLM_DataRequest2) (
                                    pLMParms->ushConnection, // connect handle
                                    NULL, 0,                 // look ahead-routing
                                    pbyte, ulBytesToWrite,   // actual data
                                    (HUSERDATA)pDataInfo);   // data handle
             #ifdef DEBUG
              {
                sprintf( logbuf, "InfraredWrite LM_DataRequest2 rc=%d BuffersSent=%d\r\n",rc, pLMParms->ulBuffersSent);
                LogCall( logbuf );
              }
             #endif /* DEBUG */
             if (rc == STSOK)
             {
               // Data was sent - update buffer counter and get out
               //  of the retry loop
               pLMParms->ulBuffersSent++;
               ulRetryCount = 0;
               break;
             }
             else
             {
               // how do we handle disconnect before last buffer is sent
               #ifdef DEBUG
                {
                  sprintf( logbuf, "InfraredWrite LM_DataRequest2 Failed!!!! rc= %d\r\n",rc);
                  LogCall( logbuf );
                }
               #endif /* DEBUG */
               ulRetryCount++;
               if (ulRetryCount >= 3)
               {
                 DosSleep(1L);         //Let someone else run
               }
             }
           }
           if (rc != STSOK)
           {
             ulBytesToWrite = 0;
           }

           ulBufferSize -= ulBytesToWrite;
           pbData       += ulBytesToWrite;
         }
       } /* end while */

    }


    /*
    ** if we sent anything wait for all of the data packets to be received
    */
    if (pLMParms->ulBuffersSent)
    {
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredWrite Waiting for data confirmation BufSent=%d\r\n", pLMParms->ulBuffersSent);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      rc = DosWaitEventSem(pLMParms->hevWriteComplete, 20000L); // 20 seconds
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredWrite No longer waiting WaitEventSem rc=%d\r\n",rc);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
    } /* endif */

    if (ulRC)
    {
      ulReturn = ERROR_WRITE_FAULT;   // return dos error
    }

    /*
    ** return the number of bytes confirmed in the call back
    */
   EnterPdrSem();
    *pcbWritten = FreeConfirmedBufs( &pLMParms->pConfirmedBufs );
   LeavePdrSem();
    if (cbData != *pcbWritten)
    {
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredWrite NotAllWritten toWrite=%d written=%d\r\n",cbData,*pcbWritten);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
    } /* endif */

  }
  else
  {
    ulRC = ERROR_WRITE_FAULT;        // return  DOS_ERROR error
    #ifdef DEBUG
     {
       sprintf( logbuf, "InfraredWrite Error writing (no connection) cbToWrite=%d cbWritten=%d\r\n",
                cbData,*pcbWritten );
       LogCall( logbuf );
     }
    #endif /* DEBUG */
  }

  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredWrite rc=%d *pcbWritten=%d\r\n",ulReturn, *pcbWritten);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  return (ulReturn);
}                                               /* end function InfraredWrite */

/***************************************************************************
 * FUNCTION NAME = WriteWithRetry
 *
 * DESCRIPTION   = Issue InfraredWrite and retry as many times as
 *                 necessary until the print timeout expires.
 *
 *
 * INPUT         = pPortInst   -> port structure printint to
 *                 pchData     -> buffer to write
 *                 cbData      -  bytes to write
 *                 pcbWritten  -> total bytes written on exit
 *
 * OUTPUT        = 0   - write successful, *pcbWritten has bytes actually sent
 *                 other - error, *pcbWritten still has bytes actually sent
 *
 * NOTES         = Must have pPortInst->hPortSem before calling to ensure
 *                  only 1 write is sent at a time.
 *
 *
 *
 **************************************************************************/
ULONG APIENTRY WriteWithRetry( PPORTINST  pPortInst, PVOID pchData,
                              ULONG cbData, PULONG pcbWritten)
{
    ULONG      rc;
    ULONG      cbWritten;
    ULONG      cbWrite;
    ULONG      ulPortTimeout ;  /* timeout for this port       */
    ULONG      ulTime ;  /* time prior to issuing Write call   */
    #ifdef DEBUG_ALERT
     CHAR logbuf[100];
    #endif

    #ifdef DEBUG
     {
       sprintf( logbuf, "WriteWithRetry Entry cb=%d pPortInst=%lX\r\n",
                cbData, (ULONG)pPortInst);
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    /*
     * Save time prior to issuing DosWrite
     * If write fails before printer timeout expired
     *   retry the request.
     * This gives Named Pipes(and monitors) a timeout
     *   period to start accepting data
     */
    rc          = 0;
    ulTime      = time() ;
    *pcbWritten = 0 ;                    /* init bytes written to zero   */

    rc = InfraredWrite(pPortInst, pchData, cbData, pcbWritten);

    if (rc || (*pcbWritten != cbData) ) {
       /*
        * Check time it took to issue write
        * If this is a known port
        *    there is a timeout for the port
        *    and the time it took to do the write was
        *    less than (PortTimeout - 1)
        * then
        *   Retry Request
        */
       if ( (pPortInst->signature == PT_SIGNATURE) &&
            (ulPortTimeout = pPortInst->ulPrintTimeOut) &&
            ( (time() - ulTime) < (ulPortTimeout - 1) )) {

           /*
            * Keep track of total bytes written
            * Reset clock if DosWrite succeeded without writting all bytes
            * Retry request
            */
           cbWritten = *pcbWritten ;
           do
           {
              /*
               * If job was aborted then return that everything printed OK
               */
              if (pPortInst->ulJobPrinting == PDR_ABORTED) {
                 #ifdef DEBUG_ALERT
                  {
                   sprintf( logbuf,
                            "WriteWithRetry abortflag while retrying\r\n");
                   LogCall( logbuf );
                  }
                 #endif /* DEBUG */
                 cbWritten = cbData;
                 break;
              }

              if ( (rc == 0) && (*pcbWritten) ) {
                 /*
                  * if DosWrite successful and bytes were written
                  *    reset timeout value
                  */
                 ulTime  = time() ;
              }

              /* Sleep one second to avoid hard loop on Write */
              DosSleep(1000) ;

              *pcbWritten = 0 ; /* reset bytes written */

              /*
               * Write buffer, taking into account bytes written so far
               */
              rc = InfraredWrite(pPortInst,
                            (PVOID)((PBYTE)pchData + cbWritten),
                                     /* index past bytes written */
                            cbData - cbWritten,/* only write remaining data */
                            pcbWritten);

              cbWritten += *pcbWritten ;

           } while ( (cbWritten < cbData) &&
                     ( (time() - ulTime) < (ulPortTimeout - 1) )) ;

           /* return actual bytes written */
           *pcbWritten = cbWritten ;
       }

    } /* end giving retry */

    #ifdef DEBUG
     {
       sprintf( logbuf, "WriteWithRetry Exit rc=%d cbWritten=%d\r\n",
                rc, *pcbWritten);
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    return(rc);
}

/*****************************************************************************
 * FUNCTION NAME = InfraredInitbind
 *
 * DESCRIPTION   = init & bind
 *                 Init and bind to the LMDLL.DLL interface.
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = (LONG)LMSTATUS STSOK=1
 *
 * RETURN-ERROR  = !=STSOK
 *
 **************************************************************************/

LONG APIENTRY InfraredInitBind( LSAPSEL lsap, PLMPFNS plmpfns,
                                LP_LM_CALLBACKS pCallBacks )
{
  LONG lRC;

  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif


  /*
  ** Init LMDLL.dll
  */
  lRC = (*plmpfns->pfnLM_Initialize)((U32)NULL);
  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredInitBind LM_Initialize rc=%d\r\n",lRC);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  DosSleep(1000L);
  if (lRC != STSOK)
  {
    DisplayError (HWND_DESKTOP, ID_IR_INIT_ERROR,
                  MB_OK | MB_APPLMODAL | MB_MOVEABLE);
  }
  else
  {
    /*
    ** Bind our callback functions to the LMDLL.DLL
    */
    lRC = (*plmpfns->pfnLM_BindLSAP) (lsap,pCallBacks);
    #ifdef DEBUG
     {
       sprintf( logbuf, "InfraredInitBind LM_BindLSAP rc=%d\r\n",lRC);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    if (lRC != STSOK)
    {
      DisplayError (HWND_DESKTOP, ID_IR_BIND_ERROR,
                    MB_OK | MB_APPLMODAL | MB_MOVEABLE);
    }
  }

  return(lRC);

} /* end function InfraredInitBind */

/*****************************************************************************
 * FUNCTION NAME = InfraredDiscover
 *
 * DESCRIPTION   = Discover up to 8 infrared devices
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = (LONG)LMSTATUS STSOK=1
 *
 * RETURN-ERROR  = !=STSOK
 *
 *****************************************************************************/

LONG APIENTRY InfraredDiscover( PLMPFNS                  plmpfns,
                                LP_DEVICE_DISCOVERY_LIST pDiscList,
                                USHORT                   uslmSize )
{
  LONG   lRC    = STSFAILED;

  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif

  lRC = (*plmpfns->pfnLM_DiscoverDevices)(8,pDiscList,&uslmSize);
  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredDiscover LM_DiscoverDevices rc=%d nrItems=%d\r\n",lRC, pDiscList->nrItems);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  return(lRC);

} /* end function InfraredDiscover */

/*****************************************************************************
 * FUNCTION NAME = InfraredConnect
 *
 * DESCRIPTION   = Connet to infrared device
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = (LONG)LMSTATUS STSOK=1
 *
 * RETURN-ERROR  = !=STSOK
 *
 **************************************************************************/

LONG APIENTRY InfraredConnect( PLMPARMS pLMParms, HDEVICE hDevice,
                               LSAPSEL lsapDest,   LSAPSEL lsapSrc)
{
  USHORT usMBid = MBID_RETRY;
  LONG   lRC;
  LONG   lReturn = STSFAILED;
  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif
  /*
  ** Create hevConnect sem in the wait state
  */
  lRC = DosCreateEventSem((PSZ)NULL,&pLMParms->hevConnect,
                          DC_SEM_SHARED, FALSE);
  ASSERTT(lRC);

  /*
  ** Create hevDisconnect sem in the wait state
  */
  if (!lRC)
  {
    lRC = DosCreateEventSem((PSZ)NULL,&pLMParms->hevDisconnect,
                          DC_SEM_SHARED, FALSE);
    ASSERTT(lRC);
  }
  /*
  ** Make a sem for tracking the completion of writes
  ** fState TRUE Make sure we do not wait on the write if a write is
  ** never done. Sem is created posted
  */
  if (!lRC)
  {
    lRC = DosCreateEventSem((PSZ)NULL,&pLMParms->hevWriteComplete,
                            DC_SEM_SHARED, TRUE);
    ASSERTT(lRC);
  }

  if(!lRC)
  {
    /*
    ** init loop vars
    */
    usMBid = MBID_RETRY;
    lRC    = STSFAILED;
    /*
    ** retry connect loop
    */
    while (usMBid == MBID_RETRY && lRC != STSOK)
    {
      /*
      ** Connect  to the device
      */
      DosSleep(1000L);
      pLMParms->fDisconnected = FALSE;  /* init disconnect flag */
      lRC = (*pLMParms->lmpfns.pfnLM_ConnectRequest)(hDevice,
             lsapDest,lsapSrc,
             (LP_CONNECT_PARMS)&pLMParms->ConnectParms,
             NULL,0,(ULONG)pLMParms,
             &pLMParms->ushConnection);
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredConnect LM_ConnectRequest rc=%d\r\n", lRC);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      if (lRC != STSOK)
      {
         usMBid = DisplayError (HWND_DESKTOP, ID_IR_CONNECT_ERROR,
                       MB_RETRYCANCEL | MB_APPLMODAL | MB_MOVEABLE);
      }
    }

    if (lRC == STSOK)
    { //OK
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredConnect Waiting for connection\r\n");
         LogCall( logbuf );
       }
      #endif /* DEBUG */

      /* 4 min max wait ???? */
      lRC = DosWaitEventSem(pLMParms->hevConnect,240000L);
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredConnect Done Waiting for connection WaitEvent rc=%d\r\n", lRC);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      if (!lRC)
      { // if we did not get a disconnect before connect confirm
        if (!pLMParms->fDisconnected)
        {
          #ifdef DEBUG
           {
             sprintf( logbuf, "InfraredConnect OK baudRate=%d windowSize=%d maxTurnTime=%d\r\n",
                      pLMParms->ConnectParms.baudRate,
                      pLMParms->ConnectParms.windowSize,
                      pLMParms->ConnectParms.maxTurnTime);
             LogCall( logbuf );
             sprintf( logbuf, "InfraredConnect    maxDataSize=%d bofCount=%d minTurnTime=%d\r\n",
                      pLMParms->ConnectParms.maxDataSize,
                      pLMParms->ConnectParms.bofCount,
                      pLMParms->ConnectParms.minTurnTime);
             LogCall( logbuf );
             sprintf( logbuf, "InfraredConnect    pollTimeout=%d discThreshold=%d\r\n",
                      pLMParms->ConnectParms.pollTimeout,
                      pLMParms->ConnectParms.discThreshold);
             LogCall( logbuf );
           }
          #endif /* DEBUG */

          lReturn = STSOK;
        }
      }
    }
    else
     lReturn = lRC;
  }

  /*
  ** if we failed free our resources
  */
  if (lReturn != STSOK)
  {
    if (pLMParms->hevConnect)
    {
      DosCloseEventSem(pLMParms->hevConnect);
      pLMParms->hevConnect = 0;
    }
    if (pLMParms->hevDisconnect)
    {
      DosCloseEventSem(pLMParms->hevDisconnect);
      pLMParms->hevDisconnect = 0;
    }
    if (pLMParms->hevWriteComplete)
    {
      DosCloseEventSem(pLMParms->hevWriteComplete);
      pLMParms->hevWriteComplete = 0;
    }
  }

  return(lReturn);

} /* end function InfraredConnect */


/*****************************************************************************
 * FUNCTION NAME = InfraredOpen
 *
 * DESCRIPTION   = Load, init, bind, discover, & connect to infrared
 *                 conected device through the LMDLL.DLL interface.
 *
 * INPUT         = pPortInst -> port instance struct for IR printer
 *                              to open a connection with
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = 0 - Success
 *
 * RETURN-ERROR  = error
 *
 * NOTE            Must be in port driver semaphore on entry/exit,
 *                 but leaves Sem and re-enters it.
 *
 **************************************************************************/

LONG APIENTRY InfraredOpen( PPORTINST pPortInst )
{
  BOOL                     fSuccess;
  ULONG                    rc;
  PSZ                      pszPortName;
  PSZ                      pszNewPrinter;
  HDEVICE                  hDevice;
  HFILE                    hFile;
  USHORT                   uslmSize;
  PLMPARMS                 pLMParms;
  LP_DEVICE_DISCOVERY_LIST pDiscList;
  CHAR                     szConnectName[MAX_CONNECTNAME_SZ];
  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif

  pszPortName = pPortInst->pszPortName;
  pLMParms    = &pPortInst->LMParms;
  rc          = 0;
  uslmSize    = 1024;
  hFile       = 0;

  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredOpen Entry for %s\r\n", pszPortName);
     LogCall( logbuf );
   }
  #endif /* DEBUG */
  /*
  ** Only allow 1 thread to open the port
  */
  if (pPortInst->flStatus & PF_OPEN_IN_PROGRESS) {
    /*
     * Set ERROR INIT IN PROGRESS      @BUGBUG
     */
    rc = PMERR_SPL_INIT_IN_PROGRESS;

  } else {

    pPortInst->flStatus |= PF_OPEN_IN_PROGRESS;
    pLMParms->pWaitingConfirmBufs = NULL;
    pLMParms->pConfirmedBufs = NULL;

   LeavePdrSem();
    rc =  LoadInfraredDLL(&pLMParms->hmodLMDLL, &pLMParms->lmpfns);
    ASSERTT(rc);
    if (!rc)
    {
      /*
      ** Get default connection name and timeouts from system INI
      */
      GetConnectionName( pszPortName, szConnectName );
      GetTimeOuts( pszPortName, pPortInst );
      /*
      ** Note: many Infrared APIs return TRUE/FALSE instead of rc
      */
      fSuccess = (*pLMParms->lmpfns.pfnLM_OpenDriver)((LP_U32)&hFile);
      #ifdef DEBUG
       {
         sprintf( logbuf, "InfraredOpen LM_OpenDriver rc=%d hFile=%d ConnectName=%s\r\n",fSuccess, hFile, szConnectName);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      if (fSuccess != STSOK)
      {
        DisplayError (HWND_DESKTOP, ID_IR_CONFIG_ERROR,
                      MB_OK | MB_APPLMODAL | MB_MOVEABLE);
      }
      /*
      ** Must wait 1 second before binding with IR device
      */
      DosSleep(1000L);

      if (fSuccess == STSOK)
      {
        /*
        ** Init & bind  LMDLL.dll
        */
        fSuccess = InfraredInitBind( 2, &pLMParms->lmpfns, &CallBacks );
        if (fSuccess == STSOK)
        {
          /*
          ** Discover the IR devices available
          ** DiscoverDevices is synchronous
          */
          pDiscList = NULL;
          rc = DosAllocMem ((PVOID)&pDiscList, uslmSize,
                             OBJ_TILE | PAG_READ | PAG_WRITE | PAG_COMMIT );
          ASSERTT(rc);

          if (!rc)
          {
            /*
            ** Get the list of IR devices available.
            */
            fSuccess = InfraredDiscover( &pLMParms->lmpfns,
                                          pDiscList,
                                          uslmSize );
            #ifdef DEBUG
             {
               sprintf( logbuf, "InfraredOpen InfraredDiscover fSuccess=%d ConnectName=%s\r\n", fSuccess, szConnectName);
               LogCall( logbuf );
             }
            #endif /* DEBUG */
            if ((fSuccess == STSOK) ||
                ((SHORT)fSuccess == (SHORT)stsErrorLinkActive))
            {
               /*
               ** InfraredDiscover will return stsErrorLinkActive if you
               **   already have a connection to an IR device,
               **   and it will fill the pDiscList from the cached
               **   list of IR devices it keeps.
               ** For now, treat this as successful.
               ** Later we may want to drop the current connection  @BUGBUG
               **   to discover any new devices in the room
               */
               fSuccess = STSOK;
               /*
               ** Find a printer we can connect to
               */
               pszNewPrinter = NULL;
               rc = FindConnectName( pDiscList,
                                     szConnectName,
                                    &hDevice,
                                    &pszNewPrinter );
               if ( pszNewPrinter &&
                    ((rc == PDR_CONNECTNAME_FOUND) ||
                     (rc == PDR_OTHER_PRINTER_FOUND)) )
               {
                  strcpy( szConnectName, pszNewPrinter );
               } else {
                  fSuccess = STSFAILED;
               }
            }
            DosFreeMem(pDiscList);
            #ifdef DEBUG
             {
               sprintf( logbuf, "InfraredOpen FindConnectName rc=%d ConnectName=%s\r\n", rc, szConnectName);
               LogCall( logbuf );
             }
            #endif /* DEBUG */
            pDiscList = NULL;
            rc = ERROR_VC_DISCONNECTED;            // assume fail
            if (fSuccess == STSOK)
            {
              fSuccess =  InfraredConnect( pLMParms, hDevice, 2, 2);
              if (fSuccess == STSOK)
              {
                rc = 0L;
              }
            }
          }
        } else {

          rc = ERROR_OPEN_FAILED;            // fail
        }
      }
    }

    if (rc)
    {
       /*
       ** Disconnect and free LMDLL if open fails
       */
       if (pLMParms->lmpfns.pfnLM_UnBindLSAP)
         (*pLMParms->lmpfns.pfnLM_UnBindLSAP)(2);

       if (pLMParms->lmpfns.pfnLM_Shutdown)
         (*pLMParms->lmpfns.pfnLM_Shutdown)();

       if (pLMParms->hevConnect)
         DosCloseEventSem(pLMParms->hevConnect);

       if (pLMParms->hevDisconnect)
         DosCloseEventSem(pLMParms->hevDisconnect);

       if (pLMParms->hevWriteComplete)
         DosCloseEventSem(pLMParms->hevWriteComplete);

       if (pLMParms->hmodLMDLL)
         DosFreeModule(pLMParms->hmodLMDLL);
      EnterPdrSem();
    } else {
      EnterPdrSem();
       pPortInst->hFile           = hFile;
       pPortInst->flStatus       |= PF_PORT_OPEN;
       pPortInst->flStatus       &= ~PF_DISCONNECT_POSTED;
       pPortInst->ulTimeLastCmd   = time();
       /*
       ** Save connection name.
       ** It will be returned as the DeviceID string for BIDI_Q_PORT
       */
       if (szConnectName[0])
       {
         strcpy( pPortInst->szNickName, szConnectName);
       }
    }
    pPortInst->flStatus &= ~PF_OPEN_IN_PROGRESS;
  }

  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredOpen rc=%d hFile=%d\r\n",rc, hFile);
     LogCall( logbuf );
   }
  #endif /* DEBUG */
  return(rc);

} /* end function InfaredOpen */

/***************************************************************************
 * FUNCTION NAME = InfraredDisconnect
 *
 * DESCRIPTION   = Disconnect connection
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = 0 - Success
 *
 * RETURN-ERROR  = LMSTATS error
 *
 **************************************************************************/

LONG APIENTRY InfraredDisconnect(PLMPARMS pLMParms)
{
  LONG    lRC;
  ULONG   ulFlags = 0;
  PLMPFNS plmpfns = &pLMParms->lmpfns;
  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif

  lRC =  (*plmpfns->pfnLM_DisconnectRequest)(pLMParms->ushConnection,NULL,
                                             0,&ulFlags);
  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredDisconnect LM_DisconnectRequest rc=%d\r\n", lRC);
     LogCall( logbuf );
   }
  #endif /* DEBUG */
  if (!pLMParms->fDisconnected)
  {
    #ifdef DEBUG
     {
       sprintf( logbuf, "InfraredDisconnect Waiting for disconnect\r\n");
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    lRC = DosWaitEventSem(pLMParms->hevDisconnect, 180000L); // 3 min
    #ifdef DEBUG
     {
       sprintf( logbuf, "InfraredDisconnect Done Waiting rc=%d fDisconnected=%d\r\n", lRC, pLMParms->fDisconnected);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
  }

  /*
  ** Free resources allocated in InfraredConnect
  */
  if (pLMParms->hevConnect)
  {
    DosCloseEventSem(pLMParms->hevConnect);
    pLMParms->hevConnect = 0;
  }
  if (pLMParms->hevDisconnect)
  {
    DosCloseEventSem(pLMParms->hevDisconnect);
    pLMParms->hevDisconnect = 0;
  }
  if (pLMParms->hevWriteComplete)
  {
    DosCloseEventSem(pLMParms->hevWriteComplete);
    pLMParms->hevWriteComplete = 0;
  }

  /*
  ** if error normalize the return code
  */
  if (lRC)
  {
    lRC = STSFAILED;
  }

  return(lRC);

}   /* end function InfraredDisconnect */

/***************************************************************************
 * FUNCTION NAME = InfraredClose
 *
 * DESCRIPTION   = Disconnect, unBind, Shutdown, and unload LMDLL.DLL
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = 0 - Success
 *
 * RETURN-ERROR  = May be either  LMSTATS error   or  Dos error
 *
 **************************************************************************/

LONG APIENTRY InfraredClose(PLMPARMS pLMParms, LSAPSEL lsap, PHFILE phFile)
{
  LONG    lRC;
  PLMPFNS plmpfns = &pLMParms->lmpfns;
  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif

  InfraredDisconnect(pLMParms);

  lRC =  (*plmpfns->pfnLM_UnBindLSAP)(lsap);
  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredClose LM_UnBindLSAP rc=%d\r\n", lRC);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  lRC =  (*plmpfns->pfnLM_Shutdown)();
  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredClose LM_Shutdown rc=%d\r\n", lRC);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  lRC = (*pLMParms->lmpfns.pfnLM_CloseDriver)((LP_U32)phFile);
  #ifdef DEBUG
   {
     sprintf( logbuf, "InfraredClose LM_CloseDriver rc=%d\r\n", lRC);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  lRC = DosFreeModule(pLMParms->hmodLMDLL);
  ASSERTT(lRC);

  return(lRC);

}   /* end function InfraredClose */

/*****************************************************************************
 * FUNCTION NAME = FindConnectName
 *
 * DESCRIPTION   = Find printer ConnectName in IR Discover list
 *
 * INPUT         = pDiscList   -> IR Discover list
 *                 pszConnectName -> IR nickname of printer to find in pDiscList
 *                 phDevice    -> place to receive IR device handle for
 *                                device matching pszConnectName
 *                 ppszPrinter -> place to store pointer to either the
 *                                pszConnectName found in pDiscList or the
 *                                first printer found in pDiscList
 *
 *
 * OUTPUT        = 0(PDR_CONNECTNAME_FOUND) - pszConnectName wanted was in
 *                                            pDiscList
 *                 1(PDR_OTHER_PRINTER_FOUND) - pszConnectName not found but
 *                                            another printer was returned
 *                 2(PDR_NO_PRINTERS_FOUND) - no printers found in pDiscList
 *
 *
 *****************************************************************************/

ULONG FindConnectName( LP_DEVICE_DISCOVERY_LIST pDiscList,
                       PSZ                      pszConnectName,
                       LP_HDEVICE               phDevice,
                       PSZ                     *ppszPrinter )
{
  ULONG  rc;
  PSZ    pszNickName;
  PSZ    pszPrinter;
  ULONG  i;
  USHORT usLenHints;
  PBYTE  pByte;
  BOOL   fMatchFound   = FALSE;
  BOOL   fPrinterFound = FALSE;
  #ifdef DEBUG
   CHAR logbuf[260];            // For calling LogCall
  #endif


  pszNickName = NULL;
  pszPrinter  = NULL;
  rc          = PDR_NO_PRINTERS_FOUND;

  #ifdef DEBUG
   {
     sprintf( logbuf, "FindConnectName (%s)\r\n", pszConnectName);
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  for (i=0; i < pDiscList->nrItems; i++)
  {
    /*
    ** We keep a pointer to the first printer found
    ** in the discovery list. This pointer will be used
    ** if the device nick name stored in the INI file
    ** is not found.  Note: the nick name is only stored
    ** in the INI file if the user  selects a device in
    ** the infrared dialog box.
    **
    ** NOTE: We Need to account for a variable # of hint bytes
    */
    pByte      = (PBYTE)pDiscList->device[i].xid.stdInfo.hints;
    usLenHints = (USHORT)pDiscList->device[i].xid.stdInfo.lenHints;
    pszNickName = (PSZ)pDiscList->device[i].xid.stdInfo.nickName;
    #ifdef DEBUG
     {
       sprintf( logbuf, "FindConnectName hdevice = %d (%s)\r\n",
                 pDiscList->device[i].hDevice,
                 pszNickName);
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    while (usLenHints--)
    {
      #ifdef DEBUG
       {
         sprintf( logbuf, "FindConnectName IR HINT BYTE = 0x%X\r\n", *pByte);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      if (*pByte & IR_HINT1_PRINTER)
      {
         if (!fPrinterFound)
         {
            *phDevice     = pDiscList->device[i].hDevice;
            pszPrinter    = pszNickName;
            fPrinterFound = TRUE;
            if (rc == PDR_NO_PRINTERS_FOUND) {
               rc = PDR_OTHER_PRINTER_FOUND;
            }
         }
         #ifdef DEBUG
          {
            sprintf( logbuf, "FindConnectName Found a Printer device in Discovery list (%s)\r\n",
                      pszNickName);
            LogCall( logbuf );
          }
         #endif /* DEBUG */
      }
      pByte++;
    }
    /*
    ** If we find one matching pszConnectName
    **    break out of loop
    **  else use first printer found.
    */
    if (!fMatchFound && pszConnectName && pszConnectName[0] &&
        !strcmp(pszConnectName, pszNickName) )
    {
      *phDevice   = pDiscList->device[i].hDevice;  // Device Found
      pszPrinter  = pszNickName;
      fMatchFound = TRUE;
      rc          = PDR_CONNECTNAME_FOUND;
      #ifdef DEBUG
       {
         sprintf( logbuf, "FindConnectName matched NickName in Discovery list (%s)\r\n", pszConnectName);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      /* break; */
    }
  } /* endfor */

  *ppszPrinter = pszPrinter;
  #ifdef DEBUG
   {
     sprintf( logbuf, "FindConnectName rc=%d (%s)\r\n", rc, pszPrinter ? pszPrinter : "NOT FOUND");
     LogCall( logbuf );
   }
  #endif /* DEBUG */

  return(rc);

} /* end function FindConnectName */

/****************************************************************************
 *
 * FUNCTION NAME = HandleQPort
 *
 * DESCRIPTION   = Process BIDI_Q_PORT command
 *                 This tries to connect to the infrared printer
 *                 selected in the IR port settings dialog and returns
 *                 the connection nickname as the printer device ID.
 *
 *
 * INPUT         = pszDeviceName   - name of port printer is attached
 *                 ulFlags         - query options(cache, short wait,...)
 *                 pOutData        - receives PRTPORT structure, which includes
 *                                   variable length strings packed
 *                                   immediately after the PRTPORT structure
 *                 pcbOutData      - Points to length of pOutData(in bytes)
 *                                   On entry this is set to length of buffer
 *                                    passed in.
 *                                   On exit this is updated with the length
 *                                    of available data, which may be more
 *                                    than put into pOutData if ERROR_MORE_DATA
 *                                    or NERR_BufTooSmall returned.
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG HandleQPort( PSZ    pszDeviceName,
                   ULONG  ulFlags,
                   PVOID  pOutData,
                   PULONG pcbOutData )
{
    ULONG       rc;
    ULONG       cbNeeded;
    PSZ         pszConnectName;
    PPORTINST   pPortInst;
    PPRTPORT    pPrtPort;
    BOOL        fPrinterFound;
    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif


    #ifdef DEBUG
     {
       sprintf( logbuf, "HandleQPort called for %s\r\n", pszDeviceName);
       LogCall( logbuf );
     }
    #endif /* DEBUG */

    rc            = 0;
    fPrinterFound = FALSE;

   EnterPdrSem();
    /*
     * Add port to the list of supported ports if it is not yet
     *   in the port list.
     */
    pPortInst = AddPortInst ( pszDeviceName );
    if (pPortInst ) {

       /*
       ** open infrared device if not yet opened
       */
       if ( !(pPortInst->flStatus & PF_PORT_OPEN) )
       {
          rc =  InfraredOpen(pPortInst);
       }
       /*
       ** At this point we should have a connection to the printer
       */
       if (pPortInst->LMParms.fDisconnected) {
          fPrinterFound = FALSE;
       } else {
          fPrinterFound = TRUE;
       }
       /*
       ** Try to load new spooler Bidi APIs
       */
       if (fLoadedPMSPL == FALSE)
       {
          LoadPMSPL();
       }
       if (pfnPrtQuery)
       {
          /*
          ** Set port flag indicating that bidi spooler is installed
          */
          pPortInst->flStatus |= PF_Q_PORT_CALLED;
       } else if (rc == 0) {

          /*
          ** If we opened the IRDD just to get the device list
          **   then we need to close the IRDD now so others can use it.
          */
          InfraredClose(&pPortInst->LMParms, 2, &pPortInst->hFile);
          pPortInst->flStatus &= ~PF_PORT_OPEN;
       }
       if (rc)
       {
         /*
         ** For now treat all InfraredOpen failures as unable to connect
         **   with printer.
         */
         rc = ERROR_VC_DISCONNECTED;

       } else {
          cbNeeded = sizeof (PRTPORT) + strlen(pPortInst->szNickName) + 1;
          if (*pcbOutData >= cbNeeded) {
              pPrtPort = (PPRTPORT) pOutData;
              memset ( pPrtPort, 0, sizeof (PRTPORT));
              if (pfnPrtQuery)
              {
                 if (fPrinterFound) {
                    pPrtPort->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE |
                                                   PRTPORT_CAPS_BIDI_ACTIVE |
                                                   PRTPORT_CAPS_Q_ALL_PORTS ;
                 } else {
                    pPrtPort->flBidiCapabilities = PRTPORT_CAPS_BIDI_CAPABLE |
                                                   PRTPORT_CAPS_BIDI_ACTIVE |
                                                   PRTPORT_CAPS_Q_ALL_PORTS |
                                                   PRTPORT_CAPS_NOT_RESPONDING;
                 }
              }
              /*
              ** For now, search the IR nickname for supporting
              **  either NPAP or PJL.
              */
              pPrtPort->flBidiProtocol = 0;
              pszConnectName = pPortInst->szNickName;
              if (*pszConnectName)
              {
                 if (strstr(pszConnectName, "NPAP"))
                 {
                    pPrtPort->flBidiProtocol |= PRTPORT_TYPE_NPAP;
                 }
                 if (strstr(pszConnectName, "PJL"))
                 {
                    pPrtPort->flBidiProtocol |= PRTPORT_TYPE_PJL;
                 }
                 if (strstr(pszConnectName, "HP LaserJet") ||
                    (strstr(pszConnectName, "DeskJet")) )
                 {
                    pPrtPort->flBidiProtocol |= PRTPORT_TYPE_PJL;
                 }
              }
              pPrtPort->ulPortType = PRTPORT_PORT_TYPE_INFRARED;
              pPrtPort->ulBidiLevel = 0;
              pPrtPort->ulMaxSendSize = 4096;
              pPrtPort->ulMaxReceiveSize = 4096;
              pPrtPort->ulMaxHeldResponses = 0;
              pPrtPort->ulpszDeviceID = sizeof(PRTPORT);
              pszConnectName = (PSZ)pPrtPort + pPrtPort->ulpszDeviceID;
              strcpy( pszConnectName, pPortInst->szNickName);
          } else {
              rc = NERR_BufTooSmall;
          }
          *pcbOutData = cbNeeded;
       }
    } else {
        rc = ERROR_NOT_ENOUGH_MEMORY;
    }
   LeavePdrSem();

    return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = HandleWaitAlert
 *
 * DESCRIPTION   = Process BIDI_WAIT_ALERT command
 *                 This holds caller until data is received from one
 *                 of the ports supported by this port driver.
 *
 * INPUT         = pszDeviceName   - name of port printer is attached
 *                 ulFlags         - query options
 *                 ulCommand       - function code for query
 *                 pInData         - command specific input data
 *                 cbInData        - length in bytes of pInData
 *                 pOutData        - returned query structure, format depends
 *                                   on ulCommand
 *                 pcbOutData      - Points to length of pOutData(in bytes)
 *                                   On entry this is set to length of buffer
 *                                    passed in.
 *                                   On exit this is updated with the length
 *                                    of available data, which may be more
 *                                    than put into pOutData
 *
 * OUTPUT        = 0  - successful.
 *                      *pcbOutData = length of data returned by query
 *                      pOutData    = query structure
 *                 234(ERROR_MORE_DATA) - partial query structure returned
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = partial query structure
 *                 2123(NERR_BufTooSmall) - buffer too small to fit any data
 *                      *pcbOutData = length of buffer required to store
 *                                    entire query structure
 *                      pOutData    = not updated, since it is much too small
 *                 Other - error code, nothing updated
 *
 *
 *
 * NOTE            Not in semaphore on entry/exit
 *
 ****************************************************************************/
ULONG HandleWaitAlert( PSZ    pszDeviceName,
                       ULONG  ulFlags,
                       ULONG  ulCommand,
                       PVOID  pInData,
                       ULONG  cbInData,
                       PVOID  pOutData,
                       PULONG pcbOutData )
{
    ULONG         rc;
    PDALERTINFO   PdAlertInfo;
    PBYTE         pPrinterData;
    ULONG         cbActual;
    ULONG         ulPostCount;
    PPORTINST     pPortInst;
    PWAITALERTBUF pWaitBuf;           /* Current WAITALERTBUF we are using  */
    PALERTBUF     pAlertBuf;
    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif

    /*
     * Find port instance
     */
    rc           = 0;
    pPrinterData = NULL;

    #ifdef DEBUG
     {
       sprintf( logbuf, "HandleWaitAlert called for %s\r\n", pszDeviceName);
       LogCall( logbuf );
     }
    #endif /* DEBUG */

   EnterPdrSem();
    pPortInst = FindPortInst ( pszDeviceName );
    if (!pPortInst ) {
        rc = ERROR_FILE_NOT_FOUND;
    } else if (!PdrWaitAlert.pCurrBuf) {
       /*
        * Must allocate WaitAlertBuffer for this port driver
        */
       rc = DosAllocMem((PVOID)&PdrWaitAlert.pCurrBuf,
                         DEF_WAITALERTBUF_SIZE,
                         PAG_READ|PAG_WRITE|PAG_COMMIT);
       if (!rc) {
          /*
          ** Initialize the waitalertbuf header
          */
          memset( PdrWaitAlert.pCurrBuf, 0, sizeof(WAITALERTBUF));
          PdrWaitAlert.pCurrBuf->cbBufSize = DEF_WAITALERTBUF_SIZE;
          PdrWaitAlert.pCurrBuf->cbBufUsed = sizeof(WAITALERTBUF) -
                                             sizeof(ALERTBUF);
          if (!PdrWaitAlert.pXlateBuf) {
             /*
              * Must allocate alternate WaitAlertBuffer for this port driver
              */
             rc = DosAllocMem((PVOID)&PdrWaitAlert.pXlateBuf,
                               DEF_WAITALERTBUF_SIZE,
                               PAG_READ|PAG_WRITE|PAG_COMMIT);
             if (!rc) {
                /*
                ** Initialize the waitalertbuf header
                */
                memset( PdrWaitAlert.pXlateBuf, 0, sizeof(WAITALERTBUF));
                PdrWaitAlert.pXlateBuf->cbBufSize = DEF_WAITALERTBUF_SIZE;
                PdrWaitAlert.pXlateBuf->cbBufUsed = sizeof(WAITALERTBUF) -
                                                    sizeof(ALERTBUF);
             }
          }
       }
    }
    if (!PdrWaitAlert.hevWaitAlert) {
       rc = DosCreateEventSem ( 0,
                                &(PdrWaitAlert.hevWaitAlert),
                                0,
                                FALSE );   // Create in Reset state
    }
   LeavePdrSem();
    if (rc) {
       return(rc);
    }

    PdAlertInfo.ulFlags = 0;
    /*
     * Only return to caller when an alert is read
     */
    while ( !(PdAlertInfo.ulFlags & PD_ALERT) && !rc) {
        memset( &PdAlertInfo, 0, sizeof(PDALERTINFO));
        cbActual = 0;
        pPrinterData = NULL;
       EnterPdrSem();
        if ((PdrWaitAlert.pCurrBuf->cAlertsRemaining == 0) &&
            (PdrWaitAlert.pXlateBuf->cAlertsRemaining == 0) &&
            (!PdrWaitAlert.pXlateBuf->fPdMoreAlerts)) {
          LeavePdrSem();
           #ifdef DEBUG
            {
              sprintf( logbuf, "HandleWaitAlert Waiting for %s\r\n", pszDeviceName);
              LogCall( logbuf );
            }
           #endif /* DEBUG */
           rc = DosWaitEventSem ( PdrWaitAlert.hevWaitAlert, SEM_INDEFINITE_WAIT );
           #ifdef DEBUG
            {
              sprintf( logbuf, "HandleWaitAlert WokeUp rc=%d\r\n", rc );
              LogCall( logbuf );
            }
           #endif /* DEBUG */
           rc = DosResetEventSem(PdrWaitAlert.hevWaitAlert, &ulPostCount);
          EnterPdrSem();
        }
        rc = 0;
        if ((PdrWaitAlert.pXlateBuf->cAlertsRemaining == 0) &&
            (PdrWaitAlert.pCurrBuf->cAlertsRemaining) &&
            (!PdrWaitAlert.pXlateBuf->fPdMoreAlerts)) {
           //
           // Swap buffers so we can process the alerts stored
           //  in pCurrBuf.
           // This thread(BIDI_WAIT_ALERT) always processes pXlateBuf.
           // When data is received from a port, it is always
           //  stored in pCurrBuf.
           // When swapping buffers, the new pCurrBuf will be reset
           //  to allow new alerts to be stored in it.
           //
           pWaitBuf = PdrWaitAlert.pCurrBuf;
           PdrWaitAlert.pCurrBuf  = PdrWaitAlert.pXlateBuf;
           PdrWaitAlert.pXlateBuf = pWaitBuf;
           memset( PdrWaitAlert.pCurrBuf, 0, sizeof(WAITALERTBUF));
           PdrWaitAlert.pCurrBuf->cbBufSize = DEF_WAITALERTBUF_SIZE;
           PdrWaitAlert.pCurrBuf->cbBufUsed = sizeof(WAITALERTBUF) -
                                              sizeof(ALERTBUF);
        }
        if ((PdrWaitAlert.pXlateBuf->cAlertsRemaining) ||
            (PdrWaitAlert.pXlateBuf->fPdMoreAlerts)) {
           pWaitBuf = PdrWaitAlert.pXlateBuf;
           //
           // Special case the first alert in the buffer.
           //
           pAlertBuf = pWaitBuf->pLastProcessed;
           if (!pAlertBuf) {
              pAlertBuf = &pWaitBuf->FirstAlertBuf;
           }
           //
           // If the last time we called SplProtXlateCmd we received
           //   PD_MOREALERTS bit set, then we must call XlateCmd
           //   again with a NULL buffer to let the protocol converter
           //   give us the additional alert from the previous buffer.
           //   This can happen if more than 1 alert is in the buffer
           //   we pass to XlateCmd().
           //
           if (PdrWaitAlert.pXlateBuf->fPdMoreAlerts) {
              cbActual = 0;
              PdAlertInfo.ulFlags = PD_NEXTALERT;
              //
              // Temp bug workaround - some code does not accept  @BUGBUG
              //   NULL for pInBuf on SplProtXlateCmd().
              // For now just point it to our stack since it is readonly
              //
              //pPrinterData = NULL;
              pPrinterData = (PBYTE)&PdAlertInfo;
           } else {
              //
              // Get past the last processed alert buffer and
              //   reset the header data.
              // Special case the first alert in the buffer.
              //
              if (pWaitBuf->pLastProcessed) {
                 pAlertBuf = (PALERTBUF)((PBYTE)pAlertBuf +
                                                sizeof(ALERTBUF) +
                                                pAlertBuf->cbData);
              }
              pPrinterData = (PBYTE)pAlertBuf + sizeof(ALERTBUF);
              cbActual = pAlertBuf->cbData;
              pWaitBuf->pLastProcessed = pAlertBuf;
              pWaitBuf->cAlertsRemaining--;

           }
           //
           // Set portname to the one we received the alert from
           //
           pszDeviceName = pAlertBuf->pszAlertPortName;
          LeavePdrSem();
           rc = MySplProtXlateCmd ( pszDeviceName,
                                    (PFN)NULL,
                                    pPrinterData,
                                    cbActual,
                                    &PdAlertInfo,
                                    pOutData,
                                    pcbOutData );
          EnterPdrSem();
           /*
           ** Must check for the protocol converter telling us
           **  that more than one alert was stored in the
           **  buffer pased to SplProtXlateCmd.
           ** In this case, we need to call the converter again
           **  with a NULL buffer to let the converter return
           **  the next alert to the caller of BIDI_WAIT_ALERT
           **/
           if (PdAlertInfo.ulFlags & PD_MOREALERTS) {
              PdrWaitAlert.pXlateBuf->fPdMoreAlerts = TRUE;
           } else {
              PdrWaitAlert.pXlateBuf->fPdMoreAlerts = FALSE;
           }
          LeavePdrSem();
        } else {
           //
           // If we woke up due to a disconnection then return
           //   that PRTALERT_TYPE_COMM_STATUS_CHANGED occurred.
           // We woke up for some reason.
           // For now, just sleep and wait for next wake-up
           //
           DosSleep( 4000 );
        }
    }
    return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = CalcBufLength
 *
 * DESCRIPTION   = Determine how big buffer is needed to store all PORTNAMES
 *                 structures
 *
 * INPUT         = hab - anchor block handle
 *
 * OUTPUT        = length of buffer necessary to store all default port names
 *                 supported by this port driver.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG CalcBufLength ( HAB hab )
{
   ULONG cbRequired;
   USHORT usID;

   cbRequired = 0;

      /*
      ** calculate length required to fit all the port info.
      */
   for (usID = PORT_ID_FIRST; usID <= PORT_ID_LAST; usID += 2)
   {
      cbRequired += CalcStructLength (hab, usID);
   }

   return(cbRequired);
}

/****************************************************************************
 *
 * FUNCTION NAME = CalcStructLength
 *
 * DESCRIPTION   = Determine size of buffer needed to store PORTNAMES structure
 *                 for given string ID.
 *
 * INPUT         = hab     - anchor block handle
 *                 usID    - string ID for port name
 *
 * OUTPUT        = length of buffer necessary to store this port name
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG CalcStructLength ( HAB hab,
                         USHORT usID )
{
   ULONG cbRequired;
   CHAR chString[STR_LEN_PORTDESC];

   cbRequired = 0;

   WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, chString);
   cbRequired += strlen (chString) + 1;
   WinLoadString(hab, hPdrMod, (USHORT)(usID + 1), STR_LEN_PORTDESC, chString);
   cbRequired += strlen (chString) + 1;
   cbRequired += sizeof (PORTNAMES);
   return(cbRequired);
}

/****************************************************************************
 *
 * FUNCTION NAME = NumPortsCanFit
 *
 * DESCRIPTION   = Determine how many ports can fit in buffer
 *
 * INPUT         = hab     - anchor block handle
 *                 cbBuf   - size in bytes of buffer to hold PORTNAMES
 *
 * OUTPUT        = count of PORTNAMES structures that can fit in buffer
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG NumPortsCanFit ( HAB hab,
                       ULONG cbBuf )
{
   ULONG cbRequired;
   USHORT usID;
   ULONG ulNumPort;

   cbRequired = 0;
   ulNumPort = 0;

      /*
      ** calculate how many ports we can fit in buf.
      */
   for (usID = PORT_ID_FIRST; usID <= PORT_ID_LAST; usID += 2)
   {
      cbRequired += CalcStructLength (hab, usID);
      if (cbRequired > cbBuf)
      {
         return(ulNumPort);
      }
      ulNumPort++;
   }

   return(ulNumPort);
}

/****************************************************************************
 *
 * FUNCTION NAME = CopyNPorts
 *
 * DESCRIPTION   = Copy given number of ports into buffer
 *
 * INPUT         = hab     - anchor block handle
 *                 pBuf    - buffer to get PORTNAMES structures
 *                 ulReturned - number of ports to return
 *
 * OUTPUT        = pBuf is updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID  CopyNPorts ( HAB hab,
                   PCH pBuf,
                   ULONG ulReturned )
{
   USHORT usID;
   ULONG ulBeginText;
   ULONG ulBeginStruct;

   ulBeginText = ulReturned * sizeof (PORTNAMES);
   ulBeginStruct = 0;

   for (usID = PORT_ID_FIRST;
        usID <= PORT_ID_LAST && ulReturned;
        usID += 2, --ulReturned)
   {
      CopyStruct (hab, usID, pBuf, &ulBeginStruct, &ulBeginText);
   }
}

/****************************************************************************
 *
 * FUNCTION NAME = CopyStruct
 *
 * DESCRIPTION   = Copy single PORTNAMES structure to buffer
 *
 * INPUT         = hab     - anchor block handle
 *                 usID    - string ID for port to return
 *                 pBuf    - buffer to get PORTNAMES structures
 *                 pulBeginStruct - offset from begin of pBuf to store next
 *                                  PORTNAMES
 *                 pulBeginText   - offset from pBuf to store next string
 *
 * OUTPUT        = pBuf is updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID CopyStruct ( HAB hab,
                  USHORT usID,
                  PCH pBuf,
                  PULONG pulBeginStruct,
                  PULONG pulBeginText )
{
   PPORTNAMES pPortNames;

   pPortNames = (PPORTNAMES)(pBuf + *pulBeginStruct);
   *pulBeginStruct += sizeof (PORTNAMES);

      /*
      ** copy port name in the structure
      */
   pPortNames->pszPortName = pBuf + *pulBeginText;
   WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, pPortNames->pszPortName);
   *pulBeginText += strlen (pPortNames->pszPortName) + 1;

      /*
      ** copy port description to the structure
      */
   pPortNames->pszPortDesc = pBuf + *pulBeginText;
   WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, pPortNames->pszPortDesc);
   *pulBeginText += strlen (pPortNames->pszPortDesc) + 1;
}

/****************************************************************************
 *
 * FUNCTION NAME = GetPortDescription
 *
 * DESCRIPTION   = Get port description from our string resources.
 *
 * INPUT         = hab     - anchor block handle
 *                 pszPortName - name of port to get description for
 *                 pszPortDesc - gets port description
 *
 * OUTPUT        = TRUE  - if portname description is found
 *                 FALSE - if not
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL GetPortDescription ( HAB hab,
                          PSZ pszPortName,
                          PSZ pszPortDesc )
{
   USHORT usID;
   CHAR chBuf[STR_LEN_PORTDESC];

   for (usID = PORT_ID_FIRST; usID <= PORT_ID_LAST; usID += 2)
   {
      WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, chBuf);
      if (!strcmp (pszPortName, chBuf))
      {
         strcpy (pszPortDesc, chBuf);
         return(TRUE);
      }
   }
   return(FALSE);
}

/****************************************************************************
 *
 * FUNCTION NAME = OpenInfraredPortDlg
 *
 * DESCRIPTION   = Display port settings dialog
 *
 * INPUT         = hab     - anchor block handle
 *                 pPortDlgStruct - pointer to the InfraRed data structure
 *
 * OUTPUT        = TRUE  - if port settings changed
 *                 FALSE - if port settings not changed
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG OpenInfraredPortDlg ( HAB hab, PPORTDLGSTRUCT pPortDlgStruct )
{
   ULONG          rc;
   PPORTSETTINGS  pPortSettings;
   PPORTDLGSTRUCT pOldPortDlgStruct;
   ULONG          cbBuf;
   ULONG          cbNeeded;
   ULONG          cRetries;
   BOOL           fRetry;
   BOOL           fAdded;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   /*
    * Check to see if port settings are already being displayed and
    *       set focus to the open dialog
    */
   pOldPortDlgStruct = FindPortDlgStruct ( pPortDlgStruct );
   if (pOldPortDlgStruct) {
       WinShowWindow ( pOldPortDlgStruct->hwndDlg, TRUE );
       WinSetFocus ( HWND_DESKTOP, pOldPortDlgStruct->hwndDlg );
       return(FALSE);
   }
   fAdded = AddPortDlgStruct ( pPortDlgStruct );
   /*
   ** Must query port driver for current settings
   **
   ** NOTE: you should start another thread that will issue
   **       the PrtQuery because it can take a long time.
   **       The thread should get the dialog handle from the
   **       pPortDlgStruct, then update the dialog when all
   **       the data from PrtQuery has been returned.
   */
   pPortSettings = NULL;
   cbBuf  = DEFAULT_BUFSIZE ;
   fRetry = FALSE;
   do
   {
      rc = DosAllocMem ((PPVOID) &pPortSettings,
                        cbBuf,
                        PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_TILE );
      if (rc) {
          #ifdef DEBUG
           {
             sprintf( logbuf, "OpenInfraredPortDlg DosAllocMem failed rc=%d\r\n",rc);
             LogCall( logbuf );
           }
          #endif /* DEBUG */
         break;

      } else {
         //
         // Must get around another thread initting this port @BUGBUG
         //    We will get rc=PMERR_SPL_INIT_IN_PROGRESS if
         //    another thread is in middle of initialization.
         //
         cRetries = 0;
         do
         {
           cbNeeded = cbBuf ;

           rc = MyPrtQuery( pPortDlgStruct->pszComputerName,
                            pPortDlgStruct->pszPortName,
                            TYPE_LONG_WAIT,
                            BIDI_Q_PORTDRV,
                            NULL,
                            0,
                            pPortSettings,
                           &cbNeeded );

         } while ( (rc == PMERR_SPL_INIT_IN_PROGRESS) &&
                   (cRetries++ < 5) && !(DosSleep( 1000 ) ) );

         if ((rc == ERROR_MORE_DATA) && (fRetry == FALSE)) {

            fRetry = TRUE ;
            cbBuf = cbNeeded ;
            DosFreeMem( pPortSettings );
            pPortSettings = NULL ;

         } else {

            fRetry = FALSE;
         }
      }
   } while ( (rc == ERROR_MORE_DATA) && fRetry );

   if (rc)
   {
      /*
       * PrtQuery failed.
       * This will happen if using the original Warp spooler(PMSPL.DLL),
       *   so for now just call the routine to get the port settings.
       * This will open the IR device driver, get the settings,
       *   then close the IR DD.
       */
      #ifdef DEBUG
       {
         sprintf( logbuf, "OpenInfraredPortDlg PrtQuery failed rc=%d\r\n",rc);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      rc = HandleQPortDRV ( pPortDlgStruct->pszPortName,
                            pPortSettings,
                            &cbNeeded );
   }
   if (rc)
   {
      DisplayError (HWND_DESKTOP, ID_IR_SETTINGS_ERROR,
                    MB_OK | MB_APPLMODAL | MB_MOVEABLE);
      if (pPortSettings)
      {
         DosFreeMem(pPortSettings);
      }
      if (fAdded) {
          RemovePortDlgStruct ( pPortDlgStruct );
      }
      return(FALSE);
   }

   /*
   ** Initialize Port Settings dialog structure to
   **  current IR port settings
   */
   pPortDlgStruct->pPortSettings    = pPortSettings;
   pPortDlgStruct->cbPortSettings   = cbBuf ;
   pPortDlgStruct->ulPrintTimeOut   = pPortSettings->ulPrintTimeOut ;
   pPortDlgStruct->ulNoQueryTimeOut = pPortSettings->ulNoQueryTimeOut ;
   pPortDlgStruct->ulNoJobTimeOut   = pPortSettings->ulNoJobTimeOut ;
   if (pPortSettings->ulpszDescription)
   {
      strcpy( pPortDlgStruct->szDescription,
              GETPTRFROMOFFSET(ulpszDescription,pPortSettings));
   }
   if (pPortSettings->ulpszConnectName)
   {
      strcpy( pPortDlgStruct->szConnectName,
              GETPTRFROMOFFSET(ulpszConnectName,pPortSettings));
   }
   WinDlgBox  (HWND_DESKTOP,
               HWND_DESKTOP,
               (PFNWP)InfraredDlg,
               (HMODULE)hPdrMod,
               IDD_PORTINFRARED,
               pPortDlgStruct);

   DosFreeMem(pPortSettings);

   if (fAdded) {
       RemovePortDlgStruct ( pPortDlgStruct );
   }
   return pPortDlgStruct->flModified;

}



/****************************************************************************
 *
 * FUNCTION NAME = InfraredDlg
 *
 * DESCRIPTION   = Port settings dialog procedure
 *
 * INPUT         = hDlg: HWND
 *                 msg:  message
 *                 mp1:  message parameter
 *                 mp2:     "       "
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

MRESULT EXPENTRY InfraredDlg( HWND hDlg,
                            USHORT msg,
                            MPARAM mp1,
                            MPARAM mp2 )
{
    PPORTDLGSTRUCT pPortDlgStruct;
    PPORTSETTINGS  pPortSettings;
    SHORT sIndex;
    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif

    switch (msg) {

        case WM_INITDLG:
            InitPortSettingsDlg( hDlg, mp2 );
            break;

        case WM_COMMAND:
            /*
            ** Check validity of pPortDlgStruct  @BUGBUG
            */
            pPortDlgStruct = (PPORTDLGSTRUCT)WinQueryWindowULong (hDlg, QWL_USER);
            pPortSettings  = pPortDlgStruct->pPortSettings;

            switch (SHORT1FROMMP(mp1)) {
               case IDC_OK:

                  WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT, SPBM_QUERYVALUE,
                                    (MPARAM)&(pPortDlgStruct->ulPrintTimeOut),
                                    (MPARAM)NULL);
                  WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT, SPBM_QUERYVALUE,
                                    (MPARAM)&(pPortDlgStruct->ulNoQueryTimeOut),
                                    (MPARAM)NULL);
                  WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT, SPBM_QUERYVALUE,
                                    (MPARAM)&(pPortDlgStruct->ulNoJobTimeOut),
                                    (MPARAM)NULL);
                  /*
                  ** Get Selected connection
                  */
                  sIndex = (SHORT)WinSendDlgItemMsg(hDlg, IDC_PPA_CONNECTIONS,
                                  LM_QUERYSELECTION,
                                  MPFROMSHORT (LIT_FIRST),
                                  (MPARAM)0);
                  if (sIndex == LIT_NONE)
                  {
                     pPortDlgStruct->szConnectName[0] = 0;
                  }
                  else
                  {
                     WinSendDlgItemMsg(hDlg, IDC_PPA_CONNECTIONS,
                             LM_QUERYITEMTEXT,
                             MPFROM2SHORT (sIndex, MAX_CONNECTNAME_SZ -1),
                             MPFROMP (pPortDlgStruct->szConnectName));
                  }

                  /*
                  ** Call function to save any changed port settings
                  */
                   if (SavePortSettings(pPortDlgStruct)) {
                      WinDismissDlg(hDlg, 0);
                  }
                  break;

               case IDC_RESET:
                  WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(pPortSettings->ulPrintTimeOut),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(pPortSettings->ulNoQueryTimeOut),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(pPortSettings->ulNoJobTimeOut),
                                    MPFROMSHORT(NULL));
                  /*
                  ** Reset connection selection
                  */
                  if (pPortSettings->ulpszConnectName)
                  {
                    sIndex = (SHORT)WinSendDlgItemMsg(hDlg, IDC_PPA_CONNECTIONS,
                                    LM_SEARCHSTRING,
                                    MPFROM2SHORT (LSS_CASESENSITIVE,LIT_FIRST),
                                    (MPARAM)GETPTRFROMOFFSET(ulpszConnectName,pPortSettings));
                    #ifdef DEBUG
                     {
                       sprintf( logbuf, "InfraredDlg IR Search (%s) index %d\r\n",
                                pPortSettings->ulpszConnectName ?
                                 GETPTRFROMOFFSET(ulpszConnectName,pPortSettings) :
                                 "NO PORTNAME",
                                 sIndex);
                       LogCall( logbuf );
                     }
                    #endif /* DEBUG */
                    if (LIT_NONE != sIndex)
                    {
                      WinSendDlgItemMsg (hDlg, IDC_PPA_CONNECTIONS,
                                             LM_SELECTITEM,
                                             (MPARAM)sIndex,
                                             (MPARAM)TRUE);
                      WinSendDlgItemMsg (hDlg, IDC_PPA_CONNECTIONS,
                                             LM_SETTOPINDEX,
                                             (MPARAM)sIndex,
                                             (MPARAM)0);
                    }
                  }

                  break;

               case IDC_DEFAULT:
                  WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_PRINT_VALUE),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_QUERY_VALUE),
                                    MPFROMSHORT(NULL));
                  WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                                     SPBM_SETCURRENTVALUE,
                                    MPFROMSHORT(DEF_TIMEOUT_JOB_VALUE),
                                    MPFROMSHORT(NULL));

                  /*
                  ** Default to no connection selected
                  ** This will cause the system to look for the
                  ** first available printer
                  */
                  sIndex = (SHORT)WinSendDlgItemMsg(hDlg, IDC_PPA_CONNECTIONS,
                                  LM_QUERYSELECTION,
                                  MPFROMSHORT (LIT_FIRST),
                                  (MPARAM)0);

                  if (LIT_NONE != sIndex)
                  {
                    WinSendDlgItemMsg (hDlg, IDC_PPA_CONNECTIONS,
                                           LM_SELECTITEM,
                                           (MPARAM)sIndex,
                                           (MPARAM)FALSE);
                  }

                  break;

               case IDC_CANCEL:
                  WinDismissDlg(hDlg, MBID_CANCEL);
                  break;

               default:
                  return(WinDefDlgProc(hDlg, msg, mp1, mp2) );
            }
           break;

        case WM_HELP:
            InitializeHelp();
            if (hwndHelp)
            {
               WinSendMsg (hwndHelp, HM_DISPLAY_HELP,
                                         (MPARAM)IDH_DLG_EXTENDED, NULL);
               return (MRESULT)TRUE;
            }
            break;

        case WM_DESTROY:

               /*
               ** if we have a help instance destroy it.
               */
            if (HelpAlreadyInitialized)
            {
               CALLDestroyHelpInstance(hwndHelp);
               hwndHelp = (HWND) NULL;
               HelpAlreadyInitialized = FALSE;
               HelpStubHookIsSet=FALSE;
            }
            ReleaseHelpStubHook();
             break;

        default:
            return(WinDefDlgProc(hDlg, msg, mp1, mp2) );
    }
    return FALSE;
}

/****************************************************************************
 *
 * FUNCTION NAME = InitPortSettingsDlg
 *
 * DESCRIPTION   = Initialize port settings dialog when WM_INITDLG received
 *
 * INPUT         = hDlg      - dialog hwnd
 *                 mp2       -> PORTSETTINGS structure
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/
VOID  InitPortSettingsDlg( HWND hDlg,
                           MPARAM mp2 )
{
    PPORTDLGSTRUCT pPortDlgStruct;
    PPORTSETTINGS  pPortSettings;
    PSZ            pszDesc;
    PSZ            pszNickName;
    CHAR           szTitle[ STR_LEN_TITLE + 1];
    CHAR           szTemp[ STR_LEN_PORTDESC ];
    ULONG          rc = 0;
    SHORT          sIndex;
    ULONG          ulcDeviceNames;  /* Number of IR printer names found      */
                                    /* This field determines how many null   */
                                    /*  terminated strings are pointed to    */
                                    /*  by ulpszDeviceList.                  */

    pPortDlgStruct = (PPORTDLGSTRUCT)mp2;
    WinSetWindowULong (hDlg, QWL_USER, (ULONG)pPortDlgStruct);
    pPortSettings = pPortDlgStruct->pPortSettings;
    pPortDlgStruct->hwndDlg = hDlg ;

    WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                           SPBM_SETLIMITS,
                           MPFROMP(TIMEOUT_UPPER_LIMIT),
                           MPFROMP(TIMEOUT_LOWER_LIMIT));

    WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                           SPBM_SETLIMITS,
                           MPFROMP(TIMEOUT_UPPER_LIMIT),
                           MPFROMP(TIMEOUT_LOWER_LIMIT));

    WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                           SPBM_SETLIMITS,
                           MPFROMP(TIMEOUT_UPPER_LIMIT),
                           MPFROMP(TIMEOUT_LOWER_LIMIT));

    pszDesc = GETPTRFROMOFFSET(ulpszDescription,pPortSettings);
    if (pszDesc)
    {
        WinSetWindowText (WinWindowFromID (hDlg,
                                           (USHORT)IDC_PPA_DESC),
                          GETPTRFROMOFFSET(ulpszDescription,pPortSettings));
        /*
         * Make dialog title consistent with the shell @154038
         *    (PortName - Properties)
         */
        rc = WinLoadString(pPortDlgStruct->hAB,
                           hPdrMod,
                           PDR_ID_PROPERTIES,
                           STR_LEN_PORTDESC, szTemp);
        /*
         * rc == 0 means an error has occurred
         */
        if (rc) {
            strcpy ( szTitle, GETPTRFROMOFFSET(ulpszPortName,pPortSettings));
            strcat ( szTitle, " - " );
            strcat ( szTitle, szTemp );
            WinSetWindowText (hDlg, szTitle);
        }
    }

    WinSendDlgItemMsg (hDlg, IDC_PPA_PRINTTIMEOUT,
                           SPBM_SETCURRENTVALUE,
                           MPFROMSHORT(pPortSettings->ulPrintTimeOut),
                           MPFROMSHORT(NULL));
    WinSendDlgItemMsg (hDlg, IDC_PPA_QUERYTIMEOUT,
                           SPBM_SETCURRENTVALUE,
                           MPFROMSHORT(pPortSettings->ulNoQueryTimeOut),
                           MPFROMSHORT(NULL));
    WinSendDlgItemMsg (hDlg, IDC_PPA_JOBTIMEOUT,
                           SPBM_SETCURRENTVALUE,
                           MPFROMSHORT(pPortSettings->ulNoJobTimeOut),
                           MPFROMSHORT(NULL));
    /*
    ** Place all the infrared print devices in the listbox.
    */
    ulcDeviceNames = pPortSettings->ulcDeviceNames;
    pszNickName = GETPTRFROMOFFSET( ulpszDeviceList, pPortSettings);
    while (ulcDeviceNames)
    {
       /*
       ** Each IR device nickname is null terminated
       */

       sIndex = (SHORT)WinSendDlgItemMsg (hDlg, IDC_PPA_CONNECTIONS,
                                          LM_INSERTITEM,
                                          MPFROMSHORT (LIT_END),
                                          MPFROMP (pszNickName));
       /*
       ** if this is the saved selection
       **  select the item
       */
       if (!strcmp(pPortDlgStruct->szConnectName, pszNickName))
       {
         WinSendDlgItemMsg (hDlg, IDC_PPA_CONNECTIONS,
                                LM_SELECTITEM,
                                (MPARAM)sIndex,
                                (MPARAM)TRUE);
         WinSendDlgItemMsg (hDlg, IDC_PPA_CONNECTIONS,
                                LM_SETTOPINDEX,
                                (MPARAM)sIndex,
                                (MPARAM)0);
       }
       ulcDeviceNames--;
       pszNickName += strlen(pszNickName) + 1;
    }
    /*
    ** create help hook for helps
    */
    SetHelpStubHook();

}

/****************************************************************************
 *
 * FUNCTION NAME = HandleQPortDRV
 *
 * DESCRIPTION   = Get InfraRed port driver settings data structure
 *
 * INPUT         = pszPortName   - Name of port
 *                 pPortSettings   -> buffer to get port settings
 *                 pcbPortSettings -> length of buffer(on entry)
 *                                    Set to length of data put
 *                                    into buffer(on exit)
 *
 * OUTPUT        = 0   - Success, pPortSettings and *pcbPortSettings updated
 *                 Other - failed
 *
 * NOTE          = Called by SplPdQuery(BIDI_Q_PORT) under bidi spooler process
 *                 or by OpenInfraredPortDlg() if bidi spooler not installed.
 *
 ****************************************************************************/

ULONG HandleQPortDRV ( PSZ           pszPortName,
                       PPORTSETTINGS pPortSettings,
                       PULONG        pcbPortSettings )
{

    ULONG                    cbNeeded;
    ULONG                    rc;
    ULONG                    fSuccess;
    PSZ                      pszNickName;
    PSZ                      pszTemp;
    ULONG                    flPortStatus;
    ULONG                    cPrintDevices;
    ULONG                    i;
    PBYTE                    pByte;
    USHORT                   usLenHints;
    USHORT                   uslmSize;
    PPORTINST                pPortInst;
    PLMPARMS                 pLMParms;
    LP_DEVICE_DISCOVERY_LIST pDiscList;
    CHAR                     chDesc[CCHMAXPATH];
    CHAR                     chAppName[CCHMAXPATH];
    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif


    #ifdef DEBUG
     {
       sprintf( logbuf, "HandleQPortDRV called for %s\r\n", pszPortName);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    rc            = 0;
    flPortStatus  = 0;
    cPrintDevices = 0;

   EnterPdrSem();
    pPortInst = AddPortInst ( pszPortName );
    if (pPortInst )
    {
       flPortStatus = pPortInst->flStatus;
    } else {
        rc = ERROR_FILE_NOT_FOUND;
    }
    /*
    ** Must get IR Device List.
    ** This requires the IR device to be open.
    */
    if ( !rc && !(pPortInst->flStatus & PF_PORT_OPEN))
    {
       rc = InfraredOpen(pPortInst);
    }
   LeavePdrSem();
    if (rc)
    {
      #ifdef DEBUG
       {
         sprintf( logbuf, "HandleQPortDRV failed prior to IRDiscover rc=%d(%s)\r\n", rc, pszPortName);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      return(rc);
    }
    /*
    ** Discover the IR devices available
    ** DiscoverDevices is synchronous
    */
    cbNeeded      = 0;
    cPrintDevices = 0;
    pDiscList     = NULL;
    uslmSize      = 1024;
    pLMParms      = &pPortInst->LMParms;
    rc = DosAllocMem ((PVOID)&pDiscList, uslmSize,
                       OBJ_TILE | PAG_READ | PAG_WRITE | PAG_COMMIT );
    ASSERTT(rc);

    if (!rc)
    {
      /*
      ** Get the list of IR devices available.
      ** If this fails, just return the other information
      **   to the caller, without any IR devices.
      */
      fSuccess = InfraredDiscover( &pLMParms->lmpfns,
                                    pDiscList,
                                    uslmSize );
      if ((fSuccess == STSOK) ||
          ((SHORT)fSuccess == (SHORT)stsErrorLinkActive))
      {
         /*
         ** InfraredDiscover will return stsErrorLinkActive if you
         **   already have a connection to an IR device,
         **   and it will fill the pDiscList from the cached
         **   list of IR devices it keeps.
         ** For now, treat this as successful.
         ** Later we may want to drop the current connection  @BUGBUG
         **   to discover any new devices in the room
         */
         fSuccess = STSOK;
         /*
         ** Count the size of the buffer needed to store all the
         **  IR print device nicknames
         */
         for (i=0; i < pDiscList->nrItems; i++)
         {
           /*
           ** NOTE: We Need to account for a variable # of hint bytes
           */
           pByte      = (PBYTE)pDiscList->device[i].xid.stdInfo.hints;
           usLenHints = (USHORT)pDiscList->device[i].xid.stdInfo.lenHints;
           pszNickName = (PSZ)pDiscList->device[i].xid.stdInfo.nickName;
           #ifdef DEBUG
            {
              sprintf( logbuf, "HandleQPortDRV hdevice = %d (%s)\r\n",
                        pDiscList->device[i].hDevice,
                        pszNickName);
              LogCall( logbuf );
            }
           #endif /* DEBUG */
           while (usLenHints--)
           {
             #ifdef DEBUG
              {
                sprintf( logbuf, "HandleQPortDRV IR HINT BYTE = 0x%X\r\n", *pByte);
                LogCall( logbuf );
              }
             #endif /* DEBUG */
             if (*pByte & IR_HINT1_PRINTER)
             {
                cbNeeded += strlen(pszNickName) + 1;
                cPrintDevices++ ;
             }
             pByte++;
           }
         } /* end for loop */
      }
    }
    /*
    ** If we opened the IRDD just to get the device list
    **   then we need to close the IRDD now so others can use it.
    */
    if ( !(flPortStatus & PF_PORT_OPEN) &&
         !(flPortStatus & PF_OPEN_IN_PROGRESS) )
    {
       if (pPortInst && !(pPortInst->flStatus & PF_Q_PORT_CALLED))
       {
         InfraredClose(&pPortInst->LMParms, 2, &pPortInst->hFile);
         pPortInst->flStatus &= ~PF_PORT_OPEN;
       }
    }
    /*
    ** Get port description from INI file
    */
    strcpy (chAppName, APPNAME_LEAD_STR);
    strcat (chAppName, pszPortName);
    chDesc[0] = '\0';
    fSuccess = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                      chAppName,
                                      KEY_DESCRIPTION,
                                      NULL,
                                      chDesc,
                                      CCHMAXPATH);
    /*
     * Get size of return data
     */
   EnterPdrSem();

    cbNeeded += sizeof(PORTSETTINGS)
                + strlen(pszPortName) + 1
                + strlen(chDesc) + 1
                + strlen(pPortInst->szNickName) + 1;
    /*
     * Check size of return buffer
     */
    if (cbNeeded > *pcbPortSettings) {
       rc = NERR_BufTooSmall;
    }

    if (!rc)
    {
       /*
        * Copy values and strings to the buffer
        */
       pPortSettings->signature        = INFR_SIGNATURE;
       pPortSettings->ulVersion        = 1;
       pPortSettings->flStatus         = pPortInst->flStatus ;
       pPortSettings->ulPrintTimeOut   = pPortInst->ulPrintTimeOut ;
       pPortSettings->ulNoQueryTimeOut = pPortInst->ulNoQueryTimeOut ;
       pPortSettings->ulNoJobTimeOut   = pPortInst->ulNoJobTimeOut ;

       pszTemp = (PSZ)((PBYTE)pPortSettings + sizeof(PORTSETTINGS));

       strcpy( pszTemp, pszPortName );
       pPortSettings->ulpszPortName = (ULONG)pszTemp;
       pszTemp += strlen(pszTemp) + 1;

       strcpy( pszTemp, chDesc );
       pPortSettings->ulpszDescription = (ULONG)pszTemp;
       pszTemp += strlen(pszTemp) + 1;

       strcpy( pszTemp, pPortInst->szNickName );
       pPortSettings->ulpszConnectName = (ULONG)pszTemp;
       pszTemp += strlen(pszTemp) + 1;

       pPortSettings->ulcDeviceNames = cPrintDevices;
       if (cPrintDevices)
       {
          pPortSettings->ulpszDeviceList = (ULONG)pszTemp;
          /*
          ** Store the null-terminated IR Device NickName List
          */
          for (i=0; i < pDiscList->nrItems; i++)
          {
            /*
            ** NOTE: We Need to account for a variable # of hint bytes
            */
            pByte      = (PBYTE)pDiscList->device[i].xid.stdInfo.hints;
            usLenHints = (USHORT)pDiscList->device[i].xid.stdInfo.lenHints;
            pszNickName = (PSZ)pDiscList->device[i].xid.stdInfo.nickName;
            while (usLenHints--)
            {
              if (*pByte & IR_HINT1_PRINTER)
              {
                 strcpy( pszTemp, pszNickName );
                 pszTemp += strlen(pszTemp) + 1;
              }
              pByte++;
            }
          } /* end for loop */
       } else {
          pPortSettings->ulpszDeviceList = 0;
       }

       /*
        * Change pointers to offsets
        */
       PTRTOOFFSET(ulpszPortName,pPortSettings);
       PTRTOOFFSET(ulpszDescription,pPortSettings);
       PTRTOOFFSET(ulpszConnectName,pPortSettings);
       if (cPrintDevices)
       {
          PTRTOOFFSET(ulpszDeviceList,pPortSettings);
       }
    }
   LeavePdrSem();

    if (pDiscList)
    {
       DosFreeMem(pDiscList);
    }
    /*
     * Set size of return buffer
     */
    *pcbPortSettings = cbNeeded;

    #ifdef DEBUG
     {
       sprintf( logbuf, "HandleQPortDRV rc=%d(%s) cbNeeded=%d\r\n", rc, pszPortName, cbNeeded);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = HandleSetPortDRV
 *
 * DESCRIPTION   = Get InfraRed port driver settings data structure
 *
 * INPUT         = pszPortName   - Name of port
 *                 pPortSettings  -> buffer to get port settings
 *                 cbPortSettings -> length of buffer(on entry)
 *
 * OUTPUT        = 0   - Success, INI entries and instance data updated
 *                 Other - failed
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG HandleSetPortDRV ( PSZ           pszPortName,
                         PPORTSETTINGS pPortSettings,
                         ULONG         cbPortSettings )
{
    ULONG        rc;
    PSZ          pszDescription;
    PSZ          pszConnectName;
    PPORTINST    pPortInst;
    CHAR         chBuf[CCHMAXPATH];
    CHAR         chAppName[CCHMAXPATH];
    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif


    #ifdef DEBUG
     {
       sprintf( logbuf, "HandleSetPortDRV called for %s\r\n", pszPortName);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    rc             = 0;
    pszDescription = NULL;
    pszConnectName = NULL;

    if ( (pPortSettings->signature != INFR_SIGNATURE) ||
         (pPortSettings->ulVersion != 1) )
    {
       return(ERROR_INVALID_PARAMETER);
    }

   EnterPdrSem();
    pPortInst = AddPortInst ( pszPortName );
    if (pPortInst )
    {
        pPortInst->ulPrintTimeOut = pPortSettings->ulPrintTimeOut ;
        pPortInst->ulNoQueryTimeOut = pPortSettings->ulNoQueryTimeOut ;
        pPortInst->ulNoJobTimeOut = pPortSettings->ulNoJobTimeOut ;
        pszDescription = GETPTRFROMOFFSET( ulpszDescription, pPortSettings);
        pszConnectName = GETPTRFROMOFFSET( ulpszConnectName, pPortSettings);
    } else {
        rc = ERROR_FILE_NOT_FOUND;
    }
   LeavePdrSem();
    if (rc)
    {
      #ifdef DEBUG
       {
         sprintf( logbuf, "HandleSetPortDRV failed rc=%d(%s)\r\n", rc, pszPortName);
         LogCall( logbuf );
       }
      #endif /* DEBUG */
      return(rc);
    }
    /*
    ** Set port data in INI file
    */
    strcpy (chAppName, APPNAME_LEAD_STR);
    strcat (chAppName, pszPortName);

    _itoa (pPortSettings->ulPrintTimeOut, chBuf, 10);
    strcat (chBuf, ";");
    if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                               chAppName,
                               KEY_TIMEOUT_PRINT,
                               chBuf))
     {
        DE ("Error in writing to system init file");
     }

    _itoa (pPortSettings->ulNoQueryTimeOut, chBuf, 10);
    strcat (chBuf, ";");
    if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                               chAppName,
                               KEY_TIMEOUT_QUERY,
                               chBuf))
     {
        DE ("Error in writing to system init file");
     }

    _itoa (pPortSettings->ulNoJobTimeOut, chBuf, 10);
    strcat (chBuf, ";");
    if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                               chAppName,
                               KEY_TIMEOUT_JOB,
                               chBuf))
     {
        DE ("Error in writing to system init file");
     }

     if (pszDescription)
     {
        if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                   chAppName,
                                   KEY_DESCRIPTION,
                                   pszDescription))
         {
            DE ("Error in writing to system init file");
         }
     }

     if (pszConnectName)
     {
        if (!PrfWriteProfileString (HINI_SYSTEMPROFILE,
                                   chAppName,
                                   KEY_CONNECTION,
                                   pszConnectName))
         {
            DE ("Error in writing to system init file");
         }
     }

    #ifdef DEBUG
     {
       sprintf( logbuf, "HandleSetPortDRV rc=%d(%s)\r\n", rc, pszPortName);
       LogCall( logbuf );
     }
    #endif /* DEBUG */
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = SavePortSettings
 *
 * DESCRIPTION   = save any changed values for this port in INI file
 *
 * INPUT         = pPortDlgStruct -> this port's information
 *
 * OUTPUT        = TRUE          -  successful
 *                 FALSE         -  PrfWrite failed
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL SavePortSettings( PPORTDLGSTRUCT pPortDlgStruct )
{
   PPORTSETTINGS pPortSettings;
   PPORTSETTINGS pNewPortSettings;
   BOOL          fModified;
   ULONG         rc;
   ULONG         cbNeeded;
   PSZ           pszTemp;
   PSZ           pszDescription;
   PSZ           pszConnectName;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   fModified        = FALSE;
   rc               = 0;
   pNewPortSettings = NULL;
   pPortSettings    = pPortDlgStruct->pPortSettings;
   pszDescription   = GETPTRFROMOFFSET( ulpszDescription, pPortSettings);
   pszConnectName   = GETPTRFROMOFFSET( ulpszConnectName, pPortSettings);

   if ( (pPortDlgStruct->ulPrintTimeOut != pPortSettings->ulPrintTimeOut) ||
        (pPortDlgStruct->ulNoQueryTimeOut != pPortSettings->ulNoQueryTimeOut) ||
        (pPortDlgStruct->ulNoJobTimeOut != pPortSettings->ulNoJobTimeOut) )
   {
     fModified = TRUE;
   } else {
      if ( (pszDescription && strcmp(pPortDlgStruct->szDescription, pszDescription) ) ||
           (!pszDescription && pPortDlgStruct->szDescription[0] ) )
      {
         fModified = TRUE;
      }  else {
         if ( (pszConnectName && strcmp(pPortDlgStruct->szConnectName, pszConnectName) ) ||
              (!pszConnectName && pPortDlgStruct->szConnectName[0] ) )
         {
            fModified = TRUE;
         }

      }
   }
   pPortDlgStruct->flModified = fModified;

   if (fModified)
   {
      cbNeeded = sizeof(PORTSETTINGS)
               + strlen(pPortDlgStruct->szDescription) + 1
               + strlen(pPortDlgStruct->szConnectName) + 1;

      rc = DosAllocMem((PVOID)&pNewPortSettings,
                       cbNeeded,
                       PAG_READ | PAG_WRITE | PAG_COMMIT);
      if (!rc)
      {
         memset (pNewPortSettings, 0, cbNeeded);
         pNewPortSettings->signature = pPortSettings->signature ;
         pNewPortSettings->ulVersion = pPortSettings->ulVersion ;
         pNewPortSettings->ulPrintTimeOut   = pPortDlgStruct->ulPrintTimeOut ;
         pNewPortSettings->ulNoQueryTimeOut = pPortDlgStruct->ulNoQueryTimeOut ;
         pNewPortSettings->ulNoJobTimeOut   = pPortDlgStruct->ulNoJobTimeOut ;
         pszTemp = (PSZ)((PBYTE)pNewPortSettings + sizeof(PORTSETTINGS));
         if (pPortDlgStruct->szDescription[0])
         {
            pNewPortSettings->ulpszDescription = (ULONG)pszTemp - (ULONG)pNewPortSettings ;
            strcpy( pszTemp, pPortDlgStruct->szDescription );
            pszTemp += strlen(pszTemp) + 1 ;
         }
         if (pPortDlgStruct->szConnectName[0])
         {
            pNewPortSettings->ulpszConnectName = (ULONG)pszTemp  - (ULONG)pNewPortSettings ;
            strcpy( pszTemp, pPortDlgStruct->szConnectName );
            pszTemp += strlen(pszTemp) + 1 ;
         }

         if (pfnPrtSet)
         {
            rc = MyPrtSet( pPortDlgStruct->pszComputerName,
                           pPortDlgStruct->pszPortName,
                           TYPE_LONG_WAIT,
                           BIDI_SET_PORTDRV,
                           pNewPortSettings,
                           cbNeeded );

         } else {
            /*
             * Bidi spooler APIs not available so call routine directly
             */
             rc = HandleSetPortDRV ( pPortDlgStruct->pszPortName,
                                     pNewPortSettings,
                                     cbNeeded );
         }
          DosFreeMem( pNewPortSettings );
      }
   }
   #ifdef DEBUG
    {
      sprintf( logbuf, "SavePortSettings rc=%d\r\n",rc);
      LogCall( logbuf );
    }
   #endif /* DEBUG */
   if (rc)
   {
      return(FALSE);
   }
   return TRUE;
}

/****************************************************************************
 *
 * FUNCTION NAME = RemoveLeadTrailBlanks
 *
 * DESCRIPTION   = Remove all the leading blanks and all the trailing blanks
 *                 from the source string. The target string contains no lead
 *                 or trail blanks. Source string is not altered.
 *
 * INPUT         = pTarget   - Target string.
 *                 pSource   - Source string.
 *
 * OUTPUT        = Void function.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 *      Note : Function accepts the same pointer for both source and target.
 *             This means that same buffer can be used as input and output.
 *
 ****************************************************************************/

VOID RemoveLeadTrailBlanks ( PCH pTarget,
                             PCH pSource )
{
   while (*pSource == ' ' || *pSource == '\t' || *pSource == '\n')
      pSource++;
   if (!(*pSource))
   {
      *pTarget = '\0';
      return;
   }
   while (*pSource)
      *pTarget++ = *pSource++ ;

   pTarget--;
   while (*pTarget == ' ' || *pTarget == '\t' || *pTarget == '\n')
      pTarget-- ;
   /*
   ** Set first blank to null
   */
   *++pTarget = '\0';

}

/****************************************************************************
 *
 * FUNCTION NAME = DisplayError
 *
 * DESCRIPTION   = Display error having string from the resource file.
 *
 * INPUT         = hwndOwner     - Owner of message box.
 *                                    if NULL, default is last active window.
 *                 usStringID    - ID of string in resource file.
 *                 usWinStyle    - Window style of message box.
 *                                    if NULL, default is MB_OK.
 *
 * OUTPUT        = User-response value returned by WimMessageBox API.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT DisplayError ( HWND hwndOwner,
                      USHORT usStringID,
                      USHORT usWinStyle )
{
   CHAR pszTitle[256];                       /*  Message-box window title   */
   CHAR pszText[256];                        /*  Message-box window message */
   USHORT usResponse;
   HAB    hAB;

   hAB = WinQueryAnchorBlock (HWND_DESKTOP);
   WinLoadString (hAB, hPdrMod, PORT_ERR_TITLE, 255, (PSZ)pszTitle);
   WinLoadString (hAB, hPdrMod, usStringID, 255, (PSZ)pszText);
   if (!hwndOwner)
   {
      hwndOwner = WinQueryActiveWindow (HWND_DESKTOP);
   }
   if (!usWinStyle)
   {
      usWinStyle = MB_OK;
   }
   usResponse = WinMessageBox (HWND_DESKTOP, hwndOwner, pszText, pszTitle, 1,
                                                           (ULONG)usWinStyle);
   return (usResponse);

}

/****************************************************************************
 *
 * FUNCTION NAME = DE
 *
 * DESCRIPTION   = Display Error message
 *
 * INPUT         = str  - error message string to display
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID DE ( PCH str )
{
       WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                 (PCH)str,
                 (PCH)"Error",
                 1,
                 MB_OKCANCEL | MB_APPLMODAL |
                 MB_MOVEABLE | MB_ICONASTERISK);

}

/****************************************************************************
 *
 * FUNCTION NAME = CreateControlThread - INTERNAL API
 *
 * DESCRIPTION   = Create thread to monitor IR port connections
 *
 * INPUT         = None
 *
 * OUTPUT        = tid   - new thread ID, 0 if failed to create thread
 *
 * NOTE          = Only called once when BIDI_INIT_PORTDRV command recieved
 *
 ****************************************************************************/

TID   CreateControlThread( VOID )
{
   TID   tid;
   ULONG rc;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   tid = 0;

   rc = DosCreateThread( &tid,
                         (PFNTHREAD)ControlThread,
                         0,
                         0,                         // Execution Flag
                         16384);                    // Stack Size 16K
   #ifdef DEBUG
    {
      sprintf( logbuf, "CreateControlThread rc=%d tidControlThread=%d\r\n",rc, tid);
      LogCall( logbuf );
    }
   #endif /* DEBUG */

   tidControlThread = tid;

   return tid;
}


/****************************************************************************
 *
 * FUNCTION NAME = ControlThread - INTERNAL API
 *
 * DESCRIPTION   = Thread to monitor and control all IR ports.
 *                 Only 1 of these threads will exist.
 *
 * INPUT         = None
 *
 * OUTPUT        = None, exits when port driver shutdown by
 *                 the BIDI_SHUTDOWN command
 *
 ****************************************************************************/

VOID EXPENTRY ControlThread( VOID )
{
    ULONG       rc;
    ULONG       ulPostCount;
    ULONG       ulWakeupTime;     /* Time(in secs) to wake up again */
    ULONG       ulWaitTime;       /* #msecs to wait for EventSem    */
    ULONG       ulNextTimeOut;    /* Largest of Job or Query time to wakeup */
    ULONG       ulCurrTime;       /* Current time in seconds */
    ULONG       ulCheckQueryTime; /* Time to close port based on last query */
    ULONG       ulCheckJobTime;   /* Time to close port based on last job   */
    PPORTINST   pPortInst;
    #ifdef DEBUG
     CHAR logbuf[260];            // For calling LogCall
    #endif

    /*
    ** Create non-shared event sem for this thread
    */
    rc = DosCreateEventSem((PSZ)NULL,&hevControl,
                           0, FALSE);
    do
    {
       rc = DosResetEventSem(hevControl, &ulPostCount);
      EnterPdrSem();
       ulCurrTime  = time();
       ulWakeupTime = SEM_INDEFINITE_WAIT;
       for (pPortInst = pPortInstList; pPortInst; pPortInst = pPortInst->pNext)
       {
          /*
          ** Search through list of IR ports
          ** If any need to be closed, close them now.
          */
          if ( (pPortInst->flStatus & PF_PORT_OPEN) &&
               (pPortInst->ulJobPrinting != PDR_PRINTING) &&
               (pPortInst->ulJobPrinting != PDR_ABORTED) )
          {

             ulCheckQueryTime = pPortInst->ulTimeLastCmd + pPortInst->ulNoQueryTimeOut;
             ulCheckJobTime   = pPortInst->ulTimeLastJob + pPortInst->ulNoJobTimeOut;
             if ( (ulCurrTime > ulCheckQueryTime) &&
                  (ulCurrTime > ulCheckJobTime) )
             {
                /*
                ** Our timeout for printing a job and waiting for
                **   the next query has expired.
                ** We will close the connection, and the next time
                **   we get a query or job in, we will generate a
                **   poweron reset alert to allow the protocol converter
                **   to re-enable alerts with the printer.
                ** Note that if a job confirmation does not come before
                **   we close the connection, the spooler will keep the
                **   job as "In Printer" status.
                */
                #ifdef DEBUG
                 {
                   sprintf( logbuf, "ControlThread closing port not in use TimeLastJob=%d TimeLastQuery=%d\r\n", pPortInst->ulTimeLastJob, pPortInst->ulTimeLastCmd);
                   LogCall( logbuf );
                 }
                #endif /* DEBUG */
                InfraredClose(&pPortInst->LMParms, 2, &pPortInst->hFile);
                pPortInst->flStatus &= ~PF_PORT_OPEN;
             } else {
                /*
                ** Now determine how long we should wait until the next
                **   time we check.
                ** If a port is printing a job
                **   we wait indefinitely because SplPdClose will wake
                **   this thread up.
                ** If any port is not printing a job
                **   we will check for the next time we should close the port.
                ** If we are not waiting indefinitely
                **   set fControlWakeupPending so that SplPdSendCmd does not
                **   keep waking up the control thread.
                */
                if ( ulCheckQueryTime > ulCheckJobTime )
                   ulNextTimeOut = ulCheckQueryTime;
                else
                   ulNextTimeOut = ulCheckJobTime;

                if (ulNextTimeOut < ulWakeupTime)
                   ulWakeupTime = ulNextTimeOut;
             }
          }

       }
       /*
       ** We have checked all the ports.
       ** Now determine how long we should wait until the next time we check.
       ** If a port is printing a job
       **   we wait indefinitely because SplPdClose will wake this thread up.
       ** If any port is not printing a job
       **   we will check for the next time we should close the port.
       ** If we are not waiting indefinitely
       **   set fControlWakeupPending so that SplPdSendCmd does not keep waking
       **   up the control thread and convert ulWakeupTime from seconds
       **   to milliseconds.
       */
       if (ulWakeupTime == (ULONG)SEM_INDEFINITE_WAIT)
       {
          fControlWakeupPending = FALSE;
          ulWaitTime = SEM_INDEFINITE_WAIT;
       } else {
          fControlWakeupPending = TRUE;
          ulWakeupTime -= ulCurrTime ;
          ulWaitTime    = ulWakeupTime * 1000;
       }
      LeavePdrSem();
       if (!fShutdownInProgress)
       {
          #ifdef DEBUG
           {
             sprintf( logbuf, "ControlThread about to Wait %d\r\n", ulWakeupTime);
             LogCall( logbuf );
           }
          #endif /* DEBUG */
          rc = DosWaitEventSem ( hevControl, ulWaitTime );
       }
    } while ( !fShutdownInProgress );
}

/*
** infrared callback functions
*/
BOOLEAN LMINDIC lmConnectIndic(
                          HCONNECTION      hConnect
                        , LP_CONNECT_PARMS pConnectParms
                        , LP_VOID          pUserData
                        , U16              size
                        , LSAPSEL          lsap   /* additional parameter */
                        )
{
   //LMSTATUS error;
   //int      rc;
   //PPDOPENINST pPdOpenInstance = xxx;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   #ifdef DEBUG
    {
      sprintf( logbuf, "lmConnectIndic\r\n");
      LogCall( logbuf );
    }
   #endif /* DEBUG */

   //fConnected  = true;
   //fConnecting = false;
   //fNewConnection = true;

   //memcpy(&pPdOpenInstance->ConnectParms,pConnectParms,sizeof(CONNECT_PARMS));
   //memcpy(connectMsg,pUserData,size);
   //connectMsg[57] ='\0';
   //hConn          = hConnect;
   //ConHandle      = hConnect;
   //rc = (*pPdOpenInstance->lmpfns.ConnectResponse)(hConnect,connectMsg,size,true,&error);

   return(TRUE);
}

BOOLEAN LMINDIC lmDisconnectIndic(
                          HCONNECTION      hConnect
                        , LP_VOID          pUserData
                        , U16              size
                        , LMSTATUS         lmStatus
                        , U32              userContext
                        )

{

   /*
   ** get out instance data
   */
   PLMPARMS pLMParms = (PLMPARMS)userContext;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   #ifdef DEBUG
    {
      sprintf( logbuf, "lmDisconnectIndic prior fDisconnect=%d\r\n", pLMParms->fDisconnected);
      LogCall( logbuf );
    }
   #endif /* DEBUG */

   /*
   ** set the disconnect flag before making sure
   ** the other  threads are not blocked
   */
   pLMParms->fDisconnected = TRUE;

   /*
   ** If we did not connect we will get
   ** a disconnect.  Therefore, we must
   ** make sure we are not waiting for the connect by
   ** posting the connect sem.
   */
   DosPostEventSem(pLMParms->hevConnect);

   /*
   ** make sure we are not waiting for a write to
   ** complete
   */
   if (pLMParms->hevWriteComplete)
     DosPostEventSem(pLMParms->hevWriteComplete);

   /*
   ** make sure we are not waiting for the disconnect
   ** posting the disconnect sem.
   */
   DosPostEventSem(pLMParms->hevDisconnect);

   return(TRUE);
}

BOOLEAN LMINDIC lmDataIndic(
                          HCONNECTION      hConnect
                        , LP_VOID          pImmedData
                        , U16              sizeImmed
                        , U16              sizeAll
                        , HUSERDATA        hUserData
                        , U32              userContext
                        )

{
   PPORTINST     pPortInst;
   PLMPARMS      pLMParms;
   PWAITALERTBUF pWaitBuf;           /* Current WAITALERTBUF we are using  */
   PALERTBUF     pAlertBuf;
   PALERTBUF     pNewAlert;
   PBYTE         pBuf;
   ULONG         cbNeeded;
   INT           iRC;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   /*
   ** get out instance data
   */
   pLMParms = (PLMPARMS)userContext;

   #ifdef DEBUG
    {
      sprintf( logbuf, "lmDataIndic sizeImmed=%d sizeAll=%d\r\n", sizeImmed, sizeAll);
      LogCall( logbuf );
    }
   #endif /* DEBUG */
   #ifdef DEBUG_ALERT
    DumpHex( (PBYTE)pImmedData, sizeImmed);
   #endif

  EnterPdrSem();
   if (pLMParms->pPortInst) {
      pPortInst = pLMParms->pPortInst;
      pWaitBuf  = PdrWaitAlert.pCurrBuf;
      if (pWaitBuf) {
         //
         // Handle case where entire buffer won't fit @BUGBUG
         // For this sample, any alert that will not fit is tossed.
         //
         cbNeeded = pWaitBuf->cbBufUsed + (USHORT)sizeAll + sizeof(ALERTBUF);
         if (cbNeeded <= pWaitBuf->cbBufSize)
         {
           //
           // There is room to store this alert data in the WaitAlertBuf
           //
           if (pWaitBuf->pLastStored)
           {
              pAlertBuf = pWaitBuf->pLastStored;
              pNewAlert = (PALERTBUF)((PBYTE)pAlertBuf +
                                             sizeof(ALERTBUF) +
                                             pAlertBuf->cbData);
           } else {
              pNewAlert = &pWaitBuf->FirstAlertBuf;
           }
           pNewAlert->pszAlertPortName = pPortInst->pszPortName;
           pNewAlert->cbData = (USHORT)sizeAll;
           pBuf = (PBYTE)pNewAlert + sizeof(ALERTBUF);
           //memcpy( pBuf, pImmedData, (USHORT)sizeAll );
           iRC = (*pLMParms->lmpfns.pfnLM_ReceiveData2)(hConnect,
                                                        hUserData,
                                                        (LP_VOID)pBuf,
                                                        0, sizeAll);
#if DEBUG
           if (iRC != STSOK)
           {
              #ifdef DEBUG
               {
                 sprintf( logbuf, "IR LM_ReceiveData2 Failed iRC=%d\r\n",iRC);
                 LogCall( logbuf );
               }
              #endif /* DEBUG */
           }
           else
           {
#ifdef DEBUG_ALERT
             DumpHex( (PBYTE)pBuf, sizeAll);
#endif
           }
#endif
           pWaitBuf->cbBufUsed = cbNeeded;
           pWaitBuf->pLastStored = pNewAlert;
           pWaitBuf->cAlertsRemaining++;
         }
         DosPostEventSem(PdrWaitAlert.hevWaitAlert);
      }
   }
  LeavePdrSem();
   return(TRUE);

}

BOOLEAN LMINDIC lmExpDataIndic(
                          HCONNECTION      hConnect
                        , LP_VOID          pImmedData
                        , U16              sizeImmed
                        , U16              sizeAll
                        , HUSERDATA        hUserData
                        , U32              userContext
                        )
{
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   #ifdef DEBUG
    {
      sprintf( logbuf, "IR Expedited Data Indication\r\n");
      LogCall( logbuf );
    }
   #endif /* DEBUG */
   //flushall();
   //rc = (*pLMParms->lmpfns.ReceiveData2)(hConnect,hUserData,&buf,0,sizeAll);
   return(TRUE);
}

// Qos not currently implemented


BOOLEAN LMINDIC lmQosIndic(
                          HCONNECTION      hConnect
                        , LP_VOID          pQoSParam
                        , U32              userContext
                        )
{
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   #ifdef DEBUG
    {
      sprintf( logbuf, "IR QosIndic\r\n");
      LogCall( logbuf );
    }
   #endif /* DEBUG */
   return(TRUE);
}

VOID LMCONF lmConnectConf(unsigned short hConnection,
                             LP_CONNECT_PARMS lpConnectParms,
                             LP_VOID pUserData,
                             U16 lenUserData,
                             U32 userContext
                            )
{
   PLMPARMS pLMParms;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   #ifdef DEBUG
    {
      sprintf( logbuf, "IR Connection Confirmation, handle = %d\r\n",hConnection);
      LogCall( logbuf );
    }
   #endif /* DEBUG */

   pLMParms = (PLMPARMS)userContext;
   pLMParms->ushConnection = hConnection;
   /*
   ** Copy the connection parameters
   */
   memcpy( (PVOID)&pLMParms->ConnectParms,
           (PVOID)lpConnectParms, sizeof(CONNECT_PARMS));
   /*
   ** let the SplPdOpen go by posting the
   ** connection semaphore
   */
   DosPostEventSem(pLMParms->hevConnect);
}


VOID LMCONF lmTransmitConf(
                          HCONNECTION      hConn
                        , HUSERDATA        hUserData
                        , LMSTATUS         status
                        , U32              userContext
                        )

{
   PDATAINFO pDataInfo;
   PLMPARMS  pLMParms = (PLMPARMS)userContext;
   ULONG     rc;
   BOOL      fFound;
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif

   #ifdef DEBUG
    {
      sprintf( logbuf, "IR Transmit Confirmation LMSTATUS=%d BufferReturned=%lX\r\n",status, (ULONG)hUserData);
      LogCall( logbuf );
    }
   #endif /* DEBUG */

   pDataInfo   = (PDATAINFO)hUserData;      /* data header&handle pointer     */
   fFound      = FALSE;

  EnterPdrSem();

   if (status == STSOK)
   {
     pLMParms->ulBytesConfirmed += pDataInfo->ulDataSize;
     pLMParms->ulBuffersConfirmed++;
     #ifdef DEBUG
      {
        sprintf( logbuf, "IR Confirm buffer OK DataSize=%d TotalBytesconfirmed=%d BuffersConfirmed=%d\r\n", pDataInfo->ulDataSize, pLMParms->ulBytesConfirmed, pLMParms->ulBuffersConfirmed);
        LogCall( logbuf );
      }
     #endif /* DEBUG */
     fFound = RemoveDataInfoBuf( &pLMParms->pWaitingConfirmBufs, pDataInfo );
   } /* endif */
   else
   {
     #ifdef DEBUG
      {
        sprintf( logbuf, "IR Confirm buffer NOT GOOD DataSize=%d TotalBytesconfirmed=%d BuffersConfirmed=%d\r\n", pDataInfo->ulDataSize, pLMParms->ulBytesConfirmed, pLMParms->ulBuffersConfirmed);
        LogCall( logbuf );
      }
     #endif /* DEBUG */
   }
   if (pDataInfo->ulDataType && IRDATATYPE_LASTBUFFER)
   {
     #ifdef DEBUG
      {
        sprintf( logbuf, "IR Confirm LASTBUFFER in transaction\r\n");
        LogCall( logbuf );
      }
     #endif /* DEBUG */
     DosPostEventSem(pLMParms->hevWriteComplete);
   } /* endif */

   /*
   ** If buffer was on our list waiting for confirmations
   ** then place it onto our confirmed list for InfraredWrite()
   **  to remove later
   ** else
   **  just free the buffer(we were not waiting for a confirmation)
   **/
   if (fFound) {
      AddDataInfoBuf( &pLMParms->pConfirmedBufs, pDataInfo );
   } else {
      rc = DosFreeMem ( (PVOID)pDataInfo );
      ASSERTT(rc);
   }
  LeavePdrSem();
}

// Qos not currently implemented

VOID LMCONF lmQosConf(
                          HCONNECTION      hConnect
                        , LP_VOID          pQoSParam
                        , U32              userContext
                     )


{
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif
   #ifdef DEBUG
    {
      sprintf( logbuf, "IR QosConf\r\n");
      LogCall( logbuf );
    }
   #endif /* DEBUG */
}

// discover is currently synchronous

VOID LMCONF lmDiscoverConf(
                       LP_DEVICE_DISCOVERY_LIST lpDevDiscList)
{
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif
   #ifdef DEBUG
    {
      sprintf( logbuf, "IR Discover Confirmation\r\n");
      LogCall( logbuf );
    }
   #endif /* DEBUG */
}

BOOLEAN LMINDIC lmDiscoverIndic(
                       LP_DEVICE_DISCOVERY_LIST lpDevDiscList)
{
   #ifdef DEBUG
    CHAR logbuf[260];            // For calling LogCall
   #endif
   #ifdef DEBUG
    {
      sprintf( logbuf, "IR Discover Indication\r\n");
      LogCall( logbuf );
    }
   #endif /* DEBUG */
   return(TRUE);
}

/*
** END file infrared.c
*/
