/************************************************************
* .FILE:        mwin.cpp                                    *
*                                                           *
* .DESCRIPTION: MandelBrot Sample Program:                  *
*                 - Class Implementation                    *
*                                                           *
* .CLASSES:     MainWindow                                  *
*                                                           *
* .COPYRIGHT:                                               *
*   IBM Open Class Library                                  *
*   Licensed Material - Property of IBM                     *
*   (C) Copyright IBM Corp. 1998, 2000. All Rights Reserved *
*                                                           *
* .DISCLAIMER:                                              *
*   The following [enclosed] code is sample code created by *
*   IBM Corporation.  This sample code is not part of any   *
*   standard IBM product and is provided to you solely for  *
*   the purpose of assisting you in the development of your *
*   applications.  The code is provided 'AS IS', without    *
*   warranty of any kind.  IBM shall not be liable for any  *
*   damages arising out of your use of the sample code,     *
*   even if they have been advised of the possibility of    *
*   such damages.                                           *
*                                                           *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT   *
*        THE SOURCE                                         *
*                                                           *
************************************************************/
#include <iframe.hpp>
#include <i2dghand.hpp>
#include <icanvas.hpp>
#include <icmdhdr.hpp>
#include <icoordsy.hpp>
#include <iexgrprt.hpp>
#include <ihandle.hpp>
#include <iimage.hpp>
#include <imenubar.hpp>
#include <ipainhdr.hpp>
#include <isysmenu.hpp>
#include <itimer.hpp>
#include <locale.h>

#include "mandel.h"
#include "mandim.hpp"
#include "progdial.hpp"

/************************************************************
* Class MainWindow                                          *
*  - Application's main window.                             *
************************************************************/
class MainWindow : public IFrameWindow,
                   public MandelbrotImageGenerator::Observer
{
public:
  // Constructor
  MainWindow ( unsigned long windowId );

  // Destructor
virtual
 ~MainWindow ( );

protected:
  // Callback function to notify this object as to how much
  // of the Mandelbrot image has been generated.
virtual void
  updateProgress ( unsigned long percent );

  // Callback function to allow image generation to be
  // canceled.
virtual bool
  cancelImage    ( );

private:
  // Hidden members.
  MainWindow ( const MainWindow& );
MainWindow
 &operator = ( const MainWindow& );

  // This is the function called by ICommandConnectionTo.
bool
  processCommand ( ICommandEvent& event );

  // This is the function called by IPaintConnectionTo.
bool
  paintWindow    ( IPaintEvent&   event );

  // Stop the secondary thread and close the application.
void
  exitApp ( );

  // Start a secondary thread which will create an image
  // of the Mandelbrot set.
void
  startImageThread ( );

  // Create an image of the Mandelbrot Set. This function
  // will be run on a secondary thread.
void
  generateImage ( );

  // Update the progress indicator to show how much of the
  // image has been generated.
void
  checkProgress ( );

  // The following are states that the primary thread can
  // be in.
enum EPrimaryState {
  kPrimaryIdle,
  kPrimaryGenerating,
  kPrimaryCanceling,
  kPrimaryExiting
  };

  // The following are states that the secondary thread can
  // be in.
enum ESecondaryState {
  kSecondaryIdle,
  kSecondaryGenerating
  };

  // Private data members.
ICanvas
  fDrawingArea;
IMenuBar
  fMenuBar;
EPrimaryState
  fPrimaryState;
ESecondaryState
  fSecondaryState;
INonGUIThread
  fWorkThread;
ITimer
  fTimer;
unsigned long
  fProgress;
ProgressDialog
 *fIndicator;
IImage
 *fImage;
MandelbrotImageGenerator
 *fMandelbrotGenerator;
ICommandConnectionTo< MainWindow >
  fCommandConnectionTo;
IPaintConnectionTo< MainWindow >
  fPaintConnectionTo;

static const int
  kIterationLimit;
}; // MainWindow

const int MainWindow::kIterationLimit = 1000;

/************************************************************
* Class MainWindow constructor                              *
************************************************************/
MainWindow::MainWindow ( unsigned long windowId )
 : IFrameWindow( windowId )
 , fDrawingArea( WND_DRAW, this, this )
 , fMenuBar( WND_MAIN, this )
 , fPrimaryState( kPrimaryIdle )
 , fSecondaryState( kSecondaryIdle )
 , fIndicator( 0 )
 , fImage( 0 )
 , fMandelbrotGenerator( 0 )
 , fCommandConnectionTo( *this, &MainWindow::processCommand )
 , fPaintConnectionTo( *this, &MainWindow::paintWindow )
{
  this->setClient( &fDrawingArea );
  fDrawingArea.setBackgroundColor( IColor::kBlack );

  // Create the progress indicator dialog.
  IResourceLibrary reslib;
  IString str(reslib.loadString(GEN_IMAGE));
  fIndicator = new ProgressDialog( this, str);

  // Start up the event handlers.
  fCommandConnectionTo
   .handleEventsFor( this )
   .handleEventsFor( fIndicator );
  fPaintConnectionTo
   .handleEventsFor( &fDrawingArea );

  this->sizeTo( IWindow::desktopWindow()->size() / 2 );
  this->setFocus();
  this->show();
}

/************************************************************
* Class MainWindow destructor                               *
************************************************************/
MainWindow::~MainWindow()
{
  // Shut down the event handlers.
  fCommandConnectionTo
   .stopHandlingEventsFor( this )
   .stopHandlingEventsFor( fIndicator );
  fPaintConnectionTo
   .stopHandlingEventsFor( &fDrawingArea );

  delete fIndicator;
  delete fImage;
  delete fMandelbrotGenerator;
}

/************************************************************
* Class MainWindow::processCommand                          *
************************************************************/
bool MainWindow::processCommand ( ICommandEvent& event )
{
  bool stopProcessingEvent = true;
  switch ( event.commandId() )
  {
     // "Generate Image" menu item
     case ID_GENERATE:
        this->startImageThread();
        break;

     // Exit menu item
     case ID_EXIT:
        this->exitApp();
        break;

     // Cancel button on the progress dialog
     case WND_CB:

        // Set the flag telling the secondary thread to
        // cancel image generation, and return to the
        // processing the message queue while waiting for it
        // to do so. This gives the secondary thread the
        // opportunity to release any resources it has
        // allocated and destroy any objects it has created.
        // On OS/2, the message queue must continue to be
        // processed while canceling the task (if canceling
        // might take some time), since there is only one
        // message queue. Failure to process the message
        // queue blocks input to other running applications.
        fPrimaryState = kPrimaryCanceling;
        break;

     case IC_ID_CLOSE:
        this->exitApp();
        break;

     default:
        stopProcessingEvent = false;
        break;
  }
  return stopProcessingEvent;
}

/************************************************************
* Class MainWindow::paintWindow                             *
************************************************************/
bool MainWindow::paintWindow ( IPaintEvent& event )
{
  // Clear the drawing area.
  event.clearBackground( IColor::kBlack );

  // If image generation has completed, display the image.
  if ( fPrimaryState == kPrimaryIdle )
  {
     if ( ! fImage  &&  fMandelbrotGenerator )
     {
        fImage = new IImage( fMandelbrotGenerator->image() );
     }
     if ( fImage )
     {
        IPresSpaceHandle presSpace = event.presSpaceHandle();
        IExtendedRootGrafPort grafPort( presSpace );
        fImage->draw( grafPort );
     }
  }

  return true;
}

/************************************************************
* Class MainWindow::imageProgress                           *
************************************************************/
void MainWindow::updateProgress ( unsigned long percent )
{
  fProgress = percent;
}

/************************************************************
* Class MainWindow::cancelImage                             *
************************************************************/
bool MainWindow::cancelImage ( )
{
  return ( fPrimaryState == kPrimaryCanceling  ||
           fPrimaryState == kPrimaryExiting );
}

/************************************************************
* Class MainWindow::exitApp                                 *
************************************************************/
void MainWindow::exitApp ( )
{
  if ( fPrimaryState == kPrimaryGenerating )
  {
     // Set the flag telling the secondary thread to cancel
     // image generation, and return to processing the
     // message queue while waiting for it to do so. This
     // gives the secondary thread the opportunity to release
     // any resources it has allocated, and destroy any
     // objects it has created. On OS/2, the message queue
     // must continue to be processed while canceling the
     // task (if canceling might take some time), since there
     // is only one message queue. Failure to process the
     // message queue blocks input to other running
     // applications.
     fPrimaryState = kPrimaryExiting;
  }
  else
  {
     this->close();
  }
}

/************************************************************
* Class MainWindow::startImageThread                        *
************************************************************/
void MainWindow::startImageThread ( )
{
  fPrimaryState = kPrimaryGenerating;
  fSecondaryState = kSecondaryGenerating;
  fProgress = 0;
  this->setMousePointer( ISystemPointerHandle(
                            ISystemPointerHandle::kWait ) );
  fIndicator->setPercentage( 0 );

  // Position the indicator dialog in the center of the
  // screen.
  ISize sz = fIndicator->size();
  IRectangle crect = IWindow::desktopWindow()->rect();
  IPoint pos( crect.minX() +
                ( crect.width() - sz.width() ) / 2,
              crect.minY() +
                ( crect.height() - sz.height() ) / 2 );
  fIndicator->moveTo( pos );
  fIndicator->show();
  fIndicator->setFocus();

  // Disable the menu item so the user start this action
  // again.
  fMenuBar.disableItem( ID_GENERATE );

  // Define the portion of the complex plane to be drawn.
  // Ensure it includes the Mandelbrot set, and has the same
  // aspect ratio as the drawing area. Center the image at
  // complex(-0.7, 0.0). For the area that should be
  // included, the upper left is at complex(-2.0, 1.3) and
  // the lower right is at complex(0.6, -1.3).
  complex upperLeft;
  complex lowerRight;
  unsigned long width = this->client()->size().width();
  unsigned long height = this->client()->size().height();
  if ( width > height )
  {
     double d = 1.3 * width / height;
     upperLeft  = complex( -0.7 - d, 1.3 );
     lowerRight = complex( -0.7 + d, -1.3 );
  }
  else
  {
     double d = 1.3 * height / width;
     upperLeft  = complex( -2.0, d );
     lowerRight = complex( 0.6, -d );
  }

  // Create a new image object representing the portion
  // of the complex plane to be drawn.
  delete fMandelbrotGenerator;
  fMandelbrotGenerator =
     new MandelbrotImageGenerator( upperLeft, lowerRight,
                                   width, height,
                                   kIterationLimit );

  // Start another thread and have it run the member function
  // that generates the image.
  fWorkThread.start( new IThreadMemberFn< MainWindow >
                     ( *this, &MainWindow::generateImage ) );

  // Start the timer which will check on the progress of the
  // image generation twice per second.
  fTimer.setInterval( 500 );
  fTimer.start( new ITimerMemberFn0< MainWindow >
                     ( *this, &MainWindow::checkProgress ) );
}

/************************************************************
* Class MainWindow::generateImage                           *
************************************************************/
void MainWindow::generateImage ( )
{
  // Generate the image. This object is passed as the
  // Observer; it will be notified of the progress of the
  // image generation (via a call to updateProgress).
  fMandelbrotGenerator->generate( *this );
  fSecondaryState = kSecondaryIdle;
}

/************************************************************
* Class MainWindow::checkProgress                           *
************************************************************/
void MainWindow::checkProgress ( )
{
  if ( fSecondaryState == kSecondaryGenerating )
  {
     fIndicator->setPercentage( fProgress );
  }
  else
  {
     bool imageGenerated = true;
     switch (fPrimaryState)
     {
        case kPrimaryCanceling:
           imageGenerated = false;
           delete fMandelbrotGenerator;
           fMandelbrotGenerator = 0;

           // Fall through to the next case.

        case kPrimaryGenerating:
           fPrimaryState = kPrimaryIdle;
           fTimer.stop();
           fIndicator->hide();
           if ( imageGenerated )
           {  // Delete the previous image and cause the new
              // one to be next.
              delete fImage;
              fImage = 0;
              fDrawingArea.refresh( IWindow::paintAll );
           }
           fMenuBar.enableItem( ID_GENERATE );
           this->setMousePointer( IPointerHandle( 0 ) );
           break;

        case kPrimaryExiting:
           imageGenerated = false;
           this->close();
           break;

        default:
           break;
     }
  }
}

/************************************************************
* main                                                      *
************************************************************/
int main ( )
{
  setlocale(LC_ALL, "");

  // Specify an upper-left coordinate system.
  ICoordinateSystem::setApplicationOrientation(
                        ICoordinateSystem::originUpperLeft );

  MainWindow mainWindow( WND_MAIN );
  IApplication::current().run();
  return 0;
}
