/************************************************************
* .FILE:        mandim.hpp                                  *
*                                                           *
* .DESCRIPTION: MandelBrot Sample Program:                  *
*                 - Class Header                            *
*                                                           *
* .CLASSES:     MandelbrotImageGenerator                    *
*                                                           *
* .COPYRIGHT:                                               *
*   IBM Open Class Library                                  *
*   Licensed Material - Property of IBM                     *
*   (C) Copyright IBM Corp. 1998, 2000. All Rights Reserved *
*                                                           *
* .DISCLAIMER:                                              *
*   The following [enclosed] code is sample code created by *
*   IBM Corporation.  This sample code is not part of any   *
*   standard IBM product and is provided to you solely for  *
*   the purpose of assisting you in the development of your *
*   applications.  The code is provided 'AS IS', without    *
*   warranty of any kind.  IBM shall not be liable for any  *
*   damages arising out of your use of the sample code,     *
*   even if they have been advised of the possibility of    *
*   such damages.                                           *
*                                                           *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT   *
*        THE SOURCE                                         *
*                                                           *
************************************************************/
#ifndef MANDIM_HPP
#define MANDIM_HPP

#include <complex.h>
#include <igimage.hpp>

/************************************************************
* Class MandelbrotImageGenerator                            *
*  - Creates an image representing points on a complex      *
*    plane. Pixels are colored according to how "close" the *
*    points are to the Mandelbrot set.  Closeness is        *
*    defined in terms of how many iterations it took to     *
*    determine the point was not in the Mandelbrot set.     *
************************************************************/
class MandelbrotImageGenerator {
public:
  // Constructor
  //  - Initialize the object. The area of the complex plane
  //    belonging to the image is defined by the points
  //    "upperLeft" and "lowerRight." The size (in pixels)
  //    of the image to generate is specified by "width" and
  //    "height." The "limit" is the maximum number of
  //    iterations to allow before deciding a point is within
  //    the Mandelbrot set.
  MandelbrotImageGenerator ( complex       upperLeft,
                             complex       lowerRight,
                             unsigned long width,
                             unsigned long height,
 	                          unsigned      limit );

  // Destructor
 ~MandelbrotImageGenerator ( );

  /********************************************************
  * Class MandelbrotImageGenerator::Observer              *
  *  - Objects of this class are notified as to the       *
  *    progress of the image being generated by a         *
  *    "MandelbrotImageGenerator" object.                 *
  ********************************************************/
  class Observer {
  public:
  virtual
    // Destructor
   ~Observer ( ) { }

  protected:
    // Protected constructor to call from derived classes
    Observer ( ) { }

    // Callback function called by a MandelbrotImageGenerator
    // object to notify of the progress in generating an
    // image. The parameter "percent" specifies what amount
    // of the overall image has been generated.
  virtual void
    updateProgress ( unsigned long percent ) = 0;

    // Callback function called periodically by a
    // MandelbrotImageGenerator object to determine if
    // generation of the image should be continued or
    // canceled.
  virtual bool
    cancelImage ( ) = 0;
  private:
    friend class MandelbrotImageGenerator;
  }; // MandelbrotImageGenerator::Observer

  // Generate the image.  The "observer" object will be
  // notified of the progress of the image generation.
void
  generate ( Observer& observer );

  // Return the image created by the generate function.
IGImage
  image    ( ) const;

private:
  // Hidden members.
  MandelbrotImageGenerator ( const MandelbrotImageGenerator& );
MandelbrotImageGenerator
 &operator=                ( const MandelbrotImageGenerator& );

  // Private data members.
complex
  fTopLeft,
  fBottomRight;
unsigned long
  fWidth,
  fHeight;
unsigned
  fLimit;
IGImage
 *fGImage;

  /********************************************************
  * Struct MandelbrotImageGenerator::ComplexCache         *
  *  - Represents a complex number. It caches the squares *
  *    of the real and imaginary components to allow for  *
  *    faster calculations.                               *
  ********************************************************/
  struct ComplexCache
  {
    double
      r,
      i,
      rsq,
      isq;

    ComplexCache ( double real, double imag )
      : r( real ),
        i( imag )
    {
      rsq = r * r;
      isq = i * i;
    }
  }; // MandelbrotImageGenerator::ComplexCache
}; // MandelbrotImageGenerator

#endif // MANDIM_HPP
