/************************************************************
* .FILE:        mandim.cpp                                  *
*                                                           *
* .DESCRIPTION: MandelBrot Sample Program:                  *
*                 - Class Implementation                    *
*                                                           *
* .CLASSES:     MandelBrotImageGenerator                    *
*                                                           *
* .COPYRIGHT:                                               *
*   IBM Open Class Library                                  *
*   Licensed Material - Property of IBM                     *
*   (C) Copyright IBM Corp. 1998, 2000. All Rights Reserved *
*                                                           *
* .DISCLAIMER:                                              *
*   The following [enclosed] code is sample code created by *
*   IBM Corporation.  This sample code is not part of any   *
*   standard IBM product and is provided to you solely for  *
*   the purpose of assisting you in the development of your *
*   applications.  The code is provided 'AS IS', without    *
*   warranty of any kind.  IBM shall not be liable for any  *
*   damages arising out of your use of the sample code,     *
*   even if they have been advised of the possibility of    *
*   such damages.                                           *
*                                                           *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT   *
*        THE SOURCE                                         *
*                                                           *
************************************************************/
#include "mandim.hpp"

/************************************************************
* Class MandelBrotImageGenerator constructor                *
************************************************************/
MandelbrotImageGenerator::MandelbrotImageGenerator
                            ( complex       upperLeft,
                              complex       lowerRight,
				                  unsigned long width,
                              unsigned long height,
				                  unsigned      limit )
 : fTopLeft( upperLeft ),
   fBottomRight( lowerRight ),
   fWidth( width ),
   fHeight( height ),
   fLimit( limit ),
   fGImage( 0 )
{
  fGImage = new IGImage( width, height,
                         IGImage::k256Color8Bit );
}

/************************************************************
* Class MandelBrotImageGenerator destructor                 *
************************************************************/
MandelbrotImageGenerator::~MandelbrotImageGenerator ( )
{
  delete fGImage;
}

/************************************************************
* Class MandelBrotImageGenerator::generate                  *
************************************************************/
void
MandelbrotImageGenerator::generate ( Observer& observer )
{
  // Create the image of the Mandelbrot set. For each complex
  // number c within the defined range, iterate over the
  // calculation z = z*z + c, where z starts at
  // complex(0.0, 0.0). If the magnitude of z ever exceeds
  // 2.0, the point is not in the Mandelbrot set. If the
  // iteration reaches the predefined limit, decide that it
  // probably is in the set.

  IGImagePixelAccessor pa( *fGImage );

  // Percent complete
  unsigned long percent = 0;

  double re, im;
  unsigned long xi, yi;
  const double STEP_RE =
      ( real( fBottomRight ) - real( fTopLeft ) ) / fWidth;
  const double STEP_IM =
      ( imag ( fBottomRight ) - imag( fTopLeft ) ) / fHeight;

  for ( re = real(fTopLeft), xi = 0;
        xi < fWidth;
        ++xi, re += STEP_RE )
  {
     for ( im = imag( fTopLeft ), yi = 0;
           yi < fHeight;
           ++yi, im += STEP_IM )
     {
        // Every tenth point, check if the observer wants to
        // cancel the image generation.
        if ( yi % 10 == 0  &&  observer.cancelImage() )
        {
           return;
        }

        // Determine if the current point, complex(re, im) is
        // in the Mandelbrot set.
        ComplexCache z( 0.0, 0.0 );
        unsigned long count;
        for ( count = 0;
              count < fLimit  &&  z.rsq + z.isq < 4.0;
              ++count)
        {
           // The following code calculates
           // z = z * z + c, where c is complex(re, im).
           double rtmp = z.rsq - z.isq + re;
           z.i = z.r * z.i * 2.0 + im;
           z.isq = z.i * z.i;
           z.r = rtmp;
           z.rsq = rtmp * rtmp;
        }
        if ( count == fLimit )
        {
           // Point is probably in the Mandelbrot set.
           pa.setPixel( IGPoint2D( xi, yi ),
                        IColor::kBlack );
        }
        else
        {
           // Point is not it the Mandelbrot set.  Set the
           // color of the pixel based on how many iterations
           // it took before deciding this. The following
           // algorithm produces an interesting looking
           // picture.
           const unsigned long
             MAX   = 500,
             LIMIT = 64,
             MULT  = 256 / LIMIT;
           unsigned long fac;
           for ( fac = 0; fac < LIMIT; ++fac )
           {
              if ( count >= MAX / ( fac + 1 ) )
              {
                 break;
              }
           }
           if ( fac < LIMIT )
           {
              CharIntensity primary =
                     (CharIntensity) ( 255 - fac * MULT );
              pa.setPixel( IGPoint2D( xi, yi ),
                           IBaseColor( primary,
                                       primary,
                                       primary ) );
           }
           else
           {
              pa.setPixel( IGPoint2D( xi, yi ),
                           IColor::kBlack );
           }
        }
     }

     // If the percentage of columns completed has changed by
     // at least one percent, notify the observer.
     unsigned long npercent = xi * 100 / fWidth;
     if ( npercent > percent )
     {
        percent = npercent;
        observer.updateProgress( percent );
     }
  }
  // Finally finished!
  observer.updateProgress( 100 );
}

/************************************************************
* Class MandelBrotImageGenerator::image                     *
************************************************************/
IGImage MandelbrotImageGenerator::image ( ) const
{
  return IGImage( *fGImage );
}
