/*
-------------------------------------------------------------------------------
Copyright (c) IBM Corporation 1997.
All Rights Reserved.

Permission is granted to copy, use, modify, and merge this software into your
applications and to permit others to do any of the foregoing. You must include
this permission and copyright notice in all copies and modified versions of
this software. THIS SOFTWARE IS PROVIDED IN ITS 'AS IS' CONDITION.
IBM DISCLAIMS ANY LIABILITY OF ANY KIND FOR DAMAGES WHATSOEVER RESULTING FROM
THE USE OF THIS SOFTWARE.
-------------------------------------------------------------------------------
*/
// Revision: 01 1.3.3.1 samples/testfw/tautotst.cpp, ocsamples-L1, ioc.v400, 001006 
#include "iargdict.hpp"
#include <math.h>
#include "tautotst.hpp"

//-----------------------------------------------------------------------------
// ITimingTest: Globals
//-----------------------------------------------------------------------------

// This must match the z-factor used in TTimingTest.
const double    kZFactor = 1.960; // 97.5 percentile of normal distribution

//-----------------------------------------------------------------------------
// IAutoSamplingTest Methods
//-----------------------------------------------------------------------------

IAutoSamplingTest::IAutoSamplingTest()
{
        fLeadSamples = 6;
}

IAutoSamplingTest::~IAutoSamplingTest()
{
}

void IAutoSamplingTest::timingTest()
{
        // No action needed here; sample values will be synthesized by oneSample.
}

void IAutoSamplingTest::setup()
{
        // We cannot set automatic sampling with any certainty, since the user can
        // turn it off from the command line.

        if (!automaticSampling()) {
                setSuccess(false);
                outputTextStream() << "Fail; please enable automatic sampling for this test!\n";
        }

        fCounter = 0;
}

void IAutoSamplingTest::cleanup()
{
        if (fDiscardedSamples != fLeadSamples) {
                outputTextStream() << "Fail: expected discarded samples = "
                        << fLeadSamples << ", actual = " << fDiscardedSamples
                        << (char)'\n';
                setSuccess(false);
        }
}

double IAutoSamplingTest::oneSample()
{
        double result;

        if (!fCounter) localSetup(); // Must do this here, rather than in Setup
        ++fCounter;
        if (fCounter % 2) result = fLowValue;
        else {
                result = (fCounter <= fLeadSamples) ? fOutsideHighValue: fInsideHighValue;
        }

        return result;
}

void IAutoSamplingTest::localSetup()
/*
        localSetup does the work that normally goes in setup. However, it has to
        happen later because it must come AFTER the arguments are parsed by the
        test, which happens in the TTimingTest::Test method.
*/
{
        const double kDefaultInsideEpsilon = 0.01;
        const double kDefaultOutsideEpsilon = 0.5;

        fLowValue = 100.0;
        fInsideEpsilon = kDefaultInsideEpsilon; // May be changed by command-line
        fOutsideEpsilon = kDefaultOutsideEpsilon; // May be changed by command-line

        if (!parseArguments()) setSuccess(false);
        else {
                // If innerEpsilon is too large, or outerEpsilon too small the test fails
                computeHighValues(fLowValue, fInsideEpsilon, fOutsideEpsilon);

                // Figure out how many digits to show of the high value
                long digits =
                        (long)-floor(log10(fHighValueThreshold - fInsideHighValue));

                // I put all these OutputTextStream calls in to make CompTech
                // faster. Otherwise it takes 9 minutes (!) to compile this one fn.
                outputTextStream()
                  // << setMaxFractionDigits(digits+1)
                        <<   "Low sample value     = " << fLowValue
                        << "\nHigh value threshold = " << fHighValueThreshold;
                outputTextStream()
                        << "\nInside high value    = " << fInsideHighValue
                        << "\nOutside high value   = " << fOutsideHighValue;
                outputTextStream()
                        // << setMaxFractionDigits(6)
                        << "\nInside epsilon  = " << fInsideEpsilon;
                outputTextStream()
                        << ((fInsideEpsilon != kDefaultInsideEpsilon)? " *ALTERED": "")
                        << "\nOutside epsilon = " << fOutsideEpsilon;
                outputTextStream()
                        << ((fOutsideEpsilon != kDefaultOutsideEpsilon)? " *ALTERED": "")
                        << "\nOut-of-tolerance (outside) samples = " << fLeadSamples << (char)'\n';
        }
}

double IAutoSamplingTest::samplingErrorMicroseconds() const
{
        return 0.0;
}

double IAutoSamplingTest::calibration()
{
        fEmptyMedianTime = 0.0;
        fClockNowOverhead = 0.0;
        return 0.0;
}

void IAutoSamplingTest::computeHighValues(double lowValue, double insideEpsilon, double outsideEpsilon)
/*
        computeHighValues computes three high values: The threshold, which is
        exactly at the auto-sampling tolerance, the inside value, which is within
        the tolerance, and the outside value, which is outside the tolerance. These
        values go into fHighValueTolerance, fInsideHighValue, and fOutsideHighValue,
        respectively. They all must be used in alternation with the fLowValue
        (see GetOneSample).

        <Epsilon> is the difference between the inside, outside, and threshold
        values. That is, the inside value is computed as (1-epsilon) times the gap
        between low and high threshold.
*/
{
        double k = kZFactor /
                (automaticSampleTolerance() * sqrt((double)sampleCount()-(double)1.0));

        fHighValueThreshold = (k+1.0)/(k-1.0) * lowValue;

        fInsideHighValue = fLowValue + (fHighValueThreshold - fLowValue) *
                (1.0 - insideEpsilon);

        fOutsideHighValue = fLowValue + (fHighValueThreshold - fLowValue) *
                (1.0 + outsideEpsilon);
}

bool IAutoSamplingTest::parseArguments()
/*
        parseArguments parses the input arguments. It understands the following:
                -out <n>        Sets the number of out-of-tolerance samples which are
                                        initially generated by oneSample.

        parseArguments sets the following values:
                fLeadSamples
*/
{
        static const IString kLeadSamplesKey("-out");
        static const IString kInsideEpsKey("-ineps");
        static const IString kOutsideEpsKey("-outeps");

        bool goodParse = true;

        // Parse input arguments into a text argument dictionary
        IArgumentDictionary args(*this);


        if (args.numberAt(kLeadSamplesKey, fLeadSamples, 0L, sampleCount()*(maximumSampleWindows()-1))){
                if (args.numberAt(kInsideEpsKey, fInsideEpsilon, 0.0, 1.0) ||
                        args.numberAt(kOutsideEpsKey, fOutsideEpsilon, fInsideEpsilon, 1.0))
                        outputTextStream()
                                << "Warning: may FAIL, inside or outside epsilon altered!\n";
        }
        else{
                goodParse = false;
        }

        return goodParse;
}

//eof



