// Revision: 44 1.38.2.1 source/ui/extapp/itbarbut.cpp, toolbar, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: itbarbut.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in itbarbut.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

// #define IC_TRACE_ALL

extern "C" {
  #define INCL_WINBUTTONS
  #define INCL_WINSYS
  #define INCL_WINWINDOWMGR
  #define INCL_WINCOUNTRY
  #define INCL_WINHOOKS
  #include <iwindefs.h>
  #ifdef IC_MOTIF
    #include <Xm/PushB.h>
  #endif
}

#pragma info(none)
  #include <iseq2.h>
#pragma info(restore)

#include <igrport.hpp>
#include <iimage.hpp>
#include <icconst.h>
#include <itbarbut.hpp>
#include <itbar.hpp>
#include <icustbdd.hpp>
#include <icustbhd.hpp>
#include <icustbev.hpp>
#include <icolor.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <ireslib.hpp>
#include <ifont.hpp>
#include <iguilock.hpp>
#include <iprimlck.hpp>
#include <ireslock.hpp>
#include <idmhndlr.hpp>
#include <itbpriv.hpp>
#include <itrace.hpp>
#ifdef IC_MOTIF
  #include <imstring.hpp>
  #include <ixmlabel.hpp>
  #include <nl_types.h>
  #include <langinfo.h>  
#endif //IC_MOTIF

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ITBARBUT_CPP_
  #include <ipagetun.h>
#endif




/*------------------------------------------------------------------------------
| Private Class to hold list of all toolbar buttons created.                   |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IStandardButtonList : public IVPtrSequence<IToolBarButton*>
{
public:
  IStandardButtonList ();
 ~IStandardButtonList ();

void
  addButton    ( IToolBarButton* button );

void
  removeButton ( IToolBarButton* button );

void
  bitmapFormatChanged ();

void
  textFormatChanged ();

private:
IPrivateResource
  fListLock;
IStandardButtonList ( const IStandardButtonList&);
IStandardButtonList& operator =( const IStandardButtonList&);
};

#ifdef IC_EXTENDED_RTTI_ENABLED
IStandardButtonList::IStandardButtonList ( const IStandardButtonList&)
{}
#endif

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IToolBarStaticData - A wrapper class for static toolbar data.  It provides   |
| an object for notational convenience and a destructor for static data.       |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarStaticData {
public:
  IToolBarStaticData ( ) { };       // Constructor does nothing
  ~IToolBarStaticData ( );

static IStandardButtonList
  &buttonList( );

static IStandardButtonList
 *fgButtonList;
static IColor
 *transparentColor;
static unsigned long
  transparentColorReset;
static IGImage
  *standardBitmapOnly,
  *standardTextOnly,
  *standardBitmapAndText;
static ISize
  standardCachedBmpSize;
static unsigned long
  standardCachedTextWidth,
  standardCachedTextLines;
static IFont
  *buttonFont;
static IColor
  *colorButtonDark,
  *colorButtonLight,
  *colorButtonMiddle,
  *colorDefaultButton;
static long
  standardCachedBgndColor,
  standardCachedParentBgndColor;
static unsigned long
  cxSysBorder,
  cySysBorder;
};

#pragma pack(pop)
#pragma enum(pop)

static IToolBarStaticData   toolBarStaticData;
IStandardButtonList* IToolBarStaticData::fgButtonList                  = 0;
IColor*              IToolBarStaticData::transparentColor              = 0;
unsigned long        IToolBarStaticData::transparentColorReset         = 0;
IGImage*             IToolBarStaticData::standardBitmapOnly            = 0;
IGImage*             IToolBarStaticData::standardTextOnly              = 0;
IGImage*             IToolBarStaticData::standardBitmapAndText         = 0;
ISize                IToolBarStaticData::standardCachedBmpSize;
unsigned long        IToolBarStaticData::standardCachedTextWidth       = 0;
unsigned long        IToolBarStaticData::standardCachedTextLines       = 0;
IFont*               IToolBarStaticData::buttonFont                    = 0;
IColor*              IToolBarStaticData::colorButtonDark               = 0;
IColor*              IToolBarStaticData::colorButtonLight              = 0;
IColor*              IToolBarStaticData::colorButtonMiddle             = 0;
IColor*              IToolBarStaticData::colorDefaultButton            = 0;
long                 IToolBarStaticData::standardCachedBgndColor       = 0;
long                 IToolBarStaticData::standardCachedParentBgndColor = 0;
unsigned long        IToolBarStaticData::cxSysBorder                   = 0;
unsigned long        IToolBarStaticData::cySysBorder                   = 0;

/*------------------------------------------------------------------------------
| IToolBarStaticData::~IToolBarStaticData                                      |
------------------------------------------------------------------------------*/
IToolBarStaticData::~IToolBarStaticData ( )
{
    IMODTRACE_DEVELOP("IToolBarStaticData::~IToolBarStaticData");
#if IC_STATIC_PRIORITY_SUPPORTED
    // These deletes are only safe if static object ordering is supported.
    if (transparentColor)
    {
      delete transparentColor;
      transparentColor = 0;
    }
    if (standardBitmapOnly)
    {
      delete standardBitmapOnly;
      standardBitmapOnly = 0;
    }
    if (standardTextOnly)
    {
      delete standardTextOnly;
      standardTextOnly = 0;
    }
    if (standardBitmapAndText)
    {
      delete standardBitmapAndText;
      standardBitmapAndText = 0;
    }
    if (buttonFont)
    {
      delete buttonFont;
      buttonFont = 0;
    }
    if (colorButtonDark)
    {
      delete colorButtonDark;
      colorButtonDark = 0;
    }
    if (colorButtonLight)
    {
      delete colorButtonLight;
      colorButtonLight = 0;
    }
    if (colorButtonMiddle)
    {
      delete colorButtonMiddle;
      colorButtonMiddle = 0;
    }
    if (colorDefaultButton)
    {
      delete colorDefaultButton;
      colorDefaultButton = 0;
    }
    if (fgButtonList)
    {
      delete fgButtonList;
      fgButtonList = 0;
    }
#endif
}

/*------------------------------------------------------------------------------
| IToolBarStaticData::buttonList                                               |
------------------------------------------------------------------------------*/
IStandardButtonList& IToolBarStaticData::buttonList( )
{
   if (!fgButtonList)
   {
      IPrimalLock lock;
      if (!fgButtonList)
         fgButtonList = new IStandardButtonList;
   }
   return *fgButtonList;
}

/*------------------------------------------------------------------------------
| IStandardButtonList::IStandardButtonList                                     |
------------------------------------------------------------------------------*/
IStandardButtonList::IStandardButtonList( )
  : IVPtrSequence<IToolBarButton*>( ),
    fListLock( )
{
  IMODTRACE_DEVELOP("IStandardButtonList::IStandardButtonList");
}

/*------------------------------------------------------------------------------
| IStandardButtonList::~IStandardButtonList                                    |
------------------------------------------------------------------------------*/
IStandardButtonList::~IStandardButtonList( )
{
  IMODTRACE_DEVELOP("IStandardButtonList::~IStandardButtonList");
}
/*------------------------------------------------------------------------------
| IStandardButtonList::addButton                                               |
------------------------------------------------------------------------------*/
void IStandardButtonList::addButton ( IToolBarButton *button )
{
  IGUIResourceLock resLock(fListLock);
  this->addAsLast(button);
}

/*------------------------------------------------------------------------------
| IStandardButtonList::removeButton                                            |
------------------------------------------------------------------------------*/
void IStandardButtonList::removeButton ( IToolBarButton *button )
{
  IGUIResourceLock resLock(fListLock);
  IStandardButtonList::Cursor cursor(*this);
  bool finished = false;
  cursor.setToFirst();
  while (cursor.isValid() && !finished)
  {
    IToolBarButton* currentButton = (IToolBarButton*)
                    this->elementAt(cursor);
    if ( currentButton == button )
    {
      this->removeAt(cursor);
      finished = true;
    }
    else
      cursor.setToNext();
  }
}

/*------------------------------------------------------------------------------
| IStandardButtonList::bitmapFormatChanged                                     |
------------------------------------------------------------------------------*/
void IStandardButtonList::bitmapFormatChanged ()
{
  IGUIResourceLock resLock(fListLock);
  IStandardButtonList::Cursor cursor(*this);
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
  {
    IToolBarButton* button = (IToolBarButton*)
                    this->elementAt(cursor);
    if ( button->isBitmapVisible() )
       button->setLayoutDistorted(IWindow::minimumSizeChanged,0);
  }
}

/*------------------------------------------------------------------------------
| IStandardButtonList::textFormatChanged                                       |
------------------------------------------------------------------------------*/
void IStandardButtonList::textFormatChanged ()
{
  IGUIResourceLock resLock(fListLock);
  IStandardButtonList::Cursor cursor(*this);
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
  {
    IToolBarButton* button = (IToolBarButton*)
                    this->elementAt(cursor);
    if ( button->isTextVisible() )
      button->setLayoutDistorted(IWindow::minimumSizeChanged,0);
  }
}

/*------------------------------------------------------------------------------
| This class does the drawing for toolbar buttons                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarButtonData : public ICustomButtonDrawHandler {
typedef ICustomButtonDrawHandler
  Inherited;
public:
  IToolBarButtonData ();
  ~IToolBarButtonData ();
IBitmapHandle
  bitmapHandle,
  latchedBitmapHandle;
IColor
  *transparentColor;
bool
  skipLayout;
IImage
  *bmp,
  *lbmp;
ISize
  minSize;
ISize
  currentStandardSize;
unsigned long
  currentStandardWidth,
  currentStandardLines;
//  values for fflags
static const unsigned long  invalid;
static const unsigned long  sizeValid;
static const unsigned long  imageValid;
unsigned long
  fflags;
#ifdef IC_PMWIN
IGImage
  *fUpCachedImage,
  *fLatchedCachedImage;
#endif
static long
  fgOptRGBDark,           // colors used in background bitmaps...used to
  fgOptRGBHighlight,      // force redraw if system colors changed.
  fgOptRGBMiddle,
  fgOptRGBBorder;
static bool
  fgOptImageValid;        // cached background image valid flag
protected:
bool
  dispatchHandlerEvent ( IEvent& event );
bool
  drawButton ( ICustomButtonDrawEvent& event );
void
  drawForeground ( ICustomButtonDrawEvent& event );
};


const unsigned long  IToolBarButtonData::invalid    = 0;
const unsigned long  IToolBarButtonData::sizeValid  = 1;
const unsigned long  IToolBarButtonData::imageValid = 2;

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| Private class to hold toolbar button data                                    |
------------------------------------------------------------------------------*/
IToolBarButtonData :: IToolBarButtonData ()
                    : bitmapHandle ( 0 ),
                      latchedBitmapHandle ( 0 ),
                      transparentColor ( 0 ),
                      skipLayout(false),
                      bmp ( 0 ),
                      lbmp ( 0 ),
                      currentStandardWidth(IToolBarButton::standardTextWidth()),
                      currentStandardLines(IToolBarButton::standardTextLines()),
                      currentStandardSize(IToolBarButton::standardBitmapSize()),
                      fflags ( IToolBarButtonData::invalid )
#ifdef IC_PMWIN
                      , fUpCachedImage( 0 ),
                      fLatchedCachedImage(0)
#endif
{
  if ( IToolBarButton::isDefaultTransparentColorSet() )
  {
    transparentColor = new IColor(0,0,0);
    *transparentColor = IToolBarButton::defaultTransparentColor();
  }
}

/*------------------------------------------------------------------------------
| Empty destructor                                                             |
------------------------------------------------------------------------------*/
IToolBarButtonData :: ~IToolBarButtonData ()
{}

/*------------------------------------------------------------------------------
| IToolBarButtonData::dispatchHandlerEvent                                     |
------------------------------------------------------------------------------*/
bool IToolBarButtonData :: dispatchHandlerEvent ( IEvent &event )
{
  IMODTRACE_ALL("IToolBarButtonData::dispatchHandlerEvent");
  bool rc = true;
  switch ( event.eventId() )
  {
#ifdef IC_PMWIN
 #ifndef IC_WIN
    case WM_SETFOCUS:
         if ( event.parameter2().number1() &&
              !((IToolBarButton *)event.window())->allowsMouseClickFocus() )
         {
            event.setResult(0);
            return true;
         }
         break;
 #endif //IC_WIN
#endif //IC_PMWIN
    case IC_UM_QRY_BMP_VISIBLE:
         event.setResult(
                  ( (IToolBarButton *)event.window() )->isBitmapVisible() );
         break;
    case IC_UM_QRY_TXT_VISIBLE:
         event.setResult(
                  ( (IToolBarButton *)event.window() )->isTextVisible() );
         break;

    case IC_UM_TOOLBAR:
    {
      switch ( event.parameter1().asUnsignedLong() )
      {
        case IC_UM_TBAR_TARGET_ENTER:
          event.setResult( true );
          break;
      }
    }
    break;

#ifdef IC_WIN
    /*********************************************************************/
    /* Only processes WM_ERASEBKGND to prevent default processing        */
    /*********************************************************************/
    case WM_ERASEBKGND:
         event.setResult( true );
         break;
#endif


    default:
         rc = Inherited::dispatchHandlerEvent ( event );
         break;
  }
  return rc;
}

/*------------------------------------------------------------------------------
| IToolBarButtonData::drawButton                                               |
------------------------------------------------------------------------------*/
bool IToolBarButtonData :: drawButton ( ICustomButtonDrawEvent& event )
{
  IMODTRACE_ALL("IToolBarButtonData::drawButton");

  // Get the toolbar button
  IToolBarButton* pTBarBtn = (IToolBarButton *)event.customButton();

  // Determine button state
  ICustomButtonDrawEvent::ButtonState buttonState = event.buttonState();
  bool isLatched  = (buttonState == ICustomButtonDrawEvent::buttonLatched);
  bool isDrawUp   = (buttonState == ICustomButtonDrawEvent::buttonUp);

  // There are 2 levels of optimization.  Each toolbar button saves a
  // bitmap image of the entire button in its current up, enabled position.
  // If valid, this image is used to paint.
  // If the image is not available, one is created when the button is drawn
  // in an up position for the first time with the current font, color, bitmap
  // and text values.  The fflags member, imageValid bit is set when this
  // is done and reset by setLayoutDistorted.
  // The second level of optimization stores bitmaps of standard button
  // border/backgrounds.  These images are shared across buttons and primarily
  // serve to speed up the initial paint of a toolbar.
  //
  // NOTE: In Motif, only the second level of optimization is handled here.
  // The image optimization is handled by ICustomButton/ICustomButtonDrawHandler.

#ifdef IC_PMWIN
  // Clean up any invalid cached image.  We rely upon setLayoutDistorted
  // being called for font, color, text or bitmap changes which affect the
  // appearance of the button.  That will cause imageValid to become false.
  // We handle disabled or draw down buttons in this routine.
  if ( !(fflags & imageValid) )
     {
     // image is not valid...either we havent drawn it yet or a
     // setLayoutDistorted has invalidated it.
     // delete any saved image for this button
     if ( this->fUpCachedImage)
        {
        delete this->fUpCachedImage;
        this->fUpCachedImage = 0;
        }

     if ( this->fLatchedCachedImage)
        {
        delete this->fLatchedCachedImage;
        this->fLatchedCachedImage = 0;
        }
     }
#endif

     // check the color settings on static cached bitmaps.  Delete bitmaps
     // if there was a color change which invalidates them.
  if ( (!fgOptImageValid) &&
       (toolBarStaticData.standardBitmapOnly ||
        toolBarStaticData.standardTextOnly ||
        toolBarStaticData.standardBitmapAndText) )
     {
     if ( (fgOptRGBDark       != toolBarStaticData.colorButtonDark->asRGBLong()) ||
          (fgOptRGBHighlight  != toolBarStaticData.colorButtonLight->asRGBLong()) ||
          (fgOptRGBMiddle     != toolBarStaticData.colorButtonMiddle->asRGBLong()) ||
          (fgOptRGBBorder     != toolBarStaticData.colorDefaultButton->asRGBLong()))
        {
        // color changes.  Delete all cached bitmaps.
        if (toolBarStaticData.standardBitmapOnly)
           {
             delete toolBarStaticData.standardBitmapOnly;
             toolBarStaticData.standardBitmapOnly = 0;
           }
        if (toolBarStaticData.standardTextOnly)
           {
             delete toolBarStaticData.standardTextOnly;
             toolBarStaticData.standardTextOnly = 0;
           }
        if (toolBarStaticData.standardBitmapAndText)
           {
             delete toolBarStaticData.standardBitmapAndText;
             toolBarStaticData.standardBitmapAndText = 0;
           }
        }

     // set image valid flag to bypass further color checking until next
     // setLayoutDistorted.
     fgOptImageValid = true;
     }

  IRectangle rect( event.drawingArea() );
  IGImage*  pOptimize (0);

  // See if we have a valid cached image of the button.
  //
#ifdef IC_PMWIN
  if ( this->fUpCachedImage && isDrawUp )          // RDL: Is a check for enabled needed here??
     {
     // use cached image
     pOptimize = this->fUpCachedImage;
     ITRACE_ALL("Up Image cached");
     }
  else if(this->fLatchedCachedImage && isLatched  )
     {
     // use cached image
     pOptimize = this->fLatchedCachedImage;
     ITRACE_ALL("Latched enabled image cached");
     }
  else
#endif //IC_PMWIN
#ifdef IC_MOTIF
  // NOTE: In Motif, the caching is handled by the base class
  // ICustomButtonDrawHandler.  The grafPort provided in the event is the
  // cached image if caching is being used, or the screen otherwise.
  // In addition, this function is only called when the image needs to be
  // drawn.  In the case when we are drawing the disabled image, we can
  // use the already drawn enabled image and simply redraw it half toned.
  ICustomButtonDrawData* drawData =
      (ICustomButtonDrawData*)(char*)event.parameter2();
  if (event.isButtonEnabled() || !drawData )
#endif //IC_MOTIF
     {
     ITRACE_ALL("Drawing image");
     // Need to paint.  We do painting into a memory graphics context, which
     // is then drawn on the screen later.

     IGImage *pBorderBMP( 0 );

     // For standard buttons, we save static bitmaps of the border and background
     // shared across all toolbar buttons
     bool   borderOptPossible =
       ( pTBarBtn->isStandardFormat() &&
         event.drawUp() &&
         (fflags & sizeValid) && (minSize == rect.size() ) );

     // Get the static bitmap for current view
     if ( borderOptPossible )
        {
        //This test is for an obscure case when using optimized (non
        //dynamic) toolbar buttons:  The user has existing standard size
        //buttons, then changes the standard size settings, then creates
        //new buttons.  The new standard size setting only effects the newly
        //created buttons.  This drawbutton function could be painting a button
        //created before the standard size setting was changed, or a button
        //created after the setting was changed.  So we need to check to see
        //if the cached bitmap is correct for the button being painted.
        //The cached bitmap is invalidated if need be, and the newly created
        //replacement is stored as the cached button.
        if ( (!IPerformanceSettings::isSettingEnabled(IPerformanceSettings::dynamicToolBarButtons) &&
             ((currentStandardSize != toolBarStaticData.standardCachedBmpSize) ||
              (currentStandardLines != toolBarStaticData.standardCachedTextLines) ||
              (currentStandardWidth != toolBarStaticData.standardCachedTextWidth)) )
           ||   //Or the background color, or the parent's background color
                //changed, then the bitmap is invalid .
           ( (toolBarStaticData.standardCachedParentBgndColor !=
              pTBarBtn->parent()->backgroundColor().asRGBLong()) ||
             (toolBarStaticData.standardCachedBgndColor !=
              pTBarBtn->backgroundColor().asRGBLong()) ) )
          {
            if (toolBarStaticData.standardBitmapOnly)
              {
                delete toolBarStaticData.standardBitmapOnly;
                toolBarStaticData.standardBitmapOnly = 0;
              }
            if (toolBarStaticData.standardTextOnly)
              {
                delete toolBarStaticData.standardTextOnly;
                toolBarStaticData.standardTextOnly = 0;
              }
            if (toolBarStaticData.standardBitmapAndText)
              {
                delete toolBarStaticData.standardBitmapAndText;
                toolBarStaticData.standardBitmapAndText = 0;
              }
            pBorderBMP = 0;
          }
          else
          {
          // Yes... we can optimize border/background painting
          // Figure out which standard bitmap we want to use
          switch ( pTBarBtn->view() )
             {
             case IToolBarButton::bitmapView:
               pBorderBMP = toolBarStaticData.standardBitmapOnly;
               break;
             case IToolBarButton::textView:
               pBorderBMP = toolBarStaticData.standardTextOnly;
               break;
             case IToolBarButton::bitmapAndTextView:
               pBorderBMP = toolBarStaticData.standardBitmapAndText;
               break;
             }
          }
        }
     // If the standard bitmap does not exist need to paint...this could be
     // because we cannot optimize or because the saved image is invalid.
     IRectangle foregroundRect;
     if ( !pBorderBMP )
        {
        // Pass the colormap as 0, so that a palette is not selected and
        // realized.
        pBorderBMP = new IGImage( (long)rect.width(),
                              (long)rect.height(),
                              IGImage::kTrueColor24Bit,
                              ICoordinateSystem::kOriginUpperLeft, 0);

        //Query grafPort into local variable for use several times later
        IGrafPort&  borderMemoryPort = *(pBorderBMP->grafPort());

        // Create Dummy Custom button event for memory drawing.
        ICustomButtonDrawEvent borderMemoryEvent( event, rect.size(), &borderMemoryPort);

        ITRACE_ALL(IString("Border not optimized - rect=") + rect.asString() );
        // Draw the border and background for the button normally
        drawBorder ( borderMemoryEvent );
        drawBackground ( borderMemoryEvent );
        foregroundRect = borderMemoryEvent.drawingArea();

        // If optimization possible save off the bitmap for next time
        if (borderOptPossible)
           {
           // save colors used in bitmap
           fgOptRGBDark       = toolBarStaticData.colorButtonDark->asRGBLong();
           fgOptRGBHighlight  = toolBarStaticData.colorButtonLight->asRGBLong();
           fgOptRGBMiddle     = toolBarStaticData.colorButtonMiddle->asRGBLong();
           fgOptRGBBorder     = toolBarStaticData.colorDefaultButton->asRGBLong();

           // Save the background color and the parent's background color
           toolBarStaticData.standardCachedParentBgndColor =
                 pTBarBtn->parent()->backgroundColor().asRGBLong();
           toolBarStaticData.standardCachedBgndColor =
                 pTBarBtn->backgroundColor().asRGBLong();

           //Cache the border image to an IGBitmap so it can be used later.
           switch ( pTBarBtn->view() )
             {
             case IToolBarButton::bitmapView:
               //Retrieve the contents of the grafport into an IGImage
               toolBarStaticData.standardBitmapOnly = pBorderBMP;
               break;

             case IToolBarButton::textView:
               //Retrieve the contents of the grafport into an IGImage
               toolBarStaticData.standardTextOnly = pBorderBMP;
               break;

             case IToolBarButton::bitmapAndTextView:
               //Retrieve the contents of the grafport into an IGImage
               toolBarStaticData.standardBitmapAndText = pBorderBMP;
               break;
             }  // switch

             //Also cache away attributes of this button
             toolBarStaticData.standardCachedBmpSize = currentStandardSize;
             toolBarStaticData.standardCachedTextLines = currentStandardLines;
             toolBarStaticData.standardCachedTextWidth = currentStandardWidth;
           }   // optimization possible
        } // no saved border bitmap



#ifdef IC_PMWIN
     // We create our tool bar image by starting with a copy of the background image.
     pOptimize = new IGImage( *pBorderBMP);

     //Query grafPort into local variable for use several times later
     IGrafPort&  memGport = *(pOptimize->grafPort());

     // Create Dummy Custom button event for memory drawing.
     ICustomButtonDrawEvent memoryEvent( event, rect.size(), &memGport);
#endif //IC_PMWIN

#ifdef IC_MOTIF
     // The grafPort provided in the ICustomButtonDrawEvent argument is
     // the correct one for either caching or no caching.  Draw into it
     // directly.
     event.grafPort().draw( *pBorderBMP);

     // Make a reference to the supplied event for the common code below.
     ICustomButtonDrawEvent& memoryEvent(event);
#endif //IC_MOTIF


     // Delete the border image if we aren't saving it.
     if ( !borderOptPossible )
        delete pBorderBMP;


     if(borderOptPossible)
       {
       // Reduce the drawing area so the foreground will paint correctly
       // Buttons painted differently in PM and Windows
       foregroundRect = rect.shrunkBy(IPair(toolBarStaticData.cxSysBorder,
                                               toolBarStaticData.cySysBorder));

       // 1 additional border top/left, 2 bottom/right
       foregroundRect = IRectangle ( foregroundRect.minX() + toolBarStaticData.cxSysBorder,
                           foregroundRect.minY() + toolBarStaticData.cySysBorder,
                           foregroundRect.maxX() - toolBarStaticData.cxSysBorder*2,
                           foregroundRect.maxY() - toolBarStaticData.cySysBorder*2);

       }
     ITRACE_ALL(IString("Optimize border - setDrawingArea=") + foregroundRect.asString() );
     memoryEvent.setDrawingArea ( foregroundRect );


     // Draw the foreground
     drawForeground ( memoryEvent );

#ifdef IC_PMWIN
     // Save the image for future use.
     if ( isDrawUp )
        {
        fUpCachedImage = pOptimize;
        if(fLatchedCachedImage)
           {
           delete fLatchedCachedImage;
           fLatchedCachedImage = 0;
           }
        fflags |= imageValid;
        }
     else if(isLatched)
        {
        fLatchedCachedImage = pOptimize;
        if(fUpCachedImage)
           {
           delete fUpCachedImage;
           fUpCachedImage = 0;
           }
        fflags |= imageValid;
        }
#endif //IC_PMWIN
    }  // else need to paint



  IGrafMatrix   adjust;
  IGPoint2D     offset( rect.minXMinY() );
  adjust.setToTranslate( offset );
  IGrafBundle   bundle;

#ifdef IC_PMWIN
  // At this point pOptimize contains the image of the entire button.
  // Paint the image to the screen.
  //If button is disabled, draw the entire button, including borders in
  //halftone, which represents disabled state.
  if (  event.isButtonEnabled() ) {
    bundle.setImageTransferMode( IImageTransferMode::kSrcCopy );
  } else {
    bundle.setImageTransferMode( IImageTransferMode::kOCHalfTone );
  }
  event.grafPort().draw( *pOptimize, bundle, adjust );

  // Clean up bitmap if not needed anymore
  if ( pOptimize != this->fUpCachedImage && pOptimize != fLatchedCachedImage)
     delete pOptimize;

#endif //IC_PMWIN
#ifdef IC_MOTIF
  // We are done if we ard drawing the enabled image.  For the disabled image,
  // create a half toned image.
  if (  !event.isButtonEnabled() )
  {
     // Get the previously drawn enabled image from the cache maintained
     // by the custom button draw handler and draw it disabled.
     pOptimize = drawData ? drawData->image( drawData->buttonState(), true ) : 0;

     if (pOptimize)
     {  // Draw the enabled image with a disabled appearance.
        bundle.setImageTransferMode( IImageTransferMode::kOCHalfTone );
        event.grafPort().draw( *pOptimize, bundle, adjust );
     }
     else
     {  // Edge case where there is no cache.  Use the default function
        // in ICustomButtonDrawHandler.
        this->drawDisabledEmphasis( event );
     }
  }
#endif //IC_MOTIF

  return true;
}

/*------------------------------------------------------------------------------
| IToolBarButtonData::drawForeground                                           |
------------------------------------------------------------------------------*/
void IToolBarButtonData :: drawForeground ( ICustomButtonDrawEvent& event )
{
  IMODTRACE_ALL("IToolBarButtonData::drawForeground");
  IRectangle       rect = event.drawingArea();
  IRectangle       newArea = rect;

  IToolBarButton*  pTBarBtn = (IToolBarButton *)event.customButton();

  bool             bStandardFormat = pTBarBtn->isStandardFormat();
  bool             bTextVisible = pTBarBtn->isTextVisible();
  bool             bDynamicToolBarButtons =
                          IPerformanceSettings::isSettingEnabled(
                          IPerformanceSettings::dynamicToolBarButtons);

  if ( pTBarBtn->isBitmapVisible() && ( bitmapHandle != 0 ))
  {
    IPoint bitmapPos;
    ISize  bitmapSize;

    bitmapSize = ISize((IPair::Coord)(bmp->geometricBounds().width()),
                       (IPair::Coord)(bmp->geometricBounds().height()) );

    // Center bitmap
    bitmapPos.setX(rect.minX()+((rect.width()-bitmapSize.width())/2));
    bitmapPos.setY(rect.minY()+((rect.height()-bitmapSize.height())/2));

    // If text is also shown, do not center vertically
    if ( bTextVisible )
    {
      if ( bStandardFormat )
      {
        if ( bDynamicToolBarButtons )
        {
          bitmapPos.setY(rect.minY() + 1 +
                        (IToolBarButton::standardBitmapSize().height() -
                         bitmapSize.height())/2);
        }
        else
        {
          bitmapPos.setY(rect.minY() + 1 +
                        (currentStandardSize.height() -
                         bitmapSize.height())/2);
        }
      }
      else
      {
        bitmapPos.setY(rect.minY() + 1);
      }

      if ( event.drawUp() == false )
        bitmapPos.setY(bitmapPos.y() + 1);
    }

    // Draw the bitmap
    {
    // Choose appropriate image.
    IImage *pImage = ( (!event.drawDown() && pTBarBtn->isLatched()) &&
                        latchedBitmapHandle != 0 ) ? lbmp : bmp;

    // Set up linked grafport to draw to specified bitmap position.
    IGrafMatrix matrix;
    matrix.translateBy( bitmapPos );
    ILinkedGrafPort linkedPort( &event.grafPort(), kViewMatrix, &matrix );

    // Draw the image.
    pImage->draw( linkedPort );
    }

    // Remove the bitmap area from the drawing area
    if ( bStandardFormat )
    {
      if ( bDynamicToolBarButtons )
      {
        newArea = IRectangle ( rect.minX(),
                               bitmapPos.y() +
                               IToolBarButton::standardBitmapSize().height() - 1,
                               rect.maxX(),
                               rect.maxY());
      }
      else
      {
        newArea = IRectangle ( rect.minX(),
                               bitmapPos.y() +
                               currentStandardSize.height() - 1,
                               rect.maxX(),
                               rect.maxY());
      }
    }
    else
    {
      newArea = IRectangle ( rect.minX(),
                             bitmapPos.y() + bitmapSize.height() - 1 ,
                             rect.maxX(),
                             rect.maxY());
    }

    if ( event.drawUp() )
      {
      newArea.sizeTo(ISize(newArea.size().width(), newArea.size().height()-1));
      newArea.moveBy( IPair(0, 1 ) );    // down 1
      }
    ITRACE_ALL(IString("new drawing area  newArea=") + newArea.asString() );
    event.setDrawingArea ( newArea );
  }

  if ( bTextVisible )
  {
    if ( bStandardFormat )
    {
      if ( bDynamicToolBarButtons )
      {
        // Remove border area from drawing area
        event.setDrawingArea ( IRectangle ( newArea.minX() + 4,
                                            newArea.minY(),
                                            newArea.minX() + 4 +
                                            IToolBarButton::standardTextWidth(),
                                            newArea.maxY() ));
      }
      else
      {
        // Remove border area from drawing area
        event.setDrawingArea ( IRectangle ( newArea.minX() + 4,
                                            newArea.minY(),
                                            newArea.minX() + 4 +
                                            currentStandardWidth,
                                            newArea.maxY() ));
      }
      Inherited::drawForeground(event);
    }
    else
      Inherited::drawForeground(event);
  }

  // Restore drawing area rectangle (since we may have changed it)
  event.setDrawingArea ( rect );
}

/*------------------------------------------------------------------------------
| IToolBarButton styles                                                        |
------------------------------------------------------------------------------*/
const IToolBarButton::Style
  IToolBarButton::bitmapVisible        ( 0, IBS_TBARBMPVISIBLE ),  //0x00000004
  IToolBarButton::textVisible          ( 0, IBS_TBARTEXTVISIBLE ), //0x00000008
  IToolBarButton::bitmapAndTextVisible ( 0, IBS_TBARBOTHVISIBLE ), //0x00000010
  IToolBarButton::useIdForBitmap       ( 0, IBS_TBARUSEBMPID ),    //0x00000020
  IToolBarButton::useIdForText         ( 0, IBS_TBARUSETEXTID ),   //0x00000040
  IToolBarButton::standardFormat       ( 0, IBS_TBARSTDFORMAT ),   //0x00000080
  IToolBarButton::transparentBitmap    ( 0, IBS_TBARTRANSBITMAP ), //0x00004000
  IToolBarButton::dragDelete           ( 0, IBS_TBARDRAGDEL ),     //0x00000200
#if (IC_OBSOLETE <= IC_OBSOLETE_3)
  IToolBarButton::noDragDelete         ( 0 ),
#endif // IC_OBSOLETE
#ifdef IC_WIN
  IToolBarButton::classDefaultStyle    ( (WS_VISIBLE ),
                                                          IBS_NOPOINTERFOCUS |
#else
  IToolBarButton::classDefaultStyle    ( (WS_VISIBLE |
                                          BS_NOPOINTERFOCUS),
#endif
                                                         (IBS_TBARBMPVISIBLE |
                                                          IBS_TBARUSEBMPID   |
                                                          IBS_TBARUSETEXTID  |
                                                          IBS_TBARDRAGDEL    |
                                                          IBS_TBARSTDFORMAT) );

IToolBarButton::Style
#ifdef IC_WIN
  IToolBarButton::fCurrentDefaultStyle ( (WS_VISIBLE) ,
                                                         IBS_NOPOINTERFOCUS |
#else
  IToolBarButton::fCurrentDefaultStyle ( (WS_VISIBLE |
                                         BS_NOPOINTERFOCUS),
#endif
                                                        (IBS_TBARBMPVISIBLE |
                                                         IBS_TBARUSEBMPID   |
                                                         IBS_TBARUSETEXTID  |
                                                         IBS_TBARDRAGDEL    |
                                                         IBS_TBARSTDFORMAT) );
  unsigned long IToolBarButton::fgStandardWidth(50);
  unsigned long IToolBarButton::fgStandardLines(1);
  ISize         IToolBarButton::fgStandardSize(22,22);
  long          IToolBarButtonData::fgOptRGBDark      (0);
  long          IToolBarButtonData::fgOptRGBHighlight (0);
  long          IToolBarButtonData::fgOptRGBMiddle    (0);
  long          IToolBarButtonData::fgOptRGBBorder    (0);
  bool       IToolBarButtonData::fgOptImageValid   (false);



/*------------------------------------------------------------------------------
| IToolBarButton ctor                                                          |
------------------------------------------------------------------------------*/
IToolBarButton :: IToolBarButton (unsigned long id,
                                  IWindow* parent,
                                  IWindow* owner,
                                  const IRectangle& initial,
                                  const Style& style ) :
#ifdef IC_PMWIN
                  ICustomButton(),   //skipLayout flag
#endif
#ifdef IC_MOTIF
                  ICustomButton(id, parent, owner, initial),
#endif //IC_MOTIF

                  fToolBarButtonData(0)
{
#ifdef IC_PMWIN
  //These borders can safely be queried once and the values used later.
  //In OS/2 and Windows, these values will not change dynamically.  They
  //cannot be set by an end user or a programmer
  if (toolBarStaticData.cxSysBorder == 0)
     toolBarStaticData.cxSysBorder = IQUERYSYSVALUE(SV_CXBORDER);
  if (toolBarStaticData.cySysBorder == 0)
     toolBarStaticData.cySysBorder = IQUERYSYSVALUE(SV_CYBORDER);
#endif
#ifdef IC_MOTIF
  if (toolBarStaticData.cxSysBorder == 0)
     toolBarStaticData.cxSysBorder = 1;
  if (toolBarStaticData.cySysBorder == 0)
     toolBarStaticData.cySysBorder = 1;
#endif

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  // Check for bad style combinations

  if ((( style & bitmapVisible ) && ( style & textVisible )) ||
      (( style & bitmapVisible ) && ( style & bitmapAndTextVisible )) ||
      (( style & textVisible )   && ( style & bitmapAndTextVisible )))
  {
     ITHROWLIBRARYERROR( IC_INVALIDSTYLE,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable );
  }

  IASSERTPARM(parent!=0);
  fToolBarButtonData = new IToolBarButtonData();

  //No need to do multiple layout requests during the construction
  //of the window (SetLayoutDistorted is called once by
  //handleEventsFor, once by setText and once by setting the window
  //font).  Later, initiate layout once, after all settings are complete.
  fToolBarButtonData->skipLayout = true;

#ifdef IC_PMWIN
  IWindowHandle whButton =
      this -> create( id,
                      0,
                      convertToGUIStyle( style ),
                      WC_BUTTON,
                      parent->handleForChildCreation(),
                      (owner == 0) ? IWindowHandle(0) : owner->handle(),
                      initial,
                      0,
                      0,
                      defaultOrdering(),
                      convertToGUIStyle( style, true ) );


  startHandlingEventsFor(whButton);
#endif

#ifdef IC_MOTIF

   // Remove the default draw handler added by ICustomButton.  We will
   // add our own (IToolBarButtonData) later.
   this->removeDefaultDrawHandler();

   // Adjust resources on widget so that the system does not draw borders,
   // and there is no shadow highlight.
   IArgList argList;

   argList.setResource( XmNhighlightThickness,  0);  //no highlight border

   XtSetValues( (Widget)this->handle(),
                (ArgList)argList.argList(),
                argList.numberOfArgs() );

   // Set up pointerFocus based on the IButton::noPointerFocus style. (the
   // convertToGUIStyle functions handle this on Intel)
   this->enableMouseClickFocus( (style & noPointerFocus) ? false : true );

#endif //IC_MOTIF

  // Add handler
  fToolBarButtonData->handleEventsFor(this);

  // Handle style information
  if ( style & useIdForBitmap )
  {
    if ( id < IC_ID_BASE )
    {
      // Try to load from user resource
      fToolBarButtonData->bitmapHandle =
                      IApplication::current().userResourceLibrary()
                                             .tryToLoadBitmap(id);
    }
    else
    {
      // Load from class library resource
      fToolBarButtonData->bitmapHandle =
                      IApplication::current().resourceLibrary()
                                             .tryToLoadBitmap(id);
    }

    // Could not load bitmap, so try to get the default bitmap
    if ( !fToolBarButtonData->bitmapHandle )
    {
      fToolBarButtonData->bitmapHandle =
                      IApplication::current().resourceLibrary()
                                             .tryToLoadBitmap(IC_ID_DEFAULT);
    }

    // If we could load a bitmap, finish setup
    if ( fToolBarButtonData->bitmapHandle )
    {
      fToolBarButtonData->bmp = new IImage( fToolBarButtonData->bitmapHandle );
      if ( (fToolBarButtonData->transparentColor) &&
           (style & transparentBitmap) )
        fToolBarButtonData->bmp->setTransparentColor ( *fToolBarButtonData->transparentColor );
    }
  }

  if ( style & useIdForText )
  {
    if ( id < IC_ID_BASE )
    {
      // Load from user resource
      setText(IApplication::current().userResourceLibrary().
              tryToLoadString(id));
    }
    else
    {
      // Load from class library resource
      setText(IApplication::current().resourceLibrary().
              tryToLoadString(id));
    }
  }

  if ( style & standardFormat )
  {
    if ( IPerformanceSettings::isSettingEnabled (
                  IPerformanceSettings::dynamicToolBarButtons ) )
      {
        toolBarStaticData.buttonList().addButton(this);
      }

    if (fToolBarButtonData->bmp)
    {
      ISize bmpSize = ISize( (IPair::Coord)(fToolBarButtonData->bmp->geometricBounds().width()),
                             (IPair::Coord)(fToolBarButtonData->bmp->geometricBounds().height()) ),
            oldSize = bmpSize;
      if ( bmpSize.width() > fgStandardSize.width() )
        bmpSize.setWidth(fgStandardSize.width());
      if ( bmpSize.height() > fgStandardSize.height() )
        bmpSize.setHeight(fgStandardSize.height());

      if ( bmpSize != oldSize )
      {
        // The bitmap is too big, so shrink it
        fToolBarButtonData->bmp->sizeTo( IGPoint2D( bmpSize.width(),
                                                    bmpSize.height() ) );
        fToolBarButtonData->bitmapHandle =
            fToolBarButtonData->bmp->image()->handle();
      }
    }
  }

#ifdef IC_PM
  if (toolBarStaticData.buttonFont == 0)
    {
      unsigned long codepage = WinQueryCp(HMQ_CURRENT);
      switch (codepage)
      {
        // Set the default button font to be Helv.8
        // Note: DBCS OS/2 systems should use the default font instead of Helv
        //       since the double-byte characters do not exist for this font
        // Note: The following list represents the DBCS code pages supported
        //       by OS/2 2.1 and Warp as of June 1995
        case  949: // Korean                      (replaces older 934/944)
        case  944: // Korean
        case  934: // Korean
        case 1381: // People's Republic of China  (replaces older 936/946)
        case  936: // People's Republic of China
        case  946: // People's Republic of China
        case  950: // Taiwan                      (replaces older 938/948)
        case  938: // Taiwan
        case  948: // Taiwan
        case  932: // Japanese
        case  942: // Japanese
        {
          toolBarStaticData.buttonFont = new IFont(this);
          toolBarStaticData.buttonFont->setPointSize(8);
        }
        break;
        default:
        {
          toolBarStaticData.buttonFont = new IFont("Helv",8);
        }
        break;
      }
    }
  setFont(*toolBarStaticData.buttonFont);
#endif
#ifdef IC_WIN
  if (toolBarStaticData.buttonFont == 0)
     toolBarStaticData.buttonFont = new IFont("Arial",9);
  setFont(*toolBarStaticData.buttonFont);
#endif
#ifdef IC_MOTIF
  if (toolBarStaticData.buttonFont == 0)
  {
    IString currentLocale = nl_langinfo(CODESET);
    if( currentLocale == "ISO8859-1" )
      toolBarStaticData.buttonFont = new IFont("Helvetica" ,8);
    else    
      toolBarStaticData.buttonFont = new IFont(IWindow::desktopWindow()->font().name(),12); 
  }   
  setFont(*toolBarStaticData.buttonFont);
#endif
  if (toolBarStaticData.colorButtonDark == 0)
     toolBarStaticData.colorButtonDark = new IColor(IColor::buttonDark);
  if (toolBarStaticData.colorButtonLight == 0)
     toolBarStaticData.colorButtonLight = new IColor(IColor::buttonLight);
  if (toolBarStaticData.colorButtonMiddle == 0)
     toolBarStaticData.colorButtonMiddle = new IColor(IColor::buttonMiddle);
  if (toolBarStaticData.colorDefaultButton  == 0)
     toolBarStaticData.colorDefaultButton = new IColor(IColor::defaultButton);


  //Earlier calls to setLayoutDistorted were ignored.  Now,
  //initiate layout once, after all settings are complete.
  //Even though layout is done on a button when it is added to the
  //toolbar, layout still has to be done here because the button might
  //never be part of a toolbar.
  //Note that colorChange not needed in this call to setLayoutDistorted
  fToolBarButtonData->skipLayout = false;
  setLayoutDistorted(IWindow::minimumSizeChanged |
                     IWindow::windowCreated |
                     IWindow::fontChanged, 0);
}

/*------------------------------------------------------------------------------
| IToolBarButton dtor                                                          |
------------------------------------------------------------------------------*/
IToolBarButton :: ~IToolBarButton()
{

  if (fToolBarButtonData)
  {
    fToolBarButtonData->stopHandlingEventsFor(this);

    if ( isStandardFormat() &&
       ( IPerformanceSettings::isSettingEnabled (
                  IPerformanceSettings::dynamicToolBarButtons ) ) )
       {
         toolBarStaticData.buttonList().removeButton(this);
       }

#ifdef IC_PMWIN
    if ( fToolBarButtonData->fUpCachedImage )
       delete fToolBarButtonData->fUpCachedImage;
    if ( fToolBarButtonData->fLatchedCachedImage )
       delete fToolBarButtonData->fLatchedCachedImage;
#endif //IC_PMWIN

    if ( fToolBarButtonData->bmp )
      delete fToolBarButtonData->bmp;
    if ( fToolBarButtonData->lbmp )
      delete fToolBarButtonData->lbmp;
    if ( fToolBarButtonData->transparentColor)
      delete fToolBarButtonData->transparentColor;
    delete fToolBarButtonData;
  }
}

/*------------------------------------------------------------------------------
| IToolBarButton::defaultStyle                                                 |
------------------------------------------------------------------------------*/
IToolBarButton::Style IToolBarButton :: defaultStyle()
{
  return fCurrentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IToolBarButton::setDefaultStyle                                              |
------------------------------------------------------------------------------*/
void IToolBarButton :: setDefaultStyle(const Style& style)
{
  fCurrentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IToolBarButton::convertToGUIStyle                                            |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IToolBarButton :: convertToGUIStyle(const IBitFlag& guiStyle,
                                                  bool bExtOnly) const
{
  // Obtain the style from the class (ICustomButton) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| IToolBarButton::defaultTransparentColor                                      |
------------------------------------------------------------------------------*/
IColor IToolBarButton :: defaultTransparentColor()
{
  if ( isDefaultTransparentColorSet() )
  {
      if ( !toolBarStaticData.transparentColor )
          toolBarStaticData.transparentColor = new IColor(255,0,255);
      return *(toolBarStaticData.transparentColor);
  }
  else
      return IColor(0,0,0);
}

/*------------------------------------------------------------------------------
| IToolBarButton::clearDefaultTransparentColor                                 |
------------------------------------------------------------------------------*/
void IToolBarButton :: clearDefaultTransparentColor ( )
{
  toolBarStaticData.transparentColorReset = 1;
  if ( toolBarStaticData.transparentColor )
  {
     delete toolBarStaticData.transparentColor;
     toolBarStaticData.transparentColor = 0;
  }
}

/*------------------------------------------------------------------------------
| IToolBarButton::setDefaultTransparentColor                                   |
------------------------------------------------------------------------------*/
void IToolBarButton :: setDefaultTransparentColor (const IColor& aColor )
{
  toolBarStaticData.transparentColorReset = 0;
  if ( !toolBarStaticData.transparentColor )
    toolBarStaticData.transparentColor = new IColor(0,0,0);
  *toolBarStaticData.transparentColor = aColor;
}

/*------------------------------------------------------------------------------
| IToolBarButton::isDefaultTransparentColorSet                                 |
------------------------------------------------------------------------------*/
bool IToolBarButton :: isDefaultTransparentColorSet ()
{
  if ( !toolBarStaticData.transparentColorReset  )
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| IToolBarButton::setBitmap                                                    |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setBitmap(const IResourceId& bitmapId )
{
  fToolBarButtonData->bitmapHandle = 0;
  if ( fToolBarButtonData->bmp )
  {
    delete fToolBarButtonData->bmp;
    fToolBarButtonData->bmp = 0;
  }

  //First attempt to load from the bitmap's resource library.  If not found
  //and Id is in IOC range, load from IOC library.  Exception thrown
  //if not found.
  if ( bitmapId != 0 )
  {
    if ( bitmapId < IC_ID_BASE )
    {
    // Try to load from user resource, throw exception if not found
    fToolBarButtonData->bitmapHandle = bitmapId.resourceLibrary().
                                       loadBitmap(bitmapId);
    }
    else
    {
      // Try to load from user resource
      fToolBarButtonData->bitmapHandle = bitmapId.resourceLibrary().
                                         tryToLoadBitmap(bitmapId);
      if ( !fToolBarButtonData->bitmapHandle )
      {
        // Load from class library resource, throw exception if not found
        fToolBarButtonData->bitmapHandle =
                        IApplication::current().resourceLibrary()
                                               .loadBitmap(bitmapId);
      }
    }
  }
  return setBitmap(fToolBarButtonData->bitmapHandle);
}

/*------------------------------------------------------------------------------
| IToolBarButton::setLatchedBitmap                                             |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setLatchedBitmap
                                  ( const IResourceId& latchedBitmapId)
{
  fToolBarButtonData->latchedBitmapHandle = 0;
  if ( fToolBarButtonData->lbmp )
  {
    delete fToolBarButtonData->lbmp;
    fToolBarButtonData->lbmp = 0;
  }

  if ( latchedBitmapId != 0 )
  {
    fToolBarButtonData->latchedBitmapHandle = latchedBitmapId.resourceLibrary().
                                loadBitmap(latchedBitmapId);
  }

  return setLatchedBitmap(fToolBarButtonData->latchedBitmapHandle);
}

/*------------------------------------------------------------------------------
| IToolBarButton::setBitmap                                                    |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setBitmap(unsigned long id )
{
  return setBitmap(IResourceId(id));
}

/*------------------------------------------------------------------------------
| IToolBarButton::setLatchedBitmap                                             |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setLatchedBitmap(unsigned long id)
{
  return setLatchedBitmap(IResourceId(id));
}

/*------------------------------------------------------------------------------
| IToolBarButton::setBitmap                                                    |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setBitmap (const IBitmapHandle& handle)
{
  fToolBarButtonData->bitmapHandle = handle;

  if ( fToolBarButtonData->bmp )
  {
    delete fToolBarButtonData->bmp;
    fToolBarButtonData->bmp = 0;
  }

  if ( handle )
  {
    fToolBarButtonData->bmp = new IImage( fToolBarButtonData->bitmapHandle );
    if ( (fToolBarButtonData->transparentColor) &&
         (extendedStyle() & transparentBitmap.asExtendedUnsignedLong()) )
      fToolBarButtonData->bmp->setTransparentColor ( *fToolBarButtonData->transparentColor );

    // Check if bitmap needs to be resized to fit in standard button
    if ( isStandardFormat() )
    {
      ISize bmpSize = ISize( (IPair::Coord)(fToolBarButtonData->bmp->geometricBounds().width()),
                             (IPair::Coord)(fToolBarButtonData->bmp->geometricBounds().height()) ),
            oldSize = bmpSize;
      // If button has standard format set, resize the bitmap to fit the
      // current size of the button.  For the non-dynamic case, do not use
      // the fgStandardSize static because it might have been reset, and
      // therefore should only effect new buttons.
      // This assumes that the button is not relying on this function to
      // size the button to the standard size, when it is originally created.
      if ( IPerformanceSettings::isSettingEnabled (
                    IPerformanceSettings::dynamicToolBarButtons ) )
        {
          if ( bmpSize.width() > fgStandardSize.width() )
            bmpSize.setWidth(fgStandardSize.width());
          if ( bmpSize.height() > fgStandardSize.height() )
            bmpSize.setHeight(fgStandardSize.height());
        }
      else
        {
          if ( bmpSize.width() > fToolBarButtonData->currentStandardSize.width() )
            bmpSize.setWidth(fToolBarButtonData->currentStandardSize.width());
          if ( bmpSize.height() > fToolBarButtonData->currentStandardSize.height() )
            bmpSize.setHeight(fToolBarButtonData->currentStandardSize.height());
        }
      if ( bmpSize != oldSize )
      {
        // The bitmap is too big, so shrink it
        fToolBarButtonData->bmp->sizeTo( IGPoint2D( bmpSize.width(),
                                                    bmpSize.height() ) );
        fToolBarButtonData->bitmapHandle =
            fToolBarButtonData->bmp->image()->handle();
      }
    }
  }

  setLayoutDistorted(IWindow::minimumSizeChanged,0);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBarButton::setLatchedBitmap                                             |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setLatchedBitmap (const IBitmapHandle& handle)
{
  fToolBarButtonData->latchedBitmapHandle = handle;

  if ( fToolBarButtonData->lbmp )
  {
    delete fToolBarButtonData->lbmp;
    fToolBarButtonData->lbmp = 0;
  }

  if ( handle )
  {
    fToolBarButtonData->lbmp = new IImage( fToolBarButtonData->latchedBitmapHandle );
    if ( (fToolBarButtonData->transparentColor) &&
         (extendedStyle() & transparentBitmap.asExtendedUnsignedLong()) )
      fToolBarButtonData->lbmp->setTransparentColor ( *fToolBarButtonData->transparentColor );

    // Check if latched bitmap needs to be resized to fit in standard button
    if ( isStandardFormat() )
    {
      ISize bmpSize = ISize( (IPair::Coord)(fToolBarButtonData->lbmp->geometricBounds().width()),
                             (IPair::Coord)(fToolBarButtonData->lbmp->geometricBounds().height()) ),
            oldSize = bmpSize;
      if ( IPerformanceSettings::isSettingEnabled (
                    IPerformanceSettings::dynamicToolBarButtons ) )
        {
          if ( bmpSize.width() > fgStandardSize.width() )
            bmpSize.setWidth(fgStandardSize.width());
          if ( bmpSize.height() > fgStandardSize.height() )
            bmpSize.setHeight(fgStandardSize.height());
        }
      else
        {
          if ( bmpSize.width() > fToolBarButtonData->currentStandardSize.width() )
            bmpSize.setWidth(fToolBarButtonData->currentStandardSize.width());
          if ( bmpSize.height() > fToolBarButtonData->currentStandardSize.height() )
            bmpSize.setHeight(fToolBarButtonData->currentStandardSize.height());
        }
      if ( bmpSize != oldSize )
      {
        // The bitmap is too big, so shrink it
        fToolBarButtonData->lbmp->sizeTo( IGPoint2D( bmpSize.width(),
                                                     bmpSize.height() ) );
        fToolBarButtonData->latchedBitmapHandle =
             fToolBarButtonData->lbmp->image()->handle();
      }
    }
  }

  setLayoutDistorted(IWindow::minimumSizeChanged,0);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBarButton::bitmap                                                       |
------------------------------------------------------------------------------*/
IBitmapHandle IToolBarButton :: bitmap () const
{
  return fToolBarButtonData->bitmapHandle;
}

/*------------------------------------------------------------------------------
| IToolBarButton::latchedBitmap                                                |
------------------------------------------------------------------------------*/
IBitmapHandle IToolBarButton :: latchedBitmap () const
{
  return fToolBarButtonData->latchedBitmapHandle;
}

/*------------------------------------------------------------------------------
| IToolBarButton::setView                                                      |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setView (IToolBarButton::View buttonView)
{
  unsigned long ulExtStyle = extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;

  switch ( buttonView )
  {
    case bitmapView:
         ulExtStyle |= bitmapVisible.asExtendedUnsignedLong();
         ulExtStyle &= ~textVisible.asExtendedUnsignedLong();
         ulExtStyle &= ~bitmapAndTextVisible.asExtendedUnsignedLong();
         break;
    case textView:
         ulExtStyle &= ~bitmapVisible.asExtendedUnsignedLong();
         ulExtStyle |= textVisible.asExtendedUnsignedLong();
         ulExtStyle &= ~bitmapAndTextVisible.asExtendedUnsignedLong();
         break;
    case bitmapAndTextView:
         ulExtStyle &= ~bitmapVisible.asExtendedUnsignedLong();
         ulExtStyle &= ~textVisible.asExtendedUnsignedLong();
         ulExtStyle |= bitmapAndTextVisible.asExtendedUnsignedLong();
         break;
  }

  if ( ulExtStyle != ulOldExtStyle )
  {
    setExtendedStyle(ulExtStyle);
    setLayoutDistorted(IWindow::minimumSizeChanged,0);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBarButton::view                                                         |
------------------------------------------------------------------------------*/
IToolBarButton::View IToolBarButton :: view () const
{
  if ( isBitmapVisible() )
  {
    if ( isTextVisible() )
      return bitmapAndTextView;
    else
      return bitmapView;
  }
  else
    return textView;
}

/*------------------------------------------------------------------------------
| IToolBarButton::isBitmapVisible                                              |
------------------------------------------------------------------------------*/
bool IToolBarButton :: isBitmapVisible () const
{
  unsigned long ulMask = bitmapVisible.asExtendedUnsignedLong() |
                         bitmapAndTextVisible.asExtendedUnsignedLong();
  return( (extendedStyle() & ulMask) ?  true : false );
}

/*------------------------------------------------------------------------------
| IToolBarButton::isTextVisible                                                |
------------------------------------------------------------------------------*/
bool IToolBarButton :: isTextVisible () const
{
  unsigned long ulMask = textVisible.asExtendedUnsignedLong() |
                         bitmapAndTextVisible.asExtendedUnsignedLong();
  return( (extendedStyle() & ulMask) ?  true : false );
}

/*------------------------------------------------------------------------------
| IToolBarButton::bitmapSize                                                   |
------------------------------------------------------------------------------*/
ISize IToolBarButton :: bitmapSize () const
{
  ISize sz;
  if ( fToolBarButtonData->bitmapHandle != 0 )
  {
    IGImage image( fToolBarButtonData->bitmapHandle );
    sz = ISize( (IPair::Coord)(image.rasterBounds().width()),
                (IPair::Coord)(image.rasterBounds().height()) );
  }
  return sz;
}

/*------------------------------------------------------------------------------
| IToolBarButton::transparentColor                                             |
------------------------------------------------------------------------------*/
IColor IToolBarButton :: transparentColor () const
{
  if (fToolBarButtonData->transparentColor)
    return (*fToolBarButtonData->transparentColor);
  else
    return IColor(0,0,0);
}

/*------------------------------------------------------------------------------
| IToolBarButton::hasTransparentColor                                          |
------------------------------------------------------------------------------*/
bool IToolBarButton :: hasTransparentColor () const
{
  if (fToolBarButtonData->transparentColor)
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| IToolBarButton::resetTransparentColor                                        |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: resetTransparentColor ()
{
  if (fToolBarButtonData->transparentColor)
  {
    delete fToolBarButtonData->transparentColor;
    fToolBarButtonData->transparentColor = 0;

    if ( extendedStyle() & transparentBitmap.asExtendedUnsignedLong() )
      {
        if ( fToolBarButtonData->bmp )
          fToolBarButtonData->bmp->resetTransparentColor();
        if ( fToolBarButtonData->lbmp )
          fToolBarButtonData->lbmp->resetTransparentColor();
        // Mark image as not valid so repaint causes cached button to be
        // recreated with new bitmap.
//        fToolBarButtonData->fflags &= ~IToolBarButtonData::imageValid;
        setLayoutDistorted( IWindow::colorChanged, 0);
        refresh();
      }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBarButton::setTransparentColor                                          |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setTransparentColor (const IColor& aColor)
{
  if (fToolBarButtonData->transparentColor)
  {
    delete fToolBarButtonData->transparentColor;
    fToolBarButtonData->transparentColor = 0;
  }
  fToolBarButtonData->transparentColor = new IColor(aColor);

  //The behavior is to immediately update the bitmap with the new
  //transparent color.  Only do so if style is set.
  if ( extendedStyle() & transparentBitmap.asExtendedUnsignedLong() )
    {
      if ( fToolBarButtonData->bmp )
        fToolBarButtonData->bmp->setTransparentColor(aColor);
      if ( fToolBarButtonData->lbmp )
        fToolBarButtonData->lbmp->setTransparentColor(aColor);
      // Mark image as not valid so repaint causes cached button to be
      // recreated with new bitmap.
//      fToolBarButtonData->fflags &= ~IToolBarButtonData::imageValid;
      setLayoutDistorted( IWindow::colorChanged, 0);
      refresh();
    }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBarButton::setLayoutDistorted                                           |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: setLayoutDistorted
                                ( unsigned long layoutAttributeOn,
                                  unsigned long layoutAttributeOff )
{
  if ( !fToolBarButtonData->skipLayout )
    {
      // If minimum size changed invalidate all caches.
      if (layoutAttributeOn & IWindow::minimumSizeChanged )
        fToolBarButtonData->fflags = IToolBarButtonData::invalid;

      // If color changed invalidate per-button bitmap image and static
      // background bitmaps
      if (layoutAttributeOn & IWindow::colorChanged)
        {
          fToolBarButtonData->fflags &= ~IToolBarButtonData::imageValid;
          IToolBarButtonData::fgOptImageValid = false;

          // Rather than create an IColor (which queries the system for the
          // actual color) each time a color is needed, cache the value
          // and update the cache here.  When the system color changes, all
          // windows will receive a setLayoutDistorted with colorChanged.
          if (toolBarStaticData.colorButtonDark)
            {
              delete toolBarStaticData.colorButtonDark;
            }
          toolBarStaticData.colorButtonDark =
                            new IColor(IColor::buttonDark);
          if (toolBarStaticData.colorButtonLight)
            {
              delete toolBarStaticData.colorButtonLight;
            }
          toolBarStaticData.colorButtonLight =
                            new IColor(IColor::buttonLight);
          if (toolBarStaticData.colorButtonMiddle)
            {
              delete toolBarStaticData.colorButtonMiddle;
            }
          toolBarStaticData.colorButtonMiddle =
                            new IColor(IColor::buttonMiddle);
          if (toolBarStaticData.colorDefaultButton)
            {
              delete toolBarStaticData.colorDefaultButton;
            }
          toolBarStaticData.colorDefaultButton =
                            new IColor(IColor::defaultButton);
        }

      // If font changed invalidate per-button bitmap image.if text visible.
      if ((layoutAttributeOn & IWindow::fontChanged ) && this->isTextVisible())
        fToolBarButtonData->fflags &= ~IToolBarButtonData::imageValid;

      Inherited::setLayoutDistorted(layoutAttributeOn,layoutAttributeOff);

    }  //end skipLayout test
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBarButton::enableDragDelete                                             |
------------------------------------------------------------------------------*/
#ifdef IC_PM
IToolBarButton& IToolBarButton :: enableDragDelete ( bool bTurnOn )
{
unsigned long ulExtStyle = extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;

  if (bTurnOn)
  {
    ulExtStyle |= dragDelete.asExtendedUnsignedLong();
  }
  else
  {
    ulExtStyle &= ~dragDelete.asExtendedUnsignedLong();
  }

  if (ulExtStyle != ulOldExtStyle)
  {
    setExtendedStyle( ulExtStyle );
  }
  return( *this );
}
#endif

#ifndef IC_PM
IToolBarButton& IToolBarButton :: enableDragDelete ( bool )
{
  return( *this );
}
#endif

/*------------------------------------------------------------------------------
| IToolBarButton::disableDragDelete                                            |
------------------------------------------------------------------------------*/
IToolBarButton& IToolBarButton :: disableDragDelete ( )
{
#ifdef IC_PM
  enableDragDelete( false );
#endif
  return( *this );
}

/*------------------------------------------------------------------------------
| IToolBarButton::allowsDragDelete                                             |
------------------------------------------------------------------------------*/
bool IToolBarButton :: allowsDragDelete ( ) const
{
#ifdef IC_PM
  return( (extendedStyle() & dragDelete.asExtendedUnsignedLong()) ? true
                                                                    : false );
#else
  return false;
#endif
}

/*------------------------------------------------------------------------------
| IToolBarButton::calcMinimumSize                                              |
------------------------------------------------------------------------------*/
ISize IToolBarButton :: calcMinimumSize () const
{
  // Set up local variables used throughout this function
  unsigned long cxBorderTimesFive = toolBarStaticData.cxSysBorder * 5;
  unsigned long cyBorderTimesFive = toolBarStaticData.cxSysBorder * 5;

  ISize  oldSize( fToolBarButtonData->minSize );

  // Check for standard formatting
  if ( isStandardFormat() )
  {
    // The minimum size includes 5 pels of border and 2 of pad
    fToolBarButtonData->minSize.setWidth  ( cxBorderTimesFive + 2);
    fToolBarButtonData->minSize.setHeight ( cyBorderTimesFive + 2);

    if ( IPerformanceSettings::isSettingEnabled (
                IPerformanceSettings::dynamicToolBarButtons ) )
    {
     if ( isBitmapVisible() )
     {
       // Add room for bitmap
       fToolBarButtonData->minSize.setWidth (
                                     fToolBarButtonData->minSize.width() +
                                     fgStandardSize.width() );
       fToolBarButtonData->minSize.setHeight (
                                     fToolBarButtonData->minSize.height() +
                                     fgStandardSize.height() );
     }

     if ( isTextVisible() )
     {
       // Check if the button is already wide enough for text
       if ( !(isBitmapVisible()) ||
          ( fgStandardWidth + 6 ) > fgStandardSize.width() )
       {
         fToolBarButtonData->minSize.setWidth(
                                     cxBorderTimesFive + 8 + fgStandardWidth);
       }

       // Add room for the lines of text
       fToolBarButtonData->minSize.setHeight (
                              fToolBarButtonData->minSize.height() +
                              ( fgStandardLines * characterSize().height() ));
     }
    }  //IPerformanceSettings::dynamicToolBarButtons
    else
    {
     // if standardsize, and not dynamic toolbarbuttons.
     if ( isBitmapVisible() )
     {
       // Add room for bitmap
       fToolBarButtonData->minSize.setWidth (
                             fToolBarButtonData->minSize.width() +
                             fToolBarButtonData->currentStandardSize.width() );
       fToolBarButtonData->minSize.setHeight (
                             fToolBarButtonData->minSize.height() +
                             fToolBarButtonData->currentStandardSize.height() );
     }

     if ( isTextVisible() )
     {
       // Check if the button is already wide enough for text
       if ( !(isBitmapVisible()) ||
          ( fToolBarButtonData->currentStandardWidth + 6 ) >
            fToolBarButtonData->currentStandardSize.width() )
       {
         fToolBarButtonData->minSize.setWidth ( cxBorderTimesFive + 8 +
                       fToolBarButtonData->currentStandardWidth );
       }

       // Add room for the lines of text
       fToolBarButtonData->minSize.setHeight (
                              fToolBarButtonData->minSize.height() +
                              ( fToolBarButtonData->currentStandardLines *
                                characterSize().height() ));
     }
    }

    fToolBarButtonData->fflags |= IToolBarButtonData::sizeValid;
    if ( fToolBarButtonData->minSize != oldSize )
       fToolBarButtonData->fflags &= ~IToolBarButtonData::imageValid;
  } // endif isStandardSize()

  // Normal sizing (based on the size of bitmap and text)
  if ( fToolBarButtonData->fflags & IToolBarButtonData::sizeValid )
  {
    return fToolBarButtonData->minSize;
  }
  else
  {
    if ( isTextVisible() )
    {
      fToolBarButtonData->minSize = Inherited::calcMinimumSize();
    }
    else
    {
      fToolBarButtonData->minSize.setWidth ( cxBorderTimesFive );
      fToolBarButtonData->minSize.setHeight ( cyBorderTimesFive );
    }

    if ( isBitmapVisible() )
    {
      // The minimum size when the bitmap is visible should include room
      // for the bitmap and 5 pels of border (for width)
      ISize minBitmapSize;
      if ( fToolBarButtonData->bmp )
         {
         ISize
           bmpSize = ISize( (IPair::Coord)(fToolBarButtonData->bmp->geometricBounds().width()),
                            (IPair::Coord)(fToolBarButtonData->bmp->geometricBounds().height()) );
         minBitmapSize = bmpSize + ISize( cxBorderTimesFive, 0 );
         }
      else
         minBitmapSize = ISize( cxBorderTimesFive, 0 );

      if ( fToolBarButtonData->minSize.width() < minBitmapSize.width() )
      {
        fToolBarButtonData->minSize.setWidth ( minBitmapSize.width() );
      }

      fToolBarButtonData->minSize.setHeight (
                                    fToolBarButtonData->minSize.height() +
                                    minBitmapSize.height() );
    }

    fToolBarButtonData->fflags |= IToolBarButtonData::sizeValid;
    if ( fToolBarButtonData->minSize != oldSize )
       fToolBarButtonData->fflags &= ~IToolBarButtonData::imageValid;
    return fToolBarButtonData->minSize;
  }
}

/*------------------------------------------------------------------------------
| IToolBarButton::standardTextWidth                                            |
------------------------------------------------------------------------------*/
unsigned long IToolBarButton::standardTextWidth ()
{
  return fgStandardWidth;
}

/*------------------------------------------------------------------------------
| IToolBarButton::standardTextLines                                            |
------------------------------------------------------------------------------*/
unsigned long IToolBarButton::standardTextLines ()
{
  return fgStandardLines;
}

/*------------------------------------------------------------------------------
| IToolBarButton::standardBitmapSize                                           |
------------------------------------------------------------------------------*/
ISize IToolBarButton::standardBitmapSize ()
{
  return fgStandardSize;
}

/*------------------------------------------------------------------------------
| ToolBarButton::setStandardTextWidth                                          |
------------------------------------------------------------------------------*/
void IToolBarButton::setStandardTextWidth ( unsigned long newWidth )
{
  if ( IPerformanceSettings::isSettingEnabled (
                IPerformanceSettings::dynamicToolBarButtons ) )
     {
      if (fgStandardWidth != newWidth)
        {
          fgStandardWidth = newWidth;
          if ( toolBarStaticData.standardTextOnly )
          {
            delete toolBarStaticData.standardTextOnly;
            toolBarStaticData.standardTextOnly = 0;
          }
          if ( toolBarStaticData.standardBitmapAndText )
          {
            delete toolBarStaticData.standardBitmapAndText;
            toolBarStaticData.standardBitmapAndText = 0;
          }
          toolBarStaticData.buttonList().textFormatChanged();
        }
     }
  else
     {
       // Set the width for use on next toolbarButton created
       fgStandardWidth = newWidth;
     }
}

/*------------------------------------------------------------------------------
| IToolBarButton::setStandardTextLines                                         |
------------------------------------------------------------------------------*/
void IToolBarButton::setStandardTextLines ( unsigned long newLines )
{
  if ( IPerformanceSettings::isSettingEnabled (
                IPerformanceSettings::dynamicToolBarButtons ) )
     {
      if (fgStandardLines != newLines)
        {
          fgStandardLines = newLines;
          if ( toolBarStaticData.standardTextOnly )
          {
            delete toolBarStaticData.standardTextOnly;
            toolBarStaticData.standardTextOnly = 0;
          }
          if ( toolBarStaticData.standardBitmapAndText )
          {
            delete toolBarStaticData.standardBitmapAndText;
            toolBarStaticData.standardBitmapAndText = 0;
          }
          toolBarStaticData.buttonList().textFormatChanged();
        }
     }
  else
     {
       // Set the lines for use on next toolbarButton created
       fgStandardLines = newLines;
     }
}

/*------------------------------------------------------------------------------
| IToolBarButton::setStandardBitmapSize                                        |
------------------------------------------------------------------------------*/
void IToolBarButton::setStandardBitmapSize ( const ISize& newSize )
{
  if ( IPerformanceSettings::isSettingEnabled (
                IPerformanceSettings::dynamicToolBarButtons ) )
     {
      if (fgStandardSize != newSize)
        {
          fgStandardSize = newSize;
          if ( toolBarStaticData.standardBitmapOnly )
          {
            delete toolBarStaticData.standardBitmapOnly;
            toolBarStaticData.standardBitmapOnly = 0;
          }
          if ( toolBarStaticData.standardBitmapAndText )
          {
            delete toolBarStaticData.standardBitmapAndText;
            toolBarStaticData.standardBitmapAndText = 0;
          }
          toolBarStaticData.buttonList().bitmapFormatChanged();
        }
     }
  else
     {
       // Set the size for use on next toolbarButton created
       fgStandardSize = newSize;
     }
}

/*------------------------------------------------------------------------------
| IToolBarButton::isStandardFormat                                             |
------------------------------------------------------------------------------*/
bool IToolBarButton :: isStandardFormat () const
{
  return( (extendedStyle() & standardFormat.asExtendedUnsignedLong()) ? true
                                                                      : false );
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| iToolBarButtonMotifCallback                                                  |
------------------------------------------------------------------------------*/
void _System iToolBarButtonMotifCallback(
                     Widget     handle,
                     XtPointer  client_data,
                     XtPointer  call_data)
{
  IMODTRACE_ALL("iToolBarButtonMotifCallback");

  return;
}
#endif  //IC_MOTIF
