// Revision: 15 1.3.1.3 source/ui/drag/iimagels.cpp, dragdrop, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iimagels.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IDMDropSource                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#include <iimagels.hpp>

#ifndef _ITRACE_
  #include <itrace.hpp>
#endif
#include <icconst.h>

#ifndef _IEXCEPT_
  #include <iexcept.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IIMAGELS_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IImageList::IImageList                                                         |
|                                                                              |
| Constructor.                                                                 |
|   Try to keep this light-weight, they are created frequently.                |
------------------------------------------------------------------------------*/
IImageList::IImageList()
  : _imageList( NULL )
  , _activeImage( 0 )

  {
  IMODTRACE_DEVELOP( "ImageList::ImageList()");
  }

/*------------------------------------------------------------------------------
| IImageList::IImageList                                                         |
|                                                                              |
| Constructor.                                                                 |
|   Try to keep this light-weight, they are created frequently.                |
------------------------------------------------------------------------------*/
IImageList::IImageList( HIMAGELIST existingImageList )
  : _imageList( existingImageList )
  , _activeImage( 0 )

  {
  IMODTRACE_DEVELOP( "ImageList::ImageList(HIMAGELIST)");
  if ( _imageList == NULL )
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable);
  }

/*------------------------------------------------------------------------------
| IImageList::~IImageList                                                         |
|                                                                              |
| destructor.                                                                 |
------------------------------------------------------------------------------*/
IImageList::~IImageList()
  {
  IMODTRACE_DEVELOP( "ImageList::~ImageList()");
  if ( _imageList != NULL )
    ::ImageList_Destroy( _imageList);
  }

/*------------------------------------------------------------------------------
| IImageList::create
|
| Create will create the image list. We don't want to create the
| image list during the ctor because we may want to throw an
| exception.
------------------------------------------------------------------------------*/
IImageList &IImageList::create( const ISize &imageSize,
                                unsigned flags,
                                int initialImageListSize,
                                int growthFactor )
  {
  IMODTRACE_DEVELOP( "ImageList::create");

  if ( _imageList != NULL )
    ::ImageList_Destroy( _imageList );

  _imageList = ::ImageList_Create( imageSize.width(),
                                   imageSize.height(),
                                   flags,
                                   initialImageListSize,
                                   growthFactor );
  if ( _imageList == NULL )
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable);
  return *this;
  }

/*------------------------------------------------------------------------------
| IImageList::setActiveImage
|
| Marks an image in the image list as being the active one.
------------------------------------------------------------------------------*/
IImageList &IImageList::setActiveImage( int imageNumber )

  {
  IMODTRACE_DEVELOP( "ImageList::setActiveImage");
  _activeImage = imageNumber;
  return *this;
  }

/*------------------------------------------------------------------------------
| IImageList::activeImage
|
| Returns the active image index.
------------------------------------------------------------------------------*/
int IImageList::activeImage() const
  {
  IMODTRACE_DEVELOP( "ImageList::activeImage()");
  return _activeImage;
  }

/*------------------------------------------------------------------------------
| IImageList::mergeImagesIntoDest
|
| Merges two images into a destination image list and returns the
| index.
------------------------------------------------------------------------------*/
int IImageList::mergeImagesIntoDest( int   backgroundImage,
                                     const IImageList &foregroundImageList,
                                     int   foregroundImage,
                                     const IPoint     &offset,
                                     IImageList &destination  )
  {
  IMODTRACE_DEVELOP( "ImageList::mergeImagesIntoDest");

  // Create the merged icon.
  IImageList mergedImageList( ::ImageList_Merge( imageList(),
                              backgroundImage,
                              foregroundImageList.imageList(),
                              foregroundImage,
                              offset.x(),
                              offset.y() ));
  mergedImageList.setActiveImage( 0 );
  return destination.addIconFromImageList(mergedImageList );
  }

/*------------------------------------------------------------------------------
| IImageList::icon
|
| Returns the image at index.
------------------------------------------------------------------------------*/
HICON IImageList::icon( int index )

  {
  IMODTRACE_DEVELOP( "ImageList::icon()");
  int whichOne = index;
  if ( index == -1 )
    whichOne = activeImage();

  HICON theIcon;
  theIcon = ::ImageList_GetIcon( imageList(),
                               whichOne,
                               ILD_TRANSPARENT );
  if ( theIcon == NULL )
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable);
  return theIcon;
  }

/*------------------------------------------------------------------------------
| IImageList::addIconFromImageList
|
| Adds active image from image list to this image list and returns index.
------------------------------------------------------------------------------*/
int IImageList::addIconFromImageList( IImageList &fromImageList )

  {
  IMODTRACE_DEVELOP( "ImageList::addIconFromImageList()");
  HICON mergedIcon = fromImageList.icon( );

  int imageIndex = ::ImageList_AddIcon( imageList(), mergedIcon );
  ::DestroyIcon( mergedIcon );

  if ( imageIndex == -1)
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable);

  return imageIndex;
  }

/*------------------------------------------------------------------------------
| IImageList::imageList
|
| Returns the imageList.
------------------------------------------------------------------------------*/
HIMAGELIST IImageList::imageList() const

  {
  return _imageList;
  }

/*------------------------------------------------------------------------------
| IImageList::addIcon
|
| Adds the icon to the image list.
------------------------------------------------------------------------------*/
int IImageList::addIcon( HICON icon )

  {
  IMODTRACE_DEVELOP( "ImageList::addIcon()");
  int index = ::ImageList_AddIcon( imageList(), icon );
  if ( index == -1 )
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable);

  return index;
  }

/*------------------------------------------------------------------------------
| IImageList::addCursor
|
| Adds the icon to the image list.
------------------------------------------------------------------------------*/
int IImageList::addCursor( HCURSOR cursor )

  {
  IMODTRACE_DEVELOP( "ImageList::addIcon()");
  HICON icon = cursor;
  return addIcon( icon );
  }


/*------------------------------------------------------------------------------
| IImageList::add
|
| Adds the bitmap to the image list.
------------------------------------------------------------------------------*/
int IImageList::add( HBITMAP bitmap )

  {
  IMODTRACE_DEVELOP( "ImageList::add()");
  int index = ::ImageList_Add( imageList(), bitmap, NULL );
  if ( index == -1 )
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable);

  return index;
  }


/*------------------------------------------------------------------------------
| IImageList::addIconAsSeeThru
|
| Modifies the icons mask to make the icon more transparent and adds the icon
| to the image list.
------------------------------------------------------------------------------*/
int IImageList::addIconAsSeeThru( HICON &icon )

  {
  IMODTRACE_DEVELOP("IImageList::addIconAsSeeThru");

  ICONINFO iconInfo;
  if ( GetIconInfo( icon, &iconInfo ) == FALSE )
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
      IBaseErrorInfo::invalidRequest,
      IException::recoverable);

  HBITMAP hSourceBitmap = iconInfo.hbmMask;
  BITMAP  iSourceBitmap;

  HDC hdc = ::GetDC( HWND_DESKTOP );

  HDC hMemSrc  = ::CreateCompatibleDC( NULL );
  HDC hMemDest = ::CreateCompatibleDC( NULL );

  ::GetObject( hSourceBitmap, sizeof(BITMAP), (LPSTR)&iSourceBitmap);

  int width  = iSourceBitmap.bmWidth;
  int height = iSourceBitmap.bmHeight;

  HBITMAP hOldSourceBitmap = (HBITMAP)
                               ::SelectObject( hMemSrc, hSourceBitmap );
  HBITMAP hDestBitmap      = ::CreateBitmap( height, width,
                                             iSourceBitmap.bmPlanes,
                                             iSourceBitmap.bmBitsPixel,
                                             NULL);

  HBITMAP hOldDestBitmap   = (HBITMAP)
                               ::SelectObject( hMemDest, hDestBitmap );

  //------------------------------------------------------------------
  // Given a monochrome mask,
  // - Toggle a constant fraction of black pixels: a repeating pattern
  // - Keep all white pixels (transparent)
  //
  // bool raster operation:
  //
  //                         Step 3
  //          |-------------------------|
  //                Step 2
  //          |----------------|
  //            Step 1
  //           |-----|
  // Result = ( ~Mask & Pattern ) | Mask
  //
  // Example of small linear mask. 1's are transparent areas:
  //
  //                  xparent   image xparent  <-- Areas of mask
  //                  ------- x x x x -------
  //    Mask        : 1 1 1 1 0 0 0 0 1 1 1 1
  //   ~Mask        : 0 0 0 0 1 1 1 1 0 0 0 0
  //    Pattern     : 1 0 1 0 1 0 1 0 1 0 1 0
  // ~Mask & Pattern: 0 0 0 0 1 0 1 0 0 0 0 0
  //   |Mask        : 1 1 1 1 1 0 1 0 1 1 1 1
  //                          x x x x
  //
  // The x x x x is a ghostly image now
  //
  // Wow! We get to use a rater op other than SRCCOPY!
  // Three of 'em!
  //------------------------------------------------------------------

  //-------------------------------
  // hMesDest <-- ~Mask
  //-------------------------------
  ::BitBlt(
      hMemDest,
      0, 0,
      width, height,
      hMemSrc,
      0, 0,
      NOTSRCCOPY   // ~source --> dest
      );

  //------------------------------------------
  // Use stock GRAY_BRUSH brush as a pattern.
  // LT_GRAY is more vacuous.
  // DK_GRAY is more solid.
  // Okay to leave stock objects selected.
  //------------------------------------------
  ::SelectObject( hMemDest, ::GetStockObject( GRAY_BRUSH ) );

  //-------------------------------
  // hMemDest <-- & pattern
  //-------------------------------
  ::BitBlt(
      hMemDest,
      0, 0,
      width, height,
      hMemDest,
      0, 0,
      MERGECOPY  // Source & pattern --> dest
      );

  //--------------------------------
  // hMemDest <--- | Mask
  //--------------------------------
  ::BitBlt(
      hMemDest,
      0, 0,
      width, height,
      hMemSrc,
      0, 0,
      SRCPAINT  // source | dest --> dest
      );

  ::ReleaseDC( HWND_DESKTOP, hdc );

  ::SelectObject(hMemSrc, hOldSourceBitmap);
  ::SelectObject(hMemDest, hOldDestBitmap);
  ::DeleteDC(hMemDest);
  ::DeleteDC(hMemSrc);

  ::DeleteObject( iconInfo.hbmMask );
  iconInfo.hbmMask = hDestBitmap;
  icon = ::CreateIconIndirect( &iconInfo );

  ::DeleteObject( hDestBitmap );

  int iconIndex = addIcon( icon );
  ::DestroyIcon( icon );

  // Clean up bitmaps created and passed from GetIconInfo
  ::DeleteObject( iconInfo.hbmColor );
  return iconIndex;
  }

/*------------------------------------------------------------------------------
| IImageList::numberOfImages
|
| Returns the numberOfImages.
------------------------------------------------------------------------------*/
int IImageList::numberOfImages() const

  {
  IMODTRACE_DEVELOP( "ImageList::numberOfImages()");
  return ::ImageList_GetImageCount( imageList() );
  }

/*------------------------------------------------------------------------------
| IImageList::imageSize
|
| Returns the image list film size.
------------------------------------------------------------------------------*/
ISize IImageList::imageSize() const
  {
  IMODTRACE_DEVELOP( "ImageList::imageSize()");
  int width, height;
  ::ImageList_GetIconSize( imageList(), &width, &height );
  return ISize( width, height);
  }

/*------------------------------------------------------------------------------
| IImageList::switchDragImage
|
| Switches the current drag image to the image that has been marked active.
------------------------------------------------------------------------------*/
IImageList &IImageList::switchDragImage( const IPoint &hotSpot,
                             const IPoint &cursorPos )

  {
  // Need to add some protection to make sure a drag is actually occurring.
  IMODTRACE_DEVELOP( "ImageList::switchDragImage()");
  ITRACE_DEVELOP( "Drag image is " + IString( activeImage() ));
  ::ImageList_DragLeave( NULL );
  ::ImageList_EndDrag( );
  ::ImageList_BeginDrag( imageList(),
                       activeImage(),
                       hotSpot.x(),
                       hotSpot.y() );
  ::ImageList_DragEnter( NULL, (int)cursorPos.x(), (int)cursorPos.y() );
  return *this;
  }

