// Revision: 08 1.8.1.5 source/ui/drag/idmtgth.cpp, dragdrop, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: idmtgth.hpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IDMTargetHandler - Drag/Drop target window handler.                      *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
extern "C" {
  /* define what subsections of PM headers to include */
  #define INCL_WINSTDDRAG       // Standard drag stuff
  #define INCL_WINSTDCNR        // Container drag stuff
  #include <iwindefs.h>
}

#include <iexcept.hpp>
#include <itrace.hpp>
#include <idmtgth.hpp>
#include <idmevent.hpp>
#include <idmtgtrn.hpp>
#include <idmcomm.hpp>
#include <idmitem.hpp>
#include <idmefit.hpp>
#include <idmmleit.hpp>
#include <idmcnrit.hpp>
#include <ictlevt.hpp>
#include <idmprov.hpp>
#include <ientryfd.hpp>
#include <imle.hpp>
#include <icnrctl.hpp>
#include <idmseq.hpp>
#include <icconst.h>

#ifdef IC_MOTIF  
    #include <idmenab.hpp>
    #include <ixevtids.h>
    #include <icnrm.h>
#endif // IC_MOTIF

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IDMTGTH_CPP_
  #include <ipagetun.h>
#endif

typedef IDM::DropIndicator DropIndicator;

/*------------------------------------------------------------------------------
| IDMTargetHandler::IDMTargetHandler                                           |
|                                                                              |
| Constructor (for IWindow *)                                                  |
------------------------------------------------------------------------------*/
IDMTargetHandler :: IDMTargetHandler (IWindow* pWin)  :
                                      IDMHandler()
{
  IMODTRACE_DEVELOP("IDMTargetHandler::ctor (win)");

  /*-------------------------------------------------------------------
     Add target renderer to target drag handler
   -------------------------------------------------------------------*/
  addRenderer( IDMRenderer::defaultTargetRenderer() );

  /*-------------------------------------------------------------------
     Configure the window as a target drag handler
   -------------------------------------------------------------------*/
  handleEventsFor(pWin);
#ifdef IC_MOTIF 
  IDMDragDropEnabler::enableDropOn( pWin->handle() );
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::IDMTargetHandler                                           |
|                                                                              |
| Constructor (for entry field)                                                |
------------------------------------------------------------------------------*/
IDMTargetHandler :: IDMTargetHandler (IEntryField* pEF) :
                    IDMHandler()
{
  IMODTRACE_DEVELOP("IDMTargetHandler::ctor (ef)");

  /*-------------------------------------------------------------------
     Add target renderer to target handler
   -------------------------------------------------------------------*/
  addRenderer( IDMRenderer::defaultTargetRenderer() );

  /*-------------------------------------------------------------------
     Designate drag item provider for the entry field
   -------------------------------------------------------------------*/
  IDMHandler::setItemProvider(pEF);

  /*-------------------------------------------------------------------
     Configure the entry field as target drag handler
   -------------------------------------------------------------------*/
  handleEventsFor(pEF);
#ifdef IC_MOTIF 
  IDMDragDropEnabler::enableDropOn( pEF->handle() );
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::IDMTargetHandler                                           |
|                                                                              |
| Constructor (for MLE)                                                        |
------------------------------------------------------------------------------*/
IDMTargetHandler :: IDMTargetHandler (IMultiLineEdit* pMLE) :
                                                  IDMHandler()
{
  IMODTRACE_DEVELOP("IDMTargetHandler::ctor (mle)");

  /*-------------------------------------------------------------------
     Add target renderer to target handler
   -------------------------------------------------------------------*/
  addRenderer( IDMRenderer::defaultTargetRenderer() );

  /*-------------------------------------------------------------------
     Designate drag item provider for the MLE
   -------------------------------------------------------------------*/
  IDMHandler::setItemProvider(pMLE);

  /*-------------------------------------------------------------------
     Configure the entry field as target drag handler
   -------------------------------------------------------------------*/
  handleEventsFor(pMLE);
#ifdef IC_MOTIF 
  IDMDragDropEnabler::enableDropOn( pMLE->handle() );
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::IDMTargetHandler                                           |
|                                                                              |
| Constructor (for container)                                                  |
------------------------------------------------------------------------------*/
IDMTargetHandler :: IDMTargetHandler(IContainerControl* pCnr)  :
                    IDMHandler()
{
  IMODTRACE_DEVELOP("IDMTargetHandler::ctor (cnr)");

  /*-------------------------------------------------------------------
     Add target renderer to target drag handler
   -------------------------------------------------------------------*/
  addRenderer( IDMRenderer::defaultTargetRenderer() );

  /*-------------------------------------------------------------------
     Designate drag item provider for the container
   -------------------------------------------------------------------*/
  IDMHandler::setItemProvider(pCnr);

  /*-------------------------------------------------------------------
     Configure the container as a target drag handler
   -------------------------------------------------------------------*/
  handleEventsFor(pCnr);
#ifdef IC_MOTIF 
  IDMDragDropEnabler::enableDropOn( pCnr->handle() );
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IDMTargetHandler:: IDMTargetHandler                                          |
|                                                                              |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
IDMTargetHandler :: IDMTargetHandler() :
                    IDMHandler()
{
  IMODTRACE_DEVELOP("IDMTargetHandler::ctor ()");

  /*-------------------------------------------------------------------
          Add target renderer to target drag handler
   -------------------------------------------------------------------*/
  addRenderer( IDMRenderer::defaultTargetRenderer() );
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::~IDMTargetHandler                                          |
|                                                                              |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
IDMTargetHandler :: ~IDMTargetHandler()
{
  IMODTRACE_DEVELOP("IDMTargetHandler::dtor ()");
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::dispatchHandlerEvent                                       |
| Overloaded handler                                                           |
| Processes the following PM Drag messages:                                    |
|      DM_DRAGOVER                                                             |
|      DM_DRAGLEAVE                                                            |
|      DM_DROP                                                                 |
|      DM_DROPHELP                                                             |
|      DM_RENDERCOMPLETE                                                       |
------------------------------------------------------------------------------*/
bool IDMTargetHandler :: dispatchHandlerEvent(IEvent& evt)

  {
  bool bProcessed = false;

  switch(evt.eventId())
    {
    case DM_DRAGOVER:
      {
      IMODTRACE_DEVELOP("IDMTargetHandler DM_DRAGOVER");

      /*---------------------------------------------------------------
          Container check
        ---------------------------------------------------------------*/
#ifdef IC_MOTIFPM 
      IWindow *pCtrlWin = evt.window();
      if (isContainerControl(pCtrlWin))
        return(false);
#endif

      IDMTargetEnterEvent event = IDMTargetEnterEvent(evt);
      bProcessed = targetEnter(event);
      evt.setResult(event.result());

      ITRACE_DEVELOP("Drop Ind   = "+IString((int)event.dropIndicator()));
      ITRACE_DEVELOP("Default Op = "+IString((int)event.defaultOperation()));
      break;
    }/* end DM_DRAGOVER */


  case DM_DRAGLEAVE:
    {
    IMODTRACE_DEVELOP("IDMTargetHandler DM_DRAGLEAVE");

#ifdef IC_MOTIFPM 
    /*---------------------------------------------------------------
      Container check
      ---------------------------------------------------------------*/
    IWindow *pCtrlWin = evt.window();
    if (isContainerControl(pCtrlWin))
      return(false);
#endif
    IDMTargetLeaveEvent event = IDMTargetLeaveEvent(evt);
    bProcessed = targetLeave(event);
    evt.setResult(event.result());
    break;
    }/* DM_DRAGLEAVE */


  case DM_DROP:
    {
    IMODTRACE_DEVELOP("IDMTargetHandler DM_DROP");
    IWindow *pCtrlWin = evt.window();

#ifdef IC_MOTIFPM 
    /*---------------------------------------------------------------
      Container check
      ---------------------------------------------------------------*/
    if (isContainerControl(pCtrlWin))
      return(false);
#endif // IC_MOTIFPM

    IDMTargetDropEvent event = IDMTargetDropEvent(evt);
    bProcessed = targetDrop(event,
        isContainerControl(pCtrlWin));
    evt.setResult(event.result());
    break;
    }/* DM_DROP */


  case DM_DROPHELP:
    {
    IMODTRACE_DEVELOP("IDMTargetHandler DM_DROPHELP");

#ifdef IC_MOTIFPM 
    /*---------------------------------------------------------------
      Container check
      ---------------------------------------------------------------*/
    IWindow *pCtrlWin = evt.window();
    if (isContainerControl(pCtrlWin))
      return(false);
#endif // IC_MOTIFPM

    IDMTargetHelpEvent event = IDMTargetHelpEvent(evt);
    bProcessed = targetHelp(event);
    evt.setResult(event.result());
    break;
  }/* DM_DROPHELP */

#ifndef IC_MOTIF 
  case DM_RENDERCOMPLETE:
    {
    IMODTRACE_DEVELOP("IDMTargetHandler DM_RENDERCOMPLETE");

    IDMTargetEndEvent event = IDMTargetEndEvent(evt);
    bProcessed = targetEnd(event);
    evt.setResult(event.result());
    break;
    }/* DM_RENDERCOMPLETE */
#endif // ~IC_MOTIF

#ifdef IC_MOTIFPM 
    case WM_CONTROL:
    {
#ifdef IC_PM
      IControlEvent ctlevt(evt);
      IWindow* pwndControl = ctlevt.controlWindow();
#else
      IWindow* pwndControl = evt.controlWindow();
#endif

      if (isContainerControl(pwndControl))
      {
        /*-------------------------------------------------------------
           Container if at this point
         -------------------------------------------------------------*/
        switch(evt.parameter1().number2())
        {
          case CN_DRAGAFTER:
          case CN_DRAGOVER:
          {
            IMODTRACE_DEVELOP("IDMTargetHandler CN_DRAGOVER/AFTER");
            IDMTargetEnterEvent event( evt );
            bProcessed = targetEnter(event);
            evt.setResult(event.result());
                                break;
          }

          case CN_DROP:
          {
            IMODTRACE_DEVELOP("IDMTargetHandler CN_DROP");
            IDMTargetDropEvent event( evt );
            /*---------------------------------------------------------
               Do the actual drop processing
             ---------------------------------------------------------*/
            bProcessed = targetDrop(event);
            evt.setResult(event.result());
            break;
          }

          case CN_DRAGLEAVE:
          {
            IMODTRACE_DEVELOP("IDMTargetHandler CN_DRAGLEAVE");
            IDMTargetLeaveEvent event( evt );
            bProcessed = targetLeave(event);
            evt.setResult(event.result());
            break;
          }

          case CN_DROPHELP:
          {
            IMODTRACE_DEVELOP("IDMTargetHandler CN_DROPHELP");
            IDMTargetHelpEvent event( evt );
            bProcessed = targetHelp(event);
            evt.setResult(event.result());
            break;
          }
        }/* end switch */
      }
      break;

    }/* end WM_CONTROL */
#endif // IC_MOTIFPM

  }/* end switch */

  return(bProcessed);
} // dispatchHandlerEvent

/*------------------------------------------------------------------------------
| IDMTargetHandler::targetEnter                                                |
|                                                                              |
| Replies to dragover message                                                  |
------------------------------------------------------------------------------*/
bool IDMTargetHandler :: targetEnter(IDMTargetEnterEvent& event)
{
  IMODTRACE_DEVELOP("IDMTargetHandler::targetEnter") ;
  /*------------------------------------------------------------------------
    Get target operation object for this target enter (i.e. drag over)
    event if it exists.
  ------------------------------------------------------------------------*/
  IDMTargetOperation::Handle pDragOp =
                             IDMTargetOperation::instanceFor(event);

  /*------------------------------------------------------------------------
    If it doesn't exist, allocate one ...
  ------------------------------------------------------------------------*/
  if (! pDragOp.valid() )
    pDragOp = allocateOperation(event);

  /*----------------------------------------------------------------------
    Get the item provider for the target window, which is set by the
    IDMTargetOperation ctor
  ----------------------------------------------------------------------*/
  IDMItemProvider *pProvider = pDragOp->targetWindow()->itemProvider();

  /*------------------------------------------------------------------------
    Provider has option of setting the drop indicator and/or target
    operation (i.e. move, copy, link, etc...).
  ------------------------------------------------------------------------*/
  pProvider->provideEnterSupport(event);

  /*------------------------------------------------------------------------
    Do not attempt renderering if the provider override for
    provideEnterSupport() sets the drop indicator to either of these
  ------------------------------------------------------------------------*/
  if ((event.dropIndicator() == IDM::neverOk)  ||
      (event.dropIndicator() == IDM::notOk))
  {
    event.setResult(IEventResult(event.dropIndicator(),
                                 event.defaultOperation()));
    return(true);
  }

  /*------------------------------------------------------------------------
     Find the appropriate target renderers for the operation
  ------------------------------------------------------------------------*/
  findRenderersFor(pDragOp);

  /*------------------------------------------------------------------------
    Determine the best case drop sceanario
  ------------------------------------------------------------------------*/
  DropIndicator dropFlag = IDM::ok;
  unsigned count = pDragOp->numberOfItems();
  for (unsigned i=1; i <= count; i++)
  {
    DropIndicator dropstatus = pDragOp->item(i)->dropStatus();
    switch (dropstatus)
    {
      case IDM::neverOk:
        dropFlag = IDM::neverOk;
        i = count;
        break;

      case IDM::operationNotOk:
        dropFlag = IDM::operationNotOk;
        break;

      case IDM::notOk:
        if (dropFlag != IDM::operationNotOk)
          dropFlag = IDM::notOk;
        break;

      default:
        break;
    }
  }

  /*------------------------------------------------------------------------
    Set the default operation
  ------------------------------------------------------------------------*/
  event.setDefaultOperation(pDragOp->operation());

  /*------------------------------------------------------------------------
    Set the drop indicator
  ------------------------------------------------------------------------*/
  event.setDropIndicator(dropFlag);

  /*----------------------------------------------------------------------
    Set the result and return ...
  ----------------------------------------------------------------------*/
  event.setResult(IEventResult(event.dropIndicator(),
                               event.defaultOperation()));
  return(true);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::allocateOperation                                          |
|                                                                              |
| Creates an instance of the IDMTargetOperation class                          |
| and returns a pointer to it.                                                 |
------------------------------------------------------------------------------*/
IDMTargetOperation::Handle IDMTargetHandler ::
                           allocateOperation(IDMTargetEnterEvent& event) const
{
  /*------------------------------------------------------------------------
    Construct the target operation and save in a static location
  ------------------------------------------------------------------------*/
  IDMTargetOperation::Handle pTgtOpH(new IDMTargetOperation(event)) ; 
  IDMTargetOperation::setTargetOperation (pTgtOpH);
  return (pTgtOpH);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::targetLeave                                                |
|                                                                              |
| Called when the dragged objects leave the target window                      |
------------------------------------------------------------------------------*/
bool IDMTargetHandler :: targetLeave(IDMTargetLeaveEvent& event)
{
  IMODTRACE_DEVELOP("IDMTargetHandler::targetLeave");

  IDMTargetOperation::Handle pTgtOpH(IDMTargetOperation::targetOperation()) ;

#ifdef IC_MOTIFPM 
  if (event.eventId() != DM_DRAGLEAVE)
  {
    /*-------------------------------------------------------------------
       Verify that this is the correct message and notification code
     -------------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            event.parameter1().number2() == CN_DRAGLEAVE);

    /*--------------------------------------------------------------------
      Set the source container to the no refresh state to prevent updates
      until the drag processing has completed
    --------------------------------------------------------------------*/
    if (pTgtOpH->sourceWindowHandle() == pTgtOpH->targetWindowHandle())
    {
      ITRACE_DEVELOP("!!!!!!!!Setting source container refresh off!!!!!!!");
      pTgtOpH->setContainerRefreshOff(
                             (IContainerControl *)pTgtOpH->sourceWindow() );
    }
  }
#endif // IC_MOTIFPM

  /*----------------------------------------------------------------------
    Get the item provider for the window, and give it a shot at
    any cleanup per the target leave event.
  ----------------------------------------------------------------------*/
  IDMItemProvider *pProvider = pTgtOpH->targetWindow()->itemProvider();

  /*----------------------------------------------------------------------
    Allow the provider to perform any necessary processing.
  ----------------------------------------------------------------------*/
  pProvider->provideLeaveSupport(event);

  /*----------------------------------------------------------------------
    Remove all items from the target operation's collection ...

    Note:  The dragleave flag is to prevent us from issuing a
           DrgDeleteDraginfoStrHandles when the dtor for
           IDMTargetOperation is called ...
  ----------------------------------------------------------------------*/
  pTgtOpH->bNotDragLeave = false;
  unsigned count = pTgtOpH->numberOfItems();
  ITRACE_DEVELOP( "Remove " + IString(count) + "items" );

  for (unsigned i = count; i; i--)
    pTgtOpH->removeItem(i);

  /*----------------------------------------------------------------------
    Reset the static target operation pointer, which causes the
    destructor for IDMTargetOperation to be called ...
  ----------------------------------------------------------------------*/
  IDMTargetOperation::setTargetOperation(0);

  /*----------------------------------------------------------------------
    Set the result and return ...
  ----------------------------------------------------------------------*/
  event.setResult(0);
  return(true);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::targetDrop                                                 |
|                                                                              |
| Called when the dragged objects are dropped on the target window.            |
------------------------------------------------------------------------------*/
bool IDMTargetHandler::targetDrop(IDMTargetDropEvent& event, bool bCnr)
{
  IMODTRACE_DEVELOP("IDMTargetHandler::targetDrop");

  /*------------------------------------------------------------------------
    Get target operation object for this target drop event and number
    of drag items to process

    Note:  The drop flag is to prevent us from issuing a DrgFreeDraginfo
           in the IDMTargetOperation dtor unless its because of a
           drop event ...
  ------------------------------------------------------------------------*/
  IDMTargetOperation::Handle pTgtOpH(IDMTargetOperation::targetOperation()) ;
  pTgtOpH->bDrop = true;
  unsigned count = pTgtOpH->numberOfItems();

  ITRACE_DEVELOP( "The number of items is " + IString( count ) );
  /*------------------------------------------------------------------------
         Set the target container to the no refresh state to prevent updates
         until the drop processing has completed
  ------------------------------------------------------------------------*/
  if (bCnr)
  {
    IContainerControl* pCnr = (IContainerControl *)pTgtOpH->targetWindow() ;
    if (pCnr->isRefreshOn())
       pTgtOpH->setContainerRefreshOff(pCnr);
  }

  /*------------------------------------------------------------------------
         Loop thru all the items defined per the drag target operation
  ------------------------------------------------------------------------*/
  for ( unsigned i = count; i; i-- )
  {
    /*----------------------------------------------------------------------
        Get the target item ...
      ----------------------------------------------------------------------*/
    IDMItem::Handle pTgtDIH(pTgtOpH->item(i)) ;

    /*----------------------------------------------------------------------
       Use the target renderer that was selected during target enter
       processing to render the target item.
      ----------------------------------------------------------------------*/
    renderer(pTgtDIH->renderer())->targetRender(event, pTgtDIH);
  }

  if (bCnr)
    pTgtOpH->setContainerRefreshOn() ;

  /*----------------------------------------------------------------------
         Set the result and return ...
  ----------------------------------------------------------------------*/
  event.setResult(0);
  return(true);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::targetHelp                                                 |
|                                                                              |
| Called when the user requests help while dragging an object over the         |
| target window                                                                |
------------------------------------------------------------------------------*/
bool IDMTargetHandler :: targetHelp(IDMTargetHelpEvent& event)
{
  IMODTRACE_DEVELOP("IDMTargetHandler::targetHelp");

  /*----------------------------------------------------------------------
    Get the item provider for the window, and give it a shot at
    providing help.
  ----------------------------------------------------------------------*/
  IDMItemProvider *pProvider;
  if (event.eventId() == DM_DROPHELP)
    pProvider = event.window()->itemProvider();
#ifdef IC_MOTIFPM 
  else
  {
    /*-------------------------------------------------------------------
       Verify that this is the correct message and notification code
     -------------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            event.parameter1().number2() == CN_DROPHELP);

    pProvider = event.controlWindow()->itemProvider();
  }
#endif // IC_MOTIFPM
  /*----------------------------------------------------------------------
    Allow the provider to perform any necessary processing.
  ----------------------------------------------------------------------*/
  pProvider->provideHelpFor(event);

  /*----------------------------------------------------------------------
    Set the result and return ...
  ----------------------------------------------------------------------*/
  event.setResult(0);
  return(true);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::targetEnd                                                  |
|                                                                              |
| Called when the source renderer has completed the rendering of a specific    |
| dragged object, and it posts this event.  Not used by the default renderers. |
------------------------------------------------------------------------------*/
bool IDMTargetHandler :: targetEnd(IDMTargetEndEvent& event)
{
  IMODTRACE_DEVELOP("IDMTargetHandler::targetEnd") ;

#ifdef IC_PMWIN 
  /*------------------------------------------------------------------------
    Puts into target item repository to end the transfer
  ------------------------------------------------------------------------*/
  renderer(event.dragItem()->renderer())->targetRenderComplete(event);

  /*----------------------------------------------------------------------
    Set the result and return ...
  ----------------------------------------------------------------------*/
  event.setResult(0);
#endif // IC_PMWIN
  return(true);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler :: findRenderersFor                                         |
|                                                                              |
| Called to find the appropriate source renderers for the operation.           |
------------------------------------------------------------------------------*/
bool IDMTargetHandler ::
        findRenderersFor(const IDMTargetOperation::Handle& pTgtOpH)
{
  DropIndicator dropFlag,
                drpflgActual;
  unsigned      position;
  bool          bFoundRenderer(true) ;

  /*----------------------------------------------------------------------
         Obtain the count of elements in the set
  ----------------------------------------------------------------------*/
  unsigned count = pTgtOpH->numberOfItems();

  /*----------------------------------------------------------------------
    Loop thru all the items defined per the drag operation
  ----------------------------------------------------------------------*/
  for (unsigned long i = 1; (i <= count); i++)
  {
    IDMItem::Handle pItemH(pTgtOpH->item(i)) ;

    if ((position = findRendererFor(pItemH)))
      pItemH->setRenderer((unsigned)position);
    else
       bFoundRenderer = false ;
  } // for

  return(bFoundRenderer);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::findRendererFor                                            |
|                                                                              |
| Called to find the appropriate source renderer for the item.  Returns the    |
| position of the renderer, 0 if none found.                                   |
------------------------------------------------------------------------------*/
unsigned IDMTargetHandler :: findRendererFor(const IDMItem::Handle& pTgtDIH)
{
  DropIndicator dropFlag;

  /*------------------------------------------------------------------------
    Obtain the count of elements in the renderer sequence
  ------------------------------------------------------------------------*/
  unsigned long ulRendererCnt = numberOfRenderers();

  /*------------------------------------------------------------------------
    Start looping through all the IDMTargetRenderers that are
    attatched to this target handler.
  ------------------------------------------------------------------------*/
  unsigned long pos(0) ;
  for (pos = 1; (pos <= ulRendererCnt); pos++)
  {
    /*-------------------------------------------------------------------
                 Get the target renderer object
    --------------------------------------------------------------------*/
    IDMTargetRenderer *pTgtRndr = renderer(pos);

    /*-------------------------------------------------------------------
       Call the renderer to see if it can render the drag item
    --------------------------------------------------------------------*/
    dropFlag = pTgtRndr->canRender(pTgtDIH);

    /*-------------------------------------------------------------------
       Weight the drop status:
         IDM::neverOk > IDM::operationNotOk > IDM::noOk ...

       Unless of course its IDM::ok in which case we bypass the
                 weighting.
    --------------------------------------------------------------------*/
    if (dropFlag != IDM::ok)
    {
      switch (pTgtDIH->dropStatus())
      {
        case IDM::neverOk:
          dropFlag = IDM::neverOk;
          break;

        case IDM::operationNotOk:
          if (dropFlag != IDM::neverOk)
            dropFlag = IDM::operationNotOk;
          break;

        case IDM::notOk:
          if ((dropFlag != IDM::operationNotOk)  &&
              (dropFlag != IDM::neverOk))
          {
            dropFlag = IDM::notOk;
          }
          break;

        default:
          ITRACE_DEVELOP("Error - bad drop status");
          break;
      }
    }

    /*-------------------------------------------------------------------
       Store the status in the item ...
    --------------------------------------------------------------------*/
    pTgtDIH->setDropStatus(dropFlag);

    /*-------------------------------------------------------------------
       If we've found a renderer, proceed to verify the operation ...
    --------------------------------------------------------------------*/
    if (dropFlag == IDM::ok)
    {
      /*-----------------------------------------------------------------
         Now verify that the current operation is supported -
         If not store status in the item ...
                ------------------------------------------------------------------*/
      if (!pTgtRndr->supportsOperation(pTgtDIH))
        pTgtDIH->setDropStatus(IDM::operationNotOk);
      else
      {
        /*-----------------------------------------------------------------
           We can support the operation (i.e. move or copy) so return
           the renderer's position ...
        ------------------------------------------------------------------*/
        return(pos);
      }
    }
  } /* end for */

  return(0);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::numberOfRenderers                                          |
|                                                                              |
| Returns the number of renderers.                                             |
------------------------------------------------------------------------------*/
unsigned IDMTargetHandler :: numberOfRenderers ()
{
  return (Inherited::numberOfRenderers());
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::renderer                                                   |
|                                                                              |
| Returns the target renderer with the given position.                         |
------------------------------------------------------------------------------*/
IDMTargetRenderer* IDMTargetHandler :: renderer (unsigned position)
{
  return((IDMTargetRenderer *)Inherited::renderer(position));
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::replaceRenderer                                            |
|                                                                              |
|  Replaces a given IDMTargetRenderer with another.                            |
------------------------------------------------------------------------------*/
IDMTargetHandler& IDMTargetHandler :: replaceRenderer (unsigned position,
                                      const IDMTargetRenderer& replacement)
{
  Inherited::replaceRenderer(position, replacement);
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::addRenderer                                                |
|                                                                              |
| Adds another IDMTargetRenderer to this target operation.                     |
------------------------------------------------------------------------------*/
IDMTargetHandler& IDMTargetHandler ::
                  addRenderer (const IDMTargetRenderer& newRenderer)
{
  Inherited::addRenderer(newRenderer);
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler :: removeRenderer                                           |
|                                                                              |
| Removes an IDMTargetRenderer from this target operation.                     |
------------------------------------------------------------------------------*/
IDMTargetHandler& IDMTargetHandler ::
                  removeRenderer (const IDMTargetRenderer& rendererToRemove)
{
  Inherited::removeRenderer(rendererToRemove);
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::setRenderer                                                |
|                                                                              |
| Sets the renderer for this target operation.  Removes any existing renderers.|
------------------------------------------------------------------------------*/
IDMTargetHandler& IDMTargetHandler ::
                  setRenderer (const IDMTargetRenderer& newRenderer)
{
  Inherited::setRenderer(newRenderer);
  return (*this);
}

/**************************************************************************/
/* Note:  The following functions are here to prevent compiler warnings   */
/*                                                                        */
/**************************************************************************/

/*------------------------------------------------------------------------------
| IDMTargetHandler::replaceRenderer                                            |
------------------------------------------------------------------------------*/
IDMHandler& IDMTargetHandler :: replaceRenderer (unsigned position,
                                           const IDMRenderer& replacement)
{
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::addRenderer                                                |
------------------------------------------------------------------------------*/
IDMHandler& IDMTargetHandler :: addRenderer (const IDMRenderer& newRenderer)
{
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::removeRenderer                                             |
------------------------------------------------------------------------------*/
IDMHandler& IDMTargetHandler :: removeRenderer (unsigned position)
{
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::removeRenderer                                             |
------------------------------------------------------------------------------*/
IDMHandler& IDMTargetHandler ::
            removeRenderer (const IDMRenderer& rendererToRemove)
{
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMTargetHandler::setRenderer                                                |
------------------------------------------------------------------------------*/
IDMHandler& IDMTargetHandler :: setRenderer (const IDMRenderer& newRenderer)
{
  return(*this);
}

