// Revision: 90 1.17.1.7 source/ui/drag/idmoper.cpp, dragdrop, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: idmoper.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IDMOperation                                                             *
*       IDMSourceOperation                                                     *
*       IDMTargetOperation                                                     *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C" {
  #define IUSING_OLE
  #ifdef WIN32_LEAN_AND_MEAN
    #undef WIN32_LEAN_AND_MEAN
  #endif
  #ifdef _OLE2_H_
    #undef _OLE2_H_
  #endif
  /* define what subsections of PM headers to include */
  #define INCL_WINPOINTERS
  #define INCL_WINSYS
  #define INCL_WININPUT
  #define INCL_GPIBITMAPS
  #define INCL_WINSTDDRAG       // Standard drag stuff
  #define INCL_WINSTDCNR        // Standard container stuff
  #include <iwindefs.h>
#ifdef IC_WIN
  #include <shlobj.h>
#endif
#ifdef IC_MOTIF  
extern "C" {
  #include <Xm/Xm.h>
  #include <Xm/DragDrop.h>
  #include <Xm/ScrolledW.h>
  #include <cnr.h>
  #include <icong.h>
}
#endif
}

#ifdef IC_MOTIF  
  #include <ibmpstat.hpp>
  #include <idmmotf.hpp>
#endif

#include <iexcept.hpp>
#include <icconst.h>
#include <itrace.hpp>
#include <ipoint.hpp>
#include <idmsrcop.hpp>
#include <idmtgtop.hpp>
#include <idmitem.hpp>
#include <idmprov.hpp>
#include <idmevent.hpp>
#include <idmcomm.hpp>
#include <ictlevt.hpp>
#include <icnrctl.hpp>
#include <icnrobj.hpp>
#include <icnrrec.hpp>
#include <iwindow.hpp>
#include <ihandle.hpp>
#include <idmimage.hpp>
#include <idmseq.hpp>

#include <iimage.hpp>

#include <icoordsy.hpp>

#ifdef IC_WIN
   #include <idmdatob.hpp>
   #include <idmdrsrc.hpp>
   #include <idmfmten.hpp>
   #include <idmformt.hpp>
#endif // IC_WIN


#ifdef IC_MOTIFWIN 
   #include <idmdnfo.hpp>
   #include <idmditm.hpp>
   #include <iplatfrm.hpp>
#endif // IC_MOTIFWIN

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IDMOPER_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_WIN

#pragma enum(4)
#pragma pack(push,4)

class IDMSourceOperationData {
public:
  IDMSourceOperationData ( )
   : imageListFlags( ILC_COLOR | ILC_MASK )
  { }
 ~IDMSourceOperationData ( )
  { }

void
  doDrag ( IDMSourceOperation *srcOper );

private:
IStream
 *marshall ( IDragInfoWrapper* pDInfoWrapper );

unsigned                                     // Result: number of images used
  stackImages( IDMImage& anImage,            // Output: image of stack
               IPoint& stackHotspot,         // Output: size of member icons
               IDMSourceOperation* pSrcOp ); // Input: source drag operation

unsigned                                   // Result: number of images used
  stackImagesPass1
            ( IDMImage& anImage,           // I/O: image of stack
              IDMSourceOperation* pSrcOp,  // Input: source drag operation
              const ISize& offset,         // Input: stagger offset in pixels
              HIMAGELIST& himagelist,      // Output: list of stackable images
              IRectangle& rect,            // Output: bounding rect of stack
              IPoint& hotspot );           // Output: working hotspot

unsigned                                   // Result: number of images used
  stackImagesPass2
            ( IDMImage& anImage,           // Output: image of stack
              const ISize& offset,         // Input: stagger offet in pixels
              const HIMAGELIST himagelist, // Input: list of images
              const IRectangle& rect);     // Input: stack image bounds

const unsigned
  imageListFlags;
}; // IDMSourceOperationData


class IDMTargetOperationData {
public:
  IDMTargetOperationData ( )
   : pDInfo( NULL )
  { }
 ~IDMTargetOperationData ( )
  { delete pDInfo; }
void
  buildItems ( IDMTargetOperation *targetOper,
               IDMTargetEnterEvent &event );
bool
  isContainerControl( const IWindow *win );
private:
_DRAGINFO
 *pDInfo;
}; // IDMTargetOperationData

#pragma enum(pop)
#pragma pack(pop)

#endif

#define DRG_DEBUG        ((PVOID) 0x80000000L)    /* prevents screen locking */

typedef IDM::DragImageStyle DragImageStyle;
typedef IDM::DropStyle      DropStyle;

/* Initialize static */
IDMTargetOperation::Handle IDMTargetOperation::pDMTgtOpHandle(0);
bool IDMOperation::fgDebugSupport = false;
#ifdef IC_MOTIFWIN 
IDMSourceOperation::Handle IDMSourceOperation::pDMSrcOpHandle(0);
#endif

/*------------------------------------------------------------------------------
| Static operation definitions.                                                |
|                                                                              |
| These values exploit the specific values used for the PM toolkit             |
| DO_* values.  Item creation must support the operations corresponding        |
| to the DO_* values.                                                          |
------------------------------------------------------------------------------*/
const unsigned long IDMOperation::drag    (DO_DEFAULT);
const unsigned long IDMOperation::move    (DO_MOVE);
const unsigned long IDMOperation::copy    (DO_COPY);
const unsigned long IDMOperation::link    (DO_LINK);
const unsigned long IDMOperation::unknown (DO_UNKNOWN);

/*------------------------------------------------------------------------------
| IDMOperation::IDMOperation                                                   |
|                                                                              |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
IDMOperation :: IDMOperation(IDMSourceBeginEvent &event)
  : src(event.source())
  , pos(event.position())
  , op(drag)
  , srcWindowHandle(0)
  , tgtWindowHandle(0)
  , pCnrObj(0)
  , pCnrRefresh(0)
#ifdef IC_PM
  , pPMDragInfo(0)
#endif
// ifdef IC_MOTIF  
//   _dragContext(0),
// endif
  , bDragResult(true)
  , pDMItemSeqCl(0)
{
   IFUNCTRACE_DEVELOP();
// ifdef IC_MOTIF  
//     if ( ( event.eventType() == IEvent::DragDropCallback) &&
//          ( event.eventId() == XmCR_DRAG ) )
//     {
//     // The callback is defined by our StartDrag routine. The only valid field
//     // are dragContext, event and reason.
//     XmDragProcCallback dpcb =
//       ( XmDragProcCallback ) event.parameter1().asUnsignedLong();
//
//     _dragContext = dpcb->dragContext;
//     }
// endif
}

/*------------------------------------------------------------------------------
| IDMOperation::IDMOperation                                                   |
|                                                                              |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
IDMOperation :: IDMOperation(IDMTargetEnterEvent &event)
  : src(IDM::pointingDevice),
    pos(IPoint(0,0)),
    op(drag),
    srcWindowHandle(0),
    tgtWindowHandle(0),
    pCnrObj(0),
    pCnrRefresh(0),
#ifdef IC_PM
    pPMDragInfo(0),
#endif
    bDragResult(true),
    pDMItemSeqCl(0)
{
   IFUNCTRACE_DEVELOP();
}

/*------------------------------------------------------------------------------
| IDMOperation::~IDMOperation                                                  |
|                                                                              |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
IDMOperation :: ~IDMOperation ( )
{
  IFUNCTRACE_DEVELOP();
  delete this->pDMItemSeqCl;
}

/*------------------------------------------------------------------------------
| IDMOperation::operation                                                      |
|                                                                              |
| Returns the current direct manipulation operation.                           |
------------------------------------------------------------------------------*/
unsigned long IDMOperation :: operation () const
{
  return (this->op);
}

/*------------------------------------------------------------------------------
| IDMOperation::setOperation                                                   |
|                                                                              |
| Sets the operation of the Direct Manipulation operation.                     |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setOperation (unsigned long Operation)
{
  this->op = Operation;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::position                                                       |
|                                                                              |
| Returns the drag or drop position within the source or target window.        |
------------------------------------------------------------------------------*/
IPoint IDMOperation :: position () const
{
  return (this->pos);
}

/*------------------------------------------------------------------------------
| IDMOperation::source                                                         |
|                                                                              |
| Returns the source of the direct manipulation operation                      |
| (i.e. always pointing device)                                                |
------------------------------------------------------------------------------*/
IDM::Source IDMOperation :: source() const
{
  return (this->src);
}

/*------------------------------------------------------------------------------
| IDMOperation::sourceWindowHandle                                             |
|                                                                              |
| Returns the handle of the source window for the direct manipulation          |
| operation.                                                                   |
------------------------------------------------------------------------------*/
IWindowHandle IDMOperation :: sourceWindowHandle () const
{
  return (this->srcWindowHandle);
}

/*------------------------------------------------------------------------------
| IDMOperation::targetWindowHandle                                             |
|                                                                              |
| Returns the handle of the target window for the direct manipulation          |
| operation.                                                                   |
------------------------------------------------------------------------------*/
IWindowHandle IDMOperation :: targetWindowHandle () const
{
  return (this->tgtWindowHandle);
}

/*------------------------------------------------------------------------------
| IDMOperation::sourceWindow                                                   |
|                                                                              |
| Returns a pointer to the source window for the direct manipulation           |
| operation.                                                                   |
------------------------------------------------------------------------------*/
IWindow* IDMOperation :: sourceWindow () const
{
  return (IWindow::windowWithHandle(this->srcWindowHandle));
}

/*------------------------------------------------------------------------------
| IDMOperation::targetWindow                                                   |
|                                                                              |
| Returns a pointer to the target window for the direct manipulation           |
| operation.                                                                   |
------------------------------------------------------------------------------*/
IWindow* IDMOperation :: targetWindow () const
{
  return (IWindow::windowWithHandle(this->tgtWindowHandle));
}

/*------------------------------------------------------------------------------
| IDMOperation::numberOfItems                                                  |
|                                                                              |
| Returns the number of items in the collection.                               |
------------------------------------------------------------------------------*/
unsigned IDMOperation :: numberOfItems ()
{
  return unsigned(itemCollection()->numberOfElements());
}

/*------------------------------------------------------------------------------
| IDMOperation::item                                                           |
|                                                                              |
| Returns the drag item with a given index.  0 is returned if the index is     |
| invalid.                                                                     |
------------------------------------------------------------------------------*/
IDMItem::Handle IDMOperation :: item(unsigned index)
{
  /*------------------------------------------------------------------------
    Get number of items
  ------------------------------------------------------------------------*/
  unsigned count = numberOfItems();

  /*------------------------------------------------------------------------
    Verify index
  ------------------------------------------------------------------------*/
  if ((index < 1)  ||  (index > count))
  {
  //Throw exception
  ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
      IBaseErrorInfo::invalidRequest,
      IException::recoverable);
  return(0);
  }

  /*------------------------------------------------------------------------
     Return item handle at index
  ------------------------------------------------------------------------*/
  return(itemCollection()->elementAtPosition(index));
}

/*------------------------------------------------------------------------------
| IDMOperation::replaceItem                                                    |
|                                                                              |
| Replace a given drag item with another.                                      |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: replaceItem (unsigned index,
                                           const IDMItem::Handle &pDI)
{
  /*------------------------------------------------------------------------
     Get number of items
  ------------------------------------------------------------------------*/
  unsigned count = numberOfItems();

  /*------------------------------------------------------------------------
    Verify index
  ------------------------------------------------------------------------*/
  if ((index < 1)  ||  (index > count))
  {
    //Throw exception
    ITHROWLIBRARYERROR(IC_INDEX_OUT_OF_RANGE,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable);
    return(*this);
  }

  /*------------------------------------------------------------------------
    Replace the item
  ------------------------------------------------------------------------*/
  removeItem(index);
  itemCollection()->addAtPosition(index, pDI);

  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::addItem                                                        |
|                                                                              |
| Associate another drag item with this operation.                             |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: addItem (const IDMItem::Handle &pDI)
{
  IMODTRACE_DEVELOP( "IDMOperation::addItem" );

  /*-------------------------------------------------------------------
     Store item in collection and bump the operation's use count.
   -------------------------------------------------------------------*/
  itemCollection()->add(pDI);
  ITRACE_DEVELOP( "IDMOperation::addItem - addRef" );
  this->addRef();
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::removeItem                                                     |
|                                                                              |
| Remove a drag item from the operation.                                       |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: removeItem (unsigned index)
{
  /*------------------------------------------------------------------------
    Define a cursor for the drag item sequence and set to the proper
    position.
  ------------------------------------------------------------------------*/
  IDMItemSeq::Cursor cursor(*(itemCollection()));
  itemCollection()->setToPosition(index, cursor);

  /*-----------------------------------------------------------------------
      Set the item handle to 0 and remove item from the collection
   -----------------------------------------------------------------------*/
  itemCollection()->elementAt(cursor) = 0;
  itemCollection()->removeAtPosition(index);
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::removeItem                                                     |
|                                                                              |
| Remove a drag item from the operation.                                       |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: removeItem (const IDMItem::Handle &pDI)
{
  /*------------------------------------------------------------------------
    Get number of items
  ------------------------------------------------------------------------*/
  unsigned count = numberOfItems();

  /*------------------------------------------------------------------------
    Loop thru the collection and remove item if a match is found
  ------------------------------------------------------------------------*/
  for (unsigned index = 1; (index <= count); index++)
  {
    if ((itemCollection()->elementAtPosition(index)) == pDI)
    {
      removeItem(index);
      break;
    }
  }

  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::containerObject                                                |
|                                                                              |
| Returns a container object.                                                  |
------------------------------------------------------------------------------*/
IContainerObject* IDMOperation :: containerObject() const
{
  return (this->pCnrObj);
}

/*------------------------------------------------------------------------------
| IDMOperation::setContainerObject                                             |
|                                                                              |
| Sets a pointer to a container object.                                        |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setContainerObject(IContainerObject *object)
{
  this->pCnrObj = object;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::setContainerRefreshOff                                         |
|                                                                              |
| Disables the container refreshes, will be re-enable it after processing.     |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setContainerRefreshOff(IContainerControl* pCnr)
{
  IMODTRACE_DEVELOP("IDMOperation::setContainerRefreshOff");

  this->pCnrRefresh = pCnr;
  pCnr->setRefreshOff();

  return(*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::setContainerRefreshOn                                          |
|                                                                              |
| Re-enables container refreshes.                                              |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setContainerRefreshOn()
{
  IMODTRACE_DEVELOP("IDMOperation::setContainerRefreshOn");

  if (this->pCnrRefresh)
  {
    if (targetWindowHandle())
    this->pCnrRefresh->setRefreshOn().refresh(true);
#ifdef IC_PM
    else
    {
      DRAGINFO*  pPMDragInfo = dragInfo();
      if (!pPMDragInfo ||
         ((unsigned long)pPMDragInfo->usOperation == IDMOperation::copy) ||
         ((unsigned long)pPMDragInfo->usOperation == IDMOperation::link) ||
          dragWasInterrupted())
      {
        this->pCnrRefresh->setRefreshOn();
      }
      else
      {
        this->pCnrRefresh->setRefreshOn().refresh();
      }
    }
#endif
#ifdef IC_MOTIFWIN 
    else
    {
      _DRAGINFO*  pPMDragInfo = dragInfo();
      unsigned long dragOp = 0;
      if (!pPMDragInfo ||
         (dragOp=((unsigned long)IDragInfoWrapper(*pPMDragInfo).dragOperation())
          == IDMOperation::copy) ||
             (dragOp == IDMOperation::link) ||
             dragWasInterrupted())
      {
        this->pCnrRefresh->setRefreshOn();
      }
      else
      {
        this->pCnrRefresh->setRefreshOn().refresh(true);
      }
    }
#endif
  }

  return(*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::setSourceWindowHandle                                          |
|                                                                              |
| Sets the source window of the direct manipulation operation during a source  |
| begin event.                                                                 |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation ::
              setSourceWindowHandle (IWindowHandle srcWindow)
{
  this->srcWindowHandle = srcWindow;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::setTargetWindowHandle                                          |
|                                                                              |
| Sets the target window of the direct manipulation operation during a target  |
| enter event.                                                                 |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation ::
              setTargetWindowHandle (IWindowHandle tgtWindow)
{
  this->tgtWindowHandle = tgtWindow;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::setSource                                                      |
|                                                                              |
| Sets the source of the Direct Manipulation operation.                        |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setSource (Source source)
{
  this->src = source;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::setPosition                                                    |
|                                                                              |
| Sets the pointing device position for the Direct Manipulation operation.     |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setPosition (IPoint position)
{
  this->pos = position;
  return (*this);
}

// #if defined(IC_PM) || defined(IC_WIN)
/*------------------------------------------------------------------------------
| IDMOperation::setDragInfo                                                    |
|                                                                              |
| Sets the pointer to the PM DRAGINFO structure.                               |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setDragInfo (_DRAGINFO *pDragInfo)
{
  this->pPMDragInfo = pDragInfo;
  return (*this);
}
// #endif

/*------------------------------------------------------------------------------
| IDMOperation::setDragResult                                                  |
|                                                                              |
| Sets the result of the drag operation.                                       |
------------------------------------------------------------------------------*/
IDMOperation& IDMOperation :: setDragResult (bool result)
{
  this->bDragResult = result;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMOperation::itemCollection                                                 |
|                                                                              |
| Returns a pointer to the source collection of IDMItem::Handles.              |
| Notes:                                                                       |
|   1) Private                                                                 |
------------------------------------------------------------------------------*/
IDMItemSeq* IDMOperation :: itemCollection()
{
  if (!this->pDMItemSeqCl)
    this->pDMItemSeqCl = new IDMItemSeq;

  return (this->pDMItemSeqCl);
}


/*------------------------------ Source Operation ----------------------------*/


/*------------------------------------------------------------------------------
| IDMSourceOperation::IDMSourceOperation                                       |
|                                                                              |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
IDMSourceOperation :: IDMSourceOperation(IDMSourceBeginEvent &event,
                                         DragImageStyle dIStyle)
  : IDMOperation(event),
    drgImgStyle(dIStyle),
    totalImgOffset(0,0),
    ptrOffset(0,0),
    stackingPct(IPair(50,50))
#ifdef IC_WIN
  , fDMSourceOperationData( new IDMSourceOperationData() )
#endif
{
  IFUNCTRACE_DEVELOP();

// ifdef IC_MOTIF 
//   // Need to check this reason code.
//   if (event.eventId() == XmCR_DRAG /* WM_BEGINDRAG */)
//     setSourceWindowHandle(event.handle());
// endif
// ifdef IC_PM
  if (event.eventId() == WM_BEGINDRAG)
    setSourceWindowHandle(event.handle());
#ifdef IC_MOTIFPM 
  else
  {
    /*-----------------------------------------------------------------
       Verify that this is the correct message and notification code
     -----------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            event.parameter1().number2() == CN_INITDRAG);

#if 0                                                    // $HT!a
    IControlEvent ctlevt(event);
    setSourceWindowHandle(ctlevt.controlWindow()->handle()); 
#endif // 0                                                 $HT!a
    this->setSourceWindowHandle( event.controlWindow()->handle() ); // $HT!a
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMSourceOperation:: ~IDMSourceOperation                                     |
|                                                                              |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
IDMSourceOperation :: ~IDMSourceOperation ()
{
  IFUNCTRACE_DEVELOP();

  /*-------------------------------------------------------------------
    Enable container refresh (this function can detect whether or
    not we're processing a container) ...
   -------------------------------------------------------------------*/
  setContainerRefreshOn();

#ifdef IC_PM
  DRAGINFO*  pPMDragInfo = dragInfo();
  DrgFreeDraginfo (pPMDragInfo);
#endif

#ifdef IC_MOTIFWIN
  _DRAGINFO*  pPMDragInfo = dragInfo();
  IDragInfoWrapper dInfoWrapper( *pPMDragInfo );
  dInfoWrapper.deleteSourceResources();

#ifdef IC_WIN 
  if (fDMSourceOperationData )
    delete fDMSourceOperationData;
#endif // IC_WIN
#endif

}

/*------------------------------------------------------------------------------
| IDMSourceOperation::begin                                                    |
|                                                                              |
| Begin the drag operation.                                                    |
------------------------------------------------------------------------------*/
IDMSourceOperation& IDMSourceOperation::begin ()
{
  IMODTRACE_DEVELOP("IDMSourceOperation::begin");

#ifdef IC_MOTIFPM  
  _DRAGINFO       *pDInfo;
#ifdef IC_PM  
  _DRAGIMAGE      DragImage;
  _DRAGIMAGE      *pDrgImg;
  _DRAGIMAGE      *pDragImage = 0;
#endif
  bool         done = false;
  unsigned long   arraySize = 0;
  unsigned short  numberOfImagesFound = 0;
  unsigned long   ulItemCnt = numberOfItems();
  DragImageStyle  diStyle   = imageStyle();

  /*-------------------------------------------------------------------
     Make sure we have some items to drag...
   -------------------------------------------------------------------*/
  if (ulItemCnt)
  {
#ifdef IC_PM  
    pDInfo = DrgAllocDraginfo (ulItemCnt);
    if (!pDInfo)
      ITHROWGUIERROR("DrgAllocDraginfo");
#endif // IC_PM
#ifdef IC_MOTIF 
    pDInfo = new IDragInfo ( (unsigned short)ulItemCnt );
#endif

    /*-------------------------------------------------------------------
       Set the window handle of the source and the drag operation
       in DRAGINFO.   All other fields in DRAGINFO are initialized
       by PM.
     -------------------------------------------------------------------*/
#ifdef IC_PM 
    pDInfo->hwndSource = sourceWindowHandle();
    pDInfo->usOperation = operation();
#endif
#ifdef IC_MOTIF 
    IDragInfoWrapper dInfoWrapper(*pDInfo);
    dInfoWrapper.setSourceWindowHandle( sourceWindowHandle() );
#endif

    /*-------------------------------------------------------------------
       Set up the drag image array if needed.
     -------------------------------------------------------------------*/
    if (diStyle == IDM::stack3AndFade)
    {
      arraySize = 4;
    }
    else if (diStyle == IDM::allStacked)
    {
      arraySize = ulItemCnt;
    }

    if (arraySize)
    {
#ifdef IC_PM 
      pDrgImg = pDragImage = new _DRAGIMAGE[arraySize];
      IASSERT(pDrgImg!=0);
#endif // IC_PM
    }

    /*-----------------------------------------------------------------
       Store "handle" to our drag information
     -----------------------------------------------------------------*/
    setDragInfo (pDInfo);

    /*-----------------------------------------------------------------
       Loop through the collection of IDMItems.  For each, copy
       the corresponding data, and set each one into the PM DragInfo
       structure.  Also, build the array of drag images to be displayed.
     -----------------------------------------------------------------*/
    for (unsigned long ulItem = 0; (ulItem < ulItemCnt); ulItem++)
    {
      IDMItem::Handle pItemHandle;
      pItemHandle = item((ulItem+1));

#ifdef IC_PM 
      _DRAGITEM      PMDragItem;

      /*---------------------------------------------------------------
         Create a PM DragItem from the corresponding IDMItem.
         Then, set the PM DragItem into the PM DragInfo structure.
       ---------------------------------------------------------------*/
      pItemHandle->asPMDragItem(&PMDragItem);
      if (!DrgSetDragitem (pDInfo, &PMDragItem, sizeof(_DRAGITEM), ulItem))
        ITHROWGUIERROR("DrgSetDragitem");

      /*---------------------------------------------------------------
         Query the PM DragItem pointer from shared memory, and set
         its value in the corresponding ICLUI drag item.
       ---------------------------------------------------------------*/
      _DRAGITEM* pdi = DrgQueryDragitemPtr (pDInfo, ulItem);
      if (pdi)
      {
        pItemHandle->setPMDragItem(pdi);
      }
      else
      {
        ITHROWGUIERROR("DrgQueryDragitemPtr");
      }
#endif // IC_PM

#ifdef IC_MOTIF
      /*---------------------------------------------------------------
         Set the DRAGITEM in the DRAGINFO for this item.
       ---------------------------------------------------------------*/
      _DRAGITEM& dItem = dInfoWrapper[ulItem];
      pItemHandle->asPMDragItem( &dItem );

      /*---------------------------------------------------------------
         Tell the ICLUI drag item where the DRAGITEM is.
       ---------------------------------------------------------------*/
      pItemHandle->setPMDragItem( &dItem );
#endif // IC_MOTIF

      /*---------------------------------------------------------------
         Set up the dragimage for this drag item.
       ---------------------------------------------------------------*/
      if ((diStyle == IDM::allStacked) ||
          (diStyle == IDM::stack3AndFade))
      {
        IDMImage &image = pItemHandle->image();

        /*-------------------------------------------------------------
           Determine if we need to add the fade icon.
         -------------------------------------------------------------*/
        if ((diStyle == IDM::stack3AndFade) && (!done) &&
            (numberOfImagesFound == 3))
        {
          /*-----------------------------------------------------------
             Add the special fade icon if we have already found the
             3 we stack.
           -----------------------------------------------------------*/
#ifdef IC_PM 
          image.asPMDragImage(pDrgImg, false, true);
          ISize imageOffset = totalImageOffset();
          pDrgImg->cxOffset = (short)imageOffset.width();
          pDrgImg->cyOffset = (short)imageOffset.height();
#endif  // IC_PM
          done = true;
          numberOfImagesFound++;
        }
        else if (!done)
        {
#ifdef IC_PM  
          image.asPMDragImage(pDrgImg);

          /*-----------------------------------------------------------
             If we got an image, increase the count.
           -----------------------------------------------------------*/
          if (pDrgImg->hImage)
          {
            ISize imageOffset;
            ISize  ptrOffset;
            numberOfImagesFound++;
            ptrOffset = image.pointerOffset();
            if (ptrOffset == 0)
            {
              imageOffset = totalImageOffset();
              pDrgImg->cxOffset = (short)imageOffset.width();
              pDrgImg->cyOffset = (short)imageOffset.height();
                storeImageOffset (&image);
            }
            else
            {
              pDrgImg->cxOffset = (short)ptrOffset.width();
              pDrgImg->cyOffset = (short)ptrOffset.height();
            }
            pDrgImg++;
          }
#endif // IC_PM
        }
      }
    }

    /*-----------------------------------------------------------------
       Give them system icons if they ask for them, or if they did not
       supply any images.
     -----------------------------------------------------------------*/
    if ((diStyle == IDM::systemImages) || (!numberOfImagesFound))
    {
      IDMImage image = IDMImage();
#ifdef IC_PM 
      image.asPMDragImage(&DragImage, true);

      /*---------------------------------------------------------------
         If we are dragging more than 1 item, give them the multiple
         file icon.  Otherwise, give them the single file icon.
       ---------------------------------------------------------------*/
      if (ulItemCnt > 1)
      {
        DragImage.hImage = (unsigned long)*(new ISystemPointerHandle (
                           ISystemPointerHandle::multipleFile));
      }
      else
      {
        DragImage.hImage = (unsigned long)*(new ISystemPointerHandle (
                           ISystemPointerHandle::singleFile));
      }
      numberOfImagesFound = 1;
      pDragImage = &DragImage;
#endif // IC_PM
    }

#ifdef IC_PM 
    /*-----------------------------------------------------------------
       Start the drag - DrgDrag executes in a synchronous manner

       Note:  Check the debug flag in order to fully enable debug
              support.
     -----------------------------------------------------------------*/
    unsigned rc = DrgDrag (pDInfo->hwndSource,
                           pDInfo,
                           pDragImage,
                           numberOfImagesFound,
                           VK_ENDDRAG,
                           ((debugSupport()) ? DRG_DEBUG
                                                : 0));

    /*-----------------------------------------------------------------
       Delete the array if we created one.
      -----------------------------------------------------------------*/
    if ((diStyle != IDM::systemImages) && (pDragImage))
    {
      delete [] pDragImage;
    }
    ITRACE_DEVELOP("IDMSourceOperation:  DrgDrag rc = "+IString(rc));

    /*-----------------------------------------------------------------
       Remove any remaining items from the source operation's
       collection.  There will be remaining items if one of the
       following conditions occurs:
          - the operation was cancelled
          - help was requested
          - the drop was not allowed
        All of these conditions return NULL
     -----------------------------------------------------------------*/
    if (!rc)
    {
      setDragResult( false );
      unsigned count = numberOfItems();
      for (unsigned i = count; i; i--)
        removeItem(i);
    }
#endif // IC_PM
  } /* end if */
// ifdef IC_MOTIF
//   /*---------------------------------------------------------------
//      Set up the dragimage for this drag item.
//   ---------------------------------------------------------------*/
//
//   IDMItem::Handle pItemHandle = item(1);
//
//   if ( (diStyle == IDM::allStacked) ||
//        (diStyle == IDM::stack3AndFade))
//     {
//     IDMImage &image = pItemHandle->image();
//
//     unsigned int  maxCursorWidth, maxCursorHeight;
//     XQueryBestCursor( XtDisplay(_dragContext),
//                      RootWindowOfScreen(XtScreen(_dragContext)), 64, 64,
//                      &maxCursorWidth, &maxCursorHeight);
//
//     Widget sourceIcon;
//
//     if (maxCursorWidth > 32 &&
//         maxCursorHeight > 32 )
//       {
//       maxCursorWidth = 64;
//       maxCursorHeight = 64;
//       }
//     else
//       {
//       maxCursorWidth = 32;
//       maxCursorHeight = 32;
//       }
//
//     Widget dragObject;
//     XtVaGetValues( _dragContext, XmNclientData, &dragObject, NULL );
//
//     Arg args[7];
//     int n = 0;
//
//     if ( image.pointer() )
//       {
//       XtSetArg( args[n], XmNpixmap, image.pointer() ); n++;
//       Pixmap shapeMask =
//         IBitmapStaticPtr::shapeMaskFor( image.pointer() );
//       if ( shapeMask )
//         {
//         XtSetArg( args[n], XmNmask, shapeMask );n++;
//         }
//       XtSetArg( args[n], XmNwidth, maxCursorWidth );n++;
//       XtSetArg( args[n], XmNheight, maxCursorHeight );n++;
// //      XtSetArg( args[n], XmNdepth,
// //          DefaultDepth( XtDisplay(dragObject), 0));n++;
//       }
//
//     // Not sure what to make the parent of the drag icon, will use the
//     // drag context client data for now.
// //    sourceIcon = XmCreateDragIcon( XtParent(dragObject), "Drag", args, n );
//     sourceIcon = XmCreateDragIcon( dragObject, "Drag", args, n );
//
//     XtVaSetValues( _dragContext,
//       XmNsourceCursorIcon, sourceIcon,
//       NULL );
//     }
// endif
//@@Rob.  Copied from Jim.
#endif // IC_MOTIFPM

#ifdef IC_WIN
  /*-------------------------------------------------------------------
     Start the drag (Windows)
   -------------------------------------------------------------------*/
  fDMSourceOperationData->doDrag(this);
#endif

  return(*this);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::imageStyle                                               |
|                                                                              |
| Returns the source operation drag image style.                               |
------------------------------------------------------------------------------*/
DragImageStyle IDMSourceOperation::imageStyle () const
{
  return (this->drgImgStyle);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::pointerOffset                                            |
|                                                                              |
| Returns the position of the pointing device relative to the source container |
| object's origin.                                                             |
------------------------------------------------------------------------------*/
ISize IDMSourceOperation::pointerOffset () const
{
  return(this->ptrOffset);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::operation                                                |
|                                                                              |
| Returns the current direct manipulation operation.                           |
------------------------------------------------------------------------------*/
unsigned long IDMSourceOperation :: operation () const
{
#ifdef IC_WIN
  /*-------------------------------------------------------------------
     This is where we add the operation support to determine
     copy, move, link, etc...
   -------------------------------------------------------------------*/
  return( Inherited::operation() );
#endif

#ifdef IC_PM
  _DRAGINFO* di = dragInfo();
  if (di)
  {
    return((unsigned long)di->usOperation);
  }
  else
  {
    return( Inherited::operation() );
  }
#endif

#ifdef IC_MOTIF   
  _DRAGINFO* di = dragInfo();
  if (di)
  {
    return (unsigned long)IDragInfoWrapper(*di).dragOperation();
  }
  else
  {
    return IDMOperation::drag;
  }
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::stackingPercentage                                       |
|                                                                              |
| Returns the stacking percentage.                                             |
------------------------------------------------------------------------------*/
IPair IDMSourceOperation::stackingPercentage () const
{
  return (this->stackingPct);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::setImageStyle                                            |
|                                                                              |
| Sets the source operation drag image style.                                  |
------------------------------------------------------------------------------*/
IDMSourceOperation& IDMSourceOperation::setImageStyle(DragImageStyle dIStyle)
{
  this->drgImgStyle = dIStyle;
  return (*this);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::setPointerOffset                                         |
|                                                                              |
| Sets the position of the pointing device relative to the source container    |
| object's origin.                                                             |
------------------------------------------------------------------------------*/
IDMSourceOperation& IDMSourceOperation::setPointerOffset (const ISize &pOff)
{
  this->ptrOffset = pOff;
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::setStackingPercentage                                    |
|                                                                              |
| Sets the stacking percentage.                                                |
------------------------------------------------------------------------------*/
IDMSourceOperation& IDMSourceOperation::
                    setStackingPercentage (const IPair& stackingPercent)
{
  this->stackingPct = stackingPercent;
  return (*this);
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IDMSourceOperation::operationFrom                                            |
|                                                                              |
| Retrieves a handle to the source drag operation from the drag information    |
| structure.                                                                   |
------------------------------------------------------------------------------*/
IDMSourceOperation::Handle IDMSourceOperation ::
                           operationFrom(_DRAGINFO *pDInfo)
{
  /*-------------------------------------------------------------------
     Access the PM DRAGINFO structure which was passed to us.
   -------------------------------------------------------------------*/
  if (!DrgAccessDraginfo (pDInfo))
   ITHROWGUIERROR("DrgAccessDraginfo");

  /*-------------------------------------------------------------------
     Get count of PM drag items
   -------------------------------------------------------------------*/
  unsigned count = pDInfo->cditem;

  /*-------------------------------------------------------------------
     Loop thru PM Drag Items until we find one we can construct
     source operation form ...
   -------------------------------------------------------------------*/
  IDMSourceOperation::Handle pSrcOpH = 0;
  for (unsigned i=0; i < count; i++)
  {
    /*-----------------------------------------------------------------
       Get PM drag item
     -----------------------------------------------------------------*/
    _DRAGITEM *pPMDItem = DrgQueryDragitemPtr (pDInfo, i);

    /*-----------------------------------------------------------------
       If the pointer and ulItemID are not NULL, assume ulItemID
       contains a pointer to the source item which in turn, can
       give us the pointer to the source operation object,
       which we can construct an IDMSourceOperation::Handle from.
     -----------------------------------------------------------------*/
    if (pPMDItem)
    {
      if (pPMDItem->ulItemID)
      {
         IDMItem *pItem = (IDMItem *) pPMDItem->ulItemID; 
         if (pItem)
         {
            pSrcOpH = pItem->sourceOperation();
         }
        break;
      }
    }
  }

  return(pSrcOpH);
}
#endif // IC_PM

#ifdef IC_MOTIF  
IDMSourceOperation::Handle IDMSourceOperation ::
                           operationFrom(_DRAGINFO *pDInfo)
{
  /*-------------------------------------------------------------------
     Access the PM DRAGINFO structure which was passed to us.
   -------------------------------------------------------------------*/
  IDragInfoWrapper dInfoWrapper(*pDInfo);
  dInfoWrapper.access();

  /*-------------------------------------------------------------------
     Get count of PM drag items
   -------------------------------------------------------------------*/
  unsigned count = dInfoWrapper.numDragItems();

  /*-------------------------------------------------------------------
     Loop thru PM Drag Items until we find one we can construct
     source operation from ...
   -------------------------------------------------------------------*/
  IDMSourceOperation::Handle pSrcOpH = 0;
  for (unsigned i=0; i < count; i++)
  {
    /*-----------------------------------------------------------------
       Get PM drag item
     -----------------------------------------------------------------*/
    IDragItemWrapper dItemWrapper( dInfoWrapper[i] );

    /*-----------------------------------------------------------------
       If the ulItemID is not NULL, assume ulItemID
       contains a pointer to the source item which in turn, can
       give us the pointer to the source operation object,
       which we can construct an IDMSourceOperation::Handle from.
     -----------------------------------------------------------------*/
    unsigned long itemID = dItemWrapper.itemID();
    if ( itemID )
    {
      pSrcOpH = ((IDMItem *)(itemID))->sourceOperation();
      break;
    }
  }

  return(pSrcOpH);
}
#endif // IC_MOTIF


/*------------------------------------------------------------------------------
| IDMSourceOperation::storeImageOffset                                         |
|                                                                              |
| Store the offset at which to place the next image.                           |
| Notes:                                                                       |
|   1) Private                                                                 |
------------------------------------------------------------------------------*/
void IDMSourceOperation::storeImageOffset(IDMImage *pIDMImage)
{
  IMODTRACE_DEVELOP("IDMSourceOperation::storeImageOffset");

#ifdef IC_PMWIN  
  unsigned long style = (pIDMImage->style()).asUnsignedLong();
  ISize stretchSize = pIDMImage->stretchSize();
  IPointerHandle ptr;
  IBitmapHandle  bmp;
#ifdef IC_PM
  POINTERINFO    PointerInfo;
#endif
#ifdef IC_WIN
  ICONINFO       PointerInfo;
#endif
  long           cxPtr,
                 cyPtr;

  /*-------------------------------------------------------------------
     Calculate the percentage that is used to position the next
     image in relation to the current image.  Default is (50,50)
     which means the nth image (where n=2, 3, or 4) is positioned
     with its lower left corner in the middle of the current image.
   -------------------------------------------------------------------*/
#ifdef IC_PM
  if (style & DRG_ICON)
  {
    ptr = pIDMImage->pointer();
    if ((unsigned long)ptr)
    {
      if (WinQueryPointerInfo ((HPOINTER)ptr, &PointerInfo))
      {
        if (PointerInfo.fPointer)
        {
          cxPtr = WinQuerySysValue (HWND_DESKTOP, SV_CXPOINTER);
          cyPtr = WinQuerySysValue (HWND_DESKTOP, SV_CYPOINTER);
          if ((!cxPtr) || (!cyPtr))
            ITHROWGUIERROR("WinQuerySysValue");
        }
        else
        {
          cxPtr = WinQuerySysValue (HWND_DESKTOP, SV_CXICON);
          cyPtr = WinQuerySysValue (HWND_DESKTOP, SV_CYICON);
          if ((!cxPtr) || (!cyPtr))
            ITHROWGUIERROR("WinQuerySysValue");
        }
        this->totalImgOffset.setWidth(this->totalImgOffset.width() +
                             (stackingPercentage().coord1() *
                             (cxPtr + stretchSize.width())) / 100);
        this->totalImgOffset.setHeight(this->totalImgOffset.height() +
                             (stackingPercentage().coord2() *
                             (cyPtr + stretchSize.height())) / 100);
      }
      else
        ITHROWGUIERROR("WinQueryPointerInfo");
    }
  }
  else
  {
#endif // IC_PM
#ifdef IC_WIN
  if (style & DRG_ICON)
  {
    ptr = pIDMImage->pointer();
    if ( ptr )
    {
      if (IQUERYPOINTERINFO( ptr, &PointerInfo))
      {
        if (PointerInfo.fIcon)
        {
          cxPtr = IQUERYSYSVALUE(SV_CXPOINTER);
          cyPtr = IQUERYSYSVALUE(SV_CYPOINTER);
          if ((!cxPtr) || (!cyPtr))
            ITHROWGUIERROR("IQUERYSYSVALUE");
        }
        else
        {
          cxPtr = IQUERYSYSVALUE (SV_CXICON);
          cyPtr = IQUERYSYSVALUE (SV_CYICON);
          if ((!cxPtr) || (!cyPtr))
            ITHROWGUIERROR("IQUERYSYSVALUE");
        }
        this->totalImgOffset.setWidth(this->totalImgOffset.width() +
                             (stackingPercentage().coord1() *
                             (cxPtr + stretchSize.width())) / 100);
        this->totalImgOffset.setHeight(this->totalImgOffset.height() +
                             (stackingPercentage().coord2() *
                             (cyPtr + stretchSize.height())) / 100);
      }
      else
        ITHROWGUIERROR("IQUERYPOINTERINFO");
    }
  }
  else
  {
#endif // IC_WIN
    bmp = pIDMImage->bitmap();
    if ((unsigned long)bmp.asUnsigned() ) // new
    {
      IGImage  bitmap(bmp);
      ISize    bmpSize = ISize( bitmap.rasterBounds().width(),
                                bitmap.rasterBounds().height() );
        this->totalImgOffset.setWidth(this->totalImgOffset.width() +
                             (stackingPercentage().coord1() *
                             (bmpSize.width() + stretchSize.width())) / 100);
        this->totalImgOffset.setHeight(this->totalImgOffset.height() +
                             (stackingPercentage().coord2() *
                             (bmpSize.height() + stretchSize.height())) / 100);
#ifdef IC_PMWIN
    }
#endif
  }
#endif // IC_PMWIN
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::totalImageOffset                                         |
|                                                                              |
| Return the total image offset.                                               |
| Notes:                                                                       |
|   1) Private                                                                 |
------------------------------------------------------------------------------*/
ISize IDMSourceOperation::totalImageOffset()
{
  return (this->totalImgOffset);
}

#ifdef IC_MOTIFWIN  
/*------------------------------------------------------------------------------
| IDMSourceOperation::sourceOperation                                          |
|                                                                              |
| Returns the source operation object handle.                                  |
------------------------------------------------------------------------------*/
IDMSourceOperation::Handle IDMSourceOperation :: sourceOperation()
{
  return (pDMSrcOpHandle);
}

/*------------------------------------------------------------------------------
| IDMSourceOperation::setSourceOperation                                       |
|                                                                              |
| Sets the source operation object handle.                                     |
------------------------------------------------------------------------------*/
void IDMSourceOperation ::
    setSourceOperation(IDMSourceOperation::Handle sourceOp)
{
  pDMSrcOpHandle = sourceOp;
}
#endif


/*------------------------------ Target Operation ----------------------------*/

/*------------------------------------------------------------------------------
| IDMTargetOperation::IDMTargetOperation                                       |
|                                                                              |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
#ifdef IC_PMWIN 
IDMTargetOperation :: IDMTargetOperation(IDMTargetEnterEvent &event)
 : IDMOperation(event),
   bWasDragAfter(0),
   sizeDropOffset(0,0),
   dropStyle(IDM::notContainer),
   bNotDragLeave(true),
   bDrop(false)
#ifdef IC_WIN
  ,fDMTargetOperationData( new IDMTargetOperationData() )
#endif
  ,bFirstTimeEntered(true)
{
  IFUNCTRACE_DEVELOP();

#ifdef IC_WIN
  /*-------------------------------------------------------------------
    Build a list of the target drag items
    -------------------------------------------------------------------*/
  fDMTargetOperationData->buildItems(this, event);

  if ( fDMTargetOperationData->isContainerControl( event.window() ))
  {
    /*-----------------------------------------------------------------
      Set the target window handle
      -----------------------------------------------------------------*/
    setTargetWindowHandle(event.window()->handle());

    /*-----------------------------------------------------------------
       Get flag for over/after event and set our default drop style ...
      -----------------------------------------------------------------*/
    this->bWasDragAfter = event.isDragAfter();
    if (this->bWasDragAfter)
      setStyle(IDM::alignFlow);
    else
      setStyle(IDM::dropAtPosition);

    /*-----------------------------------------------------------------
       Set the container object pointer for the object that the over
       or after event occurred over.  0 if over container white space.
     -----------------------------------------------------------------*/
    setContainerObject(event.object());
  }
#endif

#ifdef IC_PM
  _DRAGINFO *pDInfo;

  IDMItemProvider *pProvider;

  if (event.eventId() == DM_DRAGOVER)
  {
    /*-----------------------------------------------------------------
      Set the target window handle
      -----------------------------------------------------------------*/
    setTargetWindowHandle(event.handle());

    /*-----------------------------------------------------------------
      Get the PM DRAGINFO structure
     -----------------------------------------------------------------*/
    pDInfo = (_DRAGINFO *)event.parameter1().asUnsignedLong();

    /*-----------------------------------------------------------------
      Get pointer to the item provider
     -----------------------------------------------------------------*/
    pProvider = event.window()->itemProvider();
  }
  else
  {
    /*-----------------------------------------------------------------
       Verify that this is the correct message and notification code
     -----------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            (event.parameter1().number2() == CN_DRAGOVER ||
             event.parameter1().number2() == CN_DRAGAFTER));

    /*-----------------------------------------------------------------
       Set the target window handle
     -----------------------------------------------------------------*/
    setTargetWindowHandle(event.controlWindow()->handle());

    /*-----------------------------------------------------------------
       Get flag for over/after event and set our default drop style ...
      -----------------------------------------------------------------*/
    this->bWasDragAfter = event.isDragAfter();
    if (this->bWasDragAfter)
      setStyle(IDM::alignFlow);
    else
      setStyle(IDM::dropAtPosition);

    /*-----------------------------------------------------------------
       Set the container object pointer for the object that the over
       or after event occurred over.  0 if over container white space.
     -----------------------------------------------------------------*/
    setContainerObject(event.object());

    /*-----------------------------------------------------------------
       Get the PM DRAGINFO structure
     -----------------------------------------------------------------*/
    _CNRDRAGINFO *pCnrDInfo =
                 (_CNRDRAGINFO *)event.parameter2().asUnsignedLong();
    pDInfo = pCnrDInfo->pDragInfo;

    /*-----------------------------------------------------------------
      Get pointer to the item provider
     -----------------------------------------------------------------*/
    pProvider = event.controlWindow()->itemProvider();
  }
  /*-------------------------------------------------------------------
     Access the PM DRAGINFO structure which was passed to us.
   -------------------------------------------------------------------*/
  if (!DrgAccessDraginfo (pDInfo))
    ITHROWGUIERROR("DrgAccessDraginfo");

  /*-------------------------------------------------------------------
     Store "handle" to our drag information
   -------------------------------------------------------------------*/
  setDragInfo (pDInfo);

  /*-------------------------------------------------------------------
     Set the source window handle
   -------------------------------------------------------------------*/
  setSourceWindowHandle(pDInfo->hwndSource);

  /*-------------------------------------------------------------------
      Set the target enter (i.e. drag over) position of the pointing
     device.
    -------------------------------------------------------------------*/
  setPosition (IPoint(pDInfo->xDrop, pDInfo->yDrop));

  /*-------------------------------------------------------------------
      Set the current drag operation.
    -------------------------------------------------------------------*/
  setOperation (pDInfo->usOperation);

  /*-------------------------------------------------------------------
      Get first PM drag item
    -------------------------------------------------------------------*/
  unsigned long ulItem = 0;
  _DRAGITEM *pPMDItem = DrgQueryDragitemPtr (pDInfo, ulItem);

  /*-------------------------------------------------------------------
      For each dragitem in the draginfo, we need to create a
      corresponding IDMItem.  We construct each based upon the
      information in the PM DRAGINFO structure.
    -------------------------------------------------------------------*/
  while (pPMDItem)
  {
    /*-----------------------------------------------------------------
      Create the generic drag item based upon the PM DRAGITEM ...
      -----------------------------------------------------------------*/
    IDMItem::Handle pDIH(new IDMItem(this, pPMDItem));

    /*-----------------------------------------------------------------
      Obtain a derived target item based upon the generic drag
      item via the item provider  - Throw an exception if the
      provider fails ...
      -----------------------------------------------------------------*/
    IDMItem::Handle pTgtItemH = pProvider->provideTargetItemFor(pDIH);
    if ( ! pTgtItemH.valid() )
    {
      ITRACE_DEVELOP("IDMTargetOperation: target item provider for failed");
      //Throw exception here
      ITHROWLIBRARYERROR(IC_TARGET_ITEM_PROVIDER_FAILED,
          IBaseErrorInfo::invalidRequest,
          IException::recoverable);
      break;
    }

    /*-----------------------------------------------------------------
    Add the target item to the collection ...
      -----------------------------------------------------------------*/
    addItem (pTgtItemH);

    /*-----------------------------------------------------------------
     Get the next PM DRAGITEM, it it exists ...
     -----------------------------------------------------------------*/
    pPMDItem = DrgQueryDragitemPtr (pDInfo, ++ulItem);
  }

  /*-------------------------------------------------------------------
     Free the PM DRAGINFO structure
    -------------------------------------------------------------------*/
  DrgFreeDraginfo (pDInfo);
#endif
}
#endif // IC_PMWIN

#ifdef IC_MOTIF 
IDMTargetOperation :: IDMTargetOperation(IDMTargetEnterEvent &event)
 : IDMOperation(event)
 , bWasDragAfter(0)
 , sizeDropOffset(0,0)
 , dropStyle(IDM::notContainer)
 , bNotDragLeave(true)
 , bDrop(false)
 , bFirstTimeEntered(true)
{
  IFUNCTRACE_DEVELOP();

  _DRAGINFO *pDInfo;
  IDMItemProvider *pProvider;

  if (event.eventId() == DM_DRAGOVER)
  {
    /*-----------------------------------------------------------------
       Set the target window handle
     -----------------------------------------------------------------*/
    setTargetWindowHandle(event.handle()); 

    /*-----------------------------------------------------------------
       Get the PM DRAGINFO structure
     -----------------------------------------------------------------*/
    pDInfo = (_DRAGINFO *)event.parameter2().asUnsignedLong();

    /*-----------------------------------------------------------------
      Get pointer to the item provider
     -----------------------------------------------------------------*/
    pProvider = event.window()->itemProvider();
  }
  else
  {
    /*-----------------------------------------------------------------
       Verify that this is the correct message and notification code
     -----------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            (event.parameter1().number2() == CN_DRAGOVER ||
             event.parameter1().number2() == CN_DRAGAFTER));

    /*-----------------------------------------------------------------
       Set the target window handle
     -----------------------------------------------------------------*/
    setTargetWindowHandle(event.controlWindow()->handle()); 

    /*-----------------------------------------------------------------
       Get flag for over/after event and set our default drop style ...
     -----------------------------------------------------------------*/
    this->bWasDragAfter = event.isDragAfter();
    if (this->bWasDragAfter)
      setStyle(IDM::alignFlow);
    else
      setStyle(IDM::dropAtPosition);

    /*-----------------------------------------------------------------
       Set the container object pointer for the object that the over
       or after event occurred over.  0 if over container white space.
     -----------------------------------------------------------------*/
    setContainerObject(event.object());

    /*-----------------------------------------------------------------
       Get the PM DRAGINFO structure
     -----------------------------------------------------------------*/
    _CNRDRAGINFO *pCnrDInfo =
                 (_CNRDRAGINFO *)event.parameter2().asUnsignedLong();
    pDInfo = pCnrDInfo->pDragInfo;

    /*-----------------------------------------------------------------
      Get pointer to the item provider
     -----------------------------------------------------------------*/
    pProvider = event.controlWindow()->itemProvider();
  }

  /*-------------------------------------------------------------------
     Access the PM DRAGINFO structure which was passed to us.
   -------------------------------------------------------------------*/
  IDragInfoWrapper dInfoWrapper(*pDInfo);
  dInfoWrapper.access();

  /*-------------------------------------------------------------------
     Store "handle" to our drag information
   -------------------------------------------------------------------*/
  setDragInfo (pDInfo);

  /*-------------------------------------------------------------------
     Set the source window handle
   -------------------------------------------------------------------*/
  setSourceWindowHandle(dInfoWrapper.sourceWindowHandle());

  /*-------------------------------------------------------------------
     Set the target enter (i.e. drag over) position of the pointing
     device.
   -------------------------------------------------------------------*/
  setPosition (dInfoWrapper.dropPoint());

  /*-------------------------------------------------------------------
     Set the current drag operation.
   -------------------------------------------------------------------*/
  setOperation (dInfoWrapper.dragOperation());

  /*-------------------------------------------------------------------
     For each dragitem in the draginfo, we need to create a
     corresponding IDMItem.  We construct each based upon the
     information in the PM DRAGINFO structure.
   -------------------------------------------------------------------*/
  unsigned short numItems = dInfoWrapper.numDragItems();
  for ( unsigned long i=0; i<numItems; i++)
  {
    /*-----------------------------------------------------------------
      Create the generic drag item based upon the PM DRAGITEM ...
     -----------------------------------------------------------------*/
     IDMItem::Handle pDIH(new IDMItem(this, &dInfoWrapper[i]));
     pDIH->addRMF( IDM::rmLibrary, IDM::rfSharedMem );

    /*-----------------------------------------------------------------
      Obtain a derived target item based upon the generic drag
      item via the item provider  - Throw an exception if the
      provider fails ...
     -----------------------------------------------------------------*/
    IDMItem::Handle pTgtItemH = pProvider->provideTargetItemFor(pDIH);
    if (! pTgtItemH.valid() )
    {
      ITRACE_DEVELOP("IDMTargetOperation: target item provider for failed");
      //Throw exception here
      ITHROWLIBRARYERROR(IC_TARGET_ITEM_PROVIDER_FAILED,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
      break;
    }

    /*-----------------------------------------------------------------
      Add the target item to the collection ...
     -----------------------------------------------------------------*/
    addItem (pTgtItemH);
  }

  /*-------------------------------------------------------------------
     Free the PM DRAGINFO structure
   -------------------------------------------------------------------*/
  dInfoWrapper.release();
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| IDMTargetOperation::~IDMTargetOperation                                      |
|                                                                              |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
IDMTargetOperation :: ~IDMTargetOperation ()
{
  IFUNCTRACE_DEVELOP();

  /*-------------------------------------------------------------------
     Enable container refresh (this function can detect whether or
     not we're processing a container) ...
   -------------------------------------------------------------------*/
  setContainerRefreshOn();

#ifdef IC_PM
  DRAGINFO*  pPMDragInfo = dragInfo();
  if (this->bNotDragLeave)
    DrgDeleteDraginfoStrHandles (pPMDragInfo);

  if (this->bDrop)
    DrgFreeDraginfo (pPMDragInfo);
#endif

#ifdef IC_WIN
  if (fDMTargetOperationData )
    delete fDMTargetOperationData;
#endif

#ifdef IC_MOTIF 
  _DRAGINFO* pPMDragInfo = dragInfo();
  IDragInfoWrapper dInfoWrapper(*pPMDragInfo);
  if (this->bNotDragLeave)
    dInfoWrapper.deleteTargetResources();
  if (this->bDrop)
    dInfoWrapper.release();
#endif // IC_MOTIF

}

/*------------------------------------------------------------------------------
| IDMTargetOperation::instanceFor                                              |
|                                                                              |
| Returns the IDMTargetOperation object for the given target enter event.      |
------------------------------------------------------------------------------*/
#ifdef IC_PMWIN  
IDMTargetOperation::Handle IDMTargetOperation ::
                           instanceFor (IDMTargetEnterEvent &event)
{
  IMODTRACE_DEVELOP("IDMTargetOperation::instanceFor");
  /*-------------------------------------------------------------------
     If the target operation has not been created yet, create it and
     store it as part of the IDMTargetOperation class.
   -------------------------------------------------------------------*/
  IDMTargetOperation::Handle pTgtOpH(targetOperation());
  if ( ! pTgtOpH.valid() )
    return (0);

#ifdef IC_WIN
  /*-------------------------------------------------------------------
     If the target is a container ...
   -------------------------------------------------------------------*/
  if ( pTgtOpH->fDMTargetOperationData->isContainerControl( event.window() ))
  {
    pTgtOpH->bWasDragAfter = event.isDragAfter();
    if (pTgtOpH->bWasDragAfter)
      pTgtOpH->setStyle(IDM::alignFlow);
    else
      pTgtOpH->setStyle(IDM::dropAtPosition);

    /*-------------------------------------------------------------------
       Set the container object pointer for the object that the over
       or after event occurred over.  0 if over container white space.
     -------------------------------------------------------------------*/
    pTgtOpH->setContainerObject(event.object());

    /*-------------------------------------------------------------------
       Set the target enter (i.e. drag over) position of the pointing
       device.
      -------------------------------------------------------------------*/
    pTgtOpH->setPosition (event.position());
  }
#endif

#ifdef IC_PM
  /*------------------------------------------------------------------------
    Extract pointer to PM DRAGINFO struct from event parms
  ------------------------------------------------------------------------*/
  _DRAGINFO *pDInfo;
  if (event.eventId() == DM_DRAGOVER)
  {
    /*-----------------------------------------------------------------
       Get the PM DRAGINFO structure
     -----------------------------------------------------------------*/
    pDInfo = (_DRAGINFO *)event.parameter1().asUnsignedLong();
  }
  else
  {
    /*-----------------------------------------------------------------
       Verify that this is the correct message and notification code
     -----------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            (event.parameter1().number2() == CN_DRAGOVER ||
             event.parameter1().number2() == CN_DRAGAFTER));

    /*-----------------------------------------------------------------
       Get flag for over/after event and set our default drop style ...
     -----------------------------------------------------------------*/
    pTgtOpH->bWasDragAfter = event.isDragAfter();
    if (pTgtOpH->bWasDragAfter)
      pTgtOpH->setStyle(IDM::alignFlow);
    else
      pTgtOpH->setStyle(IDM::dropAtPosition);

    /*-------------------------------------------------------------------
       Set the container object pointer for the object that the over
       or after event occurred over.  0 if over container white space.
     -------------------------------------------------------------------*/
    pTgtOpH->setContainerObject(event.object());

    pDInfo = ((_CNRDRAGINFO *)event.parameter2().asUnsignedLong())->pDragInfo;
  }

  /*-------------------------------------------------------------------
     Access the PM DRAGINFO structure which was passed in the event.
   -------------------------------------------------------------------*/
  if (!DrgAccessDraginfo (pDInfo))
    ITHROWGUIERROR("DrgAccessDraginfo");

  /*-------------------------------------------------------------------
     Set the target enter (i.e. drag over) position of the pointing
     device.
   -------------------------------------------------------------------*/
  pTgtOpH->setPosition (IPoint(pDInfo->xDrop, pDInfo->yDrop));

  /*-------------------------------------------------------------------
     Set the current drag operation.
   -------------------------------------------------------------------*/
  pTgtOpH->setOperation (pDInfo->usOperation);

  /*-------------------------------------------------------------------
     Free the PM DRAGINFO structure
   -------------------------------------------------------------------*/
  DrgFreeDraginfo (pDInfo);
#endif

  return (pTgtOpH);
}
#endif // IC_PMWIN
#ifdef IC_MOTIF 
IDMTargetOperation::Handle IDMTargetOperation ::
                           instanceFor (IDMTargetEnterEvent &event)
{
  IFUNCTRACE_DEVELOP();
  /*-------------------------------------------------------------------
     If the target operation has not been created yet, create it and
     store it as part of the IDMTargetOperation class.
   -------------------------------------------------------------------*/
  IDMTargetOperation::Handle pTgtOpH = targetOperation();
  if ( !pTgtOpH.valid() )
    return (0);

// ifdef IC_PM
  /*------------------------------------------------------------------------
    Extract pointer to PM DRAGINFO struct from event parms
  ------------------------------------------------------------------------*/
  _DRAGINFO *pDInfo;
  if (event.eventId() == DM_DRAGOVER)
  {
    /*-----------------------------------------------------------------
       Get the PM DRAGINFO structure
     -----------------------------------------------------------------*/
    pDInfo = (_DRAGINFO *)event.parameter2().asUnsignedLong();
  }
  else
  {
    /*-----------------------------------------------------------------
       Verify that this is the correct message and notification code
     -----------------------------------------------------------------*/
    IASSERT(event.eventId() == WM_CONTROL &&
            (event.parameter1().number2() == CN_DRAGOVER ||
             event.parameter1().number2() == CN_DRAGAFTER));

    /*-----------------------------------------------------------------
       Get flag for over/after event and set our default drop style ...
     -----------------------------------------------------------------*/
    pTgtOpH->bWasDragAfter = event.isDragAfter();
    if (pTgtOpH->bWasDragAfter)
      pTgtOpH->setStyle(IDM::alignFlow);
    else
      pTgtOpH->setStyle(IDM::dropAtPosition);

    /*-------------------------------------------------------------------
       Set the container object pointer for the object that the over
       or after event occurred over.  0 if over container white space.
     -------------------------------------------------------------------*/
    pTgtOpH->setContainerObject(event.object());

    pDInfo = ((_CNRDRAGINFO *)event.parameter2().asUnsignedLong())->pDragInfo;
  }

  /*-------------------------------------------------------------------
     Access the PM DRAGINFO structure which was passed in the event.
   -------------------------------------------------------------------*/
  IDragInfoWrapper dInfoWrapper(*pDInfo);
  dInfoWrapper.access();

  /*-------------------------------------------------------------------
     Set the target enter (i.e. drag over) position of the pointing
     device.
   -------------------------------------------------------------------*/
  pTgtOpH->setPosition (dInfoWrapper.dropPoint());

  /*-------------------------------------------------------------------
     Set the current drag operation.
   -------------------------------------------------------------------*/
  pTgtOpH->setOperation (dInfoWrapper.dragOperation());

  /*-------------------------------------------------------------------
     Free the PM DRAGINFO structure
   -------------------------------------------------------------------*/
  dInfoWrapper.release();
// endif
// ifdef IC_MOTIF
//   if ( event.eventId() == XmCR_DROP_SITE_ENTER_MESSAGE )
//     {
//     XmDragProcCallback dpcb = (XmDragProcCallback)
//         event.parameter1().asUnsignedLong();
//     /*-------------------------------------------------------------------
//        Set the target enter (i.e. drag over) position of the pointing
//        device.
//      -------------------------------------------------------------------*/
//     // these values need to be recoordinated.
//     pTgtOpH->setPosition (IPoint(dpcb->x, dpcb->y));
//
//     /*-------------------------------------------------------------------
//        Set the current drag operation.
//      -------------------------------------------------------------------*/
//     if ( dpcb->operation == XmDROP_MOVE )
//       pTgtOpH->setOperation(IDMOperation::move);
//     else if ( dpcb->operation == XmDROP_COPY )
//       pTgtOpH->setOperation(IDMOperation::copy);
//     else if ( dpcb->operation == XmDROP_LINK )
//       pTgtOpH->setOperation(IDMOperation::link);
//     else
//       pTgtOpH->setOperation(IDMOperation::unknown );
//     }
// endif
// //@@Rob.  Adapted from Jim.

  return (pTgtOpH);
}
#endif

/*------------------------------------------------------------------------------
| IDMTargetOperation::targetOperation                                          |
|                                                                              |
| Returns the target operation object handle.                                  |
------------------------------------------------------------------------------*/
IDMTargetOperation::Handle IDMTargetOperation :: targetOperation()
{
  return (pDMTgtOpHandle);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::setTargetOperation                                       |
|                                                                              |
| Sets the target operation object handle.                                     |
------------------------------------------------------------------------------*/
void IDMTargetOperation ::
        setTargetOperation(IDMTargetOperation::Handle targetOp)
{
  pDMTgtOpHandle = targetOp;
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::wasDragAfter                                             |
|                                                                              |
| Returns the state of the latest DRAGOVER or DRAGAFTER.                       |
------------------------------------------------------------------------------*/
bool IDMTargetOperation :: wasDragAfter() const
{
  return(this->bWasDragAfter);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::dropOffset                                               |
|                                                                              |
| Queries the current drop offset.                                             |
------------------------------------------------------------------------------*/
ISize IDMTargetOperation :: dropOffset() const
{
  return(this->sizeDropOffset);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::style                                                    |
|                                                                              |
| Returns the drop style.                                                      |
------------------------------------------------------------------------------*/
DropStyle IDMTargetOperation :: style() const
{
  return(this->dropStyle);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::setDropOffset                                            |
|                                                                              |
| Sets the current drop offset.                                                |
------------------------------------------------------------------------------*/
IDMTargetOperation& IDMTargetOperation ::
                       setDropOffset(const ISize &offset)
{
  this->sizeDropOffset = offset;
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::setStyle                                                 |
|                                                                              |
| Sets the drop style.                                                         |
------------------------------------------------------------------------------*/
IDMTargetOperation& IDMTargetOperation ::
       setStyle(DropStyle drpStyle)
{
  this->dropStyle = drpStyle;
  return(*this);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::isStyle                                                  |
|                                                                              |
| Queries whether a specified style flag is set.                               |
------------------------------------------------------------------------------*/
bool IDMTargetOperation :: isStyle(DropStyle drpStyle)
{
  return((style() & drpStyle) == drpStyle);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::dropPosition                                             |
|                                                                              |
| Computes the drop position for this item, in container workspace             |
| coordinates, and sets the offset for the next item, according to             |
| the operation's drop style.                                                  |
------------------------------------------------------------------------------*/
IPoint IDMTargetOperation :: dropPosition(const IDMItem::Handle &pDIH,
                                          IDMTargetDropEvent &event)
{
  IMODTRACE_DEVELOP("IDMTargetOperation dropPosition");

  IPoint ptDrop;

  if (isStyle(IDM::dropAtPosition))
  {
    /*-----------------------------------------------------------------
       Drop at position
     -----------------------------------------------------------------*/
    ptDrop = positionRelativeTo(pDIH, event);
  }
  else
  {
    /*-----------------------------------------------------------------
       Drop-align
     -----------------------------------------------------------------*/
    ISize sizOff = dropOffset();

    IPoint ptMouse = positionRelativeTo(0, event);

     /*-----------------------------------------------------------------
       Initial drop guess is mouse position plus current offset
     -----------------------------------------------------------------*/
    ptDrop = ptMouse + sizOff;

    /*-----------------------------------------------------------------
       If flow, check that we are not out of screen
     -----------------------------------------------------------------*/
    IRectangle rectObject;
    if (event.object())
      rectObject = event.container()->iconRectangle(event.object(), true);
    else
      rectObject = IRectangle();


    if (isStyle(IDM::alignFlow))
    {
      IRectangle rectWorkspace = event.container()->viewPortOnWorkspace();

      /*---------------------------------------------------------------
         Correct next position if out of container window
       ---------------------------------------------------------------*/
      if (isStyle(IDM::alignVertical))
      {
        /*-------------------------------------------------------------
           Vertical alignment
          -------------------------------------------------------------*/
        if (rectWorkspace.bottom() > ptDrop.y())
        {
          sizOff = ISize((sizOff.width() + rectObject.width()), 0);
          if (!isStyle(IDM::flowBeside))
          {
            sizOff.setHeight(rectWorkspace.top() - ptMouse.y() -
                             rectObject.height());
          }

          /*-----------------------------------------------------------
           Re-calc drop point
           -----------------------------------------------------------*/
          ptDrop = ptMouse + sizOff;
        }
      }
      else
      {
        /*-------------------------------------------------------------
           Horizontal alignment
         -------------------------------------------------------------*/
        if (rectWorkspace.right() < (ptDrop.x() + rectObject.width()))
        {
          sizOff = ISize(0, (sizOff.height() - rectObject.height()));
          if (!isStyle(IDM::flowBeside))
            sizOff.setWidth(rectWorkspace.left() - ptMouse.x());

          /*-----------------------------------------------------------
             Re-calc drop point
           -----------------------------------------------------------*/
          ptDrop = ptMouse + sizOff;
        }
      }
    }

    /*-----------------------------------------------------------------
       Compute next drop offset
     -----------------------------------------------------------------*/
    if (isStyle(IDM::alignVertical))
    {
      /*---------------------------------------------------------------
         Align vertically (-y)
         ---------------------------------------------------------------*/
      sizOff.setHeight(sizOff.height() - rectObject.height());
    }
    else
    {
      /*---------------------------------------------------------------
         Align horizontally (+x)
       ---------------------------------------------------------------*/
      sizOff.setWidth(sizOff.width() + rectObject.width());
    }

    setDropOffset(sizOff);
  }

  return(ptDrop);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::positionRelativeTo                                       |
|                                                                              |
| Returns a drag item's position relative to the target container.  If the     |
| item pointer is 0, the drop point relative to the container is returned.     |
------------------------------------------------------------------------------*/
IPoint IDMTargetOperation ::
       positionRelativeTo(const IDMItem::Handle &pDIH,
                          IDMTargetDropEvent &event)
{
  IMODTRACE_DEVELOP("IDMTargetOperation::positionRelativeTo");

  IPoint ptDrop;
  IContainerControl *pCnrTgt = event.container();

  if (pCnrTgt)
  {
    /*-----------------------------------------------------------------
       Map to container window position
     -----------------------------------------------------------------*/
    IPoint dropPoint = event.dropPosition();
    ITRACE_DEVELOP( "dropPoint is " + dropPoint.asString() );

    ptDrop = IWindow::mapPoint( event.dropPosition(),
                                IWindow::desktopWindow()->handle(),
                                pCnrTgt->handle() );

    ITRACE_DEVELOP( "mapped ptDrop is " + ptDrop.asString() );
    /*-----------------------------------------------------------------
       Adjustment for workspace displacement relative to viewPort
       ---------------------------------------------------------------*/
    ptDrop += pCnrTgt->viewPortOnWorkspace().bottomLeft();
    ITRACE_DEVELOP( "vpOnWorkspace ptDrop is " + ptDrop.asString() );

    ptDrop -= pCnrTgt->viewPortOnWindow().bottomLeft();
    ITRACE_DEVELOP( "vpOnWindow ptDrop is " + ptDrop.asString() );

    /*-----------------------------------------------------------------
       Adjustment for icon offset
     -----------------------------------------------------------------*/
    if ( pDIH.valid() )
      ptDrop += pDIH->imageOffset();
    ITRACE_DEVELOP( "and the is ptDrop is " + ptDrop.asString() );

  }

  return(ptDrop);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::firstTimeEntered                                         |
|                                                                              |
| Returns a flag indicating if first time entered                              |
------------------------------------------------------------------------------*/
bool IDMTargetOperation :: firstTimeEntered() const
{
  return (this->bFirstTimeEntered);
}

/*------------------------------------------------------------------------------
| IDMTargetOperation::setFirstTimeEntered                                      |
|                                                                              |
| Sets a flag indicating if first time entered                                 |
------------------------------------------------------------------------------*/
IDMTargetOperation &IDMTargetOperation :: setFirstTimeEntered(bool flag)
{
  this->bFirstTimeEntered = flag;
  return (*this);
}

#ifdef IC_WIN
/*-------------------------------------------------------------------
| Return truth of parameter *IWindow begin a container control
-------------------------------------------------------------------*/
bool IDMTargetOperationData::isContainerControl( const IWindow *win )
{
  char pszClassName[30];
  long lLen = IQUERYCLASSNAME(win->handle(),
                              sizeof(pszClassName),
                              (PSZ)pszClassName);

  /*------------------------------------------
  | If window class is edit control...
  ------------------------------------------*/
  IString className(pszClassName);
  return ( lLen > 0 &&
           (( className == IString(WC_CONTAINER)) ||
            (className == IString(WC_NATIVECONTAINER)) ));
}


/*------------------------------------------------------------------------------
| IDMTargetOperationData::buildItems                                (PRIVATE)  |
|                                                                              |
| Builds a list of the target drag items                                       |
------------------------------------------------------------------------------*/
void  IDMTargetOperationData::buildItems(IDMTargetOperation* pTgtOp,
                                         IDMTargetEnterEvent& event )
{

  IMODTRACE_DEVELOP("IDMTargetOperation::buildItems");

  ULONG ulItemCnt = 1;
  IDMItemProvider *pProvider;

  IDataObject *dataObject =
      (IDataObject *) event.parameter2().asUnsignedLong();


  /*------------------------------------------------------------
  | Set window handle of target operation so it will be
  | picked up in DRAGINFO instance..
  ------------------------------------------------------------*/
  pTgtOp->setTargetWindowHandle(event.window()->handle());

 /*------------------------------------------------------------
  | Build a DRAGINFO wrapper from the OLE IDataObject sent to
  | us in the event.If drag from non-OpenClass window, the
  | pDInfoWrapper will default to one item with types set
  | from the formats advertised in the dataObject.
  ------------------------------------------------------------*/
  IDragInfoWrapper* pDInfoWrapper = IDragInfoWrapper::dragInfoWrapper(dataObject);
  if (pDInfoWrapper)
    ulItemCnt = pDInfoWrapper->numDragItems();

  if (event.eventId() == DM_DRAGOVER)
  {
    /*-----------------------------------------------------------------
       Set the target window handle
      ----------------------------------------------------------------*/
    pTgtOp->setTargetWindowHandle(event.handle());

    /*-----------------------------------------------------------------
       Get pointer to the item provider
      ----------------------------------------------------------------*/
    pProvider = event.window()->itemProvider();
  }

  if (pDInfoWrapper)
  {
    /*----------------------------------------------------------------
    |   Set source window handle
    ----------------------------------------------------------------*/
    pTgtOp->setSourceWindowHandle(pDInfoWrapper->sourceWindowHandle());
  }

  /*------------------------------------------------------------------
     Set the target enter (i.e. drag over) position of the pointing
     device.
  ------------------------------------------------------------------*/
  IPoint pt( event.parameter1().number1(), event.parameter2().number2());
  pTgtOp->setPosition(pt);

  /*------------------------------------------------------------------
     Set the current drag operation.
  ------------------------------------------------------------------*/
  pTgtOp->setOperation( IDMOperation::move );

  IDMTargetOperation::Handle pTgtOpH;
  pTgtOpH = pTgtOp;

  /*--------------------------------------------------------------------
    Create the generic drag item from IDataObject ...
    Get number of objects in source op.
    Iterate over the objects in source op:
      For each, do the following, through the addItem
  --------------------------------------------------------------------*/
  ITRACE_DEVELOP(IString("Target buildItems see's ") + IString(ulItemCnt) + " Items");

  /*--------------------------------------------------------------------
    Iterate over the source items adding them to the target operation
  --------------------------------------------------------------------*/
  const IString aSpace(' ');
  for (unsigned nItem = 1; nItem <= ulItemCnt; ++nItem)
  {
    _DRAGITEM& dragItem = (*pDInfoWrapper)[nItem - 1]; // ie. IDragInfo&
    IDMItem::Handle pDIH(new IDMItem(pTgtOpH.getAlias(), &dragItem));
    IDragItemWrapper dragItemWrapper(dragItem);

    pDIH->setPMDragItem(&dragItem);

  /*--------------------------------------------------------------------
    Check for existing RMFs which match ours before adding them
  --------------------------------------------------------------------*/
    if( !pDIH->supportsRMF( dragItemWrapper.rmfs() ) )
        pDIH->addRMF( dragItemWrapper.rmfs() );
    if( !pDIH->supportsRMF( IDMItem::rmfsFrom( IDM::rmLibrary, IDM::rfSharedMem ) ) )
        pDIH->addRMF( IDM::rmLibrary, IDM::rfSharedMem );

    ITRACE_DEVELOP(IString("buildItems processed item ") +
                 IString(dragItemWrapper.itemID() ));

    /*-----------------------------------------------------------------
      Obtain a derived target item based upon the generic drag
      item via the item provider  - Throw an exception if the
      provider fails ...
    -----------------------------------------------------------------*/
    IDMItem::Handle pTgtItemH;
    pTgtItemH = pProvider->provideTargetItemFor(pDIH);
    if (! pTgtItemH.valid() )
    {
      ITRACE_DEVELOP("IDMTargetOperation: target item provider for failed");
    }

    /*-----------------------------------------------------------------
      Add the target item to the collection ...
    -----------------------------------------------------------------*/
    pTgtOpH->addItem (pTgtItemH);
  }

  //---------------------------------------------------------
  // Save the _DRAGINFO because items have references to it
  // Let our destructor delete it because we destruct after
  // items are done with ref's to the contained _DRAGITEM's.
  //---------------------------------------------------------
  if (pDInfo)
    delete pDInfo;
  pDInfo = pDInfoWrapper->asDragInfoPtr ( );

  delete pDInfoWrapper;
}

/*---------------------------------------------------------------
| Marshall DRAGINFO and DRAGITEMs to byte array:
---------------------------------------------------------------*/
IStream* IDMSourceOperationData::marshall(
                    IDragInfoWrapper* pDInfoWrapper)
{
  IStream* pIStream = NULL;

  /*---------------------------------------------------------------
  | 1. Get length of marshall byte array.
  | 2. Marshall draginfo object instance to byte array.
  ---------------------------------------------------------------*/
  unsigned cbDragInfo = pDInfoWrapper->asDragInfoPtr()->marshall();

  char* pMarshalledData = new char[cbDragInfo];
  if (pMarshalledData != NULL)
  {
    ITRACE_DEVELOP("Marshalling dInfoWrapper for " +
          IString(cbDragInfo) + IString(" bytes"));

    pDInfoWrapper->asDragInfoPtr()->marshall(0, pMarshalledData);

    /*---------------------------------------------------------------
    | 3. Create hGlobal IStream of DRAGINFO and DRAGITEMs' byte array
    ---------------------------------------------------------------*/
    ITRACE_DEVELOP("Creating IStream of marshalled dInfoWrapper");

    HRESULT hr = CreateStreamOnHGlobal(0, True, &pIStream);
    ULONG cbResult = 0;

    /*---------------------------------------------------------------
    | 4. Write marshalled byte array (flattened DRAGINFO) to IStream
    ---------------------------------------------------------------*/
    if (hr == S_OK)
      hr = pIStream->Write(pMarshalledData, cbDragInfo, &cbResult);   // Write marshalled byte-array

    IASSERT(cbResult == cbDragInfo);

    delete [] pMarshalledData;
    pMarshalledData = NULL;
  }
  return pIStream;
}


/*------------------------------------------------------------------------------
| IDMSourceOperationData::doDrag                                    (PRIVATE)  |
|                                                                              |
| Starts the direct manipulation                                               |
------------------------------------------------------------------------------*/
void IDMSourceOperationData::doDrag(IDMSourceOperation* pSrcOp)
{
  IMODTRACE_DEVELOP("IDMSourceOperationData::doDrag");

  /*----------------------------------------------------------------------------
  | Make sure we have items to drag...
  ----------------------------------------------------------------------------*/
  const unsigned long   ulItemCnt = pSrcOp->numberOfItems();
  ITRACE_DEVELOP( "Number of Items is " + IString(ulItemCnt) );

  bool doTheDrag = bool(ulItemCnt);

  /*--------------------------------------------------------------
  | Count the types of the first item. We must assume that all
  | items of the drag have these types because OLE assumes
  | all advertised formats (types) apply to everything getting
  | transfered by OLE Uniform Data Transfer.
  --------------------------------------------------------------*/
  IDMItem::Handle pItemHandle;
  pItemHandle = pSrcOp->item( 1 );
  const IString& itemTypes(pItemHandle->types());
  const unsigned countTypes = itemTypes.occurrencesOf(',') + 1;

  bool filesSupported(false);
  unsigned fileFormats = 0;
  if ( pItemHandle->supportsRMF( IDM::rmFile, IDM::rfText ) )
  {
    ITRACE_DEVELOP("File Format supported");
    filesSupported = True;
    fileFormats = 3;

    // CF_HROP  // already added
    // FileName
    // ShellIDList
    // ShellIDListArray
  }

  /*--------------------------------------------------------------
  | Reserve space for all formats of first item, plus one
  | IOC_DRAGINFO format.
  --------------------------------------------------------------*/
  FORMATETC *prgfe = new FORMATETC[countTypes + 1 + fileFormats];

  /*-----------------------------------------------------------------
  | Add all types of first item to the format list in FIFO order.
  -----------------------------------------------------------------*/
  unsigned startPos = 1;
  for ( unsigned typeIndex = 0; typeIndex < countTypes; typeIndex++ )
  {
    const unsigned index = itemTypes.indexOf(',', startPos);
    const IString type = (index ?
                    itemTypes.subString(startPos, index - startPos) :
                    itemTypes.subString(startPos));

    startPos = index + 1;

    /*----------------------------------------------------------------
    | Convert type to corresponding clipboard format
    ----------------------------------------------------------------*/
    const CLIPFORMAT cfFormat = CLIPFORMAT(IDMFormat::formatFromString(type));
    ITRACE_DEVELOP("converted type " + IString(cfFormat));

    /*----------------------------------------------------------------
    | Add clipboard format (type) to array in FIFO order
    ----------------------------------------------------------------*/
    SETDefFormatEtc( prgfe[ typeIndex ], cfFormat, TYMED_HGLOBAL );

   }

  /*-----------------------------------------------------------------
  | Create an empty _DRAGINFO object for our number of items
  -----------------------------------------------------------------*/
  _DRAGINFO *pDInfo = new IDragInfo (
            (unsigned short)(unsigned long)ulItemCnt);

  /*-----------------------------------------------------------------
  | Populate _DRAGINFO and encapsulate it in a draginfo wrapper.
  -----------------------------------------------------------------*/
  IDragInfoWrapper* pDInfoWrapper =
      IDragInfoWrapper::setDragInfoWrapper( pSrcOp, pDInfo );

  /*---------------------------------------------------------------
  | Marshall draginfo wrapper to a global IStream
  | Note: we own this IStream. The IDataObject will clone
  | it to any target requesting it. Each target owns the clone
  | given to it. Each owner, including us, must release any
  | owned IStream interface.
  ---------------------------------------------------------------*/
  IStream* pIStream = marshall( pDInfoWrapper);

  /*---------------------------------------------------------------
  | Set private clipboard format for the IStream containing
  | marshalled DRAGINFO and DRAGITEMs
  ---------------------------------------------------------------*/
  unsigned restOfFormats = countTypes;
  if (pIStream != NULL)
  {
    ITRACE_DEVELOP("Registering private clipboard format DRAGINFO");
    const CLIPFORMAT clipFmt =
        CLIPFORMAT(::RegisterClipboardFormat(IDMDataObject::formatDragInfo()));

    SETDefFormatEtc( prgfe[restOfFormats], clipFmt, TYMED_ISTREAM );
    restOfFormats++;
    doTheDrag = True;
  }


  if ( filesSupported )
  {
    ITRACE_DEVELOP("Files supported");

    CLIPFORMAT clipFmt = IDMFormat::formatFromString( CFSTR_SHELLIDLIST);
    ITRACE_DEVELOP("Added format " + IString(clipFmt) );
    SETDefFormatEtc( prgfe[restOfFormats], clipFmt, TYMED_HGLOBAL );
    restOfFormats++;

    clipFmt = IDMFormat::formatFromString( CFSTR_SHELLIDLISTOFFSET);
    ITRACE_DEVELOP("Added format " + IString(clipFmt) );
    SETDefFormatEtc( prgfe[restOfFormats], clipFmt, TYMED_HGLOBAL );
    restOfFormats++;

//    ITRACE_DEVELOP("Added format " + IString(CF_HDROP) );
//    SETDefFormatEtc( prgfe[restOfFormats], CF_HDROP, TYMED_HGLOBAL );
//    restOfFormats++;

    clipFmt = IDMFormat::formatFromString( CFSTR_FILENAME);
    ITRACE_DEVELOP("Added format " + IString(clipFmt) );
    SETDefFormatEtc( prgfe[restOfFormats], clipFmt, TYMED_HGLOBAL );
    restOfFormats++;
  }

  /*---------------------------------------------------------------
  | Create IDataObject from generated formats, source window,
  | and the IStream containing simulated PM DRAGINFO, DRAGITEMs.
  ---------------------------------------------------------------*/
  // + 1 is for IOCDRAGINFO
  IDMEnumFormat formats( countTypes + 1 + fileFormats, prgfe );

  delete [] prgfe;

  IDataObject* pIDataObject = NULL;
  IDMDataObject* pIDMDataObject = NULL;

  if (doTheDrag)
  {
    pIDMDataObject = new IDMDataObject(
           IWindow::windowWithHandle(pSrcOp->sourceWindowHandle()),
           &formats, pIStream);
    if (FAILED(pIDMDataObject->QueryInterface(IID_IDataObject,
             (void **) &pIDataObject)))
    {
      doTheDrag = false;
    }
  }

  IDropSource* pIDropSource = NULL;
  IDMDropSource* pIDMDropSource = NULL;
  if (doTheDrag)
  {
    /*-------------------------------------------------------------------
    | Store "handle" to our drag information
    -------------------------------------------------------------------*/
    pDInfoWrapper->setSourceWindowHandle( pSrcOp->sourceWindowHandle() );
    pSrcOp->setDragInfo (pDInfo);

    pIDMDropSource = new IDMDropSource;

    if (FAILED(pIDMDropSource->QueryInterface(IID_IDropSource,
              (void **) &pIDropSource)))
    {
      doTheDrag = false;
    }
  }

  if ( doTheDrag )
  {
    ITRACE_DEVELOP("do the drag");
    unsigned long supportedOperations = 0,
                  operations = 0;
    IDMItem::Handle anItem;
    anItem = pSrcOp->item(1);

    if ( (IDM::allStacked == pSrcOp->imageStyle() ) ||
         (IDM::stack3AndFade == pSrcOp->imageStyle() ))
    {
      IDMImage image;
      IPoint stackHotspot;
      unsigned int numImages = 1;

      // Windows 95 has restrictions on ImageList sizes which will cause
      // an application to terminate if the icon is too big.  Try/Catch
      // block is added below to use default image in this scenario.
      try
      {
        numImages = stackImages(image, stackHotspot, pSrcOp);
      }
      catch ( IException &exp )
      {
        // If exception thrown, reset image so that default drag image
        // is used in setImage call below.
        image.setPointer(0);
        image.setBitmap(0);
      }
      if ( numImages)
        pIDMDropSource->setImage( image, stackHotspot );
      ITRACE_DEVELOP("stackHotspot is " + stackHotspot.asString());
    }

    supportedOperations = anItem->supportedOperations();
    if ( supportedOperations & IDMItem::moveable )
    {
      operations |= DROPEFFECT_MOVE;
      ITRACE_DEVELOP( "Move supported");
    }

    if ( supportedOperations & IDMItem::copyable )
    {
      operations |= DROPEFFECT_COPY;
      ITRACE_DEVELOP( "Copy supported");
    }

    if ( supportedOperations & IDMItem::linkable )
    {
      operations |= DROPEFFECT_LINK;
      ITRACE_DEVELOP( "Link supported");
    }

    /*----------------------------------------------------------------
       Perform modal drag loop through OLE
    ----------------------------------------------------------------*/
    ITRACE_DEVELOP("DoDragDrop called");
    DWORD dwEffect;
    HRESULT hResult = DoDragDrop(
       (LPDATAOBJECT) pIDataObject,
       (LPDROPSOURCE) pIDropSource,
       operations,
       &dwEffect );

    if (pSrcOp)
    {
       unsigned long ulNumItems = pSrcOp->numberOfItems();
       if (hResult != DRAGDROP_S_DROP)
       {
          ITRACE_DEVELOP("after drag cancel, remove " + IString(ulNumItems));
          for (unsigned i = ulNumItems; i; i--)
             pSrcOp->removeItem(i);
       }
       else if ((ulNumItems > 0) && (pIDMDropSource->keyState() & MK_LBUTTON))
       {
          // if we got here then we just finished a default drag to a non OpenClass
          // target.
          // Now, need to set things up so that sourceEnd() gets called on the
          // source IDMItem
          // (numItems > 0 means the target side hasn't sent DM_ENDCONVERSATION yet.
          // MK_LBUTTON means we are doing a LMB drag. if we are doing RMB
          // drag, then the target side hasn't rendered anything yet so leave
          // things as they are.
          IDM::RenderCompletion code = IDM::targetSuccessful; // extrapolating from DRAGDROP_S_DROP
          unsigned long drgOperation(IDMOperation::unknown);
          if (dwEffect & DROPEFFECT_MOVE)
             drgOperation = IDMOperation::move;
          else if (dwEffect & DROPEFFECT_COPY)
             drgOperation = IDMOperation::copy;
          else if (dwEffect & DROPEFFECT_LINK)
             drgOperation = IDMOperation::link;
          for (unsigned i = ulNumItems; i; i--) {
             IDMItem::Handle anItem;
             anItem = pSrcOp->item(i);
             IWindowHandle winH = anItem->sourceWindowHandle();
             anItem->sourceWindowHandle().sendEvent (DM_ENDCONVERSATION,
                                                     IEventParameter1((unsigned long) (void *) anItem.getAlias()),
                                                     IEventParameter2(code,drgOperation));
          } // for
       } // numItems>0 and keyState
    } // pSrcOp

  } /* end if */

  ITRACE_DEVELOP("DoDragDrop ended" );

  if (pIDMDataObject)
    pIDMDataObject->Release();

  if (pIDMDropSource)
    pIDMDropSource->Release();

  delete pDInfo;

  // We are done with the source operation, set it back to 0.
//  IDMSourceOperation::setSourceOperation( 0 );
}
#endif // IC_WIN


#ifdef IC_WIN


/*------------------------------------------------------------------------------
| IDMSourceOperationData::stackImages                                  PRIVATE
|
| Build a masked image of the stacked images. Set result as drag image.
| NOTE: Very Win32-specific. Needs to be abstracted (2D graphics,
| class IImageList, etc.
|
------------------------------------------------------------------------------*/
unsigned IDMSourceOperationData::stackImages(
              IDMImage& anImage,
              IPoint& stackHotspot,
              IDMSourceOperation* pSrcOp)
{
  IMODTRACE_DEVELOP("IDMSourceOperationData::stackImages");

  //--------------------------------------------
  // Hotspot will be adjusted for larger Win32
  // icon retangle, if necessary.
  //--------------------------------------------
  IDMItem::Handle pItemHandle;  // Caution: don't construct and assign.
  pItemHandle = pSrcOp->item(1); // Now okay to initialize it.
  IDMImage theImage(pItemHandle->image());
  ISize imageOffset = theImage.pointerOffset();
  ITRACE_DEVELOP("imageOffset is " + theImage.pointerOffset().asString() );
  if ( imageOffset == 0 )
  {
    imageOffset = pItemHandle->imageOffset();
    ITRACE_DEVELOP("imageOffset is " + imageOffset.asString() );
  }

  ITRACE_DEVELOP("Pointer offset is " + imageOffset.asString() );
  IPoint hotspot(IPoint(0,0) - imageOffset );
  // TODO : maybe ? IPoint hotspot(IPoint(0,0) - pSrcOp->pointerOffset());
  ITRACE_DEVELOP("Initial hot spot is " + hotspot.asString() );

  //--------------------------------------------
  // Hold individual input images in image list.
  //--------------------------------------------
  HIMAGELIST himagelist = NULL;

  //--------------------------------------------
  // Const stagger offset.
  //--------------------------------------------
  const ISize offset(16, 16);  // Need stacking percentage

  //--------------------------------------------
  // Bounding rectangle of stack. Will grow
  // this during pass 1. Adjust hotspot.
  //--------------------------------------------
  IRectangle rect(IPoint(0, 0), ISize(0, 0));

  //--------------------------------------------
  // Pass 1: compute number of images, bounding
  // rectangle of stack, and hotspot.
  // Return value: number of images stacked.
  //--------------------------------------------
  unsigned cImagesUsed = stackImagesPass1(
            anImage,      // Image to receive stacked images
            pSrcOp,       // Drag source operation
            offset,       // Stack stagger in pixels
            himagelist,   // Receives images to be stacked
            rect,         // Receives computed bounding rect of stack
            hotspot       // Receives intermediate hotspot
            );

  //--------------------------------------------
  // If pass 1 found stackable images, carry out
  // pass 2: stack the images.
  //--------------------------------------------
  if (cImagesUsed)
    stackImagesPass2(
                anImage,
                offset,
                himagelist,
                rect);

  //--------------------------------------------
  // If success, set stack hotspot size, then
  // free work image list.
  //--------------------------------------------
  if (himagelist)
  {
    int cx, cy;
    ::ImageList_GetIconSize(himagelist, &cx, &cy);
    ISize cellSize(ISize(cx, cy));

    stackHotspot = IPoint(hotspot.x(),
                          rect.height() - cellSize.height() + hotspot.y() );

    ::ImageList_Destroy(himagelist);
  }

  return cImagesUsed;
}

//-----------------------------------------------------------------
// PASS ONE of stack images:
// Iterate over collection of IDMItems.  For each, set its image
// into a work image list. Compute the bounding rectangle of the
// images when they're stacked, using the stagger offset.
//-----------------------------------------------------------------
unsigned IDMSourceOperationData::stackImagesPass1(
              IDMImage& anImage,           // I/O: image of stack
              IDMSourceOperation* pSrcOp,  // Input: source drag operation
              const ISize& offset,         // Input: stagger offset in pixels
              HIMAGELIST& himagelist,      // Output: list of stackable images
              IRectangle& rect,            // Output: bounding rect of stack
              IPoint& hotspot)             // Output: working hotspot
{
  IMODTRACE_DEVELOP("IDMSourceOperationData::stackImagesPass1");

  const unsigned       itemCount  = pSrcOp->numberOfItems();
  const DragImageStyle diStyle    = pSrcOp->imageStyle();

  //--------------------------------------------
  // Return value: number of images stacked
  //--------------------------------------------
  unsigned cImagesUsed = 0;

  //-------------------------------------------------------------------
  // Wer haben Item, nicht whar? Throw exception if no drag items.
  //-------------------------------------------------------------------
  if ( itemCount )
  {
    //-----------------------------------------------------------------
    // Iterate over collection of IDMItems.  For each, set its image
    // into a work image list. Compute the bounding rectangle of the
    // images when they're stacked, using the supplied offsets.
    //-----------------------------------------------------------------
    for (unsigned itemIndex = 0; itemIndex < itemCount; itemIndex++)
    {
      //-------------------------------------
      // Finished if we cached the fade icon
      //-------------------------------------
      if ( itemIndex >= 4 && diStyle == IDM::stack3AndFade )
        break;

      HICON hicon = NULL;

      IDMItem::Handle pItemHandle;  // Caution: don't construct and assign.
      pItemHandle = pSrcOp->item(itemIndex + 1); // Now okay to initialize it.

      //--------------------------------------------
      // Get image
      // It can't be a ref because we may substitute
      // the fade icon later.
      //--------------------------------------------
      IDMImage image(pItemHandle->image());

      ISize sizeBmp(0, 0);

      bool needToDelete = false;
      ICONINFO iconInfo;
      IGImage*  pbmImage = NULL; // Image bitmap

      //----------------------------------
      // Using fade and we've got three?
      // Then substitute the fade icon.
      //----------------------------------
      if ( itemIndex >= 3 && diStyle == IDM::stack3AndFade )
        image = IDMImage(ISystemPointerHandle(
                              ISystemPointerHandle::dragFade));

      //-------------------
      // If icon...
      //-------------------
      if ( image.pointer() )
      {
        hicon = image.pointer();

        //--------------------------------------------
        // Disect icon. If it's cursor, we can't
        // stack it. Many programs routinely set
        // StackAll or Stack3AndFade for cursor
        // images. We use OLE default cursors in
        // that case, so we don't want to stack
        // cursors.
        //--------------------------------------------
        if (!::GetIconInfo(hicon, &iconInfo) || !iconInfo.fIcon )
        {
          ITRACE_DEVELOP("bailing out of stackImage - 2 " + IString(itemIndex));
          return 0; // Found a cursor: use OLE default instead.
        }

        pbmImage = new IGImage(iconInfo.hbmColor);
        needToDelete = true;

        // Clean up 1 of the bitmaps created and passed from GetIconInfo
        // Cannot delete other yet since it is currently wrappered
        ::DeleteObject( iconInfo.hbmMask );
      }
      else
      //--------------------------------------------
      // Else if Bitmap...
      //--------------------------------------------
      if ( image.bitmap() )
      {
        pbmImage = new IGImage(image.bitmap());
      }
      else
      {
        return 0; // We only stack bitmaps or icons
      }

      //--------------------------------------------
      // Bitmap information
      //--------------------------------------------
      BITMAP bmpInfo;
      ::GetObject(pbmImage->handle(), sizeof bmpInfo, &bmpInfo);

      //--------------------------------------------
      // Get bounding rectangle of this image
      //--------------------------------------------
      ISize sizeIcon(bmpInfo.bmWidth, bmpInfo.bmHeight);
// 
//      if (image.style() & IDMImage::stretch)
//      {
//        sizeIcon = image.stretchSize();
//      }
// 
      IRectangle rectBmp(IPoint(0, 0), sizeIcon);

      //----------------------------------------------------------
      // Maintain the hotspot of the first image.
      // Later, after the images are stacked, we'll compute
      // another hotspot for the entire stack.
      //----------------------------------------------------------
      if (itemIndex == 0)
      {
        //---------------------------------------------------------------
        // If the hot spot is relative to the lower left than we need to
        // convert it to the upper left. ImageLists understand only
        // native window MM_TEXT mapping mode coordinates.
        //---------------------------------------------------------------
        if ( ICoordinateSystem::applicationOrientation() ==
               ICoordinateSystem::originLowerLeft )
        {
          hotspot = ICoordinateSystem::convertToNative(
              hotspot,
              sizeIcon );
          ITRACE_DEVELOP("Converted hot spot is " + hotspot.asString() );
        }
      }

      //----------------------------------------
      // Create image list sized to first icon.
      //----------------------------------------
      if (himagelist == NULL)
        himagelist = ::ImageList_Create(
                           int(bmpInfo.bmWidth),
                           int(bmpInfo.bmHeight),
                           imageListFlags,
                           4,
                           4  // May need more for StackAll
                           );

      //--------------------------------------------
      // Add icon to image list
      //--------------------------------------------
      if ( image.pointer() )
        ::ImageList_AddIcon(himagelist, hicon);
      else
        ::ImageList_AddMasked(himagelist, pbmImage->handle(),
                              RGB(255,255,255));

      //--------------------------------------------
      // Grow rect to bound icon if necessary
      //    rect.UnionRect(&rect, &rectBmp);
      //--------------------------------------------
      rect |= rectBmp;

      //--------------------------------------------
      // Grow rect by offset * position in sequence
      //--------------------------------------------
      if (itemIndex != 0)
        rect = IRectangle(IPoint(0, 0),
      ISize(ISize(rect.width(),rect.height()) + offset));

      delete pbmImage; // Carry out GDI DeleteObject

      //--------------------------------------
      // Cannot delete hbmColor until after it
      // has potentially been used by pbmImage
      //--------------------------------------
      if (needToDelete)
        ::DeleteObject( iconInfo.hbmColor );

      //-------------------------------
      // Count number of icons used
      //-------------------------------
      cImagesUsed++;

    }  // end pass 1

    //-----------------------------------------------
    // We know the final hotspot so set it in result
    //-----------------------------------------------
    anImage.setPointerOffset(hotspot);

  }
  else
      ITHROWGUIERROR("stackImagesPass1");

  return cImagesUsed++;
}

//-----------------------------------------------------------------
// PASS TWO:
// Given an imagelist of icons, stack them, last-to-first, from
// the NE to SW corners of the bounding rectangle that was
// computed in pass one. The last icons stacked is the top-most
// icon and is the icon under the mouse.
//
// Result: number of images used
// Parameters:  IDMImage& anImage,       // Output: image of stack
//              const ISize& offset,     // Input: stack stager in pixels
//              HIMAGELIST himagelist,   // Input: list of images
//              const IRectangle& rect   // Input: stack image bounds
//
//-----------------------------------------------------------------
unsigned IDMSourceOperationData::stackImagesPass2(
                IDMImage& anImage,       // Output: image of stack
                const ISize& offset,     // Input: stagger offet in pixels
                const HIMAGELIST himagelist, // Input: list of images
                const IRectangle& rect)  // Input: stack image bounds
{
  IMODTRACE_DEVELOP("IDMSourceOperationData::stackImagesPass1");

  const unsigned cImagesUsed = ::ImageList_GetImageCount(himagelist);

  ITRACE_DEVELOP(IString(cImagesUsed) + IString(" = images used"));
  ITRACE_DEVELOP("rect == " + rect.asString());

  //-------------------------------------------------------------
  // Need to build final image w/ equal sides to avoid stretching
  // the image (because it is being placed into a pointer)
  //-------------------------------------------------------------
  int maxDim = (rect.width() > rect.height()) ?
                (int)rect.width() : (int)rect.height();

  //-------------------------------------------------------
  // Build imagelist for stack image
  //-------------------------------------------------------
  HIMAGELIST hImageStack = ::ImageList_Create(
                              maxDim, maxDim,
                              imageListFlags,
                              1,
                              0      // Allow  one only
                              );

  if (hImageStack == NULL)
    ITHROWGUIERROR("stackImagesPass2");

  //-------------------------------------------------------
  // Enumerate icons last-to-first.
  // Lay each into the image, offsetting from NE to SW.
  // Lay each image's mask in the same position.
  //-------------------------------------------------------
  for (int index = cImagesUsed; index > 0; --index)
  {
    const int position = index - 1;
    HIMAGELIST himagelist2 = ::ImageList_Merge(
                                      hImageStack,
                                      0,
                                      himagelist,
                                      position,
                                      int(offset.width() * position),
                                      int(offset.height() * (cImagesUsed - index))
                                 );

    if (himagelist2 == NULL)
      ITHROWGUIERROR("stackImagesPass2");

    ::ImageList_Destroy(hImageStack);
    hImageStack = himagelist2;
  }

  //--------------------------------------------
  // Set drag image
  //--------------------------------------------
  HICON hiconStack = ::ImageList_GetIcon(hImageStack, 0, ILD_TRANSPARENT );
  if (hiconStack == NULL)
    ITHROWGUIERROR("stackImagesPass2");

  ::ImageList_Destroy(hImageStack);

  //--------------------------------------------
  // Set result: picture
  //--------------------------------------------
  anImage.setPointer(IPointerHandle (hiconStack ));


  return cImagesUsed;
}

#endif

/*------------------------------------------------------------------------------
| IDMOperation::debugSupport                                                   |
|                                                                              |
| Returns the static to indicate if debug support is set for DrgDrag           |
------------------------------------------------------------------------------*/
bool IDMOperation::debugSupport()
{
  return IDMOperation::fgDebugSupport;
}

/*------------------------------------------------------------------------------
| IDMOperation::setdebugSupport                                                |
|                                                                              |
| Sets the debugSupport flag used for DrgDrag                                  |
------------------------------------------------------------------------------*/
void IDMOperation::setDebugSupport(bool debugSupport)
{
  IDMOperation::fgDebugSupport = debugSupport;
}

