// Revision: 70 1.15.2.1 source/ui/drag/idmevent.cpp, dragdrop, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: idmevent.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IDMEvent                                                                 *
*       IDMSourceBeginEvent                                                    *
*       IDMSourceRenderEvent                                                   *
*         IDMSourcePrepareEvent                                                *
*       IDMSourceEndEvent                                                      *
*       IDMSourceDiscardEvent                                                  *
*       IDMSourcePrintEvent                                                    *
*                                                                              *
*       IDMTargetEvent                                                         *
*         IDMTargetEnterEvent                                                  *
*         IDMTargetLeaveEvent                                                  *
*         IDMTargetDropEvent                                                   *
*       IDMTargetHelpEvent                                                     *
*       IDMTargetEndEvent                                                      *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C" {
  /* include PM headers */
  #define INCL_WININPUT
  #define INCL_WINSTDDRAG
  #define INCL_WINSTDCNR
  #include <iwindefs.h>
}


#include <ihandle.hpp>
#include <ictlevt.hpp>
#include <icnrctl.hpp>
#include <icnrobj.hpp>
#include <icnrrec.hpp>
#include <iexcept.hpp> // Exception handling
#include <itrace.hpp>  // Trace Facility
#include <icoordsy.hpp>


#include <idmevent.hpp> // Include our header
#include <idmitem.hpp>
#ifdef IC_MOTIFWIN
  #include <idmditm.hpp>
  #include <idmdnfo.hpp>
#endif
#include <idmsrcop.hpp>
#include <idmtgtop.hpp>
#include <idmseq.hpp>
#include <icconst.h>

#ifdef IC_MOTIF
   #include <idmmotf.hpp>
   #include <ixdc.hpp>
#endif


// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IDMEVENT_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_MOTIFWIN
#pragma enum(4)
#pragma pack(push,4)

class IDMEventData {
public:
  IDMEventData ( )
  { }
virtual
 ~IDMEventData ( )
  { }
static bool
  isContainerControl ( IWindow *win );
}; // class IDMEventData

#pragma enum(pop)
#pragma pack(pop)

#ifdef IC_WIN
bool IDMEventData::isContainerControl( IWindow *win )
{
  char pszClassName[30];
  long lLen = IQUERYCLASSNAME(win->handle(),
                              sizeof(pszClassName),
                              (PSZ)pszClassName);

  if ( lLen > 0 )
  {
    // If window class is edit control
    IString className(pszClassName);
    if ( (className == IString(WC_CONTAINER)) ||
         (className == IString(WC_NATIVECONTAINER)) )
    {
      return true;
    }
  }
  return false;
}  // isContainerControl
#endif // IC_WIN

#ifdef IC_MOTIF
bool IDMEventData::isContainerControl( IWindow *win )
{
  return IDMMotif::isContainerControl( win );
}  // isContainerControl
#endif // IC_MOTIF
#endif // IC_MOTIFWIN

/*------------------------------------------------------------------------------
| Base Events                                                                  |
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
| IDMEvent::IDMEvent                                                           |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMEvent :: IDMEvent (const IEvent &event)
  : IEvent(event.handle(),
           event.eventId(),
           event.parameter1(),
           event.parameter2())
{
#ifdef IC_MOTIF
  // We did not use the IEvent copy constructor, because we do not want
  // a copy of the handler list and cursor pointers.  This is to prevent
  // handlers from using IWindow::dispatchRemainingHandlers to handle
  // IDMEvent's.
  //@@Rob.  Check with Mark Benge to make sure this is the intent.
  //@@Rob.  Actually, Mark says it was an oversight not to copy
  //@@Rob.  the private data.  Remember to submit a defect for next
  //@@Rob.  release of OS/2 ICLUI.             IC_NOTYET
  setPassToOwner(event.passToOwner());
#endif
#ifdef IC_PM
  if ( event.eventId() == WM_CONTROL )
  {  // Container will generate drag notifications using WM_CONTROL.
     IControlEvent
       ctlEvent( event );
     this->setControlHandle( ctlEvent.controlHandle() );
  }
#endif
}

/*------------------------------------------------------------------------------
| IDMEvent::~IDMEvent                                                          |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMEvent :: ~IDMEvent ()
{
}

/*------------------------------------------------------------------------------
| Source Events                                                                |
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::IDMSourceBeginEvent                                     |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMSourceBeginEvent :: IDMSourceBeginEvent (const IEvent &event) :
                       IDMEvent(event),
                       src(IDM::pointingDevice)
{
  /**************************************************************************/
  /* Save the pointing device's position                                    */
  /* Note: In OS/2 PM drag CAN ONLY be initiated by the pointing device ... */
  /*   This may not be the case for other platforms ...                     */
  /**************************************************************************/
}

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::~IDMSourceBeginEvent                                    |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMSourceBeginEvent :: ~IDMSourceBeginEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::position                                                |
|                                                                              |
| Return the position.                                                         |
------------------------------------------------------------------------------*/
IPoint IDMSourceBeginEvent :: position() const
{
#ifdef IC_MOTIFWIN
  IPoint pt;
  if ( eventId() == WM_BEGINDRAG )
  {
    pt = IPoint( parameter1().lowNumber(),
                 parameter1().highNumber());
    // coords already transformed to correct origin by sender
  }
  return pt;
#endif // IC_MOTIFWIN
#ifdef IC_PM
  if ( eventId() == WM_BEGINDRAG )
  {
    return IPoint( parameter1().lowNumber(),
                   parameter1().highNumber() );
  }
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_INITDRAG );

    _CNRDRAGINIT *pCnrDInit =
      (_CNRDRAGINIT *)parameter2().asUnsignedLong();
    return IPoint( pCnrDInit->x, pCnrDInit->y );
  }
#endif // IC_PM
} // position

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::source                                                  |
|                                                                              |
| Returns the source of the drag operation.                                    |
------------------------------------------------------------------------------*/
IDM::Source IDMSourceBeginEvent :: source() const
{
  return this->src;
}

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::containerId                                             |
|                                                                              |
| Returns the source container's ID.                                           |
------------------------------------------------------------------------------*/
unsigned long IDMSourceBeginEvent :: containerId() const
{
#ifdef IC_MOTIFWIN
  if ( eventId() == WM_BEGINDRAG )
  {
    if ( IDMEventData::isContainerControl( window() ))
      return window()->id();
  }
  return 0;
#endif // IC_MOTIFWIN
#ifdef IC_PM
  if ( eventId() == WM_BEGINDRAG )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_INITDRAG );

    return parameter1().number1();
  }
#endif // IC_PM
}

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::object                                                  |
|                                                                              |
| Returns a pointer to the object over which the drag started.  Returns 0 if   |
| drag started over "white space" or if non-container.                         |
------------------------------------------------------------------------------*/
IContainerObject* IDMSourceBeginEvent :: object() const
{
#ifdef IC_MOTIFWIN
  if ( IDMEventData::isContainerControl( window() ) )
  {
    IContainerControl *cnrctl = (IContainerControl *) window();
    // Position is relative to the window, not the desktop.
    IPoint pt = position();

    IPoint ptDrag = pt;
    return cnrctl->objectUnderPoint( ptDrag );
  }
  return 0;
#endif // IC_MOTIFWIN
#ifdef IC_PM
  if ( eventId() == WM_BEGINDRAG )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_INITDRAG );

    _CNRDRAGINIT *pCnrDInit =
     (_CNRDRAGINIT *)parameter2().asUnsignedLong();
    return IObjFromRec( (IMiniCnrRecord *)pCnrDInit->pRecord );
  }
#endif // IC_PM
}

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::container                                               |
|                                                                              |
| Returns a pointer to the source container for which the event occurred.      |
------------------------------------------------------------------------------*/
IContainerControl* IDMSourceBeginEvent :: container() const

{
#ifdef IC_MOTIFWIN
  if ( eventId() == WM_BEGINDRAG )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      return (IContainerControl*) this->controlWindow();
    }
  }
  return 0 ;
#endif // IC_MOTIF_WIN
#ifdef IC_PM
  if ( eventId() == WM_BEGINDRAG )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_INITDRAG );

    return (IContainerControl*) this->controlWindow();
  }
#endif // IC_PM
} // container

/*------------------------------------------------------------------------------
| IDMSourceBeginEvent::offset                                                  |
|                                                                              |
| Returns the position of the pointing device relative to the source           |
| container object's origin.                                                   |
------------------------------------------------------------------------------*/
ISize IDMSourceBeginEvent :: offset() const
{
#ifdef IC_MOTIFWIN
  IMODTRACE_DEVELOP("IDMSourceBeginEvent::offset()");

  if ( IDMEventData::isContainerControl( window() ))
  {
    IContainerControl *ctrlWin = (IContainerControl *) window();
    IContainerObject *ctrlObj = ctrlWin->objectUnderPoint( position() );
    if ( ctrlObj )
    {
      IRectangle iconRect = ctrlWin->iconRectangle( ctrlObj, false );
      ITRACE_DEVELOP( "iconRectangle is " + iconRect.asString() );
      ITRACE_DEVELOP( "position is " + position().asString() );
      IPoint ptOffset = iconRect.minXMinY() - position();
      ITRACE_DEVELOP( "ptOffset is " + ptOffset.asString() );

#ifdef IC_WIN
      char pszClassName[30];
      long lLen = IQUERYCLASSNAME( ctrlWin->handle(),
                                   sizeof(pszClassName),
                                   (PSZ)pszClassName );
      if ( lLen > 0 )
      {
        // If window class is native container control
        IString className( pszClassName );
        ITRACE_DEVELOP( "The class name is " + className );
        if ( className == IString(WC_NATIVECONTAINER) )
        {
          ISize iconSize = ctrlWin->iconSize();
          if ( iconRect.size() != iconSize )
          {
            ITRACE_DEVELOP( "Adjust the ptOffset, iconSize is " +
                            iconSize.asString() );
            // Adjust the hot spot.
            IPoint diffPt( ( iconRect.size().width()
                               - iconSize.width() )/2 ,
                           ( iconRect.size().height()
                              - iconSize.height())/2 );
            ITRACE_DEVELOP( "Diff pt is " + diffPt.asString() );
            ptOffset += diffPt;
            ITRACE_DEVELOP( "Adjusted ptOffset is " + ptOffset.asString());
          }
        }
      }
#endif // IC_WIN
      return ISize( ptOffset.x(), ptOffset.y() );
    }
  }
  return ISize();
#endif // IC_MOTIFWIN

#ifdef IC_PM
  if ( eventId() == WM_BEGINDRAG )
    return ISize();
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_INITDRAG );

    _CNRDRAGINIT *pCnrDInit =
      (_CNRDRAGINIT *)parameter2().asUnsignedLong();
    return ISize( pCnrDInit->cx, pCnrDInit->cy );
  }
#endif // IC_PM

} // offset

/*------------------------------------------------------------------------------
| IDMSourceEndEvent::IDMSourceEndEvent                                         |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMSourceEndEvent :: IDMSourceEndEvent (const IEvent &event) :
                                        IDMEvent(event),
                                        pDMSrcItemHandle(0)
{
   //@@Rob.  This comment mentions Drag Transfer, but I don't see why.
  /******************************************************************/
  /* Get PM Drag Transfer structure, source drag item handle and    */
  /* source drag operation handle                                   */
  /******************************************************************/
  this->pDMSrcItemHandle = (IDMItem *)(event.parameter1().asUnsignedLong());
  this->bWasTgtSuccessful =
        (event.parameter2().asUnsignedLong() & IDM::targetSuccessful) ? true
                                                                      : false;
}

/*------------------------------------------------------------------------------
| IDMSourceEndEvent::~IDMSourceEndEvent                                        |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMSourceEndEvent :: ~IDMSourceEndEvent ( )
{
}

/*------------------------------------------------------------------------------
| IDMSourceEndEvent::dragItem                                                  |
|                                                                              |
| Returns handle of drag item the target has completed rendering of.           |
------------------------------------------------------------------------------*/
IDMItem::Handle IDMSourceEndEvent :: dragItem() const
{
  return this->pDMSrcItemHandle;
}

/*------------------------------------------------------------------------------
| IDMSourceEndEvent::wasTargetSuccessful                                       |
|                                                                              |
| Returns whether target successfully completed its rendering.                 |
------------------------------------------------------------------------------*/
bool IDMSourceEndEvent :: wasTargetSuccessful() const
{
  return this->bWasTgtSuccessful;
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::IDMSourcePrepareEvent                                 |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMSourcePrepareEvent :: IDMSourcePrepareEvent (const IEvent &event) :
                         IDMSourceRenderEvent (event),
                         bTgtCanRetry(false),
                         bNoSrcRendering(false)
{
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::~IDMSourcePrepareEvent                                |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMSourcePrepareEvent :: ~IDMSourcePrepareEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::alternateWindow                                       |
|                                                                              |
| Returns a pointer to the alternate source window.                            |
------------------------------------------------------------------------------*/
IWindow* IDMSourcePrepareEvent :: alternateWindow() const
{
  return IWindow::windowWithHandle( alternateWindowHandle() );
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::alternateWindowHandle                                 |
|                                                                              |
| Returns the handle of the alternate source window.                           |
------------------------------------------------------------------------------*/
IWindowHandle IDMSourcePrepareEvent :: alternateWindowHandle() const
{
  // Only supported on OS/2.
#ifdef IC_PM
  return dragTransfer()->pditem->hwndItem;
#else
  return IWindowHandle( 0 );
#endif
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::setAlternateWindowHandle                              |
|                                                                              |
| Set the alternate source window handle.                                      |
------------------------------------------------------------------------------*/
IDMSourcePrepareEvent &IDMSourcePrepareEvent ::
                      setAlternateWindowHandle (const IWindowHandle &window)
{
#ifdef IC_PM
  dragTransfer()->pditem->hwndItem = window;

  /*************************************************************************/
  /* Update the source window handle in the drag item, so the target will  */
  /* be aware of an alternate source window for source rendering.          */
  /*************************************************************************/
  dragItem()->setSourceWindowHandle( window );
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::setTargetCanRetry                                     |
------------------------------------------------------------------------------*/
IDMSourcePrepareEvent &IDMSourcePrepareEvent ::
                      setTargetCanRetry (bool flag)
{
  this->bTgtCanRetry = flag;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourcePrepareEvent::setNoSourceRendering                                  |
------------------------------------------------------------------------------*/
IDMSourcePrepareEvent& IDMSourcePrepareEvent ::
                       setNoSourceRendering (bool flag)
{
  this->bNoSrcRendering = flag;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::IDMSourceRenderEvent                                   |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMSourceRenderEvent :: IDMSourceRenderEvent (const IEvent &event)
 : IDMEvent (event)
 , pDMSrcItemHandle(0)
 , bRetry(false)
 , rcCompletion(IDM::renderOk)

{
#ifdef IC_PM
  /*************************************************************************/
  /* Get access PM Drag Transfer structure, source drag item handle and    */
  /* source drag operation handle                                          */
  /*************************************************************************/
  this->pPMDragTransfer =
    (_DRAGTRANSFER *)event.parameter1().asUnsignedLong();

  if ( (!(this->pPMDragTransfer)) || ((!(this->pPMDragTransfer->pditem))) )
  {
    ITHROWLIBRARYERROR( IC_DRAGTRANSFER_INVALID,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  this->pDMSrcItemHandle = (IDMItem *)(dragTransfer()->pditem->ulItemID);

  /*************************************************************************/
  /* Save target "render to" name and target info                          */
  /*************************************************************************/
  this->strTgtName =
    IDMItem::stringFromHandle( dragTransfer()->hstrRenderToName );

  /*************************************************************************/
  /* Get the selected RMF                                                  */
  /*************************************************************************/
  IString selRMF =
    IDMItem::stringFromHandle( dragTransfer()->hstrSelectedRMF );

  /*************************************************************************/
  /* Extract selected rendering mechanism from selected RMF                */
  /*************************************************************************/
  this->strSelMechanism = IDMItem::rmFrom( selRMF );

  /*************************************************************************/
  /* Extract selected rendering format from selected RMF                   */
  /*************************************************************************/
  this->strSelFormat = IDMItem::rfFrom( selRMF );

  /*************************************************************************/
  /* Get the target window handle                                          */
  /*************************************************************************/
  this->tgtWindowHandle = IWindowHandle( dragTransfer()->hwndClient );
#endif // IC_PM
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::~IDMSourceRenderEvent                                  |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMSourceRenderEvent :: ~IDMSourceRenderEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::dragItem                                               |
|                                                                              |
| Returns handle to the IDMItem corresponding to the item being rendered.      |
------------------------------------------------------------------------------*/
IDMItem::Handle IDMSourceRenderEvent :: dragItem() const
{
  return this->pDMSrcItemHandle;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::alternateWindow                                        |
|                                                                              |
| Returns a pointer to the alternate source window.                            |
------------------------------------------------------------------------------*/
IWindow* IDMSourceRenderEvent :: alternateWindow() const
{
  return IWindow::windowWithHandle( alternateWindowHandle() );
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::alternateWindowHandle                                  |
|                                                                              |
| Returns the handle of the alternate source window.                           |
------------------------------------------------------------------------------*/
IWindowHandle IDMSourceRenderEvent :: alternateWindowHandle() const
{
#ifdef IC_PM
  return dragTransfer()->pditem->hwndItem;
#else
  // Not supported yet.
  return 0;
#endif
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::selectedMechanism                                      |
|                                                                              |
| Returns the selected rendering mechanism.                                    |
------------------------------------------------------------------------------*/
IString IDMSourceRenderEvent :: selectedMechanism() const
{
  return this->strSelMechanism;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::selectedFormat                                         |
|                                                                              |
| Returns the selected rendering format.                                       |
------------------------------------------------------------------------------*/
IString IDMSourceRenderEvent :: selectedFormat() const
{
  return this->strSelFormat;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::targetName                                             |
|                                                                              |
| Returns the target "render to" name.                                         |
------------------------------------------------------------------------------*/
IString IDMSourceRenderEvent :: targetName() const
{
  return this->strTgtName;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::targetWindow                                           |
|                                                                              |
| Returns a pointer to the target window.                                      |
------------------------------------------------------------------------------*/
IWindow* IDMSourceRenderEvent :: targetWindow() const
{
#ifdef IC_PMWIN
  return IWindow::windowWithHandle( targetWindowHandle() );
#endif
#ifdef IC_MOTIF
  return (IDMMotif::controlWindow(targetWindowHandle())) ;
#endif
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::targetWindowHandle                                     |
|                                                                              |
| Returns the handle of the target window.                                     |
------------------------------------------------------------------------------*/
IWindowHandle IDMSourceRenderEvent::targetWindowHandle() const
{
  return this->tgtWindowHandle;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::canRetry                                               |
|                                                                              |
| Indicates whether renderer processing this event will be allowed to retry if |
| a failure occurs.                                                            |
------------------------------------------------------------------------------*/
bool IDMSourceRenderEvent::canRetry() const
{
  return this->bRetry;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::completion                                             |
|                                                                              |
| Returns the flag which indicates a successful completion.                    |
------------------------------------------------------------------------------*/
IDM::RenderCompletion IDMSourceRenderEvent::completion() const
{
#ifdef IC_PMWIN
  return this->rcCompletion;
#else
  return IDM::fail;
#endif // IC_PMWIN
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::setTargetInfo                                          |
|                                                                              |
| Can be used to pass information to the target.                               |
------------------------------------------------------------------------------*/
IDMSourceRenderEvent& IDMSourceRenderEvent ::
                      setTargetInfo(unsigned long info)
{
#ifdef IC_PM
  dragTransfer()->ulTargetInfo = info;
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::targetInfo                                             |
|                                                                              |
| Returns the target defined information.                                      |
------------------------------------------------------------------------------*/
unsigned long IDMSourceRenderEvent :: targetInfo() const
{
#ifdef IC_PM
  return dragTransfer()->ulTargetInfo;
#else
  return 0;
#endif
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::setRetry                                               |
|                                                                              |
| Sets the retry flag.                                                         |
------------------------------------------------------------------------------*/
IDMSourceRenderEvent& IDMSourceRenderEvent :: setRetry (bool allowRetry)
{
  this->bRetry = allowRetry;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::setCompletion                                          |
|                                                                              |
| Sets the completion flag.                                                    |
------------------------------------------------------------------------------*/
IDMSourceRenderEvent& IDMSourceRenderEvent ::
                      setCompletion(IDM::RenderCompletion code)
{
  this->rcCompletion = code;
  return *this;
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IDMSourceRenderEvent::dragTransfer                                           |
|                                                                              |
| Returns a pointer to the DRAGTRANSFER structure.                             |
------------------------------------------------------------------------------*/
_DRAGTRANSFER* IDMSourceRenderEvent :: dragTransfer() const
{
  return this->pPMDragTransfer;
}
#endif // IC_MOTIFPM

/*------------------------------------------------------------------------------
| IDMSourceDiscardEvent::IDMSourceDiscardEvent                                 |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMSourceDiscardEvent ::
                IDMSourceDiscardEvent (const IEvent &event)  :
                IDMEvent(event),
                pSrcOpH(0),
                discardInd(IDM::sourceDiscards)
{
  /*************************************************************************/
  /* Sets the handle to the source operation for this event.               */
  /*************************************************************************/
#ifdef IC_PM
  setSourceOperation( IDMSourceOperation::operationFrom((_DRAGINFO *)
                                         event.parameter1().asUnsignedLong()) );
#endif
}

/*------------------------------------------------------------------------------
| IDMSourceDiscardEvent::~IDMSourceDiscardEvent                                |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMSourceDiscardEvent :: ~IDMSourceDiscardEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMSourceDiscardEvent::sourceOperation                                       |
|                                                                              |
| Returns the source operation for this event.                                 |
------------------------------------------------------------------------------*/
IDMSourceOperation::Handle IDMSourceDiscardEvent :: sourceOperation() const
{
#ifndef IC_MOTIF
  return this->pSrcOpH;
#else
  return 0;
#endif
}

/*------------------------------------------------------------------------------
| IDMSourceDiscardEvent::setSourceOperation                                    |
|                                                                              |
| Sets the source operation for this event.                                    |
------------------------------------------------------------------------------*/
IDMSourceDiscardEvent& IDMSourceDiscardEvent ::
                setSourceOperation(const IDMSourceOperation::Handle &pSrc)
{
  this->pSrcOpH = pSrc;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourceDiscardEvent::whoDiscards                                           |
|                                                                              |
| Returns the discard indicator that identifies who should discard the item(s).|
------------------------------------------------------------------------------*/
IDM::DiscardIndicator IDMSourceDiscardEvent :: whoDiscards() const
{
  return this->discardInd;
}

/*------------------------------------------------------------------------------
| IDMSourceDiscardEvent::setWhoDiscards                                        |
|                                                                              |
| Set the discard indicator.  Identifies whether the source or target should   |
| discard, or if the operation should be aborted. IDM::targetDiscards is the   |
| default.                                                                     |
------------------------------------------------------------------------------*/
IDMSourceDiscardEvent& IDMSourceDiscardEvent ::
                       setWhoDiscards(DiscardIndicator indicator)
{
  this->discardInd = indicator;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::IDMSourcePrintEvent                                     |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMSourcePrintEvent ::
                IDMSourcePrintEvent (const IEvent &event)  :
                IDMEvent(event),
                pSrcOpH(0),
                printInd(IDM::sourcePrints)
{
  /*************************************************************************/
  /* Sets the handle to the source operation for this event.               */
  /*************************************************************************/
#ifdef IC_PM
  _DRAGITEM *pPMDItem = (_DRAGITEM *)event.parameter1().asUnsignedLong();
  if ( pPMDItem->ulItemID )
  {
    this->pSrcOpH = ((IDMItem *)(pPMDItem->ulItemID))->sourceOperation();
  }
  else
  {
    ITHROWLIBRARYERROR( IC_DRAGITEM_INVALID,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }
#endif
}

/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::~IDMSourcePrintEvent                                    |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMSourcePrintEvent :: ~IDMSourcePrintEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::sourceOperation                                         |
|                                                                              |
| Returns the source operation for this event.                                 |
------------------------------------------------------------------------------*/
IDMSourceOperation::Handle IDMSourcePrintEvent :: sourceOperation() const
{
  return this->pSrcOpH;
}

/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::setSourceOperation                                      |
------------------------------------------------------------------------------*/
IDMSourcePrintEvent& IDMSourcePrintEvent ::
                setSourceOperation(const IDMSourceOperation::Handle &pSrc)
{
  this->pSrcOpH = pSrc;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::whoPrints                                               |
|                                                                              |
| Returns the print indicator that identifies who should print the item(s).    |
------------------------------------------------------------------------------*/
IDM::PrintIndicator IDMSourcePrintEvent :: whoPrints() const
{
  return this->printInd;
}

/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::setWhoPrints                                            |
|                                                                              |
| Set the print indicator.  Identifies whether the source or the target should |
| print, or if the operation should be aborted.  IDM::targetPrints is the      |
| default.                                                                     |
------------------------------------------------------------------------------*/
IDMSourcePrintEvent& IDMSourcePrintEvent ::
                     setWhoPrints(PrintIndicator indicator)
{
  this->printInd = indicator;
  return *this;
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IDMSourcePrintEvent::printDestination                                        |
|                                                                              |
| Returns the handle to the print destination for this event.                  |
------------------------------------------------------------------------------*/
_PRINTDEST* IDMSourcePrintEvent :: printDestination() const
{
  return (_PRINTDEST *)parameter2().asUnsignedLong();
}
#endif // IC_PM

/*------------------------------------------------------------------------------
|  Target Events                                                               |
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
| IDMTargetEvent::IDMTargetEvent                                               |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMTargetEvent :: IDMTargetEvent (const IEvent& event) :
                  IDMEvent( event ),
                  hpsEmph( IPresSpaceHandle() )
{
}

/*------------------------------------------------------------------------------
| IDMTargetEvent::~IDMTargetEvent                                              |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMTargetEvent :: ~IDMTargetEvent( )
{
  /*************************************************************************/
  /* Release the presentation space unless the user has explicitly done so */
  /*************************************************************************/
  if ( this->hpsEmph )
    releasePresSpace();
}

/*------------------------------------------------------------------------------
| IDMTargetEvent::presSpace                                                    |
|                                                                              |
| Acquires and returns the presentation space handle                           |
| to use for drawing the target emphasis.                                      |
------------------------------------------------------------------------------*/
IPresSpaceHandle IDMTargetEvent :: presSpace( )
{
  IMODTRACE_ALL( "IDMTargetEvent::presSpace" );

#ifdef IC_MOTIF
  unsigned long valuemask = 0;     // use defaults XGCValues
  XGCValues values;
  if ( hpsEmph == NULL )
//   hpsEmph = XCreateGC( XtDisplay( (Widget) handle() ),
//                        XtWindow ( (Widget) handle() ),
//                        valuemask,
//                        &values);
     hpsEmph = IPresSpaceHandle( new IXDC( XtDisplay( (Widget) handle() ),
                                           XtWindow ( (Widget) handle() ) ) );

  if (! (IXDC*) hpsEmph )
     ITHROWGUIERROR("IXDC");
#endif // IC_MOTIF

#ifdef IC_PMWIN
  /*************************************************************************/
  /* Get the presentation space unless it has already been acquired.       */
  /*************************************************************************/
  if ( !(this->hpsEmph) )
  {
#ifdef IC_WIN
     HDC hps = IGETPS( handle() );
#else
     HPS hps = DrgGetPS( handle() );
#endif

     if ( !hps )
        ITHROWGUIERROR( "DrgGetPS" );

     /***********************************************************************/
     /* Save it in case we have to free it later for performance reasons.   */
     /***********************************************************************/
     this->hpsEmph = IPresSpaceHandle( hps, this->handle() );
#ifdef IC_PM
     ITRACE_DEVELOP("   presSpaceHandle that was created = " +
                                            IString( this->hpsEmph ));
#endif
  }

#endif // IC_PMWIN

  return this->hpsEmph;
}

/*------------------------------------------------------------------------------
| IDMTargetEvent::releasePresSpace                                             |
|                                                                              |
| Releases the presentation space handle that was used                         |
| for drawing the target emphasis.                                             |
------------------------------------------------------------------------------*/
void IDMTargetEvent :: releasePresSpace( )
{
  IMODTRACE_ALL("IDMTargetEvent::releasePresSpace");
#ifdef IC_PM
  if ( !DrgReleasePS( this->hpsEmph ) )
  {
     ITHROWLIBRARYERROR( IC_RELEASEPSFAIL,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
  }
#endif // IC_PM
#ifdef IC_WIN
  IRELEASEPS( handle(), this->hpsEmph );
#endif // IC_WIN
  // Motif cleanup handled by IPresSpaceHandle.

  this->hpsEmph = IPresSpaceHandle();
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::IDMTargetEnterEvent                                     |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMTargetEnterEvent ::
              IDMTargetEnterEvent (const IEvent &event)  :
              IDMTargetEvent(event),
              dropInd(IDM::ok),
              op(IDMOperation::drag)

{
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::~IDMTargetEnterEvent                                    |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMTargetEnterEvent :: ~IDMTargetEnterEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::position                                                |
------------------------------------------------------------------------------*/
IPoint IDMTargetEnterEvent :: position() const
{
#ifdef IC_WIN
  IPoint pt;
  if ( eventId() == DM_DRAGOVER )
  {
    pt = IPoint( parameter1().lowNumber(),
                 parameter1().highNumber());
    // convert if necessary.
    if ( ICoordinateSystem::applicationOrientation() ==
         ICoordinateSystem::originLowerLeft )
    {
      pt.setY( IWindow::desktopWindow()->size().height() - pt.y() );
    }
  }
  return pt;
#endif               // IC_WIN

#ifdef IC_MOTIFPM
  if ( eventId() == DM_DRAGOVER )
  {
    return IPoint( parameter1().lowNumber(),
                   parameter1().highNumber() );
  }
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             (parameter1().number2() == CN_DRAGOVER ||
              parameter1().number2() == CN_DRAGAFTER) );

    _CNRDRAGINFO *pCnrDInfo = (_CNRDRAGINFO *)parameter2().asUnsignedLong();

#ifdef IC_PM
    if ( !DrgAccessDraginfo( pCnrDInfo->pDragInfo ) )
      ITHROWGUIERROR( "DrgAccessDraginfo" );

    IPoint pos( pCnrDInfo->pDragInfo->xDrop, pCnrDInfo->pDragInfo->yDrop );

    DrgFreeDraginfo( pCnrDInfo->pDragInfo );
    return pos;
#endif // IC_PM
#ifdef IC_MOTIF
    IDragInfoWrapper dragInfoWrapper( *(pCnrDInfo->pDragInfo) );
    dragInfoWrapper.access();
    IPoint pos( dragInfoWrapper.dropPoint() );
    dragInfoWrapper.release();
    return pos;
#endif // IC_MOTIF
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::dropIndicator                                           |
------------------------------------------------------------------------------*/
IDM::DropIndicator IDMTargetEnterEvent :: dropIndicator() const
{
  return this->dropInd;
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::defaultOperation                                        |
------------------------------------------------------------------------------*/
unsigned long IDMTargetEnterEvent :: defaultOperation() const
{
  return this->op;
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::setDropIndicator                                        |
------------------------------------------------------------------------------*/
IDMTargetEnterEvent& IDMTargetEnterEvent ::
                     setDropIndicator(DropIndicator indicator)
{
  this->dropInd = indicator;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::setDefaultOperation                                     |
------------------------------------------------------------------------------*/
IDMTargetEnterEvent& IDMTargetEnterEvent ::
                     setDefaultOperation(unsigned long operation)
{
  this->op = operation;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::isAboveFirst                                            |
|                                                                              |
| Queries whether drag-after has occurred above the first object in the list.  |
------------------------------------------------------------------------------*/
bool IDMTargetEnterEvent :: isAboveFirst() const
{
// Not implemented yet.
#ifdef IC_WIN
  return false;
#endif
#ifdef IC_MOTIFPM
  if ( eventId() == DM_DRAGOVER )
    return false;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             (parameter1().number2() == CN_DRAGOVER ||
              parameter1().number2() == CN_DRAGAFTER) );

    if ( isDragAfter() )
    {
      _CNRDRAGINFO *pCnrDInfo =
                    (_CNRDRAGINFO *)parameter2().asUnsignedLong();
      return ( (unsigned)pCnrDInfo->pRecord == CMA_FIRST );
    }
    else
      return false;
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::isDragAfter                                             |
|                                                                              |
| Queries whether the event is a drag-after or drag-over event.                |
------------------------------------------------------------------------------*/
bool IDMTargetEnterEvent :: isDragAfter() const
{
#ifdef IC_WIN
  if ( eventId() == DM_DRAGOVER )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      IContainerControl *ctrlWin = (IContainerControl *) window();
      if ( ctrlWin->isDetailsView() || ctrlWin->isTextView() ||
           ctrlWin->isNameView() )
      {
        IPoint pos = position();
        IPoint ptDrop = IWindow::mapPoint( pos,
                                 IWindow::desktopWindow()->handle(),
                                 ctrlWin->handle() );    
        
        if ( ctrlWin->isPMCompatible() && ctrlWin->isDetailsView() )
        {
          // If origin lower left we need to add the height of the
          // cnr title otherwise we subtract.
          IPair::Coord coef=-1;
          if ( ICoordinateSystem::isConversionNeeded() )
            coef=1;

          if ( ctrlWin->areDetailsViewTitlesVisible() )
            ptDrop.setY( ptDrop.y() +
                         coef * ctrlWin->detailsTitleRectangle().height() );

          if ( ctrlWin->isTitleVisible() )
            ptDrop.setY( ptDrop.y() + coef * ctrlWin->titleRectangle().height() );
        }       
        
        if ( ctrlWin->objectUnderPoint( ptDrop ) )
          return true;
      }
    }
  }
  return false;
#endif // IC_WIN
#ifdef IC_MOTIFPM
  if ( eventId() == DM_DRAGOVER )
    return false;
  else
  {
    /******************************************************************/
         /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             (parameter1().number2() == CN_DRAGOVER ||
              parameter1().number2() == CN_DRAGAFTER) );

    return ( parameter1().number2() == CN_DRAGAFTER );
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::containerId                                             |
|                                                                              |
| Returns the source container's ID.                                           |
------------------------------------------------------------------------------*/
unsigned long IDMTargetEnterEvent :: containerId() const
{
  // Not implemented yet. Can only do this when within the same
  // process.
  if ( eventId() == DM_DRAGOVER )
    return 0;
#ifdef IC_MOTIFPM
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             (parameter1().number2() == CN_DRAGOVER ||
              parameter1().number2() == CN_DRAGAFTER) );

    return parameter1().number1();
  }
#endif
  return 0 ;
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::object                                                  |
|                                                                              |
| Returns a pointer to the object that the pointing device is over.  Returns   |
| 0 if over "white space" or if non-container window.                          |
------------------------------------------------------------------------------*/
IContainerObject* IDMTargetEnterEvent :: object() const
{
#ifdef IC_WIN
  IMODTRACE_DEVELOP( "IDMTargetEnterEvent::object" );
  if ( eventId() == DM_DRAGOVER )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      IContainerControl *ctrlWin = (IContainerControl *) window();
      IPoint ptDrop = IWindow::mapPoint( position(),
                                IWindow::desktopWindow()->handle(),
                                ctrlWin->handle() );

      IContainerObject *ctrlObj = ctrlWin->objectUnderPoint( ptDrop );
      if ( ctrlObj )
      {
        return ctrlObj;
      }
    }
  }
  return 0;
#endif
#ifdef IC_MOTIFPM
  if ( eventId() == DM_DRAGOVER )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             (parameter1().number2() == CN_DRAGOVER ||
              parameter1().number2() == CN_DRAGAFTER) );

    _CNRDRAGINFO *pCnrDInfo =
                 (_CNRDRAGINFO *)parameter2().asUnsignedLong();
    if ( (pCnrDInfo->pRecord) && !(isAboveFirst()) )
      return IObjFromRec( (IMiniCnrRecord *)pCnrDInfo->pRecord );
    else
      return 0;
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetEnterEvent::container                                               |
|                                                                              |
| Returns a pointer to the target container for which the event occurred.      |
------------------------------------------------------------------------------*/
IContainerControl* IDMTargetEnterEvent :: container() const
{
#ifdef IC_WIN
  if ( eventId() == DM_DRAGOVER )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      return (IContainerControl *) window();
    }
  }
  return 0;
#endif // IC_WIN

#ifdef IC_MOTIFPM
  if ( eventId() == DM_DRAGOVER )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             (parameter1().number2() == CN_DRAGOVER ||
              parameter1().number2() == CN_DRAGAFTER) );

    return (IContainerControl*) this->controlWindow();
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetLeaveEvent::IDMTargetLeaveEvent                                     |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMTargetLeaveEvent :: IDMTargetLeaveEvent (const IEvent &event) :
                       IDMTargetEvent (event)
{
}

/*------------------------------------------------------------------------------
| IDMTargetLeaveEvent::~IDMTargetLeaveEvent                                    |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMTargetLeaveEvent :: ~IDMTargetLeaveEvent ()
{
}


/*------------------------------------------------------------------------------
| IDMTargetDropEvent::IDMTargetDropEvent                                       |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMTargetDropEvent ::IDMTargetDropEvent (const IEvent &event) :
             IDMTargetEvent(event),
             ulTgtInfo(0)
{
#ifdef IC_WIN
  setDropPosition( IPoint( parameter1().lowNumber(),
                           parameter1().highNumber() ) );
	::ShowCursor(TRUE);
#endif // IC_WIN

#ifdef IC_MOTIFPM
   _DRAGINFO *pDI;
  if ( event.eventId() == DM_DROP )
  {
    pDI = (_DRAGINFO *)parameter1().asUnsignedLong();
  }
  else
  {
    _CNRDRAGINFO *pCnrInfo =
                 (_CNRDRAGINFO *)event.parameter2().asUnsignedLong();
    pDI = pCnrInfo->pDragInfo;
  }

#ifdef IC_PM
  if ( !DrgAccessDraginfo( pDI ) )
    ITHROWGUIERROR( "DrgAccessDraginfo" );
  setDropPosition( IPoint( pDI->xDrop, pDI->yDrop ) );
#endif //IC_PM

#ifdef IC_MOTIF
  IDragInfoWrapper dragInfoWrapper( *pDI );
  dragInfoWrapper.access();
  setDropPosition( dragInfoWrapper.dropPoint() );
#endif // IC_MOTIF

#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::~IDMTargetDropEvent                                      |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMTargetDropEvent :: ~IDMTargetDropEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::targetInfo                                               |
|                                                                              |
| Returns the target defined information.                                      |
------------------------------------------------------------------------------*/
unsigned long IDMTargetDropEvent :: targetInfo() const
{
  return this->ulTgtInfo;
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::setTargetInfo                                            |
|                                                                              |
| Sets the target defined information.                                         |
------------------------------------------------------------------------------*/
IDMTargetDropEvent& IDMTargetDropEvent :: setTargetInfo(unsigned long info)
{
  this->ulTgtInfo = info;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::dropPosition                                             |
|                                                                              |
| Returns the position, expressed in desktop coordinates where the drop        |
| occurred.                                                                    |
------------------------------------------------------------------------------*/
IPoint IDMTargetDropEvent :: dropPosition () const
{
#ifdef IC_WIN
  IPoint pt = this->pos;
  // The position is in Window coordinates so convert to PM's if
  // required
  if ( ICoordinateSystem::applicationOrientation() ==
       ICoordinateSystem::originLowerLeft )
  {
    pt.setY( IWindow::desktopWindow()->size().height() - pt.y() );
  }
  return pt;
#endif // IC_WIN

#ifdef IC_MOTIFPM
  return this->pos;
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::setDropPosition                                          |
|                                                                              |
| Sets the position, expressed in desktop coordinates, where the drop occurred.|
------------------------------------------------------------------------------*/
IDMTargetDropEvent& IDMTargetDropEvent :: setDropPosition(const IPoint &pPos)
{
  this->pos = pPos;
  return *this;
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::containerId                                              |
|                                                                              |
| Returns the target container's ID.                                           |
------------------------------------------------------------------------------*/
unsigned long IDMTargetDropEvent :: containerId() const
{
#ifdef IC_WIN
  if ( eventId() == DM_DROP )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      return window()->id();
    }
  }
  return 0;
#endif // IC_WIN

#ifdef IC_MOTIFPM
  if ( eventId() == DM_DROP )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_DROP );

    return parameter1().number1();
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::object                                                   |
|                                                                              |
| Returns a pointer to the object over which the drop-on event occurred.       |
| Returns 0 if drop was on container "white space" or non-container window.    |
------------------------------------------------------------------------------*/
IContainerObject* IDMTargetDropEvent :: object() const
{
#ifdef IC_WIN
  if ( eventId() == DM_DROP )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      IContainerControl *ctrlWin = (IContainerControl *) window();
      IPoint ptDrop = IWindow::mapPoint( dropPosition(),
                              IWindow::desktopWindow()->handle(),
                              ctrlWin->handle() );

      IContainerObject *ctrlObj = ctrlWin->objectUnderPoint( ptDrop );
      if ( ctrlObj )
      {
        return ctrlObj;
      }
    }
  }
  return 0;
#endif // IC_WIN

#ifdef IC_MOTIFPM
  if ( eventId() == DM_DROP )
    return 0;
  else
  {
    /******************************************************************/
    /* Verify that this is the correct message and notification code  */
    /******************************************************************/
    IASSERT( eventId() == WM_CONTROL &&
             parameter1().number2() == CN_DROP );

    _CNRDRAGINFO *pCnrDInfo =
        (_CNRDRAGINFO *)parameter2().asUnsignedLong();
    if ( (pCnrDInfo->pRecord)  &&
         ((unsigned)pCnrDInfo->pRecord != CMA_FIRST) )
      return IObjFromRec( (IMiniCnrRecord *)pCnrDInfo->pRecord );
    else
      return 0;
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetDropEvent::container                                                |
|                                                                              |
| Returns a pointer to the container where the drop-on event occurred.         |
------------------------------------------------------------------------------*/
IContainerControl* IDMTargetDropEvent :: container() const
{
#ifdef IC_WIN
  if ( eventId() == DM_DROP )
  {
    if ( IDMEventData::isContainerControl( window() ))
    {
      return (IContainerControl *) window();
    }
  }
  return 0;
#endif // IC_WIN

#ifdef IC_MOTIFPM
  if ( eventId() == DM_DROP )
     return 0;
  else
  {
     /******************************************************************/
     /* Verify that this is the correct message and notification code  */
     /******************************************************************/
     IASSERT( eventId() == WM_CONTROL &&
              parameter1().number2() == CN_DROP );

     return (IContainerControl*) this->controlWindow();
  }
#endif // IC_MOTIFPM
}

/*------------------------------------------------------------------------------
| IDMTargetHelpEvent::IDMTargetHelpEvent                                       |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMTargetHelpEvent :: IDMTargetHelpEvent (const IEvent &event) :
                      IDMEvent (event)
{
}

/*------------------------------------------------------------------------------
| IDMTargetHelpEvent::~IDMTargetHelpEvent                                      |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMTargetHelpEvent :: ~IDMTargetHelpEvent ()
{
}

/*------------------------------------------------------------------------------
| IDMTargetHelpEvent::operation                                                |
|                                                                              |
| Returns the operation (i.e. move, copy, or link) which can be used to        |
| provide help in the context of the drag operation.                           |
------------------------------------------------------------------------------*/
unsigned long IDMTargetHelpEvent :: operation () const
{
  // TODO : do we need to do anything for IC_WIN ?
  /********************************************************************/
  /* Access the PM DRAGINFO structure which was passed to us.         */
  /********************************************************************/
  _DRAGINFO *pDInfo = (_DRAGINFO *)parameter1().asUnsignedLong();

#ifdef IC_PM
  if ( !DrgAccessDraginfo( pDInfo ) )
    ITHROWGUIERROR( "DrgAccessDraginfo" );

  /********************************************************************/
  /* Get the operation                                                */
  /********************************************************************/
  unsigned long op = pDInfo->usOperation;

  /********************************************************************/
  /* Free the PM DRAGINFO structure and return the operation          */
  /********************************************************************/
  DrgFreeDraginfo( pDInfo );
  return op;
#endif // IC_PM
#ifdef IC_MOTIF
  IDragInfoWrapper dragInfoWrapper( *pDInfo );
  dragInfoWrapper.access();

  /********************************************************************/
  /* Get the operation                                                */
  /********************************************************************/
  unsigned long op = dragInfoWrapper.dragOperation();

  /********************************************************************/
  /* Free the PM DRAGINFO structure and return the operation          */
  /********************************************************************/
  dragInfoWrapper.release();
  return op;
#endif // IC_MOTIF
  return IDMItem::copyable;
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::IDMTargetEndEvent                                         |
|                                                                              |
| Constructor.                                                                 |
------------------------------------------------------------------------------*/
IDMTargetEndEvent :: IDMTargetEndEvent (const IEvent &event) :
                     IDMEvent(event)
                     ,pDMTgtItemHandle(0)
                     ,usCompletion(0)
{
  IMODTRACE_DEVELOP( "IDMTargetEndEvent::IDMTargetEndEvent" );

#ifdef IC_PM
  /************************************************************************/
  /* Get access PM Drag Transfer structure                                */
  /************************************************************************/
  this->pPMDragTransfer = (_DRAGTRANSFER*)event.parameter1().asUnsignedLong();

  if ( !(this->pPMDragTransfer) )
  {
    ITHROWLIBRARYERROR( IC_DRAGTRANSFER_INVALID,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  /************************************************************************/
  /* Find the IDMItem based upon the PM drag item                         */
  /************************************************************************/
  this->pDMTgtItemHandle = targetItemFromPM(dragTransfer()->pditem) ;
  if ( ! pDMTgtItemHandle.valid() )
  {
    ITHROWLIBRARYERROR( IC_DRAGITEM_INVALID,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  /************************************************************************/
  /* Get completion code                                                  */
  /************************************************************************/
  this->usCompletion = event.parameter2().lowNumber();
#endif // IC_PM
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::~IDMTargetEndEvent                                        |
|                                                                              |
| Destructor.                                                                  |
------------------------------------------------------------------------------*/
IDMTargetEndEvent :: ~IDMTargetEndEvent ( )
{
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::dragItem                                                  |
|                                                                              |
| Returns handle of the drag item that source has completed rendering.         |
------------------------------------------------------------------------------*/
IDMItem::Handle IDMTargetEndEvent :: dragItem() const
{
  return this->pDMTgtItemHandle;
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::alternateWindowHandle                                     |
|                                                                              |
| Returns handle of the alternate source window.                               |
------------------------------------------------------------------------------*/
IWindowHandle IDMTargetEndEvent :: alternateWindowHandle() const
{
#ifdef IC_PM
  return dragTransfer()->pditem->hwndItem;
#else
  return 0;
#endif
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::alternateWindow                                           |
|                                                                              |
| Returns a pointer to the alternate source window.                            |
------------------------------------------------------------------------------*/
IWindow* IDMTargetEndEvent :: alternateWindow() const
{
  return IWindow::windowWithHandle( alternateWindowHandle() );
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::targetInfo                                                |
|                                                                              |
| Returns the target defined information.                                      |
------------------------------------------------------------------------------*/
unsigned long IDMTargetEndEvent :: targetInfo() const
{
#ifdef IC_PM
  return dragTransfer()->ulTargetInfo;
#else
  return 0;
#endif
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::targetCanRetry                                            |
|                                                                              |
| Source has completed the rendering operation and will allow the target to    |
| retry its part of the operation if it fails.                                 |
------------------------------------------------------------------------------*/
bool IDMTargetEndEvent :: targetCanRetry ( ) const
{
  if ( this->usCompletion & IDM::retry )
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::renderingFailed                                           |
|                                                                              |
| Source is unable to perform the rendering operation.  The target may be      |
| allowed to retry, but if it chooses not to, it must call                     |
| informSourceOfCompletion() to end the operation.                             |
------------------------------------------------------------------------------*/
bool IDMTargetEndEvent :: renderingFailed ( ) const
{
  if ( this->usCompletion & IDM::fail )
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::targetItemFromPM                                          |
|                                                                              |
| Finds IDMItem based upon the PM drag item.                                   |
------------------------------------------------------------------------------*/
IDMItem::Handle IDMTargetEndEvent ::
                targetItemFromPM(_DRAGITEM *pDragItem)
{
#ifdef IC_PM
  /************************************************************************/
  /* Access the target operation and determine the number of drag items   */
  /************************************************************************/
  IDMTargetOperation::Handle pTgtOp(IDMTargetOperation::targetOperation()) ;
  unsigned count = pTgtOp->numberOfItems();

  /************************************************************************/
  /* Loop thru all the items defined per the drag operation               */
  /************************************************************************/
  for ( unsigned index = count; index; index-- )
  {
    IDMItem::Handle pTgtItem(pTgtOp->item( index )) ;

    /**********************************************************************/
    /* Compare the PM drag item pointers and return if match found        */
    /**********************************************************************/
    if ( pTgtItem->PMDragItem() == pDragItem )
      return pTgtItem;
  }

  /************************************************************************/
  /* No match, return NULL                                                */
  /************************************************************************/
#endif // IC_PM
  return 0;
}

/*------------------------------------------------------------------------------
| IDMTargetEndEvent::dragTransfer                                              |
|                                                                              |
| Returns a pointer to the DRAGTRANSFER structure.                             |
------------------------------------------------------------------------------*/
_DRAGTRANSFER* IDMTargetEndEvent :: dragTransfer() const
{
#ifdef IC_PM
  return this->pPMDragTransfer;
#else
  return 0;
#endif // IC_PM
}
