// Revision: 88 1.8.2.1 source/ui/cnr/icnrobjp.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrobjp.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the container                     *
*   functions declared in icnrobjp.hpp.                                        *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#include <icnrobjp.hpp>

#ifdef IC_WIN
#include <icnrctlw.hpp>
#include <iexcept.hpp>
#include <iset2.h>
extern "C"
{
  #include <iclcnrw.h>
}
#endif // IC_WIN

#ifdef IC_MOTIF
#include <ilanglvl.hpp>

extern "C" {
#include <X11/IntrinsicP.h>     // for CoreP.h
#include <X11/CoreP.h>          // for core.being_destroyed
#include <cnr.h>
#include <icong.h>
}

#include <icnrrec.hpp>
#include <icnrctl.hpp>
#include <icnrctlm.hpp>
#include <icnrobj.hpp>
#include <icnrcol.hpp>
#include <icnrolst.hpp>
#include <iexcept.hpp>
#include <imstring.hpp>

#include <idate.hpp>
#include <itime.hpp>
#include <ireslib.hpp>
#include <ibmpstat.hpp>
#endif // IC_MOTIF

#include <itrace.hpp>           //ER:
#include <icconst.h>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNROBJP_CPP_
  #include <ipagetun.h>
#endif // IC_PAGETUNE

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ICnrObjSet::ICnrObjSet                                                       |
------------------------------------------------------------------------------*/
ICnrObjSet::ICnrObjSet( )
  : ISet < ICnrObjSetInfo* > ( )
{ }


/*------------------------------------------------------------------------------
| ICnrObjSet::~ICnrObjSet                                                      |
------------------------------------------------------------------------------*/
ICnrObjSet::~ICnrObjSet ( )
{
  ICnrObjSetInfo* pCOSInfo;

  Cursor cursor(*this);

  // Delete all ICnrObjSetInfo objects in the set.
  forCursor (cursor)
  {
    pCOSInfo = elementAt(cursor);
    delete pCOSInfo;
  }
  // Now remove them from the set.
  this->removeAll();
}

/*------------------------------------------------------------------------------
| ICnrObjSetInfo::ICnrObjSetInfo                                               |
------------------------------------------------------------------------------*/
ICnrObjSetInfo::ICnrObjSetInfo ( IContainerControl* pCnrCtl,
                                 HTREEITEM          pTreeViewItem,
                                 int                imageListIndex )
  : pContainer ( pCnrCtl ),
    pTVItem    ( pTreeViewItem ),
    imageIndex ( imageListIndex ),
    objectState(0)
{ }


/*------------------------------------------------------------------------------
| ICnrObjSetInfo::~ICnrObjSetInfo                                              |
------------------------------------------------------------------------------*/
ICnrObjSetInfo::~ICnrObjSetInfo ()
{ }


/*------------------------------------------------------------------------------
| ICnrObjSetInfo::ICnrObjSetInfo (copy constructor)                            |
------------------------------------------------------------------------------*/
ICnrObjSetInfo::ICnrObjSetInfo ( const ICnrObjSetInfo& object )
  : pContainer ( object.pContainer ),
    pTVItem    ( object.pTVItem ),
    imageIndex ( object.imageIndex ),
    objectState( object.objectState )

{
}


/*------------------------------------------------------------------------------
| ICnrObjSetInfo::operator=                                                    |
------------------------------------------------------------------------------*/
ICnrObjSetInfo& ICnrObjSetInfo::operator= ( const ICnrObjSetInfo& object )
{
  if (this == &object)
    return *this;
  pContainer = object.pContainer;
  pTVItem    = object.pTVItem;
  imageIndex = object.imageIndex;
  objectState = object.objectState;
  return *this;
}

// mjs begin
/*------------------------------------------------------------------------------
| ICnrObjSetInfo::setObjectState                                               |
------------------------------------------------------------------------------*/
ICnrObjSetInfo& ICnrObjSetInfo::setObjectState( unsigned long stateOff,
                                                unsigned long stateOn,
                                                IContainerObject* pCnrObj )
{
  bool sendEmphMsg(false);

  // If the inuse state is changing send a notification message.
  if (((stateOff & CRA_INUSE) && (objectState & CRA_INUSE)) ||
      ((stateOn  & CRA_INUSE) && ~(objectState & CRA_INUSE)))
    sendEmphMsg = true;

  objectState &= ~stateOff;
  objectState |= stateOn;

  if (sendEmphMsg && pCnrObj)
  {
    NOTIFYRECORDEMPHASIS notifyRecord;

    notifyRecord.hwndCnr       = pContainer->handle();
    notifyRecord.fEmphasisMask = CRA_INUSE;
    notifyRecord.pRecord       = (RECORDCORE*)IRecFromObj(pCnrObj);

    pContainer->owner()->sendEvent( WM_CONTROL,
                               IEventParameter1( pContainer->id(), CN_EMPHASIS ),
                               IEventParameter2( &notifyRecord ));
  }
  return *this;
}
// mjs end

/*------------------------------------------------------------------------------
| ICnrObjSet::treeItemFromObject                                               |
------------------------------------------------------------------------------*/
HTREEITEM ICnrObjSet::treeItemFromObject (IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    return pCOSInfo->pTVItem;
  else
    return 0;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::containsObject                                                   |
------------------------------------------------------------------------------*/
bool ICnrObjSet::containsObject (IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::remove                                                           |
------------------------------------------------------------------------------*/
bool ICnrObjSet::remove (IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo;

  Cursor cursor(*this);
  for(cursor.setToFirst(); cursor.isValid();  )
  {
    pCOSInfo = elementAt(cursor);
    if ( pCnrCtl == pCOSInfo->pContainer)
    {
      removeAt( cursor );
      delete( pCOSInfo );
      cursor.invalidate();
    }
    else
      cursor.setToNext();
  }
  return (false);
}

/*------------------------------------------------------------------------------
| ICnrObjSet::isDropOnable                                                     |
------------------------------------------------------------------------------*/
bool ICnrObjSet::isDropOnable (IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    return (bool)(pCOSInfo->objectState & CRA_DROPONABLE);
  else
    return false;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::isReadOnly                                                       |
------------------------------------------------------------------------------*/
bool ICnrObjSet::isReadOnly (IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    return (bool)(pCOSInfo->objectState & CRA_RECORDREADONLY);
  else
    return false;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::isInUse                                                          |
------------------------------------------------------------------------------*/
bool ICnrObjSet::isInUse (IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    return (bool)(pCOSInfo->objectState & CRA_INUSE);
  else
    return false;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::updateTreeItem                                                   |
------------------------------------------------------------------------------*/
ICnrObjSet& ICnrObjSet::updateTreeItem( IContainerControl* pCnrCtl,
                                        HTREEITEM treeItem )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    *pCOSInfo = ICnrObjSetInfo( pCnrCtl, treeItem, pCOSInfo->imageIndex );

  return *this;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::imageIndexFromCnr                                                |
------------------------------------------------------------------------------*/
int ICnrObjSet::imageIndexFromCnr ( IContainerControl* pCnrCtl )
{
  ICnrObjSetInfo* pCOSInfo(objectFromCnr(pCnrCtl));

  if (pCOSInfo)
    return pCOSInfo->imageIndex;
  else
    return 0;
}

/*------------------------------------------------------------------------------
| ICnrObjSet::objectFromCnr                                                    |
------------------------------------------------------------------------------*/
ICnrObjSetInfo* ICnrObjSet::objectFromCnr ( IContainerControl* pCnrCtl  )
{
  ICnrObjSetInfo* pCOSInfo(0);

  Cursor cursor(*this);
  forCursor (cursor)
  {
    pCOSInfo = elementAt(cursor);
    if ( pCnrCtl == pCOSInfo->pContainer)
      break;
    else
      pCOSInfo = 0;
  }

  if (pCOSInfo)
    return pCOSInfo;
  else
    return 0;
}
#endif // IC_WIN

/*------------------------------------------------------------------------------
| ICnrObjPrivateData::ICnrObjPrivateData                                       |
------------------------------------------------------------------------------*/
ICnrObjPrivateData::ICnrObjPrivateData(IContainerObject* newObject )
   : pObject(newObject)
    ,hptrInUseIcon(0)
    ,hptrRegularIcon(0)
#ifdef IC_WIN
    ,pCnrSet( new ICnrObjSet() )
#endif
{ }


/*------------------------------------------------------------------------------
| ICnrObjPrivateData::~ICnrObjPrivateData                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
ICnrObjPrivateData::~ICnrObjPrivateData( )
{
#ifdef IC_WIN
  if (pCnrSet)
  {
    delete pCnrSet;
  }
#endif // IC_WIN
#ifdef IC_MOTIF
  ContainedInListType::Cursor cursor(containedInList) ;
  forCursor(cursor)
  {
     ContainedInElement* pElement = containedInList.elementAt(cursor) ;
     delete pElement ;
  }
  containedInList.removeAll() ;
#endif // IC_MOTIF
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ICnrObjPrivateData::listItemFromCnr                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
int ICnrObjPrivateData::listItemFromCnr( const IContainerControl* pCnrCtl,
                                         const IContainerObject*  pObject,
                                         int   startFrom )
{
  int index(-1);

  if ( pCnrCtl )
  {
    LV_FINDINFO lvFindInfo;

    // Get the index of the lv_item
    lvFindInfo.flags  = LVFI_PARAM;
    lvFindInfo.lParam = (LPARAM)pObject;

    // Prime the lv item
    if (pCnrCtl->ppd->pListview)
      index = IContainerControl::ListViewFindItem(
                                 pCnrCtl->ppd->pListview->handle(),
                                 startFrom, &lvFindInfo );
    else
      index = -1;
  }

  return index;
}
#endif // IC_WIN


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ContainedInElement ctor                                                      |
------------------------------------------------------------------------------*/
ContainedInElement::ContainedInElement (IContainerControl* aContainer,
                                        IWindowHandle anIconGadget,
                                        IContainerObject* pObject)
       : iconGadget(anIconGadget)
         , containerKey(aContainer)
         , pParentElement(NULL)
         , pPreviousElement(NULL)
         , pNextElement(NULL)
         , pFirstChildElement(NULL)
         , pLastChildElement(NULL)
         , pCnrObject(pObject)
{ }

/*------------------------------------------------------------------------------
| ContainedInElement dtor                                                      |
------------------------------------------------------------------------------*/
ContainedInElement:: ~ContainedInElement ( )
{ }

/*------------------------------------------------------------------------------
| ContainedInElement getKey                                                    |
------------------------------------------------------------------------------*/
IContainerControl* const& ContainedInElement::getKey() const
{
return containerKey;
}

/*------------------------------------------------------------------------------
| setRefreshOn                                                                     |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData :: setRefreshOn(bool bRefreshState)
{
   ::Boolean bOn = (bRefreshState ? True : False) ;
   ContainedInListType::Cursor cursor(containedInList) ;
   forCursor(cursor)
      XtVaSetValues( (Widget)containedInList.elementAt(cursor)->iconGadget,
                     XmNrefreshGadget, &bOn,
                     NULL ) ;
}

// the following functions are to maintain the linked list of ContainedInElement
// objects used for processing of IContainerControl::Cursor
/*------------------------------------------------------------------------------
| appendToList                                                                 |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData::appendToList(ContainedInElement* pObj, IContainerControl* pCnr)
{
   ContainedInElement* pElement = IContainedInElementFromObj(pObject, pCnr) ;
   pElement->pLastChildElement = pObj ;
   if (pElement->pFirstChildElement == NULL)
      pElement->pFirstChildElement = pObj ;
} // appendToList

/*------------------------------------------------------------------------------
| insertToList                                                                 |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData::insertToList(ContainedInElement* pObj, IContainerControl* pCnr)
{
   ContainedInElement* pElement = IContainedInElementFromObj(pObject, pCnr) ;
   pElement->pFirstChildElement = pObj ;
   if (pElement->pLastChildElement == NULL)
      pElement->pLastChildElement = pObj ;
} // insertToList

/*------------------------------------------------------------------------------
| firstChild                                                                   |
------------------------------------------------------------------------------*/
ContainedInElement* ICnrObjPrivateData::firstChild(IContainerControl* pCnr)
{
   ContainedInElement* pElement = IContainedInElementFromObj(pObject, pCnr) ;
   return pElement->pFirstChildElement ;
} // firstChild

/*------------------------------------------------------------------------------
| lastChild                                                                    |
------------------------------------------------------------------------------*/
ContainedInElement* ICnrObjPrivateData::lastChild(IContainerControl* pCnr)
{
   ContainedInElement* pElement = IContainedInElementFromObj(pObject, pCnr) ;
   return pElement->pLastChildElement ;
} // lastChild

/*------------------------------------------------------------------------------
| nextObject                                                                   |
------------------------------------------------------------------------------*/
IContainerObject* ICnrObjPrivateData::nextObject(IContainerControl* pCnr)
{
   IContainerObject* next = NULL ;

   ContainedInElement* element = IContainedInElementFromObj(pObject, pCnr) ;
   if (element)
   {
      ContainedInElement* nextElement = element->pNextElement ;
      if (nextElement)
         next = nextElement->pCnrObject ;
   }
   return  next ;
} // nextObject

/*------------------------------------------------------------------------------
| previousObject                                                               |
------------------------------------------------------------------------------*/
IContainerObject* ICnrObjPrivateData::previousObject(IContainerControl* pCnr)
{
   IContainerObject* previous = NULL ;

   ContainedInElement* element = IContainedInElementFromObj(pObject, pCnr) ;
   if (element)
   {
      ContainedInElement* previousElement = element->pPreviousElement ;
      if (previousElement)
         previous = previousElement->pCnrObject ;
   }
   return  previous ;
} // previousObject

/*------------------------------------------------------------------------------
| firstChildObject                                                             |
------------------------------------------------------------------------------*/
IContainerObject* ICnrObjPrivateData::firstChildObject(IContainerControl* pCnr)
{
   IContainerObject* child = NULL ;

   ContainedInElement* element = IContainedInElementFromObj(pObject, pCnr) ;
   if (element)
   {
      ContainedInElement* childElement = element->pFirstChildElement ;
      if (childElement)
         child = childElement->pCnrObject ;
   }
   return  child ;
} // firstChildObject

/*------------------------------------------------------------------------------
| lastChildObject                                                              |
------------------------------------------------------------------------------*/
IContainerObject* ICnrObjPrivateData::lastChildObject(IContainerControl* pCnr)
{
   IContainerObject* child = NULL ;

   ContainedInElement* element = IContainedInElementFromObj(pObject, pCnr) ;
   if (element)
   {
      ContainedInElement* childElement = element->pLastChildElement ;
      if (childElement)
         child = childElement->pCnrObject ;
   }
   return  child ;
} // lastChildObject

/*------------------------------------------------------------------------------
| addToCnr                                                                     |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData :: addToCnr(IContainerControl* addTo,
                                    RECORDINSERT* precordinsert)
{
  IFUNCTRACE_DEVELOP() ;
  IMiniCnrRecord* pcnrrec = IRecFromObj(pObject);
  Widget parentIG = 0;
  Widget afterGadget = 0 ;
  int posIndex;
  IContainerObject* afterObject = NULL ;
  IContainerObject* parent = IObjFromRec(precordinsert->pRecordParent);
  if (parent != 0)
    parentIG = IGadgetFromObj(parent, addTo);

  // Determine where to place new gadget
  if (precordinsert->pRecordOrder == (IMiniCnrRecord*)CMA_END)
    // There's something strange in Container.c
    // It defines UNSPECIFIED_POSITION_INDEX to be -32767
    // Container.h defines XmUNSPECIFIED_POSITION as -1
    // Code in InsertNode() checks for equality to UNSPECIFIED_POSITION_INDEX
    // There's some reasone why they didn't use XmUNSPECIFIED_POSITION, but
    // I can't figure it out.
    // So, for now use UNSPECIFIED_POSITION_INDEX as value of posIndex to
    // prevent objects that are supposed to go to the end, ending up as the
    // first objects.
    posIndex = -32767;
  else if (precordinsert->pRecordOrder == (IMiniCnrRecord*)CMA_FIRST)
    posIndex = 0;
  else
  {
    // Wants to put it after the specified object
    int afterPosIndex;
    afterObject = IObjFromRec(precordinsert->pRecordOrder);
    afterGadget = IGadgetFromObj(afterObject, addTo);
    // add one to afterPosIndex and rely on InsertNode() in Container.c
    // to get in right spot in its linked list of Cwid's
    XtVaGetValues(afterGadget, XmNpositionIndex, &afterPosIndex, NULL);
    posIndex = afterPosIndex + 1;
  }


  // Refresh makes no difference and causes problems, skip for now

  // if (!precordinsert->fInvalidateRecord)
  //   XuiclSetRefreshOn(addTo->handle(), ::False);


  // Note that we store the IContainerObject pointer in XmNuserData.
  // This needed to support ICnrEnterEvent::object().
  // It's safe to use XmNuserData for now because UICL is only code
  // that can possibly access the IconGadget.

  Arg args[20];
  int n = 0;
  IMString labelString(pcnrrec->strIcon);

  XtSetArg(args[n], XmNlabelString, (XmString)labelString); n++;
  XtSetArg(args[n], XmNentryParent, parentIG); n++;
  XtSetArg(args[n], XmNpositionIndex, posIndex); n++;
  XtSetArg(args[n], XmNiconViewX, pcnrrec->ptlIcon.x); n++;
  XtSetArg(args[n], XmNiconViewY, pcnrrec->ptlIcon.y); n++;
  XtSetArg(args[n], XmNuserData, (XtPointer)pObject); n++;
  unsigned char usSelectedEmphasis ;
  if( pcnrrec->flRecordAttr & CRA_SELECTED )
     usSelectedEmphasis = XmSELECTED ;
  else
     usSelectedEmphasis = XmNOT_SELECTED ;
  XtSetArg(args[n], XmNvisualEmphasis, usSelectedEmphasis); n++;

  // Saw performance problems if set the icon resources when hptrIcon was 0
  if (pcnrrec->hptrIcon)
  {
     Pixmap shapeMask =
       IBitmapStaticPtr::shapeMaskFor(pcnrrec->hptrIcon);
     XtSetArg(args[n], XmNlargeIconPixmap, pcnrrec->hptrIcon); n++;
     XtSetArg(args[n], XmNsmallIconPixmap, pcnrrec->hptrIcon); n++;
     if (shapeMask)
     {
       XtSetArg(args[n], XmNsmallIconMask, shapeMask); n++;
       XtSetArg(args[n], XmNlargeIconMask, shapeMask); n++;
     }
  }

  Widget
  theGadget = XtCreateWidget(
     ((char*)pcnrrec->strIcon ? (char*) (pcnrrec->strIcon) : "IContainerObject"),
                                 xmIconGadgetClass,
                                 addTo->handleForChildCreation(),
                                 args, n);
  XtAddCallback(theGadget,
                XmNinUseEmphasisCallback,
                iwindowMotifCallback,
                addTo);
  // Skip the below optimization, since it only makes things slower.
  // Always call XtManageChild instead.
  // if (addTo->isRefreshOn())
  // {
  //    XtManageChild(theGadget);
  // }
  // else {
  //    if (addTo->ppd->pwidgetList == 0)
  //       addTo->ppd->pwidgetList = new IWidgetList(1024);
  //    addTo->ppd->pwidgetList->add(theGadget);
  // }
  XtManageChild(theGadget);

  // add new element into our linked list of objects in the cnr.
  ContainedInElement*  pObjElement = new ContainedInElement(addTo, theGadget, pObject) ;

  if (parent == NULL && afterObject)
  {
     // we came here from a call to addObjectAfter(new, after, top level)
     ContainedInElement* pAfterElement = IContainedInElementFromObj(afterObject, addTo) ;
     if (pAfterElement)
     {
        pObjElement->pPreviousElement = pAfterElement ;
        pObjElement->pNextElement     = pAfterElement->pNextElement ;
        ContainedInElement* pElement = pAfterElement->pNextElement ;
        if (pElement)
           pElement->pPreviousElement = pObjElement ;
        pAfterElement->pNextElement = pObjElement ;
        if (addTo->ppd->lastObject() == afterObject)
           addTo->ppd->lastObject(pObject) ;
     }
  }

  else if (parent == NULL && afterObject == NULL)
  {                                                                     
     // we came here from a call to addObject(new) ;
     // posIndex tells us to add to beginning or end of list
     if (posIndex == 0)
     {
        IContainerObject* pFirstObj = addTo->ppd->firstObject() ;
        if (pFirstObj) {
           ContainedInElement* pFirstElement = IContainedInElementFromObj(pFirstObj, addTo) ;
           if (pFirstElement)
           {
              pFirstElement->pPreviousElement = pObjElement ;
              pObjElement->pNextElement = pFirstElement ;
           }
        }
        addTo->ppd->firstObject(pObject) ;
     }
     else
     {
        IContainerObject* pLastObj = addTo->ppd->lastObject() ;
        if (pLastObj) {
           ContainedInElement* pLastElement = IContainedInElementFromObj(pLastObj, addTo) ;
           if (pLastElement)
           {
              pLastElement->pNextElement = pObjElement ;
              pObjElement->pPreviousElement = pLastElement ;
           }
        }
        addTo->ppd->lastObject(pObject) ;
     }
  }

  else if (parent && afterObject == NULL)
  {
     // we came here from a call to addObject(new, parent) or addObjectAfter(new, null, parent)
     // posIndex tells us to add to beginning or end of list
     if (posIndex == 0)
     {
        ContainedInElement* pFirstElement = parent->ppd->firstChild(addTo) ;
        if (pFirstElement)
        {
           pFirstElement->pPreviousElement = pObjElement ;
           pObjElement->pNextElement = pFirstElement ;
        }
        pObjElement->pParentElement = IContainedInElementFromObj(parent, addTo) ;
        parent->ppd->insertToList(pObjElement, addTo) ;
     }
     else
     {
        ContainedInElement* pLastElement = parent->ppd->lastChild(addTo) ;
        if (pLastElement)
        {
           pLastElement->pNextElement = pObjElement ;
           pObjElement->pPreviousElement = pLastElement ;
        }
        pObjElement->pParentElement = IContainedInElementFromObj(parent, addTo) ;
        parent->ppd->appendToList(pObjElement, addTo) ;
     }
  }

  else if (parent && afterObject)
  {
     // we came from a call to addObjectAfter(new, after, parent) 
     ContainedInElement* pAfterElement = IContainedInElementFromObj(afterObject, addTo) ;
     if (pAfterElement)
     {
        pObjElement->pPreviousElement = pAfterElement;
        pObjElement->pNextElement = pAfterElement->pNextElement ;
        ContainedInElement* pElement = pAfterElement->pNextElement ;
        if (pElement)
           pElement->pPreviousElement = pObjElement ;
        pAfterElement->pNextElement = pObjElement ;
     }
     pObjElement->pParentElement = IContainedInElementFromObj(parent, addTo) ;
     if (afterObject == IObjFromGadget(parent->ppd->lastChild(addTo)->iconGadget))
        parent->ppd->appendToList(pObjElement, addTo) ;
  }

  containedInList.add(pObjElement);

  // if a root object update its entryDetail resource for possible display
  // of details view
  if (parent == 0)
    updateEntryDetail(addTo);

  // Have seen refresh problems with details views if you addObject
  // after having deleted all objects. This fixes it.
  if (addTo->isDetailsView() && addTo->isRefreshOn())
    XmContainerRelayout(addTo->handleForChildCreation());


  // if (!precordinsert->fInvalidateRecord)
  //   XuiclSetRefreshOn(addTo->handle(), ::True);

}

/*------------------------------------------------------------------------------
| removeFromCnr                                                                |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData :: removeFromCnr(
                     IContainerControl* removeFrom,
                     unsigned long flags)
{
  IFUNCTRACE_DEVELOP() ;

  // flags may contain CMA_IVALIDATE indicating immediate refresh
  // if (!flags & CMA_INVALIDATE)
  //    XuiclSetRefreshOn(removeFrom->handle(), ::False);

  // Following will throw an exception if pObj not in removeFrom
  Widget iconG = IGadgetFromObj(pObject, removeFrom);

  XtRemoveCallback(
     iconG,
     XmNinUseEmphasisCallback,
     iwindowMotifCallback,
     removeFrom);

  // First recursively remove any children of this IconGadget
  // use ObjectCursor instead of calling Motif now that collections correctly
  // cover hidden objects as well 

  //WidgetList widgetList = NULL;
  //int childCount = XmContainerGetItemChildren(removeFrom->handleForChildCreation(),
  //                                            iconG, &widgetList);
  //for(int i = 0; i < childCount; i++)
  //{
  //  IContainerObject* pObj = IObjFromGadget(widgetList[i]);
  //  pObj->ppd->removeFromCnr(removeFrom, 0);
  //}
  //if( widgetList ) {
  //    XtFree( (char*) widgetList );
  //}

  ICnrObjectSet setOfChildren = removeFrom->immediateDescendentsOf(pObject) ;
  if (! setOfChildren.isEmpty())
  {
     ICnrObjectSet::Cursor cursor(setOfChildren) ;
     for (cursor.setToFirst(); cursor.isValid() ; cursor.setToNext() )
     {
        IContainerObject* pObj = cursor.element() ;
        pObj->ppd->removeFromCnr(removeFrom, 0);
     }
  }

  if ( ( (Widget) removeFrom->handle() )->core.being_destroyed )
  {
     // HACK ALERT:
     // If we are removing the object due to cleanUp processing for the
     // Destroy of the Container widget, then since the Container widget
     // is being destroyed the Intrinsics won't call ChangeManaged for an
     // XtUnmanageChild call. So, to get this IconGadget out of the
     // linked list in Container, call a hack routine that will call
     // SeverNode. We need to be sure it's out of the list so that
     // the ObjectCursor we use to find all items to delete doesn't keep
     // hitting the same deleted item.
     XuiclSeverNode(iconG);
  }
  else
  {
     // Unmanage the icon gadget widget to remove it from the display.
     // Otherwise, relying solely on XtDestroyWidget to do this can result
     // in a lengthy delay before phase 2 of the Xt destroy processing
     // actually destroys the window for the container object widget.
     XtUnmanageChild( iconG );
  }

  /* moved from before XuiclSeverNode because removal of selected item
   * might invoke callback which performs CM_QUERYCONTAINER, which queries
   * collection.
   */
  removeFromCollection(removeFrom);

  XtDestroyWidget(iconG);

  if (removeFrom->isRefreshOn())
    XmContainerRelayout(removeFrom->handleForChildCreation());    //ER: force lines drawing
                                                  //    in tree view

  // if (!flags & CMA_INVALIDATE)
  //    XuiclSetRefreshOn(removeFrom->handle(), ::True);
}

/*------------------------------------------------------------------------------
| removeFromCollection                                                         |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData::removeFromCollection(IContainerControl* pCnr)
{
   
   ContainedInListType::Cursor cursor(containedInList) ;
   if (containedInList.locateElementWithKey(pCnr, cursor))
   {
      ContainedInElement* pElement = containedInList.elementAt(cursor) ;
      ContainedInElement* pPrevious = pElement->pPreviousElement ;
      ContainedInElement* pNext = pElement->pNextElement ;
      // fix up the pointers in the previous and next elements in the linked list
      if (pPrevious)
         pPrevious->pNextElement = pElement->pNextElement ;
      if (pNext)
         pNext->pPreviousElement = pElement->pPreviousElement ;

      // fix up the pointers in the parent object in the linked list
      if (pElement->pParentElement)
      {
         if (pElement->pParentElement->pFirstChildElement == pElement)
            pElement->pParentElement->pFirstChildElement = pElement->pNextElement ;
         if (pElement->pParentElement->pLastChildElement == pElement)
            pElement->pParentElement->pLastChildElement = pElement->pPreviousElement ;
      }

      // if we are deleting the first or last object in the container
      // need to update the cnr first and last element pointers
      if (pCnr->ppd->lastObject() == pElement->pCnrObject)
         if (pPrevious)
            pCnr->ppd->lastObject(pPrevious->pCnrObject) ;
         else
            pCnr->ppd->lastObject( (IContainerObject*) NULL) ;
      if (pCnr->ppd->firstObject() == pElement->pCnrObject)
         if (pNext)
            pCnr->ppd->firstObject(pNext->pCnrObject) ;
         else
            pCnr->ppd->firstObject((IContainerObject*) NULL) ;

      containedInList.removeAt(cursor);
      delete pElement ;
   }
} // removeFromCollection

/*------------------------------------------------------------------------------
| getText                                                                      |
------------------------------------------------------------------------------*/
static IString getText(IContainerColumn* column,
                       ICnrObjPrivateData* ppd)
{
  if (column->isString())
    return column->dataAsString(ppd->pObject);

  if (column->isNumber())
    return IString(column->dataAsNumber(ppd->pObject));

  if (column->isDate())
    return column->dataAsDate(ppd->pObject).asString();

  if (column->isTime())
    return column->dataAsTime(ppd->pObject).asString("%H:%M");

  // No way we can display if isIconHandle is true or
  // if we got some new type of column we don't know about..

  return IString(" ");
};

/*------------------------------------------------------------------------------
| updateEntryDetail                                                            |
------------------------------------------------------------------------------*/
void ICnrObjPrivateData :: updateEntryDetail(IContainerControl *activeCnr)
{
  IFUNCTRACE_DEVELOP();

  unsigned long numCols = activeCnr->ppd->columnSeq.numberOfElements();

  if (numCols != 0)
  {
    // Allocate an XmStringTable for setting IconGadget's entryDetail.
    XmStringTable strTable =
      (XmStringTable) new char[numCols * sizeof(XmString)];

    ColumnSequenceType::Cursor cursor(activeCnr->ppd->columnSeq);
    int visibleCols = 0;
    for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
    {
       IContainerColumn* curCol = activeCnr->ppd->columnSeq.elementAt(cursor);
       if (curCol->isVisible())
         // Use XmStringCreateLtoR so that \n produces multi-lines
         strTable[visibleCols++] =
           XmStringCreateLtoR(getText(curCol, this), XmFONTLIST_DEFAULT_TAG);
    }

    // Set the IconGadget's entryDetail resource
    XtVaSetValues(containedInList.elementWithKey(activeCnr)->iconGadget,
                  XmNentryDetail, strTable,
                  NULL);

    // free the storage used by the XmStringTable
    for (int i=0; i<visibleCols; i++)
      XmStringFree(strTable[i]);
    delete [] strTable;
  }
}

/*------------------------------------------------------------------------------
| IGadgetFromObj                                                               |
------------------------------------------------------------------------------*/
IWindowHandle IGadgetFromObj(IContainerObject*  pObj,
                             IContainerControl* pCnr,
                             bool               throwIfNotFound)
{
  // It's very likely this can be called when pObj is not in pCnr
  // (IContainerControl::containsObject() is best example of this)
  // So we need to avoid throwing spurious exceptions, thus a calling
  // routine can set throwIfNotFound to false and get 0 returned.

  if (pObj->ppd->containedInList.containsElementWithKey(pCnr))
    return(pObj->ppd->containedInList.elementWithKey(pCnr)->iconGadget); 
  else
    if(throwIfNotFound)
    {
      ITHROWGUIERROR("Specified object not found in Container");
      // next line just here to humor the compiler so it won't give a
      // warning about return type of IWindowHandle expected
      return IWindowHandle(0);
    }
    else
      return IWindowHandle(0);
}

/*------------------------------------------------------------------------------
| IContainedInElementFromObj                                                   |
------------------------------------------------------------------------------*/
ContainedInElement* IContainedInElementFromObj(IContainerObject*  pObj,
                                         IContainerControl* pCnr)
{
  if (pObj->ppd->containedInList.containsElementWithKey(pCnr))
     return(pObj->ppd->containedInList.elementWithKey(pCnr));
  else
     return 0 ;
}

/*------------------------------------------------------------------------------
| IObjFromGadget                                                               |
------------------------------------------------------------------------------*/
IContainerObject* IObjFromGadget(IWindowHandle iconG)
{
  IContainerObject* pObj = 0;
  XtVaGetValues(iconG, XmNuserData, &pObj, NULL);
  return pObj;
}


#endif // IC_MOTIF
