// Revision: 95 1.17.2.1 source/ui/cnr/icnrctlw.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrctlw.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the native window container       *
*   functions declared in icnrctlw.hpp.                                        *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

#include <icnrctlw.hpp>

#ifdef IC_WIN

#include <iapp.hpp>
#include <ibidiset.hpp>
#include <icconst.h>
#include <icnrobjp.hpp>
#include <icoordsy.hpp>
#include <idmhndlr.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <imle.hpp>
#include <ireslib.hpp>
#include <istattxt.hpp>
#include <itimer.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTLW_CPP_
  #include <ipagetun.h>
#endif

VOID FAR CreateNLSDateString  ( LPCDATE pDate,  LPSTR pszDate  );
VOID FAR CreateNLSTimeString  ( LPCTIME pTime,  LPSTR pszTime  );
bool     CreateNLSNumberString( DWORD dwNumber, LPSTR lpNumber );

/*------------------------------------------------------------------------------
| ICnrControlData::ICnrControlData                                             |
|                                                                              |
| Control data default constructor.                                            |
------------------------------------------------------------------------------*/
ICnrControlData::ICnrControlData( )
  : pCnr( 0 ),
    cclStyle( 0 ),
    pmCompatible( true ),
    showTitle( false ),
    showTitleSeparator( false ),
    doubleClickOccurred( false ),
    bCheckingDragStart( false ),
    needArrange( false ),
    collapseFirstColumn(false),
    useScaledImages( false ),                                        /* D7877 */
    expandNotificationDone( false ),
    pListview( 0 ),
    pTreeview( 0 ),
    pActiveview( 0 ),
    pTitle( 0 ),
    pEditColumn( 0 ),
    pEditObject( 0 ),
    pDetailsEdit( 0 ),
    fTimer( 0 ),
    largeImages( 0 ),
    scaledImages( 0 ),                                               /* D7877 */
    smallImages( 0 ),
    noImages( 0 ),
    inUseImage( 0 ),
    scaledInUseImage( 0 ),                                           /* D7877 */
    smallInUseImage( 0 ),
    av( ICnrControlData::none ),
    pointerSet(0),
    columnSet(0),
    columnsToDelete(0),
    pointLButtonDown()
{}

/*------------------------------------------------------------------------------
| ICnrControlData::ICnrControlData                                             |
|                                                                              |
| Control data informed constructor.                                           |
------------------------------------------------------------------------------*/
ICnrControlData::ICnrControlData( IContainerControl* cnr,
                                  unsigned long      ulStyle,
                                  IContainerControl::Attribute attribute )
  : pCnr( cnr ),
    cclStyle( ulStyle ),
    pmCompatible( false ),
    showTitle( false ),
    showTitleSeparator( false ),
    doubleClickOccurred( false ),
    bCheckingDragStart( false ),
    needArrange( false ),
    collapseFirstColumn(false),
    useScaledImages( false ),                                        /* D7877 */
    expandNotificationDone( false ),
    pListview( 0 ),
    pTreeview( 0 ),
    pActiveview( 0 ),
    pTitle(new IStaticText( CID_CNRTITLEWND,
                            cnr,
                            cnr,
                            IRectangle(),
                            IStaticText::defaultStyle() | IStaticText::border3D )),
    pEditColumn( 0 ),
    pEditObject( 0 ),
    pDetailsEdit( 0 ),
    fTimer( 0 ),
    largeImages(ImageList_Create    ( GetSystemMetrics (SM_CXICON),
                                      GetSystemMetrics (SM_CYICON),
                                      ILC_MASK, 40, 5 )),            /* D7877 */
    scaledImages( 0 ),
    smallImages(ImageList_Create    ( GetSystemMetrics (SM_CXSMICON),
                                      GetSystemMetrics (SM_CYSMICON),
                                      ILC_MASK, 40, 5 )),            /* D7877 */
    noImages   (ImageList_Create    ( 1,   1, ILC_MASK, 1,  0 )),    /* D7877 */
    inUseImage (ImageList_Create    ( GetSystemMetrics (SM_CXICON),
                                      GetSystemMetrics (SM_CYICON),
                                      ILC_MASK, 1,  0 )),            /* D7877 */
    scaledInUseImage( 0 ),                                           /* D7877 */
    smallInUseImage(ImageList_Create( GetSystemMetrics (SM_CXSMICON),
                                      GetSystemMetrics (SM_CYSMICON),
                                      ILC_MASK, 1,  0 )),            /* D7877 */
    av( ICnrControlData::none ),
    pointerSet(new ICnrPointerSet()),
    columnSet(new IColumnSet()),
    columnsToDelete(new IColDeleteSet()),
    pointLButtonDown()
{
  pTitle->setAlignment( IStaticText::centerCenter );
  pTitle->setBackgroundColor( IGUIColor( IGUIColor::menuBgnd ));

  // Initialize the container info structure.

  cnrInfo.flWindowAttr = attribute.asUnsignedLong();
  cnrInfo.strCnrTitle  = 0;

  // Use the largeImages ImageList to set the size of the icons
  int width, height;
  if (ImageList_GetIconSize( largeImages, &width, &height) )
  {
    cnrInfo.slBitmapOrIcon.cx = width;
    cnrInfo.slBitmapOrIcon.cy = height;
  }
  // Initialize tree values to zero - change later when view is established in
  // setWindowAttr
  cnrInfo.slTreeBitmapOrIcon.cx = 0;
  cnrInfo.slTreeBitmapOrIcon.cy = 0;

  ImageList_AddIcon( inUseImage,      IApplication::current().resourceLibrary().loadIcon(IC_INUSEICON) );
  ImageList_AddIcon( smallInUseImage, IApplication::current().resourceLibrary().loadIcon(IC_SMALLINUSEICON) );
}

/*------------------------------------------------------------------------------
| ICnrControlData::~ICnrControlData                                            |
|                                                                              |
| Control data destructor.                                                     |
------------------------------------------------------------------------------*/
ICnrControlData::~ICnrControlData( )
{
  if (pListview)
  {
    delete pListview;
  }
  if (pTreeview)
  {
    delete pTreeview;
  }
  if (pTitle)
  {
    delete pTitle;
  }

// Defect icl2_8095:  Do not delete this
// object since it already is an autodelete object.
//  if (pDetailsEdit)
//    delete pDetailsEdit;

  if (fTimer)
  {
    fTimer->stop() ;
    delete fTimer;
  }
  if (pointerSet)
  {
    delete pointerSet;
  }
  if (columnSet)
  {
    delete columnSet;
  }
  if (columnsToDelete)
  {
    delete columnsToDelete;
  }
  if (largeImages)
  {
    ImageList_Destroy( largeImages );
  }
  if (scaledImages)                                                  /* D7877 */
  {
    ImageList_Destroy( scaledImages );
  }
  if (smallImages)
  {
    ImageList_Destroy( smallImages );
  }
  if (noImages)
  {
    ImageList_Destroy( noImages );
  }
  if (inUseImage)
  {
    ImageList_Destroy( inUseImage );
  }
  if (scaledInUseImage)                                              /* D7877 */
  {
    ImageList_Destroy( scaledInUseImage );
  }
  if (smallInUseImage)
  {
    ImageList_Destroy( smallInUseImage );
  }
  if (cnrInfo.strCnrTitle)
  {
    delete cnrInfo.strCnrTitle;
  }
}


/*------------------------------------------------------------------------------
| ICnrControlData::setCnrInfo                                                  |
|                                                                              |
| Change cnrInfo structure according to passed flag.                           |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::setCnrInfo(ICnrInfo* pcnrInfo, unsigned long setFlag)
{
  if (setFlag & CMA_PSORTRECORD)
  {
    cnrInfo.pSortRecord = pcnrInfo->pSortRecord;
  }

  if (setFlag & CMA_PFIELDINFOLAST)
  {
    cnrInfo.pFieldInfoLast = pcnrInfo->pFieldInfoLast;
  }

  if (setFlag & CMA_CNRTITLE)
  {
    if (cnrInfo.strCnrTitle)
      delete cnrInfo.strCnrTitle;
    cnrInfo.strCnrTitle = (char*)malloc( strlen( pcnrInfo->strCnrTitle ) + 1 );
    strcpy( cnrInfo.strCnrTitle, pcnrInfo->strCnrTitle );
    // Now force an update of the title window/
    if (pTitle)
      pTitle->setText( cnrInfo.strCnrTitle );
  }

  if (setFlag & CMA_FLWINDOWATTR)
  {
    cnrInfo.flWindowAttr = pcnrInfo->flWindowAttr;
    setWindowAttr(cnrInfo.flWindowAttr);
  }

  if (setFlag & CMA_PTLORIGIN)
  {
    cnrInfo.ptlOrigin = pcnrInfo->ptlOrigin;
  }

  if (setFlag & CMA_DELTA)
  {
    cnrInfo.cDelta = pcnrInfo->cDelta;
  }

  if (setFlag & CMA_SLBITMAPORICON)                                  /* D7877 */
  {
    // Only change the size if it is different from the current size.
    if ((cnrInfo.slBitmapOrIcon.cx != pcnrInfo->slBitmapOrIcon.cx) ||
        (cnrInfo.slBitmapOrIcon.cy != pcnrInfo->slBitmapOrIcon.cy))
    {
       cnrInfo.slBitmapOrIcon = pcnrInfo->slBitmapOrIcon;
       setImageSize();
    }
  }

  if (setFlag & CMA_SLTREEBITMAPORICON)
  {
    cnrInfo.slTreeBitmapOrIcon = pcnrInfo->slTreeBitmapOrIcon;
  }

  if (setFlag & CMA_TREEBITMAP)
  {
    cnrInfo.hbmExpanded = pcnrInfo->hbmExpanded;
    cnrInfo.hbmCollapsed = pcnrInfo->hbmCollapsed;
  }

  if (setFlag & CMA_TREEICON)
  {
    cnrInfo.hptrExpanded = pcnrInfo->hptrExpanded;
    cnrInfo.hptrCollapsed = pcnrInfo->hptrCollapsed;
  }

  if (setFlag & CMA_LINESPACING)
  {
    cnrInfo.cyLineSpacing = pcnrInfo->cyLineSpacing;
  }

  if (setFlag & CMA_CXTREEINDENT)
  {
    cnrInfo.cxTreeIndent = pcnrInfo->cxTreeIndent;
    if ( pTreeview )
      TreeView_SetIndent( pTreeview->handle(), cnrInfo.cxTreeIndent );
  }

  if (setFlag & CMA_CXTREELINE)
  {
    cnrInfo.cxTreeLine = pcnrInfo->cxTreeLine;
  }

  if (setFlag & CMA_XVERTSPLITBAR)
  {
    cnrInfo.xVertSplitbar = pcnrInfo->xVertSplitbar;
  }

  // There are other CMA_ flags we may support in the future
  return *this;
}
/* D7877 Next function*/
/*------------------------------------------------------------------------------
| ICnrControlData::setImageSize                                                |
|                                                                              |
| Set the size of the images.                                                   |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::setImageSize()
{
  bool done = false;
  int  cx, cy;
  ImageList_GetIconSize (largeImages, &cx, &cy);
  // If the size requested matches the large image list, use that.
  if ((cnrInfo.slBitmapOrIcon.cx == cx) &&
      (cnrInfo.slBitmapOrIcon.cy == cy))
  {
     useScaledImages = false;
     cnrInfo.flWindowAttr &= ~CV_MINI;
     setImageListWithView (cnrInfo.flWindowAttr);
     done = true;
  }
  else
  {
     ImageList_GetIconSize (smallImages, &cx, &cy);
     // If the size requested matches the small image list, use that.
     if ((cnrInfo.slBitmapOrIcon.cx == cx) &&
         (cnrInfo.slBitmapOrIcon.cy == cy))
     {
        useScaledImages = false;
        cnrInfo.flWindowAttr |= CV_MINI;
        setImageListWithView (cnrInfo.flWindowAttr);
        done = true;
     }
     else
     // If the size requested matches the current scaled list, use that.
     {
        if (scaledImages)
        {
           ImageList_GetIconSize (scaledImages, &cx, &cy);
           if ((cnrInfo.slBitmapOrIcon.cx == cx) &&
               (cnrInfo.slBitmapOrIcon.cy == cy))
           {
              useScaledImages = true;
              cnrInfo.flWindowAttr &= ~CV_MINI;
              setImageListWithView (cnrInfo.flWindowAttr);
              done = true;
           }
        }
     }
  }
  // Otherwise, we have to scale to get the right size.  Scale from the large
  // images.
  if (!done)
  {
     // If we had a previous scaled list, destroy it.
     if (scaledImages)
        ImageList_Destroy (scaledImages);
     if (scaledInUseImage)
        ImageList_Destroy (scaledInUseImage);

     // Create and add the scaled in use image.
     scaledInUseImage = ImageList_Create (cnrInfo.slBitmapOrIcon.cx,
                                      cnrInfo.slBitmapOrIcon.cy,
                                      ILC_MASK, 1, 0);
     ImageList_AddIcon( scaledInUseImage,
            IApplication::current().resourceLibrary().loadIcon(IC_INUSEICON) );

     // Create the scaled image list and add the icons from the large image list.
     scaledImages = ImageList_Create (cnrInfo.slBitmapOrIcon.cx,
                                      cnrInfo.slBitmapOrIcon.cy,
                                      ILC_MASK, 40, 5);
     int numberOfIcons = ImageList_GetImageCount (largeImages);
     for (int i=0; i < numberOfIcons; i++)
     {
        // Get icon, add to scaled list and free it (Image API creates it)
        HICON image = ImageList_GetIcon (largeImages, i, ILD_NORMAL);
        ImageList_AddIcon (scaledImages, image);
        DestroyIcon( image );
     }
     useScaledImages = true;
     cnrInfo.flWindowAttr &= ~CV_MINI;
     setImageListWithView (cnrInfo.flWindowAttr);
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::queryCnrInfo                                                |
|                                                                              |
| Return cnrInfo structure.                                                    |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::queryCnrInfo(ICnrInfo* pInfo)
{
  *pInfo = cnrInfo;
  return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::setSelection                                                |
|                                                                              |
| Change selection model of the container.                                     |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::setSelection(unsigned long selection)
{
  // If a listview exists, then change its selection type.
  if (pListview)
  {
    if (selection == CCS_SINGLESEL)
    {
      DWORD dwStyle = GetWindowLong(pListview->handle(), GWL_STYLE);
      if (!(dwStyle & LVS_SINGLESEL))
      {
        SetWindowLong(pListview->handle(),
                      GWL_STYLE,
                      (dwStyle | LVS_SINGLESEL | LVS_SHOWSELALWAYS));
      }
    }
    else if (selection == CCS_MULTIPLESEL)
    {
      DWORD dwStyle = GetWindowLong(pListview->handle(), GWL_STYLE);
      if (dwStyle & LVS_SINGLESEL)
      {
        SetWindowLong(pListview->handle(),
                      GWL_STYLE,
                      (dwStyle & ~LVS_SINGLESEL | LVS_SHOWSELALWAYS));
      }
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::migrateContainerObjects                                     |
|                                                                              |
| Build tree view from list view OR vice versa.                                |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::migrateContainerObjects( bool fromListToTree )
{
  if (fromListToTree)
  {
    int             numListItems(ListView_GetItemCount(pListview->handle()));
    LV_ITEM         lvItem;
    TV_ITEM         treeItem;
    TV_INSERTSTRUCT tvis;

    lvItem.mask       = LVIF_IMAGE | LVIF_PARAM | LVIF_STATE;
    lvItem.iSubItem   = 0;
    tvis.hParent      = TVI_ROOT;
    tvis.hInsertAfter = TVI_LAST;

    for (int index=0; index < numListItems; index++)
    {
      lvItem.iItem = index;
      if (ListView_GetItem(pListview->handle(), &lvItem))
      {
        IContainerObject* pCnrObj((IContainerObject*)lvItem.lParam);
        IMiniCnrRecord* pRec(IRecFromObj(pCnrObj) );

        // Add to list but ONLY if an icon is specified.
        int imageIndex = -1;
        if (pRec->hptrIcon)
          imageIndex = addIconToList(pRec->hptrIcon);

        treeItem.state          = 0;
        treeItem.mask           = TVIF_IMAGE | TVIF_TEXT | TVIF_PARAM | TVIF_SELECTEDIMAGE;
        treeItem.stateMask      = 0x0000;
        treeItem.pszText        = pRec->strIcon;
        treeItem.cchTextMax     = 64;
        treeItem.iImage         = imageIndex;
        treeItem.iSelectedImage = imageIndex;
        treeItem.lParam         = lvItem.lParam;
        tvis.item               = treeItem;

        // Insert the item into the tree view.
        HTREEITEM ti = TreeView_InsertItem(pTreeview->handle(), &tvis);

        pCnrObj->ppd->pCnrSet->updateTreeItem( pCnr, ti );
      }
    }
  }
  else
  {
    LV_ITEM          lvItem;
    TV_ITEM          treeItem;
    HTREEITEM        rootNode;

    treeItem.mask = TVIF_IMAGE | TVIF_PARAM;
    lvItem.mask   = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM;

    int index;
    for (rootNode = TreeView_GetRoot(pTreeview->handle()), index=0;
         rootNode != 0;
         rootNode = TreeView_GetNextSibling(pTreeview->handle(), rootNode), index++)
    {
      treeItem.hItem = rootNode;

      if (TreeView_GetItem(pTreeview->handle(), &treeItem))
      {
        IMiniCnrRecord* pRec(IRecFromObj( (IContainerObject*)treeItem.lParam) );

        lvItem.state      = 0;
        lvItem.stateMask  = 0x0000;
        lvItem.iItem      = index;

        lvItem.iSubItem   = 0;  // Don't need to loop for column subitems.
        lvItem.pszText    = pRec->strIcon;
        lvItem.cchTextMax = 64;
        lvItem.iImage     = treeItem.iImage;
        lvItem.lParam     = treeItem.lParam;

        ListView_InsertItem(pListview->handle(), &lvItem);
      }
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::layoutActiveView                                            |
|                                                                              |
| Position container window and title window if any.                           |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::layoutActiveView( const ISize& parentSize )
{
  int titleHeight(24);
  if (showTitle && pTitle)
  {
    SetWindowPos( pTitle->handle(),
                  HWND_TOP,
                  0,0,
                  parentSize.width(),
                  titleHeight,
                  SWP_SHOWWINDOW | SWP_FRAMECHANGED );

    if (showTitleSeparator)
      titleHeight++;

    SetWindowPos( pActiveview->handle(),
                  HWND_TOP,
                  0, titleHeight,
                  parentSize.width(),
                  parentSize.height()- titleHeight,
                  SWP_SHOWWINDOW );

  }
  else
  {
    if (pTitle)
      pTitle->hide();
    SetWindowPos( pActiveview->handle(),
                  HWND_TOP,
                  0,0,
                  parentSize.width(),
                  parentSize.height(),
                  SWP_SHOWWINDOW );
  }
  ITRACE_DEVELOP(IString("======> LAYOUT ACTIVE VIEW <====== Width: ") +
             IString(parentSize.width()) + IString("  Height: ") +
             IString(parentSize.height()));
  return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::addIconToList                                               |
|                                                                              |
| Put icon handle into pointer set and add it to the image list.               |
------------------------------------------------------------------------------*/
int ICnrControlData::addIconToList ( const IPointerHandle& iconHandle )
{
  int index;

  // Check if the pointer had already been inserted into the
  // image list.  If it has, return its index.
  if (pointerSet->containsElementWithKey( iconHandle.asUnsigned() ))
  {
    IPointerElement* pointerElement(pointerSet->elementWithKey( iconHandle.asUnsigned() ));
    index = pointerElement->imageListIndex;
  }
  // Otherwise, insert the icon into the image list.
  else
  {
    index = ImageList_AddIcon( smallImages, iconHandle );
            ImageList_AddIcon( largeImages, iconHandle );
    if (useScaledImages)
       ImageList_AddIcon( scaledImages, iconHandle );                /* D7877 */

    // Add the icon into the pointer set.
    pointerSet->addPointerElement( iconHandle, index );
  }
  return index;
}

/*------------------------------------------------------------------------------
| ICnrControlData::addInUseToList                                              |
|                                                                              |
| Put in use icon into pointer set and add it to the image list.               |
------------------------------------------------------------------------------*/
int ICnrControlData::addInUseToList ( const IPointerHandle& iconHandle )
{
  int index;

  // Check if the pointer had already been inserted into the
  // image list.  If not, return an error (-1).
  if (pointerSet->containsElementWithKey( iconHandle.asUnsigned() ))
  {
    IPointerElement* pointerElement(pointerSet->elementWithKey( iconHandle.asUnsigned() ));
    // If the image has not been merged, merge it with the in use image and
    // add it to the list.  Otherwise, return the merged icon's index.
    if (pointerElement->inUseIndex == -1)
    {
      HICON      hIcon;
      HIMAGELIST hTemp = ImageList_Merge( inUseImage,
                                          0,
                                          largeImages,
                                          pointerElement->imageListIndex,
                                          0, 0 );

      hIcon = ImageList_GetIcon( hTemp, 0, ILD_NORMAL );
      index = ImageList_AddIcon( largeImages, hIcon );
      ImageList_Destroy( hTemp );
      DestroyIcon( hIcon );

      pointerElement->inUseIndex = index;

      hTemp = ImageList_Merge( smallInUseImage, 0,
                               smallImages,
                               pointerElement->imageListIndex,
                               0, 0 );

      hIcon = ImageList_GetIcon( hTemp, 0, ILD_NORMAL );
      index = ImageList_AddIcon( smallImages, hIcon );
      ImageList_Destroy( hTemp );
      DestroyIcon( hIcon );
      if (useScaledImages)                                           /* D7877 */
      {
         hTemp = ImageList_Merge( scaledInUseImage, 0,
                                  scaledImages,
                                  pointerElement->imageListIndex,
                                  0, 0 );                            /* D7877 */

         hIcon = ImageList_GetIcon( hTemp, 0, ILD_NORMAL );          /* D7877 */
         index = ImageList_AddIcon( scaledImages, hIcon );           /* D7877 */
         ImageList_Destroy( hTemp );                                 /* D7877 */
         DestroyIcon( hIcon );                                       /* D7877 */
      }
    }
    else // Icon already merged - return its index.
    {
      index = pointerElement->inUseIndex;
    }
  }
  else  // Icon not in list - error.
  {
    index = -1;
  }
  return index;
}
/* D7877 Next function */
/*------------------------------------------------------------------------------
| ICnrControlData::setWindowAttr                                               |
|                                                                              |
| Set window attributes structure according to CV_* flags.                     |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::setWindowAttr(unsigned long flags)
{
  // Don't need to check for CV_FLOW since we've
  // hard-coded the GRID place style in createContainer.
  // Basically, the native container is always in flowed view, so
  // showTextView == showFlowedTextView, etc.
  // Illegal combinations, e.g., tree & detail will do one or the other
  // or nothing.
  // Only the combinations used by IContainerControl are considered.

  // showTextView             ( ), // CV_TEXT & !CV_TREE
  // showFlowedTextView       ( ), // CV_FLOW & CV_TEXT & !CV_TREE
  // showNameView             ( ), // CV_NAME & !CV_TREE
  // showFlowedNameView       ( ), // CV_NAME & CV_FLOW & !CV_TREE
  // showDetailsView          ( ), // CV_DETAILS
  // showTreeIconView         ( ), // CV_TREE & CV_ICON
  // showTreeTextView         ( ), // CV_TREE & CV_TEXT
  // showTreeNameView         ( ), // CV_TREE & CV_NAME
  // showIconView             ( ); // CV_ICON & !CV_TREE

  // For all view flags, set the active view, set the image list, and
  // change window word if necessary (SetWindowLong).
  if (((flags & CV_TREE) && (flags & CV_ICON)) ||
      ((flags & CV_TREE) && (flags & CV_TEXT)))
  {
    setActiveView( ICnrControlData::tree );
    setImageListWithView (flags);
    DWORD dwStyle = GetWindowLong(pTreeview->handle(), GWL_STYLE);
    SetWindowLong(pTreeview->handle(),GWL_STYLE, (dwStyle | TVS_HASBUTTONS | TVS_LINESATROOT | TVS_HASLINES));
  }

  if ((flags & CV_TREE) && (flags & CV_NAME))
  {
    setActiveView( ICnrControlData::tree );
    setImageListWithView (flags);
    DWORD dwStyle = GetWindowLong(pTreeview->handle(), GWL_STYLE);
    SetWindowLong(pTreeview->handle(),GWL_STYLE, ((dwStyle | TVS_HASLINES) & ~(TVS_HASBUTTONS | TVS_LINESATROOT)));
  }

  if (flags & CV_DETAIL)
  {
    setActiveView( ICnrControlData::list );
    setImageListWithView (flags);
    DWORD dwStyle = GetWindowLong(pListview->handle(), GWL_STYLE);
    if ((dwStyle & LVS_TYPEMASK) != LVS_REPORT)
    {
      SetWindowLong(pListview->handle(),
                    GWL_STYLE,
                    (dwStyle & ~LVS_TYPEMASK) | LVS_REPORT | LVS_SHOWSELALWAYS);
    }
    if (collapseFirstColumn)
    {
       ListView_SetColumnWidth(pListview->handle(), 0, 0) ;
       collapseFirstColumn = false ;
    }
  }

  if ( (flags & CV_ICON) && !(flags & CV_TREE) )
  {
    setActiveView( ICnrControlData::list );
    setImageListWithView (flags);
    DWORD dwStyle = GetWindowLong(pListview->handle(), GWL_STYLE);
    if ((dwStyle & LVS_TYPEMASK) != LVS_ICON)
    {
      SetWindowLong(pListview->handle(),
                    GWL_STYLE,
                    (dwStyle & ~LVS_TYPEMASK) | LVS_ICON | LVS_SHOWSELALWAYS);
    }
  }

  if (((flags & CV_NAME) && !(flags & CV_TREE)) ||
     ((flags & CV_TEXT) && !(flags & CV_TREE)))
  {
    setActiveView( ICnrControlData::list );
    setImageListWithView (flags);
    DWORD dwStyle = GetWindowLong(pListview->handle(), GWL_STYLE);
    if ((dwStyle & LVS_TYPEMASK) != LVS_LIST)
    {
      SetWindowLong(pListview->handle(),
                    GWL_STYLE,
                    (dwStyle & ~LVS_TYPEMASK) | LVS_LIST | LVS_SHOWSELALWAYS );
    }
  }

  if (flags & CA_TITLELEFT)
  {
    pTitle->setAlignment( IStaticText::centerLeft );
  }

  if (flags & CA_TITLECENTER)
  {
    pTitle->setAlignment( IStaticText::centerCenter );
  }

  if (flags & CA_TITLERIGHT)
  {
    pTitle->setAlignment( IStaticText::centerRight );
  }

  if (flags & CA_TITLESEPARATOR)
  {
    showTitleSeparator = true;
    layoutActiveView(pCnr->size());
  }
  else
  {
    showTitleSeparator = false;
    layoutActiveView(pCnr->size());
  }

  if (flags & CA_CONTAINERTITLE)
  {
      // Set the font of the title window.
      pTitle->setFont(pActiveview->font());

    if (cnrInfo.strCnrTitle)
      pTitle->setText( cnrInfo.strCnrTitle );

    showTitle = true;
    layoutActiveView(pCnr->size());
  }
  else
  {
    showTitle = false;
    layoutActiveView(pCnr->size());
  }

  if (flags & CA_TREELINE)
  {
     // Native container always has them or not.  Can't turn them on and off.
  }

  if (flags & CA_DETAILSVIEWTITLES)
  {
     // Title is always required in native container.  Can't turn it on or off.
  }

  return *this;
}
/* D7877 Next function */
/*------------------------------------------------------------------------------
| ICnrControlData::setImageListWithView                                        |
|                                                                              |
| Set the image list according to the view.                                    |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::setImageListWithView ( unsigned long flags )
{
   int width, height;
   // For all views, set the image list by checking the following: if mini,
   // if scaled, and otherwise.  Store the image width and height in
   // case it is queried.
   // treeIcon and treeName have images.
   if (((flags & CV_TREE) && (flags & CV_ICON)) ||
       ((flags & CV_TREE) && (flags & CV_NAME)))
   {
      if (flags & CV_MINI)
      {
        useScaledImages = false;
        TreeView_SetImageList(pTreeview->handle(), smallImages, TVSIL_NORMAL);
        if (ImageList_GetIconSize( smallImages, &width, &height) )
        {
          cnrInfo.slTreeBitmapOrIcon.cx = width;
          cnrInfo.slTreeBitmapOrIcon.cy = height;
          cnrInfo.slBitmapOrIcon.cx = width;
          cnrInfo.slBitmapOrIcon.cy = height;
        }
      }
     else
     if (useScaledImages)
     {
        TreeView_SetImageList(pTreeview->handle(), scaledImages, TVSIL_NORMAL);
        if (ImageList_GetIconSize( scaledImages, &width, &height) )
        {
          cnrInfo.slTreeBitmapOrIcon.cx = width;
          cnrInfo.slTreeBitmapOrIcon.cy = height;
          cnrInfo.slBitmapOrIcon.cx = width;
          cnrInfo.slBitmapOrIcon.cy = height;
        }
     }
     else
     {
       TreeView_SetImageList(pTreeview->handle(), largeImages, TVSIL_NORMAL);
       if (ImageList_GetIconSize( largeImages, &width, &height) )
       {
         cnrInfo.slTreeBitmapOrIcon.cx = width;
         cnrInfo.slTreeBitmapOrIcon.cy = height;
         cnrInfo.slBitmapOrIcon.cx = width;
         cnrInfo.slBitmapOrIcon.cy = height;
       }
     }
   }
   else
   // treeText has no images.
   if ((flags & CV_TREE) && (flags & CV_TEXT))
   {
     // Using a NULL image list causes the TreeView to draw the collapse/expand
     // icons itself.
     TreeView_SetImageList(pTreeview->handle(), NULL, TVSIL_NORMAL);
     cnrInfo.slTreeBitmapOrIcon.cx = 0;
     cnrInfo.slTreeBitmapOrIcon.cy = 0;
     cnrInfo.slBitmapOrIcon.cx = width;
     cnrInfo.slBitmapOrIcon.cy = height;
   }
   else
   // Details, icon, and name all have images.
   if ((flags & CV_DETAIL) ||
       ((flags & CV_ICON) && !(flags & CV_TREE)) ||
       ((flags & CV_NAME) && !(flags & CV_TREE)))
   {
     if (flags & CV_MINI)
     {
       useScaledImages = false;
       ListView_SetImageList(pListview->handle(), smallImages, LVSIL_SMALL );
       ListView_SetImageList(pListview->handle(), smallImages, LVSIL_NORMAL );
       if (ImageList_GetIconSize( smallImages, &width, &height) )
       {
         cnrInfo.slTreeBitmapOrIcon.cx = width;
         cnrInfo.slTreeBitmapOrIcon.cy = height;
         cnrInfo.slBitmapOrIcon.cx = width;
         cnrInfo.slBitmapOrIcon.cy = height;
       }
     }
     else
     if (useScaledImages)
     {
        ListView_SetImageList(pListview->handle(), scaledImages, LVSIL_SMALL);
        ListView_SetImageList(pListview->handle(), scaledImages, LVSIL_NORMAL);
        if (ImageList_GetIconSize( scaledImages, &width, &height) )
        {
          cnrInfo.slTreeBitmapOrIcon.cx = width;
          cnrInfo.slTreeBitmapOrIcon.cy = height;
          cnrInfo.slBitmapOrIcon.cx = width;
          cnrInfo.slBitmapOrIcon.cy = height;
        }
     }
     else
     {
       ListView_SetImageList(pListview->handle(), largeImages, LVSIL_SMALL );
       ListView_SetImageList(pListview->handle(), largeImages, LVSIL_NORMAL );
       if (ImageList_GetIconSize( largeImages, &width, &height) )
       {
         cnrInfo.slTreeBitmapOrIcon.cx = width;
         cnrInfo.slTreeBitmapOrIcon.cy = height;
         cnrInfo.slBitmapOrIcon.cx = width;
         cnrInfo.slBitmapOrIcon.cy = height;
       }
     }
   }
   else
   // Plain text view has no images.
   if ((flags & CV_TEXT) && !(flags & CV_TREE))
     // Set a dummy image list that contains zero-sized images.
     ListView_SetImageList(pListview->handle(), noImages, LVSIL_SMALL );

   return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::setActiveView                                               |
|                                                                              |
| Show the tree view or list view.  Create them if they don't exist.           |
------------------------------------------------------------------------------*/
ICnrControlData& ICnrControlData::setActiveView ( ActiveView activeView )
{
  // If this is the first time, we have to do some special processing.
  bool fFirstTime(false);
  if (activeView == ICnrControlData::none)
    fFirstTime = true;

  // If the view being requested is the one we're in, do nothing.
  if (activeView != av)
  {
    // Create a place holder for the prior view.
    INativeContainerControl *pOldview(0);
    // Set the active view.
    av = activeView;

    // View requested is tree view.
    if (activeView == ICnrControlData::tree)
    {
      // A TreeView control doesn't yet exist -> Create It.
      if (!pTreeview)
      {
        unsigned long currentStyle(pCnr->style());
        pTreeview = new INativeContainerControl( IC_NATIVE_TREEVIEW,
                                                 pCnr->handle(),
                                                 pCnr->handle(),
                                                 IRectangle(IPoint(0,0),
                                                            pCnr->size()),
                                                 convertToNativeStyle( currentStyle, false ),
                                                 false );

        // If a list view exists, populate the tree view from the list view
        if (pListview)
        {
          migrateContainerObjects( true );
          pTreeview->setFont(pListview->font());
        }

        // If the child indention has been changed from the default, set it.
        if ( cnrInfo.cxTreeIndent )
          TreeView_SetIndent( pTreeview->handle(), cnrInfo.cxTreeIndent );

      }

      // Set the active view to the tree view.
      pActiveview = pTreeview;

      // If there was a list view before, clean it up and get it read to hide.
      // Initialize characteristics of the tree view from the list view.
      if (pListview)
      {
        // Save old view.
        pOldview = pListview;
        // Position tree view using list view info.
        pTreeview->moveSizeTo( pListview->rect() );

        // Get rid of the edit field if it exists.
        ListView_EditLabel( pListview->handle(), (-1) );

        if (pCnr->ppd->pDetailsEdit)                                 /* D8097 */
          // Send a fake enter message to the mle so that it will close.
          pCnr->ppd->pDetailsEdit->sendEvent( WM_CHAR,
                                              IEventParameter1( VK_RETURN ));

        // Fake out the container to let it think it is still in a non tree
        // view so that the CM_QUERYRECORDEMPHASIS logic below will work.
        cnrInfo.flWindowAttr &= ~CV_TREE;

        // Query the cursored object in the list view.  This
        // will be the selected object in the tree view.
        IContainerObject* cursored(pCnr->cursoredObject());
        cnrInfo.flWindowAttr |= CV_TREE;

        if (cursored)
        {
          // Set the selected object in the tree view to
          // the cursored object in the tree view.
          pCnr->setSelected(cursored);
        }
      }
    }
    else
    {
      // List view was requested and it doesn't exist.
      if (!pListview)
      {
        unsigned long currentStyle(pCnr->style());
        pListview = new INativeContainerControl( IC_NATIVE_LISTVIEW,
                                                 pCnr->handle(),
                                                 pCnr->handle(),
                                                 IRectangle(IPoint(0,0),
                                                            pCnr->size()),
                                                 convertToNativeStyle( currentStyle, true ),
                                                 true );

        // If a tree view exists, populate the list view from the tree view
        if (pTreeview)
        {
          migrateContainerObjects( false );
          pListview->setFont(pTreeview->font());
        }

        // Add the required first column for details view.
        LV_COLUMN    lvColumn;

        lvColumn.mask     = LVCF_TEXT | LVCF_WIDTH;
        lvColumn.pszText  = "";
        lvColumn.cx = 0;

        ListView_InsertColumn( pListview->handle(), 0, &lvColumn );

      }

      // Set the active view to the list view.
      pActiveview = pListview;


      // If there was a tree view before, clean it up and get it read to hide.
      // Initialize characteristics of the list view from the tree view.
      if (pTreeview)
      {
        // Set the old view.
        pOldview = pTreeview;
        // Size the list view using the tree view info.
        pListview->moveSizeTo( pTreeview->rect() );

        // Get rid of the edit field if it exists.
        TreeView_EndEditLabelNow( pTreeview->handle(), false );

        // Fake out the container to let it think it is still in a tree
        // view so that the CM_QUERYRECORDEMPHASIS logic below will work.
        cnrInfo.flWindowAttr |= CV_TREE;

        // Query the cursored object in the tree view.  This
        // will be the selected object in the list view.
        IContainerObject* cursored(pCnr->cursoredObject());
        cnrInfo.flWindowAttr &= ~CV_TREE;

        if (cursored)
        {
          //We should remove selection from all objects other than the
          //cursor because of mult. selection....Or that a different object
          //may have been selected when last in the listview.
          IContainerControl::ObjectCursor  listCursor =
                              IContainerControl::ObjectCursor(*pCnr, 0, false);
          forCursor ( listCursor )
            pCnr->removeSelected( pCnr->objectAt( listCursor ) );

          // Set the selected & cursored object in the list view
          // to the root of the cursored object in the tree view.
          IContainerObject*  rootObj= pCnr->ppd->rootObject( cursored );
          pCnr->setSelected( rootObj );
          pCnr->setCursor  ( rootObj );
        }
      }
    }

    // Show the active view and hide the old one.
    pActiveview->show();
    if (pOldview)
      pOldview->hide();
  }

  if (pCnr->ppd->pDetailsEdit)                                     /* D8097 */
    // Send a fake enter message to the mle so that it will close.
    pCnr->ppd->pDetailsEdit->sendEvent( WM_CHAR,
                                        IEventParameter1( VK_RETURN ));

  // If this was the first time a view was shown, set its font.
  if (fFirstTime)
  {
    pCnr->setFont(pActiveview->font());
    // Set the font of the title window.
    pTitle->setFont(pActiveview->font());
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ICnrControlData::rootObject                                                  |
|                                                                              |
| Return the root object of the tree view.                                     |
------------------------------------------------------------------------------*/
IContainerObject* ICnrControlData::rootObject(IContainerObject* treeObj)
{
  IContainerObject*   rootObj = 0;

  while ( treeObj )
  {
    rootObj = treeObj;
    treeObj = pCnr->parentObject( treeObj );
  }

  return ( rootObj );
}

/*------------------------------------------------------------------------------
| ICnrControlData::activeView                                                  |
|                                                                              |
| Return the active view.                                                      |
------------------------------------------------------------------------------*/
ICnrControlData::ActiveView ICnrControlData::activeView( ) const
{
  return av;
}

/*------------------------------------------------------------------------------
| ICnrControlData::registerContainer                                           |
|                                                                              |
| Register the wrapper window class.                                           |
------------------------------------------------------------------------------*/
void ICnrControlData::registerContainer( )
{
  static bool fRegistered(false);

  // Register the wrapper window class and query the size of the large
  // and small image lists.
  if (!fRegistered)
  {
    InitCommonControls();

    fRegistered = true;
    WNDCLASS wndClass;

    wndClass.style         = CS_HREDRAW | CS_VREDRAW;
    wndClass.cbClsExtra    = 0;
    wndClass.cbWndExtra    = sizeof(LONG);
    wndClass.hCursor       = LoadCursor(0,IDC_ARROW);
    wndClass.hInstance     = GetModuleHandle(0);
    wndClass.lpfnWndProc   = (WNDPROC)IDEFWINDOWPROC;
    wndClass.hIcon         = 0;
    wndClass.hbrBackground = (HBRUSH) GetStockObject( BLACK_BRUSH );
    wndClass.lpszMenuName  = 0;
    wndClass.lpszClassName = WC_NATIVECONTAINER;

    if (!RegisterClass(&wndClass))
      ITHROWGUIERROR( "RegisterClass" );

  }
}

/*------------------------------------------------------------------------------
| ICnrControlData::convertToNativeStyle                                        |
|                                                                              |
| Convert CCL container style to native container style.                       |
------------------------------------------------------------------------------*/
unsigned long ICnrControlData::convertToNativeStyle( unsigned long cclStyle,
                                                     bool          fListView )
{
  unsigned long windowStyle(0);

  if (fListView)
  {
    windowStyle = LVS_ICON | LVS_SHOWSELALWAYS;

    if (!(cclStyle & CCS_READONLY))
       windowStyle |= LVS_EDITLABELS;
    if (cclStyle & CCS_AUTOPOSITION)
      windowStyle |= LVS_AUTOARRANGE;
    if (cclStyle & CCS_SINGLESEL)
      windowStyle |= LVS_SINGLESEL;
  }
  else
  {
    windowStyle = TVS_SHOWSELALWAYS | TVS_HASLINES | TVS_HASBUTTONS | TVS_LINESATROOT;

    if (!(cclStyle & CCS_READONLY))
       windowStyle |= TVS_EDITLABELS;
  }

  return windowStyle;
}

/*------------------------------------------------------------------------------
| INativeContainerControl::INativeContainerControl                             |
|                                                                              |
| Native container control constructor.                                        |
------------------------------------------------------------------------------*/
INativeContainerControl::INativeContainerControl( unsigned long id,
                                                  const IWindowHandle& parent,
                                                  const IWindowHandle& owner,
                                                  const IRectangle& initial,
                                                  unsigned long style,
                                                  bool fCreateListView )
  : INativeContainerControl::Inherited( )
{
  IWindowHandle whContainer;

  if (fCreateListView)
  {
    whContainer = this->create( IC_NATIVE_LISTVIEW,
                                0,
                                style | WS_CHILD,
                                WC_LISTVIEW,
                                parent,
                                owner,
                                initial,
                                0,
                                0,
                                defaultOrdering(),
                                WS_EX_CLIENTEDGE );
  }
  else
  {
    whContainer = this->create( IC_NATIVE_TREEVIEW,
                                0,
                                style | WS_CHILD,
                                WC_TREEVIEW,
                                parent,
                                owner,
                                initial,
                                0,
                                0,
                                defaultOrdering(),
                                WS_EX_CLIENTEDGE );
  }
  this->startHandlingEventsFor( whContainer );

  IMessageHandler::defaultHandler()->handleEventsFor( this );
}

/*------------------------------------------------------------------------------
| INativeContainerControl::~INativeContainerControl                            |
|                                                                              |
| Native container control destructor                                          |
------------------------------------------------------------------------------*/
INativeContainerControl::~INativeContainerControl( )
{
  IMessageHandler::defaultHandler()->stopHandlingEventsFor(this);
}

#pragma enum(4)
#pragma pack(push,4)

/**********************************************************************/
/* Class to handle destruction of the static INativeContainerHandler  */
/* object.                                                            */
/**********************************************************************/
class INativeContainerHandlerStatics
{
public:
~INativeContainerHandlerStatics ()
 {
   IMODTRACE_DEVELOP("INativeContainerHandlerStatics::~INativeContainerHandlerStatics");
#if IC_STATIC_PRIORITY_SUPPORTED
   // It is only safe to do the deletes when static init/term ordering can be controlled.
   if(pNativeContainerHandler)
      delete pNativeContainerHandler;
   if(pMessageHandler)
      delete pMessageHandler;
   if(pEditHandler)
      delete pEditHandler;
#endif
 }

INativeContainerHandler
 *pNativeContainerHandler;
IMessageHandler
 *pMessageHandler;
IDetailsEditHandler
 *pEditHandler;
};   // INativeContainHandlerStatics

#pragma pack(pop)
#pragma enum(pop)

static INativeContainerHandlerStatics IDefaultNCStatics;


/*------------------------------------------------------------------------------
| INativeContainerHandler::INativeContainerHandler                             |
| Native container handler constructor.                                        |
------------------------------------------------------------------------------*/
INativeContainerHandler::INativeContainerHandler( )
  : INativeContainerHandler::Inherited( )
{ }

/*------------------------------------------------------------------------------
| INativeContainerHandler::~INativeContainerHandler                            |
| Native container handler destructor.                                         |
------------------------------------------------------------------------------*/
INativeContainerHandler::~INativeContainerHandler( )
{ }

/*------------------------------------------------------------------------------
| INativeContainerHandler::defaultHandler                                      |
|                                                                              |
| Returns the default handler (creates one if necessary).                      |
------------------------------------------------------------------------------*/
INativeContainerHandler* INativeContainerHandler::defaultHandler ( )
{
   if(!IDefaultNCStatics.pNativeContainerHandler)
      IDefaultNCStatics.pNativeContainerHandler = new INativeContainerHandler();
   return IDefaultNCStatics.pNativeContainerHandler;
}

/*------------------------------------------------------------------------------
| INativeContainerHandler::containerHandleFromEvent                            |
|                                                                              |
| Extract the container handle from an event.                                  |
------------------------------------------------------------------------------*/
IWindowHandle INativeContainerHandler::containerHandleFromEvent ( const IEvent& event )
{
  IContainerControl *container((IContainerControl*)event.window());
  IWindowHandle view;
  view = container->ppd->pActiveview->handle();
  return view;
}

/*------------------------------------------------------------------------------
| INativeContainerHandler::dispatchHandlerEvent                                |
|                                                                              |
| Handle events to the native container wrapper.                               |
------------------------------------------------------------------------------*/
bool INativeContainerHandler::dispatchHandlerEvent( IEvent& event )
{
#define ptrTV_DISPINFO ((TV_DISPINFO *)event.parameter2().asUnsignedLong())
#define ptrLV_DISPINFO ((LV_DISPINFO *)event.parameter2().asUnsignedLong())

  bool fStopHandling( false );
  IContainerControl *pCnrCtl((IContainerControl*)event.window());

  switch (event.eventId())
  {
    // Eat the keyboard messages so they are not passed up the owner chain.

    case  WM_SETREDRAW:
        if (pCnrCtl->ppd->pListview)   // only if Listview
        {
            int rc = SendMessage( pCnrCtl->ppd->pListview->handle(),
                        WM_SETREDRAW, event.parameter1(), 0 );
            event.setResult( rc );      // set return code
            fStopHandling = true;       // stop the handling
        }
    break;
    case  WM_KEYDOWN:
    case  WM_KEYUP:
    case  WM_CHAR:
      // Pass tab key to owner for tab processing.
      if ((event.eventId() == WM_CHAR) && (event.parameter1() == VK_TAB))
      {
        IWindow* owner(event.window()->owner());
        owner->sendEvent( event );
      }
      event.setResult(true);
    break;

    case WM_GETDLGCODE:
      event.setResult( (long)DLGC_WANTARROWS | DLGC_WANTCHARS );
      fStopHandling = true;
    break;

    // If the wrapper window is receiving the input focus,
    // pass it onto the active control window.
    case WM_SETFOCUS:
      if (pCnrCtl->ppd->pActiveview &&
          pCnrCtl->ppd->pActiveview->isVisible())
        pCnrCtl->ppd->pActiveview->setFocus();
    break;

    case CM_ALLOCDETAILFIELDINFO:
    case CM_ALLOCRECORD:
      /*******************************************************************/
      /* These messages are not implemented because we do not call them  */
      /* for the native windows container.  The memory allocation is done*/
      /* only via the CCL allocation container.                          */
      /*******************************************************************/
      event.setResult(0);
      fStopHandling = true;
    break;

    case CM_ARRANGE:
    {
      /******************************************************************/
      /* Arrange the Native ListView Control's Icon View.               */
      /******************************************************************/
      //If we have size and the list view exists, do the arrange now.
      //Otherwise, set the needArrange flag for later when we are given
      //size (WM_SIZE), or the view is changed (CM_SETCNRINFO).

      if (pCnrCtl->ppd->pListview)
      {
         ISize cnrSize = pCnrCtl->ppd->pListview->size();
         bool hasSize = (!(cnrSize.width() == 0)) &&
                           (!(cnrSize.height() == 0));
         if (hasSize)
         {
            pCnrCtl->ppd->needArrange = false;
            ListView_Arrange( pCnrCtl->ppd->pListview->handle(), LVA_ALIGNTOP );
         }
         else
            pCnrCtl->ppd->needArrange = true;
      }
      else
         pCnrCtl->ppd->needArrange = true;

      event.setResult( true );
      fStopHandling = true;
    }
    break;

    case CM_CLOSEEDIT:
      /*********************************************************************/
      /* Close the current edit field.                                     */
      /*********************************************************************/

      //If in a treeView, close its edit field, otherwise close the
      //listview's edit field.  We assume that switching views closes
      //the edit fields, thus both views can't have an open field
      //simultaneously.
      if ( pCnrCtl->isTreeView() )
        event.setResult( TreeView_EndEditLabelNow(
                                 pCnrCtl->ppd->pTreeview->handle(), false ) );
      // If we're not in details view, let the control close the open edit field.
      else if (!pCnrCtl->isDetailsView())
           event.setResult( ListView_EditLabel( pCnrCtl->ppd->pListview->handle(),
                                                (-1) ) );
      // If we're in details view, we opened the edit field so we have to close it.
      else if (pCnrCtl->ppd->pDetailsEdit)                          /* D8017 */
      {
        // Send a fake enter message to the mle so that it will close.
        pCnrCtl->ppd->pDetailsEdit->sendEvent( WM_CHAR,
                                               IEventParameter1( VK_RETURN ));
        event.setResult(true);
      }

      fStopHandling = true;
      break;

    case CM_ERASERECORD:
      /********************************************************************/
      /* There is no Native Windows equivalent.                           */
      /* This function is not needed for moving icons in Windows Listview.*/
      /********************************************************************/
      event.setResult( true );
      fStopHandling = true;
    break;


    case CM_EXPANDTREE:
    case CM_COLLAPSETREE:
    {
      /*********************************************************************/
      /* Expand/Collapse the given object.                                 */
      /* If object=0, then expand/collapse all objects.                    */
      /*********************************************************************/
      //Set changeFlag appropriately, for the given message.
      UINT  changeFlag=0;
      if (event.eventId() == CM_EXPANDTREE)
        changeFlag = TVE_EXPAND;
      else
        changeFlag = TVE_COLLAPSE;
      IMiniCnrRecord* pRec((IMiniCnrRecord*)event.parameter1().asUnsignedLong());

      // If we have a tree view, do the expand/collapse whether it is currently
      // showing or not.
      if ( pCnrCtl->ppd->pTreeview )
      {
        IContainerObject*  pCnrObj=0;
        HTREEITEM          hItem;

        if ( pRec )    //Just expand/collapse the given object.
        {
          pCnrObj  = IObjFromRec( pRec );
          hItem = pCnrObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl);
          // If an expand or collapse actually occurs, TreeView_Expand
          // returns true, and we need to notify our owner.  TreeView_Expand
          // DOES NOT send a TVN_ITEMEXPANDED, as documented in the MSDN.
          // Check our own flag to see if expand or collapse notification has
          // been done.  Windows does notify the first time,
          // but not thereafter.
          pCnrCtl->ppd->expandNotificationDone = false;
          if (TreeView_Expand(pCnrCtl->ppd->pTreeview->handle(),
              hItem, changeFlag))
          {
             if (!pCnrCtl->ppd->expandNotificationDone)
             {
                pCnrCtl->ppd->expandNotificationDone = true;
                IWindow* owner(event.window()->owner());

                if (owner)
                {
                  if (changeFlag == TVE_EXPAND)
                  {
                    owner->sendEvent( WM_CONTROL,
                                      IEventParameter1( event.window()->id(),
                                                        CN_EXPANDTREE ),
                                      IEventParameter2( (RECORDCORE*)pRec ));
                  }
                  else
                  {
                    owner->sendEvent( WM_CONTROL,
                                      IEventParameter1( event.window()->id(),
                                                        CN_COLLAPSETREE ),
                                      IEventParameter2( (RECORDCORE*)pRec ));
                  }
                }
             }
          }
          // Always return true.  TreeView_Expand returns false if no change
          // occurred, i.e., if expand is requested on an already expanded item.
          event.setResult( true );                                   /* D7942 */
        }
        else           //Expand/Collapse the whole tree.
        {
          IContainerControl::ObjectCursor cursor(*pCnrCtl);

          //Cursor thru and expand/collapse all objects in the tree view.
          forCursor (cursor)
          {
            pCnrObj = pCnrCtl->objectAt(cursor);
            hItem = pCnrObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl);
            TreeView_Expand( pCnrCtl->ppd->pTreeview->handle(),
                             hItem, changeFlag );
          }
          //Always return true
          event.setResult( true );
        }
      }
      fStopHandling = true;
    }
    break;

    case CM_FILTER:
      //Not implemented for Native Windows.
    break;

    case CM_FREEDETAILFIELDINFO:
    case CM_FREERECORD:
      /*******************************************************************/
      /* These messages are not implemented because we do not call them  */
      /* for the native windows container.  The freeing is only done via */
      /* the CCL allocationContainer.  We do not need to explicity free  */
      /* windows objects or columns, just removing them is enough.       */
      /*******************************************************************/
      event.setResult(0);
      fStopHandling = true;
    break;

    case CM_HORZSCROLLSPLITWINDOW:
    {
      /*******************************************************************/
      /* Scroll the Details view.                                        */
      /* Only scroll the left window.  If right is requested, do nothing.*/
      /*******************************************************************/
      bool            retVal    ( false );
      unsigned short  usWindow  ( event.parameter1().lowNumber() );
      long            scrollInc ( (long)event.parameter2() );

      if ( pCnrCtl->isDetailsView() )
      {
        if ( usWindow == CMA_LEFT )
          event.setResult( ListView_Scroll( pCnrCtl->ppd->pListview->handle(),
                                            (int)scrollInc, 0 ) );
        retVal = true;
      }

      event.setResult( retVal );
      fStopHandling = true;
    }
    break;

    case CM_INSERTDETAILFIELDINFO:
    {
      /*******************************************************************/
      /* Insert a new column into the list view.                         */
      /*******************************************************************/
      LPFIELDINFO
            pFI((LPFIELDINFO)event.parameter1().asUnsignedLong());
      LPFIELDINFOINSERT
           pFIInsert((LPFIELDINFOINSERT)event.parameter2().asUnsignedLong());
      LV_COLUMN    lvColumn;
      int   columnId( 0 ),
            retVal( 1 ),
            insertPosition( 0 );

      // If a ListView control doesn't yet exist, create it.
      if ( !pCnrCtl->ppd->pListview )
      {
        unsigned long currentStyle( pCnrCtl->style() );
        pCnrCtl->ppd->pListview = new INativeContainerControl(
                                          IC_NATIVE_LISTVIEW,
                                          pCnrCtl->handle(),
                                          pCnrCtl->handle(),
                                          IRectangle(IPoint(0,0),
                                                     pCnrCtl->size()),
                                          pCnrCtl->ppd->convertToNativeStyle(
                                                           currentStyle, true ),
                                          true );

        // Set the image list for the new view.                      /* D7877 */
        pCnrCtl->ppd->setImageListWithView (pCnrCtl->ppd->cnrInfo.flWindowAttr);
        // If a tree view exists, populate the list view from the tree view
        if ( pCnrCtl->ppd->pTreeview )
          pCnrCtl->ppd->migrateContainerObjects( false );

        // Add the required first column for details view.
        lvColumn.mask     = LVCF_TEXT | LVCF_WIDTH;
        lvColumn.pszText  = "";
        lvColumn.cx = 0;

        ListView_InsertColumn( pCnrCtl->ppd->pListview->handle(),
                               0, &lvColumn );

      }

      // If the column being inserted is (DataSource=isIcon or isIconViewText)
      // then don't insert it as it is already inserted as required.
      if ( pFI->offStruct < pCnrCtl->baseRecordSize() )
      {
        pCnrCtl->ppd->collapseFirstColumn = false ;
        if ( pFI->offStruct == (unsigned long)&(((ICnrRecord *)0)->strIcon) )
        {
          // DataSource=isIconViewText, then update first column title
          // and column size.
          lvColumn.mask = LVCF_TEXT | LVCF_WIDTH;
          if ( pFI->cxWidth == 0 )
            lvColumn.cx   = 100;
          else
            lvColumn.cx   = pFI->cxWidth;

          if ( (pFI->flTitle & CFA_STRING) && pFI->pTitleData )
          {
             // In native Windows, only single line headings are supported, so
             // we will strip out any newline characters and replace them with
             // spaces.
             IString temp( (char*)pFI->pTitleData );
             temp.change( "\n", " " );
             strcpy( (char*)(pFI->pTitleData), temp );
             lvColumn.pszText = (char*)pFI->pTitleData;
          }
          else
            lvColumn.pszText = "";

          // Check if the isIconViewText column has been added previously.
          IContainerColumn* pcnrcol =
                   pCnrCtl->ppd->columnSet->containerColumnFromColumnId(0);

          // If column 0 has already been added, then remove from set
          // before adding the new one (because we are only replacing the
          // current one in the native control, don't want to count twice)
          if ( pcnrcol != 0 )
          {
            pCnrCtl->ppd->columnSet->remove( pcnrcol, false );
            pCnrCtl->ppd->columnsToDelete->add( pcnrcol );
          }

          ListView_SetColumn( pCnrCtl->ppd->pListview->handle(),
                              0, &lvColumn );

          // Add this to the columnSet for this container.
          pCnrCtl->ppd->columnSet->add( (IContainerColumn*)pFI->pUserData,
                                        0, 0, true );
        }
        else
        if ( pFI->offStruct == (unsigned long)&(((ICnrRecord *)0)->hptrIcon) )
        {
          // DataSource=isIcon, then update first column size.
          // Do this only if the isIconViewText column has not been added.
          // The application can only control the size and heading of this
          // column via the isIconViewText column.
          if ( pCnrCtl->ppd->columnSet->containerColumnFromColumnId(0) == 0 )
          {
            lvColumn.mask = LVCF_WIDTH;
            lvColumn.cx   = 100;
            ListView_SetColumn( pCnrCtl->ppd->pListview->handle(),
                                0, &lvColumn );

            // Add this to the columnSet for this container ... but only
            // if we don't already have
            pCnrCtl->ppd->columnSet->add( (IContainerColumn*)pFI->pUserData,
                                          (-1), (0), true );
          }
          else
          {
             // don't bump up the column count as we are not adding
             // extra isIcon fields
             pCnrCtl->ppd->cnrInfo.cFields--;  // this gets incremented at the end of the case
          }
        }
      }
      else
      {
        int  prevSubItemId(0);
        bool colVisible( true );
        if ( pFI->flData & CFA_BITMAPORICON )
          colVisible = false;

        lvColumn.mask = LVCF_FMT | LVCF_SUBITEM | LVCF_TEXT | LVCF_WIDTH;

        // CCL allows both the title and the column data to have separate
        // alignment styles.  ListView only has one so we will use the
        // data alignment for both.
        if ( pFI->flData & CFA_CENTER )
           lvColumn.fmt      = LVCFMT_CENTER;
        else
        {
           // Reverse the alignment if the window has an RTL
           // window layout.
           bool
             rightToLeft = false;
           if ( IBidiSettings::isBidiSupported() )
           {
              IBidiSettings
                bidiSettings( *pCnrCtl );
              if ( bidiSettings.windowLayout() ==
                     IBidiSettings::layoutRightToLeft )
              {
                 rightToLeft = true;
              }
           }
           if ( pFI->flData & CFA_RIGHT )
           {
              lvColumn.fmt = rightToLeft ? LVCFMT_LEFT : LVCFMT_RIGHT;
           }
           else  // default is left-aligned
           {
              lvColumn.fmt = rightToLeft ? LVCFMT_RIGHT : LVCFMT_LEFT;
           }
        }

        // ListView does not support icons for column titles.
        // ListView only supports strings for column titles.
        // Use an empty string if style is not string, or if
        // a string is not yet provided at this point.
        if ( (pFI->flTitle & CFA_STRING) && pFI->pTitleData )
        {
           // In native Windows, only single line headings are supported, so
           // we will strip out any newline characters and replace them with
           // spaces.
           IString temp( (char*)pFI->pTitleData );
           temp.change( "\n", " " );
           strcpy( (char*)(pFI->pTitleData), temp );
           lvColumn.pszText = (char*)pFI->pTitleData;
        }
        else
          lvColumn.pszText = "";

        // Query for the next available subItemId.
        lvColumn.iSubItem = pCnrCtl->ppd->columnSet->newSubItemId();

        // Specify the column width, otherwise use the default.
        if ( pFI->cxWidth == 0 )
          lvColumn.cx = 100;
        else
          lvColumn.cx = pFI->cxWidth;

        // Determine the position to place the new column.
        if ( pFIInsert->pFieldInfoOrder ==(PFIELDINFO)CMA_FIRST )
          insertPosition = 1;
        else if ( pFIInsert->pFieldInfoOrder == (PFIELDINFO)CMA_END )
        {
          insertPosition = pCnrCtl->ppd->columnSet->columnCount();

          //Get The current last items SubItemId
          prevSubItemId = pCnrCtl->ppd->columnSet->lastSubItemId();
        }
        else    //Assume valid column. If not valid, columnIdFromFieldInfo will throw
                // an exception.
        {
          insertPosition = pCnrCtl->ppd->columnSet->columnIdFromFieldInfo(
                                          pFIInsert->pFieldInfoOrder);
          if (insertPosition < 1)
            insertPosition = 1;
          else
            insertPosition++;

          // Update subItem so that it represents the previous subItem.
          prevSubItemId = pCnrCtl->ppd->columnSet->subItemIdFromFieldInfo(
                                         pFIInsert->pFieldInfoOrder);
        }

        lvColumn.iSubItem = insertPosition;

        // If column was a CFA_BITMAPORICON, it won't be shown.
        if ( colVisible )
          columnId = ListView_InsertColumn( pCnrCtl->ppd->pListview->handle(),
                                            insertPosition,
                                            &lvColumn );

        // Add this to the columnSet for this container.
        if ( columnId != (-1) )
          pCnrCtl->ppd->columnSet->add( (IContainerColumn*)pFI->pUserData,
                                        prevSubItemId,
                                        insertPosition,
                                        colVisible);
        else
          retVal = 0;
      }

      pCnrCtl->ppd->cnrInfo.cFields++;
      event.setResult( retVal );
      fStopHandling = true;
    }
    break;

    case CM_INSERTRECORD:
    {
      /*******************************************************************/
      /* Insert a new object.                                            */
      /*******************************************************************/
      IMiniCnrRecord*
            pRec((IMiniCnrRecord*)event.parameter1().asUnsignedLong());
      LPRECORDINSERT
           pRecordInsert((LPRECORDINSERT)event.parameter2().asUnsignedLong());
      IContainerObject* pCnrObj(IObjFromRec(pRec));

      // emulate OS/2 behavior. Should not have the same object twice in a
      // cnr
      if (pCnrCtl->containsObject(pCnrObj) )
      {
         event.setResult(false) ;
         fStopHandling = true ;
      }
      else
      {
        // If no icon, then don't add to list and pass index of -1
        int imageIndex = -1;
        if (pRec->hptrIcon)
          imageIndex = pCnrCtl->ppd->addIconToList(pRec->hptrIcon);
        ICnrObjSetInfo  *pCOSInfo;

        //Create the IContainerObject's private data if this
        //object has not been inserted before.
        if ( pCnrObj->ppd == 0 )
          pCnrObj->ppd = new ICnrObjPrivateData( pCnrObj );

        //Check if any objects are already inserted.
        //If none, then give object cursored emphasis.
        bool noObjectsInserted = true;
        if ( ((pCnrCtl->ppd->pListview) &&
              (ListView_GetItemCount(pCnrCtl->ppd->pListview->handle()) == 0)) ||
             ((pCnrCtl->ppd->pTreeview ) &&
              (TreeView_GetCount(pCnrCtl->ppd->pTreeview->handle()) == 0)) )
          pRec->flRecordAttr |= CRA_CURSORED;

        //Check and see if a child is being inserted.
        //If so, then make sure a tree view has been created before
        //we attempt to insert the child object below.
        if ( pRecordInsert->pRecordParent && !pCnrCtl->ppd->pTreeview )
        {
          //Create TreeView and migrate objects.
          unsigned long currentStyle(pCnrCtl->style());
          pCnrCtl->ppd->pTreeview = new INativeContainerControl
                                           ( IC_NATIVE_TREEVIEW,
                                             pCnrCtl->handle(),
                                             pCnrCtl->handle(),
                                             IRectangle(IPoint(0,0),
                                                        pCnrCtl->size()),
                                             pCnrCtl->ppd->convertToNativeStyle(
                                                                  currentStyle,
                                                                  false ),
                                             false );

          // If a list view exists, populate the tree view from the list view
          if (pCnrCtl->ppd->pListview)
            pCnrCtl->ppd->migrateContainerObjects( true );

          // If the child indention has been changed from the default, set it.
          if ( pCnrCtl->ppd->cnrInfo.cxTreeIndent )
            TreeView_SetIndent( pCnrCtl->ppd->pTreeview->handle(),
                                pCnrCtl->ppd->cnrInfo.cxTreeIndent );

        }

        // If we have a tree view, insert the item.
        if ( pCnrCtl->ppd->pTreeview )
        {
          TV_ITEM          treeItem;
          TV_INSERTSTRUCT  tvis;
          HTREEITEM        pTreeObj;

          treeItem.mask = TVIF_TEXT | TVIF_IMAGE | TVIF_STATE | TVIF_PARAM | TVIF_SELECTEDIMAGE;
          treeItem.state = 0;
          //if ( pRec->flRecordAttr & CRA_SELECTED )
          //  treeItem.state |= TVIS_SELECTED;
          // TVIS_FOCUSED never worked, and is no longer supported by
          // the Windows 2000 SDK.
//        if ( pRec->flRecordAttr & CRA_CURSORED )
//          treeItem.state |= TVIS_FOCUSED;
          if ( pRec->flRecordAttr & CRA_TARGET )
            treeItem.state |= TVIS_DROPHILITED;
          if ( pRec->flRecordAttr & CRA_EXPANDED )
            treeItem.state |= TVIS_EXPANDED;

          treeItem.stateMask      = 0x00FF;
          treeItem.pszText        = pRec->strIcon;
          treeItem.cchTextMax     = 64;
          treeItem.iImage         = imageIndex;
          treeItem.iSelectedImage = imageIndex;
          treeItem.lParam         = (unsigned long)IObjFromRec(pRec);

          if ( pRecordInsert->pRecordParent )
          {
            IMiniCnrRecord*   pParentRec((IMiniCnrRecord*)
                                               pRecordInsert->pRecordParent);
            IContainerObject* pParentObj(IObjFromRec(pParentRec));

            tvis.hParent = pParentObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl);

            // If parent object not found, put it as a root object.
            if (tvis.hParent == 0)
               tvis.hParent = TVI_ROOT;
          }
          else
            tvis.hParent = TVI_ROOT;

          // Set insertion order.
          if ( pRecordInsert->pRecordOrder == (PRECORDCORE)CMA_FIRST )
            tvis.hInsertAfter = TVI_FIRST;
          else if ( pRecordInsert->pRecordOrder == (PRECORDCORE)CMA_END )
          {
            tvis.hInsertAfter = TVI_LAST;
          }
          else if ( pRecordInsert->pRecordOrder != 0 )
          {
            // Insert after the given object.
            IMiniCnrRecord*   pAfterRec((IMiniCnrRecord*)
                                               pRecordInsert->pRecordOrder);
            IContainerObject* pAfterObj(IObjFromRec(pAfterRec));

            tvis.hInsertAfter =
                         pAfterObj->ppd->pCnrSet->treeItemFromObject( pCnrCtl);

            // If insertion after object not found, put it at the end.
            if (tvis.hInsertAfter == 0)
               tvis.hInsertAfter = TVI_LAST;
          }

          // Set the insertion item to the tree item.
          tvis.item = treeItem;

          // Insert the item.
          if ( pTreeObj=TreeView_InsertItem(pCnrCtl->ppd->pTreeview->handle(),
                                            &tvis) )
          {
             if ( pRec->flRecordAttr & CRA_SELECTED )
               TreeView_SelectItem(pCnrCtl->ppd->pTreeview->handle(),
                                   pTreeObj) ;

            pCOSInfo = new ICnrObjSetInfo(pCnrCtl, pTreeObj, imageIndex);
            pCnrObj->ppd->pCnrSet->add( pCOSInfo );

            event.setResult( true );
          }
          else
            event.setResult( false );
        }

        //If a ListView exists, then insert the object there, providing
        //the object is not a child object.
        if ( pCnrCtl->ppd->pListview && !pRecordInsert->pRecordParent )
        {
          // If there is no tree view, put it in the object set as it wouldn't
          // have been done before.
          if ( !pCnrCtl->ppd->pTreeview )
          {
            pCOSInfo = new ICnrObjSetInfo(pCnrCtl, 0, imageIndex);
            pCnrObj->ppd->pCnrSet->add( pCOSInfo );
          }

          LV_ITEM   item;

          item.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM | LVIF_STATE;
          item.state = 0;
          if ( pRec->flRecordAttr & CRA_SELECTED )
            item.state |= LVIS_SELECTED;
          if ( pRec->flRecordAttr & CRA_CURSORED )
            item.state |= LVIS_FOCUSED;
          if ( pRec->flRecordAttr & CRA_TARGET )
            item.state |= LVIS_DROPHILITED;

          item.stateMask = 0x000F;

          // Set the insertion order.
          if ( pRecordInsert->pRecordOrder == (PRECORDCORE)CMA_FIRST )
            item.iItem = 0;
          else if ( pRecordInsert->pRecordOrder == (PRECORDCORE)CMA_END )
          {
            int  count=ListView_GetItemCount(pCnrCtl->ppd->pListview->handle());
            item.iItem = count;
          }
          else
          {  //Need to insert after pRecordInsert->pRecordOrder
            IMiniCnrRecord*   pAfterRec((IMiniCnrRecord*)
                                               pRecordInsert->pRecordOrder);
            IContainerObject* pAfterObj(IObjFromRec(pAfterRec));
            int           itemIndex;
            LV_FINDINFO   lvFindInfo;
            lvFindInfo.flags  = LVFI_PARAM;
            lvFindInfo.lParam = (LPARAM)pAfterObj;

            // Get the objects list view index.
            itemIndex = IContainerControl::ListViewFindItem(
                                           pCnrCtl->ppd->pListview->handle(),
                                           (-1), &lvFindInfo );
            item.iItem = itemIndex + 1;
          }

          item.iSubItem = 0;
          item.pszText = pRec->strIcon;
          item.cchTextMax = 64;
          item.iImage = imageIndex;
          item.lParam = (unsigned long)IObjFromRec(pRec);

          if (ListView_InsertItem(pCnrCtl->ppd->pListview->handle(), &item) == -1)
            event.setResult( false );
          else
          {
             if ( pRec->ptlIcon.x != 0 || pRec->ptlIcon.y !=0 )
             {
                ListView_SetItemPosition32( pCnrCtl->ppd->pListview->handle(),
                                            item.iItem, pRec->ptlIcon.x,
                                            pRec->ptlIcon.y );
             }
             event.setResult( true );
          }
        }

        if ( pRec->flRecordAttr & CRA_RECORDREADONLY )
          pCOSInfo->setObjectState(0, CRA_RECORDREADONLY);
        if ( pRec->flRecordAttr & CRA_DROPONABLE )
          pCOSInfo->setObjectState(0, CRA_DROPONABLE);
       }

      fStopHandling = true;
      break;
    }

    case CM_INVALIDATEDETAILFIELDINFO:
      /*******************************************************************/
      /* Invalidate all details items.                                   */
      /*******************************************************************/
      // If not in tree view, redraw.
      if ( !pCnrCtl->isTreeView() )
      {
        ListView_RedrawItems( pCnrCtl->ppd->pListview->handle(),
                              0,
                              ListView_GetItemCount(
                                        pCnrCtl->ppd->pListview->handle()) );

        if (pCnrCtl->flClState & IContainerControl::invalidFrame)
        {
           // Force redraw of non client area (notably scroll bars).
           ITRACE_DEVELOP("Processing IContainerControl::invalidFrame");
           SetWindowPos( pCnrCtl->ppd->pListview->handle(),
                         0,     //hwndAfter (ignored)
                         0, 0,  //pos  (ignored)
                         0, 0,  //size (ignored)
                         SWP_NOACTIVATE| SWP_NOZORDER | SWP_NOMOVE | SWP_NOSIZE |
                         SWP_FRAMECHANGED );
           pCnrCtl->flClState &= ~IContainerControl::invalidFrame;
        }
        InvalidateRect( pCnrCtl->ppd->pListview->handle(), 0, true );
      }
      event.setResult( true );
      fStopHandling = true;
    break;

    case CM_INVALIDATERECORD:
    {
      /*******************************************************************/
      /* Only CMA_xx flags needed by the IContainerControl class are     */
      /* supported.  Others are ignored.                                 */
      /* Only one object or all objects can be invalidated.              */
      /*******************************************************************/
      IContainerObject* pCnrObj;
      unsigned short    cNumRecord       (event.parameter2().lowNumber());
      unsigned short    fInvalidateRecord(event.parameter2().highNumber());
      //Throw an exception if cNumRecord != 0 (invalidate all) or != 1
      // (invalidate one).
      IASSERTSTATE( cNumRecord <= 1 );

      if ( cNumRecord )
      {
        //parameter1 is a pointer to a IMiniCnrRecord*
        IMiniCnrRecord*
             pRec(*((IMiniCnrRecord**)event.parameter1().asUnsignedLong()));
        pCnrObj = IObjFromRec(pRec);

        // Add/replace image in list but ONLY if an icon is specified.
        int imageIndex = -1;
        if (pRec->hptrIcon)
          imageIndex = pCnrCtl->ppd->addIconToList(pRec->hptrIcon);
        int  itemIndex=(-1);

        // Emphasis states need to be updated.
        // We update them immediately, via CM_SETRECORDEMPHASIS.

        // If there is a list view, update its state information.
        if ( pCnrCtl->ppd->pListview )
        {
          //If a ListView exists, then update the icon view position.
          LV_FINDINFO   lvFindInfo;
          lvFindInfo.flags  = LVFI_PARAM;
          lvFindInfo.lParam = (LPARAM)pCnrObj;

          // Get the object's list view index.
          itemIndex = IContainerControl::ListViewFindItem(
                                         pCnrCtl->ppd->pListview->handle(),
                                         (-1), &lvFindInfo );

          if ( itemIndex != (-1) )
          {
            ListView_SetItemPosition32( pCnrCtl->ppd->pListview->handle(),
                                        itemIndex, pRec->ptlIcon.x,
                                        pRec->ptlIcon.y );

            if ( fInvalidateRecord & CMA_TEXTCHANGED ) // the icon changed
            {
               LV_ITEM lvItem;

               // Prime the lv item
               lvItem.iItem    = itemIndex;
               lvItem.mask     = LVIF_IMAGE | LVIF_TEXT;
               lvItem.iSubItem = 0;

               lvItem.iImage = imageIndex;
               lvItem.pszText = pRec->strIcon;
               ListView_SetItem( pCnrCtl->ppd->pListview->handle(), &lvItem );
            }
            // Redraw ListView only if we're not in tree view.
            if ( !pCnrCtl->isTreeView() )
            {
              ListView_RedrawItems( pCnrCtl->ppd->pListview->handle(),
                                    itemIndex, itemIndex);

              if (pCnrCtl->isRefreshOn())
                 UpdateWindow( pCnrCtl->ppd->pListview->handle() );
              else
                 InvalidateRect( pCnrCtl->ppd->pListview->handle(), 0, true );
            }
          }
        }
        //If there is a tree view, update its state information.
        if (pCnrCtl->ppd->pTreeview)
        {
          ICnrObjSetInfo* pObjInfo = pCnrObj->ppd->pCnrSet->objectFromCnr( pCnrCtl );
          HTREEITEM treeItem = pObjInfo->pTVItem;

          if (treeItem && (fInvalidateRecord & CMA_TEXTCHANGED)) // the icon changed
          {
            TV_ITEM tvItem;
            tvItem.hItem = treeItem;
            tvItem.mask  = TVIF_IMAGE | TVIF_SELECTEDIMAGE | TVIF_TEXT;
            tvItem.iImage =
            tvItem.iSelectedImage = imageIndex;
            tvItem.pszText = pRec->strIcon;
            TreeView_SetItem( pCnrCtl->ppd->pTreeview->handle(), &tvItem );
          }
        }
      }
      else   //All objects
      {
        // If we're not in tree view, redraw the ListView.
        if ( !pCnrCtl->isTreeView() )
        {
          ListView_RedrawItems( pCnrCtl->ppd->pListview->handle(),
                                0,
                                ListView_GetItemCount(
                                          pCnrCtl->ppd->pListview->handle()) );

          InvalidateRect( pCnrCtl->ppd->pListview->handle(), 0, true );
        }
      }

      // If we're in tree view, redraw it.
      if ( pCnrCtl->isTreeView() )
        InvalidateRect( pCnrCtl->ppd->pTreeview->handle(), 0, true );

      fStopHandling = true;
      event.setResult( true );
    }
    break;


    case CM_OPENEDIT:
    {
      event.setResult( true );

      /*********************************************************************/
      /* Open an edit field.  We assume that a valid object is coming in   */
      /* the CNREDITDATA structure.                                        */
      /*********************************************************************/
      PCNREDITDATA
           pCnrEditData((PCNREDITDATA)event.parameter1().asUnsignedLong());
      IMiniCnrRecord*   pRec((IMiniCnrRecord*)pCnrEditData->pRecord);
      IContainerObject* pCnrObj(IObjFromRec(pRec));

      // Only attempt to open the edit field if the style is not read only and
      // there is a valid object.
      if (!(pCnrCtl->style() & IContainerControl::readOnly.asUnsignedLong()) &&
          pCnrObj && pCnrObj->isWriteable())
      {
        if ( pCnrCtl->isTreeView() )     //currently in a tree view.
        {
          //Convert the ICnrObj* to an HTREEITEM and start the treeview edit.
          event.setResult( TreeView_EditLabel(
                             pCnrCtl->ppd->pTreeview->handle(),
                             pCnrObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl)));
        }
        else       //currently not in a tree view.
        {
          //Convert the ICnrObj* to an index and start the listview edit.
          int          index;
          LV_FINDINFO  lvFindInfo;

          lvFindInfo.flags  = LVFI_PARAM;
          lvFindInfo.lParam = (unsigned long)pCnrObj;

          index = IContainerControl::ListViewFindItem(
                                    pCnrCtl->ppd->pListview->handle(),
                                    (-1), &lvFindInfo);

          // Process the edit request on a details view column.
          if ( pCnrCtl->isDetailsView() )
          {
            //Find the column.
            IContainerColumn* pColumn((IContainerColumn*)pCnrEditData->pFieldInfo->pUserData);

            // Only edit if the data is of type string and is not read only.
            if ((pColumn->pfieldinfoCl->flData & CFA_STRING) &&
                pColumn->isWriteable())
            {
              // If there is already a field open, we have to close it.
              if (pCnrCtl->ppd->pDetailsEdit)                      /* D8067 */
                // Send a fake enter message to the mle so that it will close.
                pCnrCtl->ppd->pDetailsEdit->sendEvent( WM_CHAR,
                                                       IEventParameter1( VK_RETURN ));

              // If we are in details view then ensure that we are not
              // attempting to edit the required column.
              if ( pCnrEditData->pFieldInfo &&
                   (pCnrCtl->ppd->columnSet->columnIdFromFieldInfo(pCnrEditData->pFieldInfo)
                    != 0) )
              {
                // Save the column and container objects so that we can
                // retrieve the edit info when sending the CN_REALLOCPSZ
                // and CN_EDITEDIT notifications.
                pCnrCtl->ppd->pEditColumn = pColumn;
                pCnrCtl->ppd->pEditObject = pCnrObj;

                // Set up and create the MLE.
                LPTSTR *pColumnText(((LPTSTR*)(((LPBYTE)pRec) + pColumn->pfieldinfoCl->offStruct)));

                IRectangle detailsRect( pCnrCtl->detailsObjectRectangle( pCnrObj,
                                                                         pColumn ));

                IFont listviewFont( pCnrCtl->ppd->pListview->font() );

                detailsRect.sizeTo(ISize((listviewFont.textWidth( *pColumnText )*1.5),
                                         detailsRect.height()));

                detailsRect.shrinkBy( IPair(0, (detailsRect.height()-
                                 (listviewFont.maxCharHeight()+8))/2 ));

                // Shift edit control right by approx margin if left-justified.
                if (pColumn->pfieldinfoCl->flData & CFA_LEFT)
                  detailsRect.moveBy( IPair( listviewFont.textWidth( "xi" ), 0 ));

                pCnrCtl->ppd->pDetailsEdit = new IMultiLineEdit( CID_MLE,
                                                   pCnrCtl,
                                                   pCnrCtl,
                                                   detailsRect,
                                                   IWindow::visible | IMultiLineEdit::border );

                pCnrCtl->ppd->pDetailsEdit->setAutoDeleteObject( true );

                pCnrCtl->ppd->pDetailsEdit->setText( *pColumnText ).
                // select all the text in the mle
                              selectRange().
                // Set the font of the edit window to use the font of the list view control.
                              setFont( listviewFont );

                // Set the MLE handler.
                IDetailsEditHandler::defaultHandler()->handleEventsFor(
                                                        pCnrCtl->ppd->pDetailsEdit);

                // Send a notification (CN_BEGINEDIT) back to the owner.
                CNREDITDATA cnrEditData;
                cnrEditData.cb         = sizeof( CNREDITDATA );
                cnrEditData.hwndCnr    = pCnrCtl->handle();
                cnrEditData.pRecord    = (RECORDCORE*)pRec;
                cnrEditData.pFieldInfo = pColumn->pfieldinfoCl;
                cnrEditData.ppszText   = pColumnText;
                cnrEditData.cbText     = strlen( *pColumnText );
                cnrEditData.id         = pCnrCtl->id();

                pCnrCtl->owner()->sendEvent( WM_CONTROL,
                                        IEventParameter1( cnrEditData.id, CN_BEGINEDIT ),
                                        IEventParameter2( &cnrEditData ) );

                // Set focus to the MLE.
                pCnrCtl->ppd->pDetailsEdit->setFocus();
              }
              else // Details view required column edit - let ListView do it.
              {
                event.setResult( ListView_EditLabel(pCnrCtl->ppd->pListview->handle(),
                                                   index) );
              }
            }
          }
          else // Non-details list view - let ListView do it.
          {
            event.setResult( ListView_EditLabel(pCnrCtl->ppd->pListview->handle(),
                                               index) );
          }
        }
      }
      fStopHandling = true;
    }
    break;

    case CM_PAINTBACKGROUND:
      /********************************************************************/
      /* This message is generated by the CCL control.  The CCL container */
      /* sends this message to itself, thereby allowing the owner to      */
      /* subclass and process this.  There is no equivalent sent by either*/
      /* the native ListView or TreeView, thus handleDrawBackground will  */
      /* never be called.                                                 */
      /********************************************************************/
    break;

    case CM_QUERYCNRINFO:
      /********************************************************************/
      /* Query all the attributes of the container and insert them into   */
      /* the ICnrInfo structure.                                          */
      /********************************************************************/
      ((IContainerControl*)event.window())->ppd->queryCnrInfo(
                            (ICnrInfo*)event.parameter1().asUnsignedLong() );
      event.setResult( true );
      fStopHandling = true;
    break;

    case CM_QUERYDETAILFIELDINFO:
    {
      /********************************************************************/
      /* Query details attributes of the container.  Not all CMA_xx flags */
      /* are supported.                                                   */
      /********************************************************************/
      LPFIELDINFO
            pFI((LPFIELDINFO)event.parameter1().asUnsignedLong());
      unsigned short ulCmd(event.parameter2().lowNumber());

      // Only respond to this message if a list view exists.
      if ( pCnrCtl->ppd->pListview )
      {
        switch(ulCmd)
        {
          case CMA_FIRST:
            event.setResult( pCnrCtl->ppd->columnSet->firstFieldInfo() );
          break;

          case CMA_LAST:
            event.setResult( pCnrCtl->ppd->columnSet->lastFieldInfo() );
          break;

          case CMA_NEXT:
            event.setResult( pCnrCtl->ppd->columnSet->nextFieldInfo(pFI) );
          break;

          case CMA_PREV:
            event.setResult( pCnrCtl->ppd->columnSet->prevFieldInfo(pFI) );
          break;

          case CMA_DATAWIDTH:
          {
            int subItem( pCnrCtl->ppd->columnSet->subItemIdFromFieldInfo(pFI) );
            if ( subItem > (-10) )
            {
              int colIndex =
                       pCnrCtl->ppd->columnSet->columnIdFromFieldInfo(pFI);

              if ( (colIndex==(-1)) &&
                   (pCnrCtl->ppd->columnSet->containerColumnFromSubId(0)) )
                event.setResult( 0 );
              else
              {
                if ( colIndex==(-1) )
                  colIndex = 0;
                event.setResult( ListView_GetColumnWidth(
                                 pCnrCtl->ppd->pListview->handle(), colIndex) );
              }
            }
            else
              event.setResult( 0 );
          }
          break;

          default:
            event.setResult( (LPFIELDINFO)(-1) );
          break;
        }
      }
      else
        event.setResult( 0 );

      fStopHandling = true;
    }
    break;

    case CM_QUERYRECORD:
    {
      /********************************************************************/
      /* Return the requested record according to the passed in CMA_xx    */
      /* flag.                                                            */
      /* For the purposes of the IContainerControl class, the only        */
      /* implementation necessary is for the ObjectCursor case.  Other    */
      /* uses may fail.                                                   */
      /********************************************************************/
      unsigned short ulCmd(event.parameter2().lowNumber());

      unsigned long  ulArg(event.parameter1().asUnsignedLong());
      IWindowHandle  hControl;

      // If a tree view exists, use if for cursoring purposes so children
      // are included.
      if ( pCnrCtl->ppd->pTreeview )
      {
        TV_ITEM    item;
        HTREEITEM  pTreeItem, pNextItem;
        IContainerObject*  pObj=0;
        hControl = pCnrCtl->ppd->pTreeview->handle();

        IMiniCnrRecord* pRec((IMiniCnrRecord*)ulArg);

        if ( pRec )
          pObj = IObjFromRec( pRec );
        switch(ulCmd)
        {
          case CMA_FIRST:
            pTreeItem =  TreeView_GetRoot(hControl);
          break;

          case CMA_FIRSTCHILD:
            pTreeItem = TreeView_GetChild(hControl,
                            pObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) );
          break;

          case CMA_LAST:
          {
            pNextItem =
            pTreeItem =  TreeView_GetRoot(hControl);

            while (pNextItem)
            {
              pTreeItem = pNextItem;
              pNextItem = TreeView_GetNextSibling(hControl,
                                                  pNextItem );
            }
          }
          break;

          case CMA_LASTCHILD:
          {
            //Check for pObj=0, then same as CMA_LAST
            if ( pObj )
              pNextItem =
              pTreeItem = TreeView_GetChild(hControl,
                              pObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) );
            else
              pNextItem =
              pTreeItem =  TreeView_GetRoot(hControl);

            while (pNextItem)
            {
              pTreeItem = pNextItem;
              pNextItem = TreeView_GetNextSibling(hControl,
                                                  pNextItem );
            }
          }
          break;

          case CMA_NEXT:
            pTreeItem = TreeView_GetNextSibling(hControl,
                            pObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) );
          break;

          case CMA_PARENT:
            pTreeItem = TreeView_GetParent(hControl,
                            pObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) );
          break;

          case CMA_PREV:
            pTreeItem = TreeView_GetPrevSibling(hControl,
                            pObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) );
          break;

          default:
            event.setResult( -1 );
            fStopHandling=true;
          break;
        }

        // If an item was found, get the corresponding container object and
        // set the result.
        if (!fStopHandling)
        {
          if ( pTreeItem )
          {
            item.mask =  TVIF_PARAM;
            item.hItem = pTreeItem;

            TreeView_GetItem(hControl, &item);
            fStopHandling = true;
            event.setResult( (IMiniCnrRecord*)IRecFromObj(
                                              (IContainerObject*)item.lParam) );
          }
          else
            event.setResult( 0 );
        }
      }
      // If only a list view exists, use it for cursoring purposes.
      else
      {
         //----------------------------------------------------------------------------
         // DEFECT 8860 :
         // The arg passed in is supposed to be an IMiniCnrRecord*, but in
         // the IC_WIN case (except called by parentObject(), which is ignored):
         //  IF it is NOT a pm-style container, and
         //  IF it is NOT a tree-view contiiner,
         //  THEN it is an ObjectCursor*, NOT a record*.
         //----------------------------------------------------------------------------
         IContainerControl::ObjectCursor* pObjCur(0);
#ifdef IC_WIN
         if (!pCnrCtl->ppd->pmCompatible /*&& !pCnrCtl->ppd->pTreeview*/)
            pObjCur = ((IContainerControl::ObjectCursor*)ulArg);
#endif
         else
            IContainerControl::ObjectCursor* pObjCur((IContainerControl::ObjectCursor*)
                                               event.parameter1().asUnsignedLong());

        LV_ITEM   item;
        item.iSubItem = 0;
        hControl = pCnrCtl->ppd->pListview->handle() ;

        switch(ulCmd)
        {
          case CMA_FIRST:
            item.iItem=0;
          break;

          case CMA_LAST:
            item.iItem = ListView_GetItemCount(hControl) - 1 ;
          break;

          case CMA_NEXT:
            item.iItem= (pObjCur ? pObjCur->currentIndex + 1 : 0 ) ;
          break;

          case CMA_PREV:
            item.iItem = (pObjCur ? pObjCur->currentIndex - 1 : 0 ) ;
          break;

          case CMA_FIRSTCHILD:
          case CMA_LASTCHILD:
          case CMA_PARENT:
            event.setResult( 0 );
            fStopHandling=true;
          break;

          default:
            event.setResult( -1 );
            fStopHandling=true;
          break;
        }
        // If an item was found, get the corresponding container object and
        // set the result.
        if (!fStopHandling)
        {
          item.mask = LVIF_PARAM;
          if ( ListView_GetItem(hControl, &item) )
          {
            pObjCur->currentIndex = item.iItem;
            event.setResult( (IMiniCnrRecord*)IRecFromObj(
                                              (IContainerObject*)item.lParam) );
          }
          else
            event.setResult( 0 );
        }
      }
      fStopHandling = true;
    }
    break;

    case CM_QUERYRECORDEMPHASIS:
    {
      /*******************************************************************/
      /* Query the object's current emphasis states for this container.  */
      /* This is used by the cursor classes.                             */
      /*******************************************************************/
      unsigned long      ulEmphasis(event.parameter2().asUnsignedLong());
      IMiniCnrRecord*    pStartRec((IMiniCnrRecord*)event.parameter1().
                                                    asUnsignedLong());
      IContainerObject*  pStartObj=0;

      if ( pStartRec  && ( pStartRec != (IMiniCnrRecord*)CMA_FIRST ) )
        pStartObj = IObjFromRec( pStartRec );

      if ( pCnrCtl->isTreeView() )
      {
        TV_ITEM    item;
        HTREEITEM  pTreeItem;

        if ( ulEmphasis & CRA_INUSE )
        {
          if ( ulEmphasis & ( CRA_SELECTED | CRA_CURSORED ) )
          {
            pTreeItem =TreeView_GetSelection(pCnrCtl->ppd->pTreeview->handle());
            if ( pStartObj &&
                 (pStartObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) ==
                  pTreeItem) )
              event.setResult( 0 );
            else
            {
              item.mask =  TVIF_PARAM;
              item.hItem = pTreeItem;

              if (TreeView_GetItem(pCnrCtl->ppd->pTreeview->handle(), &item))
              {
                IContainerObject*  pCnrObj;
                pCnrObj = (IContainerObject*)(item.lParam);
                if ( pCnrObj->ppd->pCnrSet->isInUse(pCnrCtl) )
                  event.setResult( pCnrObj );
              }
            }
          }
          else
          {
            IContainerObject*                pCnrObj;
            IContainerControl::ObjectCursor  cursor(*pCnrCtl);

            if ( pStartObj )
              cursor.setCurrent( pStartObj );

            // Else do Nothing. Thus when setToNext() is called in
            // the cursor below, it will actually go to first(CMA_FIRST)

            event.setResult( false );

            //Cursor through and query in-use for all objects in the tree view.
            //Need to stop cursor.
            for( cursor.setToNext(); cursor.isValid(); cursor.setToNext() )
            {
              pCnrObj = pCnrCtl->objectAt(cursor);
              if ( pCnrObj->ppd->pCnrSet->isInUse(pCnrCtl) )
              {
                event.setResult( pCnrObj );
                cursor.invalidate();   // Stop cursor
              }
            }
          }
        }
        else    //No in-use emphasis to worry about.
        {
          if ( ulEmphasis & ( CRA_SELECTED | CRA_CURSORED ) )
          {
            pTreeItem =TreeView_GetSelection(pCnrCtl->ppd->pTreeview->handle());
            if (pStartObj && ( pStartObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) ==
                          pTreeItem ) )
              event.setResult( 0 );
            else
            {
              item.mask =  TVIF_PARAM;
              item.hItem = pTreeItem;

              if (TreeView_GetItem(pCnrCtl->ppd->pTreeview->handle(), &item))
                event.setResult((IMiniCnrRecord*)IRecFromObj((IContainerObject*)item.lParam));
            }
          }
          // ulEmphasis parameter must be invalid.
          else
            event.setResult( -1 );
        }
      }
      else  // ListView only.
      {
        LV_ITEM item;
        int     index=(-1);
        UINT    flags=0;

        // Get current index.
        if (pStartObj)
        {
          LV_FINDINFO  lvFindInfo;

          lvFindInfo.flags  = LVFI_PARAM;
          lvFindInfo.lParam = (unsigned long)pStartObj;

          index = IContainerControl::ListViewFindItem(
                                    pCnrCtl->ppd->pListview->handle(),
                                    (-1), &lvFindInfo);
        }
        if ( ulEmphasis & CRA_INUSE )
        {
          if ( ulEmphasis & ( CRA_SELECTED | CRA_CURSORED ) )
          {
            IContainerObject*  pCnrObj;

            if (ulEmphasis & CRA_SELECTED)
              flags |= LVNI_SELECTED;

            if (ulEmphasis & CRA_CURSORED)
              flags |= LVNI_FOCUSED;

            while ( index != (-1) )
            {
              index = ListView_GetNextItem(pCnrCtl->ppd->pListview->handle(),
                                           index, flags );

              if ( index != (-1) )
              {
                item.mask =  LVIF_PARAM;
                item.iItem = index;
                item.iSubItem = 0;
                ListView_GetItem(pCnrCtl->ppd->pListview->handle(), &item);

                pCnrObj = (IContainerObject*)(item.lParam);
                if ( pCnrObj->ppd->pCnrSet->isInUse(pCnrCtl) )
                {
                  index = (-1);
                  event.setResult( IRecFromObj(pCnrObj) );
                }
              }
              else
                event.setResult( 0 );
            }
          }
          else   // In-use only.
          {
            IContainerObject*   pCnrObj;
            bool  notFound = true;
            int  numListItems( ListView_GetItemCount(
                                         pCnrCtl->ppd->pListview->handle()) );

            for (int iter=index+1; iter < numListItems; iter++)
            {
              item.mask =  LVIF_PARAM;
              item.iItem = iter;
              item.iSubItem = 0;
              ListView_GetItem(pCnrCtl->ppd->pListview->handle(), &item);
              pCnrObj = (IContainerObject*)(item.lParam);

              if ( pCnrObj->ppd->pCnrSet->isInUse(pCnrCtl) )
              {
                notFound = false;
                iter     = numListItems;
                event.setResult( IRecFromObj(pCnrObj) );
              }
            }

            if ( notFound )
              event.setResult( 0 );
          }
        }
        else    // No in-use emphasis to worry about.
        {
          if (ulEmphasis & CRA_SELECTED)
            flags |= LVNI_SELECTED;

          if (ulEmphasis & CRA_CURSORED)
            flags |= LVNI_FOCUSED;

          item.iItem = ListView_GetNextItem(pCnrCtl->ppd->pListview->handle(),
                                            index, flags );

          if ( item.iItem != (-1) )
          {
            item.mask =  LVIF_PARAM;
            item.iSubItem = 0;
            ListView_GetItem(pCnrCtl->ppd->pListview->handle(), &item);

            event.setResult( IRecFromObj((IContainerObject*)item.lParam) );
          }
          else
            event.setResult( 0 );
        }
      }
      fStopHandling = true;
    }
    break;

    case CM_QUERYRECORDFROMRECT:
    {
      /*******************************************************************/
      /* Get the object intersecting the given rectangle.                */
      /*                                                                 */
      /* Only what IContainerControl requires has been implemented.      */
      /* We only allow a rectangle of a point to determine the           */
      /* object under that point.                                        */
      /* If the starting point is anything other than CMA_FIRST, we      */
      /* will return 0.                                                  */
      /*******************************************************************/
      IMiniCnrRecord*
          pSearchAfter( (IMiniCnrRecord*)event.parameter1().asUnsignedLong() );
      LPQUERYRECFROMRECT
       pQryRecFromRect((LPQUERYRECFROMRECT)event.parameter2().asUnsignedLong());

      // Have to start search after CMA_FIRST - all others rejected.
      if ( pSearchAfter == (IMiniCnrRecord*)CMA_FIRST )
      {
        // If we're in tree view, find the object using hit testing.
        if ( pCnrCtl->isTreeView() )
        {
          HTREEITEM        treeHandle;
          TV_HITTESTINFO   treeHitTestInfo;

          treeHitTestInfo.pt.x  = pQryRecFromRect->rect.right;
          treeHitTestInfo.pt.y  = pQryRecFromRect->rect.bottom;
          treeHitTestInfo.flags = TVHT_ONITEM | TVHT_ONITEMICON |
                                  TVHT_ONITEMLABEL;
          treeHandle = TreeView_HitTest( pCnrCtl->ppd->pTreeview->handle(),
                                         &treeHitTestInfo );
          // If tree item found, convert it to an object and set the result.
          if ( treeHandle )
          {
            TV_ITEM    treeItem;

            treeItem.hItem = treeHandle;
            treeItem.mask  = TVIF_PARAM;

            TreeView_GetItem( pCnrCtl->ppd->pTreeview->handle(), &treeItem );
            event.setResult( IRecFromObj( (IContainerObject*)treeItem.lParam) );
          }
          // No item found - set result to zero.
          else
            event.setResult( 0 );
        }
        // We're in list view - find item using hit testing.
        else
        {
          int              itemIndex;
          LV_HITTESTINFO   listHitTestInfo;

          listHitTestInfo.pt.x  = pQryRecFromRect->rect.right;
          listHitTestInfo.pt.y  = pQryRecFromRect->rect.bottom;
          listHitTestInfo.flags = LVHT_ONITEMICON | LVHT_ONITEMLABEL |
                                  LVHT_ONITEMSTATEICON;
          itemIndex = ListView_HitTest( pCnrCtl->ppd->pListview->handle(),
                                        &listHitTestInfo );
          // If list item found, convert it to an object and set the result.
          if ( itemIndex != (-1) )
          {
            LV_ITEM    listItem;

            listItem.iItem    = itemIndex;
            listItem.iSubItem = 0;
            listItem.mask     = LVIF_PARAM;

            ListView_GetItem( pCnrCtl->ppd->pListview->handle(), &listItem );
            event.setResult( IRecFromObj( (IContainerObject*)listItem.lParam) );
          }
          // No item found.
          else
          {
            // If we're in details view, do our own hit testing.
            if ( pCnrCtl->isDetailsView() &&
                 ListView_GetItemCount(pCnrCtl->ppd->pListview->handle()) > 0  )
            {
              //Check to see if point is over a column other than the first.
              //HitTest only checks the first column.

              POINT viewPortOffset;
              int   index( 0 );
              bool  notDone( true );
              long  leftSide( 0 ),
                    rightSide( 0 );

              ListView_GetItemPosition( pCnrCtl->ppd->pListview->handle(),
                                        0, &viewPortOffset );
              leftSide += viewPortOffset.x;


              //Check and see if this point fits vertically in any of
              //the details view rows.
              itemIndex = (-1);
              int numListItems(ListView_GetItemCount(
                                          pCnrCtl->ppd->pListview->handle()));
              RECTL  rect;
              for (index=0; index < numListItems; index++)
              {
                ListView_GetItemRect( pCnrCtl->ppd->pListview->handle(),
                                      index, &rect, LVIR_BOUNDS );
                if ( (rect.top <= listHitTestInfo.pt.y) &&
                     (listHitTestInfo.pt.y <= rect.bottom) )
                {
                  itemIndex = index;
                  index = numListItems;
                }
              }

              // Initialize result to zero in case item not found.
              event.setResult( 0 );

              if ( itemIndex != (-1) )
              {
                // We have a potential candidate lets see if the given
                // x value falls within the other columns.
                // If item found here, set the result.
                int  numColumns( pCnrCtl->ppd->columnSet->columnCount() );
                for (index=0; index < numColumns; index++)
                {
                  rightSide = leftSide + ListView_GetColumnWidth(
                                    pCnrCtl->ppd->pListview->handle(), index );
                  if ( rightSide > leftSide )
                  {
                    if ( (leftSide <= pQryRecFromRect->rect.right) &&
                         (pQryRecFromRect->rect.right <= rightSide ) )
                    {
                      LV_ITEM    listItem;

                      listItem.iItem    = itemIndex;
                      listItem.iSubItem = 0;
                      listItem.mask     = LVIF_PARAM;

                      ListView_GetItem( pCnrCtl->ppd->pListview->handle(),
                                        &listItem );
                      event.setResult( IRecFromObj(
                                        (IContainerObject*)listItem.lParam) );
                      index = numColumns;
                    }
                    leftSide = rightSide;
                  }
                }
              }
            }
            // No item found and not in details view - set result to zero.
            else
              event.setResult( 0 );
          }
        }
      }
      // Search after other than CMA_FIRST specified.
      else
        event.setResult( 0 );

      fStopHandling = true;
    }
    break;

    case CM_QUERYRECORDINFO:
    {
      /*******************************************************************/
      /* Update the object's flRecordAttr and ptlIcon fields.            */
      /*                                                                 */
      /* Only what IContainerControl requires has been implemented.      */
      /* We only allow cNumRecord = 1.  An exception is thrown for       */
      /* all other values.                                               */
      /*******************************************************************/
      IMiniCnrRecord*
           pRec(*((IMiniCnrRecord**)event.parameter1().asUnsignedLong()));
      unsigned short    cNumRecord   (event.parameter2().lowNumber());
      IContainerObject* pObj;
      TV_ITEM       treeItem;
      LV_FINDINFO   lvFindInfo;
      int           itemIndex;
      bool          existsInTree = false;

      IASSERTSTATE( cNumRecord = 1 );
      IASSERTSTATE( pRec != 0 );
      pRec->flRecordAttr = 0;
      pRec->ptlIcon.x = pRec->ptlIcon.y = 0;
      pObj = IObjFromRec( pRec );

     // If a ListView control exists, then get the item's icon view
      // position there even if currently in a non-tree view.
      // Otherwise, let it default to (0,0)
      if ( pCnrCtl->ppd->pListview )
      {
        lvFindInfo.flags  = LVFI_PARAM;
        lvFindInfo.lParam = (LPARAM)pObj;

        // Get the object's list view index.
        itemIndex = IContainerControl::ListViewFindItem(
                                       pCnrCtl->ppd->pListview->handle(),
                                       (-1), &lvFindInfo );

        if ( itemIndex != (-1) )
          ListView_GetItemPosition( pCnrCtl->ppd->pListview->handle(),
                                    itemIndex, &(pRec->ptlIcon) );
      }

      // If a TreeView control exists, then check for collapsed/expanded
      // there even if currently in a non-tree view.
      if ( pCnrCtl->ppd->pTreeview )
      {
        treeItem.mask      = TVIF_STATE;
        treeItem.stateMask = 0x00FF;
        treeItem.hItem     = pObj->ppd->pCnrSet->treeItemFromObject( pCnrCtl );

        existsInTree = TreeView_GetItem( pCnrCtl->ppd->pTreeview->handle(),
                                         &treeItem );
        if ( existsInTree )
        {
          if ( treeItem.state & TVIS_EXPANDED )
            pRec->flRecordAttr |= CRA_EXPANDED;
          else
            pRec->flRecordAttr |= CRA_COLLAPSED;
        }
      }
      else  //Default to collapsed, as CCL does.
        pRec->flRecordAttr |= CRA_COLLAPSED;

      // Check the object's private data for inUse emphasis.
      if ( pObj->ppd->pCnrSet->isInUse(pCnrCtl) )
        pRec->flRecordAttr |= CRA_INUSE;

      // Check the object's private data for readonly attribute.
      if ( pObj->ppd->pCnrSet->isReadOnly(pCnrCtl) )
        pRec->flRecordAttr |= CRA_RECORDREADONLY;

      // Check the object's private data for droponable attribute.
      if ( pObj->ppd->pCnrSet->isDropOnable(pCnrCtl) )
        pRec->flRecordAttr |= CRA_DROPONABLE;

      // Check the current view for the emphasis attributes.
      if ( pCnrCtl->isTreeView() )
      {
        if ( existsInTree )
        {
          // TVIS_FOCUSED never worked, and is no longer supported by
          // the Windows 2000 SDK.
//        if ( treeItem.state & TVIS_FOCUSED )
//          pRec->flRecordAttr |= CRA_CURSORED;
          if ( treeItem.state & TVIS_SELECTED )
            pRec->flRecordAttr |= CRA_SELECTED;
          if ( treeItem.state & TVIS_DROPHILITED )
            pRec->flRecordAttr |= CRA_TARGET;
        }
        else
        {
          //Couldn't find the object in the TreeView.
          //Thus it must not be in the ListView either because all
          //objects in the ListView must be in the TreeView.
          fStopHandling = true;
        }
      }
      else       //listview
      {
        unsigned int  itemState(0);

        if ( itemIndex != (-1) )
        {
          itemState = ListView_GetItemState( pCnrCtl->ppd->pListview->handle(),
                                             itemIndex, 0x000F );

          if ( itemState & LVIS_FOCUSED )
            pRec->flRecordAttr |= CRA_CURSORED;
          if ( itemState & LVIS_SELECTED )
            pRec->flRecordAttr |= CRA_SELECTED;
          if ( itemState & LVIS_DROPHILITED )
            pRec->flRecordAttr |= CRA_TARGET;

          event.setResult(true );
        }
        else
          //Couldn't find the object in the ListView.
          //Its possible it is a child object in the TreeView.
          if ( !existsInTree )
            fStopHandling = true;
      }

      // If fStopHandling = true, the object doesn't exist in this container.
      // Set the event result appropriately.
      event.setResult( !fStopHandling );
      fStopHandling = true;
    }
    break;

    case CM_QUERYRECORDRECT:
    {
      /*******************************************************************/
      /* Get the rectangle for the given object.                         */
      /*******************************************************************/
      IContainerObject*   pObj;
      LPQUERYRECORDRECT
       pQryRecordRect((LPQUERYRECORDRECT)event.parameter2().asUnsignedLong());
      PRECTL prclItem( (PRECTL)event.parameter1().asUnsignedLong() );
      bool   fSuccess;

      IASSERTSTATE( pQryRecordRect->pRecord != 0 );
      pObj = IObjFromRec( (IMiniCnrRecord*)pQryRecordRect->pRecord );


      // Get the rectangle for tree view.
      if ( pCnrCtl->isTreeView() )
      {
        HTREEITEM  treeHandle = pObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl);
        fSuccess = TreeView_GetItemRect( pCnrCtl->ppd->pTreeview->handle(),
                                         treeHandle, prclItem, true );

        // TreeView_GetItemRect doesn't include the icon.  If the query includes
        // the icon, loop, decrement x, and do hit testing until we are no longer
        // over the icon.
        if ( fSuccess && (pQryRecordRect->fsExtent & CMA_ICON) )
        {
          bool            notDone( true );
          int             index;
          TV_HITTESTINFO  hitTestInfo;

          hitTestInfo.pt.x = prclItem->left;
          hitTestInfo.pt.y = (prclItem->top + prclItem->bottom) / 2;
          for ( index=1; notDone; index++ )
          {
            hitTestInfo.pt.x--;
            TreeView_HitTest(pCnrCtl->ppd->pTreeview->handle(), &hitTestInfo);
            if (hitTestInfo.flags == TVHT_ONITEMICON)
            {
              while( hitTestInfo.flags == TVHT_ONITEMICON )
              {
                hitTestInfo.pt.x--;
                TreeView_HitTest(pCnrCtl->ppd->pTreeview->handle(),
                                 &hitTestInfo);
              }
              notDone=false;
            }
            else if ( index = 500 )
            {
              notDone=false;
              fSuccess=false;
            }
          }

          if ( fSuccess )
          {
            // If the query didn't include text, remove that from the rectangle.
            if (!(pQryRecordRect->fsExtent & CMA_TEXT))
            {
              prclItem->right = prclItem->left - index;
            }
            prclItem->left = hitTestInfo.pt.x +1;
          }
        }
      }
      // Get the rectangle for list view.
      else
      {
        int          index=0;
        int          code=0;

        LV_FINDINFO  lvFindInfo;

        lvFindInfo.flags  = LVFI_PARAM;
        lvFindInfo.lParam = (unsigned long)pObj;

        index = IContainerControl::ListViewFindItem(
                                   pCnrCtl->ppd->pListview->handle(),
                                   (-1), &lvFindInfo);

        if ( (pQryRecordRect->fsExtent & CMA_TEXT)  &&
             (pQryRecordRect->fsExtent & CMA_ICON)  )
          code = LVIR_BOUNDS;
        if ( (pQryRecordRect->fsExtent & CMA_TEXT)  &&
             !(pQryRecordRect->fsExtent & CMA_ICON)  )
          code = LVIR_LABEL;
        if ( !(pQryRecordRect->fsExtent & CMA_TEXT)  &&
             (pQryRecordRect->fsExtent & CMA_ICON)  )
          code = LVIR_ICON;

        fSuccess = ListView_GetItemRect( pCnrCtl->ppd->pListview->handle(),
                                         index, prclItem, code );
      }

      // Adjust for possible title window.
      if ( fSuccess && pCnrCtl->isTitleVisible() )
      {
        IWindow *nCnr;
        if (pCnrCtl->isTreeView() )
          nCnr = pCnrCtl->ppd->pTreeview;
        else
          nCnr = pCnrCtl->ppd->pListview;

        POINTL newOrigin;
        newOrigin.x = 0;
        newOrigin.y = 0;

        IMAPWINDOWPOINTS( nCnr->handle(), pCnrCtl->handle(), &newOrigin, 1 );

        prclItem->top    += newOrigin.y;
        prclItem->bottom += newOrigin.y;
      }

      event.setResult( fSuccess );
      fStopHandling = true;
    }
    break;

    case CM_QUERYVIEWPORTRECT:
    {
      /*******************************************************************/
      /* Query the current viewport rectangle.                           */
      /*                                                                 */
      /* This is only implemented for the viewport in window coordinates.*/
      /* The fRightSplitWindow flag is ignored.                          */
      /*******************************************************************/
      BOOL      retVal;
      PRECTL    pCnrRect( (PRECTL)event.parameter1().asUnsignedLong() );
      unsigned short  usIndicator(event.parameter2().lowNumber());
      LPRECT    pClientRect;


      pCnrRect->left   = 0;
      pCnrRect->right  = 0;
      pCnrRect->top    = 0;
      pCnrRect->bottom = 0;

      ISize  viewSize;
      if ( pCnrCtl->isTreeView() )
        viewSize = pCnrCtl->ppd->pTreeview->size();
      else
        viewSize = pCnrCtl->ppd->pListview->size();

      pCnrRect->right  = viewSize.width();
      pCnrRect->bottom = viewSize.height();

      // Adjust for Possible title window.
      if ( pCnrCtl->isTitleVisible() )
      {
        IWindow *nCnr;
        if (pCnrCtl->isTreeView() )
          nCnr = pCnrCtl->ppd->pTreeview;
        else
          nCnr = pCnrCtl->ppd->pListview;

        POINTL newOrigin;
        newOrigin.x = 0;
        newOrigin.y = 0;

        IMAPWINDOWPOINTS( nCnr->handle(), pCnrCtl->handle(), &newOrigin, 1 );

        pCnrRect->top    += newOrigin.y;
        pCnrRect->bottom += newOrigin.y;
      }

      event.setResult( true );
      fStopHandling = true;
    }
    break;

    case CM_REMOVEDETAILFIELDINFO:
    {
      /*******************************************************************/
      /* Remove the column from the ListView.                            */
      /*                                                                 */
      /* Only what IContainerControl requires has been implemented.      */
      /* We are only considering 2 possibilities:                        */
      /*   1) cNumFieldInfo = 0 -> All columns being removed.            */
      /*   2) cNumFieldInfo = 1 -> Just one being removed.               */
      /*******************************************************************/
      bool               retVal( true );
      IContainerColumn*  pCnrColumn;
      unsigned short     cNumFieldInfo   (event.parameter2().lowNumber());

      IASSERTSTATE( cNumFieldInfo <= 1 );

      // ListView must exist if we are to remove columns.
      if (pCnrCtl->ppd->pListview)
      {
        if ( cNumFieldInfo )   //Assume just one.
        {
          //parameter1 is a pointer to a PFIELDINFO
          PFIELDINFO pFI(*((PFIELDINFO*)event.parameter1().asUnsignedLong()));
          pCnrColumn = (IContainerColumn*)pFI->pUserData;
          int  colId( pCnrCtl->ppd->columnSet->columnIdFromContainerColumn(
                                                                  pCnrColumn) );

          if ( colId >= (-1) )    // (-2) is invalid.
          {
            // Remove it from the list, which also does ListView_RemoveItem.
            pCnrCtl->ppd->columnSet->remove( pCnrColumn );
            ListView_RedrawItems( pCnrCtl->ppd->pListview->handle(), 0,
                                  ListView_GetItemCount(
                                      pCnrCtl->ppd->pListview->handle() )-1 );
            InvalidateRect( pCnrCtl->ppd->pListview->handle(), 0, true );
            pCnrCtl->ppd->cnrInfo.cFields--;
          }
          else
          {
            // If failure, then this might be a column we removed because
            // of the isIconViewText restriction.  If so, then remove it
            // from our cached list because the application now owns this
            // column for deleting.
            pCnrCtl->ppd->columnsToDelete->remove( pCnrColumn );
            retVal = false;
          }
        }
        else    //Remove and Free All Columns
        {
          int colCount( pCnrCtl->ppd->columnSet->columnCount() );

          //take care of icon column here before loop.

          for ( int colId = 0; colId < colCount; colId++ )
          {

            pCnrColumn = pCnrCtl->ppd->columnSet->containerColumnFromColumnId(
                                                                         colId);

            if ( pCnrColumn )
              pCnrCtl->ppd->columnSet->remove( pCnrColumn );
          }
          pCnrCtl->ppd->cnrInfo.cFields = 0;
          // Clear out any cached columns to delete because all are removed
          pCnrCtl->ppd->columnsToDelete->removeAll();
        }
      }

      event.setResult( retVal );
      fStopHandling = true;
    }
    break;

    case CM_REMOVERECORD:
    {
      /*******************************************************************/
      /* Remove the items from the TreeView and/or the ListView.         */
      /*                                                                 */
      /* Only what IContainerControl requires has been implemented.      */
      /* We are only considering 2 possibilities:                        */
      /*   1) cNumRecord = 0 -> All records being removed.               */
      /*   2) cNumRecord = 1 -> Just one being removed.                  */
      /* We do not accept CMA_FREE, or cNumRecord > 1, as this           */
      /* is all the class library requires.                              */
      /*                                                                 */
      /* NOTE: Actually, deleteAllObjects uses CMA_FREE, but this        */
      /*       has been bypassed, because of the problems posed by       */
      /*       trying to implement noSharedObjects optimization.         */
      /*       See deleteAllObjects for more info.                       */
      /*                                                                 */
      /* We are currently ignoring CMA_INVALIDATE.                       */
      /*******************************************************************/

      IContainerObject* pCnrObj;
      unsigned short    cNumRecord   (event.parameter2().lowNumber());
      unsigned short    fRemoveRecord(event.parameter2().highNumber());
      IASSERTSTATE( cNumRecord <= 1 );
      IASSERTSTATE( !(bool)(fRemoveRecord & CMA_FREE) );

      if ( cNumRecord )   //Assume just one.
      {
        //parameter1 is a pointer to a IMiniCnrRecord*
        IMiniCnrRecord*
             pRec(*((IMiniCnrRecord**)event.parameter1().asUnsignedLong()));
        pCnrObj = IObjFromRec(pRec);

        // This container must contain the object before we can continue.  If
        // it doesn't, ignore this call like CCL does.
        if (pCnrCtl->containsObject (pCnrObj))                       /* D8111 */
        {
          //If TreeView exists, convert ICnrObj* to PTREEITEM and remove it.
          if ( pCnrCtl->ppd->pTreeview )
          {
            HTREEITEM pTreeItem;
            pTreeItem = pCnrObj->ppd->pCnrSet->treeItemFromObject( pCnrCtl );

            if ( pTreeItem )
              TreeView_DeleteItem(pCnrCtl->ppd->pTreeview->handle(), pTreeItem);
          }

          //If ListView exists, convert ICnrObj* to an index and remove it.
          if ( pCnrCtl->ppd->pListview )
          {
            int           index=0;
            LV_FINDINFO   lvFindInfo;
            lvFindInfo.flags  = LVFI_PARAM;
            lvFindInfo.lParam = (LPARAM)pCnrObj;

            // Get the object's list view index.
            index = IContainerControl::ListViewFindItem(
                                       pCnrCtl->ppd->pListview->handle(),
                                       (-1), &lvFindInfo );

            if ( index != (-1) )
              ListView_DeleteItem(pCnrCtl->ppd->pListview->handle(), index);
          }

          //Remove the corresponding element from the object's CnrSet.
          pCnrObj->ppd->pCnrSet->remove( pCnrCtl );
        }
      }
      else   //Remove All
      {
        if ( pCnrCtl->ppd->pListview && !pCnrCtl->ppd->pTreeview )
        {
          //Cursor through and remove from each objects CnrSet
          //for this ListView only container.
          //This is done because the LVN_DELETEITEM notification does
          //not return the LV_ITEM.lParam = ICnrObj field but only the
          //index.

          int     numListItems(ListView_GetItemCount(
                                     pCnrCtl->ppd->pListview->handle()));
          LV_ITEM         lvItem;

          lvItem.mask       = LVIF_PARAM;
          lvItem.iSubItem   = 0;

          for (int index=0; index < numListItems; index++)
          {
            lvItem.iItem = index;
            if (ListView_GetItem(pCnrCtl->ppd->pListview->handle(), &lvItem))
            {
              IContainerObject* pCnrObj((IContainerObject*)lvItem.lParam);
              pCnrObj->ppd->pCnrSet->remove( pCnrCtl );
            }
          }
        }

        //If the views exist, remove all from the native control.
        if ( pCnrCtl->ppd->pTreeview )
        {
          TreeView_DeleteAllItems(pCnrCtl->ppd->pTreeview->handle() );
        }

        if ( pCnrCtl->ppd->pListview )
          ListView_DeleteAllItems(pCnrCtl->ppd->pListview->handle() );
      }

      fStopHandling = true;
      break;
    }

    case CM_SCROLLWINDOW:
    {
      /*******************************************************************/
      /* Scroll the TreeView or the ListView.                            */
      /*                                                                 */
      /*******************************************************************/
      unsigned short  scrollDirection( event.parameter1().lowNumber() );
      long            scrollInc      ( (long)event.parameter2() );

      // Must be either CMA_VERTICAL or CMA_HORIZONTAL
      IASSERTSTATE( ( scrollDirection == CMA_VERTICAL)  ||
                    ( scrollDirection == CMA_HORIZONTAL)  );

      if ( pCnrCtl->isTreeView() )
      {
        int      retVal;
        RECT     cnrRect;
        POINTL   pt;

        GetWindowRect( pCnrCtl->ppd->pTreeview->handle(), &cnrRect);

        pt.x = cnrRect.left;
        pt.y = cnrRect.top;
        IMAPWINDOWPOINTS( 0, pCnrCtl->ppd->pTreeview->handle(), &pt, 1);
        cnrRect.left = pt.x;
        cnrRect.top  = pt.y;

        pt.x = cnrRect.right;
        pt.y = cnrRect.bottom;
        IMAPWINDOWPOINTS( 0, pCnrCtl->ppd->pTreeview->handle(), &pt, 1);
        cnrRect.right = pt.x;
        cnrRect.bottom  = pt.y;

        if ( scrollDirection == CMA_VERTICAL )
          retVal = ScrollWindowEx( pCnrCtl->ppd->pTreeview->handle(),
                                   0, (int)scrollInc, 0, &cnrRect, 0, 0,
                                   SW_ERASE | SW_INVALIDATE );
        else
          retVal = ScrollWindowEx( pCnrCtl->ppd->pTreeview->handle(),
                                   (int)scrollInc, 0, 0, &cnrRect, 0, 0,
                                   SW_ERASE | SW_INVALIDATE );

        if ( retVal = ERROR )
          event.setResult( false );
        else
          event.setResult( true );
      }
      else          //ListView
      {
        if ( scrollDirection == CMA_VERTICAL )
          event.setResult( ListView_Scroll( pCnrCtl->ppd->pListview->handle(),
                                            0, (int)scrollInc ) );
        else
          event.setResult( ListView_Scroll( pCnrCtl->ppd->pListview->handle(),
                                            (int)scrollInc, 0 ) );
      }

      fStopHandling = true;
    }
    break;

    case CM_SEARCHSTRING:
    {
      /*******************************************************************/
      /* Search the TreeView or the ListView for the given string.       */
      /*                                                                 */
      /* This is only implemented for listview currently and doesn't     */
      /* support details view or the treeView.                           */
      /* It currently doesn't support case sensitivity.                  */
      /* The tree view could possibly create a tree cursor and then      */
      /* call TreeView_GetItem for each one untill a string matches.     */
      /*******************************************************************/
      IMiniCnrRecord*
          pSearchAfter( (IMiniCnrRecord*)event.parameter2().asUnsignedLong() );
      LPSEARCHSTRING
          pSearchString( (LPSEARCHSTRING)event.parameter1().asUnsignedLong() );
      int   itemIndex=0;

      IASSERTSTATE( pSearchAfter != 0 );

      if ( pSearchAfter == (IMiniCnrRecord*)CMA_FIRST )
      {
        itemIndex = (-1);
      }

      // for ANY OF the tree, list, or detail views...
      {
         // 22902 : SEARCH OF (TREE, DETAIL, or LIST) VIEW FOR SPECIFIED TEXT...
         //         Use an ObjectCursor to spin through the entries, searching
         //         below the top-level ONLY for tree views.
         event.setResult( 0 );
         IContainerControl::ObjectCursor *pTreeCursor = NULL;
         if ( pCnrCtl->isTreeView() ) // search all sub-branches
            pTreeCursor = new IContainerControl::ObjectCursor(*pCnrCtl, 0, true);
         else                         // search only visible (top-level) entries
            pTreeCursor = new IContainerControl::ObjectCursor(*pCnrCtl, 0, false);

         // initialize loop to either begin at 1st, or at specified object
         if ( itemIndex == -1 )
           pTreeCursor->setToFirst();
         else
         {
           IContainerObject* pCnrObj( IObjFromRec(pSearchAfter) );
           IString searchFrom( pSearchAfter->strIcon );
           pTreeCursor->setCurrent( pCnrObj );
           pTreeCursor->setToNext();       // start PAST the searchAfter item
         }

         // loop through tree via ObjectCursor, and locate 1st match
         for( ; pTreeCursor->isValid(); pTreeCursor->setToNext())
         {
           IContainerObject* pObj( pCnrCtl->objectAt(*pTreeCursor) );
           IMiniCnrRecord* pRec( IRecFromObj(pObj) );

           // will be checking treeItem text against pSearchString text
           IString check( pRec->strIcon );
           IString match( pSearchString->pszSearch );

           // Make search smart enough to handle case-sensitive or not,
           //     must-be-prefix or not, and must-be-exact-match or not.
           if ( !pSearchString->fsCaseSensitive )
           {
             match.upperCase();
             check.upperCase();
           }

           Boolean bFound = FALSE;
           if ( pSearchString->usView & CV_EXACTMATCH )
           {
             if ( check == match )       // exact matching entry
                bFound = TRUE;
           }
           else if ( pSearchString->fsPrefix ) // match at 1st position in string
           {
             if ( check.indexOf(match) == 1 )
                bFound = TRUE;
           }
           else   // all other cases
           {
             if ( check.indexOf(match) > 0 )  // found substring
               bFound = TRUE;
           }

           if ( bFound ) // matched, according to search params
           {
             event.setResult( IRecFromObj( pObj ) );
             break;
           }
         }
         delete pTreeCursor;
         // END 22902 CODE...
      }

      fStopHandling = true;
    }
    break;

    case CM_SETCNRINFO:
      /*******************************************************************/
      /* Set the container info.                                         */
      /* Check the needArrange flag here and do an arrange if necessary  */
      /* as the user may have requested an arrange when not in icon      */
      /* view, and then changed the view to icon.                        */
      /*******************************************************************/
      ((IContainerControl*)event.window())->ppd->setCnrInfo( (ICnrInfo*)event.parameter1().asUnsignedLong(),
                                                             event.parameter2().asUnsignedLong() );
      // Do an arrange if we are in icon view, have size, and the
      // arrange flag is set
      if (pCnrCtl->ppd->pListview && pCnrCtl->ppd->needArrange)
      {
         ISize cnrSize = pCnrCtl->ppd->pListview->size();
         bool hasSize = (!(cnrSize.width() == 0)) &&
                           (!(cnrSize.height() == 0));
         if (hasSize)
         {
            pCnrCtl->ppd->needArrange = false;
            ListView_Arrange( pCnrCtl->ppd->pListview->handle(), LVA_ALIGNTOP );
         }
      }
      event.setResult( true );
      fStopHandling = true;
      break;

    case CM_SETRECORDEMPHASIS:
    {
      /********************************************************************/
      /* Sets emphasis states for the current view.                       */
      /* This only sets these styles for the current view.                */
      /* Sets the InUse state for both views if they exist.               */
      /* Support for setAttributes (CRA_RECORDREADONLY, DROPONABLE) needed*/
      /* so CM_INVALIDATERECORD doesn't need to worry about the           */
      /* flRecordAttr field.                                              */
      /********************************************************************/
      IContainerObject*  pObj=0;
      IMiniCnrRecord* pRec((IMiniCnrRecord*)event.parameter1().asUnsignedLong());
      bool            fChangeEmphasis( (bool)event.parameter2().number1() );
      unsigned short  fEmphAttribute (  event.parameter2().number2() );
      ICnrObjSetInfo* pObjInfo;
      bool            fSuccess;
      int             itemIndex(-1);
      HTREEITEM       treeItem(0);
      HWND            hwndList(0);
      HWND            hwndTree(0);

      event.setResult(true);

      // Template is placed, but we can determine details.
      // Maybe we do nothing for whole container source emphasis?
      if ( fEmphAttribute == CRA_SOURCE )
      {
        if ( pRec )
        {
          // Source Emphasis on an object.
          pObj = IObjFromRec( pRec );
          if ( fChangeEmphasis )
          {
            // Aquiring it.
          }
          else
          {
            // Losing it.
          }
        }
        else
        {
          // Source Emphasis on entire container.
          if ( fChangeEmphasis )
          {
            // Aquiring it.
          }
          else
          {
            // Losing it.
          }
        }
        event.setResult(true);
        fStopHandling = true;
        break;
      }

      IASSERTSTATE( pRec != 0 );
      pObj = IObjFromRec( pRec );

      IASSERTSTATE( pObj->ppd != 0 );
      pObjInfo = pObj->ppd->pCnrSet->objectFromCnr( pCnrCtl );

      //Turn readOnly on/off for the object.
      if ( fEmphAttribute & CRA_RECORDREADONLY )
      {
        IASSERTSTATE( pObjInfo != 0 );
        if ( fChangeEmphasis ) // Acquiring it.
          pObjInfo->setObjectState(0, CRA_RECORDREADONLY);
        else                   // Losing it.
          pObjInfo->setObjectState(CRA_RECORDREADONLY, 0);
      }

      //Turn droponable on/off for the object.
      if ( fEmphAttribute & CRA_DROPONABLE )
      {
        IASSERTSTATE( pObjInfo != 0 );
        if ( fChangeEmphasis ) // Acquiring it.
          pObjInfo->setObjectState(0, CRA_DROPONABLE);
        else                   // Losing it.
          pObjInfo->setObjectState(CRA_DROPONABLE, 0);
      }

      // Turn inUse emphasis on/off in both tree & list if they exist.
      if ( fEmphAttribute & CRA_INUSE)
      {
        if (pCnrCtl->ppd->pListview)
        {
          hwndList = pCnrCtl->ppd->pListview->handle();

          LV_FINDINFO lvFindInfo;

          // Get the index of the lv_item
          lvFindInfo.flags  = LVFI_PARAM;
          lvFindInfo.lParam = (LPARAM)pObj;

          // Prime the lv item
          itemIndex = IContainerControl::ListViewFindItem(hwndList, -1,
                                                          &lvFindInfo);

          // Make sure we get a valid index back.  It could be
          // setting inUse on a child object that doesn't exist
          // in the list view.
          if (itemIndex != -1)
          {
            LV_ITEM lvItem;

            // Prime the lv item
            lvItem.iItem    = itemIndex;
            lvItem.mask     = LVIF_IMAGE;
            lvItem.iSubItem = 0;

            IASSERTSTATE( pObjInfo != 0 );
            if ( fChangeEmphasis )  // Acquiring it.
            {
              // Set the item's in use image in the image list.
              lvItem.iImage = pCnrCtl->ppd->addInUseToList( pRec->hptrIcon );
              pObjInfo->setObjectState(0, CRA_INUSE, pObj );
            }
            else                    // Losing it.
            {
              // Set the item's normal image in the image list.
              lvItem.iImage = pCnrCtl->ppd->addIconToList( pRec->hptrIcon );
              pObjInfo->setObjectState(CRA_INUSE, 0, pObj );
            }
            fSuccess = ListView_SetItem( hwndList, &lvItem );
          }
        }

        // CRA_INUSE for tree view.
        if (pCnrCtl->ppd->pTreeview)
        {
          IASSERTSTATE( pObjInfo != 0 );
          hwndTree = pCnrCtl->ppd->pTreeview->handle();
          treeItem = pObjInfo->pTVItem;

          if (treeItem)
          {
            TV_ITEM tvItem;
            tvItem.hItem = treeItem;
            tvItem.mask  = TVIF_IMAGE | TVIF_SELECTEDIMAGE;
            if ( fChangeEmphasis ) // Acquiring it.
            {
              // Set the item's in use image in the image list.
              tvItem.iImage =
              tvItem.iSelectedImage = pCnrCtl->ppd->addInUseToList( pRec->hptrIcon );
              pObjInfo->setObjectState(0, CRA_INUSE, pObj);
            }
            else                   // Losing it.
            {
              // Set the item's normal image in the image list.
              tvItem.iImage =
              tvItem.iSelectedImage = pCnrCtl->ppd->addIconToList( pRec->hptrIcon );
              pObjInfo->setObjectState(CRA_INUSE, 0, pObj);
            }
            TreeView_SetItem( hwndTree, &tvItem );
          }
        }
      }


      //Turn selection/cursor  on/off only in the current view.
      if ( !pCnrCtl->isTreeView() )         //Consider only ListView
      {
        hwndList = pCnrCtl->ppd->pListview->handle();

        LV_FINDINFO lvFindInfo;
        lvFindInfo.flags  = LVFI_PARAM;
        lvFindInfo.lParam = (LPARAM)pObj;

        // Get the object's list view index.
        itemIndex = IContainerControl::ListViewFindItem(hwndList, -1,
                                                        &lvFindInfo);

        if( (fEmphAttribute & CRA_CURSORED) || (fEmphAttribute & CRA_SELECTED) )
        {
          unsigned int itemState(0);
          //Must first query before changing the state.
          itemState = ListView_GetItemState( hwndList, itemIndex, 0x000F );

          if ( (fEmphAttribute & CRA_CURSORED) &&  fChangeEmphasis )
            itemState |= LVIS_FOCUSED;

          if ( fEmphAttribute & CRA_SELECTED )
            if ( fChangeEmphasis )
              itemState |= LVIS_SELECTED;
            else
              itemState &= ~(LVIS_SELECTED);

          ListView_SetItemState( hwndList, itemIndex, itemState, 0x000F );
        }
        else if (!(fEmphAttribute & CRA_INUSE))                     /* D7981 */
          event.setResult( false );
      }

      else                           //Consider only TreeView.
      {
        IASSERTSTATE( pObjInfo != 0 );
        hwndTree = pCnrCtl->ppd->pTreeview->handle();
        treeItem = pObjInfo->pTVItem;

        if( ((fEmphAttribute & CRA_CURSORED) || (fEmphAttribute & CRA_SELECTED))
            && fChangeEmphasis )
        {    //MUST BE TURNING IT ON... cant turn off in tree view.
          TreeView_SelectItem( hwndTree, treeItem );
        }
        else if (!(fEmphAttribute & CRA_INUSE))
        {
          event.setResult( false );
        }
      }

      fStopHandling = true;
    }
    break;

    case CM_SORTRECORD:
    {
      /********************************************************************/
      /* Sorts the objects for both the Listview and the Treeview.        */
      /********************************************************************/

      // Process the TreeView Sort only if the TreeView is already created.
      if (pCnrCtl->ppd->pTreeview)
      {
        TV_SORTCB   treeSort;
        treeSort.lpfnCompare = (PFNTVCOMPARE)((void*)event.parameter1());
        treeSort.lParam      = (LPARAM)event.parameter2();
        treeSort.hParent     = 0;
        event.setResult( TreeView_SortChildrenCB(
                                            pCnrCtl->ppd->pTreeview->handle(),
                                            &treeSort, 0 ) );
        //Create Container object cursor and iterate all objects and
        //call TreeView sort for any object that has children.
        //This is necessary because the native TreeView does not
        //sort the entire tree hierarchy with one sort request.
        IContainerControl::ObjectCursor  treeCursor =
                         IContainerControl::ObjectCursor(*pCnrCtl, 0, true);
        forCursor ( treeCursor )
        {
          IContainerObject* pCnrObj(pCnrCtl->objectAt(treeCursor));
          HTREEITEM pTreeItem;
          pTreeItem = TreeView_GetChild(pCnrCtl->ppd->pTreeview->handle(),
                          pCnrObj->ppd->pCnrSet->treeItemFromObject(pCnrCtl) );
          if ( pTreeItem )
          {
            treeSort.hParent =
                        pCnrObj->ppd->pCnrSet->treeItemFromObject( pCnrCtl );
            TreeView_SortChildrenCB( pCnrCtl->ppd->pTreeview->handle(),
                                     &treeSort, 0 );
          }
        }
      }

      // Process the ListView Sort only if the ListView is already created.
      if (pCnrCtl->ppd->pListview)
      {  //mkb, the following is expected to return false under _WIN32S
        event.setResult( ListView_SortItems( pCnrCtl->ppd->pListview->handle(),
                                             (WPARAM)event.parameter1(),
                                             (LPARAM)event.parameter2() ) );
      }

      fStopHandling = true;
    }
    break;

    case WM_SHOWWINDOW:
      /********************************************************************/
      /* Make sure the active view is shown.                              */
      /********************************************************************/
    {
      if (event.parameter1().asUnsignedLong())
      {
        ShowWindow( pCnrCtl->ppd->pActiveview->handle(), SW_SHOW );
      }
    }
    break;

    case WM_SIZE:
    /********************************************************************/
    /* Check to see if we need to do an arrange in icon view after we   */
    /* have been sized.                                                 */
    /********************************************************************/
    {
       if (pCnrCtl->ppd->pListview && pCnrCtl->ppd->needArrange)
       {
          ISize cnrSize = pCnrCtl->ppd->pListview->size();
          bool hasSize = (!(cnrSize.width() == 0)) &&
                            (!(cnrSize.height() == 0));
          if (hasSize)
          {
             pCnrCtl->ppd->needArrange = false;
             ListView_Arrange( pCnrCtl->ppd->pListview->handle(), LVA_ALIGNTOP );
          }
       }
    }
    break;

    case WM_WINDOWPOSCHANGED:
    /********************************************************************/
    /* If the window has been sized, position the child windows         */
    /* (container and title).                                           */
    /********************************************************************/
    {                              // size/move event found
      PWINDOWPOS pswp = (PWINDOWPOS)(char*)event.parameter2();
      if (!(pswp->flags & SWP_NOSIZE))
      {                           // size event found
        pCnrCtl->ppd->layoutActiveView( ISize(pswp->cx, pswp->cy) );
      }
      break;
    }

    case WM_CONTEXTMENU:
    /********************************************************************/
    /* A context menu was requested.  Get the mouse position and send   */
    /* the CN_CONTEXTMENU notification.                                 */
    /********************************************************************/
    {
      IMiniCnrRecord* pRec(0);
      POINT cursorPos;
      cursorPos.x = event.parameter2().number1();
      cursorPos.y = event.parameter2().number2();

      GetCursorPos( &cursorPos );

      MapWindowPoints( HWND_DESKTOP, pCnrCtl->handle(), &cursorPos, 1 );

      IPoint cursorPoint( cursorPos.x, cursorPos.y );

      cursorPoint = ICoordinateSystem::convertToApplication( cursorPoint,
                                                             pCnrCtl->size());

      IContainerObject *pObject(pCnrCtl->objectUnderPoint( cursorPoint ));
      if ( pObject )
        pRec = IRecFromObj( pObject );

      CNRMENUINIT cnrMenuInit;
      cnrMenuInit.pRecord = (RECORDCORE*)pRec;
      // Wrapper the record in a cnrmenuinit structure so that the WM_CONTROL
      // message is converted by the wrapper.
      event.window()->owner()->sendEvent( WM_CONTROL,
                                          IEventParameter1( event.window()->id(),
                                                            CN_CONTEXTMENU ),
                                          IEventParameter2( &cnrMenuInit ));
      fStopHandling = true;
    }
    break;

    case WM_HELP:
    /********************************************************************/
    /* Help was requested.  If on a menu item, do nothing.  Otherwise,  */
    /* get the cursored object and send the CN_HELP notification.       */
    /********************************************************************/
    {
      HELPINFO helpInfo=*((LPHELPINFO)(void *)(event.parameter2()));
      if (helpInfo.iContextType == HELPINFO_MENUITEM)
      {
        fStopHandling = false;
      }
      else
      {
        // If no object is cursored, send -1 as parm2
        IContainerObject* cursored(pCnrCtl->cursoredObject());
        IMiniCnrRecord* pRec=(IMiniCnrRecord*)(void*)(-1);

        if (cursored)
        {
          pRec = IRecFromObj(cursored);
        }
        event.window()->owner()->sendEvent( WM_CONTROL,
                                          IEventParameter1( event.window()->id(),
                                                            CN_HELP ),
                                          IEventParameter2( (RECORDCORE*)pRec ));
        fStopHandling = true;
      }
    }
    break;

    case WM_SETFONT:
    /********************************************************************/
    /* Pass font changes on to the container parts                      */
    /********************************************************************/
    {
      // Pass font change messages onto the listview and treeview controls.
      if (pCnrCtl->ppd->pListview)
        pCnrCtl->ppd->pListview->sendEvent( event );

      if (pCnrCtl->ppd->pTreeview)
        pCnrCtl->ppd->pTreeview->sendEvent( event );

      // Also pass to title if one was specified.
      if ( pCnrCtl->ppd->pTitle )
        // pCnrCtl->ppd->pTitle->setFont(pCnrCtl->ppd->pActiveview->font());
        pCnrCtl->ppd->pTitle->resetFont() ;
    }
    break;

    case WM_DESTROY:
    /********************************************************************/
    /* Free any associated lists BEFORE the control is destroyed.       */
    /********************************************************************/
    {
      // Free any associated lists for the listview control.
      if (pCnrCtl->ppd->pListview)
      {
        ListView_SetImageList( pCnrCtl->ppd->pListview->handle(),
                               0, LVSIL_SMALL );
        ListView_SetImageList( pCnrCtl->ppd->pListview->handle(),
                               0, LVSIL_NORMAL );
      }
      // Free any associated lists for the treeview control.
      if (pCnrCtl->ppd->pTreeview)
        TreeView_SetImageList( pCnrCtl->ppd->pTreeview->handle(),
                               0, TVSIL_NORMAL);
    }
    break;


/*------------------------------------------------------------------------------
| The following events are notification messages coming FROM the tree view     |
| or list view control.                                                        |
|                                                                              |
------------------------------------------------------------------------------*/

    case WM_NOTIFY:
    {
      fStopHandling = true;
      event.setResult(false);
      switch( ((LPNMHDR)event.parameter2().asUnsignedLong())->code)
      {
        case NM_DBLCLK:
        /********************************************************************/
        /* Generate a CN_ENTER notification.                                */
        /********************************************************************/
        {
          pCnrCtl->ppd->doubleClickOccurred = true;
          NMHDR *pnmhdr((LPNMHDR)event.parameter2().asUnsignedLong());
          IWindow *owner(event.window()->owner());

          // If we're in tree view, the object which was double-clicked is the
          // cursored object.
          if (pCnrCtl->isTreeView())
          {
            // find the focused item and send a CN_ENTER notification.
            NOTIFYRECORDENTER notifyEnter;
            IContainerObject* cursored(pCnrCtl->cursoredObject());

            if (cursored)
            {
              // Fill the notify structure and send the notification.
              notifyEnter.hwndCnr = event.window()->handle();
              notifyEnter.fKey    = false;
              notifyEnter.pRecord = (RECORDCORE*)IRecFromObj(cursored);

              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(), CN_ENTER ),
                                IEventParameter2( &notifyEnter ));
            }
          }
          // If we're in list view, the object which was double-clicked is the
          // one under the mouse.
          else
          {
            POINT          cursorPos;
            LV_HITTESTINFO hitTestInfo;

            GetCursorPos( &cursorPos );

            // Map the mouse position to local coordinates.
            MapWindowPoints( HWND_DESKTOP, pCnrCtl->ppd->pListview->handle(),
                             &cursorPos, 1 );

            // Do hit testing to find the item.
            hitTestInfo.pt = cursorPos;
            ListView_HitTest( pCnrCtl->ppd->pListview->handle(), &hitTestInfo );
            if (hitTestInfo.iItem != -1)
            {
              NOTIFYRECORDENTER notifyEnter;
              LV_ITEM           lvItem;
              lvItem.iItem    = hitTestInfo.iItem;
              lvItem.mask     = LVIF_PARAM;
              lvItem.iSubItem = 0;
              ListView_GetItem( pCnrCtl->ppd->pListview->handle(), &lvItem );

              IMiniCnrRecord* pRec(IRecFromObj((IContainerObject*)lvItem.lParam));

              // Set up the notify structure and send the notification.
              notifyEnter.hwndCnr = event.window()->handle();
              notifyEnter.fKey    = false;
              notifyEnter.pRecord = (RECORDCORE*)pRec;

              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(), CN_ENTER ),
                                IEventParameter2( &notifyEnter ));

            }
          }
          ITRACE_DEVELOP(IString("---> Double Click"));
        }
        break;

        case NM_RETURN:
          ITRACE_DEVELOP(IString("---> Enter Key"));
        break;

        case NM_SETFOCUS:
        /********************************************************************/
        /* Generate a CN_SETFOCUS notification.                             */
        /********************************************************************/
        {
          IWindow *owner(event.window()->owner());
          if (owner)
            owner->sendEvent( WM_CONTROL,
                              IEventParameter1( event.window()->id(),
                                                CN_SETFOCUS ),
                              IEventParameter2( event.window()->handle() ));
        }
        break;

        case NM_KILLFOCUS:
        /********************************************************************/
        /* Generate a CN_KILLFOCUS notification.                            */
        /********************************************************************/
        {
          IWindow *owner(event.window()->owner());
          if (owner)
            owner->sendEvent( WM_CONTROL,
                              IEventParameter1( event.window()->id(),
                                                CN_KILLFOCUS ),
                              IEventParameter2( event.window()->handle() ));
        }
        break;

        case LVN_BEGINDRAG:
          ITRACE_DEVELOP(IString("---> LVN_BEGINDRAG"));
        break;

        case LVN_BEGINRDRAG:
          ITRACE_DEVELOP(IString("---> LVN_BEGINRDRAG"));
        break;

        case LVN_BEGINLABELEDIT:
        case TVN_BEGINLABELEDIT:
        /********************************************************************/
        /* Generate a CN_BEGINEDIT notification.                            */
        /********************************************************************/
        {
          IContainerObject* pCnrObj;
          unsigned long     controlId;

          // Set controlID according to view.
          if (((LPNMHDR)event.parameter2().asUnsignedLong())->code == LVN_BEGINLABELEDIT)
          {
            pCnrObj = (IContainerObject*)ptrLV_DISPINFO->item.lParam;
            controlId = IC_NATIVE_LISTVIEW;
          }
          else
          {
            pCnrObj = (IContainerObject*)ptrTV_DISPINFO->item.lParam;
            controlId = IC_NATIVE_TREEVIEW;
          }

          // If the object is read only, cancel the edit.
          if ( pCnrObj->ppd->pCnrSet->isReadOnly( pCnrCtl ) )
          {
            if ( pCnrCtl->isTreeView() )
              TreeView_EndEditLabelNow( pCnrCtl->ppd->pTreeview->handle(),
                                        false );
            else
            {
              ListView_EditLabel( pCnrCtl->ppd->pListview->handle(),
                                  (-1) );
              if (pCnrCtl->ppd->pDetailsEdit)                      /* D8067 */
                // Send a fake enter message to the mle so that it will close.
                pCnrCtl->ppd->pDetailsEdit->sendEvent( WM_CHAR,
                                                       IEventParameter1( VK_RETURN ));
            }
          }
          // The object is not read only - generate the notification.
          else
          {
            IMiniCnrRecord*   pRec(IRecFromObj(pCnrObj) );

            CNREDITDATA cnrEditData;
            cnrEditData.cb         = sizeof( CNREDITDATA );
            cnrEditData.hwndCnr    = event.window()->handle();
            cnrEditData.pRecord    = (RECORDCORE*)pRec;
            // the field info struct is 0 if we are not editing a details view.
            cnrEditData.pFieldInfo = 0;
            cnrEditData.ppszText   = (LPSTR*)&(pRec->strIcon);
            cnrEditData.cbText     = pRec->strIcon.length() + 1;
            cnrEditData.id         = event.window()->id();

            event.window()->owner()->sendEvent( WM_CONTROL,
                                       IEventParameter1( cnrEditData.id, CN_BEGINEDIT ),
                                       IEventParameter2( &cnrEditData ) );
          }
        }
        break;

        case LVN_COLUMNCLICK:
          ITRACE_DEVELOP(IString("---> LVN_COLUMNCLICK"));
        break;

        case LVN_DELETEALLITEMS:
          ITRACE_DEVELOP(IString("---> LVN_DELETEALLITEMS"));
        break;

        case LVN_DELETEITEM:
          ITRACE_DEVELOP(IString("---> LVN_DELETEITEM"));
        break;

        case LVN_ENDLABELEDIT:
        case TVN_ENDLABELEDIT:
        /********************************************************************/
        /* Generate a CN_REALLOCPSZ or CN_ENDEDIT notification.             */
        /********************************************************************/
        {
          IContainerObject* pCnrObj;
          unsigned long     controlId;
          bool              fListview;
          bool              fEditCancelled;
          IWindow*          owner(event.window()->owner());

          // Set info according to the view.
          if (((LPNMHDR)event.parameter2().asUnsignedLong())->code == LVN_ENDLABELEDIT)
          {
            pCnrObj   = (IContainerObject*)ptrLV_DISPINFO->item.lParam;
            controlId = IC_NATIVE_LISTVIEW;
            fListview = true;
            fEditCancelled = ((ptrLV_DISPINFO->item.iItem == -1) ||
                              (ptrLV_DISPINFO->item.pszText == 0));
          }
          else
          {
            pCnrObj   = (IContainerObject*)ptrTV_DISPINFO->item.lParam;
            controlId = IC_NATIVE_TREEVIEW;
            fListview = false;
            fEditCancelled = (ptrTV_DISPINFO->item.pszText == 0);
          }

          IMiniCnrRecord* pRec(IRecFromObj(pCnrObj));

          CNREDITDATA cnrEditData;
          cnrEditData.cb         = sizeof( CNREDITDATA );
          cnrEditData.hwndCnr    = event.window()->handle();
          cnrEditData.pRecord    = (RECORDCORE*)pRec;
          // the field info struct is 0 if we are not editing a details view.
          cnrEditData.pFieldInfo = 0;
          cnrEditData.ppszText   = (LPSTR*)&(pRec->strIcon);
          cnrEditData.cbText     = pRec->strIcon.length() + 1;
          cnrEditData.id         = event.window()->id();

          // If the user did not cancel editing, send a message to
          // reallocate the string.
          if (!fEditCancelled)
          {
            if (owner->sendEvent( WM_CONTROL,
                                  IEventParameter1( cnrEditData.id,
                                                    CN_REALLOCPSZ ),
                                  IEventParameter2( &cnrEditData ) )
                                 .asUnsignedLong())
            {
              // Refresh the text in the views.
              if (fListview)
              {
                pRec->strIcon = ptrLV_DISPINFO->item.pszText;
                ListView_SetItemText( pCnrCtl->ppd->pListview->handle(),
                                      ptrLV_DISPINFO->item.iItem,
                                      0,
                                      ptrLV_DISPINFO->item.pszText );

                // If a tree view exists, update the tree item text.
                if ( pCnrCtl->ppd->pTreeview )
                {
                  TV_ITEM tvItem;
                  tvItem.hItem = pCnrObj->ppd->pCnrSet->treeItemFromObject( pCnrCtl );
                  tvItem.mask = TVIF_TEXT;
                  tvItem.pszText = pRec->strIcon;
                  TreeView_SetItem( pCnrCtl->ppd->pTreeview->handle(), &tvItem);
                }
              }
              else
              {
                pRec->strIcon = ptrTV_DISPINFO->item.pszText;
                ptrTV_DISPINFO->item.mask = TVIF_TEXT;

                TreeView_SetItem( pCnrCtl->ppd->pTreeview->handle(),
                                  &(ptrTV_DISPINFO->item));

                // If a list view exists, update the list item text.
                if ( pCnrCtl->ppd->pListview )
                {
                  // If the tree item has no parent, then it exists
                  // in the listview at well.
                  if (!TreeView_GetParent(pCnrCtl->ppd->pTreeview->handle(),
                                          ptrTV_DISPINFO->item.hItem))
                  {
                    int index(ICnrObjPrivateData::listItemFromCnr( pCnrCtl,
                                                                   pCnrObj ));
                    if (index != -1)
                    {
                      ListView_SetItemText( pCnrCtl->ppd->pListview->handle(),
                                            index,
                                            0,
                                            ptrTV_DISPINFO->item.pszText );
                    }
                  }
                }
              }
            }
          }
          // Notify with CN_ENDEDIT.
          owner->sendEvent( WM_CONTROL,
                            IEventParameter1( event.window()->id(), CN_ENDEDIT ),
                            IEventParameter2( &cnrEditData ));
        }
        break;

        case LVN_GETDISPINFO:
        /********************************************************************/
        /* ListView sends this message when drawing details view.  We return*/
        /* the information to draw.                                         */
        /********************************************************************/
        {
          // If this is a valid subitem and the column is text, proceed.  Otherwise,
          // do nothing.

          if ((ptrLV_DISPINFO->item.iSubItem > 0) &&
              (ptrLV_DISPINFO->item.mask & LVIF_TEXT))
          {
            // Get the container object and the column.  The info to return is the
            // intersection of the two.
            IContainerObject*
               pCnrObj((IContainerObject*)ptrLV_DISPINFO->item.lParam);
            IContainerColumn*
               pColumn( pCnrCtl->ppd->columnSet->containerColumnFromSubId(
                                               ptrLV_DISPINFO->item.iSubItem) );
            IMiniCnrRecord*
               pRec(IRecFromObj(pCnrObj) );

            // If it's plain text, just return the object's data using the field info
            // offset.
            if ( pColumn->pfieldinfoCl->flData & CFA_STRING )
              ptrLV_DISPINFO->item.pszText = (*((LPTSTR*)(((LPBYTE)pRec) +
                                           pColumn->pfieldinfoCl->offStruct)));
            // Otherwise, it's type number, date, or time.  Get the data and convert
            // it to a string.
            // If the text string hasn't been allocated yet, we have to allocate one.
            else
            {
              if (!ptrLV_DISPINFO->item.pszText)
              {
                ptrLV_DISPINFO->item.pszText = new char[15];
              }
              // The data is type number.  Convert to string.
              if ( pColumn->pfieldinfoCl->flData & CFA_ULONG )
              {
                 CreateNLSNumberString (((unsigned long)*(LPDWORD)(((LPBYTE)pRec) +
                                                 pColumn->pfieldinfoCl->offStruct)),
                                        ptrLV_DISPINFO->item.pszText);
              }
              // The data is type date.  Convert to string.
              else if ( pColumn->pfieldinfoCl->flData & CFA_DATE )
              {
                 CreateNLSDateString (((LPCDATE)(((LPBYTE)pRec) +
                                                 pColumn->pfieldinfoCl->offStruct)),
                                      ptrLV_DISPINFO->item.pszText);
              }
              // The data is type time.  Convert to string.
              else if ( pColumn->pfieldinfoCl->flData & CFA_TIME )
              {
                 CreateNLSTimeString (((LPCTIME)(((LPBYTE)pRec) +
                                                 pColumn->pfieldinfoCl->offStruct)),
                                      ptrLV_DISPINFO->item.pszText);
              }
            }
          }
        }
        break;

        case LVN_INSERTITEM:
          ITRACE_DEVELOP(IString("---> LVN_INSERTITEM"));
        break;

        case LVN_ITEMCHANGING:
          ITRACE_DEVELOP(IString("---> LVN_ITEMCHANGING"));
        break;

        case LVN_ITEMCHANGED:
        /********************************************************************/
        /* Notification from the list view that an item changed.  We need   */
        /* to notify the owner with a CN_EMPHASIS.                          */
        /********************************************************************/
        {
          NM_LISTVIEW* pNMListView((NM_LISTVIEW*)event.parameter2().asUnsignedLong());
          NOTIFYRECORDEMPHASIS notifyRecord;
          IContainerObject* pCnrObj((IContainerObject*)pNMListView->lParam);
          IWindow* owner(event.window()->owner());

          // Only generate the notification if we have an object and an owner.
          if (pCnrObj && owner)
          {
            ICnrObjSetInfo* pCnrObjInfo(pCnrObj->ppd->pCnrSet->objectFromCnr( pCnrCtl ));

            // Set up the notify structure and send the notification.
            notifyRecord.hwndCnr       = event.window()->handle();
            notifyRecord.pRecord       = (RECORDCORE*)IRecFromObj((IContainerObject*)pNMListView->lParam);
            notifyRecord.fEmphasisMask = 0;

            if (pNMListView->uChanged & LVIF_STATE)
            {
              if ((pNMListView->uNewState & LVIS_SELECTED) ||
                  (pNMListView->uOldState & LVIS_SELECTED))
                notifyRecord.fEmphasisMask |= CRA_SELECTED;
              if ((pNMListView->uNewState & LVIS_FOCUSED) ||
                  (pNMListView->uOldState & LVIS_FOCUSED))
                notifyRecord.fEmphasisMask |= CRA_CURSORED;
            }

            owner->sendEvent( WM_CONTROL,
                              IEventParameter1( event.window()->id(), CN_EMPHASIS ),
                              IEventParameter2( &notifyRecord ));

          }
        }
        break;

        case LVN_KEYDOWN:
        case TVN_KEYDOWN:
        /********************************************************************/
        /* Notification from the views that a key was pressed.  If it was   */
        /* the enter key, generate a CN_ENTER.                              */
        /********************************************************************/
        {
          LV_KEYDOWN* pLVKeyDown((LV_KEYDOWN*)event.parameter2().asUnsignedLong());
          IWindow* owner(event.window()->owner());

          if (pLVKeyDown->wVKey == VK_RETURN && owner)
          {
            // find the focused item and send a CN_ENTER notification.
            NOTIFYRECORDENTER notifyEnter;
            IContainerObject* cursored(pCnrCtl->cursoredObject());

            if (cursored)
            {
              notifyEnter.hwndCnr = event.window()->handle();
              notifyEnter.fKey    = true;
              notifyEnter.pRecord = (RECORDCORE*)IRecFromObj(cursored);

              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(), CN_ENTER ),
                                IEventParameter2( &notifyEnter ));
            }
          }
          ITRACE_DEVELOP(IString("---> LVN_KEYDOWN | TVN_KEYDOWN"));
        }
        break;

        case LVN_SETDISPINFO:
          ITRACE_DEVELOP(IString("---> LVN_SETDISPINFO"));
        break;

        case TVN_BEGINDRAG:
          ITRACE_DEVELOP(IString("---> TVN_BEGINDRAG"));
        break;

        case TVN_BEGINRDRAG:
          ITRACE_DEVELOP(IString("---> TVN_BEGINRDRAG"));
        break;

        case TVN_DELETEITEM:
        /********************************************************************/
        /* An item was deleted from tree view.  Remove its record from the  */
        /* object set.                                                      */
        /********************************************************************/
        {
          ITRACE_DEVELOP(IString("---> TVN_DELETEITEM"));
          NM_TREEVIEW* pNMTreeView((NM_TREEVIEW*)event.parameter2().asUnsignedLong());
          IContainerObject* pCnrObj((IContainerObject*)pNMTreeView->itemOld.lParam);

          if (pCnrObj)
          {
            pCnrObj->ppd->pCnrSet->remove( pCnrCtl );
          }
        }
        break;

        case TVN_ITEMEXPANDED:
        /********************************************************************/
        /* An item was expanded/collapsed in tree view.  Generate a         */
        /* CN_EXPANDTREE or CN_COLLAPSETREE.                                */
        /********************************************************************/
        {
          NM_TREEVIEW* pNMTreeView((NM_TREEVIEW*)event.parameter2().asUnsignedLong());
          IContainerObject* pCnrObj((IContainerObject*)pNMTreeView->itemNew.lParam);
          IMiniCnrRecord*   pRec(IRecFromObj(pCnrObj) );
          IWindow* owner(event.window()->owner());

          // Set the expand or collapse notification done flag.
          pCnrCtl->ppd->expandNotificationDone = true;

          if (owner)
          {
            if (pNMTreeView->itemNew.state & TVIS_EXPANDED)
            {
              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(),
                                                  CN_EXPANDTREE ),
                                IEventParameter2( (RECORDCORE*)pRec ));
            }
            else
            {
              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(),
                                                  CN_COLLAPSETREE ),
                                IEventParameter2( (RECORDCORE*)pRec ));
            }
          }
        }
        break;

        case TVN_ITEMEXPANDING:
          ITRACE_DEVELOP(IString("---> TVN_ITEMEXPANDING"));
        break;

        case TVN_SELCHANGING:
          ITRACE_DEVELOP(IString("---> TVN_SELCHANGING"));
        break;

        case TVN_SELCHANGED:
        /********************************************************************/
        /* Tree view selection changed.  Generate a CN_EMPHASIS notification*/
        /********************************************************************/
        {
          NM_TREEVIEW* pNMTreeView((NM_TREEVIEW*)event.parameter2().asUnsignedLong());
          NOTIFYRECORDEMPHASIS notifyRecord;
          IContainerObject* pCnrObj(0);
          IWindow* owner(event.window()->owner());

          if (owner)
          {
            notifyRecord.hwndCnr       = event.window()->handle();
            notifyRecord.fEmphasisMask = CRA_SELECTED | CRA_CURSORED;

            if (pNMTreeView->itemOld.hItem)
            {
              pCnrObj = (IContainerObject*)pNMTreeView->itemOld.lParam;
              notifyRecord.pRecord = (RECORDCORE*)IRecFromObj(pCnrObj);
              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(), CN_EMPHASIS ),
                                IEventParameter2( &notifyRecord ));
            }

            if (pNMTreeView->itemNew.hItem)
            {
              pCnrObj = (IContainerObject*)pNMTreeView->itemNew.lParam;
              notifyRecord.pRecord = (RECORDCORE*)IRecFromObj(pCnrObj);
              owner->sendEvent( WM_CONTROL,
                                IEventParameter1( event.window()->id(), CN_EMPHASIS ),
                                IEventParameter2( &notifyRecord ));
            }
          }
        }
        break;

      default:
        fStopHandling = false;    // pass on all other WM_NOTIFY codes
      }  // switch on WM_NOTIFY code
    } // WM_NOTIFY case
  }
  return fStopHandling;
}
/*------------------------------------------------------------------------------
| ICnrControlData::timerPopped                                                 |
|                                                                              |
| Function to detect double-click over an item.                                |
------------------------------------------------------------------------------*/
void ICnrControlData::timerPopped ( unsigned long timerId )
{
  // Stop the timer.
  if (fTimer)
  {
    fTimer->stop();
    delete fTimer ;
    fTimer = 0    ;
  }

  // If a double click has not occured, then check if the column and
  // object can be edited.
  if (!doubleClickOccurred)
  {
    MapWindowPoints( pListview->handle(), pCnr->handle(), &editPoint, 1 );

    IPoint cursorPoint( editPoint.x, editPoint.y );

    IContainerColumn *pColumn( pCnr->columnUnderPoint( cursorPoint ));

    cursorPoint = ICoordinateSystem::convertToApplication( cursorPoint,
                                                           pCnr->size());

    IContainerObject *pObject(pCnr->objectUnderPoint( cursorPoint ));

    if (pColumn &&                                      // column exists
        pObject &&                                      // container object exists
        (pColumn->pfieldinfoCl->flData & CFA_STRING) && // column is a string
        pColumn->isWriteable() &&                       // column is editable
        pCnr->isCursored( pObject ) &&                  // object is cursored
        pCnr->isWriteable( pObject ))
    {
      pCnr->editObject( pObject, pColumn );
    }
  }

}

/*------------------------------------------------------------------------------
| IMessageHandler::IMessageHandler                                             |
| Low-level message handler for the native container views.                    |
------------------------------------------------------------------------------*/
IMessageHandler::IMessageHandler( )
  : IMessageHandler::Inherited( )
{ }

/*------------------------------------------------------------------------------
| IMessageHandler::~IMessageHandler                                            |
------------------------------------------------------------------------------*/
IMessageHandler::~IMessageHandler( )
{ }

/*------------------------------------------------------------------------------
| IMessageHandler::defaultHandler                                              |
------------------------------------------------------------------------------*/
IMessageHandler* IMessageHandler::defaultHandler ( )
{
   if(!IDefaultNCStatics.pMessageHandler)
      IDefaultNCStatics.pMessageHandler = new IMessageHandler();
   return IDefaultNCStatics.pMessageHandler;
}

/*------------------------------------------------------------------------------
| IMessageHandler::dispatchHandlerEvent                                        |
| Low-level message handler for the native container views.  Provides drag     |
| and drop support.                                                            |
------------------------------------------------------------------------------*/
bool IMessageHandler::dispatchHandlerEvent( IEvent& event )
{
  bool fStopHandling( false );

  switch (event.eventId())
  {
    // Push mouse move messages up to the wrapper window so that
    // drag drop works.
    case WM_MOUSEMOVE:
    {
      IEventData p2( event.parameter2() );
      IContainerControl* pCnrCtl((IContainerControl*)event.window()->parent());
      if ( pCnrCtl->ppd->showTitle )
      {
        // If the title is showing then we need to remap the event to
        // the container wrapper window.
        // The point must be in native coordinates regardless of which
        // orientation has been selected. IMouseEvent member function
        // mousePosition will take care of any re-coordination issues,
        // but asssumes that the point is in native coordinates.
        unsigned long mousePos = event.parameter2().asUnsignedLong();

        POINTL newOrigin;
        newOrigin.x = 0;
        newOrigin.y = 0;
        IMAPWINDOWPOINTS( event.window()->handle(),
                          event.window()->parent()->handle(),
                          &newOrigin, 1 );

        p2 = IEventData( (short)( LOWORD(mousePos) ),
                         (short)( HIWORD(mousePos) + newOrigin.y ) );
      }
      // A new event with the wrapper container window as the dispatching
      // handle must be created for dispatching.
      IEvent passUpEvent( event.window()->owner()->handle(),
                          event.eventId(),
                          event.parameter1(),
                          p2 );


      event.window()->owner()->dispatch( passUpEvent );
      fStopHandling = true;
    }
    break;

    case WM_LBUTTONDOWN:
    {
      IContainerControl* pCnrCtl((IContainerControl*)event.window()->parent());
      pCnrCtl->ppd->doubleClickOccurred = false;

      // If in details view...
      if (pCnrCtl->isDetailsView() && !(pCnrCtl->style() & IContainerControl::readOnly.asUnsignedLong()))
      {
        // If we are not editing a column...
        if (!pCnrCtl->ppd->pDetailsEdit && !ListView_GetEditControl( pCnrCtl->ppd->pListview->handle() ))
        {
          // If the timer has already been started...
          if (!pCnrCtl->ppd->fTimer)
          {
            LV_HITTESTINFO hitTestInfo;

            hitTestInfo.pt.x = event.parameter2().number1();
            hitTestInfo.pt.y = event.parameter2().number2();

            ListView_HitTest( pCnrCtl->ppd->pListview->handle(), &hitTestInfo );

            // If the listview detected the mouse is not over a list view item
            // then we can assume it is over an object not in the first column.
            // We let the list view handle editing for the first column.
            if (hitTestInfo.iItem == -1)
            {
              pCnrCtl->ppd->editPoint = hitTestInfo.pt;
              pCnrCtl->ppd->fTimer = new ITimer( new ITimerMemberFn<ICnrControlData>(
                                                           *pCnrCtl->ppd,
                                                           &ICnrControlData::timerPopped),
                                                 GetDoubleClickTime());
            }
          }
        }
        else if (pCnrCtl->ppd->pDetailsEdit)
        {

          // Send a fake enter message to the mle so that it will
          // commit the edit changes.
          pCnrCtl->ppd->pDetailsEdit->sendEvent( WM_CHAR,
                                                 IEventParameter1( VK_RETURN ));

          fStopHandling = true;
        }
      }
      if (!fStopHandling)
      {
        // Save parameter2, it may need to be changed if the title is
        // showing.
        IEventData p2( event.parameter2() );
        if ( pCnrCtl->ppd->showTitle )
        {
          // If the title is showing then we need to remap the event to
          // the container wrapper window.
          // The point must be in native coordinates regardless of which
          // orientation has been selected. IMouseEvent member function
          // mousePosition will take care of any re-coordination issues,
          // but asssumes that the point is in native coordinates.
          unsigned long mousePos = event.parameter2().asUnsignedLong();

          POINTL newOrigin;
          newOrigin.x = 0;
          newOrigin.y = 0;
          IMAPWINDOWPOINTS( event.window()->handle(),
                            event.window()->parent()->handle(),
                            &newOrigin, 1 );

          p2 = IEventData( (short)( LOWORD(mousePos) ),
                           (short)( HIWORD(mousePos) + newOrigin.y ) );
        }
        // A new event with the wrapper container window as the dispatching
        // handle must be created for dispatching.
        IEvent passUpEvent( event.window()->owner()->handle(),
                            event.eventId(),
                            event.parameter1(),
                            p2 );

        if (!event.window()->owner()->dispatch( passUpEvent ))
          this->defaultProcedure( event );
        fStopHandling = true;
      }
    }
    break;

    case WM_LBUTTONUP:
    case WM_LBUTTONDBLCLK:
    case WM_RBUTTONDOWN:
    case WM_RBUTTONUP:
    case WM_RBUTTONDBLCLK:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONUP:
    case WM_MBUTTONDBLCLK:
    {
      // Save parameter2, it may need to be changed if the title is
      // showing.
      IEventData p2( event.parameter2() );
      IContainerControl* pCnrCtl((IContainerControl*)event.window()->parent());
      if ( pCnrCtl->ppd->showTitle )
      {
        // If the title is showing then we need to remap the event to
        // the container wrapper window.
        // The point must be in native coordinates regardless of which
        // orientation has been selected. IMouseEvent member function
        // mousePosition will take care of any re-coordination issues,
        // but asssumes that the point is in native coordinates.
        unsigned long mousePos = event.parameter2().asUnsignedLong();

        POINTL newOrigin;
        newOrigin.x = 0;
        newOrigin.y = 0;
        IMAPWINDOWPOINTS( event.window()->handle(),
                          event.window()->parent()->handle(),
                          &newOrigin, 1 );

        p2 = IEventData( (short)( LOWORD(mousePos) ),
                         (short)( HIWORD(mousePos) + newOrigin.y ) );
      }
      // A new event with the wrapper container window as the dispatching
      // handle must be created for dispatching.
      IEvent passUpEvent( event.window()->owner()->handle(),
                          event.eventId(),
                          event.parameter1(),
                          p2 );

      if (!event.window()->owner()->dispatch( passUpEvent ))
        this->defaultProcedure( event );
      fStopHandling = true;
    }
    break;

    case IC_UM_IS_AGGREGATE_CTRL:
    {
        // Listview is aggregate (header), tree view can be with direct edit.
        event.setResult( true );
        fStopHandling = true;
    }
    break;

    // Need to detect when scroll bars appear/disappear.
    case WM_NCCALCSIZE:
    {
       IContainerControl* pCnrCtl =
         dynamic_cast<IContainerControl*>( event.window()->parent() ) ;
       if ( pCnrCtl && !pCnrCtl->isRefreshOn() )
       {
           //Potential scrollbar change with refresh off.  Flag for special
           //handling in refresh.
           ITRACE_DEVELOP("Setting IContainerControl::invalidFrame");
           pCnrCtl->flClState |= IContainerControl::invalidFrame;
       }
    }
    break;

  }
  return fStopHandling;
}

/*------------------------------------------------------------------------------
| IDetailsEditHandler::IDetailsEditHandler                                     |
|                                                                              |
------------------------------------------------------------------------------*/
IDetailsEditHandler::IDetailsEditHandler ( )
  : IDetailsEditHandler::Inherited( )
{ }

/*------------------------------------------------------------------------------
| IDetailsEditHandler::~IDetailsEditHandler                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IDetailsEditHandler::~IDetailsEditHandler ( )
{ }

/*------------------------------------------------------------------------------
| IDetailsEditHandler::defaultHandler                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IDetailsEditHandler* IDetailsEditHandler::defaultHandler ( )
{
   if(!IDefaultNCStatics.pEditHandler)
      IDefaultNCStatics.pEditHandler = new IDetailsEditHandler();
   return IDefaultNCStatics.pEditHandler;
}

/*------------------------------------------------------------------------------
| IDetailsEditHandler::virtualKeyPress                                         |
| An IKeyBoardHandler to handle keys which interact with an open edit field.   |
| Attached to the edit field when it is created in CM_OPENEDIT.  Notifications |
| are sent to the control owner.                                               |
------------------------------------------------------------------------------*/
bool IDetailsEditHandler::virtualKeyPress( IKeyboardEvent& event )
{
  // Escape and enter are the only keys we're interested in.
  if (event.virtualKey() == IKeyboardEvent::esc    ||
      event.virtualKey() == IKeyboardEvent::enter  ||
      event.virtualKey() == IKeyboardEvent::newLine  )
  {
    // Get the container, edit object, and column text.
    IContainerControl *pCnr((IContainerControl*)event.window()->parent());

    IMiniCnrRecord* pRec(IRecFromObj(pCnr->ppd->pEditObject) );
    LPTSTR *pColumnText(((LPTSTR*)(((LPBYTE)pRec) +
                       pCnr->ppd->pEditColumn->pfieldinfoCl->offStruct)));

    // Set up the structure to go along with the notifications.
    CNREDITDATA cnrEditData;
    cnrEditData.cb         = sizeof( CNREDITDATA );
    cnrEditData.hwndCnr    = pCnr->handle();
    cnrEditData.pRecord    = (RECORDCORE*)pRec;
    cnrEditData.pFieldInfo = pCnr->ppd->pEditColumn->pfieldinfoCl;
    cnrEditData.ppszText   = pColumnText;
    cnrEditData.cbText     = strlen( *pColumnText );
    cnrEditData.id         = pCnr->id();

    // If the key was enter, send the CN_REALLOCPSZ notification and update the view
    // if the owner returns true.
    if (event.virtualKey() == IKeyboardEvent::enter ||
        event.virtualKey() == IKeyboardEvent::newLine )
    {
      if (pCnr->owner()->sendEvent( WM_CONTROL,
                                    IEventParameter1( cnrEditData.id,
                                                      CN_REALLOCPSZ ),
                                    IEventParameter2( &cnrEditData ) )
                                   .asUnsignedLong())
      {
        IString newText(((IMultiLineEdit*)event.window())->text());

        IString *pObjectText((IString*)pColumnText);
        *pObjectText = newText;
      }
    }
    // Stop this handler.
    IDetailsEditHandler::defaultHandler()->stopHandlingEventsFor( event.window() );

    // Send the CN_ENDEDIT notification.
    pCnr->owner()->sendEvent( WM_CONTROL,
                              IEventParameter1( cnrEditData.id,
                                                CN_ENDEDIT ),
                              IEventParameter2( &cnrEditData ));

    // Destroy the edit window and reset private data.
    IDESTROYWINDOW( event.window()->handle() );

    // Get the item index of the edit object so we can redraw.
    LV_FINDINFO lvFindInfo;
    lvFindInfo.lParam = (long)pCnr->ppd->pEditObject;
    lvFindInfo.flags  = LVFI_PARAM;
    int itemIndex(IContainerControl::ListViewFindItem(
                                     pCnr->ppd->pListview->handle(),
                                    (-1), &lvFindInfo ));            /* D8067 */

    ListView_RedrawItems( pCnr->ppd->pListview->handle(),
                          itemIndex,
                          itemIndex );                               /* D8067 */

    pCnr->ppd->pEditObject  = 0;
    pCnr->ppd->pEditColumn  = 0;
    pCnr->ppd->pDetailsEdit = 0;

    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IColumnElement::IColumnElement                                               |
| IColumnElements are used in the IColumnSet.  Each element represents one     |
| column.                                                                      |
------------------------------------------------------------------------------*/
IColumnElement::IColumnElement ( IContainerColumn* pCol,
                                 int               subItemIdentifier,
                                 int               columnIdentifier,
                                 int               visibilityIdentifier)
  : pColumn(pCol), subItemId(subItemIdentifier),
    columnId( columnIdentifier ), visibilityId(visibilityIdentifier)
{
}

/*------------------------------------------------------------------------------
| IColumnElement::~IColumnElement                                              |
------------------------------------------------------------------------------*/
IColumnElement::~IColumnElement ( )
{
}

/*------------------------------------------------------------------------------
| IColumnElement::operator=                                                    |
------------------------------------------------------------------------------*/
IColumnElement& IColumnElement::operator= ( const IColumnElement& element )
{
  if (this == &element)
    return *this;
  this->pColumn   = element.pColumn;
  this->subItemId = element.subItemId;
  this->columnId  = element.columnId;
  this->visibilityId = element.visibilityId;
  return *this;
}

/*------------------------------------------------------------------------------
| IColumnElement::operator==                                                   |
------------------------------------------------------------------------------*/
bool IColumnElement::operator == ( const IColumnElement& element ) const
{
  return (bool)( this->pColumn   == element.pColumn &&
                 this->subItemId == element.subItemId &&
                 this->columnId  == element.columnId &&
                 this->visibilityId == element.visibilityId);
}

/*------------------------------------------------------------------------------
| IColumnElement::operator<                                                    |
| Less than is based on the subItemId, which indicates position.               |
------------------------------------------------------------------------------*/
bool IColumnElement::operator < ( const IColumnElement& element ) const
{
  return (bool)(this->subItemId < element.subItemId);
}

/*------------------------------------------------------------------------------
| IColumnSet::IColumnSet                                                       |
| A set of all the columns in the container (visible and hidden).  The key is  |
| the IColumnElement subItemId.                                                |
------------------------------------------------------------------------------*/
IColumnSet::IColumnSet( )
 : IKeySortedSetOnSortedTabularSequence< IColumnElement, int > ( 8 ),
   nextInvalidSubId(-10)
{}

/*------------------------------------------------------------------------------
| IColumnSet::~IColumnSet                                                      |
------------------------------------------------------------------------------*/
IColumnSet::~IColumnSet( )
{ }

/*------------------------------------------------------------------------------
| IColumnSet::add                                                              |
| Add a new column to the set.  Note this does not add it to the ListView.     |
------------------------------------------------------------------------------*/
bool IColumnSet::add( IContainerColumn* pCol,
                      int               subItemIdentifier,
                      int               columnIdentifier,
                      bool              visible)
{
  int  visibilityId( 0 );
  // We're inserting a valid and visible column.
  if ( visible )
  {
    // In this case subItemIdentifier = prevCol's subItemIdentifier
    // It could be representing either a visible or hidden column.
    // Hidden columns are represented by a value < -10.  So if its greater than -10,
    // the column we're inserting after is visible.
    if ( subItemIdentifier > (-10) )
    {
      // Required column isIcon has columnIdentifier -1, and required column isIconText
      // has columnIdentifier 0, so we need one greater than those.
      if ( columnIdentifier > 0 )
      {
        IColumnSet::Cursor cursor(*this);
        // We have to increment all the columnIds after this one we're inserting.  Cursor
        // through and do that.  We also have to increment subItemIds for visible columns,
        // as they're identical to the columnId for visible columns.
        forCursor (cursor)
        {
          if ( elementAt(cursor).columnId >= columnIdentifier )
          {
            elementAt(cursor).columnId++;

            // Only increment the subItem Id's on the visible columns
            if (elementAt(cursor).subItemId > 0 )
              elementAt(cursor).subItemId++;
          }

          // If the previous column had hidden columns after it, update their columnIds
          // so that they will occur after the column we are inserting.
          if ( (elementAt(cursor).columnId  == (columnIdentifier-1) ) &&
               elementAt(cursor).visibilityId > 0 )
          {
            elementAt(cursor).columnId++;
          }
        }
      }
    }
    // We're inserting after a hidden column.
    else
    {
      // Make sure this key is in the set.
      if (containsElementWithKey( subItemIdentifier ))
      {
        // The previous element is the one with the given subItemIdentifier.
        IColumnElement prevElement(elementWithKey( subItemIdentifier ));

        // Required column isIcon has columnIdentifier -1, and required column isIconText
        // has columnIdentifier 0, so we need one greater than those.
        if ( columnIdentifier > 0 )
        {
          IColumnSet::Cursor cursor(*this);
          // We have to increment all the columnIds after this one we're inserting.  Cursor
          // through and do that.  We also have to increment subItemIds for visible columns,
          // as they're identical to the columnId for visible columns.
          forCursor (cursor)
          {
            if ( elementAt(cursor).columnId >= columnIdentifier )
            {
              elementAt(cursor).columnId++;

              // Only increment the subItem Id's on the visible columns
              if (elementAt(cursor).subItemId > 0 )
                elementAt(cursor).subItemId++;
            }

            //Update possible hidden columns following the new column.  Update the
            // visibilityId of any hidden items after the new column.
            if ( (elementAt(cursor).columnId  == (columnIdentifier-1) ) &&
                 elementAt(cursor).visibilityId > prevElement.visibilityId )
            {
              elementAt(cursor).columnId++;
              elementAt(cursor).visibilityId -= prevElement.visibilityId;
            }
          }
        }
      }
      else
        return false;
    }

    subItemIdentifier = columnIdentifier;
  }
  // We're inserting an invalid or invisible column.
  else
  {
    // Have to reduce columnId, as hidden columns have the same column identifier as
    // their most previous visible column.
    columnIdentifier--;

    // The subItemId must already be in the list, or there was an error.
    if (containsElementWithKey( subItemIdentifier ))
    {
      // Get the previous item.
      IColumnElement prevElement(elementWithKey( subItemIdentifier ));
      // Since this item is hidden, its visibilityId will be one more than its previous
      // item.
      visibilityId = prevElement.visibilityId + 1;
    }
    else
      return false;

    // Update the visibilityIds of all hidden items following this one.
    IColumnSet::Cursor cursor(*this);
    forCursor (cursor)
    {
      if ( (elementAt(cursor).columnId == columnIdentifier) &&
           (elementAt(cursor).visibilityId >= visibilityId) )
      {
        elementAt(cursor).visibilityId++;
      }
    }

    // Invalid subItemIds start at -11 and grow downward.
    nextInvalidSubId--;
    subItemIdentifier = nextInvalidSubId;
  }

  // Finally, create the new element with the correct Ids and add it to the set.
  IColumnElement newElement( pCol, subItemIdentifier,
                             columnIdentifier, visibilityId );

  Inherited::add( newElement );
  return false;
}

/*------------------------------------------------------------------------------
| IColumnSet::remove                                                           |
| Remove an item from the set of columns AND delete it from the ListView.      |
------------------------------------------------------------------------------*/
bool IColumnSet::remove( IContainerColumn* pCol,
                         bool              collapseColumns )
{
  int colId(0);
  int visibilityId(0);

  IColumnSet::Cursor cursor(*this);

  // Find given column in our columnSet and save information.  Remove it from the set.
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if (columnElement.pColumn == pCol)
    {
      colId        = columnElement.columnId;
      visibilityId = columnElement.visibilityId;
      removeAt( cursor );
      break;
    }
  }

  // Update our columnSet, based on removing above column.
  if ( visibilityId > 0 )     //Removed column was hidden
  {
    //Only need to update the visibility id of succeeding hidden columns with the
    //same columnId.
    forCursor (cursor)
    {
      if ( (elementAt(cursor).columnId == colId) &&
           (elementAt(cursor).visibilityId > visibilityId) )
      {
        elementAt(cursor).visibilityId--;
      }
    }
  }
  else //Removed column was visible
  {
    // Required column isIcon has columnIdentifier -1, and required column isIconText
    // has columnIdentifier 0, so we need one greater than those.
    if ( colId > 0 )          //Only need to update if colId>0
    {
      ListView_DeleteColumn( pCol->container()->ppd->pListview->handle(),
                             colId );

      // Determine the number of hidden columns after previous visible column.
      int prevHiddenCount(0);
      forCursor (cursor)
      {
        if ( elementAt(cursor).columnId == (colId - 1) )
          if ( elementAt(cursor).visibilityId > prevHiddenCount )
            prevHiddenCount = elementAt(cursor).visibilityId;
      }

      // Update any hidden columns that were after the removed column by adding the number
      // of hidden columns after the previous visible column to their visibilityId.
      // Update columnId and subItemId of visible columns by subtracting one from them.
      forCursor (cursor)
      {
        if ( elementAt(cursor).columnId >= colId )
        {
          if ( elementAt(cursor).visibilityId == 0 )
            elementAt(cursor).subItemId--;
          else if ( elementAt(cursor).columnId == colId )
            elementAt(cursor).visibilityId += prevHiddenCount;

          elementAt(cursor).columnId--;
        }
      }
    }
    else
    {
      // Don't collapse for case where we are removing it just to add again
      // immediately (this prevents flashing).
      if (collapseColumns)
      {
        // When both the isIcon and isIconViewText columns have been
        // removed, then we should size the first column to 0.
        if ( !containerColumnFromSubId(0) && !containerColumnFromSubId(-1) )
           if (pCol->container()->isDetailsView())
              ListView_SetColumnWidth( pCol->container()->ppd->pListview->handle(),
                                       0, 0 );
           else
              pCol->container()->ppd->collapseFirstColumn = true ;
      }
    }
  }

  return true;
}

/*------------------------------------------------------------------------------
| IColumnSet::containerColumnFromSubId                                         |
| Return the IContainerColumn of the column with the given subItemId.  Query   |
| the set.                                                                     |
------------------------------------------------------------------------------*/
IContainerColumn* IColumnSet::containerColumnFromSubId( int subItemIdentifier ) const
{
  if (containsElementWithKey( subItemIdentifier ))
  {
    IColumnElement columnElement(elementWithKey( subItemIdentifier ));
    return columnElement.pColumn;
  }
  else
  {
    return 0;
  }
}

/*------------------------------------------------------------------------------
| IColumnSet::newSubItemId                                                     |
| Returns the next valid subItemId for a visible column.  If the column is     |
| invisible, the subItemId is adjusted in IColumnSet::add.                     |
------------------------------------------------------------------------------*/
int IColumnSet::newSubItemId( ) const
{
  int newId(1);
  if (!isEmpty())
  {
    IColumnElement columnElement(lastElement());
    // We need to check to see if the only columns inserted are invisible
    // (subItem<-1), or the isIcon column (subItem=-1), or the isIconViewText
    // column (subItem=0), or any combination of the above.  If this is the
    // case then we need to return newId=1.
    if (columnElement.subItemId > 0)
      newId = columnElement.subItemId + 1;
  }
  return newId;
}

/*------------------------------------------------------------------------------
| IColumnSet::containerColumnFromColumnId                                      |
| Return the IContainerColumn of the column with the given columnId.  Cursor   |
| the set.                                                                     |
------------------------------------------------------------------------------*/
IContainerColumn* IColumnSet::containerColumnFromColumnId( int columnId ) const
{
  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if ( columnElement.columnId == columnId )
    {
      return( columnElement.pColumn );
      break;
    }
  }

  return 0;
}

/*------------------------------------------------------------------------------
| IColumnSet::columnCount                                                      |
| Returns the number of columns in the column set, which is the next valid     |
| columnId.                                                                    |
------------------------------------------------------------------------------*/
int IColumnSet::columnCount( ) const
{
  int count( 0 );

  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    if ( elementAt(cursor).columnId > count )
      count = elementAt(cursor).columnId;
  }
  return (count + 1);  //Account for first column whose Id=0
}

/*------------------------------------------------------------------------------
| IColumnSet::lastSubItemId                                                    |
| Returns the subItemId of the last column in the set, which is used to        |
| generate the next subItemId.                                                 |
------------------------------------------------------------------------------*/
int IColumnSet::lastSubItemId( ) const
{
  int subItemId( 0 );
  int columnId( columnCount() - 1 );
  int visibilityId( -1 );

  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    if ( elementAt(cursor).columnId == columnId )
    {
      if ( elementAt(cursor).visibilityId > visibilityId)
      {
        visibilityId = elementAt(cursor).visibilityId;
        subItemId = elementAt(cursor).subItemId;
      }
    }
  }
  return (subItemId);
}

/*------------------------------------------------------------------------------
| IColumnSet::lastFieldInfo                                                    |
| Returns the pFieldInfo for the last column in the set.  Used for             |
| IContainerControl::ColumnCursor.                                             |
------------------------------------------------------------------------------*/
LPFIELDINFO IColumnSet::lastFieldInfo( ) const
{
  PFIELDINFO  pFI( 0 );
  int columnId( columnCount() - 1 );
  int visibilityId( -1 );

  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    if ( elementAt(cursor).columnId == columnId )
    {
      if ( elementAt(cursor).visibilityId > visibilityId)
      {
        visibilityId = elementAt(cursor).visibilityId;
        pFI = elementAt(cursor).pColumn->pfieldinfoCl;
      }
    }
  }
  return ( (LPFIELDINFO)pFI );
}

/*------------------------------------------------------------------------------
| IColumnSet::firstFieldInfo                                                   |
| Returns the pFieldInfo for the first column in the set.  Used for            |
| IContainerControl::ColumnCursor.                                             |
------------------------------------------------------------------------------*/
LPFIELDINFO IColumnSet::firstFieldInfo( ) const
{
  PFIELDINFO  pFI( 0 );
  int columnId(2);
  int visibilityId(1);

  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    if ( elementAt(cursor).columnId < columnId )
    {
      columnId     = elementAt(cursor).columnId;
      visibilityId = elementAt(cursor).visibilityId;
      pFI = elementAt(cursor).pColumn->pfieldinfoCl;
    }
    else if ( elementAt(cursor).columnId == columnId )
    {
      if ( elementAt(cursor).visibilityId < visibilityId )
      {
        visibilityId = elementAt(cursor).visibilityId;
        pFI = elementAt(cursor).pColumn->pfieldinfoCl;
      }
    }
  }

  return ( (LPFIELDINFO)pFI );
}

/*------------------------------------------------------------------------------
| IColumnSet::nextFieldInfo                                                    |
| Returns the pFieldInfo for the column after the column with the given        |
| pFieldInfo structure.  Used for IContainerControl::ColumnCursor.             |
------------------------------------------------------------------------------*/
LPFIELDINFO IColumnSet::nextFieldInfo(LPFIELDINFO pFI ) const
{
  PFIELDINFO  nextPFI( 0 );
  int  columnId;
  int  visibilityId;

  // First find the element with the given pFieldInfo structure.
  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    if ( elementAt(cursor).pColumn->pfieldinfoCl == pFI )
    {
      columnId = elementAt(cursor).columnId;
      visibilityId = elementAt(cursor).visibilityId;
      break;
    }
  }

  // If column 0 (isIconText) is not inserted but we do have a column -1 (isIcon),
  // then consider it a special case.
  if ( (columnId == (-1)) && !containerColumnFromSubId(0) )
  {
    forCursor (cursor)
    {
      // If there are hidden items after this column (visibilityId > 0), then return the
      // first one as the next item.
      if ( (elementAt(cursor).columnId == 0) &&
           (elementAt(cursor).visibilityId == 1) )
        nextPFI = elementAt(cursor).pColumn->pfieldinfoCl;
    }
    // If no hidden items found after this one, then the next visible item will have
    // subItemId 1.
    if ( !nextPFI )
    {
      if ( containerColumnFromSubId(1) )
        nextPFI = containerColumnFromSubId(1)->pfieldinfoCl;
    }
  }
  // Treat all other cases the same:  columnId >= -1 and both isIcon (-1) and isIconText
  // (0) columnId's present.
  else
  {
    forCursor (cursor)
    {
      // Get the next visible or hidden item.
      if ( (elementAt(cursor).columnId == (columnId + 1)) &&
           (elementAt(cursor).visibilityId == 0) )
      {
        nextPFI = elementAt(cursor).pColumn->pfieldinfoCl;
      }
      if ( (elementAt(cursor).columnId == columnId) &&
           (elementAt(cursor).visibilityId == (visibilityId + 1)) )
      {
        nextPFI = elementAt(cursor).pColumn->pfieldinfoCl;
        break;
      }
    }
  }

  return ( (LPFIELDINFO)nextPFI );
}

/*------------------------------------------------------------------------------
| IColumnSet::prevFieldInfo                                                    |
| Returns the pFieldInfo for the column prior to the column with the given     |
| pFieldInfo structure.  Used for IContainerControl::ColumnCursor.             |
------------------------------------------------------------------------------*/
LPFIELDINFO IColumnSet::prevFieldInfo(LPFIELDINFO pFI ) const
{
  PFIELDINFO  prevPFI( 0 );
  int  columnId;
  int  visibilityId;

  IColumnSet::Cursor cursor(*this);
  // First find the element with the given pFieldInfo structure.
  forCursor (cursor)
  {
    if ( elementAt(cursor).pColumn->pfieldinfoCl == pFI )
    {
      columnId = elementAt(cursor).columnId;
      visibilityId = elementAt(cursor).visibilityId;
      break;
    }
  }

  // If the column is hidden, previous column will have visibilityId one less than
  // the given column and the same columnId.
  if (visibilityId > 0)    //Column is hidden
  {
    forCursor (cursor)
    {
      if ( (elementAt(cursor).columnId == columnId) &&
           (elementAt(cursor).visibilityId == (visibilityId - 1)) )
      {
        prevPFI = elementAt(cursor).pColumn->pfieldinfoCl;
        break;
      }
    }
  }
  else                    //Column is visible
  {
    int tempColId( -2 );
    int tempVisId;

    forCursor (cursor)
    {
      if (elementAt(cursor).columnId < columnId)
      {
        // The previous column is either visible, in which case the columnId's are
        // increasing.
        if (elementAt(cursor).columnId > tempColId)
        {
          tempColId = elementAt(cursor).columnId;
          tempVisId = elementAt(cursor).visibilityId;
          prevPFI = elementAt(cursor).pColumn->pfieldinfoCl;
        }
        // Or the previous column is hidden, in which case the columnId's are remaining
        // the same, but the visibilityId's are increasing.
        else if ( (elementAt(cursor).columnId == tempColId) &&
                  (elementAt(cursor).visibilityId > tempVisId) )
        {
          tempVisId = elementAt(cursor).visibilityId;
          prevPFI = elementAt(cursor).pColumn->pfieldinfoCl;
        }
      }
    }
  }

  return ( (LPFIELDINFO)prevPFI );
}

/*------------------------------------------------------------------------------
| IColumnSet::subItemIdFromFieldInfo                                           |
| Returns the subItemId for the given pFieldInfo structure.                    |
------------------------------------------------------------------------------*/
int IColumnSet::subItemIdFromFieldInfo ( PFIELDINFO pFI ) const
{
  bool found(false);
  int  subItemId(0);
  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if ( columnElement.pColumn->pfieldinfoCl == pFI )
    {
      subItemId = columnElement.subItemId;
      found = true;
      break;
    }
  }
  IASSERTSTATE( found );
  return subItemId;
}

/*------------------------------------------------------------------------------
| IColumnSet::columnIdFromFieldInfo                                            |
| Returns the columnId for the given pFieldInfo structure.                     |
------------------------------------------------------------------------------*/
int IColumnSet::columnIdFromFieldInfo ( PFIELDINFO pFI ) const
{
  bool found(false);
  int  colId(-2);
  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if ( columnElement.pColumn->pfieldinfoCl == pFI )
    {
      colId = columnElement.columnId;
      found = true;
      break;
    }
  }
  IASSERTSTATE( found );
  return colId;
}

/*------------------------------------------------------------------------------
| IColumnSet::columnIdFromContainerColumn                                      |
| Returns the columnId for the given IContainerColumn.                         |
------------------------------------------------------------------------------*/
int IColumnSet::columnIdFromContainerColumn ( IContainerColumn* pCol ) const
{
  int colId(-2);
  IColumnSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if (columnElement.pColumn == pCol)
    {
      colId = columnElement.columnId;
      break;
    }
  }
  return colId;
}


/*------------------------------------------------------------------------------
| IColDeleteSet::IColDeleteSet                                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IColDeleteSet::IColDeleteSet( )
 : IKeySet< IColumnElement, int > ( 8 )
{}

/*------------------------------------------------------------------------------
| IColDeleteSet::~IColDeleteSet                                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IColDeleteSet::~IColDeleteSet( )
{ }

/*------------------------------------------------------------------------------
| IColDeleteSet::add                                                              |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
Boolean IColDeleteSet::add( IContainerColumn* pCol )
{
  IColumnElement newElement( pCol, (int)(void *)pCol, 0, false );
  Inherited::add( newElement );
  return false;
}

/*------------------------------------------------------------------------------
| IColDeleteSet::remove                                                           |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
Boolean IColDeleteSet::remove( IContainerColumn* pCol )
{
  IColDeleteSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if (columnElement.pColumn == pCol)
    {
      removeAt( cursor );
      return true;
    }
  }
  return false;
}


/*------------------------------------------------------------------------------
| IColDeleteSet::deleteAll                                                        |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IColDeleteSet::deleteAll()
{
  IColDeleteSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    IColumnElement columnElement(elementAt( cursor ));
    if (columnElement.pColumn)
      delete columnElement.pColumn;
  }
}

/*------------------------------------------------------------------------------
| IPointerElement::IPointerElement                                             |
| Elements for the ICnrPointerSet class.  Each represents one item in the      |
| image list(s) for the ListView and TreeView.  We keep this list to avoid     |
| having duplicates in the image list.                                         |
------------------------------------------------------------------------------*/
IPointerElement::IPointerElement( const IPointerHandle& pointer,
                                  unsigned long index )
: pointerHandle(pointer), imageListIndex(index), inUseIndex(-1), lph(pointer.asUnsigned())
{}

/*------------------------------------------------------------------------------
| IPointerElement::~IPointerElement                                            |
------------------------------------------------------------------------------*/
IPointerElement::~IPointerElement( )
{
}

/*------------------------------------------------------------------------------
| ICnrPointerSet::ICnrPointerSet                                               |
| A set of pointer handles that reflect the items in the image list(s) for the |
| ListView and TreeView.  We keep this list to avoid having duplicates in the  |
| image list.  The key is the pointer handle as unsigned long.  Used only in   |
| the addIconToList and addInUseToList functions.                              |
------------------------------------------------------------------------------*/
ICnrPointerSet::ICnrPointerSet()
  // Initially allocate 20 IPointerElements
: IKeySet < IPointerElement*, unsigned long > ( 20 )
{}

/*------------------------------------------------------------------------------
| ICnrPointerSet::~ICnrPointerSet                                              |
------------------------------------------------------------------------------*/
ICnrPointerSet::~ICnrPointerSet()
{
  ICnrPointerSet::Cursor cursor(*this);
  forCursor (cursor)
  {
    delete this->elementAt( cursor );
  }
  this->removeAll();
}

/*------------------------------------------------------------------------------
| ICnrPointerSet::addPointerElement                                            |                                                                              |
------------------------------------------------------------------------------*/
ICnrPointerSet& ICnrPointerSet::addPointerElement( const IPointerHandle& pointer,
                                                   int                   imageIndex )
{
  IPointerElement* newPointerElement(new IPointerElement(pointer, imageIndex));
  this->add( newPointerElement );
  return *this;
}

/*------------------------------------------------------------------------------
| The following set of functions are used to convert IContainerControl data    |
| to the format expected by the ListView details view.  All ListView data must |
| be as strings.  Therefore we must convert, for example, CDATEs to strings.   |
| These functions are essentially identical to those used by the CCL control.  |
------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
| reverse                                                                      |
| Reverse a string.  Used by CreateNLSNumberString.                            |
------------------------------------------------------------------------------*/
VOID FAR reverse(LPSTR psz)
{
  int     i, j;
  char    c;

  for (i = 0, j = lstrlen(psz) - 1; i<j; i++, j-- )
  {
    c = psz[i];
    psz[i] = psz[j];
    psz[j] = c;
  }
}

/*------------------------------------------------------------------------------
| BuildDateTimeString                                                          |
| Build a date/time string from the given strings.  Used by CreateNLSDateString|
| and CreateNLSTimeString.                                                     |
------------------------------------------------------------------------------*/
LPSTR FAR BuildDateTimeString( int Field, LPSTR pszDateTime,
                               LPSTR DateTimeSep, bool LeadingZero )
{
  if ((Field < 10) && (LeadingZero))
    *pszDateTime++ = '0';

  itoa (Field, pszDateTime, 10);

  /*********************************************************************/
  /* If there is a separator, add it to the end of the string.         */
  /*********************************************************************/
  if (DateTimeSep)
    lstrcat (pszDateTime, DateTimeSep);

  /*********************************************************************/
  /* Return a pointer to the end of the string, ie. to the terminating */
  /* NULL.                                                             */
  /*********************************************************************/
  while (*pszDateTime)
    pszDateTime++;

  return (pszDateTime);
}

/*------------------------------------------------------------------------------
| CreateNLSDateString                                                          |
| Create a string from a CDATE structure.                                      |
------------------------------------------------------------------------------*/
VOID FAR CreateNLSDateString( LPCDATE pDate, LPSTR pszDate )
{
 int          aDate[3];
 int          Field1;
 int          Field2;
 int          Field3;
 int          DAY   = 0;
 int          MONTH = 1;
 int          YEAR  = 2;
 static DWORD DateFormat (GetProfileInt((LPSTR)"intl", (LPSTR)"iDate", 0));
 static bool  LeadingZero(GetProfileInt((LPSTR)"intl", (LPSTR)"iLzero", 0));
 static bool  fNeedStateInfo(true);
 static char  DateSep[2];
 bool         useLeadingZero(LeadingZero);

 if (fNeedStateInfo)
 {
   fNeedStateInfo = false;
   /******************************************************************/
   /* Get the Date Separator.                                        */
   /******************************************************************/
   GetProfileString ( (LPSTR)"intl", (LPSTR)"sDate", (LPSTR)"/",
                      DateSep, 2);
 }

 /*********************************************************************/
 /* Fill up my array with the values in the CDATE structure.          */
 /*********************************************************************/
 aDate[DAY]   = pDate->day;
 aDate[MONTH] = pDate->month;
 aDate[YEAR]  = pDate->year;

 /*********************************************************************/
 /* Determine the Date Format that is to be displayed.  There are 3   */
 /* possibilities. Month, Day, Year is the default.  The values of 2, */
 /* 1, and 0 for pccinfon->DateFormat are the values returned by the  */
 /* system during the query.  See CnrInitNLSInfo.                     */
 /*********************************************************************/
 switch ((int)DateFormat)
 {
    case 2:
       Field1 = YEAR;
       Field2 = MONTH;
       Field3 = DAY;
    break;

    case 1:
       Field1 = DAY;
       Field2 = MONTH;
       Field3 = YEAR;
    break;

    case 0:
    default:
       Field1 = MONTH;
       Field2 = DAY;
       Field3 = YEAR;
    break;
 }

  if (Field1 == YEAR)
  {
    useLeadingZero = true;
  }

  pszDate = BuildDateTimeString (aDate[Field1],
                                 pszDate,
                                 DateSep,
                                 useLeadingZero);

  useLeadingZero = LeadingZero;

  if (Field2 == YEAR)
  {
    useLeadingZero = true;
  }

  pszDate = BuildDateTimeString (aDate[Field2],
                                 pszDate,
                                 DateSep,
                                 useLeadingZero);

  useLeadingZero = LeadingZero;

  if (Field3 == YEAR)
  {
    useLeadingZero = true;
  }

  /*********************************************************************/
  /* Don't pass in the Date Separator for the last field.              */
  /*********************************************************************/
  pszDate = BuildDateTimeString (aDate[Field3],pszDate,
                                 0, useLeadingZero);

  return;
}

/*------------------------------------------------------------------------------
| CreateNLSTimeString                                                          |
| Create a string from a CTIME structure.                                      |
------------------------------------------------------------------------------*/
VOID FAR CreateNLSTimeString( LPCTIME pTime, LPSTR pszTime )
{
  int         Hours;
  bool        PM(false);
  static char am[9];
  static char pm[9];
  static char TimeSep[2];
  static bool MilitaryFormat(GetProfileInt ((LPSTR)"intl", (LPSTR)"iTime", 0));
  static bool fNeedStateInfo(true);
  static bool LeadingZero(GetProfileInt((LPSTR)"intl", (LPSTR)"iLzero", 0));

  if (fNeedStateInfo)
  {
    fNeedStateInfo = false;
    /******************************************************************/
    /* Get the AM string.                                             */
    /******************************************************************/
    GetProfileString ( (LPSTR)"intl", (LPSTR)"s1159", (LPSTR)"am", am, 9);

    /******************************************************************/
    /* Get the PM string.                                             */
    /******************************************************************/
    GetProfileString ( (LPSTR)"intl", (LPSTR)"s2359", (LPSTR)"pm", pm, 9);

    /******************************************************************/
    /* Get the Time Separator.                                        */
    /******************************************************************/
    GetProfileString ( (LPSTR)"intl", (LPSTR)"sTime", (LPSTR)":",TimeSep, 2);
  }

  Hours = (int)pTime->hours;

  /*********************************************************************/
  /* If we are in the 12 hour format, we need to convert the hours to  */
  /* this format.                                                      */
  /*********************************************************************/
  if (!MilitaryFormat)
  {
    if (Hours >= 12)
    {
      Hours -= 12;
      PM = true;
    }

    if (Hours == 0)
    {
      Hours = 12;
    }
  }

  pszTime = BuildDateTimeString (Hours, pszTime, TimeSep, LeadingZero);

  pszTime = BuildDateTimeString ((int)pTime->minutes, pszTime, TimeSep, true);

  /*********************************************************************/
  /* Don't pass in the Time Separator for the last field.              */
  /*********************************************************************/
  pszTime = BuildDateTimeString ((int)pTime->seconds, pszTime, 0, true);

  /*********************************************************************/
  /* If we are in the 12 hour format, add the am/pm indicator.         */
  /*********************************************************************/
  if (!MilitaryFormat)
  {
    lstrcat (pszTime, " ");
    lstrcat (pszTime, (PM) ? pm : am);
  }

  return;
}

/*------------------------------------------------------------------------------
| CreateNLSNumberString                                                        |
| Create a string from a number.  Negative numbers not supported.              |
------------------------------------------------------------------------------*/
bool CreateNLSNumberString( DWORD dwNumber, LPSTR lpNumber )
{
 LPSTR      pszLead;           /* Temp Pointer used to create string. */
 LPSTR      pszBack;           /* Temp Pointer used to create string. */
 LPSTR      pszTemp;           /* Temp Pointer used to  free the      */
 WORD       i;                 /* Loop counter.                       */
 static char Thousands[2];
 static bool fNeedStateInfo(true);

 if (fNeedStateInfo)
 {
   fNeedStateInfo = false;
   /******************************************************************/
   /* Get the Number Separator.                                      */
   /******************************************************************/
   GetProfileString ( (LPSTR)"intl", (LPSTR)"sThousand", (LPSTR)",",
                      Thousands, 2);
 }

 /* If the number is such that we don't have to worry about adding
  * the thousands separater, convert it and leave.
  */
 if (dwNumber < 1000)
 {
#ifdef IC_WU
    sprintf (lpNumber, "%d", dwNumber);
#else
    ultoa (dwNumber, lpNumber, 10);
#endif
    return (TRUE);
 }

 /*********************************************************************/
 /* Allocate a temporary pointer to be used to create the number      */
 /* string.                                                           */
 /*********************************************************************/
 if (!(pszLead = new char[30]))
 {
   return (FALSE);
 }

 /*********************************************************************/
 /* Save this pointer so we can free the memory at the bottom of this */
 /* function.                                                         */
 /*********************************************************************/
 pszTemp = pszLead;

 /*********************************************************************/
 /* Convert the number to a string.                                   */
 /*********************************************************************/
#ifdef IC_WU
 sprintf (pszLead, "%d", dwNumber );
#else
 ultoa (dwNumber, pszLead, 10);
#endif

 /*********************************************************************/
 /* Reverse the string.                                               */
 /*********************************************************************/
 reverse (pszLead);

 /* Loop through the string and add in the thousands separator.
  */
 while (*pszLead)
 {
   i = 0;
   pszBack = pszLead;

   while ((*pszLead) && (i < 3))
   {
     pszLead = AnsiNext(pszLead);
     i++;
   }

   if (*pszLead)
   {
     strncat (lpNumber, pszBack, 3);
     lstrcat (lpNumber, Thousands);
   }
   else
   {
     lstrcat (lpNumber, pszBack);
   }
 }

 /*********************************************************************/
 /* Reverse the string again before we return.                        */
 /*********************************************************************/
 reverse (lpNumber);

 /*********************************************************************/
 /* Free the temporary storage that was allocated previously.         */
 /*********************************************************************/
 delete [] pszTemp;

 return (TRUE);
}

/*------------------------------------------------------------------------------
| IContainerControl::ListViewFindItem                                          |
|                                                                              |
| Static function that for Win32, calls the ListView_FindItem function, but    |
| for Win32s, implements the function here.  The function is broken on Win32s  |
| and we needed to provide our own version.                                    |
------------------------------------------------------------------------------*/
int IContainerControl::ListViewFindItem(const IWindowHandle& handle,
                                        int start, void* lvFindInfo)
{
  int index;

  index = ListView_FindItem(handle, start, (LV_FINDINFO*)lvFindInfo);

  return index;
}

#else

/*------------------------------------------------------------------------------
| ICnrControlData::ICnrControlData                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
ICnrControlData::ICnrControlData( )
  : pmCompatible( true )
{}

/*------------------------------------------------------------------------------
| ICnrControlData::~ICnrControlData                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
ICnrControlData::~ICnrControlData( )
{}

#endif // IC_WIN
