// Revision: 90 1.9.1.7 source/ui/cnr/icnrctl9.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrctl9.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains static objects and misc functions for                   *
*   the class IContainerControl, declared in icnrctl.hpp.                      *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

#define INCL_WINSTDCNR
#define INCL_WINSTDDRAG

#include <icnrrec.hpp>  // Must be first for OS flags
#include <icconst.h>
#include <icnrctl.hpp>
#include <icnrstat.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <iguilock.hpp>
#include <ihandler.hpp>
#include <ikeyevt.hpp>
#include <iprimlck.hpp>
#include <ireslock.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#ifdef IC_PMWIN
#include <icnrctlw.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL9_CPP_
  #include <ipagetun.h>
#endif

// Forward declares
class IPrivateResource;
class ICnrDestroyHandler;
class IWindow;

//***************************************************************
// Public Container Styles.
//***************************************************************

const IContainerControl::Style
  IContainerControl::readOnly          = CCS_READONLY,
  IContainerControl::autoPosition      = CCS_AUTOPOSITION,
  IContainerControl::extendedSelection = CCS_EXTENDSEL,
  IContainerControl::singleSelection   = CCS_SINGLESEL,
  IContainerControl::multipleSelection = CCS_MULTIPLESEL,
  IContainerControl::verifyPointers    = CCS_VERIFYPOINTERS,
  IContainerControl::noSharedObjects   (0, ICCS_NOSHAREDOBJECTS),
#ifdef IC_PMWIN
  IContainerControl::pmCompatible      (0, ICCS_PMCOMPATIBILITY),
#endif
#ifdef IC_MOTIF
  IContainerControl::pmCompatible      (0),
#endif
#ifdef IC_MOTIFWIN
  IContainerControl::classDefaultStyle ( WS_VISIBLE | CCS_SINGLESEL );
#endif
#ifdef IC_PM
  IContainerControl::classDefaultStyle ( WS_VISIBLE | CCS_SINGLESEL,
                                         ICCS_PMCOMPATIBILITY );
#endif


#ifdef IC_MOTIFWIN
  IContainerControl::Style
    IContainerControl::currentDefaultStyle ( WS_VISIBLE | CCS_SINGLESEL );
#endif
#ifdef IC_PM
 IContainerControl::Style
    IContainerControl::currentDefaultStyle ( WS_VISIBLE | CCS_SINGLESEL,
                                             ICCS_PMCOMPATIBILITY );
#endif


//***************************************************************
// Public Container Attributes.
//***************************************************************
const IContainerControl::Attribute
  IContainerControl::textView              = CV_TEXT,
  IContainerControl::iconView              = CV_ICON,
  IContainerControl::nameView              = CV_NAME,
  IContainerControl::detailsView           = CV_DETAIL,
  IContainerControl::treeView              = CV_TREE,
  IContainerControl::flowedView            = CV_FLOW,
  IContainerControl::miniIcons             = CV_MINI,
  IContainerControl::readOnlyTitle         = CA_TITLEREADONLY,
  IContainerControl::titleSeparator        = CA_TITLESEPARATOR,
  IContainerControl::detailsViewTitles     = CA_DETAILSVIEWTITLES,
  IContainerControl::visibleTitle          = CA_CONTAINERTITLE,
  IContainerControl::alignTitleCentered    = CA_TITLECENTER,
  IContainerControl::alignTitleLeft        = CA_TITLELEFT,
  IContainerControl::alignTitleRight       = CA_TITLERIGHT,
  IContainerControl::handleDrawItem        = CA_OWNERDRAW,
  IContainerControl::handleDrawBackground  = CA_OWNERPAINTBACKGROUND,
  IContainerControl::orderedTargetEmphasis = CA_ORDEREDTARGETEMPH,
  IContainerControl::mixedTargetEmphasis   = CA_MIXEDTARGETEMPH,
  IContainerControl::visibleTreeLine       = CA_TREELINE,
  IContainerControl::classDefaultAttribute = orderedTargetEmphasis |
                                             detailsViewTitles     |
                                             visibleTreeLine       |
                                             readOnlyTitle         |
                                             iconView;

  IContainerControl::Attribute
            IContainerControl::currentDefaultAttribute = classDefaultAttribute;

  // Static Pointer allocation
  static ICnrControlStaticPtr CnrCtlStatics;

IPrivateResource* ICnrControlStaticPtr::pCnrKey = 0;
ICnrDestroyHandler* ICnrControlStaticPtr::pHandler = 0;

ICnrDestroyHandler ::  ICnrDestroyHandler() { }
ICnrDestroyHandler :: ~ICnrDestroyHandler() { }


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::setAllocationContainer

 Implementation: Sets the allocation container handle for this container.
                 This is the "hidden" container in which all our other
                 containers (for this thread) reside.
------------------------------------------------------------------------------*/
void IContainerControl::setAllocationContainer(unsigned long hwndContainer)
{
   IMODTRACE_DEVELOP("CnrCtl::setAllocationContainer");
   IThread::current().setVariable("ICnrCtl::hwnd", IString(&hwndContainer,
                                                           sizeof(hwndContainer)));
   ITRACE_DEVELOP(IString("ICnrCTl::hwnd is: ") +
                  IThread::current().variable("ICnrCtl::hwnd"));
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::hwndAllocation

 Implementation: Returns the allocation container handle for this container.
------------------------------------------------------------------------------*/
IWindowHandle IContainerControl::hwndAllocation()
{
   IString strAllocate = IThread::current().variable("ICnrCtl::hwnd");
   IWindowHandle whnd;

   if (strAllocate.length() != 0)
#ifdef IC_WIN
     whnd = (IWindowHandle::Value)( *(unsigned long*)(char*)strAllocate );
#endif
#ifdef IC_MOTIFPM
     whnd = *(IWindowHandle*)(char*)strAllocate;
#endif

#ifdef IC_PMWIN
   // If not allocated or an invalid handle, call initialize and try again
   if ((strAllocate.length()==0) ||
       (!IISWINDOW(IThread::current().anchorBlock(), whnd)))
#endif
#ifdef IC_MOTIF
   // If not allocated call initialize and try again
   if (strAllocate.length()==0)
#endif
   {
      IContainerControl::initialize();
      strAllocate = IThread::current().variable("ICnrCtl::hwnd");
      if (strAllocate.length() != 0)
#ifdef IC_WIN
        whnd = (IWindowHandle::Value)( *(unsigned long*)(char*)strAllocate );
#endif
#ifdef IC_MOTIFPM
        whnd = *(IWindowHandle*)(char*)strAllocate;
#endif
   }
   IASSERT(strAllocate.length()!=0);
   return whnd;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::containerKey

 Implementation: Returns the key (semaphore) used to serialize
                 access to this container.
------------------------------------------------------------------------------*/
IPrivateResource& IContainerControl ::containerKey()
{
   if (!CnrCtlStatics.pCnrKey)
   {
      IPrimalLock lock;
      if (!CnrCtlStatics.pCnrKey)
         CnrCtlStatics.pCnrKey = new IPrivateResource;
   }
   return *CnrCtlStatics.pCnrKey;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::destroyHandler

 Implementation: Returns a reference to the handler that processes
                 WM_DESTROY requests for this container. This routine
                 creates the handler if doesn't yet exist.
------------------------------------------------------------------------------*/
ICnrDestroyHandler& IContainerControl::destroyHandler()
{
   if(CnrCtlStatics.pHandler == 0)
   {
      IGUIResourceLock lock( containerKey() );
      if (CnrCtlStatics.pHandler == 0)
         CnrCtlStatics.pHandler = new ICnrDestroyHandler();
   }
   return *CnrCtlStatics.pHandler;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl::baseRecordSize

 Implementation: Private function to get and return the container's CNRINFO
                 structure as ICnrInfo.
------------------------------------------------------------------------------*/
unsigned long IContainerControl::baseRecordSize()
{
   return sizeof(IMiniCnrRecord);
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl::containerInfo

 Implementation: Private function to query the container's CNRINFO
                 structure as ICnrInfo.
------------------------------------------------------------------------------*/
void IContainerControl::containerInfo(ICnrInfo* pCnrInfo) const
{
  unsigned fSuccess = this->sendEvent(CM_QUERYCNRINFO,
                                      MPFROMP(pCnrInfo),
                                      MPFROMLONG(sizeof(ICnrInfo)));
  if (!fSuccess)
  {
    ITHROWGUIERROR("CM_QUERYCNRINFO");
  }
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl::setContainerInfo

 Implementation: Private function to set the container's CNRINFO structure.
------------------------------------------------------------------------------*/
void IContainerControl::setContainerInfo(ICnrInfo* pCnrInfo,
                                         unsigned long flags)
{
  unsigned fSuccess = this->sendEvent(CM_SETCNRINFO,
                                      MPFROMP(pCnrInfo),
                                      MPFROMLONG(flags));
  if (!fSuccess)
  {
    ITHROWGUIERROR("CM_SETCNRINFO");
  }
}

/*------------------------------------------------------------------------------
 Function Name: IContainerControl::defaultStyle

 Implementation: Returns the default style for new container objects.
------------------------------------------------------------------------------*/
IContainerControl::Style IContainerControl::defaultStyle()
{
  return currentDefaultStyle;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::setDefaultStyle

 Implementation: Replaces the default style for new container objects.
------------------------------------------------------------------------------*/
void  IContainerControl::setDefaultStyle
                                (const IContainerControl::Style& cnrStyle)
{
  currentDefaultStyle = cnrStyle;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::convertToGUIStyle

 Implementation: Returns the base style for the window (by default), or
                 the extended style if the extended flag (bExtOnly) is set.
------------------------------------------------------------------------------*/
unsigned long IContainerControl::convertToGUIStyle(const IBitFlag& guiStyle,
                                                   bool bExtOnly) const
{
  // Obtain the style from the class (IControl) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
    // CCS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= guiStyle.asUnsignedLong() & ICCS_MASK;
#ifdef IC_WIN
    // Defect: 28824:
    // Container will not paint over siblings due to OS2 limiation.
    // NA for windows so remove limitation.
    // ulStyle |= CNR_REQUIREDSTYLES;
    ulStyle |= WS_CHILD | WS_CLIPCHILDREN;
#endif
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
 Function Name: IContainerControl::isPMCompatible

 Implementation: Returns true if the window has an OS/2 look and feel
                 (on Windows, this is determined by whether the container
                 is implemented with the CCL control).
------------------------------------------------------------------------------*/
bool IContainerControl::isPMCompatible( ) const
{
#ifdef IC_WIN
  return (bool)ppd->pmCompatible;
#endif
#ifdef IC_PM
  return true;
#endif
#ifdef IC_MOTIF
  return false;
#endif
}

/*------------------------------------------------------------------------------
 Function Name: IContainerControl::defaultAttribute

 Implementation: Returns the default attribute for new container objects.
------------------------------------------------------------------------------*/
IContainerControl::Attribute IContainerControl::defaultAttribute()
{
  return currentDefaultAttribute;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::setDefaultAttribute

 Implementation: Replaces the default attribute for new container objects.
------------------------------------------------------------------------------*/
void  IContainerControl::setDefaultAttribute
                           (const IContainerControl::Attribute& cnrAttribute)
{
  currentDefaultAttribute = cnrAttribute;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::dispatchHandlerEvent

 Implementation: Handles WM_DESTROY messages.
------------------------------------------------------------------------------*/
bool ICnrDestroyHandler::dispatchHandlerEvent(IEvent& evt)
{
   if (evt.eventId() == WM_DESTROY)
   {
//      IContainerControl* pcnrctl = (IContainerControl*)IWindow::windowWithHandle(evt.handle());
      IContainerControl* pcnrctl = (IContainerControl*)(evt.controlWindow());
      if(pcnrctl!=0)
      {
        pcnrctl->cleanUp();
      }
   }
#ifdef IC_PMWIN
   else if ( evt.eventId() == WM_HSCROLL || evt.eventId() == WM_VSCROLL )
   {
      // Make sure any open Direct Edit Windows are closed before scrolling.
      IContainerControl* pcnrctl = dynamic_cast<IContainerControl*>( evt.controlWindow() );
      if(pcnrctl && pcnrctl->currentEditMLE())
         pcnrctl->sendEvent(CM_CLOSEEDIT, 0, 0);
   }
#endif

#ifdef IC_WIN
   else if (evt.eventId() == IC_UM_IS_AGGREGATE_CTRL )
   {
      evt.setResult(true);
      return true;
   }
#endif
   return false;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::handleEventsFor

 Implementation: Enables this handler to handle events for the specified
                 container.
------------------------------------------------------------------------------*/
ICnrDestroyHandler& ICnrDestroyHandler::handleEventsFor( IContainerControl *container )
{
  IASSERTPARM(container != 0);
  IHandler::handleEventsFor(container);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::stopHandlingEventsFor

 Implementation: Disables this handler from handling events for the
                 specified container.
------------------------------------------------------------------------------*/
ICnrDestroyHandler& ICnrDestroyHandler::stopHandlingEventsFor( IContainerControl *container )
{
  IASSERTPARM(container != 0);
  IHandler::stopHandlingEventsFor(container);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::handleEventsFor

 Implementation: Ignores requests to attach this handler to a window other
                 than an IContainerControl.
------------------------------------------------------------------------------*/
IHandler& ICnrDestroyHandler::handleEventsFor ( IWindow* window  )
{
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: ICnrDestroyHandler::stopHandlingEventsFor

 Implementation: Ignores requests to remove this handler from a window
                 other than an IContainerControl.
------------------------------------------------------------------------------*/
IHandler& ICnrDestroyHandler::stopHandlingEventsFor ( IWindow* window  )
{
  return *this;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::addDefaultHandler

 Implementation: Attaches a default handler to the container.
------------------------------------------------------------------------------*/
void IContainerControl::addDefaultHandler ( )
{
  destroyHandler().handleEventsFor(this);
  return;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::removeDefaultHandler

 Implementation: Removes the default handler from the container.
------------------------------------------------------------------------------*/
void IContainerControl::removeDefaultHandler ( )
{
  destroyHandler().stopHandlingEventsFor(this);
  return;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::setDeleteObjectsOnClose

 Implementation: Flags whether the container is responsible for deleting
                 its container objects during its clean-up processing.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::setDeleteObjectsOnClose(bool fDestroy)
{
  if(fDestroy)
    flClState |= IContainerControl::autoDeleteObjects;
  else
    flClState &= ~IContainerControl::autoDeleteObjects;
  return *this;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl::setDeleteColumnsOnClose

 Implementation: Flags whether the container is responsible for deleting
                 its column objects during its clean-up processing.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::setDeleteColumnsOnClose(bool fDestroy)
{
  if(fDestroy)
    flClState |= IContainerControl::autoDeleteColumns;
  else
    flClState &= ~IContainerControl::autoDeleteColumns;
  return *this;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
 Function Name: IContainerControl::passEventToOwner

 Implementation: Prevents the container from propagating an Enter key
                 press to its owner window.
------------------------------------------------------------------------------*/
bool IContainerControl::passEventToOwner ( IEvent& event )
{
  switch ( event.eventId() )
  {
     case xEvent( KeyPress ):
     {  // Don't propagate Enter key presses to owner window.
        IKeyboardEvent
          keyEvent( event );
        if ( keyEvent.isUpTransition() == false  &&
             keyEvent.isVirtual()  &&
             ( keyEvent.virtualKey() == IKeyboardEvent::enter  ||
               keyEvent.virtualKey() == IKeyboardEvent::newLine ) )
        {
           event
            .setPassToOwner( false );
        }
        break;
     }
     default:
        Inherited::passEventToOwner( event );
        break;
  }
  return event.passToOwner();
}
#endif
