// Revision: 88 1.8.1.7 source/ui/cnr/icnrctl7.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrctl7.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the object attribute setting and  *
*   quering functions declared in icnrctl.hpp.                                 *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINSTDCNR

#include <icnrrec.hpp>  // Must be first for OS flags

#include <icconst.h>
#include <icnrctl.hpp>
#include <iexcept.hpp>
#include <itrace.hpp>


// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL7_CPP_
  #include <ipagetun.h>
#endif

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setSelected

 Implementation: Select an object.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::setSelected ( IContainerObject* object,
                                                    bool select )
{
  if ( this->isSingleSelection()  &&  ! select )
  {
     // IASSERTSTATE is better here, but we need to preserve
     // past behavior by using the same exception type as results
     // from ITHROWGUIERROR (which used to result from the call
     // to setEmphasis).
     ITHROWLIBRARYERROR1( IC_ASSERTPARM, IBaseErrorInfo::accessError,
                          IException::recoverable,
                          "this->isSingleSelection() && select==false" );
  }
  else
  {
     this->setEmphasis( object, CRA_SELECTED, select );
  }

  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeSelected

 Implementation: De-Select an object.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeSelected(IContainerObject* pcnrobj)
{
   // object asserted on setEmphasis
   setEmphasis(pcnrobj, CRA_SELECTED, false);
   return *this;
}



/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: showSourceEmphasis

 Implementation: Draw source emphasis on an object.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: showSourceEmphasis(IContainerObject* pcnrobj,
                                                    bool fSource)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  // object asserted on setEmphasis
  setEmphasis(pcnrobj, CRA_SOURCE, fSource);
  /*************************************************************************/
  /*  When removing the source emphasis from the record, invalidate the    */
  /*  record in addition to using the setrecordemphasis message.           */
  /*  This should solve the left-over painting problems that have been     */
  /*  encountered using source emphasis.                                   */
  /*  Changed to calling refresh as it calls containsObject() to be sure   */
  /*  the pm cnr's external recordcore is updated before calling invalidate*/
  /*************************************************************************/
  if ((!fSource) && pcnrobj->isRefreshOn() )
  {
    pcnrobj->refresh(this);
  }
  return *this;
#endif  //IC_PMWIN
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: showSourceEmphasis

 Implementation: Draw source emphasis on the whole container.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: showSourceEmphasis( bool fSource)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  /*************************************************************************/
  /*  When removing the source emphasis from the container, invalidate the */
  /*  container instead of using the setrecordemphasis message.            */
  /*  This should solve the left-over painting problems that have been     */
  /*  encountered using source emphasis.                                   */
  /*  NOTE: The container actually xors the painted bits when source emph  */
  /*        is being painted on the whole container(not a record).  Thus,  */
  /*        showing the source emphasis twice will actually cause it to be */
  /*        turned off.                                                    */
  /*************************************************************************/
  if (fSource)
    sendEvent( CM_SETRECORDEMPHASIS,
               0,
               MPFROM2SHORT(fSource, CRA_SOURCE));
  else
  {
    /*************************************************************************/
    /* The reason why we use cm_invalidaterecord is because the bits are xord*/
    /* by the pm container when you call CM_SETRECORDEMPHASIS the second time*/
    /* for source emphasis.  This can potentially cause some of the source   */
    /* emphasis painting to be left around.  Invalidaterecord will insure    */
    /* that the container is refreshed and there will be no source emphasis  */
    /* painting left around.                                                 */
    /* Changed to call refresh() so that the pm cnr's external recordcores   */
    /* are updated properly before calling cm_invalidaterecord               */
    /*************************************************************************/
    if (isRefreshOn())
      refresh();
  }
  return *this;
#endif  //IC_PMWIN
#ifdef IC_MOTIF
  return *this;
#endif
}


IContainerControl& IContainerControl :: hideSourceEmphasis(IContainerObject* object)
{
  ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  return showSourceEmphasis(object, false);
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


IContainerControl& IContainerControl :: hideSourceEmphasis()
{
  ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  return showSourceEmphasis(false);
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setCursor

 Implementation: Put the cursor on an object.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setCursor(IContainerObject* pcnrobj)
{
   // object asserted on setEmphasis
   setEmphasis(pcnrobj, CRA_CURSORED, true);
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: enableDataUpdate

 Implementation: Make an object editable.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: enableDataUpdate(IContainerObject* pcnrobj,
                                                         bool fEnable)
{
   IMODTRACE_DEVELOP("CnrCtl::enableDataUpdate");
   // object asserted on setAttributes
   if(fEnable)
     setAttributes(pcnrobj, CRA_RECORDREADONLY, 0);
   else
     setAttributes(pcnrobj, 0, CRA_RECORDREADONLY);
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: disableDataUpdate

 Implementation: Make an object non-editable.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: disableDataUpdate(IContainerObject* pcnrobj)
{
   // object asserted on setAttributes
   setAttributes(pcnrobj, 0, CRA_RECORDREADONLY);
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setInUse

 Implementation: Mark an object with In-Use emphasis.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setInUse(IContainerObject* pcnrobj,
                                                 bool fInUse )
{
   if (pcnrobj)
      pcnrobj->setInUse(fInUse, this) ;
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeInUse

 Implementation: Remove In-Use emphasis.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeInUse(IContainerObject* pcnrobj)
{
   if (pcnrobj)
      pcnrobj->removeInUse(this) ;
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: enableDrop

 Implementation: Allow this object to be dropped on.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: enableDrop(IContainerObject* pcnrobj,
                                                   bool fEnable)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   // object asserted on setAttributes
   if(fEnable)
     setAttributes(pcnrobj, 0, CRA_DROPONABLE);
   else
     setAttributes(pcnrobj, CRA_DROPONABLE, 0);
   return *this;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: disableDrop

 Implementation: Don't allow this object to be dropped on.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: disableDrop(IContainerObject* pcnrobj)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   // object asserted on setAttributes
   setAttributes(pcnrobj, CRA_DROPONABLE, 0);
   return *this;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isDropOnAble

 Implementation: Determine if this object allows a drop.
------------------------------------------------------------------------------*/
bool IContainerControl :: isDropOnAble(const IContainerObject* pcnrobj) const
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_DROPONABLE) ? true : false);
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return false;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isTarget

 Implementation: Determine if this object has target emphasis.
------------------------------------------------------------------------------*/
bool IContainerControl :: isTarget(const IContainerObject* pcnrobj) const
{
#ifdef IC_PMWIN
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_TARGET) ? true : false);
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return false;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isSource

 Implementation: Determine if this object has source emphasis.
------------------------------------------------------------------------------*/
bool IContainerControl :: isSource(const IContainerObject* pcnrobj) const
{
// ITRACE_MOTIF_NOTYET();
#ifdef IC_PMWIN
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_SOURCE) ? true : false);
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return false;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isSelected

 Implementation: Determine if this object is selected.
------------------------------------------------------------------------------*/
bool IContainerControl :: isSelected(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_SELECTED) ? true : false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isInUse

 Implementation: Determine if this object has InUse emphasis.
------------------------------------------------------------------------------*/
bool IContainerControl :: isInUse(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_INUSE) ? true : false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isCursored

 Implementation: Determine if this object has the cursor.
------------------------------------------------------------------------------*/
bool IContainerControl :: isCursored(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_CURSORED) ? true : false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isVisible

 Implementation: Determine if this object is filtered.
------------------------------------------------------------------------------*/
bool IContainerControl :: isVisible(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_FILTERED) ? false : true);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isExpanded

 Implementation: Determine if this object is expanded.
------------------------------------------------------------------------------*/
bool IContainerControl :: isExpanded(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_EXPANDED) ? true : false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isCollapsed

 Implementation: Determine if this object is collapsed.
------------------------------------------------------------------------------*/
bool IContainerControl :: isCollapsed(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_COLLAPSED) ? true : false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isWriteable

 Implementation: Determine if this object can be edited.
------------------------------------------------------------------------------*/
bool IContainerControl :: isWriteable(const IContainerObject* pcnrobj) const
{
   // object asserted on attributes
   unsigned long flAttr = attributes(pcnrobj);
   return ((flAttr & CRA_RECORDREADONLY) ? false : true);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: cursoredObject

 Implementation: Return the object with the cursor.
------------------------------------------------------------------------------*/
IContainerObject* IContainerControl :: cursoredObject() const
{
  IContainerControl::ObjectCursor cnrcursor(*this, IContainerObject::cursored);
  return cnrcursor.first();
}
