// Revision: 83 1.9.3.1 source/ui/cnr/icnrctl2.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrctl2.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the IContainerControl class       *
*   declared in icnrctl.hpp.                                                   *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WIN
#define INCL_WINSTDCNR
#define INCL_WINWINDOWMGR
#define INCL_GPI

#define IUSING_OLE
#ifdef WIN32_LEAN_AND_MEAN
  #undef WIN32_LEAN_AND_MEAN
#endif
#ifdef _OLE2_H_
  #undef _OLE2_H_
#endif

#include <icnrrec.hpp>  // Must be first for OS flags

#include <itrace.hpp>
#include <iexcept.hpp>
#include <irect.hpp>
#include <icnrctl.hpp>
#include <icnrcol.hpp>
#include <ifont.hpp>

#include <icnrobjp.hpp>

#ifdef IC_MOTIF
#include <icnrctlm.hpp>
#endif

#ifdef IC_PMWIN
#include <istattxt.hpp>
#include <icnrctlw.hpp>
#endif
#ifdef IC_WIN
#include <commctrl.h>
#endif
#ifdef IC_PM
#include <ithread.hpp>
#endif

#include <imcevt.hpp>
#include <icoordsy.hpp> 

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL2_CPP_
  #include <ipagetun.h>
#endif

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: viewPortOnWorkspace

 Implementation: Query the rectangle bounding the client area
            relative to the origin of the containers workspace.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: viewPortOnWorkspace() const
{
   IFUNCTRACE_DEVELOP() ;
#ifdef IC_PMWIN  
   IRectangle rect = viewport(CMA_WORKSPACE, false);
#endif
#ifdef IC_MOTIF
   IRectangle rect = ppd->viewPortOnWorkspace() ;
   IString cnrtmp = rect.asString() ;
   ITRACE_DEVELOP("viewport rect " + rect.asString()) ;
#endif
   return rect;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: detailsViewPortOnWorkspace

 Implementation: Query the rectangle bounding the client area
  relative to the origin of the containers workspace
  in the details view.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: detailsViewPortOnWorkspace(bool fRight) const
{
#ifdef IC_PMWIN
  IRectangle rect = viewport(CMA_WORKSPACE, fRight);
#endif  
#ifdef IC_MOTIF
  IRectangle rect = ppd->viewPortOnWorkspace() ;
#endif

  return rect;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: viewPortOnWindow

 Implementation: Query the rectangle bounding the client area
            in container window coordinates.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: viewPortOnWindow() const
{
   IRectangle rect = viewport(CMA_WINDOW, false);
   return rect;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: detailsViewPortOnWindow

 Implementation: Query the rectangle bounding the client area
   in container window coordinates of Details View.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: detailsViewPortOnWindow(bool fRight) const
{
  IRectangle rect = viewport(CMA_WINDOW, fRight);

  return rect;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: viewport

 Implementation: Query the rectangle bounding the client area
   relative to the passed coordinate space.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: viewport(unsigned long ulCoordinateSpace,
                                         bool fRightSplitWindow) const
{
  IFUNCTRACE_DEVELOP() ;
  RECTL rectl;
  unsigned fSuccess = sendEvent(CM_QUERYVIEWPORTRECT,
                                MPFROMP(&rectl),
                                MPFROM2SHORT(ulCoordinateSpace,
                                             fRightSplitWindow));
  if(!fSuccess)
  {
     ITHROWGUIERROR("CM_QUERYVIEWPORT");
  }

#ifdef IC_WIN
  IRectangle rect(rectl.left, rectl.bottom, rectl.right, rectl.top);
#endif
#ifdef IC_PM
  IRectangle rect(rectl.xLeft, rectl.yBottom, rectl.xRight, rectl.yTop);
#endif
#ifdef IC_MOTIF 
   IRectangle rect(rectl.lXLeft, rectl.lYBottom, rectl.lXRight, rectl.lYTop) ;
#endif // IC_MOTIF
  // convert to application coordinates if needed
  if (ICoordinateSystem::applicationOrientation() == IC_CNR_COORDSYS )
  {
    rect.moveTo( IPoint( rect.minX(),
                         size().height() - rect.maxY() ) );
  }
  ITRACE_DEVELOP("viewport rect = (" + rect.asString() + ")") ;
  return rect;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: scrollVertically

 Implementation: Scroll the container up and down.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: scrollVertically(long lPixels)
{
  scroll(lPixels, 0, false);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: scrollHorizontally

 Implementation: Scroll the container left and right.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: scrollHorizontally(long lPixels,
                                                           bool fRight)
{
   scroll(0, lPixels, fRight);
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: scroll

 Implementation: Scroll the container up/down and left/right.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: scroll(long lVerticalPixels,
                                 long lHorizontalPixels, bool fRight)
{
   unsigned fSuccess;
#ifdef IC_PMWIN
   //We need to make sure any posted CM_CLOSEEDIT is processed before we
   //scroll the window (to not lose changes).  Remove any from the queue
   //and call CM_CLOSEEDIT now.  We post this message in closeEdit().
   QMSG  qmsg;
   bool  closeFound = false;
   while ( IPEEKMSG( IThread::current().anchorBlock(), &qmsg, handle(),
                     CM_CLOSEEDIT, CM_CLOSEEDIT, PM_REMOVE ) )
   {
     closeFound = true;
   }
   if ( closeFound )
   {
     sendEvent(CM_CLOSEEDIT, 0, 0);
   }
#endif
   if(lVerticalPixels)
   {
     fSuccess = sendEvent(CM_SCROLLWINDOW,
                          MPFROMLONG(CMA_VERTICAL),
                          MPFROMLONG(lVerticalPixels));
     if(!fSuccess)
     {
        ITHROWGUIERROR("CM_SCROLLWINDOW");
     }
   }

   if(lHorizontalPixels)
   {
     if ( isDetailsView() && detailsViewSplit() )
     {
       scrollDetailsHorizontally(lHorizontalPixels, fRight);
     }
     else
     {
       fSuccess = sendEvent( CM_SCROLLWINDOW,
                             MPFROMLONG(CMA_HORIZONTAL),
                             MPFROMLONG(lHorizontalPixels));
       if(!fSuccess)
       {
          ITHROWGUIERROR("CM_SCROLLWINDOW");
       }
     }
   }
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: scrollDetailsHorizontally

 Implementation: Scroll the details view left and right.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: scrollDetailsHorizontally(long lHorizontalPixels,
                                                    bool fRight)
{
   unsigned long flSide;
   if (fRight)
     flSide = CMA_RIGHT;
   else
     flSide = CMA_LEFT;

   if (detailsViewSplit())
   {
     unsigned fSuccess = sendEvent(CM_HORZSCROLLSPLITWINDOW,
                                   MPFROMLONG(flSide),
                                   MPFROMLONG(lHorizontalPixels));
       if(!fSuccess)
       {
          ITHROWGUIERROR("CM_SCROLLSPLITWINDOW");
       }
   }
   else
     scroll( 0, lHorizontalPixels, false);

   return *this;
}



/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: scrollToObject

 Implementation: Scroll the Window to include an object.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: scrollToObject(const IContainerObject* pcnrobj)
{
#ifdef IC_PMWIN
  if ( ppd->pmCompatible )
  {

    refresh(IWindow::immediate) ;

    IRectangle rectViewport = viewPortOnWindow();
    IRectangle rectObject   = iconRectangle(pcnrobj, true);
    long lMargin = 4;

    // always want to use the top of the object and viewport as the
    // reference points.
    if (ICoordinateSystem::applicationOrientation() ==
        ICoordinateSystem::originUpperLeft )
    {
      scroll(rectObject.minY() - rectViewport.minY() + lMargin,
             rectObject.minX() - lMargin);
    }
    else
    {
      scroll(rectViewport.maxY() - rectObject.maxY() - lMargin,
             rectObject.minX() - lMargin);
    }
  }
#ifdef IC_WIN
  else   //Native Windows Control
  {
    if ( isTreeView() )
      TreeView_EnsureVisible( ppd->pTreeview->handle(),
                              pcnrobj->ppd->pCnrSet->treeItemFromObject(this) );
    else
    {
      int   itemIndex;
      LV_FINDINFO   lvFindInfo;
      lvFindInfo.flags  = LVFI_PARAM;
      lvFindInfo.lParam = (LPARAM)pcnrobj;

      // Get the objects list view index.
      itemIndex = IContainerControl::ListViewFindItem(
                                     ppd->pListview->handle(),
                                     (-1), &lvFindInfo );

      ListView_EnsureVisible( ppd->pListview->handle(), itemIndex, false );
    }
  }
#endif //IC_WIN
#endif //IC_PMWIN
#ifdef IC_MOTIF
  IRectangle rectViewport = viewPortOnWindow();
  IRectangle rectObject   = iconRectangle(pcnrobj, true);
  long lMargin = 8;
  scroll(rectViewport.top() - rectObject.top() - lMargin,
         rectObject.left() - lMargin);
#endif  //IC_MOTIF
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: scrollToObject

 Implementation: Scroll the Window to include an object in a column
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: scrollToObject(
                                           const IContainerObject* pcnrobj,
                                           const IContainerColumn* pcnrcol,
                                           bool bLeftJustify)
{
  IASSERTPARM(pcnrobj!=0);
  long        lMargin = 4;
  long        horizontalDistance=0;
  long        verticalDistance  =0;
  bool     rightSide=false;
  IRectangle  rectViewport;
  IRectangle  rectObject;

  if (pcnrcol)
  {
    rightSide = isColumnRight(pcnrcol);
    rectViewport = detailsViewPortOnWindow(rightSide);
    rectObject   = detailsObjectRectangle(pcnrobj, pcnrcol);
  }
  else
  {
    rectViewport = detailsViewPortOnWindow(false);
    rectObject   = detailsObjectRectangle(pcnrobj, false);
  }

#ifdef IC_PMWIN
  // convert coordinates so that reference points are correct in rectangles
  if (ICoordinateSystem::applicationOrientation() ==
      ICoordinateSystem::originUpperLeft )
     {
     long   cnrheight = size().height();
     rectViewport.moveTo( IPoint( rectViewport.minX(),
                          cnrheight - rectViewport.maxY() ) );
     rectObject.moveTo( IPoint( rectObject.minX(),
                        cnrheight - rectObject.maxY() ) );
     }
#endif //IC_PMWIN

  if (bLeftJustify)  //scroll to top-left corner
  {
    if (pcnrcol)     //only scroll horizontally if given a column
    {
      horizontalDistance = rectObject.left() - lMargin;
    }
    verticalDistance = rectViewport.top() - rectObject.top() - lMargin;
  }
  else             //scroll object just into view at closest border
  {
    if (pcnrcol)   //only scroll horizontally if given a column
    {
      if ( (rectObject.left() < rectViewport.left()) ||
           ( (rectObject.right()-rectObject.left()) >
             (rectViewport.right()-rectViewport.left()) ) )
      {
        horizontalDistance = rectObject.left() - rectViewport.left();
      }
      else if (rectObject.right() > rectViewport.right() )
      {
        horizontalDistance = rectObject.right() - rectViewport.right();
      }
    }

    if ( (rectObject.top() > rectViewport.top()) ||
         ( (rectObject.top()-rectObject.bottom()) >
           (rectViewport.top()-rectViewport.bottom()) ) )
    {
      verticalDistance = rectViewport.top() - rectObject.top();
    }
    else if (rectObject.bottom() < 0)
    {
      verticalDistance = rectViewport.bottom() - rectObject.bottom();
    }
  }
  scroll( verticalDistance, horizontalDistance, rightSide );

  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setIconSize

 Implementation: Set the size of icons.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setIconSize(const ISize& sizeIcon)
{
  ICnrInfo cnrInfo;
  cnrInfo.slBitmapOrIcon.cx = sizeIcon.width();
  cnrInfo.slBitmapOrIcon.cy = sizeIcon.height();
  setContainerInfo(&cnrInfo, CMA_SLBITMAPORICON);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: iconSize

 Implementation: Query the size of icons.
------------------------------------------------------------------------------*/
ISize IContainerControl :: iconSize() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  return ISize(cnrInfo.slBitmapOrIcon.cx, cnrInfo.slBitmapOrIcon.cy);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: moveIconTo

 Implementation: Move an icon in the Icon View.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: moveIconTo(IContainerObject* object,
                                     const IPoint& pt)
{
  IFUNCTRACE_DEVELOP();
  IASSERTPARM(object!=0);

  ITRACE_DEVELOP("dropPoint = " + pt.asString()) ;

  bool fContainsObject = containsObject(object);
  IMiniCnrRecord* pcnrrec = IRecFromObj(object);

  if(fContainsObject)
  {
    IASSERTSTATE(parentObject(object)==0);
    /* remove record from previous location */
    if (!sendEvent(CM_ERASERECORD,
                   MPFROMP(pcnrrec),0).asUnsignedLong())
    {
       ITHROWGUIERROR("CM_ERASERECORD");
    }
  }

  /* set new position */
  pcnrrec->ptlIcon.x = pt.x();
  pcnrrec->ptlIcon.y = pt.y();

#ifdef IC_PMWIN
  // convert to application coordinates if needed
  if (ICoordinateSystem::applicationOrientation() == IC_CNR_COORDSYS )
  {
    pcnrrec->ptlIcon.y = size().height() - pt.y() - iconSize().height();
  }

#ifdef IC_WIN
  // If this is not a PM Compatible container then we need to map the
  // coordinates to the native container, which is a child of the container
  // wrapper. The reason for this is the container may have a title which
  // will skew the coordinates by approximately the size of the title.
  if ( !ppd->pmCompatible )
  {
    // Get the native container that is currently being used
    IWindow *currentNativeCnr;
    if ( isTreeView() )
      currentNativeCnr = ppd->pTreeview;
    else
      currentNativeCnr = ppd->pListview;

    // Now map the point that is relative to our container wrapper to the
    // native container
    IPoint temp(pcnrrec->ptlIcon.x, pcnrrec->ptlIcon.y);
    temp = IWindow::mapPoint( temp,
                              handle(),
                              currentNativeCnr->handle() );
    pcnrrec->ptlIcon.x = temp.x();
    pcnrrec->ptlIcon.y = temp.y();
  }
#endif //IC_WIN
#endif //IC_PMWIN


  ITRACE_DEVELOP("cnr coords ") ;
  ITRACE_DEVELOP("x = " + IString((long) (pcnrrec->ptlIcon.x))) ;
  ITRACE_DEVELOP("y = " + IString((long) (pcnrrec->ptlIcon.y))) ;

  if(fContainsObject)
  {
    unsigned short fsFlags = CMA_REPOSITION;

    if(!sendEvent(CM_INVALIDATERECORD,
                  MPFROMP(&pcnrrec),
                  MPFROM2SHORT(1,fsFlags)).asUnsignedLong())
    {
       ITHROWGUIERROR("CM_INVALIDATERECORD");
    }
  }

  ITRACE_DEVELOP(IString("New Icon location is:")+ iconRectangle(object, false).asString());
  ITRACE_DEVELOP(IString("New Icon & Text location is:")+ iconRectangle(object, true).asString());

   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: iconRectangle

 Implementation: Query an icon's rectangle.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: iconRectangle(const IContainerObject* object,
                                             bool fIncludeText) const
{
   IASSERTSTATE(object!=0 && containsObject(object));
   RECTL rectl;
   QUERYRECORDRECT  qryrecrect;

#ifdef IC_MOTIF
   qryrecrect.fsExtent = CMA_ICON;
#endif
#ifdef IC_PMWIN
   /* Avoid error when calling CM_QUERYRECORDRECT by not setting */
   /* CMA_ICON in either the text view or the tree-text view.    */
   if ( isTextView() )
     qryrecrect.fsExtent = 0;
   else
     qryrecrect.fsExtent = CMA_ICON;
#endif  //IC_PMWIN

   if(fIncludeText!=false)
      qryrecrect.fsExtent |= CMA_TEXT;
   qryrecrect.pRecord = (PRECORDCORE)IRecFromObj(object);
   qryrecrect.cb = sizeof(QUERYRECORDRECT);
   unsigned fSuccess = sendEvent( CM_QUERYRECORDRECT,
                                  MPFROMP(&rectl),
                                  MPFROMLONG(&qryrecrect));
   if(fSuccess)
   {
#ifdef IC_WIN
     IRectangle rect(rectl.left, rectl.bottom, rectl.right, rectl.top);
#endif
#ifdef IC_PM
     IRectangle rect(rectl.xLeft, rectl.yBottom, rectl.xRight, rectl.yTop);
#endif
#ifdef IC_MOTIF
     IRectangle rect(rectl.lXLeft, rectl.lYBottom, rectl.lXRight, rectl.lYTop);
#endif

#ifdef IC_PMWIN
     // convert to application coordinates if needed
     if (ICoordinateSystem::applicationOrientation() == IC_CNR_COORDSYS )
     {
       rect.moveTo( IPoint( rect.minX(),
                            size().height() - rect.maxY() ) );
     }
#endif //IC_PMWIN
     return rect;
   }
   else
     return IRectangle();

}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: textRectangle

 Implementation: Query an object's text rectangle.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: textRectangle(const IContainerObject* object) const
{
   IASSERTSTATE(object!=0 && containsObject(object));
   RECTL rectl;
   QUERYRECORDRECT  qryrecrect;
   qryrecrect.fsExtent = CMA_TEXT;
   qryrecrect.pRecord = (PRECORDCORE)IRecFromObj(object);
   qryrecrect.cb = sizeof(QUERYRECORDRECT);
   unsigned fSuccess = sendEvent( CM_QUERYRECORDRECT,
                                  MPFROMP(&rectl),
                                  MPFROMLONG(&qryrecrect));
   if(fSuccess)
   {
#ifdef IC_WIN
     IRectangle rect(rectl.left, rectl.bottom, rectl.right, rectl.top);
#endif
#ifdef IC_PM
     IRectangle rect(rectl.xLeft, rectl.yBottom, rectl.xRight, rectl.yTop);
#endif
#ifdef IC_MOTIF
     IRectangle rect(rectl.lXLeft, rectl.lYBottom, rectl.lXRight, rectl.lYTop);
#endif
#ifdef IC_PMWIN
     // convert to application coordinates if needed
     if (ICoordinateSystem::applicationOrientation() == IC_CNR_COORDSYS )
     {
       rect.moveTo( IPoint( rect.minX(),
                            size().height() - rect.maxY() ) );
     }
#endif //IC_PMWIN
     return rect;
   }
   else
     return IRectangle();
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: detailsObjectRectangle

 Implementation: Query an object's details view rectangle.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: detailsObjectRectangle
                                       (const IContainerObject* object,
                                        bool fRightWindow ) const
{
   IASSERTSTATE(object!=0 && containsObject(object));
   RECTL rectl;
   QUERYRECORDRECT  qryrecrect;
   qryrecrect.fsExtent = 0;
   qryrecrect.pRecord = (PRECORDCORE)IRecFromObj(object);
   qryrecrect.cb = sizeof(QUERYRECORDRECT);
   qryrecrect.fRightSplitWindow = fRightWindow;
   unsigned fSuccess = sendEvent( CM_QUERYRECORDRECT,
                                  MPFROMP(&rectl),
                                  MPFROMLONG(&qryrecrect));
   if(fSuccess)
   {
#ifdef IC_WIN
     IRectangle rect(rectl.left, rectl.bottom, rectl.right, rectl.top);
#endif
#ifdef IC_PM
     IRectangle rect(rectl.xLeft, rectl.yBottom, rectl.xRight, rectl.yTop);
#endif
#ifdef IC_MOTIF
     IRectangle rect(rectl.lXLeft, rectl.lYBottom, rectl.lXRight, rectl.lYTop);
#endif
#ifdef IC_PMWIN
     // convert to application coordinates if needed
     if (ICoordinateSystem::applicationOrientation() == IC_CNR_COORDSYS )
     {
       rect.moveTo( IPoint( rect.minX(),
                            size().height() - rect.maxY() ));
     }
#endif //IC_PMWIN
     return rect;
   }
   else
     return IRectangle();
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: detailsObjectRectangle

 Implementation: Query an object's column rectangle (the cell).
                 Returns rectangle in window coordinates.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: detailsObjectRectangle
                                       (const IContainerObject* object,
                                        const IContainerColumn* column) const
{
#ifdef IC_PMWIN
  IASSERTPARM(column!=0 && object!=0);
  IASSERTSTATE(containsObject(object) && column->pfieldinfoCl);

  if ( ppd->pmCompatible )
  {
    // get the complete details rectangle for this object
    IRectangle lRectRecord = detailsObjectRectangle(object, false);

    /*-----------------------------------------------------------
    if the column is not visible, return a rectangle with a height but no width
   ------------------------------------------------------------*/
    if (!column->isVisible())
    {
      IRectangle rect(0,
                      lRectRecord.bottom(),
                      0,
                      lRectRecord.top());
      return rect;
    }

    /*-----------------------------------------------------------
   else if it is visible, continue calculating the cell size
   ------------------------------------------------------------*/
   // get font metrics
    long        LeftSide=0, RightSide=0;
    long        LeftMargin, RightMargin;
    long        lAveCharWidth;
 #ifdef IC_WIN
    HDC         hdc;
    TEXTMETRIC  TextMetric;
    hdc = GetDC( handle() );
    GetTextMetrics( hdc, &TextMetric );
    ReleaseDC( handle(), hdc );
    lAveCharWidth = TextMetric.tmAveCharWidth;
 #endif  //IC_WIN
 #ifdef IC_PM
    HPS         hps;
    FONTMETRICS FontMetrics;
    hps = WinGetPS( handle() );
    GpiQueryFontMetrics( hps, sizeof(FONTMETRICS), &FontMetrics );
    WinReleasePS( hps );
    lAveCharWidth = FontMetrics.lAveCharWidth;
 #endif  //IC_PM
    LeftMargin  = (3 * lAveCharWidth) / 2;
    RightMargin = (3 * lAveCharWidth) - LeftMargin;

    // determine which side the column is on
    IContainerControl::ColumnCursor visiblecolcur(*this, true);
    if( isColumnRight(column) )
    {
      visiblecolcur.setCurrent(detailsViewSplit());
      visiblecolcur.setToNext();
    }
    else
      visiblecolcur.setToFirst();

    // set up a second column cursor
    IContainerControl::ColumnCursor lastvisiblecur(*this, true);
    if( !isColumnRight(column)  && detailsViewSplit() )
    {
      lastvisiblecur.setCurrent(detailsViewSplit());
      if ( !(detailsViewSplit()->isVisible()) )
        lastvisiblecur.setToPrevious();
    }
    else
    {
      lastvisiblecur.setToLast();
    }

    //  determine the right & left sides side of the cell
    RightSide = columnAt(visiblecolcur)->displayWidth();
    RightSide += lAveCharWidth + RightMargin;

    while (visiblecolcur.isValid() && columnAt(visiblecolcur) != column )
    {
      LeftSide = RightSide;
      visiblecolcur.setToNext();
      RightSide += columnAt(visiblecolcur)->displayWidth();
      RightSide += LeftMargin + RightMargin;
    }

    if ( columnAt(visiblecolcur) == columnAt(lastvisiblecur) )
      RightSide += lAveCharWidth - RightMargin;

    /*-----------------------------------------------------------
     Convert the columns workspace values to window coordinates.
   ------------------------------------------------------------*/
    IRectangle lRectViewportWork;
    lRectViewportWork = detailsViewPortOnWorkspace(isColumnRight(column));
    IRectangle rect(LeftSide - lRectViewportWork.left(),
                    lRectRecord.bottom(),
                    RightSide - lRectViewportWork.left(),
                    lRectRecord.top());
    if (isColumnRight(column))
    {
      IRectangle lRectViewportWindow;

      lRectViewportWindow = detailsViewPortOnWindow(true);
      rect = IRectangle(LeftSide - lRectViewportWork.left()
                               + lRectViewportWindow.left(),
                      lRectRecord.bottom(),
                      RightSide - lRectViewportWork.left()
                               + lRectViewportWindow.left(),
                      lRectRecord.top());
    }
    return rect;
  }
#ifdef IC_WIN
  else
  {
    POINT viewPortOffset;
    int      index( 0 ),
             columnId( -1 );
    long   leftSide( 0 ),
           rightSide( 0 ),
           topSide( 0 ),
           bottomSide( 0 );

    if ( ListView_GetItemCount( ppd->pListview->handle() ) > 0 )
    {
      ListView_GetItemPosition( ppd->pListview->handle(), 0, &viewPortOffset );
      leftSide += viewPortOffset.x;
    }

    columnId = ppd->columnSet->columnIdFromContainerColumn(
                                            (IContainerColumn*)column );

    if ( columnId != (-1) )
    {
      for ( index=0; index<columnId; index++ )
      {
        leftSide += ListView_GetColumnWidth( ppd->pListview->handle(), index );
      }

      rightSide = leftSide + ListView_GetColumnWidth( ppd->pListview->handle(),
                                                      columnId );
    }
    int    objectIndex;
    LV_FINDINFO   lvFindInfo;
    lvFindInfo.flags  = LVFI_PARAM;
    lvFindInfo.lParam = (LPARAM)object;

    // Get the objects list view index.
    objectIndex = IContainerControl::ListViewFindItem(
                                     ppd->pListview->handle(),
                                     (-1), &lvFindInfo );

    RECT   itemRect;
    ListView_GetItemRect( ppd->pListview->handle(), objectIndex,
                          &itemRect, LVIR_BOUNDS );

    IRectangle rect(leftSide, itemRect.bottom, rightSide, itemRect.top);

    // Adjust for possible title window.
    if ( isTitleVisible() )
    {
      rect.moveBy( IPair( 0, ppd->pTitle->size().height() ) );
    }


    if (ICoordinateSystem::applicationOrientation() == IC_CNR_COORDSYS )
      rect = ICoordinateSystem::convertToApplication( rect, size() );

    return rect;
  }
#endif // IC_WIN
#endif // IC_PMWIN
// Stub for now on Motif until IContainerColumn is ported
// Not sure if we can ever get this function to work though...
#ifdef IC_MOTIF
   // get the complete details rectangle for this object
   IRectangle lRectRecord = detailsObjectRectangle(object, false);

   /*-----------------------------------------------------------
   if the column is not visible, return a rectangle with a height but no width
  ------------------------------------------------------------*/
   if (!column->isVisible())
   {
     IRectangle rect(0,
                     lRectRecord.bottom(),
                     0,
                     lRectRecord.top());
     return rect;
   }

  return ppd->detailsCellRectangle( (IContainerObject*)object, column );
#endif //IC_MOTIF
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: disableDrawItem

 Implementation: Turn Owner draw off.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: disableDrawItem()
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   setContainerAttributes(CA_OWNERDRAW,0);
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl::enableDrawBackground

 Implementation: Enable the container to paint the background.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: enableDrawBackground(bool fEnable)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   if(fEnable)
     setContainerAttributes(0,CA_OWNERPAINTBACKGROUND);
   else
     setContainerAttributes(CA_OWNERPAINTBACKGROUND, 0);
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isDrawBackgroundEnabled

 Implementation: Query Background drawing state.
------------------------------------------------------------------------------*/
bool IContainerControl :: isDrawBackgroundEnabled() const
{
#ifdef IC_PMWIN
  return ( (containerAttributes() & CA_OWNERPAINTBACKGROUND) ? true : false);
#endif
#ifdef IC_MOTIF
  return false;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl::disableDrawBackground

 Implementation: Don't allow the container to paint the background.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: disableDrawBackground()
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   setContainerAttributes(CA_OWNERPAINTBACKGROUND,0);
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: convertToWorkspace

 Implementation: Convert the given rectangle in window coordinates to
                 container workspace coordinates.  This conversion is valid
                 only for the current view as we need to query the current
                 viewport for the conversion.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: convertToWorkspace
                                       (const IRectangle& windowRectangle,
                                        bool fRightWindow ) const
{
#ifdef IC_PMWIN
  IRectangle lRectViewportWindow;
  IRectangle rect = windowRectangle;
  IRectangle lRectViewportWork = detailsViewPortOnWorkspace(fRightWindow);
  ISize  adjustSize;
  // convert from application coordinates if needed
  if (ICoordinateSystem::applicationOrientation() ==
      ICoordinateSystem::originUpperLeft )
     {
     adjustSize = size();
     lRectViewportWork.moveTo( IPoint( lRectViewportWork.minX(),
                                       adjustSize.height() -
                                       lRectViewportWork.maxY()));
     rect.moveTo( IPoint( rect.minX(),
                          adjustSize.height() -
                          rect.maxY()));
     }

  if (fRightWindow)
  {
    lRectViewportWindow = detailsViewPortOnWindow(true);
    rect = IRectangle(rect.left()   + lRectViewportWork.left()
                         - lRectViewportWindow.left(),
                      rect.bottom() + lRectViewportWork.bottom(),
                      rect.right()  + lRectViewportWork.left()
                         - lRectViewportWindow.left(),
                      rect.top()    + lRectViewportWork.bottom());
  }
  else
  {
    rect = IRectangle(rect.left()   + lRectViewportWork.left(),
                      rect.bottom() + lRectViewportWork.bottom(),
                      rect.right()  + lRectViewportWork.left(),
                      rect.top()    + lRectViewportWork.bottom());
  }
  // convert to application coordinates if needed
  if (ICoordinateSystem::applicationOrientation() ==
      ICoordinateSystem::originUpperLeft )
     {
     rect.moveTo( IPoint( rect.minX(),
                          adjustSize.height() -
                          rect.maxY()));
     }
#endif //IC_PMWIN
#ifdef IC_MOTIF
  IRectangle lRectViewportWindow;
  IRectangle lRectViewportWork;
  IRectangle rect;
  lRectViewportWork = detailsViewPortOnWorkspace(fRightWindow);

  if (fRightWindow)
  {
    lRectViewportWindow = detailsViewPortOnWindow(true);
    rect = IRectangle(windowRectangle.left()   + lRectViewportWork.left()
                      - lRectViewportWindow.left(),
                    windowRectangle.bottom() + lRectViewportWork.bottom(),
                    windowRectangle.right()  + lRectViewportWork.left()
                      - lRectViewportWindow.left(),
                    windowRectangle.top()    + lRectViewportWork.bottom());
  }
  else
  {
    rect = IRectangle(windowRectangle.left()   + lRectViewportWork.left(),
                    windowRectangle.bottom() + lRectViewportWork.bottom(),
                    windowRectangle.right()  + lRectViewportWork.left(),
                    windowRectangle.top()    + lRectViewportWork.bottom());
  }

  Dimension container_width, clipWindow_width, scrollBar_height, spacing;
  XtVaGetValues(ppd->container,  XmNwidth, &container_width,  NULL );
  XtVaGetValues(ppd->clipWindow, XmNwidth, &clipWindow_width, NULL );

  if( container_width > clipWindow_width )  // horizontal scrollBar visible?
  {
     XtVaGetValues(ppd->horizontalScroll, XmNheight, &scrollBar_height, NULL );
     XtVaGetValues(ppd->scrolledWindow, XmNspacing, &spacing, NULL );
  }
  else
  {
     scrollBar_height = 0;
     spacing          = 0;
  }

  rect = IRectangle( rect.left(),
                     rect.bottom() - (scrollBar_height + spacing),
                     rect.right(),
                     rect.top()    - (scrollBar_height + spacing));
#endif //IC_MOTIF
  return rect;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: columnUnderPoint

 Implementation: Determine the column under a point in window
                 coordinates.
------------------------------------------------------------------------------*/
IContainerColumn* IContainerControl :: columnUnderPoint(const IPoint& point) const
{
  IMODTRACE_DEVELOP("CnrCtl :: columnUnderPoint");
  IASSERTSTATE( isDetailsView() );

#ifdef IC_MOTIF
  return ppd->columnUnderPoint( point );
#endif
#ifdef IC_PMWIN
  if ( ppd->pmCompatible )
  {
    IRectangle lRectLeftViewportWindow;
    IRectangle lRectRightViewportWindow;
    IContainerControl::ColumnCursor visiblecolcur(*this, true);
    IContainerControl::ColumnCursor lastcolcur(*this, true);

    /*****************************************************************/
    /* Make sure x value of point is within the confines of the      */
    /* current details viewport.  Also, set up the starting column   */
    /* cursor and the last column cursor.                            */
    /*****************************************************************/
    lRectLeftViewportWindow = detailsViewPortOnWindow(false);
    if (detailsViewSplit())
    {
      lRectRightViewportWindow = detailsViewPortOnWindow(true);

      IASSERTPARM( ((point.x() >= lRectLeftViewportWindow.left()) &&
                    (point.x() <= lRectRightViewportWindow.right())) );
      /*******************************************************/
      /* Set up the start and end columns.                   */
      /* Count the split bar area as the left D.V, window.   */
      /*******************************************************/
      if ( (point.x() >= lRectLeftViewportWindow.left()) &&
           (point.x() <= lRectRightViewportWindow.left()) )
      {
        visiblecolcur.setToFirst();
        lastcolcur.setCurrent(detailsViewSplit());
        if ( !(detailsViewSplit()->isVisible()) )
          lastcolcur.setToPrevious();
      }
      else
      {
        visiblecolcur.setCurrent(detailsViewSplit());
        visiblecolcur.setToNext();
        lastcolcur.setToLast();
      }
    }
    else
    {
      IASSERTPARM( (point.x() >= lRectLeftViewportWindow.left()) &&
                   (point.x() <= lRectLeftViewportWindow.right()) );
      visiblecolcur.setToFirst();
      lastcolcur.setToLast();
    }

    /****************************************************************/
    /* Query the containers font to determine the column margins    */
    /****************************************************************/
    long        LeftSide=0, RightSide=0;
    long        LeftMargin, RightMargin;
    long        lAveCharWidth;
  #ifdef IC_WIN
     HDC         hdc;
     TEXTMETRIC  TextMetric;
     hdc = GetDC( handle() );
     GetTextMetrics( hdc, &TextMetric );
     ReleaseDC( handle(), hdc );
     lAveCharWidth = TextMetric.tmAveCharWidth;
  #endif  //IC_WIN
  #ifdef IC_PM
    HPS         hps;
    FONTMETRICS FontMetrics;
    hps = WinGetPS( handle() );
    GpiQueryFontMetrics( hps, sizeof(FONTMETRICS), &FontMetrics );
    WinReleasePS( hps );
    lAveCharWidth = FontMetrics.lAveCharWidth;
  #endif //IC_PM
    LeftMargin  = (3 * lAveCharWidth) / 2;
    RightMargin = (3 * lAveCharWidth) - LeftMargin;
    /*****************************************************************/
    /* Adjust the starting point to container window coordinates     */
    /*****************************************************************/
    IRectangle lRectViewportWork;
    lRectViewportWork = detailsViewPortOnWorkspace(isColumnRight
                             (columnAt(visiblecolcur)) );
    LeftSide -= lRectViewportWork.left();
    RightSide -= lRectViewportWork.left();
    if (isColumnRight(columnAt(visiblecolcur)))
    {
      LeftSide += lRectRightViewportWindow.left();
      RightSide += lRectRightViewportWindow.left();
    }

    /******************************************************************/
    /* Compute the first visible column, if it is it, return now.     */
    /******************************************************************/
    RightSide += columnAt(visiblecolcur)->displayWidth();
    RightSide += lAveCharWidth + RightMargin;
    if ( point.x() <= RightSide )
      return (columnAt(visiblecolcur));

    /******************************************************************/
    /* Loop thru all visible columns and return if point matches      */
    /******************************************************************/
    while (visiblecolcur.isValid() && (columnAt(visiblecolcur) !=
                                       columnAt(lastcolcur)) )
    {
      LeftSide = RightSide;
      visiblecolcur.setToNext();
      RightSide += columnAt(visiblecolcur)->displayWidth();
      RightSide += LeftMargin + RightMargin;
      if ( point.x() <= RightSide )
        return (columnAt(visiblecolcur));
    }

    return(columnAt(lastcolcur));
  }
#ifdef IC_WIN
  else       //Native ListView
  {
    POINT viewPortOffset;
    int      index( 0 );
    bool  notDone( true );
    long   leftSide( 0 ),
           rightSide( 0 );

    if ( ListView_GetItemCount( ppd->pListview->handle() ) > 0 )
    {
      ListView_GetItemPosition( ppd->pListview->handle(), 0, &viewPortOffset );
      leftSide += viewPortOffset.x;
    }

    while ( notDone )
    {
      rightSide = leftSide + ListView_GetColumnWidth( ppd->pListview->handle(),
                                                      index );
      if ( rightSide > leftSide )
      {
        if ( (leftSide <= point.x()) && (point.x() <= rightSide ) )
          return( ppd->columnSet->containerColumnFromColumnId( index ) );
        index++;
        leftSide = rightSide;
      }
      else
        notDone = false;
    }

    return (IContainerColumn*) 0;
  }
#endif //IC_WIN
#endif //IC_PMWIN
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IContainerControl::isDragStarting                                                 |
|
| Windows does not generate drag messages, so therefore it is up to a
| a control to decide if the conditions exist to start a drag. This is
| is the basic implementation that will just look for a down followed by
| a move more than xxx. If this occurs before we have an up event then
| we have a drag.
|
| Note: if you don't want the event to continue up the owner chain,
|       then set the event result to true.
------------------------------------------------------------------------------*/
bool IContainerControl::isDragStarting( IEvent &event )
{
  IMODTRACE_DEVELOP("IContainerControl::isDragStarting");

  bool brc = false;

  switch ( event.eventId() )
  {
    case WM_LBUTTONDOWN:
    case WM_RBUTTONDOWN:
    {
      ITRACE_DEVELOP("Left button down. Looking for drag");
      IMouseClickEvent mevt( event );
      // Is there a drag object?
      IContainerObject *cnrObject =
          this->objectUnderPoint( mevt.mousePosition() );
      if ( cnrObject )
      {
        // "A window that has captured the mouse receives all mouse input,
        // regardless of the position of the cursor, except when a mouse
        // button is clicked while the cursor hot spot is in the window of
        // another thread."  -- Microsoft Win32 SDK Reference
        if ( ::GetCapture() == NULL )
        {
          ::SetCapture( event.handle());
          ppd->bCheckingDragStart = True;
          ppd->pointLButtonDown = mevt.mousePosition();
        }
      }
      break;
    }

    case WM_LBUTTONUP:
    case WM_RBUTTONUP:
    {
      ITRACE_DEVELOP("WM_LBUTTONUP up event");
      if ( ppd->bCheckingDragStart )
      {
        ::ReleaseCapture();
        //SetCapture( NULL );
        ppd->bCheckingDragStart = False;
        ppd->pointLButtonDown = IPoint(0,0);
      }
      break;
    }

    case WM_MOUSEMOVE:
    {
      if ( ppd->bCheckingDragStart)
      {
        ITRACE_DEVELOP("Mouse move while checking drag start");

        if ( !( event.parameter1().asUnsignedLong() & MK_LBUTTON ) &&
             !( event.parameter1().asUnsignedLong() & MK_RBUTTON ) )
        {
          ::ReleaseCapture();
          //SetCapture( NULL );
          ppd->bCheckingDragStart = False;
          ppd->pointLButtonDown = IPoint(0,0);

        }
        else
        {
          IMouseClickEvent mevt( event );
          IPoint mPoint( ppd->pointLButtonDown );

          if ( abs(mevt.mousePosition().x() - mPoint.x()) >
                     DD_DEFDRAGMINDIST ||
               abs(mevt.mousePosition().y() - mPoint.y()) >
                     DD_DEFDRAGMINDIST )
          {
            /*-----------------------------------------
            | We're checking for a drag start. The has
            | mouse moved out of the start rectangle.
            | A true result will tell some caller to
            | nudge the OpenClass drag framwwork.
            -----------------------------------------*/
            ITRACE_DEVELOP("Starting a drag" );
            ::ReleaseCapture();
            ppd->bCheckingDragStart = FALSE;
            brc = true;
          }
        }
      }
      break;
    }
  }
  return brc;
}
#endif
