// Revision: 82 1.16.1.11 source/ui/cnr/icnrctl1.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrctl1.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the Attribute and Title functions *
*   declared in icnrctl.hpp.                                                   *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINSYS
#define INCL_WINSTDCNR
#define INCL_WINWINDOWMGR

extern "C"
{
  #include <string.h>
}

#include <icnrrec.hpp>  // Must be first for OS flags

#include <itrace.hpp>
#include <iexcept.hpp>
#include <ireslib.hpp>
#include <icnrctl.hpp>
#include <icolor.hpp>
#include <icnrfilt.hpp>

#ifdef IC_MOTIF
  #include <icnrctlm.hpp>
  #include <cnr.h>
#endif

#ifdef IC_PMWIN
  #include <icnrctlw.hpp>
#endif

#ifndef _INOTIFEV_
  #include <inotifev.hpp>
#endif


// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL1_CPP_
  #include <ipagetun.h>
#endif


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setTitle

 Implementation: Put text into the container title.

------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setTitle(const char* pszTitle)
{
  IMODTRACE_DEVELOP("Cnr::setCnrTitle");

  IString newTitle(pszTitle);
  if (newTitle != this->title())
  {
    ICnrInfo cnrInfo;
    /*******************************************************************/
    /* Store the title in the private area of IContainerControl.  This */
    /* is needed since the WC_CONTAINER requires that the app store    */
    /* the memory for the text strings it displays.                    */
    /*******************************************************************/
    strContainerTitle = newTitle;

    /*******************************************************************/
    /* The cnrInfo that we pass the WC_CONTAINER must have the pointer */
    /* to the text string to a valid string that we store.             */
    /*******************************************************************/
    cnrInfo.strCnrTitle = (char*)newTitle;
    setContainerInfo(&cnrInfo, CMA_CNRTITLE);
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setTitle

 Implementation: Put text into the container title using a
   resource id.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setTitle(const IResourceId& resid)
{
  IString strText = resid.resourceLibrary().loadString(resid);
  setTitle((char*)strText);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: title

 Implementation: Query the title of the container.
------------------------------------------------------------------------------*/
IString IContainerControl :: title() const
{
#ifdef IC_MOTIF
  return strContainerTitle ; // the motif cnr widget doesn't keep title info
#else
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  return IString(cnrInfo.strCnrTitle);
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setTitleAlignment

 Implementation: Set the justification attributes for the title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setTitleAlignment(TitleAlignment titleAlign)
{
   unsigned long newAlignment;
   switch (titleAlign)
   {
      case IContainerControl::centered:
      {
         newAlignment = CA_TITLECENTER;
         break;
      }
      case IContainerControl::left:
      {
         newAlignment = CA_TITLELEFT;
         break;
      }
      case IContainerControl::right:
      {
         newAlignment = CA_TITLERIGHT;
         break;
      }
   }

   ICnrInfo cnrInfo;
   containerInfo(&cnrInfo);
   if (!(cnrInfo.flWindowAttr & newAlignment))
   {
     setContainerAttributes(CA_TITLELEFT | CA_TITLERIGHT | CA_TITLECENTER,
                            newAlignment);

#ifdef IC_PMWIN
     refresh(IWindow::paintAll);
#endif
   }
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: showTitle

 Implementation: Show the title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: showTitle(bool fShow)
{
  bool attrChanged = false;
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);

  if (fShow)
  {
    if (!(cnrInfo.flWindowAttr & CA_CONTAINERTITLE))
    {
      setContainerAttributes(0, CA_CONTAINERTITLE);
      attrChanged = true;
    }
  }
  else if (cnrInfo.flWindowAttr & CA_CONTAINERTITLE)
  {
    setContainerAttributes(CA_CONTAINERTITLE, 0);
    attrChanged = true;
  }

  if (attrChanged)
  {
    notifyObservers(INotificationEvent(IContainerControl::titleVisibleId,
                                       *this, true, (void*)fShow));
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: hideTitle

 Implementation: Hide the title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: hideTitle()
{
  showTitle(false);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isTitleVisible

 Implementation: Determine if the title is currently displayed.
------------------------------------------------------------------------------*/
bool IContainerControl :: isTitleVisible() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  if (cnrInfo.flWindowAttr & CA_CONTAINERTITLE)
  {
    return true;
  }
  return false;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: enableTitleUpdate

 Implementation: Enable the title to be edited.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: enableTitleUpdate(bool fEnable)
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);

  if (fEnable)
  {
    if (cnrInfo.flWindowAttr & CA_TITLEREADONLY)
    {
      setContainerAttributes(CA_TITLEREADONLY,0);
    }
  }
  else if (!(cnrInfo.flWindowAttr & CA_TITLEREADONLY))
  {
    setContainerAttributes(0, CA_TITLEREADONLY);
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: disableTitleUpdate

 Implementation: Disable edit of the title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: disableTitleUpdate()
{
  enableTitleUpdate(false);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isTitleWriteable

 Implementation: Query if the container currently allows direct editing
                 of the container title.
------------------------------------------------------------------------------*/
bool IContainerControl :: isTitleWriteable() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  if (cnrInfo.flWindowAttr & CA_TITLEREADONLY)
    return (false);
  else
    return(true);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: showTitleSeparator

 Implementation: Add a separator below the title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: showTitleSeparator(bool fShow)
{
  bool attrChanged = false;
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);

  if (fShow)
  {
    if (!(cnrInfo.flWindowAttr & CA_TITLESEPARATOR))
    {
#ifdef IC_MOTIF
      ppd->containerManagement(ICnrControlData::SHOW_SEPARATOR);
#endif
      setContainerAttributes(0, CA_TITLESEPARATOR);
      attrChanged = true;
    }
  }
  else if (cnrInfo.flWindowAttr & CA_TITLESEPARATOR)
  {
#ifdef IC_MOTIF
    ppd->containerManagement(ICnrControlData::HIDE_SEPARATOR);
#endif
    setContainerAttributes(CA_TITLESEPARATOR, 0);
    attrChanged = true;
  }

  if (attrChanged)
  {
    if ( isVisible() && isRefreshOn() )
    {
      refresh();
    }
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: hideTitleSeparator

 Implementation: Remove the separator below the title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: hideTitleSeparator()
{
  showTitleSeparator(false);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isTitleSeparatorVisible

 Implementation: Determine if the title separator is currently displayed.
                 Separator is only displayed if title is currently displayed.
------------------------------------------------------------------------------*/
bool IContainerControl :: isTitleSeparatorVisible() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);

  /******************************************************************/
  /* Check both flags because the title separator is only displayed */
  /* if the title itself is visible.                                */
  /******************************************************************/
  if ((cnrInfo.flWindowAttr & CA_CONTAINERTITLE) &&
      (cnrInfo.flWindowAttr & CA_TITLESEPARATOR))
  {
    return (true);
  }
  return(false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: titleRectangle

 Implementation:  Queries the container title rectangle in container window
                  coordinates.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: titleRectangle( ) const
{

   if (! isTitleVisible() )
      return IRectangle() ;

#ifdef IC_PMWIN
  IRectangle rect ;
  bool  needToDelete = false;
  IWindow* titleWindow = IWindow::windowWithOwner(CID_CNRTITLEWND,
                                         (IContainerControl*)this);

  if (titleWindow == 0)
  {
    needToDelete = true;
    titleWindow = new IWindow(CID_CNRTITLEWND, (IContainerControl*)this);
  }

  if (titleWindow)
     rect = titleWindow->rect() ;
  else
     rect = IRectangle() ;

  if (needToDelete)
    delete titleWindow;

  return rect;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return ppd->titleRectangle();
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: detailsTitleRectangle

 Implementation:  Queries the details view column title window rectangle in
                  container window coordinates.
------------------------------------------------------------------------------*/
IRectangle IContainerControl :: detailsTitleRectangle( bool rightSide ) const
{
#ifdef IC_PMWIN
  unsigned long  windowId=CID_LEFTCOLTITLEWND;
  if (rightSide)
    windowId = CID_RIGHTCOLTITLEWND;

  bool  needToDelete = false;
  IWindow* detailTitleWindow = IWindow::windowWithOwner(windowId,
                                         (IContainerControl*)this);

  if (detailTitleWindow == 0)
  {
    HWND hwnd = IWINDOWFROMID(this->handle(), windowId) ;
    if (hwnd)
    {
       needToDelete = true;
       detailTitleWindow = new IWindow(windowId, (IContainerControl*)this);
    }
    if (detailTitleWindow == 0)
      return  IRectangle();
  }

  IRectangle  rect = detailTitleWindow->rect();

  if (needToDelete)
    delete detailTitleWindow;

  return rect;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return ppd->detailsTitleRectangle() ;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: showDetailsViewTitles

 Implementation: Show the title in Details View.title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: showDetailsViewTitles(bool fShow)
{
  bool attrChanged = false;
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);

  if (fShow)
  {
    if (!(cnrInfo.flWindowAttr & CA_DETAILSVIEWTITLES))
    {
      setContainerAttributes(0, CA_DETAILSVIEWTITLES);
      attrChanged = true;
    }
  }
  else if (cnrInfo.flWindowAttr & CA_DETAILSVIEWTITLES)
  {
    setContainerAttributes(CA_DETAILSVIEWTITLES, 0);
    attrChanged = true;
  }

  if (attrChanged)
  {
    notifyObservers(INotificationEvent(IContainerControl::detailsViewTitlesId,
                                       *this, true, (void*)fShow));
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: hideDetailsViewTitles

 Implementation: Remove the title in Details View.title.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: hideDetailsViewTitles()
{
  showDetailsViewTitles(false);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: areDetailsViewTitlesVisible

 Implementation: Queries whether the container details view
                 column titles are currently displayed for the
                 details view.
------------------------------------------------------------------------------*/
bool IContainerControl :: areDetailsViewTitlesVisible() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);

  if (cnrInfo.flWindowAttr & CA_DETAILSVIEWTITLES)
    return(true);
  else
    return (false);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setLineSpacing

 Implementation: Set the space in Pixels between lines.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setLineSpacing(long lLineSpacing )
{
  ICnrInfo cnrInfo;
  cnrInfo.cyLineSpacing = (short)lLineSpacing;
  setContainerInfo(&cnrInfo, CMA_LINESPACING);
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: lineSpacing

 Implementation: Query the space in Pixels between lines.
------------------------------------------------------------------------------*/
long IContainerControl :: lineSpacing() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  return (long)( (cnrInfo.cyLineSpacing > 0) ? cnrInfo.cyLineSpacing : 0 );
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setEmphasis

 Implementation: Protected function to change object emphasis.
------------------------------------------------------------------------------*/
void IContainerControl :: setEmphasis(IContainerObject* object,
                                     unsigned long ulEmphasisAttribute,
                                     bool fOn)
{
   IASSERTPARM(object!=0);
#ifdef IC_MOTIF
   if (fOn)
     IRecFromObj(object)->flRecordAttr |= ulEmphasisAttribute;
   else
     IRecFromObj(object)->flRecordAttr &= ~ulEmphasisAttribute;
#endif
   {
     unsigned fSuccess = sendEvent( CM_SETRECORDEMPHASIS,
                                    IRecFromObj(object),
                                    MPFROM2SHORT(fOn, ulEmphasisAttribute));
     if(!fSuccess)
     {
        ITHROWGUIERROR("CM_SETRECORDEMPHASIS");
     }
   }
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setAttributes

 Implementation: Protected function to set attributes.
------------------------------------------------------------------------------*/
void IContainerControl :: setAttributes(IContainerObject* object,
                                       unsigned long flAttrOff,
                                       unsigned long flAttrOn)
{
   IMODTRACE_DEVELOP("CnrCtl:setAttributes");
   IASSERTPARM(object!=0);
   IMiniCnrRecord* pbase= IRecFromObj(object);
#ifdef IC_MOTIF
   pbase->flRecordAttr = attributes(object);
   pbase->flRecordAttr &= ~(flAttrOff);
   pbase->flRecordAttr |= (flAttrOn);

   bool fSuccess = sendEvent(CM_INVALIDATERECORD,
             &(pbase),MPFROM2SHORT(1, CMA_NOREPOSITION));

   if(!fSuccess)
   {
      ITHROWGUIERROR("CM_INVALIDATERECORD");
   }
#endif
#ifdef IC_PMWIN
   if ( ppd->pmCompatible )
   {
     pbase->flRecordAttr = attributes(object);
     pbase->flRecordAttr &= ~(flAttrOff);
     pbase->flRecordAttr |= (flAttrOn);

     unsigned fSuccess = sendEvent(CM_INVALIDATERECORD,
                                   &(pbase),
                                   MPFROM2SHORT(1, CMA_NOREPOSITION));
     if(!fSuccess)
     {
        ITHROWGUIERROR("CM_INVALIDATERECORD");
     }
   }
   else   // Native Windows Container -> Use CM_SETRECORDEMPHASIS
   {
     if ( flAttrOff )
       sendEvent(CM_SETRECORDEMPHASIS, pbase,
                 MPFROM2SHORT(false, (unsigned short)flAttrOff));

     if ( flAttrOn )
       sendEvent(CM_SETRECORDEMPHASIS, pbase,
                 MPFROM2SHORT(true, (unsigned short)flAttrOn));
   }
#endif //IC_PMWIN
   return;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: attributes

 Implementation: Protected function to query attributes.

   Note: This function is constant so that the query functions
         can be but it does refresh the object record.
------------------------------------------------------------------------------*/
unsigned long IContainerControl :: attributes(const IContainerObject* object) const
{
   IASSERTPARM(object!=0);
   void* pbase= (IContainerObject*)IRecFromObj(object);
   unsigned fSuccess = sendEvent(CM_QUERYRECORDINFO,
             MPFROMP(&pbase),MPFROMSHORT(1));
   if(!fSuccess)
   {
      ITHROWGUIERROR("CM_QUERYRECORDINFO");
   }
   return object->pbase->flRecordAttr;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isVisible

 Implementation: Queries if the container window is visible.
------------------------------------------------------------------------------*/
bool IContainerControl::isVisible() const
{
   return Inherited::isVisible();
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: refresh

 Implementation: Refresh all records in this container.
    There area several variations of this method.
    If immediate is set to true, painting will occur
    synchronously, otherwise it will be async.
  NOTE: For MOTIF, immediate is ignored
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: refresh( bool immediate )
{
  IFUNCTRACE_DEVELOP();
#ifdef IC_PMWIN
  /*******************/
  /* Show the window */
  /*******************/
  show();
#endif //IC_PMWIN

  this->refresh( immediate ?
                   IWindow::paintAllImmediate : IWindow::paintAll );

  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: refresh
 This version is here for compilation reasons (refresh is virtual in base class)
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: refresh(
                                        const IRectangle& invalidRectangle,
                                        bool immediate)
{
  Inherited::refresh(invalidRectangle, immediate);
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: refresh
 This version is here for compilation reasons (refresh is virtual in base class)
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: refresh( IWindow::RefreshType type )
{
#ifdef IC_PMWIN
  if ( isDetailsView() )
  {
     /**************************************************************/
     /* Because the CM_INVALIDATERECORD message sometimes has      */
     /* problems updating the scroll bars in details view,         */
     /* we instead use CM_INVALIDATEDETAILFIELDINFO to simply      */
     /* refresh the screen and alleviate the scrollbar problems.   */
     /**************************************************************/
     unsigned
       success = this->sendEvent( CM_INVALIDATEDETAILFIELDINFO );
     if ( ! success )
     {
        ITHROWGUIERROR( "CM_INVALIDATEDETAILFIELDINFO" );
     }
  }
  else
  {
     if ( ppd->pmCompatible )
     {
        /*************************************************************/
        /* Iterate through all objects and update them for the given */
        /* container. We need to make sure the external PM container */
        /* recordcore is updated to represent this object in this    */
        /* container before we call CM_INVALIDATERECORD.             */
        /*************************************************************/
        IContainerControl::ObjectCursor
          cursor( *this );
        for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
        {
           this->containsObject( this->objectAt( cursor ) );
        }
     }

     unsigned long
       fsFlags = CMA_REPOSITION | CMA_ERASE;
     unsigned
       success = this->sendEvent( CM_INVALIDATERECORD, 0,
                                  MPFROM2SHORT( 0, fsFlags ) );
     if ( ! success )
     {
        ITHROWGUIERROR( "CM_INVALIDATERECORD" );
     }
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  ppd->refresh();
#endif

  flClState &= ~IContainerControl::needsRefresh;

  this->Inherited::refresh( type );

  return *this;
}

IContainerControl& IContainerControl :: refresh(IContainerObject* pcnrobj,
                                                bool immediate)
{
  IFUNCTRACE_DEVELOP() ;

  if (pcnrobj!=0)
  {
    pcnrobj->refresh(this, immediate);
  }
  return *this;
}

IContainerControl& IContainerControl ::setRefreshOff()
{
  setRefreshOn(false);
  return *this;
}

IContainerControl& IContainerControl ::setRefreshOn(bool fOn)
{

  if (fOn)
  {
    if (flClState & IContainerControl::noRefresh)
    {
#ifdef IC_PMWIN
      enableUpdate(true);
#endif

#ifdef IC_MOTIF
      ppd->setRefreshOn(true) ;
#endif
      flClState &= ~IContainerControl::noRefresh;
    }
  }
  else if (!(flClState & IContainerControl::noRefresh))
  {
#ifdef IC_PMWIN
    enableUpdate(false);
#endif

#ifdef IC_MOTIF
      ppd->setRefreshOn(false) ;
#endif
    flClState |= IContainerControl::noRefresh;
  }
  return *this;
}

#ifdef IC_MOTIF
bool IContainerControl::isRefreshOn() const
{
  return ((flClState & IContainerControl::noRefresh) ? false : true);
}
#endif

/*------------------------------------------------------------------------------
 Function Name: showObject(), hideObject()

 Implementation: show, hide the specified container object.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: showObject( IContainerObject* pcnrobj,
                                                    bool fVisible)
{
  if (pcnrobj!=0)
     pcnrobj->show(fVisible, this);
  return *this;
}

IContainerControl& IContainerControl :: hideObject( IContainerObject* pcnrobj)
{
  if (pcnrobj!=0)
     pcnrobj->show(false, this);
  return *this;
}



/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setSingleSelection

 Implementation: Change selection to single.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setSingleSelection()
{
#ifdef IC_MOTIF
   ppd->setSelection(CCS_SINGLESEL);
#endif
#ifdef IC_PMWIN
   /***********************************************************************/
   /* We could be coming from extended/multiple mode where more than one  */
   /* object is selected.  We need to remove selection from all objects   */
   /* other than the cursored object.  Additionally, in either of the     */
   /* other modes, it is possible that no object is selected.  Thus we    */
   /* need to ensure the cursored object is also selected.                */
   /*                                                                     */
   /* Don't Bother with this logic if the container was already in single */
   /* selection mode when this function(setSingleSelection) is called.    */
   /* This is important because you can't remove object selection in      */
   /* single selection mode -> an exception will be thrown.               */
   /***********************************************************************/
   if ( !isSingleSelection() )
   {
     IContainerControl::ObjectCursor selCur(*this,IContainerObject::selected);
     for (selCur.setToFirst(); selCur.isValid(); selCur.setToNext() )
     {
       if ( !isCursored( objectAt(selCur) ))
         removeSelected( objectAt(selCur) );
     }

     IContainerControl::ObjectCursor curCur(*this,IContainerObject::cursored);
     curCur.setToFirst();
     if (curCur.isValid() && !isSelected(objectAt(curCur)) )
       setSelected( objectAt(curCur) );
   }

   /***********************************************************************/
   /* Query current style, remove other 2 possibilites(extended,multiple) */
   /* and set single selection style.                                     */
   /***********************************************************************/
   unsigned long ulStyle = style();
   unsigned long ulOldStyle = ulStyle;
   ulStyle &= ~(extendedSelection.asUnsignedLong() |
                multipleSelection.asUnsignedLong());
   ulStyle |= singleSelection.asUnsignedLong();

   if (ulStyle != ulOldStyle)
     setStyle(ulStyle);

#ifdef IC_WIN
   if (!ppd->pmCompatible)
   {
     ppd->setSelection(CCS_SINGLESEL);
   }
#endif //IC_WIN
#endif //IC_PMWIN

   return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setMultipleSelection

 Implementation: Change selection to multiple.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setMultipleSelection()
{
   unsigned long ulStyle = style();
   unsigned long ulOldStyle = ulStyle;
   ulStyle &= ~(extendedSelection.asUnsignedLong() |
                singleSelection.asUnsignedLong());
   ulStyle |= multipleSelection.asUnsignedLong();

   if (ulStyle != ulOldStyle)
     setStyle(ulStyle);

#ifdef IC_MOTIFWIN
#ifdef IC_WIN
   if (!ppd->pmCompatible)
#endif
     ppd->setSelection(CCS_MULTIPLESEL);
#endif //IC_MOTIFWIN

   return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setExtendedSelection

 Implementation: Change selection to extended.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: setExtendedSelection()
{
   unsigned long ulStyle = style();
   unsigned long ulOldStyle = ulStyle;
   ulStyle &= ~(multipleSelection.asUnsignedLong() |
                singleSelection.asUnsignedLong());
   ulStyle |= extendedSelection.asUnsignedLong();

   if (ulStyle != ulOldStyle)
     setStyle(ulStyle);

#ifdef IC_WIN
   if (!ppd->pmCompatible)
     ppd->setSelection(CCS_MULTIPLESEL);
#endif //IC_WIN

#ifdef IC_MOTIF
   ppd->setSelection(CCS_EXTENDSEL);
#endif

   return *this;
}


/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: isSingleSelection
------------------------------------------------------------------------------*/
bool IContainerControl :: isSingleSelection() const
{
#ifdef IC_MOTIF
   return  ppd->isSingleSelection() ;
#endif
#ifdef IC_PMWIN
   return (style() & singleSelection.asUnsignedLong()) ? true : false;
#endif
}

/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: isMultipleSelection
------------------------------------------------------------------------------*/
bool IContainerControl :: isMultipleSelection() const
{
#ifdef IC_MOTIF
   return ppd->isMultipleSelection() ;
#endif
#ifdef IC_PMWIN
   return (style() & multipleSelection.asUnsignedLong()) ? true : false;
#endif
}

/*------------------------------------------------------------------------------
 Function Name: IContainerControl :: isExtendedSelection
------------------------------------------------------------------------------*/
bool IContainerControl :: isExtendedSelection() const
{
#ifdef IC_MOTIF
   return ppd->isExtendedSelection() ;
#endif
#ifdef IC_PMWIN
     return (style() & extendedSelection.asUnsignedLong()) ? true : false;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setContainerAttributes

 Implementation: Generic function to set container attributes.
------------------------------------------------------------------------------*/
void IContainerControl :: setContainerAttributes(unsigned long ulCnrAttrOff,
                                                 unsigned long ulCnrAttrOn)
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  cnrInfo.flWindowAttr &= ~(ulCnrAttrOff);
  cnrInfo.flWindowAttr |= ulCnrAttrOn;
  setContainerInfo(&cnrInfo, CMA_FLWINDOWATTR);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: containerAttributes

 Implementation: Generic function to query the container attributes.
------------------------------------------------------------------------------*/
unsigned long IContainerControl :: containerAttributes() const
{
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  return (cnrInfo.flWindowAttr);
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: enableCaching

 Implementation: Set record caching to the passed value.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: enableCaching(unsigned long ulDeltaValue)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  ICnrInfo cnrInfo;
  cnrInfo.cDelta = ulDeltaValue;
  setContainerInfo(&cnrInfo, CMA_DELTA);
  return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isCachingEnabled

 Implementation: Query Caching state.
------------------------------------------------------------------------------*/
bool IContainerControl :: isCachingEnabled() const
{
#ifdef IC_PMWIN
  ICnrInfo cnrInfo;
  containerInfo(&cnrInfo);
  if (cnrInfo.cDelta)
    return true;
  else
    return false;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return false;
#endif
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: enableDrawItem

 Implementation: Turn Owner draw on - OwnerDrawEvent dispatched.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: enableDrawItem(bool fEnable)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   if(fEnable)
     setContainerAttributes(0,CA_OWNERDRAW);
   else
     setContainerAttributes(CA_OWNERDRAW, 0);
   return *this;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  return *this;
#endif
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: isDrawItemEnabled

 Implementation: Query ownerdraw for items  state.
------------------------------------------------------------------------------*/
bool IContainerControl :: isDrawItemEnabled() const
{
#ifdef IC_PMWIN
  return ( (containerAttributes() & CA_OWNERDRAW) ? true : false);
#endif
#ifdef IC_MOTIF
  return false;
#endif
}

#ifdef IC_WIN
/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::setBackgroundColor

 Implementation:  Updates the background color of the container control.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::setBackgroundColor ( const IColor &color)
{
  setColor( PP_BACKGROUNDCOLOR, color );
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::setForegroundColor

 Implementation:  Updates the foreground color of the container control.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::setForegroundColor ( const IColor &color)
{
  setColor( PP_FOREGROUNDCOLOR, color );
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::setColor

 Implementation:  Updates the background or foreground color of the container
                  control.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::setColor ( unsigned long colorArea,
                                                 const IColor &color)
{
  if (colorArea == PP_FOREGROUNDCOLOR)
  {
    if (ppd->pmCompatible)
    {
      unsigned long ulResult =
         handle().sendEvent(CM_SETCNRCOLORS,
         IEventParameter1((unsigned long)CMA_FOREGROUNDCOLOR),
         IEventParameter2(NEARESTPALETTECOLOR((unsigned long) color.asRGBLong())));
      if (ulResult)
      {
        setLayoutDistorted(IWindow::colorChanged, 0);
      }
      else
      {
        ITHROWGUIERROR2("CM_SETCNRCOLORS",
                        IBaseErrorInfo::invalidParameter,
                        IException::recoverable);
      } /* endif */
    }
    else
    {
      // Native IContainerControl - Note this only changes the listview
      // control - Windows has no apis for changing the treeview colors
      if (ppd->pListview)
      {
        ListView_SetTextColor( ppd->pListview->handle(),
            NEARESTPALETTECOLOR((unsigned long)color.asRGBLong()) );
        InvalidateRect( ppd->pListview->handle(), 0, true );
      }
    }

  }
  else if ( colorArea == PP_BACKGROUNDCOLOR )
  {
    unsigned long bkColor =
             NEARESTPALETTECOLOR((unsigned long)color.asRGBLong());
    if (ppd->pmCompatible)
    {
      unsigned long ulResult = handle().sendEvent( CM_SETCNRCOLORS,
        IEventParameter1((unsigned long)CMA_BACKGROUNDCOLOR),
        IEventParameter2(bkColor));
      if (ulResult)
      {
        setLayoutDistorted(IWindow::colorChanged, 0);
      }
      else
      {
        ITHROWGUIERROR2("CM_SETCNRCOLORS",
                        IBaseErrorInfo::invalidParameter,
                        IException::recoverable);
      } /* endif */
    }
    else
    {
      // Native IContainerControl - Note this only changes the listview
      // control - Windows has no apis for changing the treeview colors
      if (ppd->pListview)
      {
        ListView_SetBkColor( ppd->pListview->handle(), bkColor );
        ListView_SetTextBkColor( ppd->pListview->handle(), bkColor );
        InvalidateRect( ppd->pListview->handle(), 0, true );
      }
    }
  }
  else
    Inherited::setColor( colorArea, color );
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::foregroundColor

 Implementation:  Returns the foreground color of the container control.
------------------------------------------------------------------------------*/
IColor IContainerControl::foregroundColor () const
{
  unsigned char ucRed, ucGreen, ucBlue;
  unsigned long ulResult;

  if (ppd->pmCompatible)
  {
    ulResult = handle().sendEvent( CM_QUERYCNRCOLORS,
                  IEventParameter1((unsigned long)CMA_FOREGROUNDCOLOR),
                  IEventParameter2((unsigned long) NULL));
  }
  else
  {
    // Native IContainerControl - Note this queries the listview
    // control - Windows has no color apis for the treeview control
    if (ppd->pListview)
      ulResult = ListView_GetTextColor( ppd->pListview->handle() );
    else
      // If only tree view exists, return default foreground color
      return (IGUIColor(IGUIColor::windowText));
  }

  ucBlue = (unsigned char)((ulResult & 0x00ff0000) >> 16);
  ucGreen = (unsigned char)((ulResult & 0x0000ff00) >> 8);
  ucRed = (unsigned char)(ulResult & 0x000000ff);
  return (IColor(ucRed, ucGreen, ucBlue));
}


/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::backgroundColor

 Implementation:  Returns the background color of the container control.
------------------------------------------------------------------------------*/
IColor IContainerControl::backgroundColor () const
{
  unsigned char ucRed, ucGreen, ucBlue;
  unsigned long ulResult;

  if (ppd->pmCompatible)
  {
    ulResult = handle().sendEvent( CM_QUERYCNRCOLORS,
                  IEventParameter1((unsigned long)CMA_BACKGROUNDCOLOR),
                  IEventParameter2((unsigned long) NULL));
  }
  else
  {
    // Native IContainerControl - Note this queries the listview
    // control - Windows has no color apis for the treeview control
    if (ppd->pListview)
      ulResult = ListView_GetBkColor( ppd->pListview->handle() );
    else
      // If only tree view exists, return default background color
      return (IColor(IColor::windowBgnd));
  }

  ucBlue = (unsigned char)((ulResult & 0x00ff0000) >> 16);
  ucGreen = (unsigned char)((ulResult & 0x0000ff00) >> 8);
  ucRed = (unsigned char)(ulResult & 0x000000ff);
  return (IColor(ucRed, ucGreen, ucBlue));
}


/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::resetForegroundColor

 Implementation:  Resets the foreground color of the container control.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::resetForegroundColor ( )
{
  IColor defaultColor(IColor::windowText);
  unsigned long defColor =
           NEARESTPALETTECOLOR((unsigned long)defaultColor.asRGBLong());
  if (ppd->pmCompatible)
  {
    unsigned long ulResult =
       handle().sendEvent(CM_SETCNRCOLORS,
       IEventParameter1((unsigned long)CMA_FOREGROUNDCOLOR),
       IEventParameter2(defColor));
  }
  else
  {
    // Native IContainerControl - Note this only changes the listview
    // control - Windows has no apis for changing the treeview colors
    if (ppd->pListview)
    {
      ListView_SetTextColor( ppd->pListview->handle(), defColor );
      InvalidateRect( ppd->pListview->handle(), 0, true );
    }
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name:  IContainerControl::resetBackgroundColor

 Implementation:  Resets the background color of the container control.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl::resetBackgroundColor ( )
{
  IColor defaultColor(IColor::windowBgnd);
  unsigned long defColor =
           NEARESTPALETTECOLOR((unsigned long)defaultColor.asRGBLong());
  if (ppd->pmCompatible)
  {
    unsigned long ulResult =
       handle().sendEvent(CM_SETCNRCOLORS,
       IEventParameter1((unsigned long)CMA_BACKGROUNDCOLOR),
       IEventParameter2(defColor));
  }
  else
  {
    // Native IContainerControl - Note this only changes the listview
    // control - Windows has no apis for changing the treeview colors
    if (ppd->pListview)
    {
      ListView_SetBkColor( ppd->pListview->handle(), defColor );
      ListView_SetTextBkColor( ppd->pListview->handle(), defColor );
      InvalidateRect( ppd->pListview->handle(), 0, true );
    }
  }
  return *this;
}
#endif //IC_WIN

