// Revision: 81 1.13.2.1 source/ui/cnr/icnrctl0.cpp, container, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: icnrctl0.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the IContainerControl function    *
*   declared in icnrctl.hpp.                                                   *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

/*--------------------------------------------------------------*/
/* declare the include file switches and the include files.     */
/*--------------------------------------------------------------*/
#define INCL_WINSTDCNR
#define INCL_GPILCIDS
#define INCL_DOSMODULEMGR

extern "C" {
  #include <stdio.h>
  #include <string.h>
  }

#include <icnrrec.hpp>  // Must be first for OS flags

#ifdef IC_MOTIF
#include <cnr.h>
#include <ifont.hpp>
#endif
#include <irect.hpp>
#include <itrace.hpp>
#include <istring.hpp>
#include <iguilock.hpp>
#include <ireslock.hpp>
#include <iexcept.hpp>

#ifdef IC_PMWIN
  #include <imphdr.hpp>
#endif

#include <ithread.hpp>
#include <ireslib.hpp>
#include <icnrobj.hpp>
#include <icnrcol.hpp>
#include <icnrctl.hpp>
#include <ihandler.hpp>
#include <icnrclst.hpp>
#include <icnrstat.hpp>
#include <icnrfilt.hpp>
#include <inotifev.hpp>
#include <iinhratt.hpp>

#include <icnrobjp.hpp>

#ifdef IC_PMWIN
#include <icnrctlw.hpp>
#endif

#ifdef IC_WIN
#include <icctlsta.hpp>         // statics to load dll
#include <commctrl.h>
#endif

#ifdef IC_MOTIF
  // have to be able to new and delete an ICnrControlData object
  // and an ICnrObjPrivateData object
  #include <icnrctlm.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCTL0_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
  Imbedded classes: declarations and implementations
------------------------------------------------------------------------------*/
ICnrControlList* ICnrControlStaticPtr::pCnrSet =0;


ICnrControlStaticPtr::~ICnrControlStaticPtr ()
{
  IMODTRACE_DEVELOP("ICnrControlStaticPtr::~ICnrControlStaticPtr");
#if IC_STATIC_PRIORITY_SUPPORTED
  // It is safe to do deletes only when static init ordering can be controlled.
  if(pCnrSet)  delete pCnrSet;
  if(pCnrKey)  delete pCnrKey;
  if(pHandler) delete pHandler;
#endif
}

ICnrInfo ::   ICnrInfo()
      : cb(sizeof(ICnrInfo)),
        pSortRecord(0), pFieldInfoLast(0), pFieldInfoObject(0),
#ifdef IC_PMWIN
        strCnrTitle(0),
#endif
#ifdef IC_MOTIF
        strCnrTitle(" "),
#endif
        flWindowAttr(0), ptlOrigin(0,0),
        cDelta(0), cRecords(0),
        slBitmapOrIcon(0,0), slTreeBitmapOrIcon(0,0),
        hbmExpanded(0), hbmCollapsed(0), hptrExpanded(0), hptrCollapsed(0),
        cyLineSpacing(0), cxTreeIndent(0), cxTreeLine(0),
        cFields(0), xVertSplitbar(0)
{
}
ICnrInfo :: ~ICnrInfo() {}
IMiniCnrRecord :: IMiniCnrRecord() :
#ifdef IC_NOBINARYCOMP
        cRecordText(0), pRecordIcon(0),
#endif
        strIcon(), hptrIcon(0) {}
IMiniCnrRecord :: ~IMiniCnrRecord() {}

#ifdef IC_PM
bool _System pfnDeleteObjects(ICnrRecord* pRecord, void* p);
bool _System pfnCleanUpObjects(ICnrRecord* pRecord, void* thisCnr);
#endif
#ifdef IC_WIN
bool CALLBACK pfnDeleteObjects(ICnrRecord* pRecord, void* p);
bool CALLBACK pfnCleanUpObjects(ICnrRecord* pRecord, void* thisCnr);
#endif

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: initialize

 Implementation: static function for allocation container
------------------------------------------------------------------------------*/
void IContainerControl :: initialize()
{
   IMODTRACE_DEVELOP("ICnrCtl::initialize");

  /*---------------------------------------------------*/
  /* Ensure PM is initialized and the container set    */
  /* is constructed.                                   */
  /*---------------------------------------------------*/
   IThread::current().initializeGUI();
#ifdef IC_MOTIF
   IContainerControl::setAllocationContainer(0xFFFFFFFF);

#endif
   {
    IGUIResourceLock collectionLock(containerKey());
    containerList();
   }

#ifdef IC_WIN
  // If needed, load the DLL containing the control code.
  IControlStatics::loadControlDLL();
#endif


  /*---------------------------------------------------*/
  /* Create the invisible container allocation window  */
  /*---------------------------------------------------*/
#ifdef IC_PMWIN
  unsigned long flAllocateStyle = CCS_MINIRECORDCORE;
#ifdef IC_WIN

#ifdef IC_WU
  // Wind/U maps the desktop window handle to NULL.  But within
  //  Wind/U, the XtCreateWidget() requires a non-NULL parent.
  //  We will use the objectWindow as parent.
  HWND hwndObject = IWindow::objectWindow()->handle();
#else
  HWND hwndObject = GetDesktopWindow();
#endif // IC_WU

  HINSTANCE hInst = GetModuleHandle(0);
  // Defect: 28824:
  // Container will not paint over siblings due to OS2 limiation.
  // NA for windows so remove limitation.
  // flAllocateStyle |= CNR_REQUIREDSTYLES;
  flAllocateStyle |= WS_CHILD | WS_CLIPCHILDREN;
#else
#ifndef HWND_OBJECT
  #define HWND_OBJECT 0
#endif
  HWND hwndObject = HWND_OBJECT;
  ULONG  hInst = 0;
#endif //IC_WIN

  unsigned long hwndAllocate =
#ifdef IC_WIN
                               (unsigned long)(void*)
#endif
              ICREATEWINDOW( hwndObject,
                             (PSZ)WC_CONTAINER,
                             0,
                             flAllocateStyle,
                             0,0,0,0,
                             0,
                             HWND_TOP,
                             0xffff,
                             0,
                             0,
                             hInst );

 if(hwndAllocate == 0)
 {
     ITHROWGUIERROR("CreateWindow(WC_CONTAINER-Allocate)");
 }
 else
    IContainerControl::setAllocationContainer(hwndAllocate);
#endif //IC_PMWIN
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: IContainerControl

 Implementation: Construct a container control
------------------------------------------------------------------------------*/
IContainerControl :: IContainerControl(unsigned long ulId,
                          IWindow* pwndParent,
                          IWindow* pwndOwner,
                          const IRectangle& rectInit,
                          const Style& style,
                          const Attribute& attribute) :
                                 ulClObjectChanges(0),
                                 ulClColumnChanges(0),
                                 pmleClEdit(0),
                                 pcnrcolClEdit(0),
                                 pcnrobjClEdit(0),
                                 pcnrobjClSingleSelectionNotification(0)
#ifdef IC_PMWIN
                                 , pcnrcolClSplitbar(0),
                                 strContainerTitle(" ")
#endif
{
  IMODTRACE_DEVELOP("ICnrCtl::ICnrctl(parent)");

  // Save the extended style to make sure we have a copy of it stored
#ifdef IC_PMWIN
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );
#endif

  /*---------------------------------------------------*/
  /* Ensure:                                           */
  /*     The allocation container is built             */
  /*     The default style is mini-records             */
  /*---------------------------------------------------*/
   if(IContainerControl::hwndAllocation() == 0)
       initialize();

#ifdef IC_MOTIF
   pcnrinfoCl = new ICnrInfo;
   unsigned long ulStyle = style.asUnsignedLong();
   ulStyle |= CCS_MINIRECORDCORE;
#endif
#ifdef IC_PMWIN
   unsigned long ulStyle = convertToGUIStyle( style );
   ulStyle |= CCS_MINIRECORDCORE | CCS_MINIICONS;
#endif

   /* Ensure only a single selection style */
   if(ulStyle & CCS_SINGLESEL)
     ulStyle &= ~(CCS_EXTENDSEL | CCS_MULTIPLESEL);
   else if(ulStyle & CCS_EXTENDSEL)
     ulStyle &= ~(CCS_SINGLESEL | CCS_MULTIPLESEL);
   else if(ulStyle & CCS_MULTIPLESEL)
     ulStyle &= ~(CCS_SINGLESEL | CCS_EXTENDSEL);
   else  /* No Selection styles set, then default to Single */
     ulStyle |= CCS_SINGLESEL;

#ifdef IC_DEVELOP
   ulStyle |= CCS_VERIFYPOINTERS;
#endif

#ifdef IC_PMWIN
   HWND ownerHandle = 0;
   if(pwndOwner)
      ownerHandle = pwndOwner->handle();


  IWindowHandle whContainer;

#ifdef IC_WIN
  bool pmCompatible(extendedStyle() &
                      (IContainerControl::pmCompatible.asExtendedUnsignedLong()));

  if (!pmCompatible)
  {
    ICnrControlData::registerContainer();

    whContainer =this -> create( ulId,
                                 0,
                                 ulStyle | WS_CLIPCHILDREN,
                                 WC_NATIVECONTAINER,
                                 pwndParent->handle(),
                                 ownerHandle,
                                 rectInit,
                                 0,
                                 0,
                                 defaultOrdering(),
                                 0 );

    startHandlingEventsFor(whContainer);
    INativeContainerHandler::defaultHandler()->handleEventsFor(this);

    ppd = new ICnrControlData( this, ulStyle, attribute );

    if (attribute & IContainerControl::treeView )
    {
      ppd->setActiveView( ICnrControlData::tree );
    }
    else
    {
      ppd->setActiveView( ICnrControlData::list );
    }

    IWindowHandle wh(ppd->pActiveview->handle());
  }
  else
#endif //IC_WIN
  {
    ppd = new ICnrControlData;

    /*---------------------------------------------------*/
    /* Create the container                              */
    /*---------------------------------------------------*/
    whContainer =
        this -> create( ulId,
                        0,
                        ulStyle,
                        WC_CONTAINER,
                        pwndParent->handle(),
                        ownerHandle,
                        rectInit,
                        0,
                        0,
                        defaultOrdering(),
                        convertToGUIStyle( style, true ) );
    startHandlingEventsFor(whContainer);
  }
  /*---------------------------------------------------*/
  /* Add the Destroy Handler and start the message     */
  /* processing.                                       */
  /*---------------------------------------------------*/
  addDefaultHandler();
  IMousePointerHandler::defaultHandler()->handleEventsFor(this);

#endif //IC_PMWIN

#ifdef IC_MOTIF
  ppd = new ICnrControlData(this);
  ppd->createContainer(ulId, pwndParent, pwndOwner, rectInit);
  // Now set whatever styles were passed
  // Not sure yet what to do with autoPosition
  if (ulStyle & CCS_AUTOPOSITION)                    /* LJO */
     ppd->ulStyle |= CCS_AUTOPOSITION;               /* LJO */
  if(ulStyle & CCS_SINGLESEL)
    setSingleSelection();
  else if(ulStyle & CCS_EXTENDSEL)
    setExtendedSelection();
  else
    setMultipleSelection();

  if(ulStyle & CCS_READONLY)
     ppd->ulStyle |= CCS_READONLY;

  Inherited::setStyle(ulStyle);

  //Setup the callbacks for the container here.
  ppd->addMotifCallbacks() ;
#endif // IC_MOTIF

  /*---------------------------------------------------*/
  /* Add the container to our list and setup the       */
  /* defaults.                                         */
  /*---------------------------------------------------*/
   addContainer();
   setDefaultContainer(attribute);
   showDetailsViewTitles(true);
   flClState = ok;
   return;
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window.  Container does
  // not inherit foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kBackgroundColor );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif
 }

#ifdef IC_PMWIN
/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: IContainerControl

 Implementation: Construct a container control on a Dialog
------------------------------------------------------------------------------*/
IContainerControl :: IContainerControl(unsigned long id,
                                       IWindow* parentDialog)
                          :      ulClObjectChanges(0),
                                 ulClColumnChanges(0),
                                 pmleClEdit(0),
                                 pcnrcolClEdit(0),
                                 pcnrobjClEdit(0),
                                 pcnrobjClSingleSelectionNotification(0),
                                 pcnrcolClSplitbar(0),
                                 strContainerTitle(" ")
{
  IMODTRACE_DEVELOP("ICnrCtl::ICnrctl(dialog)");
  ppd = new ICnrControlData;

  /*---------------------------------------------------*/
  /* Ensure the allocation container is built.         */
  /*---------------------------------------------------*/
   if(IContainerControl::hwndAllocation() == 0)
       initialize();

  /*---------------------------------------------------*/
  /* Add the Destroy Handler and start the message     */
  /* processing.                                       */
  /*---------------------------------------------------*/
   addDefaultHandler();
   IMousePointerHandler::defaultHandler()->handleEventsFor(this);
   setAutoDestroyWindow(false);
   reserveUserWindowWord( false );
   startHandlingEventsFor(id, parentDialog);

  /*---------------------------------------------------*/
  /* Add the container to our list and setup the       */
  /* defaults.                                         */
  /*---------------------------------------------------*/
   addContainer();
   setDefaultContainer(defaultAttribute());

   flClState = ok;
   return;
#ifdef IC_WIN
  // add the inheritColor attribute to the window.  Container does
  // not inherit foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kBackgroundColor );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif // IC_WIN
 }


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: IContainerControl

 Implementation: Construct from an existing IWindowHandle
------------------------------------------------------------------------------*/
IContainerControl :: IContainerControl(const IWindowHandle& handle)
                             :   ulClObjectChanges(0),
                                 ulClColumnChanges(0),
                                 pmleClEdit(0),
                                 pcnrcolClEdit(0),
                                 pcnrobjClEdit(0),
                                 pcnrobjClSingleSelectionNotification(0)
#ifdef IC_PMWIN
                                 , pcnrcolClSplitbar(0),
                                 strContainerTitle(" ")
#endif
{
   IMODTRACE_DEVELOP("ICnrCtl::ICnrctl(handle)");
#ifdef IC_PMWIN
   ppd = new ICnrControlData;
#endif

  /*---------------------------------------------------*/
  /* Ensure the allocation container is built.         */
  /*---------------------------------------------------*/
   if(IContainerControl::hwndAllocation() == 0 )
       initialize();

#ifdef IC_MOTIF
   pcnrinfoCl = new ICnrInfo;
   ppd = new ICnrControlData(this);
   ppd->createContainer(handle);

   // Add the callbacks
   ppd->addMotifCallbacks() ;
#endif // IC_MOTIF

#ifdef IC_PMWIN
  /*---------------------------------------------------*/
  /* Add the Destroy Handler and start the message     */
  /* processing.                                       */
  /*---------------------------------------------------*/
   addDefaultHandler();
   IMousePointerHandler::defaultHandler()->handleEventsFor(this);
   setAutoDestroyWindow(false);
   reserveUserWindowWord( false );
   startHandlingEventsFor(handle);
#endif

  /*---------------------------------------------------*/
  /* Add the container to our list and setup the       */
  /* defaults.                                         */
  /*---------------------------------------------------*/
   addContainer();
   setDefaultContainer(defaultAttribute());

   flClState = ok;
#ifdef IC_WIN
  // add the inheritColor attribute to the window.  Container does
  // not inherit foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kBackgroundColor );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif
   return;
 }
#endif //IC_PMWIN


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: setDefaultContainer

 Implementation: Initial Container setup.
------------------------------------------------------------------------------*/
void IContainerControl :: setDefaultContainer(const Attribute& attribute)
{
  IMODTRACE_DEVELOP("ICnrCtl::setDefaultCnr");
  ICnrInfo cnrInfo;

  /*  No splibar on Details view  */
  cnrInfo.xVertSplitbar = -1;
  lClSplitbar = -1;

  /* Line Spacing default  */
  cnrInfo.cyLineSpacing = 2;
#ifdef IC_MOTIF
  cnrInfo.strCnrTitle = "  ";
#endif
  cnrInfo.flWindowAttr = attribute.asUnsignedLong();

#ifdef IC_MOTIF
   if (cnrInfo.flWindowAttr & CA_CONTAINERTITLE) {
      ppd->containerManagement(ICnrControlData::SHOW_TITLE);
   }

   if (cnrInfo.flWindowAttr & CA_TITLESEPARATOR) {
      ppd->containerManagement(ICnrControlData::SHOW_SEPARATOR);
   }
#endif

  ulClDragEvent = CN_DRAGOVER;
  hbmClIconViewBackground = 0;
  long lFlags = CMA_FLWINDOWATTR | CMA_LINESPACING | CMA_XVERTSPLITBAR;

#ifdef IC_MOTIF
  lFlags |= CMA_CNRTITLE;
  if (cnrInfo.ptlOrigin.x != 0 && cnrInfo.ptlOrigin.y != 0)
     lFlags |= CMA_PTLORIGIN;
#endif

  setContainerInfo(&cnrInfo, lFlags);
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: addContainer()

 Implementation: Add container to static list.
------------------------------------------------------------------------------*/
void IContainerControl :: addContainer()
{
   IMODTRACE_DEVELOP("ICnrCtl::addCnr");
   IGUIResourceLock collectionLock(containerKey());

   /* Add this container to the list & increment count */
   containerList().add(this);
}



/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: ~IContainerControl

 Implementation: Destroy a container control
------------------------------------------------------------------------------*/
IContainerControl :: ~IContainerControl()
{
  IMODTRACE_DEVELOP("ICnrCtl::~ICnrCtl");

#ifdef IC_PMWIN
   IMousePointerHandler::defaultHandler()->stopHandlingEventsFor(this);
#endif // IC_PMWIN

#ifdef IC_WIN
   if (!ppd->pmCompatible)
   {
     INativeContainerHandler::defaultHandler()->stopHandlingEventsFor(this);
   }
#endif // IC_WIN

#ifdef IC_MOTIF
   // Remove the callbacks
   ppd->removeMotifCallbacks() ;
   delete pcnrinfoCl;
#endif // IC_MOTIF

  cleanUp();
  delete ppd;
}

#ifdef IC_PMWIN
/*-----------------------------------------------------------------------------
 Function Name: pfnCleanUpObjects

 Implementation: This function is called by the container for each object
                 in it.  It decrements the use count of the object then
                 checks to see if the object is not being used in any other
                 containers.  If not, the delayFree flag is set and the object
                 is freed.  The delayFree flag is set so the object dtor
                 will not attempt to free the container record.  Since this
                 function is called from IContainerControl::cleanUp(), we
                 know that the container is in the process of being destroyed.
                 The PM container will remove and free all records remaining
                 within it when it is destroyed.
------------------------------------------------------------------------------*/
#ifdef IC_PM
bool _System pfnCleanUpObjects(ICnrRecord* pRecord, void* thisCnr)
#endif
#ifdef IC_WIN
bool CALLBACK pfnCleanUpObjects(ICnrRecord* pRecord, void* thisCnr)
#endif
{
  IContainerObject* pcnrobj = IObjFromRec((IMiniCnrRecord*)(void*)pRecord);
  pcnrobj->decrementUseCount();
  ((IContainerControl*)thisCnr)->ulClObjectChanges++;
  if (pcnrobj->ulClUseCount == 0)
  {
    pcnrobj->setDelayedFree();
    delete(pcnrobj);
  }
  return true;
}
#endif //IC_PMWIN

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: cleanUp

 Implementation: Remove/Delete Objects and Columns
------------------------------------------------------------------------------*/
void IContainerControl::cleanUp()
{
  IFUNCTRACE_DEVELOP();

#ifdef IC_PMWIN
  /* If the state isn't "ok", the window is already gone */
  if(flClState&ok)
#endif
#ifdef IC_MOTIF
  // Under Motif we usually get here AFTER the Widget has been destroyed
  // and the IWindowHandle is invalid.
  // This is due to problems with how IWindow handles destruction for
  // IWindow objects that are composites (ie. have a topHandle and related
  // handles). The isValid() check will prevent throwing exceptions for
  // invalid handles, but can cause memory leaks when the Container is in
  // a secondary window that is being destroyed.
  // An explicit call to the IContainerObject destructor will execute
  // this routine and free up storage.
  if(flClState&ok && isValid())
#endif
  {
    flClState &= ~IContainerControl::ok;
    hide();
    setRefreshOff();

    /* Close the MLE(if open) so we avoid trap when closing frame. */
    if (currentEditMLE())
    {
#ifdef IC_PMWIN
      closeEdit();
#endif
#ifdef IC_MOTIF
      ppd->deleteMle();
#endif
    }

    IContainerObject* pcnrobj;
    if (willDeleteObjectsOnClose())
    {
#ifdef IC_PMWIN
      if (ppd->pmCompatible)
      {
        /***************************************************************/
        /* Call a filter function which actually deletes the container */
        /* objects.  This was done since it is a more efficient way to */
        /* delete all the objects when closing a container.            */
        /***************************************************************/
        sendEvent(CM_FILTER, MPFROMP(&pfnCleanUpObjects), MPFROMP(this));
      }
      else
      {
        /***************************************************************/
        /* For Native Windows, we must process the object deletion.    */
        /* Above we rely on the CCL container to automatically process */
        /* the freeing of objects when it is destroyed.                */
        /* See pfnCleanUpObjects for further info.                     */
        /***************************************************************/
        deleteAllObjects();
      }
#endif //IC_PMWIN
#ifdef IC_MOTIF
      //deleteAllObjects() ;
      IContainerControl::ObjectCursor cursor(*this);
      // We get all objects in the container by continually setting
      // the cursor to last after removing an object (which invalidates
      // the cursor). This ensures that we remove children objects
      // before we remove the parents (if we did the converse we would
      // not get all the objects because WC_CONTAINER would remove
      // them for us.
      for (cursor.setToLast(); cursor.isValid(); cursor.setToLast())
      {
        pcnrobj = objectAt(cursor);
        removeObject(pcnrobj);
        if (pcnrobj->ulClUseCount == 0)
           delete pcnrobj;
      }
#endif
    }
    else
    {
      /* remove all the objects  */
      removeRecords(0, 0, 0);
    }

    if(willDeleteColumnsOnClose())
    {
      IContainerControl::ColumnCursor colCursor(*this);
      IContainerColumn* pcnrcol;
      for (colCursor.setToFirst(); colCursor.isValid(); colCursor.setToFirst() )
      {
         pcnrcol = columnAt(colCursor);
         removeColumnAt(colCursor);
         delete pcnrcol;
      }
#ifdef IC_WIN
      if (!ppd->pmCompatible)
        ppd->columnsToDelete->deleteAll();
#endif // IC_WIN
    }
    else
    {
       /* remove all the columns  */
       sendEvent(CM_REMOVEDETAILFIELDINFO);
    }
#ifdef IC_PMWIN
    /* remove this container from the list */
    IContainerControl* pcnrctl;
    {
      IGUIResourceLock collectionLock(containerKey());
      ICnrControlList::Cursor cursor(containerList());
      for (cursor.setToFirst(); cursor.isValid() ; cursor.setToNext() )
      {
        pcnrctl = containerList().elementAt(cursor);
        if(pcnrctl->isValid() && pcnrctl->handle() == handle())
        {
            containerList().removeAt(cursor);
            break;
        }
      } /* end for  */
    }
#endif //IC_PMWIN
  }
#ifdef IC_MOTIF
  /* remove this container from the list */
  // For Motif we want to do this even if the handle was no longer
  // valid. Need to get the IContainerControl* out of the list to
  // prevent it being accessed once it's no longer valid.
  // That's why we use the (pcnrctl == this) under Motif
  IContainerControl* pcnrctl;
  {
    IGUIResourceLock collectionLock(containerKey());
    ICnrControlList::Cursor cursor(containerList());
    for (cursor.setToFirst(); cursor.isValid() ; cursor.setToNext() )
    {
      pcnrctl = containerList().elementAt(cursor);
      if(pcnrctl == this)
      {
        containerList().removeAt(cursor);
        break;
      }
    } /* end for  */
  }
#endif //IC_MOTIF
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: containerList()

 Implementation: Return a reference to the list of containers.
------------------------------------------------------------------------------*/
ICnrControlList& IContainerControl ::containerList()
{
   IGUIResourceLock collectionLock(containerKey());
   if(ICnrControlStaticPtr::pCnrSet == 0)
   {
      /* Start a list of Container Controls */
      ICnrControlStaticPtr::pCnrSet = new ICnrControlList();
   }

   return *ICnrControlStaticPtr::pCnrSet;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: refreshAllContainers

 Implementation: static function to refresh all containers.  If immediate
 is set to true, painting will occur synchronously, otherwise it will be
 async.
 (for Motif, immediate is ignored)
------------------------------------------------------------------------------*/
void IContainerControl :: refreshAllContainers(bool immediate)
{
   IMODTRACE_DEVELOP("ICnrCtl::refreshAllCnr");
   IContainerControl* pcnrctl;

   IGUIResourceLock collectionLock(containerKey());

   ICnrControlList::Cursor cursor(containerList());
   for (cursor.setToFirst(); cursor.isValid() ; cursor.setToNext() )
   {
      pcnrctl = containerList().elementAt(cursor);
#ifdef IC_PMWIN
      pcnrctl->refresh(immediate);
#endif
#ifdef IC_MOTIF
      pcnrctl->sendEvent(CM_INVALIDATERECORD,0ul,
              MPFROM2SHORT(0,CMA_REPOSITION | CMA_ERASE));
#endif
   }

}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: containerFromHandle

 Implementation: retrieve a container from its handle.
------------------------------------------------------------------------------*/
IContainerControl* IContainerControl::containerFromHandle(const IWindowHandle& hwnd)
{
  IGUIResourceLock collectionLock(containerKey());

  IContainerControl* pcnrctl;
  ICnrControlList::Cursor cursor(containerList());

  for (cursor.setToFirst(); cursor.isValid() ; cursor.setToNext() )
  {
     pcnrctl = containerList().elementAt(cursor);
     if(pcnrctl->handle() == hwnd)
     {
         return pcnrctl;
     }
  } /* end for  */
  return 0;
}

#ifdef IC_PMWIN
/*-----------------------------------------------------------------------------
 Function Name: pfnDeleteObjects

 Implementation: This function is called by the container for each object
                 in it.  This function will set the delayFree flag for each
                 object, then free the object.  The delayFree flag is set
                 so the object destructor will not call the container to
                 remove and free the container record.  We do that after we
                 have deleted all the objects for performance.
------------------------------------------------------------------------------*/
#ifdef IC_PM
bool _System pfnDeleteObjects(ICnrRecord* pRecord, void* p)
#endif
#ifdef IC_WIN
bool CALLBACK pfnDeleteObjects(ICnrRecord* pRecord, void* p)
#endif
{
  IContainerObject* pcnrobj = IObjFromRec((IMiniCnrRecord*)(void*)pRecord);
  pcnrobj->setDelayedFree();
  delete(pcnrobj);
  return true;
}
#endif //IC_PMWIN

/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: deleteAllObjects

 Implementation: Delete all objects in the container.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: deleteAllObjects()
{
  IMODTRACE_DEVELOP("CnrCtl::delAllObj");

  // The following code is added to ensure that only one selection notification
  // occurs using a single selection container.  Because the container
  // typically selects another object when one is deleted, selection notification
  // may occur for many objects in the container during this function, instead
  // of once for the currently selected object. Here we set the selected object
  // into pcnrobjClSingleSelectionNotification to identify the object needing
  // notification to the Notification handler.
  if(isSingleSelection() )
  {
    pcnrobjClSingleSelectionNotification = cursoredObject();
  }

#ifdef IC_PMWIN
  bool noSharedObjs = (extendedStyle() &
                 (IContainerControl::noSharedObjects.asExtendedUnsignedLong()));


  /*************************************************************************/
  /*   No (noSharedObjects) optimization for Native Windows.               */
  /*                                                                       */
  /*   (noSharedObjects) does not free memory during the deletetion        */
  /* of an object.  It assumes nothing is shared and relies on CCL         */
  /* to free all of the memory at once by calling CM_REMOVERECORD.         */
  /*                                                                       */
  /*   We must call CCL to free the memory but no objects are inserted     */
  /* into the CCL container that we use for allocation, thus               */
  /* we cant tell the CCL container to just free all the memory. Also,     */
  /* we cant destroy the CCL container, which would free all the memory    */
  /* because this could potentially free memory for objects that arent     */
  /* inserted anywhere or are inserted in another container.               */
  /* The main problem is the catch22 when trying to delete all             */
  /* the objects in the INativeContainerHandler. You cant                  */
  /* use the existing cursor, and call delete(cnrObj) because              */
  /* the cursor will be invalidated when the object is deleted.            */
  /* You cant get next, without having a valid current.  Another option    */
  /* is to use a cursor that always goes to last, but this is extremely    */
  /* inefficient, and would be worse than the default.                     */
  /*                                                                       */
  /*   We can investigate replicating the CM_FILTER callback process       */
  /* by using the TVN_DELETEITEM notification to delete objects after      */
  /* the TreeView_DeleteAllItems message is called.                        */
  /*************************************************************************/
  if (!ppd->pmCompatible)
    noSharedObjs = false;

  /****************************************************************/
  /* Check to see if this container is sharing objects with other */
  /* containers.  If it is not, send the CM_FILTER message to the */
  /* container passing in a function to be called on each         */
  /* container record.  This is done because it is by far the     */
  /* fastest way to iterate all records in a container.  The      */
  /* function passed in will be called for each container record. */
  /*                                                              */
  /* When this is done, call the container to remove and free all */
  /* container records in it.  This one call is much faster than  */
  /* removing and freeing container records individually.         */
  /****************************************************************/
  if (noSharedObjs)
  {
    // Unfilter objects to ensure not all objects filtered.
    // This will prevent the cursor changed notification trap.
    bool  refreshOn;
    if (refreshOn = isRefreshOn())
      setRefreshOff();
    sendEvent(CM_FILTER, MPFROMP(0), MPFROMP(0));

    sendEvent(CM_FILTER, MPFROMP(&pfnDeleteObjects), MPFROMP(0));
    /****************************************************************/
    /* Now remove and free all the container records with one call. */
    /****************************************************************/
    unsigned short cmd = CMA_FREE;
    if (this->isVisible() && refreshOn)
    {
      cmd |= CMA_INVALIDATE;
    }
    removeRecords(0, 0, cmd);

    if( refreshOn )
    {
      refresh();
      setRefreshOn();
    }
  }
  else
  {
    IContainerControl::ObjectCursor iter(*this);
    IContainerObject* pcnrobj = iter.first();
    while (pcnrobj != 0)
    {
      /* Make sure the object destructor does not update the container. */
      pcnrobj->flClState |= IContainerObject::nestedDelete;
      /* Delete this object */
      delete pcnrobj;
      pcnrobj = iter.first();
    }
  }
#endif //IC_PMWIN
#ifdef IC_MOTIF
  IContainerControl::ObjectCursor iter(*this);
  IContainerObject* pcnrobj = iter.first();

  while (pcnrobj!=0)
  {
    /* Make sure the object destructor does not update the container. */
    pcnrobj->flClState |= IContainerObject::nestedDelete;

    /* Delete this object */
    delete pcnrobj;

    pcnrobj = iter.first();
  }

  // For Motif, hidden objects are not seen by ObjectCursor.
  // Use private widget collection of hidden or filtered objects
  // to completely remove all objects.

  /*
  Widget     curIconG;
  ICnrIGCursor cursor(this);

  cursor.setToFirst();
  while(cursor.isValid())
  {
    curIconG = cursor.current();
    XtVaGetValues(curIconG, XmNuserData, &pcnrobj, NULL);
    delete pcnrobj;
    cursor.setToFirst();
  }
  */
#endif //IC_MOTIF
#ifdef IC_PMWIN
  if (!noSharedObjs)
#endif
  {
    // Only refresh the containers that need to be updated.
    IContainerControl* pcnrctl;

    IGUIResourceLock collectionLock(containerKey());
    ICnrControlList cnrctllist = IContainerControl :: containerList();
    ICnrControlList::Cursor cnrctlcur(cnrctllist);
    for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
    {
      pcnrctl = cnrctllist.elementAt(cnrctlcur);
      if(pcnrctl->isVisible() && pcnrctl->isRefreshOn() &&
         (pcnrctl->flClState & IContainerControl::needsRefresh))
      {
        pcnrctl->flClState &= ~IContainerControl::needsRefresh;
        pcnrctl->refresh();
      }
    }
  }

  pcnrobjClSingleSelectionNotification = 0;

  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: removeObject

 Implementation: Remove an object and all its descendents from a container,
                 or optionally from all containers.
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: removeObject(IContainerObject* newObject,
                                                     bool fAllContainers)
{
  IMODTRACE_DEVELOP("CnrCtl::removeObj");
  IASSERTPARM(newObject!=0);
  unsigned long ulFlag = 0;
  if(isRefreshOn() && newObject->isRefreshOn())
    ulFlag = CMA_INVALIDATE;

  IGUIResourceLock collectionLock(containerKey());
  // If requested to do so, remove the specified object from all containers
  if( fAllContainers != false &&                    // if asked, and ...
      containerList().numberOfElements() > 1 )      // we have >1 container
  {
    IContainerControl* pcnrctl;
    ICnrControlList::Cursor ctlcursor(containerList());

    // Iterate through all containers in our list, removing this object
    for(ctlcursor.setToFirst(); ctlcursor.isValid(); ctlcursor.setToNext())
    {
      pcnrctl = containerList().elementAt(ctlcursor);
      pcnrctl->removeObject(newObject);
    } /* endfor */
  } /* endif */
  // Otherwise, only remove the specified object from this container
  else
  {
    if(newObject->flClState & IContainerObject::hasDescendents)
    {
      /* Since removing an object from the container removes */
      /* descendents, we will ensure the use count of        */
      /* descendents gets decremented.                       */
      IContainerControl::ObjectCursor   objcursor(*this, newObject, true);

      for(objcursor.setToFirst(); objcursor.isValid(); objcursor.setToNext())
      {
        ((IContainerObject*)(objectAt(objcursor)))->decrementUseCount();
      } /* endfor */
    }

    //Call OS/2 to remove the container record.
    void* pbase= IRecFromObj(newObject);

    if(!containsObject(newObject))
       return *this;

    long lCount = removeRecords(pbase, 1, ulFlag);

    //Only decrement if container record was actually in this container.
    if (lCount != -1)
    {
      newObject->decrementUseCount();
      ulClObjectChanges++;
    }
  } /* endelse */
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerControl :: deleteSelectedObjects

 Implementation: Delete selected objects from the container.

 Notes: must be updated or over ridden for ICnrRecords - use cursors instead;
------------------------------------------------------------------------------*/
IContainerControl& IContainerControl :: deleteSelectedObjects()
{
  IMODTRACE_DEVELOP("CnrCtl::delSelObj");
  IMiniCnrRecord* pcnrrec;
  IContainerObject* pcnrobj;

  /*******************************************************************/
  /* Continually query the first selected object and then delete it  */
  /* before the suceeding query.                                     */
  /*******************************************************************/
  do
  {
    pcnrrec = (IMiniCnrRecord*)(void*)sendEvent(CM_QUERYRECORDEMPHASIS,
                                         (IContainerObject*)CMA_FIRST,
                                         MPFROMLONG(CRA_SELECTED));
    if((long)pcnrrec == -1L)
    {
      ITHROWGUIERROR("CM_QUERYRECORDEMPHASIS");
    }
    if(pcnrrec)
    {
      pcnrobj = IObjFromRec(pcnrrec);
      pcnrobj->setRefreshOff();
      delete pcnrobj;
    }

    /*******************************************************************/
    /* Need to check treeView because the container's tree view is     */
    /* always forced to single selection even if the container is not  */
    /* in single selection mode.                                       */
    /* In either case, quit after the first delete, because the cnr    */
    /* will assign a new cursored/selected object.  If we continue, we */
    /* will delete all objects in the container.                       */
    /*******************************************************************/
#ifdef IC_PMWIN
    if( isSingleSelection() ||  isTreeView() )
#endif
#ifdef IC_MOTIF
    if(isSingleSelection())
#endif
      break;

  } while (pcnrrec != 0);

  /************************************************************************/
  /* Refresh the containers that need to be updated from the above changes*/
  /************************************************************************/
  IGUIResourceLock collectionLock(containerKey());

  ICnrControlList cnrctllist = IContainerControl :: containerList();
  ICnrControlList::Cursor cnrctlcur(cnrctllist);
  IContainerControl* pcnrctl;
  for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
  {
    pcnrctl = cnrctllist.elementAt(cnrctlcur);
    if(pcnrctl->isVisible() &&  pcnrctl->isRefreshOn() &&
       (pcnrctl->flClState & IContainerControl::needsRefresh))
    {
      pcnrctl->flClState &= ~IContainerControl::needsRefresh;
      pcnrctl->refresh();
    }
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: ~IContainerObject

 Implementation: Virtual Destructor.
------------------------------------------------------------------------------*/
IContainerObject :: ~IContainerObject()
{
  IMODTRACE_ALL("CnrObj:~CnrObj");
  flClState |= deleteInProcess;   // ensure we don't delete an object twice
  IContainerControl* pcnrctl;
  IGUIResourceLock collectionLock(IContainerControl::containerKey());
  ICnrControlList cnrctllist = IContainerControl :: containerList();
  ICnrControlList::Cursor cnrctlcur(cnrctllist);

#ifdef IC_PMWIN
  if (!(this->isDelayedFree()))
#endif
  {
    // At this point, we know that at some point this object had
    // descendents, therefore we need to ensure that any children
    // get deleted as well.
    if(this->flClState & IContainerObject::hasDescendents)
    {
      // Cursor through all containers and remove all descendents by
      // iteratively removing (and possibly deleting) the last descendent.
      // After all descendents are processed, remove "this" object.
      for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
      {
        pcnrctl = cnrctllist.elementAt(cnrctlcur);
        if(pcnrctl->containsObject(this))
        {
          // remove last child repetitively

          IContainerControl::ObjectCursor  childIter(*pcnrctl, this, true);

          for (childIter.setToLast(); childIter.isValid(); childIter.setToLast())
          {

            //remove it then delete it if not already being deleted
            IContainerObject* pcnrobj = pcnrctl->objectAt(childIter);
            pcnrobj->setRefreshOff();
            pcnrctl->flClState |= IContainerControl::needsRefresh;
            pcnrctl->removeObject(pcnrobj);
            if(!(pcnrobj->flClState & deleteInProcess))
            {
              pcnrobj->flClState |= nestedDelete;
              delete pcnrobj;
            }
          }
          pcnrctl->removeObject(this);
        } /* endif (containsObject)  */
      } /* endfor (all containers)  */
    }
    else
    {
      // The simple case: no children, but more then one container
      // possible.
      for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
      {
         pcnrctl = cnrctllist.elementAt(cnrctlcur);
         if(pcnrctl->containsObject(this))
         {
  //         this->setRefreshOff();
           pcnrctl->removeObject( this );
           pcnrctl->flClState |= IContainerControl::needsRefresh;
         }
      }
    }

    /*------------------------------------------------------*/
    /* Refresh the windows if necessary if we get back to   */
    /* the root deleted object (nestedDelete not on) and the*/
    /* root object has refresh flag on.                     */
    /*------------------------------------------------------*/
    if(!(flClState & nestedDelete) && this->isRefreshOn())
    {
      for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
      {
        pcnrctl = cnrctllist.elementAt(cnrctlcur);
        if(pcnrctl->isVisible() &&  pcnrctl->isRefreshOn() &&
           (pcnrctl->flClState & IContainerControl::needsRefresh))
        {
          pcnrctl->flClState &= ~IContainerControl::needsRefresh;
          pcnrctl->refresh();
        }
      }
    }
  }

  /*------------------------------------------------------------------*/
  /* Delete the MiniRecordCore object.                                */
  /*------------------------------------------------------------------*/
  delete this->pbase;

  /*------------------------------------------------------------------*/
  /* If object uses private data then delete it.                      */
  /*------------------------------------------------------------------*/
  if (this->ppd)
    delete this->ppd;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: setEmphasis

 Implementation: Protected function to change object emphasis.
------------------------------------------------------------------------------*/
IContainerObject& IContainerObject :: setEmphasis(unsigned long ulEmphasisAttribute,
                                     bool fOn,
                                     IContainerControl* pcnrctl )
{
   /*------------------------------------------------------*/
   /* Set the emphasis on all containers.                  */
   /*------------------------------------------------------*/
   if(pcnrctl)
   {
       pcnrctl->setEmphasis(this, ulEmphasisAttribute, fOn);
   }
   else
   {
      IGUIResourceLock collectionLock(IContainerControl::containerKey());
      ICnrControlList& cnrctllist =
                                IContainerControl :: containerList();
      ICnrControlList::Cursor cnrctlcur(cnrctllist);
      for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
      {
         pcnrctl = cnrctllist.elementAt(cnrctlcur);
         if (pcnrctl->containsObject(this))
            pcnrctl->setEmphasis(this, ulEmphasisAttribute, fOn);
      }
   }
   return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: refresh

 Implementation: Invalidate this object in a specific container or
                 optionally in all containers that have refresh on.
                 This function does not consider the object's refresh state.
                 If immediate is set to true, painting will occur
                 synchronously, otherwise it will be async.  On Motif,
                 immediate is ignored.
------------------------------------------------------------------------------*/
IContainerObject& IContainerObject :: refresh(IContainerControl* pcnrctl,
                                              bool immediate)
{
  IMODTRACE_ALL("CnrObj:refresh");

  if(pcnrctl)    //update object only in this container
  {
     IMiniCnrRecord* pcnrrec = IRecFromObj(this);

     /*-----------------------------------------------------------------------*/
     /* In CCL container, we want to refresh only when visible.  In native,   */
     /* we must refresh every time.                                           */
     /*-----------------------------------------------------------------------*/
     bool isVisible = pcnrctl->isVisible();
#ifdef IC_PMWIN
     if (!pcnrctl->isPMCompatible())
        isVisible = true;                                            /* D8111 */
     else if (pcnrctl->isDetailsView())pcnrctl->refresh();
#endif
     /**********************************************************/
     /* Update the object only if following 3 are true         */
     /*   1) The container is visible.                         */
     /*   2) The object is actually in the container.          */
     /*   3) The container refresh state is on.                */
     /**********************************************************/
     if ( isVisible && pcnrctl->containsObject(this)) // &&              /* D8111 */
          // D27091 pcnrctl->isRefreshOn() )
     {
        pcnrctl->sendEvent(CM_INVALIDATERECORD,
                           MPFROMP(&pcnrrec),
                           MPFROM2SHORT(1,CMA_ERASE|CMA_TEXTCHANGED));
#ifdef IC_PMWIN
        if (immediate)
        {
          IUPDATEWINDOW(pcnrctl->handle());
        }
#endif
     }
  }
  else        //iterate through all containers and update object
  {
     IGUIResourceLock collectionLock(IContainerControl::containerKey());
     ICnrControlList& cnrctllist = IContainerControl :: containerList();
     ICnrControlList::Cursor cnrctlcur(cnrctllist);
     for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
     {
       pcnrctl = cnrctllist.elementAt(cnrctlcur);
       refresh(pcnrctl);
     }
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: setAttributes

 Implementation: Protected function to set attributes.
------------------------------------------------------------------------------*/
IContainerObject& IContainerObject :: setAttributes(unsigned long flAttrOff,
                                       unsigned long flAttrOn,
                                       IContainerControl* pcnrctl)
{
  IMODTRACE_ALL("CnrObj:setAttributes");

  if(pcnrctl)
  {
     pcnrctl->setAttributes(this, flAttrOff, flAttrOn);
  }
  else
  {
      IGUIResourceLock collectionLock(IContainerControl::containerKey());
      ICnrControlList& cnrctllist = IContainerControl :: containerList();
      ICnrControlList::Cursor cnrctlcur(cnrctllist);
      for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
      {
         pcnrctl = cnrctllist.elementAt(cnrctlcur);
         if (pcnrctl->containsObject(this))
            pcnrctl->setAttributes(this, flAttrOff, flAttrOn);
         else
         /*--------------------------------------------------------*/
         /* The following code is due to the fact that a user could*/
         /* be setting the attributes on an object before it gets  */
         /* added to a container and we want to ensure that these  */
         /* attributes apply to the object.                        */
         /*--------------------------------------------------------*/
         {
           this->pbase->flRecordAttr &= ~(flAttrOff);
           this->pbase->flRecordAttr |= (flAttrOn);
         }
      }
  }
  return *this;
}


/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: isAttribute

 Implementation: Protected function to check an attribute in all containers.
------------------------------------------------------------------------------*/
bool IContainerObject :: isAttribute(unsigned long flAttributeIn,
                                        IContainerControl* pcnrctl) const
{
  bool fTrueInAllCnr = true;
  unsigned long flAttr;
  if(pcnrctl)
  {
      flAttr = pcnrctl->attributes(this);
      return ((flAttr & flAttributeIn) ? true : false);
  }
  else
  {
       /*-----------------------------------------------------------*/
       /* Iterate all containers and check to see if the object     */
       /* exists and if the passed attribute is carried on the      */
       /* object. Return true if attribute is carried on the object */
       /* in ALL containers where it exists.                        */
       /*-----------------------------------------------------------*/
       IGUIResourceLock collectionLock(IContainerControl::containerKey());
       ICnrControlList& cnrctllist = IContainerControl :: containerList();
       ICnrControlList::Cursor cnrctlcur(cnrctllist);
       for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
       {
          pcnrctl = cnrctllist.elementAt(cnrctlcur);
          if(pcnrctl->containsObject(this))
          {
             flAttr = pcnrctl->attributes(this);
             if((flAttr & flAttributeIn) == 0  )
                 fTrueInAllCnr = false;
          }
       }
  }
  return fTrueInAllCnr;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: show

 Implementation: Adds back an object previously filtered out of the container.
------------------------------------------------------------------------------*/
IContainerObject& IContainerObject::show(bool fShow, IContainerControl* pcnrctl)
{
#define CMA_FILTER         0x1000         /* Filtering through invalidation  */

  if (pcnrctl)        //We have a container to work with.
  {
#ifdef IC_PMWIN
    unsigned long  fsFlags = 0;
    bool fInvalidateRequired=false;

    //Get current attributes for given container.
    unsigned long flAttr = pcnrctl->attributes(this);

    if (fShow)      //Showing.
    {
      // Unfilter the object and refresh if required.
      // Do nothing if already unfiltered.
      if ((flAttr & CRA_FILTERED))
      {
        pbase->flRecordAttr &= ~CRA_FILTERED;
        fInvalidateRequired=true;
      }
    }
    else          //Not showing(hiding).
      // Filter the object and refresh if required.
      // Do nothing if already filtered.
    {
      if (!(flAttr & CRA_FILTERED))
      {
        pbase->flRecordAttr |= CRA_FILTERED;
        fInvalidateRequired=true;
        //Set CMA_FILTER flag so PM container knows we are filtering
        //an object.  This flag should not be set when unfiltering.
        fsFlags |= CMA_FILTER;
      }
    }

    if (fInvalidateRequired)   //Something changed, must update container.
    {
      if (isRefreshOn() && pcnrctl->isRefreshOn())
      {
        //Since refresh is on, set flags necessary for
        //container to repaint properly.
        fsFlags |= (CMA_REPOSITION | CMA_ERASE);
        pcnrctl->sendEvent(CM_INVALIDATERECORD,&pbase,
                               MPFROM2SHORT(1, fsFlags));
      }
      else
      {
        //Refresh is off, so set flags for optimal update of container.
        fsFlags |= CMA_NOREPOSITION;
        pcnrctl->sendEvent(CM_INVALIDATERECORD,&pbase,
                               MPFROM2SHORT(1, fsFlags));
      }
    }
#endif //IC_PMWIN
#ifdef IC_MOTIF
    bool fFilterRequired = false;
    unsigned long flAttr = pcnrctl->attributes(this);
    if (fShow)
    {
       // unfilter the object and refresh if filtered
       if ( (flAttr & CRA_FILTERED) )
       {
          pbase->flRecordAttr &= ~CRA_FILTERED;
          if(isRefreshOn() && pcnrctl->isRefreshOn())
             fFilterRequired = true;
       }
    }
    else     // if not show
      // filter the object and refresh if not-filtered
    {
       if( !(flAttr & CRA_FILTERED) )
       {
          pbase->flRecordAttr |= CRA_FILTERED;
          if(isRefreshOn() && pcnrctl->isRefreshOn())
             fFilterRequired = true;
       }
    }    // end elseif fShow

    if(fFilterRequired)
    {
          IContainerHideFilter filterFn;
          pcnrctl->filter(filterFn);
    }
#endif //IC_MOTIF
  }
  else     //We have a null pcnrctl, so iterate all containers and
           //call show() for each one the object is contained in.
  {
    IGUIResourceLock collectionLock(IContainerControl::containerKey());
    ICnrControlList& cnrctllist = IContainerControl::containerList();
    ICnrControlList::Cursor cnrctlcur(cnrctllist);
    for(cnrctlcur.setToFirst(); cnrctlcur.isValid(); cnrctlcur.setToNext())
    {
      pcnrctl = cnrctllist.elementAt(cnrctlcur);
      if (pcnrctl->containsObject(this))
      {
        show(fShow, pcnrctl);
      }
    }
  }
  return *this;
}

/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: calcMinimumSize

 Implementation:  Return minimum size for a container object
------------------------------------------------------------------------------*/
ISize IContainerControl::calcMinimumSize() const
   {return ISize(50,50); }

/*-----------------------------------------------------------------------------
 Function Name: IContainerObject :: removeRecords

 Implementation: Call container to remove records.
------------------------------------------------------------------------------*/
long IContainerControl::removeRecords(void* prec, unsigned short number,
                                      unsigned short cmd)
{
  long retVal;

  if (prec)
  {
    retVal = (long)sendEvent(CM_REMOVERECORD, MPFROMP(&prec),
                             IEventData(number ,cmd));
  }
  else
  {
    retVal = (long)sendEvent(CM_REMOVERECORD, 0ul,
                             IEventData(number ,cmd));
  }

  if (retVal != -1)
  {
    notifyObservers(INotificationEvent(IContainerControl::removeId,
                                       *this, true, (void*)number));
  }
  return retVal;
}

#ifdef IC_MOTIF

// Defined in iwindow2.cpp
extern void
findFontListInChildren(Widget wh, XmFontList *fontList, bool (*criteria)(Widget) = 0);

IFont
IContainerControl::font() const
{
    XmFontList fontList = 0;
    Widget w = (Widget)handle();
    
    findFontListInChildren(w, &fontList);
    if (fontList)
        return IFont(fontList);
    else
        return IFont((IWindow*)0);
}

#endif
